package cmd

import (
	"os"

	"github.com/git-town/git-town/v7/src/cli"
	"github.com/git-town/git-town/v7/src/git"
	"github.com/git-town/git-town/v7/src/runstate"
	"github.com/git-town/git-town/v7/src/steps"
	"github.com/git-town/git-town/v7/src/userinput"
	"github.com/spf13/cobra"
)

type syncConfig struct {
	branchesToSync []string
	hasOrigin      bool
	initialBranch  string
	isOffline      bool
	shouldPushTags bool
}

var syncCmd = &cobra.Command{
	Use:   "sync",
	Short: "Updates the current branch with all relevant changes",
	Long: `Updates the current branch with all relevant changes

Synchronizes the current branch with the rest of the world.

When run on a feature branch
- syncs all ancestor branches
- pulls updates for the current branch
- merges the parent branch into the current branch
- pushes the current branch

When run on the main branch or a perennial branch
- pulls and pushes updates for the current branch
- pushes tags

If the repository contains an "upstream" remote,
syncs the main branch with its upstream counterpart.
You can disable this by running "git config git-town.sync-upstream false".`,
	Run: func(cmd *cobra.Command, args []string) {
		config, err := createSyncConfig(prodRepo)
		if err != nil {
			cli.Exit(err)
		}
		stepList, err := createSyncStepList(config, prodRepo)
		if err != nil {
			cli.Exit(err)
		}
		runState := runstate.New("sync", stepList)
		err = runstate.Execute(runState, prodRepo, nil)
		if err != nil {
			cli.Exit(err)
		}
	},
	Args: cobra.NoArgs,
	PreRunE: func(cmd *cobra.Command, args []string) error {
		if err := ValidateIsRepository(prodRepo); err != nil {
			return err
		}
		if dryRunFlag {
			currentBranch, err := prodRepo.Silent.CurrentBranch()
			if err != nil {
				return err
			}
			prodRepo.DryRun.Activate(currentBranch)
		}
		if err := validateIsConfigured(prodRepo); err != nil {
			return err
		}
		exit, err := handleUnfinishedState(prodRepo, nil)
		if err != nil {
			return err
		}
		if exit {
			os.Exit(0)
		}
		return nil
	},
}

func createSyncConfig(repo *git.ProdRepo) (result syncConfig, err error) {
	result.hasOrigin, err = repo.Silent.HasOrigin()
	if err != nil {
		return result, err
	}
	result.isOffline = prodRepo.Config.IsOffline()
	if result.hasOrigin && !result.isOffline {
		err := repo.Logging.Fetch()
		if err != nil {
			return result, err
		}
	}
	result.initialBranch, err = repo.Silent.CurrentBranch()
	if err != nil {
		return result, err
	}
	if allFlag {
		branches, err := repo.Silent.LocalBranchesMainFirst()
		if err != nil {
			return result, err
		}
		err = userinput.EnsureKnowsParentBranches(branches, repo)
		if err != nil {
			return result, err
		}
		result.branchesToSync = branches
		result.shouldPushTags = true
	} else {
		err = userinput.EnsureKnowsParentBranches([]string{result.initialBranch}, repo)
		if err != nil {
			return result, err
		}
		result.branchesToSync = append(prodRepo.Config.AncestorBranches(result.initialBranch), result.initialBranch)
		result.shouldPushTags = !prodRepo.Config.IsFeatureBranch(result.initialBranch)
	}
	return result, nil
}

func createSyncStepList(config syncConfig, repo *git.ProdRepo) (result runstate.StepList, err error) {
	for _, branchName := range config.branchesToSync {
		steps, err := runstate.SyncBranchSteps(branchName, true, repo)
		if err != nil {
			return result, err
		}
		result.AppendList(steps)
	}
	result.Append(&steps.CheckoutBranchStep{BranchName: config.initialBranch})
	if config.hasOrigin && config.shouldPushTags && !config.isOffline {
		result.Append(&steps.PushTagsStep{})
	}
	err = result.Wrap(runstate.WrapOptions{RunInGitRoot: true, StashOpenChanges: true}, repo)
	return result, err
}

func init() {
	syncCmd.Flags().BoolVar(&allFlag, "all", false, "Sync all local branches")
	syncCmd.Flags().BoolVar(&dryRunFlag, "dry-run", false, dryRunFlagDescription)
	RootCmd.AddCommand(syncCmd)
}
