package testruntime

import (
	"os"
	"path/filepath"
	"testing"

	"github.com/git-town/git-town/v17/internal/config"
	"github.com/git-town/git-town/v17/internal/config/configdomain"
	"github.com/git-town/git-town/v17/internal/config/gitconfig"
	"github.com/git-town/git-town/v17/internal/git"
	"github.com/git-town/git-town/v17/internal/git/gitdomain"
	"github.com/git-town/git-town/v17/internal/gohacks/cache"
	"github.com/git-town/git-town/v17/internal/gohacks/stringslice"
	. "github.com/git-town/git-town/v17/pkg/prelude"
	"github.com/git-town/git-town/v17/test/commands"
	testshell "github.com/git-town/git-town/v17/test/subshell"
	"github.com/shoenig/test/must"
)

// TestRuntime provides Git functionality for test code (unit and end-to-end tests).
type TestRuntime struct {
	commands.TestCommands
}

// Clone creates a clone of the repository managed by this test.Runner into the given directory.
// The cloned repo uses the same homeDir and binDir as its origin.
func Clone(original *testshell.TestRunner, targetDir string) commands.TestCommands {
	original.MustRun("git", "clone", original.WorkingDir, targetDir)
	return New(targetDir, original.HomeDir, original.BinDir)
}

// Create creates test.Runner instances.
func Create(t *testing.T) commands.TestCommands {
	t.Helper()
	dir := t.TempDir()
	workingDir := filepath.Join(dir, "repo")
	err := os.Mkdir(workingDir, 0o744)
	must.NoError(t, err)
	homeDir := filepath.Join(dir, "home")
	err = os.Mkdir(homeDir, 0o744)
	must.NoError(t, err)
	runtime := Initialize(workingDir, homeDir, homeDir)
	must.NoError(t, err)
	return runtime
}

// CreateGitTown creates a test.Runtime for use in tests,
// with a main branch and initial git town configuration.
func CreateGitTown(t *testing.T) commands.TestCommands {
	t.Helper()
	repo := Create(t)
	repo.CreateBranch(gitdomain.NewLocalBranchName("main"), gitdomain.NewBranchName("initial"))
	err := repo.Config.SetMainBranch(gitdomain.NewLocalBranchName("main"))
	must.NoError(t, err)
	err = repo.Config.NormalConfig.SetPerennialBranches(gitdomain.LocalBranchNames{})
	must.NoError(t, err)
	return repo
}

// initialize creates a fully functioning test.Runner in the given working directory,
// including necessary Git configuration to make commits. Creates missing folders as needed.
func Initialize(workingDir, homeDir, binDir string) commands.TestCommands {
	runtime := New(workingDir, homeDir, binDir)
	runtime.MustRun("git", "init", "--initial-branch=initial")
	runtime.MustRun("git", "config", "--global", "user.name", "user")
	runtime.MustRun("git", "config", "--global", "user.email", "email@example.com")
	runtime.MustRun("git", "commit", "--allow-empty", "-m", "initial commit")
	return runtime
}

// newRuntime provides a new test.Runner instance working in the given directory.
// The directory must contain an existing Git repo.
func New(workingDir, homeDir, binDir string) commands.TestCommands {
	testRunner := testshell.TestRunner{
		BinDir:           binDir,
		HomeDir:          homeDir,
		ProposalOverride: None[string](),
		Verbose:          false,
		WorkingDir:       workingDir,
	}
	gitCommands := git.Commands{
		CurrentBranchCache: &cache.LocalBranchWithPrevious{},
		RemotesCache:       &cache.Remotes{},
	}
	unvalidatedConfig := config.NewUnvalidatedConfig(config.NewUnvalidatedConfigArgs{
		Access: gitconfig.Access{
			Runner: &testRunner,
		},
		ConfigFile:    None[configdomain.PartialConfig](),
		DryRun:        false,
		FinalMessages: stringslice.NewCollector(),
		GitVersion:    git.Version{Major: 2, Minor: 38},
		GlobalConfig:  configdomain.EmptyPartialConfig(),
		LocalConfig:   configdomain.EmptyPartialConfig(),
	})
	unvalidatedConfig.UnvalidatedConfig.MainBranch = Some(gitdomain.NewLocalBranchName("main"))
	testCommands := commands.TestCommands{
		Commands:   &gitCommands,
		Config:     unvalidatedConfig,
		TestRunner: &testRunner,
	}
	return testCommands
}
