// Copyright (c) 2019 SAP SE or an SAP affiliate company. All rights reserved. This file is licensed under the Apache Software License, v. 2 except as noted otherwise in the LICENSE file
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package controlplane

import (
	"github.com/gardener/gardener/pkg/utils"

	corev1 "k8s.io/api/core/v1"
)

// MergeSecretMaps merges the 2 given secret maps.
func MergeSecretMaps(a, b map[string]*corev1.Secret) map[string]*corev1.Secret {
	x := make(map[string]*corev1.Secret)
	for _, m := range []map[string]*corev1.Secret{a, b} {
		for k, v := range m {
			x[k] = v
		}
	}
	return x
}

// ComputeChecksums computes and returns SAH256 checksums for the given secrets and configmaps.
func ComputeChecksums(secrets map[string]*corev1.Secret, cms map[string]*corev1.ConfigMap) map[string]string {
	checksums := make(map[string]string, len(secrets)+len(cms))
	for name, secret := range secrets {
		checksums[name] = utils.ComputeChecksum(secret.Data)
	}
	for name, cm := range cms {
		checksums[name] = utils.ComputeChecksum(cm.Data)
	}
	return checksums
}
