// Copyright 2019 Dolthub, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package integration_test

import (
	"context"
	"strings"
	"testing"

	"github.com/dolthub/go-mysql-server/sql"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/dolthub/dolt/go/libraries/doltcore/dtestutils"
	"github.com/dolthub/dolt/go/libraries/doltcore/sqle"
	"github.com/dolthub/dolt/go/store/types"
)

// This tests running queries against a modified subset of the stockmarket data set found here:
// https://beta.dolthub.com/repositories/Liquidata/stock-market

const createTables = `
CREATE TABLE daily_summary (
  Type varchar(80),
  Symbol varchar(80) NOT NULL,
  Country varchar(80) NOT NULL,
  TradingDate varchar(80) NOT NULL,
  Open float ,
  High float ,
  Low float ,
  Close float ,
  Volume float ,
  OpenInt varchar(80),
  PRIMARY KEY (Symbol,Country,TradingDate)
);

CREATE TABLE symbols (
  Symbol varchar(10) NOT NULL,
  Name varchar(1024),
  Sector varchar(1024),
  IPOYear int(11),
  PRIMARY KEY (Symbol)
);

CREATE TABLE join_result (
  Type varchar(80) ,
  Symbol varchar(80) NOT NULL,
  Country varchar(80) NOT NULL,
  TradingDate varchar(80)  NOT NULL,
  Open float ,
  High float ,
  Low float ,
  Close float ,
  Volume float ,
  OpenInt varchar(80) ,
  Name varchar(1024),
  Sector varchar(1024),
  IPOYear int(11),
  PRIMARY KEY (Symbol,Country,TradingDate)
);
`

const createIndexes = `
CREATE INDEX idx_country ON daily_summary (Country);
CREATE INDEX idx_ipoyear ON symbols (IPOYear);
CREATE INDEX idx_country ON join_result (Country);
`

const insertRows = `
INSERT INTO daily_summary VALUES ('stock','A','us','2017-11-01',76.2,76.65,74.85,75.3,70140,'0');
INSERT INTO daily_summary VALUES ('stock','AA','us','2017-11-01',48.91,49.27,47.45,47.7,3174960,'0');
INSERT INTO daily_summary VALUES ('stock','AAAP','us','2017-11-01',81,81.5,80.81,81.03,1598910,'0');
INSERT INTO daily_summary VALUES ('stock','AABA','us','2017-11-01',70.85,71.385,69.84,70.59,9883640,'0');
INSERT INTO daily_summary VALUES ('stock','AAC','us','2017-11-01',7.96,7.97,7.53,7.55,118113,'0');
INSERT INTO daily_summary VALUES ('etf','AADR','us','2017-11-01',57.97,57.979,57.46,57.46,33382,'0');
INSERT INTO daily_summary VALUES ('stock','AAL','us','2017-11-01',47.895,48.095,47.267,47.735,4364060,'0');
INSERT INTO daily_summary VALUES ('stock','AAME','us','2017-11-01',3.7,3.8,3.55,3.8,9641,'0');
INSERT INTO daily_summary VALUES ('stock','AAN','us','2017-11-01',37.17,37.2401,35.57,35.93,1712160,'0');
INSERT INTO daily_summary VALUES ('stock','AAOI','us','2017-11-01',40.41,41.4664,38.8301,38.97,1270560,'0');
INSERT INTO daily_summary VALUES ('stock','AAON','us','2017-11-01',35.25,35.25,34.1,34.4,118289,'0');
INSERT INTO daily_summary VALUES ('stock','AAP','us','2017-11-01',82.33,82.74,80.68,80.75,1157570,'0');
INSERT INTO daily_summary VALUES ('stock','AAPL','us','2017-11-01',169.26,169.33,165.02,166.29,33758700,'0');
INSERT INTO daily_summary VALUES ('stock','AAT','us','2017-11-01',37.25,38.87,36.791,38.76,677539,'0');
INSERT INTO daily_summary VALUES ('stock','AAU','us','2017-11-01',1.06,1.06,1,1,279964,'0');
INSERT INTO daily_summary VALUES ('stock','AAV','us','2017-11-01',5.35,5.6343,5.35,5.6,282213,'0');
INSERT INTO daily_summary VALUES ('stock','AAWW','us','2017-11-01',61.95,62.625,60.75,61.7,243782,'0');
INSERT INTO daily_summary VALUES ('etf','AAXJ','us','2017-11-01',76.23,76.35,75.85,75.96,2869470,'0');
INSERT INTO daily_summary VALUES ('stock','AAXN','us','2017-11-01',23,23.045,22.81,23.03,429431,'0');
INSERT INTO daily_summary VALUES ('stock','AB','us','2017-11-01',25.393,25.491,25.148,25.393,138047,'0');
INSERT INTO daily_summary VALUES ('stock','ABAC','us','2017-11-01',3.3836,3.4,3.3284,3.4,10415,'0');
INSERT INTO daily_summary VALUES ('stock','ABAX','us','2017-11-01',48.34,48.38,47.26,47.75,98266,'0');
INSERT INTO daily_summary VALUES ('stock','ABB','us','2017-11-01',26.47,26.48,26.24,26.24,2051220,'0');
INSERT INTO daily_summary VALUES ('stock','ABBV','us','2017-11-01',90.78,93.34,90.27,92.44,7048770,'0');
INSERT INTO daily_summary VALUES ('stock','ABC','us','2017-11-01',77.48,77.495,76.31,76.62,2546090,'0');
INSERT INTO daily_summary VALUES ('stock','ABCB','us','2017-11-01',48.3,48.65,47.15,47.65,83249,'0');
INSERT INTO daily_summary VALUES ('stock','ABCD','us','2017-11-01',6.15,6.195,5.95,6.1,44827,'0');
INSERT INTO daily_summary VALUES ('stock','ABCO','us','2017-11-01',53.95,53.95,53.675,53.8,852067,'0');
INSERT INTO daily_summary VALUES ('stock','ABDC','us','2017-11-01',9.26,9.33,8.51,8.75,226659,'0');
INSERT INTO daily_summary VALUES ('stock','ABE','us','2017-11-01',14.51,14.53,14.43,14.43,14736,'0');
INSERT INTO daily_summary VALUES ('stock','ABEO','us','2017-11-01',18.1,18.1,16.15,16.4,1337610,'0');
INSERT INTO daily_summary VALUES ('stock','ABEOW','us','2017-11-01',11.78,11.93,11.16,11.4,8701,'0');
INSERT INTO daily_summary VALUES ('stock','ABEV','us','2017-11-01',6.38,6.45,6.29,6.29,20600600,'0');
INSERT INTO daily_summary VALUES ('stock','ABG','us','2017-11-01',61.25,62.55,60.65,62.45,361247,'0');
INSERT INTO daily_summary VALUES ('stock','ABIL','us','2017-11-01',0.6404,0.66,0.6404,0.65,15969,'0');
INSERT INTO daily_summary VALUES ('stock','ABIO','us','2017-11-01',1.25,1.2719,1.21,1.225,5334,'0');
INSERT INTO daily_summary VALUES ('stock','ABLX','us','2017-11-01',20.73,21.175,20.61,21,772794,'0');
INSERT INTO daily_summary VALUES ('stock','ABM','us','2017-11-01',42.08,42.195,41,41.13,226837,'0');
INSERT INTO daily_summary VALUES ('stock','ABMD','us','2017-11-01',192.98,194.42,188.12,188.89,386670,'0');
INSERT INTO daily_summary VALUES ('stock','ABR','us','2017-11-01',8.25,8.31,8.17,8.21,253862,'0');
INSERT INTO daily_summary VALUES ('stock','ABRN','us','2017-11-01',25.31,25.31,25.25,25.26,6608,'0');
INSERT INTO daily_summary VALUES ('stock','ABR_B','us','2017-11-01',25.44,25.59,25.34,25.369,2669,'0');
INSERT INTO daily_summary VALUES ('stock','ABT','us','2017-11-01',53.89,54.57,53.6101,54,7641900,'0');
INSERT INTO daily_summary VALUES ('stock','ABTX','us','2017-11-01',39.7,39.7,38.05,38.3,38480,'0');
INSERT INTO daily_summary VALUES ('stock','ABUS','us','2017-11-01',5.8,5.89,5.35,5.4,301358,'0');
INSERT INTO daily_summary VALUES ('stock','ABX','us','2017-11-01',14.52,14.56,14.21,14.22,12455800,'0');
INSERT INTO daily_summary VALUES ('stock','ABY','us','2017-11-01',22.4,22.5,22.305,22.39,593918,'0');
INSERT INTO daily_summary VALUES ('stock','AC','us','2017-11-01',37.55,37.9,37,37.1,15044,'0');
INSERT INTO daily_summary VALUES ('stock','ACAD','us','2017-11-01',35.06,35.1992,34.08,34.36,685977,'0');
INSERT INTO daily_summary VALUES ('stock','ACBI','us','2017-11-01',16.6,16.7,16.25,16.4,119908,'0');
INSERT INTO daily_summary VALUES ('stock','ACC','us','2017-11-01',41.277,41.801,40.891,41.762,946618,'0');
INSERT INTO daily_summary VALUES ('stock','ACCO','us','2017-11-01',13.1,13.25,12.7,12.7,693354,'0');
INSERT INTO daily_summary VALUES ('stock','ACER','us','2017-11-01',17.3,17.929,17.25,17.7501,9543,'0');
INSERT INTO daily_summary VALUES ('stock','ACERW','us','2017-11-01',0.055,0.083,0.039,0.06,9720150,'0');
INSERT INTO daily_summary VALUES ('stock','ACET','us','2017-11-01',10.13,10.13,9.54,9.74,434609,'0');
INSERT INTO daily_summary VALUES ('stock','ACFC','us','2017-11-01',8.705,8.71,8.5,8.51,7309,'0');
INSERT INTO daily_summary VALUES ('stock','ACGL','us','2017-11-01',99.9,99.9,97.78,97.85,522741,'0');
INSERT INTO daily_summary VALUES ('stock','ACGLO','us','2017-11-01',25.35,25.35,25.25,25.3,71198,'0');
INSERT INTO daily_summary VALUES ('stock','ACGLP','us','2017-11-01',24.98,25,24.92,24.96,100522,'0');
INSERT INTO daily_summary VALUES ('stock','ACH','us','2017-11-01',19.58,19.6,19.33,19.33,61051,'0');
INSERT INTO daily_summary VALUES ('stock','ACHC','us','2017-11-01',31.33,31.54,29.92,30.47,2886310,'0');
INSERT INTO daily_summary VALUES ('stock','ACHN','us','2017-11-01',4.06,4.06,3.86,3.9,810516,'0');
INSERT INTO daily_summary VALUES ('stock','ACHV','us','2017-11-01',1.73,1.82,1.6,1.77,60304,'0');
INSERT INTO daily_summary VALUES ('stock','ACIA','us','2017-11-01',42.53,43.91,42.21,43.4,630425,'0');
INSERT INTO daily_summary VALUES ('etf','ACIM','us','2017-11-01',77.2196,77.359,77.0512,77.0512,1085,'0');
INSERT INTO daily_summary VALUES ('stock','ACIU','us','2017-11-01',11.77,12.2,11.732,11.95,110309,'0');
INSERT INTO daily_summary VALUES ('stock','ACIW','us','2017-11-01',24.3,24.43,23.04,23.77,955258,'0');
INSERT INTO daily_summary VALUES ('stock','ACLS','us','2017-11-01',33.3,33.55,31.115,31.8,1336220,'0');
INSERT INTO daily_summary VALUES ('stock','ACM','us','2017-11-01',35.27,35.49,34.97,35.22,485838,'0');
INSERT INTO daily_summary VALUES ('stock','ACN','us','2017-11-01',142.26,143.19,141.47,142.96,2060110,'0');
INSERT INTO daily_summary VALUES ('stock','ACNB','us','2017-11-01',27.9,28.05,27.5,27.6,4181,'0');
INSERT INTO daily_summary VALUES ('stock','ACOR','us','2017-11-01',26.65,29.6,26.65,28.15,946216,'0');
INSERT INTO daily_summary VALUES ('stock','ACP','us','2017-11-01',14.505,14.554,14.425,14.465,92529,'0');
INSERT INTO daily_summary VALUES ('stock','ACRE','us','2017-11-01',13.52,13.6,12.98,13.16,308479,'0');
INSERT INTO daily_summary VALUES ('stock','ACRS','us','2017-11-01',25.25,25.47,24.77,25.01,137399,'0');
INSERT INTO daily_summary VALUES ('stock','ACRX','us','2017-11-01',1.9,2.05,1.9,2,858306,'0');
INSERT INTO daily_summary VALUES ('stock','ACSF','us','2017-11-01',11.05,11.15,11,11.05,63597,'0');
INSERT INTO daily_summary VALUES ('stock','ACSI','us','2017-11-01',29.64,29.64,29.52,29.56,2124,'0');
INSERT INTO daily_summary VALUES ('stock','ACST','us','2017-11-01',1.2834,1.3,1.27,1.3,12107,'0');
INSERT INTO daily_summary VALUES ('stock','ACTA','us','2017-11-01',15.5,15.5,15.4,15.45,111612,'0');
INSERT INTO daily_summary VALUES ('stock','ACTG','us','2017-11-01',4.6,4.6,4.4,4.475,281153,'0');
INSERT INTO daily_summary VALUES ('stock','ACU','us','2017-11-01',22.07,22.3,20.2,22.3,10673,'0');
INSERT INTO daily_summary VALUES ('stock','ACV','us','2017-11-01',21.931,22.051,21.881,21.881,22807,'0');
INSERT INTO daily_summary VALUES ('etf','ACWF','us','2017-11-01',30.07,30.17,30,30,7180,'0');
INSERT INTO daily_summary VALUES ('etf','ACWI','us','2017-11-01',70.63,70.71,70.3868,70.42,1493180,'0');
INSERT INTO daily_summary VALUES ('etf','ACWV','us','2017-11-01',82.86,82.9,82.55,82.59,353507,'0');
INSERT INTO daily_summary VALUES ('etf','ACWX','us','2017-11-01',49.55,49.63,49.39,49.42,466069,'0');
INSERT INTO daily_summary VALUES ('stock','ACXM','us','2017-11-01',25.39,25.39,24.6,24.65,700674,'0');
INSERT INTO daily_summary VALUES ('stock','ACY','us','2017-11-01',13.8,13.85,13.3,13.3,4308,'0');
INSERT INTO daily_summary VALUES ('stock','ADAP','us','2017-11-01',7.91,7.95,7.5,7.51,213838,'0');
INSERT INTO daily_summary VALUES ('stock','ADBE','us','2017-11-01',176.59,176.94,174.7,176.25,2002920,'0');
INSERT INTO daily_summary VALUES ('stock','ADC','us','2017-11-01',47.55,47.8,47.235,47.58,127534,'0');
INSERT INTO daily_summary VALUES ('stock','ADES','us','2017-11-01',12.05,12.05,11.56,11.8,87222,'0');
INSERT INTO daily_summary VALUES ('stock','ADHD','us','2017-11-01',1.59,1.59,1.4607,1.49,472423,'0');
INSERT INTO daily_summary VALUES ('stock','ADI','us','2017-11-01',91.85,92.1104,90.59,90.99,2373950,'0');
INSERT INTO daily_summary VALUES ('stock','ADM','us','2017-11-01',40.97,41.11,40.08,40.62,7011630,'0');
INSERT INTO daily_summary VALUES ('stock','ADMA','us','2017-11-01',2.42,2.5,2.3604,2.44,67921,'0');
INSERT INTO daily_summary VALUES ('stock','ADMP','us','2017-11-01',5.25,5.25,5,5.05,452598,'0');
INSERT INTO daily_summary VALUES ('stock','ADMS','us','2017-11-01',27.05,27.87,25.5,26.62,4401750,'0');
INSERT INTO daily_summary VALUES ('stock','ADNT','us','2017-11-01',83.21,85.32,83.18,84.93,1065710,'0');
INSERT INTO daily_summary VALUES ('stock','ADOM','us','2017-11-01',5,5.07,4.31,4.55,114003,'0');
INSERT INTO daily_summary VALUES ('stock','ADP','us','2017-11-01',116.4,116.77,114.69,115.4,2751750,'0');
INSERT INTO daily_summary VALUES ('etf','ADRA','us','2017-11-01',34.3507,34.3507,34.064,34.064,606,'0');
INSERT INTO daily_summary VALUES ('etf','ADRD','us','2017-11-01',23.1,23.206,23.04,23.0715,2771,'0');
INSERT INTO daily_summary VALUES ('etf','ADRE','us','2017-11-01',43.17,43.4212,43.0508,43.09,12331,'0');
INSERT INTO daily_summary VALUES ('stock','ADRO','us','2017-11-01',6.25,8.2,6.005,7.55,1100820,'0');
INSERT INTO daily_summary VALUES ('etf','ADRU','us','2017-11-01',22.472,22.472,22.472,22.472,244,'0');
INSERT INTO daily_summary VALUES ('stock','ADS','us','2017-11-01',223.94,226.12,220.08,225.68,482828,'0');
INSERT INTO daily_summary VALUES ('stock','ADSK','us','2017-11-01',125.63,126.44,123.4,124.78,1622830,'0');
INSERT INTO daily_summary VALUES ('stock','ADSW','us','2017-11-01',24.85,24.88,24.36,24.53,356330,'0');
INSERT INTO daily_summary VALUES ('stock','ADTN','us','2017-11-01',21.25,21.35,20.85,21,283453,'0');
INSERT INTO daily_summary VALUES ('stock','ADUS','us','2017-11-01',36.15,37.595,34.625,34.7,12713,'0');
INSERT INTO daily_summary VALUES ('stock','ADVM','us','2017-11-01',3.25,3.275,3.05,3.15,171028,'0');
INSERT INTO daily_summary VALUES ('stock','ADX','us','2017-11-01',15.66,15.69,15.57,15.64,169909,'0');
INSERT INTO daily_summary VALUES ('stock','ADXS','us','2017-11-01',3.44,3.45,3.18,3.18,592393,'0');
INSERT INTO daily_summary VALUES ('stock','ADXSW','us','2017-11-01',1.42,1.45,1.31,1.314,3445,'0');
INSERT INTO daily_summary VALUES ('stock','AE','us','2017-11-01',43.52,43.94,43.12,43.49,4431,'0');
INSERT INTO daily_summary VALUES ('stock','AEB','us','2017-11-01',24.47,24.47,24.23,24.24,8078,'0');
INSERT INTO daily_summary VALUES ('stock','AED','us','2017-11-01',25.96,25.9799,25.86,25.88,28317,'0');
INSERT INTO daily_summary VALUES ('stock','AEE','us','2017-11-01',61.98,62.04,61.48,61.8,1253940,'0');
INSERT INTO daily_summary VALUES ('stock','AEG','us','2017-11-01',5.92,5.9688,5.91,5.95,1764800,'0');
INSERT INTO daily_summary VALUES ('stock','AEGN','us','2017-11-01',23.59,23.6,22.89,23.14,237886,'0');
INSERT INTO daily_summary VALUES ('stock','AEH','us','2017-11-01',25.8772,25.89,25.82,25.82,37938,'0');
INSERT INTO daily_summary VALUES ('stock','AEHR','us','2017-11-01',3.6035,3.6035,3.45,3.46,18387,'0');
INSERT INTO daily_summary VALUES ('stock','AEIS','us','2017-11-01',85.78,86.46,80.67,81.98,907263,'0');
INSERT INTO daily_summary VALUES ('stock','AEK','us','2017-11-01',26.28,26.389,26.28,26.28,30674,'0');
INSERT INTO daily_summary VALUES ('stock','AEL','us','2017-11-01',29.85,29.96,29.39,29.51,306408,'0');
INSERT INTO daily_summary VALUES ('stock','AEM','us','2017-11-01',44.92,45,44.2,44.26,1196030,'0');
INSERT INTO daily_summary VALUES ('stock','AEMD','us','2017-11-01',0.9832,0.9909,0.9684,0.9748,165145,'0');
INSERT INTO daily_summary VALUES ('stock','AEO','us','2017-11-01',13.03,13.12,12.751,12.94,3581950,'0');
INSERT INTO daily_summary VALUES ('stock','AEP','us','2017-11-01',73.809,73.997,73.105,73.402,1628510,'0');
INSERT INTO daily_summary VALUES ('stock','AER','us','2017-11-01',53.03,53.13,52.08,52.86,1452970,'0');
INSERT INTO daily_summary VALUES ('stock','AERI','us','2017-11-01',62.35,63.7,61,62.45,447568,'0');
INSERT INTO daily_summary VALUES ('stock','AES','us','2017-11-01',10.59,10.66,10.55,10.57,7046920,'0');
INSERT INTO daily_summary VALUES ('stock','AET','us','2017-11-01',170.92,172.27,169.69,171.8,2367780,'0');
INSERT INTO daily_summary VALUES ('stock','AETI','us','2017-11-01',1.55,1.6,1.55,1.5975,7497,'0');
INSERT INTO daily_summary VALUES ('stock','AEUA','us','2017-11-01',38,39.13,37.1953,38.7,8002,'0');
INSERT INTO daily_summary VALUES ('stock','AEY','us','2017-11-01',1.425,1.4258,1.39,1.418,13149,'0');
INSERT INTO daily_summary VALUES ('stock','AEZS','us','2017-11-01',1.9,1.9,1.8598,1.89,104757,'0');
INSERT INTO daily_summary VALUES ('stock','AFAM','us','2017-11-01',44.35,44.35,40.15,40.4,260745,'0');
INSERT INTO daily_summary VALUES ('stock','AFB','us','2017-11-01',13.58,13.63,13.54,13.58,47684,'0');
INSERT INTO daily_summary VALUES ('stock','AFC','us','2017-11-01',25.52,25.6,25.52,25.6,12704,'0');
INSERT INTO daily_summary VALUES ('stock','AFG','us','2017-11-01',105.02,105.305,101.015,101.81,789230,'0');
INSERT INTO daily_summary VALUES ('stock','AFGE','us','2017-11-01',26.26,26.3971,26.235,26.28,3942,'0');
INSERT INTO daily_summary VALUES ('stock','AFGH','us','2017-11-01',26.18,26.504,26.07,26.48,6333,'0');
INSERT INTO daily_summary VALUES ('stock','AFH','us','2017-11-01',19.9,19.95,19.6,19.9,13288,'0');
INSERT INTO daily_summary VALUES ('stock','AFHBL','us','2017-11-01',26.43,26.43,26.2419,26.2419,1590,'0');
INSERT INTO daily_summary VALUES ('stock','AFI','us','2017-11-01',14.93,14.98,14.29,14.47,122023,'0');
INSERT INTO daily_summary VALUES ('etf','AFK','us','2017-11-01',23.8,23.8,23.63,23.8,14310,'0');
INSERT INTO daily_summary VALUES ('stock','AFL','us','2017-11-01',84.14,84.65,83.99,84.29,1415070,'0');
INSERT INTO daily_summary VALUES ('stock','AFMD','us','2017-11-01',2.1,2.25,2,2.025,188846,'0');
INSERT INTO daily_summary VALUES ('stock','AFSD','us','2017-11-01',25.22,25.26,25.21,25.21,24794,'0');
INSERT INTO daily_summary VALUES ('stock','AFSI','us','2017-11-01',12.66,13.08,12.47,12.54,873035,'0');
INSERT INTO daily_summary VALUES ('stock','AFSI_A','us','2017-11-01',24.05,24.28,24.05,24.25,2894,'0');
INSERT INTO daily_summary VALUES ('stock','AFSI_B','us','2017-11-01',24.79,25.18,24.755,25.07,30197,'0');
INSERT INTO daily_summary VALUES ('stock','AFSI_C','us','2017-11-01',24.9,24.97,24.825,24.96,18491,'0');
INSERT INTO daily_summary VALUES ('stock','AFSI_D','us','2017-11-01',24.62,24.63,24.4,24.59,24910,'0');
INSERT INTO daily_summary VALUES ('stock','AFSI_E','us','2017-11-01',25.4255,25.71,25.336,25.65,6402,'0');
INSERT INTO daily_summary VALUES ('stock','AFSI_F','us','2017-11-01',23.09,23.14,22.83,23,55159,'0');
INSERT INTO daily_summary VALUES ('stock','AFSS','us','2017-11-01',25.88,25.995,25.822,25.995,2891,'0');
INSERT INTO daily_summary VALUES ('stock','AFST','us','2017-11-01',26.94,26.94,26.68,26.75,2462,'0');
INSERT INTO daily_summary VALUES ('stock','AFT','us','2017-11-01',16.72,16.75,16.65,16.67,56845,'0');
INSERT INTO daily_summary VALUES ('stock','AFTY','us','2017-11-01',17.95,18.0299,17.95,18,7421,'0');
INSERT INTO daily_summary VALUES ('stock','AG','us','2017-11-01',6.84,7.03,6.761,6.86,2723300,'0');
INSERT INTO daily_summary VALUES ('stock','AGC','us','2017-11-01',6.19,6.2,6.15,6.15,104371,'0');
INSERT INTO daily_summary VALUES ('stock','AGCO','us','2017-11-01',69.23,70.07,69.05,69.43,893937,'0');
INSERT INTO daily_summary VALUES ('stock','AGD','us','2017-11-01',10.66,10.69,10.6,10.63,120265,'0');
INSERT INTO daily_summary VALUES ('stock','AGEN','us','2017-11-01',3.67,3.74,3.54,3.58,1302090,'0');
INSERT INTO daily_summary VALUES ('stock','AGFS','us','2017-11-01',5.95,6.03,5.9,5.94,52326,'0');
INSERT INTO daily_summary VALUES ('stock','AGFSW','us','2017-11-01',0.3282,0.37,0.3282,0.3459,5284,'0');
INSERT INTO daily_summary VALUES ('etf','AGG','us','2017-11-01',109.17,109.41,109.17,109.29,2959360,'0');
INSERT INTO daily_summary VALUES ('stock','AGGE','us','2017-11-01',19.65,19.669,19.641,19.663,1416,'0');
INSERT INTO daily_summary VALUES ('stock','AGGP','us','2017-11-01',20.08,20.1,20.051,20.08,38951,'0');
INSERT INTO daily_summary VALUES ('stock','AGGY','us','2017-11-01',50.64,50.77,50.635,50.715,67370,'0');
INSERT INTO daily_summary VALUES ('stock','AGI','us','2017-11-01',6.44,6.44,6.24,6.25,2035030,'0');
INSERT INTO daily_summary VALUES ('stock','AGII','us','2017-11-01',63.25,63.95,62.4,62.65,139798,'0');
INSERT INTO daily_summary VALUES ('stock','AGIIL','us','2017-11-01',25.4199,25.42,25.3814,25.4199,7736,'0');
INSERT INTO daily_summary VALUES ('stock','AGIO','us','2017-11-01',66.75,66.75,61.13,62.01,893336,'0');
INSERT INTO daily_summary VALUES ('stock','AGLE','us','2017-11-01',4.34,4.41,4.2001,4.23,18268,'0');
INSERT INTO daily_summary VALUES ('stock','AGM','us','2017-11-01',74.95,76.06,73.9,74.27,62045,'0');
INSERT INTO daily_summary VALUES ('stock','AGM_A','us','2017-11-01',25.4,25.4,25.3,25.3,510,'0');
INSERT INTO daily_summary VALUES ('stock','AGM_B','us','2017-11-01',26.7318,26.79,26.67,26.77,1350,'0');
INSERT INTO daily_summary VALUES ('stock','AGN','us','2017-11-01',176.59,188.36,169.643,184.58,11863800,'0');
INSERT INTO daily_summary VALUES ('stock','AGNC','us','2017-11-01',20.16,20.229,20.05,20.2,6028620,'0');
INSERT INTO daily_summary VALUES ('stock','AGNCB','us','2017-11-01',26.15,26.3272,26.15,26.29,3943,'0');
INSERT INTO daily_summary VALUES ('stock','AGNCN','us','2017-11-01',26.75,26.75,26.42,26.49,95395,'0');
INSERT INTO daily_summary VALUES ('etf','AGND','us','2017-11-01',43.47,43.555,43.47,43.52,122140,'0');
INSERT INTO daily_summary VALUES ('stock','AGN_A','us','2017-11-01',636.86,681.48,621.22,668.42,105544,'0');
INSERT INTO daily_summary VALUES ('stock','AGO','us','2017-11-01',37.1,37.9,37.1,37.2,1231170,'0');
INSERT INTO daily_summary VALUES ('stock','AGO_B','us','2017-11-01',26.092,26.0981,26.092,26.0981,495,'0');
INSERT INTO daily_summary VALUES ('stock','AGO_E','us','2017-11-01',26.36,26.47,26.26,26.4,9157,'0');
INSERT INTO daily_summary VALUES ('stock','AGO_F','us','2017-11-01',25.2296,25.29,25.214,25.214,3866,'0');
INSERT INTO daily_summary VALUES ('etf','AGQ','us','2017-11-01',33.92,35.1699,33.84,34.99,485267,'0');
INSERT INTO daily_summary VALUES ('stock','AGR','us','2017-11-01',51.05,51.85,50.3421,51.27,661248,'0');
INSERT INTO daily_summary VALUES ('stock','AGRO','us','2017-11-01',10.25,10.43,10.19,10.33,664238,'0');
INSERT INTO daily_summary VALUES ('stock','AGRX','us','2017-11-01',4.77,4.909,4.6866,4.75,73174,'0');
INSERT INTO daily_summary VALUES ('stock','AGT','us','2017-11-01',27.76,27.76,27.4,27.4,2812,'0');
INSERT INTO daily_summary VALUES ('stock','AGTC','us','2017-11-01',3.4,3.55,3.25,3.55,176886,'0');
INSERT INTO daily_summary VALUES ('stock','AGU','us','2017-11-01',110.02,110.25,108.99,109.51,279025,'0');
INSERT INTO daily_summary VALUES ('stock','AGX','us','2017-11-01',69.3,69.3,68,68.9,170982,'0');
INSERT INTO daily_summary VALUES ('stock','AGYS','us','2017-11-01',12.5,12.5,11.93,12.415,68265,'0');
INSERT INTO daily_summary VALUES ('etf','AGZ','us','2017-11-01',113.281,113.39,113.254,113.3,9832,'0');
INSERT INTO daily_summary VALUES ('etf','AGZD','us','2017-11-01',48.27,48.27,48.142,48.142,3051,'0');
INSERT INTO daily_summary VALUES ('stock','AHC','us','2017-11-01',4.6844,4.9712,4.6844,4.9712,62565,'0');
INSERT INTO daily_summary VALUES ('stock','AHGP','us','2017-11-01',27.271,27.547,26.998,27.261,160724,'0');
INSERT INTO daily_summary VALUES ('stock','AHH','us','2017-11-01',14.3,14.35,14.17,14.27,275676,'0');
INSERT INTO daily_summary VALUES ('stock','AHL','us','2017-11-01',42.999,43.248,42.055,42.452,764077,'0');
INSERT INTO daily_summary VALUES ('stock','AHL_C','us','2017-11-01',27,27,26.92,26.96,6867,'0');
INSERT INTO daily_summary VALUES ('stock','AHL_D','us','2017-11-01',25.45,25.49,25.4,25.46,13090,'0');
INSERT INTO daily_summary VALUES ('stock','AHP','us','2017-11-01',9.82,9.82,9.51,9.69,299953,'0');
INSERT INTO daily_summary VALUES ('stock','AHPA','us','2017-11-01',9.98,9.98,9.98,9.98,308,'0');
INSERT INTO daily_summary VALUES ('stock','AHPAU','us','2017-11-01',10.3,10.3492,10.3,10.3492,1400,'0');
INSERT INTO daily_summary VALUES ('stock','AHPAW','us','2017-11-01',0.38,0.38,0.38,0.38,10600,'0');
INSERT INTO daily_summary VALUES ('stock','AHPI','us','2017-11-01',2.04,2.05,2.01,2.04,6526,'0');
INSERT INTO daily_summary VALUES ('stock','AHP_B','us','2017-11-01',20.73,20.73,20.6892,20.6892,592,'0');
INSERT INTO daily_summary VALUES ('stock','AHT','us','2017-11-01',7.06,7.11,6.94,7.04,348925,'0');
INSERT INTO daily_summary VALUES ('stock','AHT_D','us','2017-11-01',25.4832,25.4832,25.47,25.48,2589,'0');
INSERT INTO daily_summary VALUES ('stock','AHT_F','us','2017-11-01',25.2728,25.299,25.2,25.255,143366,'0');
INSERT INTO daily_summary VALUES ('stock','AHT_G','us','2017-11-01',25,25,24.95,24.955,15064,'0');
INSERT INTO daily_summary VALUES ('stock','AHT_H','us','2017-11-01',25.25,25.25,25.171,25.22,8305,'0');
INSERT INTO daily_summary VALUES ('stock','AI','us','2017-11-01',11.52,11.5351,11.35,11.38,526127,'0');
INSERT INTO daily_summary VALUES ('etf','AIA','us','2017-11-01',65.63,65.73,65.4345,65.48,57871,'0');
INSERT INTO daily_summary VALUES ('stock','AIC','us','2017-11-01',24.69,24.79,24.64,24.64,5708,'0');
INSERT INTO daily_summary VALUES ('stock','AIEQ','us','2017-11-01',25.06,25.15,24.75,24.85,269928,'0');
INSERT INTO daily_summary VALUES ('stock','AIF','us','2017-11-01',16.15,16.15,15.97,16.12,45076,'0');
INSERT INTO daily_summary VALUES ('stock','AIG','us','2017-11-01',64.92,65.09,64.51,64.66,3026600,'0');
INSERT INTO daily_summary VALUES ('stock','AIMC','us','2017-11-01',48.3,48.4,47.25,47.55,151856,'0');
INSERT INTO daily_summary VALUES ('stock','AIMT','us','2017-11-01',29.33,29.56,28.33,29.25,655112,'0');
INSERT INTO daily_summary VALUES ('stock','AIN','us','2017-11-01',60.5,60.575,59.25,59.9,129055,'0');
INSERT INTO daily_summary VALUES ('stock','AINC','us','2017-11-01',70,72,69.72,72,2393,'0');
INSERT INTO daily_summary VALUES ('stock','AINV','us','2017-11-01',5.92,5.96,5.9,5.91,567917,'0');
INSERT INTO daily_summary VALUES ('stock','AIR','us','2017-11-01',39,41.94,39,41.06,401308,'0');
INSERT INTO daily_summary VALUES ('stock','AIRG','us','2017-11-01',8.75,8.89,8.38,8.5,116264,'0');
INSERT INTO daily_summary VALUES ('stock','AIRI','us','2017-11-01',1.37,1.37,1.3,1.32,16983,'0');
INSERT INTO daily_summary VALUES ('etf','AIRR','us','2017-11-01',27.02,27.02,26.4548,26.48,13467,'0');
INSERT INTO daily_summary VALUES ('stock','AIRT','us','2017-11-01',19.95,19.95,19.505,19.65,1000,'0');
INSERT INTO daily_summary VALUES ('stock','AIT','us','2017-11-01',64.35,64.35,62.45,63.7,282108,'0');
INSERT INTO daily_summary VALUES ('stock','AIV','us','2017-11-01',44.08,44.5,43.8,44.34,1014460,'0');
INSERT INTO daily_summary VALUES ('stock','AIV_A','us','2017-11-01',26.4663,26.81,26.4663,26.7569,1186,'0');
INSERT INTO daily_summary VALUES ('stock','AIW','us','2017-11-01',24.79,24.8,24.61,24.61,1200,'0');
INSERT INTO daily_summary VALUES ('stock','AIY','us','2017-11-01',25.787,25.79,25.787,25.79,3031,'0');
INSERT INTO daily_summary VALUES ('stock','AIZ','us','2017-11-01',101.27,102.85,100.98,101.17,514548,'0');
INSERT INTO daily_summary VALUES ('stock','AI_B','us','2017-11-01',24.75,24.75,24.7,24.7499,603,'0');
INSERT INTO daily_summary VALUES ('stock','AJG','us','2017-11-01',63.57,64.05,63.45,63.6,607827,'0');
INSERT INTO daily_summary VALUES ('stock','AJRD','us','2017-11-01',31.79,31.969,30.33,30.84,937265,'0');
INSERT INTO daily_summary VALUES ('stock','AJX','us','2017-11-01',14.21,14.21,13.9,14,41917,'0');
INSERT INTO daily_summary VALUES ('stock','AJXA','us','2017-11-01',25.61,25.61,25.5916,25.597,860,'0');
INSERT INTO daily_summary VALUES ('stock','AKAM','us','2017-11-01',52.7,53.07,52.34,52.69,1610420,'0');
INSERT INTO daily_summary VALUES ('stock','AKAO','us','2017-11-01',12.84,12.9565,12.36,12.71,673207,'0');
INSERT INTO daily_summary VALUES ('stock','AKBA','us','2017-11-01',18.25,18.39,17.83,17.86,405268,'0');
INSERT INTO daily_summary VALUES ('stock','AKCA','us','2017-11-01',18.52,19.1174,17.18,17.34,215830,'0');
INSERT INTO daily_summary VALUES ('stock','AKER','us','2017-11-01',0.79,0.83,0.782,0.8,110987,'0');
INSERT INTO daily_summary VALUES ('stock','AKG','us','2017-11-01',0.934,0.9877,0.934,0.9746,515899,'0');
INSERT INTO daily_summary VALUES ('stock','AKP','us','2017-11-01',13.7,13.72,13.68,13.7,18026,'0');
INSERT INTO daily_summary VALUES ('stock','AKR','us','2017-11-01',28.3,28.66,28.085,28.58,453358,'0');
INSERT INTO daily_summary VALUES ('stock','AKRX','us','2017-11-01',32.64,32.74,32.45,32.54,3414160,'0');
INSERT INTO daily_summary VALUES ('stock','AKS','us','2017-11-01',4.7201,4.9233,4.33,4.35,58504500,'0');
INSERT INTO daily_summary VALUES ('stock','AKTS','us','2017-11-01',6.14,6.14,5.6,5.62,28385,'0');
INSERT INTO daily_summary VALUES ('stock','AKTX','us','2017-11-01',4.29,4.35,4.16,4.29,148025,'0');
INSERT INTO daily_summary VALUES ('stock','AL','us','2017-11-01',43.82,44.29,43.43,43.8,518731,'0');
INSERT INTO daily_summary VALUES ('stock','ALB','us','2017-11-01',142,142.42,140.85,142.33,907500,'0');
INSERT INTO daily_summary VALUES ('stock','ALBO','us','2017-11-01',27.73,27.95,26.6743,26.975,13365,'0');
INSERT INTO daily_summary VALUES ('stock','ALCO','us','2017-11-01',32.55,32.7,31.6,31.75,7542,'0');
INSERT INTO daily_summary VALUES ('etf','ALD','us','2017-11-01',45.08,45.1252,45.001,45.02,6038,'0');
INSERT INTO daily_summary VALUES ('stock','ALDR','us','2017-11-01',11.4,11.75,10.65,10.65,878934,'0');
INSERT INTO daily_summary VALUES ('stock','ALDW','us','2017-11-01',12.67,12.88,12.52,12.76,176027,'0');
INSERT INTO daily_summary VALUES ('stock','ALDX','us','2017-11-01',6.3,6.45,6.2,6.3,51558,'0');
INSERT INTO daily_summary VALUES ('stock','ALE','us','2017-11-01',80,80,76.91,77.32,229554,'0');
INSERT INTO daily_summary VALUES ('stock','ALEX','us','2017-11-01',45.58,45.62,44.84,45.21,110222,'0');
INSERT INTO daily_summary VALUES ('etf','ALFA','us','2017-11-01',42.2957,42.36,42.2957,42.3,706,'0');
INSERT INTO daily_summary VALUES ('stock','ALG','us','2017-11-01',106.07,110.94,105.14,109.75,121953,'0');
INSERT INTO daily_summary VALUES ('stock','ALGN','us','2017-11-01',243,244.6,232.311,234.95,1441570,'0');
INSERT INTO daily_summary VALUES ('stock','ALGT','us','2017-11-01',137.15,138.65,135.6,136.35,147559,'0');
INSERT INTO daily_summary VALUES ('stock','ALIM','us','2017-11-01',1.24,1.27,1.2,1.24,284430,'0');
INSERT INTO daily_summary VALUES ('stock','ALJJ','us','2017-11-01',3.621,3.74,3.515,3.53,32510,'0');
INSERT INTO daily_summary VALUES ('stock','ALK','us','2017-11-01',65.65,66.02,64.71,65.78,2366450,'0');
INSERT INTO daily_summary VALUES ('stock','ALKS','us','2017-11-01',49.16,49.16,48.44,48.65,848531,'0');
INSERT INTO daily_summary VALUES ('stock','ALL','us','2017-11-01',94.07,94.54,93.52,94.25,1981030,'0');
INSERT INTO daily_summary VALUES ('stock','ALLE','us','2017-11-01',83.58,84.105,82.29,82.73,700689,'0');
INSERT INTO daily_summary VALUES ('stock','ALLT','us','2017-11-01',5.48,5.52,5.41,5.41,6760,'0');
INSERT INTO daily_summary VALUES ('stock','ALLY','us','2017-11-01',26.26,26.54,26.165,26.35,6133190,'0');
INSERT INTO daily_summary VALUES ('stock','ALLY_A','us','2017-11-01',26.33,26.37,26.23,26.26,105450,'0');
INSERT INTO daily_summary VALUES ('stock','ALL_A','us','2017-11-01',25.53,25.62,25.5286,25.62,6408,'0');
INSERT INTO daily_summary VALUES ('stock','ALL_B','us','2017-11-01',26.64,26.7949,26.64,26.6532,6996,'0');
INSERT INTO daily_summary VALUES ('stock','ALL_C','us','2017-11-01',26.25,26.27,26.21,26.25,9476,'0');
INSERT INTO daily_summary VALUES ('stock','ALL_D','us','2017-11-01',26.66,26.77,26.6182,26.6182,3566,'0');
INSERT INTO daily_summary VALUES ('stock','ALL_E','us','2017-11-01',26.77,26.83,26.56,26.65,69567,'0');
INSERT INTO daily_summary VALUES ('stock','ALL_F','us','2017-11-01',26.88,26.95,26.85,26.85,6174,'0');
INSERT INTO daily_summary VALUES ('stock','ALN','us','2017-11-01',0.32,0.32,0.25,0.3,108748,'0');
INSERT INTO daily_summary VALUES ('stock','ALNY','us','2017-11-01',123.44,124.5,120.08,121.09,884697,'0');
INSERT INTO daily_summary VALUES ('stock','ALO','us','2017-11-01',3.99,4.05,3.97,4.04,62420,'0');
INSERT INTO daily_summary VALUES ('stock','ALOG','us','2017-11-01',80.55,81.05,79.05,79.95,62867,'0');
INSERT INTO daily_summary VALUES ('stock','ALOT','us','2017-11-01',12.57,12.65,12.4,12.65,27753,'0');
INSERT INTO daily_summary VALUES ('stock','ALPN','us','2017-11-01',10.16,10.3,9.875,10.3,4898,'0');
INSERT INTO daily_summary VALUES ('stock','ALP_Q','us','2017-11-01',25.53,25.55,25.5,25.55,25739,'0');
INSERT INTO daily_summary VALUES ('stock','ALQA','us','2017-11-01',3.36,3.5317,3.32,3.5,49536,'0');
INSERT INTO daily_summary VALUES ('stock','ALRM','us','2017-11-01',46.75,46.84,45.39,45.9,319307,'0');
INSERT INTO daily_summary VALUES ('stock','ALRN','us','2017-11-01',12.1,13.35,11.68,13.19,42121,'0');
INSERT INTO daily_summary VALUES ('stock','ALSK','us','2017-11-01',2.46,2.539,2.42,2.5,256958,'0');
INSERT INTO daily_summary VALUES ('stock','ALSN','us','2017-11-01',43.15,44.12,43.07,43.76,3528160,'0');
INSERT INTO daily_summary VALUES ('stock','ALT','us','2017-11-01',2.43,2.43,2.21,2.26,130838,'0');
INSERT INTO daily_summary VALUES ('stock','ALTR','us','2017-11-01',17,18.98,16.55,18.31,7533540,'0');
INSERT INTO daily_summary VALUES ('etf','ALTS','us','2017-11-01',38.53,38.56,38.491,38.491,521,'0');
INSERT INTO daily_summary VALUES ('stock','ALTY','us','2017-11-01',15.405,15.45,15.3827,15.4325,2036,'0');
INSERT INTO daily_summary VALUES ('stock','ALV','us','2017-11-01',126.65,126.85,125.2,125.36,260971,'0');
INSERT INTO daily_summary VALUES ('stock','ALX','us','2017-11-01',406.83,411.07,402.91,404.35,10245,'0');
INSERT INTO daily_summary VALUES ('stock','ALXN','us','2017-11-01',119.97,120.49,114.79,116.06,3603990,'0');
INSERT INTO daily_summary VALUES ('stock','AM','us','2017-11-01',28.9,29.19,28.64,29.08,625163,'0');
INSERT INTO daily_summary VALUES ('stock','AMAG','us','2017-11-01',15.75,16.3,15.25,15.95,1028590,'0');
INSERT INTO daily_summary VALUES ('stock','AMAT','us','2017-11-01',57,57.3408,55.2,55.71,10479900,'0');
INSERT INTO daily_summary VALUES ('stock','AMBA','us','2017-11-01',56.6,57.3999,55.45,56.38,807760,'0');
INSERT INTO daily_summary VALUES ('stock','AMBC','us','2017-11-01',16.35,16.515,16.09,16.14,203405,'0');
INSERT INTO daily_summary VALUES ('stock','AMBCW','us','2017-11-01',7.76,7.77,7.237,7.237,3800,'0');
INSERT INTO daily_summary VALUES ('stock','AMBR','us','2017-11-01',7.57,7.71,7.33,7.67,81673,'0');
INSERT INTO daily_summary VALUES ('stock','AMC','us','2017-11-01',14.1,14.1,13.1,13.45,3738660,'0');
INSERT INTO daily_summary VALUES ('stock','AMCN','us','2017-11-01',2.23,2.35,2.23,2.31,408330,'0');
INSERT INTO daily_summary VALUES ('stock','AMCX','us','2017-11-01',51.06,51.97,50.505,50.99,1292220,'0');
INSERT INTO daily_summary VALUES ('stock','AMD','us','2017-11-01',11.25,11.29,10.72,10.8,66875500,'0');
INSERT INTO daily_summary VALUES ('stock','AMDA','us','2017-11-01',4.56,4.608,4.26,4.2624,49703,'0');
INSERT INTO daily_summary VALUES ('stock','AME','us','2017-11-01',67.94,67.97,66.92,67.25,1861070,'0');
INSERT INTO daily_summary VALUES ('stock','AMED','us','2017-11-01',47.87,48.14,45.6,46.45,501952,'0');
INSERT INTO daily_summary VALUES ('stock','AMG','us','2017-11-01',187.65,188.37,185.48,185.91,303611,'0');
INSERT INTO daily_summary VALUES ('stock','AMGN','us','2017-11-01',175.3,177.31,174.72,175.42,2734850,'0');
INSERT INTO daily_summary VALUES ('stock','AMGP','us','2017-11-01',18.75,19.245,18.75,19.15,383124,'0');
INSERT INTO daily_summary VALUES ('stock','AMH','us','2017-11-01',21.22,21.51,21.22,21.46,1636080,'0');
INSERT INTO daily_summary VALUES ('stock','AMH_C','us','2017-11-01',28.37,28.3984,28.35,28.35,9490,'0');
INSERT INTO daily_summary VALUES ('stock','AMH_D','us','2017-11-01',26.94,26.94,26.78,26.78,28111,'0');
INSERT INTO daily_summary VALUES ('stock','AMH_E','us','2017-11-01',26.41,26.5,26.32,26.4426,36758,'0');
INSERT INTO daily_summary VALUES ('stock','AMH_F','us','2017-11-01',25.7705,25.8295,25.75,25.8,11358,'0');
INSERT INTO daily_summary VALUES ('stock','AMH_G','us','2017-11-01',25.99,26.08,25.93,25.99,14700,'0');
INSERT INTO daily_summary VALUES ('stock','AMID','us','2017-11-01',13.322,13.527,13.031,13.273,200250,'0');
INSERT INTO daily_summary VALUES ('etf','AMJ','us','2017-11-01',27.19,27.53,27.16,27.49,3327580,'0');
INSERT INTO daily_summary VALUES ('etf','AMJL','us','2017-11-01',19.321,19.584,19.321,19.584,2336,'0');
INSERT INTO daily_summary VALUES ('stock','AMKR','us','2017-11-01',11.62,11.88,10.72,10.8,1677770,'0');
INSERT INTO daily_summary VALUES ('etf','AMLP','us','2017-11-01',10.552,10.718,10.529,10.689,14686800,'0');
INSERT INTO daily_summary VALUES ('stock','AMLX','us','2017-11-01',11.9799,11.9799,11.9799,11.9799,0,'0');
INSERT INTO daily_summary VALUES ('stock','AMMA','us','2017-11-01',1.61,1.8,1.5601,1.581,218822,'0');
INSERT INTO daily_summary VALUES ('stock','AMN','us','2017-11-01',44.05,44.05,41.8288,42.65,1144650,'0');
INSERT INTO daily_summary VALUES ('stock','AMNB','us','2017-11-01',39.45,39.45,38.4,38.65,7431,'0');
INSERT INTO daily_summary VALUES ('stock','AMOT','us','2017-11-01',28.5,28.5,27.74,28.05,17657,'0');
INSERT INTO daily_summary VALUES ('stock','AMOV','us','2017-11-01',16.94,17.3,16.94,17,6526,'0');
INSERT INTO daily_summary VALUES ('stock','AMP','us','2017-11-01',156.9,157.8,155.92,156.57,570498,'0');
INSERT INTO daily_summary VALUES ('stock','AMPE','us','2017-11-01',1.07,1.1038,1.03,1.09,580598,'0');
INSERT INTO daily_summary VALUES ('stock','AMPH','us','2017-11-01',18.18,18.54,18.02,18.43,178700,'0');
INSERT INTO daily_summary VALUES ('stock','AMRB','us','2017-11-01',14.9633,15.18,14.9633,15.1,1543,'0');
INSERT INTO daily_summary VALUES ('stock','AMRC','us','2017-11-01',7.8,8,7.75,7.95,35350,'0');
INSERT INTO daily_summary VALUES ('stock','AMRK','us','2017-11-01',15.43,15.43,14.9,15,46408,'0');
INSERT INTO daily_summary VALUES ('stock','AMRN','us','2017-11-01',3.49,3.579,3.38,3.47,1963220,'0');
INSERT INTO daily_summary VALUES ('stock','AMRS','us','2017-11-01',3.17,3.23,3.15,3.2,282479,'0');
INSERT INTO daily_summary VALUES ('stock','AMS','us','2017-11-01',3,3.05,3,3.05,6178,'0');
INSERT INTO daily_summary VALUES ('stock','AMSC','us','2017-11-01',4.21,4.22,4.05,4.1,35924,'0');
INSERT INTO daily_summary VALUES ('stock','AMSF','us','2017-11-01',65,65.4,63.9,64.6,75348,'0');
INSERT INTO daily_summary VALUES ('stock','AMSWA','us','2017-11-01',12.376,12.386,11.979,12.187,32343,'0');
INSERT INTO daily_summary VALUES ('stock','AMT','us','2017-11-01',144.25,144.66,140.39,140.81,2819700,'0');
INSERT INTO daily_summary VALUES ('stock','AMTD','us','2017-11-01',50.187,50.386,49.032,49.092,1412990,'0');
INSERT INTO daily_summary VALUES ('stock','AMTX','us','2017-11-01',0.77,0.7939,0.756,0.76,15144,'0');
INSERT INTO daily_summary VALUES ('stock','AMT_B','us','2017-11-01',125.43,125.81,123.92,124.43,130656,'0');
INSERT INTO daily_summary VALUES ('etf','AMU','us','2017-11-01',17.05,17.29,17.05,17.26,168469,'0');
INSERT INTO daily_summary VALUES ('stock','AMWD','us','2017-11-01',96.9,97.425,95.45,95.6,103724,'0');
INSERT INTO daily_summary VALUES ('stock','AMX','us','2017-11-01',17.15,17.48,17.08,17.11,2817180,'0');
INSERT INTO daily_summary VALUES ('stock','AMZA','us','2017-11-01',8.43,8.57,8.4,8.57,1044090,'0');
INSERT INTO daily_summary VALUES ('stock','AMZN','us','2017-11-01',1105.4,1108.97,1096.74,1103.68,3755510,'0');
INSERT INTO daily_summary VALUES ('stock','AN','us','2017-11-01',47.75,47.75,46.2,47.6,2041500,'0');
INSERT INTO daily_summary VALUES ('stock','ANAB','us','2017-11-01',66.14,67.04,63.82,64.95,215385,'0');
INSERT INTO daily_summary VALUES ('stock','ANAT','us','2017-11-01',122.3,123.65,121.162,122.38,8919,'0');
INSERT INTO daily_summary VALUES ('stock','ANCB','us','2017-11-01',24.9262,24.95,24.87,24.95,6808,'0');
INSERT INTO daily_summary VALUES ('stock','ANCX','us','2017-11-01',29.17,29.329,28.514,28.603,24119,'0');
INSERT INTO daily_summary VALUES ('stock','ANDA','us','2017-11-01',10.24,10.4,10.2,10.35,22483,'0');
INSERT INTO daily_summary VALUES ('stock','ANDAR','us','2017-11-01',1.04,1.1998,1,1.08,251757,'0');
INSERT INTO daily_summary VALUES ('stock','ANDAW','us','2017-11-01',0.95,1.05,0.83,0.879,74602,'0');
INSERT INTO daily_summary VALUES ('stock','ANDE','us','2017-11-01',37.6,37.9,37,37.35,83533,'0');
INSERT INTO daily_summary VALUES ('stock','ANDV','us','2017-11-01',107.12,109.27,106.86,109.21,1171750,'0');
INSERT INTO daily_summary VALUES ('stock','ANDX','us','2017-11-01',44.13,45.01,43.993,44.805,695542,'0');
INSERT INTO daily_summary VALUES ('stock','ANET','us','2017-11-01',201.41,201.41,195.17,197.26,993718,'0');
INSERT INTO daily_summary VALUES ('stock','ANF','us','2017-11-01',12.44,12.68,11.75,11.94,5715920,'0');
INSERT INTO daily_summary VALUES ('stock','ANFI','us','2017-11-01',5.99,5.99,5.85,5.86,179883,'0');
INSERT INTO daily_summary VALUES ('stock','ANGI','us','2017-11-01',12.5,12.5803,12.265,12.37,250022,'0');
INSERT INTO daily_summary VALUES ('etf','ANGL','us','2017-11-01',30.22,30.2335,30.14,30.16,346927,'0');
INSERT INTO daily_summary VALUES ('stock','ANGO','us','2017-11-01',17.05,17.1,16.59,16.87,151976,'0');
INSERT INTO daily_summary VALUES ('stock','ANH','us','2017-11-01',5.61,5.66,5.57,5.62,594913,'0');
INSERT INTO daily_summary VALUES ('stock','ANH_A','us','2017-11-01',26.75,26.75,26.25,26.35,1089,'0');
INSERT INTO daily_summary VALUES ('stock','ANH_B','us','2017-11-01',27.75,27.75,27.67,27.707,5600,'0');
INSERT INTO daily_summary VALUES ('stock','ANH_C','us','2017-11-01',25.24,25.24,25.17,25.23,3537,'0');
INSERT INTO daily_summary VALUES ('stock','ANIK','us','2017-11-01',54.82,55.08,53.171,53.53,57611,'0');
INSERT INTO daily_summary VALUES ('stock','ANIP','us','2017-11-01',58,59.07,57.18,57.54,92126,'0');
INSERT INTO daily_summary VALUES ('stock','ANSS','us','2017-11-01',137.55,138.39,135.28,135.6,494834,'0');
INSERT INTO daily_summary VALUES ('stock','ANTH','us','2017-11-01',1.62,1.6358,1.56,1.6,76339,'0');
INSERT INTO daily_summary VALUES ('stock','ANTM','us','2017-11-01',210.03,210.98,208.67,209.53,1857760,'0');
INSERT INTO daily_summary VALUES ('stock','ANTX','us','2017-11-01',53.3,53.41,53.29,53.3,356019,'0');
INSERT INTO daily_summary VALUES ('stock','ANW','us','2017-11-01',4.5,4.525,4.4,4.45,302783,'0');
INSERT INTO daily_summary VALUES ('stock','ANY','us','2017-11-01',2.51,2.62,2.44,2.48,312700,'0');
INSERT INTO daily_summary VALUES ('etf','AOA','us','2017-11-01',54.76,54.76,54.3986,54.48,57050,'0');
INSERT INTO daily_summary VALUES ('stock','AOBC','us','2017-11-01',13.98,14.93,13.95,14.44,2078250,'0');
INSERT INTO daily_summary VALUES ('stock','AOD','us','2017-11-01',9.07,9.115,9.05,9.11,247590,'0');
INSERT INTO daily_summary VALUES ('stock','AOI','us','2017-11-01',10.65,11.15,10.6,11,39446,'0');
INSERT INTO daily_summary VALUES ('etf','AOK','us','2017-11-01',34.96,34.96,34.607,34.66,72508,'0');
INSERT INTO daily_summary VALUES ('etf','AOM','us','2017-11-01',38.37,38.37,38.16,38.24,231827,'0');
INSERT INTO daily_summary VALUES ('stock','AON','us','2017-11-01',143.82,144.71,142.69,142.84,1085940,'0');
INSERT INTO daily_summary VALUES ('etf','AOR','us','2017-11-01',45.41,45.54,45.3,45.36,95402,'0');
INSERT INTO daily_summary VALUES ('stock','AOS','us','2017-11-01',59.69,59.97,59.38,59.92,1276610,'0');
INSERT INTO daily_summary VALUES ('stock','AOSL','us','2017-11-01',18.5,18.5,17.85,18.32,96135,'0');
INSERT INTO daily_summary VALUES ('stock','AP','us','2017-11-01',16.7,17.4,16.4,16.4,29539,'0');
INSERT INTO daily_summary VALUES ('stock','APA','us','2017-11-01',41.77,42.69,41.52,42.23,3522680,'0');
INSERT INTO daily_summary VALUES ('stock','APAM','us','2017-11-01',35.9,35.9,34.35,35.45,549928,'0');
INSERT INTO daily_summary VALUES ('stock','APB','us','2017-11-01',14.55,14.59,14.4484,14.59,11930,'0');
INSERT INTO daily_summary VALUES ('stock','APC','us','2017-11-01',49.64,51.02,49.16,50.14,9065660,'0');
INSERT INTO daily_summary VALUES ('stock','APD','us','2017-11-01',159.69,160.644,158.53,159.92,899604,'0');
INSERT INTO daily_summary VALUES ('stock','APDN','us','2017-11-01',2.3,2.3,2,2.29,162694,'0');
INSERT INTO daily_summary VALUES ('stock','APDNW','us','2017-11-01',0.58,0.58,0.58,0.58,10000,'0');
INSERT INTO daily_summary VALUES ('stock','APEI','us','2017-11-01',20.15,20.35,19.45,19.85,42073,'0');
INSERT INTO daily_summary VALUES ('stock','APEN','us','2017-11-01',4.1,4.4,4.1,4.34,53872,'0');
INSERT INTO daily_summary VALUES ('stock','APF','us','2017-11-01',18,18.06,17.99,18.06,13929,'0');
INSERT INTO daily_summary VALUES ('stock','APH','us','2017-11-01',87.55,87.83,86.74,86.87,1197390,'0');
INSERT INTO daily_summary VALUES ('stock','APHB','us','2017-11-01',1.09,1.095,1.01,1.04,215498,'0');
INSERT INTO daily_summary VALUES ('stock','APLE','us','2017-11-01',18.88,18.93,18.766,18.93,1097400,'0');
INSERT INTO daily_summary VALUES ('stock','APLP','us','2017-11-01',12.976,13.138,12.78,13.064,78706,'0');
INSERT INTO daily_summary VALUES ('stock','APO','us','2017-11-01',33.08,33.37,31.5,31.63,1191850,'0');
INSERT INTO daily_summary VALUES ('stock','APOG','us','2017-11-01',48,48.31,47.6,48.29,172644,'0');
INSERT INTO daily_summary VALUES ('stock','APOP','us','2017-11-01',9.2,9.52,8.83,9.02,23507,'0');
INSERT INTO daily_summary VALUES ('stock','APOPW','us','2017-11-01',2.7,2.7,2.7,2.7,1054,'0');
INSERT INTO daily_summary VALUES ('stock','APO_A','us','2017-11-01',26.2866,26.32,26.1791,26.25,3936,'0');
INSERT INTO daily_summary VALUES ('stock','APPF','us','2017-11-01',45.95,46.25,44.35,45.1,133800,'0');
INSERT INTO daily_summary VALUES ('stock','APPN','us','2017-11-01',23.19,23.76,22.81,22.91,137785,'0');
INSERT INTO daily_summary VALUES ('stock','APPS','us','2017-11-01',1.57,1.58,1.56,1.57,58881,'0');
INSERT INTO daily_summary VALUES ('stock','APRI','us','2017-11-01',1.69,1.71,1.66,1.699,75151,'0');
INSERT INTO daily_summary VALUES ('stock','APRN','us','2017-11-01',4.75,4.85,4.61,4.67,2294260,'0');
INSERT INTO daily_summary VALUES ('stock','APT','us','2017-11-01',4.1753,4.2,4.1,4.1215,6318,'0');
INSERT INTO daily_summary VALUES ('stock','APTI','us','2017-11-01',23.38,23.7158,22.5,22.55,854077,'0');
INSERT INTO daily_summary VALUES ('stock','APTO','us','2017-11-01',1.55,1.74,1.51,1.6224,1036030,'0');
INSERT INTO daily_summary VALUES ('stock','APTS','us','2017-11-01',19.96,20.5357,19.94,20.3,468351,'0');
INSERT INTO daily_summary VALUES ('stock','APU','us','2017-11-01',44.367,44.612,44.181,44.416,232649,'0');
INSERT INTO daily_summary VALUES ('stock','APVO','us','2017-11-01',2.85,2.939,2.76,2.77,149483,'0');
INSERT INTO daily_summary VALUES ('stock','APWC','us','2017-11-01',2.855,2.855,2.855,2.855,1652,'0');
INSERT INTO daily_summary VALUES ('stock','AQB','us','2017-11-01',6.3,6.3,6.1,6.1,5635,'0');
INSERT INTO daily_summary VALUES ('stock','AQMS','us','2017-11-01',4.11,4.11,3.8,3.83,204891,'0');
INSERT INTO daily_summary VALUES ('stock','AQN','us','2017-11-01',10.71,10.75,10.62,10.7,113064,'0');
INSERT INTO daily_summary VALUES ('stock','AQXP','us','2017-11-01',11.27,11.47,11.1,11.41,71931,'0');
INSERT INTO daily_summary VALUES ('stock','AR','us','2017-11-01',19.54,19.8,19.37,19.68,4868640,'0');
INSERT INTO daily_summary VALUES ('stock','ARA','us','2017-11-01',12.25,12.38,11.39,11.43,199092,'0');
INSERT INTO daily_summary VALUES ('stock','ARAY','us','2017-11-01',4.3,4.8,4.25,4.775,642743,'0');
INSERT INTO daily_summary VALUES ('stock','ARC','us','2017-11-01',4.5,4.5,4.27,4.44,116970,'0');
INSERT INTO daily_summary VALUES ('stock','ARCB','us','2017-11-01',32.818,33.117,31.77,32.02,166053,'0');
INSERT INTO daily_summary VALUES ('stock','ARCC','us','2017-11-01',16.08,16.177,16.01,16.04,2089180,'0');
INSERT INTO daily_summary VALUES ('stock','ARCH','us','2017-11-01',77.97,79.88,76.78,78.95,593288,'0');
INSERT INTO daily_summary VALUES ('stock','ARCI','us','2017-11-01',0.88,0.92,0.88,0.8801,9839,'0');
INSERT INTO daily_summary VALUES ('stock','ARCM','us','2017-11-01',100.104,100.104,100.104,100.104,0,'0');
INSERT INTO daily_summary VALUES ('stock','ARCO','us','2017-11-01',10.1,10.3,10,10.25,407115,'0');
INSERT INTO daily_summary VALUES ('stock','ARCW','us','2017-11-01',2.6,2.7,2.55,2.6,9536,'0');
INSERT INTO daily_summary VALUES ('stock','ARCX','us','2017-11-01',16.36,16.36,16.311,16.35,98867,'0');
INSERT INTO daily_summary VALUES ('stock','ARD','us','2017-11-01',21.7,22.33,21.5,21.69,102245,'0');
INSERT INTO daily_summary VALUES ('stock','ARDC','us','2017-11-01',16.5,16.56,16.43,16.5,108019,'0');
INSERT INTO daily_summary VALUES ('stock','ARDM','us','2017-11-01',4.16,4.18,4,4.09,59631,'0');
INSERT INTO daily_summary VALUES ('stock','ARDX','us','2017-11-01',5.4,5.55,5.1,5.4,535322,'0');
INSERT INTO daily_summary VALUES ('stock','ARE','us','2017-11-01',124.68,125.72,123.93,125.21,630816,'0');
INSERT INTO daily_summary VALUES ('stock','ARES','us','2017-11-01',18.4,18.55,18.201,18.5,52773,'0');
INSERT INTO daily_summary VALUES ('stock','ARES_A','us','2017-11-01',26.8476,26.8476,26.61,26.65,23352,'0');
INSERT INTO daily_summary VALUES ('stock','AREX','us','2017-11-01',2.4,2.49,2.34,2.39,595438,'0');
INSERT INTO daily_summary VALUES ('stock','ARE_D','us','2017-11-01',36.5001,36.67,36.5001,36.66,1290,'0');
INSERT INTO daily_summary VALUES ('stock','ARGS','us','2017-11-01',0.15,0.1517,0.1425,0.1454,983445,'0');
INSERT INTO daily_summary VALUES ('etf','ARGT','us','2017-11-01',32.24,32.24,31.85,31.95,76089,'0');
INSERT INTO daily_summary VALUES ('stock','ARGX','us','2017-11-01',23,23.31,22.56,22.62,49243,'0');
INSERT INTO daily_summary VALUES ('stock','ARH_C','us','2017-11-01',25.41,25.41,25.38,25.3899,6142,'0');
INSERT INTO daily_summary VALUES ('stock','ARI','us','2017-11-01',18.08,18.205,18.04,18.09,990578,'0');
INSERT INTO daily_summary VALUES ('stock','ARII','us','2017-11-01',39.8,41.14,39.4,40.39,133799,'0');
INSERT INTO daily_summary VALUES ('stock','ARI_C','us','2017-11-01',25.5,25.525,25.4388,25.4388,1663,'0');
INSERT INTO daily_summary VALUES ('stock','ARKG','us','2017-11-01',26.03,26.0926,25.175,25.21,42699,'0');
INSERT INTO daily_summary VALUES ('stock','ARKK','us','2017-11-01',35.76,35.82,34.93,34.96,147193,'0');
INSERT INTO daily_summary VALUES ('stock','ARKQ','us','2017-11-01',33.69,33.69,32.96,33.06,38904,'0');
INSERT INTO daily_summary VALUES ('stock','ARKR','us','2017-11-01',24.06,24.31,24.06,24.31,519,'0');
INSERT INTO daily_summary VALUES ('stock','ARKW','us','2017-11-01',43.55,43.64,42.94,42.97,182168,'0');
INSERT INTO daily_summary VALUES ('stock','ARLP','us','2017-11-01',19.34,19.583,19.096,19.242,375708,'0');
INSERT INTO daily_summary VALUES ('stock','ARLZ','us','2017-11-01',1.47,1.47,1.39,1.45,182975,'0');
INSERT INTO daily_summary VALUES ('stock','ARMK','us','2017-11-01',43.86,44.12,43.25,43.26,1209680,'0');
INSERT INTO daily_summary VALUES ('stock','ARNA','us','2017-11-01',28.23,28.4499,27.1,27.38,424749,'0');
INSERT INTO daily_summary VALUES ('stock','ARNC','us','2017-11-01',25.19,25.569,24.94,25.06,4209240,'0');
INSERT INTO daily_summary VALUES ('stock','ARNC_','us','2017-11-01',89.85,90.8,89.85,90.8,301,'0');
INSERT INTO daily_summary VALUES ('stock','AROC','us','2017-11-01',12.069,12.118,11.624,11.772,274939,'0');
INSERT INTO daily_summary VALUES ('stock','AROW','us','2017-11-01',35.65,35.65,34.7,35.05,15366,'0');
INSERT INTO daily_summary VALUES ('stock','ARQL','us','2017-11-01',1.1,1.11,1.01,1.04,375517,'0');
INSERT INTO daily_summary VALUES ('stock','ARR','us','2017-11-01',25.22,25.32,24.96,25.05,670311,'0');
INSERT INTO daily_summary VALUES ('stock','ARRS','us','2017-11-01',28.72,28.83,27.91,27.95,1928700,'0');
INSERT INTO daily_summary VALUES ('stock','ARRY','us','2017-11-01',10.53,11.06,10.44,10.47,6089520,'0');
INSERT INTO daily_summary VALUES ('stock','ARR_A','us','2017-11-01',25.565,25.5799,25.5291,25.5799,647,'0');
INSERT INTO daily_summary VALUES ('stock','ARR_B','us','2017-11-01',25.09,25.14,25.09,25.1299,12675,'0');
INSERT INTO daily_summary VALUES ('stock','ARTNA','us','2017-11-01',40.717,40.717,39.459,39.843,11138,'0');
INSERT INTO daily_summary VALUES ('stock','ARTW','us','2017-11-01',2.7,2.7,2.5049,2.55,4804,'0');
INSERT INTO daily_summary VALUES ('stock','ARTX','us','2017-11-01',4.1,4.1,4,4.05,120689,'0');
INSERT INTO daily_summary VALUES ('stock','ARW','us','2017-11-01',84.18,84.18,81.83,82.15,632069,'0');
INSERT INTO daily_summary VALUES ('stock','ARWR','us','2017-11-01',3.65,3.7,3.46,3.51,373390,'0');
INSERT INTO daily_summary VALUES ('stock','ASA','us','2017-11-01',11.5,11.581,11.397,11.4,67334,'0');
INSERT INTO daily_summary VALUES ('stock','ASB','us','2017-11-01',25.3,25.5,24.9,25.15,576004,'0');
INSERT INTO daily_summary VALUES ('stock','ASB_C','us','2017-11-01',26.96,27.2,26.96,27.1487,2176,'0');
INSERT INTO daily_summary VALUES ('stock','ASB_D','us','2017-11-01',24.77,24.89,24.77,24.89,18127,'0');
INSERT INTO daily_summary VALUES ('stock','ASC','us','2017-11-01',8.5,8.575,8.3,8.5,108292,'0');
INSERT INTO daily_summary VALUES ('stock','ASCMA','us','2017-11-01',11.26,11.6,10.9,11.17,25816,'0');
INSERT INTO daily_summary VALUES ('etf','ASEA','us','2017-11-01',15.91,15.9861,15.8911,15.964,4065,'0');
INSERT INTO daily_summary VALUES ('stock','ASET','us','2017-11-01',28.26,28.31,28.23,28.23,5653,'0');
INSERT INTO daily_summary VALUES ('stock','ASFI','us','2017-11-01',7.35,7.4,7.15,7.3,5970,'0');
INSERT INTO daily_summary VALUES ('stock','ASG','us','2017-11-01',5.67,5.67,5.55,5.55,52743,'0');
INSERT INTO daily_summary VALUES ('stock','ASGN','us','2017-11-01',61.86,61.889,59.17,60.2,383893,'0');
INSERT INTO daily_summary VALUES ('stock','ASH','us','2017-11-01',68.45,69.33,68.19,68.68,781884,'0');
INSERT INTO daily_summary VALUES ('etf','ASHR','us','2017-11-01',30.7,30.7,30.62,30.67,601729,'0');
INSERT INTO daily_summary VALUES ('etf','ASHS','us','2017-11-01',36.18,36.2134,36.13,36.1905,2836,'0');
INSERT INTO daily_summary VALUES ('stock','ASHX','us','2017-11-01',22.625,22.625,22.625,22.625,110,'0');
INSERT INTO daily_summary VALUES ('stock','ASIX','us','2017-11-01',46.39,46.5,43.68,44.18,392687,'0');
INSERT INTO daily_summary VALUES ('stock','ASM','us','2017-11-01',1.32,1.35,1.3107,1.34,116826,'0');
INSERT INTO daily_summary VALUES ('stock','ASMB','us','2017-11-01',29.64,29.9,28.341,28.58,47886,'0');
INSERT INTO daily_summary VALUES ('stock','ASML','us','2017-11-01',182.68,182.98,179.54,181.4,942043,'0');
INSERT INTO daily_summary VALUES ('stock','ASNA','us','2017-11-01',2,2,1.86,1.96,2130110,'0');
INSERT INTO daily_summary VALUES ('stock','ASND','us','2017-11-01',34.05,34.385,33.325,33.56,116579,'0');
INSERT INTO daily_summary VALUES ('stock','ASPN','us','2017-11-01',4.5,4.5,4.41,4.46,7548,'0');
INSERT INTO daily_summary VALUES ('stock','ASPS','us','2017-11-01',26.07,27.06,25.88,26.74,113727,'0');
INSERT INTO daily_summary VALUES ('stock','ASPU','us','2017-11-01',8,8.1,7.7901,8.1,30080,'0');
INSERT INTO daily_summary VALUES ('stock','ASR','us','2017-11-01',179.41,180.36,175.2,175.3,91623,'0');
INSERT INTO daily_summary VALUES ('stock','ASRV','us','2017-11-01',4.14,4.14,4.0402,4.14,10878,'0');
INSERT INTO daily_summary VALUES ('stock','ASRVP','us','2017-11-01',28,28,27.6046,27.6046,240,'0');
INSERT INTO daily_summary VALUES ('stock','AST','us','2017-11-01',2.6,2.6,2.4,2.4,134211,'0');
INSERT INTO daily_summary VALUES ('stock','ASTC','us','2017-11-01',3.01,3.01,2.66,2.81,81572,'0');
INSERT INTO daily_summary VALUES ('stock','ASTE','us','2017-11-01',51.989,52.242,50.671,51.1,92686,'0');
INSERT INTO daily_summary VALUES ('stock','ASUR','us','2017-11-01',11.3,11.35,11.03,11.18,127169,'0');
INSERT INTO daily_summary VALUES ('stock','ASV','us','2017-11-01',8.1439,8.1439,8,8,4233,'0');
INSERT INTO daily_summary VALUES ('stock','ASX','us','2017-11-01',6.2,6.2,6.08,6.1,906077,'0');
INSERT INTO daily_summary VALUES ('stock','ASYS','us','2017-11-01',14.6,14.73,13.59,13.67,171152,'0');
INSERT INTO daily_summary VALUES ('stock','AT','us','2017-11-01',2.5,2.5,2.45,2.45,510542,'0');
INSERT INTO daily_summary VALUES ('stock','ATAX','us','2017-11-01',6.2,6.25,6.05,6.05,146844,'0');
INSERT INTO daily_summary VALUES ('stock','ATEC','us','2017-11-01',3.63,3.72,3.5,3.57,37569,'0');
INSERT INTO daily_summary VALUES ('stock','ATEN','us','2017-11-01',7.38,7.49,7.33,7.45,672922,'0');
INSERT INTO daily_summary VALUES ('stock','ATGE','us','2017-11-01',37.05,37.05,35.85,36.65,589132,'0');
INSERT INTO daily_summary VALUES ('stock','ATH','us','2017-11-01',52.22,52.415,51.08,51.14,1749940,'0');
INSERT INTO daily_summary VALUES ('stock','ATHM','us','2017-11-01',57.77,58.12,56.6,57.45,563279,'0');
INSERT INTO daily_summary VALUES ('stock','ATHN','us','2017-11-01',128.06,128.06,122.041,123.45,483201,'0');
INSERT INTO daily_summary VALUES ('stock','ATHX','us','2017-11-01',1.85,1.8783,1.7799,1.79,612343,'0');
INSERT INTO daily_summary VALUES ('stock','ATI','us','2017-11-01',25.9,26.3,24.65,25.12,2413410,'0');
INSERT INTO daily_summary VALUES ('stock','ATKR','us','2017-11-01',19.49,19.5,19.11,19.2,379113,'0');
INSERT INTO daily_summary VALUES ('stock','ATLC','us','2017-11-01',2.27,2.32,2.26,2.32,14312,'0');
INSERT INTO daily_summary VALUES ('stock','ATLO','us','2017-11-01',29.61,29.65,29.1,29.1,2634,'0');
INSERT INTO daily_summary VALUES ('etf','ATMP','us','2017-11-01',20.845,20.849,20.7419,20.82,67353,'0');
INSERT INTO daily_summary VALUES ('stock','ATNI','us','2017-11-01',54.52,55.39,53.17,54.36,104831,'0');
INSERT INTO daily_summary VALUES ('stock','ATNM','us','2017-11-01',0.625,0.67,0.6173,0.647,937425,'0');
INSERT INTO daily_summary VALUES ('stock','ATNX','us','2017-11-01',16.78,16.9,16.41,16.85,167878,'0');
INSERT INTO daily_summary VALUES ('stock','ATO','us','2017-11-01',87.52,87.815,86.63,86.97,339302,'0');
INSERT INTO daily_summary VALUES ('stock','ATOM','us','2017-11-01',3.31,3.31,3.07,3.22,2728,'0');
INSERT INTO daily_summary VALUES ('stock','ATOS','us','2017-11-01',0.415,0.451,0.41,0.4165,1931380,'0');
INSERT INTO daily_summary VALUES ('stock','ATR','us','2017-11-01',87.83,88.12,87.21,87.41,279320,'0');
INSERT INTO daily_summary VALUES ('stock','ATRA','us','2017-11-01',14.4,14.775,13.8,14.1,314690,'0');
INSERT INTO daily_summary VALUES ('stock','ATRC','us','2017-11-01',21.43,21.43,20.61,21.23,174583,'0');
INSERT INTO daily_summary VALUES ('stock','ATRI','us','2017-11-01',657.65,657.65,647.15,647.15,1797,'0');
INSERT INTO daily_summary VALUES ('stock','ATRO','us','2017-11-01',34.73,34.73,33.85,34.23,102981,'0');
INSERT INTO daily_summary VALUES ('stock','ATRS','us','2017-11-01',1.83,1.84,1.75,1.79,1260930,'0');
INSERT INTO daily_summary VALUES ('stock','ATSG','us','2017-11-01',24.34,24.39,23.61,23.92,743568,'0');
INSERT INTO daily_summary VALUES ('stock','ATTO','us','2017-11-01',12.285,12.478,11.849,12.043,47405,'0');
INSERT INTO daily_summary VALUES ('stock','ATTU','us','2017-11-01',6.93,7.06,6.49,6.57,271926,'0');
INSERT INTO daily_summary VALUES ('stock','ATU','us','2017-11-01',25.75,25.75,25.325,25.65,338196,'0');
INSERT INTO daily_summary VALUES ('stock','ATUS','us','2017-11-01',23.83,24.96,23.69,24.94,3810000,'0');
INSERT INTO daily_summary VALUES ('stock','ATV','us','2017-11-01',14.745,14.745,14.52,14.52,334,'0');
INSERT INTO daily_summary VALUES ('stock','ATVI','us','2017-11-01',65.69,65.78,64.25,65.33,5139100,'0');
INSERT INTO daily_summary VALUES ('stock','ATXI','us','2017-11-01',5.32,5.4,4.95,4.95,34721,'0');
INSERT INTO daily_summary VALUES ('stock','AU','us','2017-11-01',9.51,9.61,9.32,9.39,2510240,'0');
INSERT INTO daily_summary VALUES ('stock','AUDC','us','2017-11-01',6.66,6.69,6.15,6.57,449815,'0');
INSERT INTO daily_summary VALUES ('stock','AUG','us','2017-11-01',1.66,1.75,1.63,1.645,155383,'0');
INSERT INTO daily_summary VALUES ('stock','AUMN','us','2017-11-01',0.4025,0.42,0.4,0.41,67724,'0');
INSERT INTO daily_summary VALUES ('stock','AUO','us','2017-11-01',4.18,4.18,4.09,4.11,1701640,'0');
INSERT INTO daily_summary VALUES ('stock','AUPH','us','2017-11-01',5.92,5.92,5.6904,5.74,729542,'0');
INSERT INTO daily_summary VALUES ('etf','AUSE','us','2017-11-01',57.4,57.67,57.4,57.45,1247,'0');
INSERT INTO daily_summary VALUES ('stock','AUTO','us','2017-11-01',6.93,7,6.76,6.95,27148,'0');
INSERT INTO daily_summary VALUES ('stock','AUY','us','2017-11-01',2.63,2.65,2.57,2.57,8052440,'0');
INSERT INTO daily_summary VALUES ('stock','AVA','us','2017-11-01',52.13,52.2501,51.97,52.2,449009,'0');
INSERT INTO daily_summary VALUES ('stock','AVAL','us','2017-11-01',8.33,8.45,8.29,8.43,501069,'0');
INSERT INTO daily_summary VALUES ('stock','AVAV','us','2017-11-01',51.52,51.54,50.39,50.86,241355,'0');
INSERT INTO daily_summary VALUES ('stock','AVB','us','2017-11-01',181.33,183.54,180.76,183.4,609878,'0');
INSERT INTO daily_summary VALUES ('stock','AVD','us','2017-11-01',22.7,22.7,21.8,22.3,83794,'0');
INSERT INTO daily_summary VALUES ('stock','AVDL','us','2017-11-01',9.53,9.59,9.36,9.5,170883,'0');
INSERT INTO daily_summary VALUES ('stock','AVEO','us','2017-11-01',3,3.07,2.9,2.94,1268170,'0');
INSERT INTO daily_summary VALUES ('stock','AVGO','us','2017-11-01',264.24,265.64,256.89,259.29,2733250,'0');
INSERT INTO daily_summary VALUES ('stock','AVGR','us','2017-11-01',0.38,0.39,0.299,0.3268,3668300,'0');
INSERT INTO daily_summary VALUES ('stock','AVH','us','2017-11-01',7.05,7.36,7.02,7.24,80453,'0');
INSERT INTO daily_summary VALUES ('stock','AVHI','us','2017-11-01',16.85,16.85,16.35,16.45,56228,'0');
INSERT INTO daily_summary VALUES ('stock','AVID','us','2017-11-01',4.47,4.5,4.33,4.35,133287,'0');
INSERT INTO daily_summary VALUES ('stock','AVIR','us','2017-11-01',0.6201,0.6242,0.57,0.61,1397230,'0');
INSERT INTO daily_summary VALUES ('stock','AVK','us','2017-11-01',16.13,16.19,16.07,16.08,80821,'0');
INSERT INTO daily_summary VALUES ('stock','AVNW','us','2017-11-01',16.45,16.98,16.45,16.672,20750,'0');
INSERT INTO daily_summary VALUES ('stock','AVP','us','2017-11-01',2.3,2.37,2.27,2.37,4541160,'0');
INSERT INTO daily_summary VALUES ('stock','AVT','us','2017-11-01',40.3,40.3,39.36,39.49,1055860,'0');
INSERT INTO daily_summary VALUES ('stock','AVX','us','2017-11-01',18.86,18.92,18.4,18.51,204737,'0');
INSERT INTO daily_summary VALUES ('stock','AVXL','us','2017-11-01',4.27,4.27,4.07,4.18,198025,'0');
INSERT INTO daily_summary VALUES ('stock','AVXS','us','2017-11-01',105.52,106.65,97.39,99.92,435415,'0');
INSERT INTO daily_summary VALUES ('stock','AVY','us','2017-11-01',106.71,107.41,105.57,107.03,736409,'0');
INSERT INTO daily_summary VALUES ('stock','AWF','us','2017-11-01',12.9,12.98,12.9,12.95,122132,'0');
INSERT INTO daily_summary VALUES ('stock','AWI','us','2017-11-01',51.65,51.9,51.05,51.5,683218,'0');
INSERT INTO daily_summary VALUES ('stock','AWK','us','2017-11-01',87.6,87.65,86.645,86.814,835655,'0');
INSERT INTO daily_summary VALUES ('stock','AWP','us','2017-11-01',6.5,6.57,6.49,6.54,276884,'0');
INSERT INTO daily_summary VALUES ('stock','AWR','us','2017-11-01',53.88,54.32,53.44,53.62,132288,'0');
INSERT INTO daily_summary VALUES ('stock','AWRE','us','2017-11-01',4.65,4.8,4.65,4.65,14320,'0');
INSERT INTO daily_summary VALUES ('stock','AXAS','us','2017-11-01',2.18,2.24,2.12,2.16,1964710,'0');
INSERT INTO daily_summary VALUES ('stock','AXDX','us','2017-11-01',20,20.25,19.45,20.05,525883,'0');
INSERT INTO daily_summary VALUES ('stock','AXE','us','2017-11-01',69.4,69.45,66.6,67.25,197353,'0');
INSERT INTO daily_summary VALUES ('stock','AXGN','us','2017-11-01',20.6,21.15,20.25,20.6,318867,'0');
INSERT INTO daily_summary VALUES ('etf','AXJL','us','2017-11-01',69.2,69.4,69.2,69.2049,1407,'0');
INSERT INTO daily_summary VALUES ('stock','AXL','us','2017-11-01',18.12,18.28,17.7,17.81,2039070,'0');
INSERT INTO daily_summary VALUES ('stock','AXON','us','2017-11-01',5.25,5.42,5.16,5.26,1083430,'0');
INSERT INTO daily_summary VALUES ('stock','AXP','us','2017-11-01',96.29,96.565,95.44,95.79,2319970,'0');
INSERT INTO daily_summary VALUES ('stock','AXR','us','2017-11-01',6.986,7.15,6.97,7.14,5694,'0');
INSERT INTO daily_summary VALUES ('stock','AXS','us','2017-11-01',54.58,54.63,54.04,54.37,908590,'0');
INSERT INTO daily_summary VALUES ('stock','AXSM','us','2017-11-01',4.75,4.8,4.7,4.75,83595,'0');
INSERT INTO daily_summary VALUES ('stock','AXS_D','us','2017-11-01',25.32,25.46,25.32,25.4514,8238,'0');
INSERT INTO daily_summary VALUES ('stock','AXS_E','us','2017-11-01',25.4,25.43,25.31,25.32,99532,'0');
INSERT INTO daily_summary VALUES ('stock','AXTA','us','2017-11-01',33.52,33.67,32.91,33.42,3054620,'0');
INSERT INTO daily_summary VALUES ('stock','AXTI','us','2017-11-01',9.3,9.35,8.65,8.8,543985,'0');
INSERT INTO daily_summary VALUES ('stock','AXU','us','2017-11-01',1.29,1.29,1.2014,1.26,281523,'0');
INSERT INTO daily_summary VALUES ('stock','AYI','us','2017-11-01',167.89,168.21,164.87,165.72,674245,'0');
INSERT INTO daily_summary VALUES ('stock','AYR','us','2017-11-01',23.46,23.46,22.85,23.2,209365,'0');
INSERT INTO daily_summary VALUES ('etf','AYT','us','2017-11-01',41.2,42.61,41.1,41.64,3305,'0');
INSERT INTO daily_summary VALUES ('stock','AYTU','us','2017-11-01',3.3972,3.3972,3.25,3.2937,9332,'0');
INSERT INTO daily_summary VALUES ('stock','AYX','us','2017-11-01',22.6,22.6,22.1591,22.47,143051,'0');
INSERT INTO daily_summary VALUES ('stock','AZN','us','2017-11-01',34.5,34.79,34.4838,34.56,3976770,'0');
INSERT INTO daily_summary VALUES ('stock','AZO','us','2017-11-01',592.94,594.83,579.6,584.88,357093,'0');
INSERT INTO daily_summary VALUES ('stock','AZPN','us','2017-11-01',64.99,65.29,63.93,64.7,728474,'0');
INSERT INTO daily_summary VALUES ('stock','AZRE','us','2017-11-01',15.5,15.5,15.4779,15.49,9054,'0');
INSERT INTO daily_summary VALUES ('stock','AZRX','us','2017-11-01',3.1001,3.142,2.85,3.04,18033,'0');
INSERT INTO daily_summary VALUES ('stock','AZUL','us','2017-11-01',25.03,25.33,24.77,24.88,323206,'0');
INSERT INTO daily_summary VALUES ('stock','AZZ','us','2017-11-01',48.2,48.7,47.05,47.1,129127,'0');
INSERT INTO daily_summary VALUES ('stock','B','us','2017-11-01',1.86,1.9,1.86,1.88,2317,'0');
INSERT INTO daily_summary VALUES ('stock','BA','us','2017-11-01',256.91,258.02,254.87,257.12,2944710,'0');
INSERT INTO daily_summary VALUES ('stock','BAA','us','2017-11-01',0.3232,0.3329,0.3101,0.3175,230070,'0');
INSERT INTO daily_summary VALUES ('etf','BAB','us','2017-11-01',30.53,30.53,30.38,30.47,113486,'0');
INSERT INTO daily_summary VALUES ('stock','BABA','us','2017-11-01',187.88,188.88,183.58,186.08,28486000,'0');
INSERT INTO daily_summary VALUES ('stock','BABY','us','2017-11-01',42.35,42.725,41.7,42.15,298449,'0');
INSERT INTO daily_summary VALUES ('stock','BAC','us','2017-11-01',27.64,27.73,27.36,27.53,46507200,'0');
INSERT INTO daily_summary VALUES ('stock','BAC_A','us','2017-11-01',26.47,26.49,26.33,26.375,76686,'0');
INSERT INTO daily_summary VALUES ('stock','BAC_C','us','2017-11-01',26.79,26.85,26.73,26.7689,23284,'0');
INSERT INTO daily_summary VALUES ('stock','BAC_D','us','2017-11-01',25.8484,25.9,25.8188,25.88,18630,'0');
INSERT INTO daily_summary VALUES ('stock','BAC_E','us','2017-11-01',23.96,24.009,23.77,23.89,9497,'0');
INSERT INTO daily_summary VALUES ('stock','BAC_I','us','2017-11-01',26.0453,26.19,26.0453,26.12,25008,'0');
INSERT INTO daily_summary VALUES ('stock','BAC_L','us','2017-11-01',1299.59,1301.07,1295.53,1300,3896,'0');
INSERT INTO daily_summary VALUES ('stock','BAC_W','us','2017-11-01',27,27.049,26.95,26.96,32185,'0');
INSERT INTO daily_summary VALUES ('stock','BAC_Y','us','2017-11-01',26.76,26.7901,26.71,26.75,80816,'0');
INSERT INTO daily_summary VALUES ('stock','BAF','us','2017-11-01',15.1,15.1,14.99,15.01,21363,'0');
INSERT INTO daily_summary VALUES ('stock','BAH','us','2017-11-01',37.724,37.963,37.455,37.595,453030,'0');
INSERT INTO daily_summary VALUES ('stock','BAK','us','2017-11-01',31.59,31.67,30.65,31,815510,'0');
INSERT INTO daily_summary VALUES ('etf','BAL','us','2017-11-01',45.2499,45.31,45.2499,45.31,597,'0');
INSERT INTO daily_summary VALUES ('stock','BAM','us','2017-11-01',42.03,42.13,41.45,41.53,514686,'0');
INSERT INTO daily_summary VALUES ('stock','BANC','us','2017-11-01',21.2,21.325,20.675,20.7,493356,'0');
INSERT INTO daily_summary VALUES ('stock','BANC_C','us','2017-11-01',26.2954,26.35,26.2954,26.35,961,'0');
INSERT INTO daily_summary VALUES ('stock','BANC_D','us','2017-11-01',26.46,26.5917,26.42,26.5091,7264,'0');
INSERT INTO daily_summary VALUES ('stock','BANC_E','us','2017-11-01',27.7134,27.74,27.5,27.74,4332,'0');
INSERT INTO daily_summary VALUES ('stock','BANF','us','2017-11-01',55.3,55.55,53.75,54.325,31360,'0');
INSERT INTO daily_summary VALUES ('stock','BANFP','us','2017-11-01',26.5,27.2202,26.5,27.2202,400,'0');
INSERT INTO daily_summary VALUES ('stock','BANR','us','2017-11-01',57.86,58.08,56.55,57.01,197325,'0');
INSERT INTO daily_summary VALUES ('stock','BANX','us','2017-11-01',20.42,20.44,20.15,20.33,10392,'0');
INSERT INTO daily_summary VALUES ('stock','BAP','us','2017-11-01',195.2,196.9,192.17,192.74,181212,'0');
INSERT INTO daily_summary VALUES ('stock','BAR','us','2017-11-01',127.38,127.51,127.38,127.51,246,'0');
INSERT INTO daily_summary VALUES ('stock','BAS','us','2017-11-01',19.12,19.64,18.85,19.06,567811,'0');
INSERT INTO daily_summary VALUES ('stock','BASI','us','2017-11-01',2.0462,2.13,1.98,2.0455,35420,'0');
INSERT INTO daily_summary VALUES ('stock','BATRA','us','2017-11-01',23.74,23.74,22.94,23.05,127906,'0');
INSERT INTO daily_summary VALUES ('stock','BATRK','us','2017-11-01',23.85,23.85,23.13,23.27,296835,'0');
INSERT INTO daily_summary VALUES ('stock','BAX','us','2017-11-01',64.6,64.86,64.19,64.2,1848700,'0');
INSERT INTO daily_summary VALUES ('stock','BB','us','2017-11-01',10.97,11.04,10.77,10.78,2337600,'0');
INSERT INTO daily_summary VALUES ('stock','BBBY','us','2017-11-01',19.99,20.46,19.94,20.33,2691320,'0');
INSERT INTO daily_summary VALUES ('stock','BBC','us','2017-11-01',27.94,27.9652,27.14,27.14,3584,'0');
INSERT INTO daily_summary VALUES ('stock','BBD','us','2017-11-01',10.45,10.52,10.23,10.27,8469120,'0');
INSERT INTO daily_summary VALUES ('stock','BBDO','us','2017-11-01',9.665,10.065,9.665,9.715,16517,'0');
INSERT INTO daily_summary VALUES ('stock','BBF','us','2017-11-01',14.85,14.85,14.7401,14.75,16104,'0');
INSERT INTO daily_summary VALUES ('stock','BBG','us','2017-11-01',5.11,6,5.05,5.99,6179040,'0');
INSERT INTO daily_summary VALUES ('stock','BBGI','us','2017-11-01',9.45,9.75,8.8341,9.5,31165,'0');
INSERT INTO daily_summary VALUES ('etf','BBH','us','2017-11-01',127.07,127.07,126,126.338,29250,'0');
INSERT INTO daily_summary VALUES ('stock','BBK','us','2017-11-01',15.18,15.2,15.1411,15.15,10801,'0');
INSERT INTO daily_summary VALUES ('stock','BBL','us','2017-11-01',37.63,37.7,37.11,37.14,1591340,'0');
INSERT INTO daily_summary VALUES ('stock','BBN','us','2017-11-01',23.19,23.28,23.141,23.27,110227,'0');
INSERT INTO daily_summary VALUES ('stock','BBOX','us','2017-11-01',3.35,3.4,3.15,3.2,518333,'0');
INSERT INTO daily_summary VALUES ('etf','BBP','us','2017-11-01',39.52,39.52,38.69,38.76,3804,'0');
INSERT INTO daily_summary VALUES ('etf','BBRC','us','2017-11-01',18,18.01,17.87,17.87,19148,'0');
INSERT INTO daily_summary VALUES ('stock','BBRG','us','2017-11-01',2.1,2.1,2,2,61499,'0');
INSERT INTO daily_summary VALUES ('stock','BBSI','us','2017-11-01',61.25,61.25,59.07,60.1,23511,'0');
INSERT INTO daily_summary VALUES ('stock','BBT','us','2017-11-01',49.009,49.565,49.009,49.208,7743300,'0');
INSERT INTO daily_summary VALUES ('stock','BBT_D','us','2017-11-01',25.65,25.65,25.56,25.62,20284,'0');
INSERT INTO daily_summary VALUES ('stock','BBT_E','us','2017-11-01',25.54,25.54,25.45,25.47,54368,'0');
INSERT INTO daily_summary VALUES ('stock','BBT_F','us','2017-11-01',25.18,25.22,25.13,25.1788,24395,'0');
INSERT INTO daily_summary VALUES ('stock','BBT_G','us','2017-11-01',25.26,25.48,25.26,25.46,31822,'0');
INSERT INTO daily_summary VALUES ('stock','BBT_H','us','2017-11-01',26.73,26.74,26.62,26.62,9481,'0');
INSERT INTO daily_summary VALUES ('stock','BBU','us','2017-11-01',29.38,29.76,29.38,29.75,38251,'0');
INSERT INTO daily_summary VALUES ('stock','BBVA','us','2017-11-01',8.68,8.71,8.64,8.69,3151140,'0');
INSERT INTO daily_summary VALUES ('stock','BBW','us','2017-11-01',7.9,7.95,7.65,7.7,120195,'0');
INSERT INTO daily_summary VALUES ('stock','BBX','us','2017-11-01',8.04,8.15,7.96,8,88000,'0');
INSERT INTO daily_summary VALUES ('stock','BBY','us','2017-11-01',56.18,56.36,55.51,56.01,2326280,'0');
INSERT INTO daily_summary VALUES ('stock','BC','us','2017-11-01',50.74,52.14,50.71,51.46,1724310,'0');
INSERT INTO daily_summary VALUES ('stock','BCACU','us','2017-11-01',10.27,10.27,10.27,10.27,100,'0');
INSERT INTO daily_summary VALUES ('stock','BCBP','us','2017-11-01',14.01,14.158,13.861,13.96,25320,'0');
INSERT INTO daily_summary VALUES ('stock','BCC','us','2017-11-01',36,37.25,35.4,35.85,841288,'0');
INSERT INTO daily_summary VALUES ('stock','BCE','us','2017-11-01',46.16,46.44,46.02,46.25,697964,'0');
INSERT INTO daily_summary VALUES ('stock','BCEI','us','2017-11-01',34.39,35.15,33.76,35.02,245538,'0');
INSERT INTO daily_summary VALUES ('stock','BCH','us','2017-11-01',92.73,93.47,90.87,91.3,39873,'0');
INSERT INTO daily_summary VALUES ('stock','BCI','us','2017-11-01',25.3,25.34,25.235,25.259,2407,'0');
INSERT INTO daily_summary VALUES ('stock','BCLI','us','2017-11-01',4.2,4.26,3.96,4.03,139075,'0');
INSERT INTO daily_summary VALUES ('etf','BCM','us','2017-11-01',28.25,28.28,28.2152,28.2377,2302,'0');
INSERT INTO daily_summary VALUES ('stock','BCO','us','2017-11-01',75.855,76.978,74.907,76.953,719922,'0');
INSERT INTO daily_summary VALUES ('stock','BCOM','us','2017-11-01',16.0034,16.0034,16.0034,16.0034,117,'0');
INSERT INTO daily_summary VALUES ('stock','BCOR','us','2017-11-01',21.85,21.95,20.6,21.05,1373700,'0');
INSERT INTO daily_summary VALUES ('stock','BCOV','us','2017-11-01',8,8,7.675,7.7,233629,'0');
INSERT INTO daily_summary VALUES ('stock','BCPC','us','2017-11-01',84.9,84.9,82.32,82.76,88272,'0');
INSERT INTO daily_summary VALUES ('stock','BCR','us','2017-11-01',327.5,328.03,325.31,326.78,602241,'0');
INSERT INTO daily_summary VALUES ('stock','BCRH','us','2017-11-01',13.45,13.5,12.95,13,43029,'0');
INSERT INTO daily_summary VALUES ('stock','BCRX','us','2017-11-01',4.52,4.54,4.33,4.48,1389970,'0');
INSERT INTO daily_summary VALUES ('stock','BCS','us','2017-11-01',9.76,9.81,9.69,9.69,2535450,'0');
INSERT INTO daily_summary VALUES ('stock','BCS_D','us','2017-11-01',26.61,26.67,26.5,26.53,81755,'0');
INSERT INTO daily_summary VALUES ('stock','BCV','us','2017-11-01',22,22.05,21.9344,21.94,3784,'0');
INSERT INTO daily_summary VALUES ('stock','BCV_A','us','2017-11-01',25.9719,26.09,25.9719,26.0387,1170,'0');
INSERT INTO daily_summary VALUES ('stock','BCX','us','2017-11-01',8.92,9.05,8.9,9.02,270254,'0');
INSERT INTO daily_summary VALUES ('stock','BDC','us','2017-11-01',81.5,82.8,79.88,82.66,345441,'0');
INSERT INTO daily_summary VALUES ('etf','BDCL','us','2017-11-01',16.6,16.6,16.2847,16.32,262340,'0');
INSERT INTO daily_summary VALUES ('etf','BDCS','us','2017-11-01',21.07,21.07,20.71,20.72,14039,'0');
INSERT INTO daily_summary VALUES ('stock','BDC_B','us','2017-11-01',108.73,109.96,108.73,109.96,11177,'0');
INSERT INTO daily_summary VALUES ('etf','BDD','us','2017-11-01',9.95,10,9.85,9.85,7994,'0');
INSERT INTO daily_summary VALUES ('stock','BDGE','us','2017-11-01',35.8,35.85,34.525,34.85,17093,'0');
INSERT INTO daily_summary VALUES ('stock','BDJ','us','2017-11-01',9.25,9.28,9.24,9.25,347760,'0');
INSERT INTO daily_summary VALUES ('stock','BDL','us','2017-11-01',22,22,21.5924,21.5924,491,'0');
INSERT INTO daily_summary VALUES ('stock','BDN','us','2017-11-01',17.5,17.59,17.43,17.52,1044680,'0');
INSERT INTO daily_summary VALUES ('stock','BDR','us','2017-11-01',0.4147,0.415,0.39,0.4,27597,'0');
INSERT INTO daily_summary VALUES ('stock','BDSI','us','2017-11-01',2.9,2.9,2.8,2.9,516464,'0');
INSERT INTO daily_summary VALUES ('stock','BDX','us','2017-11-01',207.53,210.1,203.99,208.06,2077030,'0');
INSERT INTO daily_summary VALUES ('stock','BDXA','us','2017-11-01',56.08,57.02,56.08,56.57,144243,'0');
INSERT INTO daily_summary VALUES ('stock','BEAT','us','2017-11-01',29.1,29.1748,28.05,28.2,342591,'0');
INSERT INTO daily_summary VALUES ('stock','BEBE','us','2017-11-01',5.5,5.62,5.37,5.43,77602,'0');
INSERT INTO daily_summary VALUES ('stock','BECN','us','2017-11-01',55.69,56.36,55.29,55.67,505112,'0');
INSERT INTO daily_summary VALUES ('stock','BEDU','us','2017-11-01',24.24,24.57,23.42,23.55,324566,'0');
INSERT INTO daily_summary VALUES ('stock','BEL','us','2017-11-01',13.2,13.3,13,13.1,286004,'0');
INSERT INTO daily_summary VALUES ('stock','BELFB','us','2017-11-01',32.75,32.9,31,31.45,23159,'0');
INSERT INTO daily_summary VALUES ('stock','BEMO','us','2017-11-01',30.23,30.23,29.89,29.969,50045,'0');
INSERT INTO daily_summary VALUES ('stock','BEN','us','2017-11-01',42.45,42.73,42.25,42.41,1759350,'0');
INSERT INTO daily_summary VALUES ('stock','BEP','us','2017-11-01',33.72,34.25,33.39,34.03,127585,'0');
INSERT INTO daily_summary VALUES ('stock','BERN','us','2017-11-01',25.44,25.44,25.325,25.325,500,'0');
INSERT INTO daily_summary VALUES ('stock','BERY','us','2017-11-01',59.69,59.86,59.13,59.17,962090,'0');
INSERT INTO daily_summary VALUES ('stock','BETR','us','2017-11-01',6.44,6.55,6.38,6.52,325013,'0');
INSERT INTO daily_summary VALUES ('stock','BFAM','us','2017-11-01',86.66,86.66,85.61,86.59,142850,'0');
INSERT INTO daily_summary VALUES ('stock','BFIN','us','2017-11-01',15.869,15.978,15.739,15.759,14398,'0');
INSERT INTO daily_summary VALUES ('stock','BFK','us','2017-11-01',14.06,14.17,14.06,14.13,85570,'0');
INSERT INTO daily_summary VALUES ('stock','BFO','us','2017-11-01',15.06,15.06,14.99,15.01,5010,'0');
INSERT INTO daily_summary VALUES ('etf','BFOR','us','2017-11-01',40.53,40.53,40.16,40.305,8621,'0');
INSERT INTO daily_summary VALUES ('stock','BFR','us','2017-11-01',22.06,22.26,21.32,21.38,613531,'0');
INSERT INTO daily_summary VALUES ('stock','BFS','us','2017-11-01',61.33,62.26,60.3988,61.81,17693,'0');
INSERT INTO daily_summary VALUES ('stock','BFS_C','us','2017-11-01',25.4,25.4926,25.38,25.4926,3435,'0');
INSERT INTO daily_summary VALUES ('stock','BFY','us','2017-11-01',15.31,15.31,15.07,15.09,6022,'0');
INSERT INTO daily_summary VALUES ('stock','BFZ','us','2017-11-01',14.36,14.4271,14.29,14.29,71280,'0');
INSERT INTO daily_summary VALUES ('stock','BG','us','2017-11-01',70.48,71.75,69.24,69.56,1995330,'0');
INSERT INTO daily_summary VALUES ('stock','BGB','us','2017-11-01',15.88,15.93,15.88,15.92,151409,'0');
INSERT INTO daily_summary VALUES ('stock','BGC','us','2017-11-01',21.05,21.4,20.75,20.9,1186890,'0');
INSERT INTO daily_summary VALUES ('stock','BGCA','us','2017-11-01',25.9801,26.035,25.9801,26.0153,1300,'0');
INSERT INTO daily_summary VALUES ('stock','BGCP','us','2017-11-01',14.989,15.107,14.771,14.87,1315130,'0');
INSERT INTO daily_summary VALUES ('stock','BGFV','us','2017-11-01',7.3,7.3,6.95,7.25,1931070,'0');
INSERT INTO daily_summary VALUES ('stock','BGG','us','2017-11-01',25.48,25.49,24.98,25.16,711970,'0');
INSERT INTO daily_summary VALUES ('stock','BGH','us','2017-11-01',20.7,20.7,20.5,20.51,79328,'0');
INSERT INTO daily_summary VALUES ('stock','BGI','us','2017-11-01',2.3,2.3,2.18,2.21,9573,'0');
INSERT INTO daily_summary VALUES ('stock','BGIO','us','2017-11-01',9.96,10.02,9.93,10.02,93337,'0');
INSERT INTO daily_summary VALUES ('stock','BGNE','us','2017-11-01',93.39,93.39,88.35,88.93,392557,'0');
INSERT INTO daily_summary VALUES ('stock','BGR','us','2017-11-01',13.64,13.76,13.621,13.76,107687,'0');
INSERT INTO daily_summary VALUES ('stock','BGS','us','2017-11-01',36.35,36.75,34.75,35.85,4594250,'0');
INSERT INTO daily_summary VALUES ('stock','BGSF','us','2017-11-01',16.68,16.68,15.3934,15.98,48012,'0');
INSERT INTO daily_summary VALUES ('stock','BGT','us','2017-11-01',14.31,14.33,14.26,14.26,100462,'0');
INSERT INTO daily_summary VALUES ('stock','BGX','us','2017-11-01',16.29,16.31,16.14,16.24,42925,'0');
INSERT INTO daily_summary VALUES ('stock','BGY','us','2017-11-01',6.6,6.6,6.57,6.59,220038,'0');
INSERT INTO daily_summary VALUES ('stock','BH','us','2017-11-01',358.1,360.69,351.345,360.26,4278,'0');
INSERT INTO daily_summary VALUES ('stock','BHACR','us','2017-11-01',0.1892,0.1892,0.1892,0.1892,100,'0');
INSERT INTO daily_summary VALUES ('stock','BHACW','us','2017-11-01',0.09,0.09,0.08,0.08,545,'0');
INSERT INTO daily_summary VALUES ('stock','BHB','us','2017-11-01',30.18,30.3401,29.25,29.42,28717,'0');
INSERT INTO daily_summary VALUES ('stock','BHBK','us','2017-11-01',21.85,21.85,20.55,20.7,69534,'0');
INSERT INTO daily_summary VALUES ('stock','BHE','us','2017-11-01',31.2,31.2,30.15,30.15,330591,'0');
INSERT INTO daily_summary VALUES ('stock','BHF','us','2017-11-01',62.15,62.84,61.61,62.05,645672,'0');
INSERT INTO daily_summary VALUES ('stock','BHGE','us','2017-11-01',31.567,32.154,31.378,31.667,3145310,'0');
INSERT INTO daily_summary VALUES ('stock','BHK','us','2017-11-01',14.16,14.16,14.12,14.15,89886,'0');
INSERT INTO daily_summary VALUES ('stock','BHLB','us','2017-11-01',38.48,38.629,37.635,38.033,110467,'0');
INSERT INTO daily_summary VALUES ('stock','BHP','us','2017-11-01',42.42,42.53,41.935,42.03,4202780,'0');
INSERT INTO daily_summary VALUES ('stock','BHV','us','2017-11-01',19.27,19.27,17.42,17.67,2040,'0');
INSERT INTO daily_summary VALUES ('stock','BHVN','us','2017-11-01',30.33,30.33,28.45,29.36,356094,'0');
INSERT INTO daily_summary VALUES ('etf','BIB','us','2017-11-01',54.7,55.045,53.28,53.49,201207,'0');
INSERT INTO daily_summary VALUES ('stock','BIBL','us','2017-11-01',25.08,25.13,25.0499,25.0499,187104,'0');
INSERT INTO daily_summary VALUES ('etf','BICK','us','2017-11-01',29.7,29.73,29.45,29.52,11838,'0');
INSERT INTO daily_summary VALUES ('stock','BID','us','2017-11-01',52.08,52.18,51.04,51.42,213775,'0');
INSERT INTO daily_summary VALUES ('stock','BIDU','us','2017-11-01',247.71,251.18,245.26,245.43,3618970,'0');
INSERT INTO daily_summary VALUES ('stock','BIF','us','2017-11-01',10.61,10.65,10.59,10.59,164905,'0');
INSERT INTO daily_summary VALUES ('stock','BIG','us','2017-11-01',51.7,52.04,51.27,51.8,554546,'0');
INSERT INTO daily_summary VALUES ('stock','BIIB','us','2017-11-01',312.75,319.95,311.57,315.73,1386170,'0');
INSERT INTO daily_summary VALUES ('etf','BIL','us','2017-11-01',45.72,45.72,45.71,45.72,1172180,'0');
INSERT INTO daily_summary VALUES ('stock','BIO','us','2017-11-01',219.96,220.98,217.5,217.86,213985,'0');
INSERT INTO daily_summary VALUES ('stock','BIOA','us','2017-11-01',0.53,0.53,0.51,0.5286,851107,'0');
INSERT INTO daily_summary VALUES ('stock','BIOC','us','2017-11-01',0.88,0.88,0.83,0.84,335781,'0');
INSERT INTO daily_summary VALUES ('stock','BIOL','us','2017-11-01',0.73,0.7897,0.71,0.75,88559,'0');
INSERT INTO daily_summary VALUES ('stock','BIOS','us','2017-11-01',2.52,2.55,2.35,2.4,1165900,'0');
INSERT INTO daily_summary VALUES ('stock','BIP','us','2017-11-01',42.41,42.5499,42.16,42.48,289525,'0');
INSERT INTO daily_summary VALUES ('etf','BIS','us','2017-11-01',22.76,23.48,22.76,23.38,47135,'0');
INSERT INTO daily_summary VALUES ('stock','BIT','us','2017-11-01',18.59,18.59,18.5,18.55,106887,'0');
INSERT INTO daily_summary VALUES ('stock','BITA','us','2017-11-01',45.46,47.37,44.8601,46.01,1296080,'0');
INSERT INTO daily_summary VALUES ('etf','BIV','us','2017-11-01',84.27,84.44,84.25,84.35,515692,'0');
INSERT INTO daily_summary VALUES ('stock','BIVV','us','2017-11-01',56.69,56.88,54.17,54.65,1295410,'0');
INSERT INTO daily_summary VALUES ('etf','BIZD','us','2017-11-01',16.99,16.99,16.88,16.88,51092,'0');
INSERT INTO daily_summary VALUES ('etf','BJK','us','2017-11-01',43.08,43.459,43.08,43.25,6520,'0');
INSERT INTO daily_summary VALUES ('stock','BJRI','us','2017-11-01',31.835,31.835,31.188,31.337,456865,'0');
INSERT INTO daily_summary VALUES ('stock','BJZ','us','2017-11-01',15.07,15.07,15.01,15.01,10797,'0');
INSERT INTO daily_summary VALUES ('stock','BK','us','2017-11-01',51.67,52.0786,51.56,51.65,3561500,'0');
INSERT INTO daily_summary VALUES ('stock','BKCC','us','2017-11-01',7.15,7.17,7.12,7.14,166082,'0');
INSERT INTO daily_summary VALUES ('stock','BKD','us','2017-11-01',10.13,10.3,9.83,9.84,3230410,'0');
INSERT INTO daily_summary VALUES ('stock','BKE','us','2017-11-01',16.45,16.75,16.15,16.6,391268,'0');
INSERT INTO daily_summary VALUES ('stock','BKEP','us','2017-11-01',5.605,5.605,5.5563,5.605,44749,'0');
INSERT INTO daily_summary VALUES ('stock','BKEPP','us','2017-11-01',7.71,7.84,7.66,7.84,21690,'0');
INSERT INTO daily_summary VALUES ('etf','BKF','us','2017-11-01',44.46,44.6,44.25,44.31,15683,'0');
INSERT INTO daily_summary VALUES ('stock','BKH','us','2017-11-01',65.68,65.71,64.28,64.45,784903,'0');
INSERT INTO daily_summary VALUES ('stock','BKHU','us','2017-11-01',70.01,70.01,69.5,69.51,173529,'0');
INSERT INTO daily_summary VALUES ('stock','BKI','us','2017-11-01',45.5,45.5,44.55,45.1,814063,'0');
INSERT INTO daily_summary VALUES ('stock','BKJ','us','2017-11-01',17.25,17.35,17.25,17.35,2120,'0');
INSERT INTO daily_summary VALUES ('stock','BKK','us','2017-11-01',15.33,15.38,15.23,15.3,52419,'0');
INSERT INTO daily_summary VALUES ('etf','BKLN','us','2017-11-01',23.13,23.14,23.11,23.11,1752420,'0');
INSERT INTO daily_summary VALUES ('stock','BKMU','us','2017-11-01',10.65,10.65,10.375,10.5,110913,'0');
INSERT INTO daily_summary VALUES ('stock','BKN','us','2017-11-01',14.62,14.66,14.57,14.6,62947,'0');
INSERT INTO daily_summary VALUES ('stock','BKS','us','2017-11-01',7.1,7.2,6.85,7,823640,'0');
INSERT INTO daily_summary VALUES ('stock','BKSC','us','2017-11-01',19,19,18.7,18.9053,2154,'0');
INSERT INTO daily_summary VALUES ('stock','BKT','us','2017-11-01',6.22,6.24,6.2,6.21,91247,'0');
INSERT INTO daily_summary VALUES ('stock','BKU','us','2017-11-01',34.41,35.13,33.45,33.69,1910280,'0');
INSERT INTO daily_summary VALUES ('stock','BKYI','us','2017-11-01',2.395,2.55,2.22,2.24,75579,'0');
INSERT INTO daily_summary VALUES ('stock','BK_C','us','2017-11-01',25.17,25.1953,25.13,25.13,22721,'0');
INSERT INTO daily_summary VALUES ('stock','BL','us','2017-11-01',35.6,35.6,34.04,35.25,244246,'0');
INSERT INTO daily_summary VALUES ('stock','BLBD','us','2017-11-01',20.8,20.8,20.15,20.25,26963,'0');
INSERT INTO daily_summary VALUES ('stock','BLCM','us','2017-11-01',9.26,10.42,9.26,9.75,611364,'0');
INSERT INTO daily_summary VALUES ('stock','BLD','us','2017-11-01',66.96,66.96,65.18,65.77,304907,'0');
INSERT INTO daily_summary VALUES ('stock','BLDP','us','2017-11-01',5.06,5.1243,4.94,5.07,2238200,'0');
INSERT INTO daily_summary VALUES ('stock','BLDR','us','2017-11-01',18.15,18.21,17.8807,18.02,1052400,'0');
INSERT INTO daily_summary VALUES ('stock','BLE','us','2017-11-01',14.37,14.465,14.35,14.35,68566,'0');
INSERT INTO daily_summary VALUES ('stock','BLES','us','2017-11-01',27.73,27.78,27.33,27.36,85062,'0');
INSERT INTO daily_summary VALUES ('stock','BLFS','us','2017-11-01',5.79,5.79,5.47,5.6,75842,'0');
INSERT INTO daily_summary VALUES ('stock','BLH','us','2017-11-01',14.9199,14.9199,14.9199,14.9199,1073,'0');
INSERT INTO daily_summary VALUES ('stock','BLHY','us','2017-11-01',25.0101,25.25,25.0101,25.25,18744,'0');
INSERT INTO daily_summary VALUES ('stock','BLIN','us','2017-11-01',2.6663,2.6663,2.11,2.51,35451,'0');
INSERT INTO daily_summary VALUES ('stock','BLJ','us','2017-11-01',14.82,14.91,14.62,14.7,10277,'0');
INSERT INTO daily_summary VALUES ('stock','BLK','us','2017-11-01',473.21,473.85,468.8,469.29,449755,'0');
INSERT INTO daily_summary VALUES ('stock','BLKB','us','2017-11-01',102.25,102.265,97.87,98.9,380359,'0');
INSERT INTO daily_summary VALUES ('stock','BLL','us','2017-11-01',43.2,43.24,42.61,42.7,2264760,'0');
INSERT INTO daily_summary VALUES ('stock','BLMN','us','2017-11-01',17.766,17.866,17.477,17.517,1079210,'0');
INSERT INTO daily_summary VALUES ('stock','BLMT','us','2017-11-01',30.15,30.2,29.8,29.85,4730,'0');
INSERT INTO daily_summary VALUES ('stock','BLPH','us','2017-11-01',1.37,1.41,1.3,1.38,120335,'0');
INSERT INTO daily_summary VALUES ('stock','BLRX','us','2017-11-01',1.02,1.03,1,1.03,438365,'0');
INSERT INTO daily_summary VALUES ('stock','BLUE','us','2017-11-01',144,161.825,142.65,142.95,2465140,'0');
INSERT INTO daily_summary VALUES ('etf','BLV','us','2017-11-01',93.48,93.79,93.365,93.61,391456,'0');
INSERT INTO daily_summary VALUES ('stock','BLVD','us','2017-11-01',10,10,9.99,9.99,2307,'0');
INSERT INTO daily_summary VALUES ('stock','BLVDU','us','2017-11-01',10.31,10.31,10.3,10.3,701,'0');
INSERT INTO daily_summary VALUES ('stock','BLW','us','2017-11-01',16.05,16.08,16,16.01,39709,'0');
INSERT INTO daily_summary VALUES ('stock','BLX','us','2017-11-01',28.2,28.4,27.88,28.2,83415,'0');
INSERT INTO daily_summary VALUES ('stock','BMA','us','2017-11-01',125.35,126.36,120.5,120.7,587576,'0');
INSERT INTO daily_summary VALUES ('stock','BMCH','us','2017-11-01',21.65,21.85,21.45,21.55,285017,'0');
INSERT INTO daily_summary VALUES ('stock','BME','us','2017-11-01',36.02,36.22,36.02,36.08,19246,'0');
INSERT INTO daily_summary VALUES ('stock','BMI','us','2017-11-01',44.15,44.3,43.75,43.95,149623,'0');
INSERT INTO daily_summary VALUES ('stock','BMLP','us','2017-11-01',46.97,47,46.97,47,212,'0');
INSERT INTO daily_summary VALUES ('stock','BML_G','us','2017-11-01',22.04,22.25,22,22.1,2628,'0');
INSERT INTO daily_summary VALUES ('stock','BML_H','us','2017-11-01',21.89,21.95,21.8401,21.94,5315,'0');
INSERT INTO daily_summary VALUES ('stock','BML_I','us','2017-11-01',25.9907,25.9907,25.94,25.9698,10630,'0');
INSERT INTO daily_summary VALUES ('stock','BML_J','us','2017-11-01',23.4,23.47,23.4,23.41,13827,'0');
INSERT INTO daily_summary VALUES ('stock','BML_L','us','2017-11-01',23.57,23.57,23.4643,23.53,15615,'0');
INSERT INTO daily_summary VALUES ('stock','BMO','us','2017-11-01',76.84,76.93,76.57,76.68,259641,'0');
INSERT INTO daily_summary VALUES ('stock','BMRA','us','2017-11-01',2.74,2.95,2.74,2.9272,34648,'0');
INSERT INTO daily_summary VALUES ('stock','BMRC','us','2017-11-01',68.108,68.108,66.963,67.61,10687,'0');
INSERT INTO daily_summary VALUES ('stock','BMRN','us','2017-11-01',82.57,82.875,80.6,80.92,1964510,'0');
INSERT INTO daily_summary VALUES ('stock','BMS','us','2017-11-01',45.15,45.66,45.01,45.24,509423,'0');
INSERT INTO daily_summary VALUES ('stock','BMTC','us','2017-11-01',44,44.3,43.5,43.7,51483,'0');
INSERT INTO daily_summary VALUES ('stock','BMY','us','2017-11-01',61.86,62.35,61.49,62.16,4713230,'0');
INSERT INTO daily_summary VALUES ('stock','BNCL','us','2017-11-01',16.65,16.65,16.15,16.3,240006,'0');
INSERT INTO daily_summary VALUES ('etf','BND','us','2017-11-01',81.59,81.72,81.52,81.65,1595960,'0');
INSERT INTO daily_summary VALUES ('stock','BNDC','us','2017-11-01',25.13,25.13,25.13,25.13,5131,'0');
INSERT INTO daily_summary VALUES ('etf','BNDX','us','2017-11-01',54.87,54.91,54.85,54.88,797105,'0');
INSERT INTO daily_summary VALUES ('stock','BNED','us','2017-11-01',5.49,5.7,5.26,5.56,401287,'0');
INSERT INTO daily_summary VALUES ('stock','BNFT','us','2017-11-01',27.55,28,26.25,26.3,192397,'0');
INSERT INTO daily_summary VALUES ('stock','BNJ','us','2017-11-01',15.9,15.9,15.46,15.5,16203,'0');
INSERT INTO daily_summary VALUES ('etf','BNO','us','2017-11-01',16.5,16.5599,16.1342,16.26,289294,'0');
INSERT INTO daily_summary VALUES ('stock','BNS','us','2017-11-01',64.82,64.85,64.51,64.59,493691,'0');
INSERT INTO daily_summary VALUES ('stock','BNSO','us','2017-11-01',2.44,2.49,2.42,2.45,1407,'0');
INSERT INTO daily_summary VALUES ('stock','BNTC','us','2017-11-01',2.87,2.87,2.75,2.8272,25495,'0');
INSERT INTO daily_summary VALUES ('stock','BNY','us','2017-11-01',14.81,14.81,14.7,14.7557,15623,'0');
INSERT INTO daily_summary VALUES ('stock','BOBE','us','2017-11-01',77.2,77.24,77.11,77.11,632944,'0');
INSERT INTO daily_summary VALUES ('stock','BOCH','us','2017-11-01',12,12,11.508,11.65,35787,'0');
INSERT INTO daily_summary VALUES ('stock','BOE','us','2017-11-01',13.65,13.665,13.6,13.65,181631,'0');
INSERT INTO daily_summary VALUES ('stock','BOFI','us','2017-11-01',27.11,27.19,26.46,27.06,478608,'0');
INSERT INTO daily_summary VALUES ('stock','BOFIL','us','2017-11-01',26.1017,26.69,25.94,26.69,3046,'0');
INSERT INTO daily_summary VALUES ('stock','BOH','us','2017-11-01',82.02,82.38,81.1,81.53,118118,'0');
INSERT INTO daily_summary VALUES ('etf','BOIL','us','2017-11-01',7.11,7.27,7.1,7.26,589703,'0');
INSERT INTO daily_summary VALUES ('stock','BOJA','us','2017-11-01',12.2,12.4,11.751,12.1,1015690,'0');
INSERT INTO daily_summary VALUES ('stock','BOKF','us','2017-11-01',86.785,87.422,85.323,85.512,72534,'0');
INSERT INTO daily_summary VALUES ('stock','BOKFL','us','2017-11-01',25.54,25.54,25.32,25.41,6379,'0');
INSERT INTO daily_summary VALUES ('stock','BOLD','us','2017-11-01',26.84,27.019,25.592,25.75,355569,'0');
INSERT INTO daily_summary VALUES ('stock','BOMN','us','2017-11-01',17.74,18.4,17.5872,18.25,80445,'0');
INSERT INTO daily_summary VALUES ('etf','BOND','us','2017-11-01',106.12,106.265,106.047,106.18,65486,'0');
INSERT INTO daily_summary VALUES ('stock','BONT','us','2017-11-01',0.765,0.779,0.63,0.677,1382490,'0');
INSERT INTO daily_summary VALUES ('stock','BOOM','us','2017-11-01',21.85,21.85,21.45,21.55,99463,'0');
INSERT INTO daily_summary VALUES ('stock','BOOT','us','2017-11-01',8.23,8.39,7.97,8.11,436628,'0');
INSERT INTO daily_summary VALUES ('stock','BORN','us','2017-11-01',1.37,1.37,1.33,1.34,7561,'0');
INSERT INTO daily_summary VALUES ('stock','BOSC','us','2017-11-01',1.89,1.89,1.84,1.85,16530,'0');
INSERT INTO daily_summary VALUES ('stock','BOSS','us','2017-11-01',16.8759,16.8759,16.8665,16.8665,782,'0');
INSERT INTO daily_summary VALUES ('stock','BOTJ','us','2017-11-01',14.91,14.969,14.6,14.6,112856,'0');
INSERT INTO daily_summary VALUES ('stock','BOTZ','us','2017-11-01',23.78,23.78,23.6,23.65,1533950,'0');
INSERT INTO daily_summary VALUES ('stock','BOX','us','2017-11-01',22.08,22.27,21.575,21.82,1226940,'0');
INSERT INTO daily_summary VALUES ('stock','BP','us','2017-11-01',40.406,40.613,40.071,40.179,6849060,'0');
INSERT INTO daily_summary VALUES ('stock','BPFH','us','2017-11-01',15.888,15.987,15.391,15.54,272601,'0');
INSERT INTO daily_summary VALUES ('stock','BPFHP','us','2017-11-01',25.7768,25.7886,25.66,25.7212,5880,'0');
INSERT INTO daily_summary VALUES ('stock','BPI','us','2017-11-01',9.81,9.86,9.51,9.69,325477,'0');
INSERT INTO daily_summary VALUES ('stock','BPK','us','2017-11-01',14.99,14.99,14.98,14.99,58151,'0');
INSERT INTO daily_summary VALUES ('stock','BPL','us','2017-11-01',52.465,52.641,51.773,52.348,490450,'0');
INSERT INTO daily_summary VALUES ('stock','BPMC','us','2017-11-01',68.55,72.49,67.78,71.8,760958,'0');
INSERT INTO daily_summary VALUES ('stock','BPMP','us','2017-11-01',18,18.63,18,18.42,1656300,'0');
INSERT INTO daily_summary VALUES ('stock','BPMX','us','2017-11-01',0.19,0.1901,0.1626,0.1761,1108720,'0');
INSERT INTO daily_summary VALUES ('stock','BPOP','us','2017-11-01',36.58,37.0073,34.17,34.65,2625290,'0');
INSERT INTO daily_summary VALUES ('stock','BPOPM','us','2017-11-01',22.7228,22.7228,21.91,22.4,4571,'0');
INSERT INTO daily_summary VALUES ('stock','BPOPN','us','2017-11-01',23.28,23.6,23.28,23.3552,14271,'0');
INSERT INTO daily_summary VALUES ('stock','BPRN','us','2017-11-01',33.03,33.95,33,33.6,12766,'0');
INSERT INTO daily_summary VALUES ('stock','BPT','us','2017-11-01',19.1,19.3522,19,19.15,87367,'0');
INSERT INTO daily_summary VALUES ('stock','BPTH','us','2017-11-01',0.3978,0.4149,0.3724,0.39,212359,'0');
INSERT INTO daily_summary VALUES ('stock','BPY','us','2017-11-01',23.51,23.54,23.26,23.52,89480,'0');
INSERT INTO daily_summary VALUES ('stock','BQH','us','2017-11-01',14.84,14.84,14.71,14.8,5818,'0');
INSERT INTO daily_summary VALUES ('stock','BR','us','2017-11-01',86.14,86.72,85.88,86.42,401886,'0');
INSERT INTO daily_summary VALUES ('stock','BRAC','us','2017-11-01',9.52,9.57,9.52,9.57,1300,'0');
INSERT INTO daily_summary VALUES ('stock','BRACR','us','2017-11-01',0.29,0.29,0.28,0.28,800,'0');
INSERT INTO daily_summary VALUES ('stock','BRACU','us','2017-11-01',10.08,10.08,10.06,10.08,6434,'0');
INSERT INTO daily_summary VALUES ('stock','BRACW','us','2017-11-01',0.32,0.32,0.319979,0.319979,28000,'0');
INSERT INTO daily_summary VALUES ('stock','BRC','us','2017-11-01',38.3,38.4,37.6,37.75,138725,'0');
INSERT INTO daily_summary VALUES ('stock','BRCD','us','2017-11-01',11.68,11.77,11.63,11.74,5528570,'0');
INSERT INTO daily_summary VALUES ('stock','BREW','us','2017-11-01',18.25,18.35,17.4,17.65,71661,'0');
INSERT INTO daily_summary VALUES ('etf','BRF','us','2017-11-01',24.01,24.22,23.96,24.05,17763,'0');
INSERT INTO daily_summary VALUES ('stock','BRFS','us','2017-11-01',13.49,13.65,13.37,13.38,1234330,'0');
INSERT INTO daily_summary VALUES ('stock','BRG','us','2017-11-01',11.35,11.35,11.01,11.24,348389,'0');
INSERT INTO daily_summary VALUES ('stock','BRGL','us','2017-11-01',25.23,25.35,25.23,25.35,700,'0');
INSERT INTO daily_summary VALUES ('stock','BRG_A','us','2017-11-01',26.64,26.64,26.5882,26.6097,1216,'0');
INSERT INTO daily_summary VALUES ('stock','BRG_C','us','2017-11-01',25.96,25.96,25.9,25.935,2250,'0');
INSERT INTO daily_summary VALUES ('stock','BRG_D','us','2017-11-01',25.51,25.53,25.51,25.52,412,'0');
INSERT INTO daily_summary VALUES ('stock','BRID','us','2017-11-01',12.84,13.1,12.8001,13.1,599,'0');
INSERT INTO daily_summary VALUES ('stock','BRKL','us','2017-11-01',15.46,15.509,15.211,15.41,349555,'0');
INSERT INTO daily_summary VALUES ('stock','BRKR','us','2017-11-01',31.42,31.99,31.3,31.89,518873,'0');
INSERT INTO daily_summary VALUES ('stock','BRKS','us','2017-11-01',34.77,34.77,32.56,33.03,843981,'0');
INSERT INTO daily_summary VALUES ('stock','BRN','us','2017-11-01',1.9755,2.049,1.95,1.95,2672,'0');
INSERT INTO daily_summary VALUES ('stock','BRO','us','2017-11-01',50.06,50.44,49.89,50.09,819710,'0');
INSERT INTO daily_summary VALUES ('stock','BRQS','us','2017-11-01',4.35,4.35,4.2,4.2,3401,'0');
INSERT INTO daily_summary VALUES ('stock','BRQSW','us','2017-11-01',0.2,0.2,0.2,0.2,1000,'0');
INSERT INTO daily_summary VALUES ('stock','BRS','us','2017-11-01',9.61,9.89,9.05,9.06,610907,'0');
INSERT INTO daily_summary VALUES ('stock','BRSS','us','2017-11-01',35.338,35.837,34.44,35.089,98753,'0');
INSERT INTO daily_summary VALUES ('stock','BRT','us','2017-11-01',10.15,10.15,9.98,10.01,7267,'0');
INSERT INTO daily_summary VALUES ('stock','BRX','us','2017-11-01',17.65,18.07,17.52,17.99,5666530,'0');
INSERT INTO daily_summary VALUES ('etf','BRZU','us','2017-11-01',40.66,41.85,39.28,39.57,602696,'0');
INSERT INTO daily_summary VALUES ('stock','BSAC','us','2017-11-01',31.45,31.885,30.9,30.96,155328,'0');
INSERT INTO daily_summary VALUES ('stock','BSBR','us','2017-11-01',8.89,9.105,8.85,8.86,1939230,'0');
INSERT INTO daily_summary VALUES ('etf','BSCH','us','2017-11-01',22.58,22.61,22.58,22.6,140558,'0');
INSERT INTO daily_summary VALUES ('etf','BSCI','us','2017-11-01',21.15,21.166,21.14,21.15,1406500,'0');
INSERT INTO daily_summary VALUES ('etf','BSCJ','us','2017-11-01',21.2,21.21,21.18,21.2,140793,'0');
INSERT INTO daily_summary VALUES ('etf','BSCK','us','2017-11-01',21.46,21.46,21.41,21.43,210805,'0');
INSERT INTO daily_summary VALUES ('etf','BSCL','us','2017-11-01',21.29,21.31,21.28,21.28,107456,'0');
INSERT INTO daily_summary VALUES ('etf','BSCM','us','2017-11-01',21.34,21.41,21.29,21.29,124215,'0');
INSERT INTO daily_summary VALUES ('etf','BSCN','us','2017-11-01',21.02,21.1,21.02,21.035,58280,'0');
INSERT INTO daily_summary VALUES ('etf','BSCO','us','2017-11-01',20.97,21.02,20.931,20.931,37765,'0');
INSERT INTO daily_summary VALUES ('stock','BSCP','us','2017-11-01',20.76,20.77,20.736,20.75,17073,'0');
INSERT INTO daily_summary VALUES ('stock','BSCQ','us','2017-11-01',19.742,19.802,19.722,19.762,13393,'0');
INSERT INTO daily_summary VALUES ('stock','BSD','us','2017-11-01',13.82,13.86,13.8,13.85,6295,'0');
INSERT INTO daily_summary VALUES ('stock','BSE','us','2017-11-01',13.18,13.24,13.18,13.22,10723,'0');
INSERT INTO daily_summary VALUES ('stock','BSET','us','2017-11-01',39.235,39.235,37.839,38.139,18685,'0');
INSERT INTO daily_summary VALUES ('stock','BSF','us','2017-11-01',10.26,10.26,10.23,10.26,20891,'0');
INSERT INTO daily_summary VALUES ('stock','BSFT','us','2017-11-01',54.85,54.85,54.8,54.85,1832630,'0');
INSERT INTO daily_summary VALUES ('etf','BSJH','us','2017-11-01',25.68,25.69,25.68,25.68,85290,'0');
INSERT INTO daily_summary VALUES ('etf','BSJI','us','2017-11-01',25.21,25.23,25.21,25.23,112611,'0');
INSERT INTO daily_summary VALUES ('etf','BSJJ','us','2017-11-01',24.61,24.65,24.6,24.6,84596,'0');
INSERT INTO daily_summary VALUES ('etf','BSJK','us','2017-11-01',24.83,24.85,24.81,24.84,74178,'0');
INSERT INTO daily_summary VALUES ('etf','BSJL','us','2017-11-01',25.15,25.17,25.114,25.15,51303,'0');
INSERT INTO daily_summary VALUES ('etf','BSJM','us','2017-11-01',25.27,25.33,25.264,25.27,43756,'0');
INSERT INTO daily_summary VALUES ('stock','BSJN','us','2017-11-01',26.74,26.74,26.691,26.691,14029,'0');
INSERT INTO daily_summary VALUES ('stock','BSJO','us','2017-11-01',25.894,25.894,25.771,25.873,4534,'0');
INSERT INTO daily_summary VALUES ('stock','BSJP','us','2017-11-01',25.15,25.15,25.11,25.11,2662,'0');
INSERT INTO daily_summary VALUES ('stock','BSL','us','2017-11-01',17.43,17.4627,17.31,17.31,44820,'0');
INSERT INTO daily_summary VALUES ('stock','BSM','us','2017-11-01',18.13,18.46,18.085,18.34,110005,'0');
INSERT INTO daily_summary VALUES ('stock','BSMX','us','2017-11-01',8.42,8.45,8.23,8.25,2528860,'0');
INSERT INTO daily_summary VALUES ('stock','BSPM','us','2017-11-01',1.51,1.57,1.45,1.52,20088,'0');
INSERT INTO daily_summary VALUES ('stock','BSQR','us','2017-11-01',5.2615,5.35,4.7,4.8,70482,'0');
INSERT INTO daily_summary VALUES ('stock','BSRR','us','2017-11-01',26.61,26.61,26,26.14,23015,'0');
INSERT INTO daily_summary VALUES ('stock','BST','us','2017-11-01',26.7,26.8299,26.6,26.62,125962,'0');
INSERT INTO daily_summary VALUES ('stock','BSTC','us','2017-11-01',45.83,46.5,45.36,45.43,5289,'0');
INSERT INTO daily_summary VALUES ('stock','BSTI','us','2017-11-01',11.58,11.65,11.23,11.62,748788,'0');
INSERT INTO daily_summary VALUES ('etf','BSV','us','2017-11-01',79.54,79.57,79.5125,79.55,1045360,'0');
INSERT INTO daily_summary VALUES ('stock','BSX','us','2017-11-01',28.23,28.27,27.77,27.85,6185260,'0');
INSERT INTO daily_summary VALUES ('stock','BT','us','2017-11-01',17.59,17.63,17.52,17.55,980620,'0');
INSERT INTO daily_summary VALUES ('stock','BTA','us','2017-11-01',11.98,11.99,11.9,11.93,40201,'0');
INSERT INTO daily_summary VALUES ('stock','BTE','us','2017-11-01',2.81,2.96,2.7713,2.89,2470530,'0');
INSERT INTO daily_summary VALUES ('stock','BTEC','us','2017-11-01',29.9,29.9,29.9,29.9,200,'0');
INSERT INTO daily_summary VALUES ('stock','BTG','us','2017-11-01',2.57,2.59,2.525,2.54,2002390,'0');
INSERT INTO daily_summary VALUES ('stock','BTI','us','2017-11-01',65.01,65.18,64.59,64.79,3346210,'0');
INSERT INTO daily_summary VALUES ('stock','BTN','us','2017-11-01',6.3,6.45,6.3,6.45,9671,'0');
INSERT INTO daily_summary VALUES ('stock','BTO','us','2017-11-01',37.68,37.86,37.391,37.43,34938,'0');
INSERT INTO daily_summary VALUES ('stock','BTT','us','2017-11-01',22.27,22.38,22.26,22.38,128635,'0');
INSERT INTO daily_summary VALUES ('stock','BTU','us','2017-11-01',31.03,32.04,31.03,31.73,1571320,'0');
INSERT INTO daily_summary VALUES ('stock','BTX','us','2017-11-01',2.41,2.4154,2.285,2.31,419013,'0');
INSERT INTO daily_summary VALUES ('stock','BTZ','us','2017-11-01',13.38,13.44,13.3776,13.43,149051,'0');
INSERT INTO daily_summary VALUES ('stock','BUD','us','2017-11-01',122.44,123.24,122,122.14,1347570,'0');
INSERT INTO daily_summary VALUES ('stock','BUFF','us','2017-11-01',28.94,29.07,28.02,28.23,1483540,'0');
INSERT INTO daily_summary VALUES ('stock','BUI','us','2017-11-01',20.93,21.04,20.8075,20.96,22593,'0');
INSERT INTO daily_summary VALUES ('stock','BUR','us','2017-11-01',0.5,0.5,0.4499,0.47,59991,'0');
INSERT INTO daily_summary VALUES ('stock','BURL','us','2017-11-01',94.86,96.48,94.15,95.35,1298830,'0');
INSERT INTO daily_summary VALUES ('stock','BUSE','us','2017-11-01',31.37,31.48,30.65,30.84,141478,'0');
INSERT INTO daily_summary VALUES ('stock','BUZ','us','2017-11-01',31.3359,31.3359,31.07,31.095,3944,'0');
INSERT INTO daily_summary VALUES ('stock','BV','us','2017-11-01',4.85,4.85,4.7,4.7,89490,'0');
INSERT INTO daily_summary VALUES ('stock','BVAL','us','2017-11-01',14.64,14.64,14.64,14.64,3525,'0');
INSERT INTO daily_summary VALUES ('stock','BVN','us','2017-11-01',13.85,14.39,13.8,14.25,2083780,'0');
INSERT INTO daily_summary VALUES ('stock','BVSN','us','2017-11-01',4.1,4.1,3.9,4,2098,'0');
INSERT INTO daily_summary VALUES ('stock','BVX','us','2017-11-01',3.97,3.99,3.85,3.96,109646,'0');
INSERT INTO daily_summary VALUES ('stock','BVXV','us','2017-11-01',7.2856,7.2901,7.12,7.12,10568,'0');
INSERT INTO daily_summary VALUES ('stock','BW','us','2017-11-01',4.4,4.48,4.19,4.21,1740840,'0');
INSERT INTO daily_summary VALUES ('stock','BWA','us','2017-11-01',53.39,53.48,52.62,52.7,986595,'0');
INSERT INTO daily_summary VALUES ('stock','BWEN','us','2017-11-01',2.78,2.95,2.64,2.66,106941,'0');
INSERT INTO daily_summary VALUES ('stock','BWFG','us','2017-11-01',36.6,36.6,35.74,35.86,6827,'0');
INSERT INTO daily_summary VALUES ('stock','BWG','us','2017-11-01',13.08,13.09,13.016,13.05,75403,'0');
INSERT INTO daily_summary VALUES ('stock','BWINB','us','2017-11-01',23.1,23.1,22.7,22.8,9500,'0');
INSERT INTO daily_summary VALUES ('stock','BWLD','us','2017-11-01',118.5,119.7,114,114.15,717231,'0');
INSERT INTO daily_summary VALUES ('stock','BWP','us','2017-11-01',14.014,14.138,13.845,14.083,1381980,'0');
INSERT INTO daily_summary VALUES ('etf','BWX','us','2017-11-01',27.76,27.87,27.75,27.77,618996,'0');
INSERT INTO daily_summary VALUES ('stock','BWXT','us','2017-11-01',60.16,60.55,59.975,60.36,409661,'0');
INSERT INTO daily_summary VALUES ('etf','BWZ','us','2017-11-01',31.56,31.63,31.35,31.41,133628,'0');
INSERT INTO daily_summary VALUES ('stock','BX','us','2017-11-01',33.41,33.5,32.72,32.89,4649140,'0');
INSERT INTO daily_summary VALUES ('stock','BXC','us','2017-11-01',8.75,9.02,8.71,8.82,131997,'0');
INSERT INTO daily_summary VALUES ('stock','BXE','us','2017-11-01',2.37,2.46,2.37,2.42,201834,'0');
INSERT INTO daily_summary VALUES ('stock','BXMT','us','2017-11-01',31.92,32,31.75,31.94,644358,'0');
INSERT INTO daily_summary VALUES ('stock','BXMX','us','2017-11-01',14.1,14.15,14.0928,14.12,148179,'0');
INSERT INTO daily_summary VALUES ('stock','BXP','us','2017-11-01',121.24,122.74,121.08,121.98,614901,'0');
INSERT INTO daily_summary VALUES ('stock','BXP_B','us','2017-11-01',25.83,26.3,25.83,25.96,9700,'0');
INSERT INTO daily_summary VALUES ('stock','BXS','us','2017-11-01',32.1,32.3,31.35,31.9,770147,'0');
INSERT INTO daily_summary VALUES ('stock','BY','us','2017-11-01',20.05,20.22,19.72,19.93,85762,'0');
INSERT INTO daily_summary VALUES ('stock','BYBK','us','2017-11-01',11.95,11.95,11.7,11.9,4003,'0');
INSERT INTO daily_summary VALUES ('stock','BYD','us','2017-11-01',29.47,29.69,28.96,29.1,1106820,'0');
INSERT INTO daily_summary VALUES ('stock','BYFC','us','2017-11-01',2.411,2.5999,2.4,2.5499,4258,'0');
INSERT INTO daily_summary VALUES ('etf','BYLD','us','2017-11-01',25.08,25.08,25.037,25.06,3486,'0');
INSERT INTO daily_summary VALUES ('stock','BYM','us','2017-11-01',14.35,14.41,14.28,14.28,52292,'0');
INSERT INTO daily_summary VALUES ('stock','BYSI','us','2017-11-01',35.81,35.81,35.81,35.81,153,'0');
INSERT INTO daily_summary VALUES ('etf','BZF','us','2017-11-01',18.16,18.41,18.11,18.27,22359,'0');
INSERT INTO daily_summary VALUES ('stock','BZH','us','2017-11-01',21.15,21.46,20.845,21.34,393381,'0');
INSERT INTO daily_summary VALUES ('stock','BZM','us','2017-11-01',14.06,14.06,13.88,13.93,2071,'0');
INSERT INTO daily_summary VALUES ('etf','BZQ','us','2017-11-01',9.8,9.9599,9.55,9.94,348153,'0');
INSERT INTO daily_summary VALUES ('stock','BZUN','us','2017-11-01',31.59,32.16,30.851,31.55,969994,'0');
INSERT INTO daily_summary VALUES ('stock','C','us','2017-11-01',73.673,74.201,73.494,73.713,11407000,'0');
INSERT INTO daily_summary VALUES ('stock','CA','us','2017-11-01',32.56,32.99,32.355,32.93,2359360,'0');
INSERT INTO daily_summary VALUES ('stock','CAA','us','2017-11-01',49.45,50.42,49.16,50.3,5574250,'0');
INSERT INTO daily_summary VALUES ('stock','CAAS','us','2017-11-01',5.05,5.1,5.05,5.07,22477,'0');
INSERT INTO daily_summary VALUES ('stock','CABO','us','2017-11-01',714.93,715.46,707.095,715.46,49873,'0');
INSERT INTO daily_summary VALUES ('stock','CAC','us','2017-11-01',41.66,42.15,40.25,42,72677,'0');
INSERT INTO daily_summary VALUES ('stock','CACC','us','2017-11-01',287.9,288.37,281.52,283.08,162409,'0');
INSERT INTO daily_summary VALUES ('stock','CACG','us','2017-11-01',26.79,26.79,26.62,26.67,1071,'0');
INSERT INTO daily_summary VALUES ('stock','CACI','us','2017-11-01',145.1,145.1,139.4,142.15,192371,'0');
INSERT INTO daily_summary VALUES ('stock','CADC','us','2017-11-01',1.8,1.8,1.75,1.7908,1989,'0');
INSERT INTO daily_summary VALUES ('stock','CADE','us','2017-11-01',24.51,24.94,24.35,24.52,43213,'0');
INSERT INTO daily_summary VALUES ('stock','CAE','us','2017-11-01',17.7,18.17,17.67,18.08,566042,'0');
INSERT INTO daily_summary VALUES ('stock','CAF','us','2017-11-01',24.36,24.36,24.09,24.12,65992,'0');
INSERT INTO daily_summary VALUES ('stock','CAFD','us','2017-11-01',15.18,15.18,14.88,15.1,81856,'0');
INSERT INTO daily_summary VALUES ('etf','CAFE','us','2017-11-01',11.38,11.38,11.28,11.28,3936,'0');
INSERT INTO daily_summary VALUES ('stock','CAG','us','2017-11-01',34.26,34.53,34.1,34.13,2854680,'0');
INSERT INTO daily_summary VALUES ('stock','CAH','us','2017-11-01',60.81,62.6,60.8,62.18,3484540,'0');
INSERT INTO daily_summary VALUES ('stock','CAI','us','2017-11-01',37.42,37.645,35.3,36.23,314009,'0');
INSERT INTO daily_summary VALUES ('stock','CAJ','us','2017-11-01',37.89,37.89,37.48,37.69,230535,'0');
INSERT INTO daily_summary VALUES ('stock','CAKE','us','2017-11-01',44.7,45.38,44.26,45,2700180,'0');
INSERT INTO daily_summary VALUES ('stock','CAL','us','2017-11-01',27.59,27.82,26.89,27.33,228516,'0');
INSERT INTO daily_summary VALUES ('stock','CALA','us','2017-11-01',16.45,16.55,15.9,16,358693,'0');
INSERT INTO daily_summary VALUES ('stock','CALD','us','2017-11-01',25.45,25.45,25,25.4,199883,'0');
INSERT INTO daily_summary VALUES ('stock','CALF','us','2017-11-01',24.51,24.51,24.45,24.45,1323,'0');
INSERT INTO daily_summary VALUES ('stock','CALI','us','2017-11-01',2.42,3.33,2.2536,2.41,2088860,'0');
INSERT INTO daily_summary VALUES ('stock','CALL','us','2017-11-01',6.35,6.475,6.2,6.2,49550,'0');
INSERT INTO daily_summary VALUES ('stock','CALM','us','2017-11-01',45,45.2,43.75,43.8,351568,'0');
INSERT INTO daily_summary VALUES ('stock','CALX','us','2017-11-01',5.6,5.6,5.4,5.45,194754,'0');
INSERT INTO daily_summary VALUES ('stock','CAMP','us','2017-11-01',22.9,22.9,21.84,22.09,326339,'0');
INSERT INTO daily_summary VALUES ('stock','CAMT','us','2017-11-01',5.68,5.71,5.62,5.65,132851,'0');
INSERT INTO daily_summary VALUES ('etf','CANE','us','2017-11-01',9.73,9.76,9.64,9.65,19800,'0');
INSERT INTO daily_summary VALUES ('stock','CANF','us','2017-11-01',1.65,1.66,1.5,1.58,47257,'0');
INSERT INTO daily_summary VALUES ('etf','CAPE','us','2017-11-01',112.1,112.15,111.921,111.921,2341,'0');
INSERT INTO daily_summary VALUES ('stock','CAPL','us','2017-11-01',25.103,25.599,24.889,25.221,95856,'0');
INSERT INTO daily_summary VALUES ('stock','CAPR','us','2017-11-01',2.46,2.54,2.35,2.35,311603,'0');
INSERT INTO daily_summary VALUES ('stock','CAR','us','2017-11-01',41.64,41.64,41.04,41.15,2099890,'0');
INSERT INTO daily_summary VALUES ('stock','CARA','us','2017-11-01',12.62,12.89,12.12,12.18,906869,'0');
INSERT INTO daily_summary VALUES ('stock','CARB','us','2017-11-01',22.9,23.05,22.25,22.6,206384,'0');
INSERT INTO daily_summary VALUES ('stock','CARG','us','2017-11-01',32.93,33.23,31.61,31.72,554526,'0');
INSERT INTO daily_summary VALUES ('stock','CARO','us','2017-11-01',36.93,37.77,36.93,37.28,676831,'0');
INSERT INTO daily_summary VALUES ('stock','CARS','us','2017-11-01',23.92,24.27,23.72,23.79,907951,'0');
INSERT INTO daily_summary VALUES ('etf','CARZ','us','2017-11-01',42.02,42.0745,41.98,42,9005,'0');
INSERT INTO daily_summary VALUES ('stock','CASC','us','2017-11-01',4.64,4.65,4.26,4.36,250514,'0');
INSERT INTO daily_summary VALUES ('stock','CASH','us','2017-11-01',88.95,89.15,85.906,86.15,57573,'0');
INSERT INTO daily_summary VALUES ('stock','CASI','us','2017-11-01',2.1,2.1587,2.02,2.08,114088,'0');
INSERT INTO daily_summary VALUES ('stock','CASM','us','2017-11-01',0.7685,0.7951,0.75,0.76,48099,'0');
INSERT INTO daily_summary VALUES ('stock','CASS','us','2017-11-01',64.92,64.93,64.07,64.75,20627,'0');
INSERT INTO daily_summary VALUES ('stock','CASY','us','2017-11-01',114.74,115.95,114.18,114.52,468217,'0');
INSERT INTO daily_summary VALUES ('stock','CAT','us','2017-11-01',135.67,137.3,135.43,136.29,3628080,'0');
INSERT INTO daily_summary VALUES ('stock','CATB','us','2017-11-01',2.05,2.05,1.93,1.96,304448,'0');
INSERT INTO daily_summary VALUES ('stock','CATC','us','2017-11-01',72.2,73.5,72.2,73.5,977,'0');
INSERT INTO daily_summary VALUES ('stock','CATH','us','2017-11-01',31.925,31.95,31.8,31.83,46920,'0');
INSERT INTO daily_summary VALUES ('stock','CATM','us','2017-11-01',23.01,23.11,22.2,22.58,275390,'0');
INSERT INTO daily_summary VALUES ('stock','CATO','us','2017-11-01',12.97,13.08,12.73,12.95,178699,'0');
INSERT INTO daily_summary VALUES ('stock','CATS','us','2017-11-01',3.58,3.78,3.506,3.66,48677,'0');
INSERT INTO daily_summary VALUES ('stock','CATY','us','2017-11-01',42.23,42.32,41.2,41.72,357518,'0');
INSERT INTO daily_summary VALUES ('stock','CATYW','us','2017-11-01',21.35,21.35,21.35,21.35,500,'0');
INSERT INTO daily_summary VALUES ('stock','CAVM','us','2017-11-01',69.55,69.584,66.96,68.53,1370670,'0');
INSERT INTO daily_summary VALUES ('stock','CAW','us','2017-11-01',2.95,3.05,2.95,2.9527,3733,'0');
INSERT INTO daily_summary VALUES ('stock','CB','us','2017-11-01',150.53,152.25,150.51,151.31,1511370,'0');
INSERT INTO daily_summary VALUES ('stock','CBA','us','2017-11-01',7.99,8.11,7.99,8.1,616182,'0');
INSERT INTO daily_summary VALUES ('stock','CBAK','us','2017-11-01',1.8,1.95,1.8,1.95,98232,'0');
INSERT INTO daily_summary VALUES ('stock','CBAN','us','2017-11-01',13.8,13.9,13.8,13.9,13291,'0');
INSERT INTO daily_summary VALUES ('stock','CBAY','us','2017-11-01',9.25,9.34,8.57,8.82,1088300,'0');
INSERT INTO daily_summary VALUES ('stock','CBB','us','2017-11-01',19.35,19.35,18.45,18.75,347342,'0');
INSERT INTO daily_summary VALUES ('stock','CBB_B','us','2017-11-01',51.21,51.48,51.13,51.46,2989,'0');
INSERT INTO daily_summary VALUES ('stock','CBD','us','2017-11-01',23.16,23.5798,22.63,22.94,390986,'0');
INSERT INTO daily_summary VALUES ('stock','CBF','us','2017-11-01',40.628,40.828,40.055,40.08,240705,'0');
INSERT INTO daily_summary VALUES ('stock','CBFV','us','2017-11-01',29.5,30.2,29.45,30.148,6723,'0');
INSERT INTO daily_summary VALUES ('stock','CBG','us','2017-11-01',39.61,39.7492,39.015,39.25,2085450,'0');
INSERT INTO daily_summary VALUES ('stock','CBH','us','2017-11-01',9.4235,9.4534,9.3926,9.4534,19764,'0');
INSERT INTO daily_summary VALUES ('stock','CBI','us','2017-11-01',14.06,14.45,14,14.36,4080010,'0');
INSERT INTO daily_summary VALUES ('stock','CBIO','us','2017-11-01',4.9653,4.99,4.551,4.71,45810,'0');
INSERT INTO daily_summary VALUES ('stock','CBK','us','2017-11-01',1.18,1.21,1.17,1.19,59741,'0');
INSERT INTO daily_summary VALUES ('stock','CBL','us','2017-11-01',7.86,8.0332,7.74,7.8,2789420,'0');
INSERT INTO daily_summary VALUES ('stock','CBLI','us','2017-11-01',3,3,2.9,2.95,18004,'0');
INSERT INTO daily_summary VALUES ('stock','CBL_D','us','2017-11-01',24.728,24.74,24.6648,24.74,21019,'0');
INSERT INTO daily_summary VALUES ('stock','CBL_E','us','2017-11-01',24.8232,24.9,24.8232,24.8508,3722,'0');
INSERT INTO daily_summary VALUES ('stock','CBM','us','2017-11-01',43.4,44,42.75,42.9,593513,'0');
INSERT INTO daily_summary VALUES ('stock','CBMG','us','2017-11-01',10.7,11.05,10.7,10.75,79279,'0');
INSERT INTO daily_summary VALUES ('stock','CBMX','us','2017-11-01',7.15,7.25,6.9145,7.1,53904,'0');
INSERT INTO daily_summary VALUES ('etf','CBND','us','2017-11-01',32.3583,32.3583,32.33,32.33,410,'0');
INSERT INTO daily_summary VALUES ('stock','CBOE','us','2017-11-01',113.69,113.81,112.6,112.63,513835,'0');
INSERT INTO daily_summary VALUES ('etf','CBON','us','2017-11-01',22.7084,22.7084,22.7084,22.7084,1000,'0');
INSERT INTO daily_summary VALUES ('stock','CBPO','us','2017-11-01',78.67,78.76,76.56,76.99,224038,'0');
INSERT INTO daily_summary VALUES ('stock','CBPX','us','2017-11-01',26.95,26.95,26.65,26.85,239546,'0');
INSERT INTO daily_summary VALUES ('stock','CBRL','us','2017-11-01',156.16,157.67,155.28,156.44,231096,'0');
INSERT INTO daily_summary VALUES ('stock','CBS','us','2017-11-01',56.21,56.71,55.74,56.19,7009900,'0');
INSERT INTO daily_summary VALUES ('stock','CBSH','us','2017-11-01',58.43,58.8342,57.51,57.78,451351,'0');
INSERT INTO daily_summary VALUES ('stock','CBSHP','us','2017-11-01',26.135,26.539,26.125,26.46,3843,'0');
INSERT INTO daily_summary VALUES ('stock','CBT','us','2017-11-01',61.19,64.21,61.19,63.29,671633,'0');
INSERT INTO daily_summary VALUES ('stock','CBU','us','2017-11-01',55.64,55.815,53.85,54.44,353483,'0');
INSERT INTO daily_summary VALUES ('stock','CBX','us','2017-11-01',1.05,1.05,1.05,1.05,0,'0');
INSERT INTO daily_summary VALUES ('stock','CBZ','us','2017-11-01',17.05,17.05,16.7,16.8,64869,'0');
INSERT INTO daily_summary VALUES ('stock','CC','us','2017-11-01',57.25,57.2992,56.18,56.97,1559860,'0');
INSERT INTO daily_summary VALUES ('stock','CCA','us','2017-11-01',11.72,11.76,11.72,11.731,8225,'0');
INSERT INTO daily_summary VALUES ('stock','CCBG','us','2017-11-01',24.89,24.99,24.025,24.2,24384,'0');
INSERT INTO daily_summary VALUES ('stock','CCC','us','2017-11-01',21.7,21.9,21.6,21.85,1972080,'0');
INSERT INTO daily_summary VALUES ('stock','CCCL','us','2017-11-01',1.37,1.44,1.33,1.37,20151,'0');
INSERT INTO daily_summary VALUES ('stock','CCCR','us','2017-11-01',2.87,3.12,2.84,3,90819,'0');
INSERT INTO daily_summary VALUES ('stock','CCD','us','2017-11-01',20.5,20.55,20.363,20.51,68840,'0');
INSERT INTO daily_summary VALUES ('stock','CCE','us','2017-11-01',40.85,40.94,40.4,40.4,494571,'0');
INSERT INTO daily_summary VALUES ('stock','CCF','us','2017-11-01',118.72,120.61,115.25,120.21,27699,'0');
INSERT INTO daily_summary VALUES ('stock','CCI','us','2017-11-01',107.68,107.74,104.78,105.14,2467180,'0');
INSERT INTO daily_summary VALUES ('stock','CCIH','us','2017-11-01',1.07,1.15,1.07,1.09,428242,'0');
INSERT INTO daily_summary VALUES ('stock','CCI_A','us','2017-11-01',1094.93,1095,1088.19,1088.93,20953,'0');
INSERT INTO daily_summary VALUES ('stock','CCJ','us','2017-11-01',8.24,8.41,8.18,8.41,3004580,'0');
INSERT INTO daily_summary VALUES ('stock','CCK','us','2017-11-01',60.36,61.01,60.27,60.57,922267,'0');
INSERT INTO daily_summary VALUES ('stock','CCL','us','2017-11-01',66.58,66.66,64.145,64.57,5136420,'0');
INSERT INTO daily_summary VALUES ('stock','CCLP','us','2017-11-01',4.97,5.0999,4.661,4.8,150897,'0');
INSERT INTO daily_summary VALUES ('stock','CCM','us','2017-11-01',3.72,3.87,3.712,3.78,63844,'0');
INSERT INTO daily_summary VALUES ('stock','CCMP','us','2017-11-01',97.36,98.33,94,95.71,168385,'0');
INSERT INTO daily_summary VALUES ('stock','CCNE','us','2017-11-01',29.09,29.32,28.14,28.5,21725,'0');
INSERT INTO daily_summary VALUES ('stock','CCO','us','2017-11-01',3.75,3.85,3.6,3.8,144195,'0');
INSERT INTO daily_summary VALUES ('stock','CCOI','us','2017-11-01',54.3,54.85,53.225,53.55,681823,'0');
INSERT INTO daily_summary VALUES ('stock','CCOR','us','2017-11-01',25.5,25.5,25.33,25.33,31233,'0');
INSERT INTO daily_summary VALUES ('stock','CCRC','us','2017-11-01',12.6,12.6,12.18,12.37,145623,'0');
INSERT INTO daily_summary VALUES ('stock','CCRN','us','2017-11-01',13.79,13.915,13.38,13.63,277907,'0');
INSERT INTO daily_summary VALUES ('stock','CCS','us','2017-11-01',28.85,29.1,28.25,29,257363,'0');
INSERT INTO daily_summary VALUES ('stock','CCU','us','2017-11-01',28.72,28.72,27.62,27.73,122349,'0');
INSERT INTO daily_summary VALUES ('stock','CCUR','us','2017-11-01',6.13,6.26,6.1278,6.26,76261,'0');
INSERT INTO daily_summary VALUES ('stock','CCV','us','2017-11-01',25.49,25.49,25.35,25.409,5928,'0');
INSERT INTO daily_summary VALUES ('stock','CCXI','us','2017-11-01',6.53,6.68,6.42,6.65,124475,'0');
INSERT INTO daily_summary VALUES ('stock','CDC','us','2017-11-01',45.07,45.07,44.861,44.91,21885,'0');
INSERT INTO daily_summary VALUES ('stock','CDE','us','2017-11-01',7.69,7.74,7.52,7.53,2904160,'0');
INSERT INTO daily_summary VALUES ('stock','CDEV','us','2017-11-01',19.57,20,19.31,19.45,2548390,'0');
INSERT INTO daily_summary VALUES ('stock','CDK','us','2017-11-01',64.09,64.36,62.82,62.99,1598360,'0');
INSERT INTO daily_summary VALUES ('stock','CDL','us','2017-11-01',43.88,43.8909,43.719,43.72,8741,'0');
INSERT INTO daily_summary VALUES ('stock','CDNA','us','2017-11-01',5.85,5.85,5.6564,5.79,114641,'0');
INSERT INTO daily_summary VALUES ('stock','CDNS','us','2017-11-01',43.47,43.62,42.96,43.01,2123760,'0');
INSERT INTO daily_summary VALUES ('stock','CDOR','us','2017-11-01',10.1032,10.44,9.9,10.44,21931,'0');
INSERT INTO daily_summary VALUES ('stock','CDR','us','2017-11-01',5.4132,5.5817,5.3933,5.5718,724272,'0');
INSERT INTO daily_summary VALUES ('stock','CDR_B','us','2017-11-01',25.52,25.55,25.4835,25.55,7262,'0');
INSERT INTO daily_summary VALUES ('stock','CDR_C','us','2017-11-01',25.3,25.3,25.18,25.2999,3534,'0');
INSERT INTO daily_summary VALUES ('stock','CDTI','us','2017-11-01',1.7,1.7,1.63,1.65,52120,'0');
INSERT INTO daily_summary VALUES ('stock','CDTX','us','2017-11-01',7.75,7.8,7.35,7.4,65347,'0');
INSERT INTO daily_summary VALUES ('stock','CDW','us','2017-11-01',68.82,70.71,67.8504,68.92,1032650,'0');
INSERT INTO daily_summary VALUES ('stock','CDXC','us','2017-11-01',4.1,4.14,4,4.02,197622,'0');
INSERT INTO daily_summary VALUES ('stock','CDXS','us','2017-11-01',6.15,6.2,6.05,6.15,322815,'0');
INSERT INTO daily_summary VALUES ('stock','CDZI','us','2017-11-01',13.15,13.3,12.3,12.45,71583,'0');
INSERT INTO daily_summary VALUES ('stock','CE','us','2017-11-01',105.1,105.85,104.65,105.36,843326,'0');
INSERT INTO daily_summary VALUES ('stock','CEA','us','2017-11-01',26.34,26.5,26.34,26.42,11900,'0');
INSERT INTO daily_summary VALUES ('stock','CECE','us','2017-11-01',8.82,8.84,8.51,8.52,74257,'0');
INSERT INTO daily_summary VALUES ('stock','CECO','us','2017-11-01',10.76,10.86,10.4,10.5,126381,'0');
INSERT INTO daily_summary VALUES ('stock','CEE','us','2017-11-01',24.7,24.8,24.65,24.74,18384,'0');
INSERT INTO daily_summary VALUES ('stock','CEF','us','2017-11-01',13.21,13.3,13.17,13.24,812247,'0');
INSERT INTO daily_summary VALUES ('etf','CEFL','us','2017-11-01',18.339,18.399,18.25,18.25,89886,'0');
INSERT INTO daily_summary VALUES ('stock','CEFS','us','2017-11-01',21.47,21.48,21.3962,21.4599,10113,'0');
INSERT INTO daily_summary VALUES ('stock','CEI','us','2017-11-01',0.1562,0.1648,0.1501,0.1579,953794,'0');
INSERT INTO daily_summary VALUES ('stock','CEL','us','2017-11-01',9.58,9.58,9.4,9.5,3730,'0');
INSERT INTO daily_summary VALUES ('stock','CELC','us','2017-11-01',17.75,18.86,17.75,18.58,35375,'0');
INSERT INTO daily_summary VALUES ('stock','CELG','us','2017-11-01',101.89,102.3,100.04,100.53,8320620,'0');
INSERT INTO daily_summary VALUES ('stock','CELGZ','us','2017-11-01',1.16,1.311,1.0553,1.14,27939,'0');
INSERT INTO daily_summary VALUES ('stock','CELH','us','2017-11-01',5.2788,5.2788,4.81,4.95,58193,'0');
INSERT INTO daily_summary VALUES ('stock','CELP','us','2017-11-01',7.2314,7.28,7.1396,7.1925,17290,'0');
INSERT INTO daily_summary VALUES ('stock','CEM','us','2017-11-01',13.51,13.66,13.49,13.6,420779,'0');
INSERT INTO daily_summary VALUES ('stock','CEMB','us','2017-11-01',51.18,51.36,51.14,51.2292,17495,'0');
INSERT INTO daily_summary VALUES ('stock','CEMI','us','2017-11-01',5.969,6.04,5.969,5.9994,3498,'0');
INSERT INTO daily_summary VALUES ('stock','CEN','us','2017-11-01',9.687,9.831,9.598,9.796,323750,'0');
INSERT INTO daily_summary VALUES ('stock','CENT','us','2017-11-01',38.53,38.74,37.4,37.75,59763,'0');
INSERT INTO daily_summary VALUES ('stock','CENTA','us','2017-11-01',37.25,37.56,36.21,36.65,159633,'0');
INSERT INTO daily_summary VALUES ('stock','CENX','us','2017-11-01',14.39,14.75,13.62,13.71,2393220,'0');
INSERT INTO daily_summary VALUES ('stock','CEO','us','2017-11-01',136.46,136.89,135.69,136.11,195983,'0');
INSERT INTO daily_summary VALUES ('stock','CEQP','us','2017-11-01',24.643,25.129,24.545,24.594,258449,'0');
INSERT INTO daily_summary VALUES ('stock','CERC','us','2017-11-01',1.13,1.1301,1.0823,1.0995,37046,'0');
INSERT INTO daily_summary VALUES ('stock','CERCW','us','2017-11-01',0.049,0.049,0.049,0.049,5000,'0');
INSERT INTO daily_summary VALUES ('stock','CERN','us','2017-11-01',67.73,67.9,64.87,65.49,3290100,'0');
INSERT INTO daily_summary VALUES ('stock','CERS','us','2017-11-01',2.88,2.94,2.75,2.76,688375,'0');
INSERT INTO daily_summary VALUES ('stock','CET','us','2017-11-01',26.107,26.194,25.985,25.991,7147,'0');
INSERT INTO daily_summary VALUES ('stock','CETV','us','2017-11-01',4.65,4.65,4.45,4.55,275101,'0');
INSERT INTO daily_summary VALUES ('stock','CETX','us','2017-11-01',2.85,2.8552,2.77,2.79,103265,'0');
INSERT INTO daily_summary VALUES ('stock','CETXP','us','2017-11-01',6.4388,6.5023,6.4101,6.4823,2250,'0');
INSERT INTO daily_summary VALUES ('stock','CEV','us','2017-11-01',12.201,12.24,12.181,12.181,16004,'0');
INSERT INTO daily_summary VALUES ('stock','CEVA','us','2017-11-01',48.3,48.3,41.85,43.2,709548,'0');
INSERT INTO daily_summary VALUES ('etf','CEW','us','2017-11-01',18.8,18.86,18.79,18.79,8423,'0');
INSERT INTO daily_summary VALUES ('stock','CEY','us','2017-11-01',25.085,25.16,25.07,25.07,8198,'0');
INSERT INTO daily_summary VALUES ('stock','CEZ','us','2017-11-01',29.2953,29.44,29.2953,29.3724,3432,'0');
INSERT INTO daily_summary VALUES ('stock','CF','us','2017-11-01',38.25,38.5,37.575,38.04,4431830,'0');
INSERT INTO daily_summary VALUES ('stock','CFA','us','2017-11-01',46.69,46.8557,46.58,46.59,16615,'0');
INSERT INTO daily_summary VALUES ('stock','CFBI','us','2017-11-01',13.25,13.29,13.218,13.28,508,'0');
INSERT INTO daily_summary VALUES ('stock','CFBK','us','2017-11-01',2.45,2.4598,2.4,2.42,3916,'0');
INSERT INTO daily_summary VALUES ('stock','CFCO','us','2017-11-01',10.72,11.1,10.68,10.97,4522090,'0');
INSERT INTO daily_summary VALUES ('stock','CFCOU','us','2017-11-01',11.76,11.76,11.75,11.75,1000,'0');
INSERT INTO daily_summary VALUES ('stock','CFCOW','us','2017-11-01',1.8,1.95,1.8,1.87,142511,'0');
INSERT INTO daily_summary VALUES ('stock','CFC_B','us','2017-11-01',25.7,25.71,25.66,25.66,118439,'0');
INSERT INTO daily_summary VALUES ('stock','CFFI','us','2017-11-01',57.95,58.25,57.15,58.15,13179,'0');
INSERT INTO daily_summary VALUES ('stock','CFFN','us','2017-11-01',13.789,13.808,13.55,13.58,613815,'0');
INSERT INTO daily_summary VALUES ('stock','CFG','us','2017-11-01',38.21,38.98,38.12,38.37,5282580,'0');
INSERT INTO daily_summary VALUES ('stock','CFMS','us','2017-11-01',3.57,3.62,3.31,3.35,152706,'0');
INSERT INTO daily_summary VALUES ('stock','CFNB','us','2017-11-01',15.25,15.275,15.15,15.2,3253,'0');
INSERT INTO daily_summary VALUES ('stock','CFO','us','2017-11-01',46.85,46.85,46.551,46.605,43389,'0');
INSERT INTO daily_summary VALUES ('stock','CFR','us','2017-11-01',99.63,100.365,98.19,98.74,378635,'0');
INSERT INTO daily_summary VALUES ('stock','CFRX','us','2017-11-01',1.006,1.09,1,1.06,32159,'0');
INSERT INTO daily_summary VALUES ('stock','CFR_A','us','2017-11-01',25.26,25.39,25.26,25.34,1340,'0');
INSERT INTO daily_summary VALUES ('stock','CFX','us','2017-11-01',41.97,42.25,41.51,41.56,385704,'0');
INSERT INTO daily_summary VALUES ('stock','CG','us','2017-11-01',21.719,22.304,21.427,21.524,1171200,'0');
INSERT INTO daily_summary VALUES ('stock','CGA','us','2017-11-01',1.24,1.25,1.23,1.24,54174,'0');
INSERT INTO daily_summary VALUES ('stock','CGBD','us','2017-11-01',18.6,18.6,18.435,18.49,40100,'0');
INSERT INTO daily_summary VALUES ('stock','CGEN','us','2017-11-01',3.05,3.05,2.9,2.95,112254,'0');
INSERT INTO daily_summary VALUES ('stock','CGG','us','2017-11-01',4.6952,4.6952,4.57,4.57,2513,'0');
INSERT INTO daily_summary VALUES ('stock','CGI','us','2017-11-01',7.4,7.45,7.15,7.25,229838,'0');
INSERT INTO daily_summary VALUES ('stock','CGIX','us','2017-11-01',3.05,3.2,3.05,3.15,72302,'0');
INSERT INTO daily_summary VALUES ('stock','CGNT','us','2017-11-01',2.67,2.94,2.67,2.94,24115,'0');
INSERT INTO daily_summary VALUES ('stock','CGNX','us','2017-11-01',125,126.43,122.62,124.96,1218140,'0');
INSERT INTO daily_summary VALUES ('stock','CGO','us','2017-11-01',13.971,14.12,13.931,14.01,70208,'0');
INSERT INTO daily_summary VALUES ('etf','CGW','us','2017-11-01',35.31,35.31,35.006,35.03,68446,'0');
INSERT INTO daily_summary VALUES ('stock','CH','us','2017-11-01',8.99,9,8.91,8.96,33272,'0');
INSERT INTO daily_summary VALUES ('stock','CHA','us','2017-11-01',50.38,50.44,50.09,50.28,43437,'0');
INSERT INTO daily_summary VALUES ('stock','CHAD','us','2017-11-01',32.08,32.08,31.92,31.97,2903,'0');
INSERT INTO daily_summary VALUES ('etf','CHAU','us','2017-11-01',28.64,28.71,28.6,28.67,49791,'0');
INSERT INTO daily_summary VALUES ('stock','CHCO','us','2017-11-01',70.99,71.8015,69.29,69.83,86084,'0');
INSERT INTO daily_summary VALUES ('stock','CHCT','us','2017-11-01',27.54,27.92,27.45,27.65,104750,'0');
INSERT INTO daily_summary VALUES ('stock','CHD','us','2017-11-01',45.39,46.475,45.28,46.18,4003730,'0');
INSERT INTO daily_summary VALUES ('stock','CHDN','us','2017-11-01',209.225,209.225,205.9,208.05,51872,'0');
INSERT INTO daily_summary VALUES ('stock','CHE','us','2017-11-01',224.09,224.59,219.08,224.18,129302,'0');
INSERT INTO daily_summary VALUES ('stock','CHEF','us','2017-11-01',20.15,20.9,20,20.8,127179,'0');
INSERT INTO daily_summary VALUES ('stock','CHEK','us','2017-11-01',1.22,1.28,1.17,1.23,103270,'0');
INSERT INTO daily_summary VALUES ('stock','CHEKW','us','2017-11-01',0.4,0.4,0.4,0.4,200,'0');
INSERT INTO daily_summary VALUES ('etf','CHEP','us','2017-11-01',24.9542,25.22,24.9542,24.97,1001,'0');
INSERT INTO daily_summary VALUES ('stock','CHFC','us','2017-11-01',53.01,53.31,51.97,52.36,220865,'0');
INSERT INTO daily_summary VALUES ('stock','CHFN','us','2017-11-01',19.273,19.731,18.456,18.775,21556,'0');
INSERT INTO daily_summary VALUES ('stock','CHFS','us','2017-11-01',6.77,6.77,6.605,6.75,9073,'0');
INSERT INTO daily_summary VALUES ('stock','CHGG','us','2017-11-01',15.64,15.66,14.47,14.61,3110700,'0');
INSERT INTO daily_summary VALUES ('stock','CHH','us','2017-11-01',70.35,70.6,69.55,69.75,161589,'0');
INSERT INTO daily_summary VALUES ('stock','CHI','us','2017-11-01',11.54,11.629,11.501,11.52,226829,'0');
INSERT INTO daily_summary VALUES ('etf','CHIE','us','2017-11-01',11.6,11.6,11.55,11.55,4900,'0');
INSERT INTO daily_summary VALUES ('etf','CHII','us','2017-11-01',16.1,16.1,16.1,16.1,112,'0');
INSERT INTO daily_summary VALUES ('etf','CHIM','us','2017-11-01',21.77,21.78,21.55,21.78,3412,'0');
INSERT INTO daily_summary VALUES ('etf','CHIQ','us','2017-11-01',17.62,17.66,17.51,17.56,73797,'0');
INSERT INTO daily_summary VALUES ('etf','CHIX','us','2017-11-01',17.87,17.9126,17.84,17.85,19872,'0');
INSERT INTO daily_summary VALUES ('stock','CHK','us','2017-11-01',3.98,4.07,3.96,3.96,31422500,'0');
INSERT INTO daily_summary VALUES ('stock','CHKE','us','2017-11-01',2.2,2.225,2,2,199754,'0');
INSERT INTO daily_summary VALUES ('stock','CHKP','us','2017-11-01',107,110,102.25,103.01,9141180,'0');
INSERT INTO daily_summary VALUES ('stock','CHKR','us','2017-11-01',2.15,2.2,2.1,2.2,176402,'0');
INSERT INTO daily_summary VALUES ('stock','CHK_D','us','2017-11-01',51.69,51.7399,51,51.4,18110,'0');
INSERT INTO daily_summary VALUES ('stock','CHL','us','2017-11-01',50.61,50.8924,50.59,50.77,460978,'0');
INSERT INTO daily_summary VALUES ('stock','CHMA','us','2017-11-01',2.1,2.1,1.95,2.1,44399,'0');
INSERT INTO daily_summary VALUES ('stock','CHMG','us','2017-11-01',47.14,48.23,47.14,47.36,4635,'0');
INSERT INTO daily_summary VALUES ('stock','CHMI','us','2017-11-01',18.33,18.37,18.1844,18.27,115900,'0');
INSERT INTO daily_summary VALUES ('stock','CHMI_A','us','2017-11-01',25.6203,25.72,25.46,25.6236,5150,'0');
INSERT INTO daily_summary VALUES ('stock','CHN','us','2017-11-01',21.16,21.31,21.12,21.2,15973,'0');
INSERT INTO daily_summary VALUES ('stock','CHNR','us','2017-11-01',2.3,2.38,2.17,2.18,44379,'0');
INSERT INTO daily_summary VALUES ('etf','CHOC','us','2017-11-01',31.25,31.25,31.1201,31.1808,912,'0');
INSERT INTO daily_summary VALUES ('stock','CHRS','us','2017-11-01',11.35,11.6,11.075,11.1,477932,'0');
INSERT INTO daily_summary VALUES ('stock','CHRW','us','2017-11-01',80.9,80.94,77.83,79.33,3029710,'0');
INSERT INTO daily_summary VALUES ('stock','CHS','us','2017-11-01',7.99,8.02,7.59,7.81,2037380,'0');
INSERT INTO daily_summary VALUES ('stock','CHSCL','us','2017-11-01',28.82,28.86,28.5797,28.6448,9702,'0');
INSERT INTO daily_summary VALUES ('stock','CHSCM','us','2017-11-01',27.31,27.33,27.21,27.26,6717,'0');
INSERT INTO daily_summary VALUES ('stock','CHSCN','us','2017-11-01',27.77,27.84,27.67,27.73,24958,'0');
INSERT INTO daily_summary VALUES ('stock','CHSCO','us','2017-11-01',28.99,29.05,28.94,29.0149,9646,'0');
INSERT INTO daily_summary VALUES ('stock','CHSCP','us','2017-11-01',30,30.089,29.91,29.96,14879,'0');
INSERT INTO daily_summary VALUES ('stock','CHSP','us','2017-11-01',27.93,28.24,27.72,27.85,299391,'0');
INSERT INTO daily_summary VALUES ('stock','CHT','us','2017-11-01',33.83,33.92,33.71,33.77,223934,'0');
INSERT INTO daily_summary VALUES ('stock','CHTR','us','2017-11-01',335.62,340.74,335.465,338.22,2324200,'0');
INSERT INTO daily_summary VALUES ('stock','CHU','us','2017-11-01',14.62,14.64,14.54,14.54,263502,'0');
INSERT INTO daily_summary VALUES ('stock','CHUBA','us','2017-11-01',22.5,22.83,21.8,22.21,70769,'0');
INSERT INTO daily_summary VALUES ('stock','CHUBK','us','2017-11-01',21.52,21.52,20.79,21.14,106088,'0');
INSERT INTO daily_summary VALUES ('stock','CHUY','us','2017-11-01',22.55,22.65,21.5,21.75,147584,'0');
INSERT INTO daily_summary VALUES ('stock','CHW','us','2017-11-01',9.1008,9.1108,9.0687,9.076,165784,'0');
INSERT INTO daily_summary VALUES ('stock','CHY','us','2017-11-01',11.899,11.959,11.88,11.909,285117,'0');
INSERT INTO daily_summary VALUES ('stock','CI','us','2017-11-01',198.2,198.58,196.75,197.84,1678900,'0');
INSERT INTO daily_summary VALUES ('stock','CIA','us','2017-11-01',7.66,7.66,7.5,7.52,74121,'0');
INSERT INTO daily_summary VALUES ('stock','CIB','us','2017-11-01',37.96,38.41,37.45,37.74,980139,'0');
INSERT INTO daily_summary VALUES ('stock','CIBR','us','2017-11-01',22.43,22.43,22.111,22.1902,238098,'0');
INSERT INTO daily_summary VALUES ('stock','CID','us','2017-11-01',35.909,35.95,35.829,35.86,3229,'0');
INSERT INTO daily_summary VALUES ('stock','CIDM','us','2017-11-01',1.24,1.34,1.18,1.32,226683,'0');
INSERT INTO daily_summary VALUES ('stock','CIE','us','2017-11-01',0.98,1.04,0.94,0.9999,494259,'0');
INSERT INTO daily_summary VALUES ('stock','CIEN','us','2017-11-01',21.51,21.51,20.86,20.88,2099320,'0');
INSERT INTO daily_summary VALUES ('stock','CIF','us','2017-11-01',2.87,2.9,2.86,2.87,58545,'0');
INSERT INTO daily_summary VALUES ('stock','CIFS','us','2017-11-01',41.35,41.36,30,30.2,664089,'0');
INSERT INTO daily_summary VALUES ('stock','CIG','us','2017-11-01',2.34,2.38,2.26,2.26,3161690,'0');
INSERT INTO daily_summary VALUES ('stock','CIGI','us','2017-11-01',59.2,60.5,58.75,60.05,45815,'0');
INSERT INTO daily_summary VALUES ('stock','CII','us','2017-11-01',16,16.06,15.95,16,110979,'0');
INSERT INTO daily_summary VALUES ('stock','CIK','us','2017-11-01',3.3199,3.3298,3.3,3.3199,194940,'0');
INSERT INTO daily_summary VALUES ('stock','CIL','us','2017-11-01',40.108,40.108,40.0316,40.0316,871,'0');
INSERT INTO daily_summary VALUES ('stock','CIM','us','2017-11-01',18.3,18.415,18.29,18.38,1107140,'0');
INSERT INTO daily_summary VALUES ('stock','CIM_A','us','2017-11-01',25.8,26.039,25.8,25.962,6046,'0');
INSERT INTO daily_summary VALUES ('stock','CIM_B','us','2017-11-01',25.88,26,25.83,25.9,40198,'0');
INSERT INTO daily_summary VALUES ('stock','CINF','us','2017-11-01',70.26,71.12,70.21,70.71,564987,'0');
INSERT INTO daily_summary VALUES ('stock','CINR','us','2017-11-01',25.147,25.147,24.542,24.737,2970,'0');
INSERT INTO daily_summary VALUES ('stock','CIO','us','2017-11-01',13.08,13.2,12.96,13.03,72971,'0');
INSERT INTO daily_summary VALUES ('stock','CIO_A','us','2017-11-01',25.59,25.59,25.52,25.52,4374,'0');
INSERT INTO daily_summary VALUES ('stock','CIR','us','2017-11-01',44.25,44.7906,42.74,42.76,131669,'0');
INSERT INTO daily_summary VALUES ('stock','CISN','us','2017-11-01',13.31,13.3989,12.93,12.95,92106,'0');
INSERT INTO daily_summary VALUES ('stock','CIT','us','2017-11-01',46.755,46.869,46.017,46.117,557774,'0');
INSERT INTO daily_summary VALUES ('etf','CIU','us','2017-11-01',109.66,109.799,109.66,109.71,240646,'0');
INSERT INTO daily_summary VALUES ('stock','CIVB','us','2017-11-01',22.8,22.8,22.2,22.38,17806,'0');
INSERT INTO daily_summary VALUES ('stock','CIVI','us','2017-11-01',18.6,19,18.1,18.45,36070,'0');
INSERT INTO daily_summary VALUES ('stock','CIX','us','2017-11-01',14.85,14.85,14.4479,14.4479,523,'0');
INSERT INTO daily_summary VALUES ('stock','CIZ','us','2017-11-01',34.7,34.7491,34.65,34.65,9987,'0');
INSERT INTO daily_summary VALUES ('stock','CIZN','us','2017-11-01',23.55,24.005,23.55,24.005,370,'0');
INSERT INTO daily_summary VALUES ('stock','CJ','us','2017-11-01',29.03,29.32,27.77,27.92,649076,'0');
INSERT INTO daily_summary VALUES ('stock','CJJD','us','2017-11-01',1.5,1.53,1.41,1.47,56258,'0');
INSERT INTO daily_summary VALUES ('etf','CJNK','us','2017-11-01',26.7,26.828,26.65,26.67,17120,'0');
INSERT INTO daily_summary VALUES ('stock','CKH','us','2017-11-01',47.76,48.15,46.42,46.94,108240,'0');
INSERT INTO daily_summary VALUES ('stock','CKPT','us','2017-11-01',6.2,6.32,6.2,6.32,5802,'0');
INSERT INTO daily_summary VALUES ('stock','CL','us','2017-11-01',25.33,25.345,25.33,25.3412,4948,'0');
INSERT INTO daily_summary VALUES ('stock','CLAR','us','2017-11-01',7.3,7.3,7.1,7.1,16186,'0');
INSERT INTO daily_summary VALUES ('stock','CLB','us','2017-11-01',100.27,101.67,99.6,100.92,704972,'0');
INSERT INTO daily_summary VALUES ('stock','CLBS','us','2017-11-01',3.07,3.18,2.8,2.9,108548,'0');
INSERT INTO daily_summary VALUES ('stock','CLCT','us','2017-11-01',24.84,24.9,24.21,24.5,27500,'0');
INSERT INTO daily_summary VALUES ('stock','CLD','us','2017-11-01',4.36,4.42,4.21,4.3,1282480,'0');
INSERT INTO daily_summary VALUES ('stock','CLDC','us','2017-11-01',4.28,4.28,4.11,4.11,4664,'0');
INSERT INTO daily_summary VALUES ('stock','CLDR','us','2017-11-01',15.31,15.4096,15.08,15.26,1240660,'0');
INSERT INTO daily_summary VALUES ('stock','CLDT','us','2017-11-01',21.83,22.6,21.83,22.25,408773,'0');
INSERT INTO daily_summary VALUES ('stock','CLDX','us','2017-11-01',2.46,2.475,2.34,2.35,1330390,'0');
INSERT INTO daily_summary VALUES ('stock','CLF','us','2017-11-01',6.19,6.35,6.04,6.21,13930000,'0');
INSERT INTO daily_summary VALUES ('stock','CLFD','us','2017-11-01',13.9,13.95,13.4,13.85,48691,'0');
INSERT INTO daily_summary VALUES ('stock','CLGX','us','2017-11-01',46.62,46.755,46.18,46.47,588847,'0');
INSERT INTO daily_summary VALUES ('stock','CLH','us','2017-11-01',55.62,56.13,51.63,52.28,1043690,'0');
INSERT INTO daily_summary VALUES ('stock','CLI','us','2017-11-01',22.84,23.13,22.64,23.08,525956,'0');
INSERT INTO daily_summary VALUES ('stock','CLIR','us','2017-11-01',3.15,3.3,3.15,3.15,12612,'0');
INSERT INTO daily_summary VALUES ('stock','CLIRW','us','2017-11-01',0.8,0.8,0.8,0.8,1000,'0');
INSERT INTO daily_summary VALUES ('stock','CLLS','us','2017-11-01',34.63,34.98,29.03,29.18,508667,'0');
INSERT INTO daily_summary VALUES ('stock','CLM','us','2017-11-01',15,15.05,14.99,15.02,233455,'0');
INSERT INTO daily_summary VALUES ('stock','CLMT','us','2017-11-01',9,9.045,8.605,8.95,488357,'0');
INSERT INTO daily_summary VALUES ('stock','CLNE','us','2017-11-01',2.37,2.44,2.35,2.41,930360,'0');
INSERT INTO daily_summary VALUES ('stock','CLNS','us','2017-11-01',12.26,12.52,12.26,12.34,2032630,'0');
INSERT INTO daily_summary VALUES ('stock','CLNS_B','us','2017-11-01',25.6261,25.7,25.6192,25.65,9972,'0');
INSERT INTO daily_summary VALUES ('stock','CLNS_D','us','2017-11-01',26.0247,26.0489,26,26.0489,5097,'0');
INSERT INTO daily_summary VALUES ('stock','CLNS_E','us','2017-11-01',27.0681,27.0681,26.95,26.95,2225,'0');
INSERT INTO daily_summary VALUES ('stock','CLNS_G','us','2017-11-01',25.8524,25.8524,25.8524,25.8524,1028,'0');
INSERT INTO daily_summary VALUES ('stock','CLNS_H','us','2017-11-01',25.746,25.746,25.5799,25.679,6953,'0');
INSERT INTO daily_summary VALUES ('stock','CLNS_I','us','2017-11-01',25.5568,25.5899,25.44,25.5,62006,'0');
INSERT INTO daily_summary VALUES ('stock','CLNS_J','us','2017-11-01',25.57,25.57,25.32,25.4,44221,'0');
INSERT INTO daily_summary VALUES ('stock','CLNT','us','2017-11-01',4.1316,4.47,4.1101,4.23,45570,'0');
INSERT INTO daily_summary VALUES ('stock','CLPR','us','2017-11-01',10.855,11.024,10.726,10.756,33032,'0');
INSERT INTO daily_summary VALUES ('stock','CLR','us','2017-11-01',41.24,42.28,41.16,41.74,2397280,'0');
INSERT INTO daily_summary VALUES ('stock','CLRB','us','2017-11-01',1.51,1.5413,1.48,1.5,101442,'0');
INSERT INTO daily_summary VALUES ('stock','CLRBW','us','2017-11-01',0.15,0.15,0.147,0.147,791,'0');
INSERT INTO daily_summary VALUES ('stock','CLRBZ','us','2017-11-01',0.69,0.7,0.69,0.7,19400,'0');
INSERT INTO daily_summary VALUES ('stock','CLRO','us','2017-11-01',7.35,7.35,7.15,7.15,14256,'0');
INSERT INTO daily_summary VALUES ('stock','CLS','us','2017-11-01',10.03,10.1,9.79,10.1,1263630,'0');
INSERT INTO daily_summary VALUES ('stock','CLSD','us','2017-11-01',7.23,7.23,6.86,7.01,152407,'0');
INSERT INTO daily_summary VALUES ('stock','CLSN','us','2017-11-01',2.17,2.2,2.03,2.13,959456,'0');
INSERT INTO daily_summary VALUES ('stock','CLTL','us','2017-11-01',105.945,105.96,105.945,105.945,908,'0');
INSERT INTO daily_summary VALUES ('stock','CLUB','us','2017-11-01',6,6.05,5.8,5.85,156485,'0');
INSERT INTO daily_summary VALUES ('stock','CLVS','us','2017-11-01',76.06,76.1765,71.81,72.86,1410290,'0');
INSERT INTO daily_summary VALUES ('stock','CLW','us','2017-11-01',46.25,47.1,45.65,46.1,70104,'0');
INSERT INTO daily_summary VALUES ('stock','CLWT','us','2017-11-01',3.8,3.9,3.55,3.85,3663,'0');
INSERT INTO daily_summary VALUES ('stock','CLX','us','2017-11-01',128.85,131.59,126.45,128.02,2434660,'0');
INSERT INTO daily_summary VALUES ('stock','CLXT','us','2017-11-01',20.84,24.06,20.84,23.15,67227,'0');
INSERT INTO daily_summary VALUES ('etf','CLY','us','2017-11-01',62,62.2822,62,62.15,24199,'0');
INSERT INTO daily_summary VALUES ('stock','CLYH','us','2017-11-01',25.92,25.92,25.92,25.92,201,'0');
INSERT INTO daily_summary VALUES ('stock','CM','us','2017-11-01',88.27,88.59,88.06,88.28,300271,'0');
INSERT INTO daily_summary VALUES ('stock','CMA','us','2017-11-01',79.08,79.78,78.39,78.45,1161240,'0');
INSERT INTO daily_summary VALUES ('etf','CMBS','us','2017-11-01',51.518,51.65,51.518,51.55,7106,'0');
INSERT INTO daily_summary VALUES ('stock','CMC','us','2017-11-01',19.714,20.29,19.456,19.624,1911050,'0');
INSERT INTO daily_summary VALUES ('stock','CMCL','us','2017-11-01',5.3521,5.3705,5.3521,5.3705,681,'0');
INSERT INTO daily_summary VALUES ('stock','CMCM','us','2017-11-01',8.99,9.02,8.735,9,1331570,'0');
INSERT INTO daily_summary VALUES ('stock','CMCO','us','2017-11-01',39.849,39.849,36.802,37.071,268834,'0');
INSERT INTO daily_summary VALUES ('stock','CMCSA','us','2017-11-01',36.25,36.42,35.94,36.08,18199800,'0');
INSERT INTO daily_summary VALUES ('stock','CMCT','us','2017-11-01',16.7,17.35,16.65,17.35,5777,'0');
INSERT INTO daily_summary VALUES ('stock','CMD','us','2017-11-01',98.58,99.19,95.88,96.32,133377,'0');
INSERT INTO daily_summary VALUES ('etf','CMDT','us','2017-11-01',36.3954,36.57,36.3954,36.4856,4007,'0');
INSERT INTO daily_summary VALUES ('stock','CME','us','2017-11-01',137.99,138.29,135.61,136.14,1002580,'0');
INSERT INTO daily_summary VALUES ('etf','CMF','us','2017-11-01',59.13,59.18,59.05,59.07,70312,'0');
INSERT INTO daily_summary VALUES ('stock','CMFN','us','2017-11-01',8.85,8.85,8.7,8.75,23931,'0');
INSERT INTO daily_summary VALUES ('stock','CMG','us','2017-11-01',271.9,275.48,269.75,270.2,1035290,'0');
INSERT INTO daily_summary VALUES ('stock','CMI','us','2017-11-01',179.26,181.79,172.57,172.83,2146840,'0');
INSERT INTO daily_summary VALUES ('stock','CMLS','us','2017-11-01',0.4178,0.4495,0.39,0.43,2703060,'0');
INSERT INTO daily_summary VALUES ('stock','CMO','us','2017-11-01',8.83,8.9,8.81,8.84,682533,'0');
INSERT INTO daily_summary VALUES ('stock','CMO_E','us','2017-11-01',25.1247,25.13,25.1,25.11,11773,'0');
INSERT INTO daily_summary VALUES ('stock','CMP','us','2017-11-01',66.65,67.35,65.35,65.6,502227,'0');
INSERT INTO daily_summary VALUES ('stock','CMPR','us','2017-11-01',110,110,104.065,109.37,516137,'0');
INSERT INTO daily_summary VALUES ('stock','CMRE','us','2017-11-01',6.21,6.3,6,6,985177,'0');
INSERT INTO daily_summary VALUES ('stock','CMRE_B','us','2017-11-01',24.9,25.2794,24.655,24.8505,9375,'0');
INSERT INTO daily_summary VALUES ('stock','CMRE_C','us','2017-11-01',25.5,25.5,25.11,25.33,10418,'0');
INSERT INTO daily_summary VALUES ('stock','CMRE_D','us','2017-11-01',26.63,26.63,26.49,26.501,3470,'0');
INSERT INTO daily_summary VALUES ('stock','CMRX','us','2017-11-01',4.93,4.93,4.8,4.88,859715,'0');
INSERT INTO daily_summary VALUES ('stock','CMS','us','2017-11-01',48.167,48.356,47.561,47.74,2071400,'0');
INSERT INTO daily_summary VALUES ('stock','CMSSU','us','2017-11-01',10.0024,10.01,10,10.01,52790,'0');
INSERT INTO daily_summary VALUES ('stock','CMS_B','us','2017-11-01',104.35,104.35,103.544,104,125,'0');
INSERT INTO daily_summary VALUES ('stock','CMT','us','2017-11-01',23.15,23.4,23.01,23.08,18867,'0');
INSERT INTO daily_summary VALUES ('stock','CMTA','us','2017-11-01',16.18,16.48,16.01,16.28,42016,'0');
INSERT INTO daily_summary VALUES ('stock','CMTL','us','2017-11-01',21.64,21.885,21.31,21.52,174526,'0');
INSERT INTO daily_summary VALUES ('stock','CMU','us','2017-11-01',4.83,4.83,4.76,4.76,71801,'0');
INSERT INTO daily_summary VALUES ('etf','CN','us','2017-11-01',37.27,37.27,37.14,37.18,8098,'0');
INSERT INTO daily_summary VALUES ('stock','CNA','us','2017-11-01',54.091,54.916,54.007,54.16,350158,'0');
INSERT INTO daily_summary VALUES ('stock','CNACR','us','2017-11-01',0.285,0.29,0.285,0.29,7000,'0');
INSERT INTO daily_summary VALUES ('stock','CNAT','us','2017-11-01',4.67,4.67,4.471,4.51,286998,'0');
INSERT INTO daily_summary VALUES ('stock','CNBKA','us','2017-11-01',84.8,85.2,83.55,84.05,5505,'0');
INSERT INTO daily_summary VALUES ('stock','CNC','us','2017-11-01',94.09,97.08,94.09,96.65,2238080,'0');
INSERT INTO daily_summary VALUES ('stock','CNCE','us','2017-11-01',17.27,17.4048,16.181,16.93,92327,'0');
INSERT INTO daily_summary VALUES ('stock','CNCR','us','2017-11-01',26,26.05,25.11,25.11,21034,'0');
INSERT INTO daily_summary VALUES ('etf','CNDA','us','2017-11-01',18.11,18.11,17.981,18.01,705,'0');
INSERT INTO daily_summary VALUES ('stock','CNDF','us','2017-11-01',28.37,28.37,28.37,28.37,0,'0');
INSERT INTO daily_summary VALUES ('stock','CNDT','us','2017-11-01',15.49,15.52,15.33,15.41,2046780,'0');
INSERT INTO daily_summary VALUES ('stock','CNET','us','2017-11-01',1.21,1.28,1.16,1.18,355995,'0');
INSERT INTO daily_summary VALUES ('stock','CNHI','us','2017-11-01',13.36,13.37,13.04,13.07,3449980,'0');
INSERT INTO daily_summary VALUES ('stock','CNHX','us','2017-11-01',29.5,29.5,29.5,29.5,115,'0');
INSERT INTO daily_summary VALUES ('stock','CNI','us','2017-11-01',80.19,80.66,79.43,80.02,1237240,'0');
INSERT INTO daily_summary VALUES ('stock','CNIT','us','2017-11-01',1.29,1.37,1.29,1.35,339567,'0');
INSERT INTO daily_summary VALUES ('stock','CNK','us','2017-11-01',36.48,36.6,35.19,35.4,1458960,'0');
INSERT INTO daily_summary VALUES ('stock','CNMD','us','2017-11-01',52.22,52.76,49.09,52.36,66723,'0');
INSERT INTO daily_summary VALUES ('stock','CNNX','us','2017-11-01',16.221,16.545,16.201,16.418,197531,'0');
INSERT INTO daily_summary VALUES ('stock','CNO','us','2017-11-01',24.22,24.43,23.81,23.92,997701,'0');
INSERT INTO daily_summary VALUES ('stock','CNOB','us','2017-11-01',27.2,27.2,26.45,26.8,106013,'0');
INSERT INTO daily_summary VALUES ('stock','CNP','us','2017-11-01',29.71,29.77,29.45,29.71,3733320,'0');
INSERT INTO daily_summary VALUES ('stock','CNQ','us','2017-11-01',35.04,35.29,34.595,35.02,2923270,'0');
INSERT INTO daily_summary VALUES ('stock','CNS','us','2017-11-01',43.8,43.8,43.23,43.57,87638,'0');
INSERT INTO daily_summary VALUES ('stock','CNSF','us','2017-11-01',24.35,24.35,24.35,24.35,0,'0');
INSERT INTO daily_summary VALUES ('stock','CNSL','us','2017-11-01',19.24,19.2999,18.87,19.16,540651,'0');
INSERT INTO daily_summary VALUES ('stock','CNTF','us','2017-11-01',2.22,2.2399,2.19,2.2054,1816,'0');
INSERT INTO daily_summary VALUES ('stock','CNTY','us','2017-11-01',8.33,8.4,8.21,8.3,32563,'0');
INSERT INTO daily_summary VALUES ('stock','CNX','us','2017-11-01',16.07,16.65,16.07,16.57,5603780,'0');
INSERT INTO daily_summary VALUES ('stock','CNXC','us','2017-11-01',14.318,14.602,14.125,14.125,89336,'0');
INSERT INTO daily_summary VALUES ('stock','CNXN','us','2017-11-01',27.2,27.2895,26.56,26.72,66003,'0');
INSERT INTO daily_summary VALUES ('etf','CNXT','us','2017-11-01',35.54,35.7,35.54,35.64,7351,'0');
INSERT INTO daily_summary VALUES ('etf','CNY','us','2017-11-01',45.42,45.42,42.36,43.17,927,'0');
INSERT INTO daily_summary VALUES ('stock','CNYA','us','2017-11-01',31.7699,31.77,31.719,31.74,20356,'0');
INSERT INTO daily_summary VALUES ('stock','CO','us','2017-11-01',12.2,12.44,11.99,12.04,239435,'0');
INSERT INTO daily_summary VALUES ('stock','COBZ','us','2017-11-01',20.6,20.81,20.21,20.25,144641,'0');
INSERT INTO daily_summary VALUES ('stock','CODA','us','2017-11-01',4.97,5,4.75,4.77,4782,'0');
INSERT INTO daily_summary VALUES ('stock','CODI','us','2017-11-01',17.25,17.25,16.95,17.1,199789,'0');
INSERT INTO daily_summary VALUES ('stock','CODI_A','us','2017-11-01',25.13,25.17,25.06,25.14,30516,'0');
INSERT INTO daily_summary VALUES ('stock','CODX','us','2017-11-01',5.85,5.85,5.53,5.61,72840,'0');
INSERT INTO daily_summary VALUES ('stock','COE','us','2017-11-01',13.04,13.24,12.91,13.24,9824,'0');
INSERT INTO daily_summary VALUES ('stock','COF','us','2017-11-01',92.239,92.568,90.995,91.483,1709550,'0');
INSERT INTO daily_summary VALUES ('stock','COF_C','us','2017-11-01',26.67,26.79,26.66,26.7,21286,'0');
INSERT INTO daily_summary VALUES ('stock','COF_D','us','2017-11-01',27.05,27.0653,26.98,27,89452,'0');
INSERT INTO daily_summary VALUES ('stock','COF_F','us','2017-11-01',27.02,27.1,27.02,27.0877,16084,'0');
INSERT INTO daily_summary VALUES ('stock','COF_G','us','2017-11-01',24.94,25,24.94,25,61229,'0');
INSERT INTO daily_summary VALUES ('stock','COF_H','us','2017-11-01',26.71,26.73,26.6,26.6,87063,'0');
INSERT INTO daily_summary VALUES ('stock','COF_P','us','2017-11-01',25.38,25.4,25.3703,25.39,33782,'0');
INSERT INTO daily_summary VALUES ('stock','COG','us','2017-11-01',27.702,28.052,27.502,27.862,7176570,'0');
INSERT INTO daily_summary VALUES ('stock','COGT','us','2017-11-01',4.5,4.6,4.4,4.4,186160,'0');
INSERT INTO daily_summary VALUES ('stock','COHN','us','2017-11-01',9.78,9.875,8.84,8.84,14270,'0');
INSERT INTO daily_summary VALUES ('stock','COHR','us','2017-11-01',264.01,265.005,254,256.77,359144,'0');
INSERT INTO daily_summary VALUES ('stock','COHU','us','2017-11-01',26.11,26.11,24.31,24.62,372698,'0');
INSERT INTO daily_summary VALUES ('stock','COKE','us','2017-11-01',226.25,228.11,224.06,224.7,34565,'0');
INSERT INTO daily_summary VALUES ('stock','COL','us','2017-11-01',135.4,135.7,135.09,135.51,1057300,'0');
INSERT INTO daily_summary VALUES ('stock','COLB','us','2017-11-01',44,44.18,43.15,44.16,3264420,'0');
INSERT INTO daily_summary VALUES ('stock','COLL','us','2017-11-01',10.4,10.605,9.45,9.55,613658,'0');
INSERT INTO daily_summary VALUES ('stock','COLM','us','2017-11-01',62.29,63.09,61.97,62.18,242023,'0');
INSERT INTO daily_summary VALUES ('stock','COM','us','2017-11-01',23.922,23.922,23.922,23.922,1656,'0');
INSERT INTO daily_summary VALUES ('stock','COMB','us','2017-11-01',25.74,25.74,25.6948,25.7001,2154,'0');
INSERT INTO daily_summary VALUES ('stock','COMG','us','2017-11-01',26.23,26.23,26.23,26.23,100,'0');
INSERT INTO daily_summary VALUES ('stock','COMM','us','2017-11-01',33.25,35.36,33.25,34.23,4694000,'0');
INSERT INTO daily_summary VALUES ('etf','COMT','us','2017-11-01',36.38,36.38,35.92,35.99,15294,'0');
INSERT INTO daily_summary VALUES ('stock','CONE','us','2017-11-01',61.7,61.73,60.52,60.67,724895,'0');
INSERT INTO daily_summary VALUES ('stock','CONN','us','2017-11-01',31,31.9,30.45,31,572575,'0');
INSERT INTO daily_summary VALUES ('stock','COO','us','2017-11-01',240.54,240.57,233.45,233.97,506955,'0');
INSERT INTO daily_summary VALUES ('stock','COOL','us','2017-11-01',26.5,26.55,25.05,25.71,71461,'0');
INSERT INTO daily_summary VALUES ('stock','COP','us','2017-11-01',51.81,52.28,51.43,52.1,6455300,'0');
INSERT INTO daily_summary VALUES ('etf','COPX','us','2017-11-01',26,26.27,25.83,25.92,15098,'0');
INSERT INTO daily_summary VALUES ('stock','COR','us','2017-11-01',111.32,111.75,108.1,108.25,613864,'0');
INSERT INTO daily_summary VALUES ('stock','CORE','us','2017-11-01',34.39,34.79,34.08,34.39,235620,'0');
INSERT INTO daily_summary VALUES ('stock','CORI','us','2017-11-01',10.13,10.66,10.11,10.5,116807,'0');
INSERT INTO daily_summary VALUES ('etf','CORN','us','2017-11-01',17.25,17.331,17.25,17.29,36006,'0');
INSERT INTO daily_summary VALUES ('etf','CORP','us','2017-11-01',105.34,105.69,105.34,105.534,55900,'0');
INSERT INTO daily_summary VALUES ('stock','CORR','us','2017-11-01',36.26,36.7899,36.06,36.23,81899,'0');
INSERT INTO daily_summary VALUES ('stock','CORR_A','us','2017-11-01',26.83,26.83,26.65,26.8,4311,'0');
INSERT INTO daily_summary VALUES ('stock','CORT','us','2017-11-01',19.75,19.93,18.73,19.1,1170470,'0');
INSERT INTO daily_summary VALUES ('stock','COST','us','2017-11-01',161.97,163.29,161.29,162.69,2129360,'0');
INSERT INTO daily_summary VALUES ('stock','COT','us','2017-11-01',15.13,15.25,15,15.1,875676,'0');
INSERT INTO daily_summary VALUES ('stock','COTV','us','2017-11-01',35.9,35.95,32.13,32.59,1501720,'0');
INSERT INTO daily_summary VALUES ('stock','COTY','us','2017-11-01',15.48,15.555,15.17,15.31,5365060,'0');
INSERT INTO daily_summary VALUES ('stock','COUP','us','2017-11-01',35,35.23,34.65,35.23,405895,'0');
INSERT INTO daily_summary VALUES ('etf','COW','us','2017-11-01',25.62,25.7156,25.462,25.4701,48933,'0');
INSERT INTO daily_summary VALUES ('stock','COWN','us','2017-11-01',15.2,15.2,14.725,14.75,548306,'0');
INSERT INTO daily_summary VALUES ('stock','COWNL','us','2017-11-01',26.63,26.87,26.63,26.87,320,'0');
INSERT INTO daily_summary VALUES ('stock','COWZ','us','2017-11-01',27.18,27.18,26.98,27.09,3184,'0');
INSERT INTO daily_summary VALUES ('stock','CP','us','2017-11-01',174.6,175.12,172.915,173.33,457140,'0');
INSERT INTO daily_summary VALUES ('stock','CPA','us','2017-11-01',123.89,125.72,123.62,124.07,187156,'0');
INSERT INTO daily_summary VALUES ('stock','CPAC','us','2017-11-01',12.98,12.98,12.61,12.61,539,'0');
INSERT INTO daily_summary VALUES ('stock','CPAH','us','2017-11-01',2.6906,2.9622,2.5756,2.9622,5691,'0');
INSERT INTO daily_summary VALUES ('stock','CPB','us','2017-11-01',47.57,47.79,46.93,47.53,1701590,'0');
INSERT INTO daily_summary VALUES ('stock','CPE','us','2017-11-01',11.33,11.82,11.28,11.74,6399880,'0');
INSERT INTO daily_summary VALUES ('etf','CPER','us','2017-11-01',20.1415,20.2407,20.08,20.08,5880,'0');
INSERT INTO daily_summary VALUES ('stock','CPE_A','us','2017-11-01',51.8,51.85,51.8,51.83,4901,'0');
INSERT INTO daily_summary VALUES ('stock','CPF','us','2017-11-01',31.34,31.58,30.79,31.05,78232,'0');
INSERT INTO daily_summary VALUES ('stock','CPG','us','2017-11-01',8.31,8.51,8.1,8.18,2477810,'0');
INSERT INTO daily_summary VALUES ('stock','CPHC','us','2017-11-01',12.85,13,12.85,13,1433,'0');
INSERT INTO daily_summary VALUES ('stock','CPHI','us','2017-11-01',0.175,0.175,0.1575,0.1642,46419,'0');
INSERT INTO daily_summary VALUES ('etf','CPI','us','2017-11-01',27.5054,27.5054,27.4953,27.4953,1060,'0');
INSERT INTO daily_summary VALUES ('stock','CPIX','us','2017-11-01',7.375,7.4944,7.3418,7.42,2110,'0');
INSERT INTO daily_summary VALUES ('stock','CPK','us','2017-11-01',81,81,78.6,79.35,46772,'0');
INSERT INTO daily_summary VALUES ('stock','CPL','us','2017-11-01',16.76,16.855,16.725,16.77,225436,'0');
INSERT INTO daily_summary VALUES ('stock','CPLA','us','2017-11-01',81.6,83.45,80.3,82.95,218341,'0');
INSERT INTO daily_summary VALUES ('stock','CPLP','us','2017-11-01',3.3391,3.3683,3.2902,3.3,524181,'0');
INSERT INTO daily_summary VALUES ('stock','CPN','us','2017-11-01',14.94,15.1,14.93,15,9091830,'0');
INSERT INTO daily_summary VALUES ('stock','CPRT','us','2017-11-01',36.41,36.76,35.82,35.84,1133660,'0');
INSERT INTO daily_summary VALUES ('stock','CPRX','us','2017-11-01',2.9,2.94,2.765,2.82,1027900,'0');
INSERT INTO daily_summary VALUES ('stock','CPS','us','2017-11-01',107.02,116.3,105,115.52,278140,'0');
INSERT INTO daily_summary VALUES ('stock','CPSH','us','2017-11-01',1.11,1.1357,1.1,1.12,33319,'0');
INSERT INTO daily_summary VALUES ('stock','CPSI','us','2017-11-01',30.2,30.2,29.5,29.6,128838,'0');
INSERT INTO daily_summary VALUES ('stock','CPSS','us','2017-11-01',4.43,4.57,4.37,4.565,32483,'0');
INSERT INTO daily_summary VALUES ('stock','CPST','us','2017-11-01',1.02,1.02,0.9631,0.9753,545662,'0');
INSERT INTO daily_summary VALUES ('stock','CPT','us','2017-11-01',91.11,91.74,90.77,91.42,468488,'0');
INSERT INTO daily_summary VALUES ('stock','CPTA','us','2017-11-01',8.3,8.3968,8.24,8.29,149127,'0');
INSERT INTO daily_summary VALUES ('stock','CPTAG','us','2017-11-01',25.099,25.1,25.0101,25.0856,3654,'0');
INSERT INTO daily_summary VALUES ('stock','CPTAL','us','2017-11-01',25.35,25.4297,25.35,25.4297,533,'0');
INSERT INTO daily_summary VALUES ('stock','CQH','us','2017-11-01',25.11,25.76,25.05,25.38,93822,'0');
INSERT INTO daily_summary VALUES ('stock','CQP','us','2017-11-01',27.701,27.78,27.465,27.76,173770,'0');
INSERT INTO daily_summary VALUES ('etf','CQQQ','us','2017-11-01',60.29,60.44,59.86,60.02,161279,'0');
INSERT INTO daily_summary VALUES ('stock','CR','us','2017-11-01',83.78,84.18,83.06,83.34,271966,'0');
INSERT INTO daily_summary VALUES ('stock','CRAI','us','2017-11-01',42.58,43.26,42.23,42.88,59934,'0');
INSERT INTO daily_summary VALUES ('stock','CRAK','us','2017-11-01',28.49,28.699,28.49,28.676,3015,'0');
INSERT INTO daily_summary VALUES ('stock','CRAY','us','2017-11-01',20.65,20.695,20.05,20.25,374111,'0');
INSERT INTO daily_summary VALUES ('etf','CRBN','us','2017-11-01',115.06,115.239,114.76,114.76,5088,'0');
INSERT INTO daily_summary VALUES ('stock','CRBP','us','2017-11-01',7.15,7.199,6.8,6.85,716800,'0');
INSERT INTO daily_summary VALUES ('stock','CRC','us','2017-11-01',11.21,12.28,11.2,11.92,2757060,'0');
INSERT INTO daily_summary VALUES ('stock','CRCM','us','2017-11-01',15.38,15.38,14.75,14.98,170053,'0');
INSERT INTO daily_summary VALUES ('etf','CRED','us','2017-11-01',111.83,111.99,111.72,111.85,48450,'0');
INSERT INTO daily_summary VALUES ('stock','CREE','us','2017-11-01',35.36,35.65,33.9,34.55,2689400,'0');
INSERT INTO daily_summary VALUES ('stock','CREG','us','2017-11-01',1.2,1.3,1.17,1.2651,42622,'0');
INSERT INTO daily_summary VALUES ('stock','CRESY','us','2017-11-01',21.287,22.023,21.287,21.768,284784,'0');
INSERT INTO daily_summary VALUES ('stock','CRF','us','2017-11-01',14.88,14.89,14.8,14.85,152413,'0');
INSERT INTO daily_summary VALUES ('stock','CRH','us','2017-11-01',37.32,37.36,36.745,36.79,365659,'0');
INSERT INTO daily_summary VALUES ('stock','CRHM','us','2017-11-01',2.35,2.4,2.3,2.35,50283,'0');
INSERT INTO daily_summary VALUES ('stock','CRI','us','2017-11-01',97.2,99.18,96.24,98.48,923883,'0');
INSERT INTO daily_summary VALUES ('stock','CRIS','us','2017-11-01',1.62,1.62,1.51,1.52,639138,'0');
INSERT INTO daily_summary VALUES ('stock','CRK','us','2017-11-01',4.79,4.99,4.73,4.99,407050,'0');
INSERT INTO daily_summary VALUES ('stock','CRL','us','2017-11-01',116.59,118,116.3,117.76,508541,'0');
INSERT INTO daily_summary VALUES ('stock','CRM','us','2017-11-01',103.67,103.92,101.3,101.95,3760780,'0');
INSERT INTO daily_summary VALUES ('stock','CRMD','us','2017-11-01',0.5558,0.569,0.54,0.5598,296122,'0');
INSERT INTO daily_summary VALUES ('stock','CRME','us','2017-11-01',1.86,1.87,1.6301,1.64,257082,'0');
INSERT INTO daily_summary VALUES ('stock','CRMT','us','2017-11-01',43.3,43.3,41.65,41.75,39465,'0');
INSERT INTO daily_summary VALUES ('stock','CRNT','us','2017-11-01',2.15,2.24,2.09,2.09,505465,'0');
INSERT INTO daily_summary VALUES ('etf','CROC','us','2017-11-01',48,48,47.38,47.49,4179,'0');
INSERT INTO daily_summary VALUES ('etf','CROP','us','2017-11-01',35.43,35.43,35.31,35.31,1636,'0');
INSERT INTO daily_summary VALUES ('stock','CROX','us','2017-11-01',10.22,10.28,10.04,10.2,772084,'0');
INSERT INTO daily_summary VALUES ('stock','CRR','us','2017-11-01',8.37,8.5,7.92,8.14,1280600,'0');
INSERT INTO daily_summary VALUES ('stock','CRS','us','2017-11-01',50.7,51.88,48.89,49.1,416045,'0');
INSERT INTO daily_summary VALUES ('stock','CRSP','us','2017-11-01',19.4,19.515,18.4,18.43,171324,'0');
INSERT INTO daily_summary VALUES ('stock','CRT','us','2017-11-01',14.99,15.2,14.9031,14.92,6064,'0');
INSERT INTO daily_summary VALUES ('stock','CRTN','us','2017-11-01',0.5198,0.5198,0.44,0.4699,52507,'0');
INSERT INTO daily_summary VALUES ('stock','CRTO','us','2017-11-01',38.14,40,37.5,37.74,2947330,'0');
INSERT INTO daily_summary VALUES ('stock','CRUS','us','2017-11-01',56.4,56.55,55.28,55.72,814480,'0');
INSERT INTO daily_summary VALUES ('stock','CRVL','us','2017-11-01',60.55,60.55,56.25,56.35,24235,'0');
INSERT INTO daily_summary VALUES ('stock','CRVP','us','2017-11-01',0.7011,0.7011,0.7011,0.7011,3027,'0');
INSERT INTO daily_summary VALUES ('stock','CRVS','us','2017-11-01',13.65,13.98,13.62,13.97,25809,'0');
INSERT INTO daily_summary VALUES ('stock','CRWS','us','2017-11-01',6.133,6.25,6.1,6.1,6469,'0');
INSERT INTO daily_summary VALUES ('stock','CRY','us','2017-11-01',19.4,19.6,18.9,19,195350,'0');
INSERT INTO daily_summary VALUES ('stock','CRZO','us','2017-11-01',17.95,18.68,17.89,18.17,3982510,'0');
INSERT INTO daily_summary VALUES ('stock','CS','us','2017-11-01',15.51,15.645,15.46,15.61,3211480,'0');
INSERT INTO daily_summary VALUES ('stock','CSA','us','2017-11-01',44.8712,44.879,44.8712,44.879,906,'0');
INSERT INTO daily_summary VALUES ('stock','CSB','us','2017-11-01',42.47,42.47,42.47,42.47,431,'0');
INSERT INTO daily_summary VALUES ('stock','CSBK','us','2017-11-01',17.05,17.06,16.85,16.95,44692,'0');
INSERT INTO daily_summary VALUES ('stock','CSBR','us','2017-11-01',3.2,3.46,3.17,3.45,10835,'0');
INSERT INTO daily_summary VALUES ('stock','CSCO','us','2017-11-01',34.29,34.75,34.28,34.62,22051300,'0');
INSERT INTO daily_summary VALUES ('etf','CSD','us','2017-11-01',51.27,51.464,50.98,51.09,20184,'0');
INSERT INTO daily_summary VALUES ('stock','CSF','us','2017-11-01',44.14,44.3047,43.7986,43.81,1728,'0');
INSERT INTO daily_summary VALUES ('stock','CSFL','us','2017-11-01',26.9,27.24,26.45,26.56,379098,'0');
INSERT INTO daily_summary VALUES ('stock','CSGP','us','2017-11-01',298.02,298.02,292.56,295.13,196276,'0');
INSERT INTO daily_summary VALUES ('stock','CSGS','us','2017-11-01',42.48,42.72,42.055,42.35,147953,'0');
INSERT INTO daily_summary VALUES ('stock','CSII','us','2017-11-01',24.24,24.82,24.04,24.6,294581,'0');
INSERT INTO daily_summary VALUES ('stock','CSIQ','us','2017-11-01',17.75,18.05,17.49,17.75,1377960,'0');
INSERT INTO daily_summary VALUES ('etf','CSJ','us','2017-11-01',105.03,105.06,105,105.04,232561,'0');
INSERT INTO daily_summary VALUES ('stock','CSL','us','2017-11-01',110.71,110.71,109.345,109.83,217713,'0');
INSERT INTO daily_summary VALUES ('stock','CSLT','us','2017-11-01',3.9,4,3.8,4,306061,'0');
INSERT INTO daily_summary VALUES ('etf','CSM','us','2017-11-01',64.0164,64.02,63.79,63.8264,11234,'0');
INSERT INTO daily_summary VALUES ('stock','CSML','us','2017-11-01',28.08,28.08,27.33,27.49,48200,'0');
INSERT INTO daily_summary VALUES ('stock','CSOD','us','2017-11-01',38.76,38.76,37.88,37.98,628032,'0');
INSERT INTO daily_summary VALUES ('stock','CSPI','us','2017-11-01',11.085,11.1,11,11.0252,3726,'0');
INSERT INTO daily_summary VALUES ('stock','CSQ','us','2017-11-01',12.3,12.34,12.26,12.29,186515,'0');
INSERT INTO daily_summary VALUES ('stock','CSRA','us','2017-11-01',32.1,32.34,31.98,32.02,747207,'0');
INSERT INTO daily_summary VALUES ('stock','CSS','us','2017-11-01',30,30.2,29.71,30.07,21168,'0');
INSERT INTO daily_summary VALUES ('stock','CSSE','us','2017-11-01',7.48,7.98,7.48,7.61,37730,'0');
INSERT INTO daily_summary VALUES ('stock','CSTE','us','2017-11-01',25.1,26.9,25.05,25.75,814710,'0');
INSERT INTO daily_summary VALUES ('stock','CSTM','us','2017-11-01',11.45,11.75,11.05,11.1,13784700,'0');
INSERT INTO daily_summary VALUES ('stock','CSTR','us','2017-11-01',20.31,20.31,19.39,19.46,33461,'0');
INSERT INTO daily_summary VALUES ('stock','CSU','us','2017-11-01',13.42,13.485,12.81,12.89,318558,'0');
INSERT INTO daily_summary VALUES ('stock','CSV','us','2017-11-01',25.927,25.927,25.299,25.638,91915,'0');
INSERT INTO daily_summary VALUES ('stock','CSWC','us','2017-11-01',17.3,17.37,17.06,17.1,17530,'0');
INSERT INTO daily_summary VALUES ('stock','CSWI','us','2017-11-01',49.5,49.55,48.65,49.3,38235,'0');
INSERT INTO daily_summary VALUES ('stock','CSX','us','2017-11-01',50.73,50.96,49.77,50.15,6119850,'0');
INSERT INTO daily_summary VALUES ('stock','CTAA','us','2017-11-01',26.38,26.3952,26.1945,26.23,17413,'0');
INSERT INTO daily_summary VALUES ('stock','CTAS','us','2017-11-01',147.24,147.44,144,145.53,702760,'0');
INSERT INTO daily_summary VALUES ('stock','CTB','us','2017-11-01',32.75,33.5,32.35,32.65,1179640,'0');
INSERT INTO daily_summary VALUES ('stock','CTBB','us','2017-11-01',25.3,25.34,25.25,25.28,47298,'0');
INSERT INTO daily_summary VALUES ('stock','CTBI','us','2017-11-01',48.8,49.2,47.7978,48.5,25649,'0');
INSERT INTO daily_summary VALUES ('stock','CTDD','us','2017-11-01',25.53,25.58,25.45,25.495,50844,'0');
INSERT INTO daily_summary VALUES ('stock','CTEK','us','2017-11-01',2.89,2.89,2.85,2.85,13071,'0');
INSERT INTO daily_summary VALUES ('stock','CTG','us','2017-11-01',5.04,5.14,5.04,5.055,19096,'0');
INSERT INTO daily_summary VALUES ('stock','CTHR','us','2017-11-01',0.84,0.8603,0.8039,0.859,120679,'0');
INSERT INTO daily_summary VALUES ('stock','CTIB','us','2017-11-01',3.85,3.8575,3.75,3.75,8204,'0');
INSERT INTO daily_summary VALUES ('stock','CTIC','us','2017-11-01',3.13,3.14,3.02,3.06,149351,'0');
INSERT INTO daily_summary VALUES ('stock','CTL','us','2017-11-01',19.04,19.05,17.52,17.85,20063700,'0');
INSERT INTO daily_summary VALUES ('stock','CTLT','us','2017-11-01',42.98,43.03,41.87,42,686827,'0');
INSERT INTO daily_summary VALUES ('stock','CTMX','us','2017-11-01',20,20.3,19.6,19.92,218546,'0');
INSERT INTO daily_summary VALUES ('stock','CTO','us','2017-11-01',58.669,59.391,57.68,57.96,4906,'0');
INSERT INTO daily_summary VALUES ('stock','CTR','us','2017-11-01',11.74,11.94,11.74,11.94,172284,'0');
INSERT INTO daily_summary VALUES ('stock','CTRE','us','2017-11-01',18.93,19.18,18.82,19.11,387050,'0');
INSERT INTO daily_summary VALUES ('stock','CTRL','us','2017-11-01',29.65,29.78,28.05,28.41,456792,'0');
INSERT INTO daily_summary VALUES ('stock','CTRN','us','2017-11-01',21.79,21.79,21.1,21.61,146167,'0');
INSERT INTO daily_summary VALUES ('stock','CTRP','us','2017-11-01',48.05,49.23,47.35,47.55,12254400,'0');
INSERT INTO daily_summary VALUES ('stock','CTRV','us','2017-11-01',0.5,0.55,0.483,0.53,1035040,'0');
INSERT INTO daily_summary VALUES ('stock','CTS','us','2017-11-01',27.55,27.55,26.55,27.1,72134,'0');
INSERT INTO daily_summary VALUES ('stock','CTSH','us','2017-11-01',74.64,75,70.53,73.54,9994910,'0');
INSERT INTO daily_summary VALUES ('stock','CTSO','us','2017-11-01',6.2,6.2,5.85,6,135246,'0');
INSERT INTO daily_summary VALUES ('stock','CTT','us','2017-11-01',12.89,13.06,12.81,12.98,370465,'0');
INSERT INTO daily_summary VALUES ('stock','CTU','us','2017-11-01',25.42,25.42,25.28,25.329,16207,'0');
INSERT INTO daily_summary VALUES ('stock','CTV','us','2017-11-01',25.72,25.74,25.6,25.6557,28240,'0');
INSERT INTO daily_summary VALUES ('stock','CTW','us','2017-11-01',25.7789,25.83,25.65,25.72,9238,'0');
INSERT INTO daily_summary VALUES ('stock','CTWS','us','2017-11-01',62.39,62.71,61.55,62.17,22699,'0');
INSERT INTO daily_summary VALUES ('stock','CTX','us','2017-11-01',25.3977,25.3977,25.33,25.3512,5211,'0');
INSERT INTO daily_summary VALUES ('stock','CTXR','us','2017-11-01',3.4,3.6846,3,3.11,126820,'0');
INSERT INTO daily_summary VALUES ('stock','CTXRW','us','2017-11-01',0.595,0.595,0.595,0.595,1275,'0');
INSERT INTO daily_summary VALUES ('stock','CTXS','us','2017-11-01',83.1,83.31,81.96,82.23,1391840,'0');
INSERT INTO daily_summary VALUES ('stock','CTY','us','2017-11-01',25.42,25.42,25.32,25.34,18721,'0');
INSERT INTO daily_summary VALUES ('stock','CTZ','us','2017-11-01',25.64,25.659,25.51,25.55,25620,'0');
INSERT INTO daily_summary VALUES ('stock','CUB','us','2017-11-01',54.9,54.9,53.55,54.2,61466,'0');
INSERT INTO daily_summary VALUES ('stock','CUBA','us','2017-11-01',7.1,7.12,7.03,7.0312,50219,'0');
INSERT INTO daily_summary VALUES ('stock','CUBE','us','2017-11-01',27.16,27.44,27.07,27.36,1462640,'0');
INSERT INTO daily_summary VALUES ('stock','CUBI','us','2017-11-01',27.48,27.7,26.76,26.89,342111,'0');
INSERT INTO daily_summary VALUES ('stock','CUBI_C','us','2017-11-01',26.99,26.99,26.9128,26.9128,748,'0');
INSERT INTO daily_summary VALUES ('stock','CUBI_D','us','2017-11-01',26.75,26.75,26.75,26.75,1300,'0');
INSERT INTO daily_summary VALUES ('stock','CUBI_F','us','2017-11-01',26.41,26.41,26.351,26.41,1778,'0');
INSERT INTO daily_summary VALUES ('stock','CUBN','us','2017-11-01',24.16,24.16,23.69,23.72,1920,'0');
INSERT INTO daily_summary VALUES ('stock','CUBS','us','2017-11-01',25.88,25.88,25.83,25.83,1200,'0');
INSERT INTO daily_summary VALUES ('stock','CUDA','us','2017-11-01',23.3,23.4,22.86,22.99,482919,'0');
INSERT INTO daily_summary VALUES ('stock','CUI','us','2017-11-01',3.12,3.14,2.95,2.96,297738,'0');
INSERT INTO daily_summary VALUES ('stock','CUK','us','2017-11-01',66.76,66.87,64.72,64.99,397996,'0');
INSERT INTO daily_summary VALUES ('stock','CULP','us','2017-11-01',31.9,31.9,31.05,31.5,15649,'0');
INSERT INTO daily_summary VALUES ('stock','CUMB','us','2017-11-01',25.55,25.6584,25.55,25.55,1889,'0');
INSERT INTO daily_summary VALUES ('stock','CUR','us','2017-11-01',1.21,1.24,1.17,1.22,98530,'0');
INSERT INTO daily_summary VALUES ('etf','CURE','us','2017-11-01',44.36,45.03,44.1,44.4,114962,'0');
INSERT INTO daily_summary VALUES ('etf','CUT','us','2017-11-01',31.46,31.5,31.32,31.41,12854,'0');
INSERT INTO daily_summary VALUES ('stock','CUTR','us','2017-11-01',39.4,39.4,37.8,39.05,147923,'0');
INSERT INTO daily_summary VALUES ('stock','CUZ','us','2017-11-01',9.11,9.12,8.96,9.02,2087590,'0');
INSERT INTO daily_summary VALUES ('stock','CVA','us','2017-11-01',16.15,16.25,15.6,15.65,1206820,'0');
INSERT INTO daily_summary VALUES ('stock','CVBF','us','2017-11-01',24.01,24.09,23.44,23.62,181466,'0');
INSERT INTO daily_summary VALUES ('stock','CVCO','us','2017-11-01',157.65,157.65,155.3,156.85,18543,'0');
INSERT INTO daily_summary VALUES ('stock','CVCY','us','2017-11-01',20.019,20.508,19.66,19.82,21140,'0');
INSERT INTO daily_summary VALUES ('stock','CVE','us','2017-11-01',9.86,10.16,9.82,10.09,4428200,'0');
INSERT INTO daily_summary VALUES ('stock','CVEO','us','2017-11-01',2.14,2.17,2.1,2.15,480907,'0');
INSERT INTO daily_summary VALUES ('stock','CVG','us','2017-11-01',26.01,26.1,25.7,25.81,519772,'0');
INSERT INTO daily_summary VALUES ('stock','CVGI','us','2017-11-01',8.15,8.35,8.1,8.2,183163,'0');
INSERT INTO daily_summary VALUES ('stock','CVGW','us','2017-11-01',74.35,74.35,71.95,72.3,93063,'0');
INSERT INTO daily_summary VALUES ('stock','CVI','us','2017-11-01',27.927,29.668,27.562,29.599,1242990,'0');
INSERT INTO daily_summary VALUES ('stock','CVLT','us','2017-11-01',52.3,52.3,51,51.6,890481,'0');
INSERT INTO daily_summary VALUES ('stock','CVLY','us','2017-11-01',31.75,31.75,31,31.11,3891,'0');
INSERT INTO daily_summary VALUES ('stock','CVM','us','2017-11-01',1.67,1.74,1.65,1.67,67399,'0');
INSERT INTO daily_summary VALUES ('stock','CVNA','us','2017-11-01',14.39,14.46,13.53,14.01,367640,'0');
INSERT INTO daily_summary VALUES ('stock','CVO','us','2017-11-01',1.25,1.38,1.18,1.3,435753,'0');
INSERT INTO daily_summary VALUES ('stock','CVR','us','2017-11-01',29.55,30.308,29.55,30,1825,'0');
INSERT INTO daily_summary VALUES ('stock','CVRR','us','2017-11-01',10.907,11.882,10.582,11.649,1467300,'0');
INSERT INTO daily_summary VALUES ('stock','CVRS','us','2017-11-01',1.14,1.145,0.98,0.9897,1269160,'0');
INSERT INTO daily_summary VALUES ('stock','CVS','us','2017-11-01',68.82,69.525,68.19,69,8433580,'0');
INSERT INTO daily_summary VALUES ('stock','CVTI','us','2017-11-01',29.83,30.5,29.1433,29.72,515685,'0');
INSERT INTO daily_summary VALUES ('stock','CVU','us','2017-11-01',9.25,9.34,8.95,9,34255,'0');
INSERT INTO daily_summary VALUES ('stock','CVV','us','2017-11-01',11.63,11.63,11.25,11.25,8802,'0');
INSERT INTO daily_summary VALUES ('stock','CVX','us','2017-11-01',116.3,116.97,115.35,115.9,5030040,'0');
INSERT INTO daily_summary VALUES ('etf','CVY','us','2017-11-01',21.66,21.7,21.5668,21.65,34743,'0');
INSERT INTO daily_summary VALUES ('stock','CW','us','2017-11-01',119,119,117.06,117.85,235840,'0');
INSERT INTO daily_summary VALUES ('stock','CWAI','us','2017-11-01',25.55,25.55,25.55,25.55,735,'0');
INSERT INTO daily_summary VALUES ('stock','CWAY','us','2017-11-01',20.25,20.25,20.075,20.075,1361,'0');
INSERT INTO daily_summary VALUES ('etf','CWB','us','2017-11-01',51.84,51.94,51.62,51.77,929412,'0');
INSERT INTO daily_summary VALUES ('stock','CWBC','us','2017-11-01',10.31,10.4,10.31,10.36,1171,'0');
INSERT INTO daily_summary VALUES ('stock','CWCO','us','2017-11-01',12.3,12.4,11.3,11.55,83245,'0');
INSERT INTO daily_summary VALUES ('stock','CWEB','us','2017-11-01',50.55,50.83,49.7133,50,77170,'0');
INSERT INTO daily_summary VALUES ('stock','CWH','us','2017-11-01',42.51,42.78,41.84,42.29,841079,'0');
INSERT INTO daily_summary VALUES ('etf','CWI','us','2017-11-01',38.78,38.84,38.65,38.65,81599,'0');
INSERT INTO daily_summary VALUES ('stock','CWS','us','2017-11-01',29.68,29.68,29.65,29.6644,695,'0');
INSERT INTO daily_summary VALUES ('stock','CWST','us','2017-11-01',18.56,18.69,17.99,18.28,277584,'0');
INSERT INTO daily_summary VALUES ('stock','CWT','us','2017-11-01',42.025,42.573,41.926,42.274,120187,'0');
INSERT INTO daily_summary VALUES ('stock','CX','us','2017-11-01',8.16,8.215,7.95,7.96,9747890,'0');
INSERT INTO daily_summary VALUES ('stock','CXDC','us','2017-11-01',4.85,4.9,4.85,4.85,8145,'0');
INSERT INTO daily_summary VALUES ('stock','CXE','us','2017-11-01',5.34,5.4,5.3,5.35,192840,'0');
INSERT INTO daily_summary VALUES ('stock','CXH','us','2017-11-01',9.8,9.8,9.77,9.79,3479,'0');
INSERT INTO daily_summary VALUES ('stock','CXO','us','2017-11-01',139.84,144.89,136.57,140.26,2521810,'0');
INSERT INTO daily_summary VALUES ('stock','CXP','us','2017-11-01',22.06,22.12,21.93,22.07,542814,'0');
INSERT INTO daily_summary VALUES ('stock','CXRX','us','2017-11-01',0.5211,0.55,0.5201,0.5401,201500,'0');
INSERT INTO daily_summary VALUES ('stock','CXSE','us','2017-11-01',82.62,82.749,82.17,82.17,12090,'0');
INSERT INTO daily_summary VALUES ('stock','CXW','us','2017-11-01',24.87,24.9,24.41,24.51,840320,'0');
INSERT INTO daily_summary VALUES ('stock','CY','us','2017-11-01',15.92,15.95,15.3,15.57,7892480,'0');
INSERT INTO daily_summary VALUES ('stock','CYAD','us','2017-11-01',58.55,58.5699,57.72,58.03,3418,'0');
INSERT INTO daily_summary VALUES ('stock','CYAN','us','2017-11-01',5.1,5.1,4.75,4.95,2717,'0');
INSERT INTO daily_summary VALUES ('etf','CYB','us','2017-11-01',25.58,25.6269,25.48,25.6,1751,'0');
INSERT INTO daily_summary VALUES ('stock','CYBE','us','2017-11-01',14.6,15.65,14.6,14.8,123281,'0');
INSERT INTO daily_summary VALUES ('stock','CYBR','us','2017-11-01',42.49,42.74,40.66,41.47,1375010,'0');
INSERT INTO daily_summary VALUES ('stock','CYCC','us','2017-11-01',1.86,1.87,1.63,1.67,903894,'0');
INSERT INTO daily_summary VALUES ('stock','CYCCP','us','2017-11-01',6.1,6.1,6,6.07,501,'0');
INSERT INTO daily_summary VALUES ('stock','CYD','us','2017-11-01',22.56,22.765,22.15,22.4,48206,'0');
INSERT INTO daily_summary VALUES ('stock','CYH','us','2017-11-01',5.8,5.83,5.255,5.44,8599050,'0');
INSERT INTO daily_summary VALUES ('stock','CYHHZ','us','2017-11-01',0.0095,0.0108,0.009,0.009,120294,'0');
INSERT INTO daily_summary VALUES ('stock','CYOU','us','2017-11-01',38.73,39,38.28,38.56,335915,'0');
INSERT INTO daily_summary VALUES ('stock','CYRN','us','2017-11-01',1.8,1.8,1.695,1.7,37529,'0');
INSERT INTO daily_summary VALUES ('stock','CYRX','us','2017-11-01',7.86,7.98,7.78,7.87,111657,'0');
INSERT INTO daily_summary VALUES ('stock','CYRXW','us','2017-11-01',4.43,4.5885,4.3047,4.3305,7903,'0');
INSERT INTO daily_summary VALUES ('stock','CYS','us','2017-11-01',8,8.125,8,8.07,1558340,'0');
INSERT INTO daily_summary VALUES ('stock','CYS_A','us','2017-11-01',25.1,25.1516,25.1,25.15,6645,'0');
INSERT INTO daily_summary VALUES ('stock','CYS_B','us','2017-11-01',24.88,24.88,24.75,24.78,10480,'0');
INSERT INTO daily_summary VALUES ('stock','CYTK','us','2017-11-01',13.8,13.8,12.85,12.9,435436,'0');
INSERT INTO daily_summary VALUES ('stock','CYTR','us','2017-11-01',1.89,2.5898,1.87,2.1,1035250,'0');
INSERT INTO daily_summary VALUES ('stock','CYTX','us','2017-11-01',0.42,0.4238,0.4001,0.4001,181905,'0');
INSERT INTO daily_summary VALUES ('stock','CYTXW','us','2017-11-01',0.027,0.027,0.027,0.027,5001,'0');
INSERT INTO daily_summary VALUES ('etf','CZA','us','2017-11-01',64.98,65.02,64.6,64.66,15265,'0');
INSERT INTO daily_summary VALUES ('stock','CZFC','us','2017-11-01',23.5,23.5,23.5,23.5,246,'0');
INSERT INTO daily_summary VALUES ('stock','CZNC','us','2017-11-01',24.65,24.98,23.48,23.92,29284,'0');
INSERT INTO daily_summary VALUES ('stock','CZR','us','2017-11-01',13,13.15,12.8,12.95,15693600,'0');
INSERT INTO daily_summary VALUES ('stock','CZWI','us','2017-11-01',13.65,13.85,13.65,13.85,2928,'0');
INSERT INTO daily_summary VALUES ('stock','CZZ','us','2017-11-01',8.71,8.94,8.65,8.73,1115370,'0');
INSERT INTO daily_summary VALUES ('stock','C_C','us','2017-11-01',25.45,25.4976,25.45,25.49,15825,'0');
INSERT INTO daily_summary VALUES ('stock','C_J','us','2017-11-01',28.92,28.93,28.8262,28.89,34111,'0');
INSERT INTO daily_summary VALUES ('stock','C_K','us','2017-11-01',28.96,29.19,28.91,29.17,156356,'0');
INSERT INTO daily_summary VALUES ('stock','C_L','us','2017-11-01',26.53,26.56,26.5,26.505,16539,'0');
INSERT INTO daily_summary VALUES ('stock','C_N','us','2017-11-01',27.29,27.37,27.25,27.28,203403,'0');
INSERT INTO daily_summary VALUES ('stock','C_P','us','2017-11-01',27.98,28.7117,27.9372,28.27,16368,'0');
INSERT INTO daily_summary VALUES ('stock','C_S','us','2017-11-01',26.63,26.7605,26.63,26.73,59848,'0');
INSERT INTO daily_summary VALUES ('stock','D','us','2017-11-01',81.41,81.41,80.43,80.49,2118310,'0');
INSERT INTO daily_summary VALUES ('stock','DAC','us','2017-11-01',1.65,1.65,1.45,1.45,70965,'0');
INSERT INTO daily_summary VALUES ('etf','DAG','us','2017-11-01',2.96,2.96,2.96,2.96,250,'0');
INSERT INTO daily_summary VALUES ('stock','DAIO','us','2017-11-01',12.85,13.48,12.54,12.8,467492,'0');
INSERT INTO daily_summary VALUES ('stock','DAKT','us','2017-11-01',10.37,10.405,10.08,10.09,97320,'0');
INSERT INTO daily_summary VALUES ('stock','DAL','us','2017-11-01',50.19,50.93,49.95,50.35,5833720,'0');
INSERT INTO daily_summary VALUES ('stock','DALT','us','2017-11-01',10.17,10.18,10.1305,10.1585,36214,'0');
INSERT INTO daily_summary VALUES ('stock','DAN','us','2017-11-01',30.88,31.23,30.521,30.821,2059200,'0');
INSERT INTO daily_summary VALUES ('stock','DAR','us','2017-11-01',18.28,18.32,17.6099,17.73,1319180,'0');
INSERT INTO daily_summary VALUES ('stock','DARE','us','2017-11-01',2.5,2.64,2.3611,2.51,21376,'0');
INSERT INTO daily_summary VALUES ('stock','DATA','us','2017-11-01',81.55,81.74,79.27,80.76,1356780,'0');
INSERT INTO daily_summary VALUES ('stock','DAVE','us','2017-11-01',3.9,3.9,3.75,3.8,891,'0');
INSERT INTO daily_summary VALUES ('stock','DAX','us','2017-11-01',31.5,31.85,31.5,31.75,10495,'0');
INSERT INTO daily_summary VALUES ('stock','DB','us','2017-11-01',16.28,16.58,16.26,16.56,5113050,'0');
INSERT INTO daily_summary VALUES ('etf','DBA','us','2017-11-01',19.4,19.51,19.385,19.44,603655,'0');
INSERT INTO daily_summary VALUES ('etf','DBAW','us','2017-11-01',28.1799,28.18,28.17,28.17,2454,'0');
INSERT INTO daily_summary VALUES ('etf','DBB','us','2017-11-01',18.89,18.95,18.87,18.9,158996,'0');
INSERT INTO daily_summary VALUES ('etf','DBBR','us','2017-11-01',12.01,12.01,11.79,11.79,5306,'0');
INSERT INTO daily_summary VALUES ('etf','DBC','us','2017-11-01',16.11,16.165,15.96,16.02,5287100,'0');
INSERT INTO daily_summary VALUES ('stock','DBD','us','2017-11-01',19.25,19.3,18.35,18.5,1645280,'0');
INSERT INTO daily_summary VALUES ('etf','DBE','us','2017-11-01',13.81,13.8654,13.61,13.69,467540,'0');
INSERT INTO daily_summary VALUES ('etf','DBEF','us','2017-11-01',32.23,32.24,32.11,32.16,884994,'0');
INSERT INTO daily_summary VALUES ('etf','DBEM','us','2017-11-01',24.27,24.3785,24.08,24.17,62261,'0');
INSERT INTO daily_summary VALUES ('stock','DBES','us','2017-11-01',30.3143,30.3143,30.3143,30.3143,720,'0');
INSERT INTO daily_summary VALUES ('etf','DBEU','us','2017-11-01',29.1,29.11,28.9823,29.02,415987,'0');
INSERT INTO daily_summary VALUES ('etf','DBEZ','us','2017-11-01',31.501,31.501,31.447,31.46,1452,'0');
INSERT INTO daily_summary VALUES ('etf','DBGR','us','2017-11-01',29.36,29.39,29.301,29.3441,19830,'0');
INSERT INTO daily_summary VALUES ('etf','DBJP','us','2017-11-01',43.66,43.67,43.46,43.57,283179,'0');
INSERT INTO daily_summary VALUES ('etf','DBKO','us','2017-11-01',32.34,32.34,32.3232,32.329,2953,'0');
INSERT INTO daily_summary VALUES ('stock','DBL','us','2017-11-01',22.92,23.2,22.92,23.14,94808,'0');
INSERT INTO daily_summary VALUES ('etf','DBO','us','2017-11-01',9.32,9.34,9.16,9.22,246824,'0');
INSERT INTO daily_summary VALUES ('etf','DBP','us','2017-11-01',37.95,38.062,37.95,38.0075,2959,'0');
INSERT INTO daily_summary VALUES ('stock','DBRT','us','2017-11-01',69.25,73.15,69.25,72.83,702,'0');
INSERT INTO daily_summary VALUES ('etf','DBS','us','2017-11-01',25.91,26.33,25.91,26.22,4205,'0');
INSERT INTO daily_summary VALUES ('etf','DBV','us','2017-11-01',23.78,23.855,23.721,23.76,5031,'0');
INSERT INTO daily_summary VALUES ('stock','DBVT','us','2017-11-01',24.15,24.26,23.35,23.43,354969,'0');
INSERT INTO daily_summary VALUES ('stock','DCF','us','2017-11-01',9.97,9.99,9.96,9.96,31850,'0');
INSERT INTO daily_summary VALUES ('stock','DCI','us','2017-11-01',47.66,47.965,47.4,47.61,372278,'0');
INSERT INTO daily_summary VALUES ('stock','DCIX','us','2017-11-01',2.2407,2.422,2.0307,2.1,194501,'0');
INSERT INTO daily_summary VALUES ('stock','DCM','us','2017-11-01',24.27,24.32,24.17,24.28,104803,'0');
INSERT INTO daily_summary VALUES ('stock','DCO','us','2017-11-01',33.17,33.27,32.29,32.66,65435,'0');
INSERT INTO daily_summary VALUES ('stock','DCOM','us','2017-11-01',22.059,22.059,21.612,21.661,170710,'0');
INSERT INTO daily_summary VALUES ('stock','DCP','us','2017-11-01',32.718,33.802,32.695,33.343,423440,'0');
INSERT INTO daily_summary VALUES ('stock','DCPH','us','2017-11-01',19.23,19.85,17.3636,19.08,141301,'0');
INSERT INTO daily_summary VALUES ('stock','DCT','us','2017-11-01',58.09,58.81,58.04,58.6,520797,'0');
INSERT INTO daily_summary VALUES ('stock','DCUD','us','2017-11-01',52.33,52.33,51.9376,52.01,5024,'0');
INSERT INTO daily_summary VALUES ('stock','DD','us','2017-11-01',83.54,85.16,83.31,83.93,34861000,'0');
INSERT INTO daily_summary VALUES ('stock','DDBI','us','2017-11-01',29.61,29.61,29.57,29.575,1410,'0');
INSERT INTO daily_summary VALUES ('stock','DDC','us','2017-11-01',14.26,14.26,14.24,14.24,988206,'0');
INSERT INTO daily_summary VALUES ('stock','DDD','us','2017-11-01',9.37,9.88,8.7,9.45,17309100,'0');
INSERT INTO daily_summary VALUES ('stock','DDE','us','2017-11-01',1.01,1.01,0.9906,0.9906,16542,'0');
INSERT INTO daily_summary VALUES ('stock','DDEZ','us','2017-11-01',32.21,32.21,32.21,32.21,2073,'0');
INSERT INTO daily_summary VALUES ('stock','DDF','us','2017-11-01',10.69,10.72,10.672,10.68,11181,'0');
INSERT INTO daily_summary VALUES ('stock','DDJP','us','2017-11-01',29.78,29.81,29.78,29.79,2420,'0');
INSERT INTO daily_summary VALUES ('stock','DDLS','us','2017-11-01',32.8705,32.8705,32.864,32.864,669,'0');
INSERT INTO daily_summary VALUES ('etf','DDM','us','2017-11-01',118.87,118.92,117.689,118.16,190981,'0');
INSERT INTO daily_summary VALUES ('stock','DDR','us','2017-11-01',7.77,7.925,7.675,7.78,4597710,'0');
INSERT INTO daily_summary VALUES ('stock','DDR_A','us','2017-11-01',25.5,25.62,25.4603,25.485,23945,'0');
INSERT INTO daily_summary VALUES ('stock','DDR_J','us','2017-11-01',25.18,25.29,25.18,25.29,2744,'0');
INSERT INTO daily_summary VALUES ('stock','DDR_K','us','2017-11-01',25.12,25.285,25.111,25.285,4972,'0');
INSERT INTO daily_summary VALUES ('stock','DDS','us','2017-11-01',51.12,53.07,50.9,51.49,487746,'0');
INSERT INTO daily_summary VALUES ('stock','DDT','us','2017-11-01',25.66,25.66,25.55,25.6058,9931,'0');
INSERT INTO daily_summary VALUES ('stock','DDWM','us','2017-11-01',30.42,30.435,30.3301,30.37,27369,'0');
INSERT INTO daily_summary VALUES ('stock','DD_B','us','2017-11-01',104.2,104.45,104.2,104.45,1408,'0');
INSERT INTO daily_summary VALUES ('stock','DE','us','2017-11-01',133.87,134.73,133.25,134.27,1557990,'0');
INSERT INTO daily_summary VALUES ('stock','DEA','us','2017-11-01',20.11,20.31,20.07,20.25,193964,'0');
INSERT INTO daily_summary VALUES ('stock','DECK','us','2017-11-01',68.62,69.375,67.95,68.86,804415,'0');
INSERT INTO daily_summary VALUES ('stock','DEEF','us','2017-11-01',29.4471,29.48,29.36,29.36,2458,'0');
INSERT INTO daily_summary VALUES ('etf','DEF','us','2017-11-01',45.34,45.36,45.259,45.2829,7274,'0');
INSERT INTO daily_summary VALUES ('stock','DEFA','us','2017-11-01',29.25,29.25,29.25,29.25,174,'0');
INSERT INTO daily_summary VALUES ('stock','DEI','us','2017-11-01',39.88,40.14,39.74,39.89,464021,'0');
INSERT INTO daily_summary VALUES ('stock','DEL','us','2017-11-01',93.41,94.89,92.86,94.25,527423,'0');
INSERT INTO daily_summary VALUES ('stock','DELT','us','2017-11-01',1.36,1.4,1.29,1.3,211606,'0');
INSERT INTO daily_summary VALUES ('stock','DELTW','us','2017-11-01',0.0197,0.0198,0.0196,0.0196,16550,'0');
INSERT INTO daily_summary VALUES ('etf','DEM','us','2017-11-01',43.75,43.8837,43.62,43.64,180196,'0');
INSERT INTO daily_summary VALUES ('stock','DEMG','us','2017-11-01',28.5039,28.508,28.5,28.5,922,'0');
INSERT INTO daily_summary VALUES ('stock','DEMS','us','2017-11-01',20.2,20.2,20.2,20.2,178,'0');
INSERT INTO daily_summary VALUES ('stock','DENN','us','2017-11-01',13.07,13.24,12.83,12.84,482428,'0');
INSERT INTO daily_summary VALUES ('stock','DEO','us','2017-11-01',136.47,136.635,134.43,134.43,379841,'0');
INSERT INTO daily_summary VALUES ('stock','DEPO','us','2017-11-01',4.85,5.27,4.84,5.17,2359710,'0');
INSERT INTO daily_summary VALUES ('stock','DERM','us','2017-11-01',26.82,27.4,26.64,26.88,208564,'0');
INSERT INTO daily_summary VALUES ('etf','DES','us','2017-11-01',28.313,28.313,27.897,28.037,165390,'0');
INSERT INTO daily_summary VALUES ('stock','DESC','us','2017-11-01',34.09,34.09,33.64,33.75,13765,'0');
INSERT INTO daily_summary VALUES ('stock','DESP','us','2017-11-01',30.69,30.96,29.845,30.58,194540,'0');
INSERT INTO daily_summary VALUES ('stock','DEST','us','2017-11-01',2.32,2.439,2.22,2.42,211910,'0');
INSERT INTO daily_summary VALUES ('stock','DEUS','us','2017-11-01',31.0024,31.03,30.78,30.8751,34720,'0');
INSERT INTO daily_summary VALUES ('etf','DEW','us','2017-11-01',47.16,47.24,47.085,47.16,5622,'0');
INSERT INTO daily_summary VALUES ('stock','DEWJ','us','2017-11-01',29.37,29.37,29.3285,29.3298,868,'0');
INSERT INTO daily_summary VALUES ('stock','DEX','us','2017-11-01',11.75,11.83,11.74,11.79,37800,'0');
INSERT INTO daily_summary VALUES ('stock','DEZU','us','2017-11-01',29.76,29.76,29.729,29.729,461,'0');
INSERT INTO daily_summary VALUES ('stock','DF','us','2017-11-01',9.8,9.98,9.69,9.74,1517160,'0');
INSERT INTO daily_summary VALUES ('stock','DFBG','us','2017-11-01',1,1.1,0.95,1.05,38815,'0');
INSERT INTO daily_summary VALUES ('etf','DFE','us','2017-11-01',69.72,69.77,69.37,69.37,59437,'0');
INSERT INTO daily_summary VALUES ('stock','DFEN','us','2017-11-01',40.24,40.5,39.8,40,31376,'0');
INSERT INTO daily_summary VALUES ('stock','DFFN','us','2017-11-01',1.47,1.5,1.41,1.44,99603,'0');
INSERT INTO daily_summary VALUES ('stock','DFIN','us','2017-11-01',21.59,21.6,20.93,21.12,233010,'0');
INSERT INTO daily_summary VALUES ('etf','DFJ','us','2017-11-01',77.91,77.98,77.77,77.91,49941,'0');
INSERT INTO daily_summary VALUES ('stock','DFND','us','2017-11-01',26.142,26.142,26.142,26.142,0,'0');
INSERT INTO daily_summary VALUES ('stock','DFNL','us','2017-11-01',23.39,23.4001,23.28,23.29,4402,'0');
INSERT INTO daily_summary VALUES ('stock','DFP','us','2017-11-01',26,26.04,25.95,26.03,40446,'0');
INSERT INTO daily_summary VALUES ('stock','DFRG','us','2017-11-01',13.9,14.15,13.4,13.5,479096,'0');
INSERT INTO daily_summary VALUES ('stock','DFS','us','2017-11-01',66.68,67.175,66.3,66.9,1923130,'0');
INSERT INTO daily_summary VALUES ('stock','DG','us','2017-11-01',81.21,81.65,80.46,81.23,1487750,'0');
INSERT INTO daily_summary VALUES ('etf','DGAZ','us','2017-11-01',32.75,32.77,31.17,31.26,1788270,'0');
INSERT INTO daily_summary VALUES ('stock','DGICA','us','2017-11-01',17.5,17.68,17.32,17.53,34015,'0');
INSERT INTO daily_summary VALUES ('stock','DGICB','us','2017-11-01',15.45,15.6,15.45,15.6,788,'0');
INSERT INTO daily_summary VALUES ('stock','DGII','us','2017-11-01',10.5,10.525,9.9,10,41590,'0');
INSERT INTO daily_summary VALUES ('etf','DGL','us','2017-11-01',40.48,40.62,40.44,40.56,31384,'0');
INSERT INTO daily_summary VALUES ('etf','DGLD','us','2017-11-01',46.88,46.96,46.18,46.65,79160,'0');
INSERT INTO daily_summary VALUES ('stock','DGLY','us','2017-11-01',1.8,1.9,1.8,1.875,52326,'0');
INSERT INTO daily_summary VALUES ('etf','DGP','us','2017-11-01',23.99,24.2414,23.93,24.01,8949,'0');
INSERT INTO daily_summary VALUES ('etf','DGRE','us','2017-11-01',25.86,25.9792,25.74,25.77,18713,'0');
INSERT INTO daily_summary VALUES ('etf','DGRO','us','2017-11-01',33.25,33.359,33.2,33.24,412356,'0');
INSERT INTO daily_summary VALUES ('etf','DGRS','us','2017-11-01',34.83,34.83,34.36,34.49,10434,'0');
INSERT INTO daily_summary VALUES ('etf','DGRW','us','2017-11-01',39.27,39.32,39.13,39.22,129290,'0');
INSERT INTO daily_summary VALUES ('etf','DGS','us','2017-11-01',49.53,49.7,49.4221,49.5,83569,'0');
INSERT INTO daily_summary VALUES ('stock','DGSE','us','2017-11-01',0.9,0.9102,0.9,0.91,8962,'0');
INSERT INTO daily_summary VALUES ('etf','DGT','us','2017-11-01',82.5473,82.5473,82.5473,82.5473,195,'0');
INSERT INTO daily_summary VALUES ('stock','DGX','us','2017-11-01',93.91,94.49,93,93.1,1371900,'0');
INSERT INTO daily_summary VALUES ('etf','DGZ','us','2017-11-01',13.9294,13.948,13.9294,13.948,897,'0');
INSERT INTO daily_summary VALUES ('stock','DHDG','us','2017-11-01',24.97,24.97,24.97,24.97,20,'0');
INSERT INTO daily_summary VALUES ('stock','DHF','us','2017-11-01',3.4501,3.4699,3.4501,3.4699,92032,'0');
INSERT INTO daily_summary VALUES ('stock','DHG','us','2017-11-01',15.15,15.17,15.07,15.12,23321,'0');
INSERT INTO daily_summary VALUES ('stock','DHI','us','2017-11-01',44.46,44.92,44.37,44.73,3436780,'0');
INSERT INTO daily_summary VALUES ('stock','DHIL','us','2017-11-01',213.396,213.76,211.72,213.76,22684,'0');
INSERT INTO daily_summary VALUES ('stock','DHR','us','2017-11-01',92.37,93.13,92.09,92.91,2878850,'0');
INSERT INTO daily_summary VALUES ('etf','DHS','us','2017-11-01',70.16,70.35,70.125,70.17,37628,'0');
INSERT INTO daily_summary VALUES ('stock','DHT','us','2017-11-01',3.95,4.13,3.95,4.08,749684,'0');
INSERT INTO daily_summary VALUES ('stock','DHVW','us','2017-11-01',30.76,30.8499,30.695,30.695,5711,'0');
INSERT INTO daily_summary VALUES ('stock','DHX','us','2017-11-01',2.15,2.25,2.15,2.25,160621,'0');
INSERT INTO daily_summary VALUES ('stock','DHXM','us','2017-11-01',3.5,3.5,3,3.05,104055,'0');
INSERT INTO daily_summary VALUES ('stock','DHY','us','2017-11-01',2.84,2.85,2.8348,2.84,269450,'0');
INSERT INTO daily_summary VALUES ('etf','DIA','us','2017-11-01',234.6,235.02,233.73,234.19,2674200,'0');
INSERT INTO daily_summary VALUES ('stock','DIAL','us','2017-11-01',19.99,20,19.955,19.99,1295700,'0');
INSERT INTO daily_summary VALUES ('stock','DIAX','us','2017-11-01',17.8,17.8,17.651,17.7,88855,'0');
INSERT INTO daily_summary VALUES ('etf','DIG','us','2017-11-01',35.1,35.89,35.1,35.55,122124,'0');
INSERT INTO daily_summary VALUES ('etf','DIM','us','2017-11-01',67.944,67.944,67.7267,67.74,4718,'0');
INSERT INTO daily_summary VALUES ('stock','DIN','us','2017-11-01',47.67,47.9078,47,47.51,273876,'0');
INSERT INTO daily_summary VALUES ('stock','DIOD','us','2017-11-01',34.67,34.72,33.02,33.63,326598,'0');
INSERT INTO daily_summary VALUES ('stock','DIS','us','2017-11-01',98.13,99.46,98,99.03,6920050,'0');
INSERT INTO daily_summary VALUES ('stock','DISCA','us','2017-11-01',18.94,19.315,18.77,19.16,6748530,'0');
INSERT INTO daily_summary VALUES ('stock','DISCB','us','2017-11-01',23.55,23.55,23.55,23.55,281,'0');
INSERT INTO daily_summary VALUES ('stock','DISCK','us','2017-11-01',17.87,18.2,17.72,18.09,4043960,'0');
INSERT INTO daily_summary VALUES ('stock','DISH','us','2017-11-01',48.85,49.13,48.34,48.77,3100080,'0');
INSERT INTO daily_summary VALUES ('etf','DIV','us','2017-11-01',25.15,25.1752,25.05,25.1,82262,'0');
INSERT INTO daily_summary VALUES ('stock','DIVO','us','2017-11-01',27.518,27.5865,27.518,27.5855,1462,'0');
INSERT INTO daily_summary VALUES ('stock','DIVY','us','2017-11-01',26.0372,26.0423,26.01,26.0232,1303,'0');
INSERT INTO daily_summary VALUES ('etf','DJCI','us','2017-11-01',15.71,15.75,15.71,15.71,1881,'0');
INSERT INTO daily_summary VALUES ('stock','DJCO','us','2017-11-01',235,247.95,235,244.65,12028,'0');
INSERT INTO daily_summary VALUES ('stock','DJD','us','2017-11-01',33,33,32.61,32.72,12444,'0');
INSERT INTO daily_summary VALUES ('etf','DJP','us','2017-11-01',23.94,24.0285,23.867,23.94,238733,'0');
INSERT INTO daily_summary VALUES ('stock','DK','us','2017-11-01',26.47,27.81,26.24,27.75,1195780,'0');
INSERT INTO daily_summary VALUES ('stock','DKL','us','2017-11-01',30.082,30.423,29.642,30.375,27514,'0');
INSERT INTO daily_summary VALUES ('stock','DKS','us','2017-11-01',24.38,24.81,23.88,24.39,3466030,'0');
INSERT INTO daily_summary VALUES ('stock','DKT','us','2017-11-01',26.39,26.4,26.33,26.34,41183,'0');
INSERT INTO daily_summary VALUES ('stock','DL','us','2017-11-01',7.76,7.7692,7.65,7.67,32503,'0');
INSERT INTO daily_summary VALUES ('stock','DLA','us','2017-11-01',21.17,21.17,20.9,20.93,7837,'0');
INSERT INTO daily_summary VALUES ('stock','DLB','us','2017-11-01',57.981,58.48,57.632,58.191,273760,'0');
INSERT INTO daily_summary VALUES ('stock','DLBS','us','2017-11-01',16.78,16.78,16.78,16.78,230,'0');
INSERT INTO daily_summary VALUES ('stock','DLHC','us','2017-11-01',6.05,6.15,6.05,6.14,12505,'0');
INSERT INTO daily_summary VALUES ('etf','DLN','us','2017-11-01',88.51,88.64,88.231,88.32,41585,'0');
INSERT INTO daily_summary VALUES ('stock','DLNG','us','2017-11-01',13.15,13.25,13.04,13.15,88649,'0');
INSERT INTO daily_summary VALUES ('stock','DLNG_A','us','2017-11-01',26.971,26.971,26.835,26.95,5685,'0');
INSERT INTO daily_summary VALUES ('stock','DLPH','us','2017-11-01',100.2,100.83,98.795,99.084,1855790,'0');
INSERT INTO daily_summary VALUES ('stock','DLR','us','2017-11-01',118.85,118.85,116.86,116.94,934081,'0');
INSERT INTO daily_summary VALUES ('stock','DLR_C','us','2017-11-01',28,28,27.75,27.8693,5143,'0');
INSERT INTO daily_summary VALUES ('stock','DLR_G','us','2017-11-01',25.43,25.4995,25.41,25.4685,7349,'0');
INSERT INTO daily_summary VALUES ('stock','DLR_H','us','2017-11-01',26.71,26.83,26.71,26.77,12701,'0');
INSERT INTO daily_summary VALUES ('stock','DLR_I','us','2017-11-01',27.3,27.3,27.16,27.171,7669,'0');
INSERT INTO daily_summary VALUES ('stock','DLR_J','us','2017-11-01',25.02,25.05,24.98,25.05,11502,'0');
INSERT INTO daily_summary VALUES ('etf','DLS','us','2017-11-01',75.3,75.38,75.04,75.04,50653,'0');
INSERT INTO daily_summary VALUES ('stock','DLTH','us','2017-11-01',19.67,19.9,19.23,19.64,577518,'0');
INSERT INTO daily_summary VALUES ('stock','DLTR','us','2017-11-01',91.6,92.06,90.34,91.12,1750040,'0');
INSERT INTO daily_summary VALUES ('stock','DLX','us','2017-11-01',69.89,69.925,68.85,69.81,337885,'0');
INSERT INTO daily_summary VALUES ('stock','DM','us','2017-11-01',31.895,32.433,31.35,31.895,162844,'0');
INSERT INTO daily_summary VALUES ('stock','DMB','us','2017-11-01',12.95,12.9999,12.92,12.95,51990,'0');
INSERT INTO daily_summary VALUES ('stock','DMF','us','2017-11-01',8.9495,8.9594,8.8997,8.9594,28883,'0');
INSERT INTO daily_summary VALUES ('stock','DMLP','us','2017-11-01',14.75,14.95,14.7,14.75,104194,'0');
INSERT INTO daily_summary VALUES ('stock','DMO','us','2017-11-01',24.04,24.35,24.04,24.3,81323,'0');
INSERT INTO daily_summary VALUES ('stock','DMPI','us','2017-11-01',0.9,0.9,0.85,0.8503,592134,'0');
INSERT INTO daily_summary VALUES ('stock','DMRC','us','2017-11-01',36.15,36.15,34.65,35.3,72692,'0');
INSERT INTO daily_summary VALUES ('stock','DMTX','us','2017-11-01',6,6,5.95,5.975,75955,'0');
INSERT INTO daily_summary VALUES ('stock','DNB','us','2017-11-01',117.39,117.39,115.69,115.94,299689,'0');
INSERT INTO daily_summary VALUES ('stock','DNBF','us','2017-11-01',32.9,33.1,32.9,32.9,3465,'0');
INSERT INTO daily_summary VALUES ('stock','DNI','us','2017-11-01',13.12,13.16,13.12,13.145,24537,'0');
INSERT INTO daily_summary VALUES ('stock','DNKN','us','2017-11-01',59.35,59.7,57.81,58.76,2115080,'0');
INSERT INTO daily_summary VALUES ('etf','DNL','us','2017-11-01',58.0992,58.0992,57.811,57.811,3285,'0');
INSERT INTO daily_summary VALUES ('stock','DNN','us','2017-11-01',0.4165,0.4165,0.4021,0.4045,230686,'0');
INSERT INTO daily_summary VALUES ('etf','DNO','us','2017-11-01',62.23,63.16,62.1098,63.025,10436,'0');
INSERT INTO daily_summary VALUES ('stock','DNOW','us','2017-11-01',12.5,12.54,10.6525,11.15,6198480,'0');
INSERT INTO daily_summary VALUES ('stock','DNP','us','2017-11-01',11.24,11.2652,11.19,11.23,279751,'0');
INSERT INTO daily_summary VALUES ('stock','DNR','us','2017-11-01',1.27,1.3,1.23,1.26,6758310,'0');
INSERT INTO daily_summary VALUES ('stock','DO','us','2017-11-01',17.2,17.52,16.5333,16.74,3605500,'0');
INSERT INTO daily_summary VALUES ('stock','DOC','us','2017-11-01',17.35,17.44,17.27,17.38,1138730,'0');
INSERT INTO daily_summary VALUES ('etf','DOD','us','2017-11-01',21.8,21.8,21.4508,21.5573,4666,'0');
INSERT INTO daily_summary VALUES ('etf','DOG','us','2017-11-01',15.78,15.86,15.77,15.83,592039,'0');
INSERT INTO daily_summary VALUES ('etf','DOL','us','2017-11-01',50.19,50.2738,50.1,50.1,13123,'0');
INSERT INTO daily_summary VALUES ('etf','DON','us','2017-11-01',33.787,33.817,33.58,33.633,190050,'0');
INSERT INTO daily_summary VALUES ('etf','DOO','us','2017-11-01',43.38,43.381,43.24,43.24,4911,'0');
INSERT INTO daily_summary VALUES ('stock','DOOR','us','2017-11-01',67.65,67.65,66.15,66.95,104417,'0');
INSERT INTO daily_summary VALUES ('stock','DORM','us','2017-11-01',69.39,70.37,68.08,68.92,173867,'0');
INSERT INTO daily_summary VALUES ('stock','DOTAU','us','2017-11-01',10.42,10.42,10.42,10.42,100,'0');
INSERT INTO daily_summary VALUES ('stock','DOV','us','2017-11-01',96.31,96.4,94.51,94.54,1002510,'0');
INSERT INTO daily_summary VALUES ('stock','DOVA','us','2017-11-01',25.12,25.2,24.01,24.99,81881,'0');
INSERT INTO daily_summary VALUES ('stock','DOX','us','2017-11-01',65.2,66.985,64.19,64.37,936911,'0');
INSERT INTO daily_summary VALUES ('stock','DPG','us','2017-11-01',15.85,15.87,15.73,15.85,148833,'0');
INSERT INTO daily_summary VALUES ('etf','DPK','us','2017-11-01',12.98,13.04,12.9292,13.04,3462,'0');
INSERT INTO daily_summary VALUES ('stock','DPLO','us','2017-11-01',21.15,21.1678,20.39,20.76,488458,'0');
INSERT INTO daily_summary VALUES ('stock','DPS','us','2017-11-01',86.02,86.66,85.57,85.89,1341180,'0');
INSERT INTO daily_summary VALUES ('stock','DPST','us','2017-11-01',68.5,68.9647,65.43,66.1,15159,'0');
INSERT INTO daily_summary VALUES ('stock','DPW','us','2017-11-01',0.62,0.6289,0.5825,0.6,605519,'0');
INSERT INTO daily_summary VALUES ('stock','DPZ','us','2017-11-01',182.32,182.41,176.07,177.25,1523130,'0');
INSERT INTO daily_summary VALUES ('stock','DQ','us','2017-11-01',38.15,40.39,38.09,40.24,228545,'0');
INSERT INTO daily_summary VALUES ('stock','DRAD','us','2017-11-01',2,2.05,1.95,2,215507,'0');
INSERT INTO daily_summary VALUES ('stock','DRD','us','2017-11-01',3.42,3.47,3.38,3.39,39260,'0');
INSERT INTO daily_summary VALUES ('stock','DRE','us','2017-11-01',28.64,28.79,28.44,28.71,1681280,'0');
INSERT INTO daily_summary VALUES ('stock','DRH','us','2017-11-01',10.98,11.01,10.835,10.88,1141290,'0');
INSERT INTO daily_summary VALUES ('stock','DRI','us','2017-11-01',82.37,82.71,81.74,82.08,1262400,'0');
INSERT INTO daily_summary VALUES ('stock','DRIO','us','2017-11-01',1.79,1.79,1.61,1.64,35747,'0');
INSERT INTO daily_summary VALUES ('etf','DRIP','us','2017-11-01',16.01,16.105,14.78,15.01,2018210,'0');
INSERT INTO daily_summary VALUES ('etf','DRN','us','2017-11-01',21.88,22.14,21.7057,21.99,126654,'0');
INSERT INTO daily_summary VALUES ('stock','DRNA','us','2017-11-01',5.52,5.8,5.43,5.64,105383,'0');
INSERT INTO daily_summary VALUES ('stock','DRQ','us','2017-11-01',42.65,43.6,42.45,43,1038870,'0');
INSERT INTO daily_summary VALUES ('stock','DRRX','us','2017-11-01',0.8017,0.8099,0.7762,0.79,395463,'0');
INSERT INTO daily_summary VALUES ('stock','DRUA','us','2017-11-01',25.3,25.49,25.3,25.42,32744,'0');
INSERT INTO daily_summary VALUES ('etf','DRV','us','2017-11-01',11.24,11.29,11.07,11.14,19186,'0');
INSERT INTO daily_summary VALUES ('etf','DRW','us','2017-11-01',31.719,31.75,31.645,31.67,12845,'0');
INSERT INTO daily_summary VALUES ('stock','DRYS','us','2017-11-01',4.17,4.2808,3.731,3.89,5790070,'0');
INSERT INTO daily_summary VALUES ('stock','DS','us','2017-11-01',3.62,3.72,3.54,3.61,218945,'0');
INSERT INTO daily_summary VALUES ('stock','DSE','us','2017-11-01',5.5731,5.679,5.5731,5.6693,264209,'0');
INSERT INTO daily_summary VALUES ('stock','DSGX','us','2017-11-01',29.25,29.25,28.45,28.6,124118,'0');
INSERT INTO daily_summary VALUES ('etf','DSI','us','2017-11-01',95.53,95.68,95.249,95.37,73461,'0');
INSERT INTO daily_summary VALUES ('stock','DSKE','us','2017-11-01',13.88,13.88,13.46,13.59,89172,'0');
INSERT INTO daily_summary VALUES ('stock','DSKEW','us','2017-11-01',1.8,1.81,1.7202,1.73,57000,'0');
INSERT INTO daily_summary VALUES ('stock','DSL','us','2017-11-01',21,21,20.88,20.89,332809,'0');
INSERT INTO daily_summary VALUES ('etf','DSLV','us','2017-11-01',24.53,24.576,23.11,23.34,500423,'0');
INSERT INTO daily_summary VALUES ('stock','DSM','us','2017-11-01',8.3701,8.4,8.3602,8.3602,100206,'0');
INSERT INTO daily_summary VALUES ('stock','DSPG','us','2017-11-01',13.4,13.6,12.875,13.15,68119,'0');
INSERT INTO daily_summary VALUES ('stock','DSS','us','2017-11-01',0.6701,0.6702,0.65,0.66,4368,'0');
INSERT INTO daily_summary VALUES ('stock','DST','us','2017-11-01',58.9,59.43,58.54,59.18,387366,'0');
INSERT INTO daily_summary VALUES ('stock','DSU','us','2017-11-01',11.77,11.77,11.74,11.77,96346,'0');
INSERT INTO daily_summary VALUES ('etf','DSUM','us','2017-11-01',23.27,23.31,23.2174,23.3049,14550,'0');
INSERT INTO daily_summary VALUES ('stock','DSW','us','2017-11-01',18.61,18.85,17.7501,18.6,5016430,'0');
INSERT INTO daily_summary VALUES ('stock','DSWL','us','2017-11-01',2.72,2.72,2.69,2.72,7659,'0');
INSERT INTO daily_summary VALUES ('stock','DSX','us','2017-11-01',4.04,4.13,3.95,3.99,456876,'0');
INSERT INTO daily_summary VALUES ('stock','DSXN','us','2017-11-01',25.3,25.4402,25.13,25.26,11012,'0');
INSERT INTO daily_summary VALUES ('stock','DSX_B','us','2017-11-01',24.25,24.25,24.15,24.15,3187,'0');
INSERT INTO daily_summary VALUES ('stock','DS_B','us','2017-11-01',25.84,25.88,25.8325,25.8485,2994,'0');
INSERT INTO daily_summary VALUES ('stock','DS_C','us','2017-11-01',25.496,25.496,25.496,25.496,213,'0');
INSERT INTO daily_summary VALUES ('stock','DS_D','us','2017-11-01',25.203,25.2199,25.203,25.2199,540,'0');
INSERT INTO daily_summary VALUES ('etf','DTD','us','2017-11-01',89.3894,89.3894,89.0288,89.1192,3775,'0');
INSERT INTO daily_summary VALUES ('stock','DTE','us','2017-11-01',110.94,111.61,109.58,110.28,1301740,'0');
INSERT INTO daily_summary VALUES ('stock','DTEA','us','2017-11-01',4.25,4.25,4.2,4.25,2423,'0');
INSERT INTO daily_summary VALUES ('stock','DTF','us','2017-11-01',14.17,14.19,14.163,14.19,25495,'0');
INSERT INTO daily_summary VALUES ('etf','DTH','us','2017-11-01',44.135,44.135,43.979,44.09,38593,'0');
INSERT INTO daily_summary VALUES ('stock','DTJ','us','2017-11-01',25.6,25.93,25.6,25.91,3725,'0');
INSERT INTO daily_summary VALUES ('stock','DTK','us','2017-11-01',26.05,26.08,26,26.01,94824,'0');
INSERT INTO daily_summary VALUES ('stock','DTLA_','us','2017-11-01',28.85,28.85,28.85,28.85,219,'0');
INSERT INTO daily_summary VALUES ('etf','DTN','us','2017-11-01',85.42,85.42,85.01,85.21,11335,'0');
INSERT INTO daily_summary VALUES ('etf','DTO','us','2017-11-01',106.66,110.48,105.791,109.24,8923,'0');
INSERT INTO daily_summary VALUES ('stock','DTQ','us','2017-11-01',25.25,25.2783,25.225,25.264,6227,'0');
INSERT INTO daily_summary VALUES ('stock','DTRM','us','2017-11-01',2.15,2.15,1.97,2.11,15313,'0');
INSERT INTO daily_summary VALUES ('stock','DTV','us','2017-11-01',55.24,55.4133,55.11,55.18,1636,'0');
INSERT INTO daily_summary VALUES ('stock','DTY','us','2017-11-01',27.29,27.294,27.128,27.22,6503,'0');
INSERT INTO daily_summary VALUES ('stock','DTYS','us','2017-11-01',17.5,17.58,17.2,17.45,12640,'0');
INSERT INTO daily_summary VALUES ('stock','DUC','us','2017-11-01',9.13,9.2,9.13,9.14,101970,'0');
INSERT INTO daily_summary VALUES ('etf','DUG','us','2017-11-01',42.4,42.4,41.44,41.81,68413,'0');
INSERT INTO daily_summary VALUES ('stock','DUK','us','2017-11-01',88.56,88.56,87.56,87.86,2024610,'0');
INSERT INTO daily_summary VALUES ('stock','DUKH','us','2017-11-01',25.435,25.52,25.421,25.46,7524,'0');
INSERT INTO daily_summary VALUES ('stock','DUSA','us','2017-11-01',22.7601,22.77,22.67,22.705,18804,'0');
INSERT INTO daily_summary VALUES ('stock','DUSL','us','2017-11-01',31.2,31.2,30.81,30.81,32608,'0');
INSERT INTO daily_summary VALUES ('etf','DUST','us','2017-11-01',26.71,27.56,26.07,27.26,6345330,'0');
INSERT INTO daily_summary VALUES ('stock','DVA','us','2017-11-01',60.85,61.12,59.32,59.83,2528560,'0');
INSERT INTO daily_summary VALUES ('stock','DVAX','us','2017-11-01',22.15,22.2,20.55,21,1761160,'0');
INSERT INTO daily_summary VALUES ('stock','DVCR','us','2017-11-01',11.86,12.06,11.78,11.7815,8803,'0');
INSERT INTO daily_summary VALUES ('stock','DVD','us','2017-11-01',1.87,2.0138,1.8221,1.9659,47160,'0');
INSERT INTO daily_summary VALUES ('stock','DVEM','us','2017-11-01',32.63,32.63,32.3926,32.3926,5644,'0');
INSERT INTO daily_summary VALUES ('etf','DVHL','us','2017-11-01',20.05,20.05,19.98,19.99,2981,'0');
INSERT INTO daily_summary VALUES ('stock','DVMT','us','2017-11-01',83.32,83.32,82.04,82.4,1113030,'0');
INSERT INTO daily_summary VALUES ('stock','DVN','us','2017-11-01',37.75,39.34,37.59,39.08,9302990,'0');
INSERT INTO daily_summary VALUES ('stock','DVP','us','2017-11-01',28.47,28.47,28.47,28.47,200,'0');
INSERT INTO daily_summary VALUES ('etf','DVY','us','2017-11-01',95.02,95.235,94.53,94.6,449908,'0');
INSERT INTO daily_summary VALUES ('etf','DVYA','us','2017-11-01',48.94,48.94,48.75,48.8757,3400,'0');
INSERT INTO daily_summary VALUES ('etf','DVYE','us','2017-11-01',41.86,41.92,41.62,41.675,32807,'0');
INSERT INTO daily_summary VALUES ('etf','DVYL','us','2017-11-01',67.55,67.636,67.031,67.081,4019,'0');
INSERT INTO daily_summary VALUES ('stock','DWAQ','us','2017-11-01',102.67,102.67,100.79,101.27,4707,'0');
INSERT INTO daily_summary VALUES ('stock','DWAS','us','2017-11-01',48.3048,48.4077,47.07,47.38,23796,'0');
INSERT INTO daily_summary VALUES ('stock','DWCH','us','2017-11-01',12.6,12.6,11.65,11.8,176852,'0');
INSERT INTO daily_summary VALUES ('stock','DWDP','us','2017-11-01',73.11,73.58,72.32,73.32,7968370,'0');
INSERT INTO daily_summary VALUES ('stock','DWFI','us','2017-11-01',25.59,25.59,25.5,25.55,16475,'0');
INSERT INTO daily_summary VALUES ('stock','DWIN','us','2017-11-01',27.09,27.29,27.075,27.2,10735,'0');
INSERT INTO daily_summary VALUES ('stock','DWLD','us','2017-11-01',25.29,25.399,25.2732,25.32,7822,'0');
INSERT INTO daily_summary VALUES ('stock','DWLV','us','2017-11-01',30.415,30.415,30.415,30.415,163,'0');
INSERT INTO daily_summary VALUES ('etf','DWM','us','2017-11-01',55.18,55.23,54.9801,55.035,23709,'0');
INSERT INTO daily_summary VALUES ('stock','DWPP','us','2017-11-01',27.78,27.78,27.4603,27.4603,4526,'0');
INSERT INTO daily_summary VALUES ('stock','DWSN','us','2017-11-01',4.5927,4.5927,4.391,4.53,13720,'0');
INSERT INTO daily_summary VALUES ('stock','DWT','us','2017-11-01',18.95,20.0702,18.8,19.64,3644440,'0');
INSERT INTO daily_summary VALUES ('stock','DWTR','us','2017-11-01',28.135,28.15,27.83,27.92,7281,'0');
INSERT INTO daily_summary VALUES ('etf','DWX','us','2017-11-01',40.42,40.4462,40.2679,40.27,108590,'0');
INSERT INTO daily_summary VALUES ('stock','DX','us','2017-11-01',7,7.18,6.99,7.15,467563,'0');
INSERT INTO daily_summary VALUES ('stock','DXB','us','2017-11-01',25.73,25.79,25.73,25.771,89786,'0');
INSERT INTO daily_summary VALUES ('stock','DXC','us','2017-11-01',91.99,92.53,90.89,91.5,1145780,'0');
INSERT INTO daily_summary VALUES ('stock','DXCM','us','2017-11-01',45.12,45.63,44.5,44.86,2173980,'0');
INSERT INTO daily_summary VALUES ('etf','DXD','us','2017-11-01',9.73,9.82,9.71,9.78,576671,'0');
INSERT INTO daily_summary VALUES ('etf','DXGE','us','2017-11-01',33,33.02,32.9301,32.97,20897,'0');
INSERT INTO daily_summary VALUES ('etf','DXJ','us','2017-11-01',58.74,58.79,58.46,58.67,4804210,'0');
INSERT INTO daily_summary VALUES ('etf','DXJC','us','2017-11-01',28.659,28.669,28.6436,28.6436,16775,'0');
INSERT INTO daily_summary VALUES ('etf','DXJF','us','2017-11-01',25.7,25.71,25.5957,25.6736,5168,'0');
INSERT INTO daily_summary VALUES ('etf','DXJH','us','2017-11-01',36.0701,36.0705,36.0701,36.0705,202,'0');
INSERT INTO daily_summary VALUES ('etf','DXJR','us','2017-11-01',15.451,15.55,15.451,15.54,6679,'0');
INSERT INTO daily_summary VALUES ('etf','DXJS','us','2017-11-01',45.25,45.25,44.86,45.01,112609,'0');
INSERT INTO daily_summary VALUES ('stock','DXLG','us','2017-11-01',2,2.05,1.95,1.95,50084,'0');
INSERT INTO daily_summary VALUES ('stock','DXPE','us','2017-11-01',32.43,32.435,30.68,30.8,71657,'0');
INSERT INTO daily_summary VALUES ('etf','DXPS','us','2017-11-01',24.28,24.2827,24.2,24.2,1067,'0');
INSERT INTO daily_summary VALUES ('stock','DXR','us','2017-11-01',4.1739,4.44,4.1739,4.34,3775,'0');
INSERT INTO daily_summary VALUES ('stock','DXTR','us','2017-11-01',0.1725,0.1859,0.1725,0.1813,1177160,'0');
INSERT INTO daily_summary VALUES ('stock','DXUS','us','2017-11-01',27.0235,27.0235,27.0235,27.0235,550,'0');
INSERT INTO daily_summary VALUES ('stock','DXYN','us','2017-11-01',3.9,3.9,3.8,3.85,4022,'0');
INSERT INTO daily_summary VALUES ('stock','DX_A','us','2017-11-01',25.42,25.48,25.42,25.441,2658,'0');
INSERT INTO daily_summary VALUES ('stock','DX_B','us','2017-11-01',24.84,24.84,24.8,24.8252,9619,'0');
INSERT INTO daily_summary VALUES ('stock','DY','us','2017-11-01',88.48,89.31,87.08,87.71,492114,'0');
INSERT INTO daily_summary VALUES ('stock','DYB','us','2017-11-01',25.26,25.26,25.26,25.26,175,'0');
INSERT INTO daily_summary VALUES ('stock','DYLS','us','2017-11-01',31.1596,31.1596,30.969,31.04,28154,'0');
INSERT INTO daily_summary VALUES ('stock','DYN','us','2017-11-01',12.5,12.59,12.22,12.34,9091020,'0');
INSERT INTO daily_summary VALUES ('stock','DYNC','us','2017-11-01',82.9703,83,81.92,82.15,3113,'0');
INSERT INTO daily_summary VALUES ('stock','DYNT','us','2017-11-01',2.4982,2.7,2.4,2.4,34181,'0');
INSERT INTO daily_summary VALUES ('stock','DYSL','us','2017-11-01',1.36,1.36,1.27,1.31,79353,'0');
INSERT INTO daily_summary VALUES ('etf','DYY','us','2017-11-01',2.1094,2.1094,2.1094,2.1094,554,'0');
INSERT INTO daily_summary VALUES ('etf','DZK','us','2017-11-01',80.17,80.33,79.6747,79.6747,16602,'0');
INSERT INTO daily_summary VALUES ('stock','DZSI','us','2017-11-01',6.5303,6.5303,6.4455,6.45,5657,'0');
INSERT INTO daily_summary VALUES ('etf','DZZ','us','2017-11-01',5.61,5.6201,5.57,5.6,31798,'0');
INSERT INTO daily_summary VALUES ('stock','E','us','2017-11-01',33.01,33.21,32.9,32.99,190958,'0');
INSERT INTO daily_summary VALUES ('stock','EA','us','2017-11-01',116,116.6,111.28,114.47,9669210,'0');
INSERT INTO daily_summary VALUES ('stock','EAB','us','2017-11-01',25.14,25.14,24.8605,24.94,6345,'0');
INSERT INTO daily_summary VALUES ('stock','EACQ','us','2017-11-01',10.05,10.05,10.05,10.05,503,'0');
INSERT INTO daily_summary VALUES ('stock','EAD','us','2017-11-01',8.55,8.58,8.54,8.58,189234,'0');
INSERT INTO daily_summary VALUES ('stock','EAE','us','2017-11-01',24.8512,24.8988,24.7701,24.8121,5806,'0');
INSERT INTO daily_summary VALUES ('stock','EAGL','us','2017-11-01',10.075,10.1,10.05,10.075,1022910,'0');
INSERT INTO daily_summary VALUES ('stock','EAGLU','us','2017-11-01',10.69,10.69,10.69,10.69,525,'0');
INSERT INTO daily_summary VALUES ('stock','EAGLW','us','2017-11-01',0.7,0.7,0.68,0.7,302400,'0');
INSERT INTO daily_summary VALUES ('stock','EAI','us','2017-11-01',25,25.0248,25,25.01,23024,'0');
INSERT INTO daily_summary VALUES ('stock','EARN','us','2017-11-01',13.4,13.5,13.35,13.41,76693,'0');
INSERT INTO daily_summary VALUES ('stock','EARS','us','2017-11-01',0.7754,0.78,0.7301,0.75,213337,'0');
INSERT INTO daily_summary VALUES ('stock','EAT','us','2017-11-01',29.99,33.4,29.99,32.9,7081490,'0');
INSERT INTO daily_summary VALUES ('stock','EBAY','us','2017-11-01',37.85,38.09,37.435,37.54,7092870,'0');
INSERT INTO daily_summary VALUES ('stock','EBAYL','us','2017-11-01',27.1,27.1,26.9634,27,29692,'0');
INSERT INTO daily_summary VALUES ('stock','EBF','us','2017-11-01',20.35,20.35,19.5,19.6,79076,'0');
INSERT INTO daily_summary VALUES ('stock','EBIO','us','2017-11-01',0.88,0.9,0.69,0.74,3324330,'0');
INSERT INTO daily_summary VALUES ('stock','EBIX','us','2017-11-01',68,68.3,67.5,67.55,162773,'0');
INSERT INTO daily_summary VALUES ('stock','EBMT','us','2017-11-01',19.611,19.86,19.611,19.65,3031,'0');
INSERT INTO daily_summary VALUES ('etf','EBND','us','2017-11-01',28.98,29.14,28.98,29.07,167639,'0');
INSERT INTO daily_summary VALUES ('stock','EBR','us','2017-11-01',6.33,6.44,6.2,6.29,465374,'0');
INSERT INTO daily_summary VALUES ('stock','EBS','us','2017-11-01',41.3,41.32,39.97,40.2,231103,'0');
INSERT INTO daily_summary VALUES ('stock','EBSB','us','2017-11-01',19.95,19.95,19.425,19.55,151023,'0');
INSERT INTO daily_summary VALUES ('stock','EBTC','us','2017-11-01',36.35,36.35,35.145,35.145,3409,'0');
INSERT INTO daily_summary VALUES ('stock','EC','us','2017-11-01',11.23,11.43,11.16,11.39,2920010,'0');
INSERT INTO daily_summary VALUES ('stock','ECA','us','2017-11-01',11.89,12.31,11.88,12.13,13744600,'0');
INSERT INTO daily_summary VALUES ('stock','ECC','us','2017-11-01',19.159,19.721,19.088,19.209,96963,'0');
INSERT INTO daily_summary VALUES ('stock','ECCA','us','2017-11-01',25.71,25.799,25.7,25.701,1903,'0');
INSERT INTO daily_summary VALUES ('stock','ECCB','us','2017-11-01',26.0801,26.19,26.0801,26.19,602,'0');
INSERT INTO daily_summary VALUES ('stock','ECF','us','2017-11-01',9.1711,9.2601,9.1514,9.2503,24411,'0');
INSERT INTO daily_summary VALUES ('stock','ECF_A','us','2017-11-01',24.8,24.86,24.8,24.85,15348,'0');
INSERT INTO daily_summary VALUES ('etf','ECH','us','2017-11-01',51.245,51.57,51.21,51.285,518129,'0');
INSERT INTO daily_summary VALUES ('stock','ECHO','us','2017-11-01',24.45,24.45,23.575,24.1,659312,'0');
INSERT INTO daily_summary VALUES ('stock','ECL','us','2017-11-01',130.09,131.72,129.16,131.27,1269390,'0');
INSERT INTO daily_summary VALUES ('etf','ECNS','us','2017-11-01',51.9,52.209,51.9,52.1104,2553,'0');
INSERT INTO daily_summary VALUES ('stock','ECOL','us','2017-11-01',48.05,48.05,46.45,47.05,116633,'0');
INSERT INTO daily_summary VALUES ('stock','ECOM','us','2017-11-01',11.25,11.35,10.95,10.95,45209,'0');
INSERT INTO daily_summary VALUES ('etf','ECON','us','2017-11-01',27.57,27.64,27.36,27.36,51751,'0');
INSERT INTO daily_summary VALUES ('stock','ECPG','us','2017-11-01',46.9,47.1,45.3,45.8,201873,'0');
INSERT INTO daily_summary VALUES ('stock','ECR','us','2017-11-01',2.27,2.365,2.25,2.27,510183,'0');
INSERT INTO daily_summary VALUES ('stock','ECT','us','2017-11-01',2.35,2.45,2.35,2.45,32712,'0');
INSERT INTO daily_summary VALUES ('stock','ECYT','us','2017-11-01',4.47,4.87,4.4508,4.75,744229,'0');
INSERT INTO daily_summary VALUES ('stock','ED','us','2017-11-01',85.85,86.53,85.27,85.8,1305560,'0');
INSERT INTO daily_summary VALUES ('stock','EDAP','us','2017-11-01',3.05,3.0599,2.95,3.02,29819,'0');
INSERT INTO daily_summary VALUES ('stock','EDBI','us','2017-11-01',32.7794,32.7794,32.7794,32.7794,369,'0');
INSERT INTO daily_summary VALUES ('etf','EDC','us','2017-11-01',118.86,120.001,117.42,117.63,153476,'0');
INSERT INTO daily_summary VALUES ('stock','EDD','us','2017-11-01',7.85,7.88,7.8008,7.86,131796,'0');
INSERT INTO daily_summary VALUES ('etf','EDEN','us','2017-11-01',68.24,68.314,67.99,68.02,24321,'0');
INSERT INTO daily_summary VALUES ('stock','EDF','us','2017-11-01',16.72,16.9,16.72,16.8,92557,'0');
INSERT INTO daily_summary VALUES ('stock','EDGE','us','2017-11-01',11.4,11.4,10.7,10.73,65766,'0');
INSERT INTO daily_summary VALUES ('stock','EDGW','us','2017-11-01',6.5,6.5149,6.5,6.5,2464,'0');
INSERT INTO daily_summary VALUES ('stock','EDI','us','2017-11-01',16.29,16.3,16.15,16.3,58638,'0');
INSERT INTO daily_summary VALUES ('stock','EDIT','us','2017-11-01',24.97,25.19,22.8485,23.37,960587,'0');
INSERT INTO daily_summary VALUES ('etf','EDIV','us','2017-11-01',30.46,30.534,30.4491,30.5,224171,'0');
INSERT INTO daily_summary VALUES ('stock','EDN','us','2017-11-01',43.1,45.4999,42.1,44.68,95973,'0');
INSERT INTO daily_summary VALUES ('etf','EDOG','us','2017-11-01',24.7568,24.815,24.65,24.65,7719,'0');
INSERT INTO daily_summary VALUES ('stock','EDOM','us','2017-11-01',30.84,30.84,30.7847,30.7847,201,'0');
INSERT INTO daily_summary VALUES ('stock','EDOW','us','2017-11-01',21.07,21.08,21.05,21.05,4755,'0');
INSERT INTO daily_summary VALUES ('stock','EDR','us','2017-11-01',34.83,35.17,34.74,34.99,759904,'0');
INSERT INTO daily_summary VALUES ('stock','EDU','us','2017-11-01',83.53,84.39,81.45,82.46,2349180,'0');
INSERT INTO daily_summary VALUES ('stock','EDUC','us','2017-11-01',12.5,12.5,12.45,12.45,1878,'0');
INSERT INTO daily_summary VALUES ('etf','EDV','us','2017-11-01',117.25,118.41,117.12,117.73,63312,'0');
INSERT INTO daily_summary VALUES ('etf','EDZ','us','2017-11-01',9.84,9.97,9.745,9.95,582757,'0');
INSERT INTO daily_summary VALUES ('stock','EE','us','2017-11-01',55.6,57.2,55.45,56.2,348248,'0');
INSERT INTO daily_summary VALUES ('stock','EEA','us','2017-11-01',9.83,9.835,9.815,9.8325,8920,'0');
INSERT INTO daily_summary VALUES ('etf','EEB','us','2017-11-01',37.77,37.856,37.55,37.55,14079,'0');
INSERT INTO daily_summary VALUES ('stock','EEFT','us','2017-11-01',97.11,98.54,96.39,97.19,372752,'0');
INSERT INTO daily_summary VALUES ('stock','EEI','us','2017-11-01',11.65,11.75,11.65,11.75,1977,'0');
INSERT INTO daily_summary VALUES ('etf','EELV','us','2017-11-01',24.7,24.73,24.6,24.6,59389,'0');
INSERT INTO daily_summary VALUES ('etf','EEM','us','2017-11-01',46.7,46.8499,46.49,46.52,51925600,'0');
INSERT INTO daily_summary VALUES ('etf','EEMA','us','2017-11-01',73.88,74.03,73.53,73.53,8270,'0');
INSERT INTO daily_summary VALUES ('stock','EEMO','us','2017-11-01',21.23,21.38,21.17,21.17,92906,'0');
INSERT INTO daily_summary VALUES ('etf','EEMS','us','2017-11-01',51.4825,51.57,51.241,51.32,8990,'0');
INSERT INTO daily_summary VALUES ('etf','EEMV','us','2017-11-01',59.54,59.58,59.3,59.37,836773,'0');
INSERT INTO daily_summary VALUES ('stock','EEMX','us','2017-11-01',68.4951,68.4951,68.4951,68.4951,238,'0');
INSERT INTO daily_summary VALUES ('stock','EEP','us','2017-11-01',14.804,15.187,14.769,15.048,1372530,'0');
INSERT INTO daily_summary VALUES ('stock','EEQ','us','2017-11-01',14.46,14.74,14.46,14.6,281940,'0');
INSERT INTO daily_summary VALUES ('etf','EES','us','2017-11-01',35.569,35.62,34.99,35.21,47808,'0');
INSERT INTO daily_summary VALUES ('etf','EET','us','2017-11-01',90.14,90.6404,89.5276,89.62,37855,'0');
INSERT INTO daily_summary VALUES ('etf','EEV','us','2017-11-01',8.86,8.91,8.78,8.91,9788,'0');
INSERT INTO daily_summary VALUES ('stock','EEX','us','2017-11-01',23.41,23.45,23.06,23.15,98200,'0');
INSERT INTO daily_summary VALUES ('etf','EFA','us','2017-11-01',69.93,69.995,69.67,69.7,23674500,'0');
INSERT INTO daily_summary VALUES ('etf','EFAD','us','2017-11-01',37.99,38.03,37.85,37.87,7075,'0');
INSERT INTO daily_summary VALUES ('stock','EFAS','us','2017-11-01',18.22,18.22,18.21,18.21,1146,'0');
INSERT INTO daily_summary VALUES ('etf','EFAV','us','2017-11-01',71.92,71.96,71.66,71.66,578108,'0');
INSERT INTO daily_summary VALUES ('stock','EFBI','us','2017-11-01',17.75,17.75,17.1072,17.75,5812,'0');
INSERT INTO daily_summary VALUES ('stock','EFC','us','2017-11-01',15.28,15.3585,15.1,15.34,168512,'0');
INSERT INTO daily_summary VALUES ('stock','EFF','us','2017-11-01',16.58,16.58,16.36,16.41,25128,'0');
INSERT INTO daily_summary VALUES ('etf','EFG','us','2017-11-01',79.62,79.71,79.25,79.35,230438,'0');
INSERT INTO daily_summary VALUES ('stock','EFII','us','2017-11-01',31.28,31.35,30.25,30.96,937618,'0');
INSERT INTO daily_summary VALUES ('stock','EFL','us','2017-11-01',9.85,9.85,9.75,9.77,39120,'0');
INSERT INTO daily_summary VALUES ('stock','EFNL','us','2017-11-01',39.71,39.72,39.55,39.56,3348,'0');
INSERT INTO daily_summary VALUES ('etf','EFO','us','2017-11-01',129.14,129.14,128.399,128.54,12592,'0');
INSERT INTO daily_summary VALUES ('stock','EFOI','us','2017-11-01',2.6,2.65,2.56,2.65,5578,'0');
INSERT INTO daily_summary VALUES ('stock','EFR','us','2017-11-01',14.56,14.59,14.51,14.54,89685,'0');
INSERT INTO daily_summary VALUES ('stock','EFSC','us','2017-11-01',44.05,44.3,43.05,43.65,84104,'0');
INSERT INTO daily_summary VALUES ('stock','EFT','us','2017-11-01',14.5,14.5,14.47,14.47,62843,'0');
INSERT INTO daily_summary VALUES ('etf','EFU','us','2017-11-01',23.79,23.79,23.79,23.79,0,'0');
INSERT INTO daily_summary VALUES ('etf','EFV','us','2017-11-01',55.22,55.32,55.1,55.15,472239,'0');
INSERT INTO daily_summary VALUES ('stock','EFX','us','2017-11-01',109.12,111.29,108.885,109.8,1438880,'0');
INSERT INTO daily_summary VALUES ('etf','EFZ','us','2017-11-01',25.83,25.93,25.83,25.92,16737,'0');
INSERT INTO daily_summary VALUES ('stock','EGAN','us','2017-11-01',3.35,3.45,3.35,3.4,143104,'0');
INSERT INTO daily_summary VALUES ('stock','EGBN','us','2017-11-01',67.05,67.05,65.9,66.25,178062,'0');
INSERT INTO daily_summary VALUES ('stock','EGF','us','2017-11-01',13.48,13.56,13.47,13.56,10607,'0');
INSERT INTO daily_summary VALUES ('stock','EGHT','us','2017-11-01',13.35,13.65,13.3,13.4,1115410,'0');
INSERT INTO daily_summary VALUES ('stock','EGI','us','2017-11-01',0.434411,0.4436,0.4321,0.4436,79468,'0');
INSERT INTO daily_summary VALUES ('stock','EGIF','us','2017-11-01',16.66,16.6954,16.58,16.58,18072,'0');
INSERT INTO daily_summary VALUES ('stock','EGL','us','2017-11-01',34.03,34.03,32.83,33.41,122084,'0');
INSERT INTO daily_summary VALUES ('stock','EGLE','us','2017-11-01',4.69,4.69,4.453,4.48,745091,'0');
INSERT INTO daily_summary VALUES ('stock','EGLT','us','2017-11-01',0.99,0.99,0.92,0.939,758848,'0');
INSERT INTO daily_summary VALUES ('stock','EGN','us','2017-11-01',52.24,54.08,51.98,53.85,1471220,'0');
INSERT INTO daily_summary VALUES ('stock','EGO','us','2017-11-01',1.28,1.28,1.24,1.25,7244770,'0');
INSERT INTO daily_summary VALUES ('stock','EGOV','us','2017-11-01',17.1,17.1,16.275,16.45,437398,'0');
INSERT INTO daily_summary VALUES ('stock','EGP','us','2017-11-01',90.99,91.69,90.69,90.75,151633,'0');
INSERT INTO daily_summary VALUES ('etf','EGPT','us','2017-11-01',31.8356,31.87,31.25,31.33,10518,'0');
INSERT INTO daily_summary VALUES ('stock','EGRX','us','2017-11-01',54.15,54.245,52.78,53.58,191705,'0');
INSERT INTO daily_summary VALUES ('stock','EGY','us','2017-11-01',0.8,0.88,0.7927,0.877,524878,'0');
INSERT INTO daily_summary VALUES ('stock','EHI','us','2017-11-01',10.22,10.24,10.2,10.23,111619,'0');
INSERT INTO daily_summary VALUES ('stock','EHIC','us','2017-11-01',11.56,11.56,11.24,11.46,155728,'0');
INSERT INTO daily_summary VALUES ('stock','EHT','us','2017-11-01',10.06,10.08,10.021,10.07,46861,'0');
INSERT INTO daily_summary VALUES ('stock','EHTH','us','2017-11-01',25.38,28.59,24.98,27.82,356860,'0');
INSERT INTO daily_summary VALUES ('stock','EIA','us','2017-11-01',11.97,11.97,11.9,11.91,5351,'0');
INSERT INTO daily_summary VALUES ('etf','EIDO','us','2017-11-01',26.78,26.965,26.78,26.83,521779,'0');
INSERT INTO daily_summary VALUES ('stock','EIG','us','2017-11-01',48.05,48.15,47.203,47.651,120194,'0');
INSERT INTO daily_summary VALUES ('stock','EIGI','us','2017-11-01',8.2,8.4,8.2,8.35,406307,'0');
INSERT INTO daily_summary VALUES ('stock','EIGR','us','2017-11-01',11.25,11.275,10.305,10.6,67813,'0');
INSERT INTO daily_summary VALUES ('stock','EIM','us','2017-11-01',12.55,12.55,12.45,12.55,169690,'0');
INSERT INTO daily_summary VALUES ('stock','EIO','us','2017-11-01',12.78,12.7801,12.78,12.7801,605,'0');
INSERT INTO daily_summary VALUES ('stock','EIP','us','2017-11-01',12.615,12.6315,12.5899,12.59,3217,'0');
INSERT INTO daily_summary VALUES ('etf','EIRL','us','2017-11-01',46.76,46.85,46.58,46.58,7114,'0');
INSERT INTO daily_summary VALUES ('etf','EIS','us','2017-11-01',48.97,49.1713,48.5785,48.71,7180,'0');
INSERT INTO daily_summary VALUES ('stock','EIV','us','2017-11-01',12.17,12.3,12.17,12.3,23773,'0');
INSERT INTO daily_summary VALUES ('stock','EIX','us','2017-11-01',80.12,80.12,78.995,79.08,1378570,'0');
INSERT INTO daily_summary VALUES ('stock','EKSO','us','2017-11-01',1.07,1.08,1.04,1.045,161246,'0');
INSERT INTO daily_summary VALUES ('stock','EL','us','2017-11-01',120.15,124.8,117.14,122.12,7620400,'0');
INSERT INTO daily_summary VALUES ('stock','ELC','us','2017-11-01',24.95,24.96,24.8606,24.95,20454,'0');
INSERT INTO daily_summary VALUES ('etf','ELD','us','2017-11-01',37.6,37.82,37.6,37.73,53890,'0');
INSERT INTO daily_summary VALUES ('stock','ELECU','us','2017-11-01',10.55,10.59,10.55,10.57,1800,'0');
INSERT INTO daily_summary VALUES ('stock','ELECW','us','2017-11-01',0.3998,0.3998,0.38,0.385,500,'0');
INSERT INTO daily_summary VALUES ('stock','ELF','us','2017-11-01',21.25,21.4751,20.5326,20.83,641742,'0');
INSERT INTO daily_summary VALUES ('stock','ELGX','us','2017-11-01',5.32,5.71,5.3,5.57,810554,'0');
INSERT INTO daily_summary VALUES ('stock','ELJ','us','2017-11-01',25.19,25.21,25.15,25.16,1966,'0');
INSERT INTO daily_summary VALUES ('stock','ELLI','us','2017-11-01',90.17,91,87.315,87.48,718565,'0');
INSERT INTO daily_summary VALUES ('stock','ELLO','us','2017-11-01',7.96,7.96,7.9,7.9,3900,'0');
INSERT INTO daily_summary VALUES ('stock','ELMD','us','2017-11-01',7.49,7.49,7.25,7.35,17488,'0');
INSERT INTO daily_summary VALUES ('stock','ELON','us','2017-11-01',4.6598,4.99,4.54,4.74,25225,'0');
INSERT INTO daily_summary VALUES ('stock','ELP','us','2017-11-01',7.69,7.8,7.58,7.61,711128,'0');
INSERT INTO daily_summary VALUES ('stock','ELS','us','2017-11-01',88.63,89.16,88.18,88.95,272851,'0');
INSERT INTO daily_summary VALUES ('stock','ELSE','us','2017-11-01',3.9145,3.9145,3.9145,3.9145,161,'0');
INSERT INTO daily_summary VALUES ('stock','ELTK','us','2017-11-01',0.69,0.69,0.6499,0.66,42572,'0');
INSERT INTO daily_summary VALUES ('stock','ELU','us','2017-11-01',24.85,24.9399,24.85,24.8887,5357,'0');
INSERT INTO daily_summary VALUES ('stock','ELVT','us','2017-11-01',7.85,7.85,7.43,7.62,209340,'0');
INSERT INTO daily_summary VALUES ('stock','ELY','us','2017-11-01',14.58,14.58,14.27,14.31,914874,'0');
INSERT INTO daily_summary VALUES ('etf','EMAG','us','2017-11-01',21.39,21.5911,21.39,21.5911,310,'0');
INSERT INTO daily_summary VALUES ('stock','EMAN','us','2017-11-01',2.2,2.225,2.15,2.15,22888,'0');
INSERT INTO daily_summary VALUES ('etf','EMB','us','2017-11-01',115.84,115.97,115.695,115.83,2355960,'0');
INSERT INTO daily_summary VALUES ('stock','EMBH','us','2017-11-01',26.383,26.393,26.363,26.363,6921,'0');
INSERT INTO daily_summary VALUES ('etf','EMCB','us','2017-11-01',72.866,73.1,72.7148,73.1,1488,'0');
INSERT INTO daily_summary VALUES ('etf','EMCG','us','2017-11-01',25.38,25.4,25.221,25.26,3551,'0');
INSERT INTO daily_summary VALUES ('stock','EMCI','us','2017-11-01',28.44,29.455,28.44,29.21,6639,'0');
INSERT INTO daily_summary VALUES ('stock','EMD','us','2017-11-01',15.57,15.62,15.56,15.62,158136,'0');
INSERT INTO daily_summary VALUES ('stock','EMDV','us','2017-11-01',55.39,55.39,55.109,55.11,2431,'0');
INSERT INTO daily_summary VALUES ('stock','EME','us','2017-11-01',81.42,81.6,80.36,80.52,434995,'0');
INSERT INTO daily_summary VALUES ('stock','EMES','us','2017-11-01',8,8.5,7.91,8.17,1482900,'0');
INSERT INTO daily_summary VALUES ('stock','EMF','us','2017-11-01',17.1,17.16,17.08,17.12,37235,'0');
INSERT INTO daily_summary VALUES ('etf','EMFM','us','2017-11-01',23.26,23.3724,23.26,23.3284,1859,'0');
INSERT INTO daily_summary VALUES ('stock','EMGF','us','2017-11-01',47.93,48.14,47.8406,47.8406,10523,'0');
INSERT INTO daily_summary VALUES ('stock','EMHY','us','2017-11-01',50.64,50.66,50.45,50.54,110765,'0');
INSERT INTO daily_summary VALUES ('stock','EMI','us','2017-11-01',12.86,12.96,12.86,12.94,14843,'0');
INSERT INTO daily_summary VALUES ('etf','EMIF','us','2017-11-01',33.92,33.92,33.55,33.55,12939,'0');
INSERT INTO daily_summary VALUES ('etf','EMIH','us','2017-11-01',25.2298,25.2298,25.2298,25.2298,310,'0');
INSERT INTO daily_summary VALUES ('stock','EMITF','us','2017-11-01',2.49,2.49,2.401,2.45,11902,'0');
INSERT INTO daily_summary VALUES ('stock','EMJ','us','2017-11-01',12.66,12.6701,12.65,12.65,1973,'0');
INSERT INTO daily_summary VALUES ('stock','EMKR','us','2017-11-01',8.35,8.35,7.85,7.9,134684,'0');
INSERT INTO daily_summary VALUES ('stock','EML','us','2017-11-01',29.3,29.475,28.85,28.85,4666,'0');
INSERT INTO daily_summary VALUES ('etf','EMLC','us','2017-11-01',18.47,18.54,18.47,18.5,1483560,'0');
INSERT INTO daily_summary VALUES ('etf','EMLP','us','2017-11-01',24.62,24.7,24.567,24.67,357651,'0');
INSERT INTO daily_summary VALUES ('stock','EMMS','us','2017-11-01',3.4379,3.44,3.31,3.44,38375,'0');
INSERT INTO daily_summary VALUES ('stock','EMN','us','2017-11-01',91.46,91.55,90.29,90.98,957029,'0');
INSERT INTO daily_summary VALUES ('stock','EMO','us','2017-11-01',11.15,11.3,11.139,11.23,158372,'0');
INSERT INTO daily_summary VALUES ('stock','EMP','us','2017-11-01',25.04,25.04,24.971,25.03,18842,'0');
INSERT INTO daily_summary VALUES ('stock','EMQQ','us','2017-11-01',37.45,37.5203,37.1876,37.24,141859,'0');
INSERT INTO daily_summary VALUES ('stock','EMR','us','2017-11-01',64.56,64.89,64.04,64.07,5365550,'0');
INSERT INTO daily_summary VALUES ('stock','EMSD','us','2017-11-01',28.895,28.895,28.895,28.895,0,'0');
INSERT INTO daily_summary VALUES ('etf','EMSH','us','2017-11-01',77.87,77.87,77.87,77.87,147,'0');
INSERT INTO daily_summary VALUES ('stock','EMTL','us','2017-11-01',51.17,51.18,51.07,51.0998,7173,'0');
INSERT INTO daily_summary VALUES ('stock','EMX','us','2017-11-01',0.7,0.72,0.7,0.7099,41579,'0');
INSERT INTO daily_summary VALUES ('stock','EMXC','us','2017-11-01',53.23,53.24,53.16,53.16,514,'0');
INSERT INTO daily_summary VALUES ('stock','ENB','us','2017-11-01',38.36,38.8,38.31,38.32,2767780,'0');
INSERT INTO daily_summary VALUES ('stock','ENBL','us','2017-11-01',14.624,15.231,14.536,15.113,133235,'0');
INSERT INTO daily_summary VALUES ('stock','ENDP','us','2017-11-01',6.4,6.55,6.23,6.53,6764860,'0');
INSERT INTO daily_summary VALUES ('stock','ENFC','us','2017-11-01',26.95,27.3,26.95,27.25,12970,'0');
INSERT INTO daily_summary VALUES ('etf','ENFR','us','2017-11-01',22.685,22.73,22.6564,22.71,4912,'0');
INSERT INTO daily_summary VALUES ('stock','ENG','us','2017-11-01',1.26,1.26,1.21,1.22,3912,'0');
INSERT INTO daily_summary VALUES ('stock','ENIA','us','2017-11-01',10.7,11.1,10.53,10.64,707541,'0');
INSERT INTO daily_summary VALUES ('stock','ENIC','us','2017-11-01',5.82,5.83,5.47,5.63,344269,'0');
INSERT INTO daily_summary VALUES ('stock','ENJ','us','2017-11-01',25,25.32,24.96,25.32,3027,'0');
INSERT INTO daily_summary VALUES ('stock','ENLC','us','2017-11-01',16.1,16.2,15.7,16,312904,'0');
INSERT INTO daily_summary VALUES ('stock','ENLK','us','2017-11-01',15.38,15.82,15.32,15.79,1093740,'0');
INSERT INTO daily_summary VALUES ('stock','ENO','us','2017-11-01',25.8,25.8616,25.79,25.82,3579,'0');
INSERT INTO daily_summary VALUES ('stock','ENOR','us','2017-11-01',26.28,26.36,26.2558,26.329,17249,'0');
INSERT INTO daily_summary VALUES ('stock','ENPH','us','2017-11-01',1.53,1.53,1.38,1.4,406058,'0');
INSERT INTO daily_summary VALUES ('stock','ENR','us','2017-11-01',43.1,43.55,42.775,43.11,648854,'0');
INSERT INTO daily_summary VALUES ('stock','ENRJ','us','2017-11-01',0.2877,0.2898,0.2624,0.2644,356243,'0');
INSERT INTO daily_summary VALUES ('stock','ENS','us','2017-11-01',69.43,69.69,67.91,67.95,353321,'0');
INSERT INTO daily_summary VALUES ('stock','ENSG','us','2017-11-01',23.17,23.17,22.19,22.41,151929,'0');
INSERT INTO daily_summary VALUES ('stock','ENSV','us','2017-11-01',0.504,0.5267,0.49,0.5167,294873,'0');
INSERT INTO daily_summary VALUES ('stock','ENT','us','2017-11-01',2.46,2.47,2.125,2.16,2881040,'0');
INSERT INTO daily_summary VALUES ('stock','ENTA','us','2017-11-01',49.93,50.26,48.6505,49.21,96605,'0');
INSERT INTO daily_summary VALUES ('stock','ENTG','us','2017-11-01',33,33,31.175,31.625,1033700,'0');
INSERT INTO daily_summary VALUES ('stock','ENTL','us','2017-11-01',16.95,17.59,16.5,16.64,156084,'0');
INSERT INTO daily_summary VALUES ('stock','ENV','us','2017-11-01',53.6,53.6,52.2,52.95,166341,'0');
INSERT INTO daily_summary VALUES ('stock','ENVA','us','2017-11-01',14.85,15.15,14.5,14.6,152697,'0');
INSERT INTO daily_summary VALUES ('stock','ENX','us','2017-11-01',12.46,12.78,12.42,12.78,28304,'0');
INSERT INTO daily_summary VALUES ('etf','ENY','us','2017-11-01',8.39,8.5172,8.39,8.47,4488,'0');
INSERT INTO daily_summary VALUES ('stock','ENZ','us','2017-11-01',9.92,9.98,9.47,9.49,133757,'0');
INSERT INTO daily_summary VALUES ('etf','ENZL','us','2017-11-01',45.45,45.57,45.27,45.27,36620,'0');
INSERT INTO daily_summary VALUES ('stock','ENZY','us','2017-11-01',11.8,11.8,11.7,11.8,282968,'0');
INSERT INTO daily_summary VALUES ('stock','EOCC','us','2017-11-01',25.45,26.3,24.92,25.46,215183,'0');
INSERT INTO daily_summary VALUES ('stock','EOD','us','2017-11-01',6.26,6.31,6.22,6.25,151709,'0');
INSERT INTO daily_summary VALUES ('stock','EOG','us','2017-11-01',100.79,102.47,96.71,102.29,3953420,'0');
INSERT INTO daily_summary VALUES ('stock','EOI','us','2017-11-01',14.37,14.3954,14.32,14.32,76687,'0');
INSERT INTO daily_summary VALUES ('stock','EOS','us','2017-11-01',15.14,15.18,15.1,15.12,97072,'0');
INSERT INTO daily_summary VALUES ('stock','EOT','us','2017-11-01',22.06,22.06,21.89,21.97,31436,'0');
INSERT INTO daily_summary VALUES ('stock','EPAM','us','2017-11-01',91.78,91.78,90.06,90.61,274735,'0');
INSERT INTO daily_summary VALUES ('stock','EPAY','us','2017-11-01',32.83,33.615,31.75,32.21,512984,'0');
INSERT INTO daily_summary VALUES ('stock','EPC','us','2017-11-01',65.14,65.58,64.4,65.01,558770,'0');
INSERT INTO daily_summary VALUES ('stock','EPD','us','2017-11-01',24.65,25.13,24.57,25.02,5989100,'0');
INSERT INTO daily_summary VALUES ('stock','EPE','us','2017-11-01',2.74,2.85,2.61,2.72,846494,'0');
INSERT INTO daily_summary VALUES ('etf','EPHE','us','2017-11-01',37,37.15,36.98,37.1,168299,'0');
INSERT INTO daily_summary VALUES ('etf','EPI','us','2017-11-01',27.18,27.315,27.18,27.23,1865360,'0');
INSERT INTO daily_summary VALUES ('stock','EPIX','us','2017-11-01',0.2599,0.27,0.25,0.2547,45964,'0');
INSERT INTO daily_summary VALUES ('stock','EPM','us','2017-11-01',7.4,7.5,7.075,7.15,145704,'0');
INSERT INTO daily_summary VALUES ('etf','EPOL','us','2017-11-01',27.13,27.26,27.06,27.08,435586,'0');
INSERT INTO daily_summary VALUES ('etf','EPP','us','2017-11-01',47.08,47.2328,47.0142,47.04,520093,'0');
INSERT INTO daily_summary VALUES ('stock','EPR','us','2017-11-01',69.18,69.72,68.85,69.05,423698,'0');
INSERT INTO daily_summary VALUES ('stock','EPRF','us','2017-11-01',23.75,24.1,23.75,24.06,3642,'0');
INSERT INTO daily_summary VALUES ('stock','EPR_C','us','2017-11-01',27.8374,27.8374,27.74,27.74,725,'0');
INSERT INTO daily_summary VALUES ('stock','EPR_E','us','2017-11-01',35.7499,35.75,35.7499,35.75,1000,'0');
INSERT INTO daily_summary VALUES ('stock','EPR_F','us','2017-11-01',25.29,25.49,25.29,25.35,1366,'0');
INSERT INTO daily_summary VALUES ('etf','EPS','us','2017-11-01',29.833,29.843,29.813,29.813,9660,'0');
INSERT INTO daily_summary VALUES ('etf','EPU','us','2017-11-01',41.92,42.21,41.9,42.05,201277,'0');
INSERT INTO daily_summary VALUES ('etf','EPV','us','2017-11-01',31.81,32.08,31.75,32.08,4141,'0');
INSERT INTO daily_summary VALUES ('stock','EPZM','us','2017-11-01',16.95,16.95,15.55,16.15,510695,'0');
INSERT INTO daily_summary VALUES ('etf','EQAL','us','2017-11-01',30.02,30.08,29.87,29.96,47769,'0');
INSERT INTO daily_summary VALUES ('stock','EQBK','us','2017-11-01',34.25,34.4,33.72,34.03,37628,'0');
INSERT INTO daily_summary VALUES ('stock','EQC','us','2017-11-01',30.12,30.2,29.95,30.13,474549,'0');
INSERT INTO daily_summary VALUES ('stock','EQCO','us','2017-11-01',25.26,25.31,24.98,25.04,11258,'0');
INSERT INTO daily_summary VALUES ('stock','EQC_D','us','2017-11-01',26.25,26.29,26.25,26.29,837,'0');
INSERT INTO daily_summary VALUES ('stock','EQGP','us','2017-11-01',27.553,28.078,27.325,27.642,36411,'0');
INSERT INTO daily_summary VALUES ('stock','EQIX','us','2017-11-01',464.97,467.74,463.52,467.06,428120,'0');
INSERT INTO daily_summary VALUES ('etf','EQL','us','2017-11-01',67.5194,67.5194,67.381,67.381,1274,'0');
INSERT INTO daily_summary VALUES ('etf','EQLT','us','2017-11-01',34.77,34.77,34.7548,34.7548,815,'0');
INSERT INTO daily_summary VALUES ('stock','EQM','us','2017-11-01',72.498,73.298,71.965,73.12,179113,'0');
INSERT INTO daily_summary VALUES ('stock','EQR','us','2017-11-01',67.32,68.04,67.01,67.81,880728,'0');
INSERT INTO daily_summary VALUES ('stock','EQRR','us','2017-11-01',43.74,43.74,43.39,43.44,1702,'0');
INSERT INTO daily_summary VALUES ('stock','EQS','us','2017-11-01',2.4,2.4,2.4,2.4,100,'0');
INSERT INTO daily_summary VALUES ('stock','EQT','us','2017-11-01',63.051,63.352,61.891,62.031,3376670,'0');
INSERT INTO daily_summary VALUES ('etf','EQWL','us','2017-11-01',51.05,51.1,50.94,50.96,4417,'0');
INSERT INTO daily_summary VALUES ('etf','EQWS','us','2017-11-01',42.26,42.28,41.62,41.87,12235,'0');
INSERT INTO daily_summary VALUES ('stock','ERA','us','2017-11-01',10.84,11.155,10.6,10.86,68476,'0');
INSERT INTO daily_summary VALUES ('stock','ERC','us','2017-11-01',13.1,13.14,12.99,13.05,332825,'0');
INSERT INTO daily_summary VALUES ('stock','ERF','us','2017-11-01',9.28,9.39,9.225,9.37,1086090,'0');
INSERT INTO daily_summary VALUES ('stock','ERGF','us','2017-11-01',25.84,25.84,25.84,25.84,223,'0');
INSERT INTO daily_summary VALUES ('stock','ERH','us','2017-11-01',13.38,13.38,13.1692,13.21,17637,'0');
INSERT INTO daily_summary VALUES ('stock','ERI','us','2017-11-01',25.7,26.25,25.3,25.8,702070,'0');
INSERT INTO daily_summary VALUES ('stock','ERIC','us','2017-11-01',6.28,6.31,6.26,6.28,1914260,'0');
INSERT INTO daily_summary VALUES ('stock','ERIE','us','2017-11-01',121.16,122.89,120.52,122.18,23374,'0');
INSERT INTO daily_summary VALUES ('stock','ERII','us','2017-11-01',7.79,7.8797,7.48,7.67,354793,'0');
INSERT INTO daily_summary VALUES ('stock','ERJ','us','2017-11-01',19.35,19.61,19.05,19.09,2617760,'0');
INSERT INTO daily_summary VALUES ('stock','ERM','us','2017-11-01',21.17,21.22,21.17,21.22,1112,'0');
INSERT INTO daily_summary VALUES ('stock','ERN','us','2017-11-01',2.65,2.7,2.5,2.695,35247,'0');
INSERT INTO daily_summary VALUES ('stock','EROS','us','2017-11-01',12.3,12.9,11.875,12.7,453731,'0');
INSERT INTO daily_summary VALUES ('etf','ERUS','us','2017-11-01',33.57,33.69,33.28,33.34,533507,'0');
INSERT INTO daily_summary VALUES ('etf','ERX','us','2017-11-01',29.62,30.52,29.61,30.11,1946850,'0');
INSERT INTO daily_summary VALUES ('etf','ERY','us','2017-11-01',10.95,10.97,10.64,10.79,1336350,'0');
INSERT INTO daily_summary VALUES ('stock','ES','us','2017-11-01',62.79,63.06,61.98,62.68,2622800,'0');
INSERT INTO daily_summary VALUES ('stock','ESBA','us','2017-11-01',19.92,20.1,19.8201,20.02,12670,'0');
INSERT INTO daily_summary VALUES ('stock','ESBK','us','2017-11-01',20.7081,20.7499,20.7,20.7499,1455,'0');
INSERT INTO daily_summary VALUES ('stock','ESCA','us','2017-11-01',12.55,12.55,12.25,12.3,6039,'0');
INSERT INTO daily_summary VALUES ('stock','ESDI','us','2017-11-01',4.04,4.06,3.95,4,36128,'0');
INSERT INTO daily_summary VALUES ('stock','ESDIW','us','2017-11-01',0.44,0.44,0.43,0.43,6200,'0');
INSERT INTO daily_summary VALUES ('stock','ESE','us','2017-11-01',58.35,59.35,58.05,59.05,99591,'0');
INSERT INTO daily_summary VALUES ('stock','ESEA','us','2017-11-01',1.81,1.88,1.77,1.77,32176,'0');
INSERT INTO daily_summary VALUES ('stock','ESES','us','2017-11-01',1.46,1.46,1.39,1.41,33506,'0');
INSERT INTO daily_summary VALUES ('stock','ESG','us','2017-11-01',60.6802,60.6802,60.6802,60.6802,100,'0');
INSERT INTO daily_summary VALUES ('stock','ESGD','us','2017-11-01',67.75,67.76,67.54,67.57,16825,'0');
INSERT INTO daily_summary VALUES ('stock','ESGE','us','2017-11-01',72.25,72.25,71.8176,71.9,10772,'0');
INSERT INTO daily_summary VALUES ('stock','ESGF','us','2017-11-01',30.705,30.705,30.55,30.55,375,'0');
INSERT INTO daily_summary VALUES ('stock','ESGG','us','2017-11-01',92.41,92.43,92.41,92.43,231,'0');
INSERT INTO daily_summary VALUES ('stock','ESGL','us','2017-11-01',29.57,29.57,29.51,29.51,3619,'0');
INSERT INTO daily_summary VALUES ('stock','ESGN','us','2017-11-01',30.8675,30.8675,30.7326,30.77,2093,'0');
INSERT INTO daily_summary VALUES ('stock','ESGR','us','2017-11-01',229.65,230.3,226,227.8,13645,'0');
INSERT INTO daily_summary VALUES ('stock','ESGU','us','2017-11-01',56.56,56.56,56.5397,56.5397,849,'0');
INSERT INTO daily_summary VALUES ('stock','ESIO','us','2017-11-01',17.93,18.2,15.77,16.31,1711390,'0');
INSERT INTO daily_summary VALUES ('stock','ESL','us','2017-11-01',95.6,95.9,94.225,95,184646,'0');
INSERT INTO daily_summary VALUES ('stock','ESLT','us','2017-11-01',148.59,148.91,147.83,147.86,18273,'0');
INSERT INTO daily_summary VALUES ('stock','ESNC','us','2017-11-01',0.4301,0.46,0.4301,0.4597,103608,'0');
INSERT INTO daily_summary VALUES ('stock','ESND','us','2017-11-01',9.71,9.94,9.355,9.51,358538,'0');
INSERT INTO daily_summary VALUES ('stock','ESNT','us','2017-11-01',43.1,43.49,42.67,42.91,800121,'0');
INSERT INTO daily_summary VALUES ('stock','ESP','us','2017-11-01',22.83,23.3,22.53,22.53,12518,'0');
INSERT INTO daily_summary VALUES ('stock','ESPR','us','2017-11-01',46.14,47.55,44.42,45.09,667977,'0');
INSERT INTO daily_summary VALUES ('stock','ESQ','us','2017-11-01',16.1614,16.31,16.02,16.3,18493,'0');
INSERT INTO daily_summary VALUES ('stock','ESRT','us','2017-11-01',19.96,20.19,19.741,20.04,1695460,'0');
INSERT INTO daily_summary VALUES ('stock','ESRX','us','2017-11-01',61.81,62.61,61.52,62.32,3386900,'0');
INSERT INTO daily_summary VALUES ('stock','ESS','us','2017-11-01',262.96,264.065,260.56,262.53,291605,'0');
INSERT INTO daily_summary VALUES ('stock','ESSA','us','2017-11-01',15.9301,16.0196,15.9301,15.97,2539,'0');
INSERT INTO daily_summary VALUES ('stock','ESTE','us','2017-11-01',8.16,8.83,8.12,8.77,443614,'0');
INSERT INTO daily_summary VALUES ('stock','ESV','us','2017-11-01',5.47,5.89,5.43,5.48,17082000,'0');
INSERT INTO daily_summary VALUES ('stock','ESXB','us','2017-11-01',8.65,8.75,8.525,8.7,57887,'0');
INSERT INTO daily_summary VALUES ('stock','ETB','us','2017-11-01',16.61,16.65,16.55,16.58,41508,'0');
INSERT INTO daily_summary VALUES ('stock','ETE','us','2017-11-01',17.603,17.761,17.544,17.623,8554260,'0');
INSERT INTO daily_summary VALUES ('stock','ETFC','us','2017-11-01',43.88,44.385,43.4,43.54,1716340,'0');
INSERT INTO daily_summary VALUES ('stock','ETG','us','2017-11-01',17.26,17.35,17.26,17.34,111644,'0');
INSERT INTO daily_summary VALUES ('stock','ETH','us','2017-11-01',30.05,30.1,29.05,29.35,225049,'0');
INSERT INTO daily_summary VALUES ('stock','ETHO','us','2017-11-01',32.99,32.99,32.7574,32.8,4259,'0');
INSERT INTO daily_summary VALUES ('stock','ETJ','us','2017-11-01',9.21,9.22,9.16,9.19,252767,'0');
INSERT INTO daily_summary VALUES ('stock','ETM','us','2017-11-01',11,11.15,10.65,10.75,4430590,'0');
INSERT INTO daily_summary VALUES ('stock','ETN','us','2017-11-01',80.285,81.734,80.205,80.91,3707750,'0');
INSERT INTO daily_summary VALUES ('stock','ETO','us','2017-11-01',24.85,24.85,24.74,24.79,61895,'0');
INSERT INTO daily_summary VALUES ('stock','ETP','us','2017-11-01',16.916,17.303,16.897,17.09,13938500,'0');
INSERT INTO daily_summary VALUES ('stock','ETR','us','2017-11-01',85.435,85.475,84.366,84.792,1382840,'0');
INSERT INTO daily_summary VALUES ('stock','ETSY','us','2017-11-01',16.75,16.83,16.48,16.73,1209770,'0');
INSERT INTO daily_summary VALUES ('stock','ETV','us','2017-11-01',15.18,15.18,15.14,15.14,186670,'0');
INSERT INTO daily_summary VALUES ('stock','ETW','us','2017-11-01',11.86,11.88,11.83,11.86,186191,'0');
INSERT INTO daily_summary VALUES ('stock','ETX','us','2017-11-01',20.088,20.13,20.05,20.091,20926,'0');
INSERT INTO daily_summary VALUES ('stock','ETY','us','2017-11-01',11.68,11.7,11.62,11.63,264343,'0');
INSERT INTO daily_summary VALUES ('etf','EUDG','us','2017-11-01',26.8,26.81,26.71,26.71,26962,'0');
INSERT INTO daily_summary VALUES ('stock','EUDV','us','2017-11-01',41.68,41.78,41.42,41.42,11600,'0');
INSERT INTO daily_summary VALUES ('stock','EUFL','us','2017-11-01',48.23,48.23,48.19,48.22,1058,'0');
INSERT INTO daily_summary VALUES ('etf','EUFN','us','2017-11-01',23.14,23.21,23.09,23.09,733232,'0');
INSERT INTO daily_summary VALUES ('etf','EUM','us','2017-11-01',18.4,18.55,18.21,18.49,256317,'0');
INSERT INTO daily_summary VALUES ('etf','EUMV','us','2017-11-01',25.41,25.41,25.29,25.32,5415,'0');
INSERT INTO daily_summary VALUES ('etf','EUO','us','2017-11-01',22.5,22.54,22.383,22.46,202340,'0');
INSERT INTO daily_summary VALUES ('etf','EURL','us','2017-11-01',37.38,37.4396,36.92,36.92,25526,'0');
INSERT INTO daily_summary VALUES ('stock','EURN','us','2017-11-01',8.4,8.5,8.4,8.45,185522,'0');
INSERT INTO daily_summary VALUES ('stock','EURZ','us','2017-11-01',24.17,24.17,24.17,24.17,0,'0');
INSERT INTO daily_summary VALUES ('etf','EUSA','us','2017-11-01',53.4,53.4,53.015,53.12,6226,'0');
INSERT INTO daily_summary VALUES ('etf','EUSC','us','2017-11-01',31.47,31.47,31.36,31.39,5146,'0');
INSERT INTO daily_summary VALUES ('stock','EUXL','us','2017-11-01',30.99,31.03,30.76,30.78,2125,'0');
INSERT INTO daily_summary VALUES ('stock','EV','us','2017-11-01',51.02,51.15,50.54,50.67,294307,'0');
INSERT INTO daily_summary VALUES ('stock','EVA','us','2017-11-01',27.6,29,27.6,29,215566,'0');
INSERT INTO daily_summary VALUES ('stock','EVAR','us','2017-11-01',0.3118,0.3118,0.27,0.2801,210830,'0');
INSERT INTO daily_summary VALUES ('stock','EVBG','us','2017-11-01',26.88,27.07,26.65,26.8,376278,'0');
INSERT INTO daily_summary VALUES ('stock','EVBN','us','2017-11-01',42.35,43.334,42.25,42.8,4274,'0');
INSERT INTO daily_summary VALUES ('stock','EVC','us','2017-11-01',5.2,5.3,5.15,5.25,363749,'0');
INSERT INTO daily_summary VALUES ('stock','EVEP','us','2017-11-01',0.61,0.64,0.61,0.63,120324,'0');
INSERT INTO daily_summary VALUES ('stock','EVF','us','2017-11-01',6.5295,6.5345,6.4997,6.5096,82932,'0');
INSERT INTO daily_summary VALUES ('stock','EVG','us','2017-11-01',14.25,14.25,14.15,14.15,32383,'0');
INSERT INTO daily_summary VALUES ('stock','EVGN','us','2017-11-01',4.28,4.43,4.13,4.2,3480,'0');
INSERT INTO daily_summary VALUES ('stock','EVH','us','2017-11-01',16.4,16.4,14.55,15.3,1843500,'0');
INSERT INTO daily_summary VALUES ('stock','EVHC','us','2017-11-01',30.07,31.65,26.56,28.03,50447300,'0');
INSERT INTO daily_summary VALUES ('stock','EVI','us','2017-11-01',26.45,27.05,26.2,26.8,33272,'0');
INSERT INTO daily_summary VALUES ('stock','EVIX','us','2017-11-01',13.54,13.54,13.52,13.53,2303,'0');
INSERT INTO daily_summary VALUES ('stock','EVJ','us','2017-11-01',11.94,11.94,11.81,11.89,1657,'0');
INSERT INTO daily_summary VALUES ('stock','EVK','us','2017-11-01',2.25,2.35,2.25,2.35,1207,'0');
INSERT INTO daily_summary VALUES ('stock','EVLV','us','2017-11-01',1.16,1.18,1.1357,1.16,265010,'0');
INSERT INTO daily_summary VALUES ('stock','EVM','us','2017-11-01',11.87,11.88,11.81,11.85,39509,'0');
INSERT INTO daily_summary VALUES ('stock','EVN','us','2017-11-01',12.51,12.529,12.42,12.42,79960,'0');
INSERT INTO daily_summary VALUES ('stock','EVO','us','2017-11-01',13.32,13.37,13.32,13.37,4600,'0');
INSERT INTO daily_summary VALUES ('stock','EVOK','us','2017-11-01',3.06,3.07,2.88,2.95,111447,'0');
INSERT INTO daily_summary VALUES ('stock','EVOL','us','2017-11-01',4.6,4.65,4.55,4.6,23958,'0');
INSERT INTO daily_summary VALUES ('stock','EVP','us','2017-11-01',11.861,11.941,11.861,11.893,1485,'0');
INSERT INTO daily_summary VALUES ('stock','EVR','us','2017-11-01',80.6,80.95,78.15,78.4,294797,'0');
INSERT INTO daily_summary VALUES ('stock','EVRI','us','2017-11-01',8.31,8.34,8.22,8.32,877569,'0');
INSERT INTO daily_summary VALUES ('stock','EVT','us','2017-11-01',22.53,22.55,22.45,22.46,102509,'0');
INSERT INTO daily_summary VALUES ('stock','EVTC','us','2017-11-01',15.1,15.1,14.65,14.75,702619,'0');
INSERT INTO daily_summary VALUES ('stock','EVV','us','2017-11-01',13.909,13.959,13.884,13.959,123674,'0');
INSERT INTO daily_summary VALUES ('stock','EVY','us','2017-11-01',13.139,13.179,13.104,13.119,6485,'0');
INSERT INTO daily_summary VALUES ('stock','EW','us','2017-11-01',102.37,103.53,101.69,101.82,1797300,'0');
INSERT INTO daily_summary VALUES ('etf','EWA','us','2017-11-01',22.85,22.955,22.795,22.81,2151610,'0');
INSERT INTO daily_summary VALUES ('stock','EWBC','us','2017-11-01',60.39,60.63,59.55,59.73,577356,'0');
INSERT INTO daily_summary VALUES ('etf','EWC','us','2017-11-01',28.91,28.98,28.83,28.89,2038880,'0');
INSERT INTO daily_summary VALUES ('etf','EWD','us','2017-11-01',35.63,35.66,35.43,35.43,173686,'0');
INSERT INTO daily_summary VALUES ('etf','EWEM','us','2017-11-01',33.86,33.86,33.72,33.72,1051,'0');
INSERT INTO daily_summary VALUES ('etf','EWG','us','2017-11-01',33.38,33.44,33.335,33.35,2003550,'0');
INSERT INTO daily_summary VALUES ('stock','EWGS','us','2017-11-01',61.84,61.84,61.4953,61.56,28359,'0');
INSERT INTO daily_summary VALUES ('etf','EWH','us','2017-11-01',24.95,25.02,24.92,24.93,3936730,'0');
INSERT INTO daily_summary VALUES ('etf','EWI','us','2017-11-01',31.23,31.315,31.1,31.13,902889,'0');
INSERT INTO daily_summary VALUES ('etf','EWJ','us','2017-11-01',59.03,59.095,58.915,58.98,11338900,'0');
INSERT INTO daily_summary VALUES ('etf','EWK','us','2017-11-01',21.47,21.49,21.37,21.37,69988,'0');
INSERT INTO daily_summary VALUES ('etf','EWL','us','2017-11-01',34.46,34.56,34.33,34.33,982730,'0');
INSERT INTO daily_summary VALUES ('etf','EWM','us','2017-11-01',32.05,32.21,31.99,32.06,561854,'0');
INSERT INTO daily_summary VALUES ('stock','EWMC','us','2017-11-01',62.41,62.41,61.48,61.903,4250,'0');
INSERT INTO daily_summary VALUES ('etf','EWN','us','2017-11-01',31.78,31.8,31.66,31.67,188952,'0');
INSERT INTO daily_summary VALUES ('etf','EWO','us','2017-11-01',23.93,24.03,23.86,23.87,298857,'0');
INSERT INTO daily_summary VALUES ('etf','EWP','us','2017-11-01',33.81,33.87,33.665,33.81,2789030,'0');
INSERT INTO daily_summary VALUES ('etf','EWQ','us','2017-11-01',31.41,31.46,31.34,31.35,672850,'0');
INSERT INTO daily_summary VALUES ('stock','EWRE','us','2017-11-01',27.23,27.3664,27.23,27.35,1122,'0');
INSERT INTO daily_summary VALUES ('etf','EWS','us','2017-11-01',25.7,25.78,25.67,25.7,630321,'0');
INSERT INTO daily_summary VALUES ('stock','EWSC','us','2017-11-01',52.39,52.4999,52.2354,52.4999,1288,'0');
INSERT INTO daily_summary VALUES ('etf','EWT','us','2017-11-01',38.23,38.255,37.97,38.03,5600250,'0');
INSERT INTO daily_summary VALUES ('etf','EWU','us','2017-11-01',35.11,35.1579,34.85,34.87,2384530,'0');
INSERT INTO daily_summary VALUES ('stock','EWUS','us','2017-11-01',42.84,42.8721,42.7035,42.7035,515,'0');
INSERT INTO daily_summary VALUES ('etf','EWV','us','2017-11-01',27.79,27.79,27.75,27.75,3612,'0');
INSERT INTO daily_summary VALUES ('etf','EWW','us','2017-11-01',50.39,50.64,49.97,50.08,2293490,'0');
INSERT INTO daily_summary VALUES ('etf','EWX','us','2017-11-01',50.97,51.16,50.92,50.95,41432,'0');
INSERT INTO daily_summary VALUES ('etf','EWY','us','2017-11-01',76.19,76.455,76.0714,76.23,4141890,'0');
INSERT INTO daily_summary VALUES ('etf','EWZ','us','2017-11-01',40.28,40.7,39.83,39.9,16066700,'0');
INSERT INTO daily_summary VALUES ('etf','EWZS','us','2017-11-01',16.22,16.36,16.091,16.22,14544,'0');
INSERT INTO daily_summary VALUES ('stock','EXA','us','2017-11-01',24.23,24.24,24.21,24.23,119511,'0');
INSERT INTO daily_summary VALUES ('stock','EXAC','us','2017-11-01',41.85,42,41.8,41.9,214395,'0');
INSERT INTO daily_summary VALUES ('stock','EXAS','us','2017-11-01',54.28,54.759,52.05,53.01,3340700,'0');
INSERT INTO daily_summary VALUES ('stock','EXC','us','2017-11-01',40.32,40.36,39.98,40.15,5209990,'0');
INSERT INTO daily_summary VALUES ('stock','EXD','us','2017-11-01',11.37,11.4,11.25,11.25,57836,'0');
INSERT INTO daily_summary VALUES ('stock','EXEL','us','2017-11-01',24.95,25.36,24.19,24.23,4662800,'0');
INSERT INTO daily_summary VALUES ('stock','EXFO','us','2017-11-01',3.85,3.89,3.85,3.85,4181,'0');
INSERT INTO daily_summary VALUES ('stock','EXG','us','2017-11-01',9.35,9.39,9.35,9.37,522038,'0');
INSERT INTO daily_summary VALUES ('etf','EXI','us','2017-11-01',90.66,90.86,90.155,90.23,57695,'0');
INSERT INTO daily_summary VALUES ('stock','EXIV','us','2017-11-01',43.56,43.58,43.4,43.45,5794,'0');
INSERT INTO daily_summary VALUES ('stock','EXK','us','2017-11-01',2.13,2.29,1.94,1.98,9969970,'0');
INSERT INTO daily_summary VALUES ('stock','EXLS','us','2017-11-01',62.01,62.26,61.045,61.76,132266,'0');
INSERT INTO daily_summary VALUES ('stock','EXP','us','2017-11-01',106.42,106.63,104.66,105.34,442846,'0');
INSERT INTO daily_summary VALUES ('stock','EXPD','us','2017-11-01',58.67,58.91,58.21,58.47,660905,'0');
INSERT INTO daily_summary VALUES ('stock','EXPE','us','2017-11-01',125.88,127,121.92,122.05,3516040,'0');
INSERT INTO daily_summary VALUES ('stock','EXPO','us','2017-11-01',74.35,74.6,72.4,72.6,78273,'0');
INSERT INTO daily_summary VALUES ('stock','EXPR','us','2017-11-01',6.8,6.87,6.57,6.73,1338220,'0');
INSERT INTO daily_summary VALUES ('stock','EXR','us','2017-11-01',81.59,82,80.51,81.6,1310450,'0');
INSERT INTO daily_summary VALUES ('etf','EXT','us','2017-11-01',30.703,30.703,30.563,30.575,5775,'0');
INSERT INTO daily_summary VALUES ('stock','EXTN','us','2017-11-01',32.6,32.715,31.93,32.3,180948,'0');
INSERT INTO daily_summary VALUES ('stock','EXTR','us','2017-11-01',12.04,12.14,11.72,11.88,923385,'0');
INSERT INTO daily_summary VALUES ('stock','EXXI','us','2017-11-01',8.65,9.5,8.61,8.85,458931,'0');
INSERT INTO daily_summary VALUES ('stock','EYE','us','2017-11-01',29.05,30.98,28.27,30.37,1781340,'0');
INSERT INTO daily_summary VALUES ('stock','EYEG','us','2017-11-01',1.12,1.12,1.08,1.0967,50795,'0');
INSERT INTO daily_summary VALUES ('stock','EYES','us','2017-11-01',1.08,1.09,1.07,1.08,65110,'0');
INSERT INTO daily_summary VALUES ('stock','EYLD','us','2017-11-01',33.75,33.75,33.645,33.645,465,'0');
INSERT INTO daily_summary VALUES ('etf','EZA','us','2017-11-01',60,60.37,59.68,59.95,405633,'0');
INSERT INTO daily_summary VALUES ('etf','EZJ','us','2017-11-01',126,126.06,125.74,125.74,2256,'0');
INSERT INTO daily_summary VALUES ('etf','EZM','us','2017-11-01',38.1,38.12,37.65,37.73,73583,'0');
INSERT INTO daily_summary VALUES ('stock','EZPW','us','2017-11-01',10.35,10.4,10.05,10.15,282758,'0');
INSERT INTO daily_summary VALUES ('stock','EZT','us','2017-11-01',26.14,26.14,26,26,1708,'0');
INSERT INTO daily_summary VALUES ('etf','EZU','us','2017-11-01',43.98,44.055,43.89,43.94,4562660,'0');
INSERT INTO daily_summary VALUES ('etf','EZY','us','2017-11-01',78.16,78.16,77.63,77.87,2087,'0');
INSERT INTO daily_summary VALUES ('stock','F','us','2017-11-01',12.4,12.47,12.2752,12.35,43489600,'0');
INSERT INTO daily_summary VALUES ('stock','FAAR','us','2017-11-01',30.91,30.91,29.97,30.0479,1468,'0');
INSERT INTO daily_summary VALUES ('etf','FAB','us','2017-11-01',54.2553,54.4686,53.94,54.02,20628,'0');
INSERT INTO daily_summary VALUES ('etf','FAD','us','2017-11-01',63.75,63.7592,63.14,63.25,34511,'0');
INSERT INTO daily_summary VALUES ('stock','FAF','us','2017-11-01',54.7,55,54.44,54.51,557082,'0');
INSERT INTO daily_summary VALUES ('stock','FALN','us','2017-11-01',27.75,27.76,27.6802,27.6802,10320,'0');
INSERT INTO daily_summary VALUES ('etf','FAM','us','2017-11-01',11.6,11.6,11.49,11.5,23756,'0');
INSERT INTO daily_summary VALUES ('etf','FAN','us','2017-11-01',13.27,13.28,13.25,13.25,45480,'0');
INSERT INTO daily_summary VALUES ('stock','FANG','us','2017-11-01',108.23,111.71,108.14,109.78,1351170,'0');
INSERT INTO daily_summary VALUES ('stock','FANH','us','2017-11-01',14.99,15.75,14.8194,15.26,121400,'0');
INSERT INTO daily_summary VALUES ('stock','FANZ','us','2017-11-01',21.209,21.21,21.169,21.185,1963,'0');
INSERT INTO daily_summary VALUES ('stock','FARM','us','2017-11-01',34.3,34.4,33.3,34.3,79678,'0');
INSERT INTO daily_summary VALUES ('stock','FARO','us','2017-11-01',52.4,52.4,51.15,51.45,142446,'0');
INSERT INTO daily_summary VALUES ('etf','FAS','us','2017-11-01',60.88,61.52,60.33,60.6,1295630,'0');
INSERT INTO daily_summary VALUES ('stock','FAST','us','2017-11-01',47.23,47.23,46.4,46.78,1515560,'0');
INSERT INTO daily_summary VALUES ('stock','FAT','us','2017-11-01',10.85,11.4476,10.5,10.75,60059,'0');
INSERT INTO daily_summary VALUES ('stock','FATE','us','2017-11-01',4.85,4.885,4.51,4.69,259713,'0');
INSERT INTO daily_summary VALUES ('stock','FAX','us','2017-11-01',5.04,5.07,5.03,5.07,398658,'0');
INSERT INTO daily_summary VALUES ('etf','FAZ','us','2017-11-01',13.31,13.45,13.18,13.38,835792,'0');
INSERT INTO daily_summary VALUES ('stock','FB','us','2017-11-01',182.36,182.9,180.57,182.66,40918300,'0');
INSERT INTO daily_summary VALUES ('stock','FBC','us','2017-11-01',37.71,37.81,36.94,36.97,101473,'0');
INSERT INTO daily_summary VALUES ('etf','FBGX','us','2017-11-01',210.72,210.72,210.438,210.438,1070,'0');
INSERT INTO daily_summary VALUES ('stock','FBHS','us','2017-11-01',66.4,66.4,65,65.2,971712,'0');
INSERT INTO daily_summary VALUES ('stock','FBIO','us','2017-11-01',3.65,3.93,3.4,3.81,204896,'0');
INSERT INTO daily_summary VALUES ('stock','FBIZ','us','2017-11-01',22.151,22.151,21.663,21.862,34292,'0');
INSERT INTO daily_summary VALUES ('stock','FBK','us','2017-11-01',41.48,41.48,39.721,40.05,47147,'0');
INSERT INTO daily_summary VALUES ('stock','FBM','us','2017-11-01',13.6,13.6,13.24,13.42,46030,'0');
INSERT INTO daily_summary VALUES ('stock','FBMS','us','2017-11-01',31.96,31.96,30.961,31.71,55207,'0');
INSERT INTO daily_summary VALUES ('stock','FBNC','us','2017-11-01',36.98,37.17,36.18,36.43,54604,'0');
INSERT INTO daily_summary VALUES ('etf','FBND','us','2017-11-01',50.21,50.33,50.2,50.28,74475,'0');
INSERT INTO daily_summary VALUES ('stock','FBNK','us','2017-11-01',26.65,26.65,25.75,26,9793,'0');
INSERT INTO daily_summary VALUES ('stock','FBP','us','2017-11-01',5.17,5.19,4.99,5,881506,'0');
INSERT INTO daily_summary VALUES ('stock','FBR','us','2017-11-01',16.07,16.28,15.92,16.12,867181,'0');
INSERT INTO daily_summary VALUES ('stock','FBSS','us','2017-11-01',20.48,20.75,20.48,20.75,1469,'0');
INSERT INTO daily_summary VALUES ('etf','FBT','us','2017-11-01',122.25,122.427,120,120.46,37106,'0');
INSERT INTO daily_summary VALUES ('etf','FBZ','us','2017-11-01',15.8656,15.8656,15.8656,15.8656,276,'0');
INSERT INTO daily_summary VALUES ('stock','FC','us','2017-11-01',19.6,19.6,18.95,19.05,17428,'0');
INSERT INTO daily_summary VALUES ('etf','FCA','us','2017-11-01',29.69,29.69,29.35,29.4412,2732,'0');
INSERT INTO daily_summary VALUES ('stock','FCAL','us','2017-11-01',50.7,50.7,50.7,50.7,207,'0');
INSERT INTO daily_summary VALUES ('etf','FCAN','us','2017-11-01',25.19,25.3324,25.19,25.2614,951,'0');
INSERT INTO daily_summary VALUES ('stock','FCAP','us','2017-11-01',35.75,35.75,35.75,35.75,101,'0');
INSERT INTO daily_summary VALUES ('stock','FCAU','us','2017-11-01',17.64,17.66,17.51,17.58,2322820,'0');
INSERT INTO daily_summary VALUES ('stock','FCB','us','2017-11-01',47.2,47.3,46.15,46.4,112998,'0');
INSERT INTO daily_summary VALUES ('stock','FCBC','us','2017-11-01',30.073,32.047,29.869,30.62,110326,'0');
INSERT INTO daily_summary VALUES ('stock','FCCO','us','2017-11-01',22.85,23.1,22.783,22.85,3588,'0');
INSERT INTO daily_summary VALUES ('stock','FCCY','us','2017-11-01',18.05,18.05,17.85,17.85,2951,'0');
INSERT INTO daily_summary VALUES ('stock','FCEF','us','2017-11-01',22.09,22.09,21.9854,21.9854,5625,'0');
INSERT INTO daily_summary VALUES ('stock','FCEL','us','2017-11-01',2.2,2.23,2.11,2.14,1264850,'0');
INSERT INTO daily_summary VALUES ('stock','FCF','us','2017-11-01',14.59,14.62,14.331,14.411,460116,'0');
INSERT INTO daily_summary VALUES ('stock','FCFS','us','2017-11-01',64.099,64.249,62.703,63.551,205573,'0');
INSERT INTO daily_summary VALUES ('etf','FCG','us','2017-11-01',21.81,22.2,21.81,22.13,217020,'0');
INSERT INTO daily_summary VALUES ('stock','FCN','us','2017-11-01',42.8,42.98,41.88,42.11,460886,'0');
INSERT INTO daily_summary VALUES ('stock','FCNCA','us','2017-11-01',408.35,410.09,400.27,402.42,15880,'0');
INSERT INTO daily_summary VALUES ('stock','FCO','us','2017-11-01',8.96,9,8.94,8.96,16446,'0');
INSERT INTO daily_summary VALUES ('etf','FCOM','us','2017-11-01',31.61,31.66,31.0233,31.14,124942,'0');
INSERT INTO daily_summary VALUES ('etf','FCOR','us','2017-11-01',50.65,50.819,50.65,50.8,15548,'0');
INSERT INTO daily_summary VALUES ('stock','FCPT','us','2017-11-01',24.8,25.12,24.58,24.7,182279,'0');
INSERT INTO daily_summary VALUES ('stock','FCRE','us','2017-11-01',1.08,1.08,1,1.03,6800,'0');
INSERT INTO daily_summary VALUES ('stock','FCSC','us','2017-11-01',2.23,2.3,2.05,2.1,84561,'0');
INSERT INTO daily_summary VALUES ('etf','FCT','us','2017-11-01',13.09,13.16,13.08,13.09,54066,'0');
INSERT INTO daily_summary VALUES ('stock','FCVT','us','2017-11-01',29.1403,29.1703,29.07,29.14,14694,'0');
INSERT INTO daily_summary VALUES ('stock','FCX','us','2017-11-01',14.4,14.69,14.27,14.38,18476700,'0');
INSERT INTO daily_summary VALUES ('stock','FDC','us','2017-11-01',17.87,17.9,17.2,17.34,9962010,'0');
INSERT INTO daily_summary VALUES ('etf','FDD','us','2017-11-01',13.72,13.73,13.6301,13.6526,140939,'0');
INSERT INTO daily_summary VALUES ('stock','FDEF','us','2017-11-01',54.447,55.472,52.964,53.93,36298,'0');
INSERT INTO daily_summary VALUES ('stock','FDEU','us','2017-11-01',18.97,18.97,18.8559,18.95,34853,'0');
INSERT INTO daily_summary VALUES ('etf','FDIS','us','2017-11-01',36.64,36.7,36.44,36.51,127488,'0');
INSERT INTO daily_summary VALUES ('etf','FDIV','us','2017-11-01',51.03,51.0432,50.68,50.68,16767,'0');
INSERT INTO daily_summary VALUES ('etf','FDL','us','2017-11-01',28.88,28.94,28.8402,28.91,346369,'0');
INSERT INTO daily_summary VALUES ('stock','FDLO','us','2017-11-01',29.55,29.55,29.41,29.44,20719,'0');
INSERT INTO daily_summary VALUES ('etf','FDM','us','2017-11-01',46.92,46.92,46.482,46.74,4715,'0');
INSERT INTO daily_summary VALUES ('stock','FDMO','us','2017-11-01',30.55,30.68,30.38,30.45,13343,'0');
INSERT INTO daily_summary VALUES ('etf','FDN','us','2017-11-01',108.03,108.22,106.84,107.27,527032,'0');
INSERT INTO daily_summary VALUES ('stock','FDP','us','2017-11-01',44.72,44.72,43.02,44.01,962240,'0');
INSERT INTO daily_summary VALUES ('stock','FDRR','us','2017-11-01',29.48,29.588,29.42,29.4994,30454,'0');
INSERT INTO daily_summary VALUES ('stock','FDS','us','2017-11-01',189.93,189.93,184.756,185.39,1152790,'0');
INSERT INTO daily_summary VALUES ('etf','FDT','us','2017-11-01',61.08,61.22,60.97,60.97,65841,'0');
INSERT INTO daily_summary VALUES ('stock','FDUS','us','2017-11-01',16.74,16.94,16.65,16.77,63163,'0');
INSERT INTO daily_summary VALUES ('stock','FDVV','us','2017-11-01',27.6,27.6,27.4,27.48,32119,'0');
INSERT INTO daily_summary VALUES ('stock','FDX','us','2017-11-01',227,227.375,224.67,225.46,1149870,'0');
INSERT INTO daily_summary VALUES ('stock','FE','us','2017-11-01',32.568,32.588,32.143,32.202,3469300,'0');
INSERT INTO daily_summary VALUES ('etf','FEEU','us','2017-11-01',130.38,130.75,129.66,129.66,3147,'0');
INSERT INTO daily_summary VALUES ('etf','FEI','us','2017-11-01',14.5,14.77,14.46,14.74,193611,'0');
INSERT INTO daily_summary VALUES ('stock','FEIM','us','2017-11-01',9.34,9.34,9.34,9.34,350,'0');
INSERT INTO daily_summary VALUES ('stock','FELE','us','2017-11-01',45.6,46.1,44.9,44.95,96893,'0');
INSERT INTO daily_summary VALUES ('stock','FELP','us','2017-11-01',4.26,4.375,4.12,4.19,20414,'0');
INSERT INTO daily_summary VALUES ('etf','FEM','us','2017-11-01',27.57,27.57,27.35,27.36,111492,'0');
INSERT INTO daily_summary VALUES ('etf','FEMB','us','2017-11-01',42.01,42.01,41.53,41.7664,11734,'0');
INSERT INTO daily_summary VALUES ('etf','FEMS','us','2017-11-01',42.48,42.63,42.3,42.38,127840,'0');
INSERT INTO daily_summary VALUES ('stock','FEN','us','2017-11-01',23.11,23.33,23.03,23.22,84588,'0');
INSERT INTO daily_summary VALUES ('stock','FENC','us','2017-11-01',9.42,9.42,8.0001,9.05,23826,'0');
INSERT INTO daily_summary VALUES ('stock','FENG','us','2017-11-01',5.49,5.49,5.07,5.37,566232,'0');
INSERT INTO daily_summary VALUES ('etf','FENY','us','2017-11-01',18.89,19.0946,18.88,19.01,211899,'0');
INSERT INTO daily_summary VALUES ('etf','FEO','us','2017-11-01',16.66,16.66,16.55,16.59,11739,'0');
INSERT INTO daily_summary VALUES ('etf','FEP','us','2017-11-01',38.5,38.5991,38.3501,38.36,86035,'0');
INSERT INTO daily_summary VALUES ('stock','FET','us','2017-11-01',14.35,14.75,14.05,14.7,1233700,'0');
INSERT INTO daily_summary VALUES ('etf','FEU','us','2017-11-01',35.71,35.77,35.59,35.59,38167,'0');
INSERT INTO daily_summary VALUES ('etf','FEUZ','us','2017-11-01',43.32,43.6199,43.1171,43.5799,9951,'0');
INSERT INTO daily_summary VALUES ('etf','FEX','us','2017-11-01',56.31,56.3361,56,56.07,54745,'0');
INSERT INTO daily_summary VALUES ('stock','FEYE','us','2017-11-01',16.96,17.03,16.31,16.37,8727360,'0');
INSERT INTO daily_summary VALUES ('etf','FEZ','us','2017-11-01',41.84,41.91,41.735,41.77,2516840,'0');
INSERT INTO daily_summary VALUES ('stock','FF','us','2017-11-01',15.35,15.36,14.89,15.15,34882,'0');
INSERT INTO daily_summary VALUES ('etf','FFA','us','2017-11-01',15.6,15.6,15.51,15.57,29016,'0');
INSERT INTO daily_summary VALUES ('stock','FFBC','us','2017-11-01',27.65,27.7,26.9,27.2,292419,'0');
INSERT INTO daily_summary VALUES ('stock','FFBW','us','2017-11-01',11.34,11.39,11.3399,11.365,26459,'0');
INSERT INTO daily_summary VALUES ('stock','FFC','us','2017-11-01',20.95,21.12,20.95,21.12,70690,'0');
INSERT INTO daily_summary VALUES ('stock','FFG','us','2017-11-01',78,78,76.25,76.65,14658,'0');
INSERT INTO daily_summary VALUES ('stock','FFHG','us','2017-11-01',26.75,26.75,26.4275,26.4994,8917,'0');
INSERT INTO daily_summary VALUES ('stock','FFHL','us','2017-11-01',2.5,2.5,2.4,2.46,5834,'0');
INSERT INTO daily_summary VALUES ('stock','FFIC','us','2017-11-01',28.46,29.46,24.59,27.86,91269,'0');
INSERT INTO daily_summary VALUES ('stock','FFIN','us','2017-11-01',46.05,46.1,44.75,45.1,224581,'0');
INSERT INTO daily_summary VALUES ('stock','FFIU','us','2017-11-01',25.0399,25.0399,25.0072,25.0072,2483,'0');
INSERT INTO daily_summary VALUES ('stock','FFIV','us','2017-11-01',120.94,121.88,119.91,120.6,993044,'0');
INSERT INTO daily_summary VALUES ('stock','FFKT','us','2017-11-01',42.05,42.55,41.15,41.2,7047,'0');
INSERT INTO daily_summary VALUES ('stock','FFNW','us','2017-11-01',16.74,16.78,16.53,16.62,12740,'0');
INSERT INTO daily_summary VALUES ('etf','FFR','us','2017-11-01',44.7572,44.8576,44.7572,44.8576,654,'0');
INSERT INTO daily_summary VALUES ('stock','FFSG','us','2017-11-01',25.2,25.2,25.01,25.02,135028,'0');
INSERT INTO daily_summary VALUES ('stock','FFTG','us','2017-11-01',25.1,25.1,25.01,25.01,658424,'0');
INSERT INTO daily_summary VALUES ('stock','FFTI','us','2017-11-01',25.0903,25.11,25.0849,25.11,35129,'0');
INSERT INTO daily_summary VALUES ('stock','FFTY','us','2017-11-01',33.83,33.83,32.8,33.08,351285,'0');
INSERT INTO daily_summary VALUES ('stock','FFWM','us','2017-11-01',18.64,18.8212,18.43,18.52,73549,'0');
INSERT INTO daily_summary VALUES ('etf','FGB','us','2017-11-01',6.66,6.66,6.54,6.58,79553,'0');
INSERT INTO daily_summary VALUES ('stock','FGBI','us','2017-11-01',26.65,26.9357,26.63,26.63,3245,'0');
INSERT INTO daily_summary VALUES ('etf','FGD','us','2017-11-01',25.86,25.86,25.73,25.75,37386,'0');
INSERT INTO daily_summary VALUES ('stock','FGEN','us','2017-11-01',56.05,57,54.6,54.95,322175,'0');
INSERT INTO daily_summary VALUES ('stock','FGL','us','2017-11-01',31.2,31.2,31.15,31.2,34482,'0');
INSERT INTO daily_summary VALUES ('etf','FGM','us','2017-11-01',49.25,49.26,49.13,49.2,6900,'0');
INSERT INTO daily_summary VALUES ('stock','FGP','us','2017-11-01',4.85,4.97,4.8397,4.95,489077,'0');
INSERT INTO daily_summary VALUES ('stock','FH','us','2017-11-01',1.22,1.23,1.2,1.22,13904,'0');
INSERT INTO daily_summary VALUES ('stock','FHB','us','2017-11-01',29.45,29.68,28.99,29.15,240646,'0');
INSERT INTO daily_summary VALUES ('etf','FHK','us','2017-11-01',43.23,43.23,43.05,43.12,6441,'0');
INSERT INTO daily_summary VALUES ('etf','FHLC','us','2017-11-01',39.28,39.4105,39.19,39.19,478632,'0');
INSERT INTO daily_summary VALUES ('stock','FHN','us','2017-11-01',18.798,18.878,18.47,18.48,2330500,'0');
INSERT INTO daily_summary VALUES ('stock','FHN_A','us','2017-11-01',25.51,25.573,25.51,25.51,2523,'0');
INSERT INTO daily_summary VALUES ('etf','FHY','us','2017-11-01',13.23,13.279,13.2,13.2107,14433,'0');
INSERT INTO daily_summary VALUES ('stock','FI','us','2017-11-01',6.67,6.82,6.54,6.66,1386440,'0');
INSERT INTO daily_summary VALUES ('stock','FIBK','us','2017-11-01',39.65,39.95,38.24,38.65,288078,'0');
INSERT INTO daily_summary VALUES ('stock','FIBR','us','2017-11-01',100.55,100.64,100.47,100.6,8692,'0');
INSERT INTO daily_summary VALUES ('stock','FICO','us','2017-11-01',146.13,148.58,143.04,143.94,240967,'0');
INSERT INTO daily_summary VALUES ('etf','FIDU','us','2017-11-01',36.99,37.0038,36.731,36.75,291269,'0');
INSERT INTO daily_summary VALUES ('stock','FIEE','us','2017-11-01',174.58,174.58,172.755,172.926,1836,'0');
INSERT INTO daily_summary VALUES ('etf','FIEU','us','2017-11-01',134.42,134.605,134.42,134.605,475,'0');
INSERT INTO daily_summary VALUES ('etf','FIF','us','2017-11-01',18.53,18.54,18.29,18.51,59845,'0');
INSERT INTO daily_summary VALUES ('stock','FIG','us','2017-11-01',7.83,8.01,7.79,7.87,6700080,'0');
INSERT INTO daily_summary VALUES ('etf','FIGY','us','2017-11-01',163.92,164.295,163.585,163.8,4198,'0');
INSERT INTO daily_summary VALUES ('stock','FIHD','us','2017-11-01',164.25,164.25,163.293,163.551,9515,'0');
INSERT INTO daily_summary VALUES ('stock','FII','us','2017-11-01',31.004,31.213,30.736,30.994,775577,'0');
INSERT INTO daily_summary VALUES ('etf','FILL','us','2017-11-01',20.83,20.9,20.82,20.9,4994,'0');
INSERT INTO daily_summary VALUES ('stock','FINL','us','2017-11-01',9.3,9.53,8.88,9.13,1814290,'0');
INSERT INTO daily_summary VALUES ('etf','FINU','us','2017-11-01',92,93.2474,91.5962,91.93,4984,'0');
INSERT INTO daily_summary VALUES ('stock','FINX','us','2017-11-01',21.8,21.8,21.41,21.441,74946,'0');
INSERT INTO daily_summary VALUES ('etf','FINZ','us','2017-11-01',9.6493,9.6493,9.6493,9.6493,155,'0');
INSERT INTO daily_summary VALUES ('stock','FIS','us','2017-11-01',94.12,94.89,92.47,93.11,2849760,'0');
INSERT INTO daily_summary VALUES ('stock','FISI','us','2017-11-01',32.85,33.1,31.95,32.25,88983,'0');
INSERT INTO daily_summary VALUES ('stock','FISV','us','2017-11-01',124,125.79,120.38,125.52,5165990,'0');
INSERT INTO daily_summary VALUES ('stock','FIT','us','2017-11-01',6.25,6.38,6.2,6.25,10527400,'0');
INSERT INTO daily_summary VALUES ('stock','FITB','us','2017-11-01',29.07,29.23,28.89,28.96,3649760,'0');
INSERT INTO daily_summary VALUES ('stock','FITBI','us','2017-11-01',28.52,28.8,28.43,28.79,71930,'0');
INSERT INTO daily_summary VALUES ('stock','FIV','us','2017-11-01',9.49,9.53,9.46,9.4844,56463,'0');
INSERT INTO daily_summary VALUES ('stock','FIVE','us','2017-11-01',55.4,55.76,54.82,55.42,689096,'0');
INSERT INTO daily_summary VALUES ('stock','FIVN','us','2017-11-01',25.49,25.51,24.52,24.89,175496,'0');
INSERT INTO daily_summary VALUES ('etf','FIW','us','2017-11-01',46.79,46.79,46.315,46.42,19367,'0');
INSERT INTO daily_summary VALUES ('stock','FIX','us','2017-11-01',44.328,44.427,42.031,42.381,287488,'0');
INSERT INTO daily_summary VALUES ('stock','FIXD','us','2017-11-01',50.79,50.85,50.71,50.79,28862,'0');
INSERT INTO daily_summary VALUES ('stock','FIZZ','us','2017-11-01',98.66,99.27,95.43,95.87,390544,'0');
INSERT INTO daily_summary VALUES ('etf','FJP','us','2017-11-01',58.27,58.3745,58.25,58.25,20290,'0');
INSERT INTO daily_summary VALUES ('etf','FKO','us','2017-11-01',28.58,28.58,28.472,28.472,287,'0');
INSERT INTO daily_summary VALUES ('etf','FKU','us','2017-11-01',39.13,39.13,38.87,38.87,1814,'0');
INSERT INTO daily_summary VALUES ('stock','FL','us','2017-11-01',30.19,30.89,29.7,29.91,3000270,'0');
INSERT INTO daily_summary VALUES ('etf','FLAG','us','2017-11-01',39.66,39.66,39.24,39.25,3124,'0');
INSERT INTO daily_summary VALUES ('stock','FLC','us','2017-11-01',21.4,21.481,21.38,21.46,28390,'0');
INSERT INTO daily_summary VALUES ('stock','FLCO','us','2017-11-01',24.78,24.78,24.78,24.78,399,'0');
INSERT INTO daily_summary VALUES ('stock','FLDM','us','2017-11-01',5.85,5.93,5.68,5.7,243769,'0');
INSERT INTO daily_summary VALUES ('stock','FLEU','us','2017-11-01',156.77,157.399,155.835,156.28,22777,'0');
INSERT INTO daily_summary VALUES ('stock','FLEX','us','2017-11-01',18,18.16,17.8,18.03,6409100,'0');
INSERT INTO daily_summary VALUES ('etf','FLGE','us','2017-11-01',212.33,212.99,211.004,211.64,12428,'0');
INSERT INTO daily_summary VALUES ('stock','FLGT','us','2017-11-01',3.95,4.1532,3.81,3.89,42160,'0');
INSERT INTO daily_summary VALUES ('stock','FLIC','us','2017-11-01',31.6,31.6,29.25,29.9,108122,'0');
INSERT INTO daily_summary VALUES ('stock','FLIO','us','2017-11-01',29.62,29.62,29.5513,29.5513,347,'0');
INSERT INTO daily_summary VALUES ('stock','FLIR','us','2017-11-01',47.07,47.15,46.23,46.55,1341040,'0');
INSERT INTO daily_summary VALUES ('stock','FLKS','us','2017-11-01',2.71,3.1,2.7,3.04,176033,'0');
INSERT INTO daily_summary VALUES ('stock','FLL','us','2017-11-01',2.76,2.85,2.72,2.83,49823,'0');
INSERT INTO daily_summary VALUES ('stock','FLLV','us','2017-11-01',29.16,29.16,29.16,29.16,231,'0');
INSERT INTO daily_summary VALUES ('etf','FLM','us','2017-11-01',58.406,58.472,58.22,58.259,1130,'0');
INSERT INTO daily_summary VALUES ('etf','FLN','us','2017-11-01',21.57,21.65,21.34,21.34,5645,'0');
INSERT INTO daily_summary VALUES ('stock','FLO','us','2017-11-01',19.1,19.14,18.955,19.01,575968,'0');
INSERT INTO daily_summary VALUES ('etf','FLOT','us','2017-11-01',50.91,50.93,50.88,50.88,952180,'0');
INSERT INTO daily_summary VALUES ('stock','FLOW','us','2017-11-01',40.1,41.7099,38.95,39.27,355528,'0');
INSERT INTO daily_summary VALUES ('stock','FLQD','us','2017-11-01',28.1699,28.1699,28.11,28.11,685,'0');
INSERT INTO daily_summary VALUES ('stock','FLQE','us','2017-11-01',32.5,32.69,32.49,32.49,11140,'0');
INSERT INTO daily_summary VALUES ('stock','FLQG','us','2017-11-01',29.95,29.95,29.876,29.9399,1591,'0');
INSERT INTO daily_summary VALUES ('stock','FLQH','us','2017-11-01',25.19,25.25,25.19,25.25,7103,'0');
INSERT INTO daily_summary VALUES ('stock','FLQL','us','2017-11-01',27.415,27.415,27.415,27.415,300,'0');
INSERT INTO daily_summary VALUES ('stock','FLQM','us','2017-11-01',25.86,25.86,25.86,25.86,0,'0');
INSERT INTO daily_summary VALUES ('stock','FLQS','us','2017-11-01',25.83,25.83,25.83,25.83,0,'0');
INSERT INTO daily_summary VALUES ('stock','FLR','us','2017-11-01',43.43,44.15,43.14,43.94,2118950,'0');
INSERT INTO daily_summary VALUES ('etf','FLRN','us','2017-11-01',30.73,30.75,30.7,30.73,500374,'0');
INSERT INTO daily_summary VALUES ('stock','FLS','us','2017-11-01',44.44,44.83,43.89,43.99,2270990,'0');
INSERT INTO daily_summary VALUES ('stock','FLT','us','2017-11-01',166.5,167.82,164.01,164.31,1435440,'0');
INSERT INTO daily_summary VALUES ('etf','FLTB','us','2017-11-01',50.37,50.419,50.31,50.35,21494,'0');
INSERT INTO daily_summary VALUES ('etf','FLTR','us','2017-11-01',25.22,25.23,25.21,25.23,29119,'0');
INSERT INTO daily_summary VALUES ('stock','FLWS','us','2017-11-01',9.5,9.5,9.05,9.3,243776,'0');
INSERT INTO daily_summary VALUES ('stock','FLXN','us','2017-11-01',22.25,22.39,21.63,22.1,1137280,'0');
INSERT INTO daily_summary VALUES ('stock','FLXS','us','2017-11-01',50.56,50.56,47.07,47.6,46010,'0');
INSERT INTO daily_summary VALUES ('stock','FLY','us','2017-11-01',13.96,14.1,13.88,13.89,72738,'0');
INSERT INTO daily_summary VALUES ('etf','FM','us','2017-11-01',31.93,32.1,31.7519,31.97,310712,'0');
INSERT INTO daily_summary VALUES ('stock','FMAO','us','2017-11-01',37.4,38.5409,36.33,36.51,7983,'0');
INSERT INTO daily_summary VALUES ('etf','FMAT','us','2017-11-01',34.44,34.569,34.2651,34.42,71984,'0');
INSERT INTO daily_summary VALUES ('etf','FMB','us','2017-11-01',53.2601,53.3254,53.22,53.2496,40318,'0');
INSERT INTO daily_summary VALUES ('stock','FMBH','us','2017-11-01',38.88,39,37.62,38.29,19566,'0');
INSERT INTO daily_summary VALUES ('stock','FMBI','us','2017-11-01',23.23,23.36,22.82,23.01,385116,'0');
INSERT INTO daily_summary VALUES ('stock','FMC','us','2017-11-01',93.84,93.91,91.25,91.9,1282050,'0');
INSERT INTO daily_summary VALUES ('stock','FMCIR','us','2017-11-01',0.35,0.369,0.345,0.345,3700,'0');
INSERT INTO daily_summary VALUES ('stock','FMCIU','us','2017-11-01',10.16,10.16,10.15,10.15,48500,'0');
INSERT INTO daily_summary VALUES ('stock','FMCIW','us','2017-11-01',0.35,0.35,0.35,0.35,870,'0');
INSERT INTO daily_summary VALUES ('etf','FMF','us','2017-11-01',47.19,47.34,46.88,46.88,12220,'0');
INSERT INTO daily_summary VALUES ('stock','FMI','us','2017-11-01',45,45.05,41.7,42.05,180088,'0');
INSERT INTO daily_summary VALUES ('etf','FMK','us','2017-11-01',32.52,32.546,32.46,32.48,1702,'0');
INSERT INTO daily_summary VALUES ('stock','FMN','us','2017-11-01',14.28,14.37,14.26,14.37,6311,'0');
INSERT INTO daily_summary VALUES ('stock','FMNB','us','2017-11-01',14.75,14.8,14.25,14.35,24584,'0');
INSERT INTO daily_summary VALUES ('stock','FMO','us','2017-11-01',11.94,12.1,11.91,12.02,213932,'0');
INSERT INTO daily_summary VALUES ('stock','FMS','us','2017-11-01',48.65,48.67,48.08,48.49,695686,'0');
INSERT INTO daily_summary VALUES ('stock','FMSA','us','2017-11-01',4.4,4.47,4.02,4.2,4300340,'0');
INSERT INTO daily_summary VALUES ('stock','FMX','us','2017-11-01',87.097,87.265,85.906,86.253,459337,'0');
INSERT INTO daily_summary VALUES ('etf','FMY','us','2017-11-01',14.33,14.44,14.29,14.3093,8855,'0');
INSERT INTO daily_summary VALUES ('stock','FN','us','2017-11-01',37.65,37.65,35.84,35.84,494124,'0');
INSERT INTO daily_summary VALUES ('stock','FNB','us','2017-11-01',13.54,13.66,13.32,13.4,2604790,'0');
INSERT INTO daily_summary VALUES ('stock','FNBG','us','2017-11-01',34.57,34.64,34.11,34.11,2633,'0');
INSERT INTO daily_summary VALUES ('stock','FNB_E','us','2017-11-01',29.32,29.33,29.15,29.22,2451,'0');
INSERT INTO daily_summary VALUES ('stock','FNCF','us','2017-11-01',34.3114,34.3114,34.3114,34.3114,145,'0');
INSERT INTO daily_summary VALUES ('etf','FNCL','us','2017-11-01',39.17,39.3112,38.97,39.05,268596,'0');
INSERT INTO daily_summary VALUES ('stock','FND','us','2017-11-01',37.83,38.88,37.81,38.79,395841,'0');
INSERT INTO daily_summary VALUES ('etf','FNDA','us','2017-11-01',36.81,36.82,36.312,36.48,177629,'0');
INSERT INTO daily_summary VALUES ('etf','FNDB','us','2017-11-01',35.63,35.7,35.55,35.5825,38226,'0');
INSERT INTO daily_summary VALUES ('etf','FNDC','us','2017-11-01',35.66,35.72,35.56,35.59,148817,'0');
INSERT INTO daily_summary VALUES ('etf','FNDE','us','2017-11-01',29.74,29.8124,29.58,29.59,213534,'0');
INSERT INTO daily_summary VALUES ('etf','FNDF','us','2017-11-01',30.49,30.53,30.39,30.43,361935,'0');
INSERT INTO daily_summary VALUES ('etf','FNDX','us','2017-11-01',35.79,35.88,35.68,35.74,261148,'0');
INSERT INTO daily_summary VALUES ('stock','FNF','us','2017-11-01',37.62,38.04,37.59,37.78,1308860,'0');
INSERT INTO daily_summary VALUES ('stock','FNFV','us','2017-11-01',17.4,17.55,17.1,17.5,490312,'0');
INSERT INTO daily_summary VALUES ('stock','FNG','us','2017-11-01',22.93,22.9318,22.466,22.6205,70406,'0');
INSERT INTO daily_summary VALUES ('stock','FNGN','us','2017-11-01',36.4,37.05,35.15,35.2,205013,'0');
INSERT INTO daily_summary VALUES ('stock','FNHC','us','2017-11-01',15.51,15.74,14.89,15.03,69355,'0');
INSERT INTO daily_summary VALUES ('etf','FNI','us','2017-11-01',39.15,39.289,38.89,38.95,17890,'0');
INSERT INTO daily_summary VALUES ('stock','FNJN','us','2017-11-01',2.1,2.105,1.99,2.06,72482,'0');
INSERT INTO daily_summary VALUES ('etf','FNK','us','2017-11-01',34.82,34.8903,34.6,34.7545,7142,'0');
INSERT INTO daily_summary VALUES ('stock','FNLC','us','2017-11-01',31.59,32.1754,30.7,30.73,18602,'0');
INSERT INTO daily_summary VALUES ('stock','FNSR','us','2017-11-01',23.9,23.9,22.05,22.09,4261210,'0');
INSERT INTO daily_summary VALUES ('stock','FNTE','us','2017-11-01',9.78,9.79,9.78,9.79,200,'0');
INSERT INTO daily_summary VALUES ('stock','FNTEW','us','2017-11-01',1.25,1.25,1.25,1.25,1000,'0');
INSERT INTO daily_summary VALUES ('stock','FNV','us','2017-11-01',79.83,81.07,79.58,79.82,638036,'0');
INSERT INTO daily_summary VALUES ('stock','FNWB','us','2017-11-01',17,17,16.88,16.94,17735,'0');
INSERT INTO daily_summary VALUES ('etf','FNX','us','2017-11-01',63.68,63.68,62.9576,63.09,21373,'0');
INSERT INTO daily_summary VALUES ('etf','FNY','us','2017-11-01',38.16,38.16,37.5901,37.75,9761,'0');
INSERT INTO daily_summary VALUES ('stock','FOE','us','2017-11-01',24.07,24.07,23.64,23.86,500651,'0');
INSERT INTO daily_summary VALUES ('stock','FOF','us','2017-11-01',13.34,13.34,13.21,13.225,72447,'0');
INSERT INTO daily_summary VALUES ('stock','FOGO','us','2017-11-01',11.05,11.05,10.75,10.8,102568,'0');
INSERT INTO daily_summary VALUES ('stock','FOLD','us','2017-11-01',14.29,14.47,13.73,13.89,3592250,'0');
INSERT INTO daily_summary VALUES ('stock','FOMX','us','2017-11-01',6.12,6.28,5.945,6.07,57675,'0');
INSERT INTO daily_summary VALUES ('etf','FONE','us','2017-11-01',52.06,52.06,51.48,51.56,1146,'0');
INSERT INTO daily_summary VALUES ('stock','FONR','us','2017-11-01',31.85,32.8,31.15,32.75,34169,'0');
INSERT INTO daily_summary VALUES ('stock','FOR','us','2017-11-01',17.95,17.95,17.55,17.6,200843,'0');
INSERT INTO daily_summary VALUES ('stock','FORD','us','2017-11-01',1.17,1.2,1.17,1.2,38918,'0');
INSERT INTO daily_summary VALUES ('stock','FORK','us','2017-11-01',4.15,4.35,4,4.35,65213,'0');
INSERT INTO daily_summary VALUES ('stock','FORM','us','2017-11-01',16.55,16.85,15.8,15.8,3290050,'0');
INSERT INTO daily_summary VALUES ('stock','FORR','us','2017-11-01',44.1,45.1,44.1,44.85,46493,'0');
INSERT INTO daily_summary VALUES ('stock','FORTY','us','2017-11-01',41.02,41.02,41.02,41.02,221,'0');
INSERT INTO daily_summary VALUES ('stock','FOSL','us','2017-11-01',7.95,8.08,7.47,7.85,2186800,'0');
INSERT INTO daily_summary VALUES ('stock','FOX','us','2017-11-01',25.61,25.83,25.35,25.38,1451170,'0');
INSERT INTO daily_summary VALUES ('stock','FOXA','us','2017-11-01',26.29,26.515,25.95,25.98,4804150,'0');
INSERT INTO daily_summary VALUES ('stock','FOXF','us','2017-11-01',43.1,43.5,42.05,42.2,249929,'0');
INSERT INTO daily_summary VALUES ('etf','FPA','us','2017-11-01',35.11,35.36,35.0911,35.27,10148,'0');
INSERT INTO daily_summary VALUES ('stock','FPAY','us','2017-11-01',4.178,4.2501,4.178,4.25,1113,'0');
INSERT INTO daily_summary VALUES ('etf','FPE','us','2017-11-01',20.15,20.18,20.1302,20.18,662975,'0');
INSERT INTO daily_summary VALUES ('stock','FPEI','us','2017-11-01',20.29,20.3048,20.26,20.26,24496,'0');
INSERT INTO daily_summary VALUES ('etf','FPF','us','2017-11-01',24.72,24.85,24.65,24.67,117811,'0');
INSERT INTO daily_summary VALUES ('stock','FPH','us','2017-11-01',12.84,12.98,12.61,12.81,220249,'0');
INSERT INTO daily_summary VALUES ('stock','FPI','us','2017-11-01',8.4,8.6,8.35,8.42,201454,'0');
INSERT INTO daily_summary VALUES ('stock','FPI_B','us','2017-11-01',26.9,26.9,26.66,26.79,14760,'0');
INSERT INTO daily_summary VALUES ('etf','FPL','us','2017-11-01',11.93,12.01,11.895,11.96,140294,'0');
INSERT INTO daily_summary VALUES ('stock','FPP','us','2017-11-01',0.325,0.34,0.32,0.3279,27274,'0');
INSERT INTO daily_summary VALUES ('stock','FPRX','us','2017-11-01',45.33,46.92,43.26,43.32,285200,'0');
INSERT INTO daily_summary VALUES ('stock','FPT','us','2017-11-01',13.35,13.42,13.35,13.42,4736,'0');
INSERT INTO daily_summary VALUES ('etf','FPX','us','2017-11-01',66.16,66.18,65.6722,65.91,279229,'0');
INSERT INTO daily_summary VALUES ('etf','FPXI','us','2017-11-01',35.2513,35.2513,35.18,35.18,729,'0');
INSERT INTO daily_summary VALUES ('stock','FQAL','us','2017-11-01',30.61,30.61,30.4431,30.46,6860,'0');
INSERT INTO daily_summary VALUES ('stock','FR','us','2017-11-01',31,31.34,30.88,31.22,745624,'0');
INSERT INTO daily_summary VALUES ('stock','FRA','us','2017-11-01',14.34,14.34,14.26,14.27,61952,'0');
INSERT INTO daily_summary VALUES ('stock','FRAC','us','2017-11-01',15.71,15.99,15.26,15.56,901934,'0');
INSERT INTO daily_summary VALUES ('etf','FRAK','us','2017-11-01',15,15.2,15,15.1203,41033,'0');
INSERT INTO daily_summary VALUES ('stock','FRAN','us','2017-11-01',6.49,6.53,6.35,6.42,704579,'0');
INSERT INTO daily_summary VALUES ('stock','FRBA','us','2017-11-01',13,13.25,13,13.1,67514,'0');
INSERT INTO daily_summary VALUES ('stock','FRBK','us','2017-11-01',9.35,9.35,8.9,9.05,94502,'0');
INSERT INTO daily_summary VALUES ('stock','FRC','us','2017-11-01',97.89,98.3,96.69,96.86,1342520,'0');
INSERT INTO daily_summary VALUES ('stock','FRC_C','us','2017-11-01',25.2092,25.2351,25.18,25.2351,2167,'0');
INSERT INTO daily_summary VALUES ('stock','FRC_D','us','2017-11-01',25.2228,25.35,25.2228,25.34,31639,'0');
INSERT INTO daily_summary VALUES ('stock','FRC_E','us','2017-11-01',26.59,26.868,26.55,26.81,8790,'0');
INSERT INTO daily_summary VALUES ('stock','FRC_F','us','2017-11-01',26.66,26.89,26.66,26.88,4816,'0');
INSERT INTO daily_summary VALUES ('stock','FRC_G','us','2017-11-01',25.75,25.84,25.66,25.84,9130,'0');
INSERT INTO daily_summary VALUES ('stock','FRC_H','us','2017-11-01',25.21,25.26,25.13,25.21,19850,'0');
INSERT INTO daily_summary VALUES ('stock','FRD','us','2017-11-01',5.93,6.04,5.93,6,13643,'0');
INSERT INTO daily_summary VALUES ('stock','FRED','us','2017-11-01',4.46,4.549,4.36,4.5,763900,'0');
INSERT INTO daily_summary VALUES ('etf','FREL','us','2017-11-01',24.44,24.45,24.29,24.3,319896,'0');
INSERT INTO daily_summary VALUES ('stock','FRGI','us','2017-11-01',16.65,16.8,16.35,16.8,242450,'0');
INSERT INTO daily_summary VALUES ('etf','FRI','us','2017-11-01',22.99,23.14,22.99,23.09,45062,'0');
INSERT INTO daily_summary VALUES ('stock','FRME','us','2017-11-01',43.31,43.585,42.3,42.61,113953,'0');
INSERT INTO daily_summary VALUES ('etf','FRN','us','2017-11-01',14.6,14.6,14.48,14.5,524350,'0');
INSERT INTO daily_summary VALUES ('stock','FRO','us','2017-11-01',6.24,6.38,6.2,6.26,531937,'0');
INSERT INTO daily_summary VALUES ('stock','FRPH','us','2017-11-01',44.5,44.75,43.65,43.7,9891,'0');
INSERT INTO daily_summary VALUES ('stock','FRPT','us','2017-11-01',15.7,15.85,15.05,15.25,169601,'0');
INSERT INTO daily_summary VALUES ('stock','FRSH','us','2017-11-01',5.56,5.66,5.56,5.66,84262,'0');
INSERT INTO daily_summary VALUES ('stock','FRSX','us','2017-11-01',5.7,5.7,5.55,5.598,11933,'0');
INSERT INTO daily_summary VALUES ('stock','FRT','us','2017-11-01',120.63,121.86,119.855,121.7,488622,'0');
INSERT INTO daily_summary VALUES ('stock','FRTA','us','2017-11-01',5.03,5.13,4.85,5,559023,'0');
INSERT INTO daily_summary VALUES ('stock','FRT_C','us','2017-11-01',24.74,24.75,24.5601,24.649,19762,'0');
INSERT INTO daily_summary VALUES ('stock','FSAC','us','2017-11-01',9.74,9.74,9.73,9.73,38997,'0');
INSERT INTO daily_summary VALUES ('stock','FSACU','us','2017-11-01',10.2,10.2,10.2,10.2,8076,'0');
INSERT INTO daily_summary VALUES ('stock','FSACW','us','2017-11-01',1.12,1.13,1.11,1.11,17000,'0');
INSERT INTO daily_summary VALUES ('stock','FSAM','us','2017-11-01',4.1,4.15,4.05,4.1,157641,'0');
INSERT INTO daily_summary VALUES ('stock','FSB','us','2017-11-01',34.45,35,34.1,34.2,56548,'0');
INSERT INTO daily_summary VALUES ('stock','FSBW','us','2017-11-01',54.34,56.725,52.793,56.316,25967,'0');
INSERT INTO daily_summary VALUES ('stock','FSCT','us','2017-11-01',25.81,26.4,25.573,25.87,241048,'0');
INSERT INTO daily_summary VALUES ('etf','FSD','us','2017-11-01',16.86,16.86,16.72,16.77,164740,'0');
INSERT INTO daily_summary VALUES ('stock','FSFG','us','2017-11-01',55.16,58,55,58,1116,'0');
INSERT INTO daily_summary VALUES ('stock','FSI','us','2017-11-01',1.86,1.88,1.84,1.87,9523,'0');
INSERT INTO daily_summary VALUES ('stock','FSIC','us','2017-11-01',7.8,7.95,7.75,7.85,1522110,'0');
INSERT INTO daily_summary VALUES ('stock','FSLR','us','2017-11-01',55.36,57.869,54.71,57.21,5258280,'0');
INSERT INTO daily_summary VALUES ('stock','FSM','us','2017-11-01',4.25,4.35,4.23,4.28,963083,'0');
INSERT INTO daily_summary VALUES ('stock','FSNN','us','2017-11-01',2.82,2.88,2.64,2.68,284105,'0');
INSERT INTO daily_summary VALUES ('stock','FSP','us','2017-11-01',10,10.4,10,10.25,529244,'0');
INSERT INTO daily_summary VALUES ('stock','FSS','us','2017-11-01',21.448,21.547,20.84,21.059,401500,'0');
INSERT INTO daily_summary VALUES ('etf','FSTA','us','2017-11-01',32.04,32.15,31.99,32.03,80266,'0');
INSERT INTO daily_summary VALUES ('stock','FSTR','us','2017-11-01',25.2,25.45,24.2,24.4,53181,'0');
INSERT INTO daily_summary VALUES ('stock','FSV','us','2017-11-01',69.36,69.91,69.08,69.5,21596,'0');
INSERT INTO daily_summary VALUES ('etf','FSZ','us','2017-11-01',51.52,51.575,51.365,51.37,5045,'0');
INSERT INTO daily_summary VALUES ('stock','FT','us','2017-11-01',7.2445,7.2698,7.24,7.26,26313,'0');
INSERT INTO daily_summary VALUES ('etf','FTA','us','2017-11-01',51.73,51.8774,51.6404,51.69,42444,'0');
INSERT INTO daily_summary VALUES ('stock','FTAG','us','2017-11-01',27.96,27.96,27.96,27.96,252,'0');
INSERT INTO daily_summary VALUES ('stock','FTAI','us','2017-11-01',18.29,18.4,18.15,18.25,128250,'0');
INSERT INTO daily_summary VALUES ('etf','FTC','us','2017-11-01',59.87,59.99,59.27,59.3601,23085,'0');
INSERT INTO daily_summary VALUES ('etf','FTCS','us','2017-11-01',49.17,49.17,48.9524,48.985,42375,'0');
INSERT INTO daily_summary VALUES ('stock','FTD','us','2017-11-01',10.93,11,10.49,10.62,104794,'0');
INSERT INTO daily_summary VALUES ('etf','FTEC','us','2017-11-01',49.8,49.82,49.27,49.52,374971,'0');
INSERT INTO daily_summary VALUES ('stock','FTEK','us','2017-11-01',0.92,0.978,0.92,0.95,19976,'0');
INSERT INTO daily_summary VALUES ('stock','FTF','us','2017-11-01',11.97,11.97,11.91,11.92,107619,'0');
INSERT INTO daily_summary VALUES ('stock','FTFT','us','2017-11-01',1.49,1.54,1.3879,1.48,17913,'0');
INSERT INTO daily_summary VALUES ('etf','FTGC','us','2017-11-01',20.79,20.8032,20.6805,20.71,159100,'0');
INSERT INTO daily_summary VALUES ('etf','FTHI','us','2017-11-01',22.891,22.891,22.76,22.79,5171,'0');
INSERT INTO daily_summary VALUES ('stock','FTI','us','2017-11-01',27.81,28.389,27.615,27.76,7020630,'0');
INSERT INTO daily_summary VALUES ('stock','FTK','us','2017-11-01',5.01,5.255,4.82,4.85,725708,'0');
INSERT INTO daily_summary VALUES ('etf','FTLB','us','2017-11-01',22.5936,22.5936,22.5936,22.5936,334,'0');
INSERT INTO daily_summary VALUES ('etf','FTLS','us','2017-11-01',37.94,37.94,37.74,37.85,28744,'0');
INSERT INTO daily_summary VALUES ('stock','FTNT','us','2017-11-01',39.42,39.46,38.32,38.77,2354580,'0');
INSERT INTO daily_summary VALUES ('stock','FTR','us','2017-11-01',12.06,12.06,8.825,8.86,20818500,'0');
INSERT INTO daily_summary VALUES ('stock','FTRI','us','2017-11-01',12.16,12.2,12.16,12.17,4607,'0');
INSERT INTO daily_summary VALUES ('stock','FTRPR','us','2017-11-01',20,20,15.51,15.55,717650,'0');
INSERT INTO daily_summary VALUES ('stock','FTS','us','2017-11-01',36.9,36.9799,36.53,36.69,125393,'0');
INSERT INTO daily_summary VALUES ('etf','FTSD','us','2017-11-01',96.08,96.21,96.08,96.2099,4908,'0');
INSERT INTO daily_summary VALUES ('etf','FTSL','us','2017-11-01',48.31,48.35,48.29,48.32,236056,'0');
INSERT INTO daily_summary VALUES ('etf','FTSM','us','2017-11-01',60.02,60.03,60.01,60.03,147929,'0');
INSERT INTO daily_summary VALUES ('stock','FTV','us','2017-11-01',72.73,73.07,72.25,72.86,1222000,'0');
INSERT INTO daily_summary VALUES ('stock','FTVA','us','2017-11-01',25.08,25.14,25,25.07,315860,'0');
INSERT INTO daily_summary VALUES ('stock','FTXD','us','2017-11-01',19.6,19.6,19.6,19.6,261,'0');
INSERT INTO daily_summary VALUES ('stock','FTXG','us','2017-11-01',19.61,19.8335,19.61,19.8335,1238,'0');
INSERT INTO daily_summary VALUES ('stock','FTXL','us','2017-11-01',31.89,31.89,30.8701,31.02,11006,'0');
INSERT INTO daily_summary VALUES ('stock','FTXN','us','2017-11-01',19.41,19.41,19.32,19.37,4337,'0');
INSERT INTO daily_summary VALUES ('stock','FTXO','us','2017-11-01',28.238,28.38,28.0865,28.09,63271,'0');
INSERT INTO daily_summary VALUES ('stock','FTXR','us','2017-11-01',24.08,24.0801,24.08,24.0801,301,'0');
INSERT INTO daily_summary VALUES ('stock','FUL','us','2017-11-01',56.97,57.1825,55.78,56.5,346219,'0');
INSERT INTO daily_summary VALUES ('stock','FULT','us','2017-11-01',18.35,18.425,17.95,18.1,645994,'0');
INSERT INTO daily_summary VALUES ('stock','FUN','us','2017-11-01',62.5,63.43,62.2538,62.81,132752,'0');
INSERT INTO daily_summary VALUES ('stock','FUNC','us','2017-11-01',17.55,17.55,17.4,17.5,765,'0');
INSERT INTO daily_summary VALUES ('stock','FUND','us','2017-11-01',7.91,7.91,7.82,7.8392,25223,'0');
INSERT INTO daily_summary VALUES ('stock','FUSB','us','2017-11-01',11.1668,11.31,11.157,11.2542,4166,'0');
INSERT INTO daily_summary VALUES ('stock','FUT','us','2017-11-01',40.6,40.6,40.19,40.2,2233,'0');
INSERT INTO daily_summary VALUES ('etf','FUTY','us','2017-11-01',35.97,36,35.65,35.68,67131,'0');
INSERT INTO daily_summary VALUES ('stock','FUV','us','2017-11-01',4.25,4.44,4.24,4.44,53723,'0');
INSERT INTO daily_summary VALUES ('etf','FV','us','2017-11-01',27.17,27.2,26.91,26.97,334921,'0');
INSERT INTO daily_summary VALUES ('stock','FVAL','us','2017-11-01',31.16,31.22,31.04,31.14,27189,'0');
INSERT INTO daily_summary VALUES ('stock','FVC','us','2017-11-01',25.28,25.3348,25.095,25.15,27986,'0');
INSERT INTO daily_summary VALUES ('etf','FVD','us','2017-11-01',30.28,30.3,30.112,30.12,837276,'0');
INSERT INTO daily_summary VALUES ('stock','FVE','us','2017-11-01',1.45,1.5,1.45,1.475,21070,'0');
INSERT INTO daily_summary VALUES ('etf','FVL','us','2017-11-01',23.27,23.27,22.969,23.06,13527,'0');
INSERT INTO daily_summary VALUES ('etf','FWDB','us','2017-11-01',26.14,26.15,26.028,26.028,1446,'0');
INSERT INTO daily_summary VALUES ('etf','FWDD','us','2017-11-01',51.09,51.14,50.63,50.6558,841,'0');
INSERT INTO daily_summary VALUES ('stock','FWONA','us','2017-11-01',36.64,36.98,36.275,36.73,139969,'0');
INSERT INTO daily_summary VALUES ('stock','FWONK','us','2017-11-01',38.19,38.76,38.05,38.46,936134,'0');
INSERT INTO daily_summary VALUES ('stock','FWP','us','2017-11-01',5.62,5.8,5.44,5.58,26239,'0');
INSERT INTO daily_summary VALUES ('stock','FWRD','us','2017-11-01',57.83,57.97,54.02,57.1,113505,'0');
INSERT INTO daily_summary VALUES ('etf','FXA','us','2017-11-01',76.76,77.01,76.76,76.77,14426,'0');
INSERT INTO daily_summary VALUES ('etf','FXB','us','2017-11-01',129.17,129.33,128.805,128.84,49762,'0');
INSERT INTO daily_summary VALUES ('etf','FXC','us','2017-11-01',76.45,76.68,76.45,76.67,116362,'0');
INSERT INTO daily_summary VALUES ('etf','FXD','us','2017-11-01',38.83,38.9804,38.6803,38.76,101867,'0');
INSERT INTO daily_summary VALUES ('etf','FXE','us','2017-11-01',112.03,112.32,111.98,112.09,531789,'0');
INSERT INTO daily_summary VALUES ('stock','FXEP','us','2017-11-01',28.8253,28.8253,28.8253,28.8253,150,'0');
INSERT INTO daily_summary VALUES ('etf','FXEU','us','2017-11-01',24.18,24.19,24.072,24.072,1119,'0');
INSERT INTO daily_summary VALUES ('etf','FXF','us','2017-11-01',94.36,94.5,94.2,94.2,30600,'0');
INSERT INTO daily_summary VALUES ('etf','FXG','us','2017-11-01',45.66,45.85,45.55,45.55,38724,'0');
INSERT INTO daily_summary VALUES ('etf','FXH','us','2017-11-01',67.85,67.9,67.4,67.45,369720,'0');
INSERT INTO daily_summary VALUES ('etf','FXI','us','2017-11-01',46.42,46.49,46.2,46.23,11075000,'0');
INSERT INTO daily_summary VALUES ('etf','FXL','us','2017-11-01',51.46,51.505,50.58,50.9,400215,'0');
INSERT INTO daily_summary VALUES ('etf','FXN','us','2017-11-01',14.33,14.59,14.33,14.54,385427,'0');
INSERT INTO daily_summary VALUES ('etf','FXO','us','2017-11-01',30.47,30.52,30.28,30.33,346625,'0');
INSERT INTO daily_summary VALUES ('etf','FXP','us','2017-11-01',17.87,18.02,17.791,18,60851,'0');
INSERT INTO daily_summary VALUES ('etf','FXR','us','2017-11-01',38.23,38.23,37.9598,37.98,165565,'0');
INSERT INTO daily_summary VALUES ('etf','FXS','us','2017-11-01',113.86,113.86,113.655,113.655,894,'0');
INSERT INTO daily_summary VALUES ('etf','FXU','us','2017-11-01',27.52,27.52,27.1,27.11,226610,'0');
INSERT INTO daily_summary VALUES ('etf','FXY','us','2017-11-01',84.09,84.39,84.03,84.14,85656,'0');
INSERT INTO daily_summary VALUES ('etf','FXZ','us','2017-11-01',42.08,42.409,41.88,41.94,153725,'0');
INSERT INTO daily_summary VALUES ('etf','FYC','us','2017-11-01',42.01,42.01,41.2215,41.31,9424,'0');
INSERT INTO daily_summary VALUES ('etf','FYLD','us','2017-11-01',25.8147,25.9,25.711,25.7753,8914,'0');
INSERT INTO daily_summary VALUES ('etf','FYT','us','2017-11-01',36.04,36.04,35.3666,35.61,5563,'0');
INSERT INTO daily_summary VALUES ('etf','FYX','us','2017-11-01',59.82,59.82,58.83,59.04,15642,'0');
INSERT INTO daily_summary VALUES ('stock','G','us','2017-11-01',30.49,30.58,30.18,30.22,593528,'0');
INSERT INTO daily_summary VALUES ('etf','GAA','us','2017-11-01',27.14,27.14,27,27.08,13176,'0');
INSERT INTO daily_summary VALUES ('stock','GAB','us','2017-11-01',6.22,6.23,6.19,6.2,1012100,'0');
INSERT INTO daily_summary VALUES ('stock','GABC','us','2017-11-01',36.066,36.066,34.935,35.13,31277,'0');
INSERT INTO daily_summary VALUES ('stock','GAB_D','us','2017-11-01',26.18,26.19,26.18,26.19,617,'0');
INSERT INTO daily_summary VALUES ('stock','GAB_G','us','2017-11-01',24.77,24.77,24.6577,24.76,2276,'0');
INSERT INTO daily_summary VALUES ('stock','GAB_H','us','2017-11-01',24.9608,24.99,24.9608,24.99,906,'0');
INSERT INTO daily_summary VALUES ('stock','GAB_J','us','2017-11-01',25.4819,25.4819,25.4819,25.4819,1400,'0');
INSERT INTO daily_summary VALUES ('stock','GAIA','us','2017-11-01',12.15,12.5,12.05,12.2,26855,'0');
INSERT INTO daily_summary VALUES ('stock','GAIN','us','2017-11-01',10.15,10.28,10.125,10.26,119123,'0');
INSERT INTO daily_summary VALUES ('stock','GAINM','us','2017-11-01',25.5271,25.54,25.5072,25.5158,1321,'0');
INSERT INTO daily_summary VALUES ('stock','GAINN','us','2017-11-01',25.5001,25.5511,25.5,25.5304,1636,'0');
INSERT INTO daily_summary VALUES ('stock','GAINO','us','2017-11-01',25.26,25.31,25.26,25.31,565,'0');
INSERT INTO daily_summary VALUES ('etf','GAL','us','2017-11-01',37.68,37.68,37.5,37.53,22739,'0');
INSERT INTO daily_summary VALUES ('stock','GALE','us','2017-11-01',0.315,0.33,0.306,0.31,464156,'0');
INSERT INTO daily_summary VALUES ('stock','GALT','us','2017-11-01',2.45,2.5,2.365,2.38,108050,'0');
INSERT INTO daily_summary VALUES ('stock','GAM','us','2017-11-01',32.986,33.059,32.839,32.921,35504,'0');
INSERT INTO daily_summary VALUES ('stock','GAMR','us','2017-11-01',46.23,46.23,45.89,45.9,11974,'0');
INSERT INTO daily_summary VALUES ('stock','GAM_B','us','2017-11-01',26.53,27.09,26.53,26.58,3732,'0');
INSERT INTO daily_summary VALUES ('stock','GARD','us','2017-11-01',23.392,23.392,23.392,23.392,0,'0');
INSERT INTO daily_summary VALUES ('stock','GARS','us','2017-11-01',8.32,8.36,8.25,8.33,28026,'0');
INSERT INTO daily_summary VALUES ('etf','GASL','us','2017-11-01',23.32,24.5,23.15,24.2,219191,'0');
INSERT INTO daily_summary VALUES ('stock','GASS','us','2017-11-01',3.52,3.6,3.5001,3.55,17829,'0');
INSERT INTO daily_summary VALUES ('stock','GASX','us','2017-11-01',28.25,28.25,26.75,27.05,21967,'0');
INSERT INTO daily_summary VALUES ('stock','GATX','us','2017-11-01',60.18,60.91,59.09,59.25,278634,'0');
INSERT INTO daily_summary VALUES ('etf','GAZ','us','2017-11-01',0.24,0.2598,0.24,0.2597,26556,'0');
INSERT INTO daily_summary VALUES ('stock','GBAB','us','2017-11-01',22.38,22.38,22.26,22.31,46692,'0');
INSERT INTO daily_summary VALUES ('stock','GBCI','us','2017-11-01',38.22,38.22,37.5,37.86,485306,'0');
INSERT INTO daily_summary VALUES ('stock','GBDC','us','2017-11-01',18.98,19.08,18.81,18.96,215069,'0');
INSERT INTO daily_summary VALUES ('etf','GBF','us','2017-11-01',114.11,114.205,114.02,114.06,3775,'0');
INSERT INTO daily_summary VALUES ('stock','GBIL','us','2017-11-01',100.03,100.03,100.009,100.023,14361,'0');
INSERT INTO daily_summary VALUES ('stock','GBL','us','2017-11-01',29.01,29.53,28.66,28.8,26394,'0');
INSERT INTO daily_summary VALUES ('stock','GBLI','us','2017-11-01',42.08,42.08,40.44,41.3,5920,'0');
INSERT INTO daily_summary VALUES ('stock','GBLIL','us','2017-11-01',26.0602,26.1001,26.05,26.05,6359,'0');
INSERT INTO daily_summary VALUES ('stock','GBLIZ','us','2017-11-01',25.758,25.78,25.55,25.56,15378,'0');
INSERT INTO daily_summary VALUES ('stock','GBNK','us','2017-11-01',28.55,28.85,28.15,28.35,93050,'0');
INSERT INTO daily_summary VALUES ('stock','GBR','us','2017-11-01',1.33,1.33,1.22,1.24,40792,'0');
INSERT INTO daily_summary VALUES ('stock','GBT','us','2017-11-01',40.15,40.4,37.2,39.05,1816880,'0');
INSERT INTO daily_summary VALUES ('stock','GBX','us','2017-11-01',52.65,52.75,51.15,51.25,535459,'0');
INSERT INTO daily_summary VALUES ('stock','GCAP','us','2017-11-01',7.46,7.479,7.05,7.08,511303,'0');
INSERT INTO daily_summary VALUES ('stock','GCBC','us','2017-11-01',29.85,30.3,29.55,30.25,3267,'0');
INSERT INTO daily_summary VALUES ('etf','GCC','us','2017-11-01',19.12,19.1744,19.08,19.1451,15179,'0');
INSERT INTO daily_summary VALUES ('stock','GCH','us','2017-11-01',12.23,12.31,12.21,12.31,25933,'0');
INSERT INTO daily_summary VALUES ('stock','GCI','us','2017-11-01',8.76,8.82,8.6,8.61,845074,'0');
INSERT INTO daily_summary VALUES ('stock','GCO','us','2017-11-01',24.55,25.25,24.2,24.45,329617,'0');
INSERT INTO daily_summary VALUES ('stock','GCOW','us','2017-11-01',30.49,30.59,30.42,30.55,14964,'0');
INSERT INTO daily_summary VALUES ('stock','GCP','us','2017-11-01',29.45,29.45,28.75,29.25,426402,'0');
INSERT INTO daily_summary VALUES ('stock','GCV','us','2017-11-01',5.85,5.85,5.7,5.7328,17365,'0');
INSERT INTO daily_summary VALUES ('stock','GCVRZ','us','2017-11-01',0.4132,0.4132,0.39,0.39,31870,'0');
INSERT INTO daily_summary VALUES ('stock','GCV_B','us','2017-11-01',26.7466,26.7466,26.7466,26.7466,600,'0');
INSERT INTO daily_summary VALUES ('stock','GD','us','2017-11-01',203.57,204.21,200.54,201.62,1735270,'0');
INSERT INTO daily_summary VALUES ('stock','GDDY','us','2017-11-01',46.95,46.95,46.49,46.83,1983070,'0');
INSERT INTO daily_summary VALUES ('stock','GDEN','us','2017-11-01',26.84,26.8599,26.13,26.59,22017,'0');
INSERT INTO daily_summary VALUES ('stock','GDI','us','2017-11-01',29.16,29.2,28.87,28.94,837516,'0');
INSERT INTO daily_summary VALUES ('stock','GDL','us','2017-11-01',10.11,10.13,10.03,10.03,135417,'0');
INSERT INTO daily_summary VALUES ('stock','GDL_B','us','2017-11-01',50.48,50.48,50.4044,50.45,1482,'0');
INSERT INTO daily_summary VALUES ('stock','GDO','us','2017-11-01',18.18,18.25,18.131,18.25,44920,'0');
INSERT INTO daily_summary VALUES ('stock','GDOT','us','2017-11-01',56.94,56.945,55.56,56.22,315852,'0');
INSERT INTO daily_summary VALUES ('stock','GDP','us','2017-11-01',9.87,10.04,9.87,10.04,1056,'0');
INSERT INTO daily_summary VALUES ('stock','GDS','us','2017-11-01',15.85,16.97,15.63,16.47,909411,'0');
INSERT INTO daily_summary VALUES ('stock','GDV','us','2017-11-01',22.63,22.69,22.54,22.61,101129,'0');
INSERT INTO daily_summary VALUES ('stock','GDVD','us','2017-11-01',26.9132,26.9132,26.9132,26.9132,0,'0');
INSERT INTO daily_summary VALUES ('stock','GDV_D','us','2017-11-01',26.6,26.6,26.6,26.6,480,'0');
INSERT INTO daily_summary VALUES ('stock','GDV_G','us','2017-11-01',25.07,25.14,25.07,25.09,10888,'0');
INSERT INTO daily_summary VALUES ('etf','GDX','us','2017-11-01',22.66,22.83,22.42,22.48,32438500,'0');
INSERT INTO daily_summary VALUES ('etf','GDXJ','us','2017-11-01',32.18,32.67,32,32.12,8566500,'0');
INSERT INTO daily_summary VALUES ('etf','GDXS','us','2017-11-01',14.92,15.35,14.9,15.35,8115,'0');
INSERT INTO daily_summary VALUES ('etf','GDXX','us','2017-11-01',39.65,40.24,39.56,40,3205,'0');
INSERT INTO daily_summary VALUES ('stock','GE','us','2017-11-01',20.07,20.52,19.96,20.02,91707600,'0');
INSERT INTO daily_summary VALUES ('stock','GEB','us','2017-11-01',25.1,25.1,24.96,25.04,75249,'0');
INSERT INTO daily_summary VALUES ('stock','GEC','us','2017-11-01',3.6,3.7,3.6,3.7,21672,'0');
INSERT INTO daily_summary VALUES ('stock','GECC','us','2017-11-01',10.44,10.44,10.07,10.24,11122,'0');
INSERT INTO daily_summary VALUES ('stock','GECCL','us','2017-11-01',25.5,25.5,25.5,25.5,100,'0');
INSERT INTO daily_summary VALUES ('stock','GEF','us','2017-11-01',55.55,56.295,55.49,55.81,114708,'0');
INSERT INTO daily_summary VALUES ('stock','GEH','us','2017-11-01',25.07,25.1,24.97,25.02,43492,'0');
INSERT INTO daily_summary VALUES ('stock','GEK','us','2017-11-01',25.29,25.33,25.29,25.31,24567,'0');
INSERT INTO daily_summary VALUES ('stock','GEL','us','2017-11-01',23.48,23.9125,23.4122,23.8,367946,'0');
INSERT INTO daily_summary VALUES ('stock','GEM','us','2017-11-01',34.91,35.07,34.8087,34.83,44326,'0');
INSERT INTO daily_summary VALUES ('stock','GEMP','us','2017-11-01',9.76,10.41,9.36,9.9,41544,'0');
INSERT INTO daily_summary VALUES ('stock','GEN','us','2017-11-01',0.99,0.9901,0.9235,0.95,302175,'0');
INSERT INTO daily_summary VALUES ('stock','GENC','us','2017-11-01',17.75,17.9,17.45,17.65,17334,'0');
INSERT INTO daily_summary VALUES ('stock','GENE','us','2017-11-01',0.8199,0.825,0.7801,0.8099,96399,'0');
INSERT INTO daily_summary VALUES ('stock','GENY','us','2017-11-01',32.33,32.33,32.1076,32.12,1027,'0');
INSERT INTO daily_summary VALUES ('stock','GEO','us','2017-11-01',26.12,26.32,25.42,25.89,1126960,'0');
INSERT INTO daily_summary VALUES ('stock','GEOS','us','2017-11-01',15.22,15.74,14.76,14.86,28509,'0');
INSERT INTO daily_summary VALUES ('stock','GER','us','2017-11-01',6.07,6.19,6.06,6.16,411460,'0');
INSERT INTO daily_summary VALUES ('stock','GERN','us','2017-11-01',2.3,2.33,2.17,2.17,1745290,'0');
INSERT INTO daily_summary VALUES ('stock','GES','us','2017-11-01',16.32,16.51,16.095,16.42,912401,'0');
INSERT INTO daily_summary VALUES ('stock','GEVO','us','2017-11-01',0.64,0.64,0.603,0.61,672819,'0');
INSERT INTO daily_summary VALUES ('etf','GEX','us','2017-11-01',62.88,62.96,62.5,62.91,2796,'0');
INSERT INTO daily_summary VALUES ('stock','GF','us','2017-11-01',19.1,19.19,19.0833,19.15,13535,'0');
INSERT INTO daily_summary VALUES ('stock','GFA','us','2017-11-01',9.06,9.17,9.02,9.02,51915,'0');
INSERT INTO daily_summary VALUES ('stock','GFED','us','2017-11-01',21.3001,21.3001,21.3001,21.3001,236,'0');
INSERT INTO daily_summary VALUES ('stock','GFF','us','2017-11-01',22.8,22.8,21.55,21.8,47279,'0');
INSERT INTO daily_summary VALUES ('stock','GFI','us','2017-11-01',4,4.04,3.94,3.99,3756170,'0');
INSERT INTO daily_summary VALUES ('stock','GFN','us','2017-11-01',5.3532,5.4,5.35,5.35,4221,'0');
INSERT INTO daily_summary VALUES ('stock','GFNCP','us','2017-11-01',99.24,99.25,99.085,99.2303,737,'0');
INSERT INTO daily_summary VALUES ('stock','GFNSL','us','2017-11-01',24.5468,24.589,24.49,24.51,3487,'0');
INSERT INTO daily_summary VALUES ('stock','GFY','us','2017-11-01',17.3,17.31,17.13,17.3,4763,'0');
INSERT INTO daily_summary VALUES ('stock','GG','us','2017-11-01',13.25,13.4,13.1,13.17,7590110,'0');
INSERT INTO daily_summary VALUES ('stock','GGAL','us','2017-11-01',54.65,55.43,54,54.1,553746,'0');
INSERT INTO daily_summary VALUES ('stock','GGB','us','2017-11-01',3.31,3.355,3.21,3.22,16382800,'0');
INSERT INTO daily_summary VALUES ('stock','GGG','us','2017-11-01',132.74,132.99,130.46,130.49,206588,'0');
INSERT INTO daily_summary VALUES ('stock','GGM','us','2017-11-01',22.6,22.7,22.5,22.51,23859,'0');
INSERT INTO daily_summary VALUES ('stock','GGN','us','2017-11-01',5.45,5.49,5.4453,5.49,345964,'0');
INSERT INTO daily_summary VALUES ('stock','GGN_B','us','2017-11-01',24.74,24.8599,24.7155,24.77,21848,'0');
INSERT INTO daily_summary VALUES ('stock','GGO_A','us','2017-11-01',43.5,43.5,43.5,43.5,178,'0');
INSERT INTO daily_summary VALUES ('stock','GGP','us','2017-11-01',19.5,19.7,19.4,19.44,6865640,'0');
INSERT INTO daily_summary VALUES ('stock','GGP_A','us','2017-11-01',25.26,25.38,25.22,25.38,38875,'0');
INSERT INTO daily_summary VALUES ('stock','GGT','us','2017-11-01',8.96,9.03,8.96,8.98,38493,'0');
INSERT INTO daily_summary VALUES ('stock','GGT_B','us','2017-11-01',26.48,26.608,26.48,26.55,900,'0');
INSERT INTO daily_summary VALUES ('stock','GGT_E','us','2017-11-01',25,25.3,24.9562,25.05,44659,'0');
INSERT INTO daily_summary VALUES ('stock','GGZ','us','2017-11-01',12.6,12.64,12.4809,12.54,23885,'0');
INSERT INTO daily_summary VALUES ('stock','GGZR','us','2017-11-01',0.35,0.38,0.33,0.3375,104093,'0');
INSERT INTO daily_summary VALUES ('stock','GHC','us','2017-11-01',559.8,565,546.3,554.85,13692,'0');
INSERT INTO daily_summary VALUES ('stock','GHDX','us','2017-11-01',33.15,33.28,32.1,32.43,113714,'0');
INSERT INTO daily_summary VALUES ('etf','GHII','us','2017-11-01',28.55,28.561,28.4501,28.5,6305,'0');
INSERT INTO daily_summary VALUES ('stock','GHL','us','2017-11-01',18.3,18.3738,17.9,17.95,766628,'0');
INSERT INTO daily_summary VALUES ('stock','GHM','us','2017-11-01',19.298,19.347,18.337,18.571,35409,'0');
INSERT INTO daily_summary VALUES ('stock','GHS','us','2017-11-01',32.49,32.519,32.4874,32.519,2059,'0');
INSERT INTO daily_summary VALUES ('stock','GHY','us','2017-11-01',14.79,14.79,14.705,14.72,204804,'0');
INSERT INTO daily_summary VALUES ('stock','GHYB','us','2017-11-01',50.06,50.06,50.022,50.05,2048,'0');
INSERT INTO daily_summary VALUES ('stock','GHYG','us','2017-11-01',50.9836,51.1,50.88,50.912,22771,'0');
INSERT INTO daily_summary VALUES ('stock','GIB','us','2017-11-01',53.17,53.37,52.51,52.6,222298,'0');
INSERT INTO daily_summary VALUES ('stock','GIFI','us','2017-11-01',13.14,13.379,13.09,13.189,92794,'0');
INSERT INTO daily_summary VALUES ('stock','GIGB','us','2017-11-01',50,50.07,50,50.03,3501,'0');
INSERT INTO daily_summary VALUES ('stock','GIGM','us','2017-11-01',3.12,3.18,3.066,3.175,24881,'0');
INSERT INTO daily_summary VALUES ('etf','GII','us','2017-11-01',52.59,52.62,52.3599,52.46,11675,'0');
INSERT INTO daily_summary VALUES ('stock','GIII','us','2017-11-01',25.35,25.56,24.66,25.06,531275,'0');
INSERT INTO daily_summary VALUES ('stock','GIL','us','2017-11-01',30.55,30.77,29.18,29.3,1474020,'0');
INSERT INTO daily_summary VALUES ('stock','GILD','us','2017-11-01',74.69,75.85,74.22,74.835,8776360,'0');
INSERT INTO daily_summary VALUES ('stock','GILT','us','2017-11-01',7.35,7.44,7.25,7.3,83737,'0');
INSERT INTO daily_summary VALUES ('stock','GIM','us','2017-11-01',6.64,6.66,6.6201,6.65,172500,'0');
INSERT INTO daily_summary VALUES ('stock','GIMO','us','2017-11-01',38.55,38.95,38.55,38.95,1323460,'0');
INSERT INTO daily_summary VALUES ('stock','GIS','us','2017-11-01',51.98,52.34,51.57,52.03,2647680,'0');
INSERT INTO daily_summary VALUES ('stock','GJH','us','2017-11-01',10.1985,10.1985,10.1985,10.1985,1000,'0');
INSERT INTO daily_summary VALUES ('stock','GJR','us','2017-11-01',21.21,21.21,21.21,21.21,100,'0');
INSERT INTO daily_summary VALUES ('stock','GJS','us','2017-11-01',19.809,19.809,19.809,19.809,501,'0');
INSERT INTO daily_summary VALUES ('stock','GJV','us','2017-11-01',26.33,26.33,26.05,26.05,1260,'0');
INSERT INTO daily_summary VALUES ('stock','GKOS','us','2017-11-01',35.4,35.4,34.64,35.04,435482,'0');
INSERT INTO daily_summary VALUES ('stock','GLAD','us','2017-11-01',9.77,9.77,9.74,9.75,61098,'0');
INSERT INTO daily_summary VALUES ('stock','GLADN','us','2017-11-01',25.92,25.92,25.75,25.75,2760,'0');
INSERT INTO daily_summary VALUES ('stock','GLBL','us','2017-11-01',4.875,4.8875,4.8,4.85,312733,'0');
INSERT INTO daily_summary VALUES ('stock','GLBR','us','2017-11-01',1.25,1.3,1.25,1.25,14392,'0');
INSERT INTO daily_summary VALUES ('stock','GLBS','us','2017-11-01',0.93,0.94,0.918,0.92,132935,'0');
INSERT INTO daily_summary VALUES ('stock','GLBZ','us','2017-11-01',10.95,11.299,10.95,11.299,2662,'0');
INSERT INTO daily_summary VALUES ('etf','GLD','us','2017-11-01',120.98,121.55,120.835,121.11,4647590,'0');
INSERT INTO daily_summary VALUES ('stock','GLDD','us','2017-11-01',5.1,5.15,5.05,5.1,175587,'0');
INSERT INTO daily_summary VALUES ('etf','GLDI','us','2017-11-01',8.91,8.97,8.91,8.95,20373,'0');
INSERT INTO daily_summary VALUES ('stock','GLDW','us','2017-11-01',121.18,121.18,121.09,121.09,419,'0');
INSERT INTO daily_summary VALUES ('etf','GLL','us','2017-11-01',72.22,72.4435,71.74,71.9901,67532,'0');
INSERT INTO daily_summary VALUES ('stock','GLMD','us','2017-11-01',8.11,8.18,7.99,8,50343,'0');
INSERT INTO daily_summary VALUES ('stock','GLNG','us','2017-11-01',21.5,22.02,21.06,21.71,1271860,'0');
INSERT INTO daily_summary VALUES ('stock','GLO','us','2017-11-01',11.43,11.5,11.43,11.5,153943,'0');
INSERT INTO daily_summary VALUES ('stock','GLOB','us','2017-11-01',37.98,38.54,36.58,36.91,212699,'0');
INSERT INTO daily_summary VALUES ('stock','GLOG','us','2017-11-01',17.371,17.371,16.825,17.123,437356,'0');
INSERT INTO daily_summary VALUES ('stock','GLOG_A','us','2017-11-01',26.2059,26.25,26.2,26.249,2191,'0');
INSERT INTO daily_summary VALUES ('stock','GLOP','us','2017-11-01',23.474,23.767,22.838,23.229,170351,'0');
INSERT INTO daily_summary VALUES ('stock','GLOP_A','us','2017-11-01',26.4859,26.4859,26.2,26.29,11093,'0');
INSERT INTO daily_summary VALUES ('stock','GLOW','us','2017-11-01',0.268,0.2745,0.2619,0.2665,55816,'0');
INSERT INTO daily_summary VALUES ('stock','GLP','us','2017-11-01',17.528,17.723,17.431,17.626,21849,'0');
INSERT INTO daily_summary VALUES ('stock','GLPG','us','2017-11-01',99.47,101.18,99.4,101.14,132253,'0');
INSERT INTO daily_summary VALUES ('stock','GLPI','us','2017-11-01',36.61,36.84,36.29,36.32,821968,'0');
INSERT INTO daily_summary VALUES ('stock','GLQ','us','2017-11-01',13.7,13.79,13.6983,13.74,104285,'0');
INSERT INTO daily_summary VALUES ('stock','GLRE','us','2017-11-01',22.1,22.35,21.8,21.95,193407,'0');
INSERT INTO daily_summary VALUES ('stock','GLT','us','2017-11-01',21.18,21.99,20.7,20.76,187106,'0');
INSERT INTO daily_summary VALUES ('etf','GLTR','us','2017-11-01',63.8,64.34,63.8,64.1624,27271,'0');
INSERT INTO daily_summary VALUES ('stock','GLU','us','2017-11-01',20.07,20.07,19.87,19.9,14380,'0');
INSERT INTO daily_summary VALUES ('stock','GLUU','us','2017-11-01',4.07,4.07,3.81,3.89,4417250,'0');
INSERT INTO daily_summary VALUES ('stock','GLV','us','2017-11-01',14.12,14.22,14.05,14.07,53013,'0');
INSERT INTO daily_summary VALUES ('stock','GLW','us','2017-11-01',31.55,31.75,31.37,31.63,5163000,'0');
INSERT INTO daily_summary VALUES ('stock','GLYC','us','2017-11-01',11.69,11.84,10.89,11.2,836569,'0');
INSERT INTO daily_summary VALUES ('stock','GM','us','2017-11-01',42.72,43.37,41.9,43.13,14701000,'0');
INSERT INTO daily_summary VALUES ('stock','GME','us','2017-11-01',18.67,18.95,18.53,18.6,2488720,'0');
INSERT INTO daily_summary VALUES ('stock','GMED','us','2017-11-01',32.03,32.04,31.06,31.25,671408,'0');
INSERT INTO daily_summary VALUES ('etf','GMF','us','2017-11-01',103.88,104.205,103.55,103.62,19185,'0');
INSERT INTO daily_summary VALUES ('stock','GMLP','us','2017-11-01',21.85,22.03,21.31,21.51,228423,'0');
INSERT INTO daily_summary VALUES ('stock','GMLPP','us','2017-11-01',25.15,25.24,25.07,25.1298,281853,'0');
INSERT INTO daily_summary VALUES ('stock','GMO','us','2017-11-01',0.3168,0.3185,0.302,0.3034,81149,'0');
INSERT INTO daily_summary VALUES ('etf','GMOM','us','2017-11-01',26.91,26.92,26.76,26.76,11927,'0');
INSERT INTO daily_summary VALUES ('stock','GMRE','us','2017-11-01',8.55,8.63,8.39,8.55,87919,'0');
INSERT INTO daily_summary VALUES ('stock','GMRE_A','us','2017-11-01',25.05,25.05,25.03,25.04,12663,'0');
INSERT INTO daily_summary VALUES ('stock','GMS','us','2017-11-01',34.36,34.36,33.46,33.73,423520,'0');
INSERT INTO daily_summary VALUES ('stock','GMTA','us','2017-11-01',25.9,26.17,25.9,25.95,5773,'0');
INSERT INTO daily_summary VALUES ('stock','GMZ','us','2017-11-01',8.59,8.68,8.55,8.62,307729,'0');
INSERT INTO daily_summary VALUES ('stock','GNBC','us','2017-11-01',22.3,22.55,21.7,21.85,198336,'0');
INSERT INTO daily_summary VALUES ('stock','GNC','us','2017-11-01',6.88,7,6.71,6.78,1818160,'0');
INSERT INTO daily_summary VALUES ('stock','GNCA','us','2017-11-01',1.17,1.1742,1.08,1.1,641818,'0');
INSERT INTO daily_summary VALUES ('stock','GNCMA','us','2017-11-01',41.39,41.39,40.42,41.12,129213,'0');
INSERT INTO daily_summary VALUES ('stock','GNE','us','2017-11-01',5.729,5.729,5.2203,5.2549,77735,'0');
INSERT INTO daily_summary VALUES ('stock','GNK','us','2017-11-01',11.98,12.12,11.585,12.02,81720,'0');
INSERT INTO daily_summary VALUES ('stock','GNL','us','2017-11-01',21.572,21.661,21.363,21.502,188125,'0');
INSERT INTO daily_summary VALUES ('stock','GNL_A','us','2017-11-01',25.09,25.2,25.01,25.01,6522,'0');
INSERT INTO daily_summary VALUES ('etf','GNMA','us','2017-11-01',49.71,49.765,49.7058,49.72,8658,'0');
INSERT INTO daily_summary VALUES ('stock','GNMK','us','2017-11-01',7.5,7.54,7.29,7.3,359935,'0');
INSERT INTO daily_summary VALUES ('stock','GNMX','us','2017-11-01',1.69,1.73,1.61,1.63,214607,'0');
INSERT INTO daily_summary VALUES ('etf','GNR','us','2017-11-01',47.42,47.608,47.27,47.36,67317,'0');
INSERT INTO daily_summary VALUES ('stock','GNRC','us','2017-11-01',53.78,53.78,48.53,50.34,1433810,'0');
INSERT INTO daily_summary VALUES ('stock','GNRT','us','2017-11-01',4.67,4.81,4.62,4.74,203583,'0');
INSERT INTO daily_summary VALUES ('stock','GNRX','us','2017-11-01',23.05,23.4819,23.05,23.4819,601,'0');
INSERT INTO daily_summary VALUES ('stock','GNT','us','2017-11-01',7,7.015,6.95,6.96,69182,'0');
INSERT INTO daily_summary VALUES ('stock','GNTX','us','2017-11-01',19.52,19.63,19.29,19.32,1640070,'0');
INSERT INTO daily_summary VALUES ('stock','GNTY','us','2017-11-01',28.785,28.91,28.21,28.5,7859,'0');
INSERT INTO daily_summary VALUES ('stock','GNT_A','us','2017-11-01',24.5,24.66,24.425,24.65,23309,'0');
INSERT INTO daily_summary VALUES ('stock','GNUS','us','2017-11-01',3.18,3.27,3.09,3.2,37351,'0');
INSERT INTO daily_summary VALUES ('stock','GNW','us','2017-11-01',3.34,3.44,3.33,3.4,1822020,'0');
INSERT INTO daily_summary VALUES ('stock','GOAU','us','2017-11-01',12.3,12.3432,12.2,12.21,17890,'0');
INSERT INTO daily_summary VALUES ('stock','GOEX','us','2017-11-01',21.89,22.0617,21.7,21.7,17746,'0');
INSERT INTO daily_summary VALUES ('stock','GOF','us','2017-11-01',21.32,21.34,21.29,21.3,96007,'0');
INSERT INTO daily_summary VALUES ('stock','GOGL','us','2017-11-01',8.08,8.1625,8.01,8.07,133097,'0');
INSERT INTO daily_summary VALUES ('stock','GOGO','us','2017-11-01',9.99,10.09,9.75,9.93,2065100,'0');
INSERT INTO daily_summary VALUES ('stock','GOL','us','2017-11-01',21.06,21.66,20.98,21.29,154140,'0');
INSERT INTO daily_summary VALUES ('stock','GOLD','us','2017-11-01',98.71,99.35,97.87,98.19,530640,'0');
INSERT INTO daily_summary VALUES ('stock','GOLF','us','2017-11-01',18.47,18.5,18.15,18.21,148131,'0');
INSERT INTO daily_summary VALUES ('stock','GOOD','us','2017-11-01',22.15,22.698,21.7894,22.65,203591,'0');
INSERT INTO daily_summary VALUES ('stock','GOODM','us','2017-11-01',26.087,26.1387,26.0249,26.1387,2561,'0');
INSERT INTO daily_summary VALUES ('stock','GOODO','us','2017-11-01',25.8367,25.868,25.76,25.763,1419,'0');
INSERT INTO daily_summary VALUES ('stock','GOOG','us','2017-11-01',1017.21,1029.67,1016.95,1025.5,1373440,'0');
INSERT INTO daily_summary VALUES ('stock','GOOGL','us','2017-11-01',1036.32,1047.86,1034,1042.6,2163070,'0');
INSERT INTO daily_summary VALUES ('stock','GOOS','us','2017-11-01',21.53,21.5946,21.11,21.42,544607,'0');
INSERT INTO daily_summary VALUES ('stock','GOP','us','2017-11-01',20.19,20.19,20.005,20.005,367,'0');
INSERT INTO daily_summary VALUES ('stock','GORO','us','2017-11-01',3.5984,3.8283,3.5984,3.6284,501502,'0');
INSERT INTO daily_summary VALUES ('stock','GOV','us','2017-11-01',18.25,18.44,18.07,18.35,829101,'0');
INSERT INTO daily_summary VALUES ('stock','GOVNI','us','2017-11-01',25.86,25.91,25.73,25.79,22615,'0');
INSERT INTO daily_summary VALUES ('etf','GOVT','us','2017-11-01',25.12,25.18,25.1199,25.14,641740,'0');
INSERT INTO daily_summary VALUES ('stock','GPAC','us','2017-11-01',9.95,10.01,9.95,10.01,44597,'0');
INSERT INTO daily_summary VALUES ('stock','GPACW','us','2017-11-01',0.78,0.78,0.7778,0.78,8500,'0');
INSERT INTO daily_summary VALUES ('stock','GPC','us','2017-11-01',88.8,88.8,87.37,87.43,618256,'0');
INSERT INTO daily_summary VALUES ('stock','GPI','us','2017-11-01',79,79.71,77.4,79.3,261031,'0');
INSERT INTO daily_summary VALUES ('stock','GPJA','us','2017-11-01',25,25.09,25,25.03,191854,'0');
INSERT INTO daily_summary VALUES ('stock','GPK','us','2017-11-01',15.48,15.85,15.48,15.83,4248630,'0');
INSERT INTO daily_summary VALUES ('stock','GPL','us','2017-11-01',1.21,1.22,1.15,1.18,592461,'0');
INSERT INTO daily_summary VALUES ('stock','GPM','us','2017-11-01',8.76,8.77,8.63,8.67,371540,'0');
INSERT INTO daily_summary VALUES ('stock','GPMT','us','2017-11-01',18.6,18.92,18.52,18.78,379945,'0');
INSERT INTO daily_summary VALUES ('stock','GPMTW','us','2017-11-01',18.76,18.81,18.51,18.81,1100,'0');
INSERT INTO daily_summary VALUES ('stock','GPN','us','2017-11-01',104.39,104.47,102.13,102.55,1055550,'0');
INSERT INTO daily_summary VALUES ('stock','GPOR','us','2017-11-01',13.9,14.09,13.65,13.92,4248920,'0');
INSERT INTO daily_summary VALUES ('stock','GPP','us','2017-11-01',19.739,19.739,19.592,19.69,16858,'0');
INSERT INTO daily_summary VALUES ('stock','GPRE','us','2017-11-01',18.4,18.45,17.75,18.15,710882,'0');
INSERT INTO daily_summary VALUES ('stock','GPRK','us','2017-11-01',9.52,9.58,9.16,9.47,193951,'0');
INSERT INTO daily_summary VALUES ('stock','GPRO','us','2017-11-01',10.47,10.67,10.29,10.67,9845970,'0');
INSERT INTO daily_summary VALUES ('stock','GPS','us','2017-11-01',26.11,26.315,25.72,25.98,3563060,'0');
INSERT INTO daily_summary VALUES ('stock','GPT','us','2017-11-01',30.53,30.57,28.69,29.02,1617320,'0');
INSERT INTO daily_summary VALUES ('stock','GPX','us','2017-11-01',29.3,29.3,28.15,28.5,13915,'0');
INSERT INTO daily_summary VALUES ('etf','GQRE','us','2017-11-01',62.2,62.43,62.2,62.35,6281,'0');
INSERT INTO daily_summary VALUES ('stock','GRA','us','2017-11-01',76.49,77.37,76.225,76.6,752486,'0');
INSERT INTO daily_summary VALUES ('stock','GRAM','us','2017-11-01',4.48,4.52,4.48,4.52,74477,'0');
INSERT INTO daily_summary VALUES ('stock','GRBK','us','2017-11-01',10.85,11,10.75,10.85,32257,'0');
INSERT INTO daily_summary VALUES ('stock','GRC','us','2017-11-01',32,32.2,30.95,30.99,35897,'0');
INSERT INTO daily_summary VALUES ('etf','GREK','us','2017-11-01',9.42,9.44,9.34,9.42,1942270,'0');
INSERT INTO daily_summary VALUES ('etf','GRES','us','2017-11-01',27.36,27.4199,27.241,27.34,10384,'0');
INSERT INTO daily_summary VALUES ('stock','GRF','us','2017-11-01',8.34,8.34,8.2099,8.2099,5062,'0');
INSERT INTO daily_summary VALUES ('stock','GRFS','us','2017-11-01',23.39,23.5023,23.2,23.2,613076,'0');
INSERT INTO daily_summary VALUES ('etf','GRI','us','2017-11-01',44.3,44.3936,44.155,44.3936,1066,'0');
INSERT INTO daily_summary VALUES ('etf','GRID','us','2017-11-01',50.6401,50.6401,50.23,50.36,1321,'0');
INSERT INTO daily_summary VALUES ('stock','GRIF','us','2017-11-01',36.7,36.83,36.1,36.83,5017,'0');
INSERT INTO daily_summary VALUES ('stock','GRMN','us','2017-11-01',57.62,60.44,57.61,59.8,3488830,'0');
INSERT INTO daily_summary VALUES ('stock','GRMY','us','2017-11-01',24.6648,24.6648,24.6648,24.6648,165,'0');
INSERT INTO daily_summary VALUES ('stock','GRNB','us','2017-11-01',26.66,26.689,26.66,26.66,1706,'0');
INSERT INTO daily_summary VALUES ('stock','GROW','us','2017-11-01',2.6079,2.7179,2.548,2.6079,166082,'0');
INSERT INTO daily_summary VALUES ('stock','GRPN','us','2017-11-01',5.02,5.15,4.84,5.12,21798700,'0');
INSERT INTO daily_summary VALUES ('stock','GRR','us','2017-11-01',12.96,12.96,12.89,12.8992,17552,'0');
INSERT INTO daily_summary VALUES ('etf','GRU','us','2017-11-01',3.4397,3.4397,3.4169,3.4169,200,'0');
INSERT INTO daily_summary VALUES ('stock','GRUB','us','2017-11-01',61.1,61.365,60.42,60.83,1279490,'0');
INSERT INTO daily_summary VALUES ('stock','GRVY','us','2017-11-01',54.58,63.86,54.58,60.1,70811,'0');
INSERT INTO daily_summary VALUES ('etf','GRWN','us','2017-11-01',20.83,20.83,20.83,20.83,432,'0');
INSERT INTO daily_summary VALUES ('stock','GRX','us','2017-11-01',9.92,9.92,9.85,9.87,50225,'0');
INSERT INTO daily_summary VALUES ('stock','GRX_A','us','2017-11-01',25.83,26.03,25.83,25.97,410,'0');
INSERT INTO daily_summary VALUES ('stock','GS','us','2017-11-01',243.89,247.4,242.91,244.26,2969270,'0');
INSERT INTO daily_summary VALUES ('stock','GSAT','us','2017-11-01',1.63,1.66,1.58,1.66,4483350,'0');
INSERT INTO daily_summary VALUES ('stock','GSB','us','2017-11-01',4.03,4.1,3.98,4.01,10841,'0');
INSERT INTO daily_summary VALUES ('stock','GSBC','us','2017-11-01',54.2,54.6,53,53.45,19635,'0');
INSERT INTO daily_summary VALUES ('stock','GSBD','us','2017-11-01',22.1,22.2299,21.82,21.83,63492,'0');
INSERT INTO daily_summary VALUES ('stock','GSD','us','2017-11-01',32.35,32.389,32.35,32.3747,556,'0');
INSERT INTO daily_summary VALUES ('stock','GSEU','us','2017-11-01',31.5499,31.56,31.435,31.435,2768,'0');
INSERT INTO daily_summary VALUES ('stock','GSEW','us','2017-11-01',42.07,42.07,41.6698,41.6698,2977,'0');
INSERT INTO daily_summary VALUES ('etf','GSG','us','2017-11-01',15.56,15.64,15.43,15.48,2795980,'0');
INSERT INTO daily_summary VALUES ('stock','GSH','us','2017-11-01',28.77,28.79,28.75,28.75,2260,'0');
INSERT INTO daily_summary VALUES ('stock','GSHT','us','2017-11-01',9.95,9.95,9.9466,9.9466,1000,'0');
INSERT INTO daily_summary VALUES ('stock','GSHTU','us','2017-11-01',10.425,10.425,10.425,10.425,500,'0');
INSERT INTO daily_summary VALUES ('stock','GSIE','us','2017-11-01',29.53,29.53,29.4,29.4,78241,'0');
INSERT INTO daily_summary VALUES ('stock','GSIT','us','2017-11-01',6.96,6.96,6.83,6.88,19407,'0');
INSERT INTO daily_summary VALUES ('stock','GSJY','us','2017-11-01',33.1,33.1,33.0283,33.0495,4415,'0');
INSERT INTO daily_summary VALUES ('stock','GSK','us','2017-11-01',36.049,36.177,35.694,35.704,4641070,'0');
INSERT INTO daily_summary VALUES ('stock','GSL','us','2017-11-01',1.47,1.47,1.45,1.46,76774,'0');
INSERT INTO daily_summary VALUES ('stock','GSLC','us','2017-11-01',51.39,51.44,51.1701,51.22,205258,'0');
INSERT INTO daily_summary VALUES ('stock','GSL_B','us','2017-11-01',22.84,22.85,22.7139,22.81,7337,'0');
INSERT INTO daily_summary VALUES ('stock','GSM','us','2017-11-01',16.2,16.36,15.84,15.94,451327,'0');
INSERT INTO daily_summary VALUES ('etf','GSP','us','2017-11-01',14.69,14.73,14.52,14.55,6303,'0');
INSERT INTO daily_summary VALUES ('stock','GSS','us','2017-11-01',0.785,0.79,0.767,0.7727,1006770,'0');
INSERT INTO daily_summary VALUES ('stock','GSSC','us','2017-11-01',42.17,42.1899,42.17,42.18,1029,'0');
INSERT INTO daily_summary VALUES ('stock','GST','us','2017-11-01',0.8353,0.86,0.76,0.7873,837998,'0');
INSERT INTO daily_summary VALUES ('stock','GST_A','us','2017-11-01',17.27,17.4,16.533,17.4,7706,'0');
INSERT INTO daily_summary VALUES ('stock','GST_B','us','2017-11-01',19.03,19.25,19.02,19.25,1928,'0');
INSERT INTO daily_summary VALUES ('stock','GSUM','us','2017-11-01',8.8,9.25,8.35,8.4,274901,'0');
INSERT INTO daily_summary VALUES ('stock','GSV','us','2017-11-01',1.31,1.34,1.28,1.3,273850,'0');
INSERT INTO daily_summary VALUES ('stock','GSVC','us','2017-11-01',5.59,5.65,5.48,5.5,129348,'0');
INSERT INTO daily_summary VALUES ('etf','GSY','us','2017-11-01',50.31,50.33,50.3,50.33,168458,'0');
INSERT INTO daily_summary VALUES ('stock','GS_A','us','2017-11-01',22.71,22.87,22.71,22.82,20769,'0');
INSERT INTO daily_summary VALUES ('stock','GS_B','us','2017-11-01',25.49,25.61,25.4574,25.6,40982,'0');
INSERT INTO daily_summary VALUES ('stock','GS_C','us','2017-11-01',23.4,23.4432,23.31,23.4,13023,'0');
INSERT INTO daily_summary VALUES ('stock','GS_D','us','2017-11-01',22.89,22.91,22.71,22.89,66704,'0');
INSERT INTO daily_summary VALUES ('stock','GS_J','us','2017-11-01',26.9,26.939,26.85,26.87,54232,'0');
INSERT INTO daily_summary VALUES ('stock','GS_K','us','2017-11-01',28.82,28.82,28.74,28.81,24138,'0');
INSERT INTO daily_summary VALUES ('stock','GS_N','us','2017-11-01',27.33,27.4054,27.33,27.3649,72736,'0');
INSERT INTO daily_summary VALUES ('stock','GT','us','2017-11-01',30.18,30.73,30.14,30.23,3880910,'0');
INSERT INTO daily_summary VALUES ('stock','GTE','us','2017-11-01',2.2,2.25,2.18,2.24,2626220,'0');
INSERT INTO daily_summary VALUES ('stock','GTHX','us','2017-11-01',23.74,23.96,22.34,22.51,507551,'0');
INSERT INTO daily_summary VALUES ('stock','GTIM','us','2017-11-01',2.9,2.9,2.734,2.85,1504,'0');
INSERT INTO daily_summary VALUES ('stock','GTLS','us','2017-11-01',43.85,44.42,43.3,44.27,856798,'0');
INSERT INTO daily_summary VALUES ('stock','GTN','us','2017-11-01',15.55,15.75,15.3,15.7,702773,'0');
INSERT INTO daily_summary VALUES ('stock','GTO','us','2017-11-01',52.683,52.74,52.683,52.7,3098,'0');
INSERT INTO daily_summary VALUES ('stock','GTS','us','2017-11-01',24.16,24.45,23.73,24.05,62857,'0');
INSERT INTO daily_summary VALUES ('stock','GTT','us','2017-11-01',36.6,36.6,36.2,36.3,411357,'0');
INSERT INTO daily_summary VALUES ('stock','GTXI','us','2017-11-01',8.16,8.16,8.0427,8.13,2402,'0');
INSERT INTO daily_summary VALUES ('stock','GTY','us','2017-11-01',28.49,28.59,28.1,28.18,226266,'0');
INSERT INTO daily_summary VALUES ('stock','GTYH','us','2017-11-01',9.99,10.08,9.95,10.02,10811,'0');
INSERT INTO daily_summary VALUES ('stock','GTYHU','us','2017-11-01',10.4499,10.4499,10.447,10.4492,300,'0');
INSERT INTO daily_summary VALUES ('stock','GTYHW','us','2017-11-01',1.3347,1.3347,1.3347,1.3347,100,'0');
INSERT INTO daily_summary VALUES ('stock','GUDB','us','2017-11-01',49.99,49.99,49.98,49.98,12000,'0');
INSERT INTO daily_summary VALUES ('etf','GULF','us','2017-11-01',17.05,17.2736,17.05,17.14,4032,'0');
INSERT INTO daily_summary VALUES ('etf','GUNR','us','2017-11-01',32.02,32.13,31.9,31.95,261041,'0');
INSERT INTO daily_summary VALUES ('stock','GURE','us','2017-11-01',1.67,1.7,1.52,1.589,402885,'0');
INSERT INTO daily_summary VALUES ('etf','GURU','us','2017-11-01',28.7,28.82,28.62,28.6623,12092,'0');
INSERT INTO daily_summary VALUES ('etf','GUSH','us','2017-11-01',25.08,26.91,25,26.55,1737970,'0');
INSERT INTO daily_summary VALUES ('stock','GUT','us','2017-11-01',7.12,7.17,7.11,7.11,60054,'0');
INSERT INTO daily_summary VALUES ('stock','GUT_A','us','2017-11-01',25.7,25.8249,25.7,25.8249,400,'0');
INSERT INTO daily_summary VALUES ('stock','GUT_C','us','2017-11-01',25.43,25.5381,25.43,25.46,1304,'0');
INSERT INTO daily_summary VALUES ('stock','GV','us','2017-11-01',5.75,5.8,5.7,5.75,76634,'0');
INSERT INTO daily_summary VALUES ('stock','GVA','us','2017-11-01',64.23,64.47,62.41,62.84,358716,'0');
INSERT INTO daily_summary VALUES ('etf','GVAL','us','2017-11-01',25.21,25.27,25.14,25.1815,20924,'0');
INSERT INTO daily_summary VALUES ('etf','GVI','us','2017-11-01',110.34,110.52,110.33,110.34,60772,'0');
INSERT INTO daily_summary VALUES ('stock','GVIP','us','2017-11-01',52.72,52.84,52.5608,52.5783,11222,'0');
INSERT INTO daily_summary VALUES ('stock','GVP','us','2017-11-01',3.55,3.55,3.4,3.4,5220,'0');
INSERT INTO daily_summary VALUES ('stock','GWB','us','2017-11-01',40.721,41.205,40.065,40.632,178248,'0');
INSERT INTO daily_summary VALUES ('stock','GWGH','us','2017-11-01',9.96,9.97,9.95,9.95,1520,'0');
INSERT INTO daily_summary VALUES ('stock','GWPH','us','2017-11-01',107.89,111.06,107,110.24,421319,'0');
INSERT INTO daily_summary VALUES ('stock','GWR','us','2017-11-01',72.33,73.4,72.2,72.38,515889,'0');
INSERT INTO daily_summary VALUES ('stock','GWRE','us','2017-11-01',80.77,80.77,77.31,77.93,660760,'0');
INSERT INTO daily_summary VALUES ('stock','GWRS','us','2017-11-01',9.51,9.51,9.3622,9.41,20248,'0');
INSERT INTO daily_summary VALUES ('stock','GWW','us','2017-11-01',198.51,199.5,193.08,194.37,933852,'0');
INSERT INTO daily_summary VALUES ('etf','GWX','us','2017-11-01',35.74,35.788,35.62,35.62,35384,'0');
INSERT INTO daily_summary VALUES ('etf','GXC','us','2017-11-01',106.93,107.285,106.487,106.56,61216,'0');
INSERT INTO daily_summary VALUES ('etf','GXF','us','2017-11-01',23.9,23.92,23.848,23.88,6641,'0');
INSERT INTO daily_summary VALUES ('etf','GXG','us','2017-11-01',9.6,9.66,9.52,9.55,79769,'0');
INSERT INTO daily_summary VALUES ('stock','GXP','us','2017-11-01',32.89,32.93,32.49,32.49,2304440,'0');
INSERT INTO daily_summary VALUES ('stock','GYB','us','2017-11-01',22.79,22.79,22.78,22.79,1007,'0');
INSERT INTO daily_summary VALUES ('stock','GYC','us','2017-11-01',23.95,24,23.8929,23.8929,1077,'0');
INSERT INTO daily_summary VALUES ('etf','GYLD','us','2017-11-01',18.22,18.22,18.005,18.14,11596,'0');
INSERT INTO daily_summary VALUES ('stock','GZT','us','2017-11-01',9.48,9.48,9.46,9.46,2236,'0');
INSERT INTO daily_summary VALUES ('stock','H','us','2017-11-01',63.09,63.58,62.75,62.82,589994,'0');
INSERT INTO daily_summary VALUES ('stock','HA','us','2017-11-01',33.75,34.525,33.15,33.55,1612650,'0');
INSERT INTO daily_summary VALUES ('stock','HABT','us','2017-11-01',12.3,12.5,12.15,12.35,826351,'0');
INSERT INTO daily_summary VALUES ('stock','HACK','us','2017-11-01',30.82,30.84,30.44,30.57,299613,'0');
INSERT INTO daily_summary VALUES ('stock','HACV','us','2017-11-01',29.0354,29.0354,29.0354,29.0354,1,'0');
INSERT INTO daily_summary VALUES ('stock','HAE','us','2017-11-01',47.79,47.98,46.58,46.64,282087,'0');
INSERT INTO daily_summary VALUES ('stock','HAFC','us','2017-11-01',30.85,31.15,30.25,30.7,146816,'0');
INSERT INTO daily_summary VALUES ('stock','HAIN','us','2017-11-01',35.96,36.6,35.32,36.48,1823830,'0');
INSERT INTO daily_summary VALUES ('stock','HAIR','us','2017-11-01',7.08,7.2959,6.93,7.11,37826,'0');
INSERT INTO daily_summary VALUES ('stock','HAL','us','2017-11-01',43.15,43.93,43.03,43.05,7257570,'0');
INSERT INTO daily_summary VALUES ('stock','HALL','us','2017-11-01',11.66,11.66,11.24,11.3,9127,'0');
INSERT INTO daily_summary VALUES ('stock','HALO','us','2017-11-01',17.9,17.99,16.81,16.93,1523580,'0');
INSERT INTO daily_summary VALUES ('etf','HAO','us','2017-11-01',30.28,30.31,30.05,30.05,22724,'0');
INSERT INTO daily_summary VALUES ('etf','HAP','us','2017-11-01',36.18,36.18,36.0744,36.08,2195,'0');
INSERT INTO daily_summary VALUES ('stock','HAS','us','2017-11-01',92.9,93.33,90.32,90.47,1854220,'0');
INSERT INTO daily_summary VALUES ('stock','HASI','us','2017-11-01',24.28,24.34,23.995,24.18,237942,'0');
INSERT INTO daily_summary VALUES ('stock','HAWK','us','2017-11-01',34.2,34.25,33.5,34,550026,'0');
INSERT INTO daily_summary VALUES ('stock','HAWX','us','2017-11-01',27.1,27.1005,27.045,27.045,2407,'0');
INSERT INTO daily_summary VALUES ('stock','HAYN','us','2017-11-01',36,36.85,34.92,35.12,59766,'0');
INSERT INTO daily_summary VALUES ('stock','HBAN','us','2017-11-01',13.94,13.995,13.77,13.81,7642750,'0');
INSERT INTO daily_summary VALUES ('stock','HBANN','us','2017-11-01',25.7,25.7,25.7,25.7,388,'0');
INSERT INTO daily_summary VALUES ('stock','HBANO','us','2017-11-01',27.59,27.71,27.35,27.5993,7218,'0');
INSERT INTO daily_summary VALUES ('stock','HBANP','us','2017-11-01',1405.05,1419.49,1405.05,1410,2621,'0');
INSERT INTO daily_summary VALUES ('stock','HBB','us','2017-11-01',38.94,38.94,37.8,38.59,16499,'0');
INSERT INTO daily_summary VALUES ('stock','HBCP','us','2017-11-01',42.916,42.916,41.142,41.68,15121,'0');
INSERT INTO daily_summary VALUES ('stock','HBHC','us','2017-11-01',49.25,49.7,48.35,48.55,545079,'0');
INSERT INTO daily_summary VALUES ('stock','HBHCL','us','2017-11-01',26.12,26.24,25.9879,26.22,5872,'0');
INSERT INTO daily_summary VALUES ('stock','HBI','us','2017-11-01',22.404,22.414,21.511,21.839,10843300,'0');
INSERT INTO daily_summary VALUES ('stock','HBIO','us','2017-11-01',3.4,3.5,3.4,3.45,27681,'0');
INSERT INTO daily_summary VALUES ('stock','HBK','us','2017-11-01',14.545,14.545,14.5163,14.5163,440,'0');
INSERT INTO daily_summary VALUES ('stock','HBM','us','2017-11-01',7.6,8.05,7.55,7.75,1060310,'0');
INSERT INTO daily_summary VALUES ('stock','HBMD','us','2017-11-01',21,21.2,20.6,20.65,14395,'0');
INSERT INTO daily_summary VALUES ('stock','HBNC','us','2017-11-01',27.7,28,27.03,27.24,38760,'0');
INSERT INTO daily_summary VALUES ('stock','HBP','us','2017-11-01',6.86,6.86,6.41,6.46,34170,'0');
INSERT INTO daily_summary VALUES ('stock','HCA','us','2017-11-01',75.73,77.53,75.29,76.83,3895640,'0');
INSERT INTO daily_summary VALUES ('stock','HCAC','us','2017-11-01',9.8,9.8,9.8,9.8,35700,'0');
INSERT INTO daily_summary VALUES ('stock','HCAP','us','2017-11-01',12.95,13.193,12.95,13.02,24467,'0');
INSERT INTO daily_summary VALUES ('stock','HCAPZ','us','2017-11-01',25.5,25.58,25.42,25.42,2502,'0');
INSERT INTO daily_summary VALUES ('stock','HCC','us','2017-11-01',26.191,26.63,25.961,26.6,569889,'0');
INSERT INTO daily_summary VALUES ('stock','HCCI','us','2017-11-01',19.7,19.7,19.2,19.45,131900,'0');
INSERT INTO daily_summary VALUES ('stock','HCHC','us','2017-11-01',5.46,5.5,5.24,5.29,122616,'0');
INSERT INTO daily_summary VALUES ('stock','HCI','us','2017-11-01',37.84,37.9424,36.795,37.09,95668,'0');
INSERT INTO daily_summary VALUES ('stock','HCKT','us','2017-11-01',15.56,15.56,14.92,15.06,84007,'0');
INSERT INTO daily_summary VALUES ('stock','HCLP','us','2017-11-01',10.6,10.95,9.9,10.15,4785700,'0');
INSERT INTO daily_summary VALUES ('stock','HCM','us','2017-11-01',30.5,31.21,30.18,30.37,53296,'0');
INSERT INTO daily_summary VALUES ('stock','HCN','us','2017-11-01',66.145,66.569,65.819,66.164,1208240,'0');
INSERT INTO daily_summary VALUES ('stock','HCN_I','us','2017-11-01',60.96,61.3968,60.96,61.3,4325,'0');
INSERT INTO daily_summary VALUES ('stock','HCOM','us','2017-11-01',30.35,30.35,29.8,29.86,8359,'0');
INSERT INTO daily_summary VALUES ('stock','HCP','us','2017-11-01',25.515,25.781,25.406,25.663,5180830,'0');
INSERT INTO daily_summary VALUES ('stock','HCRF','us','2017-11-01',31.15,31.18,31.15,31.18,11987,'0');
INSERT INTO daily_summary VALUES ('stock','HCSG','us','2017-11-01',53.17,53.17,50.9999,52.38,369600,'0');
INSERT INTO daily_summary VALUES ('stock','HD','us','2017-11-01',166.42,166.74,165.12,165.38,3568020,'0');
INSERT INTO daily_summary VALUES ('stock','HDAW','us','2017-11-01',26.75,26.75,26.75,26.75,200,'0');
INSERT INTO daily_summary VALUES ('stock','HDB','us','2017-11-01',93.01,94.44,92.65,93.3,633539,'0');
INSERT INTO daily_summary VALUES ('stock','HDEE','us','2017-11-01',23.99,23.99,23.99,23.99,0,'0');
INSERT INTO daily_summary VALUES ('stock','HDEF','us','2017-11-01',25.39,25.4,25.39,25.4,2600,'0');
INSERT INTO daily_summary VALUES ('stock','HDEZ','us','2017-11-01',27.4136,27.4136,27.4136,27.4136,1863,'0');
INSERT INTO daily_summary VALUES ('etf','HDG','us','2017-11-01',45.2,45.3,45.0462,45.1891,50625,'0');
INSERT INTO daily_summary VALUES ('etf','HDGE','us','2017-11-01',8.36,8.43,8.36,8.41,1048230,'0');
INSERT INTO daily_summary VALUES ('stock','HDLV','us','2017-11-01',29.99,30.108,29.801,30.05,5783,'0');
INSERT INTO daily_summary VALUES ('stock','HDMV','us','2017-11-01',33.85,33.85,33.506,33.58,54160,'0');
INSERT INTO daily_summary VALUES ('stock','HDNG','us','2017-11-01',16.64,16.84,16.256,16.5,18709,'0');
INSERT INTO daily_summary VALUES ('stock','HDP','us','2017-11-01',16.56,16.7,16.27,16.54,880626,'0');
INSERT INTO daily_summary VALUES ('stock','HDRW','us','2017-11-01',28.86,28.86,28.86,28.86,75,'0');
INSERT INTO daily_summary VALUES ('stock','HDS','us','2017-11-01',35.53,35.78,35.1,35.46,1211550,'0');
INSERT INTO daily_summary VALUES ('stock','HDSN','us','2017-11-01',5.88,5.911,5.74,5.85,575157,'0');
INSERT INTO daily_summary VALUES ('etf','HDV','us','2017-11-01',85.98,86.1757,85.9261,86,257077,'0');
INSERT INTO daily_summary VALUES ('stock','HE','us','2017-11-01',36.55,36.55,35.96,36.13,617082,'0');
INSERT INTO daily_summary VALUES ('stock','HEAR','us','2017-11-01',0.69,0.75,0.68,0.7128,174114,'0');
INSERT INTO daily_summary VALUES ('stock','HEB','us','2017-11-01',0.305,0.31,0.3002,0.3005,160206,'0');
INSERT INTO daily_summary VALUES ('stock','HEBT','us','2017-11-01',2.99,3,2.95,2.97,164428,'0');
INSERT INTO daily_summary VALUES ('etf','HECO','us','2017-11-01',42.59,42.68,42.59,42.68,835,'0');
INSERT INTO daily_summary VALUES ('etf','HEDJ','us','2017-11-01',66.97,66.99,66.68,66.7,1076780,'0');
INSERT INTO daily_summary VALUES ('etf','HEEM','us','2017-11-01',26.79,26.95,26.68,26.68,82563,'0');
INSERT INTO daily_summary VALUES ('stock','HEES','us','2017-11-01',33.14,33.8,32.56,32.95,314223,'0');
INSERT INTO daily_summary VALUES ('etf','HEFA','us','2017-11-01',30.06,30.07,29.941,29.95,681129,'0');
INSERT INTO daily_summary VALUES ('stock','HEFV','us','2017-11-01',27.57,27.57,27.53,27.56,3753,'0');
INSERT INTO daily_summary VALUES ('stock','HEI','us','2017-11-01',90.88,90.88,89.23,89.89,208151,'0');
INSERT INTO daily_summary VALUES ('stock','HELE','us','2017-11-01',93.7,94.9,92.85,93.75,60869,'0');
INSERT INTO daily_summary VALUES ('stock','HEMV','us','2017-11-01',26.33,26.33,26.33,26.33,36,'0');
INSERT INTO daily_summary VALUES ('stock','HEP','us','2017-11-01',33.606,35.176,33.547,34.96,224438,'0');
INSERT INTO daily_summary VALUES ('stock','HEQ','us','2017-11-01',17.77,17.77,17.68,17.68,17024,'0');
INSERT INTO daily_summary VALUES ('stock','HES','us','2017-11-01',44.7,45.99,43.75,44.3,4653100,'0');
INSERT INTO daily_summary VALUES ('stock','HESM','us','2017-11-01',19.771,20.175,19.555,19.909,92559,'0');
INSERT INTO daily_summary VALUES ('stock','HES_A','us','2017-11-01',55.95,57,55.295,55.63,6585,'0');
INSERT INTO daily_summary VALUES ('stock','HEUS','us','2017-11-01',30.27,30.27,29.78,29.78,20400,'0');
INSERT INTO daily_summary VALUES ('stock','HEUV','us','2017-11-01',24.879,24.879,24.879,24.879,19,'0');
INSERT INTO daily_summary VALUES ('stock','HEWC','us','2017-11-01',26.55,26.55,26.55,26.55,3026,'0');
INSERT INTO daily_summary VALUES ('etf','HEWG','us','2017-11-01',29.87,29.89,29.81,29.84,148128,'0');
INSERT INTO daily_summary VALUES ('stock','HEWI','us','2017-11-01',17.77,17.78,17.77,17.77,2520,'0');
INSERT INTO daily_summary VALUES ('etf','HEWJ','us','2017-11-01',33.19,33.2,33.03,33.12,1133430,'0');
INSERT INTO daily_summary VALUES ('stock','HEWL','us','2017-11-01',27.57,27.6,27.53,27.6,600,'0');
INSERT INTO daily_summary VALUES ('stock','HEWP','us','2017-11-01',23.539,23.539,23.3748,23.45,18546,'0');
INSERT INTO daily_summary VALUES ('stock','HEWU','us','2017-11-01',24.1,24.1199,24.1,24.11,2504,'0');
INSERT INTO daily_summary VALUES ('stock','HEWW','us','2017-11-01',20.2699,20.2699,20.2,20.2,900,'0');
INSERT INTO daily_summary VALUES ('etf','HEZU','us','2017-11-01',31.12,31.15,31.04,31.07,314247,'0');
INSERT INTO daily_summary VALUES ('stock','HE_U','us','2017-11-01',27.15,27.15,27.15,27.15,100,'0');
INSERT INTO daily_summary VALUES ('stock','HF','us','2017-11-01',44,44.41,43.1,43.81,182473,'0');
INSERT INTO daily_summary VALUES ('stock','HFBC','us','2017-11-01',14.2188,14.2188,14.2188,14.2188,4128,'0');
INSERT INTO daily_summary VALUES ('stock','HFC','us','2017-11-01',37.74,39.7675,37.625,39.65,5335640,'0');
INSERT INTO daily_summary VALUES ('stock','HFWA','us','2017-11-01',30.575,30.625,29.88,30.277,167883,'0');
INSERT INTO daily_summary VALUES ('stock','HFXE','us','2017-11-01',20.2142,20.2142,20.19,20.2068,6473,'0');
INSERT INTO daily_summary VALUES ('stock','HFXI','us','2017-11-01',21.68,21.68,21.56,21.56,12508,'0');
INSERT INTO daily_summary VALUES ('stock','HFXJ','us','2017-11-01',22.279,22.279,22.279,22.279,304,'0');
INSERT INTO daily_summary VALUES ('stock','HGH','us','2017-11-01',29.98,30.64,29.91,30.41,69352,'0');
INSERT INTO daily_summary VALUES ('etf','HGI','us','2017-11-01',17.3,17.3245,17.3,17.3245,5262,'0');
INSERT INTO daily_summary VALUES ('stock','HGSD','us','2017-11-01',25.618,25.618,25.618,25.618,0,'0');
INSERT INTO daily_summary VALUES ('stock','HGSH','us','2017-11-01',1.25,1.25,1.23,1.2301,611,'0');
INSERT INTO daily_summary VALUES ('stock','HGT','us','2017-11-01',1.65,1.65,1.55,1.6,148822,'0');
INSERT INTO daily_summary VALUES ('stock','HGV','us','2017-11-01',41.13,41.38,40.875,41.03,1039280,'0');
INSERT INTO daily_summary VALUES ('stock','HHC','us','2017-11-01',128,128.45,126.42,126.55,166474,'0');
INSERT INTO daily_summary VALUES ('stock','HHS','us','2017-11-01',0.9905,1.07,0.99,1.04,65562,'0');
INSERT INTO daily_summary VALUES ('stock','HHYX','us','2017-11-01',27.985,27.985,27.985,27.985,143,'0');
INSERT INTO daily_summary VALUES ('stock','HI','us','2017-11-01',39.85,39.85,39.35,39.75,333114,'0');
INSERT INTO daily_summary VALUES ('stock','HIBB','us','2017-11-01',12.95,13.1,12.6,12.85,401862,'0');
INSERT INTO daily_summary VALUES ('stock','HIE','us','2017-11-01',13.19,13.27,13.0799,13.27,73968,'0');
INSERT INTO daily_summary VALUES ('stock','HIFR','us','2017-11-01',22.58,22.84,22.32,22.37,186418,'0');
INSERT INTO daily_summary VALUES ('stock','HIFS','us','2017-11-01',196.78,196.78,194,194,1154,'0');
INSERT INTO daily_summary VALUES ('stock','HIG','us','2017-11-01',55.3,55.78,55.14,55.25,2234300,'0');
INSERT INTO daily_summary VALUES ('stock','HIHO','us','2017-11-01',3.399,3.6,3.399,3.5,9138,'0');
INSERT INTO daily_summary VALUES ('stock','HII','us','2017-11-01',233.18,233.9,232.17,233.66,178130,'0');
INSERT INTO daily_summary VALUES ('stock','HIIQ','us','2017-11-01',21.5,22.05,20.75,21.3,620872,'0');
INSERT INTO daily_summary VALUES ('stock','HIL','us','2017-11-01',5.35,5.35,5.15,5.2,119643,'0');
INSERT INTO daily_summary VALUES ('etf','HILO','us','2017-11-01',15.3187,15.3187,15.3187,15.3187,137,'0');
INSERT INTO daily_summary VALUES ('stock','HIMX','us','2017-11-01',10.29,10.83,10.25,10.8,6695120,'0');
INSERT INTO daily_summary VALUES ('stock','HIO','us','2017-11-01',5.07,5.07,5.0405,5.06,423355,'0');
INSERT INTO daily_summary VALUES ('stock','HIPS','us','2017-11-01',17.49,17.49,17.49,17.49,180,'0');
INSERT INTO daily_summary VALUES ('stock','HIVE','us','2017-11-01',4.01,4.18,3.98,4.07,239279,'0');
INSERT INTO daily_summary VALUES ('stock','HIW','us','2017-11-01',50.626,51.062,50.338,50.576,479065,'0');
INSERT INTO daily_summary VALUES ('stock','HIX','us','2017-11-01',7.16,7.17,7.11,7.13,248910,'0');
INSERT INTO daily_summary VALUES ('stock','HJPX','us','2017-11-01',30.1947,30.1947,30.1947,30.1947,312,'0');
INSERT INTO daily_summary VALUES ('stock','HJV','us','2017-11-01',11.2314,11.2314,10.43,10.65,27853,'0');
INSERT INTO daily_summary VALUES ('stock','HK','us','2017-11-01',6.72,6.98,6.6501,6.82,3482790,'0');
INSERT INTO daily_summary VALUES ('stock','HL','us','2017-11-01',4.74,4.77,4.62,4.63,4701910,'0');
INSERT INTO daily_summary VALUES ('stock','HLF','us','2017-11-01',73,73.16,70.72,70.84,1199620,'0');
INSERT INTO daily_summary VALUES ('stock','HLG','us','2017-11-01',25.91,26.4001,25.7,26.15,90972,'0');
INSERT INTO daily_summary VALUES ('stock','HLI','us','2017-11-01',41.63,41.95,41.42,41.78,393767,'0');
INSERT INTO daily_summary VALUES ('stock','HLIT','us','2017-11-01',3.75,3.85,3.5,3.7,884446,'0');
INSERT INTO daily_summary VALUES ('stock','HLM_','us','2017-11-01',34.79,34.79,34.4542,34.4542,4418,'0');
INSERT INTO daily_summary VALUES ('stock','HLNE','us','2017-11-01',27.76,27.76,26.99,27.54,72192,'0');
INSERT INTO daily_summary VALUES ('stock','HLS','us','2017-11-01',46.44,46.449,45.06,45.34,714882,'0');
INSERT INTO daily_summary VALUES ('stock','HLT','us','2017-11-01',72.93,73.08,72.37,72.44,1871160,'0');
INSERT INTO daily_summary VALUES ('stock','HLTH','us','2017-11-01',1.35,1.4,1.3,1.3,213580,'0');
INSERT INTO daily_summary VALUES ('stock','HLX','us','2017-11-01',6.94,7.12,6.83,6.91,1403500,'0');
INSERT INTO daily_summary VALUES ('stock','HMC','us','2017-11-01',31.71,31.87,31.58,31.87,830473,'0');
INSERT INTO daily_summary VALUES ('stock','HMG','us','2017-11-01',11.104,11.104,11.104,11.104,106,'0');
INSERT INTO daily_summary VALUES ('stock','HMHC','us','2017-11-01',9.95,10,9.65,9.8,789228,'0');
INSERT INTO daily_summary VALUES ('stock','HMLP','us','2017-11-01',18.6,18.7,18.25,18.35,120270,'0');
INSERT INTO daily_summary VALUES ('stock','HMLP_A','us','2017-11-01',25.77,25.8,25.53,25.62,177963,'0');
INSERT INTO daily_summary VALUES ('stock','HMN','us','2017-11-01',43.9,44.4,43.3,43.4,247558,'0');
INSERT INTO daily_summary VALUES ('stock','HMNF','us','2017-11-01',18.75,18.75,18.3,18.3,1447,'0');
INSERT INTO daily_summary VALUES ('stock','HMNY','us','2017-11-01',10.67,11.1,9.5,9.92,5362040,'0');
INSERT INTO daily_summary VALUES ('stock','HMST','us','2017-11-01',29.1,29.375,28.65,28.8,180865,'0');
INSERT INTO daily_summary VALUES ('stock','HMSY','us','2017-11-01',19.33,19.47,18.7225,18.94,681856,'0');
INSERT INTO daily_summary VALUES ('stock','HMTV','us','2017-11-01',12.25,12.25,12.05,12.1,89456,'0');
INSERT INTO daily_summary VALUES ('stock','HMY','us','2017-11-01',1.77,1.82,1.74,1.8,4057680,'0');
INSERT INTO daily_summary VALUES ('stock','HNI','us','2017-11-01',34.54,34.59,33.65,34.02,210540,'0');
INSERT INTO daily_summary VALUES ('stock','HNNA','us','2017-11-01',16.61,16.6408,16.4201,16.4201,3492,'0');
INSERT INTO daily_summary VALUES ('stock','HNP','us','2017-11-01',26.56,26.68,26.31,26.39,43368,'0');
INSERT INTO daily_summary VALUES ('stock','HNRG','us','2017-11-01',5.25,5.38,5.06,5.12,56616,'0');
INSERT INTO daily_summary VALUES ('stock','HNW','us','2017-11-01',16.39,16.39,16.2,16.2,21514,'0');
INSERT INTO daily_summary VALUES ('stock','HOFT','us','2017-11-01',47.4,47.7,46.95,47.35,36895,'0');
INSERT INTO daily_summary VALUES ('stock','HOG','us','2017-11-01',47.51,48.12,47.51,47.8,877812,'0');
INSERT INTO daily_summary VALUES ('stock','HOLI','us','2017-11-01',22.71,22.9,22.52,22.64,263779,'0');
INSERT INTO daily_summary VALUES ('stock','HOLX','us','2017-11-01',37.99,38.52,37.87,38.2,3508930,'0');
INSERT INTO daily_summary VALUES ('stock','HOMB','us','2017-11-01',22.67,22.8,21.95,22.25,1191310,'0');
INSERT INTO daily_summary VALUES ('stock','HOME','us','2017-11-01',20.57,20.91,20.46,20.78,148845,'0');
INSERT INTO daily_summary VALUES ('stock','HOML','us','2017-11-01',51.47,51.6859,51.47,51.6859,697,'0');
INSERT INTO daily_summary VALUES ('stock','HON','us','2017-11-01',144.59,145.19,144.23,144.93,2161610,'0');
INSERT INTO daily_summary VALUES ('stock','HONE','us','2017-11-01',19.64,19.65,19.5,19.58,25022,'0');
INSERT INTO daily_summary VALUES ('stock','HOPE','us','2017-11-01',18.65,18.67,18.11,18.33,336269,'0');
INSERT INTO daily_summary VALUES ('stock','HOS','us','2017-11-01',3.78,3.78,3.58,3.65,317370,'0');
INSERT INTO daily_summary VALUES ('stock','HOTR','us','2017-11-01',2.16,2.4004,2.16,2.26,35612,'0');
INSERT INTO daily_summary VALUES ('stock','HOV','us','2017-11-01',2.44,2.48,2.39,2.44,1433880,'0');
INSERT INTO daily_summary VALUES ('stock','HOVNP','us','2017-11-01',7.2,7.2,7.015,7.015,238,'0');
INSERT INTO daily_summary VALUES ('stock','HP','us','2017-11-01',54.353,54.798,53.197,53.948,1480290,'0');
INSERT INTO daily_summary VALUES ('stock','HPE','us','2017-11-01',13.94,13.94,13.47,13.6,15499700,'0');
INSERT INTO daily_summary VALUES ('stock','HPF','us','2017-11-01',21.471,21.57,21.441,21.57,44071,'0');
INSERT INTO daily_summary VALUES ('stock','HPI','us','2017-11-01',21.72,21.839,21.72,21.819,44555,'0');
INSERT INTO daily_summary VALUES ('stock','HPJ','us','2017-11-01',5.7,5.7,5.05,5.125,531981,'0');
INSERT INTO daily_summary VALUES ('stock','HPP','us','2017-11-01',33.81,33.81,33.45,33.71,1039860,'0');
INSERT INTO daily_summary VALUES ('stock','HPQ','us','2017-11-01',21.6,21.64,21.39,21.47,8813660,'0');
INSERT INTO daily_summary VALUES ('stock','HPS','us','2017-11-01',18.38,18.44,18.301,18.44,56794,'0');
INSERT INTO daily_summary VALUES ('stock','HPT','us','2017-11-01',28.62,28.77,28.54,28.63,315151,'0');
INSERT INTO daily_summary VALUES ('stock','HQCL','us','2017-11-01',8,8,7.73,7.91,7478,'0');
INSERT INTO daily_summary VALUES ('stock','HQH','us','2017-11-01',23.73,23.73,23.37,23.45,108766,'0');
INSERT INTO daily_summary VALUES ('stock','HQL','us','2017-11-01',20.44,20.46,20.11,20.15,73697,'0');
INSERT INTO daily_summary VALUES ('stock','HQY','us','2017-11-01',50.58,50.96,49.26,50.45,320116,'0');
INSERT INTO daily_summary VALUES ('stock','HR','us','2017-11-01',32.38,32.42,32.03,32.37,830840,'0');
INSERT INTO daily_summary VALUES ('stock','HRB','us','2017-11-01',24.81,24.83,24.19,24.49,2503570,'0');
INSERT INTO daily_summary VALUES ('stock','HRC','us','2017-11-01',80.75,81.89,80.62,80.89,419787,'0');
INSERT INTO daily_summary VALUES ('stock','HRG','us','2017-11-01',16.36,16.36,16.15,16.2,877294,'0');
INSERT INTO daily_summary VALUES ('stock','HRI','us','2017-11-01',48.81,49.1199,48.09,48.55,162834,'0');
INSERT INTO daily_summary VALUES ('stock','HRL','us','2017-11-01',30.97,31.46,30.9,31.21,3326210,'0');
INSERT INTO daily_summary VALUES ('stock','HRS','us','2017-11-01',140.86,141.97,136.79,137.15,1037160,'0');
INSERT INTO daily_summary VALUES ('stock','HRTG','us','2017-11-01',16.04,16.125,15.41,15.6,714036,'0');
INSERT INTO daily_summary VALUES ('stock','HRTX','us','2017-11-01',15.55,15.625,15.1,15.55,701365,'0');
INSERT INTO daily_summary VALUES ('stock','HRZN','us','2017-11-01',10.68,11.2,10.68,11,119028,'0');
INSERT INTO daily_summary VALUES ('stock','HSBC','us','2017-11-01',48.77,48.89,48.51,48.68,1601400,'0');
INSERT INTO daily_summary VALUES ('stock','HSBC_A','us','2017-11-01',26.21,26.26,26.11,26.13,42625,'0');
INSERT INTO daily_summary VALUES ('stock','HSC','us','2017-11-01',21.5,21.5,20.65,20.9,385686,'0');
INSERT INTO daily_summary VALUES ('stock','HSCZ','us','2017-11-01',30.8628,30.88,30.855,30.8647,2948,'0');
INSERT INTO daily_summary VALUES ('stock','HSEA','us','2017-11-01',27.22,27.38,27.22,27.26,56064,'0');
INSERT INTO daily_summary VALUES ('stock','HSEB','us','2017-11-01',26.98,27.01,26.95,26.98,72658,'0');
INSERT INTO daily_summary VALUES ('stock','HSGX','us','2017-11-01',2.15,2.24,2.0701,2.1399,51903,'0');
INSERT INTO daily_summary VALUES ('stock','HSIC','us','2017-11-01',79.03,79.05,77.16,77.24,1714820,'0');
INSERT INTO daily_summary VALUES ('stock','HSII','us','2017-11-01',24.67,24.72,23.178,24.72,122262,'0');
INSERT INTO daily_summary VALUES ('stock','HSKA','us','2017-11-01',98.01,98.935,96.14,97.03,92867,'0');
INSERT INTO daily_summary VALUES ('stock','HSNI','us','2017-11-01',37.8,38.15,37.45,37.55,273257,'0');
INSERT INTO daily_summary VALUES ('stock','HSON','us','2017-11-01',1.51,1.51,1.4948,1.5,9901,'0');
INSERT INTO daily_summary VALUES ('etf','HSPX','us','2017-11-01',50.26,50.26,50.2107,50.26,1613,'0');
INSERT INTO daily_summary VALUES ('stock','HST','us','2017-11-01',19.66,19.88,19.58,19.73,7492050,'0');
INSERT INTO daily_summary VALUES ('stock','HSTM','us','2017-11-01',24.5,24.62,23.73,23.89,103064,'0');
INSERT INTO daily_summary VALUES ('stock','HSY','us','2017-11-01',106.57,106.87,105.61,106.73,811130,'0');
INSERT INTO daily_summary VALUES ('stock','HT','us','2017-11-01',17.76,17.95,17.64,17.76,719928,'0');
INSERT INTO daily_summary VALUES ('stock','HTA','us','2017-11-01',30.03,30.03,29.725,29.77,874365,'0');
INSERT INTO daily_summary VALUES ('stock','HTBI','us','2017-11-01',26.5,26.5,25.8,25.9,30087,'0');
INSERT INTO daily_summary VALUES ('stock','HTBK','us','2017-11-01',15.449,15.449,15.097,15.241,173892,'0');
INSERT INTO daily_summary VALUES ('stock','HTBX','us','2017-11-01',0.599,0.6,0.57,0.577,141227,'0');
INSERT INTO daily_summary VALUES ('stock','HTD','us','2017-11-01',25.571,25.75,25.571,25.72,47233,'0');
INSERT INTO daily_summary VALUES ('stock','HTFA','us','2017-11-01',25.88,25.89,25.78,25.87,4000,'0');
INSERT INTO daily_summary VALUES ('stock','HTGC','us','2017-11-01',12.2,12.299,12.186,12.21,322976,'0');
INSERT INTO daily_summary VALUES ('stock','HTGM','us','2017-11-01',2.26,2.3244,2.21,2.26,252166,'0');
INSERT INTO daily_summary VALUES ('stock','HTGX','us','2017-11-01',25.25,25.29,25.25,25.25,4111,'0');
INSERT INTO daily_summary VALUES ('stock','HTH','us','2017-11-01',23.74,23.915,23.05,23.16,288709,'0');
INSERT INTO daily_summary VALUES ('stock','HTHT','us','2017-11-01',134.95,138.249,127.36,127.59,575579,'0');
INSERT INTO daily_summary VALUES ('stock','HTLD','us','2017-11-01',21.29,21.42,20.8,21.04,525791,'0');
INSERT INTO daily_summary VALUES ('stock','HTLF','us','2017-11-01',49.55,50.8027,48.95,49.25,77081,'0');
INSERT INTO daily_summary VALUES ('stock','HTM','us','2017-11-01',3.88,3.89,3.741,3.86,58973,'0');
INSERT INTO daily_summary VALUES ('stock','HTUS','us','2017-11-01',28.32,28.32,28.16,28.25,55986,'0');
INSERT INTO daily_summary VALUES ('stock','HTY','us','2017-11-01',9,9.06,8.8799,8.9,87110,'0');
INSERT INTO daily_summary VALUES ('stock','HTZ','us','2017-11-01',25.1,25.4,24.199,24.4,2182230,'0');
INSERT INTO daily_summary VALUES ('stock','HT_D','us','2017-11-01',25.61,25.889,25.61,25.7514,8715,'0');
INSERT INTO daily_summary VALUES ('stock','HT_E','us','2017-11-01',25.39,25.39,25.1896,25.2773,2880,'0');
INSERT INTO daily_summary VALUES ('stock','HUBB','us','2017-11-01',126.56,126.72,125.61,126.16,426994,'0');
INSERT INTO daily_summary VALUES ('stock','HUBG','us','2017-11-01',43.6,43.775,42.6,43.55,451536,'0');
INSERT INTO daily_summary VALUES ('stock','HUBS','us','2017-11-01',87.4,87.85,84.15,85.95,594814,'0');
INSERT INTO daily_summary VALUES ('stock','HUM','us','2017-11-01',256.25,258.49,254.25,255.48,605100,'0');
INSERT INTO daily_summary VALUES ('stock','HUN','us','2017-11-01',32.28,32.59,31.81,31.83,6735010,'0');
INSERT INTO daily_summary VALUES ('stock','HUNTW','us','2017-11-01',0.62,0.62,0.62,0.62,100,'0');
INSERT INTO daily_summary VALUES ('stock','HURC','us','2017-11-01',45.6892,45.71,44.15,44.45,10738,'0');
INSERT INTO daily_summary VALUES ('stock','HURN','us','2017-11-01',36.95,37.1,35.25,36.3,230360,'0');
INSERT INTO daily_summary VALUES ('stock','HUSA','us','2017-11-01',0.451,0.462,0.432,0.435,511324,'0');
INSERT INTO daily_summary VALUES ('etf','HUSE','us','2017-11-01',40.245,40.25,40.13,40.17,3576,'0');
INSERT INTO daily_summary VALUES ('stock','HUSV','us','2017-11-01',22.75,22.75,22.5574,22.5795,31405,'0');
INSERT INTO daily_summary VALUES ('stock','HVT','us','2017-11-01',24.3,25.35,24.3,25.15,102508,'0');
INSERT INTO daily_summary VALUES ('stock','HWBK','us','2017-11-01',20,20.5,19.9416,20.5,2483,'0');
INSERT INTO daily_summary VALUES ('stock','HWCC','us','2017-11-01',5.4499,5.5,5.325,5.5,17517,'0');
INSERT INTO daily_summary VALUES ('stock','HWKN','us','2017-11-01',38.55,38.95,36.85,36.9,26458,'0');
INSERT INTO daily_summary VALUES ('stock','HXL','us','2017-11-01',61.23,61.88,60.69,61.62,1120380,'0');
INSERT INTO daily_summary VALUES ('stock','HY','us','2017-11-01',79.36,80.155,77.04,77.86,69134,'0');
INSERT INTO daily_summary VALUES ('stock','HYACU','us','2017-11-01',10.01,10.05,10.01,10.05,197379,'0');
INSERT INTO daily_summary VALUES ('stock','HYB','us','2017-11-01',9.8,9.89,9.8,9.85,47662,'0');
INSERT INTO daily_summary VALUES ('etf','HYD','us','2017-11-01',31.08,31.08,30.99,31.07,430272,'0');
INSERT INTO daily_summary VALUES ('stock','HYDB','us','2017-11-01',51.06,51.06,51.06,51.06,357,'0');
INSERT INTO daily_summary VALUES ('stock','HYDD','us','2017-11-01',18.5,18.6101,18.5,18.6101,2009,'0');
INSERT INTO daily_summary VALUES ('etf','HYEM','us','2017-11-01',24.81,24.81,24.74,24.78,65737,'0');
INSERT INTO daily_summary VALUES ('etf','HYG','us','2017-11-01',88.18,88.2,88.01,88.02,9505450,'0');
INSERT INTO daily_summary VALUES ('etf','HYGH','us','2017-11-01',91.43,91.43,91.31,91.31,2370,'0');
INSERT INTO daily_summary VALUES ('stock','HYGS','us','2017-11-01',9.95,9.95,9.55,9.65,42115,'0');
INSERT INTO daily_summary VALUES ('stock','HYH','us','2017-11-01',47.15,50.99,44.56,45.22,2311800,'0');
INSERT INTO daily_summary VALUES ('stock','HYHG','us','2017-11-01',68.36,68.3799,67.85,67.95,16446,'0');
INSERT INTO daily_summary VALUES ('stock','HYI','us','2017-11-01',15.45,15.4999,15.38,15.43,48872,'0');
INSERT INTO daily_summary VALUES ('etf','HYIH','us','2017-11-01',23.26,23.26,23.25,23.25,4900,'0');
INSERT INTO daily_summary VALUES ('stock','HYLB','us','2017-11-01',51.04,51.04,50.9564,50.98,132784,'0');
INSERT INTO daily_summary VALUES ('etf','HYLD','us','2017-11-01',36.18,36.34,36.1,36.2401,30130,'0');
INSERT INTO daily_summary VALUES ('etf','HYLS','us','2017-11-01',48.9,48.93,48.82,48.87,54408,'0');
INSERT INTO daily_summary VALUES ('stock','HYLV','us','2017-11-01',25.39,25.4,25.39,25.4,2572,'0');
INSERT INTO daily_summary VALUES ('etf','HYMB','us','2017-11-01',55.47,55.7046,55.42,55.46,54006,'0');
INSERT INTO daily_summary VALUES ('etf','HYND','us','2017-11-01',20.75,20.75,20.75,20.75,1183,'0');
INSERT INTO daily_summary VALUES ('etf','HYS','us','2017-11-01',101.13,101.22,100.88,101.04,620996,'0');
INSERT INTO daily_summary VALUES ('stock','HYT','us','2017-11-01',11.22,11.22,11.19,11.2,287342,'0');
INSERT INTO daily_summary VALUES ('stock','HYXE','us','2017-11-01',51.75,51.75,51.718,51.718,6469,'0');
INSERT INTO daily_summary VALUES ('stock','HYXU','us','2017-11-01',53.2,53.3,53.13,53.2,18006,'0');
INSERT INTO daily_summary VALUES ('etf','HYZD','us','2017-11-01',24.199,24.28,24.08,24.27,52726,'0');
INSERT INTO daily_summary VALUES ('stock','HZN','us','2017-11-01',16.21,16.26,15.05,15.07,306492,'0');
INSERT INTO daily_summary VALUES ('stock','HZNP','us','2017-11-01',13.65,14.1,13.59,13.83,2368850,'0');
INSERT INTO daily_summary VALUES ('stock','HZO','us','2017-11-01',18.65,19.05,18.2,18.95,648456,'0');
INSERT INTO daily_summary VALUES ('stock','I','us','2017-11-01',4.03,4.28,3.72,4.28,1327560,'0');
INSERT INTO daily_summary VALUES ('stock','IAC','us','2017-11-01',130.12,130.86,126.73,128.39,827123,'0');
INSERT INTO daily_summary VALUES ('stock','IAE','us','2017-11-01',10.84,10.89,10.81,10.88,27166,'0');
INSERT INTO daily_summary VALUES ('stock','IAF','us','2017-11-01',6.3,6.3,6.27,6.28,46396,'0');
INSERT INTO daily_summary VALUES ('stock','IAG','us','2017-11-01',5.54,5.64,5.5,5.53,2993910,'0');
INSERT INTO daily_summary VALUES ('stock','IAGG','us','2017-11-01',52.05,52.08,52.01,52.01,33713,'0');
INSERT INTO daily_summary VALUES ('etf','IAI','us','2017-11-01',58.25,58.33,57.58,57.58,12973,'0');
INSERT INTO daily_summary VALUES ('etf','IAK','us','2017-11-01',65.99,66.19,65.77,65.909,6559,'0');
INSERT INTO daily_summary VALUES ('stock','IAM','us','2017-11-01',9.85,9.85,9.85,9.85,450,'0');
INSERT INTO daily_summary VALUES ('stock','IAMXR','us','2017-11-01',0.28,0.28,0.28,0.28,400,'0');
INSERT INTO daily_summary VALUES ('stock','IAMXW','us','2017-11-01',0.28,0.28,0.28,0.28,200,'0');
INSERT INTO daily_summary VALUES ('stock','IART','us','2017-11-01',46.92,47.02,46.09,46.22,900179,'0');
INSERT INTO daily_summary VALUES ('etf','IAT','us','2017-11-01',48.01,48.26,47.66,47.77,91687,'0');
INSERT INTO daily_summary VALUES ('etf','IAU','us','2017-11-01',12.24,12.3,12.22,12.25,12474200,'0');
INSERT INTO daily_summary VALUES ('stock','IBA','us','2017-11-01',59.4,59.53,58.46,58.8,13396,'0');
INSERT INTO daily_summary VALUES ('etf','IBB','us','2017-11-01',315.3,317,311.51,312.19,1128000,'0');
INSERT INTO daily_summary VALUES ('etf','IBCC','us','2017-11-01',24.658,24.658,24.6,24.605,8757,'0');
INSERT INTO daily_summary VALUES ('etf','IBCD','us','2017-11-01',24.58,24.585,24.58,24.58,732,'0');
INSERT INTO daily_summary VALUES ('etf','IBCE','us','2017-11-01',24.5,24.5,24.4,24.4,7265,'0');
INSERT INTO daily_summary VALUES ('stock','IBCP','us','2017-11-01',22.581,22.731,22.258,22.532,53594,'0');
INSERT INTO daily_summary VALUES ('stock','IBD','us','2017-11-01',25.3,25.34,25.225,25.31,4540,'0');
INSERT INTO daily_summary VALUES ('etf','IBDB','us','2017-11-01',25.54,25.5999,25.54,25.56,6135,'0');
INSERT INTO daily_summary VALUES ('etf','IBDC','us','2017-11-01',26.16,26.2199,26.16,26.2012,6900,'0');
INSERT INTO daily_summary VALUES ('etf','IBDD','us','2017-11-01',26.55,26.695,26.55,26.64,6439,'0');
INSERT INTO daily_summary VALUES ('etf','IBDH','us','2017-11-01',25.15,25.16,25.14,25.15,29470,'0');
INSERT INTO daily_summary VALUES ('etf','IBDJ','us','2017-11-01',24.78,24.805,24.78,24.7902,24555,'0');
INSERT INTO daily_summary VALUES ('etf','IBDK','us','2017-11-01',24.97,24.97,24.9409,24.95,56994,'0');
INSERT INTO daily_summary VALUES ('etf','IBDL','us','2017-11-01',25.47,25.475,25.45,25.46,100049,'0');
INSERT INTO daily_summary VALUES ('etf','IBDM','us','2017-11-01',24.96,25,24.9091,24.97,168723,'0');
INSERT INTO daily_summary VALUES ('etf','IBDN','us','2017-11-01',25.15,25.19,25.1021,25.12,75709,'0');
INSERT INTO daily_summary VALUES ('etf','IBDO','us','2017-11-01',25.3,25.3,25.2,25.22,952326,'0');
INSERT INTO daily_summary VALUES ('etf','IBDP','us','2017-11-01',25.17,25.2,25.1,25.14,31745,'0');
INSERT INTO daily_summary VALUES ('etf','IBDQ','us','2017-11-01',25.12,25.169,25.0649,25.0753,32474,'0');
INSERT INTO daily_summary VALUES ('stock','IBDR','us','2017-11-01',24.61,24.61,24.5543,24.6,9010,'0');
INSERT INTO daily_summary VALUES ('stock','IBDS','us','2017-11-01',24.97,24.97,24.9,24.9453,4823,'0');
INSERT INTO daily_summary VALUES ('stock','IBIO','us','2017-11-01',0.3104,0.3191,0.3,0.3085,185272,'0');
INSERT INTO daily_summary VALUES ('stock','IBKC','us','2017-11-01',74.45,75,73.35,73.65,365390,'0');
INSERT INTO daily_summary VALUES ('stock','IBKCO','us','2017-11-01',28.48,28.48,27.8928,28.48,3978,'0');
INSERT INTO daily_summary VALUES ('stock','IBKCP','us','2017-11-01',27.7,28.1579,27.7,28.02,3497,'0');
INSERT INTO daily_summary VALUES ('stock','IBKR','us','2017-11-01',54.37,54.5,53.56,53.85,583857,'0');
INSERT INTO daily_summary VALUES ('etf','IBLN','us','2017-11-01',31.93,31.93,31.7,31.7,1791,'0');
INSERT INTO daily_summary VALUES ('stock','IBM','us','2017-11-01',152.57,153.37,152.31,152.51,3270650,'0');
INSERT INTO daily_summary VALUES ('etf','IBMG','us','2017-11-01',25.432,25.45,25.41,25.415,20450,'0');
INSERT INTO daily_summary VALUES ('etf','IBMH','us','2017-11-01',25.49,25.49,25.45,25.4607,19992,'0');
INSERT INTO daily_summary VALUES ('etf','IBMI','us','2017-11-01',25.62,25.62,25.58,25.6,36780,'0');
INSERT INTO daily_summary VALUES ('stock','IBMJ','us','2017-11-01',25.78,25.78,25.73,25.73,25454,'0');
INSERT INTO daily_summary VALUES ('stock','IBMK','us','2017-11-01',25.98,25.99,25.96,25.976,27596,'0');
INSERT INTO daily_summary VALUES ('stock','IBML','us','2017-11-01',25.41,25.42,25.41,25.42,8449,'0');
INSERT INTO daily_summary VALUES ('stock','IBN','us','2017-11-01',9.53,9.63,9.46,9.51,18101400,'0');
INSERT INTO daily_summary VALUES ('etf','IBND','us','2017-11-01',34.33,34.44,34.32,34.36,95433,'0');
INSERT INTO daily_summary VALUES ('stock','IBO','us','2017-11-01',24.89,24.96,24.82,24.84,1026100,'0');
INSERT INTO daily_summary VALUES ('stock','IBOC','us','2017-11-01',41.1,41.15,40.225,40.7,214259,'0');
INSERT INTO daily_summary VALUES ('stock','IBP','us','2017-11-01',69.9,70.25,69.05,69.35,164535,'0');
INSERT INTO daily_summary VALUES ('stock','IBTX','us','2017-11-01',63.151,63.501,61.404,61.953,54254,'0');
INSERT INTO daily_summary VALUES ('stock','IBUY','us','2017-11-01',37.7,37.75,37.2813,37.47,22545,'0');
INSERT INTO daily_summary VALUES ('stock','ICAD','us','2017-11-01',4.74,4.74,4.52,4.56,29521,'0');
INSERT INTO daily_summary VALUES ('stock','ICAN','us','2017-11-01',26.27,26.27,26.2699,26.2699,600,'0');
INSERT INTO daily_summary VALUES ('stock','ICB','us','2017-11-01',18.27,18.27,18.251,18.251,581,'0');
INSERT INTO daily_summary VALUES ('stock','ICBK','us','2017-11-01',32.02,32.02,31.6,31.6,8663,'0');
INSERT INTO daily_summary VALUES ('stock','ICCC','us','2017-11-01',7.85,7.98,7.85,7.9799,6036,'0');
INSERT INTO daily_summary VALUES ('stock','ICCH','us','2017-11-01',16.76,16.9599,16.76,16.9,1506,'0');
INSERT INTO daily_summary VALUES ('stock','ICD','us','2017-11-01',3.51,3.51,3.04,3.25,318185,'0');
INSERT INTO daily_summary VALUES ('stock','ICE','us','2017-11-01',66.2,66.62,66.09,66.22,3741850,'0');
INSERT INTO daily_summary VALUES ('etf','ICF','us','2017-11-01',99.7,100.46,99.7,100.22,176169,'0');
INSERT INTO daily_summary VALUES ('stock','ICFI','us','2017-11-01',54.15,54.15,52.1,53.35,51630,'0');
INSERT INTO daily_summary VALUES ('stock','ICHR','us','2017-11-01',31.5,31.7856,26.3,28.58,1697660,'0');
INSERT INTO daily_summary VALUES ('etf','ICI','us','2017-11-01',38.05,38.7,38.05,38.7,380,'0');
INSERT INTO daily_summary VALUES ('stock','ICL','us','2017-11-01',4.18,4.2,4.16,4.18,110337,'0');
INSERT INTO daily_summary VALUES ('etf','ICLN','us','2017-11-01',9.35,9.37,9.3,9.3,112267,'0');
INSERT INTO daily_summary VALUES ('stock','ICLR','us','2017-11-01',118.95,121.27,118.95,120.93,328391,'0');
INSERT INTO daily_summary VALUES ('etf','ICOL','us','2017-11-01',13.65,13.77,13.63,13.67,13704,'0');
INSERT INTO daily_summary VALUES ('stock','ICON','us','2017-11-01',1.7,2.15,1.61,2.01,18022800,'0');
INSERT INTO daily_summary VALUES ('stock','ICOW','us','2017-11-01',28.63,28.63,28.61,28.62,3547,'0');
INSERT INTO daily_summary VALUES ('stock','ICPT','us','2017-11-01',64.5,68,62.62,64.86,2263240,'0');
INSERT INTO daily_summary VALUES ('stock','ICSH','us','2017-11-01',50.13,50.13,50.05,50.1,21707,'0');
INSERT INTO daily_summary VALUES ('stock','ICUI','us','2017-11-01',191.2,191.95,188.6,189.85,76727,'0');
INSERT INTO daily_summary VALUES ('stock','ICVT','us','2017-11-01',55.72,55.79,55.47,55.48,7551,'0');
INSERT INTO daily_summary VALUES ('stock','IDA','us','2017-11-01',91.821,92.109,90.475,90.738,133983,'0');
INSERT INTO daily_summary VALUES ('stock','IDCC','us','2017-11-01',74,74.7501,73.05,74.1,284610,'0');
INSERT INTO daily_summary VALUES ('stock','IDE','us','2017-11-01',16.3,16.3,16.16,16.27,46933,'0');
INSERT INTO daily_summary VALUES ('stock','IDEV','us','2017-11-01',57.62,57.64,57.4,57.4,90258,'0');
INSERT INTO daily_summary VALUES ('stock','IDHD','us','2017-11-01',28.6,28.6,28.6,28.6,234,'0');
INSERT INTO daily_summary VALUES ('etf','IDHQ','us','2017-11-01',23.76,23.855,23.67,23.67,2295,'0');
INSERT INTO daily_summary VALUES ('stock','IDLB','us','2017-11-01',29.54,29.5745,29.45,29.455,19006,'0');
INSERT INTO daily_summary VALUES ('etf','IDLV','us','2017-11-01',33.34,33.439,33.255,33.28,89283,'0');
INSERT INTO daily_summary VALUES ('stock','IDN','us','2017-11-01',2.27,2.27,2.13,2.19,38997,'0');
INSERT INTO daily_summary VALUES ('etf','IDOG','us','2017-11-01',28.6,28.62,28.5,28.5,42969,'0');
INSERT INTO daily_summary VALUES ('stock','IDRA','us','2017-11-01',1.57,1.68,1.56,1.57,3591770,'0');
INSERT INTO daily_summary VALUES ('stock','IDSA','us','2017-11-01',1.7251,1.7251,1.53,1.62,35893,'0');
INSERT INTO daily_summary VALUES ('stock','IDSY','us','2017-11-01',7.42,7.58,7.03,7.58,32473,'0');
INSERT INTO daily_summary VALUES ('stock','IDT','us','2017-11-01',13.26,13.27,13.02,13.14,143329,'0');
INSERT INTO daily_summary VALUES ('stock','IDTI','us','2017-11-01',31.29,31.36,30.48,31.01,2201220,'0');
INSERT INTO daily_summary VALUES ('etf','IDU','us','2017-11-01',138.29,138.29,137.21,137.29,43155,'0');
INSERT INTO daily_summary VALUES ('etf','IDV','us','2017-11-01',33.63,33.66,33.47,33.5,503704,'0');
INSERT INTO daily_summary VALUES ('etf','IDX','us','2017-11-01',24.07,24.0998,23.985,24.01,27239,'0');
INSERT INTO daily_summary VALUES ('stock','IDXG','us','2017-11-01',1.21,1.22,1.17,1.17,519392,'0');
INSERT INTO daily_summary VALUES ('stock','IDXX','us','2017-11-01',167.79,167.79,159.15,159.24,535588,'0');
INSERT INTO daily_summary VALUES ('stock','IEC','us','2017-11-01',4.54,4.54,4.3356,4.41,26147,'0');
INSERT INTO daily_summary VALUES ('etf','IEF','us','2017-11-01',105.92,106.22,105.915,106.06,1321450,'0');
INSERT INTO daily_summary VALUES ('etf','IEFA','us','2017-11-01',65.58,65.6299,65.35,65.35,4615720,'0');
INSERT INTO daily_summary VALUES ('etf','IEI','us','2017-11-01',122.89,123.075,122.86,122.97,328374,'0');
INSERT INTO daily_summary VALUES ('etf','IEMG','us','2017-11-01',56.3,56.46,56.04,56.09,8822730,'0');
INSERT INTO daily_summary VALUES ('etf','IEO','us','2017-11-01',59.53,60.846,59.53,60.72,93761,'0');
INSERT INTO daily_summary VALUES ('stock','IEP','us','2017-11-01',55.93,56.6903,55.3615,55.94,83217,'0');
INSERT INTO daily_summary VALUES ('stock','IESC','us','2017-11-01',19.1,19.1,18.3,18.5,37157,'0');
INSERT INTO daily_summary VALUES ('etf','IEUR','us','2017-11-01',50.1,50.15,49.88,49.88,228722,'0');
INSERT INTO daily_summary VALUES ('etf','IEUS','us','2017-11-01',56.92,57.0512,56.66,56.66,19236,'0');
INSERT INTO daily_summary VALUES ('etf','IEV','us','2017-11-01',47.23,47.29,47.06,47.07,272049,'0');
INSERT INTO daily_summary VALUES ('stock','IEX','us','2017-11-01',129.4,129.4,127.5,128.03,256323,'0');
INSERT INTO daily_summary VALUES ('etf','IEZ','us','2017-11-01',33.45,33.89,33.04,33.21,108723,'0');
INSERT INTO daily_summary VALUES ('stock','IF','us','2017-11-01',7.71,7.7401,7.71,7.72,5256,'0');
INSERT INTO daily_summary VALUES ('etf','IFEU','us','2017-11-01',38.66,38.66,38.56,38.56,3111,'0');
INSERT INTO daily_summary VALUES ('stock','IFF','us','2017-11-01',147.74,148.935,147.51,148.32,266554,'0');
INSERT INTO daily_summary VALUES ('etf','IFGL','us','2017-11-01',29.23,29.33,29.211,29.25,78253,'0');
INSERT INTO daily_summary VALUES ('stock','IFIX','us','2017-11-01',50.2943,50.2943,50.2943,50.2943,180,'0');
INSERT INTO daily_summary VALUES ('stock','IFLY','us','2017-11-01',36.98,37.03,36.7076,36.85,23749,'0');
INSERT INTO daily_summary VALUES ('stock','IFMK','us','2017-11-01',12.9,12.9,12.9,12.9,282,'0');
INSERT INTO daily_summary VALUES ('stock','IFN','us','2017-11-01',28.05,28.2,28,28.16,86510,'0');
INSERT INTO daily_summary VALUES ('stock','IFON','us','2017-11-01',1.54,1.545,1.48,1.53,61666,'0');
INSERT INTO daily_summary VALUES ('etf','IFV','us','2017-11-01',21.92,21.92,21.75,21.79,206096,'0');
INSERT INTO daily_summary VALUES ('stock','IGA','us','2017-11-01',11.55,11.57,11.47,11.5,51353,'0');
INSERT INTO daily_summary VALUES ('stock','IGC','us','2017-11-01',0.45,0.479,0.4338,0.4449,624620,'0');
INSERT INTO daily_summary VALUES ('stock','IGD','us','2017-11-01',8.07,8.07,8,8.03,283364,'0');
INSERT INTO daily_summary VALUES ('etf','IGE','us','2017-11-01',33.3,33.74,33.3,33.55,82077,'0');
INSERT INTO daily_summary VALUES ('stock','IGEB','us','2017-11-01',50.58,50.58,50.58,50.58,12,'0');
INSERT INTO daily_summary VALUES ('etf','IGF','us','2017-11-01',45.59,45.72,45.53,45.54,155726,'0');
INSERT INTO daily_summary VALUES ('stock','IGHG','us','2017-11-01',77.41,77.41,77.0001,77.19,62883,'0');
INSERT INTO daily_summary VALUES ('stock','IGI','us','2017-11-01',21.59,22.05,21.567,21.98,15371,'0');
INSERT INTO daily_summary VALUES ('etf','IGIH','us','2017-11-01',24.39,24.4299,24.39,24.4299,3524,'0');
INSERT INTO daily_summary VALUES ('etf','IGM','us','2017-11-01',167.77,168.02,166.131,166.9,32857,'0');
INSERT INTO daily_summary VALUES ('etf','IGN','us','2017-11-01',46.69,46.85,46.25,46.41,8948,'0');
INSERT INTO daily_summary VALUES ('etf','IGOV','us','2017-11-01',48.65,48.775,48.58,48.6,102239,'0');
INSERT INTO daily_summary VALUES ('stock','IGR','us','2017-11-01',7.85,7.89,7.75,7.77,393752,'0');
INSERT INTO daily_summary VALUES ('stock','IGRO','us','2017-11-01',57.55,57.92,57.18,57.39,3040,'0');
INSERT INTO daily_summary VALUES ('stock','IGT','us','2017-11-01',23.8,24.29,23.5,24.15,2440520,'0');
INSERT INTO daily_summary VALUES ('etf','IGV','us','2017-11-01',157.33,157.33,154.73,155.62,119010,'0');
INSERT INTO daily_summary VALUES ('stock','IGVT','us','2017-11-01',48.9199,48.9199,48.9199,48.9199,146,'0');
INSERT INTO daily_summary VALUES ('stock','IHC','us','2017-11-01',27.15,27.45,27,27.3,2620,'0');
INSERT INTO daily_summary VALUES ('stock','IHD','us','2017-11-01',9.51,9.55,9.48,9.49,55490,'0');
INSERT INTO daily_summary VALUES ('etf','IHDG','us','2017-11-01',31.75,31.83,31.61,31.65,69077,'0');
INSERT INTO daily_summary VALUES ('etf','IHE','us','2017-11-01',151.11,152.46,150.75,151.85,11178,'0');
INSERT INTO daily_summary VALUES ('etf','IHF','us','2017-11-01',149.49,149.58,147.265,147.75,50865,'0');
INSERT INTO daily_summary VALUES ('stock','IHG','us','2017-11-01',56.47,56.51,56.08,56.13,148759,'0');
INSERT INTO daily_summary VALUES ('etf','IHI','us','2017-11-01',172.41,172.609,171.48,171.69,104544,'0');
INSERT INTO daily_summary VALUES ('stock','IHIT','us','2017-11-01',10.15,10.15,10.07,10.09,26162,'0');
INSERT INTO daily_summary VALUES ('stock','IHT','us','2017-11-01',1.8688,1.8688,1.8688,1.8688,118,'0');
INSERT INTO daily_summary VALUES ('etf','IHY','us','2017-11-01',25.66,25.67,25.6303,25.66,55801,'0');
INSERT INTO daily_summary VALUES ('stock','IID','us','2017-11-01',7.74,7.813,7.74,7.78,31133,'0');
INSERT INTO daily_summary VALUES ('stock','IIF','us','2017-11-01',36.44,36.47,36.35,36.35,26424,'0');
INSERT INTO daily_summary VALUES ('stock','III','us','2017-11-01',4.14,4.14,3.99,4.05,11970,'0');
INSERT INTO daily_summary VALUES ('stock','IIIN','us','2017-11-01',25.8,26.13,25.19,25.55,126894,'0');
INSERT INTO daily_summary VALUES ('stock','IIJI','us','2017-11-01',9.93,10.48,9.93,10.48,425,'0');
INSERT INTO daily_summary VALUES ('stock','IIM','us','2017-11-01',14.74,14.74,14.65,14.67,101034,'0');
INSERT INTO daily_summary VALUES ('stock','IIN','us','2017-11-01',11.5,11.5,11.15,11.25,3651,'0');
INSERT INTO daily_summary VALUES ('stock','IIPR','us','2017-11-01',19.94,19.94,19.5,19.54,21802,'0');
INSERT INTO daily_summary VALUES ('stock','IIPR_A','us','2017-11-01',25.96,26,25.53,25.75,18083,'0');
INSERT INTO daily_summary VALUES ('stock','IIVI','us','2017-11-01',45.85,46,44.25,44.3,935384,'0');
INSERT INTO daily_summary VALUES ('etf','IJH','us','2017-11-01',184.06,184.12,181.91,182.53,653328,'0');
INSERT INTO daily_summary VALUES ('etf','IJJ','us','2017-11-01',155.14,155.25,153.41,153.99,62889,'0');
INSERT INTO daily_summary VALUES ('etf','IJK','us','2017-11-01',210.48,210.48,207.82,208.49,118447,'0');
INSERT INTO daily_summary VALUES ('etf','IJR','us','2017-11-01',75.33,75.43,73.88,74.33,2406660,'0');
INSERT INTO daily_summary VALUES ('etf','IJS','us','2017-11-01',150.19,150.19,147.38,148.35,165687,'0');
INSERT INTO daily_summary VALUES ('etf','IJT','us','2017-11-01',167.69,167.8,164.12,165.43,77586,'0');
INSERT INTO daily_summary VALUES ('stock','IKNX','us','2017-11-01',8,8,8,8,104,'0');
INSERT INTO daily_summary VALUES ('etf','ILF','us','2017-11-01',34.02,34.3,33.77,33.79,3053030,'0');
INSERT INTO daily_summary VALUES ('stock','ILG','us','2017-11-01',29.9,30.06,29.54,29.65,719719,'0');
INSERT INTO daily_summary VALUES ('stock','ILMN','us','2017-11-01',206.49,208.27,205.2,207.74,750564,'0');
INSERT INTO daily_summary VALUES ('etf','ILTB','us','2017-11-01',63.55,63.905,63.55,63.76,8815,'0');
INSERT INTO daily_summary VALUES ('stock','IMAX','us','2017-11-01',24.3,24.8,23.55,24.15,1367700,'0');
INSERT INTO daily_summary VALUES ('stock','IMDZ','us','2017-11-01',5.05,5.3,4.45,4.5,2965330,'0');
INSERT INTO daily_summary VALUES ('stock','IMGN','us','2017-11-01',5.82,5.86,4.92,5.07,5525870,'0');
INSERT INTO daily_summary VALUES ('stock','IMH','us','2017-11-01',13.19,13.24,13,13.19,13977,'0');
INSERT INTO daily_summary VALUES ('stock','IMI','us','2017-11-01',1.17,1.17,1.12,1.1699,7757,'0');
INSERT INTO daily_summary VALUES ('stock','IMKTA','us','2017-11-01',23.5,23.75,22.9,23.7,61317,'0');
INSERT INTO daily_summary VALUES ('etf','IMLP','us','2017-11-01',17.93,18.03,17.93,17.98,25550,'0');
INSERT INTO daily_summary VALUES ('stock','IMMR','us','2017-11-01',8.23,8.23,7.87,7.94,167596,'0');
INSERT INTO daily_summary VALUES ('stock','IMMU','us','2017-11-01',10.76,10.8632,9.88,10.26,3008660,'0');
INSERT INTO daily_summary VALUES ('stock','IMMY','us','2017-11-01',1.52,1.526,1.45,1.47,255559,'0');
INSERT INTO daily_summary VALUES ('stock','IMNP','us','2017-11-01',0.8145,0.82,0.781,0.81,217024,'0');
INSERT INTO daily_summary VALUES ('stock','IMO','us','2017-11-01',32.43,32.7523,31.98,32.23,293188,'0');
INSERT INTO daily_summary VALUES ('stock','IMOM','us','2017-11-01',30.155,30.23,29.99,30.17,6132,'0');
INSERT INTO daily_summary VALUES ('stock','IMOS','us','2017-11-01',19.97,20.07,19.7,19.8,78980,'0');
INSERT INTO daily_summary VALUES ('stock','IMPV','us','2017-11-01',43.1,43.35,41.7,42.25,264808,'0');
INSERT INTO daily_summary VALUES ('stock','IMRN','us','2017-11-01',5.09,5.78,5.09,5.67,16450,'0');
INSERT INTO daily_summary VALUES ('stock','IMTB','us','2017-11-01',49.56,49.56,49.322,49.4,4704,'0');
INSERT INTO daily_summary VALUES ('etf','IMTM','us','2017-11-01',30.88,30.93,30.78,30.81,21421,'0');
INSERT INTO daily_summary VALUES ('stock','IMUC','us','2017-11-01',0.372,0.3789,0.345,0.351,1216910,'0');
INSERT INTO daily_summary VALUES ('stock','INAP','us','2017-11-01',4.7,4.74,4.38,4.55,733231,'0');
INSERT INTO daily_summary VALUES ('stock','INB','us','2017-11-01',9.84,9.85,9.75,9.78,87840,'0');
INSERT INTO daily_summary VALUES ('stock','INBK','us','2017-11-01',38,38,37,37.4,36298,'0');
INSERT INTO daily_summary VALUES ('etf','INCO','us','2017-11-01',45.85,46.209,45.8,45.94,12664,'0');
INSERT INTO daily_summary VALUES ('stock','INCR','us','2017-11-01',57.5,57.5,56.2,57.2,688980,'0');
INSERT INTO daily_summary VALUES ('stock','INCY','us','2017-11-01',113.42,114.79,106.36,107.96,2871730,'0');
INSERT INTO daily_summary VALUES ('etf','INDA','us','2017-11-01',35.57,35.72,35.54,35.61,2592500,'0');
INSERT INTO daily_summary VALUES ('stock','INDB','us','2017-11-01',72.7,72.95,70.9,71.9,56971,'0');
INSERT INTO daily_summary VALUES ('stock','INDF','us','2017-11-01',31.13,31.13,31.13,31.13,645,'0');
INSERT INTO daily_summary VALUES ('etf','INDL','us','2017-11-01',98.03,99.1592,97.9,98.23,65722,'0');
INSERT INTO daily_summary VALUES ('stock','INDU','us','2017-11-01',9.73,9.73,9.73,9.73,2000,'0');
INSERT INTO daily_summary VALUES ('stock','INDUU','us','2017-11-01',10.1,10.1,10.097,10.0992,21200,'0');
INSERT INTO daily_summary VALUES ('stock','INDUW','us','2017-11-01',0.4,0.4,0.4,0.4,313100,'0');
INSERT INTO daily_summary VALUES ('etf','INDY','us','2017-11-01',36.36,36.6897,36.36,36.52,170675,'0');
INSERT INTO daily_summary VALUES ('stock','INF','us','2017-11-01',12.84,12.89,12.79,12.8,113237,'0');
INSERT INTO daily_summary VALUES ('stock','INFI','us','2017-11-01',2.34,2.36,2.13,2.2,742104,'0');
INSERT INTO daily_summary VALUES ('stock','INFN','us','2017-11-01',8.44,8.58,8.29,8.4,895136,'0');
INSERT INTO daily_summary VALUES ('stock','INFO','us','2017-11-01',42.81,43.13,42.52,42.7,1966400,'0');
INSERT INTO daily_summary VALUES ('stock','INFR','us','2017-11-01',29.72,29.77,29.72,29.75,1461,'0');
INSERT INTO daily_summary VALUES ('stock','INFU','us','2017-11-01',2,2.05,2,2,8523,'0');
INSERT INTO daily_summary VALUES ('stock','INFY','us','2017-11-01',14.895,14.94,14.59,14.65,4856600,'0');
INSERT INTO daily_summary VALUES ('stock','ING','us','2017-11-01',18.59,18.64,18.51,18.52,2170930,'0');
INSERT INTO daily_summary VALUES ('stock','INGN','us','2017-11-01',99.08,99.745,97.34,98.01,129516,'0');
INSERT INTO daily_summary VALUES ('stock','INGR','us','2017-11-01',128.42,131.77,125.69,127.79,619031,'0');
INSERT INTO daily_summary VALUES ('etf','INKM','us','2017-11-01',33.24,33.2416,33.19,33.2,8049,'0');
INSERT INTO daily_summary VALUES ('stock','INN','us','2017-11-01',15.78,16,15.71,15.78,496093,'0');
INSERT INTO daily_summary VALUES ('stock','INN_B','us','2017-11-01',25.7485,25.7485,25.72,25.72,1368,'0');
INSERT INTO daily_summary VALUES ('stock','INN_C','us','2017-11-01',25.6406,25.65,25.62,25.62,1052,'0');
INSERT INTO daily_summary VALUES ('stock','INN_D','us','2017-11-01',25.6,25.6,25.6,25.6,1021,'0');
INSERT INTO daily_summary VALUES ('stock','INO','us','2017-11-01',5.85,5.879,5.67,5.7,737147,'0');
INSERT INTO daily_summary VALUES ('stock','INOD','us','2017-11-01',1.3,1.3,1.2,1.25,21811,'0');
INSERT INTO daily_summary VALUES ('stock','INOV','us','2017-11-01',16.85,16.85,16.125,16.4,717548,'0');
INSERT INTO daily_summary VALUES ('etf','INP','us','2017-11-01',85.8,86.529,85.8,86.14,2801,'0');
INSERT INTO daily_summary VALUES ('stock','INPX','us','2017-11-01',0.385,0.3998,0.363,0.3754,862252,'0');
INSERT INTO daily_summary VALUES ('stock','INS','us','2017-11-01',3.891,3.891,3.82,3.8892,2825,'0');
INSERT INTO daily_summary VALUES ('stock','INSE','us','2017-11-01',11.45,11.5,11.25,11.3,5306,'0');
INSERT INTO daily_summary VALUES ('stock','INSG','us','2017-11-01',1.5246,1.55,1.5061,1.53,246381,'0');
INSERT INTO daily_summary VALUES ('stock','INSI','us','2017-11-01',20.65,20.658,20.3452,20.39,35225,'0');
INSERT INTO daily_summary VALUES ('stock','INSM','us','2017-11-01',27.22,27.665,26.65,27.21,424307,'0');
INSERT INTO daily_summary VALUES ('stock','INST','us','2017-11-01',35,36,34.45,34.5,509700,'0');
INSERT INTO daily_summary VALUES ('stock','INSW','us','2017-11-01',20.34,20.6,19.99,20.24,166109,'0');
INSERT INTO daily_summary VALUES ('stock','INSY','us','2017-11-01',5.1,5.17,4.91,5.09,1204550,'0');
INSERT INTO daily_summary VALUES ('stock','INT','us','2017-11-01',28.05,28.68,27.69,28.57,1188850,'0');
INSERT INTO daily_summary VALUES ('stock','INTC','us','2017-11-01',45.702,46.438,45.324,46.438,48093900,'0');
INSERT INTO daily_summary VALUES ('etf','INTF','us','2017-11-01',28.74,28.785,28.68,28.69,117432,'0');
INSERT INTO daily_summary VALUES ('stock','INTG','us','2017-11-01',24.15,24.4999,24.15,24.4999,851,'0');
INSERT INTO daily_summary VALUES ('stock','INTL','us','2017-11-01',41.53,41.9,41.44,41.82,116416,'0');
INSERT INTO daily_summary VALUES ('stock','INTT','us','2017-11-01',8.6,8.6,8.45,8.5,79090,'0');
INSERT INTO daily_summary VALUES ('stock','INTU','us','2017-11-01',151.85,152.215,150.55,151.36,892377,'0');
INSERT INTO daily_summary VALUES ('stock','INTX','us','2017-11-01',2.47,2.75,2.47,2.58,9322,'0');
INSERT INTO daily_summary VALUES ('stock','INUV','us','2017-11-01',0.842,0.88,0.8326,0.88,70900,'0');
INSERT INTO daily_summary VALUES ('stock','INVA','us','2017-11-01',12.34,12.4489,12.16,12.21,569618,'0');
INSERT INTO daily_summary VALUES ('stock','INVE','us','2017-11-01',3.28,3.28,3.04,3.1,48577,'0');
INSERT INTO daily_summary VALUES ('stock','INVH','us','2017-11-01',22.71,22.85,22.51,22.65,1251170,'0');
INSERT INTO daily_summary VALUES ('stock','INWK','us','2017-11-01',10.95,11.08,10.59,10.73,151401,'0');
INSERT INTO daily_summary VALUES ('stock','INXN','us','2017-11-01',53.42,56.57,53.42,55.75,900769,'0');
INSERT INTO daily_summary VALUES ('etf','INXX','us','2017-11-01',15.73,15.795,15.71,15.75,18921,'0');
INSERT INTO daily_summary VALUES ('stock','IO','us','2017-11-01',7.9,8.1,7.65,7.8,57413,'0');
INSERT INTO daily_summary VALUES ('etf','IOIL','us','2017-11-01',10.66,10.66,10.66,10.66,550,'0');
INSERT INTO daily_summary VALUES ('stock','IONS','us','2017-11-01',57.32,58.8,57.1066,57.76,1305340,'0');
INSERT INTO daily_summary VALUES ('etf','IOO','us','2017-11-01',91.48,91.61,91.142,91.22,22416,'0');
INSERT INTO daily_summary VALUES ('stock','IOR','us','2017-11-01',9.2925,9.2925,9.2925,9.2925,154,'0');
INSERT INTO daily_summary VALUES ('stock','IOSP','us','2017-11-01',62.4,62.95,61.7,62.25,69299,'0');
INSERT INTO daily_summary VALUES ('stock','IOTS','us','2017-11-01',7.7,7.95,7.6,7.7,86246,'0');
INSERT INTO daily_summary VALUES ('stock','IOVA','us','2017-11-01',8,8.25,7.1,7.15,805638,'0');
INSERT INTO daily_summary VALUES ('stock','IP','us','2017-11-01',57.65,57.93,57.31,57.72,1778190,'0');
INSERT INTO daily_summary VALUES ('etf','IPAC','us','2017-11-01',58.94,59.07,58.73,58.88,200301,'0');
INSERT INTO daily_summary VALUES ('stock','IPAR','us','2017-11-01',46.6,46.7,45.65,46.2,68239,'0');
INSERT INTO daily_summary VALUES ('stock','IPAS','us','2017-11-01',0.6,0.6398,0.5641,0.57,658447,'0');
INSERT INTO daily_summary VALUES ('stock','IPAY','us','2017-11-01',34.13,34.13,33.7399,33.88,102021,'0');
INSERT INTO daily_summary VALUES ('stock','IPCC','us','2017-11-01',95.2,96.25,93.45,94.1,27326,'0');
INSERT INTO daily_summary VALUES ('stock','IPCI','us','2017-11-01',0.9224,0.94,0.885,0.92,149275,'0');
INSERT INTO daily_summary VALUES ('stock','IPDN','us','2017-11-01',2.77,3.4499,2.77,2.9,42453,'0');
INSERT INTO daily_summary VALUES ('etf','IPE','us','2017-11-01',56.33,56.47,56.32,56.41,634716,'0');
INSERT INTO daily_summary VALUES ('etf','IPFF','us','2017-11-01',18.51,18.51,18.38,18.4,52410,'0');
INSERT INTO daily_summary VALUES ('stock','IPG','us','2017-11-01',19.35,19.66,19.26,19.45,6169380,'0');
INSERT INTO daily_summary VALUES ('stock','IPGP','us','2017-11-01',214.71,215.52,204.135,206.44,591422,'0');
INSERT INTO daily_summary VALUES ('stock','IPHI','us','2017-11-01',40.35,42.54,40,41.15,1940770,'0');
INSERT INTO daily_summary VALUES ('stock','IPHS','us','2017-11-01',49.43,49.43,46.35,46.92,223343,'0');
INSERT INTO daily_summary VALUES ('stock','IPI','us','2017-11-01',4.01,4.17,3.87,3.91,1107830,'0');
INSERT INTO daily_summary VALUES ('etf','IPKW','us','2017-11-01',35.74,35.78,35.522,35.63,48375,'0');
INSERT INTO daily_summary VALUES ('stock','IPL_D','us','2017-11-01',25.5894,25.5894,25.33,25.33,1610,'0');
INSERT INTO daily_summary VALUES ('etf','IPO','us','2017-11-01',28.2,28.2,28.03,28.03,3398,'0');
INSERT INTO daily_summary VALUES ('stock','IPOA','us','2017-11-01',10.21,10.28,10.21,10.2689,31554,'0');
INSERT INTO daily_summary VALUES ('stock','IPWR','us','2017-11-01',2.07,2.2399,2.03,2.08,45314,'0');
INSERT INTO daily_summary VALUES ('stock','IPXL','us','2017-11-01',18.3,18.55,17.8,17.95,1709100,'0');
INSERT INTO daily_summary VALUES ('etf','IQDE','us','2017-11-01',25.3,25.3,25.2636,25.2636,2607,'0');
INSERT INTO daily_summary VALUES ('etf','IQDF','us','2017-11-01',26.52,26.5395,26.4228,26.4328,52299,'0');
INSERT INTO daily_summary VALUES ('stock','IQDG','us','2017-11-01',30.2285,30.2285,30.21,30.2202,861,'0');
INSERT INTO daily_summary VALUES ('etf','IQDY','us','2017-11-01',28.02,28.0599,28.01,28.01,2790,'0');
INSERT INTO daily_summary VALUES ('stock','IQI','us','2017-11-01',12.42,12.45,12.38,12.4,128776,'0');
INSERT INTO daily_summary VALUES ('etf','IQLT','us','2017-11-01',29.41,29.43,29.31,29.3355,18896,'0');
INSERT INTO daily_summary VALUES ('stock','IR','us','2017-11-01',89.25,89.38,87.33,87.53,1709350,'0');
INSERT INTO daily_summary VALUES ('stock','IRBT','us','2017-11-01',67.06,67.808,66.2,66.67,838039,'0');
INSERT INTO daily_summary VALUES ('stock','IRCP','us','2017-11-01',57.99,58,55.22,56,8010,'0');
INSERT INTO daily_summary VALUES ('stock','IRDM','us','2017-11-01',12.15,12.22,11.8,11.95,1241340,'0');
INSERT INTO daily_summary VALUES ('stock','IRET','us','2017-11-01',5.87,5.93,5.81,5.84,211921,'0');
INSERT INTO daily_summary VALUES ('stock','IRET_C','us','2017-11-01',25.37,25.37,25.25,25.3,11558,'0');
INSERT INTO daily_summary VALUES ('stock','IRIX','us','2017-11-01',8.6,9.07,8.39,8.45,49995,'0');
INSERT INTO daily_summary VALUES ('stock','IRL','us','2017-11-01',13.69,13.77,13.62,13.66,111252,'0');
INSERT INTO daily_summary VALUES ('stock','IRM','us','2017-11-01',40.18,40.33,39.76,39.81,826974,'0');
INSERT INTO daily_summary VALUES ('stock','IRMD','us','2017-11-01',14.55,14.6,13.6007,13.825,27986,'0');
INSERT INTO daily_summary VALUES ('stock','IROQ','us','2017-11-01',19.53,19.53,19.5,19.5,498,'0');
INSERT INTO daily_summary VALUES ('stock','IRR','us','2017-11-01',6.25,6.3,6.25,6.27,32930,'0');
INSERT INTO daily_summary VALUES ('stock','IRS','us','2017-11-01',27.962,28.555,26.988,27.657,91511,'0');
INSERT INTO daily_summary VALUES ('stock','IRT','us','2017-11-01',10.16,10.43,10.15,10.28,537195,'0');
INSERT INTO daily_summary VALUES ('stock','IRTC','us','2017-11-01',50.65,51.04,49.87,50.72,428610,'0');
INSERT INTO daily_summary VALUES ('stock','IRWD','us','2017-11-01',15.5,15.5,14.46,14.86,3294860,'0');
INSERT INTO daily_summary VALUES ('stock','ISBC','us','2017-11-01',13.737,13.836,13.558,13.588,4567800,'0');
INSERT INTO daily_summary VALUES ('stock','ISCA','us','2017-11-01',39.2,39.2,38.2,38.6,68594,'0');
INSERT INTO daily_summary VALUES ('etf','ISCF','us','2017-11-01',31.7001,31.785,31.7001,31.76,1903,'0');
INSERT INTO daily_summary VALUES ('stock','ISD','us','2017-11-01',15.16,15.21,15.16,15.18,144682,'0');
INSERT INTO daily_summary VALUES ('stock','ISDR','us','2017-11-01',16.95,16.95,16.85,16.85,2434,'0');
INSERT INTO daily_summary VALUES ('stock','ISF','us','2017-11-01',25.66,25.7,25.65,25.69,15069,'0');
INSERT INTO daily_summary VALUES ('stock','ISG','us','2017-11-01',25.6834,25.6834,25.61,25.65,5491,'0');
INSERT INTO daily_summary VALUES ('etf','ISHG','us','2017-11-01',82.55,82.5999,82.532,82.55,1688,'0');
INSERT INTO daily_summary VALUES ('stock','ISIG','us','2017-11-01',1.25,1.3252,1.22,1.29,49065,'0');
INSERT INTO daily_summary VALUES ('stock','ISL','us','2017-11-01',20.16,20.22,20.14,20.18,1913,'0');
INSERT INTO daily_summary VALUES ('stock','ISM','us','2017-11-01',24.99,25.007,24.94,24.95,3802,'0');
INSERT INTO daily_summary VALUES ('stock','ISMD','us','2017-11-01',26.43,26.43,26,26.11,18026,'0');
INSERT INTO daily_summary VALUES ('stock','ISNS','us','2017-11-01',3.35,3.35,3.3,3.3,210,'0');
INSERT INTO daily_summary VALUES ('stock','ISR','us','2017-11-01',0.4604,0.463,0.4365,0.46,122944,'0');
INSERT INTO daily_summary VALUES ('etf','ISRA','us','2017-11-01',29.6499,29.6499,29.32,29.45,5110,'0');
INSERT INTO daily_summary VALUES ('stock','ISRG','us','2017-11-01',377.31,378.39,373.865,376.04,668844,'0');
INSERT INTO daily_summary VALUES ('stock','ISSC','us','2017-11-01',3.24,3.38,3.19,3.24,51696,'0');
INSERT INTO daily_summary VALUES ('etf','ISTB','us','2017-11-01',50.02,50.09,50.02,50.04,69418,'0');
INSERT INTO daily_summary VALUES ('stock','ISTR','us','2017-11-01',23.3672,23.3672,22.85,23.05,3087,'0');
INSERT INTO daily_summary VALUES ('stock','IT','us','2017-11-01',125.75,126.22,124.45,125.6,705873,'0');
INSERT INTO daily_summary VALUES ('etf','ITA','us','2017-11-01',180.83,181.126,179.921,180.22,231765,'0');
INSERT INTO daily_summary VALUES ('etf','ITB','us','2017-11-01',39.76,40.01,39.72,39.98,4125610,'0');
INSERT INTO daily_summary VALUES ('stock','ITCB','us','2017-11-01',13.8355,13.8355,13.75,13.8,2365,'0');
INSERT INTO daily_summary VALUES ('stock','ITCI','us','2017-11-01',15.69,15.73,14.821,15.07,518076,'0');
INSERT INTO daily_summary VALUES ('etf','ITE','us','2017-11-01',59.7602,59.831,59.7602,59.8115,19578,'0');
INSERT INTO daily_summary VALUES ('stock','ITEK','us','2017-11-01',2.55,2.59,2.4876,2.52,327652,'0');
INSERT INTO daily_summary VALUES ('stock','ITEQ','us','2017-11-01',32.37,32.38,31.865,31.91,14026,'0');
INSERT INTO daily_summary VALUES ('stock','ITG','us','2017-11-01',22.6,22.93,19.5,20.01,1480850,'0');
INSERT INTO daily_summary VALUES ('stock','ITGR','us','2017-11-01',49,49.1,46.8,47.6,225978,'0');
INSERT INTO daily_summary VALUES ('stock','ITI','us','2017-11-01',6.96,7.03,6.77,6.82,51284,'0');
INSERT INTO daily_summary VALUES ('stock','ITIC','us','2017-11-01',189.27,189.27,189.27,189.27,526,'0');
INSERT INTO daily_summary VALUES ('etf','ITM','us','2017-11-01',24.06,24.07,24.0268,24.07,270592,'0');
INSERT INTO daily_summary VALUES ('etf','ITOT','us','2017-11-01',59.18,59.23,58.85,58.99,487441,'0');
INSERT INTO daily_summary VALUES ('stock','ITRI','us','2017-11-01',78.7,79.95,76.75,78.3,215159,'0');
INSERT INTO daily_summary VALUES ('stock','ITRN','us','2017-11-01',35.5,35.7,35.05,35.25,42727,'0');
INSERT INTO daily_summary VALUES ('stock','ITT','us','2017-11-01',47.21,47.46,46.78,46.8,1085510,'0');
INSERT INTO daily_summary VALUES ('stock','ITUB','us','2017-11-01',12.85,12.98,12.63,12.65,10600100,'0');
INSERT INTO daily_summary VALUES ('stock','ITUS','us','2017-11-01',2.03,2.03,1.93,1.96,218034,'0');
INSERT INTO daily_summary VALUES ('stock','ITW','us','2017-11-01',156.96,157.03,154.97,155.12,1725400,'0');
INSERT INTO daily_summary VALUES ('etf','IUSB','us','2017-11-01',50.91,50.97,50.8898,50.9,156350,'0');
INSERT INTO daily_summary VALUES ('etf','IUSG','us','2017-11-01',52.37,52.37,52.012,52.15,310814,'0');
INSERT INTO daily_summary VALUES ('etf','IUSV','us','2017-11-01',53.17,53.32,53.0199,53.12,198784,'0');
INSERT INTO daily_summary VALUES ('stock','IVAC','us','2017-11-01',8.65,8.65,8.0995,8.1,94267,'0');
INSERT INTO daily_summary VALUES ('stock','IVAL','us','2017-11-01',33.07,33.07,33,33,3224,'0');
INSERT INTO daily_summary VALUES ('stock','IVC','us','2017-11-01',15.7,15.9,15.25,15.5,225955,'0');
INSERT INTO daily_summary VALUES ('etf','IVE','us','2017-11-01',109.59,109.87,109.38,109.47,601316,'0');
INSERT INTO daily_summary VALUES ('stock','IVH','us','2017-11-01',15.67,15.69,15.58,15.58,32272,'0');
INSERT INTO daily_summary VALUES ('stock','IVLU','us','2017-11-01',26.08,26.1199,26.02,26.042,8484,'0');
INSERT INTO daily_summary VALUES ('etf','IVOG','us','2017-11-01',130.44,130.49,128.96,129.32,12873,'0');
INSERT INTO daily_summary VALUES ('etf','IVOO','us','2017-11-01',124.41,124.41,122.87,123.24,17259,'0');
INSERT INTO daily_summary VALUES ('etf','IVOV','us','2017-11-01',119.09,119.09,117.79,118.15,11685,'0');
INSERT INTO daily_summary VALUES ('stock','IVR','us','2017-11-01',17.34,17.39,17.24,17.39,649395,'0');
INSERT INTO daily_summary VALUES ('stock','IVR_A','us','2017-11-01',25.17,25.2,25.15,25.2,14285,'0');
INSERT INTO daily_summary VALUES ('stock','IVR_B','us','2017-11-01',26.24,26.4,26.21,26.28,5102,'0');
INSERT INTO daily_summary VALUES ('stock','IVR_C','us','2017-11-01',25.4,25.44,25.35,25.36,18558,'0');
INSERT INTO daily_summary VALUES ('stock','IVTY','us','2017-11-01',9,9.1,8.8226,8.95,30266,'0');
INSERT INTO daily_summary VALUES ('etf','IVV','us','2017-11-01',259.79,260.19,258.82,259.23,2512240,'0');
INSERT INTO daily_summary VALUES ('etf','IVW','us','2017-11-01',148.75,148.75,147.88,148.17,422331,'0');
INSERT INTO daily_summary VALUES ('stock','IVZ','us','2017-11-01',35.753,35.951,35.445,35.465,1749030,'0');
INSERT INTO daily_summary VALUES ('etf','IWB','us','2017-11-01',143.54,143.9,143.07,143.3,1030110,'0');
INSERT INTO daily_summary VALUES ('etf','IWC','us','2017-11-01',94.94,94.94,92.7918,93.15,49074,'0');
INSERT INTO daily_summary VALUES ('etf','IWD','us','2017-11-01',119.94,120.23,119.66,119.8,1777500,'0');
INSERT INTO daily_summary VALUES ('etf','IWF','us','2017-11-01',130.54,130.54,129.625,129.88,2883630,'0');
INSERT INTO daily_summary VALUES ('etf','IWL','us','2017-11-01',59.44,59.44,59.25,59.36,15974,'0');
INSERT INTO daily_summary VALUES ('etf','IWM','us','2017-11-01',150.5,150.53,147.44,148.27,31583700,'0');
INSERT INTO daily_summary VALUES ('etf','IWN','us','2017-11-01',125.18,125.392,123.05,123.63,688182,'0');
INSERT INTO daily_summary VALUES ('etf','IWO','us','2017-11-01',182.72,183.06,178.78,180.09,589999,'0');
INSERT INTO daily_summary VALUES ('etf','IWP','us','2017-11-01',116.98,117,115.81,116.04,439168,'0');
INSERT INTO daily_summary VALUES ('etf','IWR','us','2017-11-01',201.47,201.57,199.98,200.25,197255,'0');
INSERT INTO daily_summary VALUES ('etf','IWS','us','2017-11-01',85.98,86.26,85.64,85.72,333014,'0');
INSERT INTO daily_summary VALUES ('etf','IWV','us','2017-11-01',152.91,153.26,152.31,152.61,238303,'0');
INSERT INTO daily_summary VALUES ('etf','IWX','us','2017-11-01',50.559,50.6042,50.47,50.52,8439,'0');
INSERT INTO daily_summary VALUES ('etf','IWY','us','2017-11-01',70.89,70.934,70.49,70.6703,143787,'0');
INSERT INTO daily_summary VALUES ('stock','IX','us','2017-11-01',87.99,88.32,87.83,88.09,39553,'0');
INSERT INTO daily_summary VALUES ('etf','IXC','us','2017-11-01',34.32,34.52,34.26,34.39,200015,'0');
INSERT INTO daily_summary VALUES ('etf','IXG','us','2017-11-01',68.04,68.2035,67.7703,67.81,44461,'0');
INSERT INTO daily_summary VALUES ('etf','IXJ','us','2017-11-01',111.21,111.66,111.09,111.23,50981,'0');
INSERT INTO daily_summary VALUES ('etf','IXN','us','2017-11-01',154.02,154.68,153,153.54,114110,'0');
INSERT INTO daily_summary VALUES ('etf','IXP','us','2017-11-01',59.3,59.3,58.863,58.97,28006,'0');
INSERT INTO daily_summary VALUES ('etf','IXUS','us','2017-11-01',62.52,62.6107,62.314,62.35,354577,'0');
INSERT INTO daily_summary VALUES ('stock','IXYS','us','2017-11-01',24.5,24.5,23.15,23.4,502518,'0');
INSERT INTO daily_summary VALUES ('etf','IYC','us','2017-11-01',165.7,166.07,165.35,165.35,13707,'0');
INSERT INTO daily_summary VALUES ('etf','IYE','us','2017-11-01',37.38,37.73,37.36,37.56,435004,'0');
INSERT INTO daily_summary VALUES ('etf','IYF','us','2017-11-01',115.08,115.55,114.76,114.91,202549,'0');
INSERT INTO daily_summary VALUES ('etf','IYG','us','2017-11-01',124.2,124.803,123.61,123.82,142624,'0');
INSERT INTO daily_summary VALUES ('etf','IYH','us','2017-11-01',170.96,171.723,170.19,170.97,651527,'0');
INSERT INTO daily_summary VALUES ('etf','IYJ','us','2017-11-01',141.73,141.97,141,141.24,51793,'0');
INSERT INTO daily_summary VALUES ('etf','IYK','us','2017-11-01',121.11,121.11,120.67,120.67,13725,'0');
INSERT INTO daily_summary VALUES ('stock','IYLD','us','2017-11-01',25.486,25.586,25.486,25.546,26828,'0');
INSERT INTO daily_summary VALUES ('etf','IYM','us','2017-11-01',100.2,100.65,99.74,100.17,256369,'0');
INSERT INTO daily_summary VALUES ('etf','IYR','us','2017-11-01',80.05,80.31,79.83,79.98,5343150,'0');
INSERT INTO daily_summary VALUES ('etf','IYT','us','2017-11-01',176.99,177.13,175.4,176.1,242415,'0');
INSERT INTO daily_summary VALUES ('etf','IYW','us','2017-11-01',162.5,162.5,160.589,161.46,599672,'0');
INSERT INTO daily_summary VALUES ('etf','IYY','us','2017-11-01',129.29,129.53,128.806,128.95,13381,'0');
INSERT INTO daily_summary VALUES ('etf','IYZ','us','2017-11-01',30.23,30.48,29.4,29.67,756636,'0');
INSERT INTO daily_summary VALUES ('stock','IZEA','us','2017-11-01',3.36,4.03,3.2,3.95,833342,'0');
INSERT INTO daily_summary VALUES ('stock','JACK','us','2017-11-01',103.33,105.18,101.47,101.76,497459,'0');
INSERT INTO daily_summary VALUES ('stock','JAG','us','2017-11-01',14,14.47,13.91,14.43,1348860,'0');
INSERT INTO daily_summary VALUES ('stock','JAGX','us','2017-11-01',0.19,0.19,0.181,0.1864,370587,'0');
INSERT INTO daily_summary VALUES ('stock','JAKK','us','2017-11-01',2.8,2.85,2.55,2.6,257544,'0');
INSERT INTO daily_summary VALUES ('stock','JASN','us','2017-11-01',1.75,1.86,1.75,1.79,23772,'0');
INSERT INTO daily_summary VALUES ('stock','JASNW','us','2017-11-01',0.0115,0.0216,0.0115,0.021,4966,'0');
INSERT INTO daily_summary VALUES ('stock','JASO','us','2017-11-01',7.5,7.84,7.35,7.73,835005,'0');
INSERT INTO daily_summary VALUES ('stock','JAX','us','2017-11-01',10.65,10.8,10.45,10.45,38338,'0');
INSERT INTO daily_summary VALUES ('stock','JAZZ','us','2017-11-01',142,143.05,140.3,141.47,421232,'0');
INSERT INTO daily_summary VALUES ('stock','JBGS','us','2017-11-01',31.28,31.43,31.04,31.35,602019,'0');
INSERT INTO daily_summary VALUES ('stock','JBHT','us','2017-11-01',106.51,107.47,105.78,105.95,644011,'0');
INSERT INTO daily_summary VALUES ('stock','JBL','us','2017-11-01',28.51,28.59,28.205,28.42,1183720,'0');
INSERT INTO daily_summary VALUES ('stock','JBLU','us','2017-11-01',18.9,19.05,18.8,18.95,4993820,'0');
INSERT INTO daily_summary VALUES ('stock','JBN','us','2017-11-01',11.1006,11.1006,10.6,10.7,5000,'0');
INSERT INTO daily_summary VALUES ('stock','JBR','us','2017-11-01',11.03,11.1757,10.72,10.75,3860,'0');
INSERT INTO daily_summary VALUES ('stock','JBSS','us','2017-11-01',59,61.19,58.645,59.27,211800,'0');
INSERT INTO daily_summary VALUES ('stock','JBT','us','2017-11-01',108.15,108.15,104.15,105.35,245113,'0');
INSERT INTO daily_summary VALUES ('stock','JCAP','us','2017-11-01',20.57,20.575,20.26,20.56,150343,'0');
INSERT INTO daily_summary VALUES ('stock','JCE','us','2017-11-01',15.93,15.96,15.871,15.94,34984,'0');
INSERT INTO daily_summary VALUES ('stock','JCI','us','2017-11-01',41.76,41.85,41.1,41.2,3969590,'0');
INSERT INTO daily_summary VALUES ('stock','JCO','us','2017-11-01',9.95,10.02,9.95,10.02,42733,'0');
INSERT INTO daily_summary VALUES ('stock','JCOM','us','2017-11-01',74.25,74.61,73.49,74.16,521280,'0');
INSERT INTO daily_summary VALUES ('stock','JCP','us','2017-11-01',2.83,2.87,2.63,2.66,24804000,'0');
INSERT INTO daily_summary VALUES ('stock','JCS','us','2017-11-01',4.45,4.45,4.3,4.3442,2947,'0');
INSERT INTO daily_summary VALUES ('stock','JCTCF','us','2017-11-01',15.105,16.85,15.105,16.75,956,'0');
INSERT INTO daily_summary VALUES ('stock','JD','us','2017-11-01',38,38.31,37.72,37.86,15800100,'0');
INSERT INTO daily_summary VALUES ('stock','JDD','us','2017-11-01',12.76,12.81,12.68,12.71,71055,'0');
INSERT INTO daily_summary VALUES ('etf','JDST','us','2017-11-01',63.28,64.34,60.36,63.48,1685640,'0');
INSERT INTO daily_summary VALUES ('stock','JE','us','2017-11-01',5.58,5.62,5.565,5.58,57961,'0');
INSERT INTO daily_summary VALUES ('stock','JEC','us','2017-11-01',58.38,58.87,58.12,58.29,516183,'0');
INSERT INTO daily_summary VALUES ('stock','JELD','us','2017-11-01',37.24,37.32,36.34,36.66,315666,'0');
INSERT INTO daily_summary VALUES ('stock','JEMD','us','2017-11-01',10,10.01,10,10,106600,'0');
INSERT INTO daily_summary VALUES ('stock','JEQ','us','2017-11-01',9.19,9.23,9.19,9.21,12864,'0');
INSERT INTO daily_summary VALUES ('stock','JETS','us','2017-11-01',29.6,29.72,29.43,29.5554,39300,'0');
INSERT INTO daily_summary VALUES ('stock','JE_A','us','2017-11-01',26.0238,26.0985,25.95,25.95,1486,'0');
INSERT INTO daily_summary VALUES ('stock','JFR','us','2017-11-01',11.64,11.67,11.6,11.67,165026,'0');
INSERT INTO daily_summary VALUES ('stock','JGH','us','2017-11-01',17.42,17.42,17.21,17.24,107514,'0');
INSERT INTO daily_summary VALUES ('stock','JHA','us','2017-11-01',10,10,9.965,9.99,40338,'0');
INSERT INTO daily_summary VALUES ('stock','JHB','us','2017-11-01',10.11,10.11,10.06,10.1,129549,'0');
INSERT INTO daily_summary VALUES ('stock','JHD','us','2017-11-01',10.09,10.1,10.07,10.08,37193,'0');
INSERT INTO daily_summary VALUES ('etf','JHDG','us','2017-11-01',29.13,29.13,29.02,29.1199,33685,'0');
INSERT INTO daily_summary VALUES ('stock','JHG','us','2017-11-01',34.78,34.8,34.29,34.53,410222,'0');
INSERT INTO daily_summary VALUES ('stock','JHI','us','2017-11-01',17.84,17.93,17.8,17.86,17274,'0');
INSERT INTO daily_summary VALUES ('stock','JHMC','us','2017-11-01',28.32,28.32,28.32,28.32,427,'0');
INSERT INTO daily_summary VALUES ('stock','JHMD','us','2017-11-01',30.13,30.18,30.079,30.095,3783,'0');
INSERT INTO daily_summary VALUES ('stock','JHME','us','2017-11-01',27.55,27.99,27.55,27.99,446,'0');
INSERT INTO daily_summary VALUES ('stock','JHMF','us','2017-11-01',35.4191,35.4191,35.23,35.2489,3341,'0');
INSERT INTO daily_summary VALUES ('stock','JHMH','us','2017-11-01',30.33,30.3499,30.25,30.3499,144826,'0');
INSERT INTO daily_summary VALUES ('stock','JHMI','us','2017-11-01',33.73,33.73,33.64,33.64,254,'0');
INSERT INTO daily_summary VALUES ('stock','JHML','us','2017-11-01',33.7,33.709,33.56,33.59,35261,'0');
INSERT INTO daily_summary VALUES ('stock','JHMM','us','2017-11-01',33.28,33.28,32.86,32.9,16271,'0');
INSERT INTO daily_summary VALUES ('stock','JHMT','us','2017-11-01',41.16,41.16,40.6432,40.9,124458,'0');
INSERT INTO daily_summary VALUES ('stock','JHMU','us','2017-11-01',28.75,28.75,28.75,28.75,101,'0');
INSERT INTO daily_summary VALUES ('stock','JHS','us','2017-11-01',14.82,14.85,14.775,14.85,28671,'0');
INSERT INTO daily_summary VALUES ('stock','JHX','us','2017-11-01',15.12,15.12,15,15.12,15102,'0');
INSERT INTO daily_summary VALUES ('stock','JHY','us','2017-11-01',10.17,10.17,10.06,10.07,32409,'0');
INSERT INTO daily_summary VALUES ('stock','JILL','us','2017-11-01',5.13,5.24,5.05,5.09,277018,'0');
INSERT INTO daily_summary VALUES ('etf','JJA','us','2017-11-01',30.9432,30.98,30.94,30.98,1015,'0');
INSERT INTO daily_summary VALUES ('etf','JJC','us','2017-11-01',35.89,35.97,35.73,35.88,41729,'0');
INSERT INTO daily_summary VALUES ('etf','JJE','us','2017-11-01',5.84,5.87,5.8,5.8,942,'0');
INSERT INTO daily_summary VALUES ('etf','JJG','us','2017-11-01',25.09,25.3399,25.09,25.24,11254,'0');
INSERT INTO daily_summary VALUES ('etf','JJM','us','2017-11-01',29.5,29.51,29.4824,29.5,1730,'0');
INSERT INTO daily_summary VALUES ('etf','JJN','us','2017-11-01',15.34,15.6191,15.27,15.57,69032,'0');
INSERT INTO daily_summary VALUES ('stock','JJSF','us','2017-11-01',133.46,134.54,131.895,132.28,34075,'0');
INSERT INTO daily_summary VALUES ('etf','JJT','us','2017-11-01',44.5,44.5177,44.5,44.5,1949,'0');
INSERT INTO daily_summary VALUES ('etf','JJU','us','2017-11-01',19.01,19.01,19.01,19.01,140,'0');
INSERT INTO daily_summary VALUES ('etf','JKD','us','2017-11-01',154.49,154.55,153.869,153.95,21346,'0');
INSERT INTO daily_summary VALUES ('etf','JKE','us','2017-11-01',152.09,152.09,151.47,151.63,12385,'0');
INSERT INTO daily_summary VALUES ('etf','JKF','us','2017-11-01',101.67,101.79,101.48,101.576,5393,'0');
INSERT INTO daily_summary VALUES ('etf','JKG','us','2017-11-01',178.68,178.68,177.695,177.964,5995,'0');
INSERT INTO daily_summary VALUES ('etf','JKH','us','2017-11-01',196.13,196.13,194.799,194.83,3524,'0');
INSERT INTO daily_summary VALUES ('stock','JKHY','us','2017-11-01',110.58,110.665,109.03,109.76,544814,'0');
INSERT INTO daily_summary VALUES ('etf','JKI','us','2017-11-01',152.541,152.96,151.97,152.02,10161,'0');
INSERT INTO daily_summary VALUES ('etf','JKJ','us','2017-11-01',167.03,167.03,165.57,166.11,2607,'0');
INSERT INTO daily_summary VALUES ('etf','JKK','us','2017-11-01',176.39,176.39,172.74,173.84,2096,'0');
INSERT INTO daily_summary VALUES ('etf','JKL','us','2017-11-01',147.45,147.75,145.546,146.03,4672,'0');
INSERT INTO daily_summary VALUES ('stock','JKS','us','2017-11-01',26.63,27.5,26.63,27.15,430706,'0');
INSERT INTO daily_summary VALUES ('stock','JLL','us','2017-11-01',130,131.345,128.86,130.8,201738,'0');
INSERT INTO daily_summary VALUES ('stock','JLS','us','2017-11-01',25.84,25.86,25.72,25.86,18035,'0');
INSERT INTO daily_summary VALUES ('stock','JMBA','us','2017-11-01',8.32,8.52,8.27,8.44,57868,'0');
INSERT INTO daily_summary VALUES ('stock','JMEI','us','2017-11-01',2.7,2.76,2.67,2.67,416244,'0');
INSERT INTO daily_summary VALUES ('stock','JMF','us','2017-11-01',10.97,11.14,10.97,11.06,226217,'0');
INSERT INTO daily_summary VALUES ('stock','JMLP','us','2017-11-01',8.33,8.5,8.33,8.5,30790,'0');
INSERT INTO daily_summary VALUES ('stock','JMM','us','2017-11-01',7.6,7.6,7.59,7.59,5269,'0');
INSERT INTO daily_summary VALUES ('stock','JMP','us','2017-11-01',5.39,5.576,5.39,5.5,15197,'0');
INSERT INTO daily_summary VALUES ('stock','JMPB','us','2017-11-01',25.55,25.55,25.45,25.45,2902,'0');
INSERT INTO daily_summary VALUES ('stock','JMPC','us','2017-11-01',25.442,25.4583,25.29,25.3809,2900,'0');
INSERT INTO daily_summary VALUES ('stock','JMT','us','2017-11-01',24.51,24.66,24.5011,24.66,10958,'0');
INSERT INTO daily_summary VALUES ('stock','JMU','us','2017-11-01',0.96,0.99,0.96,0.97,4315,'0');
INSERT INTO daily_summary VALUES ('stock','JNCE','us','2017-11-01',14.13,14.46,13.84,14.33,117672,'0');
INSERT INTO daily_summary VALUES ('stock','JNJ','us','2017-11-01',139.83,140.59,139.34,139.98,5006660,'0');
INSERT INTO daily_summary VALUES ('etf','JNK','us','2017-11-01',37.1,37.11,37.03,37.04,7026870,'0');
INSERT INTO daily_summary VALUES ('stock','JNP','us','2017-11-01',4.85,4.85,4.4,4.7,58939,'0');
INSERT INTO daily_summary VALUES ('stock','JNPR','us','2017-11-01',25,25,24.51,24.86,6596900,'0');
INSERT INTO daily_summary VALUES ('etf','JNUG','us','2017-11-01',15.38,16.1,15.1381,15.28,12341300,'0');
INSERT INTO daily_summary VALUES ('etf','JO','us','2017-11-01',15.73,15.85,15.57,15.81,497403,'0');
INSERT INTO daily_summary VALUES ('stock','JOB','us','2017-11-01',2.8,2.9,2.8,2.81,12352,'0');
INSERT INTO daily_summary VALUES ('stock','JOBS','us','2017-11-01',61.68,62.5,60.96,61.26,75915,'0');
INSERT INTO daily_summary VALUES ('stock','JOE','us','2017-11-01',17.9,17.9,17.425,17.65,742619,'0');
INSERT INTO daily_summary VALUES ('stock','JOF','us','2017-11-01',12.98,13.09,12.95,13.0297,85293,'0');
INSERT INTO daily_summary VALUES ('stock','JONE','us','2017-11-01',1.32,1.35,1.26,1.31,620046,'0');
INSERT INTO daily_summary VALUES ('stock','JOUT','us','2017-11-01',74.94,75.22,72.85,73.11,25446,'0');
INSERT INTO daily_summary VALUES ('stock','JP','us','2017-11-01',22.58,23.18,21.15,21.76,478517,'0');
INSERT INTO daily_summary VALUES ('stock','JPC','us','2017-11-01',10.58,10.58,10.5,10.53,183970,'0');
INSERT INTO daily_summary VALUES ('stock','JPEH','us','2017-11-01',29.2301,29.2301,29.2301,29.2301,200,'0');
INSERT INTO daily_summary VALUES ('stock','JPEM','us','2017-11-01',57.7,57.835,57.46,57.56,10080,'0');
INSERT INTO daily_summary VALUES ('stock','JPEU','us','2017-11-01',61.46,61.58,61.3467,61.35,3425,'0');
INSERT INTO daily_summary VALUES ('stock','JPGB','us','2017-11-01',51.7299,51.7299,51.59,51.59,4783,'0');
INSERT INTO daily_summary VALUES ('etf','JPGE','us','2017-11-01',61.58,61.6,61.44,61.462,16687,'0');
INSERT INTO daily_summary VALUES ('stock','JPHF','us','2017-11-01',26.16,26.17,26.07,26.17,8016,'0');
INSERT INTO daily_summary VALUES ('stock','JPHY','us','2017-11-01',51.688,51.71,51.6704,51.6862,14371,'0');
INSERT INTO daily_summary VALUES ('stock','JPI','us','2017-11-01',25.24,25.34,25.11,25.34,34415,'0');
INSERT INTO daily_summary VALUES ('stock','JPIH','us','2017-11-01',30.1801,30.256,30.18,30.22,5219,'0');
INSERT INTO daily_summary VALUES ('stock','JPIN','us','2017-11-01',59.77,59.839,59.56,59.62,122432,'0');
INSERT INTO daily_summary VALUES ('stock','JPM','us','2017-11-01',101.1,101.7,100.73,100.92,8417170,'0');
INSERT INTO daily_summary VALUES ('stock','JPME','us','2017-11-01',62.3,62.3,61.95,62.04,4371,'0');
INSERT INTO daily_summary VALUES ('etf','JPMV','us','2017-11-01',67.2101,67.22,67.1589,67.208,2856,'0');
INSERT INTO daily_summary VALUES ('stock','JPM_A','us','2017-11-01',25.2,25.2857,25.2,25.24,27586,'0');
INSERT INTO daily_summary VALUES ('stock','JPM_B','us','2017-11-01',26.85,26.85,26.706,26.72,70916,'0');
INSERT INTO daily_summary VALUES ('stock','JPM_E','us','2017-11-01',26.6,26.68,26.6,26.66,41054,'0');
INSERT INTO daily_summary VALUES ('stock','JPM_F','us','2017-11-01',26.76,26.8156,26.71,26.775,39471,'0');
INSERT INTO daily_summary VALUES ('stock','JPM_G','us','2017-11-01',26.85,26.97,26.85,26.97,26635,'0');
INSERT INTO daily_summary VALUES ('stock','JPM_H','us','2017-11-01',27.03,27.102,27.03,27.06,26103,'0');
INSERT INTO daily_summary VALUES ('stock','JPN','us','2017-11-01',28.72,28.9965,28.72,28.9448,2324,'0');
INSERT INTO daily_summary VALUES ('etf','JPNL','us','2017-11-01',73.3,73.7,73.1,73.15,24674,'0');
INSERT INTO daily_summary VALUES ('stock','JPS','us','2017-11-01',10.34,10.34,10.29,10.33,297866,'0');
INSERT INTO daily_summary VALUES ('stock','JPSE','us','2017-11-01',28.97,28.9971,28.7699,28.87,7570,'0');
INSERT INTO daily_summary VALUES ('stock','JPST','us','2017-11-01',50.09,50.13,50.08,50.11,42064,'0');
INSERT INTO daily_summary VALUES ('stock','JPT','us','2017-11-01',25.55,25.55,25.4599,25.49,2381,'0');
INSERT INTO daily_summary VALUES ('stock','JPUS','us','2017-11-01',68.72,68.815,68.505,68.56,17960,'0');
INSERT INTO daily_summary VALUES ('stock','JPXN','us','2017-11-01',64.1,64.18,64.005,64.005,19151,'0');
INSERT INTO daily_summary VALUES ('stock','JQC','us','2017-11-01',8.37,8.39,8.31,8.32,222485,'0');
INSERT INTO daily_summary VALUES ('stock','JRI','us','2017-11-01',18.1,18.15,18.0395,18.09,65283,'0');
INSERT INTO daily_summary VALUES ('stock','JRJC','us','2017-11-01',2.3,2.34,2.19,2.26,43217,'0');
INSERT INTO daily_summary VALUES ('stock','JRJR','us','2017-11-01',0.1757,0.191,0.1757,0.1805,78099,'0');
INSERT INTO daily_summary VALUES ('stock','JRO','us','2017-11-01',11.45,11.52,11.43,11.51,110001,'0');
INSERT INTO daily_summary VALUES ('stock','JRS','us','2017-11-01',10.98,11.0649,10.94,10.96,97781,'0');
INSERT INTO daily_summary VALUES ('stock','JRVR','us','2017-11-01',42.59,43.48,41.915,42,60636,'0');
INSERT INTO daily_summary VALUES ('stock','JSD','us','2017-11-01',17.75,17.75,17.5503,17.5704,26841,'0');
INSERT INTO daily_summary VALUES ('stock','JSM','us','2017-11-01',24.24,24.4254,24.234,24.35,11928,'0');
INSERT INTO daily_summary VALUES ('stock','JSMD','us','2017-11-01',36.7,36.7,36.4401,36.6,768,'0');
INSERT INTO daily_summary VALUES ('stock','JSML','us','2017-11-01',35.91,35.91,35.37,35.6013,1061,'0');
INSERT INTO daily_summary VALUES ('stock','JSYN','us','2017-11-01',10.32,10.32,10.3,10.3,1500,'0');
INSERT INTO daily_summary VALUES ('stock','JSYNU','us','2017-11-01',10.76,10.76,10.76,10.76,100,'0');
INSERT INTO daily_summary VALUES ('stock','JSYNW','us','2017-11-01',0.2,0.2,0.2,0.2,1097,'0');
INSERT INTO daily_summary VALUES ('stock','JTA','us','2017-11-01',13.62,13.65,13.6035,13.64,10772,'0');
INSERT INTO daily_summary VALUES ('stock','JTD','us','2017-11-01',17.06,17.1,16.96,16.96,51806,'0');
INSERT INTO daily_summary VALUES ('stock','JTPY','us','2017-11-01',2.05,2.13,1.9,1.95,13821,'0');
INSERT INTO daily_summary VALUES ('stock','JUNO','us','2017-11-01',49,52,47.21,48.31,5043690,'0');
INSERT INTO daily_summary VALUES ('stock','JVA','us','2017-11-01',4.17,4.272,4.17,4.253,28861,'0');
INSERT INTO daily_summary VALUES ('stock','JWN','us','2017-11-01',39.77,40.5,39.57,39.94,1891380,'0');
INSERT INTO daily_summary VALUES ('etf','JXI','us','2017-11-01',52.04,52.05,51.83,51.83,19642,'0');
INSERT INTO daily_summary VALUES ('stock','JYNT','us','2017-11-01',4.56,4.7499,4.3,4.44,26975,'0');
INSERT INTO daily_summary VALUES ('stock','K','us','2017-11-01',62.64,63.72,61.81,63.01,4620460,'0');
INSERT INTO daily_summary VALUES ('stock','KAAC','us','2017-11-01',9.69,9.71,9.69,9.71,105228,'0');
INSERT INTO daily_summary VALUES ('stock','KAACU','us','2017-11-01',9.9979,9.9979,9.9979,9.9979,100,'0');
INSERT INTO daily_summary VALUES ('stock','KAACW','us','2017-11-01',0.82,0.82,0.82,0.82,300,'0');
INSERT INTO daily_summary VALUES ('stock','KAI','us','2017-11-01',113.1,113.1,100.25,109.75,302764,'0');
INSERT INTO daily_summary VALUES ('stock','KALA','us','2017-11-01',16.04,16.3273,14.9,15.03,128260,'0');
INSERT INTO daily_summary VALUES ('stock','KALU','us','2017-11-01',100.17,100.79,97.895,98.14,116645,'0');
INSERT INTO daily_summary VALUES ('stock','KALV','us','2017-11-01',12.86,13.0012,11.83,12.01,152995,'0');
INSERT INTO daily_summary VALUES ('stock','KAMN','us','2017-11-01',56.4,56.72,55.5,55.84,145445,'0');
INSERT INTO daily_summary VALUES ('stock','KANG','us','2017-11-01',14.28,14.59,14.22,14.32,236246,'0');
INSERT INTO daily_summary VALUES ('stock','KAP','us','2017-11-01',25.35,25.35,25.3,25.3,843,'0');
INSERT INTO daily_summary VALUES ('stock','KAR','us','2017-11-01',47.3,49.04,47.21,47.4,2503270,'0');
INSERT INTO daily_summary VALUES ('stock','KB','us','2017-11-01',53,53.35,52.87,52.95,82247,'0');
INSERT INTO daily_summary VALUES ('etf','KBA','us','2017-11-01',34.3,34.57,34.19,34.55,91335,'0');
INSERT INTO daily_summary VALUES ('stock','KBAL','us','2017-11-01',19.35,19.47,18.9835,19.29,88536,'0');
INSERT INTO daily_summary VALUES ('etf','KBE','us','2017-11-01',45.85,46.13,45.36,45.51,1672020,'0');
INSERT INTO daily_summary VALUES ('stock','KBH','us','2017-11-01',27.62,27.7301,27.33,27.54,1743010,'0');
INSERT INTO daily_summary VALUES ('stock','KBLM','us','2017-11-01',9.71,9.71,9.71,9.71,16900,'0');
INSERT INTO daily_summary VALUES ('stock','KBLMW','us','2017-11-01',0.2,0.2,0.2,0.2,60050,'0');
INSERT INTO daily_summary VALUES ('stock','KBR','us','2017-11-01',19.8,19.8,19.09,19.39,2776200,'0');
INSERT INTO daily_summary VALUES ('stock','KBSF','us','2017-11-01',2.39,2.39,2.03,2.18,12462,'0');
INSERT INTO daily_summary VALUES ('stock','KBWB','us','2017-11-01',52.61,52.94,52.41,52.53,688039,'0');
INSERT INTO daily_summary VALUES ('stock','KBWD','us','2017-11-01',23.5,23.61,23.3535,23.45,54692,'0');
INSERT INTO daily_summary VALUES ('etf','KBWP','us','2017-11-01',60.09,60.09,59.5,59.59,15050,'0');
INSERT INTO daily_summary VALUES ('etf','KBWR','us','2017-11-01',55.45,55.45,54.51,54.83,19693,'0');
INSERT INTO daily_summary VALUES ('stock','KBWY','us','2017-11-01',36.87,37.116,36.605,36.82,54957,'0');
INSERT INTO daily_summary VALUES ('stock','KCAP','us','2017-11-01',3.39,3.43,3.39,3.41,94004,'0');
INSERT INTO daily_summary VALUES ('stock','KCAPL','us','2017-11-01',25.1,25.16,25.1,25.16,5045,'0');
INSERT INTO daily_summary VALUES ('etf','KCE','us','2017-11-01',53.7,53.7,53.1111,53.32,22367,'0');
INSERT INTO daily_summary VALUES ('stock','KCNY','us','2017-11-01',34.41,34.7293,34.41,34.65,1370,'0');
INSERT INTO daily_summary VALUES ('stock','KDMN','us','2017-11-01',3.66,3.66,3.47,3.57,285720,'0');
INSERT INTO daily_summary VALUES ('stock','KE','us','2017-11-01',22.3,22.3,21.7,21.85,40899,'0');
INSERT INTO daily_summary VALUES ('stock','KED','us','2017-11-01',15.5,15.74,15.5,15.62,45506,'0');
INSERT INTO daily_summary VALUES ('stock','KEG','us','2017-11-01',10.81,11.15,10.45,10.63,78269,'0');
INSERT INTO daily_summary VALUES ('stock','KELYA','us','2017-11-01',26.61,26.61,25.87,26.3,58763,'0');
INSERT INTO daily_summary VALUES ('stock','KEM','us','2017-11-01',26.09,26.27,24.03,24.82,2642730,'0');
INSERT INTO daily_summary VALUES ('stock','KEMP','us','2017-11-01',26.2172,26.2172,26.2172,26.2172,0,'0');
INSERT INTO daily_summary VALUES ('stock','KEMQ','us','2017-11-01',24.66,24.72,24.62,24.62,3660,'0');
INSERT INTO daily_summary VALUES ('stock','KEN','us','2017-11-01',17.04,17.049,16.83,16.83,1800,'0');
INSERT INTO daily_summary VALUES ('stock','KEP','us','2017-11-01',17.39,17.45,17.26,17.39,579742,'0');
INSERT INTO daily_summary VALUES ('stock','KEQU','us','2017-11-01',28.6,29.1,28.55,28.8,2059,'0');
INSERT INTO daily_summary VALUES ('stock','KERX','us','2017-11-01',6.55,6.689,6.02,6.03,2625100,'0');
INSERT INTO daily_summary VALUES ('stock','KEX','us','2017-11-01',72,72.95,70.5,70.6,624925,'0');
INSERT INTO daily_summary VALUES ('stock','KEY','us','2017-11-01',18.33,18.575,18.27,18.28,7484560,'0');
INSERT INTO daily_summary VALUES ('stock','KEYS','us','2017-11-01',44.84,44.94,44.55,44.57,519550,'0');
INSERT INTO daily_summary VALUES ('stock','KEYW','us','2017-11-01',7.62,7.7,7.4276,7.52,278500,'0');
INSERT INTO daily_summary VALUES ('stock','KEY_I','us','2017-11-01',29.2184,29.27,29.01,29.0981,13533,'0');
INSERT INTO daily_summary VALUES ('stock','KF','us','2017-11-01',44.45,44.55,44.06,44.41,4464,'0');
INSERT INTO daily_summary VALUES ('stock','KFFB','us','2017-11-01',9.75,9.75,9.7,9.7,715,'0');
INSERT INTO daily_summary VALUES ('stock','KFN_','us','2017-11-01',25.44,25.44,25.33,25.34,9285,'0');
INSERT INTO daily_summary VALUES ('stock','KFRC','us','2017-11-01',22.5,23.55,22.1,22.8,241277,'0');
INSERT INTO daily_summary VALUES ('stock','KFS','us','2017-11-01',5.5,5.6233,5.5,5.55,15683,'0');
INSERT INTO daily_summary VALUES ('stock','KFY','us','2017-11-01',42.09,42.09,41.035,41.4,361183,'0');
INSERT INTO daily_summary VALUES ('etf','KFYP','us','2017-11-01',42.4359,42.4359,42.4359,42.4359,200,'0');
INSERT INTO daily_summary VALUES ('stock','KGC','us','2017-11-01',3.98,4.04,3.91,3.96,9675730,'0');
INSERT INTO daily_summary VALUES ('stock','KGJI','us','2017-11-01',2.2,2.28,2.15,2.26,264373,'0');
INSERT INTO daily_summary VALUES ('stock','KGRN','us','2017-11-01',25.62,25.65,25.4,25.4,53995,'0');
INSERT INTO daily_summary VALUES ('stock','KHC','us','2017-11-01',77.71,77.86,77.08,77.7,4052600,'0');
INSERT INTO daily_summary VALUES ('stock','KIDS','us','2017-11-01',18.31,19.45,18.12,18.96,43768,'0');
INSERT INTO daily_summary VALUES ('etf','KIE','us','2017-11-01',92.4,92.712,91.98,92.1,92992,'0');
INSERT INTO daily_summary VALUES ('stock','KIM','us','2017-11-01',18.22,18.73,18.12,18.55,4479290,'0');
INSERT INTO daily_summary VALUES ('stock','KIM_I','us','2017-11-01',25.17,25.21,25.17,25.21,26063,'0');
INSERT INTO daily_summary VALUES ('stock','KIM_J','us','2017-11-01',25.08,25.13,25.08,25.12,7120,'0');
INSERT INTO daily_summary VALUES ('stock','KIM_K','us','2017-11-01',25.15,25.32,25.15,25.22,6183,'0');
INSERT INTO daily_summary VALUES ('stock','KIM_L','us','2017-11-01',24.8625,24.89,24.68,24.805,32923,'0');
INSERT INTO daily_summary VALUES ('stock','KIN','us','2017-11-01',7.55,7.6,6.8,7.1,87176,'0');
INSERT INTO daily_summary VALUES ('stock','KINS','us','2017-11-01',16.35,16.4,16,16.1,27327,'0');
INSERT INTO daily_summary VALUES ('stock','KIO','us','2017-11-01',16.857,16.867,16.569,16.569,143957,'0');
INSERT INTO daily_summary VALUES ('stock','KIOR','us','2017-11-01',0.2925,0.3,0.283,0.2954,1073490,'0');
INSERT INTO daily_summary VALUES ('stock','KIQ','us','2017-11-01',0.37,0.4099,0.37,0.4099,7124,'0');
INSERT INTO daily_summary VALUES ('stock','KIRK','us','2017-11-01',11.8,11.92,11.59,11.78,90264,'0');
INSERT INTO daily_summary VALUES ('stock','KKR','us','2017-11-01',20.018,20.037,19.621,19.74,2697310,'0');
INSERT INTO daily_summary VALUES ('stock','KKR_A','us','2017-11-01',27.4699,27.55,27.39,27.43,12118,'0');
INSERT INTO daily_summary VALUES ('stock','KKR_B','us','2017-11-01',27.0325,27.25,27.01,27.01,3708,'0');
INSERT INTO daily_summary VALUES ('stock','KL','us','2017-11-01',11.8,12.55,11.79,12.4,4144050,'0');
INSERT INTO daily_summary VALUES ('stock','KLAC','us','2017-11-01',110,110.005,103.09,105.15,2316110,'0');
INSERT INTO daily_summary VALUES ('stock','KLDW','us','2017-11-01',32.82,32.82,32.65,32.6885,10122,'0');
INSERT INTO daily_summary VALUES ('stock','KLDX','us','2017-11-01',2.94,2.98,2.87,2.88,1247060,'0');
INSERT INTO daily_summary VALUES ('stock','KLIC','us','2017-11-01',22.72,22.85,21.96,22.32,464614,'0');
INSERT INTO daily_summary VALUES ('stock','KLXI','us','2017-11-01',55.35,55.97,54.2941,55.3,466073,'0');
INSERT INTO daily_summary VALUES ('stock','KMB','us','2017-11-01',111.9,113.46,111.1,112.05,2595820,'0');
INSERT INTO daily_summary VALUES ('stock','KMDA','us','2017-11-01',4.7,4.8,4.55,4.75,110285,'0');
INSERT INTO daily_summary VALUES ('stock','KMF','us','2017-11-01',13.4,13.53,13.32,13.41,125869,'0');
INSERT INTO daily_summary VALUES ('stock','KMG','us','2017-11-01',55.44,55.76,54.8,55.69,193333,'0');
INSERT INTO daily_summary VALUES ('stock','KMI','us','2017-11-01',18.25,18.38,18.05,18.13,10526300,'0');
INSERT INTO daily_summary VALUES ('stock','KMI_A','us','2017-11-01',37.6,38.099,37.5,37.55,131992,'0');
INSERT INTO daily_summary VALUES ('stock','KMM','us','2017-11-01',8.96,8.98,8.94,8.94,25086,'0');
INSERT INTO daily_summary VALUES ('stock','KMPA','us','2017-11-01',26.43,26.649,26.36,26.649,619,'0');
INSERT INTO daily_summary VALUES ('stock','KMPH','us','2017-11-01',3.8,3.95,3.6,3.6,33408,'0');
INSERT INTO daily_summary VALUES ('stock','KMPR','us','2017-11-01',64.566,65.214,63.969,64.068,234997,'0');
INSERT INTO daily_summary VALUES ('stock','KMT','us','2017-11-01',43.79,43.79,42.586,43.143,1222330,'0');
INSERT INTO daily_summary VALUES ('stock','KMX','us','2017-11-01',75.3,75.505,74.16,74.46,1035300,'0');
INSERT INTO daily_summary VALUES ('stock','KN','us','2017-11-01',16.7,16.7,16.15,16.19,842280,'0');
INSERT INTO daily_summary VALUES ('stock','KND','us','2017-11-01',6.05,6.0621,5.75,5.85,961336,'0');
INSERT INTO daily_summary VALUES ('stock','KNDI','us','2017-11-01',7.45,7.5,6.6,6.85,948936,'0');
INSERT INTO daily_summary VALUES ('stock','KNL','us','2017-11-01',21.3,21.4,20.95,20.98,273145,'0');
INSERT INTO daily_summary VALUES ('stock','KNOP','us','2017-11-01',23.25,23.25,22.5,23,156786,'0');
INSERT INTO daily_summary VALUES ('etf','KNOW','us','2017-11-01',41.86,42,41.62,41.76,33696,'0');
INSERT INTO daily_summary VALUES ('stock','KNSL','us','2017-11-01',43.76,43.98,42.48,43.18,67234,'0');
INSERT INTO daily_summary VALUES ('stock','KNX','us','2017-11-01',41.57,41.92,40.68,41.09,1123350,'0');
INSERT INTO daily_summary VALUES ('stock','KO','us','2017-11-01',45.75,46.09,45.63,45.8,7067640,'0');
INSERT INTO daily_summary VALUES ('stock','KODK','us','2017-11-01',5.45,5.6,5.05,5.15,189672,'0');
INSERT INTO daily_summary VALUES ('stock','KOF','us','2017-11-01',67.597,68.021,66.561,66.62,68914,'0');
INSERT INTO daily_summary VALUES ('etf','KOL','us','2017-11-01',15.15,15.2062,15.105,15.15,20886,'0');
INSERT INTO daily_summary VALUES ('etf','KOLD','us','2017-11-01',40.74,40.78,39.92,39.94,98969,'0');
INSERT INTO daily_summary VALUES ('stock','KONA','us','2017-11-01',3.35,3.375,3.2,3.225,31546,'0');
INSERT INTO daily_summary VALUES ('stock','KONE','us','2017-11-01',3.3,3.38,3.18,3.38,14019,'0');
INSERT INTO daily_summary VALUES ('stock','KOOL','us','2017-11-01',3.9401,3.9573,3.85,3.86,24795,'0');
INSERT INTO daily_summary VALUES ('stock','KOP','us','2017-11-01',48.95,49.35,48,48.2,75982,'0');
INSERT INTO daily_summary VALUES ('stock','KOPN','us','2017-11-01',3.54,3.54,3.36,3.39,521500,'0');
INSERT INTO daily_summary VALUES ('stock','KORS','us','2017-11-01',49.31,49.4,48.22,48.48,2549940,'0');
INSERT INTO daily_summary VALUES ('etf','KORU','us','2017-11-01',63.12,64.82,63.12,63.77,27671,'0');
INSERT INTO daily_summary VALUES ('stock','KOS','us','2017-11-01',7.77,8.1,7.71,8.02,2198520,'0');
INSERT INTO daily_summary VALUES ('stock','KOSS','us','2017-11-01',1.57,1.57,1.48,1.51,32906,'0');
INSERT INTO daily_summary VALUES ('stock','KPTI','us','2017-11-01',10.32,10.56,10.29,10.36,142754,'0');
INSERT INTO daily_summary VALUES ('stock','KR','us','2017-11-01',20.76,21.29,20.76,20.9,9975190,'0');
INSERT INTO daily_summary VALUES ('stock','KRA','us','2017-11-01',49.5,50.14,48.8,49.34,490745,'0');
INSERT INTO daily_summary VALUES ('stock','KRC','us','2017-11-01',71.14,71.94,71.06,71.69,394611,'0');
INSERT INTO daily_summary VALUES ('etf','KRE','us','2017-11-01',57.31,57.66,56.6,56.79,5179410,'0');
INSERT INTO daily_summary VALUES ('stock','KREF','us','2017-11-01',20.34,20.48,20.25,20.35,23421,'0');
INSERT INTO daily_summary VALUES ('stock','KRG','us','2017-11-01',18.79,18.92,18.6,18.78,924682,'0');
INSERT INTO daily_summary VALUES ('stock','KRMA','us','2017-11-01',18.63,18.63,18.55,18.57,1354,'0');
INSERT INTO daily_summary VALUES ('stock','KRNT','us','2017-11-01',15.6,15.7,15.4,15.55,193932,'0');
INSERT INTO daily_summary VALUES ('stock','KRNY','us','2017-11-01',15.2,15.3,14.9,15.05,291674,'0');
INSERT INTO daily_summary VALUES ('stock','KRO','us','2017-11-01',26.68,26.7558,25.9,26.1,246853,'0');
INSERT INTO daily_summary VALUES ('etf','KROO','us','2017-11-01',18.6093,18.6093,18.5196,18.5196,440,'0');
INSERT INTO daily_summary VALUES ('stock','KRP','us','2017-11-01',16.689,16.807,16.631,16.64,19253,'0');
INSERT INTO daily_summary VALUES ('stock','KRYS','us','2017-11-01',9.349,10,9.349,9.55,15987,'0');
INSERT INTO daily_summary VALUES ('stock','KS','us','2017-11-01',22.63,23.14,22.51,22.68,863318,'0');
INSERT INTO daily_summary VALUES ('stock','KSA','us','2017-11-01',25.169,25.17,24.93,24.935,75639,'0');
INSERT INTO daily_summary VALUES ('stock','KSM','us','2017-11-01',11.93,11.96,11.88,11.8953,82733,'0');
INSERT INTO daily_summary VALUES ('stock','KSS','us','2017-11-01',41.95,42.81,41.52,41.9,2970700,'0');
INSERT INTO daily_summary VALUES ('stock','KST','us','2017-11-01',12.48,12.49,12.48,12.49,1291,'0');
INSERT INTO daily_summary VALUES ('stock','KSU','us','2017-11-01',104.77,105.72,104.46,105.46,952870,'0');
INSERT INTO daily_summary VALUES ('stock','KSU_','us','2017-11-01',27.35,27.55,27.35,27.55,42,'0');
INSERT INTO daily_summary VALUES ('stock','KT','us','2017-11-01',14.4,14.41,13.91,14.13,1238720,'0');
INSERT INTO daily_summary VALUES ('stock','KTCC','us','2017-11-01',7.23,7.8,7.07,7.8,112870,'0');
INSERT INTO daily_summary VALUES ('stock','KTEC','us','2017-11-01',16.91,17.8117,16.8,16.8,16156,'0');
INSERT INTO daily_summary VALUES ('stock','KTF','us','2017-11-01',11.89,11.94,11.83,11.87,277606,'0');
INSERT INTO daily_summary VALUES ('stock','KTH','us','2017-11-01',32.41,32.53,32.41,32.53,1371,'0');
INSERT INTO daily_summary VALUES ('stock','KTN','us','2017-11-01',33.8,33.8,32.7304,33.73,3746,'0');
INSERT INTO daily_summary VALUES ('stock','KTOS','us','2017-11-01',12.16,12.3,11.65,11.9,1603390,'0');
INSERT INTO daily_summary VALUES ('stock','KTOV','us','2017-11-01',2.26,2.39,2.19,2.33,364906,'0');
INSERT INTO daily_summary VALUES ('stock','KTOVW','us','2017-11-01',0.7326,0.7535,0.7326,0.7535,15180,'0');
INSERT INTO daily_summary VALUES ('stock','KTP','us','2017-11-01',12.18,12.35,12,12.06,58878,'0');
INSERT INTO daily_summary VALUES ('stock','KTWO','us','2017-11-01',19.84,19.86,18.94,19.09,636879,'0');
INSERT INTO daily_summary VALUES ('stock','KURA','us','2017-11-01',15.45,15.5,14.45,15,285141,'0');
INSERT INTO daily_summary VALUES ('stock','KVHI','us','2017-11-01',11.65,11.7999,10.7,11,26911,'0');
INSERT INTO daily_summary VALUES ('stock','KW','us','2017-11-01',19.65,19.7,19.4,19.55,1187780,'0');
INSERT INTO daily_summary VALUES ('stock','KWEB','us','2017-11-01',57.23,57.46,56.86,57.05,634558,'0');
INSERT INTO daily_summary VALUES ('stock','KWR','us','2017-11-01',156.46,158.15,154.508,155.55,58553,'0');
INSERT INTO daily_summary VALUES ('etf','KXI','us','2017-11-01',101.68,101.86,101.46,101.49,24055,'0');
INSERT INTO daily_summary VALUES ('stock','KYE','us','2017-11-01',10.42,10.56,10.34,10.36,157514,'0');
INSERT INTO daily_summary VALUES ('stock','KYN','us','2017-11-01',15.88,16.03,15.76,15.86,1096460,'0');
INSERT INTO daily_summary VALUES ('stock','KYN_F','us','2017-11-01',25.43,25.4705,25.43,25.4633,1904,'0');
INSERT INTO daily_summary VALUES ('stock','KYO','us','2017-11-01',68,69.4048,67.51,69.39,27348,'0');
INSERT INTO daily_summary VALUES ('stock','L','us','2017-11-01',49.6,49.75,49.42,49.54,2417190,'0');
INSERT INTO daily_summary VALUES ('etf','LABD','us','2017-11-01',4.91,5.33,4.841,5.26,6054800,'0');
INSERT INTO daily_summary VALUES ('stock','LABL','us','2017-11-01',83.05,83.05,80.975,82,36929,'0');
INSERT INTO daily_summary VALUES ('etf','LABU','us','2017-11-01',78.47,79.51,72.1,73.4,1336460,'0');
INSERT INTO daily_summary VALUES ('stock','LAD','us','2017-11-01',113.59,114.25,111.71,113.68,216129,'0');
INSERT INTO daily_summary VALUES ('stock','LADR','us','2017-11-01',13.54,13.62,13.44,13.57,421206,'0');
INSERT INTO daily_summary VALUES ('stock','LAKE','us','2017-11-01',15.95,15.95,14.95,15.1,97152,'0');
INSERT INTO daily_summary VALUES ('etf','LALT','us','2017-11-01',21.23,21.23,21.2299,21.23,607,'0');
INSERT INTO daily_summary VALUES ('stock','LAMR','us','2017-11-01',70.59,71.1,68.98,69.24,772435,'0');
INSERT INTO daily_summary VALUES ('stock','LANC','us','2017-11-01',126.4,126.79,124.91,126.37,126403,'0');
INSERT INTO daily_summary VALUES ('stock','LAND','us','2017-11-01',13.21,13.2718,13.02,13.1,36287,'0');
INSERT INTO daily_summary VALUES ('stock','LANDP','us','2017-11-01',25.9,25.93,25.83,25.93,2445,'0');
INSERT INTO daily_summary VALUES ('stock','LAQ','us','2017-11-01',27.14,27.14,26.6799,26.6799,13521,'0');
INSERT INTO daily_summary VALUES ('stock','LARE','us','2017-11-01',30.0358,30.0358,30.0358,30.0358,277,'0');
INSERT INTO daily_summary VALUES ('stock','LARK','us','2017-11-01',30.24,30.24,30.24,30.24,313,'0');
INSERT INTO daily_summary VALUES ('stock','LAUR','us','2017-11-01',13.46,13.48,13.06,13.27,260065,'0');
INSERT INTO daily_summary VALUES ('stock','LAWS','us','2017-11-01',25.2,25.2,24.7,25,4331,'0');
INSERT INTO daily_summary VALUES ('stock','LAYN','us','2017-11-01',13.38,13.45,12.62,12.89,97703,'0');
INSERT INTO daily_summary VALUES ('stock','LAZ','us','2017-11-01',47.352,47.57,46.956,46.985,491455,'0');
INSERT INTO daily_summary VALUES ('stock','LB','us','2017-11-01',43.46,44.39,42.535,43.61,5351130,'0');
INSERT INTO daily_summary VALUES ('stock','LBAI','us','2017-11-01',20.65,20.65,19.854,20.053,68096,'0');
INSERT INTO daily_summary VALUES ('stock','LBIX','us','2017-11-01',1.49,2.52,1.47,2.03,13993100,'0');
INSERT INTO daily_summary VALUES ('etf','LBJ','us','2017-11-01',31.82,32.56,31.06,31.06,14357,'0');
INSERT INTO daily_summary VALUES ('stock','LBRDA','us','2017-11-01',87.02,87.6699,86.7,87.21,151015,'0');
INSERT INTO daily_summary VALUES ('stock','LBRDK','us','2017-11-01',87.49,88.71,87.46,87.98,951952,'0');
INSERT INTO daily_summary VALUES ('stock','LBTYA','us','2017-11-01',30.98,31.57,30.81,31.34,1867400,'0');
INSERT INTO daily_summary VALUES ('stock','LBTYB','us','2017-11-01',31.1,31.1,31,31,321,'0');
INSERT INTO daily_summary VALUES ('stock','LBTYK','us','2017-11-01',29.94,30.58,29.91,30.29,2255200,'0');
INSERT INTO daily_summary VALUES ('stock','LBY','us','2017-11-01',6.8339,7.346,6.7945,7.0801,515062,'0');
INSERT INTO daily_summary VALUES ('stock','LC','us','2017-11-01',5.78,5.85,5.57,5.64,4372220,'0');
INSERT INTO daily_summary VALUES ('stock','LCA','us','2017-11-01',10.03,10.1,10.01,10.01,33400,'0');
INSERT INTO daily_summary VALUES ('stock','LCAHU','us','2017-11-01',10.75,10.75,10.75,10.75,100,'0');
INSERT INTO daily_summary VALUES ('stock','LCAHW','us','2017-11-01',0.7,0.7,0.7,0.7,10000,'0');
INSERT INTO daily_summary VALUES ('stock','LCI','us','2017-11-01',20.05,21.7,19.725,21.55,1042100,'0');
INSERT INTO daily_summary VALUES ('stock','LCII','us','2017-11-01',125,125,122.45,122.8,133876,'0');
INSERT INTO daily_summary VALUES ('stock','LCM','us','2017-11-01',8.68,8.69,8.64,8.6499,63605,'0');
INSERT INTO daily_summary VALUES ('stock','LCNB','us','2017-11-01',20.6,20.6,20.05,20.55,6733,'0');
INSERT INTO daily_summary VALUES ('stock','LCUT','us','2017-11-01',18.9,18.9,17.625,18.2,13866,'0');
INSERT INTO daily_summary VALUES ('stock','LDF','us','2017-11-01',11.61,11.65,11.45,11.541,8480,'0');
INSERT INTO daily_summary VALUES ('stock','LDL','us','2017-11-01',56.5,58.7,52.4,54.7,168823,'0');
INSERT INTO daily_summary VALUES ('stock','LDOS','us','2017-11-01',62.84,62.97,62.37,62.83,818804,'0');
INSERT INTO daily_summary VALUES ('stock','LDP','us','2017-11-01',26.57,26.63,26.42,26.5,90015,'0');
INSERT INTO daily_summary VALUES ('etf','LDRI','us','2017-11-01',24.9556,24.9556,24.92,24.92,1111,'0');
INSERT INTO daily_summary VALUES ('etf','LDUR','us','2017-11-01',100.81,101.07,100.81,101,5129,'0');
INSERT INTO daily_summary VALUES ('stock','LE','us','2017-11-01',11,11.25,10.85,11.05,223789,'0');
INSERT INTO daily_summary VALUES ('stock','LEA','us','2017-11-01',177.5,178.61,176.055,177.15,596894,'0');
INSERT INTO daily_summary VALUES ('stock','LEAD','us','2017-11-01',30,30.16,29.9801,30.0767,27131,'0');
INSERT INTO daily_summary VALUES ('stock','LECO','us','2017-11-01',92.51,93.41,90.99,91.03,399409,'0');
INSERT INTO daily_summary VALUES ('stock','LEDS','us','2017-11-01',2.19,2.23,2.15,2.15,2233,'0');
INSERT INTO daily_summary VALUES ('stock','LEE','us','2017-11-01',2.25,2.3146,2.2,2.25,106640,'0');
INSERT INTO daily_summary VALUES ('stock','LEG','us','2017-11-01',47.15,47.53,46,46.5,1717090,'0');
INSERT INTO daily_summary VALUES ('stock','LEJU','us','2017-11-01',1.46,1.49,1.4,1.45,24041,'0');
INSERT INTO daily_summary VALUES ('etf','LEMB','us','2017-11-01',46.9,47.03,46.86,46.93,89409,'0');
INSERT INTO daily_summary VALUES ('stock','LEN','us','2017-11-01',56.09,56.98,55.46,56.96,6869410,'0');
INSERT INTO daily_summary VALUES ('stock','LENS','us','2017-11-01',3.6,3.6,3.4,3.417,6183,'0');
INSERT INTO daily_summary VALUES ('stock','LEO','us','2017-11-01',8.7002,8.7002,8.6006,8.6006,230215,'0');
INSERT INTO daily_summary VALUES ('stock','LEU','us','2017-11-01',3.96,3.96,3.8,3.8501,9391,'0');
INSERT INTO daily_summary VALUES ('stock','LEXEA','us','2017-11-01',46.45,46.81,45.522,45.54,536123,'0');
INSERT INTO daily_summary VALUES ('stock','LFC','us','2017-11-01',17.75,17.85,17.56,17.62,613332,'0');
INSERT INTO daily_summary VALUES ('stock','LFGR','us','2017-11-01',7.1,7.25,7.05,7.15,7961,'0');
INSERT INTO daily_summary VALUES ('stock','LFUS','us','2017-11-01',194,195.175,182.95,188.44,778663,'0');
INSERT INTO daily_summary VALUES ('stock','LFVN','us','2017-11-01',5.64,5.791,5.5,5.73,32532,'0');
INSERT INTO daily_summary VALUES ('stock','LGCY','us','2017-11-01',1.67,1.71,1.58,1.61,661087,'0');
INSERT INTO daily_summary VALUES ('stock','LGCYO','us','2017-11-01',5.23,5.5,5.23,5.48,57965,'0');
INSERT INTO daily_summary VALUES ('stock','LGCYP','us','2017-11-01',5.05,5.69,5.05,5.57,41337,'0');
INSERT INTO daily_summary VALUES ('stock','LGI','us','2017-11-01',17.04,17.139,16.95,17.07,21747,'0');
INSERT INTO daily_summary VALUES ('stock','LGIH','us','2017-11-01',60.95,61.41,60.38,61.36,241978,'0');
INSERT INTO daily_summary VALUES ('stock','LGL','us','2017-11-01',5.8718,6.05,5.8718,5.95,10517,'0');
INSERT INTO daily_summary VALUES ('stock','LGLR','us','2017-11-01',0.12,0.125,0.0999,0.1093,47481,'0');
INSERT INTO daily_summary VALUES ('etf','LGLV','us','2017-11-01',90.638,90.638,90.4295,90.48,2574,'0');
INSERT INTO daily_summary VALUES ('stock','LGND','us','2017-11-01',146.45,146.45,142.64,144.06,384907,'0');
INSERT INTO daily_summary VALUES ('stock','LH','us','2017-11-01',153.63,154.99,151.01,151.31,656099,'0');
INSERT INTO daily_summary VALUES ('stock','LHCG','us','2017-11-01',66.43,67.03,65.08,65.31,164825,'0');
INSERT INTO daily_summary VALUES ('stock','LHO','us','2017-11-01',28.45,28.55,28.1,28.24,929727,'0');
INSERT INTO daily_summary VALUES ('stock','LHO_I','us','2017-11-01',25.3,25.3,25.23,25.2602,12397,'0');
INSERT INTO daily_summary VALUES ('stock','LHO_J','us','2017-11-01',25.75,25.75,25.75,25.75,515,'0');
INSERT INTO daily_summary VALUES ('stock','LIFE','us','2017-11-01',4.85,4.9,3.85,4,154365,'0');
INSERT INTO daily_summary VALUES ('stock','LII','us','2017-11-01',192.36,192.36,189.22,189.87,276246,'0');
INSERT INTO daily_summary VALUES ('stock','LILA','us','2017-11-01',22.1,23.05,21.55,22.61,1130120,'0');
INSERT INTO daily_summary VALUES ('stock','LILAK','us','2017-11-01',22.15,23.1896,21.86,22.73,1755170,'0');
INSERT INTO daily_summary VALUES ('stock','LINC','us','2017-11-01',2.21,2.21,2.2,2.2,4219,'0');
INSERT INTO daily_summary VALUES ('stock','LIND','us','2017-11-01',10.82,10.82,10.62,10.74,59987,'0');
INSERT INTO daily_summary VALUES ('stock','LINDW','us','2017-11-01',1.75,1.7535,1.51,1.55,7991,'0');
INSERT INTO daily_summary VALUES ('stock','LINK','us','2017-11-01',7.124,7.124,7.05,7.05,926,'0');
INSERT INTO daily_summary VALUES ('stock','LINU','us','2017-11-01',1.62,1.93,1.54,1.83,1127320,'0');
INSERT INTO daily_summary VALUES ('stock','LION','us','2017-11-01',22.08,22.21,21.39,21.5,266199,'0');
INSERT INTO daily_summary VALUES ('stock','LIQT','us','2017-11-01',0.31,0.345,0.3,0.321,202318,'0');
INSERT INTO daily_summary VALUES ('etf','LIT','us','2017-11-01',39.93,40.0999,39.73,39.86,1345170,'0');
INSERT INTO daily_summary VALUES ('stock','LITB','us','2017-11-01',1.71,1.9601,1.71,1.8,41525,'0');
INSERT INTO daily_summary VALUES ('stock','LITE','us','2017-11-01',62.5,66.7,60.15,63.2,6257710,'0');
INSERT INTO daily_summary VALUES ('stock','LIVE','us','2017-11-01',12.85,13.21,12.85,13.2087,5876,'0');
INSERT INTO daily_summary VALUES ('stock','LIVN','us','2017-11-01',74.16,74.44,72.88,73.18,674046,'0');
INSERT INTO daily_summary VALUES ('stock','LJPC','us','2017-11-01',34.79,34.79,32,33.16,252833,'0');
INSERT INTO daily_summary VALUES ('stock','LKFN','us','2017-11-01',48.66,49.095,47.51,47.85,48670,'0');
INSERT INTO daily_summary VALUES ('stock','LKOR','us','2017-11-01',53.805,54.06,53.805,53.97,4200,'0');
INSERT INTO daily_summary VALUES ('stock','LKQ','us','2017-11-01',37.88,38.16,37.27,37.42,4668800,'0');
INSERT INTO daily_summary VALUES ('stock','LKSD','us','2017-11-01',16.14,16.16,15.26,15.73,287144,'0');
INSERT INTO daily_summary VALUES ('stock','LL','us','2017-11-01',31.01,31.75,29.2135,31.38,2726810,'0');
INSERT INTO daily_summary VALUES ('stock','LLEX','us','2017-11-01',5.05,5.15,4.76,4.93,123410,'0');
INSERT INTO daily_summary VALUES ('stock','LLIT','us','2017-11-01',1.3962,1.3962,1.3962,1.3962,1506,'0');
INSERT INTO daily_summary VALUES ('stock','LLL','us','2017-11-01',187.8,188.17,185.28,185.55,485973,'0');
INSERT INTO daily_summary VALUES ('stock','LLNW','us','2017-11-01',5,5.32,4.91,5.28,1642740,'0');
INSERT INTO daily_summary VALUES ('stock','LLQD','us','2017-11-01',49.7927,49.7927,49.7927,49.7927,1,'0');
INSERT INTO daily_summary VALUES ('etf','LLSP','us','2017-11-01',34.44,34.44,34.44,34.44,101,'0');
INSERT INTO daily_summary VALUES ('stock','LLY','us','2017-11-01',82.17,83.82,82,82.89,3309260,'0');
INSERT INTO daily_summary VALUES ('stock','LM','us','2017-11-01',38.49,38.96,38.2,38.52,1584560,'0');
INSERT INTO daily_summary VALUES ('stock','LMAT','us','2017-11-01',32.31,32.31,31.56,31.98,254350,'0');
INSERT INTO daily_summary VALUES ('stock','LMB','us','2017-11-01',13.584,14.159,13.584,14.1,12848,'0');
INSERT INTO daily_summary VALUES ('etf','LMBS','us','2017-11-01',51.83,51.87,51.81,51.86,105321,'0');
INSERT INTO daily_summary VALUES ('stock','LMFA','us','2017-11-01',2.0412,2.27,1.9786,2.05,32459,'0');
INSERT INTO daily_summary VALUES ('stock','LMHA','us','2017-11-01',26.87,26.87,26.72,26.738,12915,'0');
INSERT INTO daily_summary VALUES ('stock','LMHB','us','2017-11-01',25.15,25.19,25.1113,25.14,22096,'0');
INSERT INTO daily_summary VALUES ('etf','LMLP','us','2017-11-01',15.95,16.121,15.95,15.968,6338,'0');
INSERT INTO daily_summary VALUES ('stock','LMNR','us','2017-11-01',23.23,23.49,22.29,22.49,17925,'0');
INSERT INTO daily_summary VALUES ('stock','LMNX','us','2017-11-01',21.53,21.62,21.12,21.48,282497,'0');
INSERT INTO daily_summary VALUES ('stock','LMOS','us','2017-11-01',17.97,17.98,17.95,17.97,180386,'0');
INSERT INTO daily_summary VALUES ('stock','LMRK','us','2017-11-01',17.6,17.95,17.52,17.95,25811,'0');
INSERT INTO daily_summary VALUES ('stock','LMRKO','us','2017-11-01',25.3,25.3,25.207,25.263,5582,'0');
INSERT INTO daily_summary VALUES ('stock','LMRKP','us','2017-11-01',25.7472,25.7472,25.6701,25.6701,1400,'0');
INSERT INTO daily_summary VALUES ('stock','LMT','us','2017-11-01',309.05,309.66,306.23,306.6,1016080,'0');
INSERT INTO daily_summary VALUES ('stock','LN','us','2017-11-01',42.1,42.21,41.65,42.08,137375,'0');
INSERT INTO daily_summary VALUES ('stock','LNC','us','2017-11-01',76.02,76.75,74.66,75.56,1997890,'0');
INSERT INTO daily_summary VALUES ('stock','LNCE','us','2017-11-01',37.89,37.96,37.25,37.65,278421,'0');
INSERT INTO daily_summary VALUES ('stock','LND','us','2017-11-01',3.7742,3.7808,3.7742,3.7808,800,'0');
INSERT INTO daily_summary VALUES ('stock','LNDC','us','2017-11-01',13.4,13.5,13,13.15,29951,'0');
INSERT INTO daily_summary VALUES ('stock','LNG','us','2017-11-01',47.18,48.271,47.01,47.55,2055980,'0');
INSERT INTO daily_summary VALUES ('stock','LNGR','us','2017-11-01',19.41,19.66,19.2396,19.5921,8117,'0');
INSERT INTO daily_summary VALUES ('stock','LNN','us','2017-11-01',92.21,92.65,91.2,91.64,40337,'0');
INSERT INTO daily_summary VALUES ('stock','LNT','us','2017-11-01',43.24,43.35,42.9,43.07,1454700,'0');
INSERT INTO daily_summary VALUES ('stock','LNTH','us','2017-11-01',20.05,20.15,18.6,18.7,622646,'0');
INSERT INTO daily_summary VALUES ('stock','LOAN','us','2017-11-01',6.05,6.1,6.05,6.1,26121,'0');
INSERT INTO daily_summary VALUES ('stock','LOB','us','2017-11-01',23.819,24.119,23.27,23.47,64677,'0');
INSERT INTO daily_summary VALUES ('stock','LOCO','us','2017-11-01',11.55,11.65,11.3,11.35,276241,'0');
INSERT INTO daily_summary VALUES ('stock','LODE','us','2017-11-01',0.57,0.59,0.525,0.533,618846,'0');
INSERT INTO daily_summary VALUES ('stock','LOGI','us','2017-11-01',36.35,36.5,35.85,35.98,566116,'0');
INSERT INTO daily_summary VALUES ('stock','LOGM','us','2017-11-01',121.26,124.2,120.71,123.45,796021,'0');
INSERT INTO daily_summary VALUES ('stock','LOGO','us','2017-11-01',14.95,14.95,14.95,14.95,1102,'0');
INSERT INTO daily_summary VALUES ('stock','LOMA','us','2017-11-01',21.65,22.15,21.25,21.35,18872200,'0');
INSERT INTO daily_summary VALUES ('stock','LONE','us','2017-11-01',3.54,3.7899,3.48,3.58,72374,'0');
INSERT INTO daily_summary VALUES ('stock','LOPE','us','2017-11-01',89.96,89.96,83.52,85.69,586175,'0');
INSERT INTO daily_summary VALUES ('stock','LOR','us','2017-11-01',11.219,11.219,11.12,11.13,102407,'0');
INSERT INTO daily_summary VALUES ('stock','LORL','us','2017-11-01',47.5,47.5,46.3,46.8,43464,'0');
INSERT INTO daily_summary VALUES ('stock','LOV','us','2017-11-01',1.12,1.23,1.05,1.2,425310,'0');
INSERT INTO daily_summary VALUES ('stock','LOW','us','2017-11-01',79.62,80.5,79.42,79.92,4604570,'0');
INSERT INTO daily_summary VALUES ('etf','LOWC','us','2017-11-01',88.24,88.6,88.24,88.39,1286,'0');
INSERT INTO daily_summary VALUES ('stock','LOXO','us','2017-11-01',87.04,87.92,83.1,84.73,293590,'0');
INSERT INTO daily_summary VALUES ('stock','LPCN','us','2017-11-01',3.53,3.59,3.44,3.5,107257,'0');
INSERT INTO daily_summary VALUES ('stock','LPG','us','2017-11-01',7.25,7.62,7.13,7.15,205521,'0');
INSERT INTO daily_summary VALUES ('stock','LPI','us','2017-11-01',12.04,12.5,11.84,12.18,3973570,'0');
INSERT INTO daily_summary VALUES ('stock','LPL','us','2017-11-01',13.12,13.17,13.04,13.13,816024,'0');
INSERT INTO daily_summary VALUES ('stock','LPLA','us','2017-11-01',49.468,50.453,49.1,49.14,1197750,'0');
INSERT INTO daily_summary VALUES ('stock','LPNT','us','2017-11-01',48.2,48.4,46.575,46.65,762801,'0');
INSERT INTO daily_summary VALUES ('stock','LPSN','us','2017-11-01',14.05,14.125,13.5,13.95,373599,'0');
INSERT INTO daily_summary VALUES ('stock','LPT','us','2017-11-01',42.98,43.66,42.72,43.42,806286,'0');
INSERT INTO daily_summary VALUES ('stock','LPTH','us','2017-11-01',4.05,4.05,3.64,3.75,975011,'0');
INSERT INTO daily_summary VALUES ('stock','LPTX','us','2017-11-01',5.94,5.9594,5.5,5.6564,23236,'0');
INSERT INTO daily_summary VALUES ('stock','LPX','us','2017-11-01',27.42,27.69,27.11,27.12,1368660,'0');
INSERT INTO daily_summary VALUES ('stock','LQ','us','2017-11-01',17.69,17.86,17.57,17.74,2045860,'0');
INSERT INTO daily_summary VALUES ('etf','LQD','us','2017-11-01',121.05,121.15,120.85,121.01,5967420,'0');
INSERT INTO daily_summary VALUES ('etf','LQDH','us','2017-11-01',96.52,96.52,96.45,96.512,862,'0');
INSERT INTO daily_summary VALUES ('stock','LQDT','us','2017-11-01',5.7,5.75,5.575,5.65,113871,'0');
INSERT INTO daily_summary VALUES ('stock','LRAD','us','2017-11-01',2.14,2.169,2.06,2.13,29937,'0');
INSERT INTO daily_summary VALUES ('stock','LRCX','us','2017-11-01',210.45,210.45,200.544,202.92,2806360,'0');
INSERT INTO daily_summary VALUES ('stock','LRET','us','2017-11-01',25.464,25.464,25.464,25.464,1098,'0');
INSERT INTO daily_summary VALUES ('etf','LRGF','us','2017-11-01',30.94,30.99,30.7834,30.85,88460,'0');
INSERT INTO daily_summary VALUES ('stock','LRN','us','2017-11-01',16.45,16.63,15.94,16.12,193701,'0');
INSERT INTO daily_summary VALUES ('stock','LSBK','us','2017-11-01',15.76,15.806,15.72,15.72,3458,'0');
INSERT INTO daily_summary VALUES ('stock','LSCC','us','2017-11-01',5.9,5.92,5.74,5.79,860736,'0');
INSERT INTO daily_summary VALUES ('stock','LSI','us','2017-11-01',80.87,81.49,80.32,80.94,482751,'0');
INSERT INTO daily_summary VALUES ('etf','LSTK','us','2017-11-01',47.53,47.5335,47.53,47.5335,1259,'0');
INSERT INTO daily_summary VALUES ('stock','LSTR','us','2017-11-01',99.497,99.7,98.149,98.648,152005,'0');
INSERT INTO daily_summary VALUES ('stock','LSXMA','us','2017-11-01',42.06,42.205,41.38,41.6,295922,'0');
INSERT INTO daily_summary VALUES ('stock','LSXMK','us','2017-11-01',41.84,42.2,41.39,41.63,545317,'0');
INSERT INTO daily_summary VALUES ('stock','LTBR','us','2017-11-01',1.06,1.09,1.03,1.06,379145,'0');
INSERT INTO daily_summary VALUES ('stock','LTC','us','2017-11-01',46.55,46.65,46.06,46.54,176282,'0');
INSERT INTO daily_summary VALUES ('stock','LTEA','us','2017-11-01',2.45,2.54,2.43,2.46,13982,'0');
INSERT INTO daily_summary VALUES ('etf','LTL','us','2017-11-01',45.12,45.12,45.0028,45.0028,290,'0');
INSERT INTO daily_summary VALUES ('stock','LTM','us','2017-11-01',13.71,13.73,13.42,13.44,290066,'0');
INSERT INTO daily_summary VALUES ('etf','LTPZ','us','2017-11-01',66.81,67.039,66.76,66.76,6019,'0');
INSERT INTO daily_summary VALUES ('stock','LTRPA','us','2017-11-01',10.9,11,10.75,10.75,830566,'0');
INSERT INTO daily_summary VALUES ('stock','LTRX','us','2017-11-01',1.9,2.14,1.85,1.95,161924,'0');
INSERT INTO daily_summary VALUES ('stock','LTS','us','2017-11-01',3.08,3.08,2.99,3.02,136639,'0');
INSERT INTO daily_summary VALUES ('stock','LTS_A','us','2017-11-01',24.82,24.9,24.814,24.8767,19605,'0');
INSERT INTO daily_summary VALUES ('stock','LTXB','us','2017-11-01',40.179,40.448,38.894,39.303,156410,'0');
INSERT INTO daily_summary VALUES ('stock','LUB','us','2017-11-01',2.41,2.43,2.41,2.41,17566,'0');
INSERT INTO daily_summary VALUES ('stock','LUK','us','2017-11-01',25.5,25.75,25.17,25.33,1054260,'0');
INSERT INTO daily_summary VALUES ('stock','LULU','us','2017-11-01',62.21,62.88,60.81,61.49,1193910,'0');
INSERT INTO daily_summary VALUES ('stock','LUNA','us','2017-11-01',1.64,1.65,1.571,1.61,35866,'0');
INSERT INTO daily_summary VALUES ('stock','LUV','us','2017-11-01',54.03,54.46,53.54,53.99,4230090,'0');
INSERT INTO daily_summary VALUES ('stock','LVHB','us','2017-11-01',32.05,32.0799,31.7001,31.71,1567240,'0');
INSERT INTO daily_summary VALUES ('stock','LVHD','us','2017-11-01',30.7,30.7128,30.59,30.62,36499,'0');
INSERT INTO daily_summary VALUES ('stock','LVHE','us','2017-11-01',27.79,27.79,27.44,27.44,555,'0');
INSERT INTO daily_summary VALUES ('stock','LVHI','us','2017-11-01',28.27,28.29,28.2405,28.29,6784,'0');
INSERT INTO daily_summary VALUES ('stock','LVIN','us','2017-11-01',26.95,26.95,26.95,26.95,0,'0');
INSERT INTO daily_summary VALUES ('etf','LVL','us','2017-11-01',10.98,10.985,10.955,10.96,4536,'0');
INSERT INTO daily_summary VALUES ('stock','LVNTA','us','2017-11-01',57.14,57.56,56.09,56.55,569770,'0');
INSERT INTO daily_summary VALUES ('stock','LVS','us','2017-11-01',65,65,64.165,64.37,4626380,'0');
INSERT INTO daily_summary VALUES ('stock','LVUS','us','2017-11-01',25.21,25.21,25.21,25.21,0,'0');
INSERT INTO daily_summary VALUES ('stock','LW','us','2017-11-01',50.831,51.778,50.831,51.16,652963,'0');
INSERT INTO daily_summary VALUES ('stock','LWAY','us','2017-11-01',10.05,10.05,9.6851,9.7,4910,'0');
INSERT INTO daily_summary VALUES ('stock','LXFR','us','2017-11-01',12.51,12.67,12.17,12.43,23272,'0');
INSERT INTO daily_summary VALUES ('stock','LXFT','us','2017-11-01',46.55,46.63,45.3,45.7,121222,'0');
INSERT INTO daily_summary VALUES ('stock','LXP','us','2017-11-01',10.15,10.21,10.08,10.08,695110,'0');
INSERT INTO daily_summary VALUES ('stock','LXRX','us','2017-11-01',10.19,10.24,9.89,9.94,816445,'0');
INSERT INTO daily_summary VALUES ('stock','LXU','us','2017-11-01',7.59,7.85,7.28,7.54,270127,'0');
INSERT INTO daily_summary VALUES ('stock','LYB','us','2017-11-01',104.21,105.22,103.155,103.43,3111630,'0');
INSERT INTO daily_summary VALUES ('stock','LYG','us','2017-11-01',3.71,3.73,3.69,3.7,2240990,'0');
INSERT INTO daily_summary VALUES ('stock','LYL','us','2017-11-01',6.88,6.99,5.1555,5.44,204680,'0');
INSERT INTO daily_summary VALUES ('stock','LYTS','us','2017-11-01',6.9294,6.9294,6.5125,6.6415,66486,'0');
INSERT INTO daily_summary VALUES ('stock','LYV','us','2017-11-01',43.8,44.01,42.3,42.47,1588520,'0');
INSERT INTO daily_summary VALUES ('stock','LZB','us','2017-11-01',27.2,27.35,26.95,27.15,448374,'0');
INSERT INTO daily_summary VALUES ('stock','M','us','2017-11-01',18.85,19.62,18.7548,18.95,14932200,'0');
INSERT INTO daily_summary VALUES ('stock','MA','us','2017-11-01',149.95,151.49,148.78,148.89,4529760,'0');
INSERT INTO daily_summary VALUES ('stock','MAA','us','2017-11-01',102.73,103.39,102.05,103.11,563848,'0');
INSERT INTO daily_summary VALUES ('stock','MAB','us','2017-11-01',13.52,13.53,13.48,13.496,1646,'0');
INSERT INTO daily_summary VALUES ('stock','MAC','us','2017-11-01',53.903,54.575,53.834,54.279,2426370,'0');
INSERT INTO daily_summary VALUES ('stock','MACK','us','2017-11-01',11.75,11.88,11.14,11.24,171256,'0');
INSERT INTO daily_summary VALUES ('stock','MACQU','us','2017-11-01',10.47,10.52,10.47,10.52,200,'0');
INSERT INTO daily_summary VALUES ('stock','MACQW','us','2017-11-01',0.35,0.35,0.3499,0.35,4000,'0');
INSERT INTO daily_summary VALUES ('stock','MAG','us','2017-11-01',10.78,11.07,10.74,11.07,608818,'0');
INSERT INTO daily_summary VALUES ('stock','MAGA','us','2017-11-01',26.33,26.47,26.33,26.41,7795,'0');
INSERT INTO daily_summary VALUES ('stock','MAGS','us','2017-11-01',4.3,4.4,4.2587,4.3,42474,'0');
INSERT INTO daily_summary VALUES ('stock','MAIN','us','2017-11-01',40.25,40.2534,40.02,40.08,109870,'0');
INSERT INTO daily_summary VALUES ('stock','MAMS','us','2017-11-01',8.95,8.95,8.03,8.3899,13161,'0');
INSERT INTO daily_summary VALUES ('stock','MAN','us','2017-11-01',124.09,124.09,122.54,123.84,360543,'0');
INSERT INTO daily_summary VALUES ('stock','MANH','us','2017-11-01',41.75,42.77,41.14,41.39,1036130,'0');
INSERT INTO daily_summary VALUES ('stock','MANT','us','2017-11-01',46.9,47.47,45.3252,45.88,128394,'0');
INSERT INTO daily_summary VALUES ('stock','MANU','us','2017-11-01',18.05,18.1,17.9,18.05,21780,'0');
INSERT INTO daily_summary VALUES ('stock','MAR','us','2017-11-01',120.51,120.92,119.6,120.16,1381030,'0');
INSERT INTO daily_summary VALUES ('stock','MARA','us','2017-11-01',1.53,1.99,1.5,1.84,765184,'0');
INSERT INTO daily_summary VALUES ('stock','MARK','us','2017-11-01',3.71,3.74,3.38,3.45,433041,'0');
INSERT INTO daily_summary VALUES ('stock','MAS','us','2017-11-01',40.02,40.02,39.11,39.37,1985310,'0');
INSERT INTO daily_summary VALUES ('stock','MASI','us','2017-11-01',89.51,91.04,86.07,87.93,971060,'0');
INSERT INTO daily_summary VALUES ('stock','MAT','us','2017-11-01',14.18,14.34,13.54,13.55,14145200,'0');
INSERT INTO daily_summary VALUES ('stock','MATF','us','2017-11-01',33.585,33.64,33.585,33.64,200,'0');
INSERT INTO daily_summary VALUES ('etf','MATH','us','2017-11-01',32.73,32.73,32.3501,32.3501,2005,'0');
INSERT INTO daily_summary VALUES ('stock','MATR','us','2017-11-01',2.65,2.65,2.45,2.575,47481,'0');
INSERT INTO daily_summary VALUES ('stock','MATW','us','2017-11-01',63.5,63.5,61.55,62.35,65326,'0');
INSERT INTO daily_summary VALUES ('stock','MATX','us','2017-11-01',27.357,27.357,26.523,26.87,185110,'0');
INSERT INTO daily_summary VALUES ('stock','MAV','us','2017-11-01',11.41,11.41,11.33,11.35,94057,'0');
INSERT INTO daily_summary VALUES ('stock','MAXR','us','2017-11-01',63.5,63.5,62.305,62.55,52581,'0');
INSERT INTO daily_summary VALUES ('stock','MB','us','2017-11-01',32.55,32.55,30.5,30.8,759855,'0');
INSERT INTO daily_summary VALUES ('etf','MBB','us','2017-11-01',106.61,106.76,106.58,106.73,387622,'0');
INSERT INTO daily_summary VALUES ('stock','MBCN','us','2017-11-01',46,46.5307,46,46,1891,'0');
INSERT INTO daily_summary VALUES ('stock','MBFI','us','2017-11-01',46.24,46.4,45.42,45.76,235767,'0');
INSERT INTO daily_summary VALUES ('stock','MBFIP','us','2017-11-01',25.37,25.4,25.36,25.37,5116,'0');
INSERT INTO daily_summary VALUES ('etf','MBG','us','2017-11-01',26.11,26.179,26.11,26.179,2865,'0');
INSERT INTO daily_summary VALUES ('stock','MBI','us','2017-11-01',7.24,7.63,7.16,7.28,5560730,'0');
INSERT INTO daily_summary VALUES ('stock','MBII','us','2017-11-01',1,1.0001,0.9501,1,39808,'0');
INSERT INTO daily_summary VALUES ('stock','MBIN','us','2017-11-01',17.26,17.575,17.01,17.41,51739,'0');
INSERT INTO daily_summary VALUES ('stock','MBIO','us','2017-11-01',11.87,12.43,11.0101,11.46,161609,'0');
INSERT INTO daily_summary VALUES ('stock','MBOT','us','2017-11-01',1.13,1.13,1.08,1.09,305223,'0');
INSERT INTO daily_summary VALUES ('stock','MBRX','us','2017-11-01',2.05,2.169,1.95,1.95,455482,'0');
INSERT INTO daily_summary VALUES ('etf','MBSD','us','2017-11-01',23.74,23.85,23.74,23.77,5291,'0');
INSERT INTO daily_summary VALUES ('stock','MBT','us','2017-11-01',10.68,11.45,10.68,11.04,4276390,'0');
INSERT INTO daily_summary VALUES ('stock','MBTF','us','2017-11-01',10.589,10.589,10.29,10.29,19171,'0');
INSERT INTO daily_summary VALUES ('stock','MBUU','us','2017-11-01',31.46,31.46,30.59,30.79,153491,'0');
INSERT INTO daily_summary VALUES ('stock','MBVX','us','2017-11-01',0.6339,0.65,0.61,0.6301,111936,'0');
INSERT INTO daily_summary VALUES ('stock','MBWM','us','2017-11-01',36.43,36.79,35.58,35.88,21526,'0');
INSERT INTO daily_summary VALUES ('stock','MC','us','2017-11-01',42.382,42.63,42.134,42.283,388968,'0');
INSERT INTO daily_summary VALUES ('stock','MCA','us','2017-11-01',14.93,15.01,14.9,14.9,38740,'0');
INSERT INTO daily_summary VALUES ('stock','MCBC','us','2017-11-01',10.048,10.048,9.859,9.919,24528,'0');
INSERT INTO daily_summary VALUES ('stock','MCC','us','2017-11-01',5.78,5.78,5.75,5.76,142091,'0');
INSERT INTO daily_summary VALUES ('stock','MCD','us','2017-11-01',165.72,167.99,165.57,166.37,2186050,'0');
INSERT INTO daily_summary VALUES ('stock','MCEF','us','2017-11-01',18.7225,18.7225,18.69,18.69,1372,'0');
INSERT INTO daily_summary VALUES ('stock','MCEP','us','2017-11-01',1.24,1.24,1.1478,1.19,33989,'0');
INSERT INTO daily_summary VALUES ('stock','MCF','us','2017-11-01',4.01,4.35,3.94,4.11,113836,'0');
INSERT INTO daily_summary VALUES ('stock','MCFT','us','2017-11-01',23.11,23.11,22.48,22.65,143392,'0');
INSERT INTO daily_summary VALUES ('etf','MCHI','us','2017-11-01',66.03,66.27,65.68,65.75,1257720,'0');
INSERT INTO daily_summary VALUES ('stock','MCHP','us','2017-11-01',94.3,95.14,92.52,92.99,2946490,'0');
INSERT INTO daily_summary VALUES ('stock','MCHX','us','2017-11-01',3.1,3.1025,3.05,3.06,54782,'0');
INSERT INTO daily_summary VALUES ('stock','MCI','us','2017-11-01',15.9999,15.9999,15.75,15.76,12805,'0');
INSERT INTO daily_summary VALUES ('stock','MCK','us','2017-11-01',138.78,142.93,138,139.67,2843560,'0');
INSERT INTO daily_summary VALUES ('stock','MCN','us','2017-11-01',7.77,7.77,7.665,7.7,68804,'0');
INSERT INTO daily_summary VALUES ('stock','MCO','us','2017-11-01',143.79,144.27,142.5,143.49,543771,'0');
INSERT INTO daily_summary VALUES ('stock','MCR','us','2017-11-01',8.57,8.6,8.56,8.59,127124,'0');
INSERT INTO daily_summary VALUES ('stock','MCRB','us','2017-11-01',10.07,10.27,9.64,9.68,164036,'0');
INSERT INTO daily_summary VALUES ('stock','MCRI','us','2017-11-01',44.81,45.36,43.83,45.32,86396,'0');
INSERT INTO daily_summary VALUES ('stock','MCRN','us','2017-11-01',18,18.035,17.53,17.54,406680,'0');
INSERT INTO daily_summary VALUES ('etf','MCRO','us','2017-11-01',26.4966,26.4966,26.46,26.46,905,'0');
INSERT INTO daily_summary VALUES ('stock','MCS','us','2017-11-01',27.4,27.45,26.45,26.8,81889,'0');
INSERT INTO daily_summary VALUES ('stock','MCV','us','2017-11-01',25.0819,25.24,25.0819,25.24,3329,'0');
INSERT INTO daily_summary VALUES ('stock','MCX','us','2017-11-01',25.77,25.77,25.76,25.76,925,'0');
INSERT INTO daily_summary VALUES ('stock','MCY','us','2017-11-01',56.2,56.77,55.64,56.04,113570,'0');
INSERT INTO daily_summary VALUES ('stock','MD','us','2017-11-01',44.35,44.8,40.83,41.35,3397870,'0');
INSERT INTO daily_summary VALUES ('stock','MDB','us','2017-11-01',30.37,30.57,29.7018,29.8,432016,'0');
INSERT INTO daily_summary VALUES ('stock','MDC','us','2017-11-01',36.968,37.167,36.75,37.008,808913,'0');
INSERT INTO daily_summary VALUES ('stock','MDCA','us','2017-11-01',11.7,11.85,11.2,11.4,745074,'0');
INSERT INTO daily_summary VALUES ('stock','MDCO','us','2017-11-01',28.84,29.95,28.84,29.22,1388110,'0');
INSERT INTO daily_summary VALUES ('stock','MDGL','us','2017-11-01',50,50,44.2265,47.26,102069,'0');
INSERT INTO daily_summary VALUES ('stock','MDGS','us','2017-11-01',2.24,2.24,2.1442,2.2,3661,'0');
INSERT INTO daily_summary VALUES ('etf','MDIV','us','2017-11-01',19.04,19.0499,18.96,19.01,162240,'0');
INSERT INTO daily_summary VALUES ('stock','MDLQ','us','2017-11-01',25.265,25.4,25.265,25.3583,6004,'0');
INSERT INTO daily_summary VALUES ('stock','MDLX','us','2017-11-01',24.55,24.55,24.51,24.51,425,'0');
INSERT INTO daily_summary VALUES ('stock','MDLY','us','2017-11-01',5.75,5.75,5.65,5.75,12056,'0');
INSERT INTO daily_summary VALUES ('stock','MDLZ','us','2017-11-01',41.85,41.88,41.22,41.27,7619880,'0');
INSERT INTO daily_summary VALUES ('stock','MDP','us','2017-11-01',53.7,53.7,51.6,52.2,561386,'0');
INSERT INTO daily_summary VALUES ('stock','MDR','us','2017-11-01',6.94,7.47,6.91,7.15,5544340,'0');
INSERT INTO daily_summary VALUES ('stock','MDRX','us','2017-11-01',13.53,13.53,12.73,12.74,2884730,'0');
INSERT INTO daily_summary VALUES ('stock','MDSO','us','2017-11-01',75.57,75.57,70.01,70.38,702253,'0');
INSERT INTO daily_summary VALUES ('stock','MDT','us','2017-11-01',80.42,81.16,80.01,80.31,5136500,'0');
INSERT INTO daily_summary VALUES ('stock','MDU','us','2017-11-01',27.5,27.5,26.93,27.09,881400,'0');
INSERT INTO daily_summary VALUES ('stock','MDVX','us','2017-11-01',1.06,1.0773,1,1.0396,24700,'0');
INSERT INTO daily_summary VALUES ('stock','MDWD','us','2017-11-01',5,5.05,4.95,5,63504,'0');
INSERT INTO daily_summary VALUES ('stock','MDXG','us','2017-11-01',12.68,12.8,11.65,11.65,3884280,'0');
INSERT INTO daily_summary VALUES ('etf','MDY','us','2017-11-01',335.28,335.67,331.65,332.74,792132,'0');
INSERT INTO daily_summary VALUES ('etf','MDYG','us','2017-11-01',153.4,153.4,151.45,152.01,35358,'0');
INSERT INTO daily_summary VALUES ('etf','MDYV','us','2017-11-01',100.5,100.609,99.4935,99.73,55677,'0');
INSERT INTO daily_summary VALUES ('stock','MEAR','us','2017-11-01',49.97,49.99,49.96,49.9699,22803,'0');
INSERT INTO daily_summary VALUES ('stock','MED','us','2017-11-01',62.54,63.17,61.51,61.78,88312,'0');
INSERT INTO daily_summary VALUES ('stock','MEDP','us','2017-11-01',35.5,36.52,35.5,36.01,190172,'0');
INSERT INTO daily_summary VALUES ('stock','MEET','us','2017-11-01',3.42,3.4244,3.34,3.38,344862,'0');
INSERT INTO daily_summary VALUES ('stock','MEI','us','2017-11-01',47.4,47.4,46.2,46.65,328178,'0');
INSERT INTO daily_summary VALUES ('stock','MEIP','us','2017-11-01',2.63,2.6689,2.51,2.53,139318,'0');
INSERT INTO daily_summary VALUES ('stock','MELI','us','2017-11-01',242.5,243.89,233.58,239.02,689139,'0');
INSERT INTO daily_summary VALUES ('stock','MELR','us','2017-11-01',19.1,19.1,18.9,19.1,1841,'0');
INSERT INTO daily_summary VALUES ('stock','MEN','us','2017-11-01',11.74,11.81,11.74,11.81,28882,'0');
INSERT INTO daily_summary VALUES ('stock','MEOH','us','2017-11-01',49.3,50,48.95,49.85,858479,'0');
INSERT INTO daily_summary VALUES ('stock','MERC','us','2017-11-01',14.7,14.8,14.2,14.25,302978,'0');
INSERT INTO daily_summary VALUES ('stock','MER_K','us','2017-11-01',25.88,25.9667,25.84,25.94,23520,'0');
INSERT INTO daily_summary VALUES ('stock','MER_P','us','2017-11-01',26.17,26.2251,26.0405,26.19,15138,'0');
INSERT INTO daily_summary VALUES ('stock','MESO','us','2017-11-01',5.21,5.372,5.21,5.37,38515,'0');
INSERT INTO daily_summary VALUES ('stock','MET','us','2017-11-01',53.602,53.759,53.136,53.483,4392170,'0');
INSERT INTO daily_summary VALUES ('stock','METC','us','2017-11-01',5.81,6.18,5.771,6.04,153874,'0');
INSERT INTO daily_summary VALUES ('stock','MET_A','us','2017-11-01',25.15,25.2394,25.1346,25.21,14420,'0');
INSERT INTO daily_summary VALUES ('stock','MEXX','us','2017-11-01',21.83,22.1164,21.5777,21.5777,1506,'0');
INSERT INTO daily_summary VALUES ('stock','MFA','us','2017-11-01',8.26,8.36,8.24,8.33,2161050,'0');
INSERT INTO daily_summary VALUES ('stock','MFA_B','us','2017-11-01',25.53,25.53,25.47,25.53,3714,'0');
INSERT INTO daily_summary VALUES ('stock','MFC','us','2017-11-01',20.22,20.465,20.19,20.42,2779840,'0');
INSERT INTO daily_summary VALUES ('stock','MFCB','us','2017-11-01',9.06,9.2754,9.06,9.21,6145,'0');
INSERT INTO daily_summary VALUES ('etf','MFD','us','2017-11-01',13.16,13.25,13.07,13.0789,23795,'0');
INSERT INTO daily_summary VALUES ('stock','MFDX','us','2017-11-01',26.32,26.32,26.259,26.305,1537,'0');
INSERT INTO daily_summary VALUES ('stock','MFEM','us','2017-11-01',25.495,25.535,25.37,25.37,56398,'0');
INSERT INTO daily_summary VALUES ('stock','MFG','us','2017-11-01',3.68,3.68,3.65,3.68,133348,'0');
INSERT INTO daily_summary VALUES ('stock','MFGP','us','2017-11-01',35.305,35.55,35.25,35.38,1080760,'0');
INSERT INTO daily_summary VALUES ('stock','MFIN','us','2017-11-01',2.16,2.207,2.16,2.18,45495,'0');
INSERT INTO daily_summary VALUES ('stock','MFINL','us','2017-11-01',23.99,24.03,23.6015,24.03,3358,'0');
INSERT INTO daily_summary VALUES ('stock','MFL','us','2017-11-01',15.02,15.02,14.93,14.98,73506,'0');
INSERT INTO daily_summary VALUES ('stock','MFM','us','2017-11-01',6.97,6.99,6.93,6.99,99690,'0');
INSERT INTO daily_summary VALUES ('stock','MFNC','us','2017-11-01',15.51,15.51,15.12,15.35,5790,'0');
INSERT INTO daily_summary VALUES ('stock','MFO','us','2017-11-01',25.6801,25.6858,25.66,25.6858,1500,'0');
INSERT INTO daily_summary VALUES ('stock','MFSF','us','2017-11-01',38.95,39.5,38.65,38.8,5028,'0');
INSERT INTO daily_summary VALUES ('stock','MFT','us','2017-11-01',14.13,14.16,14.09,14.135,4614,'0');
INSERT INTO daily_summary VALUES ('stock','MFUS','us','2017-11-01',26.26,26.32,26.26,26.32,673,'0');
INSERT INTO daily_summary VALUES ('stock','MFV','us','2017-11-01',6.61,6.65,6.61,6.63,9892,'0');
INSERT INTO daily_summary VALUES ('stock','MG','us','2017-11-01',21.15,21.25,20.83,21.19,72971,'0');
INSERT INTO daily_summary VALUES ('stock','MGA','us','2017-11-01',54.98,55.23,54.59,55.13,1064270,'0');
INSERT INTO daily_summary VALUES ('etf','MGC','us','2017-11-01',88.76,88.852,88.49,88.61,23819,'0');
INSERT INTO daily_summary VALUES ('stock','MGCD','us','2017-11-01',8.6,8.6,8.59,8.59,480,'0');
INSERT INTO daily_summary VALUES ('stock','MGEE','us','2017-11-01',66.55,66.55,64.9,65.45,49246,'0');
INSERT INTO daily_summary VALUES ('stock','MGEN','us','2017-11-01',8.21,8.52,8.21,8.38,40741,'0');
INSERT INTO daily_summary VALUES ('stock','MGF','us','2017-11-01',4.92,4.95,4.915,4.95,43563,'0');
INSERT INTO daily_summary VALUES ('stock','MGI','us','2017-11-01',15.56,15.63,15.35,15.48,203663,'0');
INSERT INTO daily_summary VALUES ('stock','MGIC','us','2017-11-01',8.9,9,8.7699,8.95,6069,'0');
INSERT INTO daily_summary VALUES ('etf','MGK','us','2017-11-01',108.35,108.35,107.725,108,117796,'0');
INSERT INTO daily_summary VALUES ('stock','MGLN','us','2017-11-01',92,92,82.85,83.55,183567,'0');
INSERT INTO daily_summary VALUES ('stock','MGM','us','2017-11-01',31.9,31.98,30.735,30.82,8930400,'0');
INSERT INTO daily_summary VALUES ('stock','MGNX','us','2017-11-01',20,20.19,19.11,19.34,247134,'0');
INSERT INTO daily_summary VALUES ('stock','MGP','us','2017-11-01',29.45,29.84,29.45,29.75,622726,'0');
INSERT INTO daily_summary VALUES ('stock','MGPI','us','2017-11-01',68.174,68.723,62.727,62.927,171564,'0');
INSERT INTO daily_summary VALUES ('stock','MGRC','us','2017-11-01',48,49.66,47.831,49.12,211052,'0');
INSERT INTO daily_summary VALUES ('stock','MGU','us','2017-11-01',25.11,25.28,25.01,25.26,39667,'0');
INSERT INTO daily_summary VALUES ('etf','MGV','us','2017-11-01',73.49,73.69,73.366,73.48,46509,'0');
INSERT INTO daily_summary VALUES ('stock','MGYR','us','2017-11-01',12.15,12.32,12.15,12.32,215,'0');
INSERT INTO daily_summary VALUES ('stock','MHD','us','2017-11-01',16.62,16.64,16.59,16.63,18931,'0');
INSERT INTO daily_summary VALUES ('stock','MHE','us','2017-11-01',13.7,13.7,13.6,13.6,2637,'0');
INSERT INTO daily_summary VALUES ('stock','MHF','us','2017-11-01',7.5,7.58,7.5,7.58,55675,'0');
INSERT INTO daily_summary VALUES ('stock','MHH','us','2017-11-01',12.75,12.8,11.74,12,38177,'0');
INSERT INTO daily_summary VALUES ('stock','MHI','us','2017-11-01',11.74,11.75,11.66,11.68,98710,'0');
INSERT INTO daily_summary VALUES ('stock','MHK','us','2017-11-01',261.39,263.31,260.94,262.45,402237,'0');
INSERT INTO daily_summary VALUES ('stock','MHLA','us','2017-11-01',26.26,26.28,26.04,26.05,5683,'0');
INSERT INTO daily_summary VALUES ('stock','MHLD','us','2017-11-01',8.25,8.55,8.05,8.35,468112,'0');
INSERT INTO daily_summary VALUES ('stock','MHN','us','2017-11-01',13.78,13.78,13.71,13.72,71969,'0');
INSERT INTO daily_summary VALUES ('stock','MHNC','us','2017-11-01',26.749,26.749,26.11,26.15,3900,'0');
INSERT INTO daily_summary VALUES ('stock','MHO','us','2017-11-01',33.2,34,32.85,33.72,354667,'0');
INSERT INTO daily_summary VALUES ('stock','MH_A','us','2017-11-01',25.62,25.65,25.5486,25.55,5400,'0');
INSERT INTO daily_summary VALUES ('stock','MH_C','us','2017-11-01',25.96,26,25.6119,25.7,31552,'0');
INSERT INTO daily_summary VALUES ('stock','MH_D','us','2017-11-01',24.96,25,24.95,24.95,57976,'0');
INSERT INTO daily_summary VALUES ('stock','MIC','us','2017-11-01',68.261,68.819,68.114,68.623,713807,'0');
INSERT INTO daily_summary VALUES ('stock','MICR','us','2017-11-01',3.62,3.63,3.61,3.63,773,'0');
INSERT INTO daily_summary VALUES ('stock','MICT','us','2017-11-01',0.79,0.898,0.78,0.81,122113,'0');
INSERT INTO daily_summary VALUES ('stock','MICTW','us','2017-11-01',0.08,0.09,0.08,0.09,3020,'0');
INSERT INTO daily_summary VALUES ('stock','MIDD','us','2017-11-01',116.19,116.48,115.07,115.79,439764,'0');
INSERT INTO daily_summary VALUES ('etf','MIDU','us','2017-11-01',43.58,43.65,42.269,42.5,24896,'0');
INSERT INTO daily_summary VALUES ('etf','MIDZ','us','2017-11-01',15.12,15.4404,14.9,15.36,100504,'0');
INSERT INTO daily_summary VALUES ('stock','MIE','us','2017-11-01',9.87,9.96,9.83,9.87,104876,'0');
INSERT INTO daily_summary VALUES ('stock','MIII','us','2017-11-01',9.82,9.8392,9.82,9.8392,493,'0');
INSERT INTO daily_summary VALUES ('stock','MIIIU','us','2017-11-01',10.3099,10.33,10.26,10.33,3300,'0');
INSERT INTO daily_summary VALUES ('stock','MIIIW','us','2017-11-01',0.55,0.56,0.538,0.538,1800,'0');
INSERT INTO daily_summary VALUES ('stock','MIK','us','2017-11-01',19.41,19.85,19.25,19.58,1361130,'0');
INSERT INTO daily_summary VALUES ('stock','MILN','us','2017-11-01',18.62,18.62,18.54,18.5501,24471,'0');
INSERT INTO daily_summary VALUES ('stock','MIME','us','2017-11-01',31.88,31.88,30.78,31.02,271473,'0');
INSERT INTO daily_summary VALUES ('stock','MIN','us','2017-11-01',4.22,4.24,4.21,4.22,312733,'0');
INSERT INTO daily_summary VALUES ('etf','MINC','us','2017-11-01',48.71,48.72,48.62,48.65,6078,'0');
INSERT INTO daily_summary VALUES ('stock','MIND','us','2017-11-01',3.23,3.26,3.2166,3.25,140181,'0');
INSERT INTO daily_summary VALUES ('stock','MINDP','us','2017-11-01',22.8608,22.8608,22.8608,22.8608,330,'0');
INSERT INTO daily_summary VALUES ('stock','MINI','us','2017-11-01',33.4,34,33.1,33.8,145457,'0');
INSERT INTO daily_summary VALUES ('etf','MINT','us','2017-11-01',101.73,101.74,101.71,101.74,910116,'0');
INSERT INTO daily_summary VALUES ('stock','MITK','us','2017-11-01',8.9,9,8.56,8.7,251609,'0');
INSERT INTO daily_summary VALUES ('stock','MITL','us','2017-11-01',8.66,8.735,8.52,8.57,879618,'0');
INSERT INTO daily_summary VALUES ('stock','MITT','us','2017-11-01',19.05,19.49,19.05,19.35,386532,'0');
INSERT INTO daily_summary VALUES ('stock','MITT_A','us','2017-11-01',25.65,25.6586,25.59,25.6415,3829,'0');
INSERT INTO daily_summary VALUES ('stock','MITT_B','us','2017-11-01',25.3,25.4,25.285,25.4,4785,'0');
INSERT INTO daily_summary VALUES ('stock','MIXT','us','2017-11-01',10.23,10.23,9.9,10.03,52657,'0');
INSERT INTO daily_summary VALUES ('stock','MIY','us','2017-11-01',14.02,14.03,13.97,13.97,51505,'0');
INSERT INTO daily_summary VALUES ('stock','MJCO','us','2017-11-01',5.96,5.96,5.2601,5.38,10850,'0');
INSERT INTO daily_summary VALUES ('stock','MKC','us','2017-11-01',99.94,100.34,99.115,99.85,1298220,'0');
INSERT INTO daily_summary VALUES ('stock','MKL','us','2017-11-01',1086.77,1090.96,1076.68,1080.07,29658,'0');
INSERT INTO daily_summary VALUES ('stock','MKSI','us','2017-11-01',109.55,109.55,102.805,105.6,1003760,'0');
INSERT INTO daily_summary VALUES ('stock','MKTX','us','2017-11-01',175.44,177.39,174.34,175.41,329432,'0');
INSERT INTO daily_summary VALUES ('stock','MLAB','us','2017-11-01',160.08,161.945,158.09,158.09,4783,'0');
INSERT INTO daily_summary VALUES ('stock','MLCO','us','2017-11-01',25.97,26.53,25.42,26.52,6853240,'0');
INSERT INTO daily_summary VALUES ('stock','MLHR','us','2017-11-01',33.9,34,32.9,33.2,266947,'0');
INSERT INTO daily_summary VALUES ('stock','MLI','us','2017-11-01',35.15,35.5975,34.74,34.95,185147,'0');
INSERT INTO daily_summary VALUES ('stock','MLM','us','2017-11-01',216.9,217.91,213.15,217.3,735039,'0');
INSERT INTO daily_summary VALUES ('etf','MLN','us','2017-11-01',20.01,20.07,20.01,20.07,51627,'0');
INSERT INTO daily_summary VALUES ('stock','MLNK','us','2017-11-01',1.82,1.85,1.8,1.8101,54709,'0');
INSERT INTO daily_summary VALUES ('stock','MLNT','us','2017-11-01',12.25,12.25,11.25,11.25,189017,'0');
INSERT INTO daily_summary VALUES ('stock','MLNX','us','2017-11-01',46.8,47.0503,45.6,45.8,473347,'0');
INSERT INTO daily_summary VALUES ('stock','MLP','us','2017-11-01',16,16.1,15.5,15.8,23050,'0');
INSERT INTO daily_summary VALUES ('etf','MLPA','us','2017-11-01',9.583,9.74,9.583,9.73,474615,'0');
INSERT INTO daily_summary VALUES ('etf','MLPC','us','2017-11-01',15.25,15.4384,15.25,15.41,1251,'0');
INSERT INTO daily_summary VALUES ('etf','MLPI','us','2017-11-01',23.63,23.97,23.63,23.93,997736,'0');
INSERT INTO daily_summary VALUES ('stock','MLPQ','us','2017-11-01',36.479,37.06,36.27,36.95,57430,'0');
INSERT INTO daily_summary VALUES ('etf','MLPS','us','2017-11-01',12.635,12.64,12.635,12.64,381,'0');
INSERT INTO daily_summary VALUES ('etf','MLPX','us','2017-11-01',13.061,13.16,13.061,13.11,84543,'0');
INSERT INTO daily_summary VALUES ('etf','MLPY','us','2017-11-01',6.9355,6.9355,6.91,6.91,1126,'0');
INSERT INTO daily_summary VALUES ('stock','MLPZ','us','2017-11-01',40.59,40.6,40.59,40.6,272,'0');
INSERT INTO daily_summary VALUES ('stock','MLQD','us','2017-11-01',49.7095,49.7095,49.7095,49.7095,10001,'0');
INSERT INTO daily_summary VALUES ('stock','MLR','us','2017-11-01',28.55,28.55,27.25,27.55,16488,'0');
INSERT INTO daily_summary VALUES ('stock','MLSS','us','2017-11-01',1.1,1.2,1.1,1.15,31305,'0');
INSERT INTO daily_summary VALUES ('stock','MLVF','us','2017-11-01',27,27.1188,26.9,27.05,5220,'0');
INSERT INTO daily_summary VALUES ('stock','MMAC','us','2017-11-01',24.3,24.5,24.3,24.5,5409,'0');
INSERT INTO daily_summary VALUES ('stock','MMC','us','2017-11-01',81.44,82.58,81.3674,82.42,3550830,'0');
INSERT INTO daily_summary VALUES ('stock','MMD','us','2017-11-01',19.3,19.37,19.3,19.33,66517,'0');
INSERT INTO daily_summary VALUES ('stock','MMDM','us','2017-11-01',9.78,9.78,9.78,9.78,16402,'0');
INSERT INTO daily_summary VALUES ('stock','MMDMR','us','2017-11-01',0.4,0.4,0.4,0.4,129700,'0');
INSERT INTO daily_summary VALUES ('stock','MMI','us','2017-11-01',28.68,28.735,28.08,28.61,188055,'0');
INSERT INTO daily_summary VALUES ('stock','MMLP','us','2017-11-01',15.402,15.789,15.402,15.547,110394,'0');
INSERT INTO daily_summary VALUES ('stock','MMM','us','2017-11-01',231,231.76,229.11,230.18,1389920,'0');
INSERT INTO daily_summary VALUES ('stock','MMP','us','2017-11-01',67.8,68.68,67.75,68.27,603756,'0');
INSERT INTO daily_summary VALUES ('stock','MMS','us','2017-11-01',66.8,66.96,65.44,65.88,270824,'0');
INSERT INTO daily_summary VALUES ('stock','MMSI','us','2017-11-01',38.15,38.45,36.75,37.2,279913,'0');
INSERT INTO daily_summary VALUES ('stock','MMT','us','2017-11-01',6.16,6.19,6.16,6.19,182183,'0');
INSERT INTO daily_summary VALUES ('etf','MMTM','us','2017-11-01',111.117,111.117,110.79,110.79,727,'0');
INSERT INTO daily_summary VALUES ('stock','MMU','us','2017-11-01',14.04,14.04,13.94,14.01,74389,'0');
INSERT INTO daily_summary VALUES ('stock','MMV','us','2017-11-01',13.301,13.301,13.301,13.301,100,'0');
INSERT INTO daily_summary VALUES ('stock','MMYT','us','2017-11-01',26.25,27.25,24.325,24.775,2101250,'0');
INSERT INTO daily_summary VALUES ('stock','MN','us','2017-11-01',3.75,3.85,3.65,3.7,93781,'0');
INSERT INTO daily_summary VALUES ('etf','MNA','us','2017-11-01',31.0564,31.066,30.98,31,39407,'0');
INSERT INTO daily_summary VALUES ('stock','MNDO','us','2017-11-01',2.6936,2.75,2.6936,2.75,29319,'0');
INSERT INTO daily_summary VALUES ('stock','MNE','us','2017-11-01',13.73,13.77,13.73,13.77,3398,'0');
INSERT INTO daily_summary VALUES ('stock','MNGA','us','2017-11-01',0.485,0.495,0.44,0.455,3189720,'0');
INSERT INTO daily_summary VALUES ('stock','MNI','us','2017-11-01',9.86,9.99,9.17,9.2,19404,'0');
INSERT INTO daily_summary VALUES ('stock','MNK','us','2017-11-01',31.77,32.52,31.41,32.06,1896820,'0');
INSERT INTO daily_summary VALUES ('stock','MNKD','us','2017-11-01',3.28,3.36,3,3.01,5968740,'0');
INSERT INTO daily_summary VALUES ('stock','MNOV','us','2017-11-01',6.93,6.93,6.37,6.4,188738,'0');
INSERT INTO daily_summary VALUES ('stock','MNP','us','2017-11-01',15.21,15.21,15.08,15.14,29279,'0');
INSERT INTO daily_summary VALUES ('stock','MNR','us','2017-11-01',17.07,17.34,17.01,17.11,225221,'0');
INSERT INTO daily_summary VALUES ('stock','MNRO','us','2017-11-01',49.8,50.3,48.3,48.7,340646,'0');
INSERT INTO daily_summary VALUES ('stock','MNR_C','us','2017-11-01',25.19,25.2126,25.17,25.1762,14113,'0');
INSERT INTO daily_summary VALUES ('stock','MNST','us','2017-11-01',58.34,58.43,57.6,57.8,1735550,'0');
INSERT INTO daily_summary VALUES ('stock','MNTA','us','2017-11-01',12.7,14.05,12.3,12.325,1384100,'0');
INSERT INTO daily_summary VALUES ('stock','MNTX','us','2017-11-01',9.08,9.19,8.61,8.94,34657,'0');
INSERT INTO daily_summary VALUES ('stock','MO','us','2017-11-01',64.56,65.03,64.34,64.7,5668950,'0');
INSERT INTO daily_summary VALUES ('etf','MOAT','us','2017-11-01',40.54,40.72,40.48,40.48,103875,'0');
INSERT INTO daily_summary VALUES ('stock','MOBL','us','2017-11-01',3.85,3.925,3.7,3.85,1058530,'0');
INSERT INTO daily_summary VALUES ('stock','MOC','us','2017-11-01',3.12,3.26,3.12,3.258,6616,'0');
INSERT INTO daily_summary VALUES ('stock','MOD','us','2017-11-01',21.85,22.875,21.24,22.8,1368490,'0');
INSERT INTO daily_summary VALUES ('stock','MODN','us','2017-11-01',14.65,14.7,14.25,14.3,101604,'0');
INSERT INTO daily_summary VALUES ('stock','MOFG','us','2017-11-01',35.5,36.61,34.6,34.93,22530,'0');
INSERT INTO daily_summary VALUES ('stock','MOH','us','2017-11-01',67.83,68.94,67.56,68.18,1511190,'0');
INSERT INTO daily_summary VALUES ('stock','MOMO','us','2017-11-01',30.7,31,30.11,30.36,2402040,'0');
INSERT INTO daily_summary VALUES ('stock','MON','us','2017-11-01',121.15,121.49,120.98,121.14,2434680,'0');
INSERT INTO daily_summary VALUES ('etf','MOO','us','2017-11-01',60.55,60.647,60.32,60.33,40454,'0');
INSERT INTO daily_summary VALUES ('etf','MORL','us','2017-11-01',16.817,17.096,16.817,16.976,369916,'0');
INSERT INTO daily_summary VALUES ('stock','MORN','us','2017-11-01',85.67,86.14,85.09,85.56,37512,'0');
INSERT INTO daily_summary VALUES ('etf','MORT','us','2017-11-01',23.96,24.19,23.96,24.05,26015,'0');
INSERT INTO daily_summary VALUES ('stock','MOS','us','2017-11-01',23.02,23.14,22.5,22.64,7616480,'0');
INSERT INTO daily_summary VALUES ('stock','MOSY','us','2017-11-01',0.6679,0.6776,0.63,0.6401,90518,'0');
INSERT INTO daily_summary VALUES ('stock','MOTI','us','2017-11-01',35.68,35.68,35.544,35.58,3273,'0');
INSERT INTO daily_summary VALUES ('stock','MOV','us','2017-11-01',27.9,27.95,27,27.25,108488,'0');
INSERT INTO daily_summary VALUES ('stock','MPA','us','2017-11-01',14.18,14.26,14.16,14.19,15957,'0');
INSERT INTO daily_summary VALUES ('stock','MPAA','us','2017-11-01',29,29.14,27.89,28.09,284984,'0');
INSERT INTO daily_summary VALUES ('stock','MPAC','us','2017-11-01',9.73,9.73,9.73,9.73,100,'0');
INSERT INTO daily_summary VALUES ('stock','MPACW','us','2017-11-01',0.4,0.4,0.4,0.4,11333,'0');
INSERT INTO daily_summary VALUES ('stock','MPB','us','2017-11-01',29.122,30.162,29.122,29.421,1556,'0');
INSERT INTO daily_summary VALUES ('stock','MPC','us','2017-11-01',60.24,61.9,60.07,60.89,5609440,'0');
INSERT INTO daily_summary VALUES ('stock','MPCT','us','2017-11-01',59.03,59.04,58.725,58.75,2783,'0');
INSERT INTO daily_summary VALUES ('stock','MPLX','us','2017-11-01',34.931,35.196,34.538,34.685,2228430,'0');
INSERT INTO daily_summary VALUES ('stock','MPO','us','2017-11-01',15.37,16.33,15.37,15.95,825764,'0');
INSERT INTO daily_summary VALUES ('stock','MPV','us','2017-11-01',14.32,14.59,14.31,14.46,12186,'0');
INSERT INTO daily_summary VALUES ('stock','MPVD','us','2017-11-01',2.7,2.91,2.7,2.91,17298,'0');
INSERT INTO daily_summary VALUES ('stock','MPW','us','2017-11-01',13.24,13.28,13.1631,13.17,1902930,'0');
INSERT INTO daily_summary VALUES ('stock','MPWR','us','2017-11-01',122.83,122.94,119.08,120.62,223051,'0');
INSERT INTO daily_summary VALUES ('stock','MPX','us','2017-11-01',14.455,14.455,14.069,14.078,35588,'0');
INSERT INTO daily_summary VALUES ('stock','MQT','us','2017-11-01',12.96,13.03,12.95,12.95,32933,'0');
INSERT INTO daily_summary VALUES ('stock','MQY','us','2017-11-01',15.28,15.38,15.27,15.28,29472,'0');
INSERT INTO daily_summary VALUES ('stock','MRAM','us','2017-11-01',10.29,10.29,9.41,9.49,201922,'0');
INSERT INTO daily_summary VALUES ('stock','MRC','us','2017-11-01',17.34,17.44,16.49,17.04,1510300,'0');
INSERT INTO daily_summary VALUES ('stock','MRCC','us','2017-11-01',14.59,14.69,14.55,14.56,72798,'0');
INSERT INTO daily_summary VALUES ('stock','MRCY','us','2017-11-01',50.5,50.6688,49.595,50.1,246785,'0');
INSERT INTO daily_summary VALUES ('stock','MRDN','us','2017-11-01',0.99,1.0499,0.99,1,8659,'0');
INSERT INTO daily_summary VALUES ('etf','MRGR','us','2017-11-01',35.76,35.76,35.76,35.76,0,'0');
INSERT INTO daily_summary VALUES ('stock','MRIN','us','2017-11-01',14.45,14.45,14.1,14.1,10369,'0');
INSERT INTO daily_summary VALUES ('stock','MRK','us','2017-11-01',55.54,55.8,55.28,55.34,16417800,'0');
INSERT INTO daily_summary VALUES ('stock','MRLN','us','2017-11-01',21.924,23.913,21.775,23.664,67627,'0');
INSERT INTO daily_summary VALUES ('stock','MRNS','us','2017-11-01',5.36,5.897,5.3,5.79,711422,'0');
INSERT INTO daily_summary VALUES ('stock','MRO','us','2017-11-01',14.37,14.89,14.35,14.87,17081500,'0');
INSERT INTO daily_summary VALUES ('stock','MRRL','us','2017-11-01',16.648,17.881,16.648,16.969,18150,'0');
INSERT INTO daily_summary VALUES ('stock','MRSN','us','2017-11-01',16.3999,16.3999,15.01,15.18,17926,'0');
INSERT INTO daily_summary VALUES ('stock','MRT','us','2017-11-01',11.65,11.71,11.405,11.56,82626,'0');
INSERT INTO daily_summary VALUES ('stock','MRTN','us','2017-11-01',19.7,19.85,19.2,19.55,133787,'0');
INSERT INTO daily_summary VALUES ('stock','MRTX','us','2017-11-01',13.1,13.275,12.55,12.7,244677,'0');
INSERT INTO daily_summary VALUES ('stock','MRUS','us','2017-11-01',15.41,16.3,15,15.26,25866,'0');
INSERT INTO daily_summary VALUES ('stock','MRVL','us','2017-11-01',18.57,18.57,18.14,18.29,3327840,'0');
INSERT INTO daily_summary VALUES ('stock','MS','us','2017-11-01',50.22,50.545,49.5,49.63,7842840,'0');
INSERT INTO daily_summary VALUES ('stock','MSA','us','2017-11-01',80.09,80.09,78.069,79.114,179618,'0');
INSERT INTO daily_summary VALUES ('stock','MSB','us','2017-11-01',22.75,22.75,21.7,22.5,90334,'0');
INSERT INTO daily_summary VALUES ('stock','MSBF','us','2017-11-01',17.5,17.5,17.25,17.25,1673,'0');
INSERT INTO daily_summary VALUES ('stock','MSBI','us','2017-11-01',32.8,34.61,32.1,32.3,97374,'0');
INSERT INTO daily_summary VALUES ('stock','MSCA','us','2017-11-01',25.8,25.8,25.6513,25.6513,800,'0');
INSERT INTO daily_summary VALUES ('stock','MSCC','us','2017-11-01',53.84,54.19,52.94,53.16,936299,'0');
INSERT INTO daily_summary VALUES ('stock','MSCI','us','2017-11-01',118.21,119.49,117.28,117.59,548388,'0');
INSERT INTO daily_summary VALUES ('stock','MSD','us','2017-11-01',10.07,10.11,10.07,10.1,54201,'0');
INSERT INTO daily_summary VALUES ('stock','MSDI','us','2017-11-01',0.5509,0.57,0.545,0.5588,122454,'0');
INSERT INTO daily_summary VALUES ('stock','MSDIW','us','2017-11-01',0.08,0.08,0.0425,0.0425,10199,'0');
INSERT INTO daily_summary VALUES ('stock','MSEX','us','2017-11-01',43.727,44.498,42.643,42.921,23023,'0');
INSERT INTO daily_summary VALUES ('stock','MSF','us','2017-11-01',17.56,17.57,17.4991,17.5199,100493,'0');
INSERT INTO daily_summary VALUES ('stock','MSFG','us','2017-11-01',38.04,38.18,37.17,37.53,70199,'0');
INSERT INTO daily_summary VALUES ('stock','MSFT','us','2017-11-01',83.68,83.76,82.88,83.18,22307400,'0');
INSERT INTO daily_summary VALUES ('stock','MSG','us','2017-11-01',223.41,229.36,222.67,225.75,689195,'0');
INSERT INTO daily_summary VALUES ('stock','MSGN','us','2017-11-01',17.3,17.6,16.95,17,917285,'0');
INSERT INTO daily_summary VALUES ('stock','MSI','us','2017-11-01',91.14,91.74,90.16,90.56,789830,'0');
INSERT INTO daily_summary VALUES ('stock','MSL','us','2017-11-01',13.2,13.4,13.05,13.05,23587,'0');
INSERT INTO daily_summary VALUES ('stock','MSM','us','2017-11-01',83.499,83.836,79.801,80.318,798585,'0');
INSERT INTO daily_summary VALUES ('stock','MSN','us','2017-11-01',1.4287,1.5015,1.42,1.5015,30004,'0');
INSERT INTO daily_summary VALUES ('stock','MSON','us','2017-11-01',10.175,10.3,10.1,10.25,5799,'0');
INSERT INTO daily_summary VALUES ('stock','MSP','us','2017-11-01',11.7,11.8192,11.65,11.71,12446,'0');
INSERT INTO daily_summary VALUES ('stock','MSTR','us','2017-11-01',133.4,134.96,129.68,134.54,159005,'0');
INSERT INTO daily_summary VALUES ('stock','MS_A','us','2017-11-01',23.07,23.19,23.06,23.06,21373,'0');
INSERT INTO daily_summary VALUES ('stock','MS_E','us','2017-11-01',29.01,29.12,28.96,28.96,34088,'0');
INSERT INTO daily_summary VALUES ('stock','MS_F','us','2017-11-01',28.58,28.77,28.58,28.65,35363,'0');
INSERT INTO daily_summary VALUES ('stock','MS_G','us','2017-11-01',26.7,26.82,26.6342,26.68,30517,'0');
INSERT INTO daily_summary VALUES ('stock','MS_I','us','2017-11-01',27.96,28.105,27.96,28.05,22321,'0');
INSERT INTO daily_summary VALUES ('stock','MS_K','us','2017-11-01',27.13,27.13,27,27.09,222254,'0');
INSERT INTO daily_summary VALUES ('stock','MT','us','2017-11-01',29.55,29.735,29.18,29.43,6402610,'0');
INSERT INTO daily_summary VALUES ('stock','MTB','us','2017-11-01',167.63,168.83,166.86,167.28,638001,'0');
INSERT INTO daily_summary VALUES ('stock','MTBC','us','2017-11-01',3.21,3.43,3.21,3.37,1384140,'0');
INSERT INTO daily_summary VALUES ('stock','MTBCP','us','2017-11-01',25.35,25.35,25.2493,25.2493,7365,'0');
INSERT INTO daily_summary VALUES ('stock','MTCH','us','2017-11-01',26.96,26.96,26.33,26.8,1457910,'0');
INSERT INTO daily_summary VALUES ('stock','MTD','us','2017-11-01',685.48,694.48,685.48,689.11,126018,'0');
INSERT INTO daily_summary VALUES ('stock','MTDR','us','2017-11-01',26.91,27.21,26.69,27.06,1615950,'0');
INSERT INTO daily_summary VALUES ('stock','MTEM','us','2017-11-01',8.21,8.21,7.82,8.11,58331,'0');
INSERT INTO daily_summary VALUES ('stock','MTEX','us','2017-11-01',14.4,14.9,14.4,14.4,2434,'0');
INSERT INTO daily_summary VALUES ('stock','MTFB','us','2017-11-01',10.8,10.8,10.36,10.3875,1500,'0');
INSERT INTO daily_summary VALUES ('stock','MTFBW','us','2017-11-01',4.53,4.6,4.5,4.51,636,'0');
INSERT INTO daily_summary VALUES ('stock','MTG','us','2017-11-01',14.4,14.5,14.04,14.08,2685210,'0');
INSERT INTO daily_summary VALUES ('stock','MTGE','us','2017-11-01',18.15,18.7,18.15,18.45,763692,'0');
INSERT INTO daily_summary VALUES ('stock','MTH','us','2017-11-01',49.1,49.65,48.85,49.35,403399,'0');
INSERT INTO daily_summary VALUES ('stock','MTL','us','2017-11-01',5.03,5.05,4.98,5,63827,'0');
INSERT INTO daily_summary VALUES ('stock','MTLS','us','2017-11-01',15.52,15.7,14.95,15,39194,'0');
INSERT INTO daily_summary VALUES ('stock','MTL_','us','2017-11-01',0.87,0.9019,0.87,0.9,1048,'0');
INSERT INTO daily_summary VALUES ('stock','MTN','us','2017-11-01',230.42,232.47,229.66,230.47,182303,'0');
INSERT INTO daily_summary VALUES ('stock','MTNB','us','2017-11-01',1.11,1.12,1.09,1.1,91305,'0');
INSERT INTO daily_summary VALUES ('stock','MTOR','us','2017-11-01',26.2,26.5,25.86,26.13,805946,'0');
INSERT INTO daily_summary VALUES ('stock','MTP','us','2017-11-01',1.35,1.42,1.35,1.42,4418,'0');
INSERT INTO daily_summary VALUES ('stock','MTR','us','2017-11-01',15.4,15.65,15.247,15.55,9674,'0');
INSERT INTO daily_summary VALUES ('stock','MTRN','us','2017-11-01',51.996,51.996,50,50.399,143536,'0');
INSERT INTO daily_summary VALUES ('stock','MTRX','us','2017-11-01',14.3,14.3118,13.8,14,127843,'0');
INSERT INTO daily_summary VALUES ('stock','MTSC','us','2017-11-01',52.6,53.375,51.45,51.95,51455,'0');
INSERT INTO daily_summary VALUES ('stock','MTSI','us','2017-11-01',41.44,41.44,39.11,39.86,435302,'0');
INSERT INTO daily_summary VALUES ('stock','MTSL','us','2017-11-01',2.08,2.08,2.03,2.03,300,'0');
INSERT INTO daily_summary VALUES ('stock','MTT','us','2017-11-01',21.97,21.97,21.8,21.86,38608,'0');
INSERT INTO daily_summary VALUES ('stock','MTU','us','2017-11-01',6.8,6.83,6.775,6.8,1951700,'0');
INSERT INTO daily_summary VALUES ('etf','MTUM','us','2017-11-01',101.18,101.18,99.951,100.27,472936,'0');
INSERT INTO daily_summary VALUES ('stock','MTW','us','2017-11-01',10.13,10.335,9.86,10.01,3036700,'0');
INSERT INTO daily_summary VALUES ('stock','MTX','us','2017-11-01',72.65,73.55,71.7,71.9,151279,'0');
INSERT INTO daily_summary VALUES ('stock','MTZ','us','2017-11-01',44.15,44.15,42.6,42.8,605544,'0');
INSERT INTO daily_summary VALUES ('stock','MU','us','2017-11-01',44.79,45.33,43.78,44.38,38842900,'0');
INSERT INTO daily_summary VALUES ('stock','MUA','us','2017-11-01',15,15,14.946,15,59935,'0');
INSERT INTO daily_summary VALUES ('etf','MUB','us','2017-11-01',110.56,110.56,110.43,110.52,536082,'0');
INSERT INTO daily_summary VALUES ('stock','MUC','us','2017-11-01',14.7,14.76,14.67,14.67,39427,'0');
INSERT INTO daily_summary VALUES ('stock','MUE','us','2017-11-01',13.73,13.79,13.72,13.79,28043,'0');
INSERT INTO daily_summary VALUES ('stock','MUH','us','2017-11-01',14.93,14.96,14.92,14.96,5228,'0');
INSERT INTO daily_summary VALUES ('stock','MUI','us','2017-11-01',14.06,14.06,13.99,14.04,76758,'0');
INSERT INTO daily_summary VALUES ('stock','MUJ','us','2017-11-01',14.33,14.33,14.23,14.24,55202,'0');
INSERT INTO daily_summary VALUES ('stock','MULE','us','2017-11-01',23.55,23.69,23.18,23.38,1234030,'0');
INSERT INTO daily_summary VALUES ('etf','MUNI','us','2017-11-01',53.72,53.735,53.6362,53.6362,13745,'0');
INSERT INTO daily_summary VALUES ('stock','MUR','us','2017-11-01',26.883,27.547,26.674,27.16,3495680,'0');
INSERT INTO daily_summary VALUES ('stock','MUS','us','2017-11-01',13.4,13.44,13.34,13.34,21601,'0');
INSERT INTO daily_summary VALUES ('stock','MUSA','us','2017-11-01',75,75.59,74.28,75.29,504268,'0');
INSERT INTO daily_summary VALUES ('stock','MUX','us','2017-11-01',1.96,1.9601,1.91,1.95,4233110,'0');
INSERT INTO daily_summary VALUES ('stock','MVC','us','2017-11-01',10.72,10.77,10.64,10.68,49160,'0');
INSERT INTO daily_summary VALUES ('stock','MVCB','us','2017-11-01',25.281,25.33,25.281,25.33,535,'0');
INSERT INTO daily_summary VALUES ('stock','MVF','us','2017-11-01',9.73,9.73,9.65,9.68,90754,'0');
INSERT INTO daily_summary VALUES ('stock','MVIN','us','2017-11-01',46.06,46.09,46.055,46.055,983,'0');
INSERT INTO daily_summary VALUES ('stock','MVIS','us','2017-11-01',2.24,2.2645,2.17,2.21,416500,'0');
INSERT INTO daily_summary VALUES ('stock','MVO','us','2017-11-01',5.9383,5.9383,5.6814,5.92,29897,'0');
INSERT INTO daily_summary VALUES ('stock','MVT','us','2017-11-01',15.2,15.359,15.2,15.21,44803,'0');
INSERT INTO daily_summary VALUES ('etf','MVV','us','2017-11-01',115.76,115.94,113.44,114.03,33909,'0');
INSERT INTO daily_summary VALUES ('stock','MWA','us','2017-11-01',12.06,12.06,11.886,11.93,1158910,'0');
INSERT INTO daily_summary VALUES ('stock','MX','us','2017-11-01',10.45,10.65,10.25,10.4,433999,'0');
INSERT INTO daily_summary VALUES ('stock','MXC','us','2017-11-01',3.96,4.0199,3.95,3.95,3718,'0');
INSERT INTO daily_summary VALUES ('stock','MXDU','us','2017-11-01',25.15,25.15,25.15,25.15,4100,'0');
INSERT INTO daily_summary VALUES ('stock','MXE','us','2017-11-01',10.47,10.5,10.41,10.4184,15042,'0');
INSERT INTO daily_summary VALUES ('stock','MXF','us','2017-11-01',15.85,15.9,15.73,15.82,17154,'0');
INSERT INTO daily_summary VALUES ('etf','MXI','us','2017-11-01',68.3,68.49,68,68.01,24788,'0');
INSERT INTO daily_summary VALUES ('stock','MXIM','us','2017-11-01',52.9,52.96,52.07,52.33,1520000,'0');
INSERT INTO daily_summary VALUES ('stock','MXL','us','2017-11-01',24.76,24.76,23.8,23.86,482939,'0');
INSERT INTO daily_summary VALUES ('stock','MXWL','us','2017-11-01',4.82,4.82,4.57,4.57,165137,'0');
INSERT INTO daily_summary VALUES ('stock','MYC','us','2017-11-01',15.24,15.36,15.23,15.28,26603,'0');
INSERT INTO daily_summary VALUES ('stock','MYD','us','2017-11-01',14.31,14.3787,14.29,14.3,101750,'0');
INSERT INTO daily_summary VALUES ('stock','MYE','us','2017-11-01',21.8,21.9,21.15,21.35,56098,'0');
INSERT INTO daily_summary VALUES ('stock','MYF','us','2017-11-01',15.36,15.4,15.31,15.35,27662,'0');
INSERT INTO daily_summary VALUES ('stock','MYGN','us','2017-11-01',34.46,34.84,33.74,33.84,893138,'0');
INSERT INTO daily_summary VALUES ('stock','MYI','us','2017-11-01',14.05,14.13,14.03,14.12,103207,'0');
INSERT INTO daily_summary VALUES ('stock','MYJ','us','2017-11-01',15.97,16.01,15.97,15.99,29715,'0');
INSERT INTO daily_summary VALUES ('stock','MYL','us','2017-11-01',35.63,37.25,35.3,36.85,5703620,'0');
INSERT INTO daily_summary VALUES ('stock','MYN','us','2017-11-01',12.99,13.023,12.95,12.98,70987,'0');
INSERT INTO daily_summary VALUES ('stock','MYND','us','2017-11-01',4.2,4.2,3.811,4.1811,25098,'0');
INSERT INTO daily_summary VALUES ('stock','MYNDW','us','2017-11-01',0.74,0.74,0.6,0.6,2723,'0');
INSERT INTO daily_summary VALUES ('stock','MYO','us','2017-11-01',3.4,3.41,3.17,3.24,30248,'0');
INSERT INTO daily_summary VALUES ('stock','MYOK','us','2017-11-01',38.6,39.7,37.775,39,382231,'0');
INSERT INTO daily_summary VALUES ('stock','MYOS','us','2017-11-01',1.32,1.32,1.26,1.29,70201,'0');
INSERT INTO daily_summary VALUES ('stock','MYOV','us','2017-11-01',13.76,14,13.76,13.85,29955,'0');
INSERT INTO daily_summary VALUES ('stock','MYRG','us','2017-11-01',32.17,32.17,30.95,31.36,144790,'0');
INSERT INTO daily_summary VALUES ('stock','MYSZ','us','2017-11-01',0.7318,0.7598,0.7,0.7105,175342,'0');
INSERT INTO daily_summary VALUES ('etf','MYY','us','2017-11-01',11.58,11.695,11.58,11.68,17615,'0');
INSERT INTO daily_summary VALUES ('stock','MZA','us','2017-11-01',15.38,15.38,15.13,15.1873,20240,'0');
INSERT INTO daily_summary VALUES ('stock','MZF','us','2017-11-01',13.54,13.66,13.52,13.6,38934,'0');
INSERT INTO daily_summary VALUES ('stock','MZOR','us','2017-11-01',64.06,64.26,59.76,60.91,1458190,'0');
INSERT INTO daily_summary VALUES ('etf','MZZ','us','2017-11-01',19.996,20.1838,19.981,20.1838,424,'0');
INSERT INTO daily_summary VALUES ('stock','NAC','us','2017-11-01',14.55,14.55,14.46,14.48,253034,'0');
INSERT INTO daily_summary VALUES ('stock','NAD','us','2017-11-01',13.89,13.89,13.82,13.85,345191,'0');
INSERT INTO daily_summary VALUES ('stock','NAII','us','2017-11-01',10.7,10.7,10.55,10.6,3594,'0');
INSERT INTO daily_summary VALUES ('stock','NAIL','us','2017-11-01',72.71,74.5,71.5548,72.95,74771,'0');
INSERT INTO daily_summary VALUES ('stock','NAK','us','2017-11-01',1.95,2.03,1.95,1.96,1701390,'0');
INSERT INTO daily_summary VALUES ('stock','NAKD','us','2017-11-01',1.24,1.3171,1.24,1.26,20417,'0');
INSERT INTO daily_summary VALUES ('stock','NAN','us','2017-11-01',13.9,13.91,13.85,13.86,57481,'0');
INSERT INTO daily_summary VALUES ('stock','NANO','us','2017-11-01',28.53,28.59,26.98,27.47,147401,'0');
INSERT INTO daily_summary VALUES ('stock','NANR','us','2017-11-01',33.72,34.0241,33.72,33.8,21750,'0');
INSERT INTO daily_summary VALUES ('stock','NAO','us','2017-11-01',1.31,1.3853,1.28,1.31,246039,'0');
INSERT INTO daily_summary VALUES ('stock','NAP','us','2017-11-01',8.7166,8.8692,8.5831,8.7166,158474,'0');
INSERT INTO daily_summary VALUES ('stock','NAT','us','2017-11-01',4.4387,4.4982,4.2004,4.3394,1133290,'0');
INSERT INTO daily_summary VALUES ('stock','NATH','us','2017-11-01',81.761,81.761,80.05,80.2,2704,'0');
INSERT INTO daily_summary VALUES ('stock','NATI','us','2017-11-01',44.889,45.556,44.65,44.779,1441130,'0');
INSERT INTO daily_summary VALUES ('stock','NATR','us','2017-11-01',10.05,10.05,9.85,9.95,10807,'0');
INSERT INTO daily_summary VALUES ('stock','NAUH','us','2017-11-01',1.24,1.34,1.15,1.3001,30194,'0');
INSERT INTO daily_summary VALUES ('stock','NAV','us','2017-11-01',42.71,42.86,41.23,41.68,315802,'0');
INSERT INTO daily_summary VALUES ('stock','NAVB','us','2017-11-01',0.42,0.4399,0.42,0.4391,271267,'0');
INSERT INTO daily_summary VALUES ('stock','NAVG','us','2017-11-01',58.45,59.4,57.7,58.5,49202,'0');
INSERT INTO daily_summary VALUES ('stock','NAVI','us','2017-11-01',12.46,12.665,12.06,12.1,5353910,'0');
INSERT INTO daily_summary VALUES ('stock','NAZ','us','2017-11-01',14.63,14.67,14.55,14.55,12686,'0');
INSERT INTO daily_summary VALUES ('stock','NBB','us','2017-11-01',21.42,21.73,21.41,21.73,68123,'0');
INSERT INTO daily_summary VALUES ('stock','NBD','us','2017-11-01',22.43,22.43,22.371,22.43,6054,'0');
INSERT INTO daily_summary VALUES ('stock','NBEV','us','2017-11-01',2.39,2.85,2.39,2.7457,1035680,'0');
INSERT INTO daily_summary VALUES ('stock','NBH','us','2017-11-01',14.95,15,14.95,14.98,13746,'0');
INSERT INTO daily_summary VALUES ('stock','NBHC','us','2017-11-01',33.06,33.21,32.34,32.54,178658,'0');
INSERT INTO daily_summary VALUES ('stock','NBIX','us','2017-11-01',62.68,62.99,60.38,60.93,1442860,'0');
INSERT INTO daily_summary VALUES ('stock','NBL','us','2017-11-01',28.188,28.637,27.959,28.447,5610860,'0');
INSERT INTO daily_summary VALUES ('stock','NBLX','us','2017-11-01',51.535,52.06,51.016,51.416,79206,'0');
INSERT INTO daily_summary VALUES ('stock','NBN','us','2017-11-01',25.99,25.99,24.45,24.89,39550,'0');
INSERT INTO daily_summary VALUES ('stock','NBO','us','2017-11-01',12.41,12.4292,12.4,12.42,5619,'0');
INSERT INTO daily_summary VALUES ('stock','NBR','us','2017-11-01',5.72,5.98,5.6,5.77,12150700,'0');
INSERT INTO daily_summary VALUES ('stock','NBRV','us','2017-11-01',6.27,6.3392,5.8,5.95,324614,'0');
INSERT INTO daily_summary VALUES ('stock','NBTB','us','2017-11-01',38.47,38.54,37.42,37.66,134265,'0');
INSERT INTO daily_summary VALUES ('stock','NBW','us','2017-11-01',13.96,13.96,13.9,13.92,9540,'0');
INSERT INTO daily_summary VALUES ('stock','NBY','us','2017-11-01',4.45,4.65,4.35,4.65,3875,'0');
INSERT INTO daily_summary VALUES ('stock','NC','us','2017-11-01',41.6,42.3817,39.705,40.15,47338,'0');
INSERT INTO daily_summary VALUES ('stock','NCA','us','2017-11-01',10.52,10.55,10.499,10.55,39548,'0');
INSERT INTO daily_summary VALUES ('stock','NCB','us','2017-11-01',16.93,17.039,16.92,16.93,5378,'0');
INSERT INTO daily_summary VALUES ('stock','NCBS','us','2017-11-01',57.44,57.69,56.25,56.43,27051,'0');
INSERT INTO daily_summary VALUES ('stock','NCI','us','2017-11-01',17.45,17.55,16.95,17.14,392662,'0');
INSERT INTO daily_summary VALUES ('stock','NCLH','us','2017-11-01',56.06,56.1,54.04,54.2,2249090,'0');
INSERT INTO daily_summary VALUES ('stock','NCMI','us','2017-11-01',6.79,6.91,6.65,6.71,344542,'0');
INSERT INTO daily_summary VALUES ('stock','NCNA','us','2017-11-01',11.47,12.752,11.35,11.89,97113,'0');
INSERT INTO daily_summary VALUES ('stock','NCOM','us','2017-11-01',41,41.45,40.36,40.5,19073,'0');
INSERT INTO daily_summary VALUES ('stock','NCR','us','2017-11-01',32.3,32.39,31.43,31.65,1065060,'0');
INSERT INTO daily_summary VALUES ('stock','NCS','us','2017-11-01',16,16.1,15.7,15.85,324663,'0');
INSERT INTO daily_summary VALUES ('stock','NCSM','us','2017-11-01',21.95,22.27,21.1,21.41,86711,'0');
INSERT INTO daily_summary VALUES ('stock','NCTY','us','2017-11-01',0.895,0.895,0.8949,0.895,330,'0');
INSERT INTO daily_summary VALUES ('stock','NCV','us','2017-11-01',7.0791,7.0791,7.0097,7.0296,255065,'0');
INSERT INTO daily_summary VALUES ('stock','NCZ','us','2017-11-01',6.1994,6.2192,6.1598,6.1598,261780,'0');
INSERT INTO daily_summary VALUES ('stock','NDAQ','us','2017-11-01',73.02,73.02,72.36,72.41,725117,'0');
INSERT INTO daily_summary VALUES ('stock','NDLS','us','2017-11-01',4.45,4.45,4.25,4.3,39699,'0');
INSERT INTO daily_summary VALUES ('stock','NDP','us','2017-11-01',12.74,12.955,12.74,12.8,75528,'0');
INSERT INTO daily_summary VALUES ('stock','NDRA','us','2017-11-01',2.44,2.44,2.4,2.41,4517,'0');
INSERT INTO daily_summary VALUES ('stock','NDRAW','us','2017-11-01',0.5,0.637,0.5,0.5,441,'0');
INSERT INTO daily_summary VALUES ('stock','NDRO','us','2017-11-01',2.8,2.95,2.8,2.85,63220,'0');
INSERT INTO daily_summary VALUES ('stock','NDSN','us','2017-11-01',127.61,127.61,124.69,126.38,228866,'0');
INSERT INTO daily_summary VALUES ('stock','NE','us','2017-11-01',4.2,4.336,4.13,4.19,5863350,'0');
INSERT INTO daily_summary VALUES ('stock','NEA','us','2017-11-01',13.58,13.62,13.55,13.62,409236,'0');
INSERT INTO daily_summary VALUES ('etf','NEAR','us','2017-11-01',50.21,50.21,50.18,50.21,505242,'0');
INSERT INTO daily_summary VALUES ('stock','NEE','us','2017-11-01',155.67,155.72,153.09,153.38,1436900,'0');
INSERT INTO daily_summary VALUES ('stock','NEE_I','us','2017-11-01',25.28,25.32,25.23,25.23,14605,'0');
INSERT INTO daily_summary VALUES ('stock','NEE_J','us','2017-11-01',25.17,25.2767,25.17,25.2348,14312,'0');
INSERT INTO daily_summary VALUES ('stock','NEE_K','us','2017-11-01',25.76,25.76,25.67,25.7,69952,'0');
INSERT INTO daily_summary VALUES ('stock','NEE_R','us','2017-11-01',57.23,57.25,56.66,56.66,216116,'0');
INSERT INTO daily_summary VALUES ('stock','NEM','us','2017-11-01',36.36,36.46,35.59,35.69,4001660,'0');
INSERT INTO daily_summary VALUES ('stock','NEN','us','2017-11-01',72.4,72.7,72.4,72.4,2840,'0');
INSERT INTO daily_summary VALUES ('stock','NEO','us','2017-11-01',8.66,8.76,8.5601,8.67,346835,'0');
INSERT INTO daily_summary VALUES ('stock','NEOG','us','2017-11-01',80.78,81.07,78.62,80.13,111873,'0');
INSERT INTO daily_summary VALUES ('stock','NEON','us','2017-11-01',1.15,1.15,1.1,1.1,53885,'0');
INSERT INTO daily_summary VALUES ('stock','NEOS','us','2017-11-01',10.45,10.55,10,10.3,572894,'0');
INSERT INTO daily_summary VALUES ('stock','NEOT','us','2017-11-01',1.34,1.3798,1.26,1.29,884119,'0');
INSERT INTO daily_summary VALUES ('stock','NEP','us','2017-11-01',39.089,39.425,38.208,38.911,335375,'0');
INSERT INTO daily_summary VALUES ('stock','NEPT','us','2017-11-01',0.96,0.9722,0.9003,0.96,140806,'0');
INSERT INTO daily_summary VALUES ('stock','NERV','us','2017-11-01',6.4,6.4,5.875,6,89224,'0');
INSERT INTO daily_summary VALUES ('stock','NES','us','2017-11-01',11.826,12.35,11.65,12.1,10960,'0');
INSERT INTO daily_summary VALUES ('stock','NESR','us','2017-11-01',9.73,9.77,9.73,9.77,108451,'0');
INSERT INTO daily_summary VALUES ('stock','NESRW','us','2017-11-01',0.51,0.52,0.51,0.52,800,'0');
INSERT INTO daily_summary VALUES ('stock','NETE','us','2017-11-01',4.8,4.9299,4.66,4.8,94605,'0');
INSERT INTO daily_summary VALUES ('stock','NETS','us','2017-11-01',10.48,10.6,9.841,10.1,56677,'0');
INSERT INTO daily_summary VALUES ('stock','NEU','us','2017-11-01',401.4,406,400.39,405.85,37346,'0');
INSERT INTO daily_summary VALUES ('stock','NEV','us','2017-11-01',14.3,14.41,14.28,14.28,96310,'0');
INSERT INTO daily_summary VALUES ('stock','NEWA','us','2017-11-01',13.67,14.96,13.67,14.36,129734,'0');
INSERT INTO daily_summary VALUES ('stock','NEWM','us','2017-11-01',15.566,15.566,14.951,15.019,294480,'0');
INSERT INTO daily_summary VALUES ('stock','NEWR','us','2017-11-01',51.57,51.76,50.23,51.29,346988,'0');
INSERT INTO daily_summary VALUES ('stock','NEWS','us','2017-11-01',12.25,12.32,12.2,12.27,187020,'0');
INSERT INTO daily_summary VALUES ('stock','NEWT','us','2017-11-01',17.19,17.28,17.08,17.19,73078,'0');
INSERT INTO daily_summary VALUES ('stock','NEXA','us','2017-11-01',18.13,18.65,17.98,18.11,290534,'0');
INSERT INTO daily_summary VALUES ('stock','NEXT','us','2017-11-01',9.9,10.1,9.9,10.1,109632,'0');
INSERT INTO daily_summary VALUES ('stock','NEXTW','us','2017-11-01',0.7999,0.7999,0.72,0.7399,16100,'0');
INSERT INTO daily_summary VALUES ('stock','NFBK','us','2017-11-01',17.128,17.198,16.691,16.79,58149,'0');
INSERT INTO daily_summary VALUES ('stock','NFEC','us','2017-11-01',0.945,0.9798,0.9201,0.9798,4139,'0');
INSERT INTO daily_summary VALUES ('stock','NFG','us','2017-11-01',58.24,58.46,57.77,57.85,354094,'0');
INSERT INTO daily_summary VALUES ('stock','NFJ','us','2017-11-01',13.43,13.48,13.42,13.43,118904,'0');
INSERT INTO daily_summary VALUES ('stock','NFLT','us','2017-11-01',25.79,25.81,25.74,25.77,34448,'0');
INSERT INTO daily_summary VALUES ('stock','NFLX','us','2017-11-01',197.24,199.199,196,198,5017420,'0');
INSERT INTO daily_summary VALUES ('etf','NFO','us','2017-11-01',60.83,60.83,60.38,60.38,1637,'0');
INSERT INTO daily_summary VALUES ('etf','NFRA','us','2017-11-01',48.07,48.07,47.77,47.811,15316,'0');
INSERT INTO daily_summary VALUES ('stock','NFX','us','2017-11-01',31.91,31.97,30.31,30.9,7466650,'0');
INSERT INTO daily_summary VALUES ('stock','NG','us','2017-11-01',4.06,4.15,4.06,4.1,1721440,'0');
INSERT INTO daily_summary VALUES ('stock','NGD','us','2017-11-01',3.37,3.44,3.305,3.31,4850570,'0');
INSERT INTO daily_summary VALUES ('etf','NGE','us','2017-11-01',21.3,21.63,21.3,21.55,17289,'0');
INSERT INTO daily_summary VALUES ('stock','NGG','us','2017-11-01',60.83,61.05,60.47,60.55,418900,'0');
INSERT INTO daily_summary VALUES ('stock','NGHC','us','2017-11-01',20.35,20.37,19.96,20.28,158771,'0');
INSERT INTO daily_summary VALUES ('stock','NGHCN','us','2017-11-01',25.4767,25.53,25.4444,25.4997,2546,'0');
INSERT INTO daily_summary VALUES ('stock','NGHCO','us','2017-11-01',25.7,25.7,25.4887,25.56,2963,'0');
INSERT INTO daily_summary VALUES ('stock','NGHCP','us','2017-11-01',25.8473,25.8473,25.75,25.8208,1838,'0');
INSERT INTO daily_summary VALUES ('stock','NGHCZ','us','2017-11-01',25.81,25.98,25.8,25.82,6924,'0');
INSERT INTO daily_summary VALUES ('stock','NGL','us','2017-11-01',11.33,11.373,10.703,10.847,954387,'0');
INSERT INTO daily_summary VALUES ('stock','NGLS_A','us','2017-11-01',26.51,26.7,26.51,26.68,7157,'0');
INSERT INTO daily_summary VALUES ('stock','NGL_B','us','2017-11-01',24.5299,24.53,24.37,24.48,46833,'0');
INSERT INTO daily_summary VALUES ('stock','NGS','us','2017-11-01',27.9,28.25,27.25,27.8,19024,'0');
INSERT INTO daily_summary VALUES ('stock','NGVC','us','2017-11-01',4.9,5.07,4.89,4.94,67061,'0');
INSERT INTO daily_summary VALUES ('stock','NGVT','us','2017-11-01',71.95,72.43,71.22,72.38,201943,'0');
INSERT INTO daily_summary VALUES ('stock','NH','us','2017-11-01',3.65,3.7436,3.5,3.56,90136,'0');
INSERT INTO daily_summary VALUES ('stock','NHA','us','2017-11-01',9.84,9.8699,9.82,9.82,2344,'0');
INSERT INTO daily_summary VALUES ('stock','NHC','us','2017-11-01',64.57,64.57,62.69,62.85,13233,'0');
INSERT INTO daily_summary VALUES ('stock','NHF','us','2017-11-01',23.66,23.9401,23.66,23.9101,128022,'0');
INSERT INTO daily_summary VALUES ('stock','NHI','us','2017-11-01',76.43,77.08,75.77,76.74,186407,'0');
INSERT INTO daily_summary VALUES ('stock','NHLD','us','2017-11-01',2.34,2.4,2.34,2.393,4926,'0');
INSERT INTO daily_summary VALUES ('stock','NHLDW','us','2017-11-01',0.2905,0.2905,0.2125,0.2128,1879,'0');
INSERT INTO daily_summary VALUES ('stock','NHS','us','2017-11-01',12.15,12.15,12.07,12.12,48350,'0');
INSERT INTO daily_summary VALUES ('stock','NHTC','us','2017-11-01',19.126,19.894,18.732,18.801,147614,'0');
INSERT INTO daily_summary VALUES ('stock','NI','us','2017-11-01',26.99,26.99,26.41,26.58,3384350,'0');
INSERT INTO daily_summary VALUES ('etf','NIB','us','2017-11-01',25.45,25.99,25.4,25.78,48207,'0');
INSERT INTO daily_summary VALUES ('stock','NICE','us','2017-11-01',83.75,84.06,81.96,82.05,256555,'0');
INSERT INTO daily_summary VALUES ('stock','NICK','us','2017-11-01',8.8389,8.84,8.8386,8.84,1354,'0');
INSERT INTO daily_summary VALUES ('stock','NID','us','2017-11-01',13.07,13.08,13.01,13.01,127322,'0');
INSERT INTO daily_summary VALUES ('stock','NIE','us','2017-11-01',20.96,20.99,20.9,20.96,30432,'0');
INSERT INTO daily_summary VALUES ('stock','NIHD','us','2017-11-01',0.461,0.55,0.4376,0.46,7386810,'0');
INSERT INTO daily_summary VALUES ('stock','NIM','us','2017-11-01',10.24,10.255,10.2,10.21,28258,'0');
INSERT INTO daily_summary VALUES ('etf','NINI','us','2017-11-01',20.99,21.15,20.937,21.0456,2769,'0');
INSERT INTO daily_summary VALUES ('stock','NIQ','us','2017-11-01',13.15,13.21,13.1372,13.17,12529,'0');
INSERT INTO daily_summary VALUES ('stock','NITE','us','2017-11-01',22.31,23.33,21.6201,22.99,111069,'0');
INSERT INTO daily_summary VALUES ('stock','NJR','us','2017-11-01',44.7,44.8,43.85,44.1,404693,'0');
INSERT INTO daily_summary VALUES ('stock','NJV','us','2017-11-01',14.73,14.73,14.54,14.69,2420,'0');
INSERT INTO daily_summary VALUES ('stock','NK','us','2017-11-01',4.69,4.705,4.2,4.23,220505,'0');
INSERT INTO daily_summary VALUES ('stock','NKE','us','2017-11-01',55.42,55.5,54.85,55.07,7544500,'0');
INSERT INTO daily_summary VALUES ('stock','NKG','us','2017-11-01',13.2,13.24,13.1901,13.24,6997,'0');
INSERT INTO daily_summary VALUES ('stock','NKSH','us','2017-11-01',44.5,44.9,44.45,44.45,1978,'0');
INSERT INTO daily_summary VALUES ('stock','NKTR','us','2017-11-01',24.17,24.25,22.92,23.37,914610,'0');
INSERT INTO daily_summary VALUES ('stock','NKX','us','2017-11-01',15.47,15.47,15.4,15.44,57436,'0');
INSERT INTO daily_summary VALUES ('stock','NL','us','2017-11-01',13,13.2,12.15,12.45,88641,'0');
INSERT INTO daily_summary VALUES ('stock','NLNK','us','2017-11-01',9.33,9.53,9.1643,9.24,843499,'0');
INSERT INTO daily_summary VALUES ('etf','NLR','us','2017-11-01',53.72,53.72,53.51,53.51,887,'0');
INSERT INTO daily_summary VALUES ('stock','NLS','us','2017-11-01',12.9,13.25,12.9,13,964145,'0');
INSERT INTO daily_summary VALUES ('stock','NLSN','us','2017-11-01',37.18,37.225,36.065,36.16,3434540,'0');
INSERT INTO daily_summary VALUES ('stock','NLST','us','2017-11-01',0.65,0.6779,0.61,0.6321,101334,'0');
INSERT INTO daily_summary VALUES ('stock','NLY','us','2017-11-01',11.5,11.6,11.47,11.52,9241650,'0');
INSERT INTO daily_summary VALUES ('stock','NLY_C','us','2017-11-01',25.46,25.46,25.3687,25.39,39478,'0');
INSERT INTO daily_summary VALUES ('stock','NLY_D','us','2017-11-01',25.39,25.45,25.37,25.38,56353,'0');
INSERT INTO daily_summary VALUES ('stock','NLY_E','us','2017-11-01',25.3847,25.39,25.26,25.35,111126,'0');
INSERT INTO daily_summary VALUES ('stock','NLY_F','us','2017-11-01',26.48,26.48,26.23,26.34,88547,'0');
INSERT INTO daily_summary VALUES ('stock','NM','us','2017-11-01',1.84,1.865,1.78,1.81,386189,'0');
INSERT INTO daily_summary VALUES ('stock','NMFC','us','2017-11-01',14.1,14.15,13.75,13.75,435390,'0');
INSERT INTO daily_summary VALUES ('stock','NMI','us','2017-11-01',11.51,11.51,11.4168,11.4299,21156,'0');
INSERT INTO daily_summary VALUES ('stock','NMIH','us','2017-11-01',14.7,14.75,14.325,14.4,1085210,'0');
INSERT INTO daily_summary VALUES ('stock','NMK_C','us','2017-11-01',104.99,107,104.99,107,65,'0');
INSERT INTO daily_summary VALUES ('stock','NML','us','2017-11-01',8.84,8.95,8.8,8.87,407070,'0');
INSERT INTO daily_summary VALUES ('stock','NMM','us','2017-11-01',2.09,2.1,2.05,2.07,242886,'0');
INSERT INTO daily_summary VALUES ('stock','NMR','us','2017-11-01',5.74,5.76,5.73,5.76,466618,'0');
INSERT INTO daily_summary VALUES ('stock','NMRX','us','2017-11-01',4,4.21,3.97,4.03,31978,'0');
INSERT INTO daily_summary VALUES ('stock','NMS','us','2017-11-01',15.18,15.18,14.99,15.038,6389,'0');
INSERT INTO daily_summary VALUES ('stock','NMT','us','2017-11-01',14.54,14.54,14.305,14.35,7665,'0');
INSERT INTO daily_summary VALUES ('stock','NMY','us','2017-11-01',12.69,12.7,12.6535,12.68,17563,'0');
INSERT INTO daily_summary VALUES ('stock','NMZ','us','2017-11-01',13.53,13.58,13.53,13.53,122721,'0');
INSERT INTO daily_summary VALUES ('stock','NM_H','us','2017-11-01',16.3,16.68,16.3,16.5,955,'0');
INSERT INTO daily_summary VALUES ('stock','NNA','us','2017-11-01',1.26,1.27,1.25,1.25,167031,'0');
INSERT INTO daily_summary VALUES ('stock','NNBR','us','2017-11-01',29.85,29.85,28.625,28.8,95988,'0');
INSERT INTO daily_summary VALUES ('stock','NNC','us','2017-11-01',13.05,13.05,13.01,13.02,71087,'0');
INSERT INTO daily_summary VALUES ('stock','NNDM','us','2017-11-01',4.67,4.7,4.53,4.6608,7490,'0');
INSERT INTO daily_summary VALUES ('stock','NNI','us','2017-11-01',59,59.48,57.61,58.1,60475,'0');
INSERT INTO daily_summary VALUES ('stock','NNN','us','2017-11-01',40.26,40.5,40.02,40.24,745262,'0');
INSERT INTO daily_summary VALUES ('stock','NNN_E','us','2017-11-01',25.96,25.97,25.89,25.97,17775,'0');
INSERT INTO daily_summary VALUES ('stock','NNN_F','us','2017-11-01',25.11,25.11,25.01,25.08,14855,'0');
INSERT INTO daily_summary VALUES ('stock','NNVC','us','2017-11-01',1.04,1.07,1,1.01,103500,'0');
INSERT INTO daily_summary VALUES ('stock','NNY','us','2017-11-01',10.03,10.06,9.97,9.97,13254,'0');
INSERT INTO daily_summary VALUES ('stock','NOA','us','2017-11-01',4.1,4.3,4.1,4.3,64946,'0');
INSERT INTO daily_summary VALUES ('stock','NOAH','us','2017-11-01',39.59,40.1,39.33,39.47,191512,'0');
INSERT INTO daily_summary VALUES ('etf','NOBL','us','2017-11-01',60.25,60.47,60.25,60.25,187802,'0');
INSERT INTO daily_summary VALUES ('stock','NOC','us','2017-11-01',295.67,296.49,293.98,294.17,580848,'0');
INSERT INTO daily_summary VALUES ('stock','NODK','us','2017-11-01',17.94,17.94,17.6947,17.86,9497,'0');
INSERT INTO daily_summary VALUES ('stock','NOG','us','2017-11-01',0.84,0.8899,0.8,0.82,1029750,'0');
INSERT INTO daily_summary VALUES ('stock','NOK','us','2017-11-01',4.92,5,4.91,4.94,15601200,'0');
INSERT INTO daily_summary VALUES ('stock','NOM','us','2017-11-01',16.01,16.3,16.0001,16.0782,3271,'0');
INSERT INTO daily_summary VALUES ('stock','NOMD','us','2017-11-01',15.12,15.3625,15.035,15.15,702688,'0');
INSERT INTO daily_summary VALUES ('etf','NORW','us','2017-11-01',13.63,13.73,13.63,13.65,91894,'0');
INSERT INTO daily_summary VALUES ('stock','NOV','us','2017-11-01',34.49,34.99,33.86,34.06,4365760,'0');
INSERT INTO daily_summary VALUES ('stock','NOVN','us','2017-11-01',6.19,6.42,6.09,6.35,25843,'0');
INSERT INTO daily_summary VALUES ('stock','NOVT','us','2017-11-01',47.65,47.65,46.15,46.55,423381,'0');
INSERT INTO daily_summary VALUES ('stock','NOW','us','2017-11-01',127.2,127.5,123.2,124.51,1137490,'0');
INSERT INTO daily_summary VALUES ('stock','NP','us','2017-11-01',87.5,87.5,85.9,86.75,36275,'0');
INSERT INTO daily_summary VALUES ('stock','NPK','us','2017-11-01',117.25,117.9,115.95,117.4,18589,'0');
INSERT INTO daily_summary VALUES ('stock','NPN','us','2017-11-01',15.31,15.31,15.25,15.25,421,'0');
INSERT INTO daily_summary VALUES ('stock','NPO','us','2017-11-01',84.62,84.979,82.54,83.29,260055,'0');
INSERT INTO daily_summary VALUES ('stock','NPTN','us','2017-11-01',5.25,5.36,5.071,5.22,1430600,'0');
INSERT INTO daily_summary VALUES ('stock','NPV','us','2017-11-01',13.09,13.09,12.94,13.01,44944,'0');
INSERT INTO daily_summary VALUES ('stock','NQ','us','2017-11-01',3.67,3.8,3.62,3.77,394065,'0');
INSERT INTO daily_summary VALUES ('stock','NQP','us','2017-11-01',13.49,13.4997,13.46,13.48,32291,'0');
INSERT INTO daily_summary VALUES ('stock','NR','us','2017-11-01',8.75,8.9,8.6,8.6,538178,'0');
INSERT INTO daily_summary VALUES ('stock','NRCIA','us','2017-11-01',37.45,37.45,35.9501,36.3,32888,'0');
INSERT INTO daily_summary VALUES ('stock','NRCIB','us','2017-11-01',53.38,53.8399,53.38,53.63,1567,'0');
INSERT INTO daily_summary VALUES ('stock','NRE','us','2017-11-01',13.59,13.87,13.42,13.76,294339,'0');
INSERT INTO daily_summary VALUES ('stock','NRG','us','2017-11-01',25.2,25.32,24.645,24.84,7077750,'0');
INSERT INTO daily_summary VALUES ('stock','NRIM','us','2017-11-01',32.9,33.3,31.85,32.65,12342,'0');
INSERT INTO daily_summary VALUES ('stock','NRK','us','2017-11-01',12.96,12.99,12.94,12.98,105084,'0');
INSERT INTO daily_summary VALUES ('stock','NRO','us','2017-11-01',5.53,5.54,5.5,5.53,165462,'0');
INSERT INTO daily_summary VALUES ('stock','NRP','us','2017-11-01',26.902,26.902,26.067,26.067,14865,'0');
INSERT INTO daily_summary VALUES ('stock','NRT','us','2017-11-01',6.74,6.8,6.41,6.545,64000,'0');
INSERT INTO daily_summary VALUES ('stock','NRZ','us','2017-11-01',17.76,17.84,17.62,17.71,2305950,'0');
INSERT INTO daily_summary VALUES ('stock','NS','us','2017-11-01',32.548,32.829,32.113,32.248,576219,'0');
INSERT INTO daily_summary VALUES ('stock','NSA','us','2017-11-01',24.94,24.96,24.61,24.77,162367,'0');
INSERT INTO daily_summary VALUES ('stock','NSA_A','us','2017-11-01',25.3,25.35,25.26,25.34,23621,'0');
INSERT INTO daily_summary VALUES ('stock','NSC','us','2017-11-01',131.84,131.84,130.12,131.23,983322,'0');
INSERT INTO daily_summary VALUES ('stock','NSEC','us','2017-11-01',12.811,12.98,12.811,12.98,458,'0');
INSERT INTO daily_summary VALUES ('stock','NSH','us','2017-11-01',16.872,17.167,16.726,16.872,80862,'0');
INSERT INTO daily_summary VALUES ('stock','NSIT','us','2017-11-01',45.45,45.45,43.99,44.51,116149,'0');
INSERT INTO daily_summary VALUES ('stock','NSL','us','2017-11-01',6.78,6.81,6.74,6.75,84325,'0');
INSERT INTO daily_summary VALUES ('stock','NSM','us','2017-11-01',19.55,19.72,19.16,19.42,686456,'0');
INSERT INTO daily_summary VALUES ('stock','NSP','us','2017-11-01',99.05,99.1,92.35,98.35,233469,'0');
INSERT INTO daily_summary VALUES ('stock','NSPR','us','2017-11-01',0.41,0.4175,0.37,0.3998,193129,'0');
INSERT INTO daily_summary VALUES ('stock','NSS','us','2017-11-01',25.22,25.25,25.18,25.24,24716,'0');
INSERT INTO daily_summary VALUES ('stock','NSSC','us','2017-11-01',10.25,10.3,10.1,10.15,24674,'0');
INSERT INTO daily_summary VALUES ('stock','NSTG','us','2017-11-01',9.97,9.99,9.5575,9.91,210759,'0');
INSERT INTO daily_summary VALUES ('stock','NSU','us','2017-11-01',2.39,2.45,2.37,2.43,831693,'0');
INSERT INTO daily_summary VALUES ('stock','NSYS','us','2017-11-01',4.17,4.17,3.85,3.88,7624,'0');
INSERT INTO daily_summary VALUES ('stock','NS_A','us','2017-11-01',26.28,26.3675,26.2705,26.323,21649,'0');
INSERT INTO daily_summary VALUES ('stock','NS_B','us','2017-11-01',25.45,25.64,25.4472,25.6,27925,'0');
INSERT INTO daily_summary VALUES ('stock','NTAP','us','2017-11-01',44.74,44.9,44.4,44.53,1940600,'0');
INSERT INTO daily_summary VALUES ('stock','NTB','us','2017-11-01',37.337,37.387,36.158,36.594,307115,'0');
INSERT INTO daily_summary VALUES ('stock','NTC','us','2017-11-01',12.02,12.05,12.01,12.03,20146,'0');
INSERT INTO daily_summary VALUES ('stock','NTCT','us','2017-11-01',28.65,29.125,28.1,28.85,1308960,'0');
INSERT INTO daily_summary VALUES ('stock','NTEC','us','2017-11-01',8.35,8.55,8.35,8.5,51949,'0');
INSERT INTO daily_summary VALUES ('stock','NTES','us','2017-11-01',284.08,284.61,275.56,275.97,1537540,'0');
INSERT INTO daily_summary VALUES ('stock','NTG','us','2017-11-01',17,17.59,17,17.47,198855,'0');
INSERT INTO daily_summary VALUES ('stock','NTGR','us','2017-11-01',46.8,46.8,44.2,44.65,820549,'0');
INSERT INTO daily_summary VALUES ('stock','NTIC','us','2017-11-01',16.55,17,16.545,17,1250,'0');
INSERT INTO daily_summary VALUES ('stock','NTIP','us','2017-11-01',4.25,4.4,4.1,4.2,85690,'0');
INSERT INTO daily_summary VALUES ('stock','NTL','us','2017-11-01',41.85,43.39,41.85,43.39,4367,'0');
INSERT INTO daily_summary VALUES ('stock','NTLA','us','2017-11-01',30.7,30.7,25.41,25.47,1677780,'0');
INSERT INTO daily_summary VALUES ('stock','NTN','us','2017-11-01',5.61,5.9148,5.45,5.9148,1545,'0');
INSERT INTO daily_summary VALUES ('stock','NTNX','us','2017-11-01',28.78,29,27.54,27.58,3090260,'0');
INSERT INTO daily_summary VALUES ('stock','NTP','us','2017-11-01',12.15,12.2,12,12.1,43859,'0');
INSERT INTO daily_summary VALUES ('stock','NTRA','us','2017-11-01',11.17,11.17,10.3,10.57,203574,'0');
INSERT INTO daily_summary VALUES ('stock','NTRI','us','2017-11-01',50.228,50.726,48.284,48.633,629271,'0');
INSERT INTO daily_summary VALUES ('stock','NTRP','us','2017-11-01',4.475,4.475,4.3254,4.38,2200,'0');
INSERT INTO daily_summary VALUES ('stock','NTRS','us','2017-11-01',93.93,94.65,93.38,93.41,859631,'0');
INSERT INTO daily_summary VALUES ('stock','NTRSP','us','2017-11-01',26.47,26.53,26.4499,26.46,8551,'0');
INSERT INTO daily_summary VALUES ('stock','NTWK','us','2017-11-01',3.2064,3.3,3.2,3.25,9204,'0');
INSERT INTO daily_summary VALUES ('stock','NTX','us','2017-11-01',14.4,14.6212,14.3584,14.5085,9546,'0');
INSERT INTO daily_summary VALUES ('stock','NTZ','us','2017-11-01',2.041,2.13,2.03,2.08,3327,'0');
INSERT INTO daily_summary VALUES ('stock','NUAG','us','2017-11-01',24.59,24.6,24.58,24.59,6577,'0');
INSERT INTO daily_summary VALUES ('stock','NUAN','us','2017-11-01',14.75,15.05,14.64,14.75,2397270,'0');
INSERT INTO daily_summary VALUES ('stock','NUDM','us','2017-11-01',26.7599,26.7599,26.64,26.64,10380,'0');
INSERT INTO daily_summary VALUES ('stock','NUE','us','2017-11-01',58.68,59.96,58.46,58.8,2238110,'0');
INSERT INTO daily_summary VALUES ('stock','NUEM','us','2017-11-01',28.16,28.19,28.16,28.19,12260,'0');
INSERT INTO daily_summary VALUES ('etf','NUGT','us','2017-11-01',29.54,30.34,28.7875,29.15,6312200,'0');
INSERT INTO daily_summary VALUES ('stock','NULG','us','2017-11-01',30.607,30.607,30.47,30.47,10890,'0');
INSERT INTO daily_summary VALUES ('stock','NULV','us','2017-11-01',27.97,27.97,27.872,27.91,10562,'0');
INSERT INTO daily_summary VALUES ('stock','NUM','us','2017-11-01',13.44,13.44,13.38,13.42,16642,'0');
INSERT INTO daily_summary VALUES ('stock','NUMG','us','2017-11-01',28.5957,28.5957,28.3868,28.41,11265,'0');
INSERT INTO daily_summary VALUES ('stock','NUMV','us','2017-11-01',26.98,27.02,26.98,27.02,10191,'0');
INSERT INTO daily_summary VALUES ('stock','NUO','us','2017-11-01',14.73,14.79,14.73,14.76,35743,'0');
INSERT INTO daily_summary VALUES ('stock','NURE','us','2017-11-01',26.1678,26.1678,26.1678,26.1678,494,'0');
INSERT INTO daily_summary VALUES ('stock','NURO','us','2017-11-01',1.87,1.94,1.81,1.82,11395,'0');
INSERT INTO daily_summary VALUES ('stock','NUS','us','2017-11-01',64.16,64.62,63.07,63.85,587464,'0');
INSERT INTO daily_summary VALUES ('stock','NUSA','us','2017-11-01',24.95,24.96,24.935,24.96,1437,'0');
INSERT INTO daily_summary VALUES ('stock','NUSC','us','2017-11-01',27.85,27.85,27.5862,27.725,11771,'0');
INSERT INTO daily_summary VALUES ('stock','NUV','us','2017-11-01',10.15,10.17,10.12,10.16,195375,'0');
INSERT INTO daily_summary VALUES ('stock','NUVA','us','2017-11-01',56.7,57.51,55.87,56.22,1125790,'0');
INSERT INTO daily_summary VALUES ('stock','NUW','us','2017-11-01',17.18,17.23,17.17,17.18,17434,'0');
INSERT INTO daily_summary VALUES ('stock','NVAX','us','2017-11-01',1.09,1.14,1.09,1.11,3700260,'0');
INSERT INTO daily_summary VALUES ('stock','NVCN','us','2017-11-01',1.47,1.47,1.42,1.45,47034,'0');
INSERT INTO daily_summary VALUES ('stock','NVCR','us','2017-11-01',21.75,22.1075,20.8,21.5,1124140,'0');
INSERT INTO daily_summary VALUES ('stock','NVDA','us','2017-11-01',209.35,209.97,204.6,207.2,12564400,'0');
INSERT INTO daily_summary VALUES ('stock','NVEC','us','2017-11-01',85.62,85.63,82.4501,83.9,22047,'0');
INSERT INTO daily_summary VALUES ('stock','NVEE','us','2017-11-01',58.65,58.725,56.46,56.6,71901,'0');
INSERT INTO daily_summary VALUES ('stock','NVFY','us','2017-11-01',1.6098,1.6159,1.55,1.5625,160567,'0');
INSERT INTO daily_summary VALUES ('stock','NVG','us','2017-11-01',15.2,15.2,15.13,15.15,399546,'0');
INSERT INTO daily_summary VALUES ('stock','NVGN','us','2017-11-01',2.92,3,2.92,3,6707,'0');
INSERT INTO daily_summary VALUES ('stock','NVGS','us','2017-11-01',10.25,10.4,10.05,10.2,133566,'0');
INSERT INTO daily_summary VALUES ('stock','NVIV','us','2017-11-01',1.3,1.3426,1.25,1.3,128612,'0');
INSERT INTO daily_summary VALUES ('stock','NVLN','us','2017-11-01',4.78,4.8,4.58,4.71,43715,'0');
INSERT INTO daily_summary VALUES ('stock','NVMI','us','2017-11-01',29.23,29.23,23.38,25.77,1931800,'0');
INSERT INTO daily_summary VALUES ('stock','NVO','us','2017-11-01',49.46,50.03,49.25,49.69,1743930,'0');
INSERT INTO daily_summary VALUES ('stock','NVR','us','2017-11-01',3300,3300,3240,3262.29,25759,'0');
INSERT INTO daily_summary VALUES ('stock','NVRO','us','2017-11-01',88.36,89.63,86.78,88.99,278298,'0');
INSERT INTO daily_summary VALUES ('stock','NVS','us','2017-11-01',82.8,83.66,82.68,82.77,1946600,'0');
INSERT INTO daily_summary VALUES ('stock','NVTA','us','2017-11-01',8.5,8.53,8.26,8.44,348966,'0');
INSERT INTO daily_summary VALUES ('stock','NVTR','us','2017-11-01',14.21,14.21,13.3733,13.8,71110,'0');
INSERT INTO daily_summary VALUES ('stock','NVUS','us','2017-11-01',4.01,4.1,3.82,4.09,24594,'0');
INSERT INTO daily_summary VALUES ('stock','NWBI','us','2017-11-01',16.85,16.9,16.5,16.57,439036,'0');
INSERT INTO daily_summary VALUES ('stock','NWE','us','2017-11-01',59.34,59.34,58.46,58.87,408207,'0');
INSERT INTO daily_summary VALUES ('stock','NWFL','us','2017-11-01',30.19,30.19,29.5,29.73,15044,'0');
INSERT INTO daily_summary VALUES ('stock','NWHM','us','2017-11-01',11.71,11.71,11.37,11.44,133943,'0');
INSERT INTO daily_summary VALUES ('stock','NWL','us','2017-11-01',40.89,41.31,40.71,41,8572130,'0');
INSERT INTO daily_summary VALUES ('stock','NWLI','us','2017-11-01',358.93,358.93,357.31,357.33,1852,'0');
INSERT INTO daily_summary VALUES ('stock','NWN','us','2017-11-01',66.55,66.6,65.15,65.75,89437,'0');
INSERT INTO daily_summary VALUES ('stock','NWPX','us','2017-11-01',18.28,18.66,17.491,17.83,50995,'0');
INSERT INTO daily_summary VALUES ('stock','NWS','us','2017-11-01',14,14.2,13.9,14.05,447303,'0');
INSERT INTO daily_summary VALUES ('stock','NWSA','us','2017-11-01',13.73,13.8652,13.61,13.71,1900440,'0');
INSERT INTO daily_summary VALUES ('stock','NWY','us','2017-11-01',1.76,1.78,1.62,1.62,66164,'0');
INSERT INTO daily_summary VALUES ('stock','NX','us','2017-11-01',22.2,22.2,21.6,21.85,141163,'0');
INSERT INTO daily_summary VALUES ('stock','NXC','us','2017-11-01',15.34,15.34,15.21,15.21,4019,'0');
INSERT INTO daily_summary VALUES ('stock','NXE','us','2017-11-01',1.96,1.99,1.85,1.99,529052,'0');
INSERT INTO daily_summary VALUES ('stock','NXEO','us','2017-11-01',7.46,7.54,7.38,7.42,76769,'0');
INSERT INTO daily_summary VALUES ('stock','NXEOU','us','2017-11-01',7.97,7.97,7.97,7.97,111,'0');
INSERT INTO daily_summary VALUES ('stock','NXEOW','us','2017-11-01',0.52,0.52,0.45,0.52,40477,'0');
INSERT INTO daily_summary VALUES ('stock','NXJ','us','2017-11-01',13.65,13.66,13.6128,13.64,45740,'0');
INSERT INTO daily_summary VALUES ('stock','NXN','us','2017-11-01',13.61,13.63,13.6,13.62,4390,'0');
INSERT INTO daily_summary VALUES ('stock','NXP','us','2017-11-01',14.67,14.7423,14.65,14.74,28371,'0');
INSERT INTO daily_summary VALUES ('stock','NXPI','us','2017-11-01',117.05,117.73,116.885,117.3,2902890,'0');
INSERT INTO daily_summary VALUES ('stock','NXQ','us','2017-11-01',13.99,14.05,13.95,14.05,29168,'0');
INSERT INTO daily_summary VALUES ('stock','NXR','us','2017-11-01',14.9,14.93,14.8,14.82,22621,'0');
INSERT INTO daily_summary VALUES ('stock','NXRT','us','2017-11-01',23.82,25.16,23.82,25.07,119825,'0');
INSERT INTO daily_summary VALUES ('stock','NXST','us','2017-11-01',63.8,64.25,63.05,64.2,722510,'0');
INSERT INTO daily_summary VALUES ('stock','NXTD','us','2017-11-01',1.93,1.93,1.65,1.69,146637,'0');
INSERT INTO daily_summary VALUES ('stock','NXTDW','us','2017-11-01',0.16,0.161,0.16,0.161,1197,'0');
INSERT INTO daily_summary VALUES ('stock','NXTM','us','2017-11-01',26.84,27.02,26.49,26.83,464270,'0');
INSERT INTO daily_summary VALUES ('stock','NYCB','us','2017-11-01',12.43,12.529,12.282,12.321,2858150,'0');
INSERT INTO daily_summary VALUES ('stock','NYCB_A','us','2017-11-01',28.04,28.25,27.89,27.94,10392,'0');
INSERT INTO daily_summary VALUES ('stock','NYCB_U','us','2017-11-01',50.06,50.3876,50.06,50.2843,1824,'0');
INSERT INTO daily_summary VALUES ('etf','NYF','us','2017-11-01',55.81,55.84,55.78,55.81,14427,'0');
INSERT INTO daily_summary VALUES ('stock','NYH','us','2017-11-01',11.72,11.76,11.66,11.7,4050,'0');
INSERT INTO daily_summary VALUES ('stock','NYLD','us','2017-11-01',18.7,18.85,18.45,18.5,478436,'0');
INSERT INTO daily_summary VALUES ('stock','NYMT','us','2017-11-01',6.05,6.09,6.03,6.05,626471,'0');
INSERT INTO daily_summary VALUES ('stock','NYMTN','us','2017-11-01',24.88,24.97,24.78,24.9,21729,'0');
INSERT INTO daily_summary VALUES ('stock','NYMTO','us','2017-11-01',24.95,24.95,24.81,24.8527,8040,'0');
INSERT INTO daily_summary VALUES ('stock','NYMTP','us','2017-11-01',24.85,24.85,24.68,24.7204,2672,'0');
INSERT INTO daily_summary VALUES ('stock','NYMX','us','2017-11-01',3.6,3.6,3.41,3.43,82080,'0');
INSERT INTO daily_summary VALUES ('stock','NYNY','us','2017-11-01',22.4,22.4,22.05,22.25,2366,'0');
INSERT INTO daily_summary VALUES ('stock','NYRT','us','2017-11-01',7.53,7.57,7.51,7.55,1157090,'0');
INSERT INTO daily_summary VALUES ('stock','NYT','us','2017-11-01',19.8,19.8,17.9,18,3836470,'0');
INSERT INTO daily_summary VALUES ('stock','NYV','us','2017-11-01',14.64,14.64,14.63,14.63,611,'0');
INSERT INTO daily_summary VALUES ('stock','NZF','us','2017-11-01',15.03,15.04,14.99,15.01,308567,'0');
INSERT INTO daily_summary VALUES ('stock','O','us','2017-11-01',53.76,54.23,53.39,53.93,1252460,'0');
INSERT INTO daily_summary VALUES ('stock','OA','us','2017-11-01',133.1,133.19,132.74,132.77,799841,'0');
INSERT INTO daily_summary VALUES ('stock','OACQ','us','2017-11-01',10.65,10.65,10.625,10.645,6800,'0');
INSERT INTO daily_summary VALUES ('stock','OACQR','us','2017-11-01',0.4799,0.549,0.4799,0.5419,21000,'0');
INSERT INTO daily_summary VALUES ('stock','OAK','us','2017-11-01',45.278,45.624,45.179,45.377,270961,'0');
INSERT INTO daily_summary VALUES ('stock','OAKS','us','2017-11-01',4.07,4.18,4.06,4.14,65852,'0');
INSERT INTO daily_summary VALUES ('stock','OAKS_A','us','2017-11-01',25.1772,25.27,25.1704,25.27,13288,'0');
INSERT INTO daily_summary VALUES ('stock','OAPH','us','2017-11-01',32.51,32.54,30.5,31,3457,'0');
INSERT INTO daily_summary VALUES ('stock','OAS','us','2017-11-01',9.66,10.27,9.66,10.16,14776600,'0');
INSERT INTO daily_summary VALUES ('stock','OASI','us','2017-11-01',29.97,30,29.25,29.25,5141,'0');
INSERT INTO daily_summary VALUES ('stock','OASM','us','2017-11-01',1.05,1.05,0.9687,1.05,1000,'0');
INSERT INTO daily_summary VALUES ('stock','OBCI','us','2017-11-01',4.53,4.54,4.42,4.45,2400,'0');
INSERT INTO daily_summary VALUES ('stock','OBE','us','2017-11-01',1.1,1.165,1.08,1.13,2530190,'0');
INSERT INTO daily_summary VALUES ('stock','OBLN','us','2017-11-01',8.42,8.42,8.3,8.37,23503,'0');
INSERT INTO daily_summary VALUES ('stock','OBOR','us','2017-11-01',25.44,25.5,25.425,25.425,3112,'0');
INSERT INTO daily_summary VALUES ('stock','OBSV','us','2017-11-01',10.0001,10.8265,10.0001,10.66,4763,'0');
INSERT INTO daily_summary VALUES ('stock','OC','us','2017-11-01',83,83.1,81.93,82.29,1504530,'0');
INSERT INTO daily_summary VALUES ('stock','OCC','us','2017-11-01',2.45,2.5,2.45,2.5,1183,'0');
INSERT INTO daily_summary VALUES ('stock','OCFC','us','2017-11-01',27.647,27.856,26.841,27.03,106784,'0');
INSERT INTO daily_summary VALUES ('stock','OCIO','us','2017-11-01',26.17,26.21,26.15,26.17,3186,'0');
INSERT INTO daily_summary VALUES ('stock','OCIP','us','2017-11-01',6.95,7,6.95,7,1100,'0');
INSERT INTO daily_summary VALUES ('stock','OCLR','us','2017-11-01',8.3,8.34,8.02,8.02,7895670,'0');
INSERT INTO daily_summary VALUES ('stock','OCN','us','2017-11-01',3.52,3.54,3.39,3.43,1056140,'0');
INSERT INTO daily_summary VALUES ('stock','OCRX','us','2017-11-01',1,1.02,0.9836,1,82295,'0');
INSERT INTO daily_summary VALUES ('stock','OCSI','us','2017-11-01',8.94,8.949,8.8,8.87,33718,'0');
INSERT INTO daily_summary VALUES ('stock','OCSL','us','2017-11-01',5.8,5.8,5.72,5.76,418270,'0');
INSERT INTO daily_summary VALUES ('stock','OCSLL','us','2017-11-01',25.39,25.39,25.26,25.26,550,'0');
INSERT INTO daily_summary VALUES ('stock','OCUL','us','2017-11-01',5.81,5.81,5.55,5.63,272575,'0');
INSERT INTO daily_summary VALUES ('stock','OCX','us','2017-11-01',6,6.025,6,6,10522,'0');
INSERT INTO daily_summary VALUES ('stock','ODC','us','2017-11-01',42.24,42.97,41.524,42.35,8044,'0');
INSERT INTO daily_summary VALUES ('stock','ODFL','us','2017-11-01',121.98,122.59,119.45,119.97,579588,'0');
INSERT INTO daily_summary VALUES ('stock','ODP','us','2017-11-01',3.11,3.16,3.08,3.12,6688340,'0');
INSERT INTO daily_summary VALUES ('stock','OEC','us','2017-11-01',23.85,23.85,23.5,23.6,230797,'0');
INSERT INTO daily_summary VALUES ('etf','OEF','us','2017-11-01',114.35,114.58,114.08,114.28,258644,'0');
INSERT INTO daily_summary VALUES ('stock','OESX','us','2017-11-01',0.8502,0.91,0.8502,0.86,132103,'0');
INSERT INTO daily_summary VALUES ('stock','OEUH','us','2017-11-01',26.2505,27.5,26.1941,27.449,3708,'0');
INSERT INTO daily_summary VALUES ('stock','OEUR','us','2017-11-01',25.29,25.36,25.2501,25.32,14196,'0');
INSERT INTO daily_summary VALUES ('stock','OFC','us','2017-11-01',32.04,32.79,31.94,32.69,1125470,'0');
INSERT INTO daily_summary VALUES ('stock','OFED','us','2017-11-01',29.898,29.898,29.4,29.4,753,'0');
INSERT INTO daily_summary VALUES ('stock','OFG','us','2017-11-01',8.9,9,8.45,8.5,332764,'0');
INSERT INTO daily_summary VALUES ('stock','OFG_A','us','2017-11-01',24.09,24.1399,23.88,23.91,15200,'0');
INSERT INTO daily_summary VALUES ('stock','OFG_B','us','2017-11-01',22.9951,23.16,22.925,23.1,14281,'0');
INSERT INTO daily_summary VALUES ('stock','OFG_D','us','2017-11-01',23.39,23.62,23.39,23.4699,5130,'0');
INSERT INTO daily_summary VALUES ('stock','OFIX','us','2017-11-01',53.8,54.13,52.3,52.43,301546,'0');
INSERT INTO daily_summary VALUES ('stock','OFLX','us','2017-11-01',64.6,66.75,64.01,66.2,32942,'0');
INSERT INTO daily_summary VALUES ('stock','OFS','us','2017-11-01',13.04,13.08,12.85,12.9,29705,'0');
INSERT INTO daily_summary VALUES ('stock','OGCP','us','2017-11-01',19.91,19.97,19.8524,19.97,1200,'0');
INSERT INTO daily_summary VALUES ('stock','OGE','us','2017-11-01',36.94,37.02,36.6,36.6,879141,'0');
INSERT INTO daily_summary VALUES ('stock','OGEN','us','2017-11-01',0.3621,0.3621,0.3201,0.3213,42924,'0');
INSERT INTO daily_summary VALUES ('stock','OGS','us','2017-11-01',76.907,77.016,75.475,75.873,158183,'0');
INSERT INTO daily_summary VALUES ('stock','OHAI','us','2017-11-01',1.179,1.18,1.12,1.17,25209,'0');
INSERT INTO daily_summary VALUES ('stock','OHGI','us','2017-11-01',1.2,1.22,1.1401,1.15,23420,'0');
INSERT INTO daily_summary VALUES ('stock','OHI','us','2017-11-01',28.48,28.59,28.02,28.06,8039460,'0');
INSERT INTO daily_summary VALUES ('stock','OHRP','us','2017-11-01',0.64,0.64,0.6,0.61,114064,'0');
INSERT INTO daily_summary VALUES ('stock','OI','us','2017-11-01',24.24,24.44,23.9,24.05,1208180,'0');
INSERT INTO daily_summary VALUES ('stock','OIA','us','2017-11-01',7.83,7.86,7.82,7.84,74094,'0');
INSERT INTO daily_summary VALUES ('etf','OIH','us','2017-11-01',24.59,24.96,24.35,24.48,7547940,'0');
INSERT INTO daily_summary VALUES ('stock','OII','us','2017-11-01',20.29,20.76,19.95,20.15,2368780,'0');
INSERT INTO daily_summary VALUES ('stock','OIIL','us','2017-11-01',32.48,32.48,32.12,32.13,1452,'0');
INSERT INTO daily_summary VALUES ('stock','OIIM','us','2017-11-01',2.03,2.03,1.666,1.82,77080,'0');
INSERT INTO daily_summary VALUES ('etf','OIL','us','2017-11-01',5.82,5.84,5.6788,5.72,2130390,'0');
INSERT INTO daily_summary VALUES ('stock','OILB','us','2017-11-01',53.96,53.96,53.24,53.24,4764,'0');
INSERT INTO daily_summary VALUES ('stock','OILD','us','2017-11-01',14.54,15.37,14.43,15.05,474297,'0');
INSERT INTO daily_summary VALUES ('stock','OILK','us','2017-11-01',21.07,21.07,20.66,20.8,1743,'0');
INSERT INTO daily_summary VALUES ('stock','OILU','us','2017-11-01',29.3,29.5,27.74,28.33,126993,'0');
INSERT INTO daily_summary VALUES ('stock','OIS','us','2017-11-01',23.2,23.95,22.55,22.75,1378990,'0');
INSERT INTO daily_summary VALUES ('stock','OKE','us','2017-11-01',53.985,54.971,53.206,53.769,2763080,'0');
INSERT INTO daily_summary VALUES ('stock','OKTA','us','2017-11-01',29,29.16,28.675,28.97,924673,'0');
INSERT INTO daily_summary VALUES ('stock','OLBK','us','2017-11-01',30.23,30.26,29.44,29.83,20856,'0');
INSERT INTO daily_summary VALUES ('stock','OLD','us','2017-11-01',24.33,24.39,24.33,24.39,1732,'0');
INSERT INTO daily_summary VALUES ('stock','OLED','us','2017-11-01',148.5,148.95,141.3,142.7,1063680,'0');
INSERT INTO daily_summary VALUES ('etf','OLEM','us','2017-11-01',16.24,16.24,16.165,16.17,742,'0');
INSERT INTO daily_summary VALUES ('stock','OLLI','us','2017-11-01',45.35,45.6,43.45,43.9,358867,'0');
INSERT INTO daily_summary VALUES ('stock','OLN','us','2017-11-01',36.749,37.316,35.879,36.003,2023090,'0');
INSERT INTO daily_summary VALUES ('etf','OLO','us','2017-11-01',4.59,4.59,4.59,4.59,511,'0');
INSERT INTO daily_summary VALUES ('stock','OLP','us','2017-11-01',24.3,24.35,23.99,24.3,21435,'0');
INSERT INTO daily_summary VALUES ('stock','OMAA','us','2017-11-01',24.95,24.95,24.83,24.8301,10165,'0');
INSERT INTO daily_summary VALUES ('stock','OMAB','us','2017-11-01',40.39,40.52,39.45,39.96,76742,'0');
INSERT INTO daily_summary VALUES ('stock','OMAM','us','2017-11-01',15.41,15.58,15.28,15.39,540923,'0');
INSERT INTO daily_summary VALUES ('stock','OMC','us','2017-11-01',67.68,68.715,67.37,67.63,2368440,'0');
INSERT INTO daily_summary VALUES ('stock','OMCL','us','2017-11-01',49.95,49.95,47.95,49.05,313772,'0');
INSERT INTO daily_summary VALUES ('stock','OME','us','2017-11-01',21.95,21.95,21.9,21.95,322689,'0');
INSERT INTO daily_summary VALUES ('stock','OMED','us','2017-11-01',4.03,4.2,3.95,4.19,161955,'0');
INSERT INTO daily_summary VALUES ('stock','OMER','us','2017-11-01',15.9,16.24,15.34,15.39,467040,'0');
INSERT INTO daily_summary VALUES ('stock','OMEX','us','2017-11-01',3.92,3.9249,3.84,3.84,1524,'0');
INSERT INTO daily_summary VALUES ('stock','OMF','us','2017-11-01',31.81,32.19,31.05,31.17,1063300,'0');
INSERT INTO daily_summary VALUES ('stock','OMI','us','2017-11-01',24.43,24.57,20.03,20.96,4559210,'0');
INSERT INTO daily_summary VALUES ('stock','OMN','us','2017-11-01',11.15,11.4,10.9,11.35,157931,'0');
INSERT INTO daily_summary VALUES ('stock','OMNT','us','2017-11-01',3.21,3.285,3.12,3.16,24785,'0');
INSERT INTO daily_summary VALUES ('stock','OMP','us','2017-11-01',16.64,17,16.53,16.95,38344,'0');
INSERT INTO daily_summary VALUES ('stock','ON','us','2017-11-01',21.69,21.769,20.84,21.17,9163550,'0');
INSERT INTO daily_summary VALUES ('stock','ONB','us','2017-11-01',18.4,18.45,18,18.05,617058,'0');
INSERT INTO daily_summary VALUES ('stock','ONCE','us','2017-11-01',81.82,83.35,78.01,78.56,513285,'0');
INSERT INTO daily_summary VALUES ('stock','ONCS','us','2017-11-01',1.16,1.21,1.15,1.19,2367220,'0');
INSERT INTO daily_summary VALUES ('stock','ONDK','us','2017-11-01',4.99,4.99,4.81,4.9,276204,'0');
INSERT INTO daily_summary VALUES ('stock','ONEO','us','2017-11-01',74.0724,74.24,73.72,73.9,8941,'0');
INSERT INTO daily_summary VALUES ('etf','ONEQ','us','2017-11-01',266.14,266.14,263.2,264.17,40344,'0');
INSERT INTO daily_summary VALUES ('stock','ONEV','us','2017-11-01',72.88,72.88,72.79,72.79,2601,'0');
INSERT INTO daily_summary VALUES ('stock','ONP','us','2017-11-01',0.86,0.87,0.84,0.87,21174,'0');
INSERT INTO daily_summary VALUES ('stock','ONS','us','2017-11-01',1.2,1.215,1.0601,1.14,185759,'0');
INSERT INTO daily_summary VALUES ('stock','ONSIW','us','2017-11-01',0.08,0.08,0.079,0.079,8390,'0');
INSERT INTO daily_summary VALUES ('stock','ONTL','us','2017-11-01',27.4947,27.53,27.46,27.51,7067,'0');
INSERT INTO daily_summary VALUES ('stock','ONTX','us','2017-11-01',1.99,1.99,1.82,1.86,81069,'0');
INSERT INTO daily_summary VALUES ('stock','ONVI','us','2017-11-01',8.975,8.975,8.975,8.975,724,'0');
INSERT INTO daily_summary VALUES ('stock','ONVO','us','2017-11-01',1.64,1.64,1.55,1.56,674289,'0');
INSERT INTO daily_summary VALUES ('stock','OOMA','us','2017-11-01',10.5,10.65,10.2,10.3,33022,'0');
INSERT INTO daily_summary VALUES ('stock','OPB','us','2017-11-01',25.95,26.15,25.4,25.85,201171,'0');
INSERT INTO daily_summary VALUES ('stock','OPGN','us','2017-11-01',0.28,0.2899,0.271,0.2847,1006300,'0');
INSERT INTO daily_summary VALUES ('stock','OPGNW','us','2017-11-01',0.04,0.04,0.04,0.04,5112,'0');
INSERT INTO daily_summary VALUES ('stock','OPHT','us','2017-11-01',2.33,2.41,2.27,2.31,327589,'0');
INSERT INTO daily_summary VALUES ('stock','OPK','us','2017-11-01',6.77,6.8,6.52,6.52,4052250,'0');
INSERT INTO daily_summary VALUES ('stock','OPNT','us','2017-11-01',39.49,40,37.1948,38.89,54146,'0');
INSERT INTO daily_summary VALUES ('stock','OPOF','us','2017-11-01',32.06,32.6895,31.86,31.86,2975,'0');
INSERT INTO daily_summary VALUES ('stock','OPP','us','2017-11-01',18.9708,19.0799,18.84,18.91,33350,'0');
INSERT INTO daily_summary VALUES ('stock','OPTN','us','2017-11-01',20.27,20.27,19.25,19.71,177892,'0');
INSERT INTO daily_summary VALUES ('stock','OPTT','us','2017-11-01',1.23,1.3,1.22,1.23,525551,'0');
INSERT INTO daily_summary VALUES ('stock','OPY','us','2017-11-01',21.795,21.994,21.546,21.696,88859,'0');
INSERT INTO daily_summary VALUES ('stock','OR','us','2017-11-01',12.6,12.78,12.58,12.72,841393,'0');
INSERT INTO daily_summary VALUES ('stock','ORA','us','2017-11-01',65.54,65.77,64.2,64.24,135451,'0');
INSERT INTO daily_summary VALUES ('stock','ORAN','us','2017-11-01',16.48,16.5,16.42,16.47,212307,'0');
INSERT INTO daily_summary VALUES ('stock','ORBC','us','2017-11-01',11.42,11.61,11.25,11.36,682029,'0');
INSERT INTO daily_summary VALUES ('stock','ORBK','us','2017-11-01',46.53,47.62,46.2,46.47,805823,'0');
INSERT INTO daily_summary VALUES ('stock','ORC','us','2017-11-01',9.98,10.06,9.82,9.89,1157430,'0');
INSERT INTO daily_summary VALUES ('stock','ORCL','us','2017-11-01',51.1,51.17,50.57,50.64,10127900,'0');
INSERT INTO daily_summary VALUES ('stock','OREX','us','2017-11-01',1.62,1.7,1.61,1.64,41256,'0');
INSERT INTO daily_summary VALUES ('stock','ORG','us','2017-11-01',31.64,31.851,31.64,31.851,543,'0');
INSERT INTO daily_summary VALUES ('stock','ORI','us','2017-11-01',20.39,20.45,20.23,20.3,1239920,'0');
INSERT INTO daily_summary VALUES ('stock','ORIG','us','2017-11-01',26.92,29.18,26.685,26.99,365963,'0');
INSERT INTO daily_summary VALUES ('stock','ORIT','us','2017-11-01',16.779,16.928,16.483,16.582,230421,'0');
INSERT INTO daily_summary VALUES ('stock','ORLY','us','2017-11-01',211.24,212.719,205.65,207.56,972142,'0');
INSERT INTO daily_summary VALUES ('stock','ORM','us','2017-11-01',17.91,17.98,17.8,17.84,21915,'0');
INSERT INTO daily_summary VALUES ('stock','ORMP','us','2017-11-01',9.96,9.96,9.475,9.48,34222,'0');
INSERT INTO daily_summary VALUES ('stock','ORN','us','2017-11-01',7.31,7.31,6.88,7,98516,'0');
INSERT INTO daily_summary VALUES ('stock','ORPN','us','2017-11-01',2.5,2.59,2.22,2.27,83785,'0');
INSERT INTO daily_summary VALUES ('stock','ORRF','us','2017-11-01',25.531,26.277,25.382,25.531,9240,'0');
INSERT INTO daily_summary VALUES ('stock','OSB','us','2017-11-01',36.49,36.49,35.66,35.93,202677,'0');
INSERT INTO daily_summary VALUES ('stock','OSBC','us','2017-11-01',13.8,13.9,13.39,13.5,44391,'0');
INSERT INTO daily_summary VALUES ('stock','OSBCP','us','2017-11-01',10.46,10.4639,10.46,10.46,1250,'0');
INSERT INTO daily_summary VALUES ('stock','OSG','us','2017-11-01',2.37,2.42,2.26,2.3,332488,'0');
INSERT INTO daily_summary VALUES ('stock','OSIS','us','2017-11-01',88.94,89.43,87.82,88.58,203508,'0');
INSERT INTO daily_summary VALUES ('stock','OSK','us','2017-11-01',92.8,93.3,86.49,87.66,1473260,'0');
INSERT INTO daily_summary VALUES ('stock','OSLE','us','2017-11-01',25.15,25.19,25.15,25.1759,3997,'0');
INSERT INTO daily_summary VALUES ('stock','OSN','us','2017-11-01',1.86,1.89,1.845,1.89,13012,'0');
INSERT INTO daily_summary VALUES ('stock','OSTK','us','2017-11-01',46.8,48.249,44.4501,46.15,3131260,'0');
INSERT INTO daily_summary VALUES ('stock','OSUR','us','2017-11-01',19.82,20,18.93,19.02,1483540,'0');
INSERT INTO daily_summary VALUES ('stock','OTEL','us','2017-11-01',10.05,10.25,10.05,10.25,3153,'0');
INSERT INTO daily_summary VALUES ('stock','OTEX','us','2017-11-01',35.07,35.1,34.05,34.26,595357,'0');
INSERT INTO daily_summary VALUES ('stock','OTIC','us','2017-11-01',2.95,3,2.9,2.9,261537,'0');
INSERT INTO daily_summary VALUES ('stock','OTIV','us','2017-11-01',1.2799,1.2799,1.221,1.279,72888,'0');
INSERT INTO daily_summary VALUES ('stock','OTTR','us','2017-11-01',46.2,46.2787,45.0001,45.7,69187,'0');
INSERT INTO daily_summary VALUES ('stock','OTTW','us','2017-11-01',13.97,14.1,13.97,14.02,12678,'0');
INSERT INTO daily_summary VALUES ('etf','OUNZ','us','2017-11-01',12.56,12.62,12.55,12.57,96011,'0');
INSERT INTO daily_summary VALUES ('stock','OUSA','us','2017-11-01',30.48,30.54,30.4289,30.5,39897,'0');
INSERT INTO daily_summary VALUES ('stock','OUSM','us','2017-11-01',26.6295,26.64,26.4,26.455,25056,'0');
INSERT INTO daily_summary VALUES ('stock','OUT','us','2017-11-01',23.56,23.75,23.41,23.44,1282660,'0');
INSERT INTO daily_summary VALUES ('stock','OVAS','us','2017-11-01',1.55,1.55,1.45,1.53,185600,'0');
INSERT INTO daily_summary VALUES ('stock','OVBC','us','2017-11-01',35.7,35.7,35,35.15,846,'0');
INSERT INTO daily_summary VALUES ('stock','OVID','us','2017-11-01',6.66,6.7,6.2092,6.51,28153,'0');
INSERT INTO daily_summary VALUES ('stock','OVLY','us','2017-11-01',16.75,16.895,16.75,16.895,1073,'0');
INSERT INTO daily_summary VALUES ('stock','OXBR','us','2017-11-01',3.105,3.2,3.05,3.1,6055,'0');
INSERT INTO daily_summary VALUES ('stock','OXBRW','us','2017-11-01',0.2908,0.32,0.2908,0.2908,2592,'0');
INSERT INTO daily_summary VALUES ('stock','OXFD','us','2017-11-01',13.35,13.98,13.18,13.25,882215,'0');
INSERT INTO daily_summary VALUES ('stock','OXLC','us','2017-11-01',10.92,10.95,10.86,10.94,93663,'0');
INSERT INTO daily_summary VALUES ('stock','OXLCM','us','2017-11-01',25.393,25.393,25.393,25.393,394,'0');
INSERT INTO daily_summary VALUES ('stock','OXLCO','us','2017-11-01',25.374,25.374,25.36,25.36,748,'0');
INSERT INTO daily_summary VALUES ('stock','OXM','us','2017-11-01',64.86,65.01,63.23,64.04,82112,'0');
INSERT INTO daily_summary VALUES ('stock','OXY','us','2017-11-01',65.16,65.785,65.07,65.46,6184180,'0');
INSERT INTO daily_summary VALUES ('stock','OZM','us','2017-11-01',3.7761,3.8059,3.6767,3.786,367658,'0');
INSERT INTO daily_summary VALUES ('stock','OZRK','us','2017-11-01',46.91,47.4,45.935,46.09,530483,'0');
INSERT INTO daily_summary VALUES ('stock','P','us','2017-11-01',9.35,9.46,9.28,9.35,2628520,'0');
INSERT INTO daily_summary VALUES ('stock','PAA','us','2017-11-01',19.78,20.73,19.68,20.6,3136530,'0');
INSERT INTO daily_summary VALUES ('stock','PAAS','us','2017-11-01',16.54,16.56,16.2899,16.32,1101750,'0');
INSERT INTO daily_summary VALUES ('stock','PAC','us','2017-11-01',94.444,94.622,93.498,93.508,75155,'0');
INSERT INTO daily_summary VALUES ('stock','PACB','us','2017-11-01',4.27,4.3,4.15,4.16,463265,'0');
INSERT INTO daily_summary VALUES ('stock','PACW','us','2017-11-01',47.9,48.75,46.99,47.54,1325760,'0');
INSERT INTO daily_summary VALUES ('etf','PAF','us','2017-11-01',58.98,59.286,58.98,59.1492,1007,'0');
INSERT INTO daily_summary VALUES ('stock','PAG','us','2017-11-01',46.742,47.08,45.829,47.03,602992,'0');
INSERT INTO daily_summary VALUES ('etf','PAGG','us','2017-11-01',26.69,26.8416,26.69,26.78,480,'0');
INSERT INTO daily_summary VALUES ('stock','PAGP','us','2017-11-01',20.49,20.995,20.39,20.72,2185180,'0');
INSERT INTO daily_summary VALUES ('stock','PAH','us','2017-11-01',10.82,10.98,10.519,10.61,4134360,'0');
INSERT INTO daily_summary VALUES ('stock','PAHC','us','2017-11-01',37.75,37.95,36.95,37.05,72393,'0');
INSERT INTO daily_summary VALUES ('stock','PAI','us','2017-11-01',16.02,16.02,15.92,15.94,10805,'0');
INSERT INTO daily_summary VALUES ('etf','PAK','us','2017-11-01',13.8,13.86,13.68,13.75,45916,'0');
INSERT INTO daily_summary VALUES ('etf','PALL','us','2017-11-01',95.2,95.84,95.2,95.75,30520,'0');
INSERT INTO daily_summary VALUES ('stock','PAM','us','2017-11-01',68.23,68.776,67.63,68.26,339260,'0');
INSERT INTO daily_summary VALUES ('stock','PANL','us','2017-11-01',2.41,2.41,2.3472,2.3899,3934,'0');
INSERT INTO daily_summary VALUES ('stock','PANW','us','2017-11-01',147.58,147.94,142.7,143.99,1682810,'0');
INSERT INTO daily_summary VALUES ('stock','PAR','us','2017-11-01',10.98,11.44,10.92,11.35,78354,'0');
INSERT INTO daily_summary VALUES ('stock','PARR','us','2017-11-01',21.08,21.27,20.96,21.21,313851,'0');
INSERT INTO daily_summary VALUES ('stock','PATI','us','2017-11-01',19.65,19.65,19.65,19.65,124,'0');
INSERT INTO daily_summary VALUES ('stock','PATK','us','2017-11-01',94.25,95.35,90.75,92,107557,'0');
INSERT INTO daily_summary VALUES ('stock','PAVE','us','2017-11-01',15.55,15.742,15.55,15.5604,7017,'0');
INSERT INTO daily_summary VALUES ('stock','PAVM','us','2017-11-01',4.29,4.44,4.29,4.4,823,'0');
INSERT INTO daily_summary VALUES ('stock','PAVMW','us','2017-11-01',1.6,1.6,1.524,1.524,405,'0');
INSERT INTO daily_summary VALUES ('stock','PAY','us','2017-11-01',19.25,19.32,18.7,18.74,1118900,'0');
INSERT INTO daily_summary VALUES ('stock','PAYC','us','2017-11-01',78.91,78.91,74.35,77.3,2693980,'0');
INSERT INTO daily_summary VALUES ('stock','PAYX','us','2017-11-01',63.87,64.28,63.56,64.14,1637050,'0');
INSERT INTO daily_summary VALUES ('stock','PB','us','2017-11-01',66.21,66.54,64.87,65.25,291590,'0');
INSERT INTO daily_summary VALUES ('stock','PBA','us','2017-11-01',33.23,33.7,33.23,33.66,529598,'0');
INSERT INTO daily_summary VALUES ('stock','PBB','us','2017-11-01',25.68,25.75,25.68,25.74,14540,'0');
INSERT INTO daily_summary VALUES ('stock','PBBI','us','2017-11-01',10.55,10.55,10.15,10.5,78737,'0');
INSERT INTO daily_summary VALUES ('stock','PBCT','us','2017-11-01',18.75,18.78,18.46,18.53,1929380,'0');
INSERT INTO daily_summary VALUES ('stock','PBCTP','us','2017-11-01',26.9,27.1397,26.9,27.1,8303,'0');
INSERT INTO daily_summary VALUES ('etf','PBD','us','2017-11-01',13.13,13.2,13.03,13.0401,2837,'0');
INSERT INTO daily_summary VALUES ('stock','PBDM','us','2017-11-01',25.8,25.8,25.8,25.8,0,'0');
INSERT INTO daily_summary VALUES ('etf','PBE','us','2017-11-01',46.67,46.82,45.921,46.068,19782,'0');
INSERT INTO daily_summary VALUES ('stock','PBEE','us','2017-11-01',25.33,25.33,25.33,25.33,0,'0');
INSERT INTO daily_summary VALUES ('stock','PBF','us','2017-11-01',29.164,29.877,28.995,29.649,3347040,'0');
INSERT INTO daily_summary VALUES ('stock','PBFX','us','2017-11-01',19.421,19.812,19.28,19.616,58410,'0');
INSERT INTO daily_summary VALUES ('stock','PBH','us','2017-11-01',46.94,47.34,46.325,46.88,744721,'0');
INSERT INTO daily_summary VALUES ('stock','PBI','us','2017-11-01',12.57,13.23,11.2,11.4,11611800,'0');
INSERT INTO daily_summary VALUES ('stock','PBIB','us','2017-11-01',12.5,12.51,12.5,12.5001,1219,'0');
INSERT INTO daily_summary VALUES ('stock','PBIP','us','2017-11-01',18.28,18.35,18.2,18.23,4460,'0');
INSERT INTO daily_summary VALUES ('stock','PBI_B','us','2017-11-01',25.63,25.63,25.17,25.28,452676,'0');
INSERT INTO daily_summary VALUES ('etf','PBJ','us','2017-11-01',32.31,32.392,32.16,32.16,27533,'0');
INSERT INTO daily_summary VALUES ('stock','PBMD','us','2017-11-01',2.22,2.22,2.11,2.13,38962,'0');
INSERT INTO daily_summary VALUES ('stock','PBNC','us','2017-11-01',57.23,57.23,56.96,56.99,2973,'0');
INSERT INTO daily_summary VALUES ('stock','PBND','us','2017-11-01',25,25.01,25,25.01,393,'0');
INSERT INTO daily_summary VALUES ('etf','PBP','us','2017-11-01',22.44,22.44,22.28,22.38,65477,'0');
INSERT INTO daily_summary VALUES ('stock','PBPB','us','2017-11-01',12.15,12.225,11.9,11.95,273148,'0');
INSERT INTO daily_summary VALUES ('stock','PBR','us','2017-11-01',10.77,10.9,10.76,10.83,18674100,'0');
INSERT INTO daily_summary VALUES ('etf','PBS','us','2017-11-01',27.31,27.3712,27.205,27.252,3802,'0');
INSERT INTO daily_summary VALUES ('stock','PBSM','us','2017-11-01',25.96,25.96,25.96,25.96,102,'0');
INSERT INTO daily_summary VALUES ('stock','PBT','us','2017-11-01',9.3,9.3075,9.05,9.22,94731,'0');
INSERT INTO daily_summary VALUES ('stock','PBUS','us','2017-11-01',25.7,25.7,25.7,25.7,0,'0');
INSERT INTO daily_summary VALUES ('etf','PBW','us','2017-11-01',24.75,24.75,24.52,24.53,9038,'0');
INSERT INTO daily_summary VALUES ('stock','PBYI','us','2017-11-01',127.85,130.074,125.55,126.15,619033,'0');
INSERT INTO daily_summary VALUES ('stock','PCAR','us','2017-11-01',72.11,72.22,70.326,70.725,2093240,'0');
INSERT INTO daily_summary VALUES ('etf','PCEF','us','2017-11-01',23.99,24.03,23.983,24.03,175507,'0');
INSERT INTO daily_summary VALUES ('stock','PCF','us','2017-11-01',9,9.1,9,9.1,29390,'0');
INSERT INTO daily_summary VALUES ('stock','PCG','us','2017-11-01',57.61,57.68,56.91,57.24,4444620,'0');
INSERT INTO daily_summary VALUES ('stock','PCG_A','us','2017-11-01',30.3,30.3794,30,30.15,6687,'0');
INSERT INTO daily_summary VALUES ('stock','PCG_B','us','2017-11-01',28.46,28.46,28.26,28.45,677,'0');
INSERT INTO daily_summary VALUES ('stock','PCG_C','us','2017-11-01',27.3,27.3,27.3,27.3,124,'0');
INSERT INTO daily_summary VALUES ('stock','PCG_D','us','2017-11-01',25.91,26.2,25.91,26.2,3025,'0');
INSERT INTO daily_summary VALUES ('stock','PCG_E','us','2017-11-01',26.4,26.7,26.4,26.7,200,'0');
INSERT INTO daily_summary VALUES ('stock','PCG_H','us','2017-11-01',25.08,25.08,25.05,25.05,1426,'0');
INSERT INTO daily_summary VALUES ('stock','PCG_I','us','2017-11-01',25.05,25.05,25.05,25.05,100,'0');
INSERT INTO daily_summary VALUES ('stock','PCH','us','2017-11-01',52.05,52.85,51.9,52.5,589158,'0');
INSERT INTO daily_summary VALUES ('stock','PCI','us','2017-11-01',22.619,22.639,22.5,22.5,456821,'0');
INSERT INTO daily_summary VALUES ('stock','PCK','us','2017-11-01',9.971,10.01,9.941,9.951,68472,'0');
INSERT INTO daily_summary VALUES ('stock','PCLN','us','2017-11-01',1918.47,1927.73,1910.94,1913.06,379666,'0');
INSERT INTO daily_summary VALUES ('stock','PCM','us','2017-11-01',11.95,11.97,11.821,11.88,57610,'0');
INSERT INTO daily_summary VALUES ('stock','PCMI','us','2017-11-01',14.2,14.5,14,14.25,87457,'0');
INSERT INTO daily_summary VALUES ('stock','PCN','us','2017-11-01',16.881,16.961,16.841,16.871,79197,'0');
INSERT INTO daily_summary VALUES ('stock','PCO','us','2017-11-01',6.52,6.58,6.42,6.5,45238,'0');
INSERT INTO daily_summary VALUES ('stock','PCOM','us','2017-11-01',11.5,11.5,11.01,11.08,45690,'0');
INSERT INTO daily_summary VALUES ('stock','PCQ','us','2017-11-01',17.08,17.15,16.951,16.971,100898,'0');
INSERT INTO daily_summary VALUES ('stock','PCRX','us','2017-11-01',32.4,33.3,32.2,32.6,797680,'0');
INSERT INTO daily_summary VALUES ('stock','PCSB','us','2017-11-01',18.89,18.89,18.58,18.74,30142,'0');
INSERT INTO daily_summary VALUES ('stock','PCTI','us','2017-11-01',6.6104,6.6897,6.5763,6.6897,16533,'0');
INSERT INTO daily_summary VALUES ('stock','PCTY','us','2017-11-01',53.83,53.96,52.08,52.48,527129,'0');
INSERT INTO daily_summary VALUES ('etf','PCY','us','2017-11-01',29.61,29.7,29.59,29.65,2114140,'0');
INSERT INTO daily_summary VALUES ('stock','PCYG','us','2017-11-01',11.2,11.2,10.55,10.85,79519,'0');
INSERT INTO daily_summary VALUES ('stock','PCYO','us','2017-11-01',7.7,7.7,7.3,7.4,20247,'0');
INSERT INTO daily_summary VALUES ('etf','PDBC','us','2017-11-01',17.58,17.6139,17.4034,17.47,364863,'0');
INSERT INTO daily_summary VALUES ('stock','PDCE','us','2017-11-01',50.82,53.41,50.17,52.86,1078620,'0');
INSERT INTO daily_summary VALUES ('stock','PDCO','us','2017-11-01',36.99,37.08,35.98,36.12,1268260,'0');
INSERT INTO daily_summary VALUES ('stock','PDEX','us','2017-11-01',7.6,7.85,7.205,7.5,39638,'0');
INSERT INTO daily_summary VALUES ('stock','PDFS','us','2017-11-01',14.86,15,14.69,14.93,176909,'0');
INSERT INTO daily_summary VALUES ('stock','PDI','us','2017-11-01',30.21,30.22,29.893,29.913,180640,'0');
INSERT INTO daily_summary VALUES ('stock','PDLB','us','2017-11-01',15.1,15.1,15,15,35769,'0');
INSERT INTO daily_summary VALUES ('stock','PDLI','us','2017-11-01',2.97,3.04,2.96,2.99,2195670,'0');
INSERT INTO daily_summary VALUES ('stock','PDM','us','2017-11-01',19.27,19.41,19.24,19.33,829903,'0');
INSERT INTO daily_summary VALUES ('etf','PDN','us','2017-11-01',33.53,33.73,33.53,33.65,12634,'0');
INSERT INTO daily_summary VALUES ('stock','PDP','us','2017-11-01',50.93,50.93,50.33,50.36,959829,'0');
INSERT INTO daily_summary VALUES ('stock','PDS','us','2017-11-01',3.05,3.17,2.96,3.02,3600470,'0');
INSERT INTO daily_summary VALUES ('stock','PDT','us','2017-11-01',16.87,16.979,16.87,16.89,46635,'0');
INSERT INTO daily_summary VALUES ('stock','PDVW','us','2017-11-01',28.75,28.9,28.2,28.85,50147,'0');
INSERT INTO daily_summary VALUES ('stock','PE','us','2017-11-01',26.87,27.42,26.62,27.09,5412300,'0');
INSERT INTO daily_summary VALUES ('stock','PEB','us','2017-11-01',35.99,36.12,35.5,35.61,761669,'0');
INSERT INTO daily_summary VALUES ('stock','PEBK','us','2017-11-01',33.57,34.6999,33.075,33.91,3010,'0');
INSERT INTO daily_summary VALUES ('stock','PEBO','us','2017-11-01',33.199,33.636,32.593,33.05,53461,'0');
INSERT INTO daily_summary VALUES ('stock','PEB_C','us','2017-11-01',25.5,25.5,25.45,25.45,2030,'0');
INSERT INTO daily_summary VALUES ('stock','PED','us','2017-11-01',0.312,0.3242,0.3,0.31,55260,'0');
INSERT INTO daily_summary VALUES ('stock','PEG','us','2017-11-01',49.94,50.19,49.29,49.51,4266170,'0');
INSERT INTO daily_summary VALUES ('stock','PEGA','us','2017-11-01',58.75,58.9,57.4,57.9,269570,'0');
INSERT INTO daily_summary VALUES ('stock','PEGI','us','2017-11-01',23.09,23.23,22.9,23.06,469745,'0');
INSERT INTO daily_summary VALUES ('stock','PEI','us','2017-11-01',9.72,9.97,9.67,9.73,1067060,'0');
INSERT INTO daily_summary VALUES ('stock','PEIX','us','2017-11-01',4.85,4.9,4.75,4.85,178113,'0');
INSERT INTO daily_summary VALUES ('stock','PEI_B','us','2017-11-01',25.5,25.57,25.5,25.5142,5839,'0');
INSERT INTO daily_summary VALUES ('stock','PEI_C','us','2017-11-01',26.25,26.3742,26.25,26.3742,46600,'0');
INSERT INTO daily_summary VALUES ('stock','PEI_D','us','2017-11-01',25.61,25.7499,25.55,25.66,9537,'0');
INSERT INTO daily_summary VALUES ('etf','PEJ','us','2017-11-01',41.95,42.13,41.66,41.66,31347,'0');
INSERT INTO daily_summary VALUES ('etf','PEK','us','2017-11-01',47.75,47.79,47.67,47.67,14954,'0');
INSERT INTO daily_summary VALUES ('stock','PEN','us','2017-11-01',99.75,101.2,99.005,99.65,255579,'0');
INSERT INTO daily_summary VALUES ('stock','PENN','us','2017-11-01',26.26,26.78,25.97,26.07,828811,'0');
INSERT INTO daily_summary VALUES ('stock','PEO','us','2017-11-01',19.63,19.8343,19.63,19.79,91581,'0');
INSERT INTO daily_summary VALUES ('stock','PEP','us','2017-11-01',109.76,110.68,109.61,110.13,2822420,'0');
INSERT INTO daily_summary VALUES ('stock','PER','us','2017-11-01',2.7034,2.7517,2.7034,2.7034,131161,'0');
INSERT INTO daily_summary VALUES ('stock','PERI','us','2017-11-01',1.05,1.08,1,1.07,3627980,'0');
INSERT INTO daily_summary VALUES ('stock','PERY','us','2017-11-01',23.3,23.35,22.46,22.8,73441,'0');
INSERT INTO daily_summary VALUES ('stock','PES','us','2017-11-01',1.95,2.05,1.925,2,752170,'0');
INSERT INTO daily_summary VALUES ('stock','PESI','us','2017-11-01',3.75,3.85,3.75,3.85,11037,'0');
INSERT INTO daily_summary VALUES ('stock','PETQ','us','2017-11-01',24.24,24.24,23.75,24.05,107948,'0');
INSERT INTO daily_summary VALUES ('stock','PETS','us','2017-11-01',35.24,36.752,35.002,36.354,827044,'0');
INSERT INTO daily_summary VALUES ('stock','PETX','us','2017-11-01',5.77,5.78,5.6,5.64,184666,'0');
INSERT INTO daily_summary VALUES ('stock','PETZ','us','2017-11-01',16.02,18.88,15.39,18.43,180478,'0');
INSERT INTO daily_summary VALUES ('etf','PEX','us','2017-11-01',42.02,42.02,42.02,42.02,102,'0');
INSERT INTO daily_summary VALUES ('etf','PEY','us','2017-11-01',17.25,17.3032,17.15,17.17,117094,'0');
INSERT INTO daily_summary VALUES ('etf','PEZ','us','2017-11-01',49.06,49.06,48.632,48.7499,2887,'0');
INSERT INTO daily_summary VALUES ('stock','PF','us','2017-11-01',54.55,55.27,54.52,54.62,1120670,'0');
INSERT INTO daily_summary VALUES ('stock','PFBC','us','2017-11-01',61.95,62.25,60.83,61.69,93809,'0');
INSERT INTO daily_summary VALUES ('stock','PFBI','us','2017-11-01',20.67,20.93,19.942,20.18,9270,'0');
INSERT INTO daily_summary VALUES ('stock','PFBX','us','2017-11-01',14,14.25,14,14.25,2128,'0');
INSERT INTO daily_summary VALUES ('stock','PFD','us','2017-11-01',15.4,15.55,15.4,15.55,24343,'0');
INSERT INTO daily_summary VALUES ('stock','PFE','us','2017-11-01',34.693,35.406,34.584,34.941,15852600,'0');
INSERT INTO daily_summary VALUES ('etf','PFF','us','2017-11-01',38.33,38.38,38.25,38.28,2713840,'0');
INSERT INTO daily_summary VALUES ('stock','PFFD','us','2017-11-01',25.11,25.11,24.9101,24.9549,2997,'0');
INSERT INTO daily_summary VALUES ('stock','PFFR','us','2017-11-01',25.8384,25.855,25.82,25.82,2507,'0');
INSERT INTO daily_summary VALUES ('stock','PFG','us','2017-11-01',66.21,67.205,65.98,67.13,1694200,'0');
INSERT INTO daily_summary VALUES ('stock','PFGC','us','2017-11-01',28.45,28.7,28.2,28.4,420696,'0');
INSERT INTO daily_summary VALUES ('stock','PFH','us','2017-11-01',12.02,12.07,11.75,11.84,17791,'0');
INSERT INTO daily_summary VALUES ('stock','PFI','us','2017-11-01',33.97,34.22,33.9,34.06,10555,'0');
INSERT INTO daily_summary VALUES ('stock','PFIE','us','2017-11-01',1.89,1.9,1.82,1.82,59259,'0');
INSERT INTO daily_summary VALUES ('etf','PFIG','us','2017-11-01',25.62,25.63,25.582,25.61,31636,'0');
INSERT INTO daily_summary VALUES ('stock','PFIN','us','2017-11-01',7.43,7.43,7.399,7.399,1237,'0');
INSERT INTO daily_summary VALUES ('stock','PFIS','us','2017-11-01',45.55,45.55,44.72,44.72,1403,'0');
INSERT INTO daily_summary VALUES ('stock','PFK','us','2017-11-01',25.23,25.26,25.2,25.2353,8838,'0');
INSERT INTO daily_summary VALUES ('stock','PFL','us','2017-11-01',11.98,11.989,11.915,11.94,65315,'0');
INSERT INTO daily_summary VALUES ('stock','PFLT','us','2017-11-01',14.04,14.0499,13.84,13.86,478478,'0');
INSERT INTO daily_summary VALUES ('etf','PFM','us','2017-11-01',25.09,25.1714,25.09,25.095,29234,'0');
INSERT INTO daily_summary VALUES ('stock','PFMT','us','2017-11-01',1.86,1.94,1.8334,1.85,33759,'0');
INSERT INTO daily_summary VALUES ('stock','PFN','us','2017-11-01',10.61,10.64,10.56,10.63,253335,'0');
INSERT INTO daily_summary VALUES ('stock','PFNX','us','2017-11-01',3.17,3.25,2.95,3.11,154509,'0');
INSERT INTO daily_summary VALUES ('stock','PFO','us','2017-11-01',12.19,12.25,12.1892,12.24,10524,'0');
INSERT INTO daily_summary VALUES ('stock','PFPT','us','2017-11-01',92.73,93.12,89.76,91.2,697239,'0');
INSERT INTO daily_summary VALUES ('stock','PFS','us','2017-11-01',27.36,27.42,26.75,26.85,328666,'0');
INSERT INTO daily_summary VALUES ('stock','PFSI','us','2017-11-01',19.15,19.2,18.45,18.7,113270,'0');
INSERT INTO daily_summary VALUES ('stock','PFSW','us','2017-11-01',8.33,8.39,8.04,8.13,45756,'0');
INSERT INTO daily_summary VALUES ('etf','PFXF','us','2017-11-01',19.92,20.0024,19.881,19.93,91876,'0');
INSERT INTO daily_summary VALUES ('stock','PG','us','2017-11-01',86.33,87.14,86.11,86.89,4884050,'0');
INSERT INTO daily_summary VALUES ('etf','PGAL','us','2017-11-01',12.48,12.5,12.43,12.43,34536,'0');
INSERT INTO daily_summary VALUES ('stock','PGC','us','2017-11-01',35.017,35.796,34.468,34.797,52662,'0');
INSERT INTO daily_summary VALUES ('stock','PGEM','us','2017-11-01',17.1,17.1,16.5,16.55,88639,'0');
INSERT INTO daily_summary VALUES ('etf','PGF','us','2017-11-01',18.87,18.88,18.84,18.87,225295,'0');
INSERT INTO daily_summary VALUES ('stock','PGH','us','2017-11-01',1.14,1.16,1.09,1.12,1071700,'0');
INSERT INTO daily_summary VALUES ('etf','PGHY','us','2017-11-01',24.17,24.25,24.17,24.1899,48631,'0');
INSERT INTO daily_summary VALUES ('etf','PGJ','us','2017-11-01',43.16,43.3068,42.93,42.93,29926,'0');
INSERT INTO daily_summary VALUES ('stock','PGLC','us','2017-11-01',2.88,2.9,2.85,2.89,40603,'0');
INSERT INTO daily_summary VALUES ('etf','PGM','us','2017-11-01',19.0915,19.0915,19.0915,19.0915,218,'0');
INSERT INTO daily_summary VALUES ('stock','PGNX','us','2017-11-01',6.22,6.28,5.8325,5.86,873744,'0');
INSERT INTO daily_summary VALUES ('stock','PGP','us','2017-11-01',16.203,16.223,16.035,16.045,45866,'0');
INSERT INTO daily_summary VALUES ('stock','PGR','us','2017-11-01',48.85,48.93,48.69,48.8,1873490,'0');
INSERT INTO daily_summary VALUES ('stock','PGRE','us','2017-11-01',15.93,16.12,15.84,16.03,1145130,'0');
INSERT INTO daily_summary VALUES ('stock','PGTI','us','2017-11-01',14.25,14.25,13.9,14.2,403446,'0');
INSERT INTO daily_summary VALUES ('etf','PGX','us','2017-11-01',14.91,14.99,14.91,14.97,1475330,'0');
INSERT INTO daily_summary VALUES ('stock','PGZ','us','2017-11-01',17.09,17.14,17.09,17.14,41010,'0');
INSERT INTO daily_summary VALUES ('stock','PH','us','2017-11-01',184,184,181.09,181.34,845012,'0');
INSERT INTO daily_summary VALUES ('etf','PHB','us','2017-11-01',19.07,19.08,19.03,19.03,739009,'0');
INSERT INTO daily_summary VALUES ('stock','PHD','us','2017-11-01',12.03,12.045,11.99,11.99,45743,'0');
INSERT INTO daily_summary VALUES ('etf','PHDG','us','2017-11-01',26.7423,26.7423,26.67,26.67,2205,'0');
INSERT INTO daily_summary VALUES ('stock','PHG','us','2017-11-01',41.17,41.35,41.0841,41.24,2143440,'0');
INSERT INTO daily_summary VALUES ('stock','PHH','us','2017-11-01',13.24,13.34,13.015,13.09,160415,'0');
INSERT INTO daily_summary VALUES ('stock','PHI','us','2017-11-01',33.16,33.38,32.86,33.06,28967,'0');
INSERT INTO daily_summary VALUES ('stock','PHII','us','2017-11-01',11.89,11.89,11.31,11.59,430,'0');
INSERT INTO daily_summary VALUES ('stock','PHIIK','us','2017-11-01',11.86,12.098,11.46,11.92,76234,'0');
INSERT INTO daily_summary VALUES ('stock','PHK','us','2017-11-01',7.8276,7.8457,7.7089,7.7286,782689,'0');
INSERT INTO daily_summary VALUES ('stock','PHM','us','2017-11-01',30.35,30.62,30.07,30.41,6414450,'0');
INSERT INTO daily_summary VALUES ('stock','PHO','us','2017-11-01',29.09,29.45,29.09,29.29,30762,'0');
INSERT INTO daily_summary VALUES ('stock','PHT','us','2017-11-01',10,10.03,9.97,10.02,112296,'0');
INSERT INTO daily_summary VALUES ('stock','PHX','us','2017-11-01',24.5,24.55,23.85,23.9,94071,'0');
INSERT INTO daily_summary VALUES ('stock','PHYS','us','2017-11-01',10.37,10.42,10.36,10.4,562291,'0');
INSERT INTO daily_summary VALUES ('stock','PI','us','2017-11-01',34.26,34.765,32.65,32.8,867831,'0');
INSERT INTO daily_summary VALUES ('etf','PICB','us','2017-11-01',26.9,26.9594,26.78,26.85,10640,'0');
INSERT INTO daily_summary VALUES ('etf','PICK','us','2017-11-01',32.5,32.5,32.1801,32.3,145654,'0');
INSERT INTO daily_summary VALUES ('stock','PICO','us','2017-11-01',19.05,19.1,18.8,19,90838,'0');
INSERT INTO daily_summary VALUES ('stock','PID','us','2017-11-01',16.07,16.129,15.996,16.01,193366,'0');
INSERT INTO daily_summary VALUES ('etf','PIE','us','2017-11-01',19.63,19.73,19.61,19.65,32773,'0');
INSERT INTO daily_summary VALUES ('stock','PIH','us','2017-11-01',7.1,7.1,7.0863,7.0863,893,'0');
INSERT INTO daily_summary VALUES ('stock','PII','us','2017-11-01',118.76,121.36,118.26,118.64,650155,'0');
INSERT INTO daily_summary VALUES ('stock','PIM','us','2017-11-01',4.84,4.8881,4.82,4.82,102687,'0');
INSERT INTO daily_summary VALUES ('etf','PIN','us','2017-11-01',25.9,26,25.86,25.95,82591,'0');
INSERT INTO daily_summary VALUES ('stock','PINC','us','2017-11-01',32.68,32.68,31.38,31.45,629336,'0');
INSERT INTO daily_summary VALUES ('etf','PIO','us','2017-11-01',25.31,25.3567,25.22,25.26,33427,'0');
INSERT INTO daily_summary VALUES ('stock','PIR','us','2017-11-01',4.19,4.24,4.12,4.19,1061390,'0');
INSERT INTO daily_summary VALUES ('stock','PIRS','us','2017-11-01',5.04,5.06,4.8,4.84,212275,'0');
INSERT INTO daily_summary VALUES ('stock','PIXY','us','2017-11-01',3.27,3.27,2.88,3.1,44931,'0');
INSERT INTO daily_summary VALUES ('stock','PIY','us','2017-11-01',25.5,25.5056,23.36,24.59,20408,'0');
INSERT INTO daily_summary VALUES ('stock','PIZ','us','2017-11-01',27.41,27.5,27.252,27.29,156054,'0');
INSERT INTO daily_summary VALUES ('stock','PJC','us','2017-11-01',73.1,73.55,71.8,72.45,205389,'0');
INSERT INTO daily_summary VALUES ('stock','PJH','us','2017-11-01',25.37,25.42,25.3401,25.4159,23032,'0');
INSERT INTO daily_summary VALUES ('etf','PJP','us','2017-11-01',62.61,63.0894,62.61,62.88,44115,'0');
INSERT INTO daily_summary VALUES ('stock','PJT','us','2017-11-01',38.96,38.96,37.74,37.78,136622,'0');
INSERT INTO daily_summary VALUES ('stock','PK','us','2017-11-01',28.81,29.08,28.72,28.93,1093410,'0');
INSERT INTO daily_summary VALUES ('etf','PKB','us','2017-11-01',33.39,33.39,32.97,33.15,42099,'0');
INSERT INTO daily_summary VALUES ('stock','PKBK','us','2017-11-01',21.85,21.85,21.15,21.15,4881,'0');
INSERT INTO daily_summary VALUES ('stock','PKD','us','2017-11-01',1.05,1.055,1.01,1.04,827729,'0');
INSERT INTO daily_summary VALUES ('stock','PKE','us','2017-11-01',19.01,19.01,18.63,18.78,37367,'0');
INSERT INTO daily_summary VALUES ('stock','PKG','us','2017-11-01',117.11,117.546,116.06,116.57,881734,'0');
INSERT INTO daily_summary VALUES ('stock','PKI','us','2017-11-01',72.75,73.05,72.27,72.27,519148,'0');
INSERT INTO daily_summary VALUES ('stock','PKO','us','2017-11-01',26.448,26.454,25.991,26.031,44928,'0');
INSERT INTO daily_summary VALUES ('stock','PKOH','us','2017-11-01',47.314,47.314,45.071,46.167,16684,'0');
INSERT INTO daily_summary VALUES ('stock','PKW','us','2017-11-01',56.05,56.22,55.755,55.83,70290,'0');
INSERT INTO daily_summary VALUES ('stock','PKX','us','2017-11-01',73.2,73.6,72.38,73.04,193795,'0');
INSERT INTO daily_summary VALUES ('stock','PLAB','us','2017-11-01',9.8,9.825,9.35,9.45,337657,'0');
INSERT INTO daily_summary VALUES ('stock','PLAY','us','2017-11-01',48.18,48.28,46.79,47.03,1485690,'0');
INSERT INTO daily_summary VALUES ('stock','PLBC','us','2017-11-01',21.8,21.8,21,21.595,3256,'0');
INSERT INTO daily_summary VALUES ('stock','PLCE','us','2017-11-01',109,109.8,108.2,108.35,327817,'0');
INSERT INTO daily_summary VALUES ('stock','PLD','us','2017-11-01',64.6,65.05,64.32,65,1727430,'0');
INSERT INTO daily_summary VALUES ('stock','PLG','us','2017-11-01',0.407,0.44,0.4001,0.4277,627773,'0');
INSERT INTO daily_summary VALUES ('stock','PLM','us','2017-11-01',0.6594,0.6598,0.61,0.6155,235173,'0');
INSERT INTO daily_summary VALUES ('etf','PLND','us','2017-11-01',20.56,20.715,20.56,20.6072,20210,'0');
INSERT INTO daily_summary VALUES ('stock','PLNT','us','2017-11-01',26.7,27,26.46,26.51,855324,'0');
INSERT INTO daily_summary VALUES ('stock','PLOW','us','2017-11-01',42.35,42.6,41.2,41.6,92027,'0');
INSERT INTO daily_summary VALUES ('stock','PLPC','us','2017-11-01',71.2,71.7299,69.5,69.5,4448,'0');
INSERT INTO daily_summary VALUES ('stock','PLPM','us','2017-11-01',4.47,4.48,4.46,4.47,1851530,'0');
INSERT INTO daily_summary VALUES ('stock','PLSE','us','2017-11-01',24.41,24.7922,23,23.18,97346,'0');
INSERT INTO daily_summary VALUES ('stock','PLT','us','2017-11-01',49.59,53.62,48.94,52.39,873758,'0');
INSERT INTO daily_summary VALUES ('stock','PLUG','us','2017-11-01',2.9,2.9,2.76,2.82,3763090,'0');
INSERT INTO daily_summary VALUES ('stock','PLUS','us','2017-11-01',95.8,97.15,95,96.5,102116,'0');
INSERT INTO daily_summary VALUES ('stock','PLW','us','2017-11-01',32.62,32.699,32.58,32.63,8683,'0');
INSERT INTO daily_summary VALUES ('stock','PLX','us','2017-11-01',0.79,0.8,0.723,0.77,875321,'0');
INSERT INTO daily_summary VALUES ('stock','PLXP','us','2017-11-01',6.75,6.9,6.6643,6.85,1513,'0');
INSERT INTO daily_summary VALUES ('stock','PLXS','us','2017-11-01',61.69,61.86,60.5,60.66,98632,'0');
INSERT INTO daily_summary VALUES ('stock','PLYA','us','2017-11-01',10.3,10.5,10.13,10.3,310026,'0');
INSERT INTO daily_summary VALUES ('stock','PLYM','us','2017-11-01',18.0943,18.1388,17.905,18.12,17651,'0');
INSERT INTO daily_summary VALUES ('stock','PLYM_A','us','2017-11-01',24.4861,24.55,24.46,24.49,35090,'0');
INSERT INTO daily_summary VALUES ('stock','PM','us','2017-11-01',104.78,105,103.64,103.75,4102160,'0');
INSERT INTO daily_summary VALUES ('stock','PMBC','us','2017-11-01',9.45,9.5,9.3,9.35,9368,'0');
INSERT INTO daily_summary VALUES ('stock','PMC','us','2017-11-01',29.25,29.25,29.125,29.15,348742,'0');
INSERT INTO daily_summary VALUES ('stock','PMD','us','2017-11-01',17.215,17.711,17.215,17.602,12653,'0');
INSERT INTO daily_summary VALUES ('stock','PME','us','2017-11-01',3,3.1,2.7,2.75,159025,'0');
INSERT INTO daily_summary VALUES ('stock','PMF','us','2017-11-01',13.14,13.15,13.08,13.099,109030,'0');
INSERT INTO daily_summary VALUES ('stock','PML','us','2017-11-01',13.051,13.061,13.011,13.061,89170,'0');
INSERT INTO daily_summary VALUES ('stock','PMM','us','2017-11-01',7.44,7.46,7.41,7.45,80283,'0');
INSERT INTO daily_summary VALUES ('stock','PMO','us','2017-11-01',12.47,12.49,12.43,12.45,52398,'0');
INSERT INTO daily_summary VALUES ('etf','PMR','us','2017-11-01',33.45,33.455,33.33,33.41,6273,'0');
INSERT INTO daily_summary VALUES ('stock','PMT','us','2017-11-01',16.1,16.24,16.1,16.23,422032,'0');
INSERT INTO daily_summary VALUES ('stock','PMTS','us','2017-11-01',1.09,1.09,1.04,1.05,149599,'0');
INSERT INTO daily_summary VALUES ('stock','PMT_A','us','2017-11-01',25.44,25.44,25.34,25.43,14597,'0');
INSERT INTO daily_summary VALUES ('stock','PMT_B','us','2017-11-01',25.31,25.34,25.19,25.3,30537,'0');
INSERT INTO daily_summary VALUES ('stock','PMX','us','2017-11-01',11.621,11.621,11.561,11.611,53668,'0');
INSERT INTO daily_summary VALUES ('stock','PN','us','2017-11-01',1.27,1.3051,1.2501,1.26,74900,'0');
INSERT INTO daily_summary VALUES ('stock','PNC','us','2017-11-01',137.88,138.64,136.76,136.94,1207600,'0');
INSERT INTO daily_summary VALUES ('stock','PNC_P','us','2017-11-01',28.31,28.43,28.228,28.43,84132,'0');
INSERT INTO daily_summary VALUES ('stock','PNC_Q','us','2017-11-01',25.451,25.4694,25.41,25.4496,8404,'0');
INSERT INTO daily_summary VALUES ('stock','PNF','us','2017-11-01',12.86,12.88,12.84,12.86,33304,'0');
INSERT INTO daily_summary VALUES ('stock','PNFP','us','2017-11-01',66.358,66.757,65.36,65.51,683592,'0');
INSERT INTO daily_summary VALUES ('stock','PNI','us','2017-11-01',12.329,12.329,12.15,12.15,15468,'0');
INSERT INTO daily_summary VALUES ('stock','PNK','us','2017-11-01',26.14,26.435,25.53,25.89,438018,'0');
INSERT INTO daily_summary VALUES ('stock','PNM','us','2017-11-01',43.7,43.7,42.45,42.5,1030270,'0');
INSERT INTO daily_summary VALUES ('stock','PNNT','us','2017-11-01',7.5,7.51,7.46,7.48,133029,'0');
INSERT INTO daily_summary VALUES ('etf','PNQI','us','2017-11-01',114.74,114.78,113.96,114.08,169086,'0');
INSERT INTO daily_summary VALUES ('stock','PNR','us','2017-11-01',70.49,70.63,69.23,69.26,1065060,'0');
INSERT INTO daily_summary VALUES ('stock','PNRG','us','2017-11-01',51,51,51,51,1196,'0');
INSERT INTO daily_summary VALUES ('stock','PNTR','us','2017-11-01',18.9,18.9,18,18.3,27674,'0');
INSERT INTO daily_summary VALUES ('stock','PNW','us','2017-11-01',87.91,88.22,87.16,87.46,663187,'0');
INSERT INTO daily_summary VALUES ('stock','PODD','us','2017-11-01',59.29,59.29,58.155,58.42,384251,'0');
INSERT INTO daily_summary VALUES ('stock','POL','us','2017-11-01',46.42,46.54,46.01,46.32,533317,'0');
INSERT INTO daily_summary VALUES ('stock','POLA','us','2017-11-01',5.23,5.3784,5,5.01,34530,'0');
INSERT INTO daily_summary VALUES ('stock','POOL','us','2017-11-01',121.59,121.78,120.075,120.48,191147,'0');
INSERT INTO daily_summary VALUES ('stock','POPE','us','2017-11-01',70.25,71,70.25,70.726,1512,'0');
INSERT INTO daily_summary VALUES ('stock','POR','us','2017-11-01',47.74,47.75,47.13,47.22,827236,'0');
INSERT INTO daily_summary VALUES ('stock','POST','us','2017-11-01',83.37,83.3899,81.61,83.09,449253,'0');
INSERT INTO daily_summary VALUES ('stock','POT','us','2017-11-01',19.58,19.74,19.5,19.63,4757800,'0');
INSERT INTO daily_summary VALUES ('stock','POWI','us','2017-11-01',81.3,81.3,79.275,80.15,144074,'0');
INSERT INTO daily_summary VALUES ('stock','POWL','us','2017-11-01',29.14,30.11,28.58,28.92,48883,'0');
INSERT INTO daily_summary VALUES ('etf','PPA','us','2017-11-01',52.71,52.8499,52.3,52.3876,64725,'0');
INSERT INTO daily_summary VALUES ('stock','PPBI','us','2017-11-01',40.65,40.8,39.4,39.75,295213,'0');
INSERT INTO daily_summary VALUES ('stock','PPC','us','2017-11-01',31.94,32.18,31.46,31.59,781413,'0');
INSERT INTO daily_summary VALUES ('stock','PPG','us','2017-11-01',115.89,116.27,115.04,115.47,1144450,'0');
INSERT INTO daily_summary VALUES ('etf','PPH','us','2017-11-01',55.46,55.8,55.39,55.57,67985,'0');
INSERT INTO daily_summary VALUES ('stock','PPHM','us','2017-11-01',4.62,4.689,4.56,4.61,180107,'0');
INSERT INTO daily_summary VALUES ('stock','PPHMP','us','2017-11-01',24,24.25,23.7948,23.8903,1520,'0');
INSERT INTO daily_summary VALUES ('stock','PPIH','us','2017-11-01',8.3999,8.4063,8.264,8.35,15480,'0');
INSERT INTO daily_summary VALUES ('stock','PPL','us','2017-11-01',37.27,37.35,36.33,36.94,4071100,'0');
INSERT INTO daily_summary VALUES ('etf','PPLT','us','2017-11-01',88.75,89.44,88.75,89.089,47128,'0');
INSERT INTO daily_summary VALUES ('stock','PPR','us','2017-11-01',5.1297,5.1297,5.0998,5.0998,409867,'0');
INSERT INTO daily_summary VALUES ('stock','PPT','us','2017-11-01',5.39,5.42,5.379,5.4,185494,'0');
INSERT INTO daily_summary VALUES ('stock','PPX','us','2017-11-01',25.34,25.45,25.32,25.4366,5752,'0');
INSERT INTO daily_summary VALUES ('stock','PQ','us','2017-11-01',1.86,1.92,1.8,1.83,59078,'0');
INSERT INTO daily_summary VALUES ('stock','PQG','us','2017-11-01',16.38,16.5,15.78,16.48,377774,'0');
INSERT INTO daily_summary VALUES ('stock','PRA','us','2017-11-01',56.1,56.7,55.95,56.6,134626,'0');
INSERT INTO daily_summary VALUES ('stock','PRAA','us','2017-11-01',28.3,28.6,27.7,27.9,243842,'0');
INSERT INTO daily_summary VALUES ('stock','PRAH','us','2017-11-01',81.69,82.62,81.09,81.3,391528,'0');
INSERT INTO daily_summary VALUES ('stock','PRAN','us','2017-11-01',3.1,3.19,3.01,3.15,61138,'0');
INSERT INTO daily_summary VALUES ('etf','PRB','us','2017-11-01',24.4806,24.54,24.46,24.46,2827,'0');
INSERT INTO daily_summary VALUES ('stock','PRCP','us','2017-11-01',7.71,7.749,7.4467,7.63,26360,'0');
INSERT INTO daily_summary VALUES ('stock','PREF','us','2017-11-01',100.58,100.63,100.58,100.63,1217,'0');
INSERT INTO daily_summary VALUES ('stock','PRE_F','us','2017-11-01',25.67,25.8,25.67,25.8,3537,'0');
INSERT INTO daily_summary VALUES ('stock','PRE_G','us','2017-11-01',27.3,27.36,27.3,27.36,1718,'0');
INSERT INTO daily_summary VALUES ('stock','PRE_H','us','2017-11-01',29.01,29.1102,28.87,28.87,15318,'0');
INSERT INTO daily_summary VALUES ('stock','PRE_I','us','2017-11-01',26.75,26.95,26.74,26.8,14165,'0');
INSERT INTO daily_summary VALUES ('etf','PRF','us','2017-11-01',109.07,109.34,108.7,108.86,84731,'0');
INSERT INTO daily_summary VALUES ('stock','PRFT','us','2017-11-01',19.45,19.45,17.65,19.15,89884,'0');
INSERT INTO daily_summary VALUES ('etf','PRFZ','us','2017-11-01',127.71,127.71,125.68,126.38,82699,'0');
INSERT INTO daily_summary VALUES ('stock','PRGO','us','2017-11-01',81.58,83.88,81.27,82.55,1293000,'0');
INSERT INTO daily_summary VALUES ('stock','PRGS','us','2017-11-01',42.51,42.51,41.65,42.03,316296,'0');
INSERT INTO daily_summary VALUES ('stock','PRGX','us','2017-11-01',7.55,7.55,7.4,7.4,5082,'0');
INSERT INTO daily_summary VALUES ('stock','PRH','us','2017-11-01',25.63,25.6599,25.63,25.64,23292,'0');
INSERT INTO daily_summary VALUES ('stock','PRI','us','2017-11-01',89,89,87.05,87.7,196213,'0');
INSERT INTO daily_summary VALUES ('stock','PRIM','us','2017-11-01',28.51,28.51,27.32,27.89,242490,'0');
INSERT INTO daily_summary VALUES ('stock','PRK','us','2017-11-01',110.5,110.71,108.769,108.78,8961,'0');
INSERT INTO daily_summary VALUES ('stock','PRKR','us','2017-11-01',1.4,1.49,1.39,1.46,224421,'0');
INSERT INTO daily_summary VALUES ('stock','PRLB','us','2017-11-01',87.75,87.75,85.15,85.65,115825,'0');
INSERT INTO daily_summary VALUES ('stock','PRMW','us','2017-11-01',11.1,11.24,10.915,10.96,191571,'0');
INSERT INTO daily_summary VALUES ('stock','PRNT','us','2017-11-01',26.18,26.18,25.4725,25.5,23140,'0');
INSERT INTO daily_summary VALUES ('stock','PRO','us','2017-11-01',22.8,23.35,22.52,23.3,333897,'0');
INSERT INTO daily_summary VALUES ('stock','PROV','us','2017-11-01',19.325,19.325,19.117,19.117,3180,'0');
INSERT INTO daily_summary VALUES ('stock','PRPO','us','2017-11-01',2.0381,2.0381,1.6,1.69,138529,'0');
INSERT INTO daily_summary VALUES ('stock','PRQR','us','2017-11-01',3.75,4,3.75,3.8,292437,'0');
INSERT INTO daily_summary VALUES ('stock','PRSC','us','2017-11-01',55.61,55.61,53.86,54.46,27762,'0');
INSERT INTO daily_summary VALUES ('stock','PRSS','us','2017-11-01',2.001,2.05,1.97,1.97,30540,'0');
INSERT INTO daily_summary VALUES ('stock','PRTA','us','2017-11-01',58.73,59.12,56.04,57.29,396254,'0');
INSERT INTO daily_summary VALUES ('stock','PRTK','us','2017-11-01',22,22.4,21.45,21.7,387834,'0');
INSERT INTO daily_summary VALUES ('stock','PRTO','us','2017-11-01',1.76,1.9,1.6,1.9,103043,'0');
INSERT INTO daily_summary VALUES ('stock','PRTS','us','2017-11-01',2.56,2.589,2.36,2.465,371524,'0');
INSERT INTO daily_summary VALUES ('stock','PRTY','us','2017-11-01',11.2,11.35,10.75,11,1059640,'0');
INSERT INTO daily_summary VALUES ('stock','PRU','us','2017-11-01',110.98,112.13,110.37,110.58,1793470,'0');
INSERT INTO daily_summary VALUES ('stock','PSA','us','2017-11-01',208.07,208.07,205.9,206.52,691300,'0');
INSERT INTO daily_summary VALUES ('etf','PSAU','us','2017-11-01',19.16,19.4806,19.14,19.175,6839,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_A','us','2017-11-01',26.72,26.749,26.62,26.7398,6725,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_B','us','2017-11-01',25.77,25.8254,25.72,25.745,112092,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_C','us','2017-11-01',25.47,25.49,25.35,25.44,5596,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_D','us','2017-11-01',25.01,25.19,25.009,25.18,67231,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_E','us','2017-11-01',24.93,24.9573,24.871,24.9573,22749,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_F','us','2017-11-01',25.23,25.23,25.15,25.201,27477,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_G','us','2017-11-01',25.38,25.38,25.17,25.22,55116,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_U','us','2017-11-01',25.21,25.25,25.21,25.24,12297,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_V','us','2017-11-01',25.14,25.18,25.08,25.14,20276,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_W','us','2017-11-01',25.22,25.328,25.17,25.3001,17902,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_X','us','2017-11-01',25.35,25.72,25.35,25.72,109518,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_Y','us','2017-11-01',26.54,26.5754,26.49,26.5,58711,'0');
INSERT INTO daily_summary VALUES ('stock','PSA_Z','us','2017-11-01',27.04,27.04,26.9,26.98,8958,'0');
INSERT INTO daily_summary VALUES ('stock','PSB','us','2017-11-01',132.86,134.72,132.66,134.08,68554,'0');
INSERT INTO daily_summary VALUES ('stock','PSB_T','us','2017-11-01',25.28,25.3544,25.27,25.2998,84808,'0');
INSERT INTO daily_summary VALUES ('stock','PSB_U','us','2017-11-01',25.23,25.2601,25.23,25.23,5155,'0');
INSERT INTO daily_summary VALUES ('stock','PSB_V','us','2017-11-01',25.33,25.38,25.31,25.34,4255,'0');
INSERT INTO daily_summary VALUES ('stock','PSB_W','us','2017-11-01',25.54,25.54,25.24,25.47,8729,'0');
INSERT INTO daily_summary VALUES ('stock','PSB_X','us','2017-11-01',25.15,25.18,25.0747,25.18,22881,'0');
INSERT INTO daily_summary VALUES ('etf','PSCC','us','2017-11-01',73.32,73.32,72.605,72.905,1134,'0');
INSERT INTO daily_summary VALUES ('etf','PSCD','us','2017-11-01',55.58,55.58,54.983,55.2095,9947,'0');
INSERT INTO daily_summary VALUES ('etf','PSCE','us','2017-11-01',14.49,14.7699,14.41,14.5,107802,'0');
INSERT INTO daily_summary VALUES ('etf','PSCF','us','2017-11-01',54.5213,54.5213,53.8116,54.05,18163,'0');
INSERT INTO daily_summary VALUES ('etf','PSCH','us','2017-11-01',92.25,92.25,89.61,90.18,12862,'0');
INSERT INTO daily_summary VALUES ('etf','PSCI','us','2017-11-01',62.84,62.84,61.5098,61.77,1360,'0');
INSERT INTO daily_summary VALUES ('etf','PSCM','us','2017-11-01',51.27,51.5,50.505,50.6673,5340,'0');
INSERT INTO daily_summary VALUES ('etf','PSCT','us','2017-11-01',81.9,81.94,79.36,80.04,21850,'0');
INSERT INTO daily_summary VALUES ('etf','PSCU','us','2017-11-01',55.96,55.96,55.38,55.4656,2810,'0');
INSERT INTO daily_summary VALUES ('stock','PSDO','us','2017-11-01',14.83,14.87,14.24,14.47,64539,'0');
INSERT INTO daily_summary VALUES ('stock','PSDV','us','2017-11-01',1.17,1.18,1.14,1.1799,402916,'0');
INSERT INTO daily_summary VALUES ('stock','PSEC','us','2017-11-01',5.99,6,5.86,5.87,4813030,'0');
INSERT INTO daily_summary VALUES ('stock','PSF','us','2017-11-01',28.45,28.45,28.2111,28.27,31796,'0');
INSERT INTO daily_summary VALUES ('etf','PSI','us','2017-11-01',54.42,54.42,52.57,53,170273,'0');
INSERT INTO daily_summary VALUES ('etf','PSJ','us','2017-11-01',65.78,65.78,64.13,64.5732,122773,'0');
INSERT INTO daily_summary VALUES ('etf','PSK','us','2017-11-01',44.75,44.76,44.67,44.73,74644,'0');
INSERT INTO daily_summary VALUES ('stock','PSL','us','2017-11-01',62.25,62.25,61.77,61.77,9526,'0');
INSERT INTO daily_summary VALUES ('stock','PSLV','us','2017-11-01',6.33,6.445,6.33,6.425,596979,'0');
INSERT INTO daily_summary VALUES ('stock','PSMB','us','2017-11-01',13.35,13.35,13.35,13.35,843,'0');
INSERT INTO daily_summary VALUES ('stock','PSMC','us','2017-11-01',12.85,12.85,12.85,12.85,0,'0');
INSERT INTO daily_summary VALUES ('stock','PSMG','us','2017-11-01',13.5897,13.59,13.5897,13.59,430,'0');
INSERT INTO daily_summary VALUES ('stock','PSMM','us','2017-11-01',13.1,13.1,13.1,13.1,0,'0');
INSERT INTO daily_summary VALUES ('stock','PSMT','us','2017-11-01',84.15,85.35,81.55,82.1,131949,'0');
INSERT INTO daily_summary VALUES ('stock','PSO','us','2017-11-01',9.28,9.31,9.215,9.25,567833,'0');
INSERT INTO daily_summary VALUES ('etf','PSP','us','2017-11-01',12.82,12.85,12.75,12.76,49045,'0');
INSERT INTO daily_summary VALUES ('etf','PSQ','us','2017-11-01',36.32,36.61,36.309,36.49,367667,'0');
INSERT INTO daily_summary VALUES ('etf','PSR','us','2017-11-01',80.17,80.19,80.17,80.19,521,'0');
INSERT INTO daily_summary VALUES ('etf','PST','us','2017-11-01',21.5,21.5053,21.41,21.4797,164283,'0');
INSERT INTO daily_summary VALUES ('stock','PSTB','us','2017-11-01',12.56,12.62,12.33,12.46,71281,'0');
INSERT INTO daily_summary VALUES ('stock','PSTG','us','2017-11-01',16.63,16.64,16.31,16.41,962771,'0');
INSERT INTO daily_summary VALUES ('stock','PSTI','us','2017-11-01',1.64,1.685,1.64,1.66,445928,'0');
INSERT INTO daily_summary VALUES ('stock','PSX','us','2017-11-01',91.03,92.65,90.915,92.3,2152230,'0');
INSERT INTO daily_summary VALUES ('stock','PSXP','us','2017-11-01',50.63,50.92,50.46,50.7,184398,'0');
INSERT INTO daily_summary VALUES ('stock','PTC','us','2017-11-01',66.88,66.88,65.11,65.81,1529680,'0');
INSERT INTO daily_summary VALUES ('stock','PTCT','us','2017-11-01',18.93,19.06,17.62,17.77,1215720,'0');
INSERT INTO daily_summary VALUES ('stock','PTEN','us','2017-11-01',20.17,20.48,19.84,19.95,4213250,'0');
INSERT INTO daily_summary VALUES ('stock','PTEU','us','2017-11-01',31.03,31.0899,30.82,31,19882,'0');
INSERT INTO daily_summary VALUES ('stock','PTF','us','2017-11-01',53.72,53.79,52.6,52.7922,10856,'0');
INSERT INTO daily_summary VALUES ('stock','PTGX','us','2017-11-01',15.17,15.17,14.31,14.53,73351,'0');
INSERT INTO daily_summary VALUES ('stock','PTH','us','2017-11-01',69.49,69.78,67.461,67.88,107406,'0');
INSERT INTO daily_summary VALUES ('stock','PTI','us','2017-11-01',1.9143,1.94,1.79,1.87,64963,'0');
INSERT INTO daily_summary VALUES ('stock','PTIE','us','2017-11-01',3.38,3.8,3.33,3.42,143930,'0');
INSERT INTO daily_summary VALUES ('stock','PTLA','us','2017-11-01',49.91,50.53,48.51,48.58,465866,'0');
INSERT INTO daily_summary VALUES ('stock','PTLC','us','2017-11-01',27.9,27.9,27.71,27.78,136340,'0');
INSERT INTO daily_summary VALUES ('stock','PTMC','us','2017-11-01',30.46,30.46,29.95,30.115,120428,'0');
INSERT INTO daily_summary VALUES ('stock','PTN','us','2017-11-01',0.75,0.75,0.721,0.726,1069260,'0');
INSERT INTO daily_summary VALUES ('stock','PTNQ','us','2017-11-01',29.6471,29.6471,29.45,29.535,37953,'0');
INSERT INTO daily_summary VALUES ('stock','PTNR','us','2017-11-01',5.16,5.1703,5.16,5.1703,228,'0');
INSERT INTO daily_summary VALUES ('stock','PTR','us','2017-11-01',65.89,66.26,65.59,66.04,144993,'0');
INSERT INTO daily_summary VALUES ('stock','PTSI','us','2017-11-01',29.47,29.68,29.16,29.64,9184,'0');
INSERT INTO daily_summary VALUES ('stock','PTX','us','2017-11-01',2.79,2.8514,2.7701,2.78,29923,'0');
INSERT INTO daily_summary VALUES ('stock','PTY','us','2017-11-01',16.691,16.721,16.631,16.681,201820,'0');
INSERT INTO daily_summary VALUES ('stock','PUB','us','2017-11-01',30.961,31.21,30.512,30.961,17473,'0');
INSERT INTO daily_summary VALUES ('stock','PUI','us','2017-11-01',28.9,28.9,28.77,28.77,2234,'0');
INSERT INTO daily_summary VALUES ('stock','PUK','us','2017-11-01',49.21,49.35,48.94,48.95,294600,'0');
INSERT INTO daily_summary VALUES ('stock','PUK_','us','2017-11-01',26.5,26.5,26.4499,26.4938,5344,'0');
INSERT INTO daily_summary VALUES ('stock','PUK_A','us','2017-11-01',26.33,26.43,26.33,26.39,7303,'0');
INSERT INTO daily_summary VALUES ('stock','PULM','us','2017-11-01',1.75,1.77,1.6,1.67,533230,'0');
INSERT INTO daily_summary VALUES ('stock','PUMP','us','2017-11-01',15.34,16.04,15.34,15.73,1775540,'0');
INSERT INTO daily_summary VALUES ('stock','PUTW','us','2017-11-01',29.63,29.68,29.6,29.66,190317,'0');
INSERT INTO daily_summary VALUES ('etf','PUW','us','2017-11-01',26.56,26.626,26.42,26.42,2065,'0');
INSERT INTO daily_summary VALUES ('stock','PVAC','us','2017-11-01',39.48,40.35,39.17,39.46,125173,'0');
INSERT INTO daily_summary VALUES ('stock','PVBC','us','2017-11-01',23.65,23.7,23.45,23.45,3474,'0');
INSERT INTO daily_summary VALUES ('stock','PVG','us','2017-11-01',11.4,11.51,11.14,11.22,1569810,'0');
INSERT INTO daily_summary VALUES ('stock','PVH','us','2017-11-01',127.37,129.92,126.91,127.91,717237,'0');
INSERT INTO daily_summary VALUES ('etf','PVI','us','2017-11-01',24.92,24.93,24.91,24.9182,17766,'0');
INSERT INTO daily_summary VALUES ('stock','PW','us','2017-11-01',6.0959,6.0959,6.0959,6.0959,502,'0');
INSERT INTO daily_summary VALUES ('etf','PWB','us','2017-11-01',40.55,40.55,40.18,40.26,24707,'0');
INSERT INTO daily_summary VALUES ('etf','PWC','us','2017-11-01',93.81,93.81,93.6774,93.6774,961,'0');
INSERT INTO daily_summary VALUES ('stock','PWOD','us','2017-11-01',48.84,48.84,47.2,47.35,11682,'0');
INSERT INTO daily_summary VALUES ('stock','PWR','us','2017-11-01',37.79,37.97,37.27,37.34,1958650,'0');
INSERT INTO daily_summary VALUES ('etf','PWV','us','2017-11-01',38.19,38.32,38.1641,38.2,85617,'0');
INSERT INTO daily_summary VALUES ('etf','PWZ','us','2017-11-01',26.04,26.13,26.04,26.05,59818,'0');
INSERT INTO daily_summary VALUES ('stock','PX','us','2017-11-01',148.61,148.79,146.99,147.92,1317710,'0');
INSERT INTO daily_summary VALUES ('stock','PXD','us','2017-11-01',150.92,155.25,150.67,153.97,2662970,'0');
INSERT INTO daily_summary VALUES ('etf','PXE','us','2017-11-01',20.9,21.35,20.9,21.3,9183,'0');
INSERT INTO daily_summary VALUES ('etf','PXF','us','2017-11-01',45.06,45.13,44.941,44.99,34928,'0');
INSERT INTO daily_summary VALUES ('etf','PXH','us','2017-11-01',21.79,21.8278,21.63,21.68,74527,'0');
INSERT INTO daily_summary VALUES ('stock','PXI','us','2017-11-01',36.15,36.95,36.15,36.83,12201,'0');
INSERT INTO daily_summary VALUES ('etf','PXJ','us','2017-11-01',9.5013,9.55,9.34,9.38,24748,'0');
INSERT INTO daily_summary VALUES ('etf','PXLG','us','2017-11-01',44.51,44.51,44.12,44.2,4493,'0');
INSERT INTO daily_summary VALUES ('etf','PXLV','us','2017-11-01',38.13,38.13,37.81,37.86,88227,'0');
INSERT INTO daily_summary VALUES ('stock','PXLW','us','2017-11-01',5.58,5.58,5.4416,5.54,215532,'0');
INSERT INTO daily_summary VALUES ('etf','PXMG','us','2017-11-01',40.31,40.31,39.9,39.9,9606,'0');
INSERT INTO daily_summary VALUES ('etf','PXMV','us','2017-11-01',30.66,30.7704,30.58,30.62,9108,'0');
INSERT INTO daily_summary VALUES ('etf','PXQ','us','2017-11-01',45.82,45.82,45.42,45.5271,1916,'0');
INSERT INTO daily_summary VALUES ('etf','PXR','us','2017-11-01',37.075,37.075,37.034,37.0529,928,'0');
INSERT INTO daily_summary VALUES ('stock','PXS','us','2017-11-01',1.88,2,1.55,1.73,9717,'0');
INSERT INTO daily_summary VALUES ('etf','PXSG','us','2017-11-01',30.77,30.77,30.7087,30.7087,1716,'0');
INSERT INTO daily_summary VALUES ('etf','PXSV','us','2017-11-01',30.03,30.14,29.7,29.83,11512,'0');
INSERT INTO daily_summary VALUES ('stock','PYDS','us','2017-11-01',2.35,2.47,2.16,2.41,1645360,'0');
INSERT INTO daily_summary VALUES ('stock','PYN','us','2017-11-01',9.91,9.95,9.89,9.921,5211,'0');
INSERT INTO daily_summary VALUES ('stock','PYPL','us','2017-11-01',72.78,72.99,71.69,72.38,6655600,'0');
INSERT INTO daily_summary VALUES ('stock','PYS','us','2017-11-01',22.25,22.25,21.53,22,8754,'0');
INSERT INTO daily_summary VALUES ('stock','PYZ','us','2017-11-01',68.14,68.215,67.5563,67.7531,1323,'0');
INSERT INTO daily_summary VALUES ('etf','PZA','us','2017-11-01',25.54,25.56,25.511,25.56,249874,'0');
INSERT INTO daily_summary VALUES ('stock','PZC','us','2017-11-01',10.44,10.459,10.39,10.4,54209,'0');
INSERT INTO daily_summary VALUES ('etf','PZD','us','2017-11-01',43.14,43.14,42.74,42.88,9327,'0');
INSERT INTO daily_summary VALUES ('stock','PZE','us','2017-11-01',13.41,13.53,13.33,13.49,8790,'0');
INSERT INTO daily_summary VALUES ('stock','PZG','us','2017-11-01',1.21,1.2566,1.16,1.2,159908,'0');
INSERT INTO daily_summary VALUES ('etf','PZI','us','2017-11-01',19.39,19.39,19.14,19.14,1082,'0');
INSERT INTO daily_summary VALUES ('stock','PZN','us','2017-11-01',11.88,12.03,11.54,11.73,30649,'0');
INSERT INTO daily_summary VALUES ('stock','PZRX','us','2017-11-01',0.63,0.66,0.601,0.6456,48003,'0');
INSERT INTO daily_summary VALUES ('etf','PZT','us','2017-11-01',24.5,24.505,24.46,24.49,3663,'0');
INSERT INTO daily_summary VALUES ('stock','PZZA','us','2017-11-01',65.255,67.497,59.138,62.027,4997080,'0');
INSERT INTO daily_summary VALUES ('stock','Q','us','2017-11-01',108.25,109,107.59,108.43,813717,'0');
INSERT INTO daily_summary VALUES ('etf','QABA','us','2017-11-01',52.91,52.97,51.84,52.14,61474,'0');
INSERT INTO daily_summary VALUES ('stock','QADA','us','2017-11-01',37.35,37.7,37.2,37.65,50354,'0');
INSERT INTO daily_summary VALUES ('etf','QAI','us','2017-11-01',30.21,30.219,30.14,30.17,95274,'0');
INSERT INTO daily_summary VALUES ('etf','QAT','us','2017-11-01',15.14,15.14,15.0186,15.09,3464,'0');
INSERT INTO daily_summary VALUES ('stock','QBAK','us','2017-11-01',7.3,7.4451,7.19,7.2,32777,'0');
INSERT INTO daily_summary VALUES ('etf','QCLN','us','2017-11-01',20.63,20.63,20.131,20.23,23407,'0');
INSERT INTO daily_summary VALUES ('stock','QCOM','us','2017-11-01',51.47,53.49,50.51,53.46,15624400,'0');
INSERT INTO daily_summary VALUES ('stock','QCP','us','2017-11-01',15.9,16.29,15.895,16.04,561588,'0');
INSERT INTO daily_summary VALUES ('stock','QCRH','us','2017-11-01',48.15,48.15,46.85,47.3,27219,'0');
INSERT INTO daily_summary VALUES ('stock','QD','us','2017-11-01',25.9,28.15,25.76,27.05,7394860,'0');
INSERT INTO daily_summary VALUES ('etf','QDEF','us','2017-11-01',42.19,42.2,42.03,42.11,9863,'0');
INSERT INTO daily_summary VALUES ('stock','QDEL','us','2017-11-01',41.11,42,40.01,40.44,388178,'0');
INSERT INTO daily_summary VALUES ('etf','QDEU','us','2017-11-01',65.38,65.38,65.1112,65.3,21087,'0');
INSERT INTO daily_summary VALUES ('etf','QDF','us','2017-11-01',42.71,42.86,42.5883,42.6642,35417,'0');
INSERT INTO daily_summary VALUES ('etf','QED','us','2017-11-01',21.25,21.25,21.24,21.25,1858,'0');
INSERT INTO daily_summary VALUES ('etf','QEFA','us','2017-11-01',64.67,64.67,64.1001,64.25,8297,'0');
INSERT INTO daily_summary VALUES ('etf','QEMM','us','2017-11-01',62.9,63.2,62.74,62.74,7217,'0');
INSERT INTO daily_summary VALUES ('stock','QEP','us','2017-11-01',9.05,9.61,9.05,9.56,5943880,'0');
INSERT INTO daily_summary VALUES ('etf','QGBR','us','2017-11-01',52.9408,52.9408,52.5501,52.6999,19407,'0');
INSERT INTO daily_summary VALUES ('stock','QGEN','us','2017-11-01',34.06,34.3199,34.06,34.26,995923,'0');
INSERT INTO daily_summary VALUES ('stock','QGTA','us','2017-11-01',24.15,24.15,24.126,24.126,2078,'0');
INSERT INTO daily_summary VALUES ('stock','QHC','us','2017-11-01',5.77,5.8,5.62,5.68,315577,'0');
INSERT INTO daily_summary VALUES ('etf','QID','us','2017-11-01',14,14.22,13.98,14.12,1954560,'0');
INSERT INTO daily_summary VALUES ('etf','QINC','us','2017-11-01',24.64,24.695,24.55,24.6022,6909,'0');
INSERT INTO daily_summary VALUES ('stock','QIWI','us','2017-11-01',17.18,17.41,17.1,17.25,137237,'0');
INSERT INTO daily_summary VALUES ('etf','QJPN','us','2017-11-01',78,78.11,77.87,78.11,48189,'0');
INSERT INTO daily_summary VALUES ('stock','QLC','us','2017-11-01',31.986,31.986,31.986,31.986,371,'0');
INSERT INTO daily_summary VALUES ('etf','QLD','us','2017-11-01',70.6,70.71,69.5168,70.05,1040120,'0');
INSERT INTO daily_summary VALUES ('etf','QLTA','us','2017-11-01',52.69,52.79,52.69,52.7345,11912,'0');
INSERT INTO daily_summary VALUES ('stock','QLYS','us','2017-11-01',57.35,57.35,53.2,55.75,1189170,'0');
INSERT INTO daily_summary VALUES ('etf','QMN','us','2017-11-01',25.551,25.584,25.55,25.55,1346,'0');
INSERT INTO daily_summary VALUES ('stock','QMOM','us','2017-11-01',28.13,28.13,28.13,28.13,21,'0');
INSERT INTO daily_summary VALUES ('stock','QNST','us','2017-11-01',9.01,9.3,8.52,8.67,530807,'0');
INSERT INTO daily_summary VALUES ('etf','QQEW','us','2017-11-01',56.5,56.5,55.93,56.04,33810,'0');
INSERT INTO daily_summary VALUES ('etf','QQQ','us','2017-11-01',152.75,152.83,151.5,152.1,29481600,'0');
INSERT INTO daily_summary VALUES ('etf','QQQC','us','2017-11-01',29.92,30.09,29.8994,29.96,6839,'0');
INSERT INTO daily_summary VALUES ('etf','QQQE','us','2017-11-01',42,42.0096,41.67,41.73,39055,'0');
INSERT INTO daily_summary VALUES ('stock','QQQX','us','2017-11-01',23.66,23.66,23.53,23.57,109119,'0');
INSERT INTO daily_summary VALUES ('etf','QQXT','us','2017-11-01',46.1408,46.18,46.03,46.07,10893,'0');
INSERT INTO daily_summary VALUES ('stock','QRHC','us','2017-11-01',1.21,1.25,1.2076,1.21,3621,'0');
INSERT INTO daily_summary VALUES ('stock','QRVO','us','2017-11-01',76.16,76.59,73.86,74.46,1953090,'0');
INSERT INTO daily_summary VALUES ('stock','QSII','us','2017-11-01',14.16,14.16,13.16,13.35,584982,'0');
INSERT INTO daily_summary VALUES ('stock','QSR','us','2017-11-01',65.07,65.89,64.62,64.66,1117000,'0');
INSERT INTO daily_summary VALUES ('etf','QTEC','us','2017-11-01',73.2,73.2,71.92,72.35,388185,'0');
INSERT INTO daily_summary VALUES ('stock','QTM','us','2017-11-01',5.35,5.35,4.96,5.04,416047,'0');
INSERT INTO daily_summary VALUES ('stock','QTNA','us','2017-11-01',15.64,15.64,14.85,15.04,237891,'0');
INSERT INTO daily_summary VALUES ('stock','QTNT','us','2017-11-01',5.12,5.3109,4.99,5.03,471597,'0');
INSERT INTO daily_summary VALUES ('stock','QTRH','us','2017-11-01',1.9,1.912,1.85,1.85,34250,'0');
INSERT INTO daily_summary VALUES ('stock','QTS','us','2017-11-01',58.06,59.08,57.5,58.56,605580,'0');
INSERT INTO daily_summary VALUES ('stock','QTWO','us','2017-11-01',42.85,42.95,41.05,41.9,150802,'0');
INSERT INTO daily_summary VALUES ('stock','QUAD','us','2017-11-01',21.59,23.75,21.05,22.41,315858,'0');
INSERT INTO daily_summary VALUES ('etf','QUAL','us','2017-11-01',79.75,79.87,79.38,79.43,158791,'0');
INSERT INTO daily_summary VALUES ('stock','QUIK','us','2017-11-01',1.62,1.62,1.52,1.54,48543,'0');
INSERT INTO daily_summary VALUES ('stock','QUMU','us','2017-11-01',3.05,3.05,2.18,2.5192,209487,'0');
INSERT INTO daily_summary VALUES ('stock','QUOT','us','2017-11-01',15.85,15.85,15.3,15.5,1475980,'0');
INSERT INTO daily_summary VALUES ('stock','QURE','us','2017-11-01',14.9,15.34,14.22,14.54,832185,'0');
INSERT INTO daily_summary VALUES ('etf','QUS','us','2017-11-01',74.52,74.52,74.3299,74.45,2603,'0');
INSERT INTO daily_summary VALUES ('stock','QVAL','us','2017-11-01',27.07,27.24,27.07,27.1,8288,'0');
INSERT INTO daily_summary VALUES ('stock','QVCA','us','2017-11-01',22.77,23.01,22.59,22.6,3221800,'0');
INSERT INTO daily_summary VALUES ('etf','QWLD','us','2017-11-01',74.1,74.1,74.089,74.089,451,'0');
INSERT INTO daily_summary VALUES ('stock','QXGG','us','2017-11-01',27.24,27.24,27.24,27.24,0,'0');
INSERT INTO daily_summary VALUES ('stock','QXMI','us','2017-11-01',25.91,25.91,25.91,25.91,0,'0');
INSERT INTO daily_summary VALUES ('stock','QXRR','us','2017-11-01',25.57,25.57,25.57,25.57,172,'0');
INSERT INTO daily_summary VALUES ('stock','QXTR','us','2017-11-01',25.512,25.512,25.512,25.512,100,'0');
INSERT INTO daily_summary VALUES ('etf','QYLD','us','2017-11-01',24.41,24.42,24.3,24.4,49105,'0');
INSERT INTO daily_summary VALUES ('stock','R','us','2017-11-01',81.81,82.415,81.52,81.73,736407,'0');
INSERT INTO daily_summary VALUES ('stock','RA','us','2017-11-01',23.9,23.92,23.81,23.89,122378,'0');
INSERT INTO daily_summary VALUES ('stock','RACE','us','2017-11-01',121,121.14,119.88,120,643765,'0');
INSERT INTO daily_summary VALUES ('stock','RAD','us','2017-11-01',1.65,1.71,1.64,1.68,19476300,'0');
INSERT INTO daily_summary VALUES ('stock','RADA','us','2017-11-01',3.25,3.32,3.2,3.23,213821,'0');
INSERT INTO daily_summary VALUES ('stock','RAIL','us','2017-11-01',18.88,19.01,18.45,18.96,88643,'0');
INSERT INTO daily_summary VALUES ('etf','RALS','us','2017-11-01',37.331,37.331,37.331,37.331,104,'0');
INSERT INTO daily_summary VALUES ('stock','RAND','us','2017-11-01',3.0217,3.0289,3.0217,3.0289,523,'0');
INSERT INTO daily_summary VALUES ('stock','RARE','us','2017-11-01',47,48.9,46.83,47.99,771294,'0');
INSERT INTO daily_summary VALUES ('stock','RARX','us','2017-11-01',13.31,13.38,12.26,12.41,132261,'0');
INSERT INTO daily_summary VALUES ('stock','RAS','us','2017-11-01',0.5328,0.57,0.5301,0.56,728135,'0');
INSERT INTO daily_summary VALUES ('stock','RAS_A','us','2017-11-01',13.22,13.3,13.2,13.3,37666,'0');
INSERT INTO daily_summary VALUES ('stock','RAS_B','us','2017-11-01',13.65,14.3118,13.5,13.51,4613,'0');
INSERT INTO daily_summary VALUES ('stock','RAS_C','us','2017-11-01',14.29,14.5,14,14.2,37936,'0');
INSERT INTO daily_summary VALUES ('stock','RATE','us','2017-11-01',13.95,13.95,13.9,13.95,308978,'0');
INSERT INTO daily_summary VALUES ('stock','RAVE','us','2017-11-01',1.57,1.59,1.5,1.54,66354,'0');
INSERT INTO daily_summary VALUES ('etf','RAVI','us','2017-11-01',75.489,75.51,75.45,75.46,5604,'0');
INSERT INTO daily_summary VALUES ('stock','RAVN','us','2017-11-01',33.95,33.95,32.65,33.25,106929,'0');
INSERT INTO daily_summary VALUES ('stock','RBA','us','2017-11-01',28.02,28.71,28.02,28.3,649520,'0');
INSERT INTO daily_summary VALUES ('stock','RBB','us','2017-11-01',24.761,24.99,24.5896,24.82,11903,'0');
INSERT INTO daily_summary VALUES ('stock','RBC','us','2017-11-01',81.7,81.85,79.65,79.8,335812,'0');
INSERT INTO daily_summary VALUES ('stock','RBCAA','us','2017-11-01',39.68,40.43,39.19,39.62,15148,'0');
INSERT INTO daily_summary VALUES ('stock','RBCN','us','2017-11-01',7.5837,7.77,7.5837,7.71,2321,'0');
INSERT INTO daily_summary VALUES ('stock','RBIN','us','2017-11-01',25.67,25.67,25.58,25.58,2300,'0');
INSERT INTO daily_summary VALUES ('stock','RBPAA','us','2017-11-01',4.48,4.52,4.38,4.38,176195,'0');
INSERT INTO daily_summary VALUES ('stock','RBS','us','2017-11-01',7.6,7.635,7.59,7.61,796153,'0');
INSERT INTO daily_summary VALUES ('stock','RBS_S','us','2017-11-01',25.65,25.66,25.6,25.6559,29864,'0');
INSERT INTO daily_summary VALUES ('stock','RBUS','us','2017-11-01',25.38,25.38,25.38,25.38,200,'0');
INSERT INTO daily_summary VALUES ('etf','RCD','us','2017-11-01',91.54,91.57,91.29,91.42,4413,'0');
INSERT INTO daily_summary VALUES ('stock','RCG','us','2017-11-01',1.41,1.47,1.41,1.43,4840,'0');
INSERT INTO daily_summary VALUES ('stock','RCI','us','2017-11-01',51.93,52.06,51.58,51.68,336755,'0');
INSERT INTO daily_summary VALUES ('stock','RCII','us','2017-11-01',9.82,10.18,9.75,10.18,2725680,'0');
INSERT INTO daily_summary VALUES ('stock','RCKY','us','2017-11-01',18.4,18.8,17.85,18.15,50265,'0');
INSERT INTO daily_summary VALUES ('stock','RCL','us','2017-11-01',124.25,124.745,120.28,120.86,2092340,'0');
INSERT INTO daily_summary VALUES ('stock','RCM','us','2017-11-01',3.9,3.97,3.75,3.92,266914,'0');
INSERT INTO daily_summary VALUES ('stock','RCMT','us','2017-11-01',5.961,5.9872,5.801,5.8401,7843,'0');
INSERT INTO daily_summary VALUES ('stock','RCOM','us','2017-11-01',26.5399,26.5399,26.5399,26.5399,0,'0');
INSERT INTO daily_summary VALUES ('stock','RCON','us','2017-11-01',1.5,1.67,1.46,1.59,778491,'0');
INSERT INTO daily_summary VALUES ('stock','RCS','us','2017-11-01',9.0888,9.1085,8.9201,8.9399,123274,'0');
INSERT INTO daily_summary VALUES ('stock','RDC','us','2017-11-01',14.85,15.055,13.89,14.51,4315760,'0');
INSERT INTO daily_summary VALUES ('stock','RDCM','us','2017-11-01',20.25,20.45,19.4,19.6,30811,'0');
INSERT INTO daily_summary VALUES ('stock','RDFN','us','2017-11-01',23.74,23.74,23.1,23.28,259067,'0');
INSERT INTO daily_summary VALUES ('stock','RDHL','us','2017-11-01',9.2,9.2,8.85,8.9,50498,'0');
INSERT INTO daily_summary VALUES ('stock','RDI','us','2017-11-01',15.73,15.73,15.495,15.55,35689,'0');
INSERT INTO daily_summary VALUES ('etf','RDIV','us','2017-11-01',34.36,34.374,33.987,34.08,46189,'0');
INSERT INTO daily_summary VALUES ('stock','RDN','us','2017-11-01',21.15,21.39,20.76,20.81,1462210,'0');
INSERT INTO daily_summary VALUES ('stock','RDNT','us','2017-11-01',11,11,10.45,10.55,104396,'0');
INSERT INTO daily_summary VALUES ('stock','RDUS','us','2017-11-01',32.18,32.67,30.81,30.85,764981,'0');
INSERT INTO daily_summary VALUES ('etf','RDVY','us','2017-11-01',28.33,28.3792,28.175,28.25,35201,'0');
INSERT INTO daily_summary VALUES ('stock','RDWR','us','2017-11-01',17.75,19.18,17.59,19.14,850674,'0');
INSERT INTO daily_summary VALUES ('stock','RDY','us','2017-11-01',35.87,35.87,35.14,35.67,409087,'0');
INSERT INTO daily_summary VALUES ('stock','RE','us','2017-11-01',238.57,239.13,234.79,234.79,728677,'0');
INSERT INTO daily_summary VALUES ('stock','RECN','us','2017-11-01',15.85,15.85,15.55,15.65,70377,'0');
INSERT INTO daily_summary VALUES ('stock','REDU','us','2017-11-01',14.03,14.36,12.26,12.92,1758900,'0');
INSERT INTO daily_summary VALUES ('stock','REED','us','2017-11-01',1.65,1.7,1.65,1.65,12260,'0');
INSERT INTO daily_summary VALUES ('stock','REEM','us','2017-11-01',27.7019,27.7019,27.7019,27.7019,180,'0');
INSERT INTO daily_summary VALUES ('etf','REET','us','2017-11-01',25.33,25.4,25.23,25.34,155196,'0');
INSERT INTO daily_summary VALUES ('stock','REFA','us','2017-11-01',27.26,27.29,27.26,27.29,7447,'0');
INSERT INTO daily_summary VALUES ('stock','REFR','us','2017-11-01',1.06,1.16,1.06,1.12,30594,'0');
INSERT INTO daily_summary VALUES ('stock','REG','us','2017-11-01',61.56,62.58,61.5572,62.12,1043780,'0');
INSERT INTO daily_summary VALUES ('stock','REGI','us','2017-11-01',12.2,12.2,11.7,11.825,210946,'0');
INSERT INTO daily_summary VALUES ('etf','REGL','us','2017-11-01',53.68,53.7016,53,53.0818,34617,'0');
INSERT INTO daily_summary VALUES ('stock','REGN','us','2017-11-01',403.81,405.504,394.56,397,1161050,'0');
INSERT INTO daily_summary VALUES ('stock','REI','us','2017-11-01',12.93,13.31,12.885,13.28,311031,'0');
INSERT INTO daily_summary VALUES ('stock','REIS','us','2017-11-01',18.35,18.5,18.25,18.35,4761,'0');
INSERT INTO daily_summary VALUES ('etf','REK','us','2017-11-01',16.22,16.23,16.1563,16.22,6262,'0');
INSERT INTO daily_summary VALUES ('stock','RELL','us','2017-11-01',6.6601,6.6799,6.5511,6.6502,11557,'0');
INSERT INTO daily_summary VALUES ('stock','RELV','us','2017-11-01',5.84,6.1,5.78,5.86,8765,'0');
INSERT INTO daily_summary VALUES ('stock','RELX','us','2017-11-01',23.32,23.32,23.0166,23.07,84988,'0');
INSERT INTO daily_summary VALUES ('stock','RELY','us','2017-11-01',1.8,1.85,1.75,1.8,155011,'0');
INSERT INTO daily_summary VALUES ('etf','REM','us','2017-11-01',45.31,45.81,45.31,45.65,435393,'0');
INSERT INTO daily_summary VALUES ('stock','REML','us','2017-11-01',29.243,29.622,28.985,29.433,4702,'0');
INSERT INTO daily_summary VALUES ('etf','REMX','us','2017-11-01',27.69,27.87,27.4,27.4,92164,'0');
INSERT INTO daily_summary VALUES ('stock','REN','us','2017-11-01',30.64,31.96,30.15,31.12,1055260,'0');
INSERT INTO daily_summary VALUES ('stock','RENN','us','2017-11-01',11.38,11.38,10.61,10.92,446501,'0');
INSERT INTO daily_summary VALUES ('stock','RENX','us','2017-11-01',22.44,22.46,22.2514,22.33,40753,'0');
INSERT INTO daily_summary VALUES ('stock','REPH','us','2017-11-01',8.73,8.89,8.5773,8.8,61839,'0');
INSERT INTO daily_summary VALUES ('stock','RES','us','2017-11-01',24.625,25.073,24.426,24.814,1406670,'0');
INSERT INTO daily_summary VALUES ('stock','RESI','us','2017-11-01',10.73,10.78,10.57,10.69,178373,'0');
INSERT INTO daily_summary VALUES ('stock','RESN','us','2017-11-01',4.68,4.68,4.42,4.6,10729,'0');
INSERT INTO daily_summary VALUES ('stock','RETA','us','2017-11-01',30.64,30.64,29.15,29.58,78427,'0');
INSERT INTO daily_summary VALUES ('etf','RETL','us','2017-11-01',25.05,25.1802,24.015,24.55,23263,'0');
INSERT INTO daily_summary VALUES ('stock','REV','us','2017-11-01',22.8,22.95,21.75,22.45,134659,'0');
INSERT INTO daily_summary VALUES ('stock','REVG','us','2017-11-01',25.94,25.94,25.36,25.63,389024,'0');
INSERT INTO daily_summary VALUES ('etf','REW','us','2017-11-01',16.83,17.1,16.83,17.1,7149,'0');
INSERT INTO daily_summary VALUES ('stock','REX','us','2017-11-01',88.72,89.729,85.71,87.11,45233,'0');
INSERT INTO daily_summary VALUES ('stock','REXR','us','2017-11-01',27.4,30.39,27.15,29.95,494095,'0');
INSERT INTO daily_summary VALUES ('stock','REXR_A','us','2017-11-01',25.1356,25.1356,25.1356,25.1356,300,'0');
INSERT INTO daily_summary VALUES ('stock','REXX','us','2017-11-01',1.97,1.97,1.84,1.88,71524,'0');
INSERT INTO daily_summary VALUES ('etf','REZ','us','2017-11-01',62.92,63.2899,62.791,63.16,25487,'0');
INSERT INTO daily_summary VALUES ('stock','RF','us','2017-11-01',15.54,15.77,15.5,15.56,9210610,'0');
INSERT INTO daily_summary VALUES ('stock','RFAP','us','2017-11-01',60.8999,60.8999,60.4401,60.4401,5449,'0');
INSERT INTO daily_summary VALUES ('stock','RFCI','us','2017-11-01',24.79,24.81,24.7601,24.8,21031,'0');
INSERT INTO daily_summary VALUES ('stock','RFDA','us','2017-11-01',30.7199,30.72,30.491,30.51,17121,'0');
INSERT INTO daily_summary VALUES ('stock','RFDI','us','2017-11-01',64.61,64.67,64.42,64.448,73147,'0');
INSERT INTO daily_summary VALUES ('stock','RFEM','us','2017-11-01',69.59,69.61,69.305,69.4207,14743,'0');
INSERT INTO daily_summary VALUES ('stock','RFEU','us','2017-11-01',65.73,65.8699,65.49,65.49,10549,'0');
INSERT INTO daily_summary VALUES ('stock','RFFC','us','2017-11-01',31.61,31.63,31.3644,31.41,8374,'0');
INSERT INTO daily_summary VALUES ('etf','RFG','us','2017-11-01',150.56,150.56,148.92,149.14,3690,'0');
INSERT INTO daily_summary VALUES ('stock','RFI','us','2017-11-01',12.54,12.63,12.54,12.61,67719,'0');
INSERT INTO daily_summary VALUES ('stock','RFIL','us','2017-11-01',2.55,2.55,2.4,2.45,26680,'0');
INSERT INTO daily_summary VALUES ('stock','RFP','us','2017-11-01',5.95,5.95,5.7,5.85,156615,'0');
INSERT INTO daily_summary VALUES ('stock','RFT','us','2017-11-01',17.9432,18.3,17.92,18.25,7725,'0');
INSERT INTO daily_summary VALUES ('stock','RFTA','us','2017-11-01',22.38,22.38,21.43,22.37,12160,'0');
INSERT INTO daily_summary VALUES ('stock','RFUN','us','2017-11-01',26.41,26.41,26.376,26.388,3065,'0');
INSERT INTO daily_summary VALUES ('etf','RFV','us','2017-11-01',65.82,65.82,65.19,65.45,2562,'0');
INSERT INTO daily_summary VALUES ('stock','RF_A','us','2017-11-01',25.39,25.42,25.38,25.4,11043,'0');
INSERT INTO daily_summary VALUES ('stock','RF_B','us','2017-11-01',28.38,28.493,28.35,28.44,13090,'0');
INSERT INTO daily_summary VALUES ('stock','RGA','us','2017-11-01',149.35,150.65,149.17,150,330667,'0');
INSERT INTO daily_summary VALUES ('stock','RGC','us','2017-11-01',16.41,16.53,15.35,15.74,4521820,'0');
INSERT INTO daily_summary VALUES ('stock','RGCO','us','2017-11-01',29.47,29.47,27.74,28.33,18839,'0');
INSERT INTO daily_summary VALUES ('stock','RGEN','us','2017-11-01',37.6,37.87,36.68,37.54,734446,'0');
INSERT INTO daily_summary VALUES ('etf','RGI','us','2017-11-01',114.12,114.12,113.12,113.3,43098,'0');
INSERT INTO daily_summary VALUES ('stock','RGLB','us','2017-11-01',26.9461,26.95,26.9461,26.95,1100,'0');
INSERT INTO daily_summary VALUES ('stock','RGLD','us','2017-11-01',84.5,84.69,83.31,83.35,500544,'0');
INSERT INTO daily_summary VALUES ('stock','RGLS','us','2017-11-01',1,1.27,0.9901,1.07,1256810,'0');
INSERT INTO daily_summary VALUES ('stock','RGNX','us','2017-11-01',30.25,31,26.5,27.5,520092,'0');
INSERT INTO daily_summary VALUES ('stock','RGR','us','2017-11-01',46.95,52.45,45.7,51.5,847403,'0');
INSERT INTO daily_summary VALUES ('stock','RGS','us','2017-11-01',15.14,15.14,14.595,14.78,177911,'0');
INSERT INTO daily_summary VALUES ('stock','RGSE','us','2017-11-01',2.14,2.1423,1.9,1.98,1309220,'0');
INSERT INTO daily_summary VALUES ('stock','RGT','us','2017-11-01',10.55,10.62,10.53,10.5899,20680,'0');
INSERT INTO daily_summary VALUES ('stock','RH','us','2017-11-01',90.51,92.18,89.51,91.75,1113720,'0');
INSERT INTO daily_summary VALUES ('stock','RHE','us','2017-11-01',0.5126,0.515,0.48,0.48,19906,'0');
INSERT INTO daily_summary VALUES ('stock','RHE_A','us','2017-11-01',14.5826,14.6293,13.05,13.58,30760,'0');
INSERT INTO daily_summary VALUES ('stock','RHI','us','2017-11-01',51.98,52.085,51.045,51.55,856175,'0');
INSERT INTO daily_summary VALUES ('stock','RHP','us','2017-11-01',66.45,66.86,65.92,66.18,176352,'0');
INSERT INTO daily_summary VALUES ('etf','RHS','us','2017-11-01',122.05,122.5,122.05,122.13,26262,'0');
INSERT INTO daily_summary VALUES ('stock','RHT','us','2017-11-01',121.3,121.64,120.16,121.26,1197570,'0');
INSERT INTO daily_summary VALUES ('stock','RIBT','us','2017-11-01',1.25,1.2627,1.24,1.25,2585,'0');
INSERT INTO daily_summary VALUES ('stock','RIC','us','2017-11-01',8.75,8.85,8.55,8.55,585702,'0');
INSERT INTO daily_summary VALUES ('stock','RICE','us','2017-11-01',28.44,28.695,28.15,28.21,5394630,'0');
INSERT INTO daily_summary VALUES ('stock','RICK','us','2017-11-01',27.62,27.98,27.06,27.25,30486,'0');
INSERT INTO daily_summary VALUES ('stock','RIF','us','2017-11-01',19.25,19.5581,19.2499,19.31,42710,'0');
INSERT INTO daily_summary VALUES ('stock','RIG','us','2017-11-01',10.69,10.93,10.41,10.48,14980600,'0');
INSERT INTO daily_summary VALUES ('stock','RIGL','us','2017-11-01',3.8,3.81,3.64,3.73,677567,'0');
INSERT INTO daily_summary VALUES ('etf','RIGS','us','2017-11-01',25.3724,25.3999,25.33,25.35,31201,'0');
INSERT INTO daily_summary VALUES ('stock','RILY','us','2017-11-01',16.8,16.95,16.5,16.65,77053,'0');
INSERT INTO daily_summary VALUES ('stock','RILYL','us','2017-11-01',25.6884,25.97,25.6884,25.97,3750,'0');
INSERT INTO daily_summary VALUES ('stock','RILYZ','us','2017-11-01',25.7751,25.7892,25.66,25.6754,11204,'0');
INSERT INTO daily_summary VALUES ('etf','RINF','us','2017-11-01',27.385,27.385,27.38,27.38,2210,'0');
INSERT INTO daily_summary VALUES ('etf','RING','us','2017-11-01',17.99,18.205,17.9,17.99,171886,'0');
INSERT INTO daily_summary VALUES ('stock','RIO','us','2017-11-01',49.46,49.47,48.97,49.03,2660910,'0');
INSERT INTO daily_summary VALUES ('stock','RIOT','us','2017-11-01',6.98,7.2,6.75,6.95,323465,'0');
INSERT INTO daily_summary VALUES ('stock','RISE','us','2017-11-01',23.3299,23.3399,23.2672,23.2672,1626,'0');
INSERT INTO daily_summary VALUES ('stock','RIV','us','2017-11-01',20.48,20.65,20.4675,20.56,89410,'0');
INSERT INTO daily_summary VALUES ('stock','RIVR','us','2017-11-01',0.2211,0.237,0.2171,0.2235,90736,'0');
INSERT INTO daily_summary VALUES ('etf','RJA','us','2017-11-01',6.05,6.0899,6.05,6.0699,11751,'0');
INSERT INTO daily_summary VALUES ('stock','RJF','us','2017-11-01',85.38,85.48,83.73,83.81,911406,'0');
INSERT INTO daily_summary VALUES ('etf','RJI','us','2017-11-01',5.22,5.245,5.2,5.22,391306,'0');
INSERT INTO daily_summary VALUES ('etf','RJN','us','2017-11-01',2.6,2.6,2.57,2.57,7108,'0');
INSERT INTO daily_summary VALUES ('stock','RKDA','us','2017-11-01',0.285,0.3,0.275,0.293,210245,'0');
INSERT INTO daily_summary VALUES ('stock','RL','us','2017-11-01',90.21,90.31,88.51,89.45,2374390,'0');
INSERT INTO daily_summary VALUES ('stock','RLGT','us','2017-11-01',4.85,4.95,4.82,4.83,65274,'0');
INSERT INTO daily_summary VALUES ('stock','RLGT_A','us','2017-11-01',26.07,26.0793,26.01,26.06,9111,'0');
INSERT INTO daily_summary VALUES ('stock','RLGY','us','2017-11-01',32.35,32.715,32.21,32.53,1102850,'0');
INSERT INTO daily_summary VALUES ('stock','RLH','us','2017-11-01',8.9,8.95,8.6,8.6,22671,'0');
INSERT INTO daily_summary VALUES ('stock','RLI','us','2017-11-01',59.47,59.545,58.37,58.73,162645,'0');
INSERT INTO daily_summary VALUES ('stock','RLJ','us','2017-11-01',21.84,22.04,21.755,22,1541760,'0');
INSERT INTO daily_summary VALUES ('stock','RLJE','us','2017-11-01',3.5484,3.6,3.54,3.54,1727,'0');
INSERT INTO daily_summary VALUES ('stock','RLJ_A','us','2017-11-01',28.13,28.1575,27.86,27.99,29856,'0');
INSERT INTO daily_summary VALUES ('stock','RLOG','us','2017-11-01',0.34,0.35,0.333,0.347,90023,'0');
INSERT INTO daily_summary VALUES ('etf','RLY','us','2017-11-01',25.553,25.7,25.515,25.61,28704,'0');
INSERT INTO daily_summary VALUES ('stock','RM','us','2017-11-01',24.92,25.08,24.12,24.98,55007,'0');
INSERT INTO daily_summary VALUES ('stock','RMAX','us','2017-11-01',66.95,66.95,65.85,66.9,60551,'0');
INSERT INTO daily_summary VALUES ('stock','RMBL','us','2017-11-01',4.6908,4.89,4.657,4.89,39898,'0');
INSERT INTO daily_summary VALUES ('stock','RMBS','us','2017-11-01',14.7,14.9,14.55,14.66,909357,'0');
INSERT INTO daily_summary VALUES ('stock','RMCF','us','2017-11-01',11.32,11.3236,11.311,11.311,1962,'0');
INSERT INTO daily_summary VALUES ('stock','RMD','us','2017-11-01',84.321,85.516,84.141,85.406,993340,'0');
INSERT INTO daily_summary VALUES ('stock','RMGN','us','2017-11-01',1.49,1.5,1.181,1.24,132991,'0');
INSERT INTO daily_summary VALUES ('stock','RMNI','us','2017-11-01',9.75,9.825,9.6,9.75,39200,'0');
INSERT INTO daily_summary VALUES ('stock','RMNIW','us','2017-11-01',0.7699,0.81,0.7515,0.7515,7800,'0');
INSERT INTO daily_summary VALUES ('stock','RMP','us','2017-11-01',20.9,20.96,20.66,20.86,385406,'0');
INSERT INTO daily_summary VALUES ('stock','RMPL_','us','2017-11-01',25.07,25.1399,25.05,25.05,16618,'0');
INSERT INTO daily_summary VALUES ('stock','RMR','us','2017-11-01',52.9,52.95,51.85,52.65,26591,'0');
INSERT INTO daily_summary VALUES ('stock','RMT','us','2017-11-01',9.45,9.45,9.27,9.3,63779,'0');
INSERT INTO daily_summary VALUES ('stock','RMTI','us','2017-11-01',6.14,6.17,5.83,5.91,260712,'0');
INSERT INTO daily_summary VALUES ('stock','RNDB','us','2017-11-01',14.9,14.99,14.75,14.75,2092,'0');
INSERT INTO daily_summary VALUES ('stock','RNDM','us','2017-11-01',51.36,51.36,51.2199,51.2199,1112,'0');
INSERT INTO daily_summary VALUES ('stock','RNDV','us','2017-11-01',20.64,20.64,20.505,20.509,5110,'0');
INSERT INTO daily_summary VALUES ('stock','RNET','us','2017-11-01',17.5,17.85,17.025,17.15,20045,'0');
INSERT INTO daily_summary VALUES ('stock','RNG','us','2017-11-01',42.5,42.7,41.75,42.4,303057,'0');
INSERT INTO daily_summary VALUES ('stock','RNGR','us','2017-11-01',11.5,12.42,11.5,12.24,42662,'0');
INSERT INTO daily_summary VALUES ('stock','RNLC','us','2017-11-01',20.85,20.8608,20.765,20.77,6401,'0');
INSERT INTO daily_summary VALUES ('stock','RNMC','us','2017-11-01',20.98,20.98,20.767,20.77,2714,'0');
INSERT INTO daily_summary VALUES ('stock','RNN','us','2017-11-01',1.81,1.81,1.7,1.73,351694,'0');
INSERT INTO daily_summary VALUES ('stock','RNP','us','2017-11-01',21.25,21.4,21.21,21.22,73046,'0');
INSERT INTO daily_summary VALUES ('stock','RNR','us','2017-11-01',135.23,137.405,134.08,136.08,679990,'0');
INSERT INTO daily_summary VALUES ('stock','RNR_C','us','2017-11-01',25.51,25.51,25.461,25.49,2550,'0');
INSERT INTO daily_summary VALUES ('stock','RNR_E','us','2017-11-01',25.44,25.51,25.36,25.51,4643,'0');
INSERT INTO daily_summary VALUES ('stock','RNSC','us','2017-11-01',21.1799,21.1799,20.84,20.84,859,'0');
INSERT INTO daily_summary VALUES ('stock','RNST','us','2017-11-01',41.79,42.11,41,41.4,119603,'0');
INSERT INTO daily_summary VALUES ('stock','RNWK','us','2017-11-01',5,5,4.72,4.76,38614,'0');
INSERT INTO daily_summary VALUES ('stock','ROAM','us','2017-11-01',24.985,25,24.86,24.86,2569,'0');
INSERT INTO daily_summary VALUES ('etf','ROBO','us','2017-11-01',40.65,40.85,40.39,40.49,735623,'0');
INSERT INTO daily_summary VALUES ('stock','ROCK','us','2017-11-01',33.45,33.95,32.5,32.5,235143,'0');
INSERT INTO daily_summary VALUES ('stock','RODM','us','2017-11-01',28.81,28.915,28.7,28.7,20784,'0');
INSERT INTO daily_summary VALUES ('stock','ROG','us','2017-11-01',153,154,150.28,151.88,528872,'0');
INSERT INTO daily_summary VALUES ('stock','ROGS','us','2017-11-01',30.8155,30.8155,30.58,30.58,1902,'0');
INSERT INTO daily_summary VALUES ('stock','ROIC','us','2017-11-01',18.01,18.27,17.91,18.18,873662,'0');
INSERT INTO daily_summary VALUES ('stock','ROK','us','2017-11-01',201.41,201.95,194.13,194.31,1639360,'0');
INSERT INTO daily_summary VALUES ('stock','ROKA','us','2017-11-01',1.17,1.24,1.12,1.16,284653,'0');
INSERT INTO daily_summary VALUES ('stock','ROKU','us','2017-11-01',20.44,20.88,19.77,19.8,1164410,'0');
INSERT INTO daily_summary VALUES ('stock','ROL','us','2017-11-01',43.806,43.935,43.099,43.487,685065,'0');
INSERT INTO daily_summary VALUES ('stock','ROLL','us','2017-11-01',124.94,125.25,122.785,123.95,46994,'0');
INSERT INTO daily_summary VALUES ('etf','ROM','us','2017-11-01',87.78,88.13,86,86.64,49220,'0');
INSERT INTO daily_summary VALUES ('etf','ROOF','us','2017-11-01',26.87,26.9935,26.73,26.9066,75658,'0');
INSERT INTO daily_summary VALUES ('stock','ROP','us','2017-11-01',259.33,260.22,257.1,258.21,361823,'0');
INSERT INTO daily_summary VALUES ('stock','RORE','us','2017-11-01',15.4687,15.4799,15.459,15.479,1110,'0');
INSERT INTO daily_summary VALUES ('stock','ROSE','us','2017-11-01',9.93,10.1,9.44,9.71,43043,'0');
INSERT INTO daily_summary VALUES ('stock','ROSEW','us','2017-11-01',1.4,1.48,1.21,1.36,59801,'0');
INSERT INTO daily_summary VALUES ('stock','ROSG','us','2017-11-01',0.8584,0.87,0.84,0.84,45613,'0');
INSERT INTO daily_summary VALUES ('stock','ROST','us','2017-11-01',63.74,64.12,63.17,63.48,1965680,'0');
INSERT INTO daily_summary VALUES ('stock','ROUS','us','2017-11-01',29.67,29.67,29.59,29.59,642,'0');
INSERT INTO daily_summary VALUES ('stock','ROX','us','2017-11-01',1.19,1.23,1.15,1.15,321112,'0');
INSERT INTO daily_summary VALUES ('stock','ROYT','us','2017-11-01',1.6798,1.6997,1.5905,1.64,183548,'0');
INSERT INTO daily_summary VALUES ('stock','RP','us','2017-11-01',43.7,44.8,43.4,44,648886,'0');
INSERT INTO daily_summary VALUES ('stock','RPAI','us','2017-11-01',12.3,12.675,12.29,12.6,2251290,'0');
INSERT INTO daily_summary VALUES ('stock','RPD','us','2017-11-01',18.09,18.32,17.91,18.24,106402,'0');
INSERT INTO daily_summary VALUES ('etf','RPG','us','2017-11-01',104,104,103,103.22,56750,'0');
INSERT INTO daily_summary VALUES ('stock','RPM','us','2017-11-01',53.66,53.66,52.75,52.86,916207,'0');
INSERT INTO daily_summary VALUES ('stock','RPRX','us','2017-11-01',0.5062,0.5096,0.4512,0.4614,873704,'0');
INSERT INTO daily_summary VALUES ('stock','RPT','us','2017-11-01',13.05,13.61,12.66,13.35,2002660,'0');
INSERT INTO daily_summary VALUES ('stock','RPT_D','us','2017-11-01',55.2,55.2,55.2,55.2,100,'0');
INSERT INTO daily_summary VALUES ('etf','RPV','us','2017-11-01',62.67,62.92,62.57,62.66,41844,'0');
INSERT INTO daily_summary VALUES ('stock','RPXC','us','2017-11-01',13.05,13.13,12.81,12.83,308343,'0');
INSERT INTO daily_summary VALUES ('stock','RQI','us','2017-11-01',12.52,12.58,12.51,12.55,220873,'0');
INSERT INTO daily_summary VALUES ('stock','RRC','us','2017-11-01',18.38,18.585,18.19,18.43,7584430,'0');
INSERT INTO daily_summary VALUES ('stock','RRD','us','2017-11-01',9.45,9.772,8.55,8.82,1594970,'0');
INSERT INTO daily_summary VALUES ('stock','RRGB','us','2017-11-01',68.35,68.7,66.75,67.8,228022,'0');
INSERT INTO daily_summary VALUES ('stock','RRR','us','2017-11-01',24.84,25.149,24.62,24.74,574086,'0');
INSERT INTO daily_summary VALUES ('stock','RRTS','us','2017-11-01',8.84,9.06,8.68,8.74,93554,'0');
INSERT INTO daily_summary VALUES ('stock','RS','us','2017-11-01',78.06,79.99,77.37,77.75,614866,'0');
INSERT INTO daily_summary VALUES ('stock','RSG','us','2017-11-01',65.19,65.29,64.2,64.55,1353830,'0');
INSERT INTO daily_summary VALUES ('stock','RSLS','us','2017-11-01',1.96,1.98,1.87,1.88,810868,'0');
INSERT INTO daily_summary VALUES ('stock','RSO','us','2017-11-01',10.23,10.41,10.23,10.36,218689,'0');
INSERT INTO daily_summary VALUES ('stock','RSO_A','us','2017-11-01',25.0061,25.0061,24.961,24.961,3600,'0');
INSERT INTO daily_summary VALUES ('stock','RSO_B','us','2017-11-01',24.77,24.772,24.705,24.7501,5959,'0');
INSERT INTO daily_summary VALUES ('stock','RSO_C','us','2017-11-01',25.3573,25.4201,25.24,25.35,7060,'0');
INSERT INTO daily_summary VALUES ('etf','RSP','us','2017-11-01',96.9,97.1,96.52,96.61,376235,'0');
INSERT INTO daily_summary VALUES ('stock','RSPP','us','2017-11-01',34.79,36.03,34.79,35.45,2733030,'0');
INSERT INTO daily_summary VALUES ('stock','RST','us','2017-11-01',10.24,10.24,9.77,10.03,161423,'0');
INSERT INTO daily_summary VALUES ('etf','RSX','us','2017-11-01',22.13,22.21,21.93,21.99,10035300,'0');
INSERT INTO daily_summary VALUES ('etf','RSXJ','us','2017-11-01',43.63,43.74,43.37,43.4183,4339,'0');
INSERT INTO daily_summary VALUES ('stock','RSYS','us','2017-11-01',1.32,1.37,1.285,1.29,96141,'0');
INSERT INTO daily_summary VALUES ('stock','RT','us','2017-11-01',2.38,2.39,2.37,2.38,2853870,'0');
INSERT INTO daily_summary VALUES ('stock','RTEC','us','2017-11-01',28.1,28.1,26.15,26.3,229901,'0');
INSERT INTO daily_summary VALUES ('etf','RTH','us','2017-11-01',82.06,82.06,81.68,81.895,7534,'0');
INSERT INTO daily_summary VALUES ('stock','RTIX','us','2017-11-01',4.55,4.55,4.3,4.35,116362,'0');
INSERT INTO daily_summary VALUES ('etf','RTM','us','2017-11-01',108.82,108.82,107.99,108.15,70133,'0');
INSERT INTO daily_summary VALUES ('stock','RTN','us','2017-11-01',181,181.67,180.19,180.26,1522240,'0');
INSERT INTO daily_summary VALUES ('stock','RTNB','us','2017-11-01',0.54,0.77,0.5001,0.6599,399466,'0');
INSERT INTO daily_summary VALUES ('stock','RTRX','us','2017-11-01',24.92,25.17,23.98,24.08,179715,'0');
INSERT INTO daily_summary VALUES ('stock','RTTR','us','2017-11-01',0.37,0.3749,0.3421,0.3514,1511140,'0');
INSERT INTO daily_summary VALUES ('stock','RUBI','us','2017-11-01',3.59,3.59,3.35,3.37,448678,'0');
INSERT INTO daily_summary VALUES ('stock','RUN','us','2017-11-01',5.67,5.84,5.66,5.74,895458,'0');
INSERT INTO daily_summary VALUES ('stock','RUSHA','us','2017-11-01',51.42,51.63,50.52,50.87,145721,'0');
INSERT INTO daily_summary VALUES ('stock','RUSHB','us','2017-11-01',47.57,47.81,47.22,47.22,4013,'0');
INSERT INTO daily_summary VALUES ('etf','RUSL','us','2017-11-01',51.08,51.54,49.8194,50.23,90389,'0');
INSERT INTO daily_summary VALUES ('etf','RUSS','us','2017-11-01',23.5,24.05,23.12,23.85,73135,'0');
INSERT INTO daily_summary VALUES ('stock','RUTH','us','2017-11-01',20.956,21.168,20.259,20.409,210713,'0');
INSERT INTO daily_summary VALUES ('stock','RVEN','us','2017-11-01',4.59,4.59,4.42,4.58,607,'0');
INSERT INTO daily_summary VALUES ('stock','RVLT','us','2017-11-01',5.6,5.6,5.37,5.46,91456,'0');
INSERT INTO daily_summary VALUES ('stock','RVNC','us','2017-11-01',26.25,26.45,25.35,25.8,134544,'0');
INSERT INTO daily_summary VALUES ('etf','RVNU','us','2017-11-01',26.81,26.82,26.74,26.74,7130,'0');
INSERT INTO daily_summary VALUES ('stock','RVP','us','2017-11-01',0.6736,0.6891,0.6601,0.6754,65159,'0');
INSERT INTO daily_summary VALUES ('stock','RVRS','us','2017-11-01',15.16,15.16,15,15.0033,7018,'0');
INSERT INTO daily_summary VALUES ('stock','RVSB','us','2017-11-01',8.97,8.975,8.7495,8.83,89861,'0');
INSERT INTO daily_summary VALUES ('stock','RVT','us','2017-11-01',16.06,16.068,15.89,15.91,190320,'0');
INSERT INTO daily_summary VALUES ('stock','RWC','us','2017-11-01',3.5378,3.65,3.5378,3.6,16708,'0');
INSERT INTO daily_summary VALUES ('etf','RWJ','us','2017-11-01',67.9,67.9,66.46,66.94,40334,'0');
INSERT INTO daily_summary VALUES ('etf','RWK','us','2017-11-01',58.23,58.23,57.491,57.8299,17559,'0');
INSERT INTO daily_summary VALUES ('etf','RWL','us','2017-11-01',48.38,48.5594,48.35,48.35,50498,'0');
INSERT INTO daily_summary VALUES ('stock','RWLK','us','2017-11-01',1.3,1.4,1.3,1.4,229125,'0');
INSERT INTO daily_summary VALUES ('etf','RWM','us','2017-11-01',43.01,43.91,43.01,43.64,417504,'0');
INSERT INTO daily_summary VALUES ('etf','RWO','us','2017-11-01',47.35,47.54,47.33,47.41,173760,'0');
INSERT INTO daily_summary VALUES ('etf','RWR','us','2017-11-01',91.65,92.16,91.5,92.05,89071,'0');
INSERT INTO daily_summary VALUES ('stock','RWT','us','2017-11-01',15.85,16.09,15.77,15.85,509600,'0');
INSERT INTO daily_summary VALUES ('etf','RWW','us','2017-11-01',66.47,66.47,66.11,66.11,2112,'0');
INSERT INTO daily_summary VALUES ('etf','RWX','us','2017-11-01',38.39,38.4499,38.285,38.3,505759,'0');
INSERT INTO daily_summary VALUES ('etf','RXD','us','2017-11-01',31.59,31.8,31.59,31.8,265,'0');
INSERT INTO daily_summary VALUES ('stock','RXDX','us','2017-11-01',15.5,15.65,14.9,15.05,1173980,'0');
INSERT INTO daily_summary VALUES ('etf','RXI','us','2017-11-01',104.66,104.815,104.398,104.466,4790,'0');
INSERT INTO daily_summary VALUES ('stock','RXII','us','2017-11-01',0.55,0.5662,0.501,0.5186,207496,'0');
INSERT INTO daily_summary VALUES ('stock','RXIIW','us','2017-11-01',0.14,0.14,0.14,0.14,4000,'0');
INSERT INTO daily_summary VALUES ('etf','RXL','us','2017-11-01',84.295,85.1199,84.295,84.34,10698,'0');
INSERT INTO daily_summary VALUES ('stock','RXN','us','2017-11-01',25.81,25.81,25.2,25.44,836825,'0');
INSERT INTO daily_summary VALUES ('stock','RXN_A','us','2017-11-01',57.97,57.97,57.09,57.09,915,'0');
INSERT INTO daily_summary VALUES ('stock','RY','us','2017-11-01',78.35,78.45,78.03,78.31,618738,'0');
INSERT INTO daily_summary VALUES ('stock','RYAAY','us','2017-11-01',113.92,114.67,112.95,113.15,769037,'0');
INSERT INTO daily_summary VALUES ('stock','RYAM','us','2017-11-01',14.5,14.635,14.2,14.23,592605,'0');
INSERT INTO daily_summary VALUES ('stock','RYB','us','2017-11-01',25.73,26.3999,25.21,25.75,192648,'0');
INSERT INTO daily_summary VALUES ('etf','RYE','us','2017-11-01',54.76,55.55,54.76,55.18,19027,'0');
INSERT INTO daily_summary VALUES ('etf','RYF','us','2017-11-01',42.17,42.32,41.94,41.99,417466,'0');
INSERT INTO daily_summary VALUES ('etf','RYH','us','2017-11-01',175.31,175.67,174.73,174.82,79392,'0');
INSERT INTO daily_summary VALUES ('stock','RYI','us','2017-11-01',9,9.7,8.875,9,294265,'0');
INSERT INTO daily_summary VALUES ('etf','RYJ','us','2017-11-01',41.585,41.585,41.14,41.3448,5649,'0');
INSERT INTO daily_summary VALUES ('stock','RYN','us','2017-11-01',30.01,30.22,29.9,30.06,378956,'0');
INSERT INTO daily_summary VALUES ('etf','RYT','us','2017-11-01',143.09,143.12,141.36,142.04,208871,'0');
INSERT INTO daily_summary VALUES ('stock','RYTM','us','2017-11-01',25.55,26.73,25.2,26.1,243215,'0');
INSERT INTO daily_summary VALUES ('etf','RYU','us','2017-11-01',88.78,88.78,87.97,87.98,13144,'0');
INSERT INTO daily_summary VALUES ('stock','RZA','us','2017-11-01',28.13,28.22,28.0454,28.09,13501,'0');
INSERT INTO daily_summary VALUES ('stock','RZB','us','2017-11-01',27.81,27.86,27.75,27.8321,6611,'0');
INSERT INTO daily_summary VALUES ('etf','RZG','us','2017-11-01',111.416,111.416,109.716,109.9,2672,'0');
INSERT INTO daily_summary VALUES ('etf','RZV','us','2017-11-01',71.13,71.13,69.65,70.06,9704,'0');
INSERT INTO daily_summary VALUES ('stock','S','us','2017-11-01',6.58,6.6,6.34,6.46,13788100,'0');
INSERT INTO daily_summary VALUES ('stock','SA','us','2017-11-01',13.65,13.7,13.1,13.15,511753,'0');
INSERT INTO daily_summary VALUES ('etf','SAA','us','2017-11-01',95.048,95.048,92.8594,93,989,'0');
INSERT INTO daily_summary VALUES ('stock','SAB','us','2017-11-01',26.25,26.25,26.25,26.25,373,'0');
INSERT INTO daily_summary VALUES ('stock','SABR','us','2017-11-01',19.7,20.05,19.29,19.55,3885570,'0');
INSERT INTO daily_summary VALUES ('stock','SACH','us','2017-11-01',4.1871,4.2553,4.0897,4.1871,107758,'0');
INSERT INTO daily_summary VALUES ('stock','SAEX','us','2017-11-01',2.21,2.3,1.99,2.1,85723,'0');
INSERT INTO daily_summary VALUES ('stock','SAFE','us','2017-11-01',17.89,18.05,17.81,17.98,87390,'0');
INSERT INTO daily_summary VALUES ('stock','SAFM','us','2017-11-01',150.71,152.91,147.87,148.23,475808,'0');
INSERT INTO daily_summary VALUES ('stock','SAFT','us','2017-11-01',82.6,82.65,80.55,81.55,75071,'0');
INSERT INTO daily_summary VALUES ('stock','SAGE','us','2017-11-01',63.37,64.835,61.08,61.83,375937,'0');
INSERT INTO daily_summary VALUES ('etf','SAGG','us','2017-11-01',31.1548,31.1548,31.1548,31.1548,320,'0');
INSERT INTO daily_summary VALUES ('stock','SAH','us','2017-11-01',20.15,20.35,19.725,20.35,243690,'0');
INSERT INTO daily_summary VALUES ('stock','SAIA','us','2017-11-01',65.65,65.65,64.1,64.4,195832,'0');
INSERT INTO daily_summary VALUES ('stock','SAIC','us','2017-11-01',74,74.29,72.3,72.86,203754,'0');
INSERT INTO daily_summary VALUES ('stock','SAL','us','2017-11-01',46.711,47.158,46.164,46.164,774,'0');
INSERT INTO daily_summary VALUES ('stock','SALM','us','2017-11-01',6.35,6.5,6.25,6.25,16909,'0');
INSERT INTO daily_summary VALUES ('stock','SALT','us','2017-11-01',8.1,8.1,7.625,7.8,516721,'0');
INSERT INTO daily_summary VALUES ('stock','SAM','us','2017-11-01',178.25,180.9,176.8,178.3,158782,'0');
INSERT INTO daily_summary VALUES ('stock','SAMG','us','2017-11-01',16.45,16.45,16.2,16.25,17932,'0');
INSERT INTO daily_summary VALUES ('stock','SAN','us','2017-11-01',6.75,6.775,6.73,6.77,4339200,'0');
INSERT INTO daily_summary VALUES ('stock','SAND','us','2017-11-01',4.35,4.385,4.25,4.25,1437050,'0');
INSERT INTO daily_summary VALUES ('stock','SANM','us','2017-11-01',32.65,33.35,32.1,33.2,1527000,'0');
INSERT INTO daily_summary VALUES ('stock','SANW','us','2017-11-01',3.1629,3.2,3.05,3.1,26115,'0');
INSERT INTO daily_summary VALUES ('stock','SAN_A','us','2017-11-01',26.1,26.14,26.09,26.1103,4065,'0');
INSERT INTO daily_summary VALUES ('stock','SAN_B','us','2017-11-01',23.95,24.08,23.89,23.9,15612,'0');
INSERT INTO daily_summary VALUES ('stock','SAN_C','us','2017-11-01',25.79,25.84,25.79,25.84,1295,'0');
INSERT INTO daily_summary VALUES ('stock','SAN_I','us','2017-11-01',25.66,25.76,25.62,25.75,3930,'0');
INSERT INTO daily_summary VALUES ('stock','SAP','us','2017-11-01',116.36,116.9,115.93,116.27,916153,'0');
INSERT INTO daily_summary VALUES ('stock','SAR','us','2017-11-01',22.47,22.47,22.352,22.44,15203,'0');
INSERT INTO daily_summary VALUES ('stock','SASR','us','2017-11-01',40.386,40.605,39.353,39.91,86666,'0');
INSERT INTO daily_summary VALUES ('stock','SATS','us','2017-11-01',56.2,56.2,53.34,53.95,251714,'0');
INSERT INTO daily_summary VALUES ('stock','SAUC','us','2017-11-01',1.7122,1.7122,1.62,1.62,33791,'0');
INSERT INTO daily_summary VALUES ('stock','SAVE','us','2017-11-01',36.74,37.4,36.44,36.64,2065700,'0');
INSERT INTO daily_summary VALUES ('stock','SB','us','2017-11-01',3.58,3.6,3.32,3.4,672376,'0');
INSERT INTO daily_summary VALUES ('stock','SBAC','us','2017-11-01',157.25,157.64,151.43,152.58,1937440,'0');
INSERT INTO daily_summary VALUES ('etf','SBB','us','2017-11-01',35.54,35.7,35.539,35.7,2619,'0');
INSERT INTO daily_summary VALUES ('stock','SBBP','us','2017-11-01',6,6,5.65,5.8,201820,'0');
INSERT INTO daily_summary VALUES ('stock','SBBX','us','2017-11-01',25.29,25.539,25.24,25.539,41716,'0');
INSERT INTO daily_summary VALUES ('stock','SBCF','us','2017-11-01',25.01,25.18,24.42,24.69,209843,'0');
INSERT INTO daily_summary VALUES ('stock','SBCP','us','2017-11-01',23.49,23.79,23.15,23.28,20303,'0');
INSERT INTO daily_summary VALUES ('stock','SBFG','us','2017-11-01',17.43,17.43,17.43,17.43,544,'0');
INSERT INTO daily_summary VALUES ('stock','SBGI','us','2017-11-01',30.5,31.35,29.6,31.05,3338810,'0');
INSERT INTO daily_summary VALUES ('stock','SBGL','us','2017-11-01',5.25,5.37,5.23,5.26,2766120,'0');
INSERT INTO daily_summary VALUES ('stock','SBH','us','2017-11-01',17.3,17.45,17.0548,17.42,1746840,'0');
INSERT INTO daily_summary VALUES ('stock','SBI','us','2017-11-01',9.49,9.53,9.4644,9.53,35733,'0');
INSERT INTO daily_summary VALUES ('etf','SBIO','us','2017-11-01',30.61,30.68,30.0301,30.07,38285,'0');
INSERT INTO daily_summary VALUES ('stock','SBLK','us','2017-11-01',10.85,11,10.5,10.67,495688,'0');
INSERT INTO daily_summary VALUES ('stock','SBLKL','us','2017-11-01',25.15,25.2415,25.0678,25.16,14393,'0');
INSERT INTO daily_summary VALUES ('stock','SBNA','us','2017-11-01',24.4,24.75,24.4,24.7,1334,'0');
INSERT INTO daily_summary VALUES ('stock','SBNY','us','2017-11-01',131.31,131.96,128.85,129.7,414960,'0');
INSERT INTO daily_summary VALUES ('stock','SBOT','us','2017-11-01',1.15,1.16,1.11,1.12,53927,'0');
INSERT INTO daily_summary VALUES ('stock','SBOW','us','2017-11-01',22.83,22.9572,22.26,22.27,28686,'0');
INSERT INTO daily_summary VALUES ('stock','SBPH','us','2017-11-01',15.03,15.41,14.79,15.12,38453,'0');
INSERT INTO daily_summary VALUES ('stock','SBR','us','2017-11-01',42.65,42.8,42.3946,42.65,12375,'0');
INSERT INTO daily_summary VALUES ('stock','SBRA','us','2017-11-01',19.88,20.14,19.75,20.14,1558500,'0');
INSERT INTO daily_summary VALUES ('stock','SBRAP','us','2017-11-01',25.5748,25.689,25.5748,25.63,1706,'0');
INSERT INTO daily_summary VALUES ('stock','SBS','us','2017-11-01',9.18,9.37,9.14,9.27,2113840,'0');
INSERT INTO daily_summary VALUES ('stock','SBSI','us','2017-11-01',35.84,36.28,34.96,35.56,98317,'0');
INSERT INTO daily_summary VALUES ('stock','SBUX','us','2017-11-01',55.1,55.59,54.94,55.13,7189760,'0');
INSERT INTO daily_summary VALUES ('stock','SB_C','us','2017-11-01',23.76,23.76,23.667,23.74,1454,'0');
INSERT INTO daily_summary VALUES ('stock','SB_D','us','2017-11-01',23.7066,23.7114,23.61,23.63,17089,'0');
INSERT INTO daily_summary VALUES ('stock','SC','us','2017-11-01',16.76,16.97,16.651,16.79,2814330,'0');
INSERT INTO daily_summary VALUES ('stock','SCA','us','2017-11-01',25.51,25.52,25.51,25.5197,1050,'0');
INSERT INTO daily_summary VALUES ('stock','SCAC','us','2017-11-01',9.98,10.02,9.98,9.98,17000,'0');
INSERT INTO daily_summary VALUES ('stock','SCACU','us','2017-11-01',10.68,10.68,10.68,10.68,100,'0');
INSERT INTO daily_summary VALUES ('stock','SCAP','us','2017-11-01',34.26,34.26,34.26,34.26,300,'0');
INSERT INTO daily_summary VALUES ('stock','SCCO','us','2017-11-01',43.564,43.981,42.599,42.758,932641,'0');
INSERT INTO daily_summary VALUES ('stock','SCD','us','2017-11-01',14.26,14.27,14.02,14.17,81285,'0');
INSERT INTO daily_summary VALUES ('stock','SCE_C','us','2017-11-01',24.617,24.694,24.617,24.694,1094,'0');
INSERT INTO daily_summary VALUES ('stock','SCE_D','us','2017-11-01',25.05,25.24,25.05,25.24,1545,'0');
INSERT INTO daily_summary VALUES ('stock','SCE_E','us','2017-11-01',25.6923,25.6923,25.6867,25.6867,433,'0');
INSERT INTO daily_summary VALUES ('stock','SCE_G','us','2017-11-01',25.74,25.89,25.72,25.75,196004,'0');
INSERT INTO daily_summary VALUES ('stock','SCE_H','us','2017-11-01',26.92,27.15,26.92,27.07,16607,'0');
INSERT INTO daily_summary VALUES ('stock','SCE_J','us','2017-11-01',27.32,27.35,26.9462,27.0416,10293,'0');
INSERT INTO daily_summary VALUES ('stock','SCE_K','us','2017-11-01',27.43,27.6,27.41,27.47,15229,'0');
INSERT INTO daily_summary VALUES ('stock','SCE_L','us','2017-11-01',25.2,25.2,25.042,25.05,28275,'0');
INSERT INTO daily_summary VALUES ('stock','SCG','us','2017-11-01',43.22,43.63,42.57,42.63,2373090,'0');
INSERT INTO daily_summary VALUES ('etf','SCHA','us','2017-11-01',68.42,68.44,67.29,67.62,322133,'0');
INSERT INTO daily_summary VALUES ('etf','SCHB','us','2017-11-01',62.46,62.55,62.15,62.28,367696,'0');
INSERT INTO daily_summary VALUES ('etf','SCHC','us','2017-11-01',36.41,36.49,36.32,36.38,167365,'0');
INSERT INTO daily_summary VALUES ('etf','SCHD','us','2017-11-01',48.64,48.738,48.551,48.67,441219,'0');
INSERT INTO daily_summary VALUES ('etf','SCHE','us','2017-11-01',27.72,27.79,27.59,27.62,620652,'0');
INSERT INTO daily_summary VALUES ('etf','SCHF','us','2017-11-01',34.27,34.31,34.16,34.16,1258720,'0');
INSERT INTO daily_summary VALUES ('etf','SCHG','us','2017-11-01',68.37,68.37,67.9328,68.11,213791,'0');
INSERT INTO daily_summary VALUES ('etf','SCHH','us','2017-11-01',40.73,40.979,40.66,40.9,1147990,'0');
INSERT INTO daily_summary VALUES ('stock','SCHK','us','2017-11-01',25.37,25.39,25.26,25.3,127525,'0');
INSERT INTO daily_summary VALUES ('stock','SCHL','us','2017-11-01',37.11,37.26,36.49,36.87,112045,'0');
INSERT INTO daily_summary VALUES ('etf','SCHM','us','2017-11-01',51.34,51.399,50.942,51.02,151441,'0');
INSERT INTO daily_summary VALUES ('stock','SCHN','us','2017-11-01',29.694,30.563,29.098,29.396,314327,'0');
INSERT INTO daily_summary VALUES ('etf','SCHO','us','2017-11-01',50.26,50.27,50.23,50.24,198332,'0');
INSERT INTO daily_summary VALUES ('etf','SCHP','us','2017-11-01',55.21,55.3033,55.17,55.2,186851,'0');
INSERT INTO daily_summary VALUES ('etf','SCHR','us','2017-11-01',53.67,53.749,53.66,53.67,111096,'0');
INSERT INTO daily_summary VALUES ('etf','SCHV','us','2017-11-01',52.92,53.0432,52.7866,52.86,169681,'0');
INSERT INTO daily_summary VALUES ('stock','SCHW','us','2017-11-01',45.009,45.233,44.12,44.16,6477420,'0');
INSERT INTO daily_summary VALUES ('stock','SCHW_C','us','2017-11-01',27.3,27.5,27.3,27.5,27572,'0');
INSERT INTO daily_summary VALUES ('stock','SCHW_D','us','2017-11-01',27.27,27.44,27.27,27.36,30200,'0');
INSERT INTO daily_summary VALUES ('etf','SCHX','us','2017-11-01',61.68,61.75,61.417,61.54,444412,'0');
INSERT INTO daily_summary VALUES ('etf','SCHZ','us','2017-11-01',52.14,52.23,52.09,52.18,364315,'0');
INSERT INTO daily_summary VALUES ('stock','SCI','us','2017-11-01',35.51,35.65,35.01,35.06,1184770,'0');
INSERT INTO daily_summary VALUES ('etf','SCID','us','2017-11-01',28.64,28.64,28.45,28.45,614,'0');
INSERT INTO daily_summary VALUES ('etf','SCIF','us','2017-11-01',62.93,63.4899,62.93,63.21,39219,'0');
INSERT INTO daily_summary VALUES ('etf','SCIJ','us','2017-11-01',32.0392,32.0392,32.0392,32.0392,405,'0');
INSERT INTO daily_summary VALUES ('etf','SCIN','us','2017-11-01',21.7699,22,21.7699,21.79,9314,'0');
INSERT INTO daily_summary VALUES ('etf','SCIU','us','2017-11-01',30.22,30.23,30.05,30.0999,6780,'0');
INSERT INTO daily_summary VALUES ('etf','SCIX','us','2017-11-01',26.0599,26.0599,26.04,26.04,1006,'0');
INSERT INTO daily_summary VALUES ('etf','SCJ','us','2017-11-01',77.26,77.26,77.05,77.1,114412,'0');
INSERT INTO daily_summary VALUES ('stock','SCKT','us','2017-11-01',3.8,3.83,3.685,3.7,32196,'0');
INSERT INTO daily_summary VALUES ('stock','SCL','us','2017-11-01',80.5,86.25,79.83,80.26,66656,'0');
INSERT INTO daily_summary VALUES ('stock','SCM','us','2017-11-01',13.91,13.975,13.5,13.96,103415,'0');
INSERT INTO daily_summary VALUES ('stock','SCMP','us','2017-11-01',10.75,10.85,9.55,9.6,751177,'0');
INSERT INTO daily_summary VALUES ('etf','SCO','us','2017-11-01',29.82,30.93,29.64,30.48,2287110,'0');
INSERT INTO daily_summary VALUES ('stock','SCON','us','2017-11-01',0.95,1.09,0.95,1,159642,'0');
INSERT INTO daily_summary VALUES ('stock','SCS','us','2017-11-01',14.65,14.65,14.275,14.35,543602,'0');
INSERT INTO daily_summary VALUES ('stock','SCSC','us','2017-11-01',43.4,43.8,42.3,42.75,66576,'0');
INSERT INTO daily_summary VALUES ('etf','SCTO','us','2017-11-01',25.605,25.605,25.55,25.55,2815,'0');
INSERT INTO daily_summary VALUES ('stock','SCVL','us','2017-11-01',18.92,19.1213,17.94,17.96,272024,'0');
INSERT INTO daily_summary VALUES ('stock','SCWX','us','2017-11-01',10.05,10.07,9.76,10.02,109669,'0');
INSERT INTO daily_summary VALUES ('stock','SCX','us','2017-11-01',8.5,8.7027,8.4,8.5,3304,'0');
INSERT INTO daily_summary VALUES ('stock','SCYX','us','2017-11-01',1.99,2.02,1.88,1.95,109579,'0');
INSERT INTO daily_summary VALUES ('etf','SCZ','us','2017-11-01',63.38,63.405,63.12,63.2,953452,'0');
INSERT INTO daily_summary VALUES ('stock','SD','us','2017-11-01',19.06,19.545,18.74,19.21,190165,'0');
INSERT INTO daily_summary VALUES ('etf','SDEM','us','2017-11-01',16.01,16.0435,15.9382,15.9382,17417,'0');
INSERT INTO daily_summary VALUES ('etf','SDIV','us','2017-11-01',21.52,21.55,21.44,21.47,1394390,'0');
INSERT INTO daily_summary VALUES ('stock','SDLP','us','2017-11-01',3.8018,3.8797,3.792,3.8797,225073,'0');
INSERT INTO daily_summary VALUES ('etf','SDOG','us','2017-11-01',44.38,44.5,44.18,44.21,98569,'0');
INSERT INTO daily_summary VALUES ('etf','SDOW','us','2017-11-01',23.08,23.41,23.0234,23.28,825370,'0');
INSERT INTO daily_summary VALUES ('etf','SDP','us','2017-11-01',23.8,23.99,23.8,23.903,698,'0');
INSERT INTO daily_summary VALUES ('stock','SDPI','us','2017-11-01',0.8286,0.8286,0.7952,0.7952,21901,'0');
INSERT INTO daily_summary VALUES ('stock','SDR','us','2017-11-01',1.2815,1.3007,1.2728,1.2815,55432,'0');
INSERT INTO daily_summary VALUES ('stock','SDRL','us','2017-11-01',0.2605,0.2684,0.2558,0.263,1923290,'0');
INSERT INTO daily_summary VALUES ('etf','SDS','us','2017-11-01',44.3,44.62,44.15,44.47,2695180,'0');
INSERT INTO daily_summary VALUES ('stock','SDT','us','2017-11-01',1.2803,1.2803,1.2609,1.2609,39429,'0');
INSERT INTO daily_summary VALUES ('etf','SDY','us','2017-11-01',92.72,92.896,92.3948,92.45,301302,'0');
INSERT INTO daily_summary VALUES ('etf','SDYL','us','2017-11-01',78.118,78.118,77.669,77.669,738,'0');
INSERT INTO daily_summary VALUES ('stock','SE','us','2017-11-01',15.33,15.49,15.19,15.43,1247950,'0');
INSERT INTO daily_summary VALUES ('etf','SEA','us','2017-11-01',11.83,11.9242,11.78,11.82,40872,'0');
INSERT INTO daily_summary VALUES ('stock','SEAC','us','2017-11-01',2.65,2.77,2.65,2.65,70509,'0');
INSERT INTO daily_summary VALUES ('stock','SEAS','us','2017-11-01',11.56,11.805,11.4,11.68,2139420,'0');
INSERT INTO daily_summary VALUES ('stock','SEB','us','2017-11-01',4423.31,4443.45,4334.48,4393.43,224,'0');
INSERT INTO daily_summary VALUES ('stock','SECO','us','2017-11-01',7.13,7.33,7.02,7.19,155840,'0');
INSERT INTO daily_summary VALUES ('stock','SECT','us','2017-11-01',26.56,26.5799,26.3499,26.44,30499,'0');
INSERT INTO daily_summary VALUES ('stock','SEDG','us','2017-11-01',33.1,33.2,32.44,32.85,661076,'0');
INSERT INTO daily_summary VALUES ('stock','SEE','us','2017-11-01',44.31,44.49,43.82,43.96,1795890,'0');
INSERT INTO daily_summary VALUES ('stock','SEED','us','2017-11-01',1.44,1.44,1.42,1.42,21364,'0');
INSERT INTO daily_summary VALUES ('etf','SEF','us','2017-11-01',12.02,12.0852,12.02,12.075,6068,'0');
INSERT INTO daily_summary VALUES ('stock','SEIC','us','2017-11-01',64.73,65.08,63.9847,64.38,432934,'0');
INSERT INTO daily_summary VALUES ('stock','SELB','us','2017-11-01',21.4,22.01,20.68,21.04,48601,'0');
INSERT INTO daily_summary VALUES ('stock','SELF','us','2017-11-01',4.7122,4.7122,4.66,4.71,9102,'0');
INSERT INTO daily_summary VALUES ('stock','SEM','us','2017-11-01',19.15,19.35,18.5,18.6,814763,'0');
INSERT INTO daily_summary VALUES ('stock','SEMG','us','2017-11-01',26.35,27.2,26.25,26.95,788258,'0');
INSERT INTO daily_summary VALUES ('stock','SENEA','us','2017-11-01',36.15,36.15,34.5,34.8,17592,'0');
INSERT INTO daily_summary VALUES ('stock','SENS','us','2017-11-01',2.92,2.92,2.5,2.57,1032050,'0');
INSERT INTO daily_summary VALUES ('stock','SEP','us','2017-11-01',42.532,43.011,42.267,42.915,253775,'0');
INSERT INTO daily_summary VALUES ('stock','SERV','us','2017-11-01',47.01,47.34,46.43,47.16,922627,'0');
INSERT INTO daily_summary VALUES ('stock','SF','us','2017-11-01',53.09,53.66,52.32,52.54,312652,'0');
INSERT INTO daily_summary VALUES ('stock','SFB','us','2017-11-01',24.98,25.05,24.98,25.02,80982,'0');
INSERT INTO daily_summary VALUES ('stock','SFBC','us','2017-11-01',33.75,33.75,33.75,33.75,111,'0');
INSERT INTO daily_summary VALUES ('stock','SFBS','us','2017-11-01',41.46,41.5,40.74,41.03,118018,'0');
INSERT INTO daily_summary VALUES ('stock','SFE','us','2017-11-01',14.1,14.25,14,14.1,215424,'0');
INSERT INTO daily_summary VALUES ('stock','SFHY','us','2017-11-01',49.99,49.99,49.99,49.99,178,'0');
INSERT INTO daily_summary VALUES ('stock','SFIG','us','2017-11-01',49.77,49.77,49.77,49.77,0,'0');
INSERT INTO daily_summary VALUES ('stock','SFL','us','2017-11-01',14.95,15.05,14.875,15.05,652608,'0');
INSERT INTO daily_summary VALUES ('stock','SFLY','us','2017-11-01',43.02,43.43,41.6,41.97,796985,'0');
INSERT INTO daily_summary VALUES ('stock','SFM','us','2017-11-01',18.49,19.079,18.42,18.87,3176320,'0');
INSERT INTO daily_summary VALUES ('stock','SFNC','us','2017-11-01',57.95,58.3,56.6,57.35,202570,'0');
INSERT INTO daily_summary VALUES ('stock','SFR','us','2017-11-01',36.5,36.81,36.25,36.57,577084,'0');
INSERT INTO daily_summary VALUES ('stock','SFS','us','2017-11-01',6,6.2,5.95,6.1,573104,'0');
INSERT INTO daily_summary VALUES ('stock','SFST','us','2017-11-01',38.5,38.9,38.35,38.6,8002,'0');
INSERT INTO daily_summary VALUES ('stock','SFUN','us','2017-11-01',4.55,4.635,4.51,4.58,1420110,'0');
INSERT INTO daily_summary VALUES ('stock','SF_A','us','2017-11-01',26.79,26.88,26.7,26.88,19093,'0');
INSERT INTO daily_summary VALUES ('stock','SGA','us','2017-11-01',43.75,43.75,43.35,43.6,1447,'0');
INSERT INTO daily_summary VALUES ('stock','SGB','us','2017-11-01',21,21,20.5,20.95,12845,'0');
INSERT INTO daily_summary VALUES ('stock','SGBX','us','2017-11-01',5.6,5.65,5.11,5.26,47167,'0');
INSERT INTO daily_summary VALUES ('stock','SGC','us','2017-11-01',23.52,23.6,23.06,23.38,12958,'0');
INSERT INTO daily_summary VALUES ('etf','SGDJ','us','2017-11-01',32.8,33.1,32.39,32.39,18442,'0');
INSERT INTO daily_summary VALUES ('etf','SGDM','us','2017-11-01',20.46,20.58,20.28,20.3626,8740,'0');
INSERT INTO daily_summary VALUES ('stock','SGEN','us','2017-11-01',61.59,61.59,60.12,60.29,1261490,'0');
INSERT INTO daily_summary VALUES ('stock','SGF','us','2017-11-01',12.31,12.5,12.31,12.5,19969,'0');
INSERT INTO daily_summary VALUES ('etf','SGG','us','2017-11-01',29.9,30.2,29.67,29.81,59240,'0');
INSERT INTO daily_summary VALUES ('stock','SGH','us','2017-11-01',32.1,32.1623,29.88,29.96,207208,'0');
INSERT INTO daily_summary VALUES ('stock','SGLB','us','2017-11-01',1.69,2.1,1.69,1.74,83589,'0');
INSERT INTO daily_summary VALUES ('stock','SGMA','us','2017-11-01',9.54,9.69,9.52,9.63,18124,'0');
INSERT INTO daily_summary VALUES ('stock','SGMO','us','2017-11-01',12.6,12.85,11.5,11.6,1379630,'0');
INSERT INTO daily_summary VALUES ('stock','SGMS','us','2017-11-01',48.25,48.7,47.05,47.75,1775900,'0');
INSERT INTO daily_summary VALUES ('stock','SGOC','us','2017-11-01',1.18,1.2,1.1379,1.14,17664,'0');
INSERT INTO daily_summary VALUES ('etf','SGOL','us','2017-11-01',123.48,123.97,123.31,123.56,24861,'0');
INSERT INTO daily_summary VALUES ('stock','SGQI','us','2017-11-01',28.29,28.29,28.29,28.29,270,'0');
INSERT INTO daily_summary VALUES ('stock','SGRP','us','2017-11-01',1.03,1.03,1.03,1.03,1034,'0');
INSERT INTO daily_summary VALUES ('stock','SGRY','us','2017-11-01',7.5,8.55,7.1,7.75,1842520,'0');
INSERT INTO daily_summary VALUES ('stock','SGU','us','2017-11-01',11.07,11.13,10.95,10.96,54901,'0');
INSERT INTO daily_summary VALUES ('stock','SGY','us','2017-11-01',29.82,30.95,29.79,30.71,128505,'0');
INSERT INTO daily_summary VALUES ('stock','SGYP','us','2017-11-01',2.78,2.85,2.74,2.77,4353330,'0');
INSERT INTO daily_summary VALUES ('stock','SGZA','us','2017-11-01',25.15,25.219,25.15,25.219,1354,'0');
INSERT INTO daily_summary VALUES ('etf','SH','us','2017-11-01',31.335,31.459,31.285,31.41,1317010,'0');
INSERT INTO daily_summary VALUES ('stock','SHAG','us','2017-11-01',49.9897,50.06,49.9897,50.028,1376,'0');
INSERT INTO daily_summary VALUES ('stock','SHAK','us','2017-11-01',37.88,38.26,36.96,37.15,1340350,'0');
INSERT INTO daily_summary VALUES ('stock','SHBI','us','2017-11-01',16.55,16.5897,16.4,16.53,8127,'0');
INSERT INTO daily_summary VALUES ('stock','SHE','us','2017-11-01',70.855,70.855,70.538,70.584,1678,'0');
INSERT INTO daily_summary VALUES ('stock','SHEN','us','2017-11-01',38.186,38.186,36.796,37.64,256186,'0');
INSERT INTO daily_summary VALUES ('stock','SHG','us','2017-11-01',45.49,45.49,45.13,45.24,85773,'0');
INSERT INTO daily_summary VALUES ('stock','SHI','us','2017-11-01',60.91,61.11,60.62,61.11,21422,'0');
INSERT INTO daily_summary VALUES ('stock','SHIP','us','2017-11-01',0.9986,0.9986,0.9105,0.9506,187491,'0');
INSERT INTO daily_summary VALUES ('stock','SHIPW','us','2017-11-01',0.18,0.18,0.18,0.18,2539,'0');
INSERT INTO daily_summary VALUES ('stock','SHLD','us','2017-11-01',5.55,5.6,5.03,5.13,1485070,'0');
INSERT INTO daily_summary VALUES ('stock','SHLDW','us','2017-11-01',1.15,1.15,0.93,0.9901,8185,'0');
INSERT INTO daily_summary VALUES ('stock','SHLM','us','2017-11-01',39.25,39.8,39,39.7,213345,'0');
INSERT INTO daily_summary VALUES ('stock','SHLO','us','2017-11-01',9.39,9.46,8.99,9.27,81827,'0');
INSERT INTO daily_summary VALUES ('stock','SHLX','us','2017-11-01',25.62,26.09,25.54,26.04,462952,'0');
INSERT INTO daily_summary VALUES ('etf','SHM','us','2017-11-01',48.49,48.49,48.45,48.45,348823,'0');
INSERT INTO daily_summary VALUES ('stock','SHNY','us','2017-11-01',9.53,9.5888,9.22,9.32,12084,'0');
INSERT INTO daily_summary VALUES ('stock','SHO','us','2017-11-01',16.42,16.575,16.24,16.28,1779640,'0');
INSERT INTO daily_summary VALUES ('stock','SHOO','us','2017-11-01',38.95,40.945,37.6,38.45,1020120,'0');
INSERT INTO daily_summary VALUES ('stock','SHOP','us','2017-11-01',99.74,100.5,95,96.2,5504740,'0');
INSERT INTO daily_summary VALUES ('stock','SHOS','us','2017-11-01',1.5736,1.75,1.5736,1.65,160910,'0');
INSERT INTO daily_summary VALUES ('stock','SHO_E','us','2017-11-01',26.59,26.59,26.59,26.59,1120,'0');
INSERT INTO daily_summary VALUES ('stock','SHPG','us','2017-11-01',148.57,148.75,145.19,145.75,1146940,'0');
INSERT INTO daily_summary VALUES ('stock','SHSP','us','2017-11-01',4.48,4.5,4.37,4.5,189955,'0');
INSERT INTO daily_summary VALUES ('etf','SHV','us','2017-11-01',110.28,110.3,110.26,110.27,508094,'0');
INSERT INTO daily_summary VALUES ('stock','SHW','us','2017-11-01',392.65,396.19,391.43,395.77,375372,'0');
INSERT INTO daily_summary VALUES ('etf','SHY','us','2017-11-01',84.17,84.21,84.17,84.21,840321,'0');
INSERT INTO daily_summary VALUES ('etf','SHYD','us','2017-11-01',24.39,24.39,24.32,24.383,12734,'0');
INSERT INTO daily_summary VALUES ('etf','SHYG','us','2017-11-01',47.56,47.56,47.42,47.44,1144670,'0');
INSERT INTO daily_summary VALUES ('stock','SID','us','2017-11-01',2.58,2.62,2.44,2.45,3566940,'0');
INSERT INTO daily_summary VALUES ('stock','SIEB','us','2017-11-01',3.8485,3.8485,3.8485,3.8485,238,'0');
INSERT INTO daily_summary VALUES ('stock','SIEN','us','2017-11-01',14.7,14.7,13.82,13.88,118736,'0');
INSERT INTO daily_summary VALUES ('stock','SIF','us','2017-11-01',7.35,7.6032,7.1,7.1,7909,'0');
INSERT INTO daily_summary VALUES ('stock','SIFI','us','2017-11-01',15.1,15.1,14.801,14.901,7509,'0');
INSERT INTO daily_summary VALUES ('stock','SIFY','us','2017-11-01',1.38,1.45,1.38,1.43,171503,'0');
INSERT INTO daily_summary VALUES ('stock','SIG','us','2017-11-01',66.3,68.22,65.33,66.47,1724650,'0');
INSERT INTO daily_summary VALUES ('stock','SIGI','us','2017-11-01',60.05,60.4,58.25,58.95,216265,'0');
INSERT INTO daily_summary VALUES ('stock','SIGM','us','2017-11-01',6.15,6.15,5.85,5.95,307632,'0');
INSERT INTO daily_summary VALUES ('etf','SIJ','us','2017-11-01',18.52,18.52,18.52,18.52,11042,'0');
INSERT INTO daily_summary VALUES ('etf','SIL','us','2017-11-01',32.41,32.45,32.01,32.09,57349,'0');
INSERT INTO daily_summary VALUES ('stock','SILC','us','2017-11-01',71.41,72,69.55,69.76,31275,'0');
INSERT INTO daily_summary VALUES ('etf','SILJ','us','2017-11-01',11.02,11.27,11.02,11.04,61930,'0');
INSERT INTO daily_summary VALUES ('stock','SIM','us','2017-11-01',9.75,9.75,9.72,9.72,400,'0');
INSERT INTO daily_summary VALUES ('stock','SIMO','us','2017-11-01',48.368,48.617,47.653,48.159,605690,'0');
INSERT INTO daily_summary VALUES ('stock','SINA','us','2017-11-01',108.3,113.72,108.28,113.13,1837110,'0');
INSERT INTO daily_summary VALUES ('stock','SINO','us','2017-11-01',2.76,2.81,2.74,2.74,30119,'0');
INSERT INTO daily_summary VALUES ('stock','SIR','us','2017-11-01',24.26,24.54,24,24.21,272261,'0');
INSERT INTO daily_summary VALUES ('stock','SIRI','us','2017-11-01',5.4598,5.5097,5.3699,5.4298,23182100,'0');
INSERT INTO daily_summary VALUES ('stock','SITE','us','2017-11-01',63.6,64.23,62.87,63.42,376732,'0');
INSERT INTO daily_summary VALUES ('stock','SITO','us','2017-11-01',7.64,7.93,7.41,7.82,99932,'0');
INSERT INTO daily_summary VALUES ('stock','SIVB','us','2017-11-01',220.45,221.82,217.85,218.73,514783,'0');
INSERT INTO daily_summary VALUES ('etf','SIVR','us','2017-11-01',16.47,16.75,16.469,16.73,132794,'0');
INSERT INTO daily_summary VALUES ('stock','SIX','us','2017-11-01',63.11,63.43,62.67,63.09,923229,'0');
INSERT INTO daily_summary VALUES ('etf','SIZE','us','2017-11-01',81.01,81.01,80.62,80.7,13991,'0');
INSERT INTO daily_summary VALUES ('etf','SJB','us','2017-11-01',23.07,23.13,23.07,23.1283,39261,'0');
INSERT INTO daily_summary VALUES ('stock','SJI','us','2017-11-01',34.04,34.09,33.29,33.36,194951,'0');
INSERT INTO daily_summary VALUES ('stock','SJM','us','2017-11-01',105.07,105.14,104.17,104.43,714352,'0');
INSERT INTO daily_summary VALUES ('etf','SJNK','us','2017-11-01',27.82,27.83,27.77,27.8,3095450,'0');
INSERT INTO daily_summary VALUES ('stock','SJR','us','2017-11-01',22.72,22.97,22.56,22.93,976725,'0');
INSERT INTO daily_summary VALUES ('stock','SJT','us','2017-11-01',7.53,7.63,7.41,7.5,157650,'0');
INSERT INTO daily_summary VALUES ('stock','SJW','us','2017-11-01',59.735,60.173,59.067,59.795,96991,'0');
INSERT INTO daily_summary VALUES ('etf','SKF','us','2017-11-01',22.72,22.85,22.5897,22.84,26753,'0');
INSERT INTO daily_summary VALUES ('stock','SKIS','us','2017-11-01',4.6,4.6,4.55,4.55,5214,'0');
INSERT INTO daily_summary VALUES ('stock','SKLN','us','2017-11-01',2.26,2.5,2,2.09,9326590,'0');
INSERT INTO daily_summary VALUES ('stock','SKM','us','2017-11-01',26.35,26.36,26,26.12,295031,'0');
INSERT INTO daily_summary VALUES ('stock','SKT','us','2017-11-01',22.82,23.25,22.66,23.12,1420660,'0');
INSERT INTO daily_summary VALUES ('stock','SKX','us','2017-11-01',31.83,32.96,31.75,32.55,2690640,'0');
INSERT INTO daily_summary VALUES ('stock','SKY','us','2017-11-01',12.55,12.6144,11.6,11.9,56955,'0');
INSERT INTO daily_summary VALUES ('stock','SKYS','us','2017-11-01',1.781,1.83,1.781,1.819,45630,'0');
INSERT INTO daily_summary VALUES ('stock','SKYW','us','2017-11-01',47.6,47.7,46.45,47.05,265353,'0');
INSERT INTO daily_summary VALUES ('etf','SKYY','us','2017-11-01',44.13,44.13,43.6,43.82,107822,'0');
INSERT INTO daily_summary VALUES ('stock','SLAB','us','2017-11-01',95.55,95.55,92.9,93.65,183813,'0');
INSERT INTO daily_summary VALUES ('stock','SLB','us','2017-11-01',64.62,64.96,63.99,64.28,11328600,'0');
INSERT INTO daily_summary VALUES ('stock','SLCA','us','2017-11-01',30.9,31.91,29.53,30.05,2809820,'0');
INSERT INTO daily_summary VALUES ('stock','SLCT','us','2017-11-01',11.721,11.785,11.72,11.74,3417,'0');
INSERT INTO daily_summary VALUES ('stock','SLD','us','2017-11-01',15.65,15.75,15.55,15.7,41355,'0');
INSERT INTO daily_summary VALUES ('stock','SLDA','us','2017-11-01',25.5,25.5,25.4,25.45,9850,'0');
INSERT INTO daily_summary VALUES ('stock','SLF','us','2017-11-01',39.13,39.13,38.79,38.97,442131,'0');
INSERT INTO daily_summary VALUES ('stock','SLG','us','2017-11-01',95.67,96.13,95.05,95.19,676697,'0');
INSERT INTO daily_summary VALUES ('stock','SLGN','us','2017-11-01',29.36,29.52,29.18,29.29,447198,'0');
INSERT INTO daily_summary VALUES ('stock','SLG_I','us','2017-11-01',25.253,25.27,25.253,25.27,2391,'0');
INSERT INTO daily_summary VALUES ('stock','SLIM','us','2017-11-01',27.6425,27.6425,27.6425,27.6425,407,'0');
INSERT INTO daily_summary VALUES ('stock','SLM','us','2017-11-01',10.7,10.79,10.34,10.48,3184200,'0');
INSERT INTO daily_summary VALUES ('stock','SLMBP','us','2017-11-01',67.37,67.51,67.36,67.51,1775,'0');
INSERT INTO daily_summary VALUES ('stock','SLNO','us','2017-11-01',1.92,1.92,1.8882,1.909,4258,'0');
INSERT INTO daily_summary VALUES ('stock','SLNOW','us','2017-11-01',0.12,0.12,0.12,0.12,660,'0');
INSERT INTO daily_summary VALUES ('stock','SLP','us','2017-11-01',16.34,16.44,15.643,15.942,69482,'0');
INSERT INTO daily_summary VALUES ('etf','SLQD','us','2017-11-01',50.44,50.47,50.42,50.43,156356,'0');
INSERT INTO daily_summary VALUES ('stock','SLRA','us','2017-11-01',25.3,25.3,25.13,25.15,5229,'0');
INSERT INTO daily_summary VALUES ('stock','SLRC','us','2017-11-01',21.23,21.4,21.1,21.22,56847,'0');
INSERT INTO daily_summary VALUES ('stock','SLTB','us','2017-11-01',25.45,25.65,25.45,25.65,1622,'0');
INSERT INTO daily_summary VALUES ('etf','SLV','us','2017-11-01',15.95,16.235,15.9301,16.18,10688700,'0');
INSERT INTO daily_summary VALUES ('etf','SLVO','us','2017-11-01',7.99,8.1099,7.9601,8.08,43631,'0');
INSERT INTO daily_summary VALUES ('etf','SLVP','us','2017-11-01',10.5153,10.6047,10.47,10.4997,23722,'0');
INSERT INTO daily_summary VALUES ('etf','SLX','us','2017-11-01',43.52,43.64,42.54,42.72,42958,'0');
INSERT INTO daily_summary VALUES ('etf','SLY','us','2017-11-01',132.51,132.51,129.953,130.79,34554,'0');
INSERT INTO daily_summary VALUES ('etf','SLYG','us','2017-11-01',232.93,232.93,228.804,229.68,22709,'0');
INSERT INTO daily_summary VALUES ('etf','SLYV','us','2017-11-01',128.08,128.08,125.7,126.41,57109,'0');
INSERT INTO daily_summary VALUES ('stock','SM','us','2017-11-01',21.68,22.81,21.491,21.88,5585240,'0');
INSERT INTO daily_summary VALUES ('etf','SMB','us','2017-11-01',17.5093,17.525,17.5,17.5,11118,'0');
INSERT INTO daily_summary VALUES ('stock','SMBC','us','2017-11-01',37.12,38.39,36.64,37.22,7519,'0');
INSERT INTO daily_summary VALUES ('stock','SMBK','us','2017-11-01',23.48,23.48,22.84,23.26,5845,'0');
INSERT INTO daily_summary VALUES ('stock','SMCI','us','2017-11-01',20.1,20.5,20,20.3,488785,'0');
INSERT INTO daily_summary VALUES ('stock','SMCP','us','2017-11-01',24.066,24.066,24.066,24.066,554,'0');
INSERT INTO daily_summary VALUES ('etf','SMDD','us','2017-11-01',11.41,11.41,11.41,11.41,464,'0');
INSERT INTO daily_summary VALUES ('etf','SMDV','us','2017-11-01',56.41,56.41,55.5001,55.754,45337,'0');
INSERT INTO daily_summary VALUES ('stock','SMED','us','2017-11-01',4.11,4.17,4.11,4.12,3554,'0');
INSERT INTO daily_summary VALUES ('etf','SMEZ','us','2017-11-01',62.78,62.78,62.6147,62.63,3604,'0');
INSERT INTO daily_summary VALUES ('stock','SMFG','us','2017-11-01',8.06,8.1,8.04,8.07,1264690,'0');
INSERT INTO daily_summary VALUES ('stock','SMG','us','2017-11-01',99.64,100.75,99.64,100.1,201426,'0');
INSERT INTO daily_summary VALUES ('etf','SMH','us','2017-11-01',102.32,102.44,100.515,101.34,4918360,'0');
INSERT INTO daily_summary VALUES ('stock','SMHD','us','2017-11-01',18.927,18.927,18.182,18.284,81706,'0');
INSERT INTO daily_summary VALUES ('stock','SMHI','us','2017-11-01',14.27,14.43,14.2,14.27,40716,'0');
INSERT INTO daily_summary VALUES ('stock','SMI','us','2017-11-01',8.17,8.29,8.17,8.23,829306,'0');
INSERT INTO daily_summary VALUES ('stock','SMIN','us','2017-11-01',49.49,49.7248,49.46,49.595,13509,'0');
INSERT INTO daily_summary VALUES ('stock','SMIT','us','2017-11-01',2.05,2.08,1.96,1.9999,36109,'0');
INSERT INTO daily_summary VALUES ('etf','SMLF','us','2017-11-01',39.33,39.33,38.43,38.57,9017,'0');
INSERT INTO daily_summary VALUES ('stock','SMLP','us','2017-11-01',20.011,20.278,19.768,20.059,234300,'0');
INSERT INTO daily_summary VALUES ('etf','SMLV','us','2017-11-01',96.29,96.48,95.1001,95.67,16126,'0');
INSERT INTO daily_summary VALUES ('stock','SMM','us','2017-11-01',10.65,10.86,10.65,10.75,134406,'0');
INSERT INTO daily_summary VALUES ('stock','SMMD','us','2017-11-01',41.93,41.93,41.93,41.93,100,'0');
INSERT INTO daily_summary VALUES ('stock','SMMF','us','2017-11-01',27.27,27.7,26.78,26.85,22097,'0');
INSERT INTO daily_summary VALUES ('stock','SMMT','us','2017-11-01',10.3,10.3122,9.85,9.88,39018,'0');
INSERT INTO daily_summary VALUES ('etf','SMMU','us','2017-11-01',50.4383,50.4383,50.35,50.35,481,'0');
INSERT INTO daily_summary VALUES ('stock','SMMV','us','2017-11-01',28.69,28.76,28.4397,28.55,24155,'0');
INSERT INTO daily_summary VALUES ('etf','SMN','us','2017-11-01',14,14.01,13.78,14.01,9238,'0');
INSERT INTO daily_summary VALUES ('stock','SMP','us','2017-11-01',43.78,44.4,42.87,42.97,115898,'0');
INSERT INTO daily_summary VALUES ('stock','SMPL','us','2017-11-01',11.54,11.89,11.43,11.64,255545,'0');
INSERT INTO daily_summary VALUES ('stock','SMPLW','us','2017-11-01',2.19,2.24,2.19,2.24,500,'0');
INSERT INTO daily_summary VALUES ('stock','SMRT','us','2017-11-01',1.08,1.099,1.02,1.05,434134,'0');
INSERT INTO daily_summary VALUES ('stock','SMSI','us','2017-11-01',1.5387,1.73,1.5387,1.66,20490,'0');
INSERT INTO daily_summary VALUES ('stock','SMTC','us','2017-11-01',41.6,41.95,40.15,40.5,346103,'0');
INSERT INTO daily_summary VALUES ('stock','SMTS','us','2017-11-01',2.45,2.53,2.43,2.5298,15270,'0');
INSERT INTO daily_summary VALUES ('stock','SMTX','us','2017-11-01',1.45,1.48,1.4499,1.48,36147,'0');
INSERT INTO daily_summary VALUES ('stock','SN','us','2017-11-01',4.43,4.62,4.42,4.6,3898430,'0');
INSERT INTO daily_summary VALUES ('stock','SNA','us','2017-11-01',158.17,158.61,156.384,158.36,526419,'0');
INSERT INTO daily_summary VALUES ('stock','SNAK','us','2017-11-01',3.99,4,3.97,3.99,616351,'0');
INSERT INTO daily_summary VALUES ('stock','SNAP','us','2017-11-01',15.45,15.52,14.39,14.51,20435300,'0');
INSERT INTO daily_summary VALUES ('stock','SNBC','us','2017-11-01',25.5,25.55,24.75,24.825,29202,'0');
INSERT INTO daily_summary VALUES ('stock','SNBR','us','2017-11-01',32.51,34.84,31.86,33.85,436400,'0');
INSERT INTO daily_summary VALUES ('stock','SNC','us','2017-11-01',21.03,21.03,21,21.02,155301,'0');
INSERT INTO daily_summary VALUES ('stock','SNCR','us','2017-11-01',11.43,11.63,11.125,11.14,508483,'0');
INSERT INTO daily_summary VALUES ('stock','SND','us','2017-11-01',7.35,7.58,7.05,7.16,500638,'0');
INSERT INTO daily_summary VALUES ('stock','SNDE','us','2017-11-01',4.32,4.931,4.3,4.74,33415,'0');
INSERT INTO daily_summary VALUES ('stock','SNDR','us','2017-11-01',26.38,26.5,25.74,25.97,205918,'0');
INSERT INTO daily_summary VALUES ('stock','SNDX','us','2017-11-01',11.3,11.45,10.8,11.02,79049,'0');
INSERT INTO daily_summary VALUES ('stock','SNE','us','2017-11-01',43.57,43.94,43.33,43.54,2289660,'0');
INSERT INTO daily_summary VALUES ('stock','SNES','us','2017-11-01',2.72,2.84,2.6201,2.79,136857,'0');
INSERT INTO daily_summary VALUES ('stock','SNFCA','us','2017-11-01',5.4,5.4013,5.2903,5.4,33325,'0');
INSERT INTO daily_summary VALUES ('stock','SNGX','us','2017-11-01',2.01,2.03,1.9,1.96,61662,'0');
INSERT INTO daily_summary VALUES ('stock','SNGXW','us','2017-11-01',0.5999,0.5999,0.5959,0.5959,4400,'0');
INSERT INTO daily_summary VALUES ('stock','SNH','us','2017-11-01',18.43,18.5399,18.305,18.49,1203300,'0');
INSERT INTO daily_summary VALUES ('stock','SNHNI','us','2017-11-01',25.22,25.22,25.0768,25.08,6949,'0');
INSERT INTO daily_summary VALUES ('stock','SNHNL','us','2017-11-01',26.87,27.1,26.76,27.1,10352,'0');
INSERT INTO daily_summary VALUES ('stock','SNHY','us','2017-11-01',57.84,57.91,56.81,57.53,83463,'0');
INSERT INTO daily_summary VALUES ('stock','SNI','us','2017-11-01',83.45,83.83,83.08,83.52,1308570,'0');
INSERT INTO daily_summary VALUES ('etf','SNLN','us','2017-11-01',18.33,18.35,18.33,18.35,80631,'0');
INSERT INTO daily_summary VALUES ('stock','SNMP','us','2017-11-01',11.15,11.45,11.1,11.2,19047,'0');
INSERT INTO daily_summary VALUES ('stock','SNMX','us','2017-11-01',0.91,0.91,0.854,0.87,92632,'0');
INSERT INTO daily_summary VALUES ('stock','SNN','us','2017-11-01',37.46,37.6,36.875,36.91,843618,'0');
INSERT INTO daily_summary VALUES ('stock','SNNA','us','2017-11-01',19.81,20.58,17.9503,19.91,41546,'0');
INSERT INTO daily_summary VALUES ('stock','SNOA','us','2017-11-01',4.36,4.4779,4.22,4.22,24914,'0');
INSERT INTO daily_summary VALUES ('stock','SNOAW','us','2017-11-01',0.37,0.37,0.2935,0.3617,23533,'0');
INSERT INTO daily_summary VALUES ('stock','SNP','us','2017-11-01',73.75,74.38,73.69,73.81,141844,'0');
INSERT INTO daily_summary VALUES ('stock','SNPS','us','2017-11-01',87.07,87.21,85.71,86.12,765690,'0');
INSERT INTO daily_summary VALUES ('stock','SNR','us','2017-11-01',8.99,9,8.84,8.89,366005,'0');
INSERT INTO daily_summary VALUES ('stock','SNSR','us','2017-11-01',20.16,20.16,19.9255,20.01,65722,'0');
INSERT INTO daily_summary VALUES ('stock','SNSS','us','2017-11-01',2.51,2.59,2.34,2.39,199731,'0');
INSERT INTO daily_summary VALUES ('stock','SNV','us','2017-11-01',47.02,47.26,46.4,46.64,662167,'0');
INSERT INTO daily_summary VALUES ('stock','SNV_C','us','2017-11-01',26.02,26.38,26.02,26.22,7243,'0');
INSERT INTO daily_summary VALUES ('stock','SNX','us','2017-11-01',135.52,135.52,132.56,132.78,227151,'0');
INSERT INTO daily_summary VALUES ('stock','SNY','us','2017-11-01',46.98,47.15,46.37,46.48,1588900,'0');
INSERT INTO daily_summary VALUES ('stock','SO','us','2017-11-01',52.4,52.4,51.76,52.08,3957610,'0');
INSERT INTO daily_summary VALUES ('etf','SOCL','us','2017-11-01',33,33.2024,32.854,32.91,48644,'0');
INSERT INTO daily_summary VALUES ('stock','SODA','us','2017-11-01',69,69,64.92,66.71,1061990,'0');
INSERT INTO daily_summary VALUES ('stock','SOFO','us','2017-11-01',3.4201,3.4366,3.3276,3.3791,6667,'0');
INSERT INTO daily_summary VALUES ('stock','SOHO','us','2017-11-01',5.95,6.249,5.94,6.18,108290,'0');
INSERT INTO daily_summary VALUES ('stock','SOHOB','us','2017-11-01',25.38,25.38,25.335,25.35,10047,'0');
INSERT INTO daily_summary VALUES ('stock','SOHOM','us','2017-11-01',25.66,25.6714,25.66,25.66,7582,'0');
INSERT INTO daily_summary VALUES ('stock','SOHOO','us','2017-11-01',25.1826,25.2,25.18,25.2,5522,'0');
INSERT INTO daily_summary VALUES ('stock','SOHU','us','2017-11-01',57.56,58.535,57.24,57.63,804677,'0');
INSERT INTO daily_summary VALUES ('stock','SOI','us','2017-11-01',16.14,16.36,15.82,16.07,107126,'0');
INSERT INTO daily_summary VALUES ('etf','SOIL','us','2017-11-01',10.46,10.46,10.3979,10.3979,4327,'0');
INSERT INTO daily_summary VALUES ('stock','SOJA','us','2017-11-01',26.91,26.98,26.87,26.94,22377,'0');
INSERT INTO daily_summary VALUES ('stock','SOJB','us','2017-11-01',25.37,25.37,25.31,25.37,64734,'0');
INSERT INTO daily_summary VALUES ('stock','SOL','us','2017-11-01',2.79,2.94,2.76,2.77,21109,'0');
INSERT INTO daily_summary VALUES ('stock','SON','us','2017-11-01',51.567,51.964,51.328,51.497,349838,'0');
INSERT INTO daily_summary VALUES ('stock','SONA','us','2017-11-01',16.418,16.418,15.951,16.07,48178,'0');
INSERT INTO daily_summary VALUES ('stock','SONC','us','2017-11-01',25.384,25.384,24.847,24.986,467726,'0');
INSERT INTO daily_summary VALUES ('stock','SONS','us','2017-11-01',7.3,7.63,7.05,7.1,1266780,'0');
INSERT INTO daily_summary VALUES ('stock','SOR','us','2017-11-01',41.07,41.08,40.92,40.92,10102,'0');
INSERT INTO daily_summary VALUES ('stock','SORL','us','2017-11-01',4.71,4.79,4.6,4.61,137725,'0');
INSERT INTO daily_summary VALUES ('stock','SOVB','us','2017-11-01',27.36,27.36,27.3304,27.36,934,'0');
INSERT INTO daily_summary VALUES ('stock','SOV_C','us','2017-11-01',25.51,25.55,25.47,25.55,16933,'0');
INSERT INTO daily_summary VALUES ('etf','SOXL','us','2017-11-01',151.67,151.8,142.6,146.58,526556,'0');
INSERT INTO daily_summary VALUES ('etf','SOXS','us','2017-11-01',15.89,16.849,15.83,16.41,656781,'0');
INSERT INTO daily_summary VALUES ('etf','SOXX','us','2017-11-01',174.12,174.24,170.58,172.09,505477,'0');
INSERT INTO daily_summary VALUES ('etf','SOYB','us','2017-11-01',18.5,18.6,18.5,18.5532,16706,'0');
INSERT INTO daily_summary VALUES ('stock','SP','us','2017-11-01',38.7,38.75,37.7,38.35,127203,'0');
INSERT INTO daily_summary VALUES ('stock','SPA','us','2017-11-01',23.3,23.3,23.25,23.27,47576,'0');
INSERT INTO daily_summary VALUES ('stock','SPAB','us','2017-11-01',28.77,28.81,28.74,28.76,200449,'0');
INSERT INTO daily_summary VALUES ('stock','SPAR','us','2017-11-01',15.5,16.5,15.05,15.7,523475,'0');
INSERT INTO daily_summary VALUES ('stock','SPB','us','2017-11-01',110.4,110.5,109.4,109.84,382946,'0');
INSERT INTO daily_summary VALUES ('stock','SPCB','us','2017-11-01',3.83,4.09,3.62,3.73,484568,'0');
INSERT INTO daily_summary VALUES ('stock','SPDN','us','2017-11-01',31.76,31.9,31.76,31.87,2813,'0');
INSERT INTO daily_summary VALUES ('stock','SPDW','us','2017-11-01',31.39,31.4,31.27,31.3,441140,'0');
INSERT INTO daily_summary VALUES ('stock','SPE','us','2017-11-01',15.87,15.959,15.79,15.88,24491,'0');
INSERT INTO daily_summary VALUES ('stock','SPEM','us','2017-11-01',37.41,37.43,37.18,37.2,193704,'0');
INSERT INTO daily_summary VALUES ('stock','SPEX','us','2017-11-01',1.35,1.44,1.3201,1.38,143225,'0');
INSERT INTO daily_summary VALUES ('stock','SPE_B','us','2017-11-01',25.8705,25.8705,25.8705,25.8705,329,'0');
INSERT INTO daily_summary VALUES ('etf','SPFF','us','2017-11-01',12.37,12.3956,12.35,12.36,96526,'0');
INSERT INTO daily_summary VALUES ('stock','SPG','us','2017-11-01',155.99,157.73,155.29,157.09,1539810,'0');
INSERT INTO daily_summary VALUES ('stock','SPGI','us','2017-11-01',157.08,158.09,156.11,156.56,1506050,'0');
INSERT INTO daily_summary VALUES ('stock','SPG_J','us','2017-11-01',73.17,74,73,74,600,'0');
INSERT INTO daily_summary VALUES ('stock','SPH','us','2017-11-01',25.708,26.324,25.708,26.08,508042,'0');
INSERT INTO daily_summary VALUES ('etf','SPHB','us','2017-11-01',40.43,41.69,40.08,40.14,1620100,'0');
INSERT INTO daily_summary VALUES ('etf','SPHD','us','2017-11-01',41.37,41.4518,41.27,41.37,246042,'0');
INSERT INTO daily_summary VALUES ('etf','SPHQ','us','2017-11-01',29.22,29.29,29.1433,29.16,377233,'0');
INSERT INTO daily_summary VALUES ('stock','SPHS','us','2017-11-01',2.16,2.19,2.06,2.09,192171,'0');
INSERT INTO daily_summary VALUES ('stock','SPI','us','2017-11-01',1.251,1.289,1.22,1.23,278118,'0');
INSERT INTO daily_summary VALUES ('stock','SPIB','us','2017-11-01',34.45,34.46,34.4132,34.44,298939,'0');
INSERT INTO daily_summary VALUES ('stock','SPIL','us','2017-11-01',7.87,7.87,7.8,7.8,22633,'0');
INSERT INTO daily_summary VALUES ('stock','SPKE','us','2017-11-01',15.3,15.3,14.505,14.7,181369,'0');
INSERT INTO daily_summary VALUES ('stock','SPKEP','us','2017-11-01',27.1833,27.34,27.1833,27.31,979,'0');
INSERT INTO daily_summary VALUES ('stock','SPLB','us','2017-11-01',28.1,28.17,28.0688,28.09,12363,'0');
INSERT INTO daily_summary VALUES ('stock','SPLG','us','2017-11-01',30.84,30.84,30.17,30.209,291002,'0');
INSERT INTO daily_summary VALUES ('stock','SPLK','us','2017-11-01',67.56,67.779,66.52,66.99,2168660,'0');
INSERT INTO daily_summary VALUES ('stock','SPLP','us','2017-11-01',19.15,19.15,18.955,19,1788,'0');
INSERT INTO daily_summary VALUES ('stock','SPLP_A','us','2017-11-01',21.39,21.39,21.05,21.17,30826,'0');
INSERT INTO daily_summary VALUES ('stock','SPLP_T','us','2017-11-01',20.35,20.44,20.17,20.3,35762,'0');
INSERT INTO daily_summary VALUES ('etf','SPLV','us','2017-11-01',46.69,46.76,46.61,46.63,1712730,'0');
INSERT INTO daily_summary VALUES ('etf','SPLX','us','2017-11-01',50.33,50.33,50.33,50.33,880,'0');
INSERT INTO daily_summary VALUES ('stock','SPMD','us','2017-11-01',32.66,32.73,32.27,32.405,167601,'0');
INSERT INTO daily_summary VALUES ('stock','SPMO','us','2017-11-01',33.27,33.32,33.2202,33.2202,501,'0');
INSERT INTO daily_summary VALUES ('stock','SPMV','us','2017-11-01',25.92,25.92,25.92,25.92,0,'0');
INSERT INTO daily_summary VALUES ('stock','SPN','us','2017-11-01',9.02,9.09,8.38,8.47,4067750,'0');
INSERT INTO daily_summary VALUES ('stock','SPNE','us','2017-11-01',9.97,10.05,9.88,10.04,18470,'0');
INSERT INTO daily_summary VALUES ('stock','SPNS','us','2017-11-01',14.1,14.15,13.87,13.96,16127,'0');
INSERT INTO daily_summary VALUES ('stock','SPOK','us','2017-11-01',17.15,17.35,16.85,17.35,119713,'0');
INSERT INTO daily_summary VALUES ('stock','SPPI','us','2017-11-01',19.68,19.85,18.54,19.04,1138620,'0');
INSERT INTO daily_summary VALUES ('stock','SPPP','us','2017-11-01',8.86,8.95,8.86,8.91,49825,'0');
INSERT INTO daily_summary VALUES ('stock','SPR','us','2017-11-01',76.9,80.19,76,79.21,2144250,'0');
INSERT INTO daily_summary VALUES ('stock','SPRT','us','2017-11-01',2.28,2.35,2.26,2.27,39002,'0');
INSERT INTO daily_summary VALUES ('stock','SPSB','us','2017-11-01',30.6,30.61,30.59,30.6,650474,'0');
INSERT INTO daily_summary VALUES ('stock','SPSC','us','2017-11-01',49.66,49.66,47.12,48.02,330465,'0');
INSERT INTO daily_summary VALUES ('stock','SPSM','us','2017-11-01',29.71,29.71,29.0947,29.24,81820,'0');
INSERT INTO daily_summary VALUES ('stock','SPTL','us','2017-11-01',35.84,36.0723,35.84,35.98,267768,'0');
INSERT INTO daily_summary VALUES ('stock','SPTM','us','2017-11-01',32.13,32.16,31.96,32.02,226274,'0');
INSERT INTO daily_summary VALUES ('stock','SPTN','us','2017-11-01',24.72,25.79,24.23,25.76,187132,'0');
INSERT INTO daily_summary VALUES ('stock','SPTS','us','2017-11-01',30.04,30.06,30.0394,30.045,35258,'0');
INSERT INTO daily_summary VALUES ('stock','SPUN','us','2017-11-01',23.86,23.86,23.7884,23.7884,418,'0');
INSERT INTO daily_summary VALUES ('etf','SPUU','us','2017-11-01',46.55,46.55,46.48,46.48,1341,'0');
INSERT INTO daily_summary VALUES ('stock','SPVM','us','2017-11-01',27.27,27.27,27.27,27.27,0,'0');
INSERT INTO daily_summary VALUES ('stock','SPVU','us','2017-11-01',33.34,33.35,33.23,33.28,4237,'0');
INSERT INTO daily_summary VALUES ('stock','SPWH','us','2017-11-01',4.11,4.215,4.04,4.13,316153,'0');
INSERT INTO daily_summary VALUES ('stock','SPWR','us','2017-11-01',7.26,7.3,7.01,7.13,3052310,'0');
INSERT INTO daily_summary VALUES ('stock','SPXC','us','2017-11-01',29.6,29.6,28.53,28.66,178159,'0');
INSERT INTO daily_summary VALUES ('etf','SPXH','us','2017-11-01',35.2,35.2006,35.17,35.17,1647,'0');
INSERT INTO daily_summary VALUES ('etf','SPXL','us','2017-11-01',41.32,41.4953,40.85,41.07,1329960,'0');
INSERT INTO daily_summary VALUES ('etf','SPXS','us','2017-11-01',33.6,33.96,33.4116,33.79,1202900,'0');
INSERT INTO daily_summary VALUES ('etf','SPXU','us','2017-11-01',12.74,12.8929,12.69,12.82,3915220,'0');
INSERT INTO daily_summary VALUES ('stock','SPXX','us','2017-11-01',16.3,16.41,16.14,16.3,71668,'0');
INSERT INTO daily_summary VALUES ('etf','SPY','us','2017-11-01',258.04,258.43,257.07,257.49,54173400,'0');
INSERT INTO daily_summary VALUES ('stock','SPYD','us','2017-11-01',36.25,36.36,36.1701,36.24,156064,'0');
INSERT INTO daily_summary VALUES ('etf','SPYG','us','2017-11-01',32.14,32.1405,31.9494,32.02,302237,'0');
INSERT INTO daily_summary VALUES ('etf','SPYV','us','2017-11-01',29.62,29.689,29.5801,29.61,81105,'0');
INSERT INTO daily_summary VALUES ('stock','SPYX','us','2017-11-01',62.82,62.82,62.5,62.54,3697,'0');
INSERT INTO daily_summary VALUES ('stock','SQ','us','2017-11-01',37.48,37.52,35.87,36.47,10557900,'0');
INSERT INTO daily_summary VALUES ('stock','SQBG','us','2017-11-01',2.6,2.63,2.5,2.5,176102,'0');
INSERT INTO daily_summary VALUES ('stock','SQLV','us','2017-11-01',25.82,25.88,25.73,25.73,12637,'0');
INSERT INTO daily_summary VALUES ('stock','SQM','us','2017-11-01',60.13,63.32,60.052,61.28,1570310,'0');
INSERT INTO daily_summary VALUES ('stock','SQNS','us','2017-11-01',1.8,2.004,1.78,1.79,472707,'0');
INSERT INTO daily_summary VALUES ('etf','SQQQ','us','2017-11-01',23.04,23.61,23.001,23.36,6010570,'0');
INSERT INTO daily_summary VALUES ('stock','SR','us','2017-11-01',79.25,79.25,78.2,78.7,107687,'0');
INSERT INTO daily_summary VALUES ('stock','SRAX','us','2017-11-01',3.53,4.02,3.4,3.55,1460280,'0');
INSERT INTO daily_summary VALUES ('stock','SRC','us','2017-11-01',8.34,8.43,8.3,8.41,10712100,'0');
INSERT INTO daily_summary VALUES ('stock','SRCE','us','2017-11-01',51.34,51.439,50.234,50.533,24667,'0');
INSERT INTO daily_summary VALUES ('stock','SRCI','us','2017-11-01',9.64,9.9,9.58,9.85,2877830,'0');
INSERT INTO daily_summary VALUES ('stock','SRCL','us','2017-11-01',70.94,71.48,68.38,68.71,1034940,'0');
INSERT INTO daily_summary VALUES ('stock','SRCLP','us','2017-11-01',57.16,57.16,55.215,55.31,10122,'0');
INSERT INTO daily_summary VALUES ('stock','SRC_A','us','2017-11-01',24.55,24.6999,24.49,24.59,133318,'0');
INSERT INTO daily_summary VALUES ('stock','SRDX','us','2017-11-01',30,30.25,29.2,29.35,16108,'0');
INSERT INTO daily_summary VALUES ('stock','SRE','us','2017-11-01',116.97,117.27,115.91,116.63,1447850,'0');
INSERT INTO daily_summary VALUES ('etf','SRET','us','2017-11-01',15.31,15.311,15.24,15.29,19328,'0');
INSERT INTO daily_summary VALUES ('stock','SREV','us','2017-11-01',3.53,3.53,3.44,3.48,111941,'0');
INSERT INTO daily_summary VALUES ('stock','SRF','us','2017-11-01',8.64,8.7599,8.64,8.65,7185,'0');
INSERT INTO daily_summary VALUES ('stock','SRG','us','2017-11-01',41.15,41.7,40.91,41.65,274025,'0');
INSERT INTO daily_summary VALUES ('stock','SRI','us','2017-11-01',22.74,22.95,21.62,22.2,267709,'0');
INSERT INTO daily_summary VALUES ('etf','SRLN','us','2017-11-01',47.38,47.56,47.35,47.37,568044,'0');
INSERT INTO daily_summary VALUES ('stock','SRLP','us','2017-11-01',26.064,26.26,25.479,25.576,38041,'0');
INSERT INTO daily_summary VALUES ('stock','SRNE','us','2017-11-01',2.7,2.75,2.55,2.575,592057,'0');
INSERT INTO daily_summary VALUES ('stock','SRPT','us','2017-11-01',49.56,50.5,48.81,49.39,1256850,'0');
INSERT INTO daily_summary VALUES ('stock','SRRA','us','2017-11-01',2,2.08,1.93,1.99,299433,'0');
INSERT INTO daily_summary VALUES ('etf','SRS','us','2017-11-01',30.58,30.7714,30.46,30.72,49767,'0');
INSERT INTO daily_summary VALUES ('stock','SRT','us','2017-11-01',12.03,12.03,11.74,11.93,21791,'0');
INSERT INTO daily_summary VALUES ('stock','SRTS','us','2017-11-01',5.7099,5.88,5.35,5.88,14858,'0');
INSERT INTO daily_summary VALUES ('stock','SRTSW','us','2017-11-01',1.01,1.01,1.01,1.01,500,'0');
INSERT INTO daily_summary VALUES ('etf','SRTY','us','2017-11-01',34.26,36.41,34.26,35.8,472144,'0');
INSERT INTO daily_summary VALUES ('stock','SRUN','us','2017-11-01',9.99,10,9.95,9.99,734568,'0');
INSERT INTO daily_summary VALUES ('stock','SRUNU','us','2017-11-01',10.5,10.53,10.43,10.47,15279,'0');
INSERT INTO daily_summary VALUES ('stock','SRUNW','us','2017-11-01',1.5826,1.6,1.52,1.52,230400,'0');
INSERT INTO daily_summary VALUES ('stock','SRV','us','2017-11-01',11.15,11.32,11.15,11.23,53167,'0');
INSERT INTO daily_summary VALUES ('stock','SSB','us','2017-11-01',90.506,90.506,88.414,89.111,92508,'0');
INSERT INTO daily_summary VALUES ('stock','SSBI','us','2017-11-01',12.7,12.7,12.4,12.65,2159,'0');
INSERT INTO daily_summary VALUES ('stock','SSC','us','2017-11-01',2.04,2.0521,1.81,1.9099,55331,'0');
INSERT INTO daily_summary VALUES ('stock','SSD','us','2017-11-01',56.59,57.62,55.87,57.58,628159,'0');
INSERT INTO daily_summary VALUES ('stock','SSFN','us','2017-11-01',9.55,9.55,9.55,9.55,1708,'0');
INSERT INTO daily_summary VALUES ('etf','SSG','us','2017-11-01',10.36,10.6,10.36,10.44,3728,'0');
INSERT INTO daily_summary VALUES ('stock','SSI','us','2017-11-01',1.68,1.72,1.62,1.68,127922,'0');
INSERT INTO daily_summary VALUES ('stock','SSKN','us','2017-11-01',1.27,1.28,1.21,1.2784,89027,'0');
INSERT INTO daily_summary VALUES ('stock','SSL','us','2017-11-01',29.97,30,29.78,29.81,263506,'0');
INSERT INTO daily_summary VALUES ('stock','SSN','us','2017-11-01',0.33,0.371,0.33,0.3689,88577,'0');
INSERT INTO daily_summary VALUES ('stock','SSNC','us','2017-11-01',40.23,40.23,39.36,39.93,1479600,'0');
INSERT INTO daily_summary VALUES ('stock','SSNI','us','2017-11-01',16.12,16.125,16.09,16.09,2248160,'0');
INSERT INTO daily_summary VALUES ('stock','SSNT','us','2017-11-01',4.1,4.15,4.1,4.15,5824,'0');
INSERT INTO daily_summary VALUES ('etf','SSO','us','2017-11-01',101.83,102.12,101.06,101.39,997547,'0');
INSERT INTO daily_summary VALUES ('stock','SSP','us','2017-11-01',17.39,17.43,16.99,17.06,351802,'0');
INSERT INTO daily_summary VALUES ('stock','SSRM','us','2017-11-01',9.7,9.8,9.61,9.64,571157,'0');
INSERT INTO daily_summary VALUES ('stock','SSTI','us','2017-11-01',17.16,17.18,16.76,16.99,47013,'0');
INSERT INTO daily_summary VALUES ('stock','SSTK','us','2017-11-01',39.5,40.67,37.91,38.01,860468,'0');
INSERT INTO daily_summary VALUES ('stock','SSW','us','2017-11-01',7,7.12,6.71,6.72,1897730,'0');
INSERT INTO daily_summary VALUES ('stock','SSWA','us','2017-11-01',24.17,24.35,24.16,24.19,75093,'0');
INSERT INTO daily_summary VALUES ('stock','SSWN','us','2017-11-01',25.41,25.48,25.4,25.48,6447,'0');
INSERT INTO daily_summary VALUES ('stock','SSW_D','us','2017-11-01',24.91,25.0799,24.91,24.99,9459,'0');
INSERT INTO daily_summary VALUES ('stock','SSW_E','us','2017-11-01',25,25,24.875,24.89,8147,'0');
INSERT INTO daily_summary VALUES ('stock','SSW_G','us','2017-11-01',24.72,24.72,24.6,24.6384,47259,'0');
INSERT INTO daily_summary VALUES ('stock','SSW_H','us','2017-11-01',24.22,24.49,24.22,24.3,31303,'0');
INSERT INTO daily_summary VALUES ('stock','SSY','us','2017-11-01',1.48,1.48,1.44,1.44,751,'0');
INSERT INTO daily_summary VALUES ('stock','SSYS','us','2017-11-01',21.06,22.21,20.78,20.98,1982780,'0');
INSERT INTO daily_summary VALUES ('stock','ST','us','2017-11-01',49.21,49.21,48.21,48.25,1175670,'0');
INSERT INTO daily_summary VALUES ('stock','STAA','us','2017-11-01',13.3,13.35,12.85,13.1,42701,'0');
INSERT INTO daily_summary VALUES ('stock','STAF','us','2017-11-01',0.6998,0.7,0.67,0.6801,28491,'0');
INSERT INTO daily_summary VALUES ('stock','STAG','us','2017-11-01',27.4,27.785,27.3,27.64,495257,'0');
INSERT INTO daily_summary VALUES ('stock','STAG_B','us','2017-11-01',25.5,25.6,25.5,25.6,297,'0');
INSERT INTO daily_summary VALUES ('stock','STAG_C','us','2017-11-01',26.95,26.95,26.95,26.95,100,'0');
INSERT INTO daily_summary VALUES ('stock','STAR','us','2017-11-01',11.7,11.8,11.6,11.76,357727,'0');
INSERT INTO daily_summary VALUES ('stock','STAR_D','us','2017-11-01',25.65,25.69,25.65,25.69,3194,'0');
INSERT INTO daily_summary VALUES ('stock','STAR_G','us','2017-11-01',25.45,25.5117,25.33,25.45,2650,'0');
INSERT INTO daily_summary VALUES ('stock','STAR_I','us','2017-11-01',25.395,25.395,25.395,25.395,125,'0');
INSERT INTO daily_summary VALUES ('stock','STAY','us','2017-11-01',19.91,20.09,19.84,19.88,1286100,'0');
INSERT INTO daily_summary VALUES ('stock','STB','us','2017-11-01',5.85,5.9,5.85,5.85,109705,'0');
INSERT INTO daily_summary VALUES ('stock','STBA','us','2017-11-01',41.06,41.06,40.09,40.33,51231,'0');
INSERT INTO daily_summary VALUES ('stock','STBZ','us','2017-11-01',28.98,29.1,28.2823,28.59,57297,'0');
INSERT INTO daily_summary VALUES ('stock','STC','us','2017-11-01',37.97,38.28,37.74,38.13,139275,'0');
INSERT INTO daily_summary VALUES ('stock','STDY','us','2017-11-01',3.055,3.2,3.05,3.2,9686,'0');
INSERT INTO daily_summary VALUES ('stock','STE','us','2017-11-01',91.61,91.74,87.16,89.25,1042360,'0');
INSERT INTO daily_summary VALUES ('stock','STFC','us','2017-11-01',25.65,25.67,25.22,25.59,16496,'0');
INSERT INTO daily_summary VALUES ('stock','STI','us','2017-11-01',60.66,61.1,60.35,60.53,1783020,'0');
INSERT INTO daily_summary VALUES ('etf','STIP','us','2017-11-01',100.43,100.485,100.413,100.45,80909,'0');
INSERT INTO daily_summary VALUES ('stock','STI_A','us','2017-11-01',24.46,24.478,24.321,24.42,5914,'0');
INSERT INTO daily_summary VALUES ('stock','STI_E','us','2017-11-01',25.55,25.5699,25.48,25.49,18525,'0');
INSERT INTO daily_summary VALUES ('stock','STK','us','2017-11-01',23.186,23.205,22.872,22.882,39650,'0');
INSERT INTO daily_summary VALUES ('stock','STKL','us','2017-11-01',9.4,9.65,9.25,9.4,272509,'0');
INSERT INTO daily_summary VALUES ('stock','STKS','us','2017-11-01',1.4302,1.66,1.39,1.41,102982,'0');
INSERT INTO daily_summary VALUES ('stock','STL','us','2017-11-01',25.029,25.229,24.381,24.68,2297780,'0');
INSERT INTO daily_summary VALUES ('stock','STLD','us','2017-11-01',38.13,38.87,37.61,37.98,3819350,'0');
INSERT INTO daily_summary VALUES ('stock','STLRW','us','2017-11-01',0.32,0.34,0.32,0.34,200,'0');
INSERT INTO daily_summary VALUES ('stock','STLY','us','2017-11-01',1.09,1.1,1.0499,1.06,33124,'0');
INSERT INTO daily_summary VALUES ('stock','STL_A','us','2017-11-01',26.06,26.25,26.06,26.24,17278,'0');
INSERT INTO daily_summary VALUES ('stock','STM','us','2017-11-01',23.81,24,23.58,23.78,3976220,'0');
INSERT INTO daily_summary VALUES ('stock','STML','us','2017-11-01',13.6,14.445,13.55,13.95,971885,'0');
INSERT INTO daily_summary VALUES ('stock','STMP','us','2017-11-01',226.65,229.2,223.35,226.35,279509,'0');
INSERT INTO daily_summary VALUES ('stock','STN','us','2017-11-01',28.7,28.7,28.15,28.45,8846,'0');
INSERT INTO daily_summary VALUES ('stock','STNG','us','2017-11-01',3.6,3.61,3.47,3.49,2366700,'0');
INSERT INTO daily_summary VALUES ('stock','STO','us','2017-11-01',20.36,20.51,20.32,20.42,3247100,'0');
INSERT INTO daily_summary VALUES ('stock','STON','us','2017-11-01',6.9,7,6.66,6.84,200191,'0');
INSERT INTO daily_summary VALUES ('stock','STOR','us','2017-11-01',24.7,24.81,24.535,24.7,1474680,'0');
INSERT INTO daily_summary VALUES ('stock','STOT','us','2017-11-01',49.502,49.54,49.4995,49.525,780,'0');
INSERT INTO daily_summary VALUES ('stock','STPP','us','2017-11-01',31.92,31.92,31.92,31.92,150,'0');
INSERT INTO daily_summary VALUES ('etf','STPZ','us','2017-11-01',52.27,52.299,52.23,52.25,31259,'0');
INSERT INTO daily_summary VALUES ('stock','STRA','us','2017-11-01',95.53,96.17,92.395,95.5,213084,'0');
INSERT INTO daily_summary VALUES ('stock','STRL','us','2017-11-01',18.42,18.824,17.33,17.93,831214,'0');
INSERT INTO daily_summary VALUES ('stock','STRM','us','2017-11-01',1.7,1.75,1.62,1.64,71276,'0');
INSERT INTO daily_summary VALUES ('stock','STRP','us','2017-11-01',181.75,181.75,180.93,181.5,40946,'0');
INSERT INTO daily_summary VALUES ('stock','STRS','us','2017-11-01',29.45,29.8,29.4,29.4,1572,'0');
INSERT INTO daily_summary VALUES ('stock','STRT','us','2017-11-01',46.85,47.5,46,46.7,36123,'0');
INSERT INTO daily_summary VALUES ('stock','STT','us','2017-11-01',92.52,92.96,92.18,92.25,1642660,'0');
INSERT INTO daily_summary VALUES ('stock','STT_C','us','2017-11-01',25.11,25.1255,25.08,25.08,16390,'0');
INSERT INTO daily_summary VALUES ('stock','STT_D','us','2017-11-01',27.63,27.87,27.63,27.82,63121,'0');
INSERT INTO daily_summary VALUES ('stock','STT_E','us','2017-11-01',26.78,26.8393,26.74,26.75,18849,'0');
INSERT INTO daily_summary VALUES ('stock','STT_G','us','2017-11-01',27.2,27.32,27.19,27.21,20610,'0');
INSERT INTO daily_summary VALUES ('stock','STWD','us','2017-11-01',21.6,21.75,21.55,21.63,1157310,'0');
INSERT INTO daily_summary VALUES ('stock','STX','us','2017-11-01',37.38,37.63,36.66,36.96,2784310,'0');
INSERT INTO daily_summary VALUES ('stock','STZ','us','2017-11-01',218.84,219.03,215.34,215.48,1155680,'0');
INSERT INTO daily_summary VALUES ('stock','SU','us','2017-11-01',34.11,34.5,34.01,34.41,3000740,'0');
INSERT INTO daily_summary VALUES ('etf','SUB','us','2017-11-01',105.62,105.64,105.617,105.63,55405,'0');
INSERT INTO daily_summary VALUES ('stock','SUI','us','2017-11-01',90.18,91.49,90.1,91.11,319144,'0');
INSERT INTO daily_summary VALUES ('stock','SUI_A','us','2017-11-01',25.19,25.1954,25.19,25.19,8000,'0');
INSERT INTO daily_summary VALUES ('stock','SUM','us','2017-11-01',31.58,31.65,30.52,30.89,1981710,'0');
INSERT INTO daily_summary VALUES ('stock','SUMR','us','2017-11-01',1.61,1.6987,1.531,1.6987,36158,'0');
INSERT INTO daily_summary VALUES ('stock','SUN','us','2017-11-01',30.169,30.782,30.169,30.578,643894,'0');
INSERT INTO daily_summary VALUES ('stock','SUNS','us','2017-11-01',17.72,17.7434,17.5501,17.66,24724,'0');
INSERT INTO daily_summary VALUES ('stock','SUNW','us','2017-11-01',1.43,1.43,1.37,1.38,128747,'0');
INSERT INTO daily_summary VALUES ('stock','SUP','us','2017-11-01',15.75,16.05,15.65,15.65,146595,'0');
INSERT INTO daily_summary VALUES ('stock','SUPN','us','2017-11-01',41.65,42,40.75,41.95,495546,'0');
INSERT INTO daily_summary VALUES ('stock','SUPV','us','2017-11-01',27,27.13,26.28,26.49,426075,'0');
INSERT INTO daily_summary VALUES ('stock','SUSA','us','2017-11-01',108.01,108.37,107.691,107.77,10691,'0');
INSERT INTO daily_summary VALUES ('stock','SUSB','us','2017-11-01',25.0799,25.0799,25.0799,25.0799,885,'0');
INSERT INTO daily_summary VALUES ('stock','SUSC','us','2017-11-01',25.41,25.41,25.375,25.38,92101,'0');
INSERT INTO daily_summary VALUES ('stock','SVA','us','2017-11-01',7.23,7.3,7.05,7.17,171437,'0');
INSERT INTO daily_summary VALUES ('stock','SVBI','us','2017-11-01',7,7,7,7,6626,'0');
INSERT INTO daily_summary VALUES ('stock','SVM','us','2017-11-01',2.57,2.6112,2.5087,2.6,200251,'0');
INSERT INTO daily_summary VALUES ('stock','SVRA','us','2017-11-01',9.27,9.27,9.1,9.23,67832,'0');
INSERT INTO daily_summary VALUES ('stock','SVU','us','2017-11-01',16.33,16.56,16.05,16.42,1873420,'0');
INSERT INTO daily_summary VALUES ('stock','SVVC','us','2017-11-01',7.9,7.9,7.861,7.8872,10882,'0');
INSERT INTO daily_summary VALUES ('etf','SVXY','us','2017-11-01',108.39,108.652,105.62,106.71,6462500,'0');
INSERT INTO daily_summary VALUES ('stock','SWCH','us','2017-11-01',19.16,19.73,18.88,19,753265,'0');
INSERT INTO daily_summary VALUES ('stock','SWIN','us','2017-11-01',29.2399,29.279,29.2399,29.279,465,'0');
INSERT INTO daily_summary VALUES ('stock','SWIR','us','2017-11-01',22.7,23.4286,22.05,22.35,396392,'0');
INSERT INTO daily_summary VALUES ('stock','SWJ','us','2017-11-01',25.28,25.389,25.28,25.3362,21992,'0');
INSERT INTO daily_summary VALUES ('stock','SWK','us','2017-11-01',162.08,162.54,160.62,161.04,959395,'0');
INSERT INTO daily_summary VALUES ('stock','SWKS','us','2017-11-01',114.52,114.6,110.588,111.32,2614820,'0');
INSERT INTO daily_summary VALUES ('stock','SWM','us','2017-11-01',42.54,42.66,41.64,42,85444,'0');
INSERT INTO daily_summary VALUES ('stock','SWN','us','2017-11-01',5.63,5.86,5.6,5.85,21670100,'0');
INSERT INTO daily_summary VALUES ('stock','SWNC','us','2017-11-01',12.91,13.42,12.91,13.4,12200,'0');
INSERT INTO daily_summary VALUES ('stock','SWP','us','2017-11-01',118.2,118.2,117.963,117.963,5665,'0');
INSERT INTO daily_summary VALUES ('stock','SWX','us','2017-11-01',82.4,82.5,81.3801,81.95,206375,'0');
INSERT INTO daily_summary VALUES ('stock','SWZ','us','2017-11-01',12.5,12.52,12.475,12.52,54731,'0');
INSERT INTO daily_summary VALUES ('stock','SXC','us','2017-11-01',11.25,11.3,10.915,11.06,624564,'0');
INSERT INTO daily_summary VALUES ('stock','SXCP','us','2017-11-01',18.15,18.2,17.7975,18,330019,'0');
INSERT INTO daily_summary VALUES ('stock','SXE','us','2017-11-01',2.2,2.29,1.95,2.01,810578,'0');
INSERT INTO daily_summary VALUES ('stock','SXI','us','2017-11-01',107.81,107.86,100.57,106.71,99661,'0');
INSERT INTO daily_summary VALUES ('stock','SXT','us','2017-11-01',76.305,76.385,75.22,75.479,289579,'0');
INSERT INTO daily_summary VALUES ('stock','SYBT','us','2017-11-01',38.1,38.3,37.35,37.8,33230,'0');
INSERT INTO daily_summary VALUES ('stock','SYBX','us','2017-11-01',14.75,14.9099,13.84,14.01,11283,'0');
INSERT INTO daily_summary VALUES ('etf','SYE','us','2017-11-01',72.78,72.8,72.61,72.61,421,'0');
INSERT INTO daily_summary VALUES ('stock','SYF','us','2017-11-01',32.562,32.801,32.398,32.682,2947450,'0');
INSERT INTO daily_summary VALUES ('etf','SYG','us','2017-11-01',77.85,77.85,77.4,77.4604,1648,'0');
INSERT INTO daily_summary VALUES ('stock','SYK','us','2017-11-01',155.08,155.24,153.53,154.43,1285060,'0');
INSERT INTO daily_summary VALUES ('stock','SYKE','us','2017-11-01',29.08,29.08,28.52,28.81,104811,'0');
INSERT INTO daily_summary VALUES ('etf','SYLD','us','2017-11-01',36.0165,36.04,35.97,36.0048,9522,'0');
INSERT INTO daily_summary VALUES ('stock','SYMC','us','2017-11-01',32.2,32.72,31.77,32.16,10361700,'0');
INSERT INTO daily_summary VALUES ('stock','SYMX','us','2017-11-01',0.44,0.44,0.42,0.4257,19203,'0');
INSERT INTO daily_summary VALUES ('stock','SYN','us','2017-11-01',0.8353,0.8353,0.7756,0.782,888206,'0');
INSERT INTO daily_summary VALUES ('stock','SYNA','us','2017-11-01',37.28,37.38,36.25,36.96,622013,'0');
INSERT INTO daily_summary VALUES ('stock','SYNC','us','2017-11-01',2.4,2.4,2.3,2.35,143478,'0');
INSERT INTO daily_summary VALUES ('stock','SYNL','us','2017-11-01',14.3,14.4,13.7,14.05,9486,'0');
INSERT INTO daily_summary VALUES ('stock','SYNT','us','2017-11-01',23.38,23.57,22.83,23.25,465157,'0');
INSERT INTO daily_summary VALUES ('stock','SYPR','us','2017-11-01',1.45,1.5443,1.36,1.36,66435,'0');
INSERT INTO daily_summary VALUES ('stock','SYRS','us','2017-11-01',17.18,17.18,15.1401,15.19,181191,'0');
INSERT INTO daily_summary VALUES ('stock','SYT','us','2017-11-01',92.39,92.44,92.26,92.34,10398,'0');
INSERT INTO daily_summary VALUES ('etf','SYV','us','2017-11-01',64.06,64.06,64.06,64.06,147,'0');
INSERT INTO daily_summary VALUES ('stock','SYX','us','2017-11-01',29.107,32.222,28.571,31.589,145774,'0');
INSERT INTO daily_summary VALUES ('stock','SYY','us','2017-11-01',55.52,55.86,55.06,55.56,2734060,'0');
INSERT INTO daily_summary VALUES ('stock','SZC','us','2017-11-01',18.7,18.74,18.32,18.32,41565,'0');
INSERT INTO daily_summary VALUES ('etf','SZK','us','2017-11-01',15.3,15.3544,15.3,15.325,1022,'0');
INSERT INTO daily_summary VALUES ('stock','T','us','2017-11-01',33.86,33.94,33.29,33.55,37830900,'0');
INSERT INTO daily_summary VALUES ('stock','TA','us','2017-11-01',4.9,4.925,4.7,4.825,218884,'0');
INSERT INTO daily_summary VALUES ('stock','TAC','us','2017-11-01',5.85,5.95,5.85,5.9,35915,'0');
INSERT INTO daily_summary VALUES ('stock','TACO','us','2017-11-01',12.67,12.79,12.58,12.6,244645,'0');
INSERT INTO daily_summary VALUES ('stock','TACOW','us','2017-11-01',3.3098,3.3185,3.3098,3.3185,1450,'0');
INSERT INTO daily_summary VALUES ('stock','TACT','us','2017-11-01',9.6,9.8,9.6,9.8,23343,'0');
INSERT INTO daily_summary VALUES ('stock','TAHO','us','2017-11-01',4.84,4.91,4.72,4.75,1988920,'0');
INSERT INTO daily_summary VALUES ('stock','TAIL','us','2017-11-01',23.13,23.19,23.11,23.155,20724,'0');
INSERT INTO daily_summary VALUES ('stock','TAIT','us','2017-11-01',1.59,1.61,1.5785,1.58,20844,'0');
INSERT INTO daily_summary VALUES ('stock','TAL','us','2017-11-01',27.9,28.25,27.01,27.04,4347880,'0');
INSERT INTO daily_summary VALUES ('etf','TAN','us','2017-11-01',24.02,24.18,23.98,24.09,340492,'0');
INSERT INTO daily_summary VALUES ('stock','TANH','us','2017-11-01',2.801,2.88,2.8,2.81,109736,'0');
INSERT INTO daily_summary VALUES ('stock','TANNI','us','2017-11-01',25,25,24.75,24.9,5754,'0');
INSERT INTO daily_summary VALUES ('stock','TANNL','us','2017-11-01',24.6499,24.6998,24.53,24.53,4085,'0');
INSERT INTO daily_summary VALUES ('stock','TANNZ','us','2017-11-01',24.6,24.6,24.3779,24.3779,1700,'0');
INSERT INTO daily_summary VALUES ('etf','TAO','us','2017-11-01',30.19,30.19,29.8,29.9,74473,'0');
INSERT INTO daily_summary VALUES ('stock','TAP','us','2017-11-01',81.59,83.32,80.25,81.04,2501920,'0');
INSERT INTO daily_summary VALUES ('stock','TAPR','us','2017-11-01',22.49,22.49,22.49,22.49,246,'0');
INSERT INTO daily_summary VALUES ('stock','TARO','us','2017-11-01',112.41,112.89,110.65,111.94,65932,'0');
INSERT INTO daily_summary VALUES ('stock','TAST','us','2017-11-01',11.3,11.4,10.95,11.05,160477,'0');
INSERT INTO daily_summary VALUES ('stock','TAT','us','2017-11-01',0.76,0.777,0.75,0.7501,81922,'0');
INSERT INTO daily_summary VALUES ('stock','TATT','us','2017-11-01',10.75,11.25,10.75,11.2,1571,'0');
INSERT INTO daily_summary VALUES ('stock','TAX','us','2017-11-01',14,14,12.8,12.95,29617,'0');
INSERT INTO daily_summary VALUES ('stock','TAXR','us','2017-11-01',20.16,20.17,19.91,19.9325,6914,'0');
INSERT INTO daily_summary VALUES ('stock','TAYD','us','2017-11-01',12.5928,13.06,12.5928,13.06,1367,'0');
INSERT INTO daily_summary VALUES ('stock','TBBK','us','2017-11-01',8.53,8.57,8.44,8.52,520240,'0');
INSERT INTO daily_summary VALUES ('etf','TBF','us','2017-11-01',22.4,22.409,22.261,22.34,199447,'0');
INSERT INTO daily_summary VALUES ('stock','TBI','us','2017-11-01',27.4,27.4,25.5,26.05,392050,'0');
INSERT INTO daily_summary VALUES ('stock','TBK','us','2017-11-01',31.25,31.45,30.3,30.75,139448,'0');
INSERT INTO daily_summary VALUES ('stock','TBLU','us','2017-11-01',28.4963,28.4963,28.4963,28.4963,100,'0');
INSERT INTO daily_summary VALUES ('stock','TBNK','us','2017-11-01',31.4,31.439,30.972,31.131,15268,'0');
INSERT INTO daily_summary VALUES ('stock','TBPH','us','2017-11-01',29.19,29.24,28.74,28.8,152248,'0');
INSERT INTO daily_summary VALUES ('etf','TBT','us','2017-11-01',35.5,35.54,35.06,35.28,1943650,'0');
INSERT INTO daily_summary VALUES ('etf','TBX','us','2017-11-01',28.185,28.228,28.185,28.2232,936,'0');
INSERT INTO daily_summary VALUES ('stock','TCAP','us','2017-11-01',12.34,12.4,12.1803,12.25,418616,'0');
INSERT INTO daily_summary VALUES ('stock','TCBI','us','2017-11-01',86.75,87.3,84.75,85,326810,'0');
INSERT INTO daily_summary VALUES ('stock','TCBIP','us','2017-11-01',25.58,25.7299,25.52,25.6,11576,'0');
INSERT INTO daily_summary VALUES ('stock','TCBK','us','2017-11-01',41.78,41.78,40.75,41.13,60823,'0');
INSERT INTO daily_summary VALUES ('stock','TCCB','us','2017-11-01',25.48,25.48,25.45,25.45,1140,'0');
INSERT INTO daily_summary VALUES ('stock','TCCO','us','2017-11-01',4,4.099,3.8162,4,35278,'0');
INSERT INTO daily_summary VALUES ('stock','TCF','us','2017-11-01',18.33,18.54,18.1,18.28,1711890,'0');
INSERT INTO daily_summary VALUES ('stock','TCFC','us','2017-11-01',36.11,36.45,35.98,36.12,8795,'0');
INSERT INTO daily_summary VALUES ('stock','TCF_C','us','2017-11-01',25.7198,25.74,25.7198,25.74,3400,'0');
INSERT INTO daily_summary VALUES ('stock','TCF_D','us','2017-11-01',25.8,25.8,25.6,25.8,43932,'0');
INSERT INTO daily_summary VALUES ('stock','TCGP','us','2017-11-01',25.3,25.33,25.27,25.29,71135,'0');
INSERT INTO daily_summary VALUES ('stock','TCHF','us','2017-11-01',40.69,40.69,40.02,40.15,2250,'0');
INSERT INTO daily_summary VALUES ('stock','TCI','us','2017-11-01',29.97,30.17,29.97,30.17,516,'0');
INSERT INTO daily_summary VALUES ('stock','TCMD','us','2017-11-01',28.63,28.695,26.6,27.03,345186,'0');
INSERT INTO daily_summary VALUES ('stock','TCO','us','2017-11-01',47.24,47.99,47.01,47.54,487165,'0');
INSERT INTO daily_summary VALUES ('stock','TCON','us','2017-11-01',2.85,3,2.8,2.95,33527,'0');
INSERT INTO daily_summary VALUES ('stock','TCO_J','us','2017-11-01',25.25,25.3,25.25,25.3,8252,'0');
INSERT INTO daily_summary VALUES ('stock','TCO_K','us','2017-11-01',25.2103,25.235,25.2,25.235,2450,'0');
INSERT INTO daily_summary VALUES ('stock','TCP','us','2017-11-01',52.55,54.465,52.511,54.2,271564,'0');
INSERT INTO daily_summary VALUES ('stock','TCPC','us','2017-11-01',16.04,16.16,15.83,16,163448,'0');
INSERT INTO daily_summary VALUES ('stock','TCRD','us','2017-11-01',9.04,9.09,9.01,9.04,103924,'0');
INSERT INTO daily_summary VALUES ('stock','TCRX','us','2017-11-01',25.3,25.3,25.3,25.3,600,'0');
INSERT INTO daily_summary VALUES ('stock','TCRZ','us','2017-11-01',25.93,25.93,25.93,25.93,100,'0');
INSERT INTO daily_summary VALUES ('stock','TCS','us','2017-11-01',3.63,3.825,3.62,3.68,214238,'0');
INSERT INTO daily_summary VALUES ('stock','TCTL','us','2017-11-01',29.451,29.5,29.451,29.5,553,'0');
INSERT INTO daily_summary VALUES ('stock','TCX','us','2017-11-01',58.7,58.7,57.5,57.85,48274,'0');
INSERT INTO daily_summary VALUES ('stock','TD','us','2017-11-01',57.02,57.15,56.78,56.87,1227410,'0');
INSERT INTO daily_summary VALUES ('stock','TDA','us','2017-11-01',25.26,25.2799,25.08,25.17,8547,'0');
INSERT INTO daily_summary VALUES ('stock','TDC','us','2017-11-01',33.59,33.6,32.84,33.29,2116690,'0');
INSERT INTO daily_summary VALUES ('stock','TDE','us','2017-11-01',25.213,25.213,25.135,25.164,4994,'0');
INSERT INTO daily_summary VALUES ('stock','TDF','us','2017-11-01',21.55,21.68,21.55,21.66,83714,'0');
INSERT INTO daily_summary VALUES ('stock','TDG','us','2017-11-01',279.64,282.87,278.93,280.95,421241,'0');
INSERT INTO daily_summary VALUES ('stock','TDI','us','2017-11-01',25.74,25.85,25.74,25.7769,2587,'0');
INSERT INTO daily_summary VALUES ('etf','TDIV','us','2017-11-01',34.47,34.47,34.1599,34.36,53730,'0');
INSERT INTO daily_summary VALUES ('stock','TDJ','us','2017-11-01',25.41,25.46,25.41,25.42,29320,'0');
INSERT INTO daily_summary VALUES ('stock','TDOC','us','2017-11-01',33.35,33.9,30.4,30.8,1808980,'0');
INSERT INTO daily_summary VALUES ('stock','TDS','us','2017-11-01',29.23,29.25,28.34,28.46,461681,'0');
INSERT INTO daily_summary VALUES ('etf','TDTF','us','2017-11-01',25.02,25.03,24.99,25.02,45977,'0');
INSERT INTO daily_summary VALUES ('etf','TDTT','us','2017-11-01',24.57,24.58,24.54,24.57,140268,'0');
INSERT INTO daily_summary VALUES ('stock','TDW','us','2017-11-01',27.3,27.4,26.86,27.18,175568,'0');
INSERT INTO daily_summary VALUES ('stock','TDY','us','2017-11-01',170.9,171,168.6,169.86,164104,'0');
INSERT INTO daily_summary VALUES ('stock','TEAM','us','2017-11-01',48.235,48.76,46.93,48.69,1262080,'0');
INSERT INTO daily_summary VALUES ('stock','TEAR','us','2017-11-01',1.37,1.4,1.31,1.36,87629,'0');
INSERT INTO daily_summary VALUES ('stock','TECD','us','2017-11-01',93.38,93.92,92.01,92.72,495910,'0');
INSERT INTO daily_summary VALUES ('stock','TECH','us','2017-11-01',132.15,132.24,127.42,128.08,215191,'0');
INSERT INTO daily_summary VALUES ('stock','TECK','us','2017-11-01',21.02,21.35,20.85,21.1,7048050,'0');
INSERT INTO daily_summary VALUES ('etf','TECL','us','2017-11-01',109.04,109.38,106,107.63,216554,'0');
INSERT INTO daily_summary VALUES ('etf','TECS','us','2017-11-01',7.25,7.4601,7.24,7.34,226802,'0');
INSERT INTO daily_summary VALUES ('stock','TEDU','us','2017-11-01',14.69,14.769,14.38,14.46,129038,'0');
INSERT INTO daily_summary VALUES ('stock','TEF','us','2017-11-01',10.5,10.52,10.41,10.46,1672010,'0');
INSERT INTO daily_summary VALUES ('stock','TEGP','us','2017-11-01',25.25,25.8,25,25.18,175793,'0');
INSERT INTO daily_summary VALUES ('stock','TEI','us','2017-11-01',11.62,11.68,11.62,11.66,72631,'0');
INSERT INTO daily_summary VALUES ('stock','TEL','us','2017-11-01',92.86,93.11,90.23,91.87,2479580,'0');
INSERT INTO daily_summary VALUES ('stock','TELL','us','2017-11-01',10.94,11,10.6,10.77,300645,'0');
INSERT INTO daily_summary VALUES ('stock','TEN','us','2017-11-01',58.52,59.33,58.09,58.26,509908,'0');
INSERT INTO daily_summary VALUES ('stock','TENX','us','2017-11-01',0.4354,0.4515,0.42,0.438,139709,'0');
INSERT INTO daily_summary VALUES ('stock','TEO','us','2017-11-01',32.96,33.93,32.96,33.77,343464,'0');
INSERT INTO daily_summary VALUES ('stock','TEP','us','2017-11-01',43.83,44.72,43.8,44.46,171152,'0');
INSERT INTO daily_summary VALUES ('stock','TER','us','2017-11-01',43.28,43.28,41.9003,42.54,3185860,'0');
INSERT INTO daily_summary VALUES ('stock','TERM','us','2017-11-01',21.33,21.33,21.18,21.209,3725,'0');
INSERT INTO daily_summary VALUES ('stock','TERP','us','2017-11-01',13.28,13.42,13.225,13.28,791015,'0');
INSERT INTO daily_summary VALUES ('stock','TESO','us','2017-11-01',3.85,4.05,3.8,3.9,152629,'0');
INSERT INTO daily_summary VALUES ('stock','TESS','us','2017-11-01',15.692,15.914,15.495,15.742,11949,'0');
INSERT INTO daily_summary VALUES ('stock','TETF','us','2017-11-01',17.7,17.71,17.66,17.66,421,'0');
INSERT INTO daily_summary VALUES ('stock','TEUM','us','2017-11-01',1.19,1.19,1.11,1.12,1011420,'0');
INSERT INTO daily_summary VALUES ('stock','TEVA','us','2017-11-01',13.55,14.08,13.38,14.02,25748000,'0');
INSERT INTO daily_summary VALUES ('stock','TEX','us','2017-11-01',45.931,46.166,43.286,44.773,2694900,'0');
INSERT INTO daily_summary VALUES ('etf','TFI','us','2017-11-01',48.91,48.91,48.8363,48.86,346050,'0');
INSERT INTO daily_summary VALUES ('etf','TFLO','us','2017-11-01',50.19,50.22,50.19,50.22,8278,'0');
INSERT INTO daily_summary VALUES ('stock','TFSL','us','2017-11-01',15.46,15.58,15.21,15.28,94994,'0');
INSERT INTO daily_summary VALUES ('stock','TFX','us','2017-11-01',237.18,242.94,235.29,241.71,302049,'0');
INSERT INTO daily_summary VALUES ('stock','TG','us','2017-11-01',19.65,19.95,18.2,19,96005,'0');
INSERT INTO daily_summary VALUES ('stock','TGA','us','2017-11-01',1.68,1.769,1.68,1.769,151427,'0');
INSERT INTO daily_summary VALUES ('stock','TGB','us','2017-11-01',2.24,2.26,2.17,2.24,1052330,'0');
INSERT INTO daily_summary VALUES ('stock','TGC','us','2017-11-01',0.6696,0.6699,0.6499,0.6599,8050,'0');
INSERT INTO daily_summary VALUES ('stock','TGEN','us','2017-11-01',2.96,2.9941,2.861,2.9301,11124,'0');
INSERT INTO daily_summary VALUES ('stock','TGH','us','2017-11-01',19.9,19.9,18.35,18.9,191664,'0');
INSERT INTO daily_summary VALUES ('stock','TGI','us','2017-11-01',31.4,31.6,30.5,30.9,342502,'0');
INSERT INTO daily_summary VALUES ('stock','TGLS','us','2017-11-01',6.93,7.11,6.81,7.08,43562,'0');
INSERT INTO daily_summary VALUES ('stock','TGNA','us','2017-11-01',12.3,12.38,12.07,12.37,1992610,'0');
INSERT INTO daily_summary VALUES ('stock','TGP','us','2017-11-01',17.112,17.558,17.013,17.36,166014,'0');
INSERT INTO daily_summary VALUES ('stock','TGP_A','us','2017-11-01',25.73,25.81,25.57,25.5799,20551,'0');
INSERT INTO daily_summary VALUES ('stock','TGP_B','us','2017-11-01',24.92,25,24.9,24.9327,113054,'0');
INSERT INTO daily_summary VALUES ('stock','TGS','us','2017-11-01',21.3,21.7,21.01,21.6,201592,'0');
INSERT INTO daily_summary VALUES ('stock','TGT','us','2017-11-01',58.8,59.42,58.615,58.86,4710540,'0');
INSERT INTO daily_summary VALUES ('stock','TGTX','us','2017-11-01',8.3,8.375,7.8,7.8,977238,'0');
INSERT INTO daily_summary VALUES ('stock','THC','us','2017-11-01',14.09,14.41,13.9,14.35,2252590,'0');
INSERT INTO daily_summary VALUES ('etf','THD','us','2017-11-01',88.4,88.76,88.111,88.21,198627,'0');
INSERT INTO daily_summary VALUES ('stock','THFF','us','2017-11-01',48.15,48.15,47,47.65,19390,'0');
INSERT INTO daily_summary VALUES ('stock','THG','us','2017-11-01',99.05,99.7,98.52,99.45,191146,'0');
INSERT INTO daily_summary VALUES ('stock','THGA','us','2017-11-01',25.51,25.5226,25.4801,25.4801,1471,'0');
INSERT INTO daily_summary VALUES ('stock','THM','us','2017-11-01',0.434,0.4379,0.4239,0.4252,81904,'0');
INSERT INTO daily_summary VALUES ('stock','THO','us','2017-11-01',137.64,138.07,136.07,136.82,575303,'0');
INSERT INTO daily_summary VALUES ('stock','THQ','us','2017-11-01',17.63,17.63,17.5,17.53,162058,'0');
INSERT INTO daily_summary VALUES ('stock','THR','us','2017-11-01',21.73,21.73,21.16,21.27,164411,'0');
INSERT INTO daily_summary VALUES ('stock','THRM','us','2017-11-01',33.7,34.45,33.25,33.65,215357,'0');
INSERT INTO daily_summary VALUES ('stock','THS','us','2017-11-01',66.52,67.13,66.05,66.36,1109140,'0');
INSERT INTO daily_summary VALUES ('stock','THST','us','2017-11-01',1.86,1.96,1.86,1.95,11138,'0');
INSERT INTO daily_summary VALUES ('stock','THW','us','2017-11-01',13.98,14.0501,13.88,14.04,237513,'0');
INSERT INTO daily_summary VALUES ('stock','TI','us','2017-11-01',8.74,8.79,8.67,8.69,26354,'0');
INSERT INTO daily_summary VALUES ('stock','TICC','us','2017-11-01',6.38,6.43,5.88,6.03,707418,'0');
INSERT INTO daily_summary VALUES ('stock','TICCL','us','2017-11-01',26.405,26.405,26.405,26.405,591,'0');
INSERT INTO daily_summary VALUES ('stock','TIER','us','2017-11-01',19.71,19.71,19.3592,19.53,104497,'0');
INSERT INTO daily_summary VALUES ('stock','TIF','us','2017-11-01',94.3,95.31,93.84,93.98,1207370,'0');
INSERT INTO daily_summary VALUES ('stock','TIG','us','2017-11-01',22.84,22.84,22.5174,22.5174,2342,'0');
INSERT INTO daily_summary VALUES ('stock','TIK','us','2017-11-01',2.9442,2.9442,2.4001,2.494,21120,'0');
INSERT INTO daily_summary VALUES ('stock','TIL','us','2017-11-01',3.968,3.9963,3.968,3.9963,1160,'0');
INSERT INTO daily_summary VALUES ('stock','TILE','us','2017-11-01',22.78,22.88,22.531,22.631,381479,'0');
INSERT INTO daily_summary VALUES ('etf','TILT','us','2017-11-01',108.11,108.11,107.31,107.62,9616,'0');
INSERT INTO daily_summary VALUES ('stock','TIME','us','2017-11-01',11.65,11.65,11.1,11.2,812876,'0');
INSERT INTO daily_summary VALUES ('etf','TIP','us','2017-11-01',113.64,113.82,113.583,113.7,2406340,'0');
INSERT INTO daily_summary VALUES ('stock','TIPT','us','2017-11-01',6.6,6.65,6.55,6.55,9760,'0');
INSERT INTO daily_summary VALUES ('etf','TIPX','us','2017-11-01',19.49,19.5347,19.49,19.52,4574,'0');
INSERT INTO daily_summary VALUES ('etf','TIPZ','us','2017-11-01',57.91,57.9498,57.88,57.942,1214,'0');
INSERT INTO daily_summary VALUES ('stock','TIS','us','2017-11-01',12.27,12.98,11.965,12.94,107922,'0');
INSERT INTO daily_summary VALUES ('stock','TISA','us','2017-11-01',1.3,1.3,1.23,1.25,40596,'0');
INSERT INTO daily_summary VALUES ('stock','TISI','us','2017-11-01',12.45,12.45,11.85,11.95,617633,'0');
INSERT INTO daily_summary VALUES ('stock','TITN','us','2017-11-01',15.06,15.09,14.71,14.79,72273,'0');
INSERT INTO daily_summary VALUES ('stock','TIVO','us','2017-11-01',18.3,18.5,17.95,18.025,720122,'0');
INSERT INTO daily_summary VALUES ('stock','TJX','us','2017-11-01',69.961,70.08,69.065,69.404,2931100,'0');
INSERT INTO daily_summary VALUES ('stock','TK','us','2017-11-01',8.0407,8.2789,8.001,8.13,324875,'0');
INSERT INTO daily_summary VALUES ('stock','TKAT','us','2017-11-01',3.2,3.5069,3.2,3.5069,3800,'0');
INSERT INTO daily_summary VALUES ('stock','TKC','us','2017-11-01',9.54,9.79,9.54,9.77,574097,'0');
INSERT INTO daily_summary VALUES ('stock','TKF','us','2017-11-01',9.51,9.65,9.47,9.51,14041,'0');
INSERT INTO daily_summary VALUES ('stock','TKR','us','2017-11-01',47.8,47.85,46.075,46.55,991437,'0');
INSERT INTO daily_summary VALUES ('stock','TLEH','us','2017-11-01',30.6891,30.75,30.6301,30.64,2225,'0');
INSERT INTO daily_summary VALUES ('stock','TLF','us','2017-11-01',7.8391,7.85,7.65,7.85,13960,'0');
INSERT INTO daily_summary VALUES ('stock','TLGT','us','2017-11-01',5.73,5.9,5.52,5.53,217267,'0');
INSERT INTO daily_summary VALUES ('etf','TLH','us','2017-11-01',135.9,137.21,135.442,136.06,178530,'0');
INSERT INTO daily_summary VALUES ('stock','TLI','us','2017-11-01',10.8,10.82,10.73,10.79,18959,'0');
INSERT INTO daily_summary VALUES ('stock','TLK','us','2017-11-01',29.6,29.74,29.15,29.42,515514,'0');
INSERT INTO daily_summary VALUES ('stock','TLND','us','2017-11-01',41.34,42.5,41.08,42.05,98810,'0');
INSERT INTO daily_summary VALUES ('stock','TLP','us','2017-11-01',41.8,42.33,41.58,42.33,70415,'0');
INSERT INTO daily_summary VALUES ('stock','TLRA','us','2017-11-01',4.44,4.479,4.31,4.36,124962,'0');
INSERT INTO daily_summary VALUES ('stock','TLRD','us','2017-11-01',15.49,15.62,14.94,15.39,869758,'0');
INSERT INTO daily_summary VALUES ('etf','TLT','us','2017-11-01',124.35,125.14,124.3,124.75,8940440,'0');
INSERT INTO daily_summary VALUES ('etf','TLTD','us','2017-11-01',68.3,68.3,68.0403,68.14,18464,'0');
INSERT INTO daily_summary VALUES ('etf','TLTE','us','2017-11-01',58.03,58.42,58,58.08,18191,'0');
INSERT INTO daily_summary VALUES ('stock','TLYS','us','2017-11-01',11.93,12.14,11.8,11.94,84812,'0');
INSERT INTO daily_summary VALUES ('stock','TM','us','2017-11-01',124.29,124.92,124.18,124.43,194195,'0');
INSERT INTO daily_summary VALUES ('etf','TMF','us','2017-11-01',20.83,21.23,20.81,21.05,425941,'0');
INSERT INTO daily_summary VALUES ('stock','TMHC','us','2017-11-01',24.24,24.25,23.25,23.86,2481200,'0');
INSERT INTO daily_summary VALUES ('stock','TMK','us','2017-11-01',84.54,85.07,84.27,84.76,376594,'0');
INSERT INTO daily_summary VALUES ('stock','TMK_B','us','2017-11-01',25.371,25.43,25.371,25.43,1231,'0');
INSERT INTO daily_summary VALUES ('stock','TMK_C','us','2017-11-01',27.19,27.1916,27.0736,27.1916,8511,'0');
INSERT INTO daily_summary VALUES ('stock','TMO','us','2017-11-01',195.14,195.31,193.22,194.19,1056620,'0');
INSERT INTO daily_summary VALUES ('stock','TMP','us','2017-11-01',87.256,87.256,84.133,84.531,21131,'0');
INSERT INTO daily_summary VALUES ('stock','TMQ','us','2017-11-01',1.02,1.0438,1,1,55118,'0');
INSERT INTO daily_summary VALUES ('stock','TMST','us','2017-11-01',14.62,15.76,14.26,14.43,643423,'0');
INSERT INTO daily_summary VALUES ('stock','TMUS','us','2017-11-01',60.14,60.4,59.36,59.75,3352010,'0');
INSERT INTO daily_summary VALUES ('stock','TMUSP','us','2017-11-01',97.73,97.73,96.55,96.89,5536,'0');
INSERT INTO daily_summary VALUES ('etf','TMV','us','2017-11-01',19.05,19.1,18.72,18.92,854772,'0');
INSERT INTO daily_summary VALUES ('etf','TNA','us','2017-11-01',67.73,67.77,63.74,64.89,5436910,'0');
INSERT INTO daily_summary VALUES ('stock','TNAV','us','2017-11-01',6.45,6.5,6.05,6.1,139614,'0');
INSERT INTO daily_summary VALUES ('stock','TNC','us','2017-11-01',69.95,70.5,68.4,68.45,73374,'0');
INSERT INTO daily_summary VALUES ('stock','TNDM','us','2017-11-01',2.47,2.62,2.43,2.55,333106,'0');
INSERT INTO daily_summary VALUES ('stock','TNET','us','2017-11-01',34.88,34.97,33.5101,34.03,484500,'0');
INSERT INTO daily_summary VALUES ('stock','TNH','us','2017-11-01',76.94,77.2,76.73,76.89,19505,'0');
INSERT INTO daily_summary VALUES ('stock','TNK','us','2017-11-01',1.49,1.53,1.49,1.5,479808,'0');
INSERT INTO daily_summary VALUES ('stock','TNP','us','2017-11-01',4.49,4.5394,4.3911,4.4208,333016,'0');
INSERT INTO daily_summary VALUES ('stock','TNP_B','us','2017-11-01',25.19,25.23,25.17,25.17,8122,'0');
INSERT INTO daily_summary VALUES ('stock','TNP_C','us','2017-11-01',25.57,25.6,25.5546,25.5546,7515,'0');
INSERT INTO daily_summary VALUES ('stock','TNP_D','us','2017-11-01',25.57,25.6,25.57,25.59,4130,'0');
INSERT INTO daily_summary VALUES ('stock','TNP_E','us','2017-11-01',25.74,25.79,25.74,25.7658,6129,'0');
INSERT INTO daily_summary VALUES ('stock','TNTR','us','2017-11-01',3.86,3.92,3.61,3.68,158544,'0');
INSERT INTO daily_summary VALUES ('stock','TNXP','us','2017-11-01',3.7,3.8199,3.64,3.66,42263,'0');
INSERT INTO daily_summary VALUES ('stock','TOCA','us','2017-11-01',11.15,11.55,10.75,11.28,174723,'0');
INSERT INTO daily_summary VALUES ('etf','TOK','us','2017-11-01',63.63,63.74,63.491,63.491,954,'0');
INSERT INTO daily_summary VALUES ('stock','TOL','us','2017-11-01',46.26,46.63,46,46.63,1530950,'0');
INSERT INTO daily_summary VALUES ('etf','TOLZ','us','2017-11-01',42.76,42.7732,42.64,42.64,3200,'0');
INSERT INTO daily_summary VALUES ('stock','TOO','us','2017-11-01',2.5502,2.6398,2.5502,2.59,842785,'0');
INSERT INTO daily_summary VALUES ('stock','TOO_A','us','2017-11-01',23.84,23.9882,23.8299,23.884,8506,'0');
INSERT INTO daily_summary VALUES ('stock','TOO_B','us','2017-11-01',25.26,25.275,25.0346,25.0716,17654,'0');
INSERT INTO daily_summary VALUES ('stock','TOPS','us','2017-11-01',0.47,0.4801,0.4032,0.43,1431990,'0');
INSERT INTO daily_summary VALUES ('stock','TORM','us','2017-11-01',6.695,6.695,5.7,5.7,3446,'0');
INSERT INTO daily_summary VALUES ('stock','TOT','us','2017-11-01',56.23,56.5,56.0491,56.22,1454630,'0');
INSERT INTO daily_summary VALUES ('etf','TOTL','us','2017-11-01',48.88,49.01,48.88,48.97,300604,'0');
INSERT INTO daily_summary VALUES ('stock','TOUR','us','2017-11-01',6.85,6.98,6.71,6.86,225586,'0');
INSERT INTO daily_summary VALUES ('stock','TOWN','us','2017-11-01',33.9,34.1,33.1,33.4,158450,'0');
INSERT INTO daily_summary VALUES ('stock','TOWR','us','2017-11-01',30.831,31.727,30.477,31.528,173544,'0');
INSERT INTO daily_summary VALUES ('stock','TPB','us','2017-11-01',17.52,17.98,17.35,17.41,25868,'0');
INSERT INTO daily_summary VALUES ('stock','TPC','us','2017-11-01',28.5,28.6,27.7,28.15,261932,'0');
INSERT INTO daily_summary VALUES ('stock','TPGE','us','2017-11-01',9.74,9.758,9.74,9.758,1100,'0');
INSERT INTO daily_summary VALUES ('stock','TPGH','us','2017-11-01',9.8904,9.8904,9.7485,9.7485,27861,'0');
INSERT INTO daily_summary VALUES ('stock','TPH','us','2017-11-01',17.71,17.84,17.61,17.7,4747570,'0');
INSERT INTO daily_summary VALUES ('stock','TPHS','us','2017-11-01',7.25,7.25,7.08,7.11,9444,'0');
INSERT INTO daily_summary VALUES ('stock','TPIC','us','2017-11-01',25.34,25.34,24.04,24.23,160078,'0');
INSERT INTO daily_summary VALUES ('stock','TPIV','us','2017-11-01',3.01,3.0186,2.874,2.874,7746,'0');
INSERT INTO daily_summary VALUES ('stock','TPL','us','2017-11-01',408.69,414.98,408.01,408.68,14119,'0');
INSERT INTO daily_summary VALUES ('stock','TPOR','us','2017-11-01',29.45,29.599,29.45,29.599,3537,'0');
INSERT INTO daily_summary VALUES ('stock','TPR','us','2017-11-01',41,41.28,40.21,40.68,2730540,'0');
INSERT INTO daily_summary VALUES ('stock','TPRE','us','2017-11-01',16.8,16.85,16.4,16.6,389431,'0');
INSERT INTO daily_summary VALUES ('stock','TPVG','us','2017-11-01',13.92,13.99,13.7,13.82,95392,'0');
INSERT INTO daily_summary VALUES ('stock','TPVY','us','2017-11-01',25.66,25.75,25.6064,25.68,5871,'0');
INSERT INTO daily_summary VALUES ('stock','TPX','us','2017-11-01',65.69,66.87,64.11,66.49,1562680,'0');
INSERT INTO daily_summary VALUES ('stock','TPYP','us','2017-11-01',23,23.02,22.93,22.99,15340,'0');
INSERT INTO daily_summary VALUES ('stock','TPZ','us','2017-11-01',20.2,20.47,20.2,20.3,37296,'0');
INSERT INTO daily_summary VALUES ('etf','TQQQ','us','2017-11-01',131.58,131.76,128.35,129.78,3460960,'0');
INSERT INTO daily_summary VALUES ('stock','TR','us','2017-11-01',35.8,35.9,35.05,35.55,137701,'0');
INSERT INTO daily_summary VALUES ('stock','TRC','us','2017-11-01',18.95,19.05,18.5901,18.72,131842,'0');
INSERT INTO daily_summary VALUES ('stock','TRCB','us','2017-11-01',19.539,19.659,18.961,18.961,12142,'0');
INSERT INTO daily_summary VALUES ('stock','TRCH','us','2017-11-01',1.4,1.45,1.33,1.4,200357,'0');
INSERT INTO daily_summary VALUES ('stock','TRCO','us','2017-11-01',40.87,40.93,40.43,40.79,804379,'0');
INSERT INTO daily_summary VALUES ('stock','TREC','us','2017-11-01',12.3,12.35,12.175,12.25,26979,'0');
INSERT INTO daily_summary VALUES ('stock','TREE','us','2017-11-01',270,272.54,260.6,264.2,103758,'0');
INSERT INTO daily_summary VALUES ('stock','TREX','us','2017-11-01',109.79,109.79,105.09,107.35,587606,'0');
INSERT INTO daily_summary VALUES ('stock','TRGP','us','2017-11-01',41.89,42.51,41.64,42.01,3319230,'0');
INSERT INTO daily_summary VALUES ('stock','TRHC','us','2017-11-01',28.88,29.1736,26.26,26.83,132276,'0');
INSERT INTO daily_summary VALUES ('stock','TRI','us','2017-11-01',45.5,45.54,43.4,44.23,2411290,'0');
INSERT INTO daily_summary VALUES ('stock','TRIB','us','2017-11-01',5.05,5.4,4.98,5.17,130506,'0');
INSERT INTO daily_summary VALUES ('stock','TRIL','us','2017-11-01',7.85,8.7,7.75,8.25,657623,'0');
INSERT INTO daily_summary VALUES ('stock','TRIP','us','2017-11-01',37.76,38.47,37.675,37.87,2634980,'0');
INSERT INTO daily_summary VALUES ('stock','TRK','us','2017-11-01',20.003,20.003,19.467,19.467,47060,'0');
INSERT INTO daily_summary VALUES ('stock','TRMB','us','2017-11-01',41.1,41.1,40.08,40.37,1251660,'0');
INSERT INTO daily_summary VALUES ('stock','TRMK','us','2017-11-01',33.16,33.32,32.41,32.73,121355,'0');
INSERT INTO daily_summary VALUES ('stock','TRMT','us','2017-11-01',16.59,16.59,15.97,16.01,33200,'0');
INSERT INTO daily_summary VALUES ('stock','TRN','us','2017-11-01',32.84,32.8742,32.135,32.36,1620920,'0');
INSERT INTO daily_summary VALUES ('stock','TRNC','us','2017-11-01',14.87,15.02,14.57,14.75,55222,'0');
INSERT INTO daily_summary VALUES ('stock','TRNO','us','2017-11-01',36.95,37.3,36.6,37.27,337982,'0');
INSERT INTO daily_summary VALUES ('stock','TRNS','us','2017-11-01',12.75,12.75,12.35,12.4,18616,'0');
INSERT INTO daily_summary VALUES ('stock','TROV','us','2017-11-01',0.72,0.785,0.701,0.78,1544370,'0');
INSERT INTO daily_summary VALUES ('stock','TROW','us','2017-11-01',93.48,93.64,92.46,92.57,1030960,'0');
INSERT INTO daily_summary VALUES ('stock','TROX','us','2017-11-01',26.9,26.98,25.79,25.87,1295910,'0');
INSERT INTO daily_summary VALUES ('stock','TRP','us','2017-11-01',47.38,48.07,47.38,47.98,840368,'0');
INSERT INTO daily_summary VALUES ('stock','TRPX','us','2017-11-01',5.13,5.35,5.13,5.26,24629,'0');
INSERT INTO daily_summary VALUES ('stock','TRQ','us','2017-11-01',3.08,3.1227,3.08,3.09,1251880,'0');
INSERT INTO daily_summary VALUES ('stock','TRS','us','2017-11-01',26.8,27.1,26.4,26.65,118893,'0');
INSERT INTO daily_summary VALUES ('etf','TRSK','us','2017-11-01',30.3,30.3,30.29,30.29,979,'0');
INSERT INTO daily_summary VALUES ('stock','TRST','us','2017-11-01',9.2,9.25,8.925,9,204420,'0');
INSERT INTO daily_summary VALUES ('stock','TRT','us','2017-11-01',5.75,5.75,5.39,5.53,23263,'0');
INSERT INTO daily_summary VALUES ('stock','TRTN','us','2017-11-01',40.5,41.29,38.97,39.61,343513,'0');
INSERT INTO daily_summary VALUES ('stock','TRTX','us','2017-11-01',19.47,19.62,19.27,19.5,37441,'0');
INSERT INTO daily_summary VALUES ('stock','TRU','us','2017-11-01',52.83,53.62,52.33,53.48,2218940,'0');
INSERT INTO daily_summary VALUES ('stock','TRUE','us','2017-11-01',16.21,16.24,15.84,16.09,1162490,'0');
INSERT INTO daily_summary VALUES ('stock','TRUP','us','2017-11-01',28.44,28.75,27.9,28.05,243287,'0');
INSERT INTO daily_summary VALUES ('stock','TRV','us','2017-11-01',133.98,134.03,132.63,133.47,1749580,'0');
INSERT INTO daily_summary VALUES ('stock','TRVG','us','2017-11-01',7.23,7.55,7.23,7.4,1435040,'0');
INSERT INTO daily_summary VALUES ('stock','TRVN','us','2017-11-01',1.51,1.52,1.44,1.44,458848,'0');
INSERT INTO daily_summary VALUES ('stock','TRX','us','2017-11-01',0.265,0.28,0.265,0.2714,102847,'0');
INSERT INTO daily_summary VALUES ('stock','TRXC','us','2017-11-01',2.98,3.03,2.65,2.72,9364570,'0');
INSERT INTO daily_summary VALUES ('stock','TS','us','2017-11-01',27.75,27.82,27.34,27.37,1753220,'0');
INSERT INTO daily_summary VALUES ('stock','TSBK','us','2017-11-01',30.55,30.71,27.47,28.24,64408,'0');
INSERT INTO daily_summary VALUES ('stock','TSC','us','2017-11-01',22.85,22.9,22.15,22.4,61529,'0');
INSERT INTO daily_summary VALUES ('stock','TSCO','us','2017-11-01',60.4,60.605,59.18,59.28,1592850,'0');
INSERT INTO daily_summary VALUES ('stock','TSE','us','2017-11-01',71.65,71.75,70.65,71.3,478284,'0');
INSERT INTO daily_summary VALUES ('stock','TSEM','us','2017-11-01',33.27,33.3595,32.41,32.78,405868,'0');
INSERT INTO daily_summary VALUES ('stock','TSG','us','2017-11-01',20.1,20.2,19.95,20.15,156502,'0');
INSERT INTO daily_summary VALUES ('stock','TSI','us','2017-11-01',5.66,5.67,5.62,5.625,82248,'0');
INSERT INTO daily_summary VALUES ('stock','TSLA','us','2017-11-01',332.25,332.609,320.26,321.08,8457340,'0');
INSERT INTO daily_summary VALUES ('stock','TSLF','us','2017-11-01',17.28,17.28,17.09,17.17,23650,'0');
INSERT INTO daily_summary VALUES ('stock','TSLX','us','2017-11-01',20.57,20.6,20.2701,20.4,157162,'0');
INSERT INTO daily_summary VALUES ('stock','TSM','us','2017-11-01',42.48,42.5,41.86,42.1,4235660,'0');
INSERT INTO daily_summary VALUES ('stock','TSN','us','2017-11-01',73.16,73.66,73.07,73.31,1937900,'0');
INSERT INTO daily_summary VALUES ('stock','TSQ','us','2017-11-01',10.5,10.5,10.2,10.27,32019,'0');
INSERT INTO daily_summary VALUES ('stock','TSRI','us','2017-11-01',5,5,4.9,4.9,4370,'0');
INSERT INTO daily_summary VALUES ('stock','TSRO','us','2017-11-01',116.86,116.86,112.61,113.95,387939,'0');
INSERT INTO daily_summary VALUES ('stock','TSS','us','2017-11-01',72.46,72.619,71.98,72.14,1462530,'0');
INSERT INTO daily_summary VALUES ('stock','TST','us','2017-11-01',0.9025,0.95,0.9025,0.94,24172,'0');
INSERT INTO daily_summary VALUES ('stock','TSU','us','2017-11-01',18.62,18.72,18.29,18.36,459825,'0');
INSERT INTO daily_summary VALUES ('stock','TTAC','us','2017-11-01',32.12,32.12,31.965,31.965,1110,'0');
INSERT INTO daily_summary VALUES ('stock','TTAI','us','2017-11-01',26.69,26.7,26.69,26.7,500,'0');
INSERT INTO daily_summary VALUES ('stock','TTC','us','2017-11-01',63.17,63.4,62.94,63.14,569447,'0');
INSERT INTO daily_summary VALUES ('stock','TTD','us','2017-11-01',66.34,66.66,63.38,65.07,714030,'0');
INSERT INTO daily_summary VALUES ('stock','TTEC','us','2017-11-01',41.95,42.1,41,41.4,48262,'0');
INSERT INTO daily_summary VALUES ('stock','TTEK','us','2017-11-01',49.3,49.45,48.4,48.85,213961,'0');
INSERT INTO daily_summary VALUES ('stock','TTF','us','2017-11-01',10.82,10.875,10.82,10.87,14542,'0');
INSERT INTO daily_summary VALUES ('etf','TTFS','us','2017-11-01',67.8,68.11,67.79,68.1099,12494,'0');
INSERT INTO daily_summary VALUES ('stock','TTGT','us','2017-11-01',12.5,12.5,12.225,12.41,26867,'0');
INSERT INTO daily_summary VALUES ('stock','TTI','us','2017-11-01',2.91,3.02,2.84,2.92,647105,'0');
INSERT INTO daily_summary VALUES ('stock','TTM','us','2017-11-01',33.53,33.66,32.97,32.99,2126030,'0');
INSERT INTO daily_summary VALUES ('stock','TTMI','us','2017-11-01',15.97,15.98,15.64,15.8,1150060,'0');
INSERT INTO daily_summary VALUES ('stock','TTNP','us','2017-11-01',2.05,2.1,1.95,2.05,265978,'0');
INSERT INTO daily_summary VALUES ('stock','TTOO','us','2017-11-01',3.75,3.85,3.64,3.82,123365,'0');
INSERT INTO daily_summary VALUES ('stock','TTP','us','2017-11-01',17.9,18.2037,17.9,18.02,85453,'0');
INSERT INTO daily_summary VALUES ('stock','TTPH','us','2017-11-01',6.04,6.08,5.7826,5.94,491519,'0');
INSERT INTO daily_summary VALUES ('stock','TTS','us','2017-11-01',8.6,8.7,8.425,8.6,653920,'0');
INSERT INTO daily_summary VALUES ('etf','TTT','us','2017-11-01',26.33,26.35,25.82,26.08,37070,'0');
INSERT INTO daily_summary VALUES ('stock','TTWO','us','2017-11-01',110.91,111.11,108.83,110.56,1684440,'0');
INSERT INTO daily_summary VALUES ('stock','TU','us','2017-11-01',36.14,36.51,36.12,36.41,572529,'0');
INSERT INTO daily_summary VALUES ('stock','TUES','us','2017-11-01',3.2,3.2,2.95,3.15,521548,'0');
INSERT INTO daily_summary VALUES ('stock','TUP','us','2017-11-01',58.82,59.93,58.82,59.2,367549,'0');
INSERT INTO daily_summary VALUES ('etf','TUR','us','2017-11-01',42.3,42.8,42.24,42.48,300356,'0');
INSERT INTO daily_summary VALUES ('stock','TURN','us','2017-11-01',1.92,1.92,1.875,1.88,92148,'0');
INSERT INTO daily_summary VALUES ('stock','TUSK','us','2017-11-01',20.08,21.2212,19.51,19.84,259406,'0');
INSERT INTO daily_summary VALUES ('etf','TUZ','us','2017-11-01',50.48,50.505,50.48,50.49,1968,'0');
INSERT INTO daily_summary VALUES ('stock','TV','us','2017-11-01',22,22.02,21.534,21.62,5351120,'0');
INSERT INTO daily_summary VALUES ('stock','TVC','us','2017-11-01',25.39,25.4533,25.3,25.34,14529,'0');
INSERT INTO daily_summary VALUES ('stock','TVE','us','2017-11-01',25.11,25.11,25.04,25.046,16723,'0');
INSERT INTO daily_summary VALUES ('etf','TVIX','us','2017-11-01',8.19,8.629,8.14,8.46,15695900,'0');
INSERT INTO daily_summary VALUES ('etf','TVIZ','us','2017-11-01',10.56,10.82,10.53,10.82,20975,'0');
INSERT INTO daily_summary VALUES ('stock','TVPT','us','2017-11-01',15.77,15.97,15.63,15.97,1824620,'0');
INSERT INTO daily_summary VALUES ('stock','TVTY','us','2017-11-01',46.65,46.65,45.2,45.65,532257,'0');
INSERT INTO daily_summary VALUES ('stock','TWI','us','2017-11-01',9.89,10.04,9.64,9.76,252296,'0');
INSERT INTO daily_summary VALUES ('stock','TWIN','us','2017-11-01',21.66,22.83,21.3937,22.25,43797,'0');
INSERT INTO daily_summary VALUES ('stock','TWLO','us','2017-11-01',32.08,32.199,31.42,31.51,1147970,'0');
INSERT INTO daily_summary VALUES ('etf','TWM','us','2017-11-01',18.11,18.85,18.1,18.65,1135920,'0');
INSERT INTO daily_summary VALUES ('stock','TWMC','us','2017-11-01',1.8,1.8,1.65,1.725,5278,'0');
INSERT INTO daily_summary VALUES ('stock','TWN','us','2017-11-01',21.62,21.64,21.55,21.64,10610,'0');
INSERT INTO daily_summary VALUES ('stock','TWNK','us','2017-11-01',11.64,11.64,11.33,11.36,1152710,'0');
INSERT INTO daily_summary VALUES ('stock','TWNKW','us','2017-11-01',1.37,1.38,1.3,1.34,21777,'0');
INSERT INTO daily_summary VALUES ('stock','TWO','us','2017-11-01',16.077,16.192,16.045,16.11,2032940,'0');
INSERT INTO daily_summary VALUES ('stock','TWOU','us','2017-11-01',64.23,64.23,61.6101,61.78,322077,'0');
INSERT INTO daily_summary VALUES ('stock','TWOW','us','2017-11-01',16.16,16.21,16.16,16.21,3400,'0');
INSERT INTO daily_summary VALUES ('stock','TWO_A','us','2017-11-01',27.7,27.76,27.57,27.64,12015,'0');
INSERT INTO daily_summary VALUES ('stock','TWO_B','us','2017-11-01',26.61,26.63,26.46,26.5,55353,'0');
INSERT INTO daily_summary VALUES ('stock','TWTR','us','2017-11-01',20.74,20.99,20.4,20.61,18823000,'0');
INSERT INTO daily_summary VALUES ('stock','TWX','us','2017-11-01',98.62,99.1384,97.78,98.39,7608360,'0');
INSERT INTO daily_summary VALUES ('stock','TX','us','2017-11-01',31.31,31.41,28.3,29.93,1553520,'0');
INSERT INTO daily_summary VALUES ('stock','TXMD','us','2017-11-01',4.74,4.78,4.48,4.62,2515790,'0');
INSERT INTO daily_summary VALUES ('stock','TXN','us','2017-11-01',96.97,97.25,95.77,96.35,3873750,'0');
INSERT INTO daily_summary VALUES ('stock','TXRH','us','2017-11-01',50.57,50.6039,49.2,49.6,1166910,'0');
INSERT INTO daily_summary VALUES ('stock','TXT','us','2017-11-01',53.01,53.55,52.96,53.4,1508850,'0');
INSERT INTO daily_summary VALUES ('stock','TY','us','2017-11-01',26,26.08,25.8401,25.96,108961,'0');
INSERT INTO daily_summary VALUES ('etf','TYD','us','2017-11-01',44.326,44.52,44.326,44.39,534,'0');
INSERT INTO daily_summary VALUES ('stock','TYG','us','2017-11-01',26.21,26.79,26.21,26.68,231000,'0');
INSERT INTO daily_summary VALUES ('stock','TYHT','us','2017-11-01',3.42,4.15,3.42,3.8,289690,'0');
INSERT INTO daily_summary VALUES ('stock','TYL','us','2017-11-01',177.31,177.31,173.92,175.72,173556,'0');
INSERT INTO daily_summary VALUES ('stock','TYME','us','2017-11-01',4.06,4.1,3.701,3.81,61374,'0');
INSERT INTO daily_summary VALUES ('etf','TYO','us','2017-11-01',13.98,13.981,13.954,13.954,1689,'0');
INSERT INTO daily_summary VALUES ('stock','TYPE','us','2017-11-01',23.35,23.5,22.8,23.3,316622,'0');
INSERT INTO daily_summary VALUES ('stock','TY_','us','2017-11-01',50.4,50.5006,50,50.3,2116,'0');
INSERT INTO daily_summary VALUES ('etf','TZA','us','2017-11-01',13,13.79,12.98,13.56,11951600,'0');
INSERT INTO daily_summary VALUES ('stock','TZOO','us','2017-11-01',6.7,6.849,6.65,6.65,10189,'0');
INSERT INTO daily_summary VALUES ('stock','U','us','2017-11-01',10.38,10.38,10.38,10.38,100,'0');
INSERT INTO daily_summary VALUES ('stock','UA','us','2017-11-01',11.46,11.82,10.91,11.29,21607700,'0');
INSERT INTO daily_summary VALUES ('stock','UAA','us','2017-11-01',12.47,12.9475,11.98,12.05,27888600,'0');
INSERT INTO daily_summary VALUES ('etf','UAE','us','2017-11-01',17.8,17.8,17.7,17.72,7979,'0');
INSERT INTO daily_summary VALUES ('stock','UAL','us','2017-11-01',58.2,58.935,58.11,58.64,4798400,'0');
INSERT INTO daily_summary VALUES ('stock','UAMY','us','2017-11-01',0.22,0.239,0.22,0.2311,39897,'0');
INSERT INTO daily_summary VALUES ('stock','UAN','us','2017-11-01',3.39,3.485,3.16,3.45,505571,'0');
INSERT INTO daily_summary VALUES ('stock','UBA','us','2017-11-01',21.8,22.05,21.64,21.76,56865,'0');
INSERT INTO daily_summary VALUES ('stock','UBCP','us','2017-11-01',12.7,12.775,12.7,12.7,1309,'0');
INSERT INTO daily_summary VALUES ('stock','UBFO','us','2017-11-01',9.5,9.5,9.3,9.35,4122,'0');
INSERT INTO daily_summary VALUES ('stock','UBIO','us','2017-11-01',32.89,32.93,31.35,31.46,90109,'0');
INSERT INTO daily_summary VALUES ('etf','UBM','us','2017-11-01',17.84,17.84,17.734,17.734,285,'0');
INSERT INTO daily_summary VALUES ('stock','UBNK','us','2017-11-01',18.44,18.5,18.13,18.24,56452,'0');
INSERT INTO daily_summary VALUES ('stock','UBNT','us','2017-11-01',62.9,62.9,61.65,62.37,169882,'0');
INSERT INTO daily_summary VALUES ('stock','UBP_G','us','2017-11-01',26.17,26.1907,26.17,26.17,1400,'0');
INSERT INTO daily_summary VALUES ('stock','UBP_H','us','2017-11-01',26.65,26.65,26.59,26.59,2553,'0');
INSERT INTO daily_summary VALUES ('etf','UBR','us','2017-11-01',77.19,78.1453,76.25,76.25,2219,'0');
INSERT INTO daily_summary VALUES ('stock','UBRT','us','2017-11-01',137.59,137.62,128.66,131.03,1400,'0');
INSERT INTO daily_summary VALUES ('stock','UBS','us','2017-11-01',16.92,16.9664,16.86,16.9,1314000,'0');
INSERT INTO daily_summary VALUES ('stock','UBSH','us','2017-11-01',34.573,34.723,33.788,33.898,103599,'0');
INSERT INTO daily_summary VALUES ('stock','UBSI','us','2017-11-01',36.15,36.375,35.35,35.7,613970,'0');
INSERT INTO daily_summary VALUES ('etf','UBT','us','2017-11-01',80.35,81.22,80.35,80.8,15350,'0');
INSERT INTO daily_summary VALUES ('stock','UCBI','us','2017-11-01',27.74,27.8,26.76,27.03,375283,'0');
INSERT INTO daily_summary VALUES ('etf','UCC','us','2017-11-01',68.65,68.91,68.65,68.82,4300,'0');
INSERT INTO daily_summary VALUES ('stock','UCFC','us','2017-11-01',9.26,9.4193,9.1206,9.26,68879,'0');
INSERT INTO daily_summary VALUES ('etf','UCI','us','2017-11-01',14.926,14.926,14.77,14.8247,8667,'0');
INSERT INTO daily_summary VALUES ('etf','UCO','us','2017-11-01',19.85,19.95,19.11,19.39,4000360,'0');
INSERT INTO daily_summary VALUES ('stock','UCTT','us','2017-11-01',25.7,25.93,23.57,24.39,2404940,'0');
INSERT INTO daily_summary VALUES ('etf','UDN','us','2017-11-01',21.89,21.89,21.845,21.86,5129,'0');
INSERT INTO daily_summary VALUES ('etf','UDOW','us','2017-11-01',80.5,80.6,79.3,79.84,542894,'0');
INSERT INTO daily_summary VALUES ('stock','UDR','us','2017-11-01',38.86,39.18,38.72,39.1,1558390,'0');
INSERT INTO daily_summary VALUES ('stock','UE','us','2017-11-01',23.66,23.78,23.36,23.75,569768,'0');
INSERT INTO daily_summary VALUES ('stock','UEC','us','2017-11-01',1.02,1.1,0.96,1.09,3951850,'0');
INSERT INTO daily_summary VALUES ('stock','UEIC','us','2017-11-01',60.5,62.95,60.05,61.25,117511,'0');
INSERT INTO daily_summary VALUES ('stock','UEPS','us','2017-11-01',9.13,9.19,9.05,9.14,71123,'0');
INSERT INTO daily_summary VALUES ('stock','UEVM','us','2017-11-01',50.55,50.55,50.34,50.34,150504,'0');
INSERT INTO daily_summary VALUES ('stock','UFAB','us','2017-11-01',8.51,8.55,8.139,8.42,22313,'0');
INSERT INTO daily_summary VALUES ('stock','UFCS','us','2017-11-01',46.1,46.37,45.66,45.88,68547,'0');
INSERT INTO daily_summary VALUES ('stock','UFI','us','2017-11-01',38.28,38.54,37.13,37.13,44837,'0');
INSERT INTO daily_summary VALUES ('stock','UFPI','us','2017-11-01',113.7,113.7,111.6,113.3,120504,'0');
INSERT INTO daily_summary VALUES ('stock','UFPT','us','2017-11-01',31.15,31.15,29.5,30.1,15145,'0');
INSERT INTO daily_summary VALUES ('stock','UFS','us','2017-11-01',47.71,48.07,47.32,47.37,840908,'0');
INSERT INTO daily_summary VALUES ('stock','UG','us','2017-11-01',18.5,18.65,18.2413,18.65,4072,'0');
INSERT INTO daily_summary VALUES ('etf','UGA','us','2017-11-01',31.62,31.82,31.16,31.38,39828,'0');
INSERT INTO daily_summary VALUES ('etf','UGAZ','us','2017-11-01',8.08,8.505,8.08,8.5,14581200,'0');
INSERT INTO daily_summary VALUES ('etf','UGE','us','2017-11-01',44.27,44.34,44.24,44.24,358,'0');
INSERT INTO daily_summary VALUES ('stock','UGI','us','2017-11-01',48.13,48.1501,47.68,47.86,483307,'0');
INSERT INTO daily_summary VALUES ('etf','UGL','us','2017-11-01',39.16,39.42,39.015,39.1666,16418,'0');
INSERT INTO daily_summary VALUES ('etf','UGLD','us','2017-11-01',10.53,10.68,10.5038,10.56,502780,'0');
INSERT INTO daily_summary VALUES ('stock','UGP','us','2017-11-01',23.54,23.66,23.21,23.48,973756,'0');
INSERT INTO daily_summary VALUES ('stock','UHAL','us','2017-11-01',395.75,400.99,386.13,396.17,203594,'0');
INSERT INTO daily_summary VALUES ('etf','UHN','us','2017-11-01',17.91,17.91,17.61,17.649,8767,'0');
INSERT INTO daily_summary VALUES ('stock','UHS','us','2017-11-01',102.45,103.11,100.78,102.12,1338530,'0');
INSERT INTO daily_summary VALUES ('stock','UHT','us','2017-11-01',73.5,73.77,72.54,73.24,28277,'0');
INSERT INTO daily_summary VALUES ('stock','UIHC','us','2017-11-01',15.74,16.01,15.505,15.66,161405,'0');
INSERT INTO daily_summary VALUES ('stock','UIS','us','2017-11-01',8.75,8.75,7.95,8.3,887084,'0');
INSERT INTO daily_summary VALUES ('stock','UITB','us','2017-11-01',50.18,50.23,50.18,50.23,251374,'0');
INSERT INTO daily_summary VALUES ('stock','UIVM','us','2017-11-01',50.89,50.89,50.74,50.78,110210,'0');
INSERT INTO daily_summary VALUES ('stock','UL','us','2017-11-01',56.127,56.226,55.73,55.76,1149770,'0');
INSERT INTO daily_summary VALUES ('stock','ULBI','us','2017-11-01',7.45,7.65,6.95,7.35,21680,'0');
INSERT INTO daily_summary VALUES ('etf','ULE','us','2017-11-01',16.44,16.52,16.44,16.49,9245,'0');
INSERT INTO daily_summary VALUES ('stock','ULH','us','2017-11-01',21.781,21.831,21.233,21.382,7252,'0');
INSERT INTO daily_summary VALUES ('etf','ULST','us','2017-11-01',40.2776,40.29,40.21,40.2293,24098,'0');
INSERT INTO daily_summary VALUES ('stock','ULTA','us','2017-11-01',203.79,212.35,202.14,204.83,1350860,'0');
INSERT INTO daily_summary VALUES ('stock','ULTI','us','2017-11-01',203.45,205.6,200.04,203.12,893454,'0');
INSERT INTO daily_summary VALUES ('stock','ULVM','us','2017-11-01',50.48,50.48,50.18,50.18,100387,'0');
INSERT INTO daily_summary VALUES ('stock','UMBF','us','2017-11-01',74.3,74.36,72.06,72.6,197685,'0');
INSERT INTO daily_summary VALUES ('stock','UMC','us','2017-11-01',2.59,2.59,2.565,2.58,299970,'0');
INSERT INTO daily_summary VALUES ('etf','UMDD','us','2017-11-01',105.65,105.65,101.91,102.71,5891,'0');
INSERT INTO daily_summary VALUES ('stock','UMH','us','2017-11-01',15.04,15.14,14.87,14.93,99399,'0');
INSERT INTO daily_summary VALUES ('stock','UMH_B','us','2017-11-01',28.09,28.09,28.02,28.02,830,'0');
INSERT INTO daily_summary VALUES ('stock','UMH_C','us','2017-11-01',27.75,27.8065,27.66,27.8065,2135,'0');
INSERT INTO daily_summary VALUES ('stock','UMPQ','us','2017-11-01',20.65,20.7,20.26,20.4,810447,'0');
INSERT INTO daily_summary VALUES ('stock','UN','us','2017-11-01',57.527,57.557,57.14,57.14,980585,'0');
INSERT INTO daily_summary VALUES ('stock','UNAM','us','2017-11-01',9.7434,9.7434,9.7434,9.7434,459,'0');
INSERT INTO daily_summary VALUES ('stock','UNB','us','2017-11-01',49.2,50,49.15,49.7,4071,'0');
INSERT INTO daily_summary VALUES ('stock','UNF','us','2017-11-01',158.3,158.9,154.95,155.5,44666,'0');
INSERT INTO daily_summary VALUES ('stock','UNFI','us','2017-11-01',38.9,39.98,38.545,38.95,664152,'0');
INSERT INTO daily_summary VALUES ('etf','UNG','us','2017-11-01',5.87,5.97,5.86,5.97,9513930,'0');
INSERT INTO daily_summary VALUES ('stock','UNH','us','2017-11-01',211.62,212.63,208.92,209.53,1902850,'0');
INSERT INTO daily_summary VALUES ('stock','UNIT','us','2017-11-01',17.57,17.57,17.02,17.02,4185600,'0');
INSERT INTO daily_summary VALUES ('etf','UNL','us','2017-11-01',9.45,9.59,9.45,9.56,16410,'0');
INSERT INTO daily_summary VALUES ('stock','UNM','us','2017-11-01',52.29,52.79,51.97,52,1506220,'0');
INSERT INTO daily_summary VALUES ('stock','UNP','us','2017-11-01',116.59,117.24,116.09,117,2787080,'0');
INSERT INTO daily_summary VALUES ('stock','UNT','us','2017-11-01',19.15,19.73,18.58,18.99,491668,'0');
INSERT INTO daily_summary VALUES ('stock','UNTY','us','2017-11-01',19.9,19.95,19.75,19.8,7767,'0');
INSERT INTO daily_summary VALUES ('stock','UNVR','us','2017-11-01',29.98,30.19,29.78,30.13,849430,'0');
INSERT INTO daily_summary VALUES ('stock','UONE','us','2017-11-01',1.7,1.7,1.5,1.5,620,'0');
INSERT INTO daily_summary VALUES ('stock','UONEK','us','2017-11-01',1.7,1.75,1.6,1.6,4867,'0');
INSERT INTO daily_summary VALUES ('stock','UPL','us','2017-11-01',7.94,8.1201,7.77,7.89,1403200,'0');
INSERT INTO daily_summary VALUES ('stock','UPLD','us','2017-11-01',21.98,21.98,21.25,21.47,121042,'0');
INSERT INTO daily_summary VALUES ('etf','UPRO','us','2017-11-01',125.76,126.34,124.39,125.07,1184690,'0');
INSERT INTO daily_summary VALUES ('stock','UPS','us','2017-11-01',117.07,117.3,116.32,116.59,2103600,'0');
INSERT INTO daily_summary VALUES ('etf','UPV','us','2017-11-01',57.5,57.5,56.92,56.92,704,'0');
INSERT INTO daily_summary VALUES ('etf','UPW','us','2017-11-01',51.14,51.14,50.39,50.56,4225,'0');
INSERT INTO daily_summary VALUES ('stock','UQM','us','2017-11-01',1.1,1.1399,1.0997,1.1,84859,'0');
INSERT INTO daily_summary VALUES ('etf','URA','us','2017-11-01',11.95,12.1,11.92,12.08,154503,'0');
INSERT INTO daily_summary VALUES ('stock','URBN','us','2017-11-01',24.45,24.9825,24.33,24.94,2032160,'0');
INSERT INTO daily_summary VALUES ('etf','URE','us','2017-11-01',63.75,64.11,63.46,63.62,21852,'0');
INSERT INTO daily_summary VALUES ('stock','URG','us','2017-11-01',0.52,0.5355,0.52,0.5346,102129,'0');
INSERT INTO daily_summary VALUES ('stock','URGN','us','2017-11-01',31.7,32.22,29.72,32.08,136159,'0');
INSERT INTO daily_summary VALUES ('stock','URI','us','2017-11-01',143.339,144.59,142.6,143.67,1147430,'0');
INSERT INTO daily_summary VALUES ('etf','URR','us','2017-11-01',18.03,18.12,18.03,18.12,1700,'0');
INSERT INTO daily_summary VALUES ('etf','URTH','us','2017-11-01',86.24,86.25,85.89,85.89,5723,'0');
INSERT INTO daily_summary VALUES ('etf','URTY','us','2017-11-01',79.52,79.52,74.74,76.14,170143,'0');
INSERT INTO daily_summary VALUES ('stock','USA','us','2017-11-01',6.17,6.19,6.15,6.18,484941,'0');
INSERT INTO daily_summary VALUES ('stock','USAC','us','2017-11-01',15.74,16.419,15.6891,16.26,74338,'0');
INSERT INTO daily_summary VALUES ('stock','USAK','us','2017-11-01',14.19,14.89,14.18,14.39,86794,'0');
INSERT INTO daily_summary VALUES ('stock','USAP','us','2017-11-01',20.51,21.15,19.78,20.99,11147,'0');
INSERT INTO daily_summary VALUES ('stock','USAS','us','2017-11-01',3.9101,4.03,3.7525,3.95,97582,'0');
INSERT INTO daily_summary VALUES ('stock','USAT','us','2017-11-01',6.4,6.6,6.3,6.35,576111,'0');
INSERT INTO daily_summary VALUES ('stock','USATP','us','2017-11-01',24.25,24.25,24.25,24.25,1100,'0');
INSERT INTO daily_summary VALUES ('stock','USAU','us','2017-11-01',1.22,1.25,1.15,1.16,193848,'0');
INSERT INTO daily_summary VALUES ('stock','USB','us','2017-11-01',54.58,54.74,54.26,54.48,4642630,'0');
INSERT INTO daily_summary VALUES ('stock','USB_H','us','2017-11-01',23.12,23.16,23.0801,23.15,15892,'0');
INSERT INTO daily_summary VALUES ('stock','USB_M','us','2017-11-01',28.74,28.87,28.721,28.84,45997,'0');
INSERT INTO daily_summary VALUES ('stock','USB_O','us','2017-11-01',25.51,25.644,25.51,25.6,5957,'0');
INSERT INTO daily_summary VALUES ('etf','USCI','us','2017-11-01',41.19,41.35,41.0276,41.14,30587,'0');
INSERT INTO daily_summary VALUES ('stock','USCR','us','2017-11-01',78.3,78.825,77,77.45,202177,'0');
INSERT INTO daily_summary VALUES ('etf','USD','us','2017-11-01',125.78,126.12,122.3,124.68,14847,'0');
INSERT INTO daily_summary VALUES ('stock','USDP','us','2017-11-01',10.805,10.902,10.805,10.878,42344,'0');
INSERT INTO daily_summary VALUES ('etf','USDU','us','2017-11-01',26.33,26.3401,26.3,26.3,8332,'0');
INSERT INTO daily_summary VALUES ('stock','USEG','us','2017-11-01',1.18,1.18,1.08,1.1192,114992,'0');
INSERT INTO daily_summary VALUES ('stock','USEQ','us','2017-11-01',26.47,26.48,26.46,26.47,32080,'0');
INSERT INTO daily_summary VALUES ('stock','USFD','us','2017-11-01',27.38,27.82,27.285,27.49,4008880,'0');
INSERT INTO daily_summary VALUES ('stock','USG','us','2017-11-01',34.49,35.02,34.285,34.45,1123700,'0');
INSERT INTO daily_summary VALUES ('stock','USHY','us','2017-11-01',50.18,50.18,50.16,50.166,429,'0');
INSERT INTO daily_summary VALUES ('etf','USL','us','2017-11-01',19.23,19.23,18.87,18.98,78419,'0');
INSERT INTO daily_summary VALUES ('stock','USLB','us','2017-11-01',29.78,29.785,29.74,29.74,14010,'0');
INSERT INTO daily_summary VALUES ('stock','USLM','us','2017-11-01',90.76,90.94,88.425,89.07,21688,'0');
INSERT INTO daily_summary VALUES ('etf','USLV','us','2017-11-01',11.8,12.44,11.7776,12.34,2832810,'0');
INSERT INTO daily_summary VALUES ('stock','USM','us','2017-11-01',36.75,36.97,35.82,36.18,75571,'0');
INSERT INTO daily_summary VALUES ('stock','USMC','us','2017-11-01',25.2017,25.21,25.19,25.19,1427,'0');
INSERT INTO daily_summary VALUES ('stock','USMF','us','2017-11-01',26.48,26.48,26.48,26.48,103,'0');
INSERT INTO daily_summary VALUES ('etf','USMV','us','2017-11-01',51.49,51.49,51.27,51.27,987393,'0');
INSERT INTO daily_summary VALUES ('stock','USNA','us','2017-11-01',65.75,65.85,64.95,65.3,80862,'0');
INSERT INTO daily_summary VALUES ('etf','USO','us','2017-11-01',11.03,11.06,10.82,10.9,21152000,'0');
INSERT INTO daily_summary VALUES ('stock','USOD','us','2017-11-01',15.64,16.08,15.64,16.08,1205,'0');
INSERT INTO daily_summary VALUES ('stock','USOI','us','2017-11-01',25.33,25.34,25.0559,25.0559,3656,'0');
INSERT INTO daily_summary VALUES ('stock','USOU','us','2017-11-01',35.12,35.12,33.16,33.61,3636,'0');
INSERT INTO daily_summary VALUES ('stock','USPH','us','2017-11-01',68.55,68.55,67.055,68.05,76922,'0');
INSERT INTO daily_summary VALUES ('stock','USRT','us','2017-11-01',48.94,49.18,48.77,48.93,200381,'0');
INSERT INTO daily_summary VALUES ('etf','UST','us','2017-11-01',58.02,58.02,57.87,57.87,772,'0');
INSERT INTO daily_summary VALUES ('stock','USTB','us','2017-11-01',50.16,50.16,50.16,50.16,250000,'0');
INSERT INTO daily_summary VALUES ('stock','USVM','us','2017-11-01',50.32,50.37,49.81,49.81,50562,'0');
INSERT INTO daily_summary VALUES ('stock','UTES','us','2017-11-01',33.55,33.57,33.55,33.57,535,'0');
INSERT INTO daily_summary VALUES ('stock','UTF','us','2017-11-01',23.31,23.31,23.01,23.07,162692,'0');
INSERT INTO daily_summary VALUES ('stock','UTG','us','2017-11-01',31.03,31.1124,30.82,30.99,287952,'0');
INSERT INTO daily_summary VALUES ('stock','UTHR','us','2017-11-01',119.09,121.07,118.495,120.21,469151,'0');
INSERT INTO daily_summary VALUES ('stock','UTI','us','2017-11-01',3.35,3.41,3.285,3.37,24103,'0');
INSERT INTO daily_summary VALUES ('stock','UTL','us','2017-11-01',52.36,52.36,51.5,51.66,30088,'0');
INSERT INTO daily_summary VALUES ('stock','UTLF','us','2017-11-01',27.83,27.83,27.83,27.83,0,'0');
INSERT INTO daily_summary VALUES ('stock','UTMD','us','2017-11-01',75.35,77.05,75.05,75.3,8823,'0');
INSERT INTO daily_summary VALUES ('stock','UTSI','us','2017-11-01',3.17,3.21,3.15,3.2,38270,'0');
INSERT INTO daily_summary VALUES ('stock','UTSL','us','2017-11-01',30.71,30.8,30.68,30.68,698,'0');
INSERT INTO daily_summary VALUES ('stock','UTX','us','2017-11-01',120.55,121,119.93,120.12,2690620,'0');
INSERT INTO daily_summary VALUES ('etf','UUP','us','2017-11-01',24.65,24.66,24.58,24.63,1327480,'0');
INSERT INTO daily_summary VALUES ('stock','UUU','us','2017-11-01',1.6,1.6,1.5429,1.5429,5541,'0');
INSERT INTO daily_summary VALUES ('stock','UUUU','us','2017-11-01',1.38,1.41,1.38,1.39,120183,'0');
INSERT INTO daily_summary VALUES ('stock','UVE','us','2017-11-01',24.05,24.3,23.5,23.65,146112,'0');
INSERT INTO daily_summary VALUES ('stock','UVSP','us','2017-11-01',29.45,29.7,28.775,29.05,65095,'0');
INSERT INTO daily_summary VALUES ('stock','UVV','us','2017-11-01',57.5,57.65,57,57.4,152813,'0');
INSERT INTO daily_summary VALUES ('etf','UVXY','us','2017-11-01',14.99,15.79,14.92,15.45,27828200,'0');
INSERT INTO daily_summary VALUES ('etf','UWM','us','2017-11-01',68.77,68.79,66.0748,66.81,362921,'0');
INSERT INTO daily_summary VALUES ('stock','UWN','us','2017-11-01',2.13,2.18,2.1,2.13,32143,'0');
INSERT INTO daily_summary VALUES ('stock','UWT','us','2017-11-01',19.31,19.48,18.26,18.66,5357170,'0');
INSERT INTO daily_summary VALUES ('etf','UXI','us','2017-11-01',66.87,67.54,66.8699,66.89,7987,'0');
INSERT INTO daily_summary VALUES ('etf','UYG','us','2017-11-01',118.86,119.553,118.06,118.26,29176,'0');
INSERT INTO daily_summary VALUES ('etf','UYM','us','2017-11-01',70.62,71.73,70.48,70.9699,11230,'0');
INSERT INTO daily_summary VALUES ('stock','UZA','us','2017-11-01',25.44,25.47,25.36,25.44,28182,'0');
INSERT INTO daily_summary VALUES ('stock','UZB','us','2017-11-01',26.75,26.75,26.554,26.56,2147,'0');
INSERT INTO daily_summary VALUES ('stock','UZC','us','2017-11-01',26.87,26.9528,26.78,26.831,14386,'0');
INSERT INTO daily_summary VALUES ('stock','V','us','2017-11-01',99.07,100.05,99.07,100.05,1733,'0');
INSERT INTO daily_summary VALUES ('stock','VAC','us','2017-11-01',132.06,132.73,130.43,131.27,378426,'0');
INSERT INTO daily_summary VALUES ('stock','VALE','us','2017-11-01',10.02,10.16,9.96,10.04,30209100,'0');
INSERT INTO daily_summary VALUES ('stock','VALX','us','2017-11-01',28.24,28.24,27.91,27.91,1939,'0');
INSERT INTO daily_summary VALUES ('stock','VAMO','us','2017-11-01',24.798,24.798,24.798,24.798,135,'0');
INSERT INTO daily_summary VALUES ('stock','VAR','us','2017-11-01',104.77,105.06,103.98,104.22,698367,'0');
INSERT INTO daily_summary VALUES ('etf','VAW','us','2017-11-01',134.16,134.67,133.495,133.93,69324,'0');
INSERT INTO daily_summary VALUES ('etf','VB','us','2017-11-01',144.34,144.41,142.45,142.95,500166,'0');
INSERT INTO daily_summary VALUES ('stock','VBF','us','2017-11-01',20.06,20.11,19.97,20.07,29422,'0');
INSERT INTO daily_summary VALUES ('stock','VBIV','us','2017-11-01',3.7,3.9,3.63,3.72,708641,'0');
INSERT INTO daily_summary VALUES ('etf','VBK','us','2017-11-01',157.18,157.18,154.83,155.5,99794,'0');
INSERT INTO daily_summary VALUES ('stock','VBLT','us','2017-11-01',6.2,6.25,6.05,6.1,114301,'0');
INSERT INTO daily_summary VALUES ('stock','VBND','us','2017-11-01',49.85,49.95,49.8356,49.91,12404,'0');
INSERT INTO daily_summary VALUES ('etf','VBR','us','2017-11-01',129.65,129.83,128.1,128.59,270609,'0');
INSERT INTO daily_summary VALUES ('stock','VBTX','us','2017-11-01',26.39,27,25.955,26.19,157978,'0');
INSERT INTO daily_summary VALUES ('stock','VC','us','2017-11-01',126.05,126.95,125.52,125.88,183626,'0');
INSERT INTO daily_summary VALUES ('stock','VCEL','us','2017-11-01',4.5,4.65,4.4,4.4,292150,'0');
INSERT INTO daily_summary VALUES ('stock','VCF','us','2017-11-01',14.76,14.899,14.73,14.889,13084,'0');
INSERT INTO daily_summary VALUES ('etf','VCIT','us','2017-11-01',87.67,87.86,87.61,87.73,721238,'0');
INSERT INTO daily_summary VALUES ('etf','VCLT','us','2017-11-01',94.21,94.5721,94.19,94.27,130435,'0');
INSERT INTO daily_summary VALUES ('stock','VCO','us','2017-11-01',34,35.19,33.6563,34.35,7014,'0');
INSERT INTO daily_summary VALUES ('etf','VCR','us','2017-11-01',146.19,146.52,145.57,145.68,58301,'0');
INSERT INTO daily_summary VALUES ('stock','VCRA','us','2017-11-01',28.39,28.39,25.72,26.65,376241,'0');
INSERT INTO daily_summary VALUES ('etf','VCSH','us','2017-11-01',79.82,79.89,79.81,79.83,1248320,'0');
INSERT INTO daily_summary VALUES ('stock','VCV','us','2017-11-01',12.85,12.89,12.81,12.86,53795,'0');
INSERT INTO daily_summary VALUES ('stock','VCYT','us','2017-11-01',8.57,8.67,8.3,8.49,80005,'0');
INSERT INTO daily_summary VALUES ('etf','VDC','us','2017-11-01',136.99,137.39,135.97,136.9,115844,'0');
INSERT INTO daily_summary VALUES ('etf','VDE','us','2017-11-01',93.17,94.23,93.17,93.77,306929,'0');
INSERT INTO daily_summary VALUES ('stock','VDSI','us','2017-11-01',13.6,13.65,13.25,13.45,188893,'0');
INSERT INTO daily_summary VALUES ('stock','VDTH','us','2017-11-01',8.65,8.81,8.561,8.66,26553,'0');
INSERT INTO daily_summary VALUES ('etf','VEA','us','2017-11-01',44.4,44.4765,44.28,44.28,6043340,'0');
INSERT INTO daily_summary VALUES ('stock','VEAC','us','2017-11-01',9.74,9.75,9.71,9.74,103275,'0');
INSERT INTO daily_summary VALUES ('stock','VEC','us','2017-11-01',30.84,31.59,30.4,30.57,36851,'0');
INSERT INTO daily_summary VALUES ('stock','VECO','us','2017-11-01',18.3,18.7,17.35,17.45,845937,'0');
INSERT INTO daily_summary VALUES ('stock','VEDL','us','2017-11-01',21.23,21.63,21.22,21.22,345977,'0');
INSERT INTO daily_summary VALUES ('stock','VEEV','us','2017-11-01',61.43,61.8,59.95,60.55,1119100,'0');
INSERT INTO daily_summary VALUES ('etf','VEGA','us','2017-11-01',30.28,30.28,30.28,30.28,1352,'0');
INSERT INTO daily_summary VALUES ('etf','VEGI','us','2017-11-01',28.8404,28.9684,28.8404,28.89,1040,'0');
INSERT INTO daily_summary VALUES ('stock','VEON','us','2017-11-01',3.93,3.96,3.89,3.93,4405020,'0');
INSERT INTO daily_summary VALUES ('stock','VER','us','2017-11-01',7.87,7.91,7.82,7.87,8887520,'0');
INSERT INTO daily_summary VALUES ('stock','VERI','us','2017-11-01',36.22,36.86,34.22,34.97,144707,'0');
INSERT INTO daily_summary VALUES ('stock','VERU','us','2017-11-01',1.35,1.35,1.25,1.29,86960,'0');
INSERT INTO daily_summary VALUES ('stock','VER_F','us','2017-11-01',25.58,25.62,25.5601,25.6,228381,'0');
INSERT INTO daily_summary VALUES ('stock','VET','us','2017-11-01',34.49,35.44,34.49,35.14,134136,'0');
INSERT INTO daily_summary VALUES ('etf','VEU','us','2017-11-01',54.01,54.13,53.87,53.87,1956820,'0');
INSERT INTO daily_summary VALUES ('stock','VFC','us','2017-11-01',69.67,70.4,68.98,70.15,2434440,'0');
INSERT INTO daily_summary VALUES ('etf','VFH','us','2017-11-01',67.32,67.63,67.0147,67.13,415302,'0');
INSERT INTO daily_summary VALUES ('stock','VFL','us','2017-11-01',13.31,13.336,13.3,13.336,6979,'0');
INSERT INTO daily_summary VALUES ('stock','VG','us','2017-11-01',8.22,8.22,8.02,8.05,1438310,'0');
INSERT INTO daily_summary VALUES ('stock','VGFO','us','2017-11-01',25.2759,25.289,25.2759,25.289,600,'0');
INSERT INTO daily_summary VALUES ('stock','VGI','us','2017-11-01',18.232,18.242,18.014,18.242,66577,'0');
INSERT INTO daily_summary VALUES ('etf','VGIT','us','2017-11-01',64.31,64.45,64.3,64.31,80504,'0');
INSERT INTO daily_summary VALUES ('etf','VGK','us','2017-11-01',58.78,58.841,58.54,58.54,3320200,'0');
INSERT INTO daily_summary VALUES ('etf','VGLT','us','2017-11-01',76.62,77.0508,76.57,76.83,135953,'0');
INSERT INTO daily_summary VALUES ('stock','VGM','us','2017-11-01',13.03,13.09,13.01,13.09,149701,'0');
INSERT INTO daily_summary VALUES ('stock','VGR','us','2017-11-01',20.84,21.03,20.77,20.86,434914,'0');
INSERT INTO daily_summary VALUES ('etf','VGSH','us','2017-11-01',60.55,60.55,60.5,60.52,246762,'0');
INSERT INTO daily_summary VALUES ('etf','VGT','us','2017-11-01',164,164.109,162.28,163.17,749965,'0');
INSERT INTO daily_summary VALUES ('stock','VGZ','us','2017-11-01',0.68,0.6999,0.6737,0.675,184931,'0');
INSERT INTO daily_summary VALUES ('stock','VHC','us','2017-11-01',5.2,5.2,4.85,4.95,396767,'0');
INSERT INTO daily_summary VALUES ('stock','VHI','us','2017-11-01',4.4,4.49,3.7701,4.24,612322,'0');
INSERT INTO daily_summary VALUES ('etf','VHT','us','2017-11-01',151.44,151.928,151.08,151.08,319351,'0');
INSERT INTO daily_summary VALUES ('stock','VIA','us','2017-11-01',30.1,31.05,30.1,30.65,33765,'0');
INSERT INTO daily_summary VALUES ('stock','VIAB','us','2017-11-01',24.27,24.95,24.17,24.78,5218800,'0');
INSERT INTO daily_summary VALUES ('stock','VIAV','us','2017-11-01',9.35,9.38,9.12,9.17,4618790,'0');
INSERT INTO daily_summary VALUES ('stock','VICL','us','2017-11-01',2.18,2.2099,2.15,2.16,149309,'0');
INSERT INTO daily_summary VALUES ('stock','VICR','us','2017-11-01',21.9,21.9,20.575,20.75,32299,'0');
INSERT INTO daily_summary VALUES ('etf','VIDI','us','2017-11-01',27.88,27.98,27.805,27.88,31357,'0');
INSERT INTO daily_summary VALUES ('etf','VIG','us','2017-11-01',96.97,97.1,96.7,96.75,388635,'0');
INSERT INTO daily_summary VALUES ('stock','VIGI','us','2017-11-01',64.76,64.809,64.47,64.54,56190,'0');
INSERT INTO daily_summary VALUES ('stock','VII','us','2017-11-01',0.408,0.408,0.4,0.4003,27190,'0');
INSERT INTO daily_summary VALUES ('etf','VIIX','us','2017-11-01',14.19,14.53,14.151,14.39,31127,'0');
INSERT INTO daily_summary VALUES ('etf','VIIZ','us','2017-11-01',6.56,6.57,6.56,6.57,921,'0');
INSERT INTO daily_summary VALUES ('etf','VIOG','us','2017-11-01',143.62,143.62,140.846,140.93,2931,'0');
INSERT INTO daily_summary VALUES ('etf','VIOO','us','2017-11-01',137.87,137.95,135.062,135.92,39466,'0');
INSERT INTO daily_summary VALUES ('etf','VIOV','us','2017-11-01',129.33,129.33,127.21,127.85,8906,'0');
INSERT INTO daily_summary VALUES ('stock','VIPS','us','2017-11-01',8.01,8.3407,8.01,8.24,5253950,'0');
INSERT INTO daily_summary VALUES ('stock','VIRC','us','2017-11-01',5.8,5.8792,5.55,5.7,13364,'0');
INSERT INTO daily_summary VALUES ('stock','VIRT','us','2017-11-01',14.1,14.475,14.05,14.2,492042,'0');
INSERT INTO daily_summary VALUES ('etf','VIS','us','2017-11-01',135.73,135.9,134.8,134.99,119192,'0');
INSERT INTO daily_summary VALUES ('stock','VISI','us','2017-11-01',4.2,4.355,3.9,4.15,212877,'0');
INSERT INTO daily_summary VALUES ('stock','VIV','us','2017-11-01',15.53,15.61,15.38,15.41,1142460,'0');
INSERT INTO daily_summary VALUES ('stock','VIVE','us','2017-11-01',5.48,5.58,5.4,5.41,135696,'0');
INSERT INTO daily_summary VALUES ('stock','VIVO','us','2017-11-01',15,15,14.6,14.7,150504,'0');
INSERT INTO daily_summary VALUES ('etf','VIXM','us','2017-11-01',22.92,23.21,22.87,23.2,165071,'0');
INSERT INTO daily_summary VALUES ('etf','VIXY','us','2017-11-01',27.9,28.59,27.81,28.31,957005,'0');
INSERT INTO daily_summary VALUES ('stock','VJET','us','2017-11-01',5.41,5.54,4.86,5.01,106285,'0');
INSERT INTO daily_summary VALUES ('stock','VKI','us','2017-11-01',11.23,11.3,11.23,11.26,61740,'0');
INSERT INTO daily_summary VALUES ('stock','VKQ','us','2017-11-01',12.32,12.39,12.32,12.35,133992,'0');
INSERT INTO daily_summary VALUES ('stock','VKTX','us','2017-11-01',2.38,2.46,2.3,2.44,277282,'0');
INSERT INTO daily_summary VALUES ('stock','VKTXW','us','2017-11-01',1.6499,1.6499,1.416,1.4688,29050,'0');
INSERT INTO daily_summary VALUES ('stock','VLGEA','us','2017-11-01',24,24.839,23.69,23.81,87384,'0');
INSERT INTO daily_summary VALUES ('stock','VLO','us','2017-11-01',79.32,80.93,79.07,80.71,4097770,'0');
INSERT INTO daily_summary VALUES ('stock','VLP','us','2017-11-01',43.06,43.56,42.53,42.64,119299,'0');
INSERT INTO daily_summary VALUES ('stock','VLRS','us','2017-11-01',10.21,10.38,10.21,10.35,411404,'0');
INSERT INTO daily_summary VALUES ('stock','VLRX','us','2017-11-01',2.174,2.4,2.174,2.38,4288,'0');
INSERT INTO daily_summary VALUES ('stock','VLT','us','2017-11-01',14.93,14.97,14.91,14.96,9020,'0');
INSERT INTO daily_summary VALUES ('etf','VLU','us','2017-11-01',98.93,98.93,97.8783,97.8783,420,'0');
INSERT INTO daily_summary VALUES ('etf','VLUE','us','2017-11-01',79.67,79.93,79.45,79.57,101871,'0');
INSERT INTO daily_summary VALUES ('stock','VLY','us','2017-11-01',11.59,11.67,11.395,11.49,2145230,'0');
INSERT INTO daily_summary VALUES ('stock','VLY_A','us','2017-11-01',27.61,27.61,27.53,27.61,3784,'0');
INSERT INTO daily_summary VALUES ('stock','VLY_B','us','2017-11-01',26.45,26.45,26.45,26.45,457,'0');
INSERT INTO daily_summary VALUES ('stock','VMAX','us','2017-11-01',14.91,15.09,14.9,15.03,7372,'0');
INSERT INTO daily_summary VALUES ('etf','VMBS','us','2017-11-01',52.59,52.65,52.57,52.63,362544,'0');
INSERT INTO daily_summary VALUES ('stock','VMC','us','2017-11-01',122.55,122.7,120.08,121.02,1788490,'0');
INSERT INTO daily_summary VALUES ('stock','VMI','us','2017-11-01',160.15,160.3,157.8,157.9,71845,'0');
INSERT INTO daily_summary VALUES ('stock','VMIN','us','2017-11-01',38.99,38.99,37.632,38.1429,5774,'0');
INSERT INTO daily_summary VALUES ('stock','VMM','us','2017-11-01',13.74,13.88,13.68,13.83,20828,'0');
INSERT INTO daily_summary VALUES ('stock','VMO','us','2017-11-01',12.55,12.55,12.5,12.54,117166,'0');
INSERT INTO daily_summary VALUES ('stock','VMOT','us','2017-11-01',28.034,28.089,27.98,27.98,5663,'0');
INSERT INTO daily_summary VALUES ('stock','VMW','us','2017-11-01',120,120.31,118.66,119.12,1565450,'0');
INSERT INTO daily_summary VALUES ('stock','VNCE','us','2017-11-01',3.61,3.779,3.61,3.65,24386,'0');
INSERT INTO daily_summary VALUES ('stock','VNDA','us','2017-11-01',15.7,15.75,15.25,15.4,293163,'0');
INSERT INTO daily_summary VALUES ('stock','VNET','us','2017-11-01',7.55,7.68,7.44,7.51,316467,'0');
INSERT INTO daily_summary VALUES ('stock','VNLA','us','2017-11-01',50.29,50.3,50.28,50.29,20866,'0');
INSERT INTO daily_summary VALUES ('etf','VNM','us','2017-11-01',16.02,16.42,16,16.36,408960,'0');
INSERT INTO daily_summary VALUES ('stock','VNO','us','2017-11-01',74.283,74.392,73.648,73.995,1131510,'0');
INSERT INTO daily_summary VALUES ('stock','VNOM','us','2017-11-01',19.335,19.714,19.335,19.561,192499,'0');
INSERT INTO daily_summary VALUES ('stock','VNO_G','us','2017-11-01',25.542,25.56,25.542,25.56,5737,'0');
INSERT INTO daily_summary VALUES ('stock','VNO_I','us','2017-11-01',25.55,25.55,25.5182,25.52,5203,'0');
INSERT INTO daily_summary VALUES ('stock','VNO_K','us','2017-11-01',25.35,25.46,25.35,25.45,6167,'0');
INSERT INTO daily_summary VALUES ('stock','VNO_L','us','2017-11-01',25.3,25.37,25.2275,25.37,8668,'0');
INSERT INTO daily_summary VALUES ('etf','VNQ','us','2017-11-01',82.34,82.78,82.15,82.6,3507720,'0');
INSERT INTO daily_summary VALUES ('etf','VNQI','us','2017-11-01',59.73,59.9,59.6,59.6,309639,'0');
INSERT INTO daily_summary VALUES ('stock','VNRX','us','2017-11-01',2.85,2.92,2.6443,2.92,73650,'0');
INSERT INTO daily_summary VALUES ('stock','VNTR','us','2017-11-01',25.16,25.23,24.71,24.86,640718,'0');
INSERT INTO daily_summary VALUES ('stock','VNTV','us','2017-11-01',70.05,70.65,69.38,69.85,2781660,'0');
INSERT INTO daily_summary VALUES ('etf','VO','us','2017-11-01',149.89,149.94,148.75,148.95,324173,'0');
INSERT INTO daily_summary VALUES ('stock','VOC','us','2017-11-01',3.69,3.76,3.54,3.69,123011,'0');
INSERT INTO daily_summary VALUES ('stock','VOD','us','2017-11-01',29.39,29.52,29.3,29.32,3114740,'0');
INSERT INTO daily_summary VALUES ('etf','VOE','us','2017-11-01',107.11,107.309,106.5,106.57,207981,'0');
INSERT INTO daily_summary VALUES ('etf','VONE','us','2017-11-01',118.51,118.75,118.08,118.22,26778,'0');
INSERT INTO daily_summary VALUES ('etf','VONG','us','2017-11-01',133.88,133.88,132.97,133.15,40365,'0');
INSERT INTO daily_summary VALUES ('etf','VONV','us','2017-11-01',104.84,105.08,104.62,104.71,29142,'0');
INSERT INTO daily_summary VALUES ('etf','VOO','us','2017-11-01',237.03,237.39,236.13,236.54,2077840,'0');
INSERT INTO daily_summary VALUES ('etf','VOOG','us','2017-11-01',133.42,133.43,132.742,132.99,38949,'0');
INSERT INTO daily_summary VALUES ('etf','VOOV','us','2017-11-01',105.61,105.83,105.38,105.53,11732,'0');
INSERT INTO daily_summary VALUES ('etf','VOT','us','2017-11-01',124.99,125.106,123.72,123.94,99416,'0');
INSERT INTO daily_summary VALUES ('etf','VOX','us','2017-11-01',90.09,90.09,88.3,88.72,264534,'0');
INSERT INTO daily_summary VALUES ('stock','VOXX','us','2017-11-01',6.75,6.75,6.5,6.5,115308,'0');
INSERT INTO daily_summary VALUES ('stock','VOYA','us','2017-11-01',40.98,42.43,40.75,41.3,3489110,'0');
INSERT INTO daily_summary VALUES ('stock','VPG','us','2017-11-01',24.85,24.9,24.05,24.5,23004,'0');
INSERT INTO daily_summary VALUES ('etf','VPL','us','2017-11-01',71.77,71.91,71.66,71.74,375568,'0');
INSERT INTO daily_summary VALUES ('etf','VPU','us','2017-11-01',121.63,121.63,120.6,120.73,111173,'0');
INSERT INTO daily_summary VALUES ('stock','VPV','us','2017-11-01',12.11,12.11,12.035,12.07,100443,'0');
INSERT INTO daily_summary VALUES ('etf','VQT','us','2017-11-01',152.55,152.785,152.55,152.71,930,'0');
INSERT INTO daily_summary VALUES ('stock','VR','us','2017-11-01',52.04,52.3,51.69,51.7,953173,'0');
INSERT INTO daily_summary VALUES ('stock','VRA','us','2017-11-01',7.23,7.23,7.03,7.1,309379,'0');
INSERT INTO daily_summary VALUES ('stock','VRAY','us','2017-11-01',6.92,7.4898,6.92,7.29,1162440,'0');
INSERT INTO daily_summary VALUES ('stock','VREX','us','2017-11-01',34.11,34.4699,33.62,34.03,174538,'0');
INSERT INTO daily_summary VALUES ('stock','VRIG','us','2017-11-01',25.23,25.23,25.2,25.22,17811,'0');
INSERT INTO daily_summary VALUES ('stock','VRML','us','2017-11-01',1.51,1.97,1.51,1.81,1165580,'0');
INSERT INTO daily_summary VALUES ('stock','VRNA','us','2017-11-01',13.69,13.75,13.42,13.75,6890,'0');
INSERT INTO daily_summary VALUES ('stock','VRNS','us','2017-11-01',43.65,44.05,42.95,43.05,332740,'0');
INSERT INTO daily_summary VALUES ('stock','VRNT','us','2017-11-01',42.55,42.65,41.45,41.75,348938,'0');
INSERT INTO daily_summary VALUES ('etf','VRP','us','2017-11-01',25.94,25.95,25.8745,25.95,346885,'0');
INSERT INTO daily_summary VALUES ('stock','VRS','us','2017-11-01',7.09,7.19,7.04,7.15,171814,'0');
INSERT INTO daily_summary VALUES ('stock','VRSK','us','2017-11-01',86,91.34,86,89.88,2413470,'0');
INSERT INTO daily_summary VALUES ('stock','VRSN','us','2017-11-01',108.14,109.68,107.6,108.86,794625,'0');
INSERT INTO daily_summary VALUES ('stock','VRTS','us','2017-11-01',116.95,117.75,114.4,114.4,41802,'0');
INSERT INTO daily_summary VALUES ('stock','VRTSP','us','2017-11-01',104.08,104.08,101.874,102,3467,'0');
INSERT INTO daily_summary VALUES ('stock','VRTU','us','2017-11-01',38.25,38.25,37.14,37.5,87989,'0');
INSERT INTO daily_summary VALUES ('stock','VRTV','us','2017-11-01',32.45,32.5,31.9,32.2,72653,'0');
INSERT INTO daily_summary VALUES ('stock','VRTX','us','2017-11-01',147.35,147.35,142.39,142.63,2412600,'0');
INSERT INTO daily_summary VALUES ('stock','VRX','us','2017-11-01',11.68,12.02,11.54,11.93,9044970,'0');
INSERT INTO daily_summary VALUES ('stock','VR_A','us','2017-11-01',25.58,25.64,25.58,25.639,7625,'0');
INSERT INTO daily_summary VALUES ('stock','VR_B','us','2017-11-01',25.34,25.38,25.24,25.26,26333,'0');
INSERT INTO daily_summary VALUES ('stock','VSAR','us','2017-11-01',1.8,1.85,1.75,1.8,828617,'0');
INSERT INTO daily_summary VALUES ('stock','VSAT','us','2017-11-01',65.4,65.49,64.8,65.05,604298,'0');
INSERT INTO daily_summary VALUES ('stock','VSEC','us','2017-11-01',48.929,49.129,47.931,48.55,56591,'0');
INSERT INTO daily_summary VALUES ('stock','VSH','us','2017-11-01',22.5,22.55,21.45,21.65,2909140,'0');
INSERT INTO daily_summary VALUES ('stock','VSI','us','2017-11-01',4.7,4.7,4.5,4.55,459837,'0');
INSERT INTO daily_summary VALUES ('stock','VSLR','us','2017-11-01',3.7,3.75,3.45,3.45,512078,'0');
INSERT INTO daily_summary VALUES ('stock','VSM','us','2017-11-01',42.23,42.5,40.89,41.17,540853,'0');
INSERT INTO daily_summary VALUES ('stock','VSMV','us','2017-11-01',26.19,26.19,26.19,26.19,101,'0');
INSERT INTO daily_summary VALUES ('etf','VSS','us','2017-11-01',117.2,117.36,116.72,116.88,153616,'0');
INSERT INTO daily_summary VALUES ('stock','VST','us','2017-11-01',19.57,19.65,19.06,19.21,5730610,'0');
INSERT INTO daily_summary VALUES ('stock','VSTM','us','2017-11-01',3.82,4,3.47,3.47,728205,'0');
INSERT INTO daily_summary VALUES ('stock','VSTO','us','2017-11-01',20.86,20.89,20.29,20.55,931107,'0');
INSERT INTO daily_summary VALUES ('etf','VT','us','2017-11-01',72.5,72.64,72.296,72.37,865252,'0');
INSERT INTO daily_summary VALUES ('stock','VTA','us','2017-11-01',11.89,11.9,11.71,11.78,503655,'0');
INSERT INTO daily_summary VALUES ('stock','VTEB','us','2017-11-01',51.56,51.58,51.55,51.57,268431,'0');
INSERT INTO daily_summary VALUES ('stock','VTGN','us','2017-11-01',0.84,0.8401,0.7623,0.7643,192581,'0');
INSERT INTO daily_summary VALUES ('etf','VTHR','us','2017-11-01',118.72,118.72,118.13,118.31,224143,'0');
INSERT INTO daily_summary VALUES ('etf','VTI','us','2017-11-01',132.83,133.034,132.17,132.44,2275550,'0');
INSERT INTO daily_summary VALUES ('etf','VTIP','us','2017-11-01',49.54,49.57,49.51,49.51,266913,'0');
INSERT INTO daily_summary VALUES ('stock','VTL','us','2017-11-01',5.95,6.15,5.9,6.1,273113,'0');
INSERT INTO daily_summary VALUES ('stock','VTN','us','2017-11-01',13.6,13.6,13.54,13.6,34560,'0');
INSERT INTO daily_summary VALUES ('stock','VTNR','us','2017-11-01',0.8008,0.84,0.8,0.822,20490,'0');
INSERT INTO daily_summary VALUES ('stock','VTR','us','2017-11-01',62.86,63.51,62.57,63.07,1456190,'0');
INSERT INTO daily_summary VALUES ('stock','VTRB','us','2017-11-01',25.38,25.41,25.3362,25.391,2666,'0');
INSERT INTO daily_summary VALUES ('etf','VTV','us','2017-11-01',102.06,102.34,101.83,101.95,1053010,'0');
INSERT INTO daily_summary VALUES ('stock','VTVT','us','2017-11-01',6.22,6.23,5.76,6.09,39268,'0');
INSERT INTO daily_summary VALUES ('etf','VTWG','us','2017-11-01',132.55,132.55,129.794,130.813,8095,'0');
INSERT INTO daily_summary VALUES ('etf','VTWO','us','2017-11-01',120.72,120.82,118.3,119.01,67062,'0');
INSERT INTO daily_summary VALUES ('etf','VTWV','us','2017-11-01',108.51,108.51,107.025,107.28,8631,'0');
INSERT INTO daily_summary VALUES ('etf','VUG','us','2017-11-01',137.09,137.09,136.27,136.58,607935,'0');
INSERT INTO daily_summary VALUES ('etf','VUSE','us','2017-11-01',31.62,31.69,31.51,31.61,35135,'0');
INSERT INTO daily_summary VALUES ('stock','VUZI','us','2017-11-01',4.85,4.95,4.55,4.65,276642,'0');
INSERT INTO daily_summary VALUES ('etf','VV','us','2017-11-01',118.55,118.66,118.1,118.29,919569,'0');
INSERT INTO daily_summary VALUES ('stock','VVC','us','2017-11-01',68.45,68.51,67.21,67.29,430837,'0');
INSERT INTO daily_summary VALUES ('stock','VVI','us','2017-11-01',58.35,58.35,56.7,57,104984,'0');
INSERT INTO daily_summary VALUES ('stock','VVPR','us','2017-11-01',3.123,3.123,3.1208,3.1208,234,'0');
INSERT INTO daily_summary VALUES ('stock','VVR','us','2017-11-01',4.43,4.44,4.41,4.43,608356,'0');
INSERT INTO daily_summary VALUES ('stock','VVUS','us','2017-11-01',0.6958,0.6999,0.61,0.64,1392470,'0');
INSERT INTO daily_summary VALUES ('stock','VVV','us','2017-11-01',24.22,24.22,23.77,23.87,931889,'0');
INSERT INTO daily_summary VALUES ('etf','VWO','us','2017-11-01',44.92,45.04,44.71,44.74,11585500,'0');
INSERT INTO daily_summary VALUES ('etf','VWOB','us','2017-11-01',80.65,80.77,80.56,80.63,57103,'0');
INSERT INTO daily_summary VALUES ('stock','VWR','us','2017-11-01',33.11,33.125,33.1,33.1,836779,'0');
INSERT INTO daily_summary VALUES ('etf','VXF','us','2017-11-01',109.26,109.29,107.806,108.23,304515,'0');
INSERT INTO daily_summary VALUES ('etf','VXUS','us','2017-11-01',56.2,56.2855,56.01,56.02,786138,'0');
INSERT INTO daily_summary VALUES ('etf','VXX','us','2017-11-01',33.56,34.45,33.4824,34.09,31608900,'0');
INSERT INTO daily_summary VALUES ('etf','VXZ','us','2017-11-01',18.92,19.17,18.88,19.14,311648,'0');
INSERT INTO daily_summary VALUES ('stock','VYGR','us','2017-11-01',20.81,21.63,20.36,20.47,411656,'0');
INSERT INTO daily_summary VALUES ('etf','VYM','us','2017-11-01',82.76,82.95,82.585,82.68,568980,'0');
INSERT INTO daily_summary VALUES ('stock','VYMI','us','2017-11-01',66.15,66.26,65.94,65.94,65644,'0');
INSERT INTO daily_summary VALUES ('stock','VZ','us','2017-11-01',48.15,48.17,47.56,47.83,11046400,'0');
INSERT INTO daily_summary VALUES ('stock','VZA','us','2017-11-01',26.85,26.85,26.7,26.74,17835,'0');
INSERT INTO daily_summary VALUES ('stock','W','us','2017-11-01',70.6,74.5,70.18,74.41,4635120,'0');
INSERT INTO daily_summary VALUES ('stock','WAAS','us','2017-11-01',12.25,12.86,12.08,12.58,161208,'0');
INSERT INTO daily_summary VALUES ('stock','WAB','us','2017-11-01',76.877,76.987,75.45,76.518,924780,'0');
INSERT INTO daily_summary VALUES ('stock','WABC','us','2017-11-01',58.257,59.071,57.025,57.601,66740,'0');
INSERT INTO daily_summary VALUES ('stock','WAC','us','2017-11-01',0.425,0.49,0.42,0.47,350824,'0');
INSERT INTO daily_summary VALUES ('stock','WAFD','us','2017-11-01',34.848,34.848,34.102,34.351,313271,'0');
INSERT INTO daily_summary VALUES ('stock','WAGE','us','2017-11-01',64.3,64.3,62.5105,63.3,367847,'0');
INSERT INTO daily_summary VALUES ('stock','WAIR','us','2017-11-01',9.2,9.2,9,9.2,214697,'0');
INSERT INTO daily_summary VALUES ('stock','WAL','us','2017-11-01',56.26,57.04,55.78,56.03,995402,'0');
INSERT INTO daily_summary VALUES ('stock','WALA','us','2017-11-01',26.31,26.449,26.3,26.3,7792,'0');
INSERT INTO daily_summary VALUES ('stock','WASH','us','2017-11-01',55.55,55.9,54.4,54.75,24457,'0');
INSERT INTO daily_summary VALUES ('stock','WAT','us','2017-11-01',197.3,199.26,196.1,198.05,446604,'0');
INSERT INTO daily_summary VALUES ('stock','WATT','us','2017-11-01',9.91,9.989,9.31,9.39,360582,'0');
INSERT INTO daily_summary VALUES ('stock','WAYN','us','2017-11-01',18,18,17.75,17.99,4874,'0');
INSERT INTO daily_summary VALUES ('stock','WB','us','2017-11-01',93.72,96.87,93.61,96.6,2809290,'0');
INSERT INTO daily_summary VALUES ('stock','WBA','us','2017-11-01',64.964,67.41,64.795,66.625,9758730,'0');
INSERT INTO daily_summary VALUES ('stock','WBAI','us','2017-11-01',9.5,10.18,9.5,10.16,171601,'0');
INSERT INTO daily_summary VALUES ('stock','WBC','us','2017-11-01',148.22,148.485,146.28,147.26,334194,'0');
INSERT INTO daily_summary VALUES ('stock','WBIA','us','2017-11-01',25.05,25.05,24.961,25.0073,2550,'0');
INSERT INTO daily_summary VALUES ('stock','WBIB','us','2017-11-01',25.5193,25.5193,25.3816,25.3816,4918,'0');
INSERT INTO daily_summary VALUES ('stock','WBIC','us','2017-11-01',21.6662,21.6662,21.555,21.6399,18757,'0');
INSERT INTO daily_summary VALUES ('stock','WBID','us','2017-11-01',23.8197,23.8197,23.681,23.7387,5343,'0');
INSERT INTO daily_summary VALUES ('stock','WBIE','us','2017-11-01',24.8914,24.8914,24.8,24.8,3576,'0');
INSERT INTO daily_summary VALUES ('stock','WBIF','us','2017-11-01',27.3114,27.3114,27.19,27.2121,6113,'0');
INSERT INTO daily_summary VALUES ('stock','WBIG','us','2017-11-01',24.6036,24.6036,24.501,24.533,5266,'0');
INSERT INTO daily_summary VALUES ('stock','WBIH','us','2017-11-01',24.832,24.832,24.7901,24.81,20912,'0');
INSERT INTO daily_summary VALUES ('stock','WBII','us','2017-11-01',25.4215,25.44,25.4,25.4379,26830,'0');
INSERT INTO daily_summary VALUES ('stock','WBIL','us','2017-11-01',26.245,26.245,26.1646,26.183,2987,'0');
INSERT INTO daily_summary VALUES ('stock','WBIR','us','2017-11-01',25.275,25.275,25.241,25.25,3690,'0');
INSERT INTO daily_summary VALUES ('stock','WBIY','us','2017-11-01',24.445,24.5,24.429,24.429,3275,'0');
INSERT INTO daily_summary VALUES ('stock','WBK','us','2017-11-01',24.81,24.917,24.723,24.791,92905,'0');
INSERT INTO daily_summary VALUES ('stock','WBS','us','2017-11-01',55.138,55.163,53.864,53.954,550207,'0');
INSERT INTO daily_summary VALUES ('stock','WBS_E','us','2017-11-01',25.31,25.4,25.31,25.4,1131,'0');
INSERT INTO daily_summary VALUES ('stock','WBT','us','2017-11-01',22.17,22.36,21.645,21.8,723636,'0');
INSERT INTO daily_summary VALUES ('stock','WCC','us','2017-11-01',63.75,63.75,61.1,61.95,444637,'0');
INSERT INTO daily_summary VALUES ('stock','WCG','us','2017-11-01',198.85,203.97,197.25,200.91,715288,'0');
INSERT INTO daily_summary VALUES ('stock','WCN','us','2017-11-01',70.866,70.896,69.558,69.728,902399,'0');
INSERT INTO daily_summary VALUES ('stock','WD','us','2017-11-01',55.21,55.64,54.01,54.4,87786,'0');
INSERT INTO daily_summary VALUES ('stock','WDAY','us','2017-11-01',111.94,111.94,108.55,110.27,1096810,'0');
INSERT INTO daily_summary VALUES ('stock','WDC','us','2017-11-01',90.01,91.23,88.9,89.24,3779990,'0');
INSERT INTO daily_summary VALUES ('stock','WDFC','us','2017-11-01',111.4,112.7,110.6,111.65,47648,'0');
INSERT INTO daily_summary VALUES ('etf','WDIV','us','2017-11-01',68.35,68.35,68.09,68.14,3773,'0');
INSERT INTO daily_summary VALUES ('stock','WDR','us','2017-11-01',18.85,18.957,18.6,18.87,1468140,'0');
INSERT INTO daily_summary VALUES ('etf','WDTI','us','2017-11-01',39.5139,39.52,39.29,39.3324,12345,'0');
INSERT INTO daily_summary VALUES ('stock','WEA','us','2017-11-01',14.05,14.05,13.971,14.02,16933,'0');
INSERT INTO daily_summary VALUES ('stock','WEAR','us','2017-11-01',18.02,18.095,18,18.01,20064,'0');
INSERT INTO daily_summary VALUES ('etf','WEAT','us','2017-11-01',6.13,6.2193,6.13,6.18,86202,'0');
INSERT INTO daily_summary VALUES ('stock','WEB','us','2017-11-01',24.3,24.6,23.35,23.7,220102,'0');
INSERT INTO daily_summary VALUES ('stock','WEC','us','2017-11-01',67.182,67.211,66.253,66.437,1495690,'0');
INSERT INTO daily_summary VALUES ('stock','WEN','us','2017-11-01',15.2,15.22,14.8925,15.03,1451140,'0');
INSERT INTO daily_summary VALUES ('stock','WERN','us','2017-11-01',35.8,36.2,35.1231,35.25,847485,'0');
INSERT INTO daily_summary VALUES ('stock','WES','us','2017-11-01',46.5,47.46,46,47.32,870903,'0');
INSERT INTO daily_summary VALUES ('stock','WETF','us','2017-11-01',11.207,11.43,11.137,11.336,1756450,'0');
INSERT INTO daily_summary VALUES ('stock','WEX','us','2017-11-01',124.1,125,118.46,121.48,840975,'0');
INSERT INTO daily_summary VALUES ('stock','WEYS','us','2017-11-01',27.28,27.785,27.1,27.2,12649,'0');
INSERT INTO daily_summary VALUES ('stock','WF','us','2017-11-01',43.9,44.78,43.9,44.48,10539,'0');
INSERT INTO daily_summary VALUES ('stock','WFBI','us','2017-11-01',35.17,35.29,34.6,34.72,33179,'0');
INSERT INTO daily_summary VALUES ('stock','WFC','us','2017-11-01',55.929,56.446,55.8,55.82,11758400,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_J','us','2017-11-01',25.55,25.57,25.51,25.54,114179,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_L','us','2017-11-01',1318.99,1318.99,1318.5,1318.5,1860,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_N','us','2017-11-01',25.05,25.0901,25.05,25.06,22290,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_O','us','2017-11-01',25.04,25.077,25.04,25.0626,13764,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_P','us','2017-11-01',25.11,25.24,25.11,25.14,25340,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_Q','us','2017-11-01',27.23,27.32,27.23,27.3,50590,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_R','us','2017-11-01',28.66,28.75,28.6152,28.75,34634,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_T','us','2017-11-01',26.14,26.1956,26.11,26.11,31375,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_V','us','2017-11-01',26.41,26.4535,26.4,26.42,63048,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_W','us','2017-11-01',25.78,25.811,25.7768,25.7946,47683,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_X','us','2017-11-01',25.45,25.51,25.38,25.42,130041,'0');
INSERT INTO daily_summary VALUES ('stock','WFC_Y','us','2017-11-01',25.85,25.9,25.8,25.8483,16257,'0');
INSERT INTO daily_summary VALUES ('stock','WFE_A','us','2017-11-01',26.42,26.625,26.42,26.55,17547,'0');
INSERT INTO daily_summary VALUES ('stock','WFHY','us','2017-11-01',51.03,51.03,51.03,51.03,330,'0');
INSERT INTO daily_summary VALUES ('stock','WFIG','us','2017-11-01',49.56,49.56,49.56,49.56,0,'0');
INSERT INTO daily_summary VALUES ('stock','WFT','us','2017-11-01',3.69,3.89,3.57,3.67,44133800,'0');
INSERT INTO daily_summary VALUES ('stock','WG','us','2017-11-01',3.12,3.17,2.93,3.01,105016,'0');
INSERT INTO daily_summary VALUES ('stock','WGL','us','2017-11-01',86,86.02,85.46,85.7,234635,'0');
INSERT INTO daily_summary VALUES ('stock','WGO','us','2017-11-01',49.8,49.85,48.25,49.65,989479,'0');
INSERT INTO daily_summary VALUES ('stock','WGP','us','2017-11-01',38.93,39.339,38.49,38.86,134587,'0');
INSERT INTO daily_summary VALUES ('stock','WHF','us','2017-11-01',14.44,14.49,14.16,14.26,23876,'0');
INSERT INTO daily_summary VALUES ('stock','WHFBL','us','2017-11-01',25.5,25.6,25.5,25.5,5971,'0');
INSERT INTO daily_summary VALUES ('stock','WHG','us','2017-11-01',65.09,65.09,64.01,64.16,13377,'0');
INSERT INTO daily_summary VALUES ('stock','WHLM','us','2017-11-01',6.6,6.922,6.499,6.5,7820,'0');
INSERT INTO daily_summary VALUES ('stock','WHLR','us','2017-11-01',9.38,9.38,9.2,9.26,33274,'0');
INSERT INTO daily_summary VALUES ('stock','WHLRD','us','2017-11-01',20.6,20.6,20.09,20.29,1970,'0');
INSERT INTO daily_summary VALUES ('stock','WHLRP','us','2017-11-01',19.63,19.6888,18.75,19.01,11403,'0');
INSERT INTO daily_summary VALUES ('stock','WHLRW','us','2017-11-01',0.0213,0.045,0.0201,0.045,50409,'0');
INSERT INTO daily_summary VALUES ('stock','WHR','us','2017-11-01',165,167.45,163.63,164.42,1137850,'0');
INSERT INTO daily_summary VALUES ('stock','WIA','us','2017-11-01',11.55,11.59,11.5401,11.59,56181,'0');
INSERT INTO daily_summary VALUES ('stock','WIFI','us','2017-11-01',23.5,23.61,22.99,23.17,282713,'0');
INSERT INTO daily_summary VALUES ('stock','WILC','us','2017-11-01',5.45,5.45,5.45,5.45,515,'0');
INSERT INTO daily_summary VALUES ('stock','WIN','us','2017-11-01',1.9,1.9,1.81,1.84,4225380,'0');
INSERT INTO daily_summary VALUES ('stock','WINA','us','2017-11-01',131.79,132.02,131.04,131.04,1974,'0');
INSERT INTO daily_summary VALUES ('stock','WING','us','2017-11-01',33.9,33.9,32.95,32.97,433048,'0');
INSERT INTO daily_summary VALUES ('etf','WIP','us','2017-11-01',56.07,56.29,55.95,56.19,119907,'0');
INSERT INTO daily_summary VALUES ('stock','WIRE','us','2017-11-01',45.65,47.75,44.75,45.35,367839,'0');
INSERT INTO daily_summary VALUES ('stock','WIT','us','2017-11-01',5.36,5.37,5.2,5.21,1450220,'0');
INSERT INTO daily_summary VALUES ('stock','WIW','us','2017-11-01',11.26,11.29,11.26,11.28,61005,'0');
INSERT INTO daily_summary VALUES ('stock','WIX','us','2017-11-01',70,70.7538,68.3,70.5,371946,'0');
INSERT INTO daily_summary VALUES ('stock','WK','us','2017-11-01',22.4,22.65,22.2,22.55,116913,'0');
INSERT INTO daily_summary VALUES ('stock','WKHS','us','2017-11-01',2.63,2.6529,2.56,2.6,117860,'0');
INSERT INTO daily_summary VALUES ('stock','WLB','us','2017-11-01',1.78,1.92,1.65,1.865,392737,'0');
INSERT INTO daily_summary VALUES ('stock','WLDN','us','2017-11-01',29.98,29.98,28.6976,29.07,35607,'0');
INSERT INTO daily_summary VALUES ('stock','WLFC','us','2017-11-01',25.15,25.2,24.62,24.63,8341,'0');
INSERT INTO daily_summary VALUES ('stock','WLH','us','2017-11-01',28.03,29.94,27.99,29.29,789728,'0');
INSERT INTO daily_summary VALUES ('stock','WLK','us','2017-11-01',85.9,86.69,85.289,85.93,650273,'0');
INSERT INTO daily_summary VALUES ('stock','WLKP','us','2017-11-01',21.822,22.117,21.748,22.018,119865,'0');
INSERT INTO daily_summary VALUES ('stock','WLL','us','2017-11-01',24.6,26.22,24.4,25.24,8074860,'0');
INSERT INTO daily_summary VALUES ('stock','WLTW','us','2017-11-01',161.5,163,159.9,160.91,931003,'0');
INSERT INTO daily_summary VALUES ('stock','WM','us','2017-11-01',82.21,82.45,80.86,81.1,2755130,'0');
INSERT INTO daily_summary VALUES ('stock','WMB','us','2017-11-01',28.84,29.11,28.69,28.79,6665800,'0');
INSERT INTO daily_summary VALUES ('stock','WMC','us','2017-11-01',10.11,10.17,10.05,10.11,260621,'0');
INSERT INTO daily_summary VALUES ('etf','WMCR','us','2017-11-01',34.45,34.45,33.21,33.42,5638,'0');
INSERT INTO daily_summary VALUES ('stock','WMGI','us','2017-11-01',26.18,26.18,25.59,25.93,931810,'0');
INSERT INTO daily_summary VALUES ('stock','WMGIZ','us','2017-11-01',1.5231,1.55,1.5231,1.55,399,'0');
INSERT INTO daily_summary VALUES ('stock','WMIH','us','2017-11-01',0.84,0.8493,0.81,0.81,294284,'0');
INSERT INTO daily_summary VALUES ('stock','WMK','us','2017-11-01',38.889,39.872,38.611,39.425,92279,'0');
INSERT INTO daily_summary VALUES ('stock','WMLP','us','2017-11-01',3.052,3.0563,3.0037,3.0424,5080,'0');
INSERT INTO daily_summary VALUES ('stock','WMS','us','2017-11-01',19.75,19.8,19.2,19.35,312848,'0');
INSERT INTO daily_summary VALUES ('stock','WMT','us','2017-11-01',87.07,88.395,87,87.94,6176040,'0');
INSERT INTO daily_summary VALUES ('etf','WMW','us','2017-11-01',28.31,28.31,28.22,28.28,7176,'0');
INSERT INTO daily_summary VALUES ('stock','WNC','us','2017-11-01',20.74,22.44,20.25,21.3,2868860,'0');
INSERT INTO daily_summary VALUES ('stock','WNEB','us','2017-11-01',10.619,10.619,10.32,10.469,65239,'0');
INSERT INTO daily_summary VALUES ('stock','WNS','us','2017-11-01',37.8,38.73,37.8,38.6,225610,'0');
INSERT INTO daily_summary VALUES ('etf','WOOD','us','2017-11-01',70.2,70.4678,70.1,70.27,18749,'0');
INSERT INTO daily_summary VALUES ('stock','WOR','us','2017-11-01',46.12,46.66,45.13,45.18,589549,'0');
INSERT INTO daily_summary VALUES ('stock','WOW','us','2017-11-01',12.89,12.9,12.717,12.87,372596,'0');
INSERT INTO daily_summary VALUES ('stock','WPC','us','2017-11-01',67.9,68.51,67.87,68.23,287309,'0');
INSERT INTO daily_summary VALUES ('stock','WPCS','us','2017-11-01',1.48,1.6,1.42,1.47,619185,'0');
INSERT INTO daily_summary VALUES ('stock','WPG','us','2017-11-01',7.85,7.98,7.7,7.72,2463450,'0');
INSERT INTO daily_summary VALUES ('stock','WPG_H','us','2017-11-01',25.21,25.29,25.145,25.28,13024,'0');
INSERT INTO daily_summary VALUES ('stock','WPG_I','us','2017-11-01',25.05,25.05,24.895,24.95,4461,'0');
INSERT INTO daily_summary VALUES ('stock','WPM','us','2017-11-01',20.94,21.1,20.69,20.73,2441150,'0');
INSERT INTO daily_summary VALUES ('stock','WPPGY','us','2017-11-01',87.65,87.905,87.29,87.33,198846,'0');
INSERT INTO daily_summary VALUES ('stock','WPRT','us','2017-11-01',3.59,3.82,3.53,3.71,1692920,'0');
INSERT INTO daily_summary VALUES ('etf','WPS','us','2017-11-01',38.43,38.49,38.29,38.38,2571,'0');
INSERT INTO daily_summary VALUES ('stock','WPX','us','2017-11-01',11.43,11.845,11.42,11.7,8610700,'0');
INSERT INTO daily_summary VALUES ('stock','WPXP','us','2017-11-01',51.73,53.7,51.73,53.7,97326,'0');
INSERT INTO daily_summary VALUES ('stock','WPZ','us','2017-11-01',36.747,37.593,36.747,37.15,1370490,'0');
INSERT INTO daily_summary VALUES ('stock','WR','us','2017-11-01',53.9,53.9,52.83,52.85,782112,'0');
INSERT INTO daily_summary VALUES ('stock','WRB','us','2017-11-01',68.8,69.09,68.08,68.54,311797,'0');
INSERT INTO daily_summary VALUES ('stock','WRB_B','us','2017-11-01',25.04,25.16,24.9,25.11,83432,'0');
INSERT INTO daily_summary VALUES ('stock','WRB_C','us','2017-11-01',26.22,26.28,26.2,26.261,4357,'0');
INSERT INTO daily_summary VALUES ('stock','WRB_D','us','2017-11-01',26.2,26.45,26.17,26.38,21165,'0');
INSERT INTO daily_summary VALUES ('stock','WRD','us','2017-11-01',13.25,13.47,13.09,13.4,402078,'0');
INSERT INTO daily_summary VALUES ('stock','WRE','us','2017-11-01',32.28,32.58,32.09,32.46,189839,'0');
INSERT INTO daily_summary VALUES ('etf','WREI','us','2017-11-01',47,47,46.54,46.7725,629,'0');
INSERT INTO daily_summary VALUES ('stock','WRI','us','2017-11-01',30.57,30.99,30.4,30.94,858677,'0');
INSERT INTO daily_summary VALUES ('stock','WRK','us','2017-11-01',60.886,61.154,60.389,60.886,2440140,'0');
INSERT INTO daily_summary VALUES ('stock','WRLD','us','2017-11-01',77.8,82.75,77.41,78.465,435199,'0');
INSERT INTO daily_summary VALUES ('stock','WRLS','us','2017-11-01',9.67,9.68,9.67,9.68,226150,'0');
INSERT INTO daily_summary VALUES ('stock','WRLSR','us','2017-11-01',0.4,0.41,0.4,0.41,25800,'0');
INSERT INTO daily_summary VALUES ('stock','WRLSU','us','2017-11-01',10.37,10.37,10.33,10.33,39348,'0');
INSERT INTO daily_summary VALUES ('stock','WRLSW','us','2017-11-01',0.54,0.54,0.53,0.53,11000,'0');
INSERT INTO daily_summary VALUES ('stock','WRN','us','2017-11-01',1.01,1.06,1.01,1.02,94900,'0');
INSERT INTO daily_summary VALUES ('stock','WS','us','2017-11-01',3.15,3.15,3.1059,3.1059,2250,'0');
INSERT INTO daily_summary VALUES ('stock','WSBC','us','2017-11-01',40.78,41.07,39.86,40.24,56282,'0');
INSERT INTO daily_summary VALUES ('stock','WSBF','us','2017-11-01',19.2,19.3,18.65,18.75,85871,'0');
INSERT INTO daily_summary VALUES ('stock','WSCI','us','2017-11-01',3.35,3.35,3.35,3.35,477,'0');
INSERT INTO daily_summary VALUES ('stock','WSFS','us','2017-11-01',50.159,50.209,48.861,49.211,54964,'0');
INSERT INTO daily_summary VALUES ('stock','WSKY','us','2017-11-01',31.08,31.09,31.056,31.09,1140,'0');
INSERT INTO daily_summary VALUES ('stock','WSM','us','2017-11-01',51.76,52.26,51.26,51.51,753549,'0');
INSERT INTO daily_summary VALUES ('stock','WSO','us','2017-11-01',167.22,167.22,164.62,165.43,157400,'0');
INSERT INTO daily_summary VALUES ('stock','WSR','us','2017-11-01',13.286,13.395,13.176,13.385,233674,'0');
INSERT INTO daily_summary VALUES ('stock','WST','us','2017-11-01',101.35,101.92,100.7,100.89,500076,'0');
INSERT INTO daily_summary VALUES ('stock','WSTG','us','2017-11-01',13.782,14.474,13.782,14.474,15293,'0');
INSERT INTO daily_summary VALUES ('stock','WSTL','us','2017-11-01',2.93,2.93,2.78,2.83,48864,'0');
INSERT INTO daily_summary VALUES ('stock','WTBA','us','2017-11-01',24.572,24.72,23.927,24.076,20291,'0');
INSERT INTO daily_summary VALUES ('stock','WTFC','us','2017-11-01',81.764,82.013,79.253,80.136,297776,'0');
INSERT INTO daily_summary VALUES ('stock','WTFCM','us','2017-11-01',28.01,28.3592,28.01,28.34,2287,'0');
INSERT INTO daily_summary VALUES ('stock','WTI','us','2017-11-01',3.19,3.37,3.1884,3.31,3503750,'0');
INSERT INTO daily_summary VALUES ('stock','WTID','us','2017-11-01',20.62,21.539,20.5799,21.37,14733,'0');
INSERT INTO daily_summary VALUES ('stock','WTIU','us','2017-11-01',18.51,18.61,17.65,17.81,17791,'0');
INSERT INTO daily_summary VALUES ('stock','WTM','us','2017-11-01',891,903.26,890.28,895.9,36459,'0');
INSERT INTO daily_summary VALUES ('stock','WTR','us','2017-11-01',35.73,35.84,35.2,35.43,492744,'0');
INSERT INTO daily_summary VALUES ('stock','WTRX','us','2017-11-01',31.1171,31.1171,31.1171,31.1171,111,'0');
INSERT INTO daily_summary VALUES ('stock','WTS','us','2017-11-01',67.85,67.85,66.55,66.9,103966,'0');
INSERT INTO daily_summary VALUES ('stock','WTT','us','2017-11-01',1.69,1.73,1.563,1.72,72242,'0');
INSERT INTO daily_summary VALUES ('stock','WTTR','us','2017-11-01',16.53,16.77,16.26,16.36,165078,'0');
INSERT INTO daily_summary VALUES ('stock','WTW','us','2017-11-01',45.21,46.27,44.35,45.15,729995,'0');
INSERT INTO daily_summary VALUES ('stock','WU','us','2017-11-01',19.97,20.04,19.83,19.98,3219470,'0');
INSERT INTO daily_summary VALUES ('stock','WUBA','us','2017-11-01',67.59,67.855,66.68,67.47,1274940,'0');
INSERT INTO daily_summary VALUES ('stock','WVE','us','2017-11-01',23.25,23.25,22.45,23,222230,'0');
INSERT INTO daily_summary VALUES ('stock','WVFC','us','2017-11-01',16.0278,16.0278,16.0278,16.0278,101,'0');
INSERT INTO daily_summary VALUES ('stock','WVVI','us','2017-11-01',8.625,8.71,8.46,8.48,5516,'0');
INSERT INTO daily_summary VALUES ('stock','WVVIP','us','2017-11-01',5.6,5.6112,5.6,5.6112,420,'0');
INSERT INTO daily_summary VALUES ('stock','WWD','us','2017-11-01',77.936,78.685,77.272,77.766,196977,'0');
INSERT INTO daily_summary VALUES ('stock','WWE','us','2017-11-01',26.75,26.869,26.27,26.5,956376,'0');
INSERT INTO daily_summary VALUES ('stock','WWR','us','2017-11-01',0.88,0.88,0.8038,0.8293,214385,'0');
INSERT INTO daily_summary VALUES ('stock','WWW','us','2017-11-01',27.63,27.63,26.79,27.13,933345,'0');
INSERT INTO daily_summary VALUES ('stock','WY','us','2017-11-01',36,36.3,35.69,35.81,3466080,'0');
INSERT INTO daily_summary VALUES ('stock','WYDE','us','2017-11-01',32.54,32.54,32.54,32.54,5173,'0');
INSERT INTO daily_summary VALUES ('stock','WYN','us','2017-11-01',108.56,110.04,106.76,107.22,972957,'0');
INSERT INTO daily_summary VALUES ('stock','WYNN','us','2017-11-01',152.38,153.56,149.87,151.55,3463860,'0');
INSERT INTO daily_summary VALUES ('stock','WYY','us','2017-11-01',0.53,0.56,0.53,0.554,118055,'0');
INSERT INTO daily_summary VALUES ('stock','X','us','2017-11-01',27.799,29.276,26.621,27.25,38703200,'0');
INSERT INTO daily_summary VALUES ('etf','XAR','us','2017-11-01',81.61,81.71,81.01,81.41,147741,'0');
INSERT INTO daily_summary VALUES ('etf','XBI','us','2017-11-01',84.42,84.75,82.04,82.47,4271590,'0');
INSERT INTO daily_summary VALUES ('stock','XBIO','us','2017-11-01',2.129,2.229,1.9206,2.21,16399,'0');
INSERT INTO daily_summary VALUES ('stock','XBIT','us','2017-11-01',4.23,4.23,4.15,4.2,38457,'0');
INSERT INTO daily_summary VALUES ('stock','XBKS','us','2017-11-01',32.18,32.35,31.47,31.74,39771,'0');
INSERT INTO daily_summary VALUES ('stock','XCO','us','2017-11-01',1.47,1.53,1.3801,1.45,202288,'0');
INSERT INTO daily_summary VALUES ('stock','XCRA','us','2017-11-01',9.87,9.92,9.8449,9.9,609013,'0');
INSERT INTO daily_summary VALUES ('stock','XEC','us','2017-11-01',117.75,121.43,117.75,121.21,1181590,'0');
INSERT INTO daily_summary VALUES ('stock','XEL','us','2017-11-01',49.66,49.71,49.19,49.23,2640580,'0');
INSERT INTO daily_summary VALUES ('stock','XELA','us','2017-11-01',5.47,5.72,5.32,5.63,9483,'0');
INSERT INTO daily_summary VALUES ('stock','XELB','us','2017-11-01',3.2,3.2,2.825,2.85,16687,'0');
INSERT INTO daily_summary VALUES ('stock','XENE','us','2017-11-01',3.25,3.25,3.1,3.1,31436,'0');
INSERT INTO daily_summary VALUES ('stock','XENT','us','2017-11-01',29.65,29.7,28.25,28.3,119002,'0');
INSERT INTO daily_summary VALUES ('etf','XES','us','2017-11-01',15.68,15.949,15.41,15.53,694287,'0');
INSERT INTO daily_summary VALUES ('stock','XFLT','us','2017-11-01',11,11,10.7001,10.84,10612,'0');
INSERT INTO daily_summary VALUES ('stock','XGTI','us','2017-11-01',1.43,1.58,1.42,1.52,503044,'0');
INSERT INTO daily_summary VALUES ('etf','XHB','us','2017-11-01',41.58,41.59,41.28,41.45,1430050,'0');
INSERT INTO daily_summary VALUES ('etf','XHE','us','2017-11-01',64.4,64.4,63.42,63.45,7121,'0');
INSERT INTO daily_summary VALUES ('stock','XHR','us','2017-11-01',21.82,21.92,21.61,21.81,325320,'0');
INSERT INTO daily_summary VALUES ('etf','XHS','us','2017-11-01',58.81,58.81,57.58,57.71,7580,'0');
INSERT INTO daily_summary VALUES ('stock','XIN','us','2017-11-01',5.62,5.7,5.6,5.64,121492,'0');
INSERT INTO daily_summary VALUES ('stock','XINA','us','2017-11-01',24.045,24.07,24,24,10200,'0');
INSERT INTO daily_summary VALUES ('stock','XITK','us','2017-11-01',80.07,80.07,78.9055,79.31,2681,'0');
INSERT INTO daily_summary VALUES ('etf','XIV','us','2017-11-01',112.91,113.15,109.95,111.11,7904000,'0');
INSERT INTO daily_summary VALUES ('stock','XIVH','us','2017-11-01',67.85,67.8675,66.89,67.02,3325,'0');
INSERT INTO daily_summary VALUES ('stock','XL','us','2017-11-01',40.6,40.78,40.1,40.34,3137210,'0');
INSERT INTO daily_summary VALUES ('etf','XLB','us','2017-11-01',59.3,59.48,59.07,59.29,5598750,'0');
INSERT INTO daily_summary VALUES ('etf','XLE','us','2017-11-01',68.32,68.995,68.29,68.68,17304700,'0');
INSERT INTO daily_summary VALUES ('etf','XLF','us','2017-11-01',26.71,26.83,26.611,26.64,40919600,'0');
INSERT INTO daily_summary VALUES ('etf','XLG','us','2017-11-01',183.86,184.08,183.22,183.457,11757,'0');
INSERT INTO daily_summary VALUES ('etf','XLI','us','2017-11-01',71.75,71.92,71.45,71.49,9104780,'0');
INSERT INTO daily_summary VALUES ('etf','XLK','us','2017-11-01',63.25,63.33,62.63,62.96,9549830,'0');
INSERT INTO daily_summary VALUES ('stock','XLNX','us','2017-11-01',74.06,74.11,72.485,73.04,1624230,'0');
INSERT INTO daily_summary VALUES ('etf','XLP','us','2017-11-01',53.24,53.4,53.09,53.26,7556260,'0');
INSERT INTO daily_summary VALUES ('stock','XLRE','us','2017-11-01',32.55,32.6,32.415,32.47,2057710,'0');
INSERT INTO daily_summary VALUES ('stock','XLRN','us','2017-11-01',39.4,39.42,37.56,37.83,211745,'0');
INSERT INTO daily_summary VALUES ('etf','XLU','us','2017-11-01',55.21,55.21,54.7147,54.81,10879200,'0');
INSERT INTO daily_summary VALUES ('etf','XLV','us','2017-11-01',81.25,81.65,81.09,81.31,7427810,'0');
INSERT INTO daily_summary VALUES ('etf','XLY','us','2017-11-01',92.29,92.59,91.98,92.02,4417220,'0');
INSERT INTO daily_summary VALUES ('etf','XME','us','2017-11-01',32.42,32.8,31.7852,31.9,3893620,'0');
INSERT INTO daily_summary VALUES ('etf','XMLV','us','2017-11-01',45.19,45.19,44.71,44.79,160538,'0');
INSERT INTO daily_summary VALUES ('etf','XMPT','us','2017-11-01',26.55,26.59,26.5,26.5,150144,'0');
INSERT INTO daily_summary VALUES ('stock','XNCR','us','2017-11-01',20,20.1,19.24,19.33,159129,'0');
INSERT INTO daily_summary VALUES ('stock','XNET','us','2017-11-01',9,12.1,8.66,12.01,12126800,'0');
INSERT INTO daily_summary VALUES ('stock','XNTK','us','2017-11-01',84.34,84.34,83.2101,83.59,26560,'0');
INSERT INTO daily_summary VALUES ('stock','XNY','us','2017-11-01',1.23,1.23,1.23,1.23,300,'0');
INSERT INTO daily_summary VALUES ('stock','XOG','us','2017-11-01',16.23,16.8,16.22,16.37,1968210,'0');
INSERT INTO daily_summary VALUES ('stock','XOM','us','2017-11-01',82.625,83.17,82.497,83.101,9517160,'0');
INSERT INTO daily_summary VALUES ('stock','XOMA','us','2017-11-01',24.41,24.52,22.8466,23,154054,'0');
INSERT INTO daily_summary VALUES ('stock','XON','us','2017-11-01',16.45,16.64,15.71,15.89,1414310,'0');
INSERT INTO daily_summary VALUES ('stock','XONE','us','2017-11-01',10.31,10.5683,10.025,10.37,191501,'0');
INSERT INTO daily_summary VALUES ('etf','XOP','us','2017-11-01',34.56,35.43,34.55,35.3,28988600,'0');
INSERT INTO daily_summary VALUES ('stock','XOXO','us','2017-11-01',20.18,20.18,19.74,19.94,121566,'0');
INSERT INTO daily_summary VALUES ('stock','XPER','us','2017-11-01',23.05,23.05,22.15,22.4,440480,'0');
INSERT INTO daily_summary VALUES ('etf','XPH','us','2017-11-01',40.73,41.18,40.58,40.96,124876,'0');
INSERT INTO daily_summary VALUES ('stock','XPL','us','2017-11-01',0.6299,0.639,0.605,0.61,62206,'0');
INSERT INTO daily_summary VALUES ('stock','XPLR','us','2017-11-01',3.28,3.29,3.11,3.12,10913,'0');
INSERT INTO daily_summary VALUES ('stock','XPO','us','2017-11-01',70.13,70.82,69.09,69.7,2229550,'0');
INSERT INTO daily_summary VALUES ('etf','XPP','us','2017-11-01',82,82.25,81.39,81.39,4863,'0');
INSERT INTO daily_summary VALUES ('stock','XRAY','us','2017-11-01',61.37,61.74,61.06,61.16,1820210,'0');
INSERT INTO daily_summary VALUES ('stock','XRF','us','2017-11-01',8.09,8.34,7.86,7.99,225398,'0');
INSERT INTO daily_summary VALUES ('etf','XRLV','us','2017-11-01',32.54,32.55,32.4332,32.46,44596,'0');
INSERT INTO daily_summary VALUES ('stock','XRM','us','2017-11-01',4.75,4.75,4.4732,4.49,211858,'0');
INSERT INTO daily_summary VALUES ('etf','XRT','us','2017-11-01',39.74,39.9453,39.27,39.58,5581460,'0');
INSERT INTO daily_summary VALUES ('stock','XRX','us','2017-11-01',30.34,30.52,29.98,30.29,2784790,'0');
INSERT INTO daily_summary VALUES ('etf','XSD','us','2017-11-01',71.42,71.42,69.69,70.33,101265,'0');
INSERT INTO daily_summary VALUES ('stock','XSHD','us','2017-11-01',25.24,25.25,25.0501,25.1096,11112,'0');
INSERT INTO daily_summary VALUES ('stock','XSHQ','us','2017-11-01',26.0047,26.0047,26.0047,26.0047,0,'0');
INSERT INTO daily_summary VALUES ('etf','XSLV','us','2017-11-01',46.75,46.75,46.0944,46.33,82147,'0');
INSERT INTO daily_summary VALUES ('etf','XSOE','us','2017-11-01',30.9603,30.99,30.827,30.827,3530,'0');
INSERT INTO daily_summary VALUES ('etf','XSW','us','2017-11-01',68.5,68.5,67.9,67.9,2207,'0');
INSERT INTO daily_summary VALUES ('etf','XT','us','2017-11-01',35.27,35.27,34.855,34.9,114732,'0');
INSERT INTO daily_summary VALUES ('stock','XTH','us','2017-11-01',79.69,79.69,78.14,78.14,1205,'0');
INSERT INTO daily_summary VALUES ('etf','XTL','us','2017-11-01',69.36,69.36,68.02,68.41,19477,'0');
INSERT INTO daily_summary VALUES ('stock','XTLB','us','2017-11-01',2.3,2.38,2.3,2.3499,4576,'0');
INSERT INTO daily_summary VALUES ('etf','XTN','us','2017-11-01',60.44,60.5612,59.871,60.001,14930,'0');
INSERT INTO daily_summary VALUES ('stock','XTNT','us','2017-11-01',0.6821,0.7498,0.6798,0.7064,678369,'0');
INSERT INTO daily_summary VALUES ('stock','XUSA','us','2017-11-01',27.004,27.004,27.004,27.004,0,'0');
INSERT INTO daily_summary VALUES ('stock','XWEB','us','2017-11-01',74.25,74.25,74.13,74.13,281,'0');
INSERT INTO daily_summary VALUES ('stock','XXII','us','2017-11-01',2.09,2.185,2.05,2.16,2414590,'0');
INSERT INTO daily_summary VALUES ('stock','XYL','us','2017-11-01',67.02,67.48,65.98,66.15,776720,'0');
INSERT INTO daily_summary VALUES ('stock','Y','us','2017-11-01',567.63,569.06,564.5,564.57,66137,'0');
INSERT INTO daily_summary VALUES ('etf','YANG','us','2017-11-01',6.32,6.41,6.281,6.41,188686,'0');
INSERT INTO daily_summary VALUES ('etf','YAO','us','2017-11-01',35.15,35.187,35.0994,35.0994,2743,'0');
INSERT INTO daily_summary VALUES ('etf','YCL','us','2017-11-01',56.24,56.5,56.16,56.3,636,'0');
INSERT INTO daily_summary VALUES ('etf','YCS','us','2017-11-01',76.63,76.75,76.09,76.5392,51506,'0');
INSERT INTO daily_summary VALUES ('etf','YDIV','us','2017-11-01',18.3176,18.68,18.3176,18.62,2236,'0');
INSERT INTO daily_summary VALUES ('stock','YECO','us','2017-11-01',3,3,2.525,2.6635,10068,'0');
INSERT INTO daily_summary VALUES ('stock','YELP','us','2017-11-01',47.22,48.17,45.95,46.03,3906050,'0');
INSERT INTO daily_summary VALUES ('stock','YERR','us','2017-11-01',15.8,16.5999,15.67,16.06,39583,'0');
INSERT INTO daily_summary VALUES ('stock','YESR','us','2017-11-01',27.6499,27.68,27.6332,27.68,3333,'0');
INSERT INTO daily_summary VALUES ('stock','YEXT','us','2017-11-01',11.47,11.7,11.23,11.65,192069,'0');
INSERT INTO daily_summary VALUES ('stock','YGE','us','2017-11-01',2.26,2.26,2.2,2.22,24798,'0');
INSERT INTO daily_summary VALUES ('stock','YGYI','us','2017-11-01',4.8993,4.9708,4.71,4.71,822,'0');
INSERT INTO daily_summary VALUES ('stock','YIN','us','2017-11-01',12.1,12.599,11.88,12.39,182311,'0');
INSERT INTO daily_summary VALUES ('etf','YINN','us','2017-11-01',33.64,33.85,33.2366,33.26,798526,'0');
INSERT INTO daily_summary VALUES ('etf','YLCO','us','2017-11-01',12.458,12.475,12.373,12.45,6051,'0');
INSERT INTO daily_summary VALUES ('stock','YLD','us','2017-11-01',41.68,41.68,41.59,41.59,230,'0');
INSERT INTO daily_summary VALUES ('etf','YMLI','us','2017-11-01',13.358,13.525,13.329,13.523,2567,'0');
INSERT INTO daily_summary VALUES ('etf','YMLP','us','2017-11-01',23.402,23.647,23.402,23.529,24403,'0');
INSERT INTO daily_summary VALUES ('stock','YNDX','us','2017-11-01',34.12,34.83,33.94,34.03,2412360,'0');
INSERT INTO daily_summary VALUES ('stock','YOGA','us','2017-11-01',2.55,2.7539,2.5215,2.65,28683,'0');
INSERT INTO daily_summary VALUES ('stock','YORW','us','2017-11-01',35.65,35.65,33.25,33.4,35477,'0');
INSERT INTO daily_summary VALUES ('stock','YPF','us','2017-11-01',24.6,24.84,24.16,24.5,1514450,'0');
INSERT INTO daily_summary VALUES ('stock','YRCW','us','2017-11-01',13.65,13.65,12.905,13.14,498913,'0');
INSERT INTO daily_summary VALUES ('stock','YRD','us','2017-11-01',43.51,45.3,42.08,42.52,627568,'0');
INSERT INTO daily_summary VALUES ('stock','YTEN','us','2017-11-01',2.82,3.0999,2.75,2.87,68194,'0');
INSERT INTO daily_summary VALUES ('stock','YTRA','us','2017-11-01',8.85,8.85,8.49,8.55,175676,'0');
INSERT INTO daily_summary VALUES ('stock','YUM','us','2017-11-01',74.54,74.95,74.26,74.31,1887010,'0');
INSERT INTO daily_summary VALUES ('stock','YUMA','us','2017-11-01',0.96,0.989,0.93,0.945,281556,'0');
INSERT INTO daily_summary VALUES ('stock','YUMC','us','2017-11-01',40.47,40.849,39.88,39.89,1857160,'0');
INSERT INTO daily_summary VALUES ('stock','YUME','us','2017-11-01',4.27,4.27,4.12,4.19,128023,'0');
INSERT INTO daily_summary VALUES ('etf','YXI','us','2017-11-01',19.63,19.75,19.63,19.74,1500,'0');
INSERT INTO daily_summary VALUES ('stock','YY','us','2017-11-01',90.84,91.58,88.39,89.24,776080,'0');
INSERT INTO daily_summary VALUES ('etf','YYY','us','2017-11-01',19.67,19.73,19.67,19.7046,61518,'0');
INSERT INTO daily_summary VALUES ('stock','Z','us','2017-11-01',41.46,41.46,40.1,40.3,967342,'0');
INSERT INTO daily_summary VALUES ('stock','ZAGG','us','2017-11-01',15.9,16.145,14.85,15.7,1057470,'0');
INSERT INTO daily_summary VALUES ('stock','ZAIS','us','2017-11-01',3.7539,3.77,3.74,3.77,4437,'0');
INSERT INTO daily_summary VALUES ('stock','ZAYO','us','2017-11-01',36.27,36.38,35.41,35.58,1356480,'0');
INSERT INTO daily_summary VALUES ('stock','ZBH','us','2017-11-01',111.91,117.125,110.79,115.09,4569990,'0');
INSERT INTO daily_summary VALUES ('stock','ZBIO','us','2017-11-01',12.07,12.65,12.07,12.56,15901,'0');
INSERT INTO daily_summary VALUES ('stock','ZBK','us','2017-11-01',29.97,30.425,29.97,30.03,1373,'0');
INSERT INTO daily_summary VALUES ('stock','ZBRA','us','2017-11-01',117.05,117.05,113.12,113.77,265221,'0');
INSERT INTO daily_summary VALUES ('stock','ZBZX','us','2017-11-01',0.95,1.01,0.91,0.97,731000,'0');
INSERT INTO daily_summary VALUES ('stock','ZB_G','us','2017-11-01',27.56,27.6015,27.56,27.6015,4649,'0');
INSERT INTO daily_summary VALUES ('stock','ZB_H','us','2017-11-01',25.86,25.86,25.85,25.86,5090,'0');
INSERT INTO daily_summary VALUES ('stock','ZDGE','us','2017-11-01',2.89,3.03,2.89,2.92,30950,'0');
INSERT INTO daily_summary VALUES ('stock','ZEAL','us','2017-11-01',18.28,18.49,18.11,18.48,10175,'0');
INSERT INTO daily_summary VALUES ('stock','ZEN','us','2017-11-01',31.26,31.46,29.57,30.61,1338160,'0');
INSERT INTO daily_summary VALUES ('stock','ZEUS','us','2017-11-01',19.2,20.405,18.99,19.35,138041,'0');
INSERT INTO daily_summary VALUES ('stock','ZF','us','2017-11-01',12.69,12.75,12.64,12.67,90867,'0');
INSERT INTO daily_summary VALUES ('stock','ZFGN','us','2017-11-01',3.57,3.67,3.47,3.65,103165,'0');
INSERT INTO daily_summary VALUES ('stock','ZG','us','2017-11-01',41.42,41.42,40.3,40.53,385988,'0');
INSERT INTO daily_summary VALUES ('stock','ZGNX','us','2017-11-01',37.55,38.95,37.5,38.65,884102,'0');
INSERT INTO daily_summary VALUES ('stock','ZION','us','2017-11-01',46.71,47.06,46.16,46.22,1400060,'0');
INSERT INTO daily_summary VALUES ('stock','ZIONZ','us','2017-11-01',11.33,11.33,11.07,11.07,1000,'0');
INSERT INTO daily_summary VALUES ('stock','ZIOP','us','2017-11-01',4.68,4.82,4.45,4.49,2022530,'0');
INSERT INTO daily_summary VALUES ('etf','ZIV','us','2017-11-01',82.25,82.419,81.17,81.36,192771,'0');
INSERT INTO daily_summary VALUES ('stock','ZIXI','us','2017-11-01',4.9,4.9,4.66,4.7,185910,'0');
INSERT INTO daily_summary VALUES ('stock','ZKIN','us','2017-11-01',7,7.3,7,7.15,18456,'0');
INSERT INTO daily_summary VALUES ('stock','ZLAB','us','2017-11-01',27.13,28.52,27.11,27.99,77354,'0');
INSERT INTO daily_summary VALUES ('etf','ZMLP','us','2017-11-01',16.28,16.3747,16.24,16.31,37322,'0');
INSERT INTO daily_summary VALUES ('stock','ZN','us','2017-11-01',2.58,2.66,2.51,2.55,121133,'0');
INSERT INTO daily_summary VALUES ('stock','ZNGA','us','2017-11-01',3.91,3.91,3.785,3.84,8750380,'0');
INSERT INTO daily_summary VALUES ('stock','ZNH','us','2017-11-01',38.55,38.88,38.46,38.87,27405,'0');
INSERT INTO daily_summary VALUES ('stock','ZOES','us','2017-11-01',12.31,12.47,12.21,12.25,367371,'0');
INSERT INTO daily_summary VALUES ('etf','ZROZ','us','2017-11-01',116.2,117.42,116.2,116.78,17171,'0');
INSERT INTO daily_summary VALUES ('stock','ZSAN','us','2017-11-01',0.68,0.767,0.67,0.7648,592437,'0');
INSERT INTO daily_summary VALUES ('etf','ZSL','us','2017-11-01',31.79,31.79,30.63,30.79,269774,'0');
INSERT INTO daily_summary VALUES ('stock','ZTO','us','2017-11-01',15.99,17.175,15.98,16.92,8703150,'0');
INSERT INTO daily_summary VALUES ('stock','ZTR','us','2017-11-01',13.251,13.341,13.222,13.321,96948,'0');
INSERT INTO daily_summary VALUES ('stock','ZTS','us','2017-11-01',64.106,64.99,63.926,64.416,3070440,'0');
INSERT INTO daily_summary VALUES ('stock','ZUMZ','us','2017-11-01',17.75,17.75,17.2,17.35,368016,'0');
INSERT INTO daily_summary VALUES ('stock','ZVV','us','2017-11-01',100.14,100.14,100.14,100.14,0,'0');
INSERT INTO daily_summary VALUES ('stock','ZX','us','2017-11-01',1.18,1.18,1.1599,1.1599,2535,'0');
INSERT INTO daily_summary VALUES ('stock','ZYME','us','2017-11-01',8.39,8.39,8.06,8.06,6411,'0');
INSERT INTO daily_summary VALUES ('stock','ZYNE','us','2017-11-01',9.7,9.93,9.41,9.69,301604,'0');

INSERT INTO symbols VALUES ('A','Agilent Technologies, Inc.','Capital Goods',1999);
INSERT INTO symbols VALUES ('AA','Alcoa Corporation','Basic Industries',2016);
INSERT INTO symbols VALUES ('AABA','Altaba Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AAC','AAC Holdings, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('AAL','American Airlines Group, Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('AAMC','Altisource Asset Management Corp','Finance',NULL);
INSERT INTO symbols VALUES ('AAME','Atlantic American Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('AAN','Aaron&#39;s,  Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AAOI','Applied Optoelectronics, Inc.','Technology',2013);
INSERT INTO symbols VALUES ('AAON','AAON, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AAP','Advance Auto Parts Inc','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AAPL','Apple Inc.','Technology',1980);
INSERT INTO symbols VALUES ('AAT','American Assets Trust, Inc.','Consumer Services',2011);
INSERT INTO symbols VALUES ('AAU','Almaden Minerals, Ltd.','Basic Industries',2015);
INSERT INTO symbols VALUES ('AAWW','Atlas Air Worldwide Holdings','Transportation',NULL);
INSERT INTO symbols VALUES ('AAXJ','iShares MSCI All Country Asia ex Japan Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('AAXN','Axon Enterprise, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AB','AllianceBernstein Holding L.P.','Finance',NULL);
INSERT INTO symbols VALUES ('ABB','ABB Ltd','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('ABBV','AbbVie Inc.','Health Care',2012);
INSERT INTO symbols VALUES ('ABC','AmerisourceBergen Corporation (Holding Co)','Health Care',NULL);
INSERT INTO symbols VALUES ('ABCB','Ameris Bancorp','Finance',1994);
INSERT INTO symbols VALUES ('ABDC','Alcentra Capital Corp.',NULL,NULL);
INSERT INTO symbols VALUES ('ABEO','Abeona Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ABEOW','Abeona Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ABEV','Ambev S.A.','Consumer Non-Durables',2013);
INSERT INTO symbols VALUES ('ABG','Asbury Automotive Group Inc','Consumer Durables',2002);
INSERT INTO symbols VALUES ('ABIL','Ability Inc.','Capital Goods',2014);
INSERT INTO symbols VALUES ('ABIO','ARCA biopharma, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ABM','ABM Industries Incorporated','Finance',NULL);
INSERT INTO symbols VALUES ('ABMD','ABIOMED, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ABR','Arbor Realty Trust','Consumer Services',2004);
INSERT INTO symbols VALUES ('ABR^A','Arbor Realty Trust',NULL,NULL);
INSERT INTO symbols VALUES ('ABR^B','Arbor Realty Trust',NULL,NULL);
INSERT INTO symbols VALUES ('ABR^C','Arbor Realty Trust',NULL,NULL);
INSERT INTO symbols VALUES ('ABT','Abbott Laboratories','Health Care',NULL);
INSERT INTO symbols VALUES ('ABTX','Allegiance Bancshares, Inc.','Finance',2015);
INSERT INTO symbols VALUES ('ABUS','Arbutus Biopharma Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('AC','Associated Capital Group, Inc.','Finance',2015);
INSERT INTO symbols VALUES ('ACA','Arcosa, Inc.','Capital Goods',2018);
INSERT INTO symbols VALUES ('ACAD','ACADIA Pharmaceuticals Inc.','Health Care',1985);
INSERT INTO symbols VALUES ('ACAM','Acamar Partners Acquisition Corp.','Finance',2019);
INSERT INTO symbols VALUES ('ACAMU','Acamar Partners Acquisition Corp.','Finance',2019);
INSERT INTO symbols VALUES ('ACAMW','Acamar Partners Acquisition Corp.','Finance',2019);
INSERT INTO symbols VALUES ('ACB','Aurora Cannabis Inc.','Consumer Durables',2018);
INSERT INTO symbols VALUES ('ACBI','Atlantic Capital Bancshares, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('ACC','American Campus Communities Inc','Consumer Services',2004);
INSERT INTO symbols VALUES ('ACCO','Acco Brands Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('ACCP','Accelerated Pharma, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('ACER','Acer Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ACGL','Arch Capital Group Ltd.','Finance',NULL);
INSERT INTO symbols VALUES ('ACGLO','Arch Capital Group Ltd.','Finance',NULL);
INSERT INTO symbols VALUES ('ACGLP','Arch Capital Group Ltd.','Finance',NULL);
INSERT INTO symbols VALUES ('ACH','Aluminum Corporation of China Limited','Basic Industries',2001);
INSERT INTO symbols VALUES ('ACHC','Acadia Healthcare Company, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ACHN','Achillion Pharmaceuticals, Inc.','Health Care',2006);
INSERT INTO symbols VALUES ('ACHV','Achieve Life Sciences, Inc. ','Health Care',NULL);
INSERT INTO symbols VALUES ('ACIA','Acacia Communications, Inc.','Technology',2016);
INSERT INTO symbols VALUES ('ACIU','AC Immune SA','Health Care',2016);
INSERT INTO symbols VALUES ('ACIW','ACI Worldwide, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('ACLS','Axcelis Technologies, Inc.','Technology',2000);
INSERT INTO symbols VALUES ('ACM','AECOM','Consumer Services',2007);
INSERT INTO symbols VALUES ('ACMR','ACM Research, Inc.','Technology',2017);
INSERT INTO symbols VALUES ('ACN','Accenture plc','Miscellaneous',2001);
INSERT INTO symbols VALUES ('ACNB','ACNB Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('ACOR','Acorda Therapeutics, Inc.','Health Care',2006);
INSERT INTO symbols VALUES ('ACP','Aberdeen Income Credit Strategies Fund',NULL,2011);
INSERT INTO symbols VALUES ('ACRE','Ares Commercial Real Estate Corporation','Consumer Services',2012);
INSERT INTO symbols VALUES ('ACRS','Aclaris Therapeutics, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('ACRX','AcelRx Pharmaceuticals, Inc.','Health Care',2011);
INSERT INTO symbols VALUES ('ACST','Acasti Pharma, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ACT','AdvisorShares Vice ETF',NULL,NULL);
INSERT INTO symbols VALUES ('ACTG','Acacia Research Corporation','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('ACTTU','Act II Global Acquisition Corp.','Finance',2019);
INSERT INTO symbols VALUES ('ACU','Acme United Corporation.','Capital Goods',1988);
INSERT INTO symbols VALUES ('ACV','AllianzGI Diversified Income & Convertible Fund',NULL,2015);
INSERT INTO symbols VALUES ('ACWI','iShares MSCI ACWI Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('ACWX','iShares MSCI ACWI ex US Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('ACY','AeroCentury Corp.','Technology',NULL);
INSERT INTO symbols VALUES ('ADAP','Adaptimmune Therapeutics plc','Health Care',2015);
INSERT INTO symbols VALUES ('ADBE','Adobe Inc.','Technology',1986);
INSERT INTO symbols VALUES ('ADC','Agree Realty Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ADES','Advanced Emissions Solutions, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('ADI','Analog Devices, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('ADIL','Adial Pharmaceuticals, Inc','Health Care',2018);
INSERT INTO symbols VALUES ('ADILW','Adial Pharmaceuticals, Inc','Health Care',2018);
INSERT INTO symbols VALUES ('ADM','Archer-Daniels-Midland Company','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('ADMA','ADMA Biologics Inc','Health Care',NULL);
INSERT INTO symbols VALUES ('ADMP','Adamis Pharmaceuticals Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('ADMS','Adamas Pharmaceuticals, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('ADNT','Adient plc','Capital Goods',2016);
INSERT INTO symbols VALUES ('ADOM','ADOMANI, Inc.','Capital Goods',2017);
INSERT INTO symbols VALUES ('ADP','Automatic Data Processing, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('ADRA','Invesco BLDRS Asia 50 ADR Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('ADRD','Invesco BLDRS Developed Markets 100 ADR Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('ADRE','Invesco BLDRS Emerging Markets 50 ADR Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('ADRO','Aduro Biotech, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('ADRU','Invesco BLDRS Europe Select ADR Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('ADS','Alliance Data Systems Corporation','Miscellaneous',2001);
INSERT INTO symbols VALUES ('ADSK','Autodesk, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('ADSW','Advanced Disposal Services, Inc.','Public Utilities',2016);
INSERT INTO symbols VALUES ('ADT','ADT Inc.','Consumer Services',2018);
INSERT INTO symbols VALUES ('ADTN','ADTRAN, Inc.','Public Utilities',1994);
INSERT INTO symbols VALUES ('ADUS','Addus HomeCare Corporation','Health Care',2009);
INSERT INTO symbols VALUES ('ADVM','Adverum Biotechnologies, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('ADX','Adams Diversified Equity Fund, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('ADXS','Advaxis, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('AE','Adams Resources & Energy, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('AEB','Aegon NV','Finance',NULL);
INSERT INTO symbols VALUES ('AED','Aegon NV','Finance',NULL);
INSERT INTO symbols VALUES ('AEE','Ameren Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('AEF','Aberdeen Emerging Markets Equity Income Fund, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('AEG','Aegon NV','Finance',NULL);
INSERT INTO symbols VALUES ('AEGN','Aegion Corp','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AEH','Aegon NV','Finance',NULL);
INSERT INTO symbols VALUES ('AEHR','Aehr Test Systems','Capital Goods',1997);
INSERT INTO symbols VALUES ('AEIS','Advanced Energy Industries, Inc.','Capital Goods',1995);
INSERT INTO symbols VALUES ('AEL','American Equity Investment Life Holding Company','Finance',2003);
INSERT INTO symbols VALUES ('AEM','Agnico Eagle Mines Limited','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AEMD','Aethlon Medical, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AEO','American Eagle Outfitters, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AEP','American Electric Power Company, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('AEP^B','American Electric Power Company, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('AER','Aercap Holdings N.V.','Technology',2006);
INSERT INTO symbols VALUES ('AERI','Aerie Pharmaceuticals, Inc.','Health Care',2013);
INSERT INTO symbols VALUES ('AES','The AES Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AETI','American Electric Technologies, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('AEY','ADDvantage Technologies Group, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AEYE','AudioEye, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AEZS','AEterna Zentaris Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('AFB','Alliance National Municipal Income Fund Inc',NULL,2002);
INSERT INTO symbols VALUES ('AFC','Ares Capital Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('AFG','American Financial Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('AFGB','American Financial Group, Inc.','Finance',2019);
INSERT INTO symbols VALUES ('AFGE','American Financial Group, Inc.','Finance',2014);
INSERT INTO symbols VALUES ('AFGH','American Financial Group, Inc.','Finance',2015);
INSERT INTO symbols VALUES ('AFH','Atlas Financial Holdings, Inc.','Finance',2013);
INSERT INTO symbols VALUES ('AFHBL','Atlas Financial Holdings, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('AFI','Armstrong Flooring, Inc.','Consumer Non-Durables',2016);
INSERT INTO symbols VALUES ('AFIN','American Finance Trust, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AFINP','American Finance Trust, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AFL','Aflac Incorporated','Finance',NULL);
INSERT INTO symbols VALUES ('AFMD','Affimed N.V.','Health Care',2014);
INSERT INTO symbols VALUES ('AFT','Apollo Senior Floating Rate Fund Inc.',NULL,2011);
INSERT INTO symbols VALUES ('AG','First Majestic Silver Corp.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AGCO','AGCO Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AGD','Aberdeen Global Dynamic Dividend Fund',NULL,2006);
INSERT INTO symbols VALUES ('AGE','AgeX Therapeutics, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('AGEN','Agenus Inc.','Health Care',2000);
INSERT INTO symbols VALUES ('AGFS','AgroFresh Solutions, Inc.','Finance',2014);
INSERT INTO symbols VALUES ('AGFSW','AgroFresh Solutions, Inc.','Finance',2014);
INSERT INTO symbols VALUES ('AGI','Alamos Gold Inc.','Basic Industries',2015);
INSERT INTO symbols VALUES ('AGIO','Agios Pharmaceuticals, Inc.','Health Care',2013);
INSERT INTO symbols VALUES ('AGLE','Aeglea BioTherapeutics, Inc.','Health Care',2016);
INSERT INTO symbols VALUES ('AGM','Federal Agricultural Mortgage Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('AGM.A','Federal Agricultural Mortgage Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('AGMH','AGM Group Holdings Inc.','Technology',2018);
INSERT INTO symbols VALUES ('AGM^A','Federal Agricultural Mortgage Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('AGM^B','Federal Agricultural Mortgage Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('AGM^C','Federal Agricultural Mortgage Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('AGN','Allergan plc.','Health Care',2013);
INSERT INTO symbols VALUES ('AGNC','AGNC Investment Corp.','Consumer Services',2008);
INSERT INTO symbols VALUES ('AGNCB','AGNC Investment Corp.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AGNCM','AGNC Investment Corp.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AGNCN','AGNC Investment Corp.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AGND','WisdomTree Barclays Negative Duration U.S. Aggregate Bond Fund',NULL,NULL);
INSERT INTO symbols VALUES ('AGO','Assured Guaranty Ltd.','Finance',2004);
INSERT INTO symbols VALUES ('AGO^B','Assured Guaranty Ltd.',NULL,NULL);
INSERT INTO symbols VALUES ('AGO^E','Assured Guaranty Ltd.',NULL,NULL);
INSERT INTO symbols VALUES ('AGO^F','Assured Guaranty Ltd.',NULL,NULL);
INSERT INTO symbols VALUES ('AGR','Avangrid, Inc.','Public Utilities',2015);
INSERT INTO symbols VALUES ('AGRO','Adecoagro S.A.','Consumer Non-Durables',2011);
INSERT INTO symbols VALUES ('AGRX','Agile Therapeutics, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('AGS','PlayAGS, Inc.','Consumer Durables',2018);
INSERT INTO symbols VALUES ('AGTC','Applied Genetic Technologies Corporation','Health Care',2014);
INSERT INTO symbols VALUES ('AGX','Argan, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AGYS','Agilysys, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AGZD','WisdomTree Interest Rate Hedged U.S. Aggregate Bond Fund',NULL,NULL);
INSERT INTO symbols VALUES ('AHC','A.H. Belo Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AHH','Armada Hoffler Properties, Inc.','Finance',2013);
INSERT INTO symbols VALUES ('AHL^C','Aspen Insurance Holdings Limited',NULL,NULL);
INSERT INTO symbols VALUES ('AHL^D','Aspen Insurance Holdings Limited',NULL,NULL);
INSERT INTO symbols VALUES ('AHPI','Allied Healthcare Products, Inc.','Health Care',1992);
INSERT INTO symbols VALUES ('AHT','Ashford Hospitality Trust Inc','Consumer Services',2003);
INSERT INTO symbols VALUES ('AHT^D','Ashford Hospitality Trust Inc',NULL,NULL);
INSERT INTO symbols VALUES ('AHT^F','Ashford Hospitality Trust Inc',NULL,NULL);
INSERT INTO symbols VALUES ('AHT^G','Ashford Hospitality Trust Inc',NULL,NULL);
INSERT INTO symbols VALUES ('AHT^H','Ashford Hospitality Trust Inc',NULL,NULL);
INSERT INTO symbols VALUES ('AHT^I','Ashford Hospitality Trust Inc',NULL,NULL);
INSERT INTO symbols VALUES ('AI','Arlington Asset Investment Corp','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AIA','iShares Asia 50 ETF',NULL,NULL);
INSERT INTO symbols VALUES ('AIC','Arlington Asset Investment Corp','Consumer Services',2015);
INSERT INTO symbols VALUES ('AIF','Apollo Tactical Income Fund Inc.',NULL,2013);
INSERT INTO symbols VALUES ('AIG','American International Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('AIG.WS','American International Group, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('AIG^A','American International Group, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('AIHS','Senmiao Technology Limited','Finance',2018);
INSERT INTO symbols VALUES ('AIMC','Altra Industrial Motion Corp.','Capital Goods',2006);
INSERT INTO symbols VALUES ('AIMT','Aimmune Therapeutics, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('AIN','Albany International Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AINC','Ashford Inc.','Consumer Services',2014);
INSERT INTO symbols VALUES ('AINV','Apollo Investment Corporation',NULL,2004);
INSERT INTO symbols VALUES ('AIPT','Precision Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('AIQ','Global X Future Analytics Tech ETF',NULL,NULL);
INSERT INTO symbols VALUES ('AIR','AAR Corp.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AIRG','Airgain, Inc.','Technology',2016);
INSERT INTO symbols VALUES ('AIRI','Air Industries Group','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AIRR','First Trust RBA American Industrial Renaissance ETF',NULL,NULL);
INSERT INTO symbols VALUES ('AIRT','Air T, Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('AIT','Applied Industrial Technologies, Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('AITB','AIT Therapeutics, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('AIV','Apartment Investment and Management Company','Consumer Services',1994);
INSERT INTO symbols VALUES ('AIV^A','Apartment Investment and Management Company',NULL,NULL);
INSERT INTO symbols VALUES ('AIW','Arlington Asset Investment Corp','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AIY','Apollo Investment Corporation',NULL,2013);
INSERT INTO symbols VALUES ('AIZ','Assurant, Inc.','Finance',2004);
INSERT INTO symbols VALUES ('AIZP','Assurant, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('AI^B','Arlington Asset Investment Corp',NULL,NULL);
INSERT INTO symbols VALUES ('AI^C','Arlington Asset Investment Corp',NULL,NULL);
INSERT INTO symbols VALUES ('AJG','Arthur J. Gallagher & Co.','Finance',NULL);
INSERT INTO symbols VALUES ('AJRD','Aerojet Rocketdyne Holdings, Inc. ','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AJX','Great Ajax Corp.','Consumer Services',2015);
INSERT INTO symbols VALUES ('AJXA','Great Ajax Corp.','Consumer Services',2017);
INSERT INTO symbols VALUES ('AKAM','Akamai Technologies, Inc.','Miscellaneous',1999);
INSERT INTO symbols VALUES ('AKBA','Akebia Therapeutics, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('AKCA','Akcea Therapeutics, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('AKER','Akers Biosciences Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('AKG','Asanko Gold Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AKO.A','Embotelladora Andina S.A.',NULL,NULL);
INSERT INTO symbols VALUES ('AKO.B','Embotelladora Andina S.A.',NULL,NULL);
INSERT INTO symbols VALUES ('AKR','Acadia Realty Trust','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AKRX','Akorn, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('AKS','AK Steel Holding Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AKTS','Akoustis Technologies, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('AKTX','Akari Therapeutics Plc','Health Care',NULL);
INSERT INTO symbols VALUES ('AL','Air Lease Corporation','Technology',2011);
INSERT INTO symbols VALUES ('ALAC','Alberton Acquisition Corporation','Finance',2018);
INSERT INTO symbols VALUES ('ALACR','Alberton Acquisition Corporation','Finance',2018);
INSERT INTO symbols VALUES ('ALACU','Alberton Acquisition Corporation','Finance',2018);
INSERT INTO symbols VALUES ('ALACW','Alberton Acquisition Corporation','Finance',2018);
INSERT INTO symbols VALUES ('ALB','Albemarle Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('ALBO','Albireo Pharma, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ALC','Alcon Inc.','Health Care',2019);
INSERT INTO symbols VALUES ('ALCO','Alico, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('ALDR','Alder BioPharmaceuticals, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('ALDX','Aldeyra Therapeutics, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('ALE','Allete, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ALEC','Alector, Inc.','Health Care',2019);
INSERT INTO symbols VALUES ('ALEX','Alexander & Baldwin, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ALG','Alamo Group, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ALGN','Align Technology, Inc.','Health Care',2001);
INSERT INTO symbols VALUES ('ALGR','Allegro Merger Corp.','Finance',2018);
INSERT INTO symbols VALUES ('ALGRR','Allegro Merger Corp.','Finance',2018);
INSERT INTO symbols VALUES ('ALGRU','Allegro Merger Corp.','Finance',2018);
INSERT INTO symbols VALUES ('ALGRW','Allegro Merger Corp.','Finance',2018);
INSERT INTO symbols VALUES ('ALGT','Allegiant Travel Company','Transportation',2006);
INSERT INTO symbols VALUES ('ALIM','Alimera Sciences, Inc.','Health Care',2010);
INSERT INTO symbols VALUES ('ALIT','Alight Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('ALJJ','ALJ Regional Holdings, Inc.','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('ALK','Alaska Air Group, Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('ALKS','Alkermes plc','Health Care',1991);
INSERT INTO symbols VALUES ('ALL','Allstate Corporation (The)','Finance',NULL);
INSERT INTO symbols VALUES ('ALLE','Allegion plc','Consumer Services',2013);
INSERT INTO symbols VALUES ('ALLK','Allakos Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('ALLO','Allogene Therapeutics, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('ALLT','Allot Ltd.','Technology',2006);
INSERT INTO symbols VALUES ('ALLY','Ally Financial Inc.','Finance',2014);
INSERT INTO symbols VALUES ('ALLY^A','Ally Financial Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('ALL^A','Allstate Corporation (The)',NULL,NULL);
INSERT INTO symbols VALUES ('ALL^B','Allstate Corporation (The)',NULL,NULL);
INSERT INTO symbols VALUES ('ALL^D','Allstate Corporation (The)',NULL,NULL);
INSERT INTO symbols VALUES ('ALL^E','Allstate Corporation (The)',NULL,NULL);
INSERT INTO symbols VALUES ('ALL^F','Allstate Corporation (The)',NULL,NULL);
INSERT INTO symbols VALUES ('ALL^G','Allstate Corporation (The)',NULL,NULL);
INSERT INTO symbols VALUES ('ALNA','Allena Pharmaceuticals, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('ALNY','Alnylam Pharmaceuticals, Inc.','Health Care',2004);
INSERT INTO symbols VALUES ('ALO','Alio Gold Inc.','Basic Industries',2011);
INSERT INTO symbols VALUES ('ALOT','AstroNova, Inc.','Technology',1983);
INSERT INTO symbols VALUES ('ALPN','Alpine Immune Sciences, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('ALP^Q','Alabama Power Company',NULL,NULL);
INSERT INTO symbols VALUES ('ALRM','Alarm.com Holdings, Inc.','Technology',2015);
INSERT INTO symbols VALUES ('ALRN','Aileron Therapeutics, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('ALSK','Alaska Communications Systems Group, Inc.','Public Utilities',1999);
INSERT INTO symbols VALUES ('ALSN','Allison Transmission Holdings, Inc.','Capital Goods',2012);
INSERT INTO symbols VALUES ('ALT','Altimmune, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ALTM','Altus Midstream Company','Public Utilities',2017);
INSERT INTO symbols VALUES ('ALTR','Altair Engineering Inc.','Technology',2017);
INSERT INTO symbols VALUES ('ALTY','Global X SuperDividend Alternatives ETF',NULL,NULL);
INSERT INTO symbols VALUES ('ALV','Autoliv, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ALX','Alexander&#39;s, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ALXN','Alexion Pharmaceuticals, Inc.','Health Care',1996);
INSERT INTO symbols VALUES ('ALYA','Alithya Group inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AL^A','Air Lease Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('AM','Antero Midstream Corporation','Public Utilities',2017);
INSERT INTO symbols VALUES ('AMAG','AMAG Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('AMAL','Amalgamated Bank','Consumer Non-Durables',2018);
INSERT INTO symbols VALUES ('AMAT','Applied Materials, Inc.','Technology',1972);
INSERT INTO symbols VALUES ('AMBA','Ambarella, Inc.','Technology',2012);
INSERT INTO symbols VALUES ('AMBC','Ambac Financial Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('AMBCW','Ambac Financial Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('AMBO','Ambow Education Holding Ltd.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AMBR','Amber Road, Inc.','Technology',2014);
INSERT INTO symbols VALUES ('AMC','AMC Entertainment Holdings, Inc.','Consumer Services',2013);
INSERT INTO symbols VALUES ('AMCA','iShares Russell 1000 Pure U.S. Revenue ETF',NULL,NULL);
INSERT INTO symbols VALUES ('AMCI','AMCI Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('AMCIU','AMCI Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('AMCIW','AMCI Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('AMCN','AirMedia Group Inc.','Technology',2007);
INSERT INTO symbols VALUES ('AMCX','AMC Networks Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AMD','Advanced Micro Devices, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AME','AMTEK, Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('AMED','Amedisys Inc','Health Care',NULL);
INSERT INTO symbols VALUES ('AMEH','Apollo Medical Holdings, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AMG','Affiliated Managers Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('AMGN','Amgen Inc.','Health Care',1983);
INSERT INTO symbols VALUES ('AMH','American Homes 4 Rent','Consumer Services',2013);
INSERT INTO symbols VALUES ('AMH^D','American Homes 4 Rent',NULL,NULL);
INSERT INTO symbols VALUES ('AMH^E','American Homes 4 Rent',NULL,NULL);
INSERT INTO symbols VALUES ('AMH^F','American Homes 4 Rent',NULL,NULL);
INSERT INTO symbols VALUES ('AMH^G','American Homes 4 Rent',NULL,NULL);
INSERT INTO symbols VALUES ('AMH^H','American Homes 4 Rent',NULL,NULL);
INSERT INTO symbols VALUES ('AMID','American Midstream Partners, LP','Public Utilities',2011);
INSERT INTO symbols VALUES ('AMKR','Amkor Technology, Inc.','Technology',1998);
INSERT INTO symbols VALUES ('AMN','AMN Healthcare Services Inc','Technology',NULL);
INSERT INTO symbols VALUES ('AMNB','American National Bankshares, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('AMOT','Allied Motion Technologies, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AMP','AMERIPRISE FINANCIAL SERVICES, INC.','Finance',NULL);
INSERT INTO symbols VALUES ('AMPE','Ampio Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('AMPH','Amphastar Pharmaceuticals, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('AMR','Alta Mesa Resources, Inc.','Energy',2017);
INSERT INTO symbols VALUES ('AMRB','American River Bankshares','Finance',NULL);
INSERT INTO symbols VALUES ('AMRC','Ameresco, Inc.','Basic Industries',2010);
INSERT INTO symbols VALUES ('AMRH','Ameri Holdings, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AMRHW','Ameri Holdings, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AMRK','A-Mark Precious Metals, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AMRN','Amarin Corporation plc','Health Care',NULL);
INSERT INTO symbols VALUES ('AMRS','Amyris, Inc.','Basic Industries',2010);
INSERT INTO symbols VALUES ('AMRWW','Alta Mesa Resources, Inc.','Energy',2017);
INSERT INTO symbols VALUES ('AMRX','Amneal Pharmaceuticals, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('AMS','American Shared Hospital Services','Health Care',NULL);
INSERT INTO symbols VALUES ('AMSC','American Superconductor Corporation','Consumer Durables',1991);
INSERT INTO symbols VALUES ('AMSF','AMERISAFE, Inc.','Finance',2005);
INSERT INTO symbols VALUES ('AMSWA','American Software, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AMT','American Tower Corporation (REIT)','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AMTB','Mercantil Bank Holding Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('AMTBB','Mercantil Bank Holding Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('AMTD','TD Ameritrade Holding Corporation','Finance',1997);
INSERT INTO symbols VALUES ('AMTX','Aemetis, Inc','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AMWD','American Woodmark Corporation','Basic Industries',1986);
INSERT INTO symbols VALUES ('AMX','America Movil, S.A.B. de C.V.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('AMZN','Amazon.com, Inc.','Consumer Services',1997);
INSERT INTO symbols VALUES ('AN','AutoNation, Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('ANAB','AnaptysBio, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('ANAT','American National Insurance Company','Finance',NULL);
INSERT INTO symbols VALUES ('ANCN','Anchiano Therapeutics Ltd.','Health Care',2019);
INSERT INTO symbols VALUES ('ANDA','Andina Acquisition Corp. III','Finance',2019);
INSERT INTO symbols VALUES ('ANDAR','Andina Acquisition Corp. III','Finance',2019);
INSERT INTO symbols VALUES ('ANDAU','Andina Acquisition Corp. III','Finance',2019);
INSERT INTO symbols VALUES ('ANDAW','Andina Acquisition Corp. III','Finance',2019);
INSERT INTO symbols VALUES ('ANDE','The Andersons, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ANDX','Andeavor Logistics LP','Energy',2011);
INSERT INTO symbols VALUES ('ANET','Arista Networks, Inc.','Technology',2014);
INSERT INTO symbols VALUES ('ANF','Abercrombie & Fitch Company','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ANFI','Amira Nature Foods Ltd','Consumer Non-Durables',2012);
INSERT INTO symbols VALUES ('ANGI','ANGI Homeservices Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ANGO','AngioDynamics, Inc.','Health Care',2004);
INSERT INTO symbols VALUES ('ANH','Anworth Mortgage Asset  Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ANH^A','Anworth Mortgage Asset  Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('ANH^B','Anworth Mortgage Asset  Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('ANH^C','Anworth Mortgage Asset  Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('ANIK','Anika Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ANIP','ANI Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ANIX','Anixa Biosciences, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ANSS','ANSYS, Inc.','Technology',1996);
INSERT INTO symbols VALUES ('ANTM','Anthem, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ANY','Sphere 3D Corp.','Technology',NULL);
INSERT INTO symbols VALUES ('AOBC','American Outdoor Brands Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AOD','Aberdeen Total Dynamic Dividend Fund',NULL,2007);
INSERT INTO symbols VALUES ('AON','Aon plc','Finance',NULL);
INSERT INTO symbols VALUES ('AOS','A.O Smith Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('AOSL','Alpha and Omega Semiconductor Limited','Technology',2010);
INSERT INTO symbols VALUES ('AP','Ampco-Pittsburgh Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('APA','Apache Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('APAM','Artisan Partners Asset Management Inc.','Finance',2013);
INSERT INTO symbols VALUES ('APC','Anadarko Petroleum Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('APD','Air Products and Chemicals, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('APDN','Applied DNA Sciences Inc','Consumer Services',NULL);
INSERT INTO symbols VALUES ('APDNW','Applied DNA Sciences Inc','Consumer Services',NULL);
INSERT INTO symbols VALUES ('APEI','American Public Education, Inc.','Consumer Services',2007);
INSERT INTO symbols VALUES ('APEN','Apollo Endosurgery, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('APH','Amphenol Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('APHA','Aphria Inc.','Consumer Durables',2018);
INSERT INTO symbols VALUES ('APHB','AmpliPhi Biosciences Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('APLE','Apple Hospitality REIT, Inc.','Consumer Services',2015);
INSERT INTO symbols VALUES ('APLS','Apellis Pharmaceuticals, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('APM','Aptorum Group Limited','Health Care',2018);
INSERT INTO symbols VALUES ('APO','Apollo Global Management, LLC','Finance',2011);
INSERT INTO symbols VALUES ('APOG','Apogee Enterprises, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('APOP','Cellect Biotechnology Ltd.','Health Care',2016);
INSERT INTO symbols VALUES ('APOPW','Cellect Biotechnology Ltd.','Health Care',2016);
INSERT INTO symbols VALUES ('APO^A','Apollo Global Management, LLC',NULL,NULL);
INSERT INTO symbols VALUES ('APO^B','Apollo Global Management, LLC',NULL,NULL);
INSERT INTO symbols VALUES ('APPF','AppFolio, Inc.','Technology',2015);
INSERT INTO symbols VALUES ('APPN','Appian Corporation','Technology',2017);
INSERT INTO symbols VALUES ('APPS','Digital Turbine, Inc.','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('APRN','Blue Apron Holdings, Inc.','Consumer Services',2017);
INSERT INTO symbols VALUES ('APT','Alpha Pro Tech, Ltd.','Health Care',NULL);
INSERT INTO symbols VALUES ('APTO','Aptose Biosciences, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('APTS','Preferred Apartment Communities, Inc.','Consumer Services',2011);
INSERT INTO symbols VALUES ('APTV','Aptiv PLC','Capital Goods',2011);
INSERT INTO symbols VALUES ('APTX','Aptinyx Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('APU','AmeriGas Partners, L.P.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('APVO','Aptevo Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('APWC','Asia Pacific Wire & Cable Corporation Limited','Basic Industries',NULL);
INSERT INTO symbols VALUES ('APY','Apergy Corporation','Technology',2018);
INSERT INTO symbols VALUES ('APYX','Apyx Medical Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('AQ','Aquantia Corp.','Technology',2017);
INSERT INTO symbols VALUES ('AQB','AquaBounty Technologies, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('AQMS','Aqua Metals, Inc.','Basic Industries',2015);
INSERT INTO symbols VALUES ('AQN','Algonquin Power & Utilities Corp.','Public Utilities',2016);
INSERT INTO symbols VALUES ('AQNA','Algonquin Power & Utilities Corp.','Public Utilities',2018);
INSERT INTO symbols VALUES ('AQST','Aquestive Therapeutics, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('AQUA','Evoqua Water Technologies Corp.','Technology',2017);
INSERT INTO symbols VALUES ('AQXP','Aquinox Pharmaceuticals, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('AR','Antero Resources Corporation','Energy',2013);
INSERT INTO symbols VALUES ('ARA','American Renal Associates Holdings, Inc','Health Care',2016);
INSERT INTO symbols VALUES ('ARAV','Aravive, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('ARAY','Accuray Incorporated','Health Care',2007);
INSERT INTO symbols VALUES ('ARC','ARC Document Solutions, Inc.','Consumer Services',2004);
INSERT INTO symbols VALUES ('ARCB','ArcBest Corporation','Transportation',NULL);
INSERT INTO symbols VALUES ('ARCC','Ares Capital Corporation',NULL,2004);
INSERT INTO symbols VALUES ('ARCE','Arco Platform Limited','Consumer Services',2018);
INSERT INTO symbols VALUES ('ARCH','Arch Coal, Inc.','Energy',2016);
INSERT INTO symbols VALUES ('ARCI','Appliance Recycling Centers of America, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ARCO','Arcos Dorados Holdings Inc.','Consumer Services',2011);
INSERT INTO symbols VALUES ('ARCT','Arcturus Therapeutics Ltd.','Health Care',2013);
INSERT INTO symbols VALUES ('ARCW','ARC Group Worldwide, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ARD','Ardagh Group S.A.','Consumer Durables',2017);
INSERT INTO symbols VALUES ('ARDC','Ares Dynamic Credit Allocation Fund, Inc.',NULL,2012);
INSERT INTO symbols VALUES ('ARDS','Aridis Pharmaceuticals Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('ARDX','Ardelyx, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('ARE','Alexandria Real Estate Equities, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AREC','American Resources Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ARES','Ares Management Corporation','Finance',2014);
INSERT INTO symbols VALUES ('ARES^A','Ares Management Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('AREX','Approach Resources Inc.','Energy',2007);
INSERT INTO symbols VALUES ('ARE^D','Alexandria Real Estate Equities, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('ARGD','Argo Group International Holdings, Ltd.','Finance',NULL);
INSERT INTO symbols VALUES ('ARGO','Argo Group International Holdings, Ltd.','Finance',NULL);
INSERT INTO symbols VALUES ('ARGX','argenx SE','Health Care',2017);
INSERT INTO symbols VALUES ('ARI','Apollo Commercial Real Estate Finance','Consumer Services',2009);
INSERT INTO symbols VALUES ('ARI^C','Apollo Commercial Real Estate Finance',NULL,NULL);
INSERT INTO symbols VALUES ('ARKR','Ark Restaurants Corp.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ARL','American Realty Investors, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('ARLO','Arlo Technologies, Inc.','Consumer Non-Durables',2018);
INSERT INTO symbols VALUES ('ARLP','Alliance Resource Partners, L.P.','Energy',1999);
INSERT INTO symbols VALUES ('ARMK','Aramark','Consumer Services',2013);
INSERT INTO symbols VALUES ('ARNA','Arena Pharmaceuticals, Inc.','Health Care',2000);
INSERT INTO symbols VALUES ('ARNC','Arconic Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ARNC^','Arconic Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('AROC','Archrock, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('AROW','Arrow Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('ARPO','Aerpio Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ARQL','ArQule, Inc.','Health Care',1996);
INSERT INTO symbols VALUES ('ARR','ARMOUR Residential REIT, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ARRY','Array BioPharma Inc.','Health Care',2000);
INSERT INTO symbols VALUES ('ARR^A','ARMOUR Residential REIT, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('ARR^B','ARMOUR Residential REIT, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('ARTNA','Artesian Resources Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ARTW','Art&#39;s-Way Manufacturing Co., Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ARTX','Arotech Corporation','Miscellaneous',1994);
INSERT INTO symbols VALUES ('ARVN','Arvinas, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('ARW','Arrow Electronics, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('ARWR','Arrowhead Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ARYA','ARYA Sciences Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('ARYAU','ARYA Sciences Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('ARYAW','ARYA Sciences Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('ASA','ASA Gold and Precious Metals Limited',NULL,NULL);
INSERT INTO symbols VALUES ('ASB','Associated Banc-Corp','Finance',NULL);
INSERT INTO symbols VALUES ('ASB^C','Associated Banc-Corp',NULL,NULL);
INSERT INTO symbols VALUES ('ASB^D','Associated Banc-Corp',NULL,NULL);
INSERT INTO symbols VALUES ('ASB^E','Associated Banc-Corp',NULL,NULL);
INSERT INTO symbols VALUES ('ASC','Ardmore Shipping Corporation','Transportation',2013);
INSERT INTO symbols VALUES ('ASCMA','Ascent Capital Group, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ASET','FlexShares Real Assets Allocation Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('ASFI','Asta Funding, Inc.','Finance',1995);
INSERT INTO symbols VALUES ('ASG','Liberty All-Star Growth Fund, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('ASGN','ASGN Incorporated','Technology',1992);
INSERT INTO symbols VALUES ('ASH','Ashland Global Holdings Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('ASIX','AdvanSix Inc.','Basic Industries',2016);
INSERT INTO symbols VALUES ('ASLN','ASLAN Pharmaceuticals Limited','Health Care',2018);
INSERT INTO symbols VALUES ('ASM','Avino Silver','Basic Industries',NULL);
INSERT INTO symbols VALUES ('ASMB','Assembly Biosciences, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ASML','ASML Holding N.V.','Technology',1995);
INSERT INTO symbols VALUES ('ASNA','Ascena Retail Group, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ASND','Ascendis Pharma A/S','Health Care',2015);
INSERT INTO symbols VALUES ('ASPN','Aspen Aerogels, Inc.','Consumer Services',2014);
INSERT INTO symbols VALUES ('ASPS','Altisource Portfolio Solutions S.A.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ASPU','Aspen Group Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ASR','Grupo Aeroportuario del Sureste, S.A. de C.V.','Transportation',NULL);
INSERT INTO symbols VALUES ('ASRT','Assertio Therapeutics, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ASRV','AmeriServ Financial Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('ASRVP','AmeriServ Financial Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('ASTC','Astrotech Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ASTE','Astec Industries, Inc.','Capital Goods',1986);
INSERT INTO symbols VALUES ('ASUR','Asure Software Inc','Technology',NULL);
INSERT INTO symbols VALUES ('ASV','ASV Holdings, Inc.','Capital Goods',2017);
INSERT INTO symbols VALUES ('ASX','ASE Technology Holding Co., Ltd.','Technology',2018);
INSERT INTO symbols VALUES ('ASYS','Amtech Systems, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AT','Atlantic Power Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('ATAI','ATA Inc.','Consumer Services',2008);
INSERT INTO symbols VALUES ('ATAX','America First Multifamily Investors, L.P.','Finance',NULL);
INSERT INTO symbols VALUES ('ATEC','Alphatec Holdings, Inc.','Health Care',2006);
INSERT INTO symbols VALUES ('ATEN','A10 Networks, Inc.','Technology',2014);
INSERT INTO symbols VALUES ('ATEST','NASDAQ TEST STOCK',NULL,NULL);
INSERT INTO symbols VALUES ('ATEST.A','NASDAQ TEST STOCK',NULL,NULL);
INSERT INTO symbols VALUES ('ATEST.B','NASDAQ TEST STOCK',NULL,NULL);
INSERT INTO symbols VALUES ('ATEST.C','NASDAQ TEST STOCK',NULL,NULL);
INSERT INTO symbols VALUES ('ATGE','Adtalem Global Education Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ATH','Athene Holding Ltd.','Finance',2016);
INSERT INTO symbols VALUES ('ATHE','Alterity Therapeutics Limited','Health Care',NULL);
INSERT INTO symbols VALUES ('ATHM','Autohome Inc.','Technology',2013);
INSERT INTO symbols VALUES ('ATHX','Athersys, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ATI','Allegheny Technologies Incorporated','Basic Industries',NULL);
INSERT INTO symbols VALUES ('ATIF','ATIF Holdings Limited',NULL,2019);
INSERT INTO symbols VALUES ('ATIS','Attis Industries Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('ATISW','Attis Industries Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('ATKR','Atkore International Group Inc.','Miscellaneous',2016);
INSERT INTO symbols VALUES ('ATLC','Atlanticus Holdings Corporation','Finance',1995);
INSERT INTO symbols VALUES ('ATLO','Ames National Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('ATNI','ATN International, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ATNM','Actinium Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ATNX','Athenex, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('ATO','Atmos Energy Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ATOM','Atomera Incorporated','Technology',2016);
INSERT INTO symbols VALUES ('ATOS','Atossa Genetics Inc.','Health Care',2012);
INSERT INTO symbols VALUES ('ATR','AptarGroup, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('ATRA','Atara Biotherapeutics, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('ATRC','AtriCure, Inc.','Health Care',2005);
INSERT INTO symbols VALUES ('ATRI','Atrion Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('ATRO','Astronics Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ATRS','Antares Pharma, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ATSG','Air Transport Services Group, Inc','Transportation',NULL);
INSERT INTO symbols VALUES ('ATTO','Atento S.A.','Public Utilities',2014);
INSERT INTO symbols VALUES ('ATTU','Attunity Ltd.','Technology',NULL);
INSERT INTO symbols VALUES ('ATU','Actuant Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('ATUS','Altice USA, Inc.','Consumer Services',2017);
INSERT INTO symbols VALUES ('ATV','Acorn International, Inc.','Consumer Services',2007);
INSERT INTO symbols VALUES ('ATVI','Activision Blizzard, Inc','Technology',NULL);
INSERT INTO symbols VALUES ('ATXI','Avenue Therapeutics, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('AU','AngloGold Ashanti Limited','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AUBN','Auburn National Bancorporation, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('AUDC','AudioCodes Ltd.','Public Utilities',1999);
INSERT INTO symbols VALUES ('AUG','Auryn Resources Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AUMN','Golden Minerals Company','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AUO','AU Optronics Corp','Technology',2002);
INSERT INTO symbols VALUES ('AUPH','Aurinia Pharmaceuticals Inc','Health Care',NULL);
INSERT INTO symbols VALUES ('AUTL','Autolus Therapeutics plc','Health Care',2018);
INSERT INTO symbols VALUES ('AUTO','AutoWeb, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AUY','Yamana Gold Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AVA','Avista Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('AVAL','Grupo Aval Acciones y Valores S.A.','Finance',2014);
INSERT INTO symbols VALUES ('AVAV','AeroVironment, Inc.','Capital Goods',2007);
INSERT INTO symbols VALUES ('AVB','AvalonBay Communities, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AVCO','Avalon GloboCare Corp.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AVD','American Vanguard Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AVDL','Avadel Pharmaceuticals plc','Health Care',NULL);
INSERT INTO symbols VALUES ('AVDR','Avedro, Inc','Health Care',2019);
INSERT INTO symbols VALUES ('AVEO','AVEO Pharmaceuticals, Inc.','Health Care',2010);
INSERT INTO symbols VALUES ('AVGO','Broadcom Inc.','Technology',2009);
INSERT INTO symbols VALUES ('AVGR','Avinger, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('AVH','Avianca Holdings S.A.','Transportation',2013);
INSERT INTO symbols VALUES ('AVID','Avid Technology, Inc.','Technology',1993);
INSERT INTO symbols VALUES ('AVK','Advent Claymore Convertible Securities and Income Fund',NULL,2003);
INSERT INTO symbols VALUES ('AVLR','Avalara, Inc.','Technology',2018);
INSERT INTO symbols VALUES ('AVNS','Avanos Medical, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('AVNW','Aviat Networks, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('AVP','Avon Products, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('AVRO','AVROBIO, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('AVT','Avnet, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('AVX','AVX Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AVXL','Anavex Life Sciences Corp.','Health Care',NULL);
INSERT INTO symbols VALUES ('AVY','Avery Dennison Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('AVYA','Avaya Holdings Corp.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('AWF','Alliance World Dollar Government Fund II',NULL,1993);
INSERT INTO symbols VALUES ('AWI','Armstrong World Industries Inc','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('AWK','American Water Works','Public Utilities',2008);
INSERT INTO symbols VALUES ('AWP','Aberdeen Global Premier Properties Fund',NULL,2007);
INSERT INTO symbols VALUES ('AWR','American States Water Company','Public Utilities',NULL);
INSERT INTO symbols VALUES ('AWRE','Aware, Inc.','Technology',1996);
INSERT INTO symbols VALUES ('AWSM','Cool Holdings Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('AWX','Avalon Holdings Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('AX','Axos Financial, Inc.','Finance',2018);
INSERT INTO symbols VALUES ('AXAS','Abraxas Petroleum Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('AXDX','Accelerate Diagnostics, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AXE','Anixter International Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('AXGN','AxoGen, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('AXGT','Axovant Gene Therapies Ltd.','Health Care',NULL);
INSERT INTO symbols VALUES ('AXL','American Axle & Manufacturing Holdings, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('AXNX','Axonics Modulation Technologies, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('AXO','Axos Financial, Inc.','Finance',2018);
INSERT INTO symbols VALUES ('AXP','American Express Company','Finance',NULL);
INSERT INTO symbols VALUES ('AXR','AMREP Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AXS','Axis Capital Holdings Limited','Finance',2003);
INSERT INTO symbols VALUES ('AXSM','Axsome Therapeutics, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('AXS^D','Axis Capital Holdings Limited',NULL,NULL);
INSERT INTO symbols VALUES ('AXS^E','Axis Capital Holdings Limited',NULL,NULL);
INSERT INTO symbols VALUES ('AXTA','Axalta Coating Systems Ltd.','Basic Industries',2014);
INSERT INTO symbols VALUES ('AXTI','AXT Inc','Technology',1998);
INSERT INTO symbols VALUES ('AXU','Alexco Resource Corp','Basic Industries',NULL);
INSERT INTO symbols VALUES ('AY','Atlantica Yield plc','Public Utilities',2014);
INSERT INTO symbols VALUES ('AYI','Acuity Brands, Inc. ','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('AYR','Aircastle Limited','Technology',2006);
INSERT INTO symbols VALUES ('AYTU','Aytu BioScience, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('AYX','Alteryx, Inc.','Technology',2017);
INSERT INTO symbols VALUES ('AZN','Astrazeneca PLC','Health Care',NULL);
INSERT INTO symbols VALUES ('AZO','AutoZone, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('AZPN','Aspen Technology, Inc.','Technology',1994);
INSERT INTO symbols VALUES ('AZRE','Azure Power Global Limited','Public Utilities',2016);
INSERT INTO symbols VALUES ('AZRX','AzurRx BioPharma, Inc.','Health Care',2016);
INSERT INTO symbols VALUES ('AZUL','Azul S.A.','Transportation',2017);
INSERT INTO symbols VALUES ('AZZ','AZZ Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('B','Barnes Group, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BA','Boeing Company (The)','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BABA','Alibaba Group Holding Limited','Miscellaneous',2014);
INSERT INTO symbols VALUES ('BABY','Natus Medical Incorporated','Health Care',2001);
INSERT INTO symbols VALUES ('BAC','Bank of America Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BAC^A','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BAC^B','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BAC^C','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BAC^E','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BAC^K','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BAC^L','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BAC^W','Bank of America Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BAC^Y','Bank of America Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BAF','BlackRock Income Investment Quality Trust',NULL,2002);
INSERT INTO symbols VALUES ('BAH','Booz Allen Hamilton Holding Corporation','Consumer Services',2010);
INSERT INTO symbols VALUES ('BAK','Braskem S.A.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('BAM','Brookfield Asset Management Inc','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BANC','Banc of California, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BANC^D','Banc of California, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BANC^E','Banc of California, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BAND','Bandwidth Inc.','Technology',2017);
INSERT INTO symbols VALUES ('BANF','BancFirst Corporation','Finance',1993);
INSERT INTO symbols VALUES ('BANFP','BancFirst Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BANR','Banner Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BANX','StoneCastle Financial Corp',NULL,NULL);
INSERT INTO symbols VALUES ('BAP','Credicorp Ltd.','Finance',NULL);
INSERT INTO symbols VALUES ('BAS','Basic Energy Services, Inc.','Energy',2005);
INSERT INTO symbols VALUES ('BASI','Bioanalytical Systems, Inc.','Health Care',1997);
INSERT INTO symbols VALUES ('BATRA','Liberty Media Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BATRK','Liberty Media Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BAX','Baxter International Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BB','BlackBerry Limited','Technology',NULL);
INSERT INTO symbols VALUES ('BBBY','Bed Bath & Beyond Inc.','Consumer Services',1992);
INSERT INTO symbols VALUES ('BBC','Virtus LifeSci Biotech Clinical Trials ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BBCP','Concrete Pumping Holdings, Inc. ','Basic Industries',2017);
INSERT INTO symbols VALUES ('BBD','Banco Bradesco Sa','Finance',NULL);
INSERT INTO symbols VALUES ('BBDC','Barings BDC, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BBDO','Banco Bradesco Sa','Finance',NULL);
INSERT INTO symbols VALUES ('BBF','BlackRock Municipal Income Investment Trust',NULL,NULL);
INSERT INTO symbols VALUES ('BBGI','Beasley Broadcast Group, Inc.','Consumer Services',2000);
INSERT INTO symbols VALUES ('BBH','VanEck Vectors Biotech ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BBK','Blackrock Municipal Bond Trust',NULL,2002);
INSERT INTO symbols VALUES ('BBL','BHP Group Plc','Energy',NULL);
INSERT INTO symbols VALUES ('BBN','BalckRock Taxable Municipal Bond Trust',NULL,2010);
INSERT INTO symbols VALUES ('BBP','Virtus LifeSci Biotech Products ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BBRX','Braeburn Pharmaceuticals, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BBSI','Barrett Business Services, Inc.','Technology',1993);
INSERT INTO symbols VALUES ('BBT','BB&T Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BBT^D','BB&T Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BBT^E','BB&T Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BBT^F','BB&T Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BBT^G','BB&T Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BBT^H','BB&T Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BBU','Brookfield Business Partners L.P.','Basic Industries',2016);
INSERT INTO symbols VALUES ('BBVA','Banco Bilbao Viscaya Argentaria S.A.','Finance',NULL);
INSERT INTO symbols VALUES ('BBW','Build-A-Bear Workshop, Inc.','Consumer Services',2004);
INSERT INTO symbols VALUES ('BBX','BBX Capital Corporation','Finance',2017);
INSERT INTO symbols VALUES ('BBY','Best Buy Co., Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BC','Brunswick Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('BCAC','Bison Capital Acquisition Corp.','Health Care',2017);
INSERT INTO symbols VALUES ('BCACR','Bison Capital Acquisition Corp.','Health Care',2017);
INSERT INTO symbols VALUES ('BCACU','Bison Capital Acquisition Corp.','Health Care',2017);
INSERT INTO symbols VALUES ('BCACW','Bison Capital Acquisition Corp.','Health Care',2017);
INSERT INTO symbols VALUES ('BCBP','BCB Bancorp, Inc. (NJ)','Finance',2005);
INSERT INTO symbols VALUES ('BCC','Boise Cascade, L.L.C.','Consumer Services',2013);
INSERT INTO symbols VALUES ('BCE','BCE, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('BCEI','Bonanza Creek Energy, Inc.','Energy',2017);
INSERT INTO symbols VALUES ('BCH','Banco De Chile','Finance',NULL);
INSERT INTO symbols VALUES ('BCLI','Brainstorm Cell Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BCML','BayCom Corp','Finance',2018);
INSERT INTO symbols VALUES ('BCNA','Reality Shares Nasdaq NexGen Economy China ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BCO','Brink&#39;s Company (The)','Transportation',NULL);
INSERT INTO symbols VALUES ('BCOM','B Communications Ltd.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BCOR','Blucora, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BCOV','Brightcove Inc.','Technology',2012);
INSERT INTO symbols VALUES ('BCOW','1895 Bancorp of Wisconsin, Inc.','Finance',2019);
INSERT INTO symbols VALUES ('BCPC','Balchem Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('BCRH','Blue Capital Reinsurance Holdings Ltd.','Finance',2013);
INSERT INTO symbols VALUES ('BCRX','BioCryst Pharmaceuticals, Inc.','Health Care',1994);
INSERT INTO symbols VALUES ('BCS','Barclays PLC','Finance',NULL);
INSERT INTO symbols VALUES ('BCSF','Bain Capital Specialty Finance, Inc.',NULL,2018);
INSERT INTO symbols VALUES ('BCTF','Bancorp 34, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BCV','Bancroft Fund Limited',NULL,NULL);
INSERT INTO symbols VALUES ('BCV^A','Bancroft Fund Limited',NULL,NULL);
INSERT INTO symbols VALUES ('BCX','BlackRock Resources',NULL,2011);
INSERT INTO symbols VALUES ('BC^A','Brunswick Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BC^B','Brunswick Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BC^C','Brunswick Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BDC','Belden Inc','Basic Industries',NULL);
INSERT INTO symbols VALUES ('BDC^B','Belden Inc',NULL,NULL);
INSERT INTO symbols VALUES ('BDGE','Bridge Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BDJ','Blackrock Enhanced Equity Dividend Trust',NULL,2005);
INSERT INTO symbols VALUES ('BDL','Flanigan&#39;s Enterprises, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BDN','Brandywine Realty Trust','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BDR','Blonder Tongue Laboratories, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('BDSI','BioDelivery Sciences International, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BDX','Becton, Dickinson and Company','Health Care',NULL);
INSERT INTO symbols VALUES ('BDXA','Becton, Dickinson and Company','Health Care',NULL);
INSERT INTO symbols VALUES ('BE','Bloom Energy Corporation','Energy',2018);
INSERT INTO symbols VALUES ('BEAT','BioTelemetry, Inc.','Health Care',2008);
INSERT INTO symbols VALUES ('BECN','Beacon Roofing Supply, Inc.','Consumer Services',2004);
INSERT INTO symbols VALUES ('BEDU','Bright Scholar Education Holdings Limited','Consumer Services',2017);
INSERT INTO symbols VALUES ('BELFA','Bel Fuse Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BELFB','Bel Fuse Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BEN','Franklin Resources, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BEP','Brookfield Renewable Partners L.P.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('BERY','Berry Global Group, Inc.','Consumer Non-Durables',2012);
INSERT INTO symbols VALUES ('BEST','BEST Inc.','Transportation',2017);
INSERT INTO symbols VALUES ('BF.A','Brown Forman Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BF.B','Brown Forman Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BFAM','Bright Horizons Family Solutions Inc.','Miscellaneous',2013);
INSERT INTO symbols VALUES ('BFC','Bank First National Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BFIN','BankFinancial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BFIT','Global X Health & Wellness Thematic ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BFK','BlackRock Municipal Income Trust',NULL,NULL);
INSERT INTO symbols VALUES ('BFL','BankFlorida',NULL,NULL);
INSERT INTO symbols VALUES ('BFO','Blackrock Florida Municipal 2020 Term Trust',NULL,2003);
INSERT INTO symbols VALUES ('BFR','BBVA Banco Frances S.A.','Finance',NULL);
INSERT INTO symbols VALUES ('BFRA','Biofrontera AG','Health Care',2018);
INSERT INTO symbols VALUES ('BFS','Saul Centers, Inc.','Consumer Services',1993);
INSERT INTO symbols VALUES ('BFST','Business First Bancshares, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BFS^C','Saul Centers, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BFS^D','Saul Centers, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BFY','BlackRock New York Municipal Income Trust II',NULL,2002);
INSERT INTO symbols VALUES ('BFZ','BlackRock California Municipal Income Trust',NULL,2001);
INSERT INTO symbols VALUES ('BG','Bunge Limited','Consumer Non-Durables',2001);
INSERT INTO symbols VALUES ('BGB','Blackstone / GSO Strategic Credit Fund',NULL,2012);
INSERT INTO symbols VALUES ('BGCP','BGC Partners, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BGFV','Big 5 Sporting Goods Corporation','Consumer Services',2002);
INSERT INTO symbols VALUES ('BGG','Briggs & Stratton Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('BGH','Babson Global Short Duration High Yield Fund',NULL,2012);
INSERT INTO symbols VALUES ('BGI','Birks Group Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BGIO','BlackRock 2022 Global Income Opportunity Trust',NULL,2017);
INSERT INTO symbols VALUES ('BGNE','BeiGene, Ltd.','Health Care',2016);
INSERT INTO symbols VALUES ('BGR','BlackRock Energy and Resources Trust',NULL,2004);
INSERT INTO symbols VALUES ('BGRN','iShares Global Green Bond ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BGS','B&G Foods, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('BGSF','BG Staffing Inc','Technology',NULL);
INSERT INTO symbols VALUES ('BGT','Blackrock Global',NULL,2004);
INSERT INTO symbols VALUES ('BGX','Blackstone GSO Long Short Credit Income Fund',NULL,2011);
INSERT INTO symbols VALUES ('BGY','BLACKROCK INTERNATIONAL, LTD.',NULL,2007);
INSERT INTO symbols VALUES ('BH','Biglari Holdings Inc.','Consumer Services',2018);
INSERT INTO symbols VALUES ('BH.A','Biglari Holdings Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BHB','Bar Harbor Bankshares, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BHC','Bausch Health Companies Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BHE','Benchmark Electronics, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('BHF','Brighthouse Financial, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BHFAL','Brighthouse Financial, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BHFAP','Brighthouse Financial, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BHGE','Baker Hughes, a GE company','Energy',2017);
INSERT INTO symbols VALUES ('BHK','Blackrock Core Bond Trust',NULL,2001);
INSERT INTO symbols VALUES ('BHLB','Berkshire Hills Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BHP','BHP Group Limited','Basic Industries',NULL);
INSERT INTO symbols VALUES ('BHR','Braemar Hotels & Resorts Inc.','Consumer Services',2013);
INSERT INTO symbols VALUES ('BHR^B','Braemar Hotels & Resorts Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BHR^D','Braemar Hotels & Resorts Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BHTG','BioHiTech Global, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BHV','BlackRock Virginia Municipal Bond Trust',NULL,2002);
INSERT INTO symbols VALUES ('BHVN','Biohaven Pharmaceutical Holding Company Ltd.','Health Care',2017);
INSERT INTO symbols VALUES ('BIB','ProShares Ultra Nasdaq Biotechnology',NULL,NULL);
INSERT INTO symbols VALUES ('BICK','First Trust BICK Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('BID','Sotheby&#39;s','Miscellaneous',1988);
INSERT INTO symbols VALUES ('BIDU','Baidu, Inc.','Technology',2005);
INSERT INTO symbols VALUES ('BIF','USLIFE Income Fund, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BIG','Big Lots, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BIIB','Biogen Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BILI','Bilibili Inc.','Technology',2018);
INSERT INTO symbols VALUES ('BIMI','NF Energy Saving Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BIO','Bio-Rad Laboratories, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BIO.B','Bio-Rad Laboratories, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BIOC','Biocept, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('BIOL','Biolase, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BIOS','BioScrip, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BIOX','Bioceres Crop Solutions Corp.','Basic Industries',2018);
INSERT INTO symbols VALUES ('BIOX.WS','Bioceres Crop Solutions Corp.',NULL,NULL);
INSERT INTO symbols VALUES ('BIP','Brookfield Infrastructure Partners LP','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BIQI','BIQI International Holdings Corporation','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('BIS','ProShares UltraShort Nasdaq Biotechnology',NULL,NULL);
INSERT INTO symbols VALUES ('BIT','BlackRock Multi-Sector Income Trust',NULL,2013);
INSERT INTO symbols VALUES ('BITA','Bitauto Holdings Limited','Technology',2010);
INSERT INTO symbols VALUES ('BJ','BJ&#39;s Wholesale Club Holdings, Inc.','Consumer Services',2018);
INSERT INTO symbols VALUES ('BJRI','BJ&#39;s Restaurants, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BK','Bank Of New York Mellon Corporation (The)','Finance',NULL);
INSERT INTO symbols VALUES ('BKCC','BlackRock Capital Investment Corporation',NULL,2007);
INSERT INTO symbols VALUES ('BKCH','AdvisorShares Sabretooth ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BKD','Brookdale Senior Living Inc.','Health Care',2005);
INSERT INTO symbols VALUES ('BKE','Buckle, Inc. (The)','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BKEP','Blueknight Energy Partners L.P., L.L.C.','Energy',2011);
INSERT INTO symbols VALUES ('BKEPP','Blueknight Energy Partners L.P., L.L.C.','Energy',NULL);
INSERT INTO symbols VALUES ('BKH','Black Hills Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('BKI','Black Knight, Inc.','Technology',2017);
INSERT INTO symbols VALUES ('BKJ','Bancorp of New Jersey, Inc','Finance',NULL);
INSERT INTO symbols VALUES ('BKK','Blackrock Municipal 2020 Term Trust',NULL,2003);
INSERT INTO symbols VALUES ('BKN','BlackRock Investment Quality Municipal Trust Inc. (The)',NULL,1993);
INSERT INTO symbols VALUES ('BKNG','Booking Holdings Inc.','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('BKS','Barnes & Noble, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BKSC','Bank of South Carolina Corp.','Finance',NULL);
INSERT INTO symbols VALUES ('BKT','BlackRock Income Trust Inc. (The)',NULL,NULL);
INSERT INTO symbols VALUES ('BKTI','BK Technologies Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('BKU','BankUnited, Inc.','Finance',2011);
INSERT INTO symbols VALUES ('BKYI','BIO-key International, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('BK^C','Bank Of New York Mellon Corporation (The)',NULL,NULL);
INSERT INTO symbols VALUES ('BL','BlackLine, Inc.','Technology',2016);
INSERT INTO symbols VALUES ('BLBD','Blue Bird Corporation','Capital Goods',2014);
INSERT INTO symbols VALUES ('BLCM','Bellicum Pharmaceuticals, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('BLCN','Reality Shares Nasdaq NextGen Economy ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BLD','TopBuild Corp.','Basic Industries',2015);
INSERT INTO symbols VALUES ('BLDP','Ballard Power Systems, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('BLDR','Builders FirstSource, Inc.','Consumer Services',2005);
INSERT INTO symbols VALUES ('BLE','BlackRock Municipal Income Trust II',NULL,2002);
INSERT INTO symbols VALUES ('BLFS','BioLife Solutions, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BLIN','Bridgeline Digital, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('BLK','BlackRock, Inc.','Finance',1999);
INSERT INTO symbols VALUES ('BLKB','Blackbaud, Inc.','Technology',2004);
INSERT INTO symbols VALUES ('BLL','Ball Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('BLMN','Bloomin&#39; Brands, Inc.','Consumer Services',2012);
INSERT INTO symbols VALUES ('BLNK','Blink Charging Co.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('BLNKW','Blink Charging Co.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('BLPH','Bellerophon Therapeutics, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('BLRX','BioLineRx Ltd.','Health Care',NULL);
INSERT INTO symbols VALUES ('BLUE','bluebird bio, Inc.','Health Care',2013);
INSERT INTO symbols VALUES ('BLW','Citigroup Inc.','Finance',2003);
INSERT INTO symbols VALUES ('BLX','Banco Latinoamericano de Comercio Exterior, S.A.','Finance',1992);
INSERT INTO symbols VALUES ('BMA','Macro Bank Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BMCH','BMC Stock Holdings, Inc.','Consumer Services',2013);
INSERT INTO symbols VALUES ('BME','Blackrock Health Sciences Trust',NULL,2005);
INSERT INTO symbols VALUES ('BMI','Badger Meter, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BMLP','Bank Of Montreal','Finance',NULL);
INSERT INTO symbols VALUES ('BML^G','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BML^H','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BML^J','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BML^L','Bank of America Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('BMO','Bank Of Montreal','Finance',NULL);
INSERT INTO symbols VALUES ('BMRA','Biomerica, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BMRC','Bank of Marin Bancorp','Finance',NULL);
INSERT INTO symbols VALUES ('BMRN','BioMarin Pharmaceutical Inc.','Health Care',1999);
INSERT INTO symbols VALUES ('BMS','Bemis Company, Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('BMTC','Bryn Mawr Bank Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BMY','Bristol-Myers Squibb Company','Health Care',NULL);
INSERT INTO symbols VALUES ('BND','Vanguard Total Bond Market ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BNDW','Vanguard Total World Bond ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BNDX','Vanguard Total International Bond ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BNED','Barnes & Noble Education, Inc','Consumer Services',2015);
INSERT INTO symbols VALUES ('BNFT','Benefitfocus, Inc.','Technology',2013);
INSERT INTO symbols VALUES ('BNGO','Bionano Genomics, Inc.','Capital Goods',2018);
INSERT INTO symbols VALUES ('BNGOW','Bionano Genomics, Inc.','Capital Goods',2018);
INSERT INTO symbols VALUES ('BNKL','Bionik Laboratories Corp.',NULL,NULL);
INSERT INTO symbols VALUES ('BNS','Bank of Nova Scotia (The)','Finance',NULL);
INSERT INTO symbols VALUES ('BNSO','Bonso Electronics International, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BNTC','Benitec Biopharma Limited','Health Care',2015);
INSERT INTO symbols VALUES ('BNTCW','Benitec Biopharma Limited','Health Care',2015);
INSERT INTO symbols VALUES ('BNY','BlackRock New York Investment Quality Municipal Trust Inc. (Th',NULL,NULL);
INSERT INTO symbols VALUES ('BOCH','Bank of Commerce Holdings (CA)','Finance',NULL);
INSERT INTO symbols VALUES ('BOE','Blackrock Global',NULL,2005);
INSERT INTO symbols VALUES ('BOH','Bank of Hawaii Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BOKF','BOK Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BOKFL','BOK Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BOLD','Audentes Therapeutics, Inc.','Health Care',2016);
INSERT INTO symbols VALUES ('BOMN','Boston Omaha Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BOOM','DMC Global Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BOOT','Boot Barn Holdings, Inc.','Consumer Services',2014);
INSERT INTO symbols VALUES ('BORN','China New Borun Corporation','Consumer Non-Durables',2010);
INSERT INTO symbols VALUES ('BOSC','B.O.S. Better Online Solutions','Technology',NULL);
INSERT INTO symbols VALUES ('BOTJ','Bank of the James Financial Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BOTZ','Global X Robotics & Artificial Intelligence ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BOX','Box, Inc.','Technology',2015);
INSERT INTO symbols VALUES ('BOXL','Boxlight Corporation','Miscellaneous',2017);
INSERT INTO symbols VALUES ('BP','BP p.l.c.','Energy',NULL);
INSERT INTO symbols VALUES ('BPFH','Boston Private Financial Holdings, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BPL','Buckeye Partners L.P.','Energy',1986);
INSERT INTO symbols VALUES ('BPMC','Blueprint Medicines Corporation','Health Care',2015);
INSERT INTO symbols VALUES ('BPMP','BP Midstream Partners LP','Energy',2017);
INSERT INTO symbols VALUES ('BPMX','BioPharmX Corporation','Health Care',2015);
INSERT INTO symbols VALUES ('BPOP','Popular, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BPOPM','Popular, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BPOPN','Popular, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BPR','Brookfield Property REIT Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BPRAP','Brookfield Property REIT Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BPRN','The Bank of Princeton',NULL,NULL);
INSERT INTO symbols VALUES ('BPT','BP Prudhoe Bay Royalty Trust','Energy',NULL);
INSERT INTO symbols VALUES ('BPTH','Bio-Path Holdings, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BPY','Brookfield Property Partners L.P.','Finance',NULL);
INSERT INTO symbols VALUES ('BPYPP','Brookfield Property Partners L.P.','Finance',NULL);
INSERT INTO symbols VALUES ('BQH','Blackrock New York Municipal Bond Trust',NULL,2002);
INSERT INTO symbols VALUES ('BR','Broadridge Financial Solutions, Inc.','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('BRAC','Black Ridge Acquisition Corp.','Consumer Services',2017);
INSERT INTO symbols VALUES ('BRACR','Black Ridge Acquisition Corp.','Consumer Services',2017);
INSERT INTO symbols VALUES ('BRACU','Black Ridge Acquisition Corp.','Consumer Services',2017);
INSERT INTO symbols VALUES ('BRACW','Black Ridge Acquisition Corp.','Consumer Services',2017);
INSERT INTO symbols VALUES ('BRC','Brady Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('BREW','Craft Brew Alliance, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('BRFS','BRF S.A.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('BRG','Bluerock Residential Growth REIT, Inc.','Consumer Services',2014);
INSERT INTO symbols VALUES ('BRG^A','Bluerock Residential Growth REIT, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BRG^C','Bluerock Residential Growth REIT, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BRG^D','Bluerock Residential Growth REIT, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BRID','Bridgford Foods Corporation','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('BRK.A','Berkshire Hathaway Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BRK.B','Berkshire Hathaway Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BRKL','Brookline Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BRKR','Bruker Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BRKS','Brooks Automation, Inc.','Technology',1995);
INSERT INTO symbols VALUES ('BRN','Barnwell Industries, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('BRO','Brown & Brown, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('BROG','Twelve Seas Investment Company','Finance',2018);
INSERT INTO symbols VALUES ('BROGR','Twelve Seas Investment Company','Finance',2018);
INSERT INTO symbols VALUES ('BROGU','Twelve Seas Investment Company','Finance',2018);
INSERT INTO symbols VALUES ('BROGW','Twelve Seas Investment Company','Finance',2018);
INSERT INTO symbols VALUES ('BRPA','Big Rock Partners Acquisition Corp.','Finance',2017);
INSERT INTO symbols VALUES ('BRPAR','Big Rock Partners Acquisition Corp.','Finance',2017);
INSERT INTO symbols VALUES ('BRPAU','Big Rock Partners Acquisition Corp.','Finance',2017);
INSERT INTO symbols VALUES ('BRPAW','Big Rock Partners Acquisition Corp.','Finance',2017);
INSERT INTO symbols VALUES ('BRPM','B. Riley Principal Merger Corp.','Finance',2019);
INSERT INTO symbols VALUES ('BRPM.U','B. Riley Principal Merger Corp.',NULL,2019);
INSERT INTO symbols VALUES ('BRPM.WS','B. Riley Principal Merger Corp.',NULL,NULL);
INSERT INTO symbols VALUES ('BRQS','Borqs Technologies, Inc. ','Technology',2015);
INSERT INTO symbols VALUES ('BRS','Bristow Group Inc','Transportation',NULL);
INSERT INTO symbols VALUES ('BRSS','Global Brass and Copper Holdings, Inc.','Capital Goods',2013);
INSERT INTO symbols VALUES ('BRT','BRT Apartments Corp.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BRX','Brixmor Property Group Inc.','Consumer Services',2013);
INSERT INTO symbols VALUES ('BRY','Berry Petroleum Corporation','Energy',2018);
INSERT INTO symbols VALUES ('BSA','BrightSphere Investment Group plc','Finance',2016);
INSERT INTO symbols VALUES ('BSAC','Banco Santander Chile','Finance',NULL);
INSERT INTO symbols VALUES ('BSBR','Banco Santander Brasil SA','Finance',2009);
INSERT INTO symbols VALUES ('BSD','BlackRock Strategic Municipal Trust Inc. (The)',NULL,1999);
INSERT INTO symbols VALUES ('BSE','Blackrock New York Municipal Income Quality Trust',NULL,2002);
INSERT INTO symbols VALUES ('BSET','Bassett Furniture Industries, Incorporated','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('BSGM','BioSig Technologies, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BSIG','BrightSphere Investment Group plc','Finance',2014);
INSERT INTO symbols VALUES ('BSL','Blackstone GSO Senior Floating Rate Term Fund',NULL,2010);
INSERT INTO symbols VALUES ('BSM','Black Stone Minerals, L.P.','Energy',2015);
INSERT INTO symbols VALUES ('BSMX','Banco Santander Mexico, S.A., Institucion de Ban','Finance',2012);
INSERT INTO symbols VALUES ('BSQR','BSQUARE Corporation','Miscellaneous',1999);
INSERT INTO symbols VALUES ('BSRR','Sierra Bancorp','Finance',NULL);
INSERT INTO symbols VALUES ('BST','BlackRock Science and Technology Trust',NULL,2014);
INSERT INTO symbols VALUES ('BSTC','BioSpecifics Technologies Corp','Health Care',NULL);
INSERT INTO symbols VALUES ('BSVN','Bank7 Corp.','Finance',2018);
INSERT INTO symbols VALUES ('BSX','Boston Scientific Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('BT','BT Group plc','Public Utilities',NULL);
INSERT INTO symbols VALUES ('BTA','BlackRock Long-Term Municipal Advantage Trust',NULL,2006);
INSERT INTO symbols VALUES ('BTAI','BioXcel Therapeutics, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('BTE','Baytex Energy Corp','Energy',NULL);
INSERT INTO symbols VALUES ('BTEC','Principal Healthcare Innovators Index ETF',NULL,NULL);
INSERT INTO symbols VALUES ('BTG','B2Gold Corp','Basic Industries',NULL);
INSERT INTO symbols VALUES ('BTI','British American Tobacco p.l.c.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('BTN','Ballantyne Strong, Inc','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('BTO','John Hancock Financial Opportunities Fund',NULL,1994);
INSERT INTO symbols VALUES ('BTT','BlackRock Municipal Target Term Trust Inc. (The)',NULL,2012);
INSERT INTO symbols VALUES ('BTU','Peabody Energy Corporation','Energy',2017);
INSERT INTO symbols VALUES ('BTX','BioTime, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('BTZ','BlackRock Credit Allocation Income Trust',NULL,2006);
INSERT INTO symbols VALUES ('BUD','Anheuser-Busch Inbev SA','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('BUI','BlackRock Utility, Infrastructure & Power Opportun',NULL,2011);
INSERT INTO symbols VALUES ('BURG','Chanticleer Holdings, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BURL','Burlington Stores, Inc.','Consumer Services',2013);
INSERT INTO symbols VALUES ('BUSE','First Busey Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BV','BrightView Holdings, Inc.','Consumer Non-Durables',2018);
INSERT INTO symbols VALUES ('BVN','Buenaventura Mining Company Inc.','Basic Industries',1996);
INSERT INTO symbols VALUES ('BVNSC','Brandes Investment Trust',NULL,NULL);
INSERT INTO symbols VALUES ('BVSN','BroadVision, Inc.','Technology',1996);
INSERT INTO symbols VALUES ('BVXV','BiondVax Pharmaceuticals Ltd.','Health Care',2015);
INSERT INTO symbols VALUES ('BVXVW','BiondVax Pharmaceuticals Ltd.','Health Care',2015);
INSERT INTO symbols VALUES ('BW','Babcock','Capital Goods',2015);
INSERT INTO symbols VALUES ('BWA','BorgWarner Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BWAY','Brainsway Ltd.','Health Care',2019);
INSERT INTO symbols VALUES ('BWB','Bridgewater Bancshares, Inc.','Finance',2018);
INSERT INTO symbols VALUES ('BWEN','Broadwind Energy, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('BWFG','Bankwell Financial Group, Inc.','Finance',2014);
INSERT INTO symbols VALUES ('BWG','BrandywineGLOBAL Global Income Opportunities Fund ',NULL,2012);
INSERT INTO symbols VALUES ('BWL.A','Bowl America, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BWMC','Boxwood Merger Corp.','Finance',2019);
INSERT INTO symbols VALUES ('BWMCU','Boxwood Merger Corp.','Finance',2018);
INSERT INTO symbols VALUES ('BWMCW','Boxwood Merger Corp.','Finance',2019);
INSERT INTO symbols VALUES ('BWXT','BWX Technologies, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('BX','The Blackstone Group L.P.','Finance',2007);
INSERT INTO symbols VALUES ('BXC','BlueLinx Holdings Inc.','Capital Goods',2004);
INSERT INTO symbols VALUES ('BXG','Bluegreen Vacations Corporation','Finance',2017);
INSERT INTO symbols VALUES ('BXMT','Capital Trust, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('BXMX','Nuveen S&P 500 Buy-Write Income Fund',NULL,NULL);
INSERT INTO symbols VALUES ('BXP','Boston Properties, Inc.','Consumer Services',1997);
INSERT INTO symbols VALUES ('BXP^B','Boston Properties, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('BXS','BancorpSouth Bank',NULL,NULL);
INSERT INTO symbols VALUES ('BY','Byline Bancorp, Inc.','Finance',2017);
INSERT INTO symbols VALUES ('BYD','Boyd Gaming Corporation','Consumer Services',1993);
INSERT INTO symbols VALUES ('BYFC','Broadway Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('BYM','Blackrock Municipal Income Quality Trust',NULL,2002);
INSERT INTO symbols VALUES ('BYND','Beyond Meat, Inc.',NULL,2019);
INSERT INTO symbols VALUES ('BYSI','BeyondSpring, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('BZH','Beazer Homes USA, Inc.','Capital Goods',1994);
INSERT INTO symbols VALUES ('BZM','BlackRock Maryland Municipal Bond Trust',NULL,2002);
INSERT INTO symbols VALUES ('BZUN','Baozun Inc.','Consumer Services',2015);
INSERT INTO symbols VALUES ('C','Citigroup Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CAAP','Corporacion America Airports SA','Transportation',2018);
INSERT INTO symbols VALUES ('CAAS','China Automotive Systems, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CABO','Cable One, Inc.','Consumer Services',2015);
INSERT INTO symbols VALUES ('CAC','Camden National Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CACC','Credit Acceptance Corporation','Finance',1992);
INSERT INTO symbols VALUES ('CACG','ClearBridge All Cap Growth ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CACI','CACI International, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('CADC','China Advanced Construction Materials Group, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CADE','Cadence Bancorporation','Finance',2017);
INSERT INTO symbols VALUES ('CAE','CAE Inc','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('CAF','Morgan Stanley China A Share Fund Inc.',NULL,2006);
INSERT INTO symbols VALUES ('CAG','ConAgra Brands, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CAH','Cardinal Health, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CAI','CAI International, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('CAI^A','CAI International, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CAI^B','CAI International, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CAJ','Canon, Inc.','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('CAKE','The Cheesecake Factory Incorporated','Consumer Services',1992);
INSERT INTO symbols VALUES ('CAL','Caleres, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CALA','Calithera Biosciences, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('CALM','Cal-Maine Foods, Inc.','Consumer Non-Durables',1996);
INSERT INTO symbols VALUES ('CALX','Calix, Inc','Consumer Services',2010);
INSERT INTO symbols VALUES ('CAMP','CalAmp Corp.','Technology',1983);
INSERT INTO symbols VALUES ('CAMT','Camtek Ltd.','Capital Goods',2000);
INSERT INTO symbols VALUES ('CANF','Can-Fite Biopharma Ltd','Health Care',NULL);
INSERT INTO symbols VALUES ('CANG','Cango Inc.','Technology',2018);
INSERT INTO symbols VALUES ('CAPL','CrossAmerica Partners LP','Energy',2012);
INSERT INTO symbols VALUES ('CAPR','Capricor Therapeutics, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CAR','Avis Budget Group, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CARA','Cara Therapeutics, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('CARB','Carbonite, Inc.','Technology',2011);
INSERT INTO symbols VALUES ('CARE','Carter Bank & Trust',NULL,NULL);
INSERT INTO symbols VALUES ('CARG','CarGurus, Inc.','Technology',2017);
INSERT INTO symbols VALUES ('CARO','Carolina Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CARS','Cars.com Inc.','Technology',2017);
INSERT INTO symbols VALUES ('CART','Carolina Trust BancShares, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CARV','Carver Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CARZ','First Trust NASDAQ Global Auto Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('CASA','Casa Systems, Inc.','Technology',2017);
INSERT INTO symbols VALUES ('CASH','Meta Financial Group, Inc.','Finance',1993);
INSERT INTO symbols VALUES ('CASI','CASI Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CASS','Cass Information Systems, Inc','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('CASY','Caseys General Stores, Inc.','Consumer Durables',1983);
INSERT INTO symbols VALUES ('CAT','Caterpillar, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CATB','Catabasis Pharmaceuticals, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('CATC','Cambridge Bancorp','Finance',NULL);
INSERT INTO symbols VALUES ('CATH','Global X S&P 500 Catholic Values ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CATM','Cardtronics plc','Miscellaneous',2007);
INSERT INTO symbols VALUES ('CATO','Cato Corporation (The)','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CATS','Catasys, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CATY','Cathay General Bancorp','Finance',NULL);
INSERT INTO symbols VALUES ('CB','Chubb Limited','Finance',NULL);
INSERT INTO symbols VALUES ('CBAN','Colony Bankcorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CBAT','CBAK Energy Technology, Inc.','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('CBAY','CymaBay Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CBB','Cincinnati Bell Inc','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CBB^B','Cincinnati Bell Inc',NULL,NULL);
INSERT INTO symbols VALUES ('CBD','Companhia Brasileira de Distribuicao','Consumer Services',1997);
INSERT INTO symbols VALUES ('CBFV','CB Financial Services, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CBH','AllianzGI Convertible & Income 2024 Target Term Fund',NULL,2017);
INSERT INTO symbols VALUES ('CBIO','Catalyst Biosciences, Inc. ','Health Care',NULL);
INSERT INTO symbols VALUES ('CBL','CBL & Associates Properties, Inc.','Consumer Services',1993);
INSERT INTO symbols VALUES ('CBLI','Cleveland BioLabs, Inc.','Health Care',2006);
INSERT INTO symbols VALUES ('CBLK','Carbon Black, Inc.','Technology',2018);
INSERT INTO symbols VALUES ('CBL^D','CBL & Associates Properties, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CBL^E','CBL & Associates Properties, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CBM','Cambrex Corporation','Health Care',1987);
INSERT INTO symbols VALUES ('CBMB','CBM Bancorp, Inc.','Finance',2018);
INSERT INTO symbols VALUES ('CBMG','Cellular Biomedicine Group, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CBNK','Capital Bancorp, Inc.','Finance',2018);
INSERT INTO symbols VALUES ('CBO','CBO (Listing Market - NYSE - Networks A/E)',NULL,NULL);
INSERT INTO symbols VALUES ('CBOE','Cboe Global Markets, Inc.',NULL,2010);
INSERT INTO symbols VALUES ('CBPO','China Biologic Products Holdings, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CBPX','Continental Building Products, Inc.','Capital Goods',2014);
INSERT INTO symbols VALUES ('CBRE','CBRE Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CBRL','Cracker Barrel Old Country Store, Inc.','Consumer Services',1981);
INSERT INTO symbols VALUES ('CBS','CBS Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CBS.A','CBS Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CBSH','Commerce Bancshares, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CBSHP','Commerce Bancshares, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CBT','Cabot Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CBTX','CBTX, Inc.','Finance',2017);
INSERT INTO symbols VALUES ('CBU','Community Bank System, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CBUS','Cibus Corp.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CBX','CBX (Listing Market NYSE Networks AE',NULL,NULL);
INSERT INTO symbols VALUES ('CBZ','CBIZ, Inc.','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('CC','Chemours Company (The)','Basic Industries',2015);
INSERT INTO symbols VALUES ('CCA','MFS California Insured Municipal Trust',NULL,1999);
INSERT INTO symbols VALUES ('CCB','Coastal Financial Corporation','Finance',2018);
INSERT INTO symbols VALUES ('CCBG','Capital City Bank Group','Finance',NULL);
INSERT INTO symbols VALUES ('CCC','Churchill Capital Corp','Technology',2018);
INSERT INTO symbols VALUES ('CCC.U','Churchill Capital Corp',NULL,2018);
INSERT INTO symbols VALUES ('CCC.WS','Churchill Capital Corp',NULL,NULL);
INSERT INTO symbols VALUES ('CCCL','China Ceramics Co., Ltd.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CCD','Calamos Dynamic Convertible & Income Fund',NULL,NULL);
INSERT INTO symbols VALUES ('CCEP','Coca-Cola European Partners plc','Consumer Non-Durables',2016);
INSERT INTO symbols VALUES ('CCF','Chase Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CCH','Collier Creek Holdings','Finance',2018);
INSERT INTO symbols VALUES ('CCH.U','Collier Creek Holdings',NULL,2018);
INSERT INTO symbols VALUES ('CCH.WS','Collier Creek Holdings',NULL,NULL);
INSERT INTO symbols VALUES ('CCI','Crown Castle International Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CCIH','ChinaCache International Holdings Ltd.','Technology',2010);
INSERT INTO symbols VALUES ('CCI^A','Crown Castle International Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CCJ','Cameco Corporation','Basic Industries',1996);
INSERT INTO symbols VALUES ('CCK','Crown Holdings, Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CCL','Carnival Corporation','Consumer Services',1987);
INSERT INTO symbols VALUES ('CCLP','CSI Compressco LP','Energy',2011);
INSERT INTO symbols VALUES ('CCM','Concord Medical Services Holdings Limited','Health Care',2009);
INSERT INTO symbols VALUES ('CCMP','Cabot Microelectronics Corporation','Technology',2000);
INSERT INTO symbols VALUES ('CCNE','CNB Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CCNI','Command Center, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('CCO','Clear Channel Outdoor Holdings, Inc.','Consumer Services',2005);
INSERT INTO symbols VALUES ('CCOI','Cogent Communications Holdings, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CCR','CONSOL Coal Resources LP','Energy',2015);
INSERT INTO symbols VALUES ('CCRC','China Customer Relations Centers, Inc.','Miscellaneous',2015);
INSERT INTO symbols VALUES ('CCRN','Cross Country Healthcare, Inc.','Technology',2001);
INSERT INTO symbols VALUES ('CCS','Century Communities, Inc.','Capital Goods',2014);
INSERT INTO symbols VALUES ('CCU','Compania Cervecerias Unidas, S.A.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CCXI','ChemoCentryx, Inc.','Health Care',2012);
INSERT INTO symbols VALUES ('CCZ','Comcast Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CDAY','Ceridian HCM Holding Inc.','Technology',2018);
INSERT INTO symbols VALUES ('CDC','VictoryShares US EQ Income Enhanced Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CDE','Coeur Mining, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CDEV','Centennial Resource Development, Inc.','Energy',2016);
INSERT INTO symbols VALUES ('CDK','CDK Global, Inc.','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('CDL','VictoryShares US Large Cap High Div Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CDLX','Cardlytics, Inc.','Technology',2018);
INSERT INTO symbols VALUES ('CDMO','Avid Bioservices, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CDMOP','Avid Bioservices, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CDNA','CareDx, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('CDNS','Cadence Design Systems, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('CDR','Cedar Realty Trust, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CDR^B','Cedar Realty Trust, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CDR^C','Cedar Realty Trust, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CDTX','Cidara Therapeutics, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('CDW','CDW Corporation','Consumer Services',2013);
INSERT INTO symbols VALUES ('CDXC','ChromaDex Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CDXS','Codexis, Inc.','Basic Industries',2010);
INSERT INTO symbols VALUES ('CDZI','Cadiz, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CE','Celanese Corporation','Basic Industries',2005);
INSERT INTO symbols VALUES ('CEA','China Eastern Airlines Corporation Ltd.','Transportation',1997);
INSERT INTO symbols VALUES ('CECE','CECO Environmental Corp.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CECO','Career Education Corporation','Consumer Services',1998);
INSERT INTO symbols VALUES ('CEE','The Central and Eastern Europe Fund, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CEI','Camber Energy, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('CEIX','CNX Resources Corporation','Energy',2017);
INSERT INTO symbols VALUES ('CEL','Cellcom Israel, Ltd.','Public Utilities',2007);
INSERT INTO symbols VALUES ('CELC','Celcuity Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('CELG','Celgene Corporation','Health Care',1987);
INSERT INTO symbols VALUES ('CELGZ','Celgene Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('CELH','Celsius Holdings, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CELP','Cypress Energy Partners, L.P.','Energy',2014);
INSERT INTO symbols VALUES ('CEM','ClearBridge MLP and Midstream Fund Inc.',NULL,2010);
INSERT INTO symbols VALUES ('CEMI','Chembio Diagnostics, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CEN','Center Coast Brookfield MLP & Energy Infrastructur',NULL,2013);
INSERT INTO symbols VALUES ('CENT','Central Garden & Pet Company','Consumer Durables',1993);
INSERT INTO symbols VALUES ('CENTA','Central Garden & Pet Company','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CENX','Century Aluminum Company','Basic Industries',1996);
INSERT INTO symbols VALUES ('CEO','CNOOC Limited','Energy',2001);
INSERT INTO symbols VALUES ('CEPU','Central Puerto S.A.','Public Utilities',2018);
INSERT INTO symbols VALUES ('CEQP','Crestwood Equity Partners LP','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CERC','Cerecor Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('CERN','Cerner Corporation','Technology',1986);
INSERT INTO symbols VALUES ('CERS','Cerus Corporation','Health Care',1997);
INSERT INTO symbols VALUES ('CET','Central Securities Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CETV','Central European Media Enterprises Ltd.','Consumer Services',1994);
INSERT INTO symbols VALUES ('CETX','Cemtrex Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CETXP','Cemtrex Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CETXW','Cemtrex Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CEV','Eaton Vance California Municipal Income Trust',NULL,1999);
INSERT INTO symbols VALUES ('CEVA','CEVA, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('CEY','VictoryShares Emerging Market High Div Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CEZ','VictoryShares Emerging Market Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CF','CF Industries Holdings, Inc.','Basic Industries',2005);
INSERT INTO symbols VALUES ('CFA','VictoryShares US 500 Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CFBI','Community First Bancshares, Inc.','Finance',2017);
INSERT INTO symbols VALUES ('CFBK','Central Federal Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CFFA','CF Finance Acquisition Corp.','Finance',2019);
INSERT INTO symbols VALUES ('CFFAU','CF Finance Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('CFFAW','CF Finance Acquisition Corp.','Finance',2019);
INSERT INTO symbols VALUES ('CFFI','C&F Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CFFN','Capitol Federal Financial, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CFG','Citizens Financial Group, Inc.','Finance',2014);
INSERT INTO symbols VALUES ('CFG^D','Citizens Financial Group, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CFMS','ConforMIS, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('CFO','VictoryShares US 500 Enhanced Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CFR','Cullen/Frost Bankers, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CFRX','ContraFect Corporation','Health Care',2014);
INSERT INTO symbols VALUES ('CFR^A','Cullen/Frost Bankers, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CFX','Colfax Corporation','Capital Goods',2008);
INSERT INTO symbols VALUES ('CFXA','Colfax Corporation','Capital Goods',2019);
INSERT INTO symbols VALUES ('CG','The Carlyle Group L.P.','Finance',2012);
INSERT INTO symbols VALUES ('CGA','China Green Agriculture, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CGBD','TCG BDC, Inc.',NULL,2017);
INSERT INTO symbols VALUES ('CGC','Canopy Growth Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CGEN','Compugen Ltd.','Health Care',2000);
INSERT INTO symbols VALUES ('CGIX','Cancer Genetics, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CGNX','Cognex Corporation','Capital Goods',1989);
INSERT INTO symbols VALUES ('CGO','Calamos Global Total Return Fund',NULL,2005);
INSERT INTO symbols VALUES ('CGVIC','Causeway ETMF Trust',NULL,NULL);
INSERT INTO symbols VALUES ('CHA','China Telecom Corp Ltd','Public Utilities',2002);
INSERT INTO symbols VALUES ('CHAC','Chardan Healthcare Acquisition Corp.','Finance',2019);
INSERT INTO symbols VALUES ('CHAC.U','Chardan Healthcare Acquisition Corp.',NULL,2018);
INSERT INTO symbols VALUES ('CHAC.WS','Chardan Healthcare Acquisition Corp.',NULL,NULL);
INSERT INTO symbols VALUES ('CHAP','Chaparral Energy, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('CHCI','Comstock Holding Companies, Inc.','Capital Goods',2004);
INSERT INTO symbols VALUES ('CHCO','City Holding Company','Finance',NULL);
INSERT INTO symbols VALUES ('CHCT','Community Healthcare Trust Incorporated','Consumer Services',2015);
INSERT INTO symbols VALUES ('CHD','Church & Dwight Company, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CHDN','Churchill Downs, Incorporated','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CHE','Chemed Corp.','Health Care',NULL);
INSERT INTO symbols VALUES ('CHEF','The Chefs&#39; Warehouse, Inc.','Consumer Non-Durables',2011);
INSERT INTO symbols VALUES ('CHEK','Check-Cap Ltd.','Health Care',2015);
INSERT INTO symbols VALUES ('CHEKW','Check-Cap Ltd.','Health Care',2015);
INSERT INTO symbols VALUES ('CHEKZ','Check-Cap Ltd.','Health Care',NULL);
INSERT INTO symbols VALUES ('CHFC','Chemical Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CHFS','CHF Solutions, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CHGG','Chegg, Inc.','Consumer Services',2013);
INSERT INTO symbols VALUES ('CHH','Choice Hotels International, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CHI','Calamos Convertible Opportunities and Income Fund',NULL,2002);
INSERT INTO symbols VALUES ('CHIC','Global X Funds Global X MSCI China Communication Services ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CHK','Chesapeake Energy Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('CHKE','Cherokee Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CHKP','Check Point Software Technologies Ltd.','Technology',1996);
INSERT INTO symbols VALUES ('CHKR','Chesapeake Granite Wash Trust','Energy',2011);
INSERT INTO symbols VALUES ('CHK^D','Chesapeake Energy Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CHL','China Mobile (Hong Kong) Ltd.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CHMA','Chiasma, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('CHMG','Chemung Financial Corp','Finance',NULL);
INSERT INTO symbols VALUES ('CHMI','Cherry Hill Mortgage Investment Corporation','Consumer Services',2013);
INSERT INTO symbols VALUES ('CHMI^A','Cherry Hill Mortgage Investment Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CHMI^B','Cherry Hill Mortgage Investment Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CHN','China Fund, Inc. (The)',NULL,NULL);
INSERT INTO symbols VALUES ('CHNA','Loncar China BioPharma ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CHNR','China Natural Resources, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CHRA','Charah Solutions, Inc.','Basic Industries',2018);
INSERT INTO symbols VALUES ('CHRS','Coherus BioSciences, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('CHRW','C.H. Robinson Worldwide, Inc.','Transportation',1997);
INSERT INTO symbols VALUES ('CHS','Chico&#39;s FAS, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CHSCL','CHS Inc','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CHSCM','CHS Inc','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CHSCN','CHS Inc','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CHSCO','CHS Inc','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CHSCP','CHS Inc','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CHSP','Chesapeake Lodging Trust','Consumer Services',2010);
INSERT INTO symbols VALUES ('CHT','Chunghwa Telecom Co., Ltd.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CHTR','Charter Communications, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CHU','China Unicom (Hong Kong) Ltd','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CHUY','Chuy&#39;s Holdings, Inc.','Consumer Services',2012);
INSERT INTO symbols VALUES ('CHW','Calamos Global Dynamic Income Fund',NULL,2007);
INSERT INTO symbols VALUES ('CHY','Calamos Convertible and High Income Fund',NULL,2003);
INSERT INTO symbols VALUES ('CI','Cigna Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('CIA','Citizens, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CIB','BanColombia S.A.','Finance',NULL);
INSERT INTO symbols VALUES ('CIBR','First Trust NASDAQ Cybersecurity ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CIC','Capitol Investment Corp. IV','Finance',2017);
INSERT INTO symbols VALUES ('CIC.U','Capitol Investment Corp. IV',NULL,2017);
INSERT INTO symbols VALUES ('CIC.WS','Capitol Investment Corp. IV',NULL,NULL);
INSERT INTO symbols VALUES ('CID','VictoryShares International High Div Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CIDM','Cinedigm Corp','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CIEN','Ciena Corporation','Public Utilities',1997);
INSERT INTO symbols VALUES ('CIF','Colonial Intermediate High Income Fund',NULL,NULL);
INSERT INTO symbols VALUES ('CIFS','China Internet Nationwide Financial Services Inc.','Consumer Services',2017);
INSERT INTO symbols VALUES ('CIG','Comp En De Mn Cemig ADS','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CIG.C','Comp En De Mn Cemig ADS',NULL,NULL);
INSERT INTO symbols VALUES ('CIGI','Colliers International Group Inc. ','Finance',NULL);
INSERT INTO symbols VALUES ('CII','Blackrock Capital and Income Strategies Fund Inc',NULL,2004);
INSERT INTO symbols VALUES ('CIK','Credit Suisse Asset Management Income Fund, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CIL','VictoryShares International Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CIM','Chimera Investment Corporation','Consumer Services',2007);
INSERT INTO symbols VALUES ('CIM^A','Chimera Investment Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CIM^B','Chimera Investment Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CIM^C','Chimera Investment Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CIM^D','Chimera Investment Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CINF','Cincinnati Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CINR','Ciner Resources LP','Basic Industries',2013);
INSERT INTO symbols VALUES ('CIO','City Office REIT, Inc.','Consumer Services',2014);
INSERT INTO symbols VALUES ('CIO^A','City Office REIT, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CIR','CIRCOR International, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CISN','Cision Ltd.','Technology',2017);
INSERT INTO symbols VALUES ('CIT','CIT Group Inc (DEL)','Finance',NULL);
INSERT INTO symbols VALUES ('CIVB','Civista Bancshares, Inc. ','Finance',NULL);
INSERT INTO symbols VALUES ('CIVBP','Civista Bancshares, Inc. ','Finance',NULL);
INSERT INTO symbols VALUES ('CIVEC','Causeway ETMF Trust',NULL,NULL);
INSERT INTO symbols VALUES ('CIX','CompX International Inc.','Capital Goods',1998);
INSERT INTO symbols VALUES ('CIZ','VictoryShares Developed Enhanced Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CIZN','Citizens Holding Company','Finance',NULL);
INSERT INTO symbols VALUES ('CJ','C&J Energy Services, Inc','Energy',2017);
INSERT INTO symbols VALUES ('CJJD','China Jo-Jo Drugstores, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CKH','SEACOR Holdings, Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('CKPT','Checkpoint Therapeutics, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CKX','CKX Lands, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('CL','Colgate-Palmolive Company','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CLAR','Clarus Corporation','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CLB','Core Laboratories N.V.','Energy',1995);
INSERT INTO symbols VALUES ('CLBK','Columbia Financial, Inc.','Finance',2018);
INSERT INTO symbols VALUES ('CLBS','Caladrius Biosciences, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CLCT','Collectors Universe, Inc.','Miscellaneous',1999);
INSERT INTO symbols VALUES ('CLDB','Cortland Bancorp','Finance',NULL);
INSERT INTO symbols VALUES ('CLDC','China Lending Corporation','Finance',2014);
INSERT INTO symbols VALUES ('CLDR','Cloudera, Inc.','Technology',2017);
INSERT INTO symbols VALUES ('CLDT','Chatham Lodging Trust (REIT)','Consumer Services',2010);
INSERT INTO symbols VALUES ('CLDX','Celldex Therapeutics, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CLF','Cleveland-Cliffs Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CLFD','Clearfield, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CLGN','CollPlant Holdings, Ltd.','Health Care',NULL);
INSERT INTO symbols VALUES ('CLGX','CoreLogic, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('CLH','Clean Harbors, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CLI','Mack-Cali Realty Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CLIR','ClearSign Combustion Corporation','Capital Goods',2012);
INSERT INTO symbols VALUES ('CLLS','Cellectis S.A.','Health Care',2015);
INSERT INTO symbols VALUES ('CLM','Cornerstone Strategic Value Fund, Inc.',NULL,1987);
INSERT INTO symbols VALUES ('CLMT','Calumet Specialty Products Partners, L.P.','Energy',2006);
INSERT INTO symbols VALUES ('CLNC','Colony Credit Real Estate, Inc.','Consumer Services',2018);
INSERT INTO symbols VALUES ('CLNE','Clean Energy Fuels Corp.','Public Utilities',2007);
INSERT INTO symbols VALUES ('CLNY','Colony Capital, Inc.','Consumer Services',2014);
INSERT INTO symbols VALUES ('CLNY^B','Colony Capital, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CLNY^E','Colony Capital, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CLNY^G','Colony Capital, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CLNY^H','Colony Capital, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CLNY^I','Colony Capital, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CLNY^J','Colony Capital, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CLOU','Global X Cloud Computing ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CLPR','Clipper Realty Inc.','Consumer Services',2017);
INSERT INTO symbols VALUES ('CLPS','CLPS Incorporation','Technology',2018);
INSERT INTO symbols VALUES ('CLR','Continental Resources, Inc.','Energy',2007);
INSERT INTO symbols VALUES ('CLRB','Cellectar Biosciences, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CLRBW','Cellectar Biosciences, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CLRBZ','Cellectar Biosciences, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CLRG','IQ Chaikin U.S. Large Cap ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CLRO','ClearOne, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CLS','Celestica, Inc.','Technology',1998);
INSERT INTO symbols VALUES ('CLSD','Clearside Biomedical, Inc.','Health Care',2016);
INSERT INTO symbols VALUES ('CLSN','Celsion Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('CLUB','Town Sports International Holdings, Inc.','Consumer Services',2006);
INSERT INTO symbols VALUES ('CLVS','Clovis Oncology, Inc.','Health Care',2011);
INSERT INTO symbols VALUES ('CLW','Clearwater Paper Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CLWT','Euro Tech Holdings Company Limited','Consumer Durables',1997);
INSERT INTO symbols VALUES ('CLX','Clorox Company (The)','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CLXT','Calyxt, Inc.','Basic Industries',2017);
INSERT INTO symbols VALUES ('CM','Canadian Imperial Bank of Commerce','Finance',NULL);
INSERT INTO symbols VALUES ('CMA','Comerica Incorporated','Finance',NULL);
INSERT INTO symbols VALUES ('CMC','Commercial Metals Company','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CMCL','Caledonia Mining Corporation Plc','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CMCM','Cheetah Mobile Inc.','Technology',2014);
INSERT INTO symbols VALUES ('CMCO','Columbus McKinnon Corporation','Capital Goods',1996);
INSERT INTO symbols VALUES ('CMCSA','Comcast Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CMCT','CIM Commercial Trust Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CMCTP','CIM Commercial Trust Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CMD','Cantel Medical Corp.','Health Care',NULL);
INSERT INTO symbols VALUES ('CME','CME Group Inc.','Finance',2002);
INSERT INTO symbols VALUES ('CMFN','CM Finance Inc',NULL,2014);
INSERT INTO symbols VALUES ('CMFNL','CM Finance Inc',NULL,NULL);
INSERT INTO symbols VALUES ('CMG','Chipotle Mexican Grill, Inc.','Consumer Services',2006);
INSERT INTO symbols VALUES ('CMI','Cummins Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('CMLS','Cumulus Media Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CMO','Capstead Mortgage Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CMO^E','Capstead Mortgage Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CMP','Compass Minerals International, Inc.','Basic Industries',2003);
INSERT INTO symbols VALUES ('CMPR','Cimpress N.V','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('CMRE','Costamare Inc.','Transportation',2010);
INSERT INTO symbols VALUES ('CMRE^B','Costamare Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CMRE^C','Costamare Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CMRE^D','Costamare Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CMRE^E','Costamare Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CMRX','Chimerix, Inc.','Health Care',2013);
INSERT INTO symbols VALUES ('CMS','CMS Energy Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CMSA','CMS Energy Corporation','Public Utilities',2018);
INSERT INTO symbols VALUES ('CMSC','CMS Energy Corporation','Public Utilities',2018);
INSERT INTO symbols VALUES ('CMSD','CMS Energy Corporation','Public Utilities',2019);
INSERT INTO symbols VALUES ('CMS^B','CMS Energy Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CMT','Core Molding Technologies Inc','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CMTL','Comtech Telecommunications Corp.','Technology',NULL);
INSERT INTO symbols VALUES ('CMU','Colonial Municipal Income Trust',NULL,1987);
INSERT INTO symbols VALUES ('CNA','CNA Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CNAC','Constellation Alpha Capital Corp.','Health Care',2017);
INSERT INTO symbols VALUES ('CNACR','Constellation Alpha Capital Corp.','Health Care',2017);
INSERT INTO symbols VALUES ('CNACU','Constellation Alpha Capital Corp.','Health Care',2017);
INSERT INTO symbols VALUES ('CNACW','Constellation Alpha Capital Corp.','Health Care',2017);
INSERT INTO symbols VALUES ('CNAT','Conatus Pharmaceuticals Inc.','Health Care',2013);
INSERT INTO symbols VALUES ('CNBKA','Century Bancorp, Inc.','Finance',1987);
INSERT INTO symbols VALUES ('CNC','Centene Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('CNCE','Concert Pharmaceuticals, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('CNCR','Loncar Cancer Immunotherapy ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CNDT','Conduent Incorporated','Miscellaneous',2016);
INSERT INTO symbols VALUES ('CNET','ChinaNet Online Holdings, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('CNF','CNFinance Holdings Limited','Finance',2018);
INSERT INTO symbols VALUES ('CNFR','Conifer Holdings, Inc.','Finance',2015);
INSERT INTO symbols VALUES ('CNFRL','Conifer Holdings, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CNHI','CNH Industrial N.V.','Capital Goods',2013);
INSERT INTO symbols VALUES ('CNI','Canadian National Railway Company','Transportation',NULL);
INSERT INTO symbols VALUES ('CNK','Cinemark Holdings Inc','Consumer Services',2007);
INSERT INTO symbols VALUES ('CNMD','CONMED Corporation','Health Care',1987);
INSERT INTO symbols VALUES ('CNNE','Cannae Holdings, Inc.','Consumer Services',2017);
INSERT INTO symbols VALUES ('CNO','CNO Financial Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CNOB','ConnectOne Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CNP','CenterPoint Energy, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CNP^B','CenterPoint Energy, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CNQ','Canadian Natural Resources Limited','Energy',NULL);
INSERT INTO symbols VALUES ('CNS','Cohen & Steers Inc','Finance',2004);
INSERT INTO symbols VALUES ('CNSL','Consolidated Communications Holdings, Inc.','Public Utilities',2005);
INSERT INTO symbols VALUES ('CNST','Constellation Pharmaceuticals, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('CNTF','China TechFaith Wireless Communication Technology Limited','Miscellaneous',2005);
INSERT INTO symbols VALUES ('CNTX','Centrexion Therapeutics Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('CNTY','Century Casinos, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CNX','CNX Resources Corporation','Energy',1999);
INSERT INTO symbols VALUES ('CNXM','CNX Midstream Partners LP','Public Utilities',2014);
INSERT INTO symbols VALUES ('CNXN','PC Connection, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CO','Global Cord Blood Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('COCP','Cocrystal Pharma, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CODA','Coda Octopus Group, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CODI','Compass Diversified Holdings','Consumer Durables',2006);
INSERT INTO symbols VALUES ('CODI^A','Compass Diversified Holdings',NULL,NULL);
INSERT INTO symbols VALUES ('CODI^B','Compass Diversified Holdings',NULL,NULL);
INSERT INTO symbols VALUES ('CODX','Co-Diagnostics, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('COE','China Online Education Group','Consumer Services',2016);
INSERT INTO symbols VALUES ('COF','Capital One Financial Corporation','Finance',1994);
INSERT INTO symbols VALUES ('COF^C','Capital One Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('COF^D','Capital One Financial Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('COF^F','Capital One Financial Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('COF^G','Capital One Financial Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('COF^H','Capital One Financial Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('COF^P','Capital One Financial Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('COG','Cabot Oil & Gas Corporation','Energy',1990);
INSERT INTO symbols VALUES ('COHN','Cohen & Company Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('COHR','Coherent, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('COHU','Cohu, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('COKE','Coca-Cola Consolidated, Inc.','Consumer Non-Durables',1972);
INSERT INTO symbols VALUES ('COLB','Columbia Banking System, Inc.','Finance',1992);
INSERT INTO symbols VALUES ('COLD','Americold Realty Trust','Consumer Services',2018);
INSERT INTO symbols VALUES ('COLL','Collegium Pharmaceutical, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('COLM','Columbia Sportswear Company','Consumer Non-Durables',1998);
INSERT INTO symbols VALUES ('COMM','CommScope Holding Company, Inc.','Technology',2013);
INSERT INTO symbols VALUES ('COMT','iShares Commodities Select Strategy ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CONE','CyrusOne Inc','Consumer Services',2013);
INSERT INTO symbols VALUES ('CONN','Conn&#39;s, Inc.','Consumer Services',2003);
INSERT INTO symbols VALUES ('COO','The Cooper Companies, Inc. ','Health Care',NULL);
INSERT INTO symbols VALUES ('COOP','Mr. Cooper Group Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('COP','ConocoPhillips','Energy',NULL);
INSERT INTO symbols VALUES ('COR','CoreSite Realty Corporation','Consumer Services',2010);
INSERT INTO symbols VALUES ('CORE','Core-Mark Holding Company, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CORR','CorEnergy Infrastructure Trust, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CORR^A','CorEnergy Infrastructure Trust, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CORT','Corcept Therapeutics Incorporated','Health Care',2004);
INSERT INTO symbols VALUES ('CORV','Correvio Pharma Corp.','Health Care',NULL);
INSERT INTO symbols VALUES ('COST','Costco Wholesale Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('COT','Cott Corporation','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('COTY','Coty Inc.','Consumer Non-Durables',2013);
INSERT INTO symbols VALUES ('COUP','Coupa Software Incorporated','Technology',2016);
INSERT INTO symbols VALUES ('COWN','Cowen Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('COWNL','Cowen Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('COWNZ','Cowen Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CP','Canadian Pacific Railway Limited','Transportation',NULL);
INSERT INTO symbols VALUES ('CPA','Copa Holdings, S.A.','Transportation',2005);
INSERT INTO symbols VALUES ('CPAC','Cementos Pacasmayo S.A.A.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CPAH','CounterPath Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('CPB','Campbell Soup Company','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CPE','Callon Petroleum Company','Energy',NULL);
INSERT INTO symbols VALUES ('CPE^A','Callon Petroleum Company','Energy',NULL);
INSERT INTO symbols VALUES ('CPF','CPB Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CPG','Crescent Point Energy Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('CPHC','Canterbury Park Holding Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CPHI','China Pharma Holdings, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CPIX','Cumberland Pharmaceuticals Inc.','Health Care',2009);
INSERT INTO symbols VALUES ('CPK','Chesapeake Utilities Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CPL','CPFL Energia S.A.','Public Utilities',2004);
INSERT INTO symbols VALUES ('CPLG','CorePoint Lodging Inc.','Consumer Services',2018);
INSERT INTO symbols VALUES ('CPLP','Capital Product Partners L.P.','Transportation',2007);
INSERT INTO symbols VALUES ('CPRI','Capri Holdings Limited','Consumer Non-Durables',2011);
INSERT INTO symbols VALUES ('CPRT','Copart, Inc.','Consumer Durables',1994);
INSERT INTO symbols VALUES ('CPRX','Catalyst Pharmaceuticals, Inc.','Health Care',2006);
INSERT INTO symbols VALUES ('CPS','Cooper-Standard Holdings Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CPSH','CPS Technologies Corp.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CPSI','Computer Programs and Systems, Inc.','Technology',2002);
INSERT INTO symbols VALUES ('CPSS','Consumer Portfolio Services, Inc.','Finance',1992);
INSERT INTO symbols VALUES ('CPST','Capstone Turbine Corporation','Energy',2000);
INSERT INTO symbols VALUES ('CPT','Camden Property Trust','Consumer Services',1993);
INSERT INTO symbols VALUES ('CPTA','Capitala Finance Corp.',NULL,2013);
INSERT INTO symbols VALUES ('CPTAG','Capitala Finance Corp.',NULL,NULL);
INSERT INTO symbols VALUES ('CPTAL','Capitala Finance Corp.',NULL,NULL);
INSERT INTO symbols VALUES ('CQP','Cheniere Energy Partners, LP','Public Utilities',2007);
INSERT INTO symbols VALUES ('CR','Crane Co.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CRAI','CRA International,Inc.','Miscellaneous',1998);
INSERT INTO symbols VALUES ('CRAY','Cray Inc','Technology',NULL);
INSERT INTO symbols VALUES ('CRBP','Corbus Pharmaceuticals Holdings, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CRC','California Resources Corporation','Energy',2014);
INSERT INTO symbols VALUES ('CRCM','Care.com, Inc.','Consumer Services',2014);
INSERT INTO symbols VALUES ('CRD.A','Crawford & Company',NULL,NULL);
INSERT INTO symbols VALUES ('CRD.B','Crawford & Company',NULL,NULL);
INSERT INTO symbols VALUES ('CREE','Cree, Inc.','Technology',1993);
INSERT INTO symbols VALUES ('CREG','China Recycling Energy Corporation','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('CRESY','Cresud S.A.C.I.F. y A.','Finance',1997);
INSERT INTO symbols VALUES ('CREX','Creative Realities, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('CREXW','Creative Realities, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('CRF','Cornerstone Strategic Return Fund, Inc. (The)',NULL,NULL);
INSERT INTO symbols VALUES ('CRH','CRH PLC','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CRHM','CRH Medical Corporation','Health Care',2015);
INSERT INTO symbols VALUES ('CRI','Carter&#39;s, Inc.','Consumer Non-Durables',2003);
INSERT INTO symbols VALUES ('CRIS','Curis, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CRK','Comstock Resources, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('CRL','Charles River Laboratories International, Inc.','Health Care',2000);
INSERT INTO symbols VALUES ('CRM','Salesforce.com Inc','Technology',2004);
INSERT INTO symbols VALUES ('CRMD','CorMedix Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CRMT','America&#39;s Car-Mart, Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CRNT','Ceragon Networks Ltd.','Technology',NULL);
INSERT INTO symbols VALUES ('CRNX','Crinetics Pharmaceuticals, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('CRON','Cronos Group Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CROX','Crocs, Inc.','Consumer Non-Durables',2006);
INSERT INTO symbols VALUES ('CRR','Carbo Ceramics, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CRS','Carpenter Technology Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CRSA','Crescent Acquisition Corp','Finance',2019);
INSERT INTO symbols VALUES ('CRSAU','Crescent Acquisition Corp','Finance',2019);
INSERT INTO symbols VALUES ('CRSAW','Crescent Acquisition Corp','Finance',2019);
INSERT INTO symbols VALUES ('CRSP','CRISPR Therapeutics AG','Health Care',2016);
INSERT INTO symbols VALUES ('CRT','Cross Timbers Royalty Trust','Energy',1992);
INSERT INTO symbols VALUES ('CRTO','Criteo S.A.','Technology',2013);
INSERT INTO symbols VALUES ('CRUS','Cirrus Logic, Inc.','Technology',1989);
INSERT INTO symbols VALUES ('CRUSC','Calvert Management Series',NULL,NULL);
INSERT INTO symbols VALUES ('CRVL','CorVel Corp.','Finance',NULL);
INSERT INTO symbols VALUES ('CRVS','Corvus Pharmaceuticals, Inc.','Health Care',2016);
INSERT INTO symbols VALUES ('CRWS','Crown Crafts, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CRY','CryoLife, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CRZO','Carrizo Oil & Gas, Inc.','Energy',1997);
INSERT INTO symbols VALUES ('CS','Credit Suisse Group','Finance',NULL);
INSERT INTO symbols VALUES ('CSA','VictoryShares US Small Cap Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CSB','VictoryShares US Small Cap High Div Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CSBR','Champions Oncology, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CSCO','Cisco Systems, Inc.','Technology',1990);
INSERT INTO symbols VALUES ('CSF','VictoryShares US Discovery Enhanced Volatility Wtd ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CSFL','CenterState Bank Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CSGP','CoStar Group, Inc.','Miscellaneous',1998);
INSERT INTO symbols VALUES ('CSGS','CSG Systems International, Inc.','Technology',1996);
INSERT INTO symbols VALUES ('CSII','Cardiovascular Systems, Inc.','Health Care',1981);
INSERT INTO symbols VALUES ('CSIQ','Canadian Solar Inc.','Technology',2006);
INSERT INTO symbols VALUES ('CSL','Carlisle Companies Incorporated','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CSLT','Castlight Health, inc.','Technology',2014);
INSERT INTO symbols VALUES ('CSML','IQ Chaikin U.S. Small Cap ETF',NULL,NULL);
INSERT INTO symbols VALUES ('CSOD','Cornerstone OnDemand, Inc.','Technology',2011);
INSERT INTO symbols VALUES ('CSPI','CSP Inc.','Technology',1982);
INSERT INTO symbols VALUES ('CSQ','Calamos Strategic Total Return Fund',NULL,2004);
INSERT INTO symbols VALUES ('CSS','CSS Industries, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CSSE','Chicken Soup for the Soul Entertainment, Inc.','Consumer Services',2017);
INSERT INTO symbols VALUES ('CSSEP','Chicken Soup for the Soul Entertainment, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CSTE','Caesarstone Ltd.','Capital Goods',2012);
INSERT INTO symbols VALUES ('CSTM','Constellium N.V.','Basic Industries',2013);
INSERT INTO symbols VALUES ('CSTR','CapStar Financial Holdings, Inc.','Finance',2016);
INSERT INTO symbols VALUES ('CSU','Capital Senior Living Corporation','Health Care',1997);
INSERT INTO symbols VALUES ('CSV','Carriage Services, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CSWC','Capital Southwest Corporation','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CSWCL','Capital Southwest Corporation','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CSWI','CSW Industrials, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CSX','CSX Corporation','Transportation',NULL);
INSERT INTO symbols VALUES ('CTAA','Qwest Corporation','Public Utilities',2016);
INSERT INTO symbols VALUES ('CTAC','ChaSerg Technology Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('CTACU','ChaSerg Technology Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('CTACW','ChaSerg Technology Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('CTAS','Cintas Corporation','Consumer Non-Durables',1983);
INSERT INTO symbols VALUES ('CTB','Cooper Tire & Rubber Company','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CTBB','Qwest Corporation','Public Utilities',2016);
INSERT INTO symbols VALUES ('CTBI','Community Trust Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CTDD','Qwest Corporation','Public Utilities',2017);
INSERT INTO symbols VALUES ('CTEK','CynergisTek, Inc.','Miscellaneous',2017);
INSERT INTO symbols VALUES ('CTG','Computer Task Group, Incorporated','Technology',NULL);
INSERT INTO symbols VALUES ('CTHR','Charles & Colvard Ltd.','Consumer Durables',1997);
INSERT INTO symbols VALUES ('CTIB','CTI Industries Corporation','Basic Industries',1997);
INSERT INTO symbols VALUES ('CTIC','CTI BioPharma Corp.','Health Care',1997);
INSERT INTO symbols VALUES ('CTK','CooTek (Cayman) Inc.','Technology',2018);
INSERT INTO symbols VALUES ('CTL','CenturyLink, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CTLT','Catalent, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('CTMX','CytomX Therapeutics, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('CTO','Consolidated-Tomoka Land Co.','Finance',NULL);
INSERT INTO symbols VALUES ('CTR','ClearBridge MLP and Midstream Total Return Fund In',NULL,2012);
INSERT INTO symbols VALUES ('CTRA','Contura Energy, Inc.','Energy',2018);
INSERT INTO symbols VALUES ('CTRC','Centric Brands Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CTRE','CareTrust REIT, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CTRL','Control4 Corporation','Capital Goods',2013);
INSERT INTO symbols VALUES ('CTRM','Castor Maritime Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('CTRN','Citi Trends, Inc.','Consumer Services',2005);
INSERT INTO symbols VALUES ('CTRP','Ctrip.com International, Ltd.','Miscellaneous',2003);
INSERT INTO symbols VALUES ('CTRV','ContraVir Pharmaceuticals Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CTS','CTS Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('CTSH','Cognizant Technology Solutions Corporation','Technology',1998);
INSERT INTO symbols VALUES ('CTSO','Cytosorbents Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('CTST','CannTrust Holdings Inc.','Consumer Durables',2019);
INSERT INTO symbols VALUES ('CTT','CatchMark Timber Trust, Inc.','Consumer Services',2013);
INSERT INTO symbols VALUES ('CTV','Qwest Corporation','Public Utilities',2014);
INSERT INTO symbols VALUES ('CTWS','Connecticut Water Service, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CTXR','Citius Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CTXRW','Citius Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CTXS','Citrix Systems, Inc.','Technology',1995);
INSERT INTO symbols VALUES ('CTY','Qwest Corporation','Public Utilities',2013);
INSERT INTO symbols VALUES ('CTZ','Qwest Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CUB','Cubic Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CUBA','The Herzfeld Caribbean Basin Fund, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CUBE','CubeSmart','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CUBI','Customers Bancorp, Inc','Finance',NULL);
INSERT INTO symbols VALUES ('CUBI^C','Customers Bancorp, Inc',NULL,NULL);
INSERT INTO symbols VALUES ('CUBI^D','Customers Bancorp, Inc',NULL,NULL);
INSERT INTO symbols VALUES ('CUBI^E','Customers Bancorp, Inc',NULL,NULL);
INSERT INTO symbols VALUES ('CUBI^F','Customers Bancorp, Inc',NULL,NULL);
INSERT INTO symbols VALUES ('CUE','Cue Biopharma, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('CUI','CUI Global, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CUK','Carnival Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CULP','Culp, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CUO','Continental Materials Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CUR','Neuralstem, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CURO','CURO Group Holdings Corp.','Finance',2017);
INSERT INTO symbols VALUES ('CUTR','Cutera, Inc.','Health Care',2004);
INSERT INTO symbols VALUES ('CUZ','Cousins Properties Incorporated','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CVA','Covanta Holding Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CVBF','CVB Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CVCO','Cavco Industries, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('CVCY','Central Valley Community Bancorp','Finance',NULL);
INSERT INTO symbols VALUES ('CVE','Cenovus Energy Inc','Energy',NULL);
INSERT INTO symbols VALUES ('CVEO','Civeo Corporation','Consumer Services',2014);
INSERT INTO symbols VALUES ('CVET','Covetrus, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CVGI','Commercial Vehicle Group, Inc.','Capital Goods',2004);
INSERT INTO symbols VALUES ('CVGW','Calavo Growers, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('CVI','CVR Energy Inc.','Energy',2007);
INSERT INTO symbols VALUES ('CVIA','Covia Holdings Corporation','Basic Industries',2018);
INSERT INTO symbols VALUES ('CVLT','Commvault Systems, Inc.','Technology',2006);
INSERT INTO symbols VALUES ('CVLY','Codorus Valley Bancorp, Inc','Finance',NULL);
INSERT INTO symbols VALUES ('CVM','Cel-Sci Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('CVNA','Carvana Co.','Consumer Durables',2017);
INSERT INTO symbols VALUES ('CVR','Chicago Rivet & Machine Co.','Technology',NULL);
INSERT INTO symbols VALUES ('CVRS','Corindus Vascular Robotics, Inc.','Health Care',2015);
INSERT INTO symbols VALUES ('CVS','CVS Health Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('CVTI','Covenant Transportation Group, Inc.','Transportation',1994);
INSERT INTO symbols VALUES ('CVU','CPI Aerostructures, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CVV','CVD Equipment Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('CVX','Chevron Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('CW','Curtiss-Wright Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('CWBC','Community West Bancshares','Finance',NULL);
INSERT INTO symbols VALUES ('CWBR','CohBar, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CWCO','Consolidated Water Co. Ltd.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CWEN','Clearway Energy, Inc.','Public Utilities',2015);
INSERT INTO symbols VALUES ('CWEN.A','Clearway Energy, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('CWH','Camping World Holdings, Inc.','Consumer Durables',2016);
INSERT INTO symbols VALUES ('CWK','Cushman & Wakefield plc','Finance',2018);
INSERT INTO symbols VALUES ('CWST','Casella Waste Systems, Inc.','Public Utilities',1997);
INSERT INTO symbols VALUES ('CWT','California Water  Service Group Holding','Public Utilities',NULL);
INSERT INTO symbols VALUES ('CX','Cemex S.A.B. de C.V.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CXDC','China XD Plastics Company Limited','Capital Goods',NULL);
INSERT INTO symbols VALUES ('CXE','Colonial High Income Municipal Trust',NULL,1989);
INSERT INTO symbols VALUES ('CXH','Colonial Investment Grade Municipal Trust',NULL,1989);
INSERT INTO symbols VALUES ('CXO','Concho Resources Inc.','Energy',2007);
INSERT INTO symbols VALUES ('CXP','Columbia Property Trust, Inc.','Consumer Services',2013);
INSERT INTO symbols VALUES ('CXSE','WisdomTree China ex-State-Owned Enterprises Fund',NULL,NULL);
INSERT INTO symbols VALUES ('CXW','CoreCivic, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('CY','Cypress Semiconductor Corporation','Technology',1986);
INSERT INTO symbols VALUES ('CYAD','Celyad SA','Health Care',2015);
INSERT INTO symbols VALUES ('CYAN','Cyanotech Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('CYBE','CyberOptics Corporation','Capital Goods',1987);
INSERT INTO symbols VALUES ('CYBR','CyberArk Software Ltd.','Technology',2014);
INSERT INTO symbols VALUES ('CYCC','Cyclacel Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CYCCP','Cyclacel Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CYCN','Cyclerion Therapeutics, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CYD','China Yuchai International Limited','Energy',1994);
INSERT INTO symbols VALUES ('CYH','Community Health Systems, Inc.','Health Care',2000);
INSERT INTO symbols VALUES ('CYOU','Changyou.com Limited','Technology',2009);
INSERT INTO symbols VALUES ('CYRN','CYREN Ltd.','Technology',NULL);
INSERT INTO symbols VALUES ('CYRX','CryoPort, Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('CYRXW','CryoPort, Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('CYTK','Cytokinetics, Incorporated','Health Care',2004);
INSERT INTO symbols VALUES ('CYTR','CytRx Corporation','Health Care',1986);
INSERT INTO symbols VALUES ('CYTX','Cytori Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CYTXZ','Cytori Therapeutics Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('CZFC','Citizens First Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('CZNC','Citizens & Northern Corp','Finance',NULL);
INSERT INTO symbols VALUES ('CZR','Caesars Entertainment Corporation','Consumer Services',2012);
INSERT INTO symbols VALUES ('CZWI','Citizens Community Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('CZZ','Cosan Limited','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('C^J','Citigroup Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('C^K','Citigroup Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('C^N','Citigroup Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('C^S','Citigroup Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('D','Dominion Energy, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('DAC','Danaos Corporation','Transportation',2006);
INSERT INTO symbols VALUES ('DAIO','Data I/O Corporation','Capital Goods',1981);
INSERT INTO symbols VALUES ('DAKT','Daktronics, Inc.','Consumer Durables',1994);
INSERT INTO symbols VALUES ('DAL','Delta Air Lines, Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('DALI','First Trust DorseyWright DALI 1 ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DAN','Dana Incorporated','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DAR','Darling Ingredients Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('DARE','Dare Bioscience, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('DATA','Tableau Software, Inc.','Technology',2013);
INSERT INTO symbols VALUES ('DAVA','Endava plc','Technology',2018);
INSERT INTO symbols VALUES ('DAVE','Famous Dave&#39;s of America, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DAX','Global X DAX Germany ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DB','Deutsche Bank AG','Finance',NULL);
INSERT INTO symbols VALUES ('DBD','Diebold Nixdorf Incorporated','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('DBI','Designer Brands Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DBL','DoubleLine Opportunistic Credit Fund',NULL,2012);
INSERT INTO symbols VALUES ('DBVT','DBV Technologies S.A.','Health Care',2014);
INSERT INTO symbols VALUES ('DBX','Dropbox, Inc.','Technology',2018);
INSERT INTO symbols VALUES ('DCAR','DropCar, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DCF','Dreyfus Alcentra',NULL,2017);
INSERT INTO symbols VALUES ('DCI','Donaldson Company, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DCIX','Performance Shipping Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('DCO','Ducommun Incorporated','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DCOM','Dime Community Bancshares, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('DCP','DCP Midstream LP','Public Utilities',NULL);
INSERT INTO symbols VALUES ('DCPH','Deciphera Pharmaceuticals, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('DCP^B','DCP Midstream LP',NULL,NULL);
INSERT INTO symbols VALUES ('DCP^C','DCP Midstream LP',NULL,NULL);
INSERT INTO symbols VALUES ('DCUD','Dominion Energy, Inc.','Public Utilities',2016);
INSERT INTO symbols VALUES ('DDD','3D Systems Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('DDF','Delaware Investments Dividend & Income Fund, Inc.',NULL,1993);
INSERT INTO symbols VALUES ('DDIV','First Trust Dorsey Wright Momentum & Dividend ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DDMX','DD3 Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('DDMXU','DD3 Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('DDMXW','DD3 Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('DDOC','DERMAdoctor, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DDS','Dillard&#39;s, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DDT','Dillard&#39;s, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DD^A','E.I. du Pont de Nemours and Company','Basic Industries',NULL);
INSERT INTO symbols VALUES ('DD^B','E.I. du Pont de Nemours and Company','Basic Industries',NULL);
INSERT INTO symbols VALUES ('DE','Deere & Company','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DEA','Easterly Government Properties, Inc.','Consumer Services',2015);
INSERT INTO symbols VALUES ('DECK','Deckers Outdoor Corporation','Consumer Non-Durables',1993);
INSERT INTO symbols VALUES ('DEI','Douglas Emmett, Inc.','Consumer Services',2006);
INSERT INTO symbols VALUES ('DELL','Dell Technologies Inc.','Technology',2018);
INSERT INTO symbols VALUES ('DENN','Denny&#39;s Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DEO','Diageo plc','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('DERM','Dermira, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('DESP','Despegar.com, Corp.','Consumer Services',2017);
INSERT INTO symbols VALUES ('DEST','Destination Maternity Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DEX','Delaware Enhanced Global Dividend',NULL,2007);
INSERT INTO symbols VALUES ('DF','Dean Foods Company','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('DFBH','DFB Healthcare Acquisitions Corp.','Finance',2018);
INSERT INTO symbols VALUES ('DFBHU','DFB Healthcare Acquisitions Corp.','Finance',2018);
INSERT INTO symbols VALUES ('DFBHW','DFB Healthcare Acquisitions Corp.','Finance',2018);
INSERT INTO symbols VALUES ('DFFN','Diffusion Pharmaceuticals Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('DFIN','Donnelley Financial Solutions, Inc.','Consumer Services',2016);
INSERT INTO symbols VALUES ('DFNL','Davis Select Financial ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DFP','Flaherty & Crumrine Dynamic Preferred and Income Fund Inc.',NULL,2013);
INSERT INTO symbols VALUES ('DFRG','Del Frisco&#39;s Restaurant Group, Inc.','Consumer Services',2012);
INSERT INTO symbols VALUES ('DFS','Discover Financial Services','Finance',NULL);
INSERT INTO symbols VALUES ('DFVL','Barclays PLC','Finance',NULL);
INSERT INTO symbols VALUES ('DFVS','Barclays PLC','Finance',NULL);
INSERT INTO symbols VALUES ('DG','Dollar General Corporation','Consumer Services',2009);
INSERT INTO symbols VALUES ('DGICA','Donegal Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('DGICB','Donegal Group, Inc.','Finance',1986);
INSERT INTO symbols VALUES ('DGII','Digi International Inc.','Technology',1989);
INSERT INTO symbols VALUES ('DGLD','Credit Suisse AG','Finance',NULL);
INSERT INTO symbols VALUES ('DGLY','Digital Ally, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('DGRE','WisdomTree Emerging Markets Quality Dividend Growth Fund',NULL,NULL);
INSERT INTO symbols VALUES ('DGRS','WisdomTree U.S. SmallCap Quality Dividend Growth Fund',NULL,NULL);
INSERT INTO symbols VALUES ('DGRW','WisdomTree U.S. Quality Dividend Growth Fund',NULL,NULL);
INSERT INTO symbols VALUES ('DGSE','DGSE Companies, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DGX','Quest Diagnostics Incorporated','Health Care',NULL);
INSERT INTO symbols VALUES ('DHF','Dreyfus High Yield Strategies Fund',NULL,1998);
INSERT INTO symbols VALUES ('DHI','D.R. Horton, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DHIL','Diamond Hill Investment Group, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('DHR','Danaher Corporation','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DHR^A','Danaher Corporation',NULL,NULL);
INSERT INTO symbols VALUES ('DHT','DHT Holdings, Inc.','Transportation',2005);
INSERT INTO symbols VALUES ('DHX','DHI Group, Inc.','Miscellaneous',2007);
INSERT INTO symbols VALUES ('DHXM','DHX Media Ltd.',NULL,NULL);
INSERT INTO symbols VALUES ('DHY','Credit Suisse High Yield Bond Fund',NULL,1998);
INSERT INTO symbols VALUES ('DIAX','Nuveen Dow 30SM Dynamic Overwrite Fund',NULL,2014);
INSERT INTO symbols VALUES ('DIN','Dine Brands Global, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DINT','Davis Select International ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DIOD','Diodes Incorporated','Technology',NULL);
INSERT INTO symbols VALUES ('DIS','Walt Disney Company (The)','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DISCA','Discovery, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DISCB','Discovery, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DISCK','Discovery, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DISH','DISH Network Corporation','Consumer Services',1995);
INSERT INTO symbols VALUES ('DIT','AMCON Distributing Company','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('DJCO','Daily Journal Corp. (S.C.)','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DK','Delek US Holdings, Inc.','Energy',2017);
INSERT INTO symbols VALUES ('DKL','Delek Logistics Partners, L.P.','Energy',2012);
INSERT INTO symbols VALUES ('DKS','Dick&#39;s Sporting Goods Inc','Consumer Services',2002);
INSERT INTO symbols VALUES ('DKT','Deutsch Bk Contingent Cap Tr V','Finance',NULL);
INSERT INTO symbols VALUES ('DL','China Distance Education Holdings Limited','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DLA','Delta Apparel, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('DLB','Dolby Laboratories','Miscellaneous',2005);
INSERT INTO symbols VALUES ('DLBS','Barclays PLC','Finance',NULL);
INSERT INTO symbols VALUES ('DLHC','DLH Holdings Corp.','Technology',NULL);
INSERT INTO symbols VALUES ('DLNG','Dynagas LNG Partners LP','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DLNG^A','Dynagas LNG Partners LP',NULL,NULL);
INSERT INTO symbols VALUES ('DLNG^B','Dynagas LNG Partners LP',NULL,NULL);
INSERT INTO symbols VALUES ('DLPH','Delphi Technologies PLC','Capital Goods',2017);
INSERT INTO symbols VALUES ('DLPN','Dolphin Entertainment, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DLPNW','Dolphin Entertainment, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DLR','Digital Realty Trust, Inc.','Consumer Services',2004);
INSERT INTO symbols VALUES ('DLR^C','Digital Realty Trust, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DLR^G','Digital Realty Trust, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DLR^I','Digital Realty Trust, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DLR^J','Digital Realty Trust, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DLR^K','Digital Realty Trust, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DLTH','Duluth Holdings Inc.','Consumer Services',2015);
INSERT INTO symbols VALUES ('DLTR','Dollar Tree, Inc.','Consumer Services',1995);
INSERT INTO symbols VALUES ('DLX','Deluxe Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('DMAC','DiaMedica Therapeutics Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('DMB','Dreyfus Municipal Bond Infrastructure Fund, Inc.',NULL,2013);
INSERT INTO symbols VALUES ('DMF','Dreyfus Municipal Income, Inc.',NULL,1988);
INSERT INTO symbols VALUES ('DMLP','Dorchester Minerals, L.P.','Energy',NULL);
INSERT INTO symbols VALUES ('DMO','Western Asset Mortgage Defined Opportunity Fund Inc',NULL,2010);
INSERT INTO symbols VALUES ('DMPI','DelMar Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('DMRC','Digimarc Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('DNBF','DNB Financial Corp','Finance',NULL);
INSERT INTO symbols VALUES ('DNI','Dividend and Income Fund',NULL,NULL);
INSERT INTO symbols VALUES ('DNJR','GOLDEN BULL LIMITED','Finance',2018);
INSERT INTO symbols VALUES ('DNKN','Dunkin&#39; Brands Group, Inc.','Consumer Services',2011);
INSERT INTO symbols VALUES ('DNLI','Denali Therapeutics Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('DNN','Denison Mine Corp','Basic Industries',NULL);
INSERT INTO symbols VALUES ('DNOW','NOW Inc.','Energy',2014);
INSERT INTO symbols VALUES ('DNP','Duff & Phelps Utilities Income, Inc.',NULL,1987);
INSERT INTO symbols VALUES ('DNR','Denbury Resources Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('DO','Diamond Offshore Drilling, Inc.','Energy',1995);
INSERT INTO symbols VALUES ('DOC','Physicians Realty Trust','Consumer Services',2013);
INSERT INTO symbols VALUES ('DOCU','DocuSign, Inc.','Technology',2018);
INSERT INTO symbols VALUES ('DOGZ','Dogness (International) Corporation','Consumer Durables',2017);
INSERT INTO symbols VALUES ('DOMO','Domo, Inc.','Technology',2018);
INSERT INTO symbols VALUES ('DOOO','BRP Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DOOR','Masonite International Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('DORM','Dorman Products, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DOV','Dover Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('DOVA','Dova Pharmaceuticals, Inc.','Health Care',2017);
INSERT INTO symbols VALUES ('DOW','Dow Inc.','Basic Industries',2019);
INSERT INTO symbols VALUES ('DOX','Amdocs Limited','Technology',NULL);
INSERT INTO symbols VALUES ('DPG','Duff & Phelps Global Utility Income Fund Inc.',NULL,2011);
INSERT INTO symbols VALUES ('DPHC','DiamondPeak Holdings Corp.','Finance',2019);
INSERT INTO symbols VALUES ('DPHCU','DiamondPeak Holdings Corp.','Finance',2019);
INSERT INTO symbols VALUES ('DPHCW','DiamondPeak Holdings Corp.',NULL,2019);
INSERT INTO symbols VALUES ('DPLO','Diplomat Pharmacy, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('DPW','DPW Holdings, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DPZ','Domino&#39;s Pizza Inc','Consumer Non-Durables',2004);
INSERT INTO symbols VALUES ('DQ','DAQO New Energy Corp.','Technology',2010);
INSERT INTO symbols VALUES ('DRAD','Digirad Corporation','Health Care',2004);
INSERT INTO symbols VALUES ('DRD','DRDGOLD Limited','Basic Industries',NULL);
INSERT INTO symbols VALUES ('DRE','Duke Realty Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DRH','Diamondrock Hospitality Company','Consumer Services',2005);
INSERT INTO symbols VALUES ('DRI','Darden Restaurants, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DRIO','DarioHealth Corp.','Health Care',NULL);
INSERT INTO symbols VALUES ('DRIOW','DarioHealth Corp.','Health Care',NULL);
INSERT INTO symbols VALUES ('DRIV','Global X Autonomous & Electric Vehicles ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DRNA','Dicerna Pharmaceuticals, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('DRQ','Dril-Quip, Inc.','Energy',1997);
INSERT INTO symbols VALUES ('DRRX','DURECT Corporation','Health Care',2000);
INSERT INTO symbols VALUES ('DRUA','Dominion Energy, Inc.','Public Utilities',2016);
INSERT INTO symbols VALUES ('DRYS','DryShips Inc.','Transportation',2005);
INSERT INTO symbols VALUES ('DS','Drive Shack Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DSE','Duff & Phelps Select MLP and Midstream Energy Fund',NULL,2014);
INSERT INTO symbols VALUES ('DSGX','The Descartes Systems Group Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('DSKE','Daseke, Inc.','Consumer Services',2015);
INSERT INTO symbols VALUES ('DSKEW','Daseke, Inc.','Consumer Services',2015);
INSERT INTO symbols VALUES ('DSL','DoubleLine Income Solutions Fund',NULL,2013);
INSERT INTO symbols VALUES ('DSLV','Credit Suisse AG','Finance',NULL);
INSERT INTO symbols VALUES ('DSM','Dreyfus Strategic Municipal Bond Fund, Inc.',NULL,1989);
INSERT INTO symbols VALUES ('DSPG','DSP Group, Inc.','Technology',1994);
INSERT INTO symbols VALUES ('DSS','Document Security Systems, Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('DSSI','Diamond S Shipping Inc.','Consumer Services',2019);
INSERT INTO symbols VALUES ('DSU','Blackrock Debt Strategies Fund, Inc.',NULL,1998);
INSERT INTO symbols VALUES ('DSWL','Deswell Industries, Inc.','Consumer Non-Durables',1995);
INSERT INTO symbols VALUES ('DSX','Diana Shipping inc.','Transportation',2005);
INSERT INTO symbols VALUES ('DSX^B','Diana Shipping inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DS^B','Drive Shack Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DS^C','Drive Shack Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DS^D','Drive Shack Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DTE','DTE Energy Company','Public Utilities',NULL);
INSERT INTO symbols VALUES ('DTEA','DAVIDsTEA Inc.','Consumer Non-Durables',2015);
INSERT INTO symbols VALUES ('DTF','Duff & Phelps Utilities Tax-Free Income, Inc.',NULL,1991);
INSERT INTO symbols VALUES ('DTIL','Precision BioSciences, Inc.','Health Care',2019);
INSERT INTO symbols VALUES ('DTJ','DTE Energy Company','Public Utilities',2016);
INSERT INTO symbols VALUES ('DTLA^','Brookfield DTLA Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DTQ','DTE Energy Company','Public Utilities',NULL);
INSERT INTO symbols VALUES ('DTSS','Datasea Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('DTUL','Barclays PLC','Finance',NULL);
INSERT INTO symbols VALUES ('DTUS','Barclays PLC','Finance',NULL);
INSERT INTO symbols VALUES ('DTV','DTE Energy Company','Public Utilities',2016);
INSERT INTO symbols VALUES ('DTW','DTE Energy Company','Public Utilities',2017);
INSERT INTO symbols VALUES ('DTY','DTE Energy Company','Public Utilities',2016);
INSERT INTO symbols VALUES ('DTYL','Barclays PLC','Finance',NULL);
INSERT INTO symbols VALUES ('DTYS','Barclays PLC','Finance',NULL);
INSERT INTO symbols VALUES ('DUC','Duff & Phelps Utility & Corporate Bond Trust, Inc.',NULL,1993);
INSERT INTO symbols VALUES ('DUK','Duke Energy Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('DUKB','Duke Energy Corporation','Public Utilities',2018);
INSERT INTO symbols VALUES ('DUKH','Duke Energy Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('DUK^A','Duke Energy Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('DUSA','Davis Select U.S. Equity ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DVA','DaVita Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('DVAX','Dynavax Technologies Corporation','Health Care',2004);
INSERT INTO symbols VALUES ('DVCR','Diversicare Healthcare Services Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('DVD','Dover Motorsports, Inc.','Consumer Services',1996);
INSERT INTO symbols VALUES ('DVLU','First Trust Dorsey Wright Momentum & Value ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DVN','Devon Energy Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('DVOL','First Trust Dorsey Wright Momentum & Low Volatility ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DVY','iShares Select Dividend ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWAQ','Invesco DWA NASDAQ Momentum ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWAS','Invesco DWA SmallCap Momentum ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWAT','Arrow DWA Tactical ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWCR','Arrow DWA Country Rotation ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWDP','DowDuPont Inc.','Basic Industries',2017);
INSERT INTO symbols VALUES ('DWFI','SPDR Dorsey Wright Fixed Income Allocation ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWIN','Invesco DWA Tactical Multi-Asset Income ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWLD','Davis Select Worldwide ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWMC','AdvisorShares Dorsey Wright Micro-Cap ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWPP','First Trust Dorsey Wright People&#39;s Portfolio ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWSH','AdvisorShares Dorsey Wright Short ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DWSN','Dawson Geophysical Company','Energy',NULL);
INSERT INTO symbols VALUES ('DWTR','Invesco DWA Tactical Sector Rotation ETF',NULL,NULL);
INSERT INTO symbols VALUES ('DX','Dynex Capital, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DXB','Deutsche Bank AG','Finance',NULL);
INSERT INTO symbols VALUES ('DXC','DXC Technology Company','Technology',NULL);
INSERT INTO symbols VALUES ('DXCM','DexCom, Inc.','Health Care',2005);
INSERT INTO symbols VALUES ('DXF','Dunxin Financial Holdings Limited','Finance',NULL);
INSERT INTO symbols VALUES ('DXGE','WisdomTree Germany Hedged Equity Fund',NULL,NULL);
INSERT INTO symbols VALUES ('DXJS','WisdomTree Japan Hedged SmallCap Equity Fund',NULL,NULL);
INSERT INTO symbols VALUES ('DXLG','Destination XL Group, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('DXPE','DXP Enterprises, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DXR','Daxor Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('DXYN','The Dixie Group, Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('DX^A','Dynex Capital, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DX^B','Dynex Capital, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('DY','Dycom Industries, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('DYAI','Dyadic International, Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('DYNC','Vistra Energy Corp.','Public Utilities',2016);
INSERT INTO symbols VALUES ('DYNT','Dynatronics Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('DYSL','Dynasil Corporation of America','Capital Goods',NULL);
INSERT INTO symbols VALUES ('DZSI','DASAN Zhone Solutions, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('E','ENI S.p.A.','Energy',NULL);
INSERT INTO symbols VALUES ('EA','Electronic Arts Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('EAB','Entergy Arkansas, LLC','Public Utilities',NULL);
INSERT INTO symbols VALUES ('EAD','Wells Fargo Income Opportunities Fund',NULL,2003);
INSERT INTO symbols VALUES ('EAE','Entergy Arkansas, LLC','Public Utilities',2013);
INSERT INTO symbols VALUES ('EAF','GrafTech International Ltd.','Energy',2018);
INSERT INTO symbols VALUES ('EAI','Entergy Arkansas, LLC','Public Utilities',2016);
INSERT INTO symbols VALUES ('EARN','Ellington Residential Mortgage REIT','Consumer Services',2013);
INSERT INTO symbols VALUES ('EARS','Auris Medical Holding Ltd.','Health Care',2014);
INSERT INTO symbols VALUES ('EAST','Eastside Distilling, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('EAT','Brinker International, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('EB','Eventbrite, Inc.','Technology',2018);
INSERT INTO symbols VALUES ('EBAY','eBay Inc.','Miscellaneous',1998);
INSERT INTO symbols VALUES ('EBAYL','eBay Inc.','Miscellaneous',NULL);
INSERT INTO symbols VALUES ('EBF','Ennis, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('EBIX','Ebix, Inc.','Technology',NULL);
INSERT INTO symbols VALUES ('EBIZ','Global X E-commerce ETF',NULL,NULL);
INSERT INTO symbols VALUES ('EBMT','Eagle Bancorp Montana, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('EBR','Centrais Electricas Brasileiras S.A.- Eletrobras','Public Utilities',2016);
INSERT INTO symbols VALUES ('EBR.B','Centrais Electricas Brasileiras S.A.- Eletrobras',NULL,2016);
INSERT INTO symbols VALUES ('EBS','Emergent Biosolutions, Inc.','Health Care',2006);
INSERT INTO symbols VALUES ('EBSB','Meridian Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('EBTC','Enterprise Bancorp Inc','Finance',NULL);
INSERT INTO symbols VALUES ('EC','Ecopetrol S.A.','Energy',NULL);
INSERT INTO symbols VALUES ('ECA','Encana Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('ECC','Eagle Point Credit Company Inc.',NULL,2014);
INSERT INTO symbols VALUES ('ECCA','Eagle Point Credit Company Inc.',NULL,2015);
INSERT INTO symbols VALUES ('ECCB','Eagle Point Credit Company Inc.',NULL,2016);
INSERT INTO symbols VALUES ('ECCX','Eagle Point Credit Company Inc.',NULL,2018);
INSERT INTO symbols VALUES ('ECCY','Eagle Point Credit Company Inc.',NULL,2017);
INSERT INTO symbols VALUES ('ECF','Ellsworth Growth and Income Fund Ltd.',NULL,1986);
INSERT INTO symbols VALUES ('ECF^A','Ellsworth Growth and Income Fund Ltd.',NULL,NULL);
INSERT INTO symbols VALUES ('ECHO','Echo Global Logistics, Inc.','Transportation',2009);
INSERT INTO symbols VALUES ('ECL','Ecolab Inc.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('ECOL','US Ecology, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ECOM','ChannelAdvisor Corporation','Technology',2013);
INSERT INTO symbols VALUES ('ECOR','electroCore, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('ECOW','Pacer Emerging Markets Cash Cows 100 ETF',NULL,NULL);
INSERT INTO symbols VALUES ('ECPG','Encore Capital Group Inc','Finance',NULL);
INSERT INTO symbols VALUES ('ECT','ECA Marcellus Trust I','Energy',2010);
INSERT INTO symbols VALUES ('ED','Consolidated Edison Inc','Public Utilities',NULL);
INSERT INTO symbols VALUES ('EDAP','EDAP TMS S.A.','Health Care',1997);
INSERT INTO symbols VALUES ('EDD','Morgan Stanley Emerging Markets Domestic Debt Fund, Inc.',NULL,2007);
INSERT INTO symbols VALUES ('EDF','Stone Harbor Emerging Markets Income Fund',NULL,2010);
INSERT INTO symbols VALUES ('EDI','Stone Harbor Emerging Markets Total Income Fund',NULL,2012);
INSERT INTO symbols VALUES ('EDIT','Editas Medicine, Inc.','Health Care',2016);
INSERT INTO symbols VALUES ('EDN','Empresa Distribuidora Y Comercializadora Norte S.A. (Edenor)','Public Utilities',2007);
INSERT INTO symbols VALUES ('EDNT','Edison Nation, Inc.','Consumer Non-Durables',2018);
INSERT INTO symbols VALUES ('EDRY','EuroDry Ltd.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('EDTX','EdtechX Holdings Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('EDTXU','EdtechX Holdings Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('EDTXW','EdtechX Holdings Acquisition Corp.','Finance',2018);
INSERT INTO symbols VALUES ('EDU','New Oriental Education & Technology Group, Inc.','Consumer Services',2006);
INSERT INTO symbols VALUES ('EDUC','Educational Development Corporation','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('EE','El Paso Electric Company','Public Utilities',NULL);
INSERT INTO symbols VALUES ('EEA','European Equity Fund, Inc. (The)',NULL,NULL);
INSERT INTO symbols VALUES ('EEFT','Euronet Worldwide, Inc.','Finance',1997);
INSERT INTO symbols VALUES ('EEI','Ecology and Environment, Inc.','Consumer Services',1987);
INSERT INTO symbols VALUES ('EEMA','iShares MSCI Emerging Markets Asia ETF',NULL,NULL);
INSERT INTO symbols VALUES ('EEX','Emerald Expositions Events, Inc.','Miscellaneous',2017);
INSERT INTO symbols VALUES ('EFAS','Global X MSCI SuperDividend EAFE ETF',NULL,NULL);
INSERT INTO symbols VALUES ('EFBI','Eagle Financial Bancorp, Inc.','Finance',2017);
INSERT INTO symbols VALUES ('EFC','Ellington Financial Inc.','Finance',2010);
INSERT INTO symbols VALUES ('EFF','Eaton vance Floating-Rate Income Plus Fund',NULL,2013);
INSERT INTO symbols VALUES ('EFII','Electronics for Imaging, Inc.','Technology',1992);
INSERT INTO symbols VALUES ('EFL','Eaton Vance Floating-Rate 2022 Target Term Trust',NULL,2017);
INSERT INTO symbols VALUES ('EFOI','Energy Focus, Inc.','Consumer Durables',NULL);
INSERT INTO symbols VALUES ('EFR','Eaton Vance Senior Floating-Rate Fund',NULL,2003);
INSERT INTO symbols VALUES ('EFSC','Enterprise Financial Services Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('EFT','Eaton Vance Floating Rate Income Trust',NULL,2004);
INSERT INTO symbols VALUES ('EFX','Equifax, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('EGAN','eGain Corporation','Technology',1999);
INSERT INTO symbols VALUES ('EGBN','Eagle Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('EGF','Blackrock Enhanced Government Fund, Inc',NULL,2005);
INSERT INTO symbols VALUES ('EGI','Entree Resources Ltd.','Basic Industries',2017);
INSERT INTO symbols VALUES ('EGIF','Eagle Growth and Income Opportunities Fund',NULL,2015);
INSERT INTO symbols VALUES ('EGLE','Eagle Bulk Shipping Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('EGO','Eldorado Gold Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('EGOV','NIC Inc.','Miscellaneous',1999);
INSERT INTO symbols VALUES ('EGP','EastGroup Properties, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('EGRX','Eagle Pharmaceuticals, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('EGY','VAALCO Energy, Inc. ','Energy',NULL);
INSERT INTO symbols VALUES ('EHC','Encompass Health Corporation','Health Care',NULL);
INSERT INTO symbols VALUES ('EHI','Western Asset Global High Income Fund Inc',NULL,2003);
INSERT INTO symbols VALUES ('EHR','Energy Hunter Resources, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('EHT','Eaton Vance High Income 2021 Target Term Trust',NULL,2016);
INSERT INTO symbols VALUES ('EHTH','eHealth, Inc.','Finance',2006);
INSERT INTO symbols VALUES ('EIDX','Eidos Therapeutics, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('EIG','Employers Holdings Inc','Finance',2007);
INSERT INTO symbols VALUES ('EIGI','Endurance International Group Holdings, Inc.','Technology',2013);
INSERT INTO symbols VALUES ('EIGR','Eiger BioPharmaceuticals, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('EIM','Eaton Vance Municipal Bond Fund',NULL,2002);
INSERT INTO symbols VALUES ('EIX','Edison International','Public Utilities',NULL);
INSERT INTO symbols VALUES ('EKSO','Ekso Bionics Holdings, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('EL','Estee Lauder Companies, Inc. (The)','Consumer Non-Durables',1995);
INSERT INTO symbols VALUES ('ELAN','Elanco Animal Health Incorporated','Health Care',2018);
INSERT INTO symbols VALUES ('ELC','Entergy Louisiana, Inc.','Public Utilities',2016);
INSERT INTO symbols VALUES ('ELF','e.l.f. Beauty, Inc.','Consumer Non-Durables',2016);
INSERT INTO symbols VALUES ('ELGX','Endologix, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ELJ','Entergy Louisiana, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ELLO','Ellomay Capital Ltd.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ELMD','Electromed, Inc.','Health Care',2011);
INSERT INTO symbols VALUES ('ELOX','Eloxx Pharmaceuticals, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ELP','Companhia Paranaense de Energia (COPEL)','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ELS','Equity Lifestyle Properties, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ELSE','Electro-Sensors, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ELTK','Eltek Ltd.','Technology',1997);
INSERT INTO symbols VALUES ('ELU','Entergy Louisiana, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ELVT','Elevate Credit, Inc.','Finance',2017);
INSERT INTO symbols VALUES ('ELY','Callaway Golf Company','Consumer Non-Durables',1992);
INSERT INTO symbols VALUES ('EMAN','eMagin Corporation','Technology',NULL);
INSERT INTO symbols VALUES ('EMB','iShares J.P. Morgan USD Emerging Markets Bond ETF',NULL,NULL);
INSERT INTO symbols VALUES ('EMCB','WisdomTree Emerging Markets Corporate Bond Fund',NULL,NULL);
INSERT INTO symbols VALUES ('EMCF','Emclaire Financial Corp','Finance',NULL);
INSERT INTO symbols VALUES ('EMCG','WisdomTree Emerging Markets Consumer Growth Fund',NULL,NULL);
INSERT INTO symbols VALUES ('EMCI','EMC Insurance Group Inc.','Finance',1982);
INSERT INTO symbols VALUES ('EMD','Western Asset Emerging Markets Debt Fund Inc',NULL,NULL);
INSERT INTO symbols VALUES ('EME','EMCOR Group, Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('EMES','Emerge Energy Services LP','Basic Industries',2013);
INSERT INTO symbols VALUES ('EMF','Templeton Emerging Markets Fund',NULL,1987);
INSERT INTO symbols VALUES ('EMIF','iShares S&P Emerging Markets Infrastructure Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('EMKR','EMCORE Corporation','Technology',1997);
INSERT INTO symbols VALUES ('EML','Eastern Company (The)','Capital Goods',NULL);
INSERT INTO symbols VALUES ('EMMS','Emmis Communications Corporation','Consumer Services',1994);
INSERT INTO symbols VALUES ('EMN','Eastman Chemical Company','Basic Industries',NULL);
INSERT INTO symbols VALUES ('EMO','ClearBridge Energy Midstream Opportunity Fund Inc.',NULL,2011);
INSERT INTO symbols VALUES ('EMP','Entergy Mississippi, LLC','Public Utilities',2016);
INSERT INTO symbols VALUES ('EMR','Emerson Electric Company','Energy',NULL);
INSERT INTO symbols VALUES ('EMX','EMX Royalty Corporation','Basic Industries',NULL);
INSERT INTO symbols VALUES ('EMXC','iShares MSCI Emerging Markets ex China ETF',NULL,NULL);
INSERT INTO symbols VALUES ('ENB','Enbridge Inc','Energy',NULL);
INSERT INTO symbols VALUES ('ENBA','Enbridge Inc','Energy',2018);
INSERT INTO symbols VALUES ('ENBL','Enable Midstream Partners, LP','Public Utilities',2014);
INSERT INTO symbols VALUES ('ENDP','Endo International plc','Health Care',NULL);
INSERT INTO symbols VALUES ('ENFC','Entegra Financial Corp.','Finance',2014);
INSERT INTO symbols VALUES ('ENG','ENGlobal Corporation','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ENIA','Enel Americas S.A.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ENIC','Enel Chile S.A.','Public Utilities',2016);
INSERT INTO symbols VALUES ('ENJ','Entergy New Orleans, LLC','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ENLC','EnLink Midstream, LLC','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ENLV','Enlivex Therapeutics Ltd.','Health Care',2014);
INSERT INTO symbols VALUES ('ENO','Entergy New Orleans, LLC','Public Utilities',2016);
INSERT INTO symbols VALUES ('ENOB','Enochian Biosciences, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ENPH','Enphase Energy, Inc.','Technology',2012);
INSERT INTO symbols VALUES ('ENR','Energizer Holdings, Inc.','Miscellaneous',2015);
INSERT INTO symbols VALUES ('ENR^A','Energizer Holdings, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('ENS','Enersys','Consumer Non-Durables',2004);
INSERT INTO symbols VALUES ('ENSG','The Ensign Group, Inc.','Health Care',2007);
INSERT INTO symbols VALUES ('ENSV','ENSERVCO Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('ENT','Global Eagle Entertainment Inc.','Consumer Services',2011);
INSERT INTO symbols VALUES ('ENTA','Enanta Pharmaceuticals, Inc.','Health Care',2013);
INSERT INTO symbols VALUES ('ENTG','Entegris, Inc.','Consumer Non-Durables',2000);
INSERT INTO symbols VALUES ('ENTX','Entera Bio Ltd.','Health Care',2018);
INSERT INTO symbols VALUES ('ENTXW','Entera Bio Ltd.','Health Care',2018);
INSERT INTO symbols VALUES ('ENV','Envestnet, Inc','Miscellaneous',2010);
INSERT INTO symbols VALUES ('ENVA','Enova International, Inc.','Finance',2014);
INSERT INTO symbols VALUES ('ENX','Eaton Vance New York Municipal Bond Fund',NULL,2002);
INSERT INTO symbols VALUES ('ENZ','Enzo Biochem, Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('ENZL','iShares MSCI New Zealand ETF',NULL,NULL);
INSERT INTO symbols VALUES ('EOD','Wells Fargo Global Dividend Opportunity Fund',NULL,2007);
INSERT INTO symbols VALUES ('EOG','EOG Resources, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('EOI','Eaton Vance Enhance Equity Income Fund',NULL,2004);
INSERT INTO symbols VALUES ('EOLS','Evolus, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('EOS','Eaton Vance Enhanced Equity Income Fund II',NULL,2005);
INSERT INTO symbols VALUES ('EOT','Eaton Vance Municipal Income Trust',NULL,2009);
INSERT INTO symbols VALUES ('EPAM','EPAM Systems, Inc.','Technology',2012);
INSERT INTO symbols VALUES ('EPAY','Bottomline Technologies, Inc.','Technology',1999);
INSERT INTO symbols VALUES ('EPC','Energizer Holdings, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('EPD','Enterprise Products Partners L.P.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('EPE','EP Energy Corporation','Energy',2014);
INSERT INTO symbols VALUES ('EPIX','ESSA Pharma Inc.','Health Care',NULL);
INSERT INTO symbols VALUES ('EPM','Evolution Petroleum Corporation, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('EPR','EPR Properties','Consumer Services',1997);
INSERT INTO symbols VALUES ('EPRT','Essential Properties Realty Trust, Inc.','Consumer Services',2018);
INSERT INTO symbols VALUES ('EPR^C','EPR Properties',NULL,NULL);
INSERT INTO symbols VALUES ('EPR^E','EPR Properties',NULL,NULL);
INSERT INTO symbols VALUES ('EPR^G','EPR Properties',NULL,NULL);
INSERT INTO symbols VALUES ('EPSN','Epsilon Energy Ltd.','Energy',NULL);
INSERT INTO symbols VALUES ('EPZM','Epizyme, Inc.','Health Care',2013);
INSERT INTO symbols VALUES ('EP^C','El Paso Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('EQ','Equillium, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('EQBK','Equity Bancshares, Inc.','Finance',2015);
INSERT INTO symbols VALUES ('EQC','Equity Commonwealth','Consumer Services',NULL);
INSERT INTO symbols VALUES ('EQC^D','Equity Commonwealth',NULL,NULL);
INSERT INTO symbols VALUES ('EQH','AXA Equitable Holdings, Inc.','Finance',2018);
INSERT INTO symbols VALUES ('EQIX','Equinix, Inc.','Consumer Services',2000);
INSERT INTO symbols VALUES ('EQM','EQM Midstream Partners, LP','Public Utilities',2012);
INSERT INTO symbols VALUES ('EQNR','Equinor ASA','Energy',NULL);
INSERT INTO symbols VALUES ('EQR','Equity Residential','Consumer Services',NULL);
INSERT INTO symbols VALUES ('EQRR','ProShares Equities for Rising Rates ETF',NULL,NULL);
INSERT INTO symbols VALUES ('EQS','Equus Total Return, Inc.',NULL,NULL);
INSERT INTO symbols VALUES ('EQT','EQT Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('ERA','Era Group, Inc.','Transportation',NULL);
INSERT INTO symbols VALUES ('ERC','Wells Fargo Multi-Sector Income Fund',NULL,2003);
INSERT INTO symbols VALUES ('ERF','Enerplus Corporation','Energy',NULL);
INSERT INTO symbols VALUES ('ERH','Wells Fargo Utilities and High Income Fund',NULL,2004);
INSERT INTO symbols VALUES ('ERI','Eldorado Resorts, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('ERIC','Ericsson','Technology',NULL);
INSERT INTO symbols VALUES ('ERIE','Erie Indemnity Company','Finance',NULL);
INSERT INTO symbols VALUES ('ERII','Energy Recovery, Inc.','Technology',2008);
INSERT INTO symbols VALUES ('ERJ','Embraer S.A.','Capital Goods',2000);
INSERT INTO symbols VALUES ('EROS','Eros International PLC','Consumer Services',2013);
INSERT INTO symbols VALUES ('ERYP','Erytech Pharma S.A.','Health Care',2017);
INSERT INTO symbols VALUES ('ES','Eversource Energy','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ESBK','Elmira Savings Bank NY (The)','Finance',NULL);
INSERT INTO symbols VALUES ('ESCA','Escalade, Incorporated','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('ESE','ESCO Technologies Inc.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ESEA','Euroseas Ltd.','Transportation',NULL);
INSERT INTO symbols VALUES ('ESG','FlexShares STOXX US ESG Impact Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('ESGD','iShares ESG MSCI EAFE ETF',NULL,NULL);
INSERT INTO symbols VALUES ('ESGE','iShares ESG MSCI EM ETF',NULL,NULL);
INSERT INTO symbols VALUES ('ESGG','FlexShares STOXX Global ESG Impact Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('ESGR','Enstar Group Limited','Finance',NULL);
INSERT INTO symbols VALUES ('ESGRO','Enstar Group Limited','Finance',NULL);
INSERT INTO symbols VALUES ('ESGRP','Enstar Group Limited','Finance',NULL);
INSERT INTO symbols VALUES ('ESGU','iShares ESG MSCI USA ETF',NULL,NULL);
INSERT INTO symbols VALUES ('ESI','Element Solutions Inc.','Basic Industries',2014);
INSERT INTO symbols VALUES ('ESLT','Elbit Systems Ltd.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ESNT','Essent Group Ltd.','Finance',2013);
INSERT INTO symbols VALUES ('ESP','Espey Mfg. & Electronics Corp.','Capital Goods',NULL);
INSERT INTO symbols VALUES ('ESPR','Esperion Therapeutics, Inc.','Health Care',2013);
INSERT INTO symbols VALUES ('ESQ','Esquire Financial Holdings, Inc.','Finance',2017);
INSERT INTO symbols VALUES ('ESRT','Empire State Realty Trust, Inc.','Consumer Services',2013);
INSERT INTO symbols VALUES ('ESS','Essex Property Trust, Inc.','Consumer Services',1994);
INSERT INTO symbols VALUES ('ESSA','ESSA Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('ESTA','Establishment Labs Holdings Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('ESTC','Elastic N.V.','Technology',2018);
INSERT INTO symbols VALUES ('ESTE','Earthstone Energy, Inc.','Energy',NULL);
INSERT INTO symbols VALUES ('ESTR','Estre Ambiental, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ESTRW','Estre Ambiental, Inc.','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ESV','Ensco Rowan plc','Energy',NULL);
INSERT INTO symbols VALUES ('ESXB','Community Bankers Trust Corporation.','Finance',NULL);
INSERT INTO symbols VALUES ('ET','Energy Transfer L.P.','Public Utilities',1996);
INSERT INTO symbols VALUES ('ETB','Eaton Vance Tax-Managed Buy-Write Income Fund',NULL,2005);
INSERT INTO symbols VALUES ('ETFC','E*TRADE Financial Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('ETG','Eaton Vance Tax-Advantaged Global Dividend Income Fund',NULL,2004);
INSERT INTO symbols VALUES ('ETH','Ethan Allen Interiors Inc.','Consumer Durables',1993);
INSERT INTO symbols VALUES ('ETJ','Eaton Vance Risk-Managed Diversified Equity Income Fund',NULL,2007);
INSERT INTO symbols VALUES ('ETM','Entercom Communications Corp.','Consumer Services',1999);
INSERT INTO symbols VALUES ('ETN','Eaton Corporation, PLC','Technology',NULL);
INSERT INTO symbols VALUES ('ETO','Eaton Vance Tax-Advantage Global Dividend Opp',NULL,2004);
INSERT INTO symbols VALUES ('ETON','Eton Pharmaceuticals, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('ETP^C','Energy Transfer Operating, L.P.',NULL,NULL);
INSERT INTO symbols VALUES ('ETP^D','Energy Transfer Operating, L.P.',NULL,NULL);
INSERT INTO symbols VALUES ('ETP^E','Energy Transfer Operating, L.P.',NULL,NULL);
INSERT INTO symbols VALUES ('ETR','Entergy Corporation','Public Utilities',NULL);
INSERT INTO symbols VALUES ('ETRN','Equitrans Midstream Corporation','Public Utilities',2018);
INSERT INTO symbols VALUES ('ETSY','Etsy, Inc.','Miscellaneous',2015);
INSERT INTO symbols VALUES ('ETTX','Entasis Therapeutics Holdings Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('ETV','Eaton Vance Corporation',NULL,2005);
INSERT INTO symbols VALUES ('ETW','Eaton Vance Corporation',NULL,2005);
INSERT INTO symbols VALUES ('ETX','Eaton Vance Municipal Income 2028 Term Trust',NULL,2013);
INSERT INTO symbols VALUES ('ETY','Eaton Vance Tax-Managed Diversified Equity Income Fund',NULL,2006);
INSERT INTO symbols VALUES ('EUFN','iShares MSCI Europe Financials Sector Index Fund',NULL,NULL);
INSERT INTO symbols VALUES ('EURN','Euronav NV','Transportation',2015);
INSERT INTO symbols VALUES ('EV','Eaton Vance Corporation','Finance',NULL);
INSERT INTO symbols VALUES ('EVA','Enviva Partners, LP','Basic Industries',2015);
INSERT INTO symbols VALUES ('EVBG','Everbridge, Inc.','Technology',2016);
INSERT INTO symbols VALUES ('EVBN','Evans Bancorp, Inc.','Finance',NULL);
INSERT INTO symbols VALUES ('EVC','Entravision Communications Corporation','Consumer Services',2000);
INSERT INTO symbols VALUES ('EVER','EverQuote, Inc.','Technology',2018);
INSERT INTO symbols VALUES ('EVF','Eaton Vance Senior Income Trust',NULL,1998);
INSERT INTO symbols VALUES ('EVFM','Evofem Biosciences, Inc.','Health Care',2014);
INSERT INTO symbols VALUES ('EVFTC','Eaton Vance NextShares Trust II',NULL,NULL);
INSERT INTO symbols VALUES ('EVG','Eaton Vance Short Diversified Income Fund',NULL,2005);
INSERT INTO symbols VALUES ('EVGBC','Eaton Vance NextShares Trust',NULL,NULL);
INSERT INTO symbols VALUES ('EVGN','Evogene Ltd.','Basic Industries',NULL);
INSERT INTO symbols VALUES ('EVH','Evolent Health, Inc','Consumer Services',2015);
INSERT INTO symbols VALUES ('EVI','EVI Industries, Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('EVK','Ever-Glory International Group, Inc.','Consumer Non-Durables',NULL);
INSERT INTO symbols VALUES ('EVLMC','Eaton Vance NextShares Trust II',NULL,NULL);
INSERT INTO symbols VALUES ('EVLO','Evelo Biosciences, Inc.','Health Care',2018);
INSERT INTO symbols VALUES ('EVLV','EVINE Live Inc.','Consumer Services',NULL);
INSERT INTO symbols VALUES ('EVM','Eaton Vance California Municipal Bond Fund',NULL,2002);
INSERT INTO symbols VALUES ('EVN','Eaton Vance Municipal Income Trust',NULL,1999);
INSERT INTO symbols VALUES ('EVOK','Evoke Pharma, Inc.','Health Care',2013);
INSERT INTO symbols VALUES ('EVOL','Evolving Systems, Inc.','Technology',1998);
INSERT INTO symbols VALUES ('EVOP','EVO Payments, Inc.','Miscellaneous',2018);
INSERT INTO symbols VALUES ('EVR','Evercore Inc.','Finance',2006);
INSERT INTO symbols VALUES ('EVRG','Evergy, Inc.','Public Utilities',2018);
INSERT INTO symbols VALUES ('EVRI','Everi Holdin