// Copyright 2021 Dolthub, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package shim

import (
	"github.com/dolthub/vitess/go/vt/proto/query"

	"github.com/dolthub/dolt/go/libraries/doltcore/schema"
	"github.com/dolthub/dolt/go/store/prolly"
	"github.com/dolthub/dolt/go/store/prolly/tree"
	"github.com/dolthub/dolt/go/store/types"
	"github.com/dolthub/dolt/go/store/val"
)

func NodeFromValue(v types.Value) tree.Node {
	return tree.NodeFromBytes(v.(types.SerialMessage))
}

func ValueFromMap(m prolly.Map) types.Value {
	return tree.ValueFromNode(m.Node())
}

func ValueFromArtifactMap(m prolly.ArtifactMap) types.Value {
	return tree.ValueFromNode(m.Node())
}

func MapFromValue(v types.Value, sch schema.Schema, ns tree.NodeStore) prolly.Map {
	root := NodeFromValue(v)
	kd := KeyDescriptorFromSchema(sch)
	vd := ValueDescriptorFromSchema(sch)
	return prolly.NewMap(root, ns, kd, vd)
}

func MapDescriptorsFromSchema(sch schema.Schema) (kd, vd val.TupleDesc) {
	kd = KeyDescriptorFromSchema(sch)
	vd = ValueDescriptorFromSchema(sch)
	return
}

func KeyDescriptorFromSchema(sch schema.Schema) val.TupleDesc {
	if schema.IsKeyless(sch) {
		return val.KeylessTupleDesc
	}

	var tt []val.Type
	_ = sch.GetPKCols().Iter(func(tag uint64, col schema.Column) (stop bool, err error) {
		tt = append(tt, val.Type{
			Enc:      encodingFromSqlType(col.TypeInfo.ToSqlType().Type()),
			Nullable: columnNullable(col),
		})
		return
	})
	return val.NewTupleDescriptor(tt...)
}

func columnNullable(col schema.Column) bool {
	for _, cnst := range col.Constraints {
		if cnst.GetConstraintType() == schema.NotNullConstraintType {
			return false
		}
	}
	return true
}

func ValueDescriptorFromSchema(sch schema.Schema) val.TupleDesc {
	var tt []val.Type
	if schema.IsKeyless(sch) {
		tt = []val.Type{val.KeylessCardType}
	}

	_ = sch.GetNonPKCols().Iter(func(tag uint64, col schema.Column) (stop bool, err error) {
		tt = append(tt, val.Type{
			Enc:      encodingFromSqlType(col.TypeInfo.ToSqlType().Type()),
			Nullable: col.IsNullable(),
		})
		return
	})
	return val.NewTupleDescriptor(tt...)
}

func encodingFromSqlType(typ query.Type) val.Encoding {
	return val.Encoding(schema.EncodingFromSqlType(typ))
}
