// Copyright 2022 The Sigstore Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package clusterimagepolicy

import (
	"context"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"fmt"
	"testing"

	logtesting "knative.dev/pkg/logging/testing"

	"github.com/sigstore/cosign/pkg/apis/config"
	"github.com/sigstore/cosign/pkg/apis/cosigned/v1alpha1"
	fakecosignclient "github.com/sigstore/cosign/pkg/client/injection/client/fake"
	"github.com/sigstore/cosign/pkg/client/injection/reconciler/cosigned/v1alpha1/clusterimagepolicy"
	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	clientgotesting "k8s.io/client-go/testing"
	fakekubeclient "knative.dev/pkg/client/injection/kube/client/fake"
	"knative.dev/pkg/configmap"
	"knative.dev/pkg/controller"
	"knative.dev/pkg/system"
	"knative.dev/pkg/tracker"

	. "github.com/sigstore/cosign/pkg/reconciler/testing/v1alpha1"
	. "knative.dev/pkg/reconciler/testing"
	_ "knative.dev/pkg/system/testing"

	"github.com/sigstore/sigstore/pkg/signature/kms/fake"
)

const (
	cipName           = "test-cip"
	cipKMSName        = "test-kms-cip"
	testKey           = "test-cip"
	cipName2          = "test-cip-2"
	testKey2          = "test-cip-2"
	keySecretName     = "publickey-key"
	keylessSecretName = "publickey-keyless"
	glob              = "ghcr.io/example/*"
	kmsKey            = "azure-kms://foo/bar"
	fakeKMSKey        = "fakekms://keycip"

	// Just some public key that was laying around, only format matters.
	validPublicKeyData = `-----BEGIN PUBLIC KEY-----
MFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAExB6+H6054/W1SJgs5JR6AJr6J35J
RCTfQ5s1kD+hGMSE1rH7s46hmXEeyhnlRnaGF8eMU/SBJE/2NKPnxE7WzQ==
-----END PUBLIC KEY-----`

	// This is the patch for replacing a single entry in the ConfigMap
	replaceCIPPatch = `[{"op":"replace","path":"/data/test-cip","value":"{\"images\":[{\"glob\":\"ghcr.io/example/*\"}],\"authorities\":[{\"key\":{\"publicKeys\":[{\"Curve\":{\"P\":115792089210356248762697446949407573530086143415290314195533631308867097853951,\"N\":115792089210356248762697446949407573529996955224135760342422259061068512044369,\"B\":41058363725152142129326129780047268409114441015993725554835256314039467401291,\"Gx\":48439561293906451759052585252797914202762949526041747995844080717082404635286,\"Gy\":36134250956749795798585127919587881956611106672985015071877198253568414405109,\"BitSize\":256,\"Name\":\"P-256\"},\"X\":88707635920070641755721733257804385133633538985728333477702186279848359565508,\"Y\":60084629679813308197062610650536235858799576629081565335301707208097929287373}]}}]}"}]`

	// This is the patch for adding an entry for non-existing KMS for cipName2
	addCIP2Patch = `[{"op":"add","path":"/data/test-cip-2","value":"{\"images\":[{\"glob\":\"ghcr.io/example/*\"}],\"authorities\":[{\"key\":{}}]}"}]`

	// This is the patch for removing the last entry, leaving just the
	// configmap objectmeta, no data.
	removeDataPatch = `[{"op":"remove","path":"/data"}]`

	// This is the patch for removing only a single entry from a map that has
	// two entries but only one is being removed. For key entry
	removeSingleEntryKeyPatch = `[{"op":"remove","path":"/data/test-cip"}]`

	// This is the patch for removing only a single entry from a map that has
	// two entries but only one is being removed. For keyless entry.
	removeSingleEntryKeylessPatch = `[{"op":"remove","path":"/data/test-cip-2"}]`

	// This is the patch for inlined secret for key ref data
	inlinedSecretKeyPatch = `[{"op":"replace","path":"/data/test-cip","value":"{\"images\":[{\"glob\":\"ghcr.io/example/*\"}],\"authorities\":[{\"key\":{\"publicKeys\":[{\"Curve\":{\"P\":115792089210356248762697446949407573530086143415290314195533631308867097853951,\"N\":115792089210356248762697446949407573529996955224135760342422259061068512044369,\"B\":41058363725152142129326129780047268409114441015993725554835256314039467401291,\"Gx\":48439561293906451759052585252797914202762949526041747995844080717082404635286,\"Gy\":36134250956749795798585127919587881956611106672985015071877198253568414405109,\"BitSize\":256,\"Name\":\"P-256\"},\"X\":88707635920070641755721733257804385133633538985728333477702186279848359565508,\"Y\":60084629679813308197062610650536235858799576629081565335301707208097929287373}]}}]}"}]`

	// This is the patch for inlined secret for keyless cakey ref data
	inlinedSecretKeylessPatch = `[{"op":"replace","path":"/data/test-cip-2","value":"{\"images\":[{\"glob\":\"ghcr.io/example/*\"}],\"authorities\":[{\"keyless\":{\"ca-cert\":{\"data\":\"-----BEGIN PUBLIC KEY-----\\nMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAExB6+H6054/W1SJgs5JR6AJr6J35J\\nRCTfQ5s1kD+hGMSE1rH7s46hmXEeyhnlRnaGF8eMU/SBJE/2NKPnxE7WzQ==\\n-----END PUBLIC KEY-----\"}}}]}"}]`
)

func TestReconcile(t *testing.T) {
	privKMSKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
	if err != nil {
		t.Fatalf("error generating ecdsa private key: %v", err)
	}
	mainContext := context.WithValue(context.Background(), fake.KmsCtxKey{}, privKMSKey)

	table := TableTest{{
		Name: "bad workqueue key",
		// Make sure Reconcile handles bad keys.
		Key: "too/many/parts",
	}, {
		Name: "key not found",
		// Make sure Reconcile handles good keys that don't exist.
		Key: "foo/not-found",
	}, {
		Name: "ClusterImagePolicy not found",
		Key:  testKey,
	}, {
		Name: "ClusterImagePolicy is being deleted, doesn't exist, no changes",
		Key:  testKey,
		Objects: []runtime.Object{
			NewClusterImagePolicy(cipName,
				WithClusterImagePolicyDeletionTimestamp),
		},
	}, {
		Name: "ClusterImagePolicy with glob and inline key data, added to cm and finalizer",
		Key:  testKey,

		SkipNamespaceValidation: true, // Cluster scoped
		Objects: []runtime.Object{
			NewClusterImagePolicy(cipName,
				WithImagePattern(v1alpha1.ImagePattern{
					Glob: glob,
				}),
				WithAuthority(v1alpha1.Authority{
					Key: &v1alpha1.KeyRef{
						Data: validPublicKeyData,
					}}))},
		WantCreates: []runtime.Object{
			makeConfigMap(),
		},
		WantPatches: []clientgotesting.PatchActionImpl{
			patchFinalizers(system.Namespace(), cipName),
		},
		WantEvents: []string{
			Eventf(corev1.EventTypeNormal, "FinalizerUpdate", `Updated "test-cip" finalizers`),
		},
	}, {
		Name: "ClusterImagePolicy with glob and inline key data, already exists, no patch",
		Key:  testKey,

		SkipNamespaceValidation: true, // Cluster scoped
		Objects: []runtime.Object{
			NewClusterImagePolicy(cipName,
				WithFinalizer,
				WithImagePattern(v1alpha1.ImagePattern{
					Glob: glob,
				}),
				WithAuthority(v1alpha1.Authority{
					Key: &v1alpha1.KeyRef{
						Data: validPublicKeyData,
					}})),
			makeConfigMap(),
		},
	}, {
		Name: "ClusterImagePolicy with glob and inline key data, needs a patch",
		Key:  testKey,

		SkipNamespaceValidation: true, // Cluster scoped
		Objects: []runtime.Object{
			NewClusterImagePolicy(cipName,
				WithFinalizer,
				WithImagePattern(v1alpha1.ImagePattern{
					Glob: glob,
				}),
				WithAuthority(v1alpha1.Authority{
					Key: &v1alpha1.KeyRef{
						Data: validPublicKeyData,
					}})),
			makeDifferentConfigMap(),
		},
		WantPatches: []clientgotesting.PatchActionImpl{
			makePatch(replaceCIPPatch),
		},
	}, {
		Name: "ClusterImagePolicy with glob and KMS key data, added as a patch",
		Key:  testKey2,

		SkipNamespaceValidation: true, // Cluster scoped
		Objects: []runtime.Object{
			NewClusterImagePolicy(cipName2,
				WithFinalizer,
				WithImagePattern(v1alpha1.ImagePattern{
					Glob: glob,
				}),
				WithAuthority(v1alpha1.Authority{
					Key: &v1alpha1.KeyRef{
						Data: kmsKey,
					}})),
			makeConfigMap(), // Make the existing configmap
		},
		WantPatches: []clientgotesting.PatchActionImpl{
			makePatch(addCIP2Patch),
		},
	},
		{
			Name: "ClusterImagePolicy with glob and inline key data, already exists, deleted",
			Key:  testKey,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							Data: validPublicKeyData,
						}}),
					WithClusterImagePolicyDeletionTimestamp),
				makeConfigMap(),
			},
			WantPatches: []clientgotesting.PatchActionImpl{
				patchRemoveFinalizers(system.Namespace(), cipName),
				makePatch(removeDataPatch),
			},
			WantEvents: []string{
				Eventf(corev1.EventTypeNormal, "FinalizerUpdate", `Updated "test-cip" finalizers`),
			},
		}, {
			Name: "Two entries, remove only one",
			Key:  testKey2,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName2,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							Data: validPublicKeyData,
						}}),
					WithClusterImagePolicyDeletionTimestamp),
				makeConfigMapWithTwoEntries(),
			},
			WantPatches: []clientgotesting.PatchActionImpl{
				patchRemoveFinalizers(system.Namespace(), cipName2),
				makePatch(removeSingleEntryKeylessPatch),
			},
			WantEvents: []string{
				Eventf(corev1.EventTypeNormal, "FinalizerUpdate", `Updated "test-cip-2" finalizers`),
			},
		}, {
			Name: "Key with secret, secret does not exist, no entry in configmap",
			Key:  testKey,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							SecretRef: &corev1.SecretReference{
								Name: keySecretName,
							},
						}}),
				),
				makeEmptyConfigMap(),
			},
			WantErr: true,
			WantEvents: []string{
				Eventf(corev1.EventTypeWarning, "InternalError", `secret "publickey-key" not found`),
			},
			PostConditions: []func(*testing.T, *TableRow){
				AssertTrackingSecret(system.Namespace(), keySecretName),
			},
		}, {
			Name: "Key with secret, secret does not exist, entry removed from configmap",
			Key:  testKey,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							SecretRef: &corev1.SecretReference{
								Name: keySecretName,
							},
						}}),
				),
				makeConfigMapWithTwoEntries(),
			},
			WantErr: true,
			WantEvents: []string{
				Eventf(corev1.EventTypeWarning, "InternalError", `secret "publickey-key" not found`),
			},
			WantPatches: []clientgotesting.PatchActionImpl{
				makePatch(removeSingleEntryKeyPatch),
			},
			PostConditions: []func(*testing.T, *TableRow){
				AssertTrackingSecret(system.Namespace(), keySecretName),
			},
		}, {
			Name: "Key with secret, secret does not exist, cm does not exist",
			Key:  testKey,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							SecretRef: &corev1.SecretReference{
								Name: keySecretName,
							},
						}}),
				),
			},
			WantErr: true,
			WantEvents: []string{
				Eventf(corev1.EventTypeWarning, "InternalError", `secret "publickey-key" not found`),
			},
			PostConditions: []func(*testing.T, *TableRow){
				AssertTrackingSecret(system.Namespace(), keySecretName),
			},
		}, {
			Name: "Keyless with secret, secret does not exist.",
			Key:  testKey,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Keyless: &v1alpha1.KeylessRef{
							CACert: &v1alpha1.KeyRef{
								SecretRef: &corev1.SecretReference{
									Name: keylessSecretName,
								},
							},
						}}),
				),
				makeConfigMapWithTwoEntries(),
			},
			WantErr: true,
			WantEvents: []string{
				Eventf(corev1.EventTypeWarning, "InternalError", `secret "publickey-keyless" not found`),
			},
			WantPatches: []clientgotesting.PatchActionImpl{
				makePatch(removeSingleEntryKeyPatch),
			},
			PostConditions: []func(*testing.T, *TableRow){
				AssertTrackingSecret(system.Namespace(), keylessSecretName),
			},
		}, {
			Name: "Key with secret, no data.",
			Key:  testKey,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							SecretRef: &corev1.SecretReference{
								Name: keySecretName,
							},
						}}),
				),
				&corev1.Secret{
					ObjectMeta: metav1.ObjectMeta{
						Namespace: system.Namespace(),
						Name:      keySecretName,
					},
				},
				makeEmptyConfigMap(),
			},
			WantErr: true,
			WantEvents: []string{
				Eventf(corev1.EventTypeWarning, "InternalError", `secret "publickey-key" contains no data`),
			},
			PostConditions: []func(*testing.T, *TableRow){
				AssertTrackingSecret(system.Namespace(), keySecretName),
			},
		}, {
			Name: "Key with secret, multiple data entries.",
			Key:  testKey,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							SecretRef: &corev1.SecretReference{
								Name: keySecretName,
							},
						}}),
				),
				&corev1.Secret{
					ObjectMeta: metav1.ObjectMeta{
						Namespace: system.Namespace(),
						Name:      keySecretName,
					},
					Data: map[string][]byte{
						"first":  []byte("first data"),
						"second": []byte("second data"),
					},
				},
				makeEmptyConfigMap(),
			},
			WantErr: true,
			WantEvents: []string{
				Eventf(corev1.EventTypeWarning, "InternalError", `secret "publickey-key" contains multiple data entries, only one is supported`),
			},
			PostConditions: []func(*testing.T, *TableRow){
				AssertTrackingSecret(system.Namespace(), keySecretName),
			},
		}, {
			Name: "Key with secret, secret exists, invalid public key",
			Key:  testKey,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							SecretRef: &corev1.SecretReference{
								Name: keySecretName,
							},
						}}),
				),
				makeEmptyConfigMap(),
				makeSecret(keySecretName, "garbage secret value, not a public key"),
			},
			WantErr: true,
			WantEvents: []string{
				Eventf(corev1.EventTypeWarning, "InternalError", `secret "publickey-key" contains an invalid public key`),
			},
			PostConditions: []func(*testing.T, *TableRow){
				AssertTrackingSecret(system.Namespace(), keySecretName),
			},
		}, {
			Name: "Key with secret, secret exists, inlined",
			Key:  testKey,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							SecretRef: &corev1.SecretReference{
								Name: keySecretName,
							},
						}}),
				),
				makeConfigMapWithTwoEntriesNotPublicKeyFromSecret(),
				makeSecret(keySecretName, validPublicKeyData),
			},
			WantPatches: []clientgotesting.PatchActionImpl{
				makePatch(inlinedSecretKeyPatch),
			},
			PostConditions: []func(*testing.T, *TableRow){
				AssertTrackingSecret(system.Namespace(), keySecretName),
			},
		}, {
			Name: "Keyless with secret, secret exists, inlined",
			Key:  testKey2,

			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipName2,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Keyless: &v1alpha1.KeylessRef{
							CACert: &v1alpha1.KeyRef{
								SecretRef: &corev1.SecretReference{
									Name: keylessSecretName,
								}},
						}}),
				),
				makeConfigMapWithTwoEntries(),
				makeSecret(keylessSecretName, validPublicKeyData),
			},
			WantPatches: []clientgotesting.PatchActionImpl{
				makePatch(inlinedSecretKeylessPatch),
			},
			PostConditions: []func(*testing.T, *TableRow){
				AssertTrackingSecret(system.Namespace(), keylessSecretName),
			},
		}, {
			Name:                    "ClusterImagePolicy with glob and KMS key, added the data after querying the fake signer",
			Key:                     cipKMSName,
			SkipNamespaceValidation: true, // Cluster scoped
			Objects: []runtime.Object{
				NewClusterImagePolicy(cipKMSName,
					WithFinalizer,
					WithImagePattern(v1alpha1.ImagePattern{
						Glob: glob,
					}),
					WithAuthority(v1alpha1.Authority{
						Key: &v1alpha1.KeyRef{
							KMS: fakeKMSKey,
						}})),
				makeEmptyConfigMap(), // Make the existing configmap
			},
			WantPatches: []clientgotesting.PatchActionImpl{
				patchKMS(mainContext, t, fakeKMSKey),
			},
		}, {}}

	logger := logtesting.TestLogger(t)
	table.Test(t, MakeFactory(func(ctx context.Context, listers *Listers, cmw configmap.Watcher) controller.Reconciler {
		r := &Reconciler{
			secretlister:    listers.GetSecretLister(),
			configmaplister: listers.GetConfigMapLister(),
			kubeclient:      fakekubeclient.Get(ctx),
			tracker:         ctx.Value(TrackerKey).(tracker.Interface),
		}
		return clusterimagepolicy.NewReconciler(ctx, logger,
			fakecosignclient.Get(ctx), listers.GetClusterImagePolicyLister(),
			controller.GetEventRecorder(ctx),
			r)
	},
		false,
		logger,
		privKMSKey,
	))
}

func makeSecret(name, secret string) *corev1.Secret {
	return &corev1.Secret{
		ObjectMeta: metav1.ObjectMeta{
			Namespace: system.Namespace(),
			Name:      name,
		},
		Data: map[string][]byte{
			"publicKey": []byte(secret),
		},
	}
}

func makeEmptyConfigMap() *corev1.ConfigMap {
	return &corev1.ConfigMap{
		ObjectMeta: metav1.ObjectMeta{
			Namespace: system.Namespace(),
			Name:      config.ImagePoliciesConfigName,
		},
	}
}

func makeConfigMap() *corev1.ConfigMap {
	return &corev1.ConfigMap{
		ObjectMeta: metav1.ObjectMeta{
			Namespace: system.Namespace(),
			Name:      config.ImagePoliciesConfigName,
		},
		Data: map[string]string{
			cipName: `{"images":[{"glob":"ghcr.io/example/*"}],"authorities":[{"key":{"publicKeys":[{"Curve":{"P":115792089210356248762697446949407573530086143415290314195533631308867097853951,"N":115792089210356248762697446949407573529996955224135760342422259061068512044369,"B":41058363725152142129326129780047268409114441015993725554835256314039467401291,"Gx":48439561293906451759052585252797914202762949526041747995844080717082404635286,"Gy":36134250956749795798585127919587881956611106672985015071877198253568414405109,"BitSize":256,"Name":"P-256"},"X":88707635920070641755721733257804385133633538985728333477702186279848359565508,"Y":60084629679813308197062610650536235858799576629081565335301707208097929287373}]}}]}`,
		},
	}
}

func patchKMS(ctx context.Context, t *testing.T, kmsKey string) clientgotesting.PatchActionImpl {
	pubKey, err := getKMSPublicKey(ctx, kmsKey)
	if err != nil {
		t.Fatalf("Failed to read KMS key ID %q: %v", kmsKey, err)
	}

	authorityKeys, err := convertAuthorityKeys(ctx, pubKey)
	if err != nil || len(authorityKeys) == 0 {
		t.Fatalf("Failed to convert KMS public key ID %q: %v", kmsKey, err)
	}

	pubKeyString := fmt.Sprintf(`{\"Curve\":{\"P\":115792089210356248762697446949407573530086143415290314195533631308867097853951,\"N\":115792089210356248762697446949407573529996955224135760342422259061068512044369,\"B\":41058363725152142129326129780047268409114441015993725554835256314039467401291,\"Gx\":48439561293906451759052585252797914202762949526041747995844080717082404635286,\"Gy\":36134250956749795798585127919587881956611106672985015071877198253568414405109,\"BitSize\":256,\"Name\":\"P-256\"},\"X\":%d,\"Y\":%d}`, authorityKeys[0].X, authorityKeys[0].Y)
	patch := `[{"op":"add","path":"/data","value":{"test-kms-cip":"{\"images\":[{\"glob\":\"ghcr.io/example/*\"}],\"authorities\":[{\"key\":{\"publicKeys\":[` + pubKeyString + `]}}]}"}}]`

	return clientgotesting.PatchActionImpl{
		ActionImpl: clientgotesting.ActionImpl{
			Namespace: system.Namespace(),
		},
		Name:  config.ImagePoliciesConfigName,
		Patch: []byte(patch),
	}
}

// Same as above, just forcing an update by changing PUBLIC => NOTPUBLIC
func makeDifferentConfigMap() *corev1.ConfigMap {
	return &corev1.ConfigMap{
		ObjectMeta: metav1.ObjectMeta{
			Namespace: system.Namespace(),
			Name:      config.ImagePoliciesConfigName,
		},
		Data: map[string]string{
			cipName: `{"images":[{"glob":"ghcr.io/example/*"}],"authorities":[{"key":{"data":"-----BEGIN NOTPUBLIC KEY-----\nMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAExB6+H6054/W1SJgs5JR6AJr6J35J\nRCTfQ5s1kD+hGMSE1rH7s46hmXEeyhnlRnaGF8eMU/SBJE/2NKPnxE7WzQ==\n-----END NOTPUBLIC KEY-----","publicKeys":[{"Curve":{"P":115792089210356248762697446949407573530086143415290314195533631308867097853951,"N":115792089210356248762697446949407573529996955224135760342422259061068512044369,"B":41058363725152142129326129780047268409114441015993725554835256314039467401291,"Gx":48439561293906451759052585252797914202762949526041747995844080717082404635286,"Gy":36134250956749795798585127919587881956611106672985015071877198253568414405109,"BitSize":256,"Name":"P-256"},"X":88707635920070641755721733257804385133633538985728333477702186279848359565508,"Y":60084629679813308197062610650536235858799576629081565335301707208097929287373}]}}]}`,
		},
	}
}

// Same as MakeConfigMap but a placeholder for second entry so we can remove it.
func makeConfigMapWithTwoEntries() *corev1.ConfigMap {
	return &corev1.ConfigMap{
		ObjectMeta: metav1.ObjectMeta{
			Namespace: system.Namespace(),
			Name:      config.ImagePoliciesConfigName,
		},
		Data: map[string]string{
			cipName:  `{"images":[{"glob":"ghcr.io/example/*"}],"authorities":[{"key":{\"publicKeys\":[{\"Curve\":{\"P\":115792089210356248762697446949407573530086143415290314195533631308867097853951,\"N\":115792089210356248762697446949407573529996955224135760342422259061068512044369,\"B\":41058363725152142129326129780047268409114441015993725554835256314039467401291,\"Gx\":48439561293906451759052585252797914202762949526041747995844080717082404635286,\"Gy\":36134250956749795798585127919587881956611106672985015071877198253568414405109,\"BitSize\":256,\"Name\":\"P-256\"},\"X\":88707635920070641755721733257804385133633538985728333477702186279848359565508,\"Y\":60084629679813308197062610650536235858799576629081565335301707208097929287373}]}}]}`,
			cipName2: "remove me please",
		},
	}
}

// Same as MakeConfigMapWithTwoEntries but the inline data is not the secret
// so we will replace it with the secret data
func makeConfigMapWithTwoEntriesNotPublicKeyFromSecret() *corev1.ConfigMap {
	return &corev1.ConfigMap{
		ObjectMeta: metav1.ObjectMeta{
			Namespace: system.Namespace(),
			Name:      config.ImagePoliciesConfigName,
		},
		Data: map[string]string{
			cipName:  `{"images":[{"glob":"ghcr.io/example/*"}],"authorities":[{"key":{"data":"NOT A REAL PUBLIC KEY"}}]}`,
			cipName2: "remove me please",
		},
	}
}

func makePatch(patch string) clientgotesting.PatchActionImpl {
	return clientgotesting.PatchActionImpl{
		ActionImpl: clientgotesting.ActionImpl{
			Namespace: system.Namespace(),
		},
		Name:  config.ImagePoliciesConfigName,
		Patch: []byte(patch),
	}
}

func patchFinalizers(namespace, name string) clientgotesting.PatchActionImpl {
	action := clientgotesting.PatchActionImpl{}
	action.Name = name
	action.Namespace = namespace
	patch := `{"metadata":{"finalizers":["` + finalizerName + `"],"resourceVersion":""}}`
	action.Patch = []byte(patch)
	return action
}

func patchRemoveFinalizers(namespace, name string) clientgotesting.PatchActionImpl {
	action := clientgotesting.PatchActionImpl{}
	action.Name = name
	action.Namespace = namespace
	patch := `{"metadata":{"finalizers":[],"resourceVersion":""}}`
	action.Patch = []byte(patch)
	return action
}
