//
// Copyright 2021 The Sigstore Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package sget

import (
	"context"
	"io"

	"github.com/google/go-containerregistry/pkg/authn"
	"github.com/google/go-containerregistry/pkg/name"
	"github.com/google/go-containerregistry/pkg/v1/remote"
	"github.com/pkg/errors"

	"github.com/sigstore/cosign/cmd/cosign/cli"
	"github.com/sigstore/cosign/cmd/cosign/cli/fulcio"
	"github.com/sigstore/cosign/pkg/cosign"
)

func New(image, key string, out io.Writer) *SecureGet {
	return &SecureGet{
		ImageRef: image,
		KeyRef:   key,
		Out:      out,
	}
}

type SecureGet struct {
	ImageRef string
	KeyRef   string
	Out      io.Writer
}

func (sg *SecureGet) Do(ctx context.Context) error {
	ref, err := name.ParseReference(sg.ImageRef)
	if err != nil {
		return err
	}

	if _, ok := ref.(name.Tag); ok {
		if sg.KeyRef == "" && !cli.EnableExperimental() {
			return errors.New("public key must be specified when fetching by tag, you must fetch by digest or supply a public key")
		}
	}

	co := &cosign.CheckOpts{
		ClaimVerifier: cosign.SimpleClaimVerifier,
		VerifyBundle:  true,
		RegistryClientOpts: []remote.Option{
			remote.WithAuthFromKeychain(authn.DefaultKeychain),
			remote.WithContext(ctx),
		},
	}
	if sg.KeyRef != "" {
		pub, err := cli.LoadPublicKey(ctx, sg.KeyRef)
		if err != nil {
			return err
		}
		co.SigVerifier = pub
	}

	if co.SigVerifier != nil || cli.EnableExperimental() {
		co.RootCerts = fulcio.GetRoots()
		sigRepo, err := cli.TargetRepositoryForImage(ref)
		if err != nil {
			return err
		}
		co.SignatureRepo = sigRepo

		sp, err := cosign.Verify(ctx, ref, co)
		if err != nil {
			return err
		}
		cli.PrintVerificationHeader(sg.ImageRef, co)
		cli.PrintVerification(sg.ImageRef, sp, "text")
	}

	img, err := remote.Image(ref, co.RegistryClientOpts...)
	if err != nil {
		return err
	}
	layers, err := img.Layers()
	if err != nil {
		return err
	}
	if len(layers) != 1 {
		return errors.New("invalid artifact")
	}
	rc, err := layers[0].Compressed()
	if err != nil {
		return err
	}

	_, err = io.Copy(sg.Out, rc)
	if err != nil {
		return err
	}
	return nil
}
