package kafka

import (
	"fmt"
	"strings"

	"github.com/spf13/cobra"

	pcmd "github.com/confluentinc/cli/v4/pkg/cmd"
	"github.com/confluentinc/cli/v4/pkg/examples"
	"github.com/confluentinc/cli/v4/pkg/output"
)

func (c *clusterCommand) newConfigurationDescribeCommand() *cobra.Command {
	cmd := &cobra.Command{
		Use:   "describe <name>",
		Short: "Describe a Kafka cluster configuration.",
		Args:  cobra.ExactArgs(1),
		RunE:  c.configurationDescribe,
		Example: examples.BuildExampleString(
			examples.Example{
				Text: `Describe Kafka cluster configuration "auto.create.topics.enable".`,
				Code: "confluent kafka cluster configuration describe auto.create.topics.enable",
			},
		),
	}

	pcmd.AddContextFlag(cmd, c.CLICommand)
	pcmd.AddEnvironmentFlag(cmd, c.AuthenticatedCLICommand)
	pcmd.AddClusterFlag(cmd, c.AuthenticatedCLICommand)
	pcmd.AddOutputFlag(cmd)

	return cmd
}

func (c *clusterCommand) configurationDescribe(cmd *cobra.Command, args []string) error {
	kafkaREST, err := c.GetKafkaREST()
	if err != nil {
		return err
	}

	config, err := kafkaREST.CloudClient.GetKafkaClusterConfig(args[0])
	if err != nil {
		return catchConfigurationNotFound(err, args[0])
	}

	table := output.NewTable(cmd)
	table.Add(&configurationOut{
		Name:     config.GetName(),
		Value:    config.GetValue(),
		ReadOnly: config.GetIsReadOnly(),
	})
	return table.Print()
}

func catchConfigurationNotFound(err error, configuration string) error {
	if err != nil && strings.Contains(err.Error(), "Not Found") {
		return fmt.Errorf(`configuration "%s" not found`, configuration)
	}
	return err
}
