--  Copyright 1994 Grady Booch
--  Copyright 1994-1997 David Weller
--  Copyright 1998-2002 Simon Wright <simon@pushface.org>

--  This package is free software; you can redistribute it and/or
--  modify it under terms of the GNU General Public License as
--  published by the Free Software Foundation; either version 2, or
--  (at your option) any later version. This package is distributed in
--  the hope that it will be useful, but WITHOUT ANY WARRANTY; without
--  even the implied warranty of MERCHANTABILITY or FITNESS FOR A
--  PARTICULAR PURPOSE. See the GNU General Public License for more
--  details. You should have received a copy of the GNU General Public
--  License distributed with this package; see file COPYING.  If not,
--  write to the Free Software Foundation, 59 Temple Place - Suite
--  330, Boston, MA 02111-1307, USA.

--  As a special exception, if other files instantiate generics from
--  this unit, or you link this unit with other files to produce an
--  executable, this unit does not by itself cause the resulting
--  executable to be covered by the GNU General Public License.  This
--  exception does not however invalidate any other reasons why the
--  executable file might be covered by the GNU Public License.

--  $Revision: 1391 $
--  $Date: 2009-01-12 21:55:33 +0100 (Mo, 12 Jan 2009) $
--  $Author: simonjwright $

with System;

package body BC.Containers.Stacks is

   procedure Process_Top (S : in out Abstract_Stack'Class) is
   begin
      Process (Item_At (S, 1).all);
   end Process_Top;

   function Are_Equal (Left, Right : Abstract_Stack'Class) return Boolean is
   begin
      if System."=" (Left'Address, Right'Address) then
         return True;
      end if;
      if Depth (Left) /= Depth (Right) then
         return False;
      end if;
      declare
         Left_Iter : Iterator'Class := New_Iterator (Left);
         Right_Iter : Iterator'Class := New_Iterator (Right);
      begin
         while not Is_Done (Left_Iter) and then
           not Is_Done (Right_Iter) loop
            if Current_Item (Left_Iter) /= Current_Item (Right_Iter) then
               return False;
            end if;
            Next (Left_Iter);
            Next (Right_Iter);
         end loop;
         return True;
      end;
   end Are_Equal;

   procedure Copy (From : Abstract_Stack'Class;
                   To : in out Abstract_Stack'Class) is
      Iter : Iterator'Class := New_Iterator (From);
   begin
      if System."/=" (From'Address, To'Address) then
         Clear (To);
         Reset (Iter);
         while not Is_Done (Iter) loop
            Add (To, Current_Item (Iter));
            Next (Iter);
         end loop;
      end if;
   end Copy;

   function Available (S : in Abstract_Stack) return Natural is
      pragma Warnings (Off, S);
   begin
      return Natural'Last;
   end Available;

   --  Subprograms to be overridden

   procedure Add (S : in out Abstract_Stack; Elem : Item) is
   begin
      raise Should_Have_Been_Overridden;
   end Add;

   procedure Remove (S : in out Abstract_Stack; From : Positive) is
   begin
      raise Should_Have_Been_Overridden;
   end Remove;

   --  Iterators

   procedure Reset (It : in out Stack_Iterator) is
      S : Abstract_Stack'Class
        renames Abstract_Stack'Class (It.For_The_Container.all);
   begin
      if Depth (S) = 0 then
         It.Index := 0;
      else
         It.Index := 1;
      end if;
   end Reset;

   procedure Next (It : in out Stack_Iterator) is
   begin
      It.Index := It.Index + 1;
   end Next;

   function Is_Done (It : Stack_Iterator) return Boolean is
      S : Abstract_Stack'Class
     renames Abstract_Stack'Class (It.For_The_Container.all);
   begin
      return It.Index = 0 or else It.Index > Depth (S);
   end Is_Done;

   function Current_Item (It : Stack_Iterator) return Item is
   begin
      if Is_Done (It) then
         raise BC.Not_Found;
      end if;
      return Item_At (It.For_The_Container.all, It.Index).all;
   end Current_Item;

   function Current_Item_Ptr (It : Stack_Iterator) return Item_Ptr is
   begin
      if Is_Done (It) then
         raise BC.Not_Found;
      end if;
      return Item_At (It.For_The_Container.all, It.Index);
   end Current_Item_Ptr;

   procedure Delete_Item_At (It : in out Stack_Iterator) is
      S : Abstract_Stack'Class
        renames Abstract_Stack'Class (It.For_The_Container.all);
   begin
      if Is_Done (It) then
         raise BC.Not_Found;
      end if;
      Remove (S, It.Index);
   end Delete_Item_At;

end BC.Containers.Stacks;
