/*-
 * GMTK_GMParms.cc
 *        All parameters associated with a GM.
 *
 *
 * Written by Jeff Bilmes <bilmes@ee.washington.edu>
 *
 * Copyright (C) 2001 Jeff Bilmes
 * Licensed under the Open Software License version 3.0
 * See COPYING or http://opensource.org/licenses/OSL-3.0
 *
 *
 */


#if HAVE_CONFIG_H
#include <config.h>
#endif
#if HAVE_HG_H
#include "hgstamp.h"
#endif
#if HAVE_DLFCN_H
#include <dlfcn.h>
#endif

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <iostream>
#include <errno.h>
#include <string.h>
#include <float.h>
#include <assert.h>

#include "general.h"
#include "error.h"

#include "GMTK_GMParms.h"

#include "GMTK_Dense1DPMF.h"
#include "GMTK_Sparse1DPMF.h"
#include "GMTK_MeanVector.h"
#include "GMTK_DiagCovarVector.h"
#include "GMTK_RealMatrix.h"
#include "GMTK_DoubleMatrix.h"
#include "GMTK_GammaComponent.h"
#include "GMTK_BetaComponent.h"
#include "GMTK_DlinkMatrix.h"
#include "GMTK_Dlinks.h"
#include "GMTK_DirichletTable.h"
#include "GMTK_MDCPT.h"
#include "GMTK_MSCPT.h"
#include "GMTK_MTCPT.h"
#include "GMTK_USCPT.h"
#include "GMTK_NGramCPT.h"
#include "GMTK_FNGramCPT.h"
#include "GMTK_VECPT.h"
#include "GMTK_DeepNN.h"
#include "GMTK_DeepVECPT.h"
#include "GMTK_DeepCPT.h"
#include "GMTK_Vocab.h"
#include "GMTK_LatticeADT.h"
#include "GMTK_LatticeNodeCPT.h"
#include "GMTK_LatticeEdgeCPT.h"
#include "GMTK_NameCollection.h"

// particular Gaussian components
#include "GMTK_GaussianComponent.h"
#include "GMTK_DiagGaussian.h"
#include "GMTK_LinMeanCondDiagGaussian.h"

#include "GMTK_MixtureCommon.h"
#include "GMTK_Mixture.h"
#include "GMTK_ZeroScoreMixture.h"
#include "GMTK_UnityScoreMixture.h"

#include "GMTK_ObservationSource.h"

#include "GMTK_CFunctionDeterministicMappings.h"

VCID(HGID)

/////////////////////////////////
// This is to allow dynamically loaded C mappers to get
// access to the ObservationSource
extern ObservationSource *globalObservationMatrix;


/////////////////////////////////
// an integer that specifies the maximum number of objects (such
// as means, covariances, DTs, etc.) that may be specified at
// one time in a file. This can be safely increased (to the
// extend that memory on the machine exists), but is here
// many for checking obviously invalid values.
const unsigned GMPARMS_MAX_NUM = 900000000;

////////////////////////////////
// Magic String definitions
#define MAGIC_DT_FILE "GMTK_DT_FILE"
#define MAGIC_PRM_FILE "GMTK_PRM_FILE"

////////////////////////////////////////////////
// Special name for global collection name. I.e.,
// with these names the arrays will index the global 
// entries.
#define NAMED_COLLECTION_GLOBAL_NAME "global"

// possibly use this at some point for ascii files.
// static const char* sectionSeparator = "############################################################################";


////////////////////////////////////////////////////////////////////
//        General create, read, destroy routines 
////////////////////////////////////////////////////////////////////


// from tieSupport
Mixture* find_Mixture(std::string &name);

GMParms::GMParms()
{
  // Read in all interal C-function deterministic mappers.
  registerAllCFunctionDeterministicMappings(*this);
}



void GMParms::clearParms() {
    mdCpts.clear();
    mdCptsMap.clear();

    msCpts.clear();
    msCptsMap.clear();

    mtCpts.clear();
    mtCptsMap.clear();

    dts.clear();
    dtsMap.clear();

    ncls.clear();
    nclsMap.clear();

    dPmfs.clear();
    dPmfsMap.clear();

    means.clear();
    meansMap.clear();

    covars.clear();
    covarsMap.clear();

    components.clear();
    componentsMap.clear();

    mixtures.clear();
    mixturesMap.clear();

    dirichletTabs.clear();
    dirichletTabsMap.clear();

    //Add back the built-in C mapping functions back to the DT map.
    registerAllCFunctionDeterministicMappings(*this);
}



GMParms & GMParms::operator=(const GMParms & that) {
        if(this  == &that) return *this;

        //registerAllCFunctionDeterministicMappings(*this);
        this->dPmfs = that.dPmfs;
        this->dPmfsMap = that.dPmfsMap;

        this->sPmfs = that.sPmfs;
        this->sPmfsMap = that.sPmfsMap;

        this->means = that.means;
        this->meansMap = that.meansMap;
        
        this->covars = that.covars;
        this->covarsMap = that.covarsMap;

        this->dLinkMats = that.dLinkMats;
        this->dLinkMatsMap = that.dLinkMatsMap;

        this->realMats = that.realMats;
        this->realMatsMap = that.realMatsMap;

        this->doubleMats = that.doubleMats;
        this->doubleMatsMap = that.doubleMatsMap;

        this->dirichletTabs = that.dirichletTabs;
        this->dirichletTabsMap = that.dirichletTabsMap;

        this->components = that.components;
        this->componentsMap = that.componentsMap;



        this->mdCpts = that.mdCpts;
        this->mdCptsMap = that.mdCptsMap;

        this->msCpts = that.msCpts;
        this->msCptsMap = that.msCptsMap;

        this->mtCpts = that.mtCpts;
        this->mtCptsMap = that.mtCptsMap;

        this->vocabs = that.vocabs;
        this->vocabsMap = that.vocabsMap;



        this->ngramCpts = that.ngramCpts;
        this->ngramCptsMap = that.ngramCptsMap;

        this->fngramCpts = that.fngramCpts;
        this->fngramCptsMap = that.fngramCptsMap;

        this->fngramImps = that.fngramImps;
        this->fngramImpsMap = that.fngramImpsMap;



        this->latticeAdts = that.latticeAdts;
        this->latticeAdtsMap = that.latticeAdtsMap;

        this->iterableLatticeAdts = that.iterableLatticeAdts;

        this->latticeNodeCpts = that.latticeNodeCpts;
        this->latticeNodeCptsMap = that.latticeNodeCptsMap;

        this->latticeEdgeCpts = that.latticeEdgeCpts;
        this->latticeEdgeCptsMap = that.latticeEdgeCptsMap;



        this->veCpts = that.veCpts;
        this->veCptsMap = that.veCptsMap;

        this->deepVECpts = that.deepVECpts;
        this->deepVECptsMap = that.deepVECptsMap;

        this->deepCpts = that.deepCpts;
        this->deepCptsMap = that.deepCptsMap;

        this->deepNNs = that.deepNNs;
        this->deepNNsMap = that.deepNNsMap;



        this->mixtures = that.mixtures;
        this->mixturesMap = that.mixturesMap;

        this->gausSwitchingMixtures = that.gausSwitchingMixtures;
        this->gausSwitchingMixturesMap = that.gausSwitchingMixturesMap;

        this->logitSwitchingMixtures = that.logitSwitchingMixtures;
        this->logitSwitchingMixturesMap = that.logitSwitchingMixturesMap;

        this->mlpSwitchingMixtures = that.mlpSwitchingMixtures;
        this->mlpSwitchingMixturesMap = that.mlpSwitchingMixturesMap;



        this->dts = that.dts;
        this->dtsMap = that.dtsMap;

        this->iterableDts = that.iterableDts;

        this->dLinks = that.dLinks;
        this->dLinksMap = that.dLinksMap;
        
        this->ncls = that.ncls;
        this->nclsMap = that.nclsMap;
    
        this->gm = that.gm;

        this->firstUtterance = that.firstUtterance;

        return *this;
    }


GMParms::~GMParms()
{
  deleteObsInVector(dPmfs);
  deleteObsInVector(sPmfs);
  deleteObsInVector(means);
  deleteObsInVector(covars);
  deleteObsInVector(dLinkMats);
  deleteObsInVector(realMats);
  deleteObsInVector(doubleMats);
  deleteObsInVector(dirichletTabs);
  deleteObsInVector(components);
  deleteObsInVector(mdCpts);
  deleteObsInVector(msCpts);
  deleteObsInVector(mtCpts);
  deleteObsInVector(vocabs);
  deleteObsInVector(ngramCpts);
  deleteObsInVector(fngramCpts);
  deleteObsInVector(latticeAdts);
  deleteObsInVector(latticeNodeCpts);
  deleteObsInVector(veCpts);
  deleteObsInVector(deepVECpts);
  deleteObsInVector(deepNNs);
  deleteObsInVector(mixtures);
  // deleteObsInVector(gausSwitchingMixtures);
  // deleteObsInVector(logitSwitchingMixtures);
  // deleteObsInVector(mlpSwitchingMixtures);
  deleteObsInVector(dts);
  deleteObsInVector(dLinks);
}

////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
//        Adding  Routines
////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////

void GMParms::add(Dense1DPMF* ob) { add(ob,dPmfs,dPmfsMap); }
void GMParms::add(Sparse1DPMF* ob) { add(ob,sPmfs,sPmfsMap); }
void GMParms::add(MeanVector* ob){ add(ob,means,meansMap); }
void GMParms::add(DiagCovarVector* ob){ add(ob,covars,covarsMap); }
void GMParms::add(DlinkMatrix* ob) { add(ob,dLinkMats,dLinkMatsMap); }
void GMParms::add(RealMatrix*ob) { add(ob,realMats,realMatsMap); }
void GMParms::add(DoubleMatrix*ob) { add(ob,doubleMats,doubleMatsMap); }
void GMParms::add(DirichletTable*ob) { add(ob,dirichletTabs,dirichletTabsMap); }
void GMParms::add(Component*ob){ add(ob,
				     components,
				     componentsMap); }
void GMParms::add(MDCPT*ob) { add(ob,mdCpts,mdCptsMap); }
void GMParms::add(MSCPT*ob) { add(ob,msCpts,msCptsMap); }
void GMParms::add(MTCPT*ob) { add(ob,mtCpts,mtCptsMap); }
void GMParms::add(Vocab* ob) { add(ob, vocabs, vocabsMap); }
void GMParms::add(NGramCPT* ob) { add(ob,ngramCpts,ngramCptsMap); }
void GMParms::add(FNGramCPT* ob) { add(ob,fngramCpts,fngramCptsMap); }
void GMParms::add(FNGramImp* ob) { add(ob,fngramImps,fngramImpsMap); }
void GMParms::add(LatticeADT* ob) {
	add(ob,latticeAdts,latticeAdtsMap);
	if ( ob->iterable() )
		iterableLatticeAdts.push_back(ob);
}
void GMParms::add(LatticeNodeCPT* ob) { add(ob,latticeNodeCpts,latticeNodeCptsMap); }
void GMParms::add(LatticeEdgeCPT* ob) { add(ob,latticeEdgeCpts,latticeEdgeCptsMap); }
void GMParms::add(VECPT*ob) { add(ob,veCpts,veCptsMap); }
void GMParms::add(DeepNN*ob) { add(ob,deepNNs,deepNNsMap); }
void GMParms::add(DeepVECPT*ob) { add(ob,deepVECpts,deepVECptsMap); }
void GMParms::add(Mixture*ob) { add(ob,mixtures,mixturesMap); }
void GMParms::add(GausSwitchingMixture*ob) { assert (0); }
void GMParms::add(LogitSwitchingMixture*ob) { assert (0); }
void GMParms::add(MLPSwitchingMixture*ob) { assert (0); }
void GMParms::add(RngDecisionTree*ob) {
  add(ob,dts,dtsMap);
  if (ob->iterable())
    iterableDts.push_back(ob);
}
void GMParms::add(Dlinks* ob) { add(ob,dLinks,dLinksMap); }


////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
//        READING Routines
////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////


/*-
 *-----------------------------------------------------------------------
 * readDPmfs
 *      Read in the dense PMF functions from a file.
 *
 * Preconditions:
 *      No conditions.
 *
 * Postconditions:
 *      All dense PMFs from file are now contained in internal arrays.
 *
 * Side Effects:
 *      Modifies internal arrays of object.
 *
 * Results:
 *      none
 *
 *-----------------------------------------------------------------------
 */
void
GMParms::readDPmfs(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num Dense PMFs");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of dense PMFs (%d) in file '%s' line %d exceeds maximum",num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    dPmfs.resize(num);
  } else {
    start = dPmfs.size();
    dPmfs.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    Dense1DPMF* ob;

    is.read(cnt,"Can't read DPMF number");
    if (cnt != i) 
      error("ERROR: dense PMF count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new Dense1DPMF;
    ob->read(is);
    if (dPmfsMap.find(ob->name()) != dPmfsMap.end())
      error("ERROR: dense PMF named '%s' already defined but is specified for a second time in file '%s' line %d",ob->name().c_str(),is.fileName(),is.lineNo());
    dPmfs[i+start] = ob;
    dPmfsMap[ob->name()] = i+start;
  }
}


/*-
 *-----------------------------------------------------------------------
 * readSPmfs
 *      Read in the sparse PMF functions from a file.
 * 
 * Preconditions:
 *      No conditions.
 *
 * Postconditions:
 *      All sparse PMFs from file are now contained in internal arrays.
 *
 * Side Effects:
 *      Modifies internal arrays of object.
 *
 * Results:
 *      none
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::readSPmfs(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num Sparse PMFs");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of sparse PMFs (%d) in file '%s' line %d exceeds maximum",num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    sPmfs.resize(num);
  } else {
    start = sPmfs.size();
    sPmfs.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    Sparse1DPMF* ob;

    is.read(cnt,"Can't read SPMF number");
    if (cnt != i) 
      error("ERROR: dense PMF count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new Sparse1DPMF;
    ob->read(is);
    if (sPmfsMap.find(ob->name()) != sPmfsMap.end())
      error("ERROR: sparse PMF named '%s' already defined but is specified for a second time in file '%s' line %d",ob->name().c_str(),is.fileName(),is.lineNo());
    sPmfs[i+start] = ob;
    sPmfsMap[ob->name()] = i+start;
  }
}


/*-
 *-----------------------------------------------------------------------
 * readMeans
 *      Read in the mean from a file.
 * 
 * Preconditions:
 *      No conditions.
 *
 * Postconditions:
 *      All means from file are now contained in internal arrays.
 *
 * Side Effects:
 *      Modifies internal arrays of object.
 *
 * Results:
 *      none
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::readMeans(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num means");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of means (%d) in file '%s' line %d exceeds maximum",num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    means.resize(num);
  } else {
    start = means.size();
    means.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    MeanVector* ob;

    is.read(cnt,"Can't read mean number");
    if (cnt != i) 
      error("ERROR: mean count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new MeanVector();
    ob->read(is);
    // printf("New mean, ob's name = %s\n",ob->name().c_str());
    if (meansMap.find(ob->name()) != meansMap.end())
      error("ERROR: mean named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    means[i+start] = ob;
    meansMap[ob->name()] = i+start;
  }
}


void 
GMParms::readCovars(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num covars");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of covars (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    covars.resize(num);
  } else {
    start = covars.size();
    covars.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    DiagCovarVector* ob;

    is.read(cnt,"Can't read covar num");
    if (cnt != i) 
      error("ERROR: covar count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new DiagCovarVector;
    ob->read(is);
    if (covarsMap.find(ob->name()) != covarsMap.end())
      error("ERROR: covar named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    covars[i+start] = ob;
    covarsMap[ob->name()] = i+start;
  }
}


void 
GMParms::readDLinkMats(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num dlink matrices");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of dlink matrices (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    dLinkMats.resize(num);
  } else {
    start = dLinkMats.size();
    dLinkMats.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    DlinkMatrix* ob;

    is.read(cnt,"Can't read dlink matrix number");
    if (cnt != i) 
      error("ERROR: dlink matrix count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new DlinkMatrix;
    ob->read(is);
    if (dLinkMatsMap.find(ob->name()) != dLinkMatsMap.end())
      error("ERROR: dlink matrix named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    dLinkMats[i+start] = ob;
    dLinkMatsMap[ob->name()] = i+start;
  }
}


void 
GMParms::readDLinks(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't rad num dlinks");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of dlinks (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    dLinks.resize(num);
  } else {
    start = dLinks.size();
    dLinks.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    Dlinks* ob;

    is.read(cnt,"Can't read dlinks num");
    if (cnt != i) 
      error("ERROR: dlink count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new Dlinks;
    ob->read(is);
    if (dLinksMap.find(ob->name()) != dLinksMap.end())
      error("ERROR: dlink structure named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    dLinks[i+start] = ob;
    dLinksMap[ob->name()] = i+start;
  }
}


void 
GMParms::readDoubleMats(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Cant' read num double matrices");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of double matrices (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    doubleMats.resize(num);
  } else {
    start = doubleMats.size();
    doubleMats.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    DoubleMatrix* ob;

    is.read(cnt,"Can't read double mat num");
    if (cnt != i) 
      error("ERROR: double matrix count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new DoubleMatrix;
    ob->read(is);
    if (doubleMatsMap.find(ob->name()) != doubleMatsMap.end())
      error("ERROR: double matrix named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    doubleMats[i+start] = ob;
    doubleMatsMap[ob->name()] = i+start;
  }
}


void 
GMParms::readRealMats(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Cant' read num real matrices");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of real matrices (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    realMats.resize(num);
  } else {
    start = realMats.size();
    realMats.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    RealMatrix* ob;

    is.read(cnt,"Can't read real mat num");
    if (cnt != i) 
      error("ERROR: real matrix count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new RealMatrix;
    ob->read(is);
    if (realMatsMap.find(ob->name()) != realMatsMap.end())
      error("ERROR: real matrix named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    realMats[i+start] = ob;
    realMatsMap[ob->name()] = i+start;
  }
}



void 
GMParms::readDirichletTabs(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Cant' read num dirichlet tables");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of dirichlet tables (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    dirichletTabs.resize(num);
  } else {
    start = dirichletTabs.size();
    dirichletTabs.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    DirichletTable* ob;

    is.read(cnt,"Can't read dirichlet table num");
    if (cnt != i) 
      error("ERROR: dirichlet table count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new DirichletTable();
    ob->read(is);
    if (dirichletTabsMap.find(ob->name()) != dirichletTabsMap.end())
      error("ERROR: dirichlet table named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    dirichletTabs[i+start] = ob;
    dirichletTabsMap[ob->name()] = i+start;
  }
}


void 
GMParms::readMdCpts(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num DenseCPTs");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of Dense CPTs (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    mdCpts.resize(num);
  } else {
    start = mdCpts.size();
    mdCpts.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    MDCPT* ob;

    is.read(cnt,"Can't read DenseCPT num");
    if (cnt != i) 
      error("ERROR: Dense CPT count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new MDCPT;
    ob->read(is);
    if (mdCptsMap.find(ob->name()) != mdCptsMap.end()) {
      if (ob->name() == string(USMDCPT_NAME))
	error("ERROR: special internal unity score DenseCPT named '%s' must not be used in parameter files, as it is used internally",USMDCPT_NAME);
      else
	error("ERROR: Dense CPT named '%s' already defined but is specified for a second time in file '%s' line %d",
	      ob->name().c_str(),is.fileName(),is.lineNo());
    }
    mdCpts[i+start] = ob;
    mdCptsMap[ob->name()] = i+start;
  }
}


void 
GMParms::readMsCpts(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num SparseCPTs");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of Sparse CPTs (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    msCpts.resize(num);
  } else {
    start = msCpts.size();
    msCpts.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    MSCPT* ob;

    is.read(cnt,"Can't read SparseCPT num");
    if (cnt != i) 
      error("ERROR: Sparse CPT count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new MSCPT;
    ob->read(is);
    if (msCptsMap.find(ob->name()) != msCptsMap.end())
      error("ERROR: Sparse CPT named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    msCpts[i+start] = ob;
    msCptsMap[ob->name()] = i+start;
  }
}



void 
GMParms::readMtCpts(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num DeterministicCPTs");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of deterministic CPTs (%d) exceeds maximum",num);
  if (reset) {
    start = 0;
    mtCpts.resize(num);
  } else {
    start = mtCpts.size();
    mtCpts.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    MTCPT* ob;

    is.read(cnt,"Can't read DeterministicCPT num");
    if (cnt != i) 
      error("ERROR: deterministic CPT count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new MTCPT;
    ob->read(is);
    if (mtCptsMap.find(ob->name()) != mtCptsMap.end())
      error("ERROR: deterministic CPT named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    mtCpts[i+start] = ob;
    mtCptsMap[ob->name()] = i+start;
  }
}


void GMParms::readVocabs(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num Vocabs");
  if ( num > GMPARMS_MAX_NUM )
    error("ERROR: number of Vocabs (%d) exceeds maximum", num);
  if ( reset ) {
    start = 0;
    vocabs.resize(num);
  } else {
    start = vocabs.size();
    vocabs.resize(start + num);
  }
  for ( unsigned i = 0; i <num;i++ ) {
    // first read the count
    Vocab* ob;

    is.read(cnt, "Can't read Vocab num");
    if ( cnt != i ) 
      error("ERROR: Vocab count (%d), out of order in file '%s' line %d, expecting %d", 
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new Vocab();
    ob->read(is);
    if ( vocabsMap.find(ob->name()) != vocabsMap.end() )
      error("ERROR: Vocab named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    vocabs[i + start] = ob;
    vocabsMap[ob->name()] = i + start;
  }
}


void GMParms::readNgramCpts(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Cant' read num NgramCPTs");
  if ( num > GMPARMS_MAX_NUM )
    error("ERROR: number of NGram CPTs (%d) exceeds maximum", num);
  if ( reset ) {
    start = 0;
    ngramCpts.resize(num);
  } else {
    start = ngramCpts.size();
    ngramCpts.resize(start + num);
  }
  for ( unsigned i = 0; i <num;i++ ) {
    // first read the count
    NGramCPT* ob;

    is.read(cnt, "Can't read NgramCPT num");
    if ( cnt != i )
      error("ERROR: NGramCPT count (%d), out of order in file '%s' line %d, expecting %d", 
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new NGramCPT();
    ob->read(is);
    if ( ngramCptsMap.find(ob->name()) != ngramCptsMap.end() )
      error("ERROR: NGramCPT named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    ngramCpts[i + start] = ob;
    ngramCptsMap[ob->name()] = i + start;
  }
}


void GMParms::readFNgramImps(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num, "Cant' read num FNgramCPTs");
  if ( num > GMPARMS_MAX_NUM )
    error("ERROR: number of FNGram CPTs (%d) exceeds maximum", num);
  if ( reset ) {
    start = 0;
    fngramImps.resize(num);
  } else {
    start = fngramImps.size();
    fngramImps.resize(start + num);
  }
  for ( unsigned i = 0; i <num; i++ ) {
    // first read the count
    FNGramImp* ob;

    is.read(cnt, "Can't read FNGramCPT num");
    if ( cnt != i )
      error("ERROR: FNGramCPT count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt, is.fileName(),is.lineNo(), i);

    ob = new FNGramImp();
    ob->read(is);
    if ( fngramImpsMap.find(ob->name()) != fngramImpsMap.end() )
      error("ERROR: FNGramCPT named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(), is.fileName(),is.lineNo());
    fngramImps[i + start] = ob;
    fngramImpsMap[ob->name()] = i + start;
  }
}


void GMParms::readLatticeAdts(iDataStreamFile& is, bool reset) {
	unsigned num;
	unsigned cnt;
	unsigned start = 0;

	is.read(num, "Can't read num Lattice CPTs");
	if ( num > GMPARMS_MAX_NUM )
		error("ERROR: number of Lattice CPTs (%d) exceeds maximum", num);

	if ( reset ) {
		start = 0;
		latticeAdts.resize(num);
		latticeNodeCpts.resize(num);
		latticeEdgeCpts.resize(num);
	} else {
		start = latticeAdts.size();
		latticeAdts.resize(start + num);
		latticeNodeCpts.resize(start + num);
		latticeEdgeCpts.resize(start + num);
	}

	for ( unsigned i = 0; i < num; i++ ) {
		is.read(cnt, "Can't read Lattice CPT index");
		if ( cnt != i )
		  error("ERROR: Lattice CPT count (%d), out of order in file '%s' line %d, expecting %d", cnt, is.fileName(),is.lineNo(), i);
		LatticeADT* ob = new LatticeADT();
		ob->read(is);
		if ( latticeAdtsMap.find(ob->name()) != latticeAdtsMap.end() )
			error("ERROR: Lattice CPT named '%s' already defined but is specified for a second time in file '%s' line %d", ob->name().c_str(), is.fileName(),is.lineNo());
		latticeAdts[i+start] = ob;
		latticeAdtsMap[ob->name()] = i + start;

		// add if it iterable
		if ( ob->iterable() )
			iterableLatticeAdts.push_back(ob);

		// add node CPT
		LatticeNodeCPT* ndCpt = new LatticeNodeCPT();
		ndCpt->setLatticeADT(*ob);
		ndCpt->setName(ob->name());
		latticeNodeCpts[i+start] = ndCpt;
		latticeNodeCptsMap[ob->name()] = i + start;

		// add edge CPT
		LatticeEdgeCPT* edgeCpt = new LatticeEdgeCPT();
		edgeCpt->setLatticeADT(*ob);
		edgeCpt->setName(ob->name());
		latticeEdgeCpts[i+start] = edgeCpt;
		latticeEdgeCptsMap[ob->name()] = i + start;
	}
}


void GMParms::readVECpts(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num, "Can't read num VirtualEvidenceCPTs");
  if ( num > GMPARMS_MAX_NUM )
    error("ERROR: number of Ve CPTs (%d) exceeds maximum", num);
  if ( reset ) {
    start = 0;
    veCpts.resize(num);
  } else {
    start = veCpts.size();
    veCpts.resize(start + num);
  }
  for ( unsigned i = 0; i <num; i++ ) {
    // first read the count
    VECPT* ob;

    is.read(cnt, "Can't read VirtualEvidenceCPT num");
    if ( cnt != i )
      error("ERROR: VECPT count (%d), out of order in file '%s' line %d, expecting %d", 
	    cnt, is.fileName(), is.lineNo(),i);

    ob = new VECPT();
    ob->read(is);
    if ( veCptsMap.find(ob->name()) != veCptsMap.end() )
      error("ERROR: VECPT named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(), is.fileName(),is.lineNo());
    veCpts[i + start] = ob;
    veCptsMap[ob->name()] = i + start;
  }
}


void GMParms::readDeepCpts(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num, "Can't read num DeepCPTs");
  if ( num > GMPARMS_MAX_NUM )
    error("ERROR: number of DeepCPTs (%d) exceeds maximum", num);
  if ( reset ) {
    start = 0;
    deepCpts.resize(num);
  } else {
    start = deepCpts.size();
    deepCpts.resize(start + num);
  }
  for ( unsigned i = 0; i <num; i++ ) {
    // first read the count
    DeepCPT* ob;

    is.read(cnt, "Can't read DeepCPT num");
    if ( cnt != i )
      error("ERROR: DeepCPT count (%d), out of order in file '%s' line %d, expecting %d", 
	    cnt, is.fileName(), is.lineNo(),i);

    ob = new DeepCPT();
    ob->read(is);
    if ( deepCptsMap.find(ob->name()) != deepCptsMap.end() )
      error("ERROR: DeepCPT named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(), is.fileName(),is.lineNo());
    deepCpts[i + start] = ob;
    deepCptsMap[ob->name()] = i + start;
  }
}


void GMParms::readDeepVECpts(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num, "Can't read num DeepVirtualEvidenceCPTs");
  if ( num > GMPARMS_MAX_NUM )
    error("ERROR: number of Deep VE CPTs (%d) exceeds maximum", num);
  if ( reset ) {
    start = 0;
    deepVECpts.resize(num);
  } else {
    start = deepVECpts.size();
    deepVECpts.resize(start + num);
  }
  for ( unsigned i = 0; i <num; i++ ) {
    // first read the count
    DeepVECPT* ob;

    is.read(cnt, "Can't read DeepVirtualEvidenceCPT num");
    if ( cnt != i )
      error("ERROR: DeepVECPT count (%d), out of order in file '%s' line %d, expecting %d", 
	    cnt, is.fileName(), is.lineNo(),i);

    ob = new DeepVECPT();
    ob->read(is);
    if ( deepVECptsMap.find(ob->name()) != deepVECptsMap.end() )
      error("ERROR: DeepVECPT named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(), is.fileName(),is.lineNo());
    deepVECpts[i + start] = ob;
    deepVECptsMap[ob->name()] = i + start;
  }
}



void GMParms::readDeepNNs(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num, "Can't read num DeepNNs");
  if ( num > GMPARMS_MAX_NUM )
    error("ERROR: number of Deep Neural Networks (%d) exceeds maximum", num);
  if ( reset ) {
    start = 0;
    deepNNs.resize(num);
  } else {
    start = deepNNs.size();
    deepNNs.resize(start + num);
  }
  for ( unsigned i = 0; i <num; i++ ) {
    // first read the count
    DeepNN* ob;

    is.read(cnt, "Can't read DeepNN num");
    if ( cnt != i )
      error("ERROR: DeepNN count (%d), out of order in file '%s' line %d, expecting %d", 
	    cnt, is.fileName(), is.lineNo(),i);

    ob = new DeepNN();
    ob->read(is);
    if ( deepNNsMap.find(ob->name()) != deepNNsMap.end() )
      error("ERROR: DeepNN named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(), is.fileName(),is.lineNo());
    deepNNs[i + start] = ob;
    deepNNsMap[ob->name()] = i + start;
  }
}



void
GMParms::readDTs(
  iDataStreamFile& is, 
  bool reset 
  )
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num Decision Trees");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of DTs (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    // TODO: this is dangerous to do, as this also removes all
    // the internal C function mapping functions. Think carefully
    // if you ever call this function with 'reset = true'.
    start = 0;
    dts.resize(num);
    //dtsMap.clear();
  } else {
    start = dts.size();
    dts.resize(start+num);
  }

  for (unsigned i=0;i<num;i++) {
    // first read the count
    RngDecisionTree* ob;

    is.read(cnt,"Can't read DecisionTree num");
    if (cnt != i) 
      error("ERROR: DT count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    ob = new RngDecisionTree;
    ob->read(is);
    if (dtsMap.find(ob->name()) != dtsMap.end())
      error("ERROR: DT named '%s' already defined but is specified for a second time in file '%s' line %d",
	    ob->name().c_str(),is.fileName(),is.lineNo());
    dts[i+start] = ob;
    dtsMap[ob->name()] = i+start;
    if (ob->iterable()) {
      iterableDts.push_back(ob);
    } 
  }
}


void 
GMParms::readComponents(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;

  unsigned start = 0;
  is.read(num,"Can't read num components");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of components (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    start = 0;
    components.resize(num);
  } else {
    start = components.size();
    components.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    Component* gc = NULL;

    is.read(cnt,"Can't read component num");
    if (cnt != i) 
      error("ERROR: component count (%d), out of order in file '%s' line %d , expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    // next read the dimension of this component
    int dim;
    is.read(dim,"Can't read component dimension");

    // read the Gaussian type, note that when
    // this is written, the object itself will write the type.
    int t;
    is.read(t,"Can't read component type");
    if (t == Component::DiagGaussian) {
      gc = new DiagGaussian(dim);
    } else if (t == Component::LinMeanCondDiagGaussian) {
      gc = new LinMeanCondDiagGaussian(dim);
    } else if (t == Component::PolyNLinMeanCondDiagGaussian) {
      error("PolyNLinMeanCondDiag not implemented");
      // gc = new PolyNLinMeanCondDiagGaussian(dim);
    } else if (t == Component::GammaComponent) {
      gc = new GammaComponent(dim);
    } else if (t == Component::BetaComponent) {
      gc = new BetaComponent(dim);
    } else {
      error("Error: reading file %s line %d, unknown component type %d in file",
	    is.fileName(),is.lineNo(),t);
    }
    gc->read(is);

    // this next check is redundant since the dim > 0 check is
    // already done by the mean, variance, etc. objects. We leave
    // it here, however, since 1) it costs almost nothing, and 2) as new object types 
    // are added, we might need such a check here.
    if (dim <= 0)
      error("ERROR: component named '%s' in file '%s' line %d specifies a non-positive dimension (%d). Must be > 0.",
	    gc->name().c_str(),is.fileName(),is.lineNo(),dim);

    if (componentsMap.find(gc->name()) != componentsMap.end())
      error("ERROR: component named '%s' already defined but is specified for a second time in file '%s' line %d",
	    gc->name().c_str(),is.fileName(),is.lineNo());
    components[i+start] = gc;
    componentsMap[gc->name()] = i+start;
  }
}


void 
GMParms::readMixtures(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num mixtures");
  if (num > GMPARMS_MAX_NUM) error("ERROR: number of mixtures (%d) in file '%s' line %d exceeds maximum",
				   num,is.fileName(),is.lineNo());
  if (reset) {
    // this isn't implemented at the moment.
    assert(0);
    // start = 0;
    // mixtures.resize(num);
  } else {
    start = mixtures.size();
    mixtures.resize(start+num);
  }


  for (unsigned i=0;i<num;i++) {
    // first read the count
    Mixture* gm;

    is.read(cnt,"Can't read mixture num");
    if (cnt != i) 
      error("ERROR: reading file %s line %d, mixture count (%d), out of order, expecting %d",
	    is.fileName(),is.lineNo(),cnt,i);


    // next read the dimension of this mixture
    int dim;
    is.read(dim,"Can't read mixture dimension");

    gm = new Mixture(dim);
    gm->read(is);

    // this next check is redundant since the dim > 0 check is
    // already done by the mean, variance, and component objects. We leave
    // it here, however, since 1) it costs almost nothing, and 2) as new object types 
    // are added, we might need such a check here.
    if (dim <= 0)
      error("ERROR: mixture named '%s' in file '%s' line %d specifies a non-positive dimension (%d). Must be > 0.",
	    gm->name().c_str(),is.fileName(),is.lineNo(),dim);

    if (mixturesMap.find(gm->name()) != mixturesMap.end()) {
      if (gm->name() == string(ZEROSCOREMIXTURE_NAME))
	error("ERROR: special internal mixture named '%s' must not be used in parameter files, as it is used internally",ZEROSCOREMIXTURE_NAME);
      else if (gm->name() == string(UNITYSCOREMIXTURE_NAME))
	error("ERROR: special internal mixture named '%s' must not be used in parameter files, as it is used internally",UNITYSCOREMIXTURE_NAME);
      else
	error("ERROR: mixture named '%s' already defined but is specified for a second time in file '%s' line %d",
	      gm->name().c_str(),is.fileName(),is.lineNo());
    }


    mixtures[i+start] = gm;
    mixturesMap[gm->name()] = i+start;
  }
}


void 
GMParms::readNameCollections(iDataStreamFile& is, bool reset)
{
  unsigned num;
  unsigned cnt;
  unsigned start = 0;

  is.read(num,"Can't read num NamedCollections");
  if (reset) {
    start = 0;
    ncls.resize(num);
    //nclsMap.clear();
  } else {
    start = ncls.size();
    ncls.resize(start+num);
  }
  for (unsigned i=0;i<num;i++) {
    // first read the count
    NameCollection* nc;

    is.read(cnt,"Can't read NameCollection num");
    if (cnt != i) 
      error("ERROR: collection order count (%d), out of order in file '%s' line %d, expecting %d",
	    cnt,is.fileName(),is.lineNo(),i);

    nc = new NameCollection();
    nc->read(is);

    if (nc->name() == NAMED_COLLECTION_GLOBAL_NAME) {
      error("ERROR: special internal collection name '%s' can not be defined in file '%s' line %d",
	    nc->name().c_str(),is.fileName(),is.lineNo());

    }

    if (nclsMap.find(nc->name()) != nclsMap.end()) {
      if (nc->name() == string(NAMED_COLLECTION_GLOBAL_NAME))
	error("ERROR: special internal collection named '%s' must not be used in parameter files, as it is used internally to refer to global table.",NAMED_COLLECTION_GLOBAL_NAME);
      else
	error("ERROR: collection named '%s' already defined but is specified for a second time in file '%s' line %d",
	      nc->name().c_str(),is.fileName(),is.lineNo());
    }

    ncls[i+start] = nc;
    nclsMap[nc->name()] = i+start;
  }
}



void 
GMParms::readGausSwitchMixtures(iDataStreamFile& is, bool reset)
{
  unsigned num;
  is.read(num,"Can't read num GSMGs");
  if (num > 0)
    error("ERROR: reading file '%s' line %d, GausSwitchMixtures not implemented just yet",
	  is.fileName(),is.lineNo());
}

void 
GMParms::readLogitSwitchMixtures(iDataStreamFile& is, bool reset)
{
  unsigned num;
  is.read(num,"Can't read num LSMGs");
  if (num > 0)
    error("ERROR: reading file '%s' line %d, LogitSwitchMixtures not implemented just yet",
	  is.fileName(),is.lineNo());
}

void 
GMParms::readMlpSwitchMixtures(iDataStreamFile& is, bool reset)
{
  unsigned num;
  is.read(num,"Can't read num MSMGs");
  if (num > 0)
    error("ERROR: reading file '%s' line %d, MLPSwitchMixtures not implemented just yet",
	  is.fileName(),is.lineNo());
}



/*-
 *-----------------------------------------------------------------------
 * readAll
 *   read in everything (all parameters) in one go.
 * 
 * Preconditions:
 *      none
 *
 * Postconditions:
 *      files read in
 *
 * Side Effects:
 *      changes internal arrays.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void
GMParms::readAll(iDataStreamFile& is)
{
  // just read everything in one go.

  // first read structural items
  readDTs(is);
  readDLinks(is);

  // then read basic numeric items.
  readDPmfs(is);
  readSPmfs(is);
  readMeans(is);
  readCovars(is);
  readDLinkMats(is);
  readRealMats(is);
#if DOUBLEMATS_EVERYWHERE
  readDoubleMats(is);
#endif
  readDirichletTabs(is);
  readMdCpts(is);
  readMsCpts(is);
  readMtCpts(is);
  readVocabs(is);
  readNgramCpts(is);
  readFNgramImps(is);
  readLatticeAdts(is);
  readVECpts(is);
  readDeepNNs(is);
  readDeepVECpts(is);
  // next read definitional items
  readComponents(is);
  readMixtures(is);
  readGausSwitchMixtures(is);
  readLogitSwitchMixtures(is);
  readMlpSwitchMixtures(is);  
}


/*-
 *-----------------------------------------------------------------------
 * readTrainable
 *   read in just the trainable parameters, i.e.,
 *   the things that might be modified by the
 *   program when training is occuring.
 *
 *   Note that the union of this routine
 *   and readNonTrainable should be the same as 
 *   readAll
 * 
 * Preconditions:
 *      none
 *
 * Postconditions:
 *      files read in
 *
 * Side Effects:
 *      changes internal arrays.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::readTrainable(iDataStreamFile& is)
{
  infoMsg(Low+9,"Reading Dense Prob. Mass Functions\n");
  readDPmfs(is);
  infoMsg(Low+9,"Reading Sparse Prob. Mass Functions\n");
  readSPmfs(is);
  infoMsg(Low+9,"Reading Means Functions\n");
  readMeans(is);
  infoMsg(Low+9,"Reading Covars Functions\n");
  readCovars(is);
  infoMsg(Low+9,"Reading Dlink Mats\n");
  readDLinkMats(is);
  infoMsg(Low+9,"Reading Real Mats\n");
  readRealMats(is);  
#if DOUBLEMATS_EVERYWHERE
  infoMsg(Low+9,"Reading Double Mats\n");
  readDoubleMats(is);  
#endif
  infoMsg(Low+9,"Reading Dense CPTs\n");
  readMdCpts(is);

  // next read definitional items
  infoMsg(Low+9,"Reading Components\n");
  readComponents(is);
  infoMsg(Low+9,"Reading Mixtures\n");
  readMixtures(is);
  readGausSwitchMixtures(is);
  readLogitSwitchMixtures(is);
  readMlpSwitchMixtures(is);  
}



/*-
 *-----------------------------------------------------------------------
 * readnonTrainable
 *   read in just the *NON*-trainable parameters, i.e.,
 *   the things that, guaranteed, will not be
 *   modified when the program is training.
 *
 *   Note that the union of this routine
 *   and readTrainable should be the same as 
 *   readAll
 * 
 * Preconditions:
 *      none
 *
 * Postconditions:
 *      files read in
 *
 * Side Effects:
 *      changes internal arrays.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::readNonTrainable(iDataStreamFile& is)
{
  // first read structural items
  infoMsg(Low+9,"Reading Decision Trees\n");
  readDTs(is);
  infoMsg(Low+9,"Reading Dlinks\n");
  readDLinks(is);

  infoMsg(Low+9,"Reading SparseCPTs\n");
  readMsCpts(is);
  infoMsg(Low+9,"Reading Deterministic CPTs\n");
  readMtCpts(is);
  infoMsg(Low+9,"Reading Vocabs\n");
  readVocabs(is);
  infoMsg(Low+9,"Reading NgramCPTs\n");
  readNgramCpts(is);
  infoMsg(Low+9,"Reading FNgramCPTs\n");
  readFNgramImps(is);
  infoMsg(Low+9,"Reading VirtualEvidenceCPTs\n");
  readVECpts(is);
  infoMsg(Low+9,"Reading DeepNNss\n");
  readDeepNNs(is);
  infoMsg(Low+9,"Reading DeepVirtualEvidenceCPTs\n");
  readDeepVECpts(is);
}


/*
void 
GMParms::read(
  iDataStreamFile& is
  )
{
  // read a file consisting of a list of keyword,filename
  // pairs. the keyword says which structure to read in,
  // and the filename says where to get it.
  string keyword;
  string fileName;
  string binStatus;

  const string INLINE_FILE_KEYWORD("inline");
  
  map<string,iDataStreamFile*> fileNameMap;
  fileNameMap[INLINE_FILE_KEYWORD] = &is;

  while (is.readString(keyword)) {

    // fprintf(stderr,"read keyword '%s'\n",keyword.c_str());

    if (!is.readString(fileName)) {
      error("ERROR: while reading file '%s' line %d , got keyword '%s' without a filename",
	    is.fileName(),is.lineNo(),keyword.c_str());
    }

    bool binary_p = is.binary();
    if (fileName != INLINE_FILE_KEYWORD) {
      // read binary status of file if this is not an inline declarator
      if (!is.readString(binStatus)) {
	error("ERROR: while reading file '%s' line %d, got keyword '%s' and filename '%s' without a binary status",
	      is.fileName(),is.lineNo(),keyword.c_str(),fileName.c_str());
      }
      if (binStatus == "ascii" || binStatus == "ASCII")
	binary_p = false;
      else if (binStatus == "binary" || binStatus == "BINARY")
	binary_p = true;
      else {
	error("ERROR: while reading file '%s' line %d, got string '%s' when expecting 'ascii'/'binary' keyword",
	      is.fileName(),is.lineNo(),binStatus.c_str());
      }
      infoMsg(Low+9,"Reading keyword '%s' from %s file '%s'.\n",
	      keyword.c_str(),(binary_p?"binary":"ASCII"),
	      fileName.c_str());
    } else {
      infoMsg(Low+9,"Reading keyword '%s' from inline.\n",keyword.c_str());
    }

    map<string,iDataStreamFile*>::iterator it = fileNameMap.find(fileName);
    if (it == fileNameMap.end()) {
      fileNameMap[fileName] = new iDataStreamFile(fileName.c_str(),binary_p);
      it = fileNameMap.find(fileName);
    } else if (((*it).second)->binary() != binary_p) {
      error("ERROR: reading '%s'. File '%s' line %d had binary status = %d, now binary status = %d. Can't mix binary and ASCII files",
	    is.fileName(),is.lineNo(),fileName.c_str(),((*it).second)->binary(),binary_p);
    }

    if (keyword == "DPMF_IN_FILE") {
      readDPmfs(*((*it).second),false);

    } else if (keyword == "SPMF_IN_FILE") {
      readSPmfs(*((*it).second),false);

    } else if (keyword == "MEAN_IN_FILE") {
      readMeans(*((*it).second),false);

    } else if (keyword == "COVAR_IN_FILE") {
      readCovars(*((*it).second),false);

    } else if (keyword == "DLINK_MAT_IN_FILE") {
      readDLinkMats(*((*it).second),false);

    } else if (keyword == "DLINK_IN_FILE") {
      readDLinks(*((*it).second),false);

    } else if (keyword == "WEIGHT_MAT_IN_FILE") {
      // TODO: evenually remove this backwards compatibility case. 
      readRealMats(*((*it).second),false);
    } else if (keyword == "REAL_MAT_IN_FILE") {
      readRealMats(*((*it).second),false);

    } else if (keyword == "DOUBLE_MAT_IN_FILE") {
      readDoubleMats(*((*it).second),false);

    } else if (keyword == "DIRICHLET_TAB_IN_FILE") {
      readDirichletTabs(*((*it).second),false);

    } else if (keyword == "DENSE_CPT_IN_FILE") {
      readMdCpts(*((*it).second),false);

    } else if (keyword == "SPARSE_CPT_IN_FILE") {
      readMsCpts(*((*it).second),false);

    } else if (keyword == "DETERMINISTIC_CPT_IN_FILE") {
      readMtCpts(*((*it).second),false);

    } else if (keyword == "VOCAB_IN_FILE") {
      readVocabs(*((*it).second),false);

    } else if (keyword == "NGRAM_CPT_IN_FILE") {
      readNgramCpts(*((*it).second),false);

    } else if (keyword == "FNGRAM_CPT_IN_FILE") {
      readFNgramImps(*((*it).second),false);

    } else if (keyword == "LATTICE_CPT_IN_FILE") {
	    readLatticeAdts(*((*it).second),false);

    } else if (keyword == "VE_CPT_IN_FILE") {
      readVECpts(*((*it).second),false);

    } else if (keyword == "DEEP_NN_IN_FILE") {
      readDeepNNs(*((*it).second),false);

    } else if (keyword == "DEEP_VE_CPT_IN_FILE") {
      readDeepVECpts(*((*it).second),false);

    } else if (keyword == "DEEP_CPT_IN_FILE") {
      readDeepCpts(*((*it).second),false);

    } else if (keyword == "DT_IN_FILE") {
      readDTs(*((*it).second),false);

    } else if (keyword == "MC_IN_FILE") {
      readComponents(*((*it).second),false);

    } else if (keyword == "MX_IN_FILE") {
      readMixtures(*((*it).second),false);

    } else if (keyword == "NAME_COLLECTION_IN_FILE") {
      readNameCollections(*((*it).second),false);

    } else if (keyword == "GSMG_IN_FILE") {
      error("GSMG_IN_FILE in file '%s' line %d, not implemented",
	    is.fileName(),is.lineNo());

    } else if (keyword == "LSMG_IN_FILE") {
      error("LSMG_IN_FILE in file '%s' line %d, not implemented",
	    is.fileName(),is.lineNo());

    } else if (keyword == "MSMG_IN_FILE") {
      error("MSMG_IN_FILE in file '%s' line %d, not implemented",
	    is.fileName(),is.lineNo());

    } else {
      error("ERROR: encountered unknown file type '%s' in file '%s' line %d",
	    keyword.c_str(),is.fileName(),is.lineNo());
    }
  }

  // now go through and delete all the input files
  for (map<string,iDataStreamFile*>::iterator it = fileNameMap.begin();
       it != fileNameMap.end(); it++) {
    if ((*it).first != INLINE_FILE_KEYWORD) {
      // don't delete is
      delete ((*it).second);
    }
  }

}
*/



void 
GMParms::read(
  iDataStreamFile& is, bool reset
  )
{
  // read a file consisting of a list of keyword,filename
  // pairs. the keyword says which structure to read in,
  // and the filename says where to get it.
  string keyword;
  string fileName;
  string binStatus;

  const string INLINE_FILE_KEYWORD("inline");
  
  map<string,iDataStreamFile*> fileNameMap;
  fileNameMap[INLINE_FILE_KEYWORD] = &is;

  while (is.readString(keyword)) {

    // fprintf(stderr,"read keyword '%s'\n",keyword.c_str());

    if (!is.readString(fileName)) {
      error("ERROR: while reading file '%s' line %d , got keyword '%s' without a filename",
	    is.fileName(),is.lineNo(),keyword.c_str());
    }

    bool binary_p = is.binary();
    if (fileName != INLINE_FILE_KEYWORD) {
      // read binary status of file if this is not an inline declarator
      if (!is.readString(binStatus)) {
	error("ERROR: while reading file '%s' line %d, got keyword '%s' and filename '%s' without a binary status",
	      is.fileName(),is.lineNo(),keyword.c_str(),fileName.c_str());
      }
      if (binStatus == "ascii" || binStatus == "ASCII")
	binary_p = false;
      else if (binStatus == "binary" || binStatus == "BINARY")
	binary_p = true;
      else {
	error("ERROR: while reading file '%s' line %d, got string '%s' when expecting 'ascii'/'binary' keyword",
	      is.fileName(),is.lineNo(),binStatus.c_str());
      }
      infoMsg(Low+9,"Reading keyword '%s' from %s file '%s'.\n",
	      keyword.c_str(),(binary_p?"binary":"ASCII"),
	      fileName.c_str());
    } else {
      infoMsg(Low+9,"Reading keyword '%s' from inline.\n",keyword.c_str());
    }

    map<string,iDataStreamFile*>::iterator it = fileNameMap.find(fileName);
    if (it == fileNameMap.end()) {
      fileNameMap[fileName] = new iDataStreamFile(fileName.c_str(),binary_p);
      it = fileNameMap.find(fileName);
    } else if (((*it).second)->binary() != binary_p) {
      error("ERROR: reading '%s'. File '%s' line %d had binary status = %d, now binary status = %d. Can't mix binary and ASCII files",
	    is.fileName(),is.lineNo(),fileName.c_str(),((*it).second)->binary(),binary_p);
    }

    if (keyword == "DPMF_IN_FILE") {
      readDPmfs(*((*it).second),reset);

    } else if (keyword == "SPMF_IN_FILE") {
      readSPmfs(*((*it).second),reset);

    } else if (keyword == "MEAN_IN_FILE") {
      readMeans(*((*it).second),reset);

    } else if (keyword == "COVAR_IN_FILE") {
      readCovars(*((*it).second),reset);

    } else if (keyword == "DLINK_MAT_IN_FILE") {
      readDLinkMats(*((*it).second),reset);

    } else if (keyword == "DLINK_IN_FILE") {
      readDLinks(*((*it).second),reset);

    } else if (keyword == "WEIGHT_MAT_IN_FILE") {
      // TODO: evenually remove this backwards compatibility case. 
      readRealMats(*((*it).second),reset);
    } else if (keyword == "REAL_MAT_IN_FILE") {
      readRealMats(*((*it).second),reset);

    } else if (keyword == "DOUBLE_MAT_IN_FILE") {
      readDoubleMats(*((*it).second),reset);

    } else if (keyword == "DIRICHLET_TAB_IN_FILE") {
      readDirichletTabs(*((*it).second),reset);

    } else if (keyword == "DENSE_CPT_IN_FILE") {
      readMdCpts(*((*it).second),reset);

    } else if (keyword == "SPARSE_CPT_IN_FILE") {
      readMsCpts(*((*it).second),reset);

    } else if (keyword == "DETERMINISTIC_CPT_IN_FILE") {
      readMtCpts(*((*it).second),reset);

    } else if (keyword == "VOCAB_IN_FILE") {
      readVocabs(*((*it).second),reset);

    } else if (keyword == "NGRAM_CPT_IN_FILE") {
      readNgramCpts(*((*it).second),reset);

    } else if (keyword == "FNGRAM_CPT_IN_FILE") {
      readFNgramImps(*((*it).second),reset);

    } else if (keyword == "LATTICE_CPT_IN_FILE") {
	    readLatticeAdts(*((*it).second),reset);

    } else if (keyword == "VE_CPT_IN_FILE") {
      readVECpts(*((*it).second),reset);

    } else if (keyword == "DEEP_NN_IN_FILE") {
      readDeepNNs(*((*it).second),reset);

    } else if (keyword == "DEEP_VE_CPT_IN_FILE") {
      readDeepVECpts(*((*it).second),reset);

    } else if (keyword == "DEEP_CPT_IN_FILE") {
      readDeepCpts(*((*it).second),reset);

    } else if (keyword == "DT_IN_FILE") {
      readDTs(*((*it).second),reset);

    } else if (keyword == "MC_IN_FILE") {
      readComponents(*((*it).second),reset);

    } else if (keyword == "MX_IN_FILE") {
      readMixtures(*((*it).second),reset);

    } else if (keyword == "NAME_COLLECTION_IN_FILE") {
      readNameCollections(*((*it).second),reset);

    } else if (keyword == "GSMG_IN_FILE") {
      error("GSMG_IN_FILE in file '%s' line %d, not implemented",
	    is.fileName(),is.lineNo());

    } else if (keyword == "LSMG_IN_FILE") {
      error("LSMG_IN_FILE in file '%s' line %d, not implemented",
	    is.fileName(),is.lineNo());

    } else if (keyword == "MSMG_IN_FILE") {
      error("MSMG_IN_FILE in file '%s' line %d, not implemented",
	    is.fileName(),is.lineNo());

    } else {
      error("ERROR: encountered unknown file type '%s' in file '%s' line %d",
	    keyword.c_str(),is.fileName(),is.lineNo());
    }
  }

  // now go through and delete all the input files
  for (map<string,iDataStreamFile*>::iterator it = fileNameMap.begin();
       it != fileNameMap.end(); it++) {
    if ((*it).first != INLINE_FILE_KEYWORD) {
      // don't delete is
      delete ((*it).second);
    }
  }

}



/*-
 *-----------------------------------------------------------------------
 * GMParms::writeDecisionTreeIndexFiles()
 *
 * Preconditions:
 *    The iterableDts vector has been filled in using readDTs 
 *
 * Postconditions:
 *    An index file is written for each iterable DT
 *
 * Side Effects:
 *    The first decison tree is set to 0 for all trees. 
 *
 * Results:
 *    none
 *-----------------------------------------------------------------------
 */
void 
GMParms::
writeDecisionTreeIndexFiles()
{
  vector<RngDecisionTree*>::iterator crrnt_tree; 
  vector<RngDecisionTree*>::iterator end_tree; 

  for (crrnt_tree = iterableDts.begin(), 
       end_tree   = iterableDts.end();
       crrnt_tree != end_tree;
       crrnt_tree++) {
 
    (*crrnt_tree)->setFirstDecisionTree(0);
    (*crrnt_tree)->writeIndexFile();
  } 
}




/*-
 *-----------------------------------------------------------------------
 * GMParms::registerDeterministicCMapper()
 *
 * Preconditions:
 *    none
 *
 * Postconditions:
 *    C function "decision trees" are registered and usable.
 *
 * Side Effects:
 *    modifies the DT data strucures.
 *
 * Results:
 *    none
 *-----------------------------------------------------------------------
 */
void 
GMParms::
registerDeterministicCMapper(const char *name,
			     unsigned num_features,
			     CFunctionMapperType func)
{
  unsigned start = dts.size();
  dts.resize(start+1);
  RngDecisionTree* ob = new RngDecisionTree(name,func,num_features);
  // make sure that the name is not alread used. If this is the case,
  // the user who added decision trees has made an error in the function
  // GMTK_CFunctionDeterministicMappings.cc. Names *must* be unique.
  if (dtsMap.find(ob->name()) != dtsMap.end())
    error("ERROR: C function deterministic mapper named '%s' already defined but is specified for a second time. Need o check .cc file",ob->name().c_str());
  // everything looks ok, so continue.
  dts[start] = ob;
  dtsMap[ob->name()] = start;
  // printf("regsitered internal DT %s\n",ob->name().c_str());
  // these are never iterable, so nothing more to do.
}




/*-
 *-----------------------------------------------------------------------
 * GMParms::finalizeParameters()
 *   load internal global objects, after all other parameters
 *   have been read in.
 *
 * Preconditions:
 *      Should be called after all internal objects have been read in. This routine
 *      should not be called before other GMTK objects have
 *      been loaded. Note also this should be done *before* any association
 *      with the RV structure, so that it doesn't have any undefined values.
 *
 * Postconditions:
 *      New global objects have been done.
 *
 * Side Effects:
 *      changes internal GMTK object arrays. Note that
 *      this routine will add to the internal GMKT object arrays
 *      by appending at the end. This routine should be called last,
 *      after all other objects have been allocated. Also, when
 *      writing out any of these objects, we should be sure
 *      not to write out any of the objects which are being
 *      stored here.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::finalizeParameters()
{
  ///////////////////////////////////////////////////////
  // Now that presumably everything has been read in,
  // we insert the global internal objects:
  //     1) a named collection which references the global arrays.
  //     2) two special scoring mixtures (unity and zero)
  //     3) one special scoring MDCPT (unity)

  // Load the global named collection.  first, make sure that the name
  // hasn't already been defined, meaning either that this routine has
  // not been called or nobody else defined a collection with this name.
  assert (nclsMap.find(string(NAMED_COLLECTION_GLOBAL_NAME)) == nclsMap.end());


  // TODO: delete this at some point in the future.
  NameCollection* nc = new NameCollection();
  nc->_name = NAMED_COLLECTION_GLOBAL_NAME;
  // copy the tables:
  // TODO: figure out a better way than copying the entire collection
  // to global. Perhaps, however, STL does the right thing and will
  // share the internal array, but should check this.
  nc->mxTable = mixtures;
  nc->spmfTable = sPmfs;
  ncls.push_back(nc);
  nclsMap[nc->name()] = ncls.size()-1;

  /////////////////////////////////////////////////////////////////////
  // now we load 2 extra mixtures. 
  // IMPORTANT: when making changes here, also see routine:  GMParms::writeMixtures(oDataStreamFile& os)

  // Load the zero scoring Mixture
  assert (mixturesMap.find(string(ZEROSCOREMIXTURE_NAME)) == mixturesMap.end());
  ZeroScoreMixture* zs = new ZeroScoreMixture();
  mixtures.push_back(zs);
  mixturesMap[zs->name()] = mixtures.size()-1;

  // Load the zero scoring Mixture
  assert (mixturesMap.find(string(UNITYSCOREMIXTURE_NAME)) == mixturesMap.end());
  UnityScoreMixture* us = new UnityScoreMixture();
  mixtures.push_back(us);
  mixturesMap[us->name()] = mixtures.size()-1;


  /////////////////////////////////////////////////////////////////////
  // and we load 1 extra MDCPT
  // IMPORTANT: when making changes here, also see routine:  GMParms::writeMdCpts(oDataStreamFile& os)

  // load the unity scoring MDCPT.  Note, this CPT corresponds to
  // a random variable:
  //   1) of any cardinality
  //   2) with no switching or conditional parents
  //   3) that is observed
  //   4) that has probability = 1 regardless of the
  //      observed value, so that this is a conditional rather than
  //      a scoring observation, similar to the way conditional
  //      floating point observations work.
  assert (mdCptsMap.find(string(USMDCPT_NAME)) == mdCptsMap.end());
  USCPT *uscpt = new USCPT();
  mdCpts.push_back(uscpt);
  mdCptsMap[uscpt->name()] = mdCpts.size()-1;

}


////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
//        WRITING Routines
////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////


/*-
 *-----------------------------------------------------------------------
 * write<OBJECTS>
 *      Write the <OBJECT> to a file specified by os
 * 
 * Preconditions:
 *      None, as if the arrays are zero sized, nothing will be written
 *      other than the number 0
 *
 * Postconditions:
 *      All objects have been written.
 *      
 * Side Effects:
 *      None.
 *
 * Results:
 *      none
 *
 *-----------------------------------------------------------------------
 */


void 
GMParms::writeDPmfs(oDataStreamFile& os)
{
  os.nl(); os.writeComment("dense PMFs");os.nl();
  os.write((unsigned)dPmfs.size(),"num dPMFs"); os.nl();
  for (unsigned i=0;i<dPmfs.size();i++) {
    // first write the count
    os.write(i,"dDPMF cnt");
    os.nl();
    dPmfs[i]->write(os);
  }
  os.nl();
}



void 
GMParms::writeSPmfs(oDataStreamFile& os)
{
  os.nl(); os.writeComment("sparse PMFs");os.nl();
  os.write((unsigned)sPmfs.size(),"num sPMFs"); os.nl();
  for (unsigned i=0;i<sPmfs.size();i++) {
    // first write the count
    os.write(i,"sPMFs cnt");
    os.nl();
    sPmfs[i]->write(os);
  }
  os.nl();
}


/*-
 *-----------------------------------------------------------------------
 * writeMeans
 *      writes out the means 
 * 
 * Preconditions:
 *      markUsedMixtureComponents() should have been called immediately before.
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeMeans(oDataStreamFile& os)
{
  os.nl(); os.writeComment("means");os.nl();
  // first scan through and find the number that are used.
  unsigned used = 0;
  for (unsigned i=0;i<means.size();i++)
    if (means[i]->emUsedBitIsSet())
      used++;
  if (used != means.size())
    warning("NOTE: saving only %d used means out of a total of %d",
	    used,means.size());
  os.write(used,"num Means"); os.nl();
  unsigned index=0;
  for (unsigned i=0;i<means.size();i++) {
    if (means[i]->emUsedBitIsSet()) {
      // first write the count
      os.write(index++,"means cnt");
      os.nl();
      means[i]->write(os);
    }
  }


  assert ( used == index );
  os.nl();
}



/*-
 *-----------------------------------------------------------------------
 * writeCovars
 *      writes out the covariance matrices
 * 
 * Preconditions:
 *      markUsedMixtureComponents() should have been called immediately before.
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeCovars(oDataStreamFile& os)
{
  os.nl(); os.writeComment("diagonal covariance matrices");os.nl();
  unsigned used = 0;
  for (unsigned i=0;i<covars.size();i++)
    if (covars[i]->emUsedBitIsSet())
      used++;
  if (used != covars.size())
    warning("NOTE: saving only %d used covariances out of a total of %d",
	    used,covars.size());
  os.write(used,"num covars"); os.nl();
  unsigned index=0;
  for (unsigned i=0;i<covars.size();i++) {
    if (covars[i]->emUsedBitIsSet()) {
      // first write the count
      os.write(index++,"covar cnt");
      os.nl();
      covars[i]->write(os);
    }
  }
  assert ( used == index );
  os.nl();
}



/*-
 *-----------------------------------------------------------------------
 * writeDlinkmats
 *      writes out the dlink mats
 * 
 * Preconditions:
 *      markUsedMixtureComponents() should have been called immediately before.
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeDLinkMats(oDataStreamFile& os)
{
  os.nl(); os.writeComment("dlink matrices");os.nl();
  unsigned used = 0;
  for (unsigned i=0;i<dLinkMats.size();i++) {
    if (dLinkMats[i]->emUsedBitIsSet())
      used++;
  }
  if (used != dLinkMats.size())
    warning("NOTE: saving only %d used dlink matrices out of a total of %d",
	    used,dLinkMats.size());
  os.write(used,"num dlink mats"); os.nl();
  unsigned index=0;
  for (unsigned i=0;i<dLinkMats.size();i++) {
    if (dLinkMats[i]->emUsedBitIsSet()) {
      // first write the count
      os.write(index++,"dlink mat cnt");
      os.nl();
      dLinkMats[i]->write(os);
    }
  }
  assert ( used == index );
  os.nl();
}



/*-
 *-----------------------------------------------------------------------
 * writeDlinks
 *      writes out the dlinks
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeDLinks(oDataStreamFile& os)
{
  os.nl(); os.writeComment("dlink structures");os.nl();
  os.write((unsigned)dLinks.size(),"num dlinks"); os.nl();
  for (unsigned i=0;i<dLinks.size();i++) {
    // first write the count
    os.write(i,"dlink cnt");
    os.nl();
    dLinks[i]->write(os);
  }
  os.nl();
}



/*-
 *-----------------------------------------------------------------------
 * writeRealMats
 *      writes out the real mats
 * 
 * Preconditions:
 *      markUsedMixtureComponents() should have been called immediately before.
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeRealMats(oDataStreamFile& os)
{
  os.nl(); os.writeComment("real valued N_i x M_i matrices");os.nl();
  unsigned used = 0;
  for (unsigned i=0;i<realMats.size();i++) {
    if (realMats[i]->emUsedBitIsSet())
      used++;
  }
  if (used != realMats.size())
    warning("NOTE: saving only %d used real matrices out of a total of %d",
	    used,realMats.size());
  os.write(used,"num real mats"); os.nl();
  unsigned index = 0;
  for (unsigned i=0;i<realMats.size();i++) {
    if (realMats[i]->emUsedBitIsSet()) {
      // first write the count
      os.write(index++,"real mat cnt");
      os.nl();
      realMats[i]->write(os);
    }
  }
  assert ( used == index );
  os.nl();
}



/*-
 *-----------------------------------------------------------------------
 * writeDoubleMats
 *      writes out the double mats
 * 
 * Preconditions:
 *      markUsedMixtureComponents() should have been called immediately before.
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeDoubleMats(oDataStreamFile& os)
{
  os.nl(); os.writeComment("double valued N_i x M_i matrices");os.nl();
  unsigned used = 0;
  for (unsigned i=0;i<doubleMats.size();i++) {
    if (doubleMats[i]->emUsedBitIsSet())
      used++;
  }
  if (used != doubleMats.size())
    warning("NOTE: saving only %d used double matrices out of a total of %d",
	    used,doubleMats.size());
  os.write(used,"num double mats"); os.nl();
  unsigned index = 0;
  for (unsigned i=0;i<doubleMats.size();i++) {
    if (doubleMats[i]->emUsedBitIsSet()) {
      // first write the count
      os.write(index++,"double mat cnt");
      os.nl();
      doubleMats[i]->write(os);
    }
  }
  assert ( used == index );
  os.nl();
}



/*-
 *-----------------------------------------------------------------------
 * writeDiricletTabs
 *      writes out the Dirichlet Tables
 * 
 * Preconditions:
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeDirichletTabs(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Dirichlet Tables");os.nl();
  os.write((unsigned)dirichletTabs.size(),"num dirichlet tabs"); os.nl();
  for (unsigned i=0;i<dirichletTabs.size();i++) {
    // first write the count
    os.write(i,"diriclet tab cnt");
    os.nl();
    dirichletTabs[i]->write(os);
  }
  os.nl();
}



/*-
 *-----------------------------------------------------------------------
 * writeMdCpts
 *
 *   Writes out all the MDCPTs in the global object. If there are
 *   any internal pre-defined CPTs in this set, then they are
 *   assumed to be kept at the beginning of the array, and are 
 *   not written out (since they are automatically created anew each
 *   time the program loads). See routine loadGlobal() for more details.
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeMdCpts(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Dense CPTs");os.nl();
  // leave out the 1st one (ie., the -1) as it is an internal
  // object. See routine loadGlobal()
  os.write((unsigned)mdCpts.size()-1,"num Dense CPTs"); os.nl();

  // Next, get a pointer to the unity score CPT that we should not
  // write out.  Note that it potentially might not be at the end of
  // the array since we might have done automatic allocation of MDCPTs
  // when reading in the .str file.
  const string usname = string(USMDCPT_NAME);
  assert (mdCptsMap.find(usname) != mdCptsMap.end());
  const unsigned idx = mdCptsMap[string(USMDCPT_NAME)];
  assert ( idx < mdCpts.size() );
  USCPT *uscpt = (USCPT*) mdCpts[idx];
  unsigned cnt = 0;
  for (unsigned i=0;i<mdCpts.size();i++) {
    if (mdCpts[i] == uscpt)
      continue;
    // first write the count
    os.write(cnt++,"Dense CPT cnt");
    os.nl();
    mdCpts[i]->write(os);
  }
  os.nl();
}


/*-
 *-----------------------------------------------------------------------
 * writemscpts
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeMsCpts(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Sparse CPTs");os.nl();
  os.write((unsigned)msCpts.size(),"num Sparse CPTs"); os.nl();
  for (unsigned i=0;i<msCpts.size();i++) {
    // first write the count
    os.write(i,"Sparse CPT cnt");
    os.nl();
    msCpts[i]->write(os);
  }
  os.nl();
}



/*-
 *-----------------------------------------------------------------------
 * writemtcpts
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeMtCpts(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Deterministic CPTs");os.nl();
  os.write((unsigned)mtCpts.size(),"num deterministic CPTs"); os.nl();
  for (unsigned i=0;i<mtCpts.size();i++) {
    // first write the count
    os.write(i,"deterministic CPT cnt");
    os.nl();
    mtCpts[i]->write(os);
  }
  os.nl();
}


/*-
 *-----------------------------------------------------------------------
 * writeDTs
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeDTs(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Decision Trees");os.nl();
  os.write((unsigned)dts.size(),"num DTS"); os.nl();
  for (unsigned i=0;i<dts.size();i++) {
    // first write the count
    os.write(i,"DTS cnt");
    os.nl();
    dts[i]->write(os);
  }
  os.nl();
}



/*-
 *-----------------------------------------------------------------------
 * writeComponents 
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeComponents(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Components");os.nl();
  unsigned used = 0;
  for (unsigned i=0;i<components.size();i++) {
    if (components[i]->emUsedBitIsSet())
      used++;
  }
  if (used != components.size())
    warning("NOTE: saving only %d used components out of a total of %d",
	    used,components.size());
  os.write(used,"num GCs"); os.nl();
  unsigned index = 0;  
  for (unsigned i=0;i<components.size();i++) {
    if (components[i]->emUsedBitIsSet()) {
      // first write the count
      os.write(index++,"GC cnt");
      os.nl();

      // next write the dimension of this component
      os.write(components[i]->dim(),"GC dim");
      os.nl();

      ////////////////////////////////////////////////////////////
      // Assume that the GC's write routine will 
      // itself write the component type
      components[i]->write(os);
    }
  }
  os.nl();
}


/*-
 *-----------------------------------------------------------------------
 * writeMixtures
 *
 *   Writes out all the mixtures in the global object. If there are
 *   any internal pre-defined mixtures in this set, then they are assumed
 *   to be kept at the beginning of the array, and are not written out
 *   (since they are automatically created anew each time the program
 *   loads). See routine loadGlobal() for more details.
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out, so moves file pointers.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeMixtures(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Mixtures of components");os.nl();


  // We do not save the final two mixtures as they are internal
  // objects. See routine finalizeParameters().  This is the reason
  // for the (-2) in the below.


  // first scan through and find the number that are used.
  const unsigned maxPossibleNumUsed = mixtures.size()-2;
  unsigned used = 0;
  for (unsigned i=0;i<maxPossibleNumUsed;i++)
    if (mixtures[i]->emUsedBitIsSet())
      used++;

  if (used != maxPossibleNumUsed)
    warning("NOTE: saving only %d used mixtures out of a total of %d",
	    used,mixtures.size());

  os.write(used,"num MIXCOMPONENTS"); os.nl();

  unsigned index=0;
  for (unsigned i=0;i<maxPossibleNumUsed;i++) {

    if (mixtures[i]->emUsedBitIsSet()){

      // first write the count
      os.write(index++,"MIXCOMPONENTS cnt");
      os.nl();

      // next write the dimension of this mixture
      os.write(mixtures[i]->dim(),"MG dim");
      os.nl();
      
      mixtures[i]->write(os);
    }
  }

  assert(used==index);

  os.nl();
}





/*-
 *-----------------------------------------------------------------------
 * writeNameCollections
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeNameCollections(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Collection of Names");os.nl();
  unsigned numToWrite = ncls.size();
  if (nclsMap.find(string(NAMED_COLLECTION_GLOBAL_NAME)) != nclsMap.end()) {
    numToWrite --;
  }
  os.write(numToWrite,"num collections"); os.nl();
  for (unsigned i=0;i<ncls.size();i++) {
    if (ncls[i]->name() == NAMED_COLLECTION_GLOBAL_NAME)
      continue;
    // write the count
    os.write(i,"NCLS cnt");
    os.nl();
    ncls[i]->write(os);
  }
  os.nl();
}





/*-
 *-----------------------------------------------------------------------
 * writeGausSwitchMixtures
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeGausSwitchMixtures(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Gaussian Switching Mixtures");os.nl();
  os.write(0,"num GausSwitchMIXTURE"); os.nl();
}




/*-
 *-----------------------------------------------------------------------
 * writeLogitSwitchMixtures
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeLogitSwitchMixtures(oDataStreamFile& os)
{
  os.nl(); os.writeComment("Logistic-Regression-based Switching Mixtures");os.nl();
  os.write(0,"num GausSwitchMIXTURE"); os.nl();
}




/*-
 *-----------------------------------------------------------------------
 * writeMlpSwitchMixtures
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      one
 *
 * Side Effects:
 *      all "used" parameters are written out.
 *
 * Results:
 *      nil.
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeMlpSwitchMixtures(oDataStreamFile& os)
{
  os.nl(); os.writeComment("MLP-based Switching Mixtures");os.nl();
  os.write(0,"num GausSwitchMIXTURE"); os.nl();
}


/*
 *  
 * ==================================================================
 *			 GROUP WRITE ROUTINES
 * ==================================================================
 *
 */



/*-
 *-----------------------------------------------------------------------
 * writeAll
 *   write out everything (all parameters) in one go.
 * 
 * Preconditions:
 *      none
 *
 * Postconditions:
 *      files write out are written.
 *
 * Side Effects:
 *      none
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeAll(oDataStreamFile& os, bool remove_unnamed)
{
  markUsedMixtureComponents(remove_unnamed);

  // just write everything in one go.

  // write structural items
  writeDTs(os);
  writeDLinks(os);

  // then write basic numeric items.
  writeDPmfs(os);
  writeSPmfs(os);
  writeMeans(os);
  writeCovars(os);
  writeDLinkMats(os);
  writeRealMats(os);  
#if DOUBLEMATS_EVERYWHERE
  writeDoubleMats(os);  
#endif
  writeMdCpts(os);
  writeMsCpts(os);
  writeMtCpts(os);

  // next write definitional items
  writeComponents(os);
  writeMixtures(os);
  writeGausSwitchMixtures(os);
  writeLogitSwitchMixtures(os);
  writeMlpSwitchMixtures(os);

}


/*-
 *-----------------------------------------------------------------------
 * writeTrainable
 *   write out trainable parameters, see readTrainable for docs. 
 * 
 * Preconditions:
 *      none
 *
 * Postconditions:
 *      files write out are written.
 *
 * Side Effects:
 *      none
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeTrainable(oDataStreamFile& os, bool remove_unnamed)
{

  markUsedMixtureComponents(remove_unnamed);

  // just write everything in one go.

  // then write basic numeric items.
  writeDPmfs(os);
  writeSPmfs(os);
  writeMeans(os);
  writeCovars(os);
  writeDLinkMats(os);
  writeRealMats(os);  
#if DOUBLEMATS_EVERYWHERE
  writeDoubleMats(os);  
#endif
  writeMdCpts(os);

  // next write definitional items
  writeComponents(os);
  writeMixtures(os);
  writeGausSwitchMixtures(os);
  writeLogitSwitchMixtures(os);
  writeMlpSwitchMixtures(os);
}


/*-
 *-----------------------------------------------------------------------
 * writeNonTrainable
 *   write out non-trainable parameters, see readNonTrainable for docs. 
 * 
 * Preconditions:
 *      none
 *
 * Postconditions:
 *      files write out are written.
 *
 * Side Effects:
 *      none
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void 
GMParms::writeNonTrainable(oDataStreamFile& os, bool remove_unnamed)
{
  // just write everything in one go.
  markUsedMixtureComponents(remove_unnamed);

  // write structural items
  writeDTs(os);
  writeDLinks(os);

  writeMsCpts(os);
  writeMtCpts(os);
}



void 
GMParms::write(const char *const outputFileFormat, const char * const cppCommandOptions, const int intTag, bool remove_unnamed)
{
  //
  // read a file consisting of a list of 
  // 
  //      <keyword>  <filename> {ascii|binary}
  //
  // triplets. 
  // The keyword says which structure to write out,
  // The filename says where to write it.
  // And the {ascii|binary} tag says if we should write it in ascii or
  //   binary.
  //

  if (outputFileFormat == NULL)
    return;

  string keyword;
  string fileName;
  string binStatus;

  map<string,oDataStreamFile*> fileNameMap;
  char buff[2048];
  iDataStreamFile is(outputFileFormat,false,true,cppCommandOptions);

  markUsedMixtureComponents(remove_unnamed);

  while (is.readString(keyword)) {

    if (!is.readString(fileName)) {
      error("ERROR: while reading file '%s' line %d, got keyword '%s' without a filename",
	    is.fileName(),is.lineNo(),keyword.c_str());
    }
    

    bool binary_p = is.binary();
    // read binary status of file
    if (!is.readString(binStatus)) {
      error("ERROR: while reading file '%s' line %d, got keyword '%s' and filename '%s' without a binary status",
	    is.fileName(),is.lineNo(),keyword.c_str(),fileName.c_str());
    }
    if (binStatus == "ascii" || binStatus == "ASCII")
      binary_p = false;
    else if (binStatus == "binary" || binStatus == "BINARY")
      binary_p = true;
    else {
      error("ERROR: while reading file '%s' line %d, got string '%s' when expecting 'ascii'/'binary' keyword",
	    is.fileName(),is.lineNo(),binStatus.c_str());
    }

    copyStringWithTag(buff,fileName.c_str(),intTag,sizeof(buff));
    fileName = buff;

    map<string,oDataStreamFile*>::iterator it = fileNameMap.find(fileName);
    if (it == fileNameMap.end()) {
      // then this is a previously unencountered filename.
      fileNameMap[fileName] = new oDataStreamFile(fileName.c_str(),binary_p);
      it = fileNameMap.find(fileName);
    } else if (((*it).second)->binary() != binary_p) {
      // need to close the file and re-open it with the appropriate binary status, and
      // in append mode
      // delete ((*it).second);
      // fileNameMap[fileName] = new oDataStreamFile(fileName.c_str(),binary_p,true);
      // it = fileNameMap.find(fileName);
      error("ERROR: reading '%s' line %d. File '%s' had binary status = %d, now binary status = %d. Can't mix binary and ASCII files",
	    is.fileName(),is.lineNo(),fileName.c_str(),((*it).second)->binary(),binary_p);
    }

    if (keyword == "DPMF_OUT_FILE") {
      writeDPmfs(*((*it).second));

    } else if (keyword == "SPMF_OUT_FILE") {
      writeSPmfs(*((*it).second));

    } else if (keyword == "MEAN_OUT_FILE") {
      writeMeans(*((*it).second));

    } else if (keyword == "COVAR_OUT_FILE") {
      writeCovars(*((*it).second));

    } else if (keyword == "DLINK_MAT_OUT_FILE") {
      writeDLinkMats(*((*it).second));

    } else if (keyword == "DLINK_OUT_FILE") {
      writeDLinks(*((*it).second));

    } else if (keyword == "WEIGHT_MAT_OUT_FILE") {
      // TODO: remove this backward compatibility case.
      writeRealMats(*((*it).second));
    } else if (keyword == "REAL_MAT_OUT_FILE") {
      writeRealMats(*((*it).second));

    } else if (keyword == "DOUBLE_MAT_OUT_FILE") {
      writeDoubleMats(*((*it).second));

    } else if (keyword == "DIRICHLET_TAB_OUT_FILE") {
      writeDirichletTabs(*((*it).second));

    } else if (keyword == "DENSE_CPT_OUT_FILE") {
      writeMdCpts(*((*it).second));

    } else if (keyword == "SPARSE_CPT_OUT_FILE") {
      writeMsCpts(*((*it).second));

    } else if (keyword == "DETERMINISTIC_CPT_OUT_FILE") {
      writeMtCpts(*((*it).second));

    }
#if 0
    // ticket 536: don't write out DTs
     else if (keyword == "DT_OUT_FILE") {
      writeDTs(*((*it).second));

    } 
#endif
    else if (keyword == "MC_OUT_FILE") {
      writeComponents(*((*it).second));

    } else if (keyword == "MX_OUT_FILE") {
      writeMixtures(*((*it).second));

    } else if (keyword == "NAME_COLLECTION_OUT_FILE") {
      writeNameCollections(*((*it).second));

    } else if (keyword == "GSMG_OUT_FILE") {
      error("GSMG_OUT_FILE in file '%s' line %d, not implemented",
	    is.fileName(),is.lineNo());

    } else if (keyword == "LSMG_OUT_FILE") {
      error("LSMG_OUT_FILE in file '%s' line %d, not implemented",
	    is.fileName(),is.lineNo());

    } else if (keyword == "MSMG_OUT_FILE") {
      error("MSMG_OUT_FILE in file '%s' line %d, not implemented",
	    is.fileName(),is.lineNo());

    } else {
      error("ERROR: encountered unknown file type '%s' in file '%s' line %d",
	    keyword.c_str(),is.fileName(),is.lineNo());
    }
  }

  // now go through and delete all the input files
  for (map<string,oDataStreamFile*>::iterator it = fileNameMap.begin();
       it != fileNameMap.end(); it++) {
      delete ((*it).second);
  }
}





////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
//        Misc Routines
////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////



/*-
 *-----------------------------------------------------------------------
 * begin()
 *      do any necessary bookkeeping work with regard to the
 *      parameters in order that we properly instantiate the first example.
 * 
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      nil
 *
 * Side Effects:
 *      might change internal objects.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void
GMParms::begin()
{
  for(unsigned i = 0; i<iterableDts.size(); i++) {
    iterableDts[i]->setFirstDecisionTree(firstUtterance);
    iterableDts[i]->beginIterableDT();
  }
  for ( unsigned i = 0; i < iterableLatticeAdts.size(); i++ ) {
	  iterableLatticeAdts[i]->beginIterableLattice();
  }
  for (unsigned i=0;i<dLinks.size();i++) {
    dLinks[i]->clearArrayCache();
  }
  // clear any mixtures component cache
  for (unsigned i=0;i<mixtures.size();i++) {
    mixtures[i]->emptyComponentCache();
  }
}



/*-
 *-----------------------------------------------------------------------
 * next()
 *      Move on to the next example.
 * 
 * Preconditions:
 *      begin() must have been called.
 *
 * Postconditions:
 *      nil
 *
 * Side Effects:
 *      might change internal objects.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void
GMParms::next()
{
  for(unsigned i = 0; i<iterableDts.size(); i++) {
    iterableDts[i]->nextIterableDT();
  }
  for ( unsigned i = 0; i < iterableLatticeAdts.size(); i++ ) {
	  iterableLatticeAdts[i]->nextIterableLattice();
  }
  for (unsigned i=0;i<dLinks.size();i++) {
    dLinks[i]->clearArrayCache();
  }
  // clear any mixtures component cache
  for (unsigned i=0;i<mixtures.size();i++) {
    mixtures[i]->emptyComponentCache();
  }

}




/*-
 *-----------------------------------------------------------------------
 * setSegment
 *      do any necessary bookkeeping work with regard to the
 *      parameters in order that we properly move to the segment
 *      given by the argument to this function. We do this
 *      here since some of the parameters can be segment dependent.
 *
 * Preconditions:
 *      nil
 *
 * Postconditions:
 *      nil
 *
 * Side Effects:
 *      might change internal objects.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
unsigned
GMParms::setSegment(const unsigned segmentNo)
{
  globalObservationMatrix->openSegment(segmentNo);
  unsigned numFrames = globalObservationMatrix->numFrames();

  for(unsigned i = 0; i<iterableDts.size(); i++) {
    iterableDts[i]->seek(segmentNo);
    iterableDts[i]->nextIterableDT();
  }
  for (unsigned i=0; i< veCpts.size(); i++) {
    veCpts[i]->setSegment(segmentNo);
    
    // FIXME - investigate if veCpts work with stream input (where
    //         the number of frames isn't known)
    if (numFrames != 0 && veCpts[i]->numFrames() != numFrames) 
      error("ERROR: number of frames in segment %d for main observation matrix is %d, but VirtualEvidenceCPT '%s' observation matrix has %d frames in that segment",segmentNo,numFrames,veCpts[i]->name().c_str(),veCpts[i]->numFrames());
  }

  for (unsigned i=0; i< deepVECpts.size(); i++) {
    deepVECpts[i]->setSegment(segmentNo);

#if 0
// This is not needed since DeepVECPTs only take input from the
// global observation matrix
    // FIXME - investigate if DeepVECpts work with stream input (where
    //         the number of frames isn't known)
    if (numFrames != 0 && deepVECpts[i]->numFrames() != numFrames) 
      error("ERROR: number of frames in segment %d for main observation matrix is %d, but DeepVirtualEvidenceCPT '%s' observation matrix has %d frames in that segment",segmentNo,numFrames,veCpts[i]->name().c_str(),veCpts[i]->numFrames());
#endif
  }

  // FIXME - investigate if lattice CPTs work with stream input (where
  //         the number of frames isn't known)
  if (numFrames != 0) {

  // set the lattice CPT frame indices
  for (unsigned i=0; i < latticeAdts.size(); i++ ) {
	  // I cannot call iterableLatticeAdts here because
	  // there is overlap of the object and reset frame indices
	  // needs to be done to all lattices
	  if ( latticeAdts[i]->iterable() ) {
		  latticeAdts[i]->seek(segmentNo);
		  latticeAdts[i]->nextIterableLattice();
	  }
	  latticeAdts[i]->resetFrameIndices(numFrames);
  }

  }

  for (unsigned i=0;i<dLinks.size();i++) {
    dLinks[i]->clearArrayCache();
  }
  // clear any mixtures component cache
  for (unsigned i=0;i<mixtures.size();i++) {
    mixtures[i]->emptyComponentCache();
  }

  return numFrames;
}




/*-
 *-----------------------------------------------------------------------
 * setStride()
 *      Inform all objects what the current stride will be from the
 *      observation matrix that will be used.
 * 
 * Preconditions:
 *      none
 *
 * Postconditions:
 *      internal objects are set to use stride.
 *
 * Side Effects:
 *      internal objects are set to use stride.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void
GMParms::setStride(const unsigned stride)
{
  for (unsigned i=0;i<dLinks.size();i++) {
    dLinks[i]->preCompute(stride);
  }
}




/*-
 *-----------------------------------------------------------------------
 * checkConsistentWithGlobalObservationStream()
 *      Check that the parameters are consistent
 *      with the observation matrix
 * 
 * Preconditions:
 *      global observation matrix must have been read in.
 *
 * Postconditions:
 *      none
 *
 * Side Effects:
 *      possible error if problems occur.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void
GMParms::checkConsistentWithGlobalObservationStream()
{
  if ((int)globalObservationMatrix->startSkip() < -(int)Dlinks::_globalMinLag)
    error("ERROR: a start skip of %d is invalid for a minimum dlink lag of %d\n",
	  globalObservationMatrix->startSkip(),
	  Dlinks::_globalMinLag);

  if (globalObservationMatrix->startSkip() < DeepVECPT::minSkip())
    error("ERROR: start skip must be at least %u due to DeepVECPT radii\n", DeepVECPT::minSkip());

  if ((int)globalObservationMatrix->endSkip() < (int)Dlinks::_globalMaxLag)
    error("ERROR: an end skip of %d is invalid for a maximum dlink lag of %d\n",
	  globalObservationMatrix->endSkip(),
	  Dlinks::_globalMaxLag);
  
  if (globalObservationMatrix->randomAccess() && globalObservationMatrix->endSkip() < DeepVECPT::minSkip())
    error("ERROR: end skip must be at least %u due to DeepVECPT radii\n", DeepVECPT::minSkip());


  if ((int)globalObservationMatrix->numContinuous() <= (int)Dlinks::_globalMaxOffset)
    error("ERROR: there is a dlink ofset of value %d which is too large for the observation matrix with only %d continuous features.",
	  Dlinks::_globalMaxOffset,
	  globalObservationMatrix->numContinuous());
}


/*-
 *-----------------------------------------------------------------------
 * totalNumberParameters
 *      return the total number of parameters used by this system.
 *      NOTE: The code in this routine is very dependent on the
 *      way that the internal objects are implemented, which means
 *      that not all objects are queried for their number of parameters.
 *      For example, sparsePMFs use dense PMFs, so we only query
 *      all the dense ones to get the total number of parameters.
 * 
 * Preconditions:
 *      parameters should be read in to have non zero value
 *
 * Postconditions:
 *      none
 *
 * Side Effects:
 *      none
 *
 * Results:
 *      obvious
 *
 *-----------------------------------------------------------------------
 */
unsigned GMParms::totalNumberParameters()
{
  unsigned sum=0;

  // gets dpmfs, and covers spmfs also. This covers mixture coefficients.
  for (unsigned i=0;i<dPmfs.size();i++)
    sum += dPmfs[i]->totalNumberParameters();

  // components, gets means, vars, dlinks, real mats
  for (unsigned i=0;i<components.size();i++)
    sum += components[i]->totalNumberParameters();

  // for discrete RVs
  for (unsigned i=0;i<mdCpts.size();i++)
    sum += mdCpts[i]->totalNumberParameters();
  for (unsigned i=0;i<msCpts.size();i++)
    sum += msCpts[i]->totalNumberParameters();
  for (unsigned i=0;i<mtCpts.size();i++)
    sum += mtCpts[i]->totalNumberParameters();
  for (unsigned i=0;i<ngramCpts.size();i++)
    sum += ngramCpts[i]->totalNumberParameters();
  for (unsigned i=0;i<fngramCpts.size();i++)
    sum += fngramCpts[i]->totalNumberParameters();
  for (unsigned i=0;i<veCpts.size();i++)
    sum += veCpts[i]->totalNumberParameters();
  for (unsigned i=0;i<deepNNs.size();i++)
    sum += deepNNs[i]->totalNumberParameters();
  for (unsigned i=0;i<deepVECpts.size();i++)
    sum += deepVECpts[i]->totalNumberParameters();
  return sum;

}



/*-
 *-----------------------------------------------------------------------
 * markOnlyNamedMixtures()
 *      mark only mixture distributions that are named in one of the
 *      named collections
 * 
 * Preconditions:
 *      all candidates for deletion should have had their UsedBit
 *      cleared
 *
 * Postconditions:
 *      none
 *
 * Side Effects:
 *      none
 *
 * Results:
 *      obvious
 *
 *-----------------------------------------------------------------------
 */
void GMParms::markOnlyNamedMixtures()
{

  // rebuild the mixture map, in case it got out of sync with the
  // mixtures vector - is this ever necessary?
  /*
  mixturesMap.clear();
  int j=0;
  for(vector<Mixture*>::iterator mi=mixtures.begin();mi!=mixtures.end();mi++,j++)
    mixturesMap[(*mi)->name()]=j;
  */

  // do each name collection in turn
  infoMsg(IM::Huge,"Marking mixtures that are listed in a loaded name collection: ");
  unsigned n=0,nu=0;
  for(vector<NameCollection*>::iterator nc=ncls.begin();nc!=ncls.end();nc++){
    for(vector<std::string>::iterator nci=(*nc)->table.begin(); nci!=(*nc)->table.end(); nci++){
      if (!mixtures[mixturesMap[*nci]]->emUsedBitIsSet()){
	n++;
	mixtures[mixturesMap[*nci]]->recursivelySetUsedBit();
      } else
	nu++;
      
    }
  } 
  infoMsg(IM::Huge,"%d were marked (there are %d further mixtures that are not listed in a name collection) \n",n,nu);

  /*
  cerr << "actually removing unused mixtures" << endl;

  too slow


  // now we must actually delete all unused Mixtures, because all the
  // various saving routines will call markUsedMixtureComponents(),
  // which will mark *all* mixtures as being used, even if they are
  // not
  for (vector<Mixture*>::iterator i=mixtures.begin();i!=mixtures.end();){
    if(! (*i)->emUsedBitIsSet())
      i=mixtures.erase(i);
    else 
      i++;
  }

  */

  infoMsg(IM::Huge,"Deleting unused DPMFs: ");
  unsigned old_size=dPmfs.size();

  vector< Dense1DPMF* > new_dPmfs;

  int s=0;
  for (vector<Dense1DPMF*>::iterator i=dPmfs.begin();i!=dPmfs.end();i++)
    if((*i)->emUsedBitIsSet())
      s++;

  new_dPmfs.reserve(s);

  // and the same for DPMFs (should probably fix writeDPmfs instead)
  for (vector<Dense1DPMF*>::iterator i=dPmfs.begin();i!=dPmfs.end();i++){
    if((*i)->emUsedBitIsSet())
      new_dPmfs.push_back(*i);
  }

  dPmfsMap.clear();
  dPmfs.clear();
  dPmfs.reserve(new_dPmfs.size());

  unsigned j=0;
  for (vector<Dense1DPMF*>::iterator i=new_dPmfs.begin();i!=new_dPmfs.end();i++,j++){
    dPmfsMap[(*i)->name()]=j;
    dPmfs.push_back(*i);
  }

  infoMsg(IM::Default," %d of %d DPMFs were deleted; %d were retained\n",old_size-dPmfs.size(),old_size,dPmfs.size());


}

/*-
 *-----------------------------------------------------------------------
 * markUsedMixtureComponents()
 *      Remove all mixture component objects that are not used by anyone
 *      (possibly because of vanishing).
 * 
 * Preconditions:
 *      parameters should be read in to have non zero value
 *
 * Postconditions:
 *      none
 *
 * Side Effects:
 *      will remove all paramters not used.
 *
 * Results:
 *      obvious
 *
 *-----------------------------------------------------------------------
 */
void GMParms::markUsedMixtureComponents(bool remove_unnamed)
{

  ///////////////////////////////////////////////////
  // First, go through *all* component related parameters
  // and mark as not used.
  for (unsigned i=0;i<means.size();i++)
    means[i]->recursivelyClearUsedBit();
  for (unsigned i=0;i<covars.size();i++)
    covars[i]->recursivelyClearUsedBit();
  for (unsigned i=0;i<dLinkMats.size();i++)
    dLinkMats[i]->recursivelyClearUsedBit();
  for (unsigned i=0;i<realMats.size();i++)
    realMats[i]->recursivelyClearUsedBit();
  for (unsigned i=0;i<doubleMats.size();i++)
    doubleMats[i]->recursivelyClearUsedBit();
  for (unsigned i=0;i<components.size();i++)
    components[i]->recursivelyClearUsedBit();
  for (unsigned i=0;i<mixtures.size();i++)
    mixtures[i]->recursivelyClearUsedBit();

  ///////////////////////////////////////////////
  // Now, set only those bits that are used
  // by some mixture.

  if(remove_unnamed)
    markOnlyNamedMixtures();
  else
    for (unsigned i=0;i<mixtures.size();i++)
      mixtures[i]->recursivelySetUsedBit();

  for (unsigned i=0; i < deepNNs.size(); i+=1) {
    vector<DoubleMatrix *> mats = deepNNs[i]->getMatrices();
    for (unsigned m=0; m < mats.size(); m += 1) {
      mats[m]->recursivelySetUsedBit();
    }
  }
}




/*-
 *-----------------------------------------------------------------------
 *  markObjectsToNotTrain(fn)
 *      Using the file given by file name, mark all objects
 *      listed in that file so that they are not trained by EM. 
 *  The format of the file is a list of pairs, where
 *  the first element in the pair specifies an object type
 *  and the 2nd gives an object name.
 *
 *
 * 
 * Preconditions:
 *      parameters should be read in to have non zero value
 *
 * Postconditions:
 *      none
 *
 * Side Effects:
 *      will set all objects listed in file so that they are not trained.
 *
 * Results:
 *      il
 *
 *-----------------------------------------------------------------------
 */
void GMParms::markObjectsToNotTrain(const char*const fileName,
				    const char *const cppOptions)
{
  if (fileName == NULL)
    return;

  iDataStreamFile is(fileName, // name
			false,    // not binary 
			true,     // run cpp
			cppOptions);

  string objType;
  string objName;
  while (is.readString(objType)) {
    if (!is.readString(objName)) {
      error("ERROR: while reading file '%s' line %d, got object type '%s' without an object name",
	    is.fileName(),is.lineNo(),
	    objType.c_str());
    }

#define EMCLEARAMTRAININGBIT_CODE(mapName,name) \
      if (objName == "*") { \
         for (unsigned i=0;i<name.size();i++)  \
            name[i]->emClearAmTrainingBit(); \
      } else { \
	if ((it=mapName.find(objName)) == mapName.end()) \
	  error("ERROR: can't find object '%s' of type '%s' listed in file '%s' line %d of objects to not train.", \
	      objName.c_str(), \
	      objType.c_str(), \
	      is.fileName(),is.lineNo()); \
        name[(*it).second]->emClearAmTrainingBit(); }


    ObjectMapType::iterator it;
    if (objType == "DPMF") {
      EMCLEARAMTRAININGBIT_CODE(dPmfsMap,dPmfs);
    } else if (objType == "SPMF") {
      EMCLEARAMTRAININGBIT_CODE(sPmfsMap,sPmfs);
    } else if (objType == "MEAN") {
      EMCLEARAMTRAININGBIT_CODE(meansMap,means);
    } else if (objType == "COVAR") {
      EMCLEARAMTRAININGBIT_CODE(covarsMap,covars);
    } else if (objType == "DLINKMAT") {
      EMCLEARAMTRAININGBIT_CODE(dLinkMatsMap,dLinkMats);
    } else if (objType == "WEIGHTMAT") {
      EMCLEARAMTRAININGBIT_CODE(realMatsMap,realMats);
    } else if (objType == "REALMAT") {
      EMCLEARAMTRAININGBIT_CODE(realMatsMap,realMats);
    } else if (objType == "DOUBLEMAT") {
      EMCLEARAMTRAININGBIT_CODE(doubleMatsMap,doubleMats);
    } else if (objType == "COMPONENT") {
      EMCLEARAMTRAININGBIT_CODE(componentsMap,components);
    } else if (objType == "DENSECPT") {
      EMCLEARAMTRAININGBIT_CODE(mdCptsMap,mdCpts);
    } else if (objType == "SPARSECPT") {
      EMCLEARAMTRAININGBIT_CODE(msCptsMap,msCpts);
    } else if (objType == "DETERMINISTICCPT") {
      EMCLEARAMTRAININGBIT_CODE(mtCptsMap,mtCpts);
    } else if (objType == "MIXTURE") {
      EMCLEARAMTRAININGBIT_CODE(mixturesMap,mixtures);
    } else {
      error("ERROR: bad object type name '%s' in file '%s' line %d of objects to not train.",
	    objType.c_str(),
	    is.fileName(),is.lineNo());
    }

  }
}



 /*-
  *-----------------------------------------------------------------------
  * makeRandom
  *      make everyone random
  * 
  * Preconditions:
  *      all must be read in.
  *
  * Postconditions:
  *      What is true after the function is called.
  *
  * Side Effects:
  *      all variable parameters are changed
  *
  * Results:
  *      nil
  *
  *-----------------------------------------------------------------------
  */
 void
 GMParms::makeRandom()
 {
   // go through all EMable objects possibly
   // used by any RV and make the call

   // first do the basic objects
   // which also does everything for continuous RVs
   for (unsigned i=0;i<dPmfs.size();i++)
     dPmfs[i]->makeRandom();
   for (unsigned i=0;i<sPmfs.size();i++)
     sPmfs[i]->makeRandom();
   for (unsigned i=0;i<means.size();i++)
     means[i]->makeRandom();
   for (unsigned i=0;i<covars.size();i++)
     covars[i]->makeRandom();
   for (unsigned i=0;i<dLinkMats.size();i++)
     dLinkMats[i]->makeRandom();
   for (unsigned i=0;i<realMats.size();i++)
     realMats[i]->makeRandom();
   for (unsigned i=0;i<doubleMats.size();i++)
     doubleMats[i]->makeRandom();

   // components
   for (unsigned i=0;i<components.size();i++)
     components[i]->makeRandom();

   // for discrete RVs
   for (unsigned i=0;i<mdCpts.size();i++)
     mdCpts[i]->makeRandom();
   for (unsigned i=0;i<msCpts.size();i++)
     msCpts[i]->makeRandom();
   for (unsigned i=0;i<mtCpts.size();i++)
     mtCpts[i]->makeRandom();

 }



void
GMParms::makeUniform()
{
  // go through all EMable objects possibly
  // used by any RV and make the call

   // first do the basic objects
   // which also does everything for continuous RVs
  for (unsigned i=0;i<dPmfs.size();i++)
    dPmfs[i]->makeUniform();
  for (unsigned i=0;i<sPmfs.size();i++)
    sPmfs[i]->makeUniform();
  for (unsigned i=0;i<means.size();i++)
    means[i]->makeUniform();
  for (unsigned i=0;i<covars.size();i++)
    covars[i]->makeUniform();
  for (unsigned i=0;i<dLinkMats.size();i++)
    dLinkMats[i]->makeUniform();
  for (unsigned i=0;i<realMats.size();i++)
    realMats[i]->makeUniform();
  for (unsigned i=0;i<doubleMats.size();i++)
    doubleMats[i]->makeUniform();

   // components
  for (unsigned i=0;i<components.size();i++)
    components[i]->makeUniform();

   // for discrete RVs
  for (unsigned i=0;i<mdCpts.size();i++)
    mdCpts[i]->makeUniform();
  for (unsigned i=0;i<msCpts.size();i++)
    msCpts[i]->makeUniform();
  for (unsigned i=0;i<mtCpts.size();i++)
    mtCpts[i]->makeUniform();

}


////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
//        EM Routines
////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////



/*-
 *-----------------------------------------------------------------------
 * emEndIteration()
 *      Call emEndIteration() for all top level objects (top level
 *      objects are those who will not be pointed to by other objects).
 *
 * Preconditions:
 *      em should be running.
 *
 * Postconditions:
 *      EM iteration is finished.
 *
 * Side Effects:
 *      Changes almost all of the parameter objects.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void
GMParms::emEndIteration()
{
  // go through all EMable objects possibly
  // used by any RV and make the call

   /////////////////////////////////////////////////////////////////
   // First, do the mixtures. This will recursively call
   // this for all mean-like objects, covariance-like objects, and so
   // on, so there is no need to do those here.  
   //
   // Mean-like objects include:
   //
   //       components    
   //       means
   //       linCondMeans
   //       nonLinCondMeans
   //       ...
   // 
   // Variance-like objects include
   // 
   //       covars
   // 
  for (unsigned i=0;i<mixtures.size();i++)
    mixtures[i]->emEndIteration();

   //////////////////////////////////////////////////////////////
   // We don't do the following code  here because those objects use
   // ref counts needed for sharing. The components will
   // themselves end the EM iteration for those objects, and will call
   // it the appropriate number of times so that the ref counts are set
   // properly.
   //
   //    for (unsigned i=0;i<means.size();i++)
   //      means[i]->emEndIteration();
   //    for (unsigned i=0;i<covars.size();i++)
   //      covars[i]->emEndIteration();
   //    for (unsigned i=0;i<dLinkMats.size();i++)
   //      dLinkMats[i]->emEndIteration();
   //    for (unsigned i=0;i<realMats.size();i++)
   //      realMats[i]->emEndIteration();
   //
   //////////////////////////////////////////////////////////////

   // finish up the spmfs and dpmfs, some of which
   // might already have been ended by the
   // above code via the mixture component densities. 
  for (unsigned i=0;i<dPmfs.size();i++)
    dPmfs[i]->emEndIteration();
  for (unsigned i=0;i<sPmfs.size();i++)
    sPmfs[i]->emEndIteration();

   // finally, end all the CPT. for discrete RVs
  for (unsigned i=0;i<mdCpts.size();i++)
    mdCpts[i]->emEndIteration();
  for (unsigned i=0;i<msCpts.size();i++)
    msCpts[i]->emEndIteration();
  for (unsigned i=0;i<mtCpts.size();i++)
    mtCpts[i]->emEndIteration();


#if 0
  // uncomment this when the objects get written.
  for (unsigned i=0;i<gausSwitchingMixtures.size();i++)
    gausSwitchingMixtures[i]->emEndIteration();
  for (unsigned i=0;i<logitSwitchingMixtures.size();i++)
    logitSwitchingMixtures[i]->emEndIteration();
  for (unsigned i=0;i<mlpSwitchingMixtures.size();i++)
    mlpSwitchingMixtures[i]->emEndIteration();
#endif

}




/*-
 *-----------------------------------------------------------------------
 * emSwapCurAndNew()
 *      Call emSwapCurAndNew() for all top level objects (top level
 *      objects are those who will not be pointed to by other objects).
 *
 * Preconditions:
 *      EM should not be running.
 *
 * Postconditions:
 *      EM iteration is finished.
 *
 * Side Effects:
 *      Changes almost all of the parameter objects.
 *
 * Results:
 *      nil
 *
 *-----------------------------------------------------------------------
 */
void
GMParms::emSwapCurAndNew()
{
  // go through all EMable objects possibly
  // used by any RV and make the call

   // for continuous RVs, this will recursively
   // call swap for all sub objects.
  for (unsigned i=0;i<mixtures.size();i++)
    mixtures[i]->emSwapCurAndNew();

   // make sure that all dpmfs and spmfs have been swapped.
  for (unsigned i=0;i<dPmfs.size();i++)
    dPmfs[i]->emSwapCurAndNew();
  for (unsigned i=0;i<sPmfs.size();i++)
    sPmfs[i]->emSwapCurAndNew();

  // Don't swap these since it should have
  // been done by the mixture swapping above.
  //    for (unsigned i=0;i<means.size();i++)
  //      means[i]->emSwapCurAndNew();
  //    for (unsigned i=0;i<covars.size();i++)
  //      covars[i]->emSwapCurAndNew();
  //    for (unsigned i=0;i<dLinkMats.size();i++)
  //      dLinkMats[i]->emSwapCurAndNew();
  //    for (unsigned i=0;i<realMats.size();i++)
  //      realMats[i]->emSwapCurAndNew();
  //    for (unsigned i=0;i<components.size();i++)
  //      components[i]->emSwapCurAndNew();

   // for discrete RVs
  for (unsigned i=0;i<mdCpts.size();i++)
    mdCpts[i]->emSwapCurAndNew();
  for (unsigned i=0;i<msCpts.size();i++)
    msCpts[i]->emSwapCurAndNew();
  for (unsigned i=0;i<mtCpts.size();i++)
    mtCpts[i]->emSwapCurAndNew();


#if 0
  // uncomment this when the objects get written.
  for (unsigned i=0;i<gausSwitchingMixtures.size();i++)
    gausSwitchingMixtures[i]->emSwapCurAndNew();
  for (unsigned i=0;i<logitSwitchingMixtures.size();i++)
    logitSwitchingMixtures[i]->emSwapCurAndNew();
  for (unsigned i=0;i<mlpSwitchingMixtures.size();i++)
    mlpSwitchingMixtures[i]->emSwapCurAndNew();
#endif

  // clear out the cloning maps, under the assumption
  // that all clones become 'real' objects.
  MixtureCommon::vanishingComponentSet.clear();
  MixtureCommon::splittingComponentSet.clear();

  MixtureCommon::meanCloneMap.clear();
  MixtureCommon::dLinkMatCloneMap.clear();
  MixtureCommon::diagCovarCloneMap.clear();
  MixtureCommon::mcCloneMap.clear();

}

void
GMParms::emStoreAccumulators(oDataStreamFile& ofile)
{
  // first do the basic objects
  for (unsigned i=0;i<dPmfs.size();i++)
    dPmfs[i]->emStoreAccumulators(ofile);
  for (unsigned i=0;i<sPmfs.size();i++)
    sPmfs[i]->emStoreAccumulators(ofile);
  for (unsigned i=0;i<means.size();i++)
    means[i]->emStoreAccumulators(ofile);
  for (unsigned i=0;i<covars.size();i++)
    covars[i]->emStoreAccumulators(ofile);
  for (unsigned i=0;i<dLinkMats.size();i++)
    dLinkMats[i]->emStoreAccumulators(ofile);
  for (unsigned i=0;i<realMats.size();i++)
    realMats[i]->emStoreAccumulators(ofile);
#if DOUBLEMATS_EVERYWHERE
  for (unsigned i=0;i<doubleMats.size();i++)
    doubleMats[i]->emStoreAccumulators(ofile);
#endif

   // components
  for (unsigned i=0;i<components.size();i++)
    components[i]->emStoreAccumulators(ofile);

   // for discrete RVs
  for (unsigned i=0;i<mdCpts.size();i++)
    mdCpts[i]->emStoreAccumulators(ofile);
  for (unsigned i=0;i<msCpts.size();i++)
    msCpts[i]->emStoreAccumulators(ofile);
  for (unsigned i=0;i<mtCpts.size();i++)
    mtCpts[i]->emStoreAccumulators(ofile);

   // for continuous RVs
  for (unsigned i=0;i<mixtures.size();i++)
    mixtures[i]->emStoreAccumulators(ofile);
#if 0
  // uncomment this when the objects get written.
  for (unsigned i=0;i<gausSwitchingMixtures.size();i++)
    gausSwitchingMixtures[i]->emStoreAccumulators(ofile);
  for (unsigned i=0;i<logitSwitchingMixtures.size();i++)
    logitSwitchingMixtures[i]->emStoreAccumulators(ofile);
  for (unsigned i=0;i<mlpSwitchingMixtures.size();i++)
    mlpSwitchingMixtures[i]->emStoreAccumulators(ofile);
#endif

}

void
GMParms::emLoadAccumulators(iDataStreamFile& ifile)
{
  /////////////////////////////////////////////////////////////
  // First, make sure the EM iterations have been
  // started. Like in the emEndIteration function above,
  // we do not call this for all objects. This call
  // ensure that all objects internal EM data structures
  // have been allocated, so that they can be accumulated to.

  // components
  for (unsigned i=0;i<components.size();i++)
    components[i]->emStartIteration();

  // do the basic discrete objects
  for (unsigned i=0;i<dPmfs.size();i++)
    dPmfs[i]->emStartIteration();
  for (unsigned i=0;i<sPmfs.size();i++)
    sPmfs[i]->emStartIteration();

  // for discrete RVs
  for (unsigned i=0;i<mdCpts.size();i++)
    mdCpts[i]->emStartIteration();
  for (unsigned i=0;i<msCpts.size();i++)
    msCpts[i]->emStartIteration();
  for (unsigned i=0;i<mtCpts.size();i++)
    mtCpts[i]->emStartIteration();

  // for continuous RVs
  for (unsigned i=0;i<mixtures.size();i++)
    mixtures[i]->emStartIteration();
  /////////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////////

  /////////////////////////////////////////////////////////////
  // Now, we actually load the accumulators.

  // first do the basic objects
  for (unsigned i=0;i<dPmfs.size();i++)
    dPmfs[i]->emLoadAccumulators(ifile);
  for (unsigned i=0;i<sPmfs.size();i++)
    sPmfs[i]->emLoadAccumulators(ifile);
  for (unsigned i=0;i<means.size();i++)
    means[i]->emLoadAccumulators(ifile);
  for (unsigned i=0;i<covars.size();i++)
    covars[i]->emLoadAccumulators(ifile);
  for (unsigned i=0;i<dLinkMats.size();i++)
    dLinkMats[i]->emLoadAccumulators(ifile);
  for (unsigned i=0;i<realMats.size();i++)
    realMats[i]->emLoadAccumulators(ifile);
#if DOUBLEMATS_EVERYWHERE
  for (unsigned i=0;i<doubleMats.size();i++)
    doubleMats[i]->emLoadAccumulators(ifile);
#endif

  // components
  for (unsigned i=0;i<components.size();i++)
    components[i]->emLoadAccumulators(ifile);

  // for discrete RVs
  for (unsigned i=0;i<mdCpts.size();i++)
    mdCpts[i]->emLoadAccumulators(ifile);
  for (unsigned i=0;i<msCpts.size();i++)
    msCpts[i]->emLoadAccumulators(ifile);
  for (unsigned i=0;i<mtCpts.size();i++)
    mtCpts[i]->emLoadAccumulators(ifile);

  // for continuous RVs
  for (unsigned i=0;i<mixtures.size();i++)
    mixtures[i]->emLoadAccumulators(ifile);
#if 0
  // uncomment this when the objects get written.
  for (unsigned i=0;i<gausSwitchingMixtures.size();i++)
    gausSwitchingMixtures[i]->emLoadAccumulators(ifile);
  for (unsigned i=0;i<logitSwitchingMixtures.size();i++)
    logitSwitchingMixtures[i]->emLoadAccumulators(ifile);
  for (unsigned i=0;i<mlpSwitchingMixtures.size();i++)
    mlpSwitchingMixtures[i]->emLoadAccumulators(ifile);
#endif
  /////////////////////////////////////////////////////////////
}

void
GMParms::emAccumulateAccumulators(iDataStreamFile& ifile)
{
  // first do the basic objects
  for (unsigned i=0;i<dPmfs.size();i++)
    dPmfs[i]->emAccumulateAccumulators(ifile);
  for (unsigned i=0;i<sPmfs.size();i++)
    sPmfs[i]->emAccumulateAccumulators(ifile);
  for (unsigned i=0;i<means.size();i++)
    means[i]->emAccumulateAccumulators(ifile);
  for (unsigned i=0;i<covars.size();i++)
    covars[i]->emAccumulateAccumulators(ifile);
  for (unsigned i=0;i<dLinkMats.size();i++)
    dLinkMats[i]->emAccumulateAccumulators(ifile);
  for (unsigned i=0;i<realMats.size();i++)
    realMats[i]->emAccumulateAccumulators(ifile);
#if DOUBLEMATS_EVERYWHERE
  for (unsigned i=0;i<doubleMats.size();i++)
    doubleMats[i]->emAccumulateAccumulators(ifile);
#endif

  // components
  for (unsigned i=0;i<components.size();i++)
    components[i]->emAccumulateAccumulators(ifile);

  // for discrete RVs
  for (unsigned i=0;i<mdCpts.size();i++)
    mdCpts[i]->emAccumulateAccumulators(ifile);
  for (unsigned i=0;i<msCpts.size();i++)
    msCpts[i]->emAccumulateAccumulators(ifile);
  for (unsigned i=0;i<mtCpts.size();i++)
    mtCpts[i]->emAccumulateAccumulators(ifile);

  // for continuous RVs
  for (unsigned i=0;i<mixtures.size();i++)
    mixtures[i]->emAccumulateAccumulators(ifile);
#if 0
  // uncomment this when the objects get written.
  for (unsigned i=0;i<gausSwitchingMixtures.size();i++)
    gausSwitchingMixtures[i]->emAccumulateAccumulators(ifile);
  for (unsigned i=0;i<logitSwitchingMixtures.size();i++)
    logitSwitchingMixtures[i]->emAccumulateAccumulators(ifile);
  for (unsigned i=0;i<mlpSwitchingMixtures.size();i++)
    mlpSwitchingMixtures[i]->emAccumulateAccumulators(ifile);
#endif
}



void
GMParms::emInitAccumulators(bool startEMIteration)
{

  if (startEMIteration) {
    /////////////////////////////////////////////////////////////
    // First, make sure the EM iterations have been
    // started. Like in the emEndIteration function above,
    // we do not call this for all objects. This call
    // ensure that all objects internal EM data structures
    // have been allocated, so that they can be accumulated to.

    // components
    for (unsigned i=0;i<components.size();i++)
      components[i]->emStartIteration();

    // do the basic discrete objects
    for (unsigned i=0;i<dPmfs.size();i++)
      dPmfs[i]->emStartIteration();
    for (unsigned i=0;i<sPmfs.size();i++)
      sPmfs[i]->emStartIteration();

    // for discrete RVs
    for (unsigned i=0;i<mdCpts.size();i++)
      mdCpts[i]->emStartIteration();
    for (unsigned i=0;i<msCpts.size();i++)
      msCpts[i]->emStartIteration();
    for (unsigned i=0;i<mtCpts.size();i++)
      mtCpts[i]->emStartIteration();

    // for continuous RVs
    for (unsigned i=0;i<mixtures.size();i++)
      mixtures[i]->emStartIteration();
    /////////////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////
  }

  /////////////////////////////////////////////////////////////
  // Now, we actually initialize the accumulators.

  // first do the basic objects
  for (unsigned i=0;i<dPmfs.size();i++)
    dPmfs[i]->emInitAccumulators();
  for (unsigned i=0;i<sPmfs.size();i++)
    sPmfs[i]->emInitAccumulators();
  for (unsigned i=0;i<means.size();i++)
    means[i]->emInitAccumulators();
  for (unsigned i=0;i<covars.size();i++)
    covars[i]->emInitAccumulators();
  for (unsigned i=0;i<dLinkMats.size();i++)
    dLinkMats[i]->emInitAccumulators();
  for (unsigned i=0;i<realMats.size();i++)
    realMats[i]->emInitAccumulators();
  for (unsigned i=0;i<doubleMats.size();i++)
    doubleMats[i]->emInitAccumulators();

  // components
  for (unsigned i=0;i<components.size();i++)
    components[i]->emInitAccumulators();

  // for discrete RVs
  for (unsigned i=0;i<mdCpts.size();i++)
    mdCpts[i]->emInitAccumulators();
  for (unsigned i=0;i<msCpts.size();i++)
    msCpts[i]->emInitAccumulators();
  for (unsigned i=0;i<mtCpts.size();i++)
    mtCpts[i]->emInitAccumulators();

  // for continuous RVs
  for (unsigned i=0;i<mixtures.size();i++)
    mixtures[i]->emInitAccumulators();
#if 0
  // uncomment this when the objects get written.
  for (unsigned i=0;i<gausSwitchingMixtures.size();i++)
    gausSwitchingMixtures[i]->emInitAccumulators();
  for (unsigned i=0;i<logitSwitchingMixtures.size();i++)
    logitSwitchingMixtures[i]->emInitAccumulators();
  for (unsigned i=0;i<mlpSwitchingMixtures.size();i++)
    mlpSwitchingMixtures[i]->emInitAccumulators();
#endif
  /////////////////////////////////////////////////////////////
}


void
GMParms::emWriteUnencodedAccumulators(oDataStreamFile& ofile, bool writeLogVals)
{
  // first do the basic objects
  if (EMable::fisherKernelMode) {
    ofile.writeComment("Fisher kernel accumulators\n");
  } else {
    ofile.writeComment("EM accumulators\n");
  }
  ofile.writeComment("%u dPmfs (there may be fewer due to -objsNotToUTilize)\n", dPmfs.size());
  for (unsigned i=0;i<dPmfs.size();i++)
    dPmfs[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
  ofile.writeComment("%u sPmfs (there may be fewer due to -objsNotToUTilize)\n", sPmfs.size());
  for (unsigned i=0;i<sPmfs.size();i++)
    sPmfs[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
  ofile.writeComment("%u means (there may be fewer due to -objsNotToUTilize)\n", means.size());
  for (unsigned i=0;i<means.size();i++)
    means[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
  ofile.writeComment("%u covars (there may be fewer due to -objsNotToUTilize)\n", covars.size());
  for (unsigned i=0;i<covars.size();i++)
    covars[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
  ofile.writeComment("%u dLinkMats (there may be fewer due to -objsNotToUTilize)\n", dLinkMats.size());
  for (unsigned i=0;i<dLinkMats.size();i++)
    dLinkMats[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
  ofile.writeComment("%u realMats (there may be fewer due to -objsNotToUTilize)\n", realMats.size());
  for (unsigned i=0;i<realMats.size();i++)
    realMats[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
#if DOUBLEMATS_EVERYWHERE
  ofile.writeComment("%u doubleMats (there may be fewer due to -objsNotToUTilize)\n", doubleMats.size());
  for (unsigned i=0;i<doubleMats.size();i++)
    doubleMats[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
#endif

   // components
  ofile.writeComment("%u components (there may be fewer due to -objsNotToUTilize)\n", components.size());
  for (unsigned i=0;i<components.size();i++)
    components[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);

   // for discrete RVs
  ofile.writeComment("%u mdCpts (there may be fewer due to -objsNotToUTilize)\n", mdCpts.size());
  for (unsigned i=0;i<mdCpts.size();i++)
    mdCpts[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
  ofile.writeComment("%u msCpts (there may be fewer due to -objsNotToUTilize)\n", msCpts.size());
  for (unsigned i=0;i<msCpts.size();i++)
    msCpts[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
  ofile.writeComment("%u mtCpts (there may be fewer due to -objsNotToUTilize)\n", mtCpts.size());
  for (unsigned i=0;i<mtCpts.size();i++)
    mtCpts[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);

   // for continuous RVs
  ofile.writeComment("%u mixtrues (there may be fewer due to -objsNotToUTilize)\n", mixtures.size());
  for (unsigned i=0;i<mixtures.size();i++)
    mixtures[i]->emWriteUnencodedAccumulators(ofile,writeLogVals);
#if 0
  // uncomment this when the objects get written.
  for (unsigned i=0;i<gausSwitchingMixtures.size();i++)
    gausSwitchingMixtures[i]->emWriteUnencodedAccumulators(ofile,written);
  for (unsigned i=0;i<logitSwitchingMixtures.size();i++)
    logitSwitchingMixtures[i]->emWriteUnencodedAccumulators(ofile,written);
  for (unsigned i=0;i<mlpSwitchingMixtures.size();i++)
    mlpSwitchingMixtures[i]->emWriteUnencodedAccumulators(ofile,written);
#endif

}


/*-
 *-----------------------------------------------------------------------
 * setFirstUtterance
 *      Set the index of the first utterance so that the parameters to
 *      identify which per-utterance decision trees to use. 
 *
 * Preconditions:
 *      none 
 *
 * Postconditions:
 *      firstUtterance is set 
 *
 * Side Effects:
 *      none 
 *
 * Results:
 *      nil
 *-----------------------------------------------------------------------
 */
void
GMParms::setFirstUtterance(
  unsigned first_index 
  )
{
  firstUtterance = first_index;
}





void GMParms::sort_name_collections()
{
  for(vector<NameCollection*>::iterator i=ncls.begin();i!=ncls.end();i++)
    (*i)->sort();
}

void
GMParms::unsort_name_collections()
{
  for(vector<NameCollection*>::iterator i=ncls.begin();i!=ncls.end();i++)
    (*i)->unsort();
}



void 
GMParms::commit_nc_changes()
{
  for(vector<NameCollection*>::iterator i=ncls.begin();i!=ncls.end();i++)
    (*i)->commit_all_searches_and_replacements();
}




void 
dlopenDeterministicMaps(char **dlopenFilenames, unsigned maxFilenames) {
  for (unsigned i=0; i < maxFilenames; i+=1) {
    if (dlopenFilenames[i]) {
#if HAVE_DLFCN_H
      void *handle = dlopen(dlopenFilenames[i], RTLD_NOW|RTLD_LOCAL);
      if (!handle) {
	error("Failed to load '%s': %s", dlopenFilenames[i], dlerror());
      }
      dlerror(); // clear errors

      // POSIX defines dlsym to return a void*. ISO C & C++ do not
      // require that a void* can be cast to a function pointer,
      // and there are architectures (including x86 in certain
      // modes) where it's not possible (sizeof(void*) != 
      // sizeof(function pointer)). So, such architectures/modes
      // cannot be POSIX compliant. The union below avoids the
      // compiler warning about this situation, but it assumes
      // that the void* returned by dlsym can safely be cast to
      // a function pointer.

      typedef void (*register_t)();
      assert(sizeof(void*) == sizeof(register_t)); // necessary, but not sufficient?
      typedef union {
        void *obj_ptr;
        register_t fn_ptr;
      } registerUnion;
      registerUnion registerMappers;
      registerMappers.obj_ptr = dlsym(handle, "registerMappers");

      const char *dlsym_error = dlerror();
      if (dlsym_error) {
	error("Failed to find registerMappers() function in '%s': %s", dlopenFilenames[i], dlsym_error);
      }
      registerMappers.fn_ptr();
      
      
      dlerror(); // clear errors
      vector<CFunctionMapperType> *mapperFunctions = (vector<CFunctionMapperType> *) dlsym(handle, "mapperFunctions");
      dlsym_error = dlerror();
      if (dlsym_error) {
	error("Failed to find mapperFunctions in '%s': %s", dlopenFilenames[i], dlsym_error);
      }
      
      dlerror(); // clear errors
      vector<unsigned>    *mapperNumFeatures = (vector<unsigned> *) dlsym(handle, "mapperNumFeatures");
      if (dlsym_error) {
	error("Failed to find mapperNumFeatures in '%s': %s", dlopenFilenames[i], dlsym_error);
      }
      
      dlerror(); // clear errors
      vector<char const*> *mapperNames = (vector<char const*> *) dlsym(handle, "mapperNames");
      if (dlsym_error) {
	error("Failed to find mapperNames in '%s': %s", dlopenFilenames[i], dlsym_error);
      }
      
      dlerror(); // clear errors
      ObservationSource **externalObsSrc = (ObservationSource **) dlsym(handle, "externalObsSrc");
      if (dlsym_error) {
	error("Failed to find external observation source in '%s': %s", dlopenFilenames[i], dlsym_error);
      }
      *externalObsSrc = globalObservationMatrix; // make GOM available to mapper functions

      for (unsigned j=0; j < mapperFunctions->size(); j+=1) {
	infoMsg(IM::Moderate,"registering %s[%u] from %s\n", (*mapperNames)[j], (*mapperNumFeatures)[j], dlopenFilenames[i]);
	GM_Parms.registerDeterministicCMapper((*mapperNames)[j], (*mapperNumFeatures)[j], (*mapperFunctions)[j]);
      }
#else
      error("dynamic loading of mapping functions not supported");
#endif
    }
  }
}






////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////





#ifdef MAIN

#include "rand.h"
#if 0
#  include "GMTK_ObservationMatrix.h"
#else
#  include "GMTK_FileSource.h"
#endif

RAND rnd(false);
GMParms GM_Parms;
#if 0
ObservationMatrix globalObservationMatrix;
#else
FileSource globalObservationMatrix;
#endif


int
main()
{

  // read in a file of decision trees
  {
  iDataStreamFile isdt("ex.means",false);
  GM_Parms.readMeans(isdt);

  oDataStreamFile os("-");
  GM_Parms.writeMeans(os);  
  }
  {
  iDataStreamFile isdt("ex.covars",false);
  GM_Parms.readCovars(isdt);

  oDataStreamFile os("-");
  GM_Parms.writeCovars(os);  
  }



#if 0
  // read in basic structures
  iDataStreamFile is("dataFiles/test1.gmb",false);

  // write both out again to stdout (i.e., "-")
  oDataStreamFile os("-");
  GM_Parms.writeDTs(os);
#endif

}




#endif
