#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pwd.h>
#include <string.h>
#include <getopt.h>

#include <glib/gkeyfile.h>

#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>
#include <gtk/gtk.h>

#include <gconf/gconf-client.h>
#include <gnome-keyring.h>

#include <gksuui.h>
#include <gksu.h>

#include "defines.h"
#include "../config.h"

#include "util.h"

#define BASE_PATH "/apps/gksu/"

/* GLOBALS */
GConfClient *gconf_client = NULL;
gboolean print_pass = FALSE;
gboolean force_grab = FALSE;
gboolean grab = TRUE;
gboolean sudo_mode = FALSE;
gboolean message_changed = FALSE;
gboolean prompt = FALSE;
gboolean always_ask_pass = FALSE;

struct option long_opts[] = {
    /*
     * { name  has_arg  *flag  val }
     */
    {"help", no_argument, NULL, 'h'},
    {"login", no_argument, NULL, 'l'},
    {"preserv-env", no_argument, NULL, 'k'},
    {"preserve-env", no_argument, NULL, 'k'},
    {"user", required_argument, NULL, 'u'},
    {"print-pass", no_argument, NULL, 'p'},
    {"message", required_argument, NULL, 'm'},
    {"title", required_argument, NULL, 't'},
    {"icon", required_argument, NULL, 'i'},
    {"disable-grab", optional_argument, NULL, 'g'},
    {"ssh-fwd", no_argument, NULL, 's'},
    {"debug", no_argument, NULL, 'd'},
    {"sudo-mode", optional_argument, NULL, 'S'},
    {"prompt", optional_argument, NULL, 'P'},
    {"always-ask-password", optional_argument, NULL, 'a'},
    {"desktop", required_argument, NULL, 'D'},
    {0, 0, 0, 0}
};

/*
 * code 'stolen' from gnome-session's logout.c
 *
 * Written by Owen Taylor <otaylor@redhat.com>
 * Copyright (C) Red Hat
 */
typedef struct {
  GdkScreen    *screen;
  int           monitor;
  GdkRectangle  area;
  int           rowstride;
  GdkWindow    *root_window;
  GdkWindow    *draw_window;
  GdkPixbuf    *start_pb, *end_pb, *frame;
  guchar       *start_p, *end_p, *frame_p;
  GTimeVal      start_time;
  GdkGC        *gc;
} FadeoutData;

FadeoutData *fade_data = NULL;
static GList *fadeout_windows = NULL;

#define FADE_DURATION 500.0

int
gsm_screen_get_width (GdkScreen *screen,
		      int        monitor)
{
	GdkRectangle geometry;

	gdk_screen_get_monitor_geometry (screen, monitor, &geometry);

	return geometry.width;
}

int
gsm_screen_get_height (GdkScreen *screen,
		       int        monitor)
{
	GdkRectangle geometry;

	gdk_screen_get_monitor_geometry (screen, monitor, &geometry);

	return geometry.height;
}

int
gsm_screen_get_x (GdkScreen *screen,
		  int        monitor)
{
	GdkRectangle geometry;

	gdk_screen_get_monitor_geometry (screen, monitor, &geometry);

	return geometry.x;
}

int
gsm_screen_get_y (GdkScreen *screen,
		  int        monitor)
{
	GdkRectangle geometry;

	gdk_screen_get_monitor_geometry (screen, monitor, &geometry);

	return geometry.y;
}

static void
get_current_frame (FadeoutData *fadeout,
		   double    sat)
{
  guchar *sp, *ep, *fp;
  int i, j, width, offset;

  width = fadeout->area.width * 3;
  offset = 0;

  for (i = 0; i < fadeout->area.height; i++)
    {
      sp = fadeout->start_p + offset;
      ep = fadeout->end_p   + offset;
      fp = fadeout->frame_p + offset;

      for (j = 0; j < width; j += 3)
	{
	  guchar r = abs (*(sp++) - ep[0]);
	  guchar g = abs (*(sp++) - ep[1]);
	  guchar b = abs (*(sp++) - ep[2]);

	  *(fp++) = *(ep++) + r * sat;
	  *(fp++) = *(ep++) + g * sat;
	  *(fp++) = *(ep++) + b * sat;
	}

      offset += fadeout->rowstride;
    }
}

static void
darken_pixbuf (GdkPixbuf *pb)
{
  int width, height, rowstride;
  int i, j;
  guchar *p, *pixels;

  width     = gdk_pixbuf_get_width (pb) * 3;
  height    = gdk_pixbuf_get_height (pb);
  rowstride = gdk_pixbuf_get_rowstride (pb);
  pixels    = gdk_pixbuf_get_pixels (pb);

  for (i = 0; i < height; i++)
    {
      p = pixels + (i * rowstride);
      for (j = 0; j < width; j++)
	p [j] >>= 1;
    }
}

static gboolean
fadeout_callback (FadeoutData *fadeout)
{
  GTimeVal current_time;
  double elapsed, percent;

  g_get_current_time (&current_time);
  elapsed = ((((double)current_time.tv_sec - fadeout->start_time.tv_sec) * G_USEC_PER_SEC +
	      (current_time.tv_usec - fadeout->start_time.tv_usec))) / 1000.0;

  if (elapsed < 0)
    {
      g_warning ("System clock seemed to go backwards?");
      elapsed = G_MAXDOUBLE;
    }

  if (elapsed > FADE_DURATION)
    {
      gdk_draw_pixbuf (fadeout->draw_window,
		       fadeout->gc,
		       fadeout->end_pb,
		       0, 0,
		       0, 0,
		       fadeout->area.width,
		       fadeout->area.height,
		       GDK_RGB_DITHER_NONE,
		       0, 0);

      return FALSE;
    }

  percent = elapsed / FADE_DURATION;

  get_current_frame (fadeout, 1.0 - percent);
  gdk_draw_pixbuf (fadeout->draw_window,
		   fadeout->gc,
		   fadeout->frame,
		   0, 0,
		   0, 0,
		   fadeout->area.width,
		   fadeout->area.height,
		   GDK_RGB_DITHER_NONE,
		   0, 0);

  gdk_flush ();

  return TRUE;
}

static void
hide_fadeout_windows (void)
{
  GList *l;

  for (l = fadeout_windows; l; l = l->next)
    {
      gdk_window_hide (GDK_WINDOW (l->data));
      g_object_unref (l->data);
    }

  g_list_free (fadeout_windows);
  fadeout_windows = NULL;
}

static gboolean
fadein_callback (FadeoutData *fadeout)
{
  GTimeVal current_time;
  double elapsed, percent;

  g_get_current_time (&current_time);
  elapsed = ((((double)current_time.tv_sec - fadeout->start_time.tv_sec) * G_USEC_PER_SEC +
	      (current_time.tv_usec - fadeout->start_time.tv_usec))) / 1000.0;

  if (elapsed < 0)
    {
      g_warning ("System clock seemed to go backwards?");
      elapsed = G_MAXDOUBLE;
    }

  if (elapsed > FADE_DURATION)
    {
      gdk_draw_pixbuf (fadeout->draw_window,
		       fadeout->gc,
		       fadeout->end_pb,
		       0, 0,
		       0, 0,
		       fadeout->area.width,
		       fadeout->area.height,
		       GDK_RGB_DITHER_NONE,
		       0, 0);

      g_object_unref (fadeout->gc);
      g_object_unref (fadeout->start_pb);
      g_object_unref (fadeout->end_pb);
      g_object_unref (fadeout->frame);

      g_free (fadeout);

      hide_fadeout_windows ();

      return FALSE;
    }

  percent = elapsed / FADE_DURATION;

  get_current_frame (fadeout, percent);
  gdk_draw_pixbuf (fadeout->draw_window,
		   fadeout->gc,
		   fadeout->frame,
		   0, 0,
		   0, 0,
		   fadeout->area.width,
		   fadeout->area.height,
		   GDK_RGB_DITHER_NONE,
		   0, 0);

  gdk_flush ();

  return TRUE;
}

static void
fadeout_screen (GdkScreen *screen,
		int        monitor)
{
  GdkWindowAttr attr;
  int attr_mask;
  GdkGCValues values;
  FadeoutData *fadeout;

  fadeout = g_new (FadeoutData, 1);

  fadeout->screen = screen;
  fadeout->monitor = monitor;

  fadeout->area.x = gsm_screen_get_x (screen, monitor);
  fadeout->area.y = gsm_screen_get_y (screen, monitor);
  fadeout->area.width = gsm_screen_get_width (screen, monitor);
  fadeout->area.height = gsm_screen_get_height (screen, monitor);

  fadeout->root_window = gdk_screen_get_root_window (screen);
  attr.window_type = GDK_WINDOW_CHILD;
  attr.x = fadeout->area.x;
  attr.y = fadeout->area.y;
  attr.width = fadeout->area.width;
  attr.height = fadeout->area.height;
  attr.wclass = GDK_INPUT_OUTPUT;
  attr.visual = gdk_screen_get_system_visual (fadeout->screen);
  attr.colormap = gdk_screen_get_default_colormap (fadeout->screen);
  attr.override_redirect = TRUE;
  attr_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP | GDK_WA_NOREDIR;

  fadeout->draw_window = gdk_window_new (fadeout->root_window, &attr, attr_mask);
  fadeout_windows = g_list_prepend (fadeout_windows, fadeout->draw_window);

  fadeout->start_pb = gdk_pixbuf_get_from_drawable (NULL,
						    fadeout->root_window,
						    NULL,
						    fadeout->area.x,
						    fadeout->area.y,
						    0, 0,
						    fadeout->area.width,
						    fadeout->area.height);

  fadeout->end_pb = gdk_pixbuf_copy (fadeout->start_pb);
  darken_pixbuf (fadeout->end_pb);

  fadeout->frame = gdk_pixbuf_copy (fadeout->start_pb);
  fadeout->rowstride = gdk_pixbuf_get_rowstride (fadeout->start_pb);

  fadeout->start_p = gdk_pixbuf_get_pixels (fadeout->start_pb);
  fadeout->end_p   = gdk_pixbuf_get_pixels (fadeout->end_pb);
  fadeout->frame_p = gdk_pixbuf_get_pixels (fadeout->frame);

  values.subwindow_mode = GDK_INCLUDE_INFERIORS;

  fadeout->gc = gdk_gc_new_with_values (fadeout->root_window, &values, GDK_GC_SUBWINDOW);

  gdk_window_set_back_pixmap (fadeout->draw_window, NULL, FALSE);
  gdk_window_show (fadeout->draw_window);
  gdk_draw_pixbuf (fadeout->draw_window,
		   fadeout->gc,
		   fadeout->frame,
		   0, 0,
		   0, 0,
		   fadeout->area.width,
		   fadeout->area.height,
		   GDK_RGB_DITHER_NONE,
		   0, 0);

  g_get_current_time (&fadeout->start_time);
  g_idle_add ((GSourceFunc) fadeout_callback, fadeout);

  fade_data = fadeout;
}

/* End of 'stolen' code */


/**
 * help:
 * @cmdname:  name of the command which was called by the user
 * (argv[0])
 *
 * This function is a simple 'usage'-style printing function.
 * It is called if the user calls the program with --help or -h
 */
void
help (gchar *cmdname)
{
  gchar *help_trans;
  const gchar* help_text[] = {
    N_("GKsu version %s\n\n"),
    N_("Usage: %s [-u <user>] [-k] [-l] <command>\n\n"),
    N_("  --always-ask-password, -a\n"
       "    Do not try to check if a password is really\n"
       "    needed for running the command, or if there\n"
       "    are other means of obtaining it: simply ask for it.\n"),
    N_("  --debug, -d\n"
       "    Print information on the screen that might be\n"
       "    useful for diagnosing and/or solving problems.\n"),
    N_("  --disable-grab, -g\n"
       "    Disable the \"locking\" of the keyboard, mouse,\n"
       "    and focus done by the program when asking for\n"
       "    password.\n"),
    N_("  --icon <icon>, -i <icon>\n"
       "    Replace the default window icon with the argument.\n"),
    N_("  --message <message>, -m <message>\n"
       "    Replace the standard message shown to ask for\n"
       "    password for the argument passed to the option.\n"),
    N_("  --print-pass, -p\n"
       "    Ask gksu to print the password to stdout, just\n"
       "    like ssh-askpass. Useful to use in scripts with\n"
       "    programs that accept receiving the password on\n"
       "    stdin.\n"),
    N_("  --prompt, -P\n"
       "    Ask the user if they want to have their keyboard\n"
       "    and mouse grabbed before doing so.\n"),
    N_("  --ssh-fwd, -s\n"
       "    Strip the host part of the $DISPLAY variable, so that\n"
       "    GKSu will work on SSH X11 Forwarding.\n"),
    N_("  --sudo-mode, -S\n"
       "    Make GKSu use sudo instead of su, as if it had been\n"
       "    run as \"gksudo\".\n"),
    N_("  --title <title>, -t <title>\n"
       "    Replace the default title with the argument.\n"),
    N_("  --user <user>, -u <user>\n"
       "    Call <command> as the specified user.\n"),
    N_("  --desktop <file>, -D <file>\n"
       "    Use a .desktop file to get the name of the application\n"
       "    and the icon from.\n"
       "\n"),
    N_("  --preserve-env, -k\n"
       "    Preserve the current environments, does not set $HOME\n"
       "    nor $PATH, for example.\n"),
    N_("  --login, -l\n"
       "    Make this a login shell. Beware this may cause\n"
       "    problems with the Xauthority magic. Run xhost\n"
       "    to allow the target user to open windows on your\n"
       "    display!\n"
       "\n"
       "\n")
  };

  help_trans = g_strconcat(_(help_text[0]), _(help_text[1]),
			   _(help_text[2]), _(help_text[3]),
			   _(help_text[4]), _(help_text[5]),
			   _(help_text[6]), _(help_text[7]),
			   _(help_text[8]), _(help_text[9]),
			   _(help_text[10]), _(help_text[11]),
			   _(help_text[12]), _(help_text[13]),
			   _(help_text[14]), _(help_text[15]),
			   NULL);
  g_print (_(help_trans), PACKAGE_VERSION, cmdname);
  g_free (help_trans);
}

/* copied from gnome-ssh-askpass */
#define GRAB_TRIES	16
#define GRAB_WAIT	250 /* milliseconds */

typedef enum
  {
    FAILED_GRAB_MOUSE,
    FAILED_GRAB_KEYBOARD
  } FailedGrabWhat;

void
report_failed_grab (FailedGrabWhat what)
{
  GtkWidget *dialog;

  dialog = g_object_new (GTK_TYPE_MESSAGE_DIALOG,
			 "message-type", GTK_MESSAGE_WARNING,
			 "buttons", GTK_BUTTONS_CLOSE,
			 NULL);

  switch (what)
    {
    case FAILED_GRAB_MOUSE:
      gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG(dialog),
				     _("<b>Could not grab your mouse.</b>\n\n"
				       "A malicious client may be eavesdropping\n"
				       "on your session or you may have just clicked\n"
				       "a menu or some application just decided to get\n"
				       "focus.\n\n"
				       "Try again."));

      break;
    case FAILED_GRAB_KEYBOARD:
      gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG(dialog),
				     _("Could not grab your keyboard.\n"
				       "A malicious client may be eavesdropping\n"
				       "on your session or you may have just clicked\n"
				       "a menu or some application just decided to get\n"
				       "focus.\n\n"
				       "Try again."));
      break;
    }

  gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);
  gtk_dialog_run (GTK_DIALOG(dialog));
  gtk_widget_destroy (dialog);

  while (gtk_events_pending ())
    gtk_main_iteration ();

}

pid_t test_lock(const char* fname)
{
   int FD = open(fname, 0);
   if(FD < 0) {
      if(errno == ENOENT) {
	 // File does not exist
	 return 0;
      } else {
	 perror("open");
	 return(-1);
      }
   }
   struct flock fl;
   fl.l_type = F_WRLCK;
   fl.l_whence = SEEK_SET;
   fl.l_start = 0;
   fl.l_len = 0;
   if (fcntl(FD, F_GETLK, &fl) < 0) {
      g_critical("fcntl error");
      close(FD);
      return(-1);
   }
   close(FD);
   // lock is available
   if(fl.l_type == F_UNLCK)
      return(0);
   // file is locked by another process
   return (fl.l_pid);
}

int get_lock(const char *File)
{
   int FD = open(File,O_RDWR | O_CREAT | O_TRUNC,0640);
   if (FD < 0)
   {
      // Read only .. cant have locking problems there.
      if (errno == EROFS)
      {
	 g_warning(_("Not using locking for read only lock file %s"),File);
	 return dup(0);       // Need something for the caller to close
      }

      // Feh.. We do this to distinguish the lock vs open case..
      errno = EPERM;
      return -1;
   }
   fcntl(FD,F_SETFD, FD_CLOEXEC);

   // Aquire a write lock
   struct flock fl;
   fl.l_type = F_WRLCK;
   fl.l_whence = SEEK_SET;
   fl.l_start = 0;
   fl.l_len = 0;
   if (fcntl(FD,F_SETLK,&fl) == -1)
   {
      if (errno == ENOLCK)
      {
	 g_warning(_("Not using locking for nfs mounted lock file %s"), File);
	 unlink(File);
	 close(FD);
	 return dup(0);       // Need something for the caller to close
      }

      int Tmp = errno;
      close(FD);
      errno = Tmp;
      return -1;
   }

   return FD;
}

static void
keyring_create_item_cb (GnomeKeyringResult result,
                        guint32 id, gpointer keyring_loop)
{
  g_main_loop_quit (keyring_loop);
}

gboolean
try_gnome_keyring_password (GksuContext *context)
{
  GnomeKeyringAttributeList *attributes;
  GnomeKeyringAttribute attribute;
  GnomeKeyringResult result;
  GList *list;
  GError *error = NULL;
  gboolean keyring_has_password;
  gchar *keyring_password;
  gchar *keyring_command;

  attributes = gnome_keyring_attribute_list_new ();

  attribute.name = g_strdup ("user");
  attribute.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
  attribute.value.string = g_strdup (gksu_context_get_user (context));
  g_array_append_val (attributes, attribute);

  attribute.name = g_strdup ("type");
  attribute.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
  attribute.value.string = g_strdup ("local");
  g_array_append_val (attributes, attribute);

  attribute.name = g_strdup ("creator");
  attribute.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
  attribute.value.string = g_strdup ("gksu");
  g_array_append_val (attributes, attribute);

  list = g_list_alloc();
  keyring_has_password = FALSE;

  result = gnome_keyring_find_items_sync (GNOME_KEYRING_ITEM_GENERIC_SECRET,
					  attributes,
					  &list);
  gnome_keyring_attribute_list_free (attributes);
  if (
      (result == GNOME_KEYRING_RESULT_OK) &&
      (g_list_length(list) == 1)
      )
    {
      GnomeKeyringFound *found = list->data;
      keyring_password = g_strdup(found->secret);

      keyring_command = g_strdup (gksu_context_get_command (context));
      gksu_context_set_command (context, "/bin/echo test > /dev/null");
      gksu_context_set_password (context, keyring_password);
      gksu_context_run (context, &error);

      if (!error)
	keyring_has_password = TRUE;
      gksu_context_set_command (context, keyring_command);
    }

  if (keyring_has_password)
    {
      gchar *password;

      password = g_locale_from_utf8 (keyring_password,
				     strlen (keyring_password),
				     NULL, NULL, NULL);
      gksu_context_set_password (context, password);
      return FALSE; /* we do not need a password */
    }

  return TRUE;
}

int
grab_keyboard_and_mouse (GtkWidget *dialog)
{
  GdkGrabStatus status;
  gint grab_tries = 0;
  gint lock = -1;

  gchar *fname = g_strdup_printf ("%s/.gksu.lock", getenv ("HOME"));
  pid_t pid = test_lock (fname);

  if (pid != 0)
    {
      g_warning ("Lock taken by pid: %i. Exiting.", pid);
      exit (0);
    }

  lock = get_lock(fname);
  if( lock < 0)
    g_warning ("Unable to create lock file.");
  g_free (fname);

  fadeout_screen (gdk_screen_get_default (), 0);
  gtk_widget_show_all (dialog);

  /* reset cursor */
  gdk_window_set_cursor(dialog->window, gdk_cursor_new(GDK_LEFT_PTR));

  for(;;)
    {
      status = gdk_pointer_grab ((GTK_WIDGET(dialog))->window, TRUE, 0, NULL,
				 NULL, GDK_CURRENT_TIME);
      if (status == GDK_GRAB_SUCCESS)
	break;
      usleep (GRAB_WAIT * 1000);
      if (++grab_tries > GRAB_TRIES)
	{
	  gtk_widget_hide (dialog);
	  g_get_current_time (&fade_data->start_time);
	  while (fadein_callback (fade_data) != FALSE);
	  report_failed_grab (FAILED_GRAB_MOUSE);
	  exit (1);
	  break;
	}
    }

  for(;;)
    {
      status = gdk_keyboard_grab ((GTK_WIDGET(dialog))->window,
				  FALSE, GDK_CURRENT_TIME);
      if (status == GDK_GRAB_SUCCESS)
	break;

      usleep(GRAB_WAIT * 1000);

      if (++grab_tries > GRAB_TRIES)
	{
	  gtk_widget_hide (dialog);
	  g_get_current_time (&fade_data->start_time);
	  while (fadein_callback (fade_data) != FALSE);
	  report_failed_grab (FAILED_GRAB_KEYBOARD);
	  exit (1);
	  break;
	}
    }

  /* we "raise" the window because there is a race here for
   * focus-follow-mouse and auto-raise WMs that may put the window
   * in the background and confuse users
   */
  gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);

  while (gtk_events_pending ())
    gtk_main_iteration ();

  return lock;
}

void
ungrab_keyboard_and_mouse (int lock)
{
  /* Ungrab */
  gdk_pointer_ungrab(GDK_CURRENT_TIME);
  gdk_keyboard_ungrab(GDK_CURRENT_TIME);
  gdk_flush();

  g_get_current_time (&fade_data->start_time);
  while (fadein_callback (fade_data) != FALSE);

  close(lock);
}

void
get_configuration_options (GksuContext *context)
{
  grab = !gconf_client_get_bool (gconf_client, BASE_PATH "disable-grab",
				 NULL);
  force_grab = gconf_client_get_bool (gconf_client, BASE_PATH "force-grab",
				      NULL);
  sudo_mode = gconf_client_get_bool (gconf_client, BASE_PATH "sudo-mode",
				     NULL);
  prompt = gconf_client_get_bool (gconf_client, BASE_PATH "prompt",
				  NULL);
  always_ask_pass = gconf_client_get_bool (gconf_client,
					   BASE_PATH "always-ask-password",
					   NULL);
}

gboolean
su_ask_password (GksuContext *context, gchar *prompt,
		 gpointer data, GError **error)
{
  GtkWidget *dialog = GTK_WIDGET(data);

  int retvalue = 0;

  if (prompt)
    gksuui_dialog_set_prompt (GKSUUI_DIALOG(dialog), prompt);

  if (!message_changed)
    {
      gchar *msg;

      if (sudo_mode)
	msg = g_strdup_printf (_("<b>Please enter your password\n"
				 "to run %s as user %s</b>"),
			       gksu_context_get_command (context),
			       gksu_context_get_user (context));
      else
	msg = g_strdup_printf (_("<b>To run the program \"%s\" you need to "
				 "enter the %s password</b>"),
			       gksu_context_get_command (context),
			       gksu_context_get_user (context));

      gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), msg);
      g_free (msg);
    }

  int lock = 0;
  if (grab)
    lock = grab_keyboard_and_mouse (dialog);
  retvalue = gtk_dialog_run (GTK_DIALOG(dialog));
  gtk_widget_hide (dialog);
  if (grab)
    ungrab_keyboard_and_mouse (lock);

  while (gtk_events_pending ())
    gtk_main_iteration ();

  if (retvalue != GTK_RESPONSE_OK)
    return 1;

  {
    gchar *password, *tmp;
    tmp = gksuui_dialog_get_password (GKSUUI_DIALOG(dialog));
    password = g_locale_from_utf8 (tmp, strlen (tmp), NULL, NULL, NULL);
    g_free (tmp);
    gksu_context_set_password (context, password);
  }

  return 0;
}

void
set_sensitivity_cb (GtkWidget *button, gpointer data)
{
  GtkWidget *widget = (GtkWidget*)data;
  gboolean sensitive;

  sensitive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(button));
  gtk_widget_set_sensitive (widget, sensitive);
}

void
cb_toggled_cb (GtkWidget *button, gpointer data)
{
  gchar *key;
  gboolean toggled;
  gchar *key_name;

  g_return_if_fail (data != NULL);

  key_name = (gchar*)data;

  toggled = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(button));

  key = g_strdup_printf (BASE_PATH "%s", key_name);

  if (!strcmp (key_name, "save-keyring"))
    {
      if (toggled)
	gconf_client_set_string (gconf_client, key, "session", NULL);
      else
	gconf_client_set_string (gconf_client, key, "default", NULL);
    }
  else if (!strcmp (key_name, "display-no-pass-info"))
    {
      /* the meaning of the key is the exact opposite of the meaning
	 of the answer - when the check button is checked the key must
	 be off
      */
      gconf_client_set_bool (gconf_client, key, !toggled, NULL);
    }
  else
    gconf_client_set_bool (gconf_client, key, toggled, NULL);

  g_free (key);
}

void
no_pass (GksuContext *context, gpointer data)
{
  GtkWidget *dialog;
  GtkWidget *alignment;
  GtkWidget *check_button;
  gboolean toggled;

  toggled = gconf_client_get_bool (gconf_client, BASE_PATH "display-no-pass-info", NULL);

  /* configuration tells us to not show this message */
  if (!toggled)
    return;

  dialog = gtk_message_dialog_new_with_markup (NULL, 0,
					       GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
					       _("<b>The \"%s\" program was started with "
						 "the privileges of the %s user without "
						 "the need to ask for a password, due to "
						 "your system's authentication mechanism "
						 "setup.</b>"
						 "\n\n"
						 "It is possible that you are being allowed "
						 "to run specific programs as user %s "
						 "without the need for a password, or that "
						 "the password is cached."
						 "\n"
						 "This is not a problem report; it's "
						 "simply a notification to make sure "
						 "you are aware of this."),
					       gksu_context_get_command (context),
					       gksu_context_get_user (context),
					       gksu_context_get_user (context));

  alignment = gtk_alignment_new (0.5, 0.5, 0.6, 1);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), alignment, TRUE, TRUE, 2);

  check_button = gtk_check_button_new_with_mnemonic (_("Do _not display this message again"));
  g_signal_connect (G_OBJECT(check_button), "toggled",
		    G_CALLBACK(cb_toggled_cb), "display-no-pass-info");
  gtk_container_add (GTK_CONTAINER(alignment), check_button);

  gtk_widget_show_all (dialog);
  gtk_dialog_run (GTK_DIALOG(dialog));
  gtk_widget_destroy (GTK_WIDGET(dialog));

  while (gtk_events_pending ())
    gtk_main_iteration ();
}

void
set_dialog_from_desktop (GksuuiDialog *dialog, GksuContext *context,
			 gchar *file_name)
{
  GKeyFile *desktop;
  GError *error = NULL;
  gchar *buffer = NULL;

  desktop = g_key_file_new ();

  g_key_file_load_from_file (desktop, file_name, G_KEY_FILE_NONE, &error);
  if (error)
    {
      gchar *error_msg;

      error_msg = g_strdup_printf ("Could not load desktop file: %s",
				   error->message);
      g_warning (error_msg);
      g_free (error_msg);
      g_error_free (error);
      return;
    }

  buffer = g_key_file_get_locale_string (desktop, "Desktop Entry",
					 "Name", NULL, NULL);
  if (buffer)
    {
      gchar *msg;

      if (sudo_mode)
	msg = g_strdup_printf (_("<b>Authentication required</b>\n\n"
				 "You need your password to run:\n\"%s\"."),
			       buffer);
      else
	msg = g_strdup_printf (_("<b>Authentication required</b>\n\n"
				 "You need to type %s's password to run:\n\"%s\"."),
			       gksu_context_get_user (context), buffer);

      gksuui_dialog_set_message (dialog, msg);

      g_free (buffer);
      g_free (msg);
      message_changed = TRUE;
    }

  buffer = g_key_file_get_locale_string (desktop, "Desktop Entry",
					 "Icon", NULL, NULL);
  if (buffer)
    {
      GdkPixbuf *pixbuf = NULL;
      gboolean is_absolute = FALSE;
      gchar *icon_path;

      is_absolute = !(buffer[0] != '/');

      if (is_absolute)
	{
	  icon_path = buffer;
	  buffer = NULL;
	}
      else
	{
	  icon_path = g_strdup_printf (DATA_DIR"/pixmaps/%s", buffer);
	  g_free (buffer);
	}

      pixbuf = gdk_pixbuf_new_from_file (icon_path, &error);
      if (error)
	{
	  gchar *error_msg;

	  error_msg = g_strdup_printf ("Could not load icon file: %s",
				       error->message);
	  g_warning (error_msg);
	  g_free (error_msg);
	  g_error_free (error);

	  if (!is_absolute)
	    g_free (icon_path);

	  return;
	}

      gksuui_dialog_set_icon (dialog, pixbuf);

      if (!is_absolute)
	g_free (icon_path);
    }
}

int
main (int argc, char **argv)
{
  GtkWidget *dialog;
  GksuContext *context;

  gchar *password = NULL;
  GError *error = NULL;

  gint newargc = 0;
  gchar **newargv = NULL;

  gchar *title = NULL, *message = NULL;
  gchar *desktop_file_name = NULL;
  GdkPixbuf *icon = NULL;

  int retvalue = 0;
  int c = 0;

  setlocale (LC_ALL, "");
  bindtextdomain(PACKAGE_NAME, LOCALEDIR);
  bind_textdomain_codeset (PACKAGE_NAME, "UTF-8");
  textdomain(PACKAGE_NAME);

  /*
   * bad, bad code... adds a second -- right after the first one,
   * because gtk_init will remove one of them...
   */
  {
    /* to check whether a -- was already found when parsing arguments */
    gboolean separator_found = 0;

    for (c = 0; c < argc; c++)
      {
	if (!strcmp ("--", argv[c]) && (!separator_found))
	  {
	    newargv = g_realloc (newargv, sizeof(char*) * (newargc + 2));
	    newargv[newargc] = g_strdup (argv[c]);
	    newargv[newargc + 1] = g_strdup (argv[c]);

	    newargc = newargc + 2;
	    separator_found = TRUE;
	  }
	else
	  {
	    newargv = g_realloc (newargv, sizeof(char*) * (newargc + 1));
	    newargv[newargc] = g_strdup (argv[c]);

	    newargc++;
	  }
      }
  }

  gtk_init (&newargc, &newargv);

  context = gksu_context_new ();
  gconf_client = gconf_client_get_default ();
  get_configuration_options (context);
  while ((c = getopt_long(newargc, newargv, "?hu:lpm:kt:i:gdsS::P::aD:", long_opts, NULL))
	 != EOF)
    {
      switch (c)
	{
	case 0:
	  break;

	case 'h':
	  help (newargv[0]);
	  exit(0);
	  break;
	case '?':
	  help (newargv[0]);
	  exit(0);
	  break;
	case 'u':
	  gksu_context_set_user (context, optarg);
	  break;
	case 'l':
	  gksu_context_set_login_shell (context, TRUE);
	  break;
	case 'p':
	  print_pass = TRUE;
	  break;
	case 't':
	  title = g_strdup (optarg);
	  break;
	case 'm':
	  message = g_strdup (optarg);
	  message_changed = TRUE;
	  break;
	case 'i':
	  icon = gdk_pixbuf_new_from_file (optarg, NULL);
	  break;
	case 'k':
	  gksu_context_set_keep_env (context, TRUE);
	  break;
	case 'g':
	  grab = FALSE;

	  if (optarg != NULL)
	    {
	      if (!strcasecmp (optarg, "yes")); /* ignore, already set */
	      else if (!strcasecmp (optarg, "no"))
		grab = FALSE;
	      else
		{
		  fprintf (stderr, _("Option not accepted for --disable-grab: %s\n"),
			   optarg);
		  return 1;
		}
	    }

	  break;
	case 'd':
	  gksu_context_set_debug (context, TRUE);
	  break;
	case 's':
	  gksu_context_set_ssh_fwd (context, TRUE);
	  break;
	case 'S':
	  sudo_mode = TRUE;

	  if (optarg != NULL)
	    {
	      if (!strcasecmp (optarg, "yes")); /* ignore, already set */
	      else if (!strcasecmp (optarg, "no"))
		sudo_mode = FALSE;
	      else
		{
		  fprintf (stderr, _("Option not accepted for --sudo-mode: %s\n"),
			   optarg);
		  return 1;
		}
	    }

	  break;
	case 'P':
	  prompt = TRUE;

	  if (optarg != NULL)
	    {
	      if (!strcasecmp (optarg, "yes")); /* ignore, already set */
	      else if (!strcasecmp (optarg, "no"))
		prompt = FALSE;
	      else
		{
		  fprintf (stderr, _("Option not accepted for --prompt: %s\n"),
			   optarg);
		  return 1;
		}
	    }

	  break;
	case 'a':
	  always_ask_pass = TRUE;
	  break;
	case 'D':
	  desktop_file_name = g_strdup (optarg);
	  break;
	}
    }

  { /* support gksu_sudo_run */
    gchar *myname = g_path_get_basename (argv[0]);
    if (!strcmp(myname, "gksudo"))
      sudo_mode = TRUE;
    g_free (myname);
  }

  if (force_grab)
    grab = TRUE;

  if (prompt)
    {
      GtkWidget *d;

      d = gtk_message_dialog_new_with_markup (NULL, 0, GTK_MESSAGE_QUESTION,
					      GTK_BUTTONS_YES_NO,
					      _("<b>Would you like your screen to be \"grabbed\"\n"
						"while you enter the password?</b>"
						"\n\n"
						"This means all applications will be paused to avoid\n"
						"the eavesdropping of your password by a a malicious\n"
						"application while you type it."));

      if (gtk_dialog_run (GTK_DIALOG(d)) == GTK_RESPONSE_NO)
	grab = FALSE;
      else
	grab = TRUE;

      gtk_widget_destroy (d);
    }

  if (grab)
    dialog = g_object_new (GKSUUI_TYPE_DIALOG,
			   "type", GTK_WINDOW_POPUP,
			   NULL);
  else
    dialog = gksuui_dialog_new ();

  if (desktop_file_name)
    set_dialog_from_desktop (GKSUUI_DIALOG(dialog), context,
			     desktop_file_name);

  if (title)
    gtk_window_set_title (GTK_WINDOW(dialog), title);
  if (message)
    gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), message);
  if (icon)
    gksuui_dialog_set_icon (GKSUUI_DIALOG(dialog), icon);

  if (print_pass)
    {
      if (!gksuui_dialog_get_message (GKSUUI_DIALOG(dialog)))
	{
	  gchar *msg =
	    g_strdup_printf (_("<b>Please enter %s's password</b>"),
			     gksu_context_get_user (context));

	  gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), msg);
	  g_free (msg);
	}

      int lock = 0;
      if (grab)
	lock = grab_keyboard_and_mouse (dialog);
      retvalue = gtk_dialog_run (GTK_DIALOG(dialog));
      gtk_widget_hide (dialog);
      if (grab)
	ungrab_keyboard_and_mouse (lock);

      /*
	 the user may have pressed cancel or
	 closed the window
      */
      if (retvalue != GTK_RESPONSE_OK)
	  return 2;

      {
	gchar *tmp;
	tmp = gksuui_dialog_get_password (GKSUUI_DIALOG(dialog));
	password = g_locale_from_utf8 (tmp, strlen (tmp), NULL, NULL, NULL);
	g_free (tmp);
      }

      if (password)
	printf ("%s\n", password);
      return 0;
    }

  /* now we can begin to care about a command */
  if (newargc <= optind)
    {
      gk_dialog (GTK_MESSAGE_ERROR, _("Missing command to run."));
      return 1;
    }

  {
    gchar *command = g_strdup (newargv[optind]);
    gchar *tmp = NULL;
    gint i = 0;

    if (!strncmp ("--", command, 2))
      {
	optind = optind + 1;

	if (newargc <= optind)
	  {
	    gk_dialog (GTK_MESSAGE_ERROR, _("Missing command to run."));
	    return 1;
	  }

	g_free (command);
	command = g_strdup (newargv[optind]);
      }

    for (i = optind + 1; i < newargc; i++)
      {
  	tmp = g_strconcat (command, " '", newargv[i], "'", NULL);
	g_free (command);
	command = tmp;
      }
    gksu_context_set_command (context, command);
    g_free (command);
  }

  {
    struct passwd *pwentry;

    pwentry = getpwnam (gksu_context_get_user (context));

    if (!pwentry)
      {
	gk_dialog (GTK_MESSAGE_ERROR, _("User %s does not exist."),
		   gksu_context_get_user (context));
	return 1;
      }

    if (pwentry->pw_uid == geteuid ())
      return g_spawn_command_line_sync (gksu_context_get_command (context),
					NULL, NULL, NULL, NULL);
  }


  if (sudo_mode)
    {
      gksu_context_sudo_run_full (context, su_ask_password, (gpointer)dialog,
				  no_pass, NULL, &error);
      if (error)
	{
	  gk_dialog (GTK_MESSAGE_ERROR,
		     _("<b>Failed to run %s as user %s.</b>\n\n%s"),
		     gksu_context_get_command (context),
		     gksu_context_get_user (context),
		     error->message);
	  return 3;
	}
    }
  else
    {
      if (!always_ask_pass)
	{
	  GtkWidget *vbox;
	  GtkWidget *check_button;
	  GtkWidget *alignment;
	  GtkWidget *radio_vbox;
	  GtkWidget *radio_session, *radio_default;
	  gboolean remember_password;
	  gchar *tmp = NULL;

	  vbox = gtk_vbox_new (2, TRUE);
	  gtk_box_pack_start (GTK_BOX(GKSUUI_DIALOG(dialog)->entry_vbox), vbox, TRUE, TRUE, 0);
	  gtk_widget_show (vbox);

	  check_button = gtk_check_button_new_with_label (_("Remember password"));
	  g_signal_connect (G_OBJECT(check_button), "toggled", G_CALLBACK(cb_toggled_cb), "save-to-keyring");

	  remember_password = gconf_client_get_bool (gconf_client, BASE_PATH"save-to-keyring", NULL);
	  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(check_button), remember_password);
	  gtk_box_pack_start (GTK_BOX(vbox), check_button, TRUE, TRUE, 0);
	  gtk_widget_show (check_button);

	  alignment = gtk_alignment_new (0.5, 0.5, 0.6, 1);
	  gtk_box_pack_start (GTK_BOX(vbox), alignment, TRUE, TRUE, 2);
	  gtk_widget_show (alignment);

	  radio_vbox = gtk_vbox_new (2, TRUE);
	  gtk_container_add (GTK_CONTAINER(alignment), radio_vbox);
	  gtk_widget_set_sensitive (radio_vbox, remember_password);
	  gtk_widget_show (radio_vbox);

	  radio_session = gtk_radio_button_new_with_label (NULL, _("Save for this session"));
	  g_signal_connect (G_OBJECT(radio_session), "toggled", G_CALLBACK(cb_toggled_cb), "save-keyring");
	  gtk_box_pack_start (GTK_BOX(radio_vbox), radio_session, TRUE, TRUE, 0);
	  gtk_widget_show (radio_session);

	  radio_default = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(radio_session), _("Save in the keyring"));
	  gtk_box_pack_start (GTK_BOX(radio_vbox), radio_default, TRUE, TRUE, 0);
	  gtk_widget_show (radio_default);

	  g_signal_connect (G_OBJECT(check_button), "toggled", G_CALLBACK(set_sensitivity_cb), radio_vbox);

	  tmp = gconf_client_get_string (gconf_client, BASE_PATH"save-keyring", NULL);
	  if (tmp && (!strcmp (tmp, "default")))
	    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(radio_default), TRUE);
	  g_free (tmp);

	  if (gksu_context_try_need_password (context))
	    try_gnome_keyring_password (context);
	}

      gksu_context_run_full (context, su_ask_password, (gpointer)dialog,
			     no_pass, NULL, &error);

      if (error)
	{
	  gk_dialog (GTK_MESSAGE_ERROR,
		     _("Failed to run %s as user %s:\n %s"),
		     gksu_context_get_command (context),
		     gksu_context_get_user (context),
		     error->message);
	  return 3;
	}
      else
	{
	  gboolean save_to_keyring;

	  save_to_keyring = gconf_client_get_bool (gconf_client, BASE_PATH"save-to-keyring", NULL);

	  if (context->password && save_to_keyring)
	    {
	      static GMainLoop *keyring_loop = NULL;
	      GnomeKeyringAttributeList *attributes;
	      GnomeKeyringAttribute attribute;

	      gchar *keyring_name;
	      gchar *key_name;

	      attributes = gnome_keyring_attribute_list_new ();

	      attribute.name = g_strdup ("user");
	      attribute.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
	      attribute.value.string = g_strdup (gksu_context_get_user (context));
	      g_array_append_val (attributes, attribute);

	      attribute.name = g_strdup ("type");
	      attribute.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
	      attribute.value.string = g_strdup ("local");
	      g_array_append_val (attributes, attribute);

	      attribute.name = g_strdup ("creator");
	      attribute.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
	      attribute.value.string = g_strdup ("gksu");
	      g_array_append_val (attributes, attribute);

	      key_name = g_strdup_printf ("Local password for user %s",
					  gksu_context_get_user (context));

	      keyring_loop = g_main_loop_new (NULL, FALSE);

	      keyring_name = gconf_client_get_string (gconf_client, BASE_PATH"save-keyring", NULL);
	      if (keyring_name == NULL)
		keyring_name = g_strdup ("session");
	      gnome_keyring_item_create (keyring_name,
					 GNOME_KEYRING_ITEM_GENERIC_SECRET,
					 key_name,
					 attributes,
					 gksu_context_get_password (context),
					 TRUE,
					 keyring_create_item_cb,
					 keyring_loop, NULL);
	      gnome_keyring_attribute_list_free (attributes);
	      g_free (keyring_name);
	      g_main_loop_run (keyring_loop);
	    }
	}
    }

  return 0;
}
