!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_compress
  use cube_types
  use cubetools_structure
  use cubeadm_cubeid_types
  use cubeadm_cubeprod_types
  ! use cubetopology_cuberegion_types
  use cubemain_messaging
  !
  public :: compress
  private
  !
  type :: compress_comm_t
     type(option_t), pointer :: comm
     type(option_t), pointer :: factor
     ! type(cuberegion_comm_t) :: region
     type(cubeid_arg_t), pointer :: incube
     type(cube_prod_t),  pointer :: oucube
   contains
     procedure, public  :: register => cubemain_compress_comm_register
     procedure, private :: parse    => cubemain_compress_comm_parse
     procedure, private :: main     => cubemain_compress_comm_main
  end type compress_comm_t
  type(compress_comm_t) :: compress
  !
  type compress_user_t
     type(cubeid_user_t) :: cubeids
     logical             :: dofactor
     character(len=24)   :: factor
     ! type(cuberegion_user_t) :: region
   contains
     procedure, private :: toprog => cubemain_compress_user_toprog
  end type compress_user_t
  !
  type compress_prog_t
     ! type(cuberegion_prog_t) :: region
     type(cube_t), pointer   :: incube
     integer(kind=4)         :: factor
     type(cube_t), pointer   :: oucube
   contains
     procedure, private :: header => cubemain_compress_prog_header
     procedure, private :: data   => cubemain_compress_prog_data
     procedure, private :: loop   => cubemain_compress_prog_loop
     procedure, private :: act    => cubemain_compress_prog_act
  end type compress_prog_t
  !
contains
  !
  subroutine cubemain_compress_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(compress_user_t) :: user
    character(len=*), parameter :: rname='COMPRESS>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call compress%parse(line,user,error)
    if (error) return
    call compress%main(user,error)
    if (error) continue
  end subroutine cubemain_compress_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_compress_comm_register(comm,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compress_comm_t), intent(inout) :: comm
    logical,                intent(inout) :: error
    !
    type(cubeid_arg_t) :: incube
    type(standard_arg_t) :: stdarg
    type(cube_prod_t) :: oucube
    character(len=*), parameter :: comm_abstract='Compress an image or cube by a factor 2 in spatial directions'
    character(len=*), parameter :: rname='COMPRESS>COMM>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Syntax
    call cubetools_register_command(&
         'COMPRESS','[cubeid]',&
         comm_abstract,&
         strg_id,&
         cubemain_compress_command,&
         comm%comm,&
         error)
    if (error) return
    call incube%register(&
         'INPUT',&
         'Signal cube',&
         strg_id,&
         code_arg_optional,&
         [flag_any],&
         code_read,&
         code_access_imaset,&
         comm%incube,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'FACTOR','N',&
         'Customize the compression factor along axes',&
         strg_id,&
         comm%factor,error)
    if (error) return
    call stdarg%register(&
         'N',&
         'Compression factor',&
         'Default is 2.',&
         code_arg_optional,&
         error)
    if (error) return
    !
    ! call comm%region%register(error)
    ! if (error) return
    !
    ! Products
    call oucube%register(&
         'COMPRESSED',&
         'Output cube',&
         strg_id,&
         [flag_compress],&
         comm%oucube,&
         error,&
         flagmode=keep_prod)
    if (error)  return
  end subroutine cubemain_compress_comm_register
  !
  subroutine cubemain_compress_comm_parse(comm,line,user,error)
    !----------------------------------------------------------------------
    ! COMPRESS [CubeId]
    !----------------------------------------------------------------------
    class(compress_comm_t), intent(in)    :: comm
    character(len=*),       intent(in)    :: line
    type(compress_user_t),  intent(out)   :: user
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='COMPRESS>COMM>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,comm%comm,user%cubeids,error)
    if (error) return
    !
    call comm%factor%present(line,user%dofactor,error)
    if (error) return
    if (user%dofactor) then
       call cubetools_getarg(line,comm%factor,1,user%factor,mandatory,error)
       if (error) return
    else
       user%factor = strg_star
    endif
    !
    ! call comm%region%parse(line,user%region,error)
    ! if (error) return
  end subroutine cubemain_compress_comm_parse
  !
  subroutine cubemain_compress_comm_main(comm,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compress_comm_t), intent(in)    :: comm
    type(compress_user_t),  intent(inout) :: user
    logical,                   intent(inout) :: error
    !
    type(compress_prog_t) :: prog
    character(len=*), parameter :: rname='COMPRESS>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(comm,prog,error)
    if (error) return
    call prog%header(comm,error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_compress_comm_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_compress_user_toprog(user,comm,prog,error)
    use cubetools_unit
    use cubetools_user2prog
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compress_user_t), intent(in)    :: user
    type(compress_comm_t),  intent(in)    :: comm
    type(compress_prog_t),  intent(out)   :: prog
    logical,                   intent(inout) :: error
    !
    integer(kind=4), parameter :: default=2
    character(len=*), parameter :: rname='COMPRESS>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_get_header(comm%incube,user%cubeids,prog%incube,error)
    if (error) return
    !
    call cubetools_user2prog_resolve_star(user%factor,default,prog%factor,error)
    if (error) return
    !
    ! call user%region%toprog(prog%incube,prog%region,error)
    ! if (error) return
    ! User feedback about the interpretation of his command line
    ! call prog%region%list(error)
    ! if (error) return
  end subroutine cubemain_compress_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_compress_prog_header(prog,comm,error)
    use cubetools_axis_types
    use cubetools_header_methods
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compress_prog_t), intent(inout) :: prog
    type(compress_comm_t),  intent(in)    :: comm
    logical,                intent(inout) :: error
    !
    type(axis_t) :: axis
    character(len=*), parameter :: rname='COMPRESS>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(comm%oucube,prog%incube,prog%oucube,error)
    if (error) return
    !
    call cubetools_header_get_axis_head_l(prog%oucube%head,axis,error)
    if (error) return
    call cubemain_compress_axis(axis,prog%factor,error)
    if (error) return
    call cubetools_header_update_axset_l(axis,prog%oucube%head,error)
    if (error) return
    !
    call cubetools_header_get_axis_head_m(prog%oucube%head,axis,error)
    if (error) return
    call cubemain_compress_axis(axis,prog%factor,error)
    if (error) return
    call cubetools_header_update_axset_m(axis,prog%oucube%head,error)
    if (error) return
    !
    ! call prog%region%header(prog%oucube,error)
    ! if (error) return
  end subroutine cubemain_compress_prog_header
  !
  subroutine cubemain_compress_axis(axis,factor,error)
    use cubetools_axis_types
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    type(axis_t),    intent(inout)  :: axis
    integer(kind=4), intent(in)     :: factor
    logical,         intent(inout)  :: error
    !
    integer(kind=4), parameter :: limit=10
    character(len=*), parameter :: rname='COMPRESS>AXIS'
    !
    axis%n = (axis%n-1)/factor+1   ! Take care of non-integer division
    if (axis%n.lt.limit) then
      call cubemain_message(seve%e,rname,  &
        'Output cube would have less than 10 pixels along '//axis%name)
      error = .true.
      return
    endif
    axis%ref = 0.5d0 - (0.5d0-axis%ref)/factor
   !axis%val = unchanged
    axis%inc = factor * axis%inc
  end subroutine cubemain_compress_axis
  !
  subroutine cubemain_compress_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compress_prog_t), intent(inout) :: prog
    logical,                intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='COMPRESS>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error) exit
       !$OMP TASK SHARED(prog,error) FIRSTPRIVATE(iter)
       if (.not.error) &
         call prog%loop(iter,error)
       !$OMP END TASK
    enddo
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_compress_prog_data
  !
  subroutine cubemain_compress_prog_loop(prog,iter,error)
    use cubeadm_taskloop
    use cubeadm_image_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compress_prog_t),    intent(inout) :: prog
    type(cubeadm_iterator_t),  intent(inout) :: iter
    logical,                   intent(inout) :: error
    !
    type(image_t) :: inima
    type(image_t) :: ouima
    character(len=*), parameter :: rname='COMPRESS>PROG>LOOP'
    !
    call inima%associate('inima',prog%incube,iter,error)
    if (error) return
    call ouima%allocate('ouima',prog%oucube,iter,error)
    if (error) return
    !
    do while (iter%iterate_entry(error))
      call prog%act(iter%ie,inima,ouima,error)
      if (error) return
    enddo ! ie
  end subroutine cubemain_compress_prog_loop
  !
  subroutine cubemain_compress_prog_act(prog,ie,inima,ouima,error)
    use cubeadm_image_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compress_prog_t), intent(inout) :: prog
    integer(kind=entr_k),   intent(in)    :: ie
    type(image_t),          intent(inout) :: inima
    type(image_t),          intent(inout) :: ouima
    logical,                intent(inout) :: error
    !
    integer(kind=pixe_k) :: ix,iy,ox,oy,lx,ly,n,n2
    real(kind=8) :: s
    character(len=*), parameter :: rname='COMPRESS>PROG>ACT'
    !
    call inima%get(ie,error)
    if (error) return
    !
    n = prog%factor  ! Just for clarity
    lx = inima%nx/n  ! Integer division (as opposed to ouima%nx value)
    ly = inima%ny/n
    !
    ! Main loop for "full" pixels. For efficiency purpose, do not
    ! compute the non-integer boundary pixels, if any. This avoids "if"
    ! tests or min(n*ox,inima%nx) for ALL pixels while we know these
    ! apply only on boundary and this command is meant to be executed
    ! on huge images.
    n2 = n*n
    do oy=1,ly
      do ox=1,lx
        ! Do not call compute_pixel also for efficiency, loop is
        ! simpler anyway.
        s = 0.d0
        do iy=n*oy-n+1,n*oy
          do ix=n*ox-n+1,n*ox
            s = s + inima%val(ix,iy)
          enddo  ! ix
        enddo  ! iy
        ouima%val(ox,oy) = s/n2
      enddo ! ox
    enddo ! oy
    !
    ! Last column with upper IX truncated to NX
    if (lx.lt.ouima%nx) then
      ox = ouima%nx
      do oy=1,ly
        ouima%val(ox,oy) = compute_pixel(ox,oy,inima%nx,n*oy)
      enddo
    endif
    !
    ! Last row with upper IY truncated to NY
    if (ly.lt.ouima%ny) then
      oy = ouima%ny
      do ox=1,lx
        ouima%val(ox,oy) = compute_pixel(ox,oy,n*ox,inima%ny)
      enddo
    endif
    !
    ! Last pixel with upper IX and IY truncated to NX and NY
    if (lx.lt.ouima%nx .and. ly.lt.ouima%ny) then
      ox = ouima%nx
      oy = ouima%ny
      ouima%val(ox,oy) = compute_pixel(ox,oy,inima%nx,inima%ny)
    endif
    !
    call ouima%put(ie,error)
    if (error) return
    !
  contains
    !
    function compute_pixel(ox,oy,mx,my)
      real(kind=4) :: compute_pixel
      integer(kind=pixe_k), intent(in) :: ox,oy
      integer(kind=pixe_k), intent(in) :: mx,my
      s = 0.d0
      n2 = 0
      do iy=n*oy-n+1,my
        do ix=n*ox-n+1,mx
          n2 = n2+1
          s = s + inima%val(ix,iy)
        enddo  ! ix
      enddo  ! iy
      compute_pixel = s/n2
    end function compute_pixel
  end subroutine cubemain_compress_prog_act
end module cubemain_compress
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
