!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_stack_spectral
  use cubemain_messaging
  use cube_types
  use cubemain_image_real
  use cubemain_windowing
  use cubemain_spaelli_types
  use cubemain_spapos_types
  use cubetools_nan
  !
  public :: cubemain_stack_spectral_domean
  public :: cubemain_stack_spectral_noaperture,cubemain_stack_spectral_aperture
  private
  !
  integer(kind=entr_k), parameter :: one = 1
  !
  type stack_spectral_prog_t
     type(window_array_t),pointer :: wind               ! Window to be spectrally stacked
     type(spapos_prog_t), pointer :: center             ! Aperture center
     type(ellipse_prog_t),pointer :: aperture           ! Aperture
     type(cube_t),pointer         :: incube             ! Input cube
     type(cube_t),pointer         :: oucube             ! Output spectrum
     type(cube_t),pointer         :: mask               ! Mask
     type(cube_t),pointer         :: noise              ! Noise reference
     real(kind=sign_k)            :: factor             ! brightness conversion factor
     logical                      :: mask2d = .false.   ! Is the mask 2d?
     logical                      :: domean = .false.   ! Output is a mean spectrum
     logical                      :: domask             ! Use a mask
     logical                      :: donoise            ! Use weighting by noise
     logical                      :: contaminate        ! NaNs contaminate spectrum
     logical,allocatable          :: include(:)         ! Channel is going to be included?
     integer(kind=pixe_k)         :: nl,nm              ! Number of pixels in l and m directions
     !
     ! Deferred procedures
     procedure(stack_interface_header), pointer :: header => null()
     procedure(stack_interface_loop),   pointer :: loop   => null()
   contains
     generic   :: init            => init_aperture,init_noaperture
     procedure :: init_aperture   => cubemain_stack_spectral_aperture_init
     procedure :: init_noaperture => cubemain_stack_spectral_noaperture_init
     procedure :: allocate        => cubemain_stack_spectral_allocate
     procedure :: set_scale       => cubemain_stack_spectral_set_scale
     procedure :: data            => cubemain_stack_spectral_data
     procedure :: weight_image    => cubemain_stack_spectral_weight_image
     procedure :: image_nomask    => cubemain_stack_spectral_image_nomask
     procedure :: image_mask      => cubemain_stack_spectral_image_mask
  end type stack_spectral_prog_t
  !
contains
  !
  !----------Interfaces--------------------------------------------------
  !
  subroutine stack_interface_header(job,error)
    !----------------------------------------------------------------------
    ! Interface for deffered procedures
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    logical,                      intent(inout) :: error
  end subroutine stack_interface_header
  !
  subroutine stack_interface_loop(job,first,last,error)
    !----------------------------------------------------------------------
    ! Interface for deffered procedures
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    integer(kind=entr_k),         intent(in)    :: first
    integer(kind=entr_k),         intent(in)    :: last
    logical,                      intent(inout) :: error
  end subroutine stack_interface_loop
  !
  !----------Common-utility----------------------------------------------
  !
  subroutine cubemain_stack_spectral_domean(cube,domean,error)
    use cubetools_header_methods
    use cubetools_brightness
    !----------------------------------------------------------------------
    ! If User has not chosen a mean or a sum decide based on cube unit
    !----------------------------------------------------------------------
    type(cube_t), intent(in)    :: cube
    logical,      intent(out)   :: domean
    logical,      intent(inout) :: error
    !
    logical :: valid
    character(len=unit_l) :: unitin
    integer(kind=code_k) :: codein
    character(len=*), parameter :: rname = 'STACK>SPECTRAL>DOMEAN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_header_get_array_unit(cube%head,unitin,error)
    if (error) return
    call cubetools_brightness_valid_brightness_unit(unitin,codein,valid,error)
    if (error) return
    if (valid) then
       select case(codein)
       case (code_unit_jyperbeam,code_unit_jyperpixel,code_unit_mjypersr)
          domean = .false.
       case (code_unit_tmb)
          domean = .true.
       case (code_unit_tas)
          call cubemain_message(seve%e,rname,'Convert it with CUBE\CONVERT first')
          error = .true.
          return
       case default
          call cubemain_message(seve%e,rname,'Unknown brightness unit '//trim(unitin))
          error = .true.
          return
       end select
    else
       call cubemain_message(seve%w,rname,'Default to averaging for unit '//trim(unitin))
       domean = .true.
    endif
  end subroutine cubemain_stack_spectral_domean
  !
  !----------Entry-points------------------------------------------------
  !
  subroutine cubemain_stack_spectral_noaperture(domean,window,incube,domask,&
       mask,donoise,noise,oucube,error)
    !----------------------------------------------------------------------
    ! Does a stack without aperture
    !----------------------------------------------------------------------
    logical,                     intent(in)    :: domean
    type(window_array_t),        intent(in)    :: window
    type(cube_t),pointer,        intent(inout) :: incube
    logical,                     intent(in)    :: domask
    type(cube_t),pointer,        intent(inout) :: mask
    logical,                     intent(in)    :: donoise
    type(cube_t),pointer,        intent(inout) :: noise
    type(cube_t),pointer,        intent(inout) :: oucube
    logical,                     intent(inout) :: error
    !
    type(stack_spectral_prog_t) :: job
    character(len=*), parameter :: rname='STACK>SPECTRAL>NOAPERTURE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call job%init(domean,window,incube,domask,mask,donoise,noise,oucube,error)
    if (error) return
    call job%header(error)
    if (error) return
    call job%data(error)
    if (error) return
  end subroutine cubemain_stack_spectral_noaperture
  !
  subroutine cubemain_stack_spectral_aperture(domean,window,incube,center,aperture,oucube,error)
    !----------------------------------------------------------------------
    ! Does a stack without aperture
    !----------------------------------------------------------------------
    logical,                     intent(in)    :: domean
    type(window_array_t),        intent(in)    :: window
    type(cube_t),pointer,        intent(inout) :: incube
    type(spapos_prog_t), target, intent(in)    :: center            
    type(ellipse_prog_t),target, intent(in)    :: aperture          
    type(cube_t),pointer,        intent(inout) :: oucube
    logical,                     intent(inout) :: error
    !
    type(stack_spectral_prog_t) :: job
    character(len=*), parameter :: rname='STACK>SPECTRAL>APERTURE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call job%init(domean,window,incube,center,aperture,oucube,error)
    if (error) return
    call job%header(error)
    if (error) return
    call job%data(error)
    if (error) return
  end subroutine cubemain_stack_spectral_aperture
  !
  !----------Common-code-base--------------------------------------------
  !
  subroutine cubemain_stack_spectral_allocate(job,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    ! allocates stack_spectral_prog_t
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    logical,                      intent(inout) :: error
    !
    integer(kind=4) :: ier,iw
    character(len=*), parameter :: rname='STACK>SPECTRAL>ALLOCATE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    allocate(job%include(job%incube%head%arr%n%c),stat=ier)
    if (failed_allocate(rname,'Channels to be included',ier,error)) return
    job%include(:) = .false.
    do iw=1,job%wind%n
       job%include(job%wind%val(iw)%o(1):job%wind%val(iw)%o(2)) = .true.
    enddo
    !
    job%nl = job%incube%head%arr%n%l
    job%nm = job%incube%head%arr%n%m
  end subroutine cubemain_stack_spectral_allocate
  !
  subroutine cubemain_stack_spectral_set_scale(job,error)
    use cubetools_header_methods
    use cubetools_brightness
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    logical,                      intent(inout) :: error
    !
    logical :: valid
    integer(kind=code_k) :: incode
    character(len=unit_l) :: inunit,ouunit
    real(kind=sign_k), parameter :: feff=1.0
    real(kind=sign_k), parameter :: beff=1.0
    character(len=*), parameter :: rname='STACK>SPECTRAL>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_header_get_array_unit(job%incube%head,inunit,error)
    if (error) return
    call cubetools_brightness_valid_brightness_unit(inunit,incode,valid,error)
    if (error) return
    if (valid) then
       if (job%domean) then
          call cubetools_header_brightness2brightness(job%incube%head,&
               .not.applyeff,feff,beff,code_unit_tmb,job%factor,error)
          if (error) return
          ouunit = brightness_unit(code_unit_tmb)
       else
          call cubetools_header_brightness2flux(job%incube%head,code_unit_jy,job%factor,error)
          if(error) return
          ouunit = flux_unit(code_unit_jy)
       endif
    else
       job%factor = 1
       ouunit = inunit
    endif
    call cubetools_header_put_array_unit(ouunit,job%oucube%head,error)
    if (error) return
  end subroutine cubemain_stack_spectral_set_scale
  !
  subroutine cubemain_stack_spectral_data(job,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    logical,                      intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='STACK>SPECTRAL>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !
    !$OMP PARALLEL DEFAULT(none) SHARED(job,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(job) FIRSTPRIVATE(iter,error)
       if (.not.error) then
          call job%loop(iter%first,iter%last,error)
       endif
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_stack_spectral_data
  !
  subroutine cubemain_stack_spectral_weight_image(job,weight,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(in)    :: job
    type(image_t),                intent(out)   :: weight
    logical,                      intent(inout) :: error
    !
    integer(kind=pixe_k) :: il,im
    character(len=*), parameter :: rname='STACK>SPECTRAL>WEIGHT>IMAGE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    if (job%donoise) then
       call weight%init(job%noise,error)
       if (error) return
       call weight%get(job%noise,one,error)
       if (error) return
       do il=1,job%nl
          do im=1,job%nm
             weight%z(il,im) = 1/weight%z(il,im)**2
          enddo
       enddo
    else
       call weight%reallocate('weight',job%nl,job%nm,error)
       if (error) return
       do il=1,job%nl
          do im=1,job%nm
             weight%z(il,im) = 1
          enddo
       enddo
    endif
  end subroutine cubemain_stack_spectral_weight_image
  ! 
  subroutine cubemain_stack_spectral_image_nomask(job,inimg,weight,ouimg,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(in)    :: job
    type(image_t),                intent(in)    :: inimg
    type(image_t),                intent(in)    :: weight
    type(image_t),                intent(inout) :: ouimg
    logical,                      intent(inout) :: error
    !
    real(kind=sign_k) :: val,wei
    integer(kind=pixe_k) :: il,im
    character(len=*), parameter :: rname='STACK>SPECTRAL>IMAGE>NOMASK'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    val = 0
    wei = 0
    if (job%contaminate) then
       do il=1,job%nl
          do im=1,job%nm
             val = val + inimg%z(il,im)*weight%z(il,im)
             wei = wei + weight%z(il,im)
          enddo
       enddo
    else
       do il=1,job%nl
          do im=1,job%nm
             if (.not.ieee_is_nan(inimg%z(il,im)).and..not.ieee_is_nan(weight%z(il,im))) then
                val = val + inimg%z(il,im)*weight%z(il,im)
                wei = wei + weight%z(il,im)
             endif
          enddo
       enddo
    endif
    if (wei.gt.0) then
       ouimg%z(one,one) = val/wei*job%factor
    else
       ouimg%z(one,one) = gr4nan
    end if
  end subroutine cubemain_stack_spectral_image_nomask
  !
  subroutine cubemain_stack_spectral_image_mask(job,inimg,weight,mask,ouimg,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(in)    :: job
    type(image_t),                intent(in)    :: inimg
    type(image_t),                intent(in)    :: weight
    type(image_t),                intent(in)    :: mask
    type(image_t),                intent(inout) :: ouimg
    logical,                      intent(inout) :: error
    !
    real(kind=sign_k) :: val,wei
    integer(kind=pixe_k) :: il,im
    character(len=*), parameter :: rname='STACK>SPECTRAL>IMAGE>MASK'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    val = 0
    wei = 0
    if (job%contaminate) then
       do il=1,job%nl
          do im=1,job%nm
             if (.not.ieee_is_nan(mask%z(il,im))) then
                val = val + inimg%z(il,im)*weight%z(il,im)
                wei = wei + weight%z(il,im)*mask%z(il,im)
             endif
          enddo
       enddo
    else
       do il=1,job%nl
          do im=1,job%nm
             if (.not.ieee_is_nan(inimg%z(il,im)).and..not.ieee_is_nan(weight%z(il,im))&
                  .and..not.ieee_is_nan(mask%z(il,im))) then
                val = val + inimg%z(il,im)*weight%z(il,im)
                wei = wei + weight%z(il,im)
             endif
          enddo
       enddo
    endif
    if (wei.gt.0) then
       ouimg%z(one,one) = val/wei*job%factor
    else
       ouimg%z(one,one) = gr4nan
    end if
  end subroutine cubemain_stack_spectral_image_mask
  !
  !----------Aperture-specific-------------------------------------------
  !
  subroutine cubemain_stack_spectral_aperture_init(job,domean,wind,incube,center,aperture,oucube,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    ! Initializes stack_spectral_prog_t
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(out)   :: job
    logical,                      intent(in)    :: domean
    type(window_array_t),target,  intent(in)    :: wind
    type(cube_t),pointer,         intent(in)    :: incube
    type(spapos_prog_t), target,  intent(in)    :: center            
    type(ellipse_prog_t),target,  intent(in)    :: aperture          
    type(cube_t),pointer,         intent(in)    :: oucube
    logical,                      intent(inout) :: error
    !
    character(len=*), parameter :: rname='STACK>SPECTRAL>APERTURE>INIT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    job%domean   =  domean
    job%wind     => wind
    job%incube   => incube
    job%domask   =  .false.
    job%mask2d   =  .false.
    job%mask     => null()
    job%donoise  =  .false.
    job%noise    => null()
    job%oucube   => oucube
    job%center   => center
    job%aperture => aperture
    !
    job%contaminate = .false.
    !
    call job%allocate(error)
    if (error) return
    !
    job%header => cubemain_stack_spectral_aperture_header
    job%loop   => cubemain_stack_spectral_aperture_loop
  end subroutine cubemain_stack_spectral_aperture_init
  !
  subroutine cubemain_stack_spectral_aperture_header(job,error)
    use phys_const
    use gkernel_types, only: projection_t
    use cubetools_header_methods
    use cubetools_axis_types
    use cubetools_beam_types
    use cubetools_spapro_types
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    logical,                      intent(inout) :: error
    !
    type(projection_t) :: gproj
    type(spapro_t) :: spapro
    type(axis_t) :: axis
    type(beam_t) :: beam
    character(len=*), parameter :: rname='STACK>SPECTRAL>APERTURE>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(job%incube,[flag_aperture,flag_spectrum],&
         job%oucube,error,access=code_cube_imaset)
    if (error) return
    !
    call job%set_scale(error)
    if (error) return
    !
     ! VVV The following code aproximates the header to describe a
    ! shape similar to the aperture. L axis is aligned with aperture
    ! major axis, m axis with the minor axis, projection angle is the
    ! aperture position angle. This is only an approximation as the
    ! projection effects are not taken into account.
    !
    ! l axis
    call cubetools_header_get_axis_head_l(job%oucube%head,axis,error)
    if (error) return
    axis%ref = 0.0
    axis%val = 0.0
    if (job%aperture%major.gt.abs(axis%inc)) axis%inc = job%aperture%major
    axis%n = 1_pixe_k
    call cubetools_header_update_axset_l(axis,job%oucube%head,error)
    if (error) return
    !
    ! m axis
    call cubetools_header_get_axis_head_m(job%oucube%head,axis,error)
    if (error) return
    axis%ref = 0.0
    axis%val = 0.0
    if (job%aperture%minor.gt.abs(axis%inc)) axis%inc = job%aperture%minor
    axis%n = 1_pixe_k
    call cubetools_header_update_axset_m(axis,job%oucube%head,error)
    if (error) return
    !
    ! Projection
    call cubetools_header_get_spapro(job%oucube%head,spapro,gproj,error)
    if (error) return
    spapro%l0 = job%center%abso(1)
    spapro%m0 = job%center%abso(2)
    spapro%pa = job%aperture%pang
    call cubetools_header_put_spapro(spapro,job%oucube%head,error)
    if (error) return
    !
    ! Beam
    call cubetools_header_get_spabeam(job%oucube%head,beam,error)
    if (error) return
    ! VVV is this the correct choice for the output beam?
    if (beam%major*beam%minor.lt.job%aperture%major*job%aperture%minor) then
       beam%major = job%aperture%major
       beam%minor = job%aperture%minor
       beam%pang  = job%aperture%pang
       call cubetools_header_update_spabeam(beam,job%oucube%head,error)
       if (error) return
    endif
    !
  end subroutine cubemain_stack_spectral_aperture_header
  !
  subroutine cubemain_stack_spectral_aperture_loop(job,first,last,error)
    use cubeadm_entryloop
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    integer(kind=entr_k),         intent(in)    :: first
    integer(kind=entr_k),         intent(in)    :: last
    logical,                      intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: inimg,ouimg,weight,mask
    character(len=*), parameter :: rname='STACK>SPECTRAL>APERTURE>LOOP'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call inimg%init(job%incube,error)
    if (error) return
    call job%aperture%tomask(job%center,mask,error)
    if (error) return
    call job%weight_image(weight,error)
    if (error) return
    call ouimg%reallocate('stacked',one,one,error)
    if (error) return
    !
    do ie=first,last
       call cubeadm_entryloop_iterate(ie,error)
       if (error)  return
       if (job%include(ie)) then
          call inimg%get(job%incube,ie,error)
          if (error) return
          call job%image_mask(inimg,weight,mask,ouimg,error)
          if (error) return
       else
          ouimg%z(one,one) = gr4nan
       endif
       call ouimg%put(job%oucube,ie,error)
       if (error) return
    enddo
  end subroutine cubemain_stack_spectral_aperture_loop
  !
  !----------No-aperture-code--------------------------------------------
  !
  subroutine cubemain_stack_spectral_noaperture_init(job,domean,wind,incube,domask,&
       mask,donoise,noise,oucube,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    ! Initializes stack_spectral_prog_t
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(out)   :: job
    logical,                      intent(in)    :: domean
    type(window_array_t),target,  intent(in)    :: wind
    type(cube_t),pointer,         intent(in)    :: incube
    logical,                      intent(in)    :: domask
    type(cube_t),pointer,         intent(in)    :: mask
    logical,                      intent(in)    :: donoise
    type(cube_t),pointer,         intent(in)    :: noise
    type(cube_t),pointer,         intent(in)    :: oucube
    logical,                      intent(inout) :: error
    !
    character(len=*), parameter :: rname='STACK>SPECTRAL>NOAPERTURE>INIT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    job%domean  =  domean
    job%wind    => wind
    job%incube  => incube
    job%domask  =  domask
    job%mask    => mask
    job%donoise =  donoise
    job%noise   => noise
    job%oucube  => oucube
    job%center     => null()
    job%aperture   => null()
    !
    job%contaminate = .false.
    !
    call job%allocate(error)
    if (error) return
    !
    job%header => cubemain_stack_spectral_noaperture_header
    if (job%domask) then
       job%mask2d = job%mask%head%arr%n%c.eq.1
       job%loop   => cubemain_stack_spectral_noaperture_loop_mask
    else       
       job%loop   => cubemain_stack_spectral_noaperture_loop_nomask
    endif
  end subroutine cubemain_stack_spectral_noaperture_init
  !
  subroutine cubemain_stack_spectral_noaperture_header(job,error)
    use phys_const
    use cubetools_header_methods
    use cubetools_axis_types
    use cubetools_beam_types
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    logical,                      intent(inout) :: error
    !
    real(kind=coor_k) :: size(2)
    type(axis_t) :: axis
    type(beam_t) :: beam
    !
    character(len=*), parameter :: rname='STACK>SPECTRAL>HEADER>NOAPERTURE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(job%incube,[flag_stack,flag_spectrum],&
         job%oucube,error,access=code_cube_imaset)
    if (error) return
    !
    call job%set_scale(error)
    if (error) return
    !
    ! l axis
    call cubetools_header_get_axis_head_l(job%oucube%head,axis,error)
    if (error) return
    ! *** JP The following code is only a first approximation. We want:
    ! *** JP 1. to set the position to the average of the unprojected sky position
    ! *** JP    inside the mask. For the moment, take the projection center.
    ! *** JP 2. to set the increment to the rms of the unprojected sky distance from
    ! *** JP    the averaged position. For the moment, take the size of the image.
    size(1) = abs(axis%inc*axis%n)
    axis%ref = 0.0
    axis%val = 0.0
    axis%inc = size(1)
    axis%n = 1_pixe_k
    call cubetools_header_update_axset_l(axis,job%oucube%head,error)
    if (error) return
    !
    ! m axis
    call cubetools_header_get_axis_head_m(job%oucube%head,axis,error)
    if (error) return
    ! *** JP The following code is only a first approximation. We want:
    ! *** JP 1. to set the position to the average of the unprojected sky position
    ! *** JP    inside the mask. For the moment, take the projection center.
    ! *** JP 2. to set the increment to the rms of the unprojected sky distance from
    ! *** JP    the averaged position. For the moment, take the size of the image.
    size(2) = abs(axis%inc*axis%n)
    axis%ref = 0.0
    axis%val = 0.0
    axis%inc = size(2)
    axis%n = 1_pixe_k
    call cubetools_header_update_axset_m(axis,job%oucube%head,error)
    if (error) return
    !
    ! Beam
    call cubetools_header_get_spabeam(job%oucube%head,beam,error)
    if (error) return
    beam%major = max(size(1),size(2))
    beam%minor = min(size(1),size(2))
    ! *** JP Unclear whether this is the correct angle convention
    if (size(1).ge.size(2)) then
       beam%pang = 0.0
    else
       beam%pang = pi/2
    endif
    call cubetools_header_update_spabeam(beam,job%oucube%head,error)
    if (error) return
    !
  end subroutine cubemain_stack_spectral_noaperture_header
  !
  subroutine cubemain_stack_spectral_noaperture_loop_nomask(job,first,last,error)
    use cubeadm_entryloop
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    integer(kind=entr_k),         intent(in)    :: first
    integer(kind=entr_k),         intent(in)    :: last
    logical,                      intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: inimg,ouimg,weight
    character(len=*), parameter :: rname='STACK>SPECTRAL>NOAPERTURE>LOOP>NOMASK'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call inimg%init(job%incube,error)
    if (error) return
    call job%weight_image(weight,error)
    if (error) return
    call ouimg%reallocate('stacked',one,one,error)
    if (error) return
    !
    do ie=first,last
       call cubeadm_entryloop_iterate(ie,error)
       if (error)  return
       if (job%include(ie)) then
          call inimg%get(job%incube,ie,error)
          if (error) return
          call job%image_nomask(inimg,weight,ouimg,error)
          if (error) return
       else
          ouimg%z(one,one) = gr4nan
       endif
       call ouimg%put(job%oucube,ie,error)
       if (error) return
    enddo
  end subroutine cubemain_stack_spectral_noaperture_loop_nomask
  !
  subroutine cubemain_stack_spectral_noaperture_loop_mask(job,first,last,error)
    use cubeadm_entryloop
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(stack_spectral_prog_t), intent(inout) :: job
    integer(kind=entr_k),         intent(in)    :: first
    integer(kind=entr_k),         intent(in)    :: last
    logical,                      intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: inimg,ouimg,weight,mask
    character(len=*), parameter :: rname='STACK>SPECTRAL>NOAPERTURE>LOOP>MASK'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call inimg%init(job%incube,error)
    if (error) return
    call mask%init(job%mask,error)
    if (error) return
    call job%weight_image(weight,error)
    if (error) return
    call ouimg%reallocate('stacked',one,one,error)
    if (error) return
    !
    if (job%mask2d) then
      call mask%get(job%mask,one,error)
      if (error) return 
    endif
    !
    do ie=first,last
       call cubeadm_entryloop_iterate(ie,error)
       if (error)  return
       if (job%include(ie)) then
          if (.not.job%mask2d) then
             call mask%get(job%mask,ie,error)
             if (error) return 
          endif
          call inimg%get(job%incube,ie,error)
          if (error) return
          call job%image_mask(inimg,weight,mask,ouimg,error)
          if (error) return
       else
          ouimg%z(one,one) = gr4nan
       endif
       call ouimg%put(job%oucube,ie,error)
       if (error) return
    enddo
  end subroutine cubemain_stack_spectral_noaperture_loop_mask
  !
end module cubemain_stack_spectral
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
