!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_convert
  use cubetools_structure
  use cube_types
  use cubeadm_cubeid_types
  use cubemain_messaging
  !
  public :: convert
  public :: cubemain_convert_command
  private
  !
  type :: convert_comm_t
     type(option_t),      pointer :: comm
     type(option_t),      pointer :: unit
     type(option_t),      pointer :: efficiencies
     type(option_t),      pointer :: factor
   contains
     procedure, public  :: register           => cubemain_convert_register
     procedure, private :: parse              => cubemain_convert_parse
     procedure, private :: parse_unit         => cubemain_convert_parse_unit
     procedure, private :: parse_factor       => cubemain_convert_parse_factor
     procedure, private :: parse_efficiencies => cubemain_convert_parse_efficiencies
     procedure, private :: main               => cubemain_convert_main
  end type convert_comm_t
  type(convert_comm_t) :: convert
  !
  integer(kind=4), parameter :: icube = 1
  type convert_user_t
     type(cubeid_user_t)   :: cubeids
     character(len=argu_l) :: unit    ! Name of output unit
     real(kind=sign_k)     :: beff    ! Beam Efficiency
     real(kind=sign_k)     :: feff    ! Forward Efficiency
     real(kind=sign_k)     :: factor  ! User defined convertion factor
     logical               :: dofactor
     logical               :: dounit
     logical               :: doapplyeff
   contains
     procedure, private :: toprog => cubemain_convert_user_toprog
  end type convert_user_t
  type convert_prog_t
     type(cube_t), pointer :: cube       ! Input cube
     type(cube_t), pointer :: converted  ! Output cube
     character(len=unit_l) :: unit_name  ! Name of output unit
     integer(kind=code_k)  :: unit_code  ! Code of output unit
     real(kind=sign_k)     :: factor     ! Convertion factor
     integer(kind=pixe_k)  :: nl,nm      ! Image size
     integer(kind=chan_k)  :: nc         ! Spectrum size
   contains
     procedure, private :: header             => cubemain_convert_prog_header
     procedure, private :: data               => cubemain_convert_prog_data
     procedure, private :: data_image_loop    => cubemain_convert_prog_data_image_loop
     procedure, private :: data_spectrum_loop => cubemain_convert_prog_data_spectrum_loop
  end type convert_prog_t
  !
contains
  !
  subroutine cubemain_convert_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(convert_user_t) :: user
    character(len=*), parameter :: rname='CONVERT>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call convert%parse(line,user,error)
    if (error) return
    call convert%main(user,error)
    if (error) continue
  end subroutine cubemain_convert_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_convert_register(convert,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(convert_comm_t), intent(inout) :: convert
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract = &
         'Convert cube between units'
    character(len=*), parameter :: comm_help = &
         'Convertion can be done automatically between units of the&
         & same kind, e.g. K (Tmb) to Jy/beam, or km/s to m/s. When&
         & trying to convert between units of different kinds or even&
         & unknown units the option /FACTOR allows the user to&
         & explicitly define a multiplicative convertion factor&
         & between input and output units.'
    character(len=*), parameter :: rname='CONVERT>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'CONVERT','[cube]',&
         comm_abstract,&
         comm_help,&
         cubemain_convert_command,&
         convert%comm,error)
    if (error) return
    call cubearg%register( &
         'CUBE', &
         'Signal cube',  &
         strg_id,&
         code_arg_optional,  &
         [flag_cube], &
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'UNIT','name',&
         'Define converted cube unit',&
         strg_id,&
         convert%unit,error)
    if (error) return
    call stdarg%register( &
         'name',  &
         'Unit for converted cube', &
         strg_id,&
         code_arg_mandatory, &
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'FACTOR','fac',&
         'Explicitly define a unit convertion factor',&
         'This option gives the user the ability to do convertions&
         & that are not supported. The factor given here is used as a&
         & multiplicative factor over the whole cube and the new unit&
         & is taken to be the unit given with option /UNIT. WARNING:&
         & No checks are performed on the given convertion factor',&
         convert%factor,error)
    if (error) return
    call stdarg%register( &
         'fac',  &
         'Unit convertion factor', &
         strg_id,&
         code_arg_mandatory, &
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'EFFICIENCIES','Beff [Feff]',&
         'Convert to and From K (Ta*)',&
         'Specify the Beam and Forward efficiencies to use when&
         & converting to and from K (Ta*). If not given explicitly,&
         & Feff is assumed to be 0.85',&
         convert%efficiencies,error)
    if (error) return
    call stdarg%register( &
         'Beff',  &
         'Beam efficiency', &
         strg_id,&
         code_arg_mandatory, &
         error)
    if (error) return
    call stdarg%register( &
         'Feff',  &
         'Forward efficiency', &
         'If not given assumed to be 0.95',&
         code_arg_optional, &
         error)
    if (error) return
  end subroutine cubemain_convert_register
  !
  subroutine cubemain_convert_parse(convert,line,user,error)
    use cubetools_parse
    !----------------------------------------------------------------------
    ! CONVERT cubname
    ! /UNIT name
    ! /EFFICIENCIES Beff [Feff]
    ! /FACTOR fac
    !----------------------------------------------------------------------
    class(convert_comm_t), intent(in)    :: convert
    character(len=*),      intent(in)    :: line
    type(convert_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='CONVERT>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,convert%comm,user%cubeids,error)
    if (error) return
    call convert%parse_unit(line,user,error)
    if (error) return
    call convert%parse_factor(line,user,error)
    if (error) return
    call convert%parse_efficiencies(line,user,error)
    if (error) return
    if (.not.(user%dounit.or.user%doapplyeff)) then
       call cubemain_message(seve%e,rname,"At least one option must be given")
       error = .true.
       return
    endif
  end subroutine cubemain_convert_parse
  !
  subroutine cubemain_convert_parse_unit(convert,line,user,error)
    !----------------------------------------------------------------------
    ! /UNIT name
    !----------------------------------------------------------------------
    class(convert_comm_t), intent(in)    :: convert
    character(len=*),      intent(in)    :: line
    type(convert_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='CONVERT>PARSE>UNIT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call convert%unit%present(line,user%dounit,error)
    if (error) return
    if (user%dounit) then
       call cubetools_getarg(line,convert%unit,1,user%unit,mandatory,error)
       if(error) return
    endif
  end subroutine cubemain_convert_parse_unit
  !
  subroutine cubemain_convert_parse_factor(convert,line,user,error)
    !----------------------------------------------------------------------
    ! /FACTOR fac
    !----------------------------------------------------------------------
    class(convert_comm_t), intent(in)    :: convert
    character(len=*),      intent(in)    :: line
    type(convert_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='CONVERT>PARSE>FACTOR'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call convert%factor%present(line,user%dofactor,error)
    if (error) return
    if (user%dofactor) then
       call cubetools_getarg(line,convert%factor,1,user%factor,mandatory,error)
       if(error) return
    endif
  end subroutine cubemain_convert_parse_factor
  !
  subroutine cubemain_convert_parse_efficiencies(convert,line,user,error)
    !----------------------------------------------------------------------
    ! /EFFICIENCIES name
    !----------------------------------------------------------------------
    class(convert_comm_t), intent(in)    :: convert
    character(len=*),      intent(in)    :: line
    type(convert_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='CONVERT>PARSE>EFFICIENCIES'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call convert%efficiencies%present(line,user%doapplyeff,error)
    if (error) return
    if (user%doapplyeff) then
       call cubetools_getarg(line,convert%efficiencies,1,user%beff,mandatory,error)
       if(error) return
       user%feff = 0.95
       call cubetools_getarg(line,convert%efficiencies,2,user%feff,.not.mandatory,error)
       if(error) return
    endif
  end subroutine cubemain_convert_parse_efficiencies
  !
  subroutine cubemain_convert_main(convert,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(convert_comm_t), intent(in)    :: convert
    type(convert_user_t),  intent(in)    :: user
    logical,               intent(inout) :: error
    !
    type(convert_prog_t) :: prog
    character(len=*), parameter :: rname='CONVERT>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_convert_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_convert_user_toprog(user,prog,error)
    use cubetools_header_methods
    use cubetools_brightness
    use cubetools_disambiguate
    use cubetools_unit
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(convert_user_t), intent(in)    :: user
    type(convert_prog_t),  intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    character(len=unit_l) :: unitin,unitou
    type(unit_user_t) :: inunit,ouunit
    character(len=mess_l) :: mess
    integer(kind=code_k) :: inunit_kind,ouunit_kind
    logical :: convertion_error
    character(len=*), parameter :: rname='CONVERT>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(convert%comm,icube,user%cubeids,code_access_imaset_or_speset,  &
      code_read,prog%cube,error)
    if (error) return
    !
    prog%nl = prog%cube%head%arr%n%l
    prog%nm = prog%cube%head%arr%n%m
    prog%nc = prog%cube%head%arr%n%c
    !
    call cubetools_header_get_array_unit(prog%cube%head,unitin,error)
    if (error) return
    call cubetools_unit_get_kind(unitin,inunit_kind,error)
    if (error) return
    call cubetools_unit_get_kind(user%unit,ouunit_kind,error)
    if (error) return
    !
    if (user%dofactor) then ! User has given a factor, simple case
       prog%factor = user%factor
       unitou = user%unit
       if (ouunit_kind.eq.code_unit_unk) &
            call cubemain_message(seve%w,rname,'Unsupported unit '//trim(user%unit))
    else ! Automatic convertion factor 
       convertion_error = .false.
       !
       select case(inunit_kind)
       case(code_unit_unk)
          convertion_error = .true.
       case(code_unit_brig)
          if (ouunit_kind.eq.code_unit_brig) then
             call cubetools_disambiguate_strict(user%unit,brightness_unit,prog%unit_code,unitou,error)
             if (error) return
             unitou = brightness_unit(prog%unit_code)
             call cubetools_header_brightness2brightness(prog%cube%head,&
                  user%doapplyeff,user%feff,user%beff,&
                  prog%unit_code,prog%factor,error)
             if (error) return
          else
             convertion_error = .true.
          endif
       case(code_unit_flux)
          if (ouunit_kind.eq.code_unit_flux) then
             call cubetools_disambiguate_strict(user%unit,flux_unit,prog%unit_code,unitou,error)
             if (error) return
             unitou = flux_unit(prog%unit_code)
             call cubetools_header_flux2flux(prog%cube%head,prog%unit_code,prog%factor,error)
             if (error) return
          else
             convertion_error = .true.
          endif
       case default
          select case(inunit_kind)
          case(code_unit_freq,code_unit_velo,code_unit_wave&
               &,code_unit_pixe,code_unit_chan,code_unit_dist)
             ! Non equivalent units, convertion only possible if kind is
             ! the same
             convertion_error = inunit_kind.ne.ouunit_kind
          case (code_unit_fov,code_unit_beam)
             ! All unit kinds here are angle units and can hence be
             ! treated together
             convertion_error = .not.(ouunit_kind.eq.code_unit_fov.or.ouunit_kind.eq.code_unit_beam)
          case default
             call cubemain_message(seve%e,rname,'Programming error')
             error = .true.
             return
          end select
          if (.not.convertion_error) then
             call cubetools_unit_get(unitin,inunit_kind,inunit,error)
             if (error) return
             call cubetools_unit_get(user%unit,ouunit_kind,ouunit,error)
             if (error) return
             prog%factor = inunit%prog_per_user*ouunit%user_per_prog
             unitou = ouunit%name
          else
             ! Do nothing
          endif
       end select
       !
       if (convertion_error) then
          write(mess,'(8a)') 'Cannot convert from ',trim(unitin),' of kind ',trim(unitkinds(inunit_kind)),&
               ' to ',trim(user%unit),' of kind ',trim(unitkinds(ouunit_kind))
          call cubemain_message(seve%e,rname,mess)
          error = .true.
          return
       endif
    endif
    !
    ! User feedback
    write(mess,'(5a,1pg14.7)') 'Converting from ',trim(unitin),' to ',trim(unitou),&
         ', factor: ',prog%factor
    call cubemain_message(seve%i,rname,mess)
    prog%unit_name = unitou
  end subroutine cubemain_convert_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_convert_prog_header(prog,error)
    use cubetools_header_methods
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(convert_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='CONVERT>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(prog%cube,[flag_convert,flag_cube],prog%converted,error)
    if (error) return
    call cubetools_header_put_array_unit(prog%unit_name,prog%converted%head,error)
    if (error) return
  end subroutine cubemain_convert_prog_header
  !
  subroutine cubemain_convert_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(convert_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='CONVERT>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error) then
          select case (prog%cube%order())
          case (code_cube_imaset)
             call prog%data_image_loop(iter%first,iter%last,error)
          case (code_cube_speset)
             call prog%data_spectrum_loop(iter%first,iter%last,error)
          case default
             call cubemain_message(seve%e,rname,'Unsupported file access')
             error = .true.
          end select
       endif
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_convert_prog_data
  !
  subroutine cubemain_convert_prog_data_image_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(convert_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    integer(kind=pixe_k) :: il,im
    type(image_t) :: inimg, ouimg
    character(len=*), parameter :: rname='CONVERT>PROG>DATA>IMAGE>LOOP'
    !
    call inimg%init(prog%cube,error)
    if (error) return
    call ouimg%reallocate('ouimg',prog%nl,prog%nm,error)
    if (error) return
    !
    do ie = first,last
       call cubeadm_entryloop_iterate(ie,error)
       if(error) return
       call inimg%get(prog%cube,ie,error)
       if (error) return
       do im=1,prog%nm
          do il=1,prog%nl
             ouimg%z(il,im) = inimg%z(il,im)*prog%factor
          end do
       end do
       call ouimg%put(prog%converted,ie,error)
       if (error) return
    enddo ! ie
  end subroutine cubemain_convert_prog_data_image_loop
  !
  subroutine cubemain_convert_prog_data_spectrum_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(convert_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    integer(kind=chan_k) :: ic
    type(spectrum_t) :: inspec,ouspec
    character(len=*), parameter :: rname='CONVERT>PROG>DATA>SPECTRUM>LOOP'
    !
    call inspec%reassociate_and_init(prog%cube,error)
    if (error) return
    call ouspec%reallocate('out',prog%nc,error)
    if (error) return
    !
    do ie = first,last
       call cubeadm_entryloop_iterate(ie,error)
       if(error) return
       call inspec%get(prog%cube,ie,error)
       if (error) return
       do ic=1,prog%nc
          ouspec%t(ic) = inspec%t(ic)*prog%factor
       end do
       call ouspec%put(prog%converted,ie,error)
       if (error) return
    enddo ! ie
  end subroutine cubemain_convert_prog_data_spectrum_loop
end module cubemain_convert
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
