!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubeadm_import
  use cubetools_structure
  use cubedag_parameters
  use cubedag_types
  use cubedag_node
  use cubedag_dag
  use cubedag_link
  use cubeadm_messaging
  use cubeadm_directory_type
  !
  type(cubedag_link_t) :: im  ! List of imported cubes
  !
  public :: cubeadm_import_command,cubeadm_import_register,cubeadm_head_to_node
  private
  !
  type :: import_comm_t
     type(option_t), pointer :: import
     type(option_t), pointer :: mapping
     type(option_t), pointer :: class
  end type import_comm_t
  type(import_comm_t) :: comm
  !
  type import_user_t
     logical :: domapping = .false.
     logical :: doclass   = .false.
  end type import_user_t
  type(import_user_t) :: user
  !
contains
  !
  subroutine cubeadm_import_register(error)
    !---------------------------------------------------------------------
    ! 
    !---------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract = &
         'Import files into the DAG'
    character(len=*), parameter :: comm_help = &
         'With a file pattern as argument (e.g. raw/*{.lmv|.fits}),&
         & import files using this pattern (pattern is assumed&
         & relative from current directory, if relevant). With a&
         & directory name as argument, import all files (*.*, only&
         & Gildas Data Format or FITS format files are imported) from&
         & this directory (no recursion). With a DAG file name as&
         & argument (*.dag), reload it from disk.'&
         & //strg_cr//strg_cr//&
         & 'Without argument, import files from the RAW directory.'
    character(len=*), parameter :: rname='IMPORT>REGISTER'
    !
    call cubeadm_message(admseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'IMPORT','[Directory|DagFile|FilePattern]',&
         comm_abstract,&
         comm_help,&
         cubeadm_import_command,&
         comm%import,error)
    if (error) return
    call stdarg%register( &
         'Target',  &
         'Import target, a DagFile, a directory or a File Pattern', &
         strg_id,&
         code_arg_optional, &
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'MAPPING','',&
         'Import MAPPING files with flags based on extension',&
         strg_id,&
         comm%mapping,error)
    if (error) return
    !
    call cubetools_register_option(&
         'CLASS','',&
         'Import CLASS files with flags based on extension',&
         strg_id,&
         comm%class,error)
    if (error) return
  end subroutine cubeadm_import_register
  !
  subroutine cubeadm_import_command(line,error)
    use cubedag_find
    use cubedag_history
    use cubeadm_snapshot
    use cubeadm_setup
    use gkernel_interfaces
    !---------------------------------------------------------------------
    ! Support routine for command:
    !   IMPORT
    !   IMPORT DagFile
    !   IMPORT Directory
    !   IMPORT Pattern
    !---------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    character(len=file_l) :: argum
    integer(kind=4) :: nc
    type(cubedag_find_t) :: criter
    logical :: do_cx
    type(cubedag_link_t) :: void
    integer(kind=entr_k) :: hid
    character(len=*), parameter :: rname='IMPORT>COMMAND'
    !
    if (comm%import%getnarg().ge.1) then
      call cubetools_getarg(line,comm%import,1,argum,.not.mandatory,error)
      if (error)  return
    else
      argum = dir%raw
    endif
    !
    call comm%mapping%present(line,user%domapping,error)
    if (error)  return
    call comm%class%present(line,user%doclass,error)
    if (error)  return
    !
    if (user%domapping.and.user%doclass) then
       call cubeadm_message(seve%e,rname,'Options /MAPPING and /CLASS conflict with each other')
       error = .true.
       return
    end if
    !
    nc = len_trim(argum)
    if (nc.ge.4 .and. argum(nc-3:nc).eq.'.dag') then
       do_cx = .false.  ! Implicitly done by cubeadm_snapshot_reimport
       call cubeadm_snapshot_reimport(argum,cubeadm_snapshot_histname(),.false.,error)
    elseif (gag_isdir(argum(1:nc)).eq.0) then
       do_cx = cubset%index%default.eq.code_index_current
       call cubeadm_import_directory(argum,nc,error)
    else
       do_cx = cubset%index%default.eq.code_index_current
       call cubeadm_import_pattern(argum,error)
    endif
    if (error)  return
    !
    if (do_cx) then
      ! Implicitly FIND everything in current index
      call cubedag_find_ix2cx(criter,error)
      if (error)  return
    endif
    !
    ! Insert in history
    call cubedag_history_add('IMPORT',line,void,im,hid,error)
    if (error)  return
    call cubedag_node_history(im,hid,error)
    if (error)  return
    !
    call cubeadm_imported_reset()
    !
  end subroutine cubeadm_import_command
  !
  subroutine cubeadm_import_directory(dirname,nc,error)
    !---------------------------------------------------------------------
    ! Import *.* files from a directory into the DAG
    !---------------------------------------------------------------------
    character(len=*), intent(in)    :: dirname
    integer(kind=4),  intent(in)    :: nc
    logical,          intent(inout) :: error
    !
    character(len=*), parameter :: rname='IMPORT>DIRECTORY'
    !
    if (dirname(nc:nc).eq.'/') then
      call cubeadm_import_pattern(dirname(1:nc)//'*.*',error)
    else
      call cubeadm_import_pattern(dirname(1:nc)//'/*.*',error)
    endif
    if (error)  return
  end subroutine cubeadm_import_directory
  !
  recursive subroutine cubeadm_import_pattern(pattern,error)
    use gkernel_interfaces
    !---------------------------------------------------------------------
    ! Import files given a pattern into the DAG
    !---------------------------------------------------------------------
    character(len=*), intent(in)    :: pattern
    logical,          intent(inout) :: error
    !
    integer(kind=4) :: nfile,ifile,filekind,nimported,nskipped,nc,myseve
    logical :: skipped
    character(len=file_l), allocatable :: filelist(:)
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='IMPORT>PATTERN'
    !
    ! Get the file names
    call gag_directory('.',trim(pattern),filelist,nfile,error)
    if (error)  return
    !
    ! Import the GDF and FITS files in the DAG
    nimported = 0
    nskipped = 0
    do ifile=1,nfile
      nc = len_trim(filelist(ifile))
      if (gag_isdir(filelist(ifile)(1:nc)).eq.0) then
        ! This is a directory. NB: this is not precisely a recursion
        ! as cubeadm_import_directory imports "somedir/*.*"
        call cubeadm_import_directory(filelist(ifile),nc,error)
        if (error)  error = .false.  ! Skip errors indexing file
      else
        ! This is not a directory (regular file...)
        call gag_file_guess(rname,filelist(ifile),filekind,error)
        if (error)  error = .false.  ! Skip errors indexing file
        if (filekind.eq.1 .or. filekind.eq.2) then
          call cubeadm_import_onefile(filelist(ifile),skipped,error)
          if (error)  exit
          if (skipped) then
            nskipped = nskipped+1
          else
            nimported = nimported+1
          endif
        endif
      endif
    enddo
    deallocate(filelist)
    !
    write(mess,'(I0,A,A)')  nimported,' new file(s) imported from ',trim(pattern)
    if (nimported.le.0) then
      myseve = seve%w
    else
      myseve = seve%r
    endif
    if (nskipped.gt.0) then
      write(mess,'(A,A,I0,A)')  &
        trim(mess),', skipped ',nskipped,' duplicate file(s)'
    endif
    call cubeadm_message(myseve,rname,mess)
  end subroutine cubeadm_import_pattern
  !
  subroutine cubeadm_import_onefile(file,skipped,error)
    use gkernel_interfaces
    use cube_types
    use cubedag_allflags
    use cubeadm_init
    use cubeadm_cubeid_types
    !---------------------------------------------------------------------
    !
    !---------------------------------------------------------------------
    character(len=*), intent(in)    :: file
    logical,          intent(out)   :: skipped
    logical,          intent(inout) :: error
    !
    character(len=base_l) :: filebase,cubeid
    class(cubedag_node_object_t), pointer :: dno
    type(cube_t), pointer :: cub
    type(flag_t), allocatable :: flags(:)
    character(len=exte_l) :: fileext
    integer(kind=4) :: ier
    character(len=*), parameter :: rname='IMPORT>ONEFILE'
    !
    if (cubedag_dag_contains(file)) then
      skipped = .true.
      return
    else
      skipped = .false.
    endif
    !
    ! Attach a new cube to the DAG
    call cubedag_dag_newbranch(dno,code_ftype_cube,error)
    if (error)  return
    !
    ! Attach its generic attributes
    call sic_parse_name(file,filebase,fileext)
    if (user%domapping) then
       call cubedag_mapping_ext2flag(fileext,flags,error)
       if (error)  return
    else if (user%doclass) then
       call cubedag_class_ext2flag(fileext,flags,error)
       if (error)  return
    else
       ! No special flags
    endif
    if (.not.allocated(flags)) then
      allocate(flags(1),stat=ier)
      if (failed_allocate(rname,'flags',ier,error)) return
      flags(:) = [flag_cube]
    endif
    call cubedag_node_set_origin(dno,code_origin_imported,error)
    if (error)  return
    call cubedag_node_set_family(dno,filebase,error)
    if (error)  return
    call cubedag_node_set_flags(dno,flags,error)
    if (error)  return
    call cubeadm_cubeid_familyflags2string(filebase,flags,cubeid,error)
    if (error)  return
    !
    cub => cubetuple_cube_ptr(dno,error)
    if (error)  return
    ! Set and fill the tuple
    call cubeadm_found_createtuple(file,cubeid,cub%node%tuple,cub,error)
    if (error) then
      ! Not a supported file, skip (but let it in DAG)
      call cubeadm_message(seve%w,rname,'Error importing file '//file)
      error = .false.
    endif
    !
    ! Set the header in index
    call cubeadm_head_to_node(cub%head,cub,error)
    if (error)  return
    !
    ! Need some cleaning. At least the GIO slots. We could go through
    ! the cubeadm_finalize_all at the end of the command execution, but
    ! IMPORT can open hundreds of cubes, and GIO slots are more limited
    ! than this. We have to avoid accumulating them:
    call cub%close(error)
    if (error)  return
    !
    call cubeadm_imported_add(dno)
    !
  end subroutine cubeadm_import_onefile
  !
  subroutine cubeadm_head_to_node(head,cube,error)
    use gkernel_interfaces
    use cubetools_header_types
    use cubetools_beam_types
    use cubetools_header_methods
    use cube_types
    !---------------------------------------------------------------------
    ! Transfer to the input index the relevant parts of the header
    !---------------------------------------------------------------------
    type(cube_header_t),  intent(in)    :: head
    type(cube_t),         intent(inout) :: cube
    logical,              intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='HEAD>TO>NODE'
    integer(kind=4) :: itel,ier
    type(beam_t) :: beam
    character(len=12), allocatable :: teles(:)
    !
    call cubedag_node_set_unit(cube,head%arr%unit,error)
    if (error)  return
    call cubedag_node_set_position(cube,head%spa%source,    &
                                       head%spa%pro%code,  &
                                       head%spa%pro%l0,    &
                                       head%spa%pro%m0,    &
                                       head%spa%pro%pa,    &
                                       head%spa%inc%l,     &
                                       head%spa%inc%m,error)
    if (error)  return
    call cubedag_node_set_spectro(cube,head%spe%line,   &
                                      head%spe%ref%f,  &
                                      head%spe%inc%f,  &
                                      head%spe%ref%v,error)
    if (error)  return
    call cubetools_header_get_spabeam(head,beam,error)
    if (error) return
    call cubedag_node_set_reso(cube,beam%major,  &
                                   beam%minor,  &
                                   beam%pang,error)
    if (error)  return
    !
    allocate(teles(head%obs%ntel),stat=ier)
    if (failed_allocate(rname,'telescopes',ier,error)) return
    do itel=1,head%obs%ntel
      teles(itel) = head%obs%tel(itel)%name
    enddo
    call cubedag_node_set_teles(cube,teles,error)
    if (error)  return
    deallocate(teles)
  end subroutine cubeadm_head_to_node
  !
  subroutine cubeadm_imported_add(dno)
    class(cubedag_node_object_t), pointer :: dno
    logical :: error
    error = .false.
    call cubedag_link_reallocate(im,im%n+1,error)
    im%n = im%n+1
    im%list(im%n)%p => dno
    im%flag(im%n) = 0
  end subroutine cubeadm_imported_add
  !
  subroutine cubeadm_imported_reset()
    logical :: error
    error = .false.
    call cubedag_link_final(im,error)
  end subroutine cubeadm_imported_reset
  !
end module cubeadm_import
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
