program uvshort_main
  use image_def
  use gbl_message
  use gkernel_interfaces, only : sysexi
  !------------------------------------------------------------------------
  ! Task  UV_SHORT
  !
  !     compute a short spacings uv table from a single-dish table
  !     by gridding, extending to 0, filtering in uv plane, multiplication
  !     by interferometer primary beam, and sampling in uv plane.
  !
  ! input :
  !   a Single-Dish table or image, and optionally a mosaic UV Table
  ! output :
  !   a UV table  (or one per field)
  !------------------------------------------------------------------------
  !
  ! By CONTAIN association, known to "uvshort_convol" and "uvshort_create_lmv"
  real ubias,vbias,ubuff(8192),vbuff(8192)
  !
  logical :: error
  !
  call uv_short_parameters(error)
  if (error) call sysexi(fatale)
  write(*,'(A)') 'S-UV_SHORT,  Successful completion'
  !
  contains
!
!--------------------------------------------------------------------------
!
subroutine uv_short_parameters(error)
  use gildas_def
  use gkernel_interfaces
  logical, intent(out) :: error
  !
  integer, parameter :: code_short_old=0    ! Old method: one file per field
  integer                           :: short_mode
  character(len=filename_length)    :: table
  character(len=filename_length)    :: uv_table
  character(len=filename_length)    :: map_name 
  real                              :: sd_factor
  real                              :: minw
  real                              :: tole
  real                              :: uv_trunc
  real                              :: sd_weight
  integer                           :: xcol
  integer                           :: ycol
  integer                           :: wcol
  integer                           :: mcol(2)
  real                              :: sd_beam
  real                              :: sd_diam
  real                              :: ip_beam
  real                              :: ip_diam
  character(len=12)                 :: weight_mode 
  logical                           :: do_single
  logical                           :: do_primary
  character(len=16)                 :: chra
  character(len=16)                 :: chde
  integer                           :: nf
  real, allocatable    :: raoff(:)
  real, allocatable    :: deoff(:)
  !
  ! Input parameters
  ! ----------------
  !
  call gildas_open
  call gildas_inte('MODE$',short_mode,1)  !SG
  !
  call gildas_char('TABLE$',table)
  call gildas_char('UV_TABLE$', uv_table)
  call gildas_char('MAP$',map_name)
  !
  call gildas_real('SD_FACTOR$', sd_factor,1)     ! unit conversion factor. 
  !                     If zero, will assume the Jy/K factor from the 
  !                     single dish beam size and observing frequency
  !                     if the unit is K. No default otherwise.
  !
  ! All other beyond can in general be default to 0 in MODE # 0
  !
  call gildas_real('MIN_WEIGHT$',minw,1)          ! Something like a few 0.01
  !                     If zero, 0.01 
  !
  call gildas_real('TOLE$',tole,1)                ! Pointing tolerance in radians
  !                     If zero, will assume 1/10th of single dish beam  TBD
  !
  call gildas_real('UV_TRUNC$',uv_trunc,1)        ! truncation radius (m)
  !                     If zero, will be sd_beam - ip_beam
  !
  call gildas_real('SD_WEIGHT$', sd_weight,1)     ! single dish weight factor, normally 1
  !                     If zero, 1.0 
  !
  ! These are only used in the TABLE mode, and can be defaulted to zero
  call gildas_inte('XCOL$',xcol,1)                ! 1
  call gildas_inte('YCOL$',ycol,1)                ! 2
  call gildas_inte('WCOL$',wcol,1)                ! 3
  call gildas_inte('MCOL$',mcol,2)                ! 4 0
  !
  ! These normally should be defaulted to 0 if MODE$ # 0 
  call gildas_real('SD_BEAM$', sd_beam,1)         ! single dish beam (rad)
  call gildas_real('SD_DIAM$', sd_diam,1)         ! single dish diam. (m)
  call gildas_real('IP_BEAM$', ip_beam,1)         ! interf. primary beam (rad)
  call gildas_real('IP_DIAM$', ip_diam,1)         ! interf. diam. (m)
  !
  call gildas_char('WEIGHT_MODE$',weight_mode)    ! Normally UN or NA, but little effect
  !
  ! These are for tests only
  call gildas_logi('DO_SINGLE$', do_single, 1)    ! Correct for Single dish beam
  call gildas_logi('DO_PRIMARY$', do_primary, 1)  ! Correct for Interferometer Primary beam
  !
  if (short_mode.eq.code_short_old .or. abs(short_mode).gt.10) then !SG Begin
    !
    ! These are only needed for Backward compatiblity or Tests
    call gildas_char('MOSAIC_RA$',chra)
    call gildas_char('MOSAIC_DEC$',chde)
    !
    call gildas_inte('NF$',nf,1)
    nf = max(nf,1)
    allocate(raoff(nf),deoff(nf))
    call gildas_real('RA$',raoff,nf)
    call gildas_real('DEC$',deoff,nf)
    !
  else
    nf = 0
    allocate(raoff(1),deoff(1))
  endif           ! SG End
  !
  call gildas_close
  !
  call uvshort_sub(short_mode, table, uv_table, map_name, sd_factor,   &
  & minw, tole, uv_trunc, sd_weight,  xcol, ycol, wcol, mcol,    &
  & sd_beam, sd_diam, ip_beam, ip_diam, weight_mode, do_single,  &
  & do_primary, chra, chde, nf, raoff, deoff, error)
  !
end subroutine uv_short_parameters
!
subroutine uvshort_sub(short_mode, table, uv_table, map_name, sd_factor,   &
  & minw, tole, uv_trunc, sd_weight,  xcol, ycol, wcol, mcol,    &
  & sd_beam, sd_diam, ip_beam, ip_diam, weight_mode, do_single,  &
  & do_primary, chra, chde, in_nf, in_raoff, in_deoff, error)
  use image_def
  use gbl_format
  use gbl_message
  use gkernel_types
  use gkernel_interfaces
  use mapping_interfaces, only : mosaic_getfields, map_message, telescope_beam
  !
  ! Task UV_SHORT
  !   Main subroutine
  !
  integer, intent(inout)            :: short_mode
  character(len=*), intent(inout)   :: table
  character(len=*), intent(inout)   :: uv_table
  character(len=*), intent(inout)   :: map_name 
  real, intent(inout)               :: sd_factor
  real, intent(inout)               :: minw
  real, intent(inout)               :: tole
  real, intent(inout)               :: uv_trunc
  real, intent(inout)               :: sd_weight
  integer, intent(inout)            :: xcol
  integer, intent(inout)            :: ycol
  integer, intent(inout)            :: wcol
  integer, intent(inout)            :: mcol(2)
  real, intent(inout)               :: sd_beam
  real, intent(inout)               :: sd_diam
  real, intent(inout)               :: ip_beam
  real, intent(inout)               :: ip_diam
  character(len=*), intent(inout)   :: weight_mode 
  logical, intent(inout)            :: do_single
  logical, intent(inout)            :: do_primary
  character(len=*), intent(inout)   :: chra
  character(len=*), intent(inout)   :: chde
  integer, intent(in)               :: in_nf
  real, intent(in)                  :: in_raoff(*)
  real, intent(in)                  :: in_deoff(*)
  logical, intent(out) :: error
  !
  real(8), parameter :: pi=3.141592653589793d0
  real(8), parameter :: f_to_k=2.d0*pi/299792458.d-6
  real(8), parameter :: clight=299792458d0
  real(8), parameter :: kbolt=1.3806488d-23
  !
  type (gildas), save :: lmv ! Input or Output Single Dish data cube in LMV order
  type (gildas), save :: uvt ! Output UV table (Short spacings only or complete)
  type (gildas), save :: hin ! Input UV table when used to append
  type (gildas), save :: sdt ! Single Dish Table for input (Zero spacings only or complete)
  real, allocatable :: lmv_data(:,:,:)     ! LMV data cube
  real, allocatable :: uvt_data(:,:)       ! computed UV data
  !
  real, allocatable :: gr_im_w(:)          ! gridded weights in image plane
  complex, allocatable :: gr_uv_w(:,:)     ! gridded weights in UV plane
  real, allocatable :: sd_lobe(:,:)        ! Single dish primary beam
  real, allocatable :: int_lobe(:,:)       ! Interferometer primary beam
  real, allocatable :: fftws(:)            ! FFT work space
  complex, allocatable :: ws_data(:,:,:)   ! Work space of SD UV table values
  complex, allocatable :: int_lobe_comp(:,:) ! Interferometer beam in complex plane
  !
  ! Dummy variables
  !
  real smooth
  integer nx,ny,nu,nc,np,n,nn(2),ndim
  integer ier
  character(len=filename_length) name
  character(len=4) extension
  integer nvis, if, nf, i
  real, allocatable :: raoff(:), deoff(:)
  real sfactor
  !
  real diam_sd, beam_sd, diam_ip, beam_ip
  real, parameter :: beam_tole=0.05
  real, parameter :: all_tole=0.001
  real, parameter :: diam_tole=0.02
  integer :: sever
  !
  logical lmv_file
  character(len=8) string
  character(len=80) mess
  !
  character(len=*), parameter :: rname='UV_SHORT'
  !
  logical :: positions = .false.
  !
  ! SG Begin
  integer, parameter :: code_short_old=0    ! Old method: one file per field
  integer, parameter :: code_short_phase=1  ! Mosaic UV table with phase offsets
  integer, parameter :: code_short_point=2  ! Mosaic UV table with pointing offsets
  integer, parameter :: code_short_auto=3   ! Automatic derivation of UV table offsets type
  integer :: xoff, yoff, loff, moff, nv, nprec
  integer(kind=index_length) :: mvis
  real(8) :: freq
  real(4) :: umax
  real(4) :: weight, scale_weight
  real(8), allocatable :: rpos(:,:)
  real(4) :: cs(2)
  logical :: precise = .true.
  real, allocatable :: duv(:,:), offxy(:,:) ! SG End
  logical :: keep_offset=.false.
  !
  ! Zero spacings
  type(projection_t) :: uv_proj, lm_proj  
  real(8) :: alpha,delta,dx,dy
  integer :: ix,iy,is,icol,ocol,jf
  integer :: gdate, last, ixcol, iycol
  real, allocatable :: zero_uvdata(:,:)
  real, allocatable :: sdt_data(:,:)
  real, allocatable :: spectrum(:)
  real :: ws, dist, dmin, total_weight
  logical :: do_zero, allow_diam_overrun
  real, parameter :: rad_to_sec=180.0*3600.0/pi
  !------------------------------------------------------------------------
  !
  ! V2.0  Support for Mosaic UV tables, automatic determination of parameters
  !       as much as possible in this case (using the Telescope section if
  !       present)
  ! 'Version 2.0   15-Nov-2016'
  !
  ! V2.1  Reorder parameters and provide sensible defaults
  !       to allow a simpler input file
  ! 'Version 2.1   06-Jun-2017'
  !
  ! V2.2  Fall back on Zero Spacings when SD_DIAM = IP_DIAM
  !------------------------------------------------------------------------
  character(len=*), parameter :: version = 'Version 2.2   06-oct-2017'!
  !
  call map_message(seve%i,rname,version)
  !
  error = .false.
  !
  call map_message(seve%i,rname,'Weighting mode is '//weight_mode)
  if (.not.do_single) then
    call map_message(seve%w,rname,' ')
    call map_message(seve%w,rname,'  *** DO_SINGLE is not set *** Test mode only !')
    call map_message(seve%w,rname,' ')
  endif
  if (.not.do_primary) then
    call map_message(seve%w,rname,' ')
    call map_message(seve%w,rname,'  *** DO_PRIMARY is not set *** Test mode only !')
    call map_message(seve%w,rname,' ')
  endif
  !
  !
  !-----------------------------------------------------------------------
  !
  ! Various checks
  ! --------------
  !
  lmv_file=.false.    ! Because of compiler warning only
  !
  error = .false.
  n = len_trim(table)
  if (n.eq.0) then
    call map_message(seve%f,rname,'Input table name empty')
    error = .true.
  else
    ! Check if it's a table or a cube lmv input file
    extension = table(n-3:n)
    call map_message(seve%i,rname,'Input file extension is '//extension)
    if (extension.eq.'.tab') then
      lmv_file=.false.
    elseif (extension.eq.'.lmv') then
      lmv_file=.true.
    else
      call map_message(seve%f,rname,'You should give input file name with his extension')
      error = .true.
    endif
  endif
  if (len_trim(map_name).eq.0) then
    call map_message(seve%f,rname,'Output image name empty')
    error = .true.
  endif
  if (short_mode.ne.code_short_old) then
    if (len_trim(uv_table).eq.0) then
      call map_message(seve%f,rname,'Empty UV table name')
      error = .true.
    endif
  endif
  if (error) return
  !
  !
  ! Initialize gildas headers
  !
  call gildas_null(lmv)
  call gildas_null(uvt, type = 'UVT')
  !
  !
  if (abs(short_mode).gt.10) then ! SG Begin
    !
    ! Keep_offset indicates to take the coordinates from the Parameter file,
    ! not from the Mosaic UV table
    keep_offset = .true.
    if (short_mode.lt.0) then
      short_mode = short_mode+10
    else
      short_mode = short_mode-10
    endif
  else
    keep_offset = .false.
  endif
  !
  ! Check if the UV Table is given as a Mosaic already
  if (short_mode.ne.code_short_old) then ! SG Begin
    !
    allow_diam_overrun = abs(short_mode).gt.code_short_auto
    !
    name  = trim(uv_table)
    call sic_parsef(name,uvt%file,' ','.uvt')
    call map_message(seve%i,rname,'Input UV Table is '//trim(uvt%file))
    !
    call gdf_read_header(uvt,error)
    if (error) return
    !
    ! Recover the number of Fiels
    loff = uvt%gil%column_pointer(code_uvt_loff)
    moff = uvt%gil%column_pointer(code_uvt_moff)
    !
    xoff = uvt%gil%column_pointer(code_uvt_xoff)
    yoff = uvt%gil%column_pointer(code_uvt_yoff)
    !
    if (abs(short_mode).eq.code_short_phase) then
      if (loff.ne.0 .or. moff.ne.0) then
        if (moff.ne.loff+1) then
          call map_message(seve%f,rname,'Improper Mosaic UV table')
          error = .true.
          return
        endif
        uvt%blc(1:2) = [loff,0]
        uvt%trc(1:2) = [moff,0]
        write(mess,'(A,I0,A,I0)') 'Phase offset columns at Loff ',loff,', Moff ',moff
        call map_message(seve%i,rname,mess)
      else
        call map_message(seve%f,rname,'Mosaic UV table does not have Phase offsets')
        error = .true.
        return
      endif
    else if (abs(short_mode).eq.code_short_point) then
      if (xoff.ne.0 .or. yoff.ne.0) then
        if (yoff.ne.xoff+1) then
          call map_message(seve%f,rname,'Improper Mosaic UV table')
          error = .true.
          return
        endif
        uvt%blc(1:2) = [xoff,0]
        uvt%trc(1:2) = [yoff,0]
        write(mess,'(A,I0,A,I0)') 'Pointing offset columns at Xoff ',xoff,', Yoff ',yoff
        call map_message(seve%i,rname,mess)
      else
        call map_message(seve%f,rname,'Mosaic UV table does not have Pointing offsets')
        error = .true.
        return
      endif
    else
      !
      ! Automatic determination
      if (loff.ne.0 .or. moff.ne.0) then
        if (moff.ne.loff+1) then
          call map_message(seve%f,rname,'Improper Mosaic UV table')
          error = .true.
          return
        endif
        uvt%blc(1:2) = [loff,0]
        uvt%trc(1:2) = [moff,0]
        write(mess,'(A,I0,A,I0)') 'Phase offset columns at Loff ',loff,', Moff ',moff
        call map_message(seve%i,rname,mess)
        short_mode = sign(code_short_phase,short_mode) ! Preserve Sign 
      else if (xoff.ne.0 .or. yoff.ne.0) then
        uvt%blc(1:2) = [xoff,0]
        uvt%trc(1:2) = [yoff,0]
        write(mess,'(A,I0,A,I0)') 'Pointing offset columns at Xoff ',xoff,', Yoff ',yoff
        call map_message(seve%i,rname,mess)
        short_mode = sign(code_short_point,short_mode) ! Preserve Sign 
      else
        call map_message(seve%f,rname,'Input UV table is not a Mosaic')
        error = .true.
        return
      endif      
    endif
    !
    ! Read the corresponding columns only
    if (keep_offset) then
      if (in_nf.ne.nf) then
        write(mess,'(A,I0,A,I0,A)') 'Number of input fields ',in_nf, &
          & ' does not match ',nf,' in UV Mosaic Table'
        call map_message(seve%e,rname,mess)
        error = .true.
        return
      endif
      raoff(:) = in_raoff(1:nf)
      deoff(:) = in_deoff(1:nf)
      write(mess,'(A,I0,A)') 'Selected ',nf,' fields at [Ra, Dec] offsets'
    else
      np = 2
      nv = uvt%gil%dim(2)
      allocate (duv(np,nv),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%w,rname,'Cannot allocate memory for Field columns')
        error = .true.
        return
      endif
      call gdf_read_data(uvt,duv,error)
      if (error) return
      call mosaic_getfields (duv,np,nv,1,2,nf,offxy)
      deallocate (duv)
      !
      ! nf is the number of fields
      ! offxy(2,nf) are the offsets
      nf = max(nf,1)
      allocate(raoff(nf),deoff(nf),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%w,rname,'Cannot allocate memory for Offsets')
        error = .true.
        return
      endif
      raoff(:) = offxy(1,:)
      deoff(:) = offxy(2,:)
      !
      ! Get the RA and DEC - One may use variable "new", but currently, just fill chra & chde
      ! for simplicity, just fill CHRA & CHDE as "empty"
      chra = " "
      chde = " "
      !
      ! We may take the IP_BEAM and IP_DIAM from the UV Table here
      write(mess,'(A,I0,A)') 'Found ',nf,' fields at [Ra, Dec] offsets'
    endif
    !
    call map_message(seve%i,rname,mess)
    do i = 1,nf
      write(6,'(A,f8.2,a,F8.2,A)') '     [',raoff(i)*180*3600/pi,&
          & ', ',deoff(i)*180*3600/pi,'] '
    enddo
    !
    ! Read the Single Dish header (table or datacube), in "lmv" structure for the time being.
    call gildas_null(lmv)
    name = trim(table)
    call sic_parsef(name,lmv%file,' ','.tab')
    call map_message(seve%i,rname,'Single Dish Table is '//trim(lmv%file))
    call gdf_read_header(lmv,error)
    if (gildas_error(lmv,rname,error)) then
      call map_message(seve%f,rname,'Cannot read header from Single Dish table')
      return
    endif
    !
    ! Check Beam, diameters and truncation
    if (lmv%gil%nteles.ne.0) then
      diam_sd = lmv%gil%teles(1)%diam
      if (sd_diam.eq.0) then
        write(mess,'(A,F6.1,A)') 'Using SD diameter derived from Telescope section: ', &
          &   diam_sd,' m'
        sever = seve%i
        sd_diam = diam_sd
      else if (abs(diam_sd-sd_diam).gt.sd_diam*all_tole) then
        if (abs(diam_sd-sd_diam).gt.sd_diam*diam_tole) then
          if (allow_diam_overrun) then
            write(mess,'(A,F6.1,A,F6.1,A)') 'SD diameter: ',sd_diam, &
              & ' m overrides Telescope diameter: ',diam_sd,' m'
            sever = seve%w
          else
            write(mess,'(A,F6.1,A,F6.1,A)') 'SD diameter: ',sd_diam, &
              & ' m does not match Telescope diameter: ',diam_sd,' m'
            error = .true.
            sever = seve%e
          endif
        else
          write(mess,'(A,F6.1,A,F6.1,A)') 'Used SD diameter: ',sd_diam, &
          & ' m does not match Telescope diameter: ',diam_sd,' m'
          sever = seve%w
        endif
      else
        write(mess,'(A,F6.1,A)') 'Using SD diameter: ',sd_diam,' m'
        sever = seve%i
      endif
    else
      write(mess,'(A,F6.1,A)') 'Using SD diameter: ',sd_diam,' m'
      sever = seve%i
    endif
    call map_message(sever,rname,mess)
    !
    ! New CODE for beam size
    beam_sd = telescope_beam(rname,lmv)
    if (beam_sd.eq.0.0) then
      write(mess,'(A,F8.1,A)') 'Using specified SD beam: ',sd_beam*3600*180/pi,'"'
      sever = seve%i
    else
      write(mess,'(A,F6.1,A)') 'Using SD Beam derived from data: ', &
        &   beam_sd*3600*180/pi,'"'
      sd_beam = beam_sd
      sever = seve%i
    endif
    call map_message(sever,rname,mess)
    !
    ! Now Interferometer beam and diameter
    !
    if (uvt%gil%nteles.ne.0) then
      diam_ip = uvt%gil%teles(1)%diam
      if (ip_diam.eq.0) then
        write(mess,'(A,F6.1,A)') 'Using IP diameter derived from Telescope section: :', &
            & diam_ip,' m'
        sever = seve%i
        ip_diam = diam_ip
      else if (abs(diam_ip-ip_diam).gt.ip_diam*all_tole) then
        write(mess,'(A,F5.1,A,F5.1,A)') 'Used IP diameter:',ip_diam, &
          & ' m does not match Telescope diameter:',diam_ip,' m'
        if (abs(diam_ip-ip_diam).gt.ip_diam*diam_tole) then
          error = .true.
          sever = seve%e
        else
          sever = seve%w
        endif
      else
        write(mess,'(A,F6.1,A)') 'Using IP diameter: ',ip_diam,' m'
        sever = seve%i
      endif
    else
      write(mess,'(A,F6.1,A)') 'Using IP diameter: ',ip_diam,' m'
      sever = seve%i
    endif
    call map_message(sever,rname,mess)
    !
    ! New CODE for beam size
    beam_ip = telescope_beam(rname,uvt)
    if (beam_ip.eq.0.0) then
      write(mess,'(A,F8.1,A)') 'Using specified IP beam: ',ip_beam*3600*180/pi,'"'
      sever = seve%i
    else
      write(mess,'(A,F6.1,A)') 'Using IP Beam derived from data: ', &
        &   beam_ip*3600*180/pi,'"'
      ip_beam = beam_ip
      sever = seve%i
    endif
    !
    call map_message(sever,rname,mess)
    !
  else
    !
    ! Old mode: use fields as specified
    nf = in_nf
    allocate(raoff(nf),deoff(nf),stat=ier)
    raoff(:) = in_raoff(1:nf)
    deoff(:) = in_deoff(1:nf)
    write(mess,'(A,I0,A)') 'Selected ',nf,' fields at [Ra, Dec] offsets'
    call map_message(seve%i,rname,mess)
    do i = 1,nf
      write(6,'(A,f8.2,a,F8.2,A)') '     [',raoff(i)*180*3600/pi,&
          & ', ',deoff(i)*180*3600/pi,'] '
    enddo
  endif   ! SG End
  !
  ! Verify all parameters are known and reasonable
  if (sd_beam.eq.0) then
    call map_message(seve%f,rname,'SD_BEAM must be specified')
    error = .true.
  endif
  if (ip_beam.eq.0) then
    call map_message(seve%f,rname,'IP_BEAM must be specified')
    error = .true.
  endif
  if (sd_diam.eq.0) then
    call map_message(seve%f,rname,'SD_DIAM must be specified')
    error = .true.
  endif
  if (ip_diam.eq.0) then
    call map_message(seve%f,rname,'IP_DIAM must be specified')
    error = .true.
  endif
  if (error) return
  !
  ! Verify if Short or Zero spacings can be computed
  if (abs(sd_diam-ip_diam).le.all_tole*sd_diam) then
    call map_message(seve%i,rname,'SD diameter = IP diameter, using Zero spacing only')
    do_zero = .true.
  else if (sd_diam.lt.ip_diam) then
    call map_message(seve%f,rname,'SD diameter too small')
    error = .true.
    return
  else  
    do_zero = .false.
  endif
  !
  ! Check the Unit conversion factor
  if (sd_factor.eq.0.0) then
    if (lmv%gil%freq.eq.0.0) then
      mess = 'SD_FACTOR$ not specified, and no frequency in input data'
      call map_message(seve%e,rname,mess)
      error = .true.
    else if (lmv%char%unit.ne.'K') then
      if (lmv%char%unit(1:2).eq.'K ') then
        sd_factor = 2.0*kbolt*1d26*pi*sd_beam**2/(4.d0*log(2.d0))*(lmv%gil%freq*1d6/clight)**2
        write(mess,'(A,1PG11.3,A)') 'Unit conversion factor would be ',sd_factor,' Jy per K (Tmb)'
        call map_message(seve%w,rname,mess)
      endif
      mess = 'SD_FACTOR$ not specified, and unit is not K, but '//lmv%char%unit 
      call map_message(seve%e,rname,mess)
      error = .true.
    else
      sd_factor = 2.0*kbolt*1d26*pi*sd_beam**2/(4.d0*log(2.d0))*(lmv%gil%freq*1d6/clight)**2
      write(mess,'(A,1PG11.3,A)') 'Unit conversion factor set to ',sd_factor,' Jy per K'
      call map_message(seve%i,rname,mess)
    endif
    if (error) return
  endif
  !
  ! Check UV_TRUNC
  if (.not.do_zero) then
    if (uv_trunc.eq.0.0) then
      call map_message(seve%w,rname,'Setting default truncation radius')
      uv_trunc = sd_diam-ip_diam
    else if (uv_trunc-sd_diam+ip_diam.gt.0.0) then
      call map_message(seve%w,rname,'Incoherent input parameters, re-setting UV_TRUNC')
      uv_trunc = sd_diam-ip_diam
    endif
    write(mess,'(A,F5.1,A)') 'Using UV truncation radius:  ',uv_trunc,' m'
    call map_message(seve%i,rname,mess)
  endif
  !
  ! Check TOLE$
  if (tole.eq.0.0) then
    tole = sd_beam/16.0 
    write(mess,'(A,F5.1,A)') 'Position tolerance TOLE$ set to SD_BEAM$/16.0 = ', &
    & tole*180*3600/pi,' "'
    call map_message(seve%i,rname,mess)
  endif
  !
  ! Check SD_WEIGHT$
  if (sd_weight.eq.0.0) then
    sd_weight = 1.0
    call map_message(seve%i,rname,'Weight scaling factor SD_WEIGHT$ set to 1.0')
  endif
  !
  ! Check MIN_WEIGHT$
  if (minw.eq.0.0) then
    minw = 0.01
    call map_message(seve%i,rname,'Minimum weight MIN_WEIGHT$ set to 0.01')
  endif
  !
  !-----------------------------------------------------------------------
  !
  if (do_zero) then
    !
    ! Only this coded for the time being...
    if (short_mode.eq.0) then
      call map_message(seve%e,rname,'Zero spacing only works for Mosaic UV tables')
      error = .true.
      return
    endif
    !
    ! We already have the reference UV table header here
    ! Allocate some work space
    !
    call gwcs_projec(uvt%gil%a0,uvt%gil%d0,-uvt%gil%pang,uvt%gil%ptyp,uv_proj,error)
    call sic_gagdate(gdate)
    ! ...
    last = uvt%gil%fcol + uvt%gil%natom * uvt%gil%nchan - 1
    ixcol = last+1
    iycol = last+2
    !
    if (lmv_file) then
      ! Read LMV file
      ! Input file is a lmv cube
      call uvshort_read_lmv(rname,table,lmv,lmv_data,gr_im_w,error)
      !
      ! We can get the weight from the Noise in the map
      weight = sd_factor*max(lmv%gil%rms, lmv%gil%noise)
      if (weight.ne.0.0) then
        weight = 1d-6/weight**2
        ! weight = sd_weight * weight ! Is that useful ?...
      else
        call map_message(seve%w,rname,'No weight in LMV file')
        weight = 1.0
      endif
      !
      nc=lmv%gil%dim(3)
      call uv_short_consistency(rname,nc,uvt,lmv,tole,error)
      if (error) return
      !
      allocate(zero_uvdata(uvt%gil%dim(1),nf),spectrum(nc),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%e,rname,'Zero spacing Memory allocation failure')
        error = .true.
        return
      endif
      !
      ! Extract appropriate directions from it: loop on fields
      call gwcs_projec(lmv%gil%a0,lmv%gil%d0,-lmv%gil%pang,lmv%gil%ptyp,lm_proj,error)
      do if=1,nf
        call rel_to_abs(uv_proj,dble(raoff(if)),dble(deoff(if)),alpha,delta,1)
        call abs_to_rel(lm_proj,alpha,delta,dx,dy,1)
        ix = nint((dx-lmv%gil%val(1))/lmv%gil%inc(1)+lmv%gil%ref(1))
        iy = nint((dy-lmv%gil%val(2))/lmv%gil%inc(2)+lmv%gil%ref(2))
        if (ix.lt.1 .or. ix.gt.nx .or. iy.lt.1 .or. iy.gt.ny) then
          write(mess,*) 'Field ',if,' is out of map ',ix,iy
          call map_message(seve%e,rname,mess)
          error = .true.
          return
        endif
        !
        ! Put it in place...
        spectrum(:) = sd_factor * lmv_data(ix,iy,:) 
        call spectrum_to_zero(nc,spectrum,zero_uvdata(:,if),gdate,weight)
        zero_uvdata(ixcol,if) = raoff(if)
        zero_uvdata(iycol,if) = deoff(if)        
      enddo
      !
      total_weight = nf*weight
    else
      !
      nc = uvt%gil%nchan
      !
      call uvshort_read_class_table(rname,table,sdt,sdt_data,error)
      if (error) return
      !      
      ! Set valid defaults for Class Tables
      if (xcol.eq.0) xcol = 1
      if (ycol.eq.0) ycol = 2
      if (wcol.eq.0) wcol = 3
      if (mcol(1).eq.0) mcol(1) = 4
      icol = mcol(1)
      if (mcol(2).ne.0) then
        ocol = mcol(2)
        if (ocol-icol+1.ne.nc) then
          Print *,'MCOL ',mcol,' ICOL OCOL ',icol,ocol,' NC ',nc
          call map_message(seve%e,rname,'Number of channels mismatch')
          error = .true.
          return
        endif
      else
        ocol = icol+nc-1
        if (ocol.gt.sdt%gil%dim(1)) then
          Print *,'MCOL ',mcol,' ICOL OCOL ',icol,ocol,' Sdt ',sdt%gil%dim(1)
          call map_message(seve%e,rname,'Number of channels mismatch')
          error = .true.
          return
        endif
      endif
      !
      allocate(zero_uvdata(uvt%gil%dim(1),nf),spectrum(nc),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%e,rname,'Zero spacing Memory allocation failure')
        error = .true.
        return
      endif!
      !
      ! Find all spectra towards the appropriate directions,
      ! with the specified tolerance
      total_weight =0.0
      call gwcs_projec(sdt%gil%a0,sdt%gil%d0,-sdt%gil%pang,sdt%gil%ptyp,lm_proj,error)
      do if=1,nf
        spectrum(:) = 0.0
        weight = 0.0
        jf = 0
        dmin = 1e36
        !
        call rel_to_abs(uv_proj,dble(raoff(if)),dble(deoff(if)),alpha,delta,1)
        call abs_to_rel(lm_proj,alpha,delta,dx,dy,1)
        !
        do is=1,sdt%gil%dim(2)
          dist = (sdt_data(xcol,is)-dx)**2+(sdt_data(ycol,is)-dy)**2
          if (dist .le. tole**2) then
            jf = jf+1
!            Print *,'Field ',if,' X',(sdt_data(xcol,is)-dx)*rad_to_sec,   & 
!            & '  Y',(sdt_data(ycol,is)-dy)*rad_to_sec,         &
!            & ' Tole ',tole*rad_to_sec
            ws = sdt_data(wcol,is)
            weight = weight + ws
            spectrum(:) = spectrum(:) + ws * sdt_data(icol:ocol,is)
            dmin = min(dmin,dist)
          endif
        enddo
        dmin = sqrt(dmin)*rad_to_sec
        if (weight.eq.0) then
          write(mess,'(A,I0,A,1PG10.2)') 'Field ',if,' has no Zero spacing' &
          &  //' Min distance ',dmin
          call map_message(seve%e,rname,mess)
          error = .true.
        else
          write(mess,'(I0,A,I0,A,1PG10.2)') jf,' spectra added to Field ',if &
          &  ,'; Min distance ',dmin
          call map_message(seve%i,rname,mess)
          total_weight = total_weight + weight
        endif
        !
        ! Put it in place...
        spectrum = spectrum/weight
        call spectrum_to_zero(nc,spectrum,zero_uvdata(:,if),gdate,weight)
        zero_uvdata(ixcol,if) = raoff(if)
        zero_uvdata(iycol,if) = deoff(if)        
      enddo
    endif
    if (error) return
    !
    ! Now concatenate the Zero spacing data to the Mosaic UV Table
    ! or Create a Zero spacing UV table
    if (short_mode.lt.0) then
      ! short_mode < 0
      !   Here we create a single UV table containing only
      !   the short spacings  for all fields
      name = trim(uv_table)//"-short"
      call sic_parsef(name,uvt%file,' ','.uvt')
      call map_message(seve%i,rname,'Creating UV table '//trim(uvt%file))
      uvt%gil%nvisi = nf
      uvt%gil%dim(2) = nf
      call gdf_create_image(uvt,error)
      uvt%blc = 0
      uvt%trc = 0
    else
      ! short_mode > 0
      !   Here we create a UV table containing the short spacings
      !   for all fields appended to the previous UV table
      call gildas_null(hin)
      name = trim(uv_table)
      call sic_parsef(name,hin%file,' ','.uvt')
      call gdf_read_header(hin,error)
      if (error) return
      !
      name = trim(uv_table)//"-merged"
      call sic_parsef(name,uvt%file,' ','.uvt')
      call map_message(seve%i,rname,'UV Table to be appended is '//trim(uvt%file))
      wcol = (nc+2)/3
      ! Copy UV table and get Total weight up to UMAX
      umax = 2.0*sd_diam  ! Here Sd_diam = Ip_diam
      call uvshort_copy_uvtable(rname,hin,uvt,umax,wcol,weight,error)
      if (error) return
      !
      ! Scaling factor for Weights
      call map_message(seve%i,rname,'Appending UV table '//trim(uvt%file))
      string = 'appended'
      uvt%blc(2) = uvt%gil%dim(2)+1
      mvis = uvt%gil%dim(2)+nf      ! Add NF visibilities
      uvt%trc(2) = mvis
      call gdf_extend_image(uvt,mvis,error)
      uvt%gil%nvisi = uvt%gil%dim(2)
      !
      ! Re-normalize the Weight to have a nice beam, including
      ! User-specified Weight factor SD_WEIGHT
      scale_weight = weight*sd_weight/4.0/total_weight
      do if=1,nf
        do is = 10,7+3*nc,3
          zero_uvdata(is,if) = zero_uvdata(is,if) * scale_weight
        enddo
      enddo
    endif
    !
    ! Write the Zero spacings in place now
    call gdf_write_data(uvt,zero_uvdata,error)
    if (error) then
      call map_message(seve%e,rname,'Error writing UV data ')
      return
    endif
    write(mess,'(I0,A)') nf,' Zero spacings added'
    call map_message(seve%i,rname,mess)
    !
    ! Finish the job. Update the number of visibilities
    call gdf_update_header(uvt,error)
    call gdf_close_image(uvt,error)
    return  
    !
    ! End of Zero Spacings case
  endif
  !
  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  !
  ! From here, this is only for Short Spacings
  ! 
  if (lmv_file) then
    !
    ! Input file is a lmv cube
    call uvshort_read_lmv(rname,table,lmv,lmv_data,gr_im_w,error)
  else
    !
    ! Input file is a single-dish table
    !
    ! Set valid defaults for Class Tables
    if (xcol.eq.0) xcol = 1
    if (ycol.eq.0) ycol = 2
    if (wcol.eq.0) wcol = 3
    if (mcol(1).eq.0) mcol(1) = 4
    !
    call uvshort_create_lmv(rname,table,lmv,lmv_data,gr_im_w,error, &
    & uvt,map_name,xcol,ycol,wcol,mcol,sd_beam,tole,chra,chde,minw )
  endif
  if (error) return
  !
  !
  nx=lmv%gil%dim(1)
  ny=lmv%gil%dim(2)
  nc=lmv%gil%dim(3)
  !
  !=====================================================================
  !
  ! Processing for all types .lmv or .tab of input file
  !
  !=====================================================================
  !
  ! Memory allocations
  !--------------------
  !
  error = .true.
  allocate(ws_data(nx,ny,nc),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%w,rname,'Cannot allocate memory for work space')
    return
  endif
  allocate(sd_lobe(nx,ny),stat=ier)       ! for SD primary beam
  if (ier.ne.0) then
    call map_message(seve%w,rname,'Cannot allocate memory for SD primary beam')
    return
  endif
  allocate(int_lobe(nx,ny),stat=ier)      ! for interferometer beam
  if (ier.ne.0) then
    call map_message(seve%w,rname,'Cannot allocate memory for interferometer beam')
    return
  endif
  allocate(gr_uv_w(nx,ny),stat=ier)     ! for gridded weights in uv plane
  if (ier.ne.0) then
    call map_message(seve%w,rname,'Cannot allocate memory for gridded weights')
    return
  endif
  allocate(fftws(2*max(nx,ny)),stat=ier)         ! for fft
  if (ier.ne.0) then
    call map_message(seve%w,rname,'Cannot allocate memory for FFT computation')
    return
  endif
  allocate(int_lobe_comp(nx,ny),stat=ier)    ! for int. Primary Beam
  if (ier.ne.0) then
    call map_message(seve%w,rname,'Cannot allocate memory for primary beam')
    return
  endif
  !
  !
  !-----------------------------------------------------------------------
  !
  ! Visibilities extraction
  ! -----------------------
  !
  ! Update parameters for FFT computation
  !
  ndim = 2
  nn(1) = nx
  nn(2) = ny
  !
  ! Get inverse of FT of (primary beam + convolving function)
  ! CAUTION: output of dosdft function is a real, but with
  ! complex convention
  !
  sfactor = 1.0+1.0/9.0    ! take into account gridding function
  smooth = sd_beam*sqrt(sfactor)
  call uvshort_dosdft(smooth,sd_diam,sd_lobe,nx,ny,sfactor,lmv)
  !
  ! Start loop on mosaic fields
  !
  Print *,'SHORT MODE ',short_mode, code_short_old
  !
  if (short_mode.eq.code_short_old) then
    Print *,'OLD Mode NF ',nf
    do if=1,nf
      !
      ! 1) Compute interferometer primary beam
      ! --------------------------------------
      !
      ! 1.a) Int. Primary Beam = gaussian, computed in image plane
      !
      ! call uvshort_doprim(ip_beam,int_lobe,nx,ny,raoff(if),deoff(if),lmv)
      !
      ! 1.b) Int. Primary Beam = truncated gaussian in the uv plane then FT
      !
      call uvshort_dointft(ip_beam,ip_diam,int_lobe,nx,ny,1.0,lmv)
      int_lobe_comp(:,:) = cmplx(int_lobe)
      call uvshort_shift(int_lobe_comp,nx,ny,raoff(if),deoff(if),lmv)
      call fourt(int_lobe_comp,nn,ndim,1,1,fftws)
      call cmtore(int_lobe_comp,int_lobe,nx,ny)
      !
      ! Normalize peak to 1
      !
      int_lobe(:,:) = int_lobe/maxval(int_lobe) ! ,1,.true.)
      !
      ! 2) Compute pseudo-visibilities
      ! ------------------------------
      !
      write(mess,'(A,I0)') 'Filtering field ',if
      call map_message(seve%i,rname,mess)
      !
      ! Start loop on channels
      do i = 1,nc
        !
        ! FT of SD image
        call retocm (lmv_data(:,:,i),ws_data(:,:,i),nx,ny)
        call fourt  (ws_data(:,:,i),nn,ndim,1,1,fftws)
        !
        ! SD beam correction (in uv plane)
        if (do_single) then
          ws_data(:,:,i) = ws_data(:,:,i)*sd_lobe
        endif
        !
        ! Apply interferometer primary beam (in image plane)
        if (do_primary) then
          call fourt  (ws_data(:,:,i),nn,ndim,-1,1,fftws)
          call uvshort_prmult (ws_data(:,:,i),int_lobe,nx,ny)
          call fourt  (ws_data(:,:,i),nn,ndim,1,1,fftws)
          ws_data(:,:,i) = ws_data(:,:,i)/(nx*ny)
        endif
        !
        ! End loop on channels
      enddo
      !
      ! 3) Compute weights
      ! ------------------
      !
      ! Weights: do something
      !
      if (weight_mode.eq.'UN') then
        gr_uv_w = 1.
      else
        call retocm (gr_im_w,gr_uv_w,nx,ny)
        call fourt  (gr_uv_w,nn,ndim,1,1,fftws)
        gr_uv_w(:,:) = gr_uv_w/(nx*ny)
      endif
      if (do_single) then
!        gr_uv_w = 1
!        call wcorr(gr_uv_w,sd_lobe,nx*ny)
        where (sd_lobe.ne.0)
          gr_uv_w = 1.0/sd_lobe**2
        else where
          gr_uv_w = 0.0
        end where
      endif
      !
      ! 4) Create uv table
      ! ------------------
      !
      ! Compute number of visibilities < uv_trunc
      call uvshort_uvcount(nx,ny,nvis,uv_trunc,lmv)
      !
      ! Output uv table
      n = len_trim(uv_table)
      name  = uv_table(1:n)
      if (nf.gt.1) then
        name(n+1:) = '-'
        n = n+2
        write(name(n:),'(i0)') if
      endif
      call sic_parsef(name,uvt%file,' ','.uvt')
      call map_message(seve%i,rname,'Creating UV table '//trim(uvt%file))
      !
      call uvshort_fill(lmv,uvt,error,nvis,nc,raoff(if),deoff(if),positions,nu)
      !
      ! Allocate memory for uv table
      allocate(uvt_data(uvt%gil%dim(1),uvt%gil%dim(2)),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%e,rname,'Cannot allocate memory for UV table')
        error = .true.
        return
      endif
      !
      ! Fill in uv table
      call uvshort_uvtable(nx,ny,nu,nc,ws_data,uvt_data,gr_uv_w,&
        &  nvis, uv_trunc, sd_weight, sd_factor,lmv)
      !
      ! Write uv table
      call gdf_write_image(uvt,uvt_data,error)
      if (error) then
        call map_message(seve%e,rname,'Error writing UV table')
        return
      endif
      write(mess,'(I0,A)') nvis,' visibilities written'
      call map_message(seve%i,rname,mess)
      deallocate(uvt_data)
      !
      ! End loop on mosaic fields
    enddo
    !
  else  ! short_mode # 0
    !
    ! Single UV Table as output
    ! The output UV Header is taken from input UV table
    ! Matching with the Class Table file must be checked
    !
    call uv_short_consistency(rname,nc,uvt,lmv,tole,error)
    if (error) return
    !
    cs = [1.0,0.0]
    !
    if (short_mode.lt.0) then
      ! short_mode < 0
      !   Here we create a single UV table containing only
      !   the short spacings  for all fields
      name = trim(uv_table)//"-short"
      call sic_parsef(name,uvt%file,' ','.uvt')
    else
      ! short_mode > 0
      !   Here we create a UV table containing the short spacings
      !   for all fields appended to the previous UV table
      call gildas_null(hin)
      name = trim(uv_table)
      call sic_parsef(name,hin%file,' ','.uvt')
      call gdf_read_header(hin,error)
      if (error) return
      !
      name = trim(uv_table)//"-merged"
      call sic_parsef(name,uvt%file,' ','.uvt')
      call map_message(seve%i,rname,'UV Table to be appended is '//trim(uvt%file))
      wcol = (nc+2)/3
      umax = sd_diam
      call uvshort_copy_uvtable(rname,hin,uvt,umax,wcol,weight,error)
      if (error) return
      !
      ! Scaling factor for Weights
      ! It attempts to match the weight density found in the UV table
      ! up to the Single Dish diameter, which is OK for 30-m / NOEMA
      ! pair with compact configuration included, but not for other cases.
      !   It will need to be refined, e.g. by computing the weight density
      ! from the shortest baseline up to the shortest baseline + IP dish diameter
      !   In addition, SD_WEIGHT$ provides a user controlled fudge factor,
      ! with 1 as a reasonable default
      scale_weight = sd_weight/4.0/(sd_diam/uv_trunc)**2
      write(mess,'(A,1PG11.4)') 'Weight scaling factor ',scale_weight
      call map_message(seve%i,rname,mess)
    endif
    !
    if (abs(short_mode).eq.code_short_phase) then
      if (precise) then
        nprec = nc
      else
        nprec = 1
      endif
      allocate(rpos(2,nprec),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%e,rname,'cannot allocate phase shift memory')
        error = .true.
        return
      endif
    endif
    !
    !
    do if=1,nf
      !
      ! 1) Compute interferometer primary beam
      ! --------------------------------------
      !
      ! 1.a) Int. Primary Beam = gaussian, computed in image plane
      !
      ! call uvshort_doprim(ip_beam,int_lobe,nx,ny,raoff(if),deoff(if),lmv)
      !
      ! 1.b) Int. Primary Beam = truncated gaussian in the uv plane then FT
      !
      call uvshort_dointft(ip_beam,ip_diam,int_lobe,nx,ny,1.0,lmv)
      int_lobe_comp(:,:) = cmplx(int_lobe)
      call uvshort_shift(int_lobe_comp,nx,ny,raoff(if),deoff(if),lmv)
      call fourt(int_lobe_comp,nn,ndim,1,1,fftws)
      call cmtore(int_lobe_comp,int_lobe,nx,ny)
      !
      ! Normalize peak to 1
      !
      int_lobe(:,:) = int_lobe/maxval(int_lobe)
      !
      ! 2) Compute pseudo-visibilities
      ! ------------------------------
      !
      write(mess,'(A,i0,A,i0)') 'Filtering  field ',if,' / ',nf
      call map_message(seve%i,rname,mess)
      !
      ! Start loop on channels
      do i = 1,nc
        !
        ! FT of SD image
        call retocm (lmv_data(:,:,i),ws_data(:,:,i),nx,ny)
        call fourt  (ws_data(:,:,i),nn,ndim,1,1,fftws)
        !
        ! SD beam correction (in uv plane)
        if (do_single) then
          ws_data(:,:,i) = ws_data(:,:,i)*sd_lobe
        endif
        !
        ! Apply interferometer primary beam (in image plane)
        if (do_primary) then
          call fourt  (ws_data(:,:,i),nn,ndim,-1,1,fftws)
          call uvshort_prmult (ws_data(:,:,i),int_lobe,nx,ny)
          call fourt  (ws_data(:,:,i),nn,ndim,1,1,fftws)
          ws_data(:,:,i) = ws_data(:,:,i)/(nx*ny)
        endif
        !
      enddo        ! End loop on channels
      !
      ! 3) Compute weights
      ! ------------------
      !
      ! Weights: do something
      if (weight_mode.eq.'UN') then
        gr_uv_w = 1.
      else
        call retocm (gr_im_w,gr_uv_w,nx,ny)
        call fourt  (gr_uv_w,nn,ndim,1,1,fftws)
        gr_uv_w(:,:) = gr_uv_w/(nx*ny)
      endif
      if (do_single) then
!        gr_uv_w = 1
!        call wcorr(gr_uv_w,sd_lobe,nx*ny)
        where (sd_lobe.ne.0)
          gr_uv_w = 1.0/sd_lobe**2
        else where
          gr_uv_w = 0.0
        end where
      endif
      !
      ! 4) Create or Append uv table
      ! ----------------------------
      !
      ! Compute number of visibilities < uv_trunc
      call uvshort_uvcount(nx,ny,nvis,uv_trunc,lmv)
      !
      uvt%blc(1) = 1
      uvt%trc(1) = uvt%gil%dim(1)
      !
      if (if.eq.1 .and. short_mode.lt.0) then
        call map_message(seve%i,rname,'Creating UV table '//trim(uvt%file))
        uvt%gil%nvisi = nvis
        uvt%gil%dim(2) = nvis
        call gdf_create_image(uvt,error)
        uvt%blc(2) = 1
        uvt%trc(2) = nvis
        string = 'written'
      else
        if (if.eq.1) then
          call map_message(seve%i,rname,'Appending UV table '//trim(uvt%file))
          string = 'appended'
        else
          call gdf_close_image(uvt,error)
        endif
        uvt%blc(2) = uvt%gil%dim(2)+1
        mvis = uvt%gil%dim(2)+nvis      ! Add nvis visibilities
        uvt%trc(2) = mvis
        call gdf_extend_image(uvt,mvis,error)
        uvt%gil%nvisi = uvt%gil%dim(2)
      endif
      !
      ! Allocate memory for UV table
      nu = uvt%gil%dim(1)
      allocate(uvt_data(nu,nvis),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%i,rname,'Cannot allocate memory for UV table ')
        error = .true.
        return
      endif
      !
      ! Fill in uv table
      if (short_mode.gt.0) then
        ! Scale the weight to have a reasonable beam.
        sd_weight = weight*scale_weight
      endif
      call uvshort_uvtable(nx,ny,nu,nc,ws_data,uvt_data,gr_uv_w,&
        &    nvis, uv_trunc, sd_weight, sd_factor, lmv)
      !
      if (abs(short_mode).eq.code_short_phase) then
        !
        ! Shift phase center to current pointing center
        if (nprec.gt.1) then
          do i=1,nprec
            freq = gdf_uv_frequency(uvt,dble(i))
            rpos(1,i) = freq * f_to_k * raoff(if)
            rpos(2,i) = freq * f_to_k * deoff(if)
          enddo
        else
          freq = gdf_uv_frequency(uvt)
          rpos(1,1) = freq * f_to_k * raoff(if)
          rpos(2,1) = freq * f_to_k * deoff(if)
        endif
        rpos = -rpos    ! Use the correct phase shift sign
        !
        ! Initialize the loff & moff columns
        uvt_data(loff,:) = 0
        uvt_data(moff,:) = 0
        !
        ! then shift the UV data set
        call shift_uvdata (uvt,nu,nvis,uvt_data,cs,nprec,rpos)
        !
        ! an only ultimately put the final loff & moff values
        uvt_data(loff,:) = raoff(if)
        uvt_data(moff,:) = deoff(if)
      else
        ! Put the xoff & yoff columns
        uvt_data(xoff,:) = raoff(if)
        uvt_data(yoff,:) = deoff(if)
      endif
      !
      call gdf_write_data(uvt,uvt_data,error)
      if (error) then
        call map_message(seve%e,rname,'Error writing UV data ')
        return
      endif
      write(mess,'(I0,A,I0)') nvis,' visibilities '//string//' for field ',if
      call map_message(seve%i,rname,mess)
      !
      deallocate(uvt_data)
    enddo
    ! Finish the job. Update the number of visibilities
    call gdf_update_header(uvt,error)
    call gdf_close_image(uvt,error)
  endif
  !
  ! Write output lmv image if needed
  if (.not.lmv_file) then
    call gdf_write_image(lmv,lmv_data,error)
    if (error) then
      call map_message(seve%w,rname,'Error writing LMV image')
      return
    endif
  endif
  !
  ! Delete scratch spaces
  deallocate(lmv_data)
  deallocate(ws_data,fftws,gr_im_w,gr_uv_w,sd_lobe,int_lobe)
  !
  if (.not.do_single) then
    call map_message(seve%w,rname,' ')
    call map_message(seve%w,rname,'  *** DO_SINGLE is not set *** Test mode only !')
    call map_message(seve%w,rname,' ')
  endif
  if (.not.do_primary) then
    call map_message(seve%w,rname,' ')
    call map_message(seve%w,rname,'  *** DO_PRIMARY is not set *** Test mode only !')
    call map_message(seve%w,rname,' ')
  endif
end subroutine uvshort_sub
!
subroutine uvshort_create_lmv(rname,table,lmv,lmv_data,gr_im_w,error, &
  & uvt,map_name,xcol,ycol,wcol,mcol,sd_beam,tole,chra,chde,minw)
  use gbl_message
  use image_def
  use gkernel_interfaces
  use mapping_interfaces
  !
  ! Task UV_SHORT
  !   Internal routine
  !
  !   Creates a "well behaved" Single Dish map from a Single Dish table
  !   for derivation of the short spacings of an interferometer mosaic
  !
  !   Use convolution by a fraction of the beam, and smooth extrapolation
  !   to zero beyond the mosaic edge
  !
  character(len=*), intent(in) :: rname   ! Caller's task name
  character(len=*), intent(in) :: table   ! CLASS table name
  type(gildas), intent(inout) :: lmv      ! LMV data set
  type(gildas), intent(inout) :: uvt      ! UVT data set
  real, allocatable, intent(out) :: lmv_data(:,:,:)   ! lmv data
  real, allocatable, intent(out) :: gr_im_w(:)        ! Weights
  logical, intent(out) :: error        ! Error flag
  character(len=*), intent(in) :: map_name  ! Map name
  integer, intent(in) :: xcol          ! X offset column
  integer, intent(in) :: ycol          ! Y offset column
  integer, intent(in) :: wcol          ! Weight column
  integer, intent(inout) :: mcol(2)    ! Mapped columns
  real, intent(in) :: sd_beam          ! Single dish beam
  real, intent(in) :: tole             ! Position tolerance
  character(len=*), intent(in) :: chra ! RA string
  character(len=*), intent(in) :: chde ! Declination string
  real, intent(inout) :: minw          ! Position tolerance
  !
  real(8), parameter :: pi=3.141592653589793d0
  !
  character(len=filename_length) :: name
  type(gildas) :: sdt             ! Single Dish data table
  real, allocatable :: sdt_data(:,:)
  integer :: np, nd, nc, lcol, ocol, nx, ny, nxmore, nymore
  integer :: ier, i
  real, allocatable :: sdw(:)     ! weights table
  real, allocatable :: rawcube(:) ! work space for sdt gridded values
  real, allocatable :: xcoord(:)  ! lmv X axis gridded coordinates
  real, allocatable :: ycoord(:)  ! lmv Y axis gridded coordinates
  real :: xmin,xmax,ymin,ymax
  real(kind=8) :: xconv(3),yconv(3),new(2),old(2),tmp
  real :: smooth
  integer ctypx,ctypy
  real maxw,xparm(10),yparm(10),support(2),cell(2)
  character(len=256) :: mess
  !
  call gildas_null(sdt)
  !
  name = trim(table)
  call sic_parsef(name,sdt%file,' ','.tab')
  !
  ! Read sdt header and check format
  !
  call gdf_read_header(sdt,error)
  if (gildas_error(sdt,rname,error)) then
    call map_message(seve%f,rname,'Cannot read header from Single Dish table')
    return
  endif
  if (sdt%gil%form.ne.fmt_r4) then
    call map_message(seve%f,rname,'Only real format supported')
    error = .true.
    return
  endif
  !
  nd = sdt%gil%dim(1)
  np = sdt%gil%dim(2)
  !
  ! Check xcol,ycol input parameters
  if ((xcol.gt.nd).or. (ycol.gt.nd)) then
    call map_message(seve%e,rname,'X or Y column does not exist')
    error = .true.
    return
  endif
  !
  ! Allocate memory space and read data
  allocate(sdt_data(nd,np), stat=sdt%status)
  if (gildas_error(sdt,rname,error)) then
    call map_message(seve%e,rname,'Cannot allocate memory for Single Dish table')
    return
  endif
  !
  call gdf_read_data(sdt,sdt_data,error)
  if (gildas_error(sdt,rname,error)) then
    call map_message(seve%e,rname,'Cannot read data from Single Dish table')
    return
  endif
  !
  ! Allocate memory for weight table
  allocate(sdw(2*max(np,nd)),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Cannot allocate memory for weigths table')
    error = .true.
    return
  endif
  !
  !-----------------------------------------------------------------------
  !
  ! Arrange data
  ! ------------
  !
  ! Recompute offsets if reference position is to be modified
  if (chra.eq.'0' .and. chde.eq.'0') then
    continue
  else
    if (chra.eq.' ' .and. chde.eq.' ') then
      new(1) = uvt%gil%a0
      new(2) = uvt%gil%d0
    else
      print *,'RA  '//chra
      print *,'Dec '//chde
      call sic_decode(chra,new(1),24,error)
      if (error) return
      call sic_decode(chde,new(2),360,error)
      if (error) return
    endif
    ! Declinations should be between [-pi/2,pi/2] and RA between [0,2*pi]
    if (new(1).lt.0 .or.new(1).gt.2d0*pi) then
       call map_message(seve%w,rname,'RA value outside [0,2*pi] => Using modulo')
       new(1) = modulo(new(1),2d0*pi)
    endif
    if (new(2).lt.-0.5d0*pi .or.new(2).gt.0.5d0*pi) then
       call map_message(seve%e,rname,'Dec value outside [-pi/2,pi/2] => Using modulo')
       new(2) = modulo(new(2)+0.5d0*pi,pi)-0.5d0*pi
    endif
    !
    old(1) = sdt%gil%a0
    old(2) = sdt%gil%d0
    !
    ! Recompute offsets
    !
    call uvshort_dopoint (sdt_data,nd,np,xcol,ycol,old,new)
    sdt%gil%a0 = new(1)
    sdt%gil%d0 = new(2)
  endif
  !
  ! Set the sinus type projection
  sdt%gil%ptyp = p_azimuthal
  !
  ! Order sdt_data with Y in increasing order
  ! sdw is used as work space only
  call uvshort_dosor (rname,sdt_data,nd,np,sdw,ycol,error)
  if (error) return
  !
  ! Read weights in sdw table
  call uvshort_dowei (sdt_data,nd,np,sdw,wcol)
  !
  !-----------------------------------------------------------------------
  !
  ! Find lmv image size
  ! -------------------
  !
  ! Find min,max offsets of the SD observations
  ! (subroutine working only on Y ordered array)
  !
  call uvshort_finsiz (sdt_data,nd,np,xcol,ycol,sdw,xmin,xmax,ymin,ymax)
  !
  ! Find increment: 4 pixels per SD beam
  !
  xconv(3) = -sd_beam/4.0
  yconv(3) = sd_beam/4.0
  !
  ! ! Old code: used if sd_beam = 0
  ! ! pixel size = min distance between SD points
  ! xinc = xmax-xmin
  ! yinc = ymax-ymin
  ! call fininc (sdt_data,nd,np,xcol,ycol,sdw,xinc,yinc,tole)
  ! xconv(3) = -xinc
  ! yconv(3) = +yinc
  !
  ! Find size of output SD image
  !
  nx = 2 * max ( nint(abs(xmax/xconv(3))+1), nint(abs(xmin/xconv(3))+1) )
  ny = 2 * max ( nint(abs(ymax/yconv(3))+1), nint(abs(ymin/yconv(3))+1) )
  nxmore = nint(4*sd_beam/abs(xconv(3)))+1
  nymore = nint(4*sd_beam/abs(yconv(3)))+1
  nx = nx+2*nxmore
  ny = ny+2*nymore
  !
  ! Extend nx,ny to nearest power of two
  i = 32
  do while(i.lt.nx)
    i = i*2
  enddo
  nx = i
  i = 32
  do while(i.lt.ny)
    i = i*2
  enddo
  ny = i
  !
  ! Reference position for lmv cube header
  !
  xconv(1) = nx/2+1
  xconv(2) = 0.0
  yconv(1) = ny/2+1
  yconv(2) = 0.0
  !
  tmp = 0.1*nint(yconv(3)*10*180*3600/pi)
  write(mess,'(A,I0,A,I0,A)') 'Creating a cube with ',nx,' by ',ny,' pixels'
  call map_message(seve%i,rname,mess)
  write(mess,'(A,F8.3,A)') 'Pixel size: ',tmp,' arcsec'
  call map_message(seve%i,rname,mess)
  !
  ! Warn for big images
  if (nx.gt.512 .or. ny.gt.512) then
    if (nx.gt.8192 .or. ny.gt.8192) then
      call map_message(seve%e,rname,'More than 8192 pixels in X or Y')
      error = .true.
    else
      call map_message(seve%e,rname,'More than 512 pixels in X or Y')
    endif
    write(mess,*) 'Offset extrema are: ',xmin,xmax,ymin,ymax
    call map_message(seve%i,rname,mess)
    write(mess,*) 'Pixel sizes are', xconv(3),yconv(3)
    call map_message(seve%i,rname,mess)
    if (error) return
  endif
  !
  ! Number of channels
  if (mcol(2).eq.0) mcol(2) = nd
  mcol(1) = max(1,min(mcol(1),nd))
  mcol(2) = max(1,min(mcol(2),nd))
  ocol = min(mcol(1),mcol(2))       ! first channel to grid
  lcol = max(mcol(1),mcol(2))       ! last channel to grid
  nc = lcol-ocol+1
  write(mess,'(A,I0,A,I0,A,I0,A)') 'Creating ',nc,' channels from [',mcol(1),',',mcol(2),']'
  call map_message(seve%i,rname,mess)
  ocol = ocol-1 ! Because it is used as such in "doconv"
  !
  !-----------------------------------------------------------------------
  !
  ! Create lmv image
  ! ----------------
  !
  ! Copy header from input SD table
  call gdf_copy_header(sdt,lmv,error)
  !
  ! Create image (in order l m v)
  name = trim(map_name)
  call sic_parsef(name,lmv%file,' ','.lmv')
  call map_message(seve%i,rname,'Creating map file '//trim(lmv%file))
  !
  ! Fill in header
  lmv%gil%ndim = 3
  lmv%gil%dim(1) = nx
  lmv%gil%dim(2) = ny
  lmv%gil%dim(3) = nc
  lmv%gil%dim(4) = 1
  lmv%loca%size = nx*ny*nc
  lmv%gil%ref(3) = sdt%gil%ref(1)-ocol
  lmv%gil%val(3) = lmv%gil%voff
  lmv%gil%inc(3) = lmv%gil%vres
  lmv%gil%convert(:,1) = xconv
  lmv%gil%convert(:,2) = yconv
  lmv%char%code(1) = sdt%char%code(2)
  lmv%char%code(2) = sdt%char%code(3)
  ! Patch a bug in Header transmission:
  !   the table has only 2 dimensions, so the 3rd axis is undefined
  if (len_trim(lmv%char%code(2)).eq.0) lmv%char%code(2)='DEC'
  lmv%char%code(3) = sdt%char%code(1)
  lmv%gil%coor_words = 6*gdf_maxdims             ! not a table
  lmv%gil%extr_words = 0                   ! extrema not computed
  lmv%gil%xaxi = 1                         ! reset projected axis
  lmv%gil%yaxi = 2
  lmv%gil%faxi = 3
  lmv%gil%form = fmt_r4
  !
  !-----------------------------------------------------------------------
  !
  ! Memory allocations
  ! ------------------
  !
  error = .true.
  allocate(lmv_data(lmv%gil%dim(1),lmv%gil%dim(2),lmv%gil%dim(3)),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Cannot allocate memory for output lmv image')
    return
  endif
  allocate(rawcube(nc*nx*ny),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Cannot allocate memory for raw cube lmv')
    return
  endif
  allocate(gr_im_w(2*nx*ny),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Cannot allocate memory for work space')
    return
  endif
  allocate(xcoord(nx),ycoord(ny),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Cannot allocate memory for lmv X or Y axis')
    return
  endif
  error = .false.
  !
  !-----------------------------------------------------------------------
  !
  ! Gridding of the SD data
  ! -----------------------
  !
  ! Resampling in space of the original spectra on a regular grid,
  ! using a convolution kernel
  !
  ! Compute gridding function: a small (1/3 of SD beam) gaussian
  ! Note: since xconv and yconv depends on SD beam, all gridding
  ! parameters do depend on SD beam value...
  !
  smooth = sd_beam/3.0
  ctypx = 2
  ctypy = 2
  support(1) = 5*smooth              ! go far enough...
  support(2) = 5*smooth
  xparm(1) = support(1)/abs(xconv(3))
  yparm(1) = support(2)/abs(yconv(3))
  xparm(2) = smooth/(2*sqrt(log(2.0)))/abs(xconv(3))
  yparm(2) = smooth/(2*sqrt(log(2.0)))/abs(yconv(3))
  xparm(3) = 2
  yparm(3) = 2
  !
  call convfn (ctypx, xparm, ubuff, ubias)
  call convfn (ctypy, yparm, vbuff, vbias)
  cell(1) = xconv(3)
  cell(2) = yconv(3)
  !
  ! Compute grid coordinates
  xcoord(:) = (/(real((dble(i)-lmv%gil%ref(1))*lmv%gil%inc(1)+lmv%gil%val(1)),i=1,nx)/)
  ycoord(:) = (/(real((dble(i)-lmv%gil%ref(2))*lmv%gil%inc(2)+lmv%gil%val(2)),i=1,ny)/)
  !
  ! Grid data: output = rawcube
  ! **WW** poids sdw --> poids gr_im_w = pour chaque pixel, somme des
  !        [poids des points pris en compte * gaussienne]
  !        (NON normalise)
  !        --> a normaliser par somme des gaussiennes ('results')?
  !
  gr_im_w = 0.
  call uvshort_doconv (&
    & nd,np,&                             ! number of input points
    & sdt_data,&                          ! input values
    & xcol,ycol,ocol,&                    ! pointers to special values
    & sdw,&                               ! weights
    & gr_im_w,&                           ! gridded weights
    & nc,nx,ny,&                          ! cube size
    & rawcube,&                           ! gridded data (output)
    & xcoord,ycoord,&                     ! cube coordonates
    & support,cell,maxw)
  !
  ! Min and max weights
  minw = maxw*minw
  call map_message(seve%i,rname,'Done Single Dish table gridding')
  !
  !
  !-----------------------------------------------------------------------
  !
  ! Extrapolation to zero outside the convex hull of the mapped region
  ! ------------------------------------------------------------------
  !
  ! Compute smoothing function = SD primary beam
  !
  ctypx = 2
  ctypy = 2
  support(1) = 3*sd_beam
  support(2) = 3*sd_beam
  xparm(1) = support(1)/abs(xconv(3))
  yparm(1) = support(2)/abs(yconv(3))
  xparm(2) = sd_beam/(2*sqrt(log(2.0)))/abs(xconv(3))
  yparm(2) = sd_beam/(2*sqrt(log(2.0)))/abs(yconv(3))
  xparm(3) = 2
  yparm(3) = 2
  !
  call convfn (ctypx, xparm, ubuff, ubias)
  call convfn (ctypy, yparm, vbuff, vbias)
  cell(1) = xconv(3)
  cell(2) = yconv(3)
  !
  ! Smooth data by gaussian = SD beam
  !       input = rawcube
  !       output = lmv_data (used as work space)
  !
  ! Gridded weights are in the call sequence, but unused 
  call uvshort_dosmoo (&
    & rawcube,&                   ! raw gridded values
    & gr_im_w,&                   ! gridded weights
    & nc,nx,ny,&                  ! cube size
    & lmv_data,&                  ! smoothed cube
    & xcoord,ycoord,&             ! cube coordinates
    & support,cell)
  call map_message(seve%i,rname,'Done image smoothing')
  !
  ! Apodisation: - smooth image (lmv_data) is apodised by gaussian
  !                at image edges
  !              - replace input image (rawcube) by smoothed-apodisated
  !                image at map edges
  !              - replace input image (rawcube) by smoothed image
  !                at points where weights < minw
  ! Output = rawcube
  !
  ! Gridded weights are used but not modified
  call uvshort_doapod (xmin,xmax,ymin,ymax,tole,sd_beam,&
    & nc,nx,ny,&
    & lmv_data,&              ! input smoothed cube
    & rawcube,&               ! output after apodisation
    & xcoord,ycoord,&         ! cube coordinates
    & gr_im_w, minw)
  call map_message(seve%i,rname,'Done image apodisation')
  !
  ! Transpose to the lmv order and put the result in lmv_data
  call uvshort_dotrans(rawcube,lmv_data,nc,nx*ny)
  call map_message(seve%i,rname,'Done transposition to lmv order')
  !
  ! Free memory
  deallocate(sdt_data,sdw,rawcube,xcoord,ycoord)
end subroutine uvshort_create_lmv
!
subroutine uvshort_convol (du,dv,resu)
  !
  ! Compute convolving factor resu
  ! resu is the result of the multiplication of the convolution functions,
  ! for u and v axes, at point (du,dv)
  !
  ! Convolution function is defined in ubuff & ubias and in vbuff & vbias,
  ! by CONTAIN association in the main program
  !
  real, intent(out) :: resu
  real, intent(in) :: du,dv
  !
  ! Local variables
  integer iu,iv
  !
  ! convolving functions values are tabulated every 1/100 cell
  iu = nint(100.0*du+ubias)
  iv = nint(100.0*dv+vbias)
  !
  ! Participation of u and v axes convolution functions
  resu = ubuff(iu)*vbuff(iv)
  if (resu.lt.1e-20) resu = 0.0
end subroutine uvshort_convol
!
!
subroutine uvshort_read_lmv(rname,table,lmv,lmv_data,gr_im_w,error)
  use gbl_message
  use image_def
  use gkernel_interfaces
  use mapping_interfaces, only : map_message
  ! Task UV_SHORT
  !   Internal routine
  !
  ! Read an existing LMV data cube to be used to compute
  ! short spacings.
  !
  character(len=*), intent(in) :: rname   ! Caller's name
  character(len=*), intent(in) :: table   ! File name
  type(gildas), intent(inout) :: lmv      ! Cube header
  real, allocatable, intent(out) :: lmv_data(:,:,:) ! Cube data
  real, allocatable, intent(out) :: gr_im_w(:)      ! Weight array
  logical, intent(out) :: error           ! Error flag
  !
  character(len=filename_length) :: name
  character(len=256) :: mess
  integer :: ier
  !
  name = trim(table)
  call sic_parsef(name,lmv%file,' ','.lmv')
  !
  call gdf_read_header(lmv,error)
  if (gildas_error(lmv,rname,error)) then
    call map_message(seve%e,rname,'Cannot read header from input cube lmv')
    return
  endif
  !
  ! Allocate memory
  allocate(lmv_data(lmv%gil%dim(1),lmv%gil%dim(2),lmv%gil%dim(3)),stat=lmv%status)
  if (gildas_error(lmv,rname,error)) then
    call map_message(seve%e,rname,'Cannot allocate memory for output lmv image')
    return
  endif
  !
  call gdf_read_data(lmv,lmv_data,error)
  if (gildas_error(lmv,rname,error)) then
    call map_message(seve%e,rname,'Cannot read data from input cube lmv')
    return
  endif
  !
  ! Update image lmv variables
  if (any(lmv%gil%dim(1:3).eq.0)) then
    call map_message(seve%e,rname,'Inconsistent input lmv cube dimensions')
    error = .true.
    return
  endif
  !
  ! What will we do about weights ?  Set them to 1.0 (Uniform...)
  allocate(gr_im_w(2*lmv%gil%dim(1)*lmv%gil%dim(2)),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Cannot allocate memory for work space')
    error = .true.
    return
  endif
  gr_im_w=1.0
  !
  write(mess,'(A,I0,A,I0)') 'Image cube lmv with ',lmv%gil%dim(1),' by ',lmv%gil%dim(2),' pixels'
  call map_message(seve%i,rname,mess)
  write(mess,*) 'Pixel sizes are', lmv%gil%inc(1),lmv%gil%inc(2)
  call map_message(seve%i,rname,mess)
end subroutine uvshort_read_lmv
!
!=========================================================================
!
subroutine uvshort_doconv (nd,np,visi,jx,jy,jo,we,gwe,&
  &   nc,nx,ny,map,mapx,mapy,sup,cell,maxw)
  !-------------------------
  ! Task UV_SHORT
  !   Internal routine
  !
  ! Convolution of 'map' by function defined by vbuff & vbias
  ! (via subroutine uvshort_convol, available by CONTAIN association).
  ! Used for gridding of Single Dish data.
  !
  integer, intent(in) :: np              ! number of visibilities
  integer, intent(in) :: nd              ! visibility size
  integer, intent(in) :: nc              ! number of channels
  integer, intent(in) :: nx,ny           ! map size
  integer, intent(in) :: jx,jy           ! x coord, y coord location in visi
  real, intent(in) :: we(np)             ! weights
  integer, intent(in) :: jo              ! offset for data in visi
  real, intent(in) :: visi(nd,np)        ! values
  real, intent(out) :: gwe(nx,ny)        ! gridded weights
  real, intent(out) :: map(nc,nx,ny)     ! gridded values
  real, intent(in) :: mapx(nx),mapy(ny)  ! coordinates of grid
  real, intent(in) :: sup(2)             ! support of convolving function in user units
  real, intent(in) :: cell(2)            ! cell size in user units
  real, intent(out) :: maxw              ! maximum weight
  !
  ! Local variables
  !
  integer ifirs,ilast                ! range to be considered
  integer ix,iy,i
  real result,weight
  real u,v,du,dv,um,up,vm,vp
  !
  ! Code
  !
  maxw = 0.0
  !
  ! Loop on Y
  !
  ifirs = 1
  do iy=1,ny
    v = mapy(iy)
    !
    ! sup is the support of the gridding function
    vm = v-sup(2)
    vp = v+sup(2)
    !
    ! Find points to be considered.
    ! Optimized dichotomic search, taking into account the
    ! fact that mapy is an ordered array.
    call uvshort_findr (np,nd,jy,visi,vm,ifirs)
    ilast = ifirs
    call uvshort_findr (np,nd,jy,visi,vp,ilast)
    ilast = ilast-1
    !
    ! Initialize x column
    map(1:nc,1:nx,iy) = 0.0
    !
    if (ilast.ge.ifirs) then
      !
      ! Loop on x cells
      do ix=1,nx
        u = mapx(ix)
        um = u-sup(1)
        up = u+sup(1)
        weight = 0.0
        !
        ! Loop on relevant data points
        do i=ifirs,ilast
          !
          ! Test if X position is within the range to be
          ! considered
          if (visi(jx,i).ge.um .and. visi(jx,i).le.up) then
            !
            ! Compute convolving factor
            du = (u-visi(jx,i))/cell(1)
            dv = (v-visi(jy,i))/cell(2)
            call uvshort_convol (du,dv,result)
            if (result.ne.0.0) then
              !
              ! Do the convolution: map(pixel) = sum of
              ! relevant values * convolving factor * weight
              result = result*we(i)
              weight = weight + result
              map (1:nc,ix,iy) = map (1:nc,ix,iy) + visi((1+jo):(nc+jo),i)*result
            endif
          endif
        enddo
        !
        ! gwe is the sum of the (convolving factor * weight) ie
        ! the sum of the weighting factors applied to the data
        gwe(ix,iy) = weight
        maxw = max(maxw,weight)
        !
        ! Normalization (only in cells where some data exists)
        if (weight.ne.0) then
          map (1:nc,ix,iy) = map(1:nc,ix,iy)/weight
        endif
      enddo
    endif
  enddo
end subroutine uvshort_doconv
!
!---------------------------------------------------------------------------------------
!
subroutine uvshort_dosmoo (raw,we,nc,nx,ny,map,mapx,mapy,sup,cell)
  ! Task UV_SHORT
  !   Internal routine
  !
  ! Smooth an input data cube raw in vlm along l and m by convolution
  ! by a gaussian function defined in vbuff & vbias via subroutine
  !   uvshort_convol  (available by CONTAIN association)
  !
  integer, intent(in) :: nc,nx,ny          ! map size
  real, intent(in) :: we(nx,ny)            ! weights
  real, intent(in) :: raw(nc,nx,ny)        ! raw map
  real, intent(out) :: map(nc,nx,ny)       ! smoothed map
  real, intent(in) :: mapx(nx),mapy(ny)    ! coordinates of grid
  real, intent(in) :: sup(2)               ! support of convolving function in user units
  real, intent(in) :: cell(2)              ! cell size in user units
  !
  ! Local variables
  integer yfirs,ylast                ! range to be considered
  integer xfirs,xlast                ! range to be considered
  integer ix,iy
  integer jx,jy                      ! x coord, y coord location in raw
  real result,weight
  real u,v,du,dv,um,up,vm,vp,dx,dy
  !
  ! Code
  dx = abs(mapx(2)-mapx(1))
  dy = abs(mapy(2)-mapy(1))
  !
  ! Loop on y rows
  do iy=1,ny
    !
    ! Compute extrema positions on axe y
    ! of gaussian function centered on map(1:nc,ix,iy)
    v = mapy(iy)
    vm = v-sup(2)
    vp = v+sup(2)
    !
    ! Compute extrema positions on axe y
    ! of relevant data points for map(1:nc,ix,iy) convolution
    yfirs = max(1,nint((iy-sup(2)/dy)))
    ylast = min(ny,nint((iy+sup(2)/dy)))
    !
    ! Initialize x colum
    map(1:nc,1:nx,iy) = 0.0
    !
    ! Loop on x cells
    !
    if (yfirs.le.ylast) then
      do ix=1,nx
        !
        ! Compute extrema positions on axe x, idem y
        u = mapx(ix)
        um = u-sup(1)
        up = u+sup(1)
        weight = 0.0
        xfirs = max(1,nint(ix-sup(1)/dx))
        xlast = min(nx,nint(ix+sup(1)/dx))
        !
        ! Loop on relevant data points
        if (xfirs.le.xlast) then
          do jy=yfirs,ylast
            dv = (v-mapy(jy))/cell(2)
            do jx=xfirs,xlast
              du = (u-mapx(jx))/cell(1)
              !
              ! Compute convolving factor
              call uvshort_convol (du,dv,result)
              if (result.ne.0.0) then
                !
                ! Do the convolution: map(pixel) = sum of
                ! relevant values * convolving factor
                !
                weight = weight + result
                map (1:nc,ix,iy) = map (1:nc,ix,iy) + raw(1:nc,jx,jy)*result
              endif
            enddo
          enddo
          !
          ! Normalize weight only in cells where some data exists...
          if (weight.ne.0) then
            map (1:nc,ix,iy) = map(1:nc,ix,iy)/weight
          endif
        endif
      enddo
    endif
  enddo
end subroutine uvshort_dosmoo
!
!-----------------------------------------------------------------------
!
subroutine uvshort_dowei (visi,nd,np,we,iw)
  ! Task UV_SHORT
  !   Internal routine
  ! Fill in weights array from the input table
  !
  integer, intent(in) :: nd       ! Visibility size
  integer, intent(in) :: np       ! Number of visibilities
  integer, intent(in) :: iw       ! Weight column
  real, intent(in) :: visi(nd,np) ! Visibilities
  real, intent(out) :: we(np)     ! Weight values
  !
  ! Code
  if (iw.le.0 .or. iw.gt.nd) then
    !
    ! Weight column does not exist...
    we(1:np) = 1.0
  else
    !
    ! Weight colum do exist
    we(1:np) = visi(iw,1:np)
  endif
end subroutine uvshort_dowei
!
!-----------------------------------------------------------------------
!
subroutine uvshort_findr (nv,nc,ic,xx,xlim,nlim)
  !  Task UV_SHORT
  !     internal routine
  !
  ! Find nlim such as
  !   xx(ic,nlim-1) < xlim < xx(ic,nlim)
  ! for input data ordered, retrieved from memory
  ! assumes nlim already preset so that xx(ic,nlim-1) < xlim
  !
  integer, intent(in) :: nv       ! Number of visibilities
  integer, intent(in) :: nc       ! Size of a visibility
  integer, intent(in) :: ic       ! COlumn to be tested
  integer, intent(inout) :: nlim
  real, intent(in) ::  xx(nc,nv),xlim
  !
  ! Local variables
  integer ninf,nsup,nmid
  !
  ! Code
  if (nlim.gt.nv) return
  !
  ! Define limits of searching area in the table
  !
  if (xx(ic,nlim).gt.xlim) then
    return
  elseif (xx(ic,nv).lt.xlim) then
    nlim = nv+1
    return
  endif
  ninf = nlim
  nsup = nv
  !
  ! Loop while : dichotomic search for input data ordered
  do while(nsup.gt.ninf+1)
    !
    ! Define middle of the searching area on the table
    nmid = (nsup + ninf)/2
    !
    ! If it's not in the last part, it's in the first one...
    ! then defined new searching limits area
    if (xx(ic,nmid).lt.xlim) then
      ninf = nmid
    else
      nsup = nmid
    endif
  enddo
  !
  ! Output
  nlim = nsup
end subroutine uvshort_findr
!
!------------------------------------------------------------------------
!
subroutine uvshort_finsiz (x,nd,np,ix,iy,we,xmin,xmax,ymin,ymax)
  ! Task UV_SHORT
  !   Internal routine
  !
  ! Find extrema xmin, xmax in ix column values
  !          and ymin, ymax in iy column values,
  ! in table x(nd,np) for points where weight is not null
  ! taking in account that table x is ordered on iy column values.
  !
  integer, intent(in) :: nd,np    ! Table size
  integer, intent(in) :: ix,iy    ! X and Y column pointers
  real, intent(in) :: x(nd,np)    ! Table data
  real, intent(in) :: we(np)      ! Weights
  real, intent(out) :: xmin,xmax,ymin,ymax  ! Min-Max
  !
  ! Local variables
  integer i,j
  !
  ! Code
  i = 1
  !
  ! Loop to start after null weights measurements
  do while (we(i).eq.0)
    i = i+1
  enddo
  !
  ! ymin is first y value with weight not null
  ymin = x(iy,i)
  !
  ! initialize xmin and xmax for searching loop
  xmin = x(ix,i)
  xmax = x(ix,i)
  !
  ! Loop on table lines to find xmin and xmax
  i = i+1
  do j=i,np
    if (we(j).ne.0) then
      if (x(ix,j).lt.xmin) then
        xmin = x(ix,j)
      elseif (x(ix,j).gt.xmax) then
        xmax = x(ix,j)
      endif
    endif
  enddo
  !
  ! Loop to find ymax = last y values with weight not null
  i = np
  do while (we(i).eq.0)
    i = i-1
  enddo
  ymax = x(iy,i)
  !
end subroutine uvshort_finsiz
!
!-------------------------------------------------------------------------
subroutine uvshort_dosor (rname,visi,nd,np,we,iy,error)
  use gildas_def
  use gbl_message
  use gkernel_interfaces
  use mapping_interfaces, only : map_message
  !
  ! Output visi(nd,np) will contain ycol column values in increasing order
  ! Use procedure trione
  ! we is used as work space only
  !
  ! Dummy variables
  !
  character(len=*), intent(in) :: rname
  integer, intent(in) :: nd,np,iy
  real, intent(inout) :: visi(nd,np)   ! Visibilities,we(nd)
  real, intent(inout) :: we(nd)        ! Work space for sorting
  logical, intent(out) :: error
  !
  ! Local variables
  integer i,ier ! ,trione
  !
  ! Code
  error = .false.
  do i=1,np-1
    if (visi(iy,i).gt.visi(iy,i+1)) then
      call map_message(seve%i,rname,'Sorting input table')
      ier = trione (visi,nd,np,iy,we)
      if (ier.ne.1) then
        call map_message(seve%e,rname,'Insufficient sorting space')
        error = .true.
      endif
      return
    endif
  enddo
  call map_message(seve%i,rname,'Input table is sorted')
  !
end subroutine uvshort_dosor
!
!-------------------------------------------------------------------------
function trione (x,nd,n,ix,work)
  !
  !   sorting program that uses a quicksort algorithm.
  ! sort on one row
  ! x r*4(*)  unsorted array        input
  ! nd  i first dimension of x      input
  ! n i second dimension of x     input
  ! ix  i x(ix,*) is the key for sorting    input
  ! work  r*4(nd) work space for exchange     input
  !
  !
  ! Dummy variables
  !
  integer nd,n,ix
  real*4 x(nd,n), work(nd)
  !
  ! Local variables
  !
  integer trione, maxstack, nstop
  parameter (maxstack=1000,nstop=15)
  integer*4 i, j, k, l1, r1, l, r, m
  integer*4 lstack(maxstack), rstack(maxstack), sp
  real*4 key
  logical mgtl, lgtr, rgtm
  !
  ! Code
  !
  trione = 1
  if (n.le.nstop) goto 50
  sp = 0
  sp = sp + 1
  lstack(sp) = 1
  rstack(sp) = n
  !
  ! Sort a subrecord off the stack
  ! set key = median of x(l), x(m), x(r)
  ! no! this is not reasonable, as systematic very inequal partitioning will
  ! occur in some cases (especially for nearly already sorted files)
  ! to fix this problem, i found (but i cannot prove it) that it is best to
  ! select the estimation of the median value from intermediate records. p.v.
  !
1 l = lstack(sp)
  r = rstack(sp)
  sp = sp - 1
  m = (l + r) / 2
  l1=(2*l+r)/3
  r1=(l+2*r)/3
  !
  mgtl = x(ix,m) .gt. x(ix,l)
  rgtm = x(ix,r) .gt. x(ix,m)
  !
  ! Algorithm to select the median key. the original one from mongo
  ! was completely wrong. p. valiron, 24-jan-84 .
  !
  !       mgtl  rgtm  lgtr  mgtl.eqv.lgtr median_key
  !
  ! kl < km < kr  t t * *   km
  ! kl > km > kr  f f * *   km
  !
  ! kl < km > kr  t f f f   kr
  ! kl < km > kr  t f t t   kl
  !
  ! kl > km < kr  f t f t   kl
  ! kl > km < kr  f t t f   kr
  !
  if (mgtl .eqv. rgtm) then
     key = x(ix,m)
  else
     lgtr = x(ix,l) .gt. x(ix,r)
     if (mgtl .eqv. lgtr) then
        key = x(ix,l)
     else
        key = x(ix,r)
     endif
  endif
  i = l
  j = r
  !
  ! Find a big record on the left
  !
10 if (x(ix,i).ge.key) goto 11
  i = i + 1
  goto 10
11 continue
  !
  ! Find a small record on the right
  !
20 if (x(ix,j).le.key) goto 21
  j = j - 1
  goto 20
21 continue
  if (i.ge.j) goto 2
  !
  ! Exchange records
  !
  call r4tor4 (x(1,i),work,nd)
  call r4tor4 (x(1,j),x(1,i),nd)
  call r4tor4 (work,x(1,j),nd)
  i = i + 1
  j = j - 1
  goto 10
  !
  ! Subfile is partitioned into two halves, left .le. right
  ! push the two halves on the stack
  !
2 continue
  if (j-l+1 .gt. nstop) then
     sp = sp + 1
     if (sp.gt.maxstack) then
        write(6,*) 'E-UV_SHORT, Stack overflow ',sp
        trione = 0
        return
     endif
     lstack(sp) = l
     rstack(sp) = j
  endif
  if (r-j .gt. nstop) then
     sp = sp + 1
     if (sp.gt.maxstack) then
        write(6,*) 'E-UV_SHORT, Stack overflow ',sp
        trione = 0
        return
     endif
     lstack(sp) = j+1
     rstack(sp) = r
  endif
  !
  ! anything left to process?
  !
  if (sp.gt.0) goto 1
  !
50 continue
  !
  do 110 j = n-1,1,-1
     k = j
     do i = j+1,n
        if (x(ix,j).le.x(ix,i)) goto 121
        k = i
     enddo
121  continue
     if (k.eq.j) goto 110
     call r4tor4 (x(1,j),work,nd)
     do i = j+1,k
        call r4tor4 (x(1,i),x(1,i-1),nd)
     enddo
     call r4tor4 (work,x(1,k),nd)
110  continue
  !
end function trione
!
!-------------------------------------------------------------------------
!
subroutine uvshort_doapod (xmin,xmax,ymin,ymax,tole,beam,&
     nc,nx,ny,map,raw,mapx,mapy, weight,wmin)
  !
  ! Replace map edges and bad quality values of raw input data cube
  ! with smoothed values contains in mapx data cube.
  ! Map edges corresponding to the part of the map between
  ! max SD observations locations xmin,xmax,ymin,ymax, and map size nx,ny
  ! Bad quality values corresponding to weights < wmin
  !
  ! Dummy variables
  !
  integer nc,nx,ny
  real mapx(nx),mapy(ny)
  real map(nc,nx,ny)
  real raw(nc,nx,ny)
  real beam,tole,xmin,xmax,ymin,ymax
  real weight(nx,ny),wmin
  !
  ! Local variables
  !
  integer ix,iy
  real lobe,apod,disty,distx
  !
  ! Code
  !
  ! Tests
  !
  ! pi = acos(-1.0)
  ! write(6,*) 'I-UV_SHORT, do apodisation :'
  ! write(6,*) 'min-max ',xmin,xmax,ymin,ymax
  ! write(6,*) 'beam et inc ',beam*180*3600/pi,tole*180*3600/pi&
  !      ,(mapx(1)-mapx(2))*180*3600/pi
  !
  ! Apodisation by a gaussian, twice as large than the SD beam
  !
  lobe = log(2.0)/beam**2
  !
  ! Loop on pixels
  do iy=1,ny
    !
    ! Compute disty : distance between map size ny
    ! and ymin or ymax SD observations limits
    if (mapy(iy).le.ymin-tole) then
      disty = ymin-mapy(iy)
    elseif (mapy(iy).ge.ymax+tole) then
      disty = mapy(iy)-ymax
    else
      disty = 0.0
    endif
    !
    do ix=1,nx
      !
      ! Idem on X, compute distx
      if (mapx(ix).le.xmin-tole) then
        distx = xmin-mapx(ix)
      elseif (mapx(ix).ge.xmax+tole) then
        distx = mapx(ix)-xmax
      else
        distx = 0.0
      endif
      !
      ! Apodisation factor
      apod = (distx**2+disty**2)*lobe
      !
      ! 'raw' is replaced by something else only in two cases
      if (apod.gt.80) then
        !
        ! Map edges
        raw(1:nc,ix,iy) = 0.0
      elseif (apod.ne.0.0) then
        !
        ! Map edges
        apod = exp(-apod)
        raw(1:nc,ix,iy) = map(1:nc,ix,iy)*apod
      elseif (weight(ix,iy).lt.wmin) then
        !
        ! Low weight point within the map
        raw(1:nc,ix,iy) = map(1:nc,ix,iy)
      endif
    enddo
  enddo
  !
end subroutine uvshort_doapod
!
!-------------------------------------------------------------------------
!
subroutine uvshort_dosdft(beam,diam,f,nx,ny,fact,lmv)
  use image_def
  !
  ! computes inverse of ft of single-dish beam
  ! (uses a gaussian truncated at dish size)
  !
  type(gildas), intent(in) :: lmv     ! LMV header
  integer, intent(in) ::  nx, ny      ! Problem size
  real, intent(out) ::  f(nx,ny)      ! (real part of the) TF of beam
  real, intent(in) ::  beam           ! Beam size in radian
  real, intent(in) ::  diam           ! Antenna diameter in meter
  real, intent(in) ::  fact           ! Scale factor
  !
  ! Local variables
  !
  real(8), parameter :: pi=3.141592653589793d0
  real(8), parameter :: clight=299792458d-6    ! frequency in mhz
  integer :: i,j, ii, jj
  real :: a, b, xx, yy
  real(8) :: dx, dy
  !
  ! Code
  !
  dx = clight/lmv%gil%freq/(lmv%gil%inc(1)*lmv%gil%dim(1))
  dy = clight/lmv%gil%freq/(lmv%gil%inc(2)*lmv%gil%dim(2))
  b = (pi*beam/2/clight*lmv%gil%freq)**2/alog(2.)
  !
  ! Equivalent beam area in square pixels ...
  !
  a = abs(4*alog(2.)/pi/beam**2*lmv%gil%inc(2)*lmv%gil%inc(1))*fact
  !
  ! Loop on pixels
  !
  do j = 1, ny
    !
    ! Loop on Y, pixels locations on Fourier plane
    jj = mod(j-1+ny/2,ny)-ny/2
    yy = ( jj*dy )**2
    !
    do i = 1, nx
      !
      ! Loop on X, pixels locations on Fourier plane
      ii = mod(i-1+nx/2,nx)-nx/2
      xx = ( ii*dx )**2
      !
      ! Truncation of the gaussian at diam
      if (xx+yy.le.diam**2) then
        f(i,j) = exp(b*(xx+yy))*a
      else
        f(i,j) = 0.0
      endif
    enddo
  enddo
  !
end subroutine uvshort_dosdft
!
!--------------------------------------------------------------------------
!
subroutine uvshort_dointft(beam,diam,f,nx,ny,fact,lmv)
  use image_def
  !
  ! computes ft of single-dish beam
  ! (uses a gaussian truncated at dish size)
  !
  type(gildas), intent(in) :: lmv     ! LMV header
  integer, intent(in) ::  nx, ny      ! Problem size
  real, intent(out) ::  f(nx,ny)      ! (real part of the) TF of beam
  real, intent(in) ::  beam           ! Beam size in radian
  real, intent(in) ::  diam           ! Antenna diameter in meter
  real, intent(in) ::  fact           ! Scale factor
  !
  ! Local variables
  !
  real(8), parameter :: pi=3.141592653589793d0
  real(8), parameter :: clight=299792458d-6    ! frequency in MHz
  integer :: i,j, ii, jj
  real :: a, b, xx, yy
  real(8) :: dx, dy
  !
  ! Code
  !
  dx = clight/lmv%gil%freq/(lmv%gil%inc(1)*lmv%gil%dim(1))
  dy = clight/lmv%gil%freq/(lmv%gil%inc(2)*lmv%gil%dim(2))
  b = (pi*beam/2/clight*lmv%gil%freq)**2/alog(2.)
  !
  ! Equivalent beam area in square pixels ...
  !
  a = abs(pi*beam**2/lmv%gil%inc(2)/lmv%gil%inc(1))*fact/abs(4*alog(2.))
  !
  ! Loop on pixels
  !
  do j = 1, ny
    !
    ! Loop on Y, pixels locations on Fourier plane
    jj = mod(j-1+ny/2,ny)-ny/2
    yy = ( jj*dy )**2
    !
    do i = 1, nx
      !
      ! Loop on X, pixels locations on Fourier plane
      ii = mod(i-1+nx/2,nx)-nx/2
      xx = ( ii*dx )**2
      !
      ! Truncation of the gaussian at diam
      if (xx+yy.le.diam**2) then
        f(i,j) = exp(-b*(xx+yy))*a
      else
        f(i,j) = 0.0
      endif
    enddo
  enddo
  !
end subroutine uvshort_dointft
!
!-------------------------------------------------------------------------
!
subroutine uvshort_shift(f,nx,ny,offra,offdec,lmv)
  use image_def
  !
  ! Centered interferometer beam f(nx,ny) of a mosaic field
  ! on is right position : offra and offdec shifted in uv plane
  !
  type(gildas), intent(in) :: lmv
  integer, intent(in) :: nx, ny
  complex, intent(inout) :: f(nx,ny)
  real, intent(in) :: offra, offdec
  !
  real(8), parameter :: pi=3.141592653589793d0
  !
  ! Local variables
  integer i, j, ii, jj
  real phi, sp, cp, xx, yy, re, im
  real(8) :: du, dv
  !
  ! Code
  !
  du = 1.d0/(lmv%gil%inc(1)*lmv%gil%dim(1))
  dv = 1.d0/(lmv%gil%inc(2)*lmv%gil%dim(2))
  !
  ! Loop on pixels
  do j = 1, ny
    !
    ! Loop on Y, pixels locations on Fourier plane
    jj = mod(j-1+ny/2,ny)-ny/2
    yy = jj*dv
    !
    do i = 1, nx
      !
      ! Loop on X, pixels locations on Fourier plane
      ii = mod(i-1+nx/2,nx)-nx/2
      xx = ii*du
      !
      phi = -2.d0*pi*(offra*xx + offdec*yy)
      cp = cos(phi)
      sp = sin(phi)
      !
      re = real(f(i,j))*cp - imag(f(i,j))*sp
      im = real(f(i,j))*sp + imag(f(i,j))*cp
      f(i,j) = cmplx(re,im)
    enddo
  enddo
  !
end subroutine uvshort_shift
!
!-------------------------------------------------------------------------
!
subroutine uvshort_dotrans (a,b,n,m)
  !
  ! Output table "b" is table "a" transposed in line/column order
  !
  integer, intent(in) :: n,m
  real, intent(in) :: a(n,m)
  real, intent(out) ::  b(m,n)
  !
  ! Local variables
  integer i,j
  !
  ! Code
  do i=1,m
    do j=1,n
      b(i,j) = a(j,i)
    enddo
  enddo
end subroutine uvshort_dotrans
!
!-------------------------------------------------------------------------
!
subroutine uvshort_prmult(z,f,nx,ny)
  !
  ! Update z(nx,ny) table with z*f, calculate in uv plane
  ! Used to multiply by interferometer primary beam
  !
  integer, intent(in) :: nx, ny         ! Problem size
  complex, intent(inout) :: z(nx, ny)   ! Complex values
  real, intent(in) :: f(nx,ny)          ! Multiplication function
  !
  ! Local variables
  integer i, j, ii, jj
  !
  ! Code
  do j = 1, ny
    jj = mod(j+ny/2-1,ny)+1
    do i = 1, nx
      ii = mod(i+nx/2-1,nx)+1
      z(ii,jj) = z(ii,jj) * f(i,j)
    enddo
  enddo
  !
end subroutine uvshort_prmult
!
!-------------------------------------------------------------------------
!
subroutine uvshort_uvcount(nx,ny,nvis,diam,lmv)
  use image_def
  !
  ! Compute number of visibilities nvis sampled on
  ! a regular grid of steps dx,dy inside the dish defined by diam
  ! nx and ny needed because uvcount is working in Fourier plane
  !
  type(gildas), intent(in) :: lmv   ! Input GILDAS image
  integer, intent(in) :: nx         ! X image size
  integer, intent(in) :: ny         ! Y image size
  integer, intent(out) :: nvis      ! Number of visibilities
  real, intent(in) :: diam          ! Telescope diameter
  !
  ! Local variables
  !
  integer :: i, j, ii, jj
  real :: uu, vv, diam2
  real(8), parameter :: clight=299792458d-6    ! frequency in mhz
  real(8) :: dx, dy
  !
  ! Code
  !
  dx = clight/lmv%gil%freq/(lmv%gil%inc(1)*lmv%gil%dim(1))
  dy = clight/lmv%gil%freq/(lmv%gil%inc(2)*lmv%gil%dim(2))
  diam2 = diam**2
  !
  nvis = 0
  do j = 1, ny
    jj = mod(j-1+ny/2,ny)-ny/2
    vv = jj*dy
    do i = 1, nx/2
      ii = mod(i-1+nx/2,nx)-nx/2
      uu = ii*dx
      if (uu*uu+vv*vv.le.diam2) then
        nvis = nvis + 1
      endif
    enddo
  enddo
  !
end subroutine uvshort_uvcount
!
!--------------------------------------------------------------------------
!
subroutine uvshort_uvtable(nx,ny,nd,nc,v,w,ww,nvis,diam,wfactor,factor,lmv)
  use image_def
  !
  ! Tabulate the visibilities
  !
  integer, intent(in) :: nx   ! X Image size
  integer, intent(in) :: ny   ! Y Image size
  integer, intent(in) :: nc   ! Number of channels
  integer, intent(in) :: nd   ! Visibility size
  integer, intent(in) :: nvis ! Number of visibilities
  complex, intent(in) :: v(nx,ny,nc)  ! Gridded visibilities
  complex, intent(in) :: ww(nx,ny)    ! Complex weights
  real, intent(out) :: w(nd,nvis)     ! Resulting visibilities
  real, intent(in) :: wfactor ! Weight factor
  real, intent(in) :: diam    ! Telescope diameter, factor
  real, intent(in) :: factor  ! Intensity factor
  type(gildas), intent(in) :: lmv     ! Input image
  !
  ! Local variables
  !
  real(8), parameter :: pi=3.141592653589793d0
  real(8), parameter :: clight=299792458d-6    ! frequency in mhz
  integer :: i, j, k ,kk, ii, jj, kvis, k00
  integer :: gdate
  real(4) :: uu, vv, sw, we, duv, wfact, diam2
  real(8) :: dx, dy
  !
  ! Code
  !
  dx = clight/lmv%gil%freq/(lmv%gil%inc(1)*lmv%gil%dim(1))
  dy = clight/lmv%gil%freq/(lmv%gil%inc(2)*lmv%gil%dim(2))
  diam2 = diam**2
  !
  wfact = 1.0/(factor**2)
  kvis = 0
  sw = 0.0
  !
  !!Print *,' Wfactor ',wfactor,'     Wfact ',wfact
  !
  call sic_gagdate(gdate)
  !
  ! Loop on pixels of the visibility map
  !
  do j = 1, ny
    jj = mod(j-1+ny/2,ny)-ny/2
    vv = jj*dy
    do i = 1, nx/2
      ii = mod(i-1+nx/2,nx)-nx/2
      uu = ii*dx
      duv = uu**2+vv**2
      !
      ! Keep only points inside circle defined by diam
      !
      if (duv.le.diam2) then
        kvis = kvis + 1
        w(1,kvis) = uu
        w(2,kvis) = vv
        w(3,kvis) = 0
        w(4,kvis) = gdate   ! Current date
        w(5,kvis) = 0
        w(6:7,kvis) = -1.0  ! Convention: Antenna # -1 for Short spacings
        kk = 7
        !
        ! Weight
        !
        we = real(ww(i,j))
        if (i.eq.1 .and. j.ne.1) then
          we = we*0.5
        endif
        if (we.lt.0) we = -we
        !
        ! u=0 v=0 point
        !
        if (duv.eq.0) k00 = kvis
        !
        ! Extract visibilities
        ! - apply K-to-Jy conversion factor
        ! - wfact = wfactor/factor**2
        !
        do k=1, nc
          w(kk+1,kvis) = real(v(i,j,k))*factor
          w(kk+2,kvis) = imag(v(i,j,k))*factor
          w(kk+3,kvis) = we*wfact
          kk = kk + 3
        enddo
        sw = sw+we*wfact
      endif
    enddo
  enddo
  !
  ! Test number of visibilities
  !
  if (kvis.ne.nvis) then
    write(6,*) 'W-UV_SHORT, Inconsistent number of visibilities'
  endif
  !
  ! Normalize the weights **WW**
  !
  if (sw.ne.0.) then
    sw = 1/sw
    do i=1,nvis
      do k=1,nc
        w(7+k*3,i) = w(7+k*3,i)*sw*wfactor
      enddo
    enddo
  endif
end subroutine uvshort_uvtable
!
!-------------------------------------------------------------------------
!
subroutine uvshort_dopoint(data,nd,np,xcol,ycol,old,new)
  !
  ! Recompute data(nd,np) xcol and ycol coordinates values
  ! in case of changing reference position a0 d0 from old to new
  !
  integer, intent(in) :: nd         ! Size of a visibility
  integer, intent(in) :: np          ! Number of visibilities
  integer, intent(in) :: xcol        ! RA offset pointer
  integer, intent(in) :: ycol        ! Dec offset pointer
  real, intent(inout) :: data(nd,np) ! Visibilities
  real(8), intent(in) :: old(2)      ! Old RA and Dec center
  real(8), intent(in) :: new(2)      ! New Ra and Dec center
  !
  ! Local variables
  !
  real(8) :: dra,dde,ra,de,uncde,cde
  integer :: i
  !
  ! The code here is only for the "Radio" projection.  Things
  ! should be done better
  !
  uncde = 1.d0/cos(old(2))
  cde = cos(new(2))
  !
  do i=1,np
    ra = old(1) + dble(data(xcol,i))*uncde
    de = old(2) + dble(data(ycol,i))
    dra = (ra - new(1)) * cde
    dde = de - new(2)
    data(xcol,i) = dra
    data(ycol,i) = dde
  enddo
  !
end subroutine uvshort_dopoint
!
subroutine uvshort_copy_uvtable(rname,hin,hou,umax,wcol,weight,error)
  use image_def
  use gkernel_interfaces
  use gbl_message
  !
  character(len=*), intent(in) :: rname   ! Callers name
  type(gildas), intent(inout) :: hin      ! Input UV table
  type(gildas), intent(inout) :: hou      ! Output UV table
  integer, intent(in) :: wcol             ! Weight column
  real, intent(in) :: umax                ! Maximum radius
  real, intent(out) :: weight             ! Total weight
  logical, intent(out) :: error           ! Logical error flag
  !
  integer :: nblock, ier, iloop, j, jcol
  real, allocatable :: din(:,:)
  real :: umax2, u2
  !
  ! OK copy the whole stuff...
  call gdf_nitems('SPACE_GILDAS',nblock,hin%gil%dim(1))
  nblock = min(nblock,hin%gil%dim(2))
  !
  allocate (din(hin%gil%dim(1),nblock),stat=ier)
  if (ier.ne.0) then
    call gag_message(seve%e,rname,'Memory allocation error ')
    error = .true.
    return
  endif
  !
  umax2 = umax*umax
  weight = 0
  jcol = hin%gil%fcol+3*wcol-1
  !
  hin%blc = 0
  hin%trc = 0
  hou%blc = 0
  hou%trc = 0
  call gdf_create_image(hou,error)
  if (error) return
  !
  call gag_message(seve%i,rname,'Copying UV data to be appended')
  do iloop = 1,hin%gil%dim(2),nblock
    hin%blc(2) = iloop
    hin%trc(2) = min(iloop+nblock-1,hin%gil%dim(2))
    hou%blc(2) = iloop
    hou%trc(2) = hin%trc(2)
    call gdf_read_data (hin,din,error)
    if (error) return
    do j=1,hin%trc(2)-hin%blc(2)+1
      u2 = din(1,j)**2+din(2,j)**2
      if (u2.lt.umax2) then
        weight = weight+din(jcol,j)
      endif
    enddo
    call gdf_write_data (hou, din, error)
    if (error) return
  enddo
  call gdf_close_image(hou,error)
  !
end subroutine uvshort_copy_uvtable
!
subroutine uvshort_fill(lmv,uvt,error,nvis,nc,ra_off,de_off,positions,last)
  use image_def
  use gkernel_interfaces
  ! Task UV_SHORT
  !   Internal routine
  ! @ private
  !   Fill in header of short spacings UV data
  !
  type(gildas), intent(in) :: lmv       ! Header of low resolution data cube
  type(gildas), intent(inout) :: uvt    ! Header of short spacings UV table
  logical, intent(out) :: error
  integer, intent(in) :: nvis ! Number of visibilities
  integer, intent(in) :: nc   ! Number of channels
  real, intent(in) :: ra_off  ! RA Offset
  real, intent(in) :: de_off  ! Declination Offset
  logical, intent(in) :: positions ! Do we add offset position columns ?
  integer, intent(out) :: last  ! Size of a visibility
  !
  integer :: i
  !
  call gdf_copy_header(lmv,uvt,error)
  uvt%char%code(2) = 'RANDOM'
  uvt%char%code(1) = 'UV-RAW'
  uvt%gil%coor_words = 6*gdf_maxdims
  uvt%gil%blan_words = 2
  uvt%gil%extr_words = 10
  uvt%gil%desc_words = 18
  uvt%gil%posi_words = 12
  uvt%gil%proj_words = 9
  uvt%gil%spec_words = 12
  uvt%gil%reso_words = 3
  uvt%gil%dim(2) = nvis
  uvt%gil%dim(1) = 3*nc+7        ! 7 daps + (real, imag, weight)*nchannels
  uvt%gil%convert = 0
  uvt%gil%ref(1) = lmv%gil%ref(3)
  uvt%gil%inc(1) = lmv%gil%fres
  uvt%gil%val(1) = lmv%gil%freq
  uvt%gil%inc(2) = 1.              ! needed to avoid funny crash in graphic...
  uvt%gil%ndim = 2
  uvt%gil%dim(3) = 1
  uvt%gil%dim(4) = 1
  !
  ! Here we could change the logic, keep A0,D0 and set the offsets
  ! in the UV table.  But this has consequences in the whole package.
  !
  ! The formula below is WRONG, especially at high declinations
  uvt%gil%ra = uvt%gil%a0+ra_off/cos(uvt%gil%d0)
  ! a better conversion should be used...
  uvt%gil%dec = uvt%gil%d0+de_off
  uvt%char%type = 'GILDAS_UVFIL'
  uvt%char%unit = 'Jy'
  uvt%gil%nchan = nc
  !
  ! Here define the order in which you want the extra "columns"
  uvt%gil%column_pointer = 0
  uvt%gil%column_size = 0
  uvt%gil%column_pointer(code_uvt_u) = 1
  uvt%gil%column_pointer(code_uvt_v) = 2
  uvt%gil%column_pointer(code_uvt_w) = 3
  uvt%gil%column_pointer(code_uvt_date) = 4
  uvt%gil%column_pointer(code_uvt_time) = 5
  uvt%gil%column_pointer(code_uvt_anti) = 6
  uvt%gil%column_pointer(code_uvt_antj) = 7
  uvt%gil%natom = 3
  uvt%gil%nstokes = 1
  uvt%gil%fcol = 8
  last = uvt%gil%fcol + uvt%gil%natom * uvt%gil%nchan - 1
  !
  uvt%gil%form = fmt_r4
  !
  if (positions) then
    last = last+1
    uvt%gil%column_pointer(code_uvt_loff) = last
    last = last+1
    uvt%gil%column_pointer(code_uvt_moff) = last
  endif
  uvt%gil%dim(1) = last
  !
  do i=1,code_uvt_last
    if (uvt%gil%column_pointer(i).ne.0) uvt%gil%column_size(i) = 1
  enddo
  !
  uvt%gil%nvisi = nvis
  uvt%gil%type_gdf = code_gdf_uvt
  call gdf_setuv (uvt,error)
  uvt%loca%size = uvt%gil%dim(1) * uvt%gil%dim(2)
  !!print *, 'Into newuvt_init ',xima%gil%nvisi, xima%gil%nchan, xima%gil%dim(1:2)
  !!print *, 'Into newuvt_init UVDA_WORDS', xima%gil%uvda_words
end subroutine uvshort_fill
!
subroutine uv_short_consistency(rname,nc,uvt,lmv,tole,error)
  use image_def
  use gbl_message
  use mapping_interfaces, only : map_message
  !
  ! UV_SHORT
  !   Verify spectral axis consistency
  !
  character(len=*), intent(in) :: rname
  integer, intent(in) :: nc ! Number of channels
  type(gildas), intent(in) :: uvt
  type(gildas), intent(in) :: lmv
  real, intent(in) :: tole
  logical, intent(out) :: error
  !
  real(8) :: freq
  real(4) :: velo
  character(len=message_length) :: mess
  !
  error = .false.
  ! Number of channels
  if (uvt%gil%nchan.ne.nc) then
    print *,'Mismatch in number of channels ',uvt%gil%nchan,nc
    call map_message(seve%w,rname,mess)
    error = .true.
  endif
  !
  ! Check here the spectral axis mismatch
  if (abs(uvt%gil%vres-lmv%gil%vres).gt.abs(lmv%gil%vres*tole)) then
    print *,'Mismatch in spectral resolution ',uvt%gil%vres,lmv%gil%vres
    call map_message(seve%w,rname,mess)
    error = .true.
  endif
  freq = (uvt%gil%ref(1) - lmv%gil%ref(3))*lmv%gil%fres + lmv%gil%freq
  if (abs(freq-lmv%gil%freq).gt.abs(lmv%gil%fres*tole)) then
    print *,'Mismatch in frequency axis ',freq,lmv%gil%freq
    call map_message(seve%w,rname,mess)
    error = .true.
  endif
  !
  ! Velocity should be checked too
  velo = (uvt%gil%ref(1) - lmv%gil%ref(3))*lmv%gil%vres + lmv%gil%voff
  if (abs(velo-lmv%gil%voff).gt.abs(lmv%gil%vres*tole)) then
    print *,'Mismatch in velocity axis ',velo,lmv%gil%voff
    call map_message(seve%w,rname,mess)
    error = .true.
  endif
end subroutine uv_short_consistency
!
subroutine spectrum_to_zero(nc,spectrum,uvdata,date,weight)
  !
  ! Convert a spectrum into a Zero spacing
  integer, intent(in) :: nc
  real, intent(in) :: spectrum(nc) 
  real, intent(out) :: uvdata(:)
  integer, intent(in) :: date
  real, intent(in) :: weight
  !
  integer :: ic
  !
  uvdata = 0
  uvdata(4) = date
  uvdata(6:7) = -1.0      ! Conventional antenna number
  do ic=1,nc
    uvdata(5+3*ic) = spectrum(ic)
    uvdata(7+3*ic) = weight
  enddo
end subroutine spectrum_to_zero
!
subroutine uvshort_read_class_table(rname,table,sdt,sdt_data,error)
  use image_def
  use gbl_message
  use gkernel_types
  use gkernel_interfaces
  !
  character(len=*), intent(in) :: rname
  character(len=*), intent(in) :: table
  type(gildas), intent(out) :: sdt
  real, allocatable, intent(out) :: sdt_data(:,:)
  logical, intent(out) :: error
  !
  character(len=256) :: name
  integer :: np,nd
  !
  call gildas_null(sdt)
  name = trim(table)
  call sic_parsef(name,sdt%file,' ','.tab')
  !
  ! Read sdt header and check format
  !
  error = .false.
  call gdf_read_header(sdt,error)
  if (gildas_error(sdt,rname,error)) then
    call map_message(seve%f,rname,'Cannot read header from Single Dish table')
    return
  endif
  if (sdt%gil%form.ne.fmt_r4) then
    call map_message(seve%f,rname,'Only real format supported')
    error = .true.
    return
  endif
  !
  nd = sdt%gil%dim(1)
  np = sdt%gil%dim(2)
  !
  ! Allocate memory space and read data
  allocate(sdt_data(nd,np), stat=sdt%status)
  if (gildas_error(sdt,rname,error)) then
    call map_message(seve%e,rname,'Cannot allocate memory for Single Dish table')
    return
  endif
  !
  call gdf_read_data(sdt,sdt_data,error)
  if (gildas_error(sdt,rname,error)) then
    call map_message(seve%e,rname,'Cannot read data from Single Dish table')
    return
  endif
end subroutine uvshort_read_class_table
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
end program uvshort_main
