module cubedag_history
  use gkernel_interfaces
  use cubetools_parameters
  use cubedag_parameters
  use cubedag_messaging
  use cubedag_types

  integer(kind=entr_k), parameter :: history_optimize_minalloc=100
  integer(kind=4), parameter :: command_length=16  ! ZZZ Duplicated from SIC

  type history_optimize_t
    integer(kind=entr_k) :: next=1  ! Number of commands in history index + 1
    integer(kind=iden_l),              allocatable :: id(:)
    character(len=command_length),     allocatable :: command(:)
    character(len=commandline_length), allocatable :: line(:)
    type(cubedag_link_t),              allocatable :: inputs(:)
    type(cubedag_link_t),              allocatable :: outputs(:)
  contains
    procedure, public  :: reallocate   => cubedag_history_reallocate_expo
    generic,   public  :: add          => add_fromargs,add_fromid
    procedure, private :: add_fromargs => cubedag_history_add_fromargs
    procedure, private :: add_fromid   => cubedag_history_add_fromid
    procedure, public  :: list         => cubedag_history_list
    final              :: cubedag_history_final
  end type history_optimize_t

  type(history_optimize_t), target :: hx  ! History index

  public :: command_length
  public :: history_optimize_t
  public :: cubedag_history_add_tohx,cubedag_history_list_hx,cubedag_history_tostr
  public :: cubedag_history_removenode,cubedag_history_removecommand
  public :: cubedag_history_destroy
  public :: hx
  private

contains

  subroutine cubedag_history_reallocate_expo(optx,mcomms,error)
    !---------------------------------------------------------------------
    !  Reallocate the 'optimize' type arrays. If current allocation is not
    ! enough, double the size of allocation. Since this reallocation
    ! routine is used in a context of adding more and more data, data
    ! is always preserved after reallocation.
    !---------------------------------------------------------------------
    class(history_optimize_t), intent(inout) :: optx    !
    integer(kind=entr_k),      intent(in)    :: mcomms  ! Requested size
    logical,                   intent(inout) :: error   ! Logical error flag
    ! Local
    integer(kind=entr_k) :: ncomms
    !
    if (allocated(optx%command)) then
      ncomms = size(optx%command,kind=8)
      if (ncomms.ge.mcomms)  return           ! Enough size yet
      ncomms = 2_8*ncomms                     ! Request twice more place than before
      if (ncomms.lt.mcomms)  ncomms = mcomms  ! Twice is not enough, use mcomms
    else
      ncomms = max(mcomms,history_optimize_minalloc)  ! No allocation yet, use mcomms
    endif
    !
    call cubedag_history_reallocate(optx,ncomms,.true.,error)
    if (error)  return
  end subroutine cubedag_history_reallocate_expo

  subroutine cubedag_history_reallocate(optx,mcomms,keep,error)
    !---------------------------------------------------------------------
    !  Allocate the 'optimize' type arrays. Enlarge the arrays to the
    ! requested size, if needed. No shrink possible. Keep data if
    ! requested.
    !---------------------------------------------------------------------
    type(history_optimize_t), intent(inout) :: optx    !
    integer(kind=entr_k),     intent(in)    :: mcomms  ! Requested size
    logical,                  intent(in)    :: keep    ! Keep previous data?
    logical,                  intent(inout) :: error   ! Logical error flag
    ! Local
    character(len=*), parameter :: rname='HISTORY>REALLOCATE'
    integer(kind=4) :: ier
    integer(kind=entr_k) :: ncomms,inode
    integer(kind=iden_l), allocatable :: bufid(:)
    character(len=command_length), allocatable :: bufcc(:)
    character(len=commandline_length), allocatable :: bufcl(:)
    type(cubedag_link_t), allocatable :: buflk(:)
    !
    if (allocated(optx%command)) then
      ncomms = size(optx%command,kind=8)  ! Size of allocation
      if (ncomms.ge.mcomms) then
        ! Index is already allocated with a larger size. Keep it like this.
        ! Shouldn't we deallocate huge allocations if user requests a small one?
        return
      endif
    elseif (mcomms.eq.0) then
      ! No problem: can occur when dealing with empty files (e.g. nothing
      ! was written in an output file)
      return
    elseif (mcomms.lt.0) then
      call cubedag_message(seve%e,rname,'Can not allocate empty indexes')
      error = .true.
      return
    endif
    !
    ncomms = min(ncomms,optx%next-1)  ! Used part of the arrays
    if (keep) then
      allocate(bufid(ncomms),stat=ier)
      allocate(bufcc(ncomms),stat=ier)
      allocate(bufcl(ncomms),stat=ier)
      allocate(buflk(ncomms),stat=ier)
      if (failed_allocate(rname,'buf arrays',ier,error)) then
        error = .true.
        return
      endif
    endif
    !
    call reallocate_optimize_id(rname,'id array',     optx%id,     mcomms,keep,bufid,error)
    if (error)  return
    call reallocate_optimize_ch(rname,'command array',optx%command,mcomms,keep,bufcc,error)
    if (error)  return
    call reallocate_optimize_ch(rname,'line array',   optx%line,   mcomms,keep,bufcl,error)
    if (error)  return
    call reallocate_optimize_lk(rname,'inputs array', optx%inputs, mcomms,keep,buflk,error)
    if (error)  return
    call reallocate_optimize_lk(rname,'outputs array',optx%outputs,mcomms,keep,buflk,error)
    if (error)  return
    !
    if (keep) then
      if (allocated(bufid))  deallocate(bufid)
      if (allocated(bufcc))  deallocate(bufcc)
      if (allocated(bufcl))  deallocate(bufcl)
      if (allocated(buflk))  deallocate(buflk)
    endif
    !
    ! Initialize the new components
    do inode=ncomms+1,mcomms
      call history_optimize_init(optx,inode,error)
      if (error)  return
    enddo
  end subroutine cubedag_history_reallocate
  !
  subroutine reallocate_optimize_id(rname,name,val,mcomms,keep,buf,error)
    character(len=*),     intent(in)    :: rname
    character(len=*),     intent(in)    :: name
    integer(kind=iden_l), allocatable   :: val(:)
    integer(kind=entr_k), intent(in)    :: mcomms
    logical,              intent(in)    :: keep
    integer(kind=iden_l), allocatable   :: buf(:)  ! Not allocated if keep is .false.
    logical,              intent(inout) :: error
    ! Local
    integer(kind=4) :: ier
    integer(kind=entr_k) :: ncomms
    !
    if (keep) then
      ncomms = size(buf)
      buf(:) = val(1:ncomms)
    endif
    if (allocated(val)) deallocate(val)
    allocate(val(mcomms),stat=ier)
    if (failed_allocate(rname,name,ier,error))  return
    if (keep) val(1:ncomms) = buf(:)
  end subroutine reallocate_optimize_id
  !
  subroutine reallocate_optimize_ch(rname,name,val,mcomms,keep,buf,error)
    character(len=*),     intent(in)    :: rname
    character(len=*),     intent(in)    :: name
    character(len=*),     allocatable   :: val(:)
    integer(kind=entr_k), intent(in)    :: mcomms
    logical,              intent(in)    :: keep
    character(len=*),     allocatable   :: buf(:)  ! Not allocated if keep is .false.
    logical,              intent(inout) :: error
    ! Local
    integer(kind=4) :: ier
    integer(kind=entr_k) :: ncomms
    !
    if (keep) then
      ncomms = size(buf)
      buf(:) = val(1:ncomms)
    endif
    if (allocated(val)) deallocate(val)
    allocate(val(mcomms),stat=ier)
    if (failed_allocate(rname,name,ier,error))  return
    if (keep) val(1:ncomms) = buf(:)
  end subroutine reallocate_optimize_ch
  !
  subroutine reallocate_optimize_lk(rname,name,val,mcomms,keep,buf,error)
    character(len=*),     intent(in)    :: rname
    character(len=*),     intent(in)    :: name
    type(cubedag_link_t), allocatable   :: val(:)
    integer(kind=entr_k), intent(in)    :: mcomms
    logical,              intent(in)    :: keep
    type(cubedag_link_t), allocatable   :: buf(:)  ! Not allocated if keep is .false.
    logical,              intent(inout) :: error
    ! Local
    integer(kind=4) :: ier
    integer(kind=entr_k) :: icomm,ncomms
    !
    if (keep) then
      ncomms = size(buf)
      do icomm=1,ncomms
        buf(icomm)%n    =  val(icomm)%n
        buf(icomm)%list => val(icomm)%list
        buf(icomm)%flag => val(icomm)%flag
      enddo
    endif
    if (allocated(val)) deallocate(val)
    allocate(val(mcomms),stat=ier)
    if (failed_allocate(rname,name,ier,error))  return
    if (keep) then
      do icomm=1,ncomms
        val(icomm)%n    =  buf(icomm)%n
        val(icomm)%list => buf(icomm)%list
        val(icomm)%flag => buf(icomm)%flag
      enddo
    endif
  end subroutine reallocate_optimize_lk

  subroutine cubedag_history_final(hoptx)
    type(history_optimize_t), intent(inout) :: hoptx
    ! Local
    integer(kind=entr_k) :: ient
    logical :: error
    !
    hoptx%next = 1
    if (allocated(hoptx%id))       deallocate(hoptx%id)
    if (allocated(hoptx%command))  deallocate(hoptx%command)
    if (allocated(hoptx%line))     deallocate(hoptx%line)
    !
    error = .false.
    if (allocated(hoptx%inputs)) then
      do ient=1,size(hoptx%inputs)
        call hoptx%inputs(ient)%final(error)
        if (error)  continue
      enddo
    endif
    if (allocated(hoptx%outputs)) then
      do ient=1,size(hoptx%outputs)
        call hoptx%outputs(ient)%final(error)
        if (error)  continue
      enddo
    endif
  end subroutine cubedag_history_final

  subroutine history_optimize_init(optx,i,error)
    !---------------------------------------------------------------------
    ! Initialize the i-th component in the index
    !---------------------------------------------------------------------
    type(history_optimize_t), intent(inout) :: optx
    integer(kind=entr_k),     intent(in)    :: i
    logical,                  intent(inout) :: error
    !
    optx%id(i)        = 0
    optx%command(i)   = strg_unk
    optx%line(i)      = strg_unk
    optx%inputs(i)%n  = 0
    optx%outputs(i)%n = 0
  end subroutine history_optimize_init

  subroutine cubedag_history_add_tohx(command,line,inputs,outputs,hid,error)
    !-------------------------------------------------------------------
    ! Add a new command in the HISTORY index. Return the associated
    ! history identifier.
    !-------------------------------------------------------------------
    character(len=*),     intent(in)    :: command
    character(len=*),     intent(in)    :: line
    type(cubedag_link_t), intent(in)    :: inputs
    type(cubedag_link_t), intent(in)    :: outputs
    integer(kind=entr_k), intent(out)   :: hid
    logical,              intent(inout) :: error
    !
    if (outputs%n.le.0) then
      ! The commands which do not create an output cube are not
      ! registered in history
      hid = 0
      return
    endif
    !
    call hx%add(command,line,inputs,outputs,error)
    if (error)  return
    hid = hx%id(hx%next-1)
  end subroutine cubedag_history_add_tohx

  subroutine cubedag_history_add_fromargs(hoptx,command,line,inputs,outputs,error)
    !-------------------------------------------------------------------
    ! Add a new command in the HISTORY index
    !-------------------------------------------------------------------
    class(history_optimize_t), intent(inout) :: hoptx
    character(len=*),          intent(in)    :: command
    character(len=*),          intent(in)    :: line
    type(cubedag_link_t),      intent(in)    :: inputs
    type(cubedag_link_t),      intent(in)    :: outputs
    logical,                   intent(inout) :: error
    ! Local
    integer(kind=entr_k) :: ient
    !
    ient = hoptx%next
    call hoptx%reallocate(ient,error)
    if (error)  return
    !
    hoptx%id(ient) = ient
    hoptx%command(ient) = command
    hoptx%line(ient) = line
    call inputs%copy(hoptx%inputs(ient),error)
    if (error)  return
    call outputs%copy(hoptx%outputs(ient),error)
    if (error)  return
    !
    hoptx%next = hoptx%next+1
  end subroutine cubedag_history_add_fromargs

  subroutine cubedag_history_add_fromid(hoptx,hid,error)
    !-------------------------------------------------------------------
    ! Add a command in the given index, duplicating one command from
    ! the main history index
    !-------------------------------------------------------------------
    class(history_optimize_t), intent(inout) :: hoptx
    integer(kind=iden_l),      intent(in)    :: hid
    logical,                   intent(inout) :: error
    ! Local
    integer(kind=entr_k) :: ihist
    !
    ihist = cubedag_history_entrynum(hid,error)
    if (error)  return
    !
    call cubedag_history_add_fromargs(hoptx,  &
      hx%command(ihist),  &
      hx%line(ihist),  &
      hx%inputs(ihist),  &
      hx%outputs(ihist),  &
      error)
    if (error)  return
    !
    ! Beware the ID must be kept from the original index
    hoptx%id(hoptx%next-1) = hx%id(ihist)
  end subroutine cubedag_history_add_fromid

  subroutine cubedag_history_list_hx(error)
    !-------------------------------------------------------------------
    ! List the history index
    !-------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    call hx%list(error)
    if (error)  return
  end subroutine cubedag_history_list_hx

  subroutine cubedag_history_list(hoptx,error)
    !-------------------------------------------------------------------
    ! List the history index
    !-------------------------------------------------------------------
    class(history_optimize_t), intent(in)    :: hoptx
    logical,                   intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='HISTORY>LIST'
    integer(kind=entr_k) :: ient
    character(len=10) :: tmpi,tmpo
    integer(kind=4) :: nd
    character(len=16) :: forma
    !
    if (hoptx%next.le.1) then
      call cubedag_message(seve%w,rname,'History index is empty')
      error = .true.
      return
    endif
    !
    nd = ceiling(log10(real(maxval(hoptx%id(1:hoptx%next-1))+1,kind=8)))
    write(forma,'(A,I0,A)') '(I',nd,',4(2X,A))'
    !
    do ient=1,hoptx%next-1
      call hoptx%inputs(ient)%repr('i=',tmpi)
      call hoptx%outputs(ient)%repr('o=',tmpo)
      write(*,forma)  hoptx%id(ient),hoptx%command(ient),tmpi,tmpo,trim(hoptx%line(ient))
    enddo
  end subroutine cubedag_history_list

  function cubedag_history_entrynum(id,error)
    !-------------------------------------------------------------------
    ! Resolve the entry number corresponding to the given ID.
    ! - If ID>0, the entry is resolved by searching in all the history
    ! index. This resolution is based on two strong assumptions:
    !  1) the identifier is UNIQUE,
    !  2) the identifier list is SORTED
    ! - If ID<=0, the ID is assumed to be a position from the end
    ! (0=last, same as CubeID).
    !-------------------------------------------------------------------
    integer(kind=entr_k) :: cubedag_history_entrynum
    integer(kind=iden_l), intent(in)    :: id
    logical,              intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='HISTORY>ENTRYNUM'
    integer(kind=entr_k) :: inf,mid,sup
    character(len=mess_l) :: mess
    !
    if (id.le.0) then
      ! Search by position from the end
      if (-id.gt.hx%next-2) then
        write(mess,'(A,I0,A)')  'No such position ',id,' in history'
        call cubedag_message(seve%e,rname,mess)
        cubedag_history_entrynum = 0
        error = .true.
        return
      endif
      cubedag_history_entrynum = hx%next-1+id
      return
    endif
    !
    ! Dichotomic search
    if (hx%id(1).eq.id) then
      cubedag_history_entrynum = 1
      return
    endif
    !
    inf = 1
    sup = hx%next-1
    do while (sup.gt.inf+1)
      mid = (inf+sup)/2  ! Integer division
      if (hx%id(mid).lt.id) then
        inf = mid
      else
        sup = mid
      endif
    enddo
    !
    if (hx%id(sup).eq.id) then
      cubedag_history_entrynum = sup
    else
      write(mess,'(A,I0,A)')  'No such identifier ',id,' in history'
      call cubedag_message(seve%e,rname,mess)
      cubedag_history_entrynum = 0
      error = .true.
      return
    endif
    !
  end function cubedag_history_entrynum

  subroutine cubedag_history_tostr(hid,str,error)
    integer(kind=iden_l), intent(in)    :: hid
    character(len=*),     intent(out)   :: str
    logical,              intent(inout) :: error
    ! Local
    integer(kind=4) :: ni,nc
    integer(kind=entr_k) :: ient
    !
    ient = cubedag_history_entrynum(hid,error)
    if (error)  return
    !
    write(str,'(I0)') hid
    if (ient.le.0) then
      ! Valid: root has no history
      return
    elseif (ient.lt.hx%next) then
      ni = len_trim(str)
      nc = min(len_trim(hx%command(ient)),len(str)-ni-3)
      write(str(ni+1:),'(3A)')  ' (',hx%command(ient)(1:nc),')'
    else
      ni = len_trim(str)
      nc = min(7,len(str)-ni-3)
      write(str(ni+1:),'(3A)')  ' (','UNKNOWN',')'
    endif
    !
  end subroutine cubedag_history_tostr

  subroutine cubedag_history_removenode(id,error)
    use cubedag_dag
    !-------------------------------------------------------------------
    ! Properly remove a DAG node from the history.
    ! This might leave some commands without inputs or without outputs
    !-------------------------------------------------------------------
    integer(kind=iden_l), intent(in)    :: id  ! Node identifier
    logical,              intent(inout) :: error
    ! Local
    class(cubedag_node_object_t), pointer :: object
    integer(kind=entr_k) :: ihist
    !
    call cubedag_dag_get_object(id,object,error)
    if (error)  return
    ihist = object%node%history
    !
    ! Remove from command which created it
    call hx%outputs(ihist)%unlink(object,error)
    if (error)  return
    !
    ! Remove from commands which used it as input. As of today, there
    ! is no backpointer to these commands (is this really desired?).
    ! Loop unefficiently on all commands:
    do ihist=1,hx%next-1
      call hx%inputs(ihist)%unlink(object,error)
      if (error)  return
    enddo
    !
  end subroutine cubedag_history_removenode

  subroutine cubedag_history_removecommand(hid,error)
    use cubedag_dag
    !-------------------------------------------------------------------
    ! Properly a command from the history.
    ! This might leave other commands without inputs and/or without
    ! outputs
    !-------------------------------------------------------------------
    integer(kind=iden_l), intent(in)    :: hid  ! History identifier
    logical,              intent(inout) :: error
    ! Local
    class(cubedag_node_object_t), pointer :: object
    integer(kind=entr_k) :: ihist,inode,jhist
    integer(kind=iden_l) :: nodeid
    !
    ihist = cubedag_history_entrynum(hid,error)
    if (error)  return
    !
    do inode=1,hx%outputs(ihist)%n
      nodeid = hx%outputs(ihist)%list(inode)%p%node%id
      call cubedag_dag_get_object(nodeid,object,error)
      if (error)  return
      !
      ! Remove from commands which used it as input. As of today, there
      ! is no backpointer to these commands (is this really desired?).
      ! Loop unefficiently on all commands:
      do jhist=1,hx%next-1
        call hx%inputs(jhist)%unlink(object,error)
        if (error)  return
      enddo
      !
      ! Actually remove the node from DAG
      call cubedag_dag_removenode(nodeid,error)
      if (error)  return
    enddo
    !
    ! Now remove the command from history index
    call cubedag_history_removecommand_from(hx,ihist,error)
    if (error)  return
    !
  end subroutine cubedag_history_removecommand

  subroutine cubedag_history_removecommand_from(optx,ihist,error)
    !-------------------------------------------------------------------
    ! Properly remove a command from the history index
    !-------------------------------------------------------------------
    type(history_optimize_t), intent(inout) :: optx
    integer(kind=entr_k),     intent(in)    :: ihist
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='HISTORY>REMOVE>COMMAND'
    integer(kind=entr_k) :: ient
    !
    call hx%inputs(ihist)%final(error)
    if (error)  return
    call hx%outputs(ihist)%final(error)
    if (error)  return
    !
    do ient=ihist+1,optx%next-1
      optx%id(ient-1)      = optx%id(ient)
      optx%command(ient-1) = optx%command(ient)
      optx%line(ient-1)    = optx%line(ient)
      call optx%inputs(ient)%copy(optx%inputs(ient-1),error)
      if (error)  return
      call optx%outputs(ient)%copy(optx%outputs(ient-1),error)
      if (error)  return
    enddo
    optx%next = optx%next-1
    !
  end subroutine cubedag_history_removecommand_from

  subroutine cubedag_history_destroy(error)
    !-------------------------------------------------------------------
    ! Brute-force destroy the whole HISTORY
    ! No care of the backpointer links to the cubes in the DAG
    !-------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    call cubedag_history_final(hx)
    !
  end subroutine cubedag_history_destroy

end module cubedag_history
