subroutine gfits_load_header(fhead,check,error)
  use gfits_interfaces, except_this=>gfits_load_header
  use gfits_types
  !---------------------------------------------------------------------
  ! @ public
  ! Load the FITS header in the dictionary structure.
  !  COMMENT lines are ignored
  !  HIERARCH lines are interpreted as appropriate
  ! This could be customized
  !
  ! ZZZ This duplicates the same from Class !!!
  !---------------------------------------------------------------------
  type(fits_header_t), intent(out)   :: fhead  !
  logical,             intent(in)    :: check  !
  logical,             intent(inout) :: error  !
  ! Local
  integer(kind=4) :: icard
  !
  ! Read cards until END is reached
  fhead%ncard = 0
  do
    icard = fhead%ncard+1
    call gfits_get(fhead%diccard(icard)%key,fhead%diccard(icard)%val,  &
      check,error,continue=.true.,comment=fhead%diccard(icard)%comment)
    if (error)  return
    if (fhead%diccard(icard)%key.eq.'COMMENT')  cycle
    if (fhead%diccard(icard)%key.eq.'')         cycle  ! Must be a comment
    if (fhead%diccard(icard)%key.eq.'END')      exit
    if (fhead%diccard(icard)%key.eq.'HIERARCH') then
      call gfits_hierarch(fhead%diccard(icard)%key,fhead%diccard(icard)%val,error)
      if (error)  return
    endif
    ! That's a valid card: keep it
    fhead%ncard = icard
    fhead%sort(icard) = icard
  enddo
  !
  ! Compute the sorting array
  call gfits_setsort(fhead,error)
  if (error)  return
  !
contains
  !
  subroutine print_header()
    ! Debug
    integer(kind=4) :: jcard
    do icard=1,fhead%ncard
      jcard = fhead%sort(icard)
      write(*,'(I0,A)') jcard,':'
      write(*,'(2X,A)') fhead%diccard(jcard)%key
      write(*,'(2X,A)') fhead%diccard(jcard)%val
      write(*,'(2X,A)') fhead%diccard(jcard)%comment
    enddo
  end subroutine print_header
  !
end subroutine gfits_load_header
!
subroutine gfits_setsort(fh,error)
  use gfits_dependencies_interfaces
  use gfits_interfaces, except_this=>gfits_setsort
  use gfits_types
  !---------------------------------------------------------------------
  ! @ private
  !  Compute the sorting list of the input fits header (fh%sort)
  !  Remark: fh%sort() must contain the unsorted elements before
  ! entering gi0_quicksort_index_with_user_gtge. It can be partially
  ! sorted, or not at all, this will work.
  !---------------------------------------------------------------------
  type(fits_header_t), intent(inout) :: fh
  logical,             intent(inout) :: error
  !
  call gi0_quicksort_index_with_user_gtge(fh%sort,fh%ncard,  &
    gfits_sort_gt,gfits_sort_ge,error)
  if (error)  return
  !
contains
  !
  ! Sorting functions: just sort the cards by key names
  function gfits_sort_gt(m,l)
    logical :: gfits_sort_gt
    integer(kind=4), intent(in) :: m,l
    gfits_sort_gt = lgt(fh%diccard(m)%key,fh%diccard(l)%key)
  end function gfits_sort_gt
  !
  function gfits_sort_ge(m,l)
    logical :: gfits_sort_ge
    integer(kind=4), intent(in) :: m,l
    gfits_sort_ge = lge(fh%diccard(m)%key,fh%diccard(l)%key)
  end function gfits_sort_ge
  !
end subroutine gfits_setsort
!
subroutine gfits_find_value(fh,key,found,value,error)
  use gfits_dependencies_interfaces
  use gfits_interfaces, except_this=>gfits_find_value
  use gfits_types
  !---------------------------------------------------------------------
  ! @ private
  ! Get the value in the dictionary from the given key. Output is the
  ! raw character string (not evaluated).
  !---------------------------------------------------------------------
  type(fits_header_t), intent(in)    :: fh
  character(len=*),    intent(in)    :: key
  logical,             intent(out)   :: found
  character(len=*),    intent(out)   :: value
  logical,             intent(inout) :: error
  ! Local
  integer(kind=4) :: icard
  !
  value = ''
  found = .false.
  !
  ! Unefficient access looping on all elements:
  ! do icard=1,fh%ncard
  !   if (fh%diccard(icard)%key.eq.key) then
  !     found = .true.
  !     value = fh%diccard(icard)%val
  !     return
  !   endif
  ! enddo
  !
  ! Efficient access using dichotomic search within the sorted cards
  if (llt(key,fh%diccard(fh%sort(1       ))%key)  .or.  &
      lgt(key,fh%diccard(fh%sort(fh%ncard))%key)) then
    ! Key is beyond the sorted cards list: nothing to find
    return
  endif
  call gi0_dicho_with_user_ltgt(fh%ncard,.false.,icard,  &
    find_key_lt,find_key_gt,error)
  if (error)  return
  ! Using floor mode, result satisfies X(ival) <= xval < X(ival+1)
  found = fh%diccard(fh%sort(icard))%key.eq.key
  if (found)  value = fh%diccard(fh%sort(icard))%val
  !
contains
  function find_key_lt(icard)
    logical :: find_key_lt
    integer(kind=4) :: icard
    find_key_lt = llt(fh%diccard(fh%sort(icard))%key,key)
  end function find_key_lt
  function find_key_gt(icard)
    logical :: find_key_gt
    integer(kind=4) :: icard
    find_key_gt = lgt(fh%diccard(fh%sort(icard))%key,key)
  end function find_key_gt
end subroutine gfits_find_value
!
subroutine gfits_get_char(fh,key,found,value,error)
  use gfits_interfaces, except_this=>gfits_get_char
  use gfits_types
  !---------------------------------------------------------------------
  ! @ public-generic gfits_get_value
  ! Get the value in the dictionary from the given key, and evaluate it
  ! as character string. Quotes are removed if any. The output value is
  ! not modified if its key is not found.
  !---------------------------------------------------------------------
  type(fits_header_t), intent(in)    :: fh
  character(len=*),    intent(in)    :: key
  logical,             intent(out)   :: found
  character(len=*),    intent(inout) :: value
  logical,             intent(inout) :: error
  ! Local
  character(len=80) :: string
  !
  call gfits_find_value(fh,key,found,string,error)
  if (error .or. .not.found)  return
  !
  value = gfits_unquote(string)
  !
end subroutine gfits_get_char
!
subroutine gfits_get_inte(fh,key,found,value,error)
  use gbl_message
  use gfits_interfaces, except_this=>gfits_get_inte
  use gfits_types
  !---------------------------------------------------------------------
  ! @ public-generic gfits_get_value
  ! Get the value in the dictionary from the given key, and evaluate it
  ! as an integer. The output value is not modified if its key is not
  ! found.
  !---------------------------------------------------------------------
  type(fits_header_t), intent(in)    :: fh
  character(len=*),    intent(in)    :: key
  logical,             intent(out)   :: found
  integer(kind=4),     intent(inout) :: value
  logical,             intent(inout) :: error
  ! Local
  integer(kind=4) :: ier
  character(len=80) :: string
  !
  call gfits_find_value(fh,key,found,string,error)
  if (error .or. .not.found)  return
  !
  read(string,'(I20)',iostat=ier)  value
  if (ier.ne.0) then
    call gfits_message(seve%e,'FITS','Error decoding string '//trim(string)//' to I*4')
    error = .true.
    return
  endif
  !
end subroutine gfits_get_inte
!
subroutine gfits_get_long(fh,key,found,value,error)
  use gbl_message
  use gfits_interfaces, except_this=>gfits_get_long
  use gfits_types
  !---------------------------------------------------------------------
  ! @ public-generic gfits_get_value
  ! Get the value in the dictionary from the given key, and evaluate it
  ! as long integer. The output value is not modified if its key is not
  ! found.
  !---------------------------------------------------------------------
  type(fits_header_t), intent(in)    :: fh
  character(len=*),    intent(in)    :: key
  logical,             intent(out)   :: found
  integer(kind=8),     intent(inout) :: value
  logical,             intent(inout) :: error
  ! Local
  integer(kind=4) :: ier
  character(len=80) :: string
  !
  call gfits_find_value(fh,key,found,string,error)
  if (error .or. .not.found)  return
  !
  read(string,'(I20)',iostat=ier)  value
  if (ier.ne.0) then
    call gfits_message(seve%e,'FITS','Error decoding string '//trim(string)//' to I*8')
    error = .true.
    return
  endif
  !
end subroutine gfits_get_long
!
subroutine gfits_get_real(fh,key,found,value,error)
  use gbl_message
  use gfits_interfaces, except_this=>gfits_get_real
  use gfits_types
  !---------------------------------------------------------------------
  ! @ public-generic gfits_get_value
  ! Get the value in the dictionary from the given key, and evaluate it
  ! as single precision float. The output value is not modified if its
  ! key is not found.
  !---------------------------------------------------------------------
  type(fits_header_t), intent(in)    :: fh
  character(len=*),    intent(in)    :: key
  logical,             intent(out)   :: found
  real(kind=4),        intent(inout) :: value
  logical,             intent(inout) :: error
  ! Local
  integer(kind=4) :: ier
  character(len=80) :: string
  !
  call gfits_find_value(fh,key,found,string,error)
  if (error .or. .not.found)  return
  !
  read(string,*,iostat=ier)  value
  if (ier.ne.0) then
    call gfits_message(seve%e,'FITS','Error decoding string '//trim(string)//' to R*4')
    error = .true.
    return
  endif
  !
end subroutine gfits_get_real
!
subroutine gfits_get_dble(fh,key,found,value,error)
  use gbl_message
  use gfits_interfaces, except_this=>gfits_get_dble
  use gfits_types
  !---------------------------------------------------------------------
  ! @ public-generic gfits_get_value
  ! Get the value in the dictionary from the given key, and evaluate it
  ! as double precision float. The output value is not modified if its
  ! key is not found.
  !---------------------------------------------------------------------
  type(fits_header_t), intent(in)    :: fh
  character(len=*),    intent(in)    :: key
  logical,             intent(out)   :: found
  real(kind=8),        intent(inout) :: value
  logical,             intent(inout) :: error
  ! Local
  integer(kind=4) :: ier
  character(len=80) :: string
  !
  call gfits_find_value(fh,key,found,string,error)
  if (error .or. .not.found)  return
  !
  read(string,*,iostat=ier)  value
  if (ier.ne.0) then
    call gfits_message(seve%e,'FITS','Error decoding string '//trim(string)//' to R*8')
    error = .true.
    return
  endif
  !
end subroutine gfits_get_dble
