! (Copyright IRAM, the GILDAS team)
!
! Author: Stephane Guilloteau
!
! This file contains the full source code to make an Image from a
! UV data file in the "uvt" order (i.e. the natural channels / time order).
!
! It is memory limited, and requires the UV data set (and a copy of it in case
! it needs sorting) to fit into memory
!
! It will utilize memory as efficiently as possible, treating as many
! channels as possible at once. The memory size should be controlled by the
! SIC logical name GILDAS_SPACE (currently defaults to 128 Mbytes).
!
! To be done
!    - Implement a DFT option for consistency. ? See also UV_DFT
!    - Handle the case with one beam for a "few" channels
!    (which will imply also modifying CLEAN in consequence)
!
program p_uvmap
  use gkernel_interfaces
  use gildas_def
  use uvmap_def_task
  !
  ! Compute Dirty images from .TUV tables
  !
  character(len=filename_length) name,uvdata
  logical error
  type (par_uvmap) :: map
  !
  error = .false.
  call get_uvmap_par ('UV_MAP',uvdata,name,map,error)
  if (error) call sysexi(fatale)
  map%uvcode = code_uvt   ! Must be 1
  call t_uvmap (uvdata,name,map,error)
  if (error) call sysexi (fatale)
!
  contains
!<FF>
subroutine t_uvmap(uvdata,name,map,error)
  use gkernel_interfaces
  use mapping_interfaces
  use gbl_message
  use image_def
  use uvmap_def_task
  use phys_const
  !
  character(len=*), intent(in) :: uvdata   ! UV data file name
  character(len=*), intent(in) :: name     ! Output file name
  type (par_uvmap), intent(inout) :: map   ! Imaging parameters
  logical, intent(out) :: error
  !
  type (gridding) :: conv
  real uvmin,uvmax, cpu1, cpu0
  !
  character(len=*), parameter :: rname='UV_MAP'
  character(len=80) chain,mess
  type (gildas), target :: huv
  type (gildas), target :: hbeam
  type (gildas), target :: hdirty
  real, allocatable :: uuv(:), vuv(:)
  real, allocatable, target :: duv(:,:)
  real, pointer :: sduv(:,:)
  real, allocatable :: wuv(:), ouv(:)
  real, allocatable :: work(:)
  integer muv, nuv, ier, ic, jc, kc
  logical sort_data
  real, allocatable :: w_mapu(:), w_mapv(:)
  real, allocatable :: w_xgrid(:), w_ygrid(:), w_grid(:,:)
  complex, allocatable :: tfgrid(:,:,:)
  complex, allocatable :: ftbeam(:,:)
  real, pointer :: beam(:,:)
  real, allocatable, target :: dirty(:,:,:)
  real wold, wall, rmi, rma, uvmx
  real rms
  real space_gildas
  integer fcol, sblock, nblock, kz, iblock, istart
  integer nc,nx,ny,lx,ly,nd,nn(2),ndim
  integer imi,ima
  integer(kind=size_length) :: jmi,jma
  integer iv
  integer blc(4), trc(4)
  real null_taper(4)
  real uvcell(2)
  real support(2)
  real(8) local_freq
  !
  integer i,lt
  real df,dc
  real, parameter :: cell_precis=0.1
  integer nident
  !
  call gildas_null (huv, type = 'UVT')
  call gildas_null (hdirty)
  call gildas_null (hbeam)
  !
  error = .false.
  blc = 0
  trc = 0
  null_taper = 0.
  call gag_cpu(cpu0)
  !
  ! Read Headers
  call sic_parse_file(uvdata,' ','.uvt',huv%file)
  call gdf_read_header (huv,error)
  if (error) return
  !
  ! Correct for new phase center if required
  if (map%shift) then
    if (huv%gil%ptyp.eq.p_none) then
      call gagout('W-SHIFT,  No previous phase center info')
      huv%gil%a0 = huv%gil%ra
      huv%gil%d0 = huv%gil%dec
      huv%gil%pang = 0.
      huv%gil%ptyp = p_azimuthal
    elseif (huv%gil%ptyp.ne.p_azimuthal) then
      call gagout('W-SHIFT,  Previous projection type not SIN')
      huv%gil%ptyp = p_azimuthal
    endif
    call uv_shift (map%new,huv%gil%a0,huv%gil%d0,huv%gil%pang,map%off,map%shift)
    huv%gil%posi_words = 12
    huv%gil%proj_words = 9
  endif
  !
  ! Define channels and Center frequency
  call t_channel (huv,map)
  nc = map%channels(2)-map%channels(1)+1
  !
  ! Compute observing frequency, and new phase center in wavelengths
  if (map%shift) then
    sort_data = .true.
    huv%gil%a0 = map%new(1)
    huv%gil%d0 = map%new(2)
    huv%gil%pang = map%new(3)
    map%cs(1)  =  cos(map%off(3))
    map%cs(2)  = -sin(map%off(3))
    ! Note that the new phase center is counter-rotated because rotations
    ! are applied before phase shift.
    map%xy(1) = - map%freq * f_to_k * ( map%off(1)*map%cs(1) - map%off(2)*map%cs(2) )
    map%xy(2) = - map%freq * f_to_k * ( map%off(2)*map%cs(1) + map%off(1)*map%cs(2) )
  else
    sort_data = .false.
    map%xy = 0.0
    map%cs = (/1.0,0.0/)
  endif
  !
  ! Read whole data
  muv = huv%gil%dim(1)
  nuv = huv%gil%dim(2)
  huv%blc = 0
  huv%trc = 0
  allocate (duv(muv,nuv),sduv(muv,nuv),stat=huv%status)
  if (gildas_error(huv,rname,error)) return
  call gdf_read_data(huv,duv,error)
  if (gildas_error(huv,rname,error)) return
  !
  ! Apply phase shift and copy to output visibilities
  call t_uvsort (muv,nuv,duv,sduv,map%xy,map%cs,uvmax,uvmin,sort_data,error)
  deallocate (duv)
  !
  if (sort_data) then
    !
    ! Write header and data
!!     call gdf_update_header(huv,error)    !! That can be bugged on old data...
!!     call gdf_write_data(huv,sduv,error)  !!
    call gdf_write_image(huv,sduv,error)   ! Re-write in New Format
    call gag_cpu(cpu1)
    !
    write(chain,102) 'I-UV_MAP,  Finished Sort & Write CPU ',cpu1-cpu0
    call gagout(chain)
  endif
  !
  ! Define the Map characteristics
  call t_map ('UV_MAP',map,huv,uvmin,uvmax,conv)
  !
  ! Define blocking factor.
  ! Use logical name SPACE_GILDAS which defines what memory (in mega bytes)
  ! should be used
  space_gildas = 64.0  ! [MiB]
  ier = sic_ramlog('SPACE_MAPPING',space_gildas)
  sblock = max(int(256.0*space_gildas*1024.0)/(map%size(1)*map%size(2)),1)
  !
  ! Process sorted UV Table according to the type of beam produced
  !
  allocate (uuv(nuv),vuv(nuv),wuv(nuv),ouv(nuv),stat=ier)
  if (ier.ne.0) then
    call gagout('E-UV_MAP,  Cannot allocate Weight array')
    error = .true.
    return
  endif
  !
  uuv(:) = sduv(1,:)
  vuv(:) = sduv(2,:)
  !
  ! Code valid for One or Many beam, FFT or DFT, remains here
  nx = map%size(1)
  ny = map%size(2)
  nd = nx*ny
  allocate(w_mapu(nx),w_mapv(ny),work(2*nx),stat=ier)
  if (ier.ne.0) then
    call gagout('E-UV_MAP,  Cannot allocate U V axes arrays')
    error = .true.
    return
  endif
  !
  uvmx = uvmax / (map%freq*f_to_k)
  lx = (uvmx+map%support(1))/map%uvcell(1) + 2
  ly = (uvmx+map%support(2))/map%uvcell(2) + 2
  lx = 2*lx
  ly = 2*ly
  if (ly.gt.ny) then
    write(chain,'(a,a,f8.3)') 'E-UV_MAP,  Map cell is too large ', &
        & ' Undersampling ratio ',float(ly)/float(ny)
    call gagout(chain)
    ly = min(ly,ny)
    lx = min(lx,nx)
  else
    write(chain,'(a,a,f8.3)') 'I-UV_MAP,  Map cell ', &
        & ' Oversampling ratio ',float(ny)/float(ly)
    call gagout(chain)
  endif
  call docoor (lx,-map%uvcell(1),w_mapu)
  call docoor (ly,map%uvcell(2),w_mapv)
  !
  ! Gridding function
  allocate(w_xgrid(nx),w_ygrid(ny),w_grid(nx,ny),stat=ier)
  if (ier.ne.0) then
    call gagout('E-UV_MAP,  Cannot allocate gridding arrays')
    error = .true.
    return
  endif
  call grdtab (ny, conv%vbuff, conv%vbias, w_ygrid)
  call grdtab (nx, conv%ubuff, conv%ubias, w_xgrid)
  !
  ! TFGRID is allocated as the largest tolerable / required workspace
  !
  if (map%beam.eq.0) then
    ! Optimize SBLOCK now, allowing some additional memory if NBLOCK small
    if (sblock.gt.0) then
      nblock = (nc+sblock-1)/sblock
      kz = mod(nc,sblock)
      if (kz.ne.0 .and. kz.lt.(sblock/(nblock+1))) then
         if (nblock.ne.1) nblock = nblock-1
      endif
      sblock = (nc+nblock-1)/nblock
      kz = min(sblock,nc)
    else
      kz = nc
      nblock = 1
    endif
    !
    ! Another way to compute SBLOCK.
    ! Allow 10 % pixel difference at map edge (for example, cell_precis parameter)
    ! Relative Delta Frequency per channel is
    df = abs(huv%gil%fres/map%freq)
    ! so scale error at map edge is, in pixel units
    dc = df * max(map%size(1),map%size(2)) / 2
    ! so allowed number of channels with same beam is
    nident = max(1,nint(cell_precis/dc))
    !
    write(chain,'(i12)') nident
    mess = 'Maximum number of channels for same beam '//trim(adjustl(chain))//', Bandwidth '
    write(chain,'(f10.1)') nident*abs(huv%gil%fres)
    lt = len_trim(mess)+2
    mess(lt:) = trim(adjustl(chain))//' MHz'
    call map_message(seve%i,rname,mess)
    !
    call t_setbeam (huv,hbeam,map,2)     ! Define Beam header
    Print *,'Beam 2  Dimensions ',hbeam%gil%ndim, hbeam%gil%dim(1:3)
  else
    kz = 1 ! For one beam per channel
    call t_setbeam (huv,hbeam,map,3)     ! Define Beam header
    !
    Print *,'Beam 3 Dimensions ',hbeam%gil%ndim, hbeam%gil%dim(1:3)
  endif
  !
  call gag_cpu(cpu1)
  write(chain,102) 'I-UV_MAP,  Finished weighting CPU ',cpu1-cpu0
  call gagout(chain)
  !
  allocate(tfgrid(kz+1,lx,ly),ftbeam(nx,ny),stat=ier)
  if (ier.ne.0) then
    call gagout('E-UV_MAP,  Cannot allocate Fourier buffers')
    error = .true.
    return
  endif
  ! FFT setup
  nn(1) = nx
  nn(2) = ny
  ndim = 2
  work = 0
  call fourt_plan(ftbeam,nn,ndim,-1,1)
  !
  ! Prepare extrema
  rma = -1e38
  rmi = 1e38
  !
  allocate(dirty(nx,ny,kz),stat=ier)
  if (ier.ne.0) then
    call gagout('E-UV_MAP,  Cannot allocate Dirty buffer')
    error = .true.
    return
  endif
  !
  call t_setdirty(huv,hdirty,map,wall)
  call sic_parse_file(name,' ','.lmv',hdirty%file)
  call gdf_create_image(hdirty,error)
  !
  ! Beam allocation
  allocate(beam(nx,ny),stat=ier)
  if (ier.ne.0) then
    call gagout('E-UV_MAP,  Cannot allocate Beam')
    error = .true.
    return
  endif
  call t_setbeam (huv,hbeam,map,3)     ! Define Beam header
  call sic_parse_file(name,' ','.beam',hbeam%file)
  !
  ! Here comes the code which depends on options
  !
  if (map%beam.eq.0) then
    !
    ! Case for One Weight channel (a single dirty beam)
    if (map%channels(3).le.0) then
      wuv = 1.0
    else
      wuv(:) = sduv(huv%gil%fcol-1+3*map%channels(3),:)
    endif
    ouv(:) = wuv   ! Save original weights
    !
    !
    ! Compute the weights from this
    call t_doweig (nuv,uuv,vuv,wuv,map%uniform(1),map%uniform(2),error)
    if (error) return
    !
    ! Apply taper if needed
    call t_dotaper(nuv,uuv,vuv,wuv,map%taper)
    !
    ! Compute the noise (after re-weighting and tapering...)
    wall = sump(nuv,wuv)
    if (wall.eq.0.0) then
      write(chain,'(a,i0,a)') 'F-UV_MAP,  Plane ',map%channels(3), &
           & ' has Zero weight'
      call gagout(chain)
      error = .true.
      return
    else
      !
      ! Noise definition
      wall = 1e-3/sqrt(wall)
      call prnoise('UV_MAP','Natural',wall,rms)
      !
      ! Re-normalize the weights and re-count the noise
      call scawei (nuv,wuv,ouv,wall)
      wall = 1e-3/sqrt(wall)
      call prnoise('UV_MAP','Expected',wall,rms)
    endif
    !
    support = map%support
    uvcell = map%uvcell
    !
    ! Compute FFT's
    fcol = map%channels(1)
    call dofft (muv,nuv,        &   ! Size of visibility array
        &    sduv,             &   ! Visibilities
        &    1,2,              &   ! U, V pointers
        &    fcol,             &   ! First channel to map
        &    kz,lx,ly,         &   ! Cube size
        &    tfgrid,           &   ! FFT cube
        &    w_mapu,w_mapv,    &   ! U and V grid coordinates
        &    support,uvcell,null_taper, &  ! Gridding parameters
        &    wuv, vuv,                  &  ! Weight array + V Visibilities
        &    conv%ubias,conv%vbias,conv%ubuff,conv%vbuff,map%ctype)
    call gag_cpu(cpu1)
    write(6,102) 'I-UV_MAP,  Finished gridding CPU ',cpu1-cpu0
    !
    call extracs(kz+1,nx,ny,kz+1,tfgrid,ftbeam,lx,ly)
    call fourt  (ftbeam, nn,ndim,-1,1,work)
    call cmtore (ftbeam, beam ,nx,ny)
    call chkfft (beam, nx,ny,error)
    if (error) then
      call gagout('E-UV_MAP,  Inconsistent pixel size')
      return
    endif
    !
    ! Compute grid correction,
    ! Normalization factor is applied to grid correction, for further
    ! use on channel maps.
    !
    ! Make beam, not normalized
    call dogrid (w_grid,w_xgrid,w_ygrid,nx,ny,beam)  ! grid correction
    !
    ! Normalize and Free beam
    call docorr (beam,w_grid,nx*ny)
    !
    rma = -1e38
    rmi = 1e38
    call domima (beam,rmi,rma,imi,ima,nx*ny)
    !
    call t_setbeam (huv,hbeam,map,2)     ! Define Beam header
    call sic_parse_file(name,' ','.beam',hbeam%file)
    hbeam%gil%extr_words = 6
    hbeam%gil%rmax = rma
    hbeam%gil%rmin = rmi
    !
    ! Write beam
    hbeam%blc = 0
    hbeam%trc = 0
    call gdf_write_image(hbeam,beam,error)
    call gag_cpu(cpu1)
    write(chain,102) 'I-UV_MAP,  Finished beam CPU ',cpu1-cpu0
    call gagout(chain)
    deallocate(beam)
    ! Done Beam
    !
    ! Prepare image
    rma = -1e38
    rmi = 1e38
    !
    ! Loop on channels
    fcol = map%channels(1)
    !
    jma = 1
    jmi = 1
    kc = 1
    do iblock = 1,nblock
      !
      istart = fcol+(iblock-1)*sblock
      blc(3) = kc   ! Correct if only a subset is Imaged...
      kz = min (sblock,nc-sblock*(iblock-1))
      trc(3) = blc(3)-1+kz
      if (iblock.gt.1) then!
         call dofft (muv,nuv,      &   ! Size of visibility array
              &    sduv,             &   ! Visibilities
              &    1,2,              &   ! U, V pointers
              &    istart,           &   ! First channel to map
              &    kz,lx,ly,         &   ! Cube size
              &    tfgrid,           &   ! FFT cube
              &    w_mapu,w_mapv,    &   ! U and V grid coordinates
              &    support,uvcell,null_taper,   &  ! Gridding parameters
              &    wuv, vuv,                  & ! Weight array + V Visibilities
              &    conv%ubias,conv%vbias,conv%ubuff,conv%vbuff,map%ctype)
      endif
      do i=1,kz
         beam => dirty(:,:,i) !! = beam
         call extracs(kz+1,nx,ny,i,tfgrid,ftbeam,lx,ly)
         call fourt  (ftbeam,     nn,ndim,-1,1,work)
         call cmtore (ftbeam,     beam,nx,ny)
         call docorr (beam, w_grid, nd)
         call domima (beam, rmi,rma,imi,ima,nd)
         if (ima.ne.0) jma = ima+(kc-1)*nd
         if (imi.ne.0) jmi = imi+(kc-1)*nd
         kc = kc+1
      enddo
      ! Write the subset
      hdirty%blc(1:4) = blc
      hdirty%trc(1:4) = trc
      call gdf_write_data(hdirty,dirty,error)
      call gag_cpu(cpu1)
      write(6,'(a,i5,a,i5,a,f9.2)') 'I-UV_MAP,  Finished planes ', &
           & blc(3),' to ',trc(3),' CPU ',cpu1-cpu0
    enddo
    !
    !
  ELSE
    !
    ! Case for One Beam per Channel
    !
    call gdf_create_image(hbeam,error)
    hbeam%gil%extr_words = 0
    wold = 0 ! Last valid weight value, used for printout
    !
    kc = 1
    do ic =map%channels(1),map%channels(2)
      blc(3) = kc
      trc(3) = kc
      wuv(:) = sduv(7+3*ic,:)
      !
      wall = sump(nuv,wuv)
      if (wall.eq.0) then
         beam = 0
         dirty = 0
         cycle
      endif
      ouv(:) = wuv
      !
      ! Compute the weights from this
      call t_doweig (nuv,uuv,vuv,wuv,map%uniform(1),map%uniform(2),error)
      if (error) return
      !
      ! Apply taper if needed
      call t_dotaper(nuv,uuv,vuv,wuv,map%taper)
      !
      ! Re-normalize the weights and re-count the noise
      if (wall.ne.wold) then
         if (wold.eq.0) then
            wold = 1e-3/sqrt(wall)
            call prnoise('UV_MAP','Natural',wold,rms)
         endif
         wold = wall
         call scawei (nuv,wuv,ouv,wall)
         wall = 1e-3/sqrt(wall)
         call prnoise('UV_MAP','Expected',wall,rms)
      endif
      !
      ! Then compute the Dirty Beam
      local_freq =  huv%gil%val(1) + huv%gil%fres * (ic-huv%gil%ref(1))
      !
      uvcell = map%uvcell * map%freq / (local_freq)
      support = map%support * map%freq / (local_freq)
      call docoor (lx,-uvcell(1),w_mapu)
      call docoor (ly,uvcell(2),w_mapv)
      !
      ! Compute FFT's
      fcol = ic
      call dofft (muv,nuv,        &   ! Size of visibility array
           &    sduv,             &   ! Visibilities
           &    1,2,              &   ! U, V pointers
           &    fcol,             &   ! First channel to map
           &    kz,lx,ly,         &   ! Cube size
           &    tfgrid,           &   ! FFT cube
           &    w_mapu,w_mapv,    &   ! U and V grid coordinates
           &    support,uvcell,null_taper, &  ! Gridding parameters
           &    wuv, vuv,                  & ! Weight array + V Visibilities
           &    conv%ubias,conv%vbias,conv%ubuff,conv%vbuff,map%ctype)
      !
      call extracs(kz+1,nx,ny,kz+1,tfgrid,ftbeam,lx,ly)
      call fourt  (ftbeam, nn,ndim,-1,1,work)
      call cmtore (ftbeam, beam ,nx,ny)
      call chkfft (beam, nx,ny,error)
      if (error) then
         call gagout('E-UV_MAP,  Inconsistent pixel size')
         return
      endif
      !
      ! Compute grid correction,
      ! Normalization factor is applied to grid correction, for further
      ! use on channel maps.
      !
      ! Make beam, not normalized
      call dogrid (w_grid,w_xgrid,w_ygrid,nx,ny,beam)  ! grid correction
      !
      ! Normalize and Free beam
      call docorr (beam,w_grid,nx*ny)
      !
      ! Write beam
      hbeam%blc(1:4) = blc
      hbeam%trc(1:4) = trc
      call gdf_write_data(hbeam,beam,error)
      ! --- Done beam
      !
      call extracs(kz+1,nx,ny,1,tfgrid,ftbeam,lx,ly)
      call fourt  (ftbeam,     nn,ndim,-1,1,work)
      call cmtore (ftbeam,     beam,nx,ny)
      call docorr (beam, w_grid, nd)
      call domima (beam, rmi,rma,imi,ima,nd)
      if (ima.ne.0) jma = ima+(blc(3)-1)*nd
      if (imi.ne.0) jmi = imi+(blc(3)-1)*nd
      ! Write the subset
      hdirty%blc(1:4) = blc
      hdirty%trc(1:4) = trc
      call gdf_write_data(hdirty,beam,error)
      !
      kc = kc+1 ! Next channel
    enddo
    !
    call gdf_close_image(hbeam,error)
    deallocate(beam)
    !
  ENDIF
  !
  ! Set extrema section
  call t_setextrema(hdirty,rmi,jmi,rma,jma)
  ! Set noise section
  hdirty%gil%noise = wall
  !
  call gdf_update_header(hdirty,error)
  call gdf_close_image(hdirty,error)
  !
  call gag_cpu(cpu1)
  write(chain,'(a,f9.2)') 'I-UV_MAP,  Finished maps CPU ',cpu1-cpu0
  call gagout(chain)
  !
  deallocate(dirty)
  !
  deallocate(tfgrid,ftbeam,sduv)
  deallocate(w_mapu,w_mapv,w_grid)
  deallocate(uuv,vuv,wuv,ouv)
  !
102 FORMAT(A,F9.2)
end subroutine t_uvmap
!
end program p_uvmap
