function togdf(fits,istyle,what,check,error,getsymbol,wpr)
  use phys_const
  use gbl_message
  use image_def
  use gio_dependencies_interfaces
  use gio_interfaces, except_this=>togdf
  use gio_params
  use gio_fitsdef
  !---------------------------------------------------------------------
  ! @ public
  !     Read a FITS file from tape and convert it to GDF format in
  !     virtual memory.
  !     From the header, we have NX*NY integer pixel values of NBITS
  !     each to read. The blocksize is 2880 Bytes.
  !     The conversion to REAL*4 is done using BSCAL and BZERO
  !---------------------------------------------------------------------
  integer(kind=4) :: togdf  ! Function value on return
  type(gildas),    intent(inout) :: fits       !
  integer(kind=4), intent(in)    :: istyle     ! A cryptic style code, 1,2 or something else
  logical,         intent(in)    :: what       ! Prompt on unknown axis types ?
  logical,         intent(in)    :: check      ! Verbose flag
  logical,         intent(out)   :: error      ! Error flag
  external                       :: getsymbol  !
  external                       :: wpr        !
  ! Local
  character(len=*), parameter :: rname='FITS'
  logical :: have_altrpix, have_altrval, have_velref, have_velo, have_restfreq
  real(8) :: x, rblank, v, vv, f, altrpix, altrval
  real(4) :: xxx,yyy
  integer, parameter :: fits_naxis=7
  integer(kind=index_length) :: nr(fits_naxis)
  integer(4) :: naxis, iproj, plus, moins, cd_matrix, naxisinfile
  integer(4) :: i, i1,i2, narg
  integer :: l,ier,iaxis,iv
  real(4) :: new_epoch
  real(8) :: rota(gdf_maxdims),ihap_ra,ihap_de,cd(gdf_maxdims,gdf_maxdims)
  real(8) :: ratio1,ratio2,angle1,angle2
  character(len=8) :: trans,comm
  character(len=1) :: answer
  character(len=70) :: argu, cmore
  character(len=12) :: rccode(gdf_maxdims),cunit(gdf_maxdims)
  character(len=8) :: dummy
  character(len=64) :: filter
  character(len=12) :: projection,chain
  character(len=filename_length) :: old_gdfname
  logical :: blanki,ihap,lhap_ra,lhap_de,err
  character(len=message_length) :: mess
  real(8) :: convert(3,gdf_maxdims), xxproj(3), rswap(3), my_convert(3,gdf_maxdims)
  real(8) :: scale, values(3)
  logical :: has_cdelt(gdf_maxdims), has_cdij(gdf_maxdims)
  !
  togdf = 0
  !
  ! Reset
  my_convert = 0.d0       ! Default conversion formula
  my_convert(1,:) = 1.d0  ! Default reference pixel is 1
  error = .false.
  call gildas_null(fits)
  !
  fits%gil%spec_words = 0
  fits%gil%extr_words = 0
  fits%gil%reso_words = 0
  fits%gil%epoc = equinox_null
  !
  cd_matrix = 0
  cd = 0
  has_cdelt(:) = .false.
  has_cdij(:) = .false.
  !
  v = 0.
  vv = 0.
  f = 0.
  altrpix = 0.d0
  have_altrpix = .false.
  altrval = 0.d0
  have_altrval = .false.
  have_velref = .false.
  have_velo = .false.
  have_restfreq = .false.
  old_gdfname = gdfname
  call sic_parsef(gdfname,fits%file,' ','.gdf')
  blanki = .false.
  bscal = 1.0
  bzero = 0.0
  ihap = .false.
  lhap_ra = .false.
  lhap_de = .false.
  filter = ' '    ! Presumably some ISO satellite stuff ?
  ihap_ra = 0
  ihap_de = 0
  rota = 0.0d0
  rccode = ' '
  cunit(:) = ' '
  !
  ! Use pixel units as default if increments are not defined
  call gfits_flush(error)
  if (error) return
  ! First line SIMPLE
  call gfits_get(comm,argu,check,error)
  if (error) return
  if (comm.ne.'SIMPLE') then
    call gio_message(seve%e,rname,'Not a FITS file at all')
    goto 99
  endif
  narg = len(argu)
  call sic_blanc(argu,narg)
  if (argu.ne.' T' .and. argu.ne.'T') then
    call gio_message(seve%w,rname,'Not a SIMPLE FITS file, trying...')
  endif
  !
  ! Second line BITPIX
  call gfits_get(comm,argu,check,error)
  if (error) return
  ! JLM compatibility patch
  if (comm.eq.'BLOCKED') then
    call gfits_get(comm,argu,check,error)
    if (error) return
  endif
  ! end patch
  !
  if (comm.ne.'BITPIX') then
    call gio_message(seve%e,rname,'Not a standard FITS file, no BITPIX')
    goto 99
  endif
  narg = len(argu)
  call sic_blanc(argu,narg)
  read(argu,*,iostat=ier) x
  nbit=nint(x)
  if (nbit.ne.8 .and. nbit.ne.16 .and. nbit.ne.32 .and. nbit.ne.-32 .and.  &
      nbit.ne.-64) then
    write(mess,101) 'Cannot handle ',nbit,' bits'
    call gio_message(seve%e,rname,mess)
    goto 99
  endif
  ! Set default blanking value for floating point FITS files
  if (nbit.eq.-32.or.nbit.eq.-64) then
    fits%gil%bval = bval0
    fits%gil%eval = -1
    fits%gil%blan_words = 0
  endif
  !
  ! Third line NAXIS
  call gfits_get(comm,argu,check,error)
  if (error) return
  read(argu,*,iostat=ier) x
  naxis=int(x)
  if (naxis.lt.1) then
    call gio_message(seve%e,rname,'No image in file')
    goto 99
  endif
  if (naxis.gt.fits_naxis) then
    call gio_message(seve%e,rname,'Too many dimensions')
    goto 99
  endif
  !
  ! Next 'NAXIS' lines are axis dimensions NAXIS1, NAXIS2, ...
  do iaxis=1,naxis
    call gfits_get(comm,argu,check,error)
    if (error) return
    dummy='NAXIS'//char(48+iaxis)
    if (comm(1:8).ne.dummy(1:8)) then
      call gio_message(seve%e,rname,dummy//' keyword not found')
      goto 99
    endif
    read(argu,*,iostat=ier) x
    if (ier.ne.0) call gio_message(seve%e,rname,'Error reading '//comm//argu)
    nr(iaxis)=nint(x)
  enddo
  !
  ! UVFITS?
  if (nr(1).eq.0) then
    call gio_message(seve%w,rname,'FITS file may be UVFITS, trying...')
    gdfname = old_gdfname
    togdf = 1
    return
  endif
  !
  ! Check dimensions and degeneracy if required
  naxisinfile = naxis  ! save naxis as written in fits file, in order to read header informations
  iaxis = naxis
  no_degen = .true.  ! Should be customizable at some point
  do while (iaxis.ge.1)
    ! Get rid of last degenerate dimensions (if required)
    if (no_degen .and. iaxis.eq.naxis .and. nr(iaxis).eq.1) then
      naxis = iaxis-1
    endif
    ! Reject more than gdf_maxdims non-degenerate dimensions
    if (iaxis.gt.gdf_maxdims .and. nr(iaxis).ne.1) then
      call gio_message(seve%e,rname,'Too many dimensions')
      goto 99
    endif
    !
    iaxis = iaxis-1
  enddo
  !
  ! Maximum GDF_MAXDIMS axis allowed
  fits%gil%ndim = min(naxis,gdf_maxdims)
  do iaxis = 1,fits%gil%ndim
    fits%gil%dim(iaxis) = nr(iaxis)
  enddo
  ier = 0
  !
  ! Now find the real information.
  !
20 if (ier.ne.0)  call gio_message(seve%e,rname,'Error reading '//comm//argu)
  ier = 0
  call gfits_get(comm,argu,check,error,comment=cmore)
  if (error) return
  call getsymbol (comm,trans,error)
  if (error) then
    error = .false.
  else
    comm = trans
  endif
  !
  if (comm.eq.'BITPIX') then
    read(argu,*,iostat=ier) x
    nbit=nint(x)
    if (nbit.ne.8 .and. nbit.ne.16 .and. nbit.ne.32 .and. nbit.ne.-32 .and.  &
        nbit.ne.-64) then
      write(mess,101) 'Cannot handle ',nbit,' bits'
      call gio_message(seve%e,rname,mess)
      goto 99
    endif
  elseif (comm.eq.'BSCALE') then   ! Tape scaling factor
    read(argu,*,iostat=ier) bscal
  elseif (comm.eq.'BZERO') then    ! Tape offset
    read(argu,*,iostat=ier) bzero
  elseif (comm.eq.'BUNIT') then    ! Array units
    i1 = index(argu,'''')+1
    i2 = index(argu(i1:),'''')-2+i1
    l = i2-i1+1
    call no_blanc(argu(i1:i2),fits%char%unit,l)
  elseif (comm.eq.'BLANK') then
    read(argu,*,iostat=ier) rblank
    blanki = .true.
    !
    ! Axis Keywords
  elseif (comm(1:5).eq.'CRVAL' .and. comm(7:).eq.'') then   ! Value at reference pixel
    read(comm(6:6),'(I1)') iaxis
    read(argu,*,iostat=ier) my_convert(2,iaxis) !! fits%gil%val1
  elseif (comm(1:5).eq.'CRPIX' .and. comm(7:).eq.'') then   ! Reference pixel
    read(comm(6:6),'(I1)') iaxis
    read(argu,*,iostat=ier) my_convert(1,iaxis) !! fits%gil%ref1
  elseif (comm(1:5).eq.'CDELT' .and. comm(7:).eq.'') then   ! Increment
    read(comm(6:6),'(I1)') iaxis
    read(argu,*,iostat=ier) my_convert(3,iaxis) !! fits%gil%inc1
    has_cdelt(iaxis) = .true.
  elseif (comm(1:5).eq.'CTYPE' .and. comm(7:).eq.'') then   ! Type of coordinate
    read(comm(6:6),'(I1)') iaxis
    i1 = index(argu,'''')+1
    i2 = index(argu(i1:),'''')-2+i1
    l = i2-i1+1
    call no_blanc(argu(i1:i2),rccode(iaxis),l)
  elseif (comm(1:5).eq.'CUNIT' .and. comm(7:).eq.'') then   ! Axis unit
    read(comm(6:6),'(I1)') iaxis
    i1 = index(argu,'''')+1
    i2 = index(argu(i1:),'''')-2+i1
    l = i2-i1+1
    call no_blanc(argu(i1:i2),cunit(iaxis),l)
  elseif (comm(1:5).eq.'CROTA' .and. comm(7:).eq.'') then
    read(comm(6:6),'(I1)') iaxis
    read(argu,*,iostat=ier)  rota(iaxis)
    !
    ! CDi_j Matrix. Order needs to be checked in FITS reference document
  elseif (comm.eq.'CD1_1') then
    read(argu,*,iostat=ier)  cd(1,1)
    has_cdij(1) = .true.
    cd_matrix = cd_matrix+1
  elseif (comm.eq.'CD2_1') then
    read(argu,*,iostat=ier)  cd(2,1)
    cd_matrix = cd_matrix+1
  elseif (comm.eq.'CD1_2') then
    read(argu,*,iostat=ier)  cd(1,2)
    cd_matrix = cd_matrix+1
  elseif (comm.eq.'CD2_2') then
    has_cdij(2) = .true.
    read(argu,*,iostat=ier)  cd(2,2)
    cd_matrix = cd_matrix+1
    !
    ! This one can happen too instead of CDELT3
  elseif (comm.eq.'CD3_3') then
    has_cdij(3) = .true.
    read(argu,*,iostat=ier)  cd(3,3)
    !
    ! Miscellaneous
    !
  elseif (comm.eq.'DATAMAX') then
    read(argu,*,iostat=ier)  fits%gil%rmax
    fits%gil%extr_words = max(fits%gil%extr_words,2)
  elseif (comm.eq.'DATAMIN') then
    read(argu,*,iostat=ier)  fits%gil%rmin
    fits%gil%extr_words = max(fits%gil%extr_words,1)
  elseif (comm.eq.'OBJECT') then
    i1 = index(argu,'''')+1
    i2 = index(argu(i1:),'''')-2+i1
    if (i2.lt.i1) i2=len(argu)
    l = i2-i1+1
    call no_blanc(argu(i1:i2),fits%char%name,l)
  elseif (comm.eq.'LINE' .or. comm.eq.'LINENAME') then
    i1 = index(argu,'''')+1
    i2 = index(argu(i1:),'''')-2+i1
    if (i2.lt.i1) i2=len(argu)
    l = i2-i1+1
    call no_blanc(argu(i1:i2),fits%char%line,l)
  elseif (comm.eq.'FLTRNR') then
    ! What is this "filter" stuff ?
    i1 = index(argu,'''')+1
    i2 = index(argu(i1:),'''')-2+i1
    if (i2.lt.i1) i2=len(argu)
    l = i2-i1+1
    call no_blanc(argu(i1:i2),filter,l)
  elseif (comm.eq.'TELESCOP') then
    i1 = index(argu,'''')+1
    i2 = index(argu(i1:),'''')-2+i1
    call gdf_addteles(fits,'TELE',argu(i1:i2),values,error)
  elseif (comm.eq.'EPOCH'.or.comm.eq.'EQUINOX') then
    read(argu,*,iostat=ier) fits%gil%epoc
    if (ier.ne.0) then
      if (argu(2:2).eq.'J') then
        read(argu(3:),1003,iostat=ier) fits%gil%epoc
      elseif (argu(2:2).eq.'B') then
        read(argu(3:),1003,iostat=ier) fits%gil%epoc
      endif
    endif
    if (ier.ne.0) then
      call gio_message(seve%e,rname,'Undecipherable Equinox '//argu)
      fits%gil%epoc = equinox_null
    endif
  elseif (comm.eq.'RADECSYS') then !see Greisen & Calabretta 1997
    fits%char%syst = 'EQUATORIAL'
    new_epoch = fits%gil%epoc
    if (argu(2:9).eq.'FK4-NO-E') then
      new_epoch = 1950.0
    elseif (argu(2:4).eq.'FK4') then
      new_epoch = 1950.0
    elseif (argu(2:4).eq.'FK5') then
      new_epoch = 2000.0
    elseif (argu(2:6).eq.'GAPPT') then
      new_epoch = equinox_null
    endif
    if (fits%gil%epoc.ne.new_epoch .and. fits%gil%epoc.ne.equinox_null) &
      & call gio_message(seve%e,rname,'Equinox reset from RADECSYS')
    fits%gil%epoc = new_epoch
  elseif (comm.eq.'BMAJ') then
    fits%gil%reso_words = 3
    read(argu,*,iostat=ier)  fits%gil%majo
    fits%gil%majo = fits%gil%majo*pi/180.0
  elseif (comm.eq.'BMIN') then
    read(argu,*,iostat=ier)  fits%gil%mino
    fits%gil%mino = fits%gil%mino*pi/180.0
  elseif (comm.eq.'BPA') then
    read(argu,*,iostat=ier)  fits%gil%posa
    fits%gil%posa = fits%gil%posa*pi/180.0
    !
    ! Frequency / Velocity axis
  elseif (comm.eq.'SPECSYS') then
    i1 = index(argu,'''')+1
    i2 = index(argu(i1:),'''')-2+i1
    chain = argu(i1:i2)
    if (chain.eq.'LSRK') then
      fits%gil%vtyp = vel_lsr
    else if (chain.eq.'HEL') then
      fits%gil%vtyp = vel_hel
    else if (chain.eq.'TOPOCENT') then
      fits%gil%vtyp = vel_obs
    endif
  elseif (comm.eq.'VELO-LSR') then
    read(argu,*,iostat=ier)  v
    have_velo = .true.
    fits%gil%spec_words = 12
  elseif (comm.eq.'VELREF') then
    read(argu,*,iostat=ier)  vv
    have_velref = .true.
    fits%gil%spec_words = 12
  elseif (comm.eq.'RESTFREQ' .or. comm.eq.'RESTFRQ') then  ! CASA patch
    read(argu,*,iostat=ier)  f
    have_restfreq = .true.
    fits%gil%spec_words = 12
!
! This code was for FCRAO CO survey, but is highly non standard.
!  elseif (comm.eq.'LINEFREQ') then
!    read(argu,*,iostat=ier)  f
!    have_restfreq = .true.
!    ! Unit may be in comment field
!    ! normal usage is e.g. [GHz], but some FITS writers may write simply GHZ
!    cmore = adjustl(cmore)
!    call unit_prefix_scale(cmore,scale,'Hz',error)
!    f = f*scale
  elseif (comm.eq.'ALTRVAL') then
    read(argu,*,iostat=ier)  altrval
    have_altrval = .true.
    fits%gil%spec_words = 12
  elseif (comm.eq.'ALTRPIX') then
    read(argu,*,iostat=ier)  altrpix
    have_altrpix = .true.
    fits%gil%spec_words = 12
    !
    ! IHAP compatibility patch (may be specific to OHP version??). Position
    ! information is stored in custom keywords instead of making use of the
    ! standard axis mechanism.
  elseif (comm(1:6).eq.'POSTN-') then
    ihap = .true.
    read(argu,*,iostat=ier) x
    if (comm(7:8).eq.'RA') then
      ihap_ra = x
    elseif (comm(7:8).eq.'DE') then
      ihap_de = x
    else
      call gio_message(seve%w,rname,'Unknown IHAP position keyword '//comm)
    endif
    !
    ! GILDAS specific code for RA and DEC
  elseif (comm.eq.'RA') then
    if (argu(1:1).eq."'") then
      i2 = len_trim(argu)-1
      ! Assume this is a sexagesimal notation in a text string
      call sic_decode(argu(2:i2),x,24,err)
    else
      read(argu,*,iostat=ier) x
      x = x*pi/180d0
    endif
    lhap_ra = .true.
    ihap_ra = x
  elseif (comm.eq.'DEC') then
    if (argu(1:1).eq."'") then
      ! Assume this is a sexagesimal notation in a text string
      i2 = len_trim(argu)-1
      call sic_decode(argu(2:i2),x,360,err)
    else
      read(argu,*,iostat=ier) x
      x = x*pi/180d0
    endif
    lhap_de = .true.
    ihap_de = x
  elseif (comm.eq.'END') then
    goto 30
  endif
  goto 20
  !
  ! Decode from FITS syntax to internal header structure
30 continue
  ! IHAP compatibility patch
  if (rccode(1).eq.' '.and.rccode(2).eq.' '.and.ihap) then
    call gio_message(seve%w,rname,'Only primitive positional information, '//  &
    'presumably from IHAP.')
    call gio_message(seve%w,rname,'Will try to make sense out of this '//  &
    'mess, but success is not guaranteed')
    ! Assume first two axes are RA and Dec, and take mid pixel as reference
    rccode(1) = 'RA'
    rccode(2) = 'DEC'
    my_convert(2,1) = ihap_ra
    my_convert(2,2) = ihap_de
    my_convert(1,1) = fits%gil%dim(1)/2.d0
    my_convert(1,2) = fits%gil%dim(2)/2.d0
  endif
  if (fits%char%line.eq.' '.and.filter.ne.' ') fits%char%line = filter
  !
  ! Decode CD Matrix if present
  if (cd_matrix.ne.0) then
    call gio_message(seve%w,rname,'Using CD Matrix')
    if (cd_matrix.ne.4) then
      call gio_message(seve%w,rname,'CDi_j matrix is incomplete')
    endif
    ratio1 = cd(1,1)/cd(2,2)
    ratio2 = sqrt(cd(1,1)**2+cd(2,1)**2) / sqrt(cd(1,2)**2+cd(2,2)**2)
    if ((abs(ratio1)-ratio2)/ratio2.gt.1e-4) then
      ! This is not A rotation, but a more complex matrix
      call gio_message(seve%w,rname,'CDi_j matrix is skewed')
    endif
    !
    ! Verification de coherence: les CDELTi ne doivent pas etre presents
    if (has_cdelt(1).or.has_cdelt(2)) then
      call gio_message(seve%w,rname,'FITS file error: CDi_j overrides CDELTi increment')
    endif
    my_convert(3,1) = sqrt(cd(1,1)**2+cd(2,1)**2)
    my_convert(3,2) = sqrt(cd(1,2)**2+cd(2,2)**2)
    if (ratio1/ratio2.lt.0) then
      ! One axis has been inverted
      ! Assume this is the first one ...
      cd(1,1) = -cd(1,1)
      cd(2,1) = -cd(2,1)
      my_convert(3,1) = -my_convert(3,1)
    endif
    !
    ! Rotation ...
    angle1 = atan2(cd(2,1),cd(1,1))
    angle2 = -atan2(cd(1,2),cd(2,2))
    rota(1) = angle1*180.0/pi
    rota(2) = angle2*180.0/pi  ! Must be in degree
    !
  endif
  !
  ! Check if CDi_i diagonal matrix has been used instead of CDELTi
  do i=3,gdf_maxdims
    if (has_cdij(i)) then
      !
      ! Verification de coherence: les CDELTi ne doivent pas etre presents
      if (has_cdelt(i)) then
        call gio_message(seve%w,rname,'FITS file error: CDi_j overrides CDELT3i increment')
      endif
      my_convert(3,i) = cd(i,i)
    endif
  enddo
  !
  ! Setup a default increment if Zero
  do i=1,gdf_maxdims
    if (my_convert(3,i).eq.0.0d0) my_convert(3,i) = 1.0d0
  enddo
  !
  ! Setup the unit types for correct astronomical labelling
  fits%gil%ptyp = p_none
  convert = my_convert
  call r8tor8 (fits%gil%a0,xxproj,3)
  !
  do i=1,min(naxisinfile,gdf_maxdims)   ! loop on naxisinfile
    !
    ! Check for standard projections
    moins = index(rccode(i),'-')
    iproj = 0
    if (moins.ne.0) then
      plus = moins+1
      do while (plus.le.len(rccode(i)) .and. rccode(i)(plus:plus).eq.'-')
        plus = plus+1
      enddo
      projection = rccode(i)(plus:)
      rccode(i) = rccode(i)(1:moins-1)
    else
      projection = ' '
    endif
    !
    ! These are patches for old AIPS tapes and CPC tapes
    if (rccode(i).eq.'LL' .or. (rccode(i).eq.'X' .and. istyle.eq.2)) then
      fits%gil%ptyp = p_gnomonic
      iproj = 1
      fits%char%code(i) = 'RA'
      fits%char%syst = 'EQUATORIAL'
      xxproj(1) = convert(2,i)*pi/180.d0
      convert(2,i) = 0.d0
      convert(3,i) = convert(3,i)*pi/180.d0
      if (istyle.eq.2) convert(3,i) = -convert(3,i)
      fits%gil%ra = xxproj(1)
    elseif (rccode(i).eq.'MM' .or. (rccode(i).eq.'Y' .and. istyle.eq.2)) then
      fits%gil%ptyp = p_gnomonic
      iproj = 2
      fits%char%code(i) = 'DEC'
      fits%char%syst = 'EQUATORIAL'
      xxproj(2) = convert(2,i)*pi/180.d0
      convert(2,i) = 0.d0
      convert(3,i) = convert(3,i)*pi/180.d0
      fits%gil%dec = xxproj(2)
      !
      ! Standard FITS projections definitions.
    else
      select case(rccode(i))
      case ('RA')
        fits%char%code(i) = 'RA'
        fits%char%syst = 'EQUATORIAL'
        convert(2,i) = convert(2,i)*pi/180.d0
        convert(3,i) = convert(3,i)*pi/180.d0
        fits%gil%ra = convert(2,i)
        iproj = 1
      case ('DEC')
        fits%char%code(i) = 'DEC'
        fits%char%syst = 'EQUATORIAL'
        convert(2,i) = convert(2,i)*pi/180.d0
        convert(3,i) = convert(3,i)*pi/180.d0
        fits%gil%dec = convert(2,i)
        iproj = 2
      case ('LAT','GLAT')
        fits%char%code(i) = 'BII'
        fits%char%syst = 'GALACTIC'
        convert(2,i) = convert(2,i)*pi/180.d0
        convert(3,i) = convert(3,i)*pi/180.d0
        fits%gil%bii = convert(2,i)
        iproj = 2
      case ('LON','GLON')
        fits%char%code(i) = 'LII'
        fits%char%syst = 'GALACTIC'
        convert(2,i) = convert(2,i)*pi/180.d0
        convert(3,i) = convert(3,i)*pi/180.d0
        fits%gil%lii = convert(2,i)
        iproj = 1
      case ('VELOCITY','VELO','VRAD')
        fits%char%code(i) = 'VELOCITY'
        fits%gil%faxi = i
        fits%gil%spec_words = 12
      case ('FREQUENCY','FREQ')
        fits%char%code(i) = 'FREQUENCY'
        fits%gil%faxi = i
        fits%gil%spec_words = 12
      case ('LAMBDA')
        fits%char%code(i) = 'LAMBDA'
        fits%gil%faxi = i
        fits%gil%spec_words = 12
      case ('STOKES')
        fits%char%code(i) = 'STOKES'
      case default
        fits%char%code(i) = rccode(i)
        if (what) then
          write(mess,100) 'Unknown axis type ',rccode(i),' for axis ',i
          call gio_message(seve%w,rname,mess)
          call wpr('Is the axis unit Degrees ? [Y/N]',answer)
          if (answer.eq.'Y' .or. answer.eq.'y') then
            call gio_message(seve%w,rname,'Converting from degrees to radians')
            convert(2,i) = convert(2,i)*pi/180.d0
            convert(3,i) = convert(3,i)*pi/180.d0
            fits%char%code(i) = 'ANGLE'
          endif
        endif
      end select
    endif
    !
    ! Some warnings...
    if (fits%char%code(i).eq.'FREQUENCY') then
      write(mess,'(A,I0,A)') 'Axis #',i,' is FREQUENCY, support is not yet fully supported - Proceed at your own risk'
      call gio_message(seve%w,rname,mess)
      call gio_message(seve%i,rname,'CASA users: you may re-export your cube with the task exportfits and argument velocity=True')
    endif
    !
    ! Convert from MKSA units to internal.
    if (fits%char%code(i).eq.'VELOCITY') then
      if (cunit(i).ne.'km/s') then
        ! Assume 'm/s', or nothing = MKSA
        convert(2,i) = convert(2,i)*1.0d-3
        convert(3,i) = convert(3,i)*1.0d-3
      endif
    elseif (fits%char%code(i).eq.'FREQUENCY') then
      call unit_prefix_scale(cunit(i),scale,'Hz',error)
      scale = scale*1.0d-6
      if (error) then
        write(mess,'(A,I0,A,A,A)') 'Axis #',i,fits%char%code(i),' -- unsupported unit ',cunit(i)
        call gio_message(seve%w,rname,mess)
      endif
      convert(2:3,i) = convert(2:3,i)*scale
    endif
    !
    if (iproj.ne.0) then
      if (istyle.eq.1) then    ! Radio projection for CPC-IRAS patch
        xxproj(iproj) = convert(2,i)
        convert(2,i) = 0.0
        fits%gil%ptyp = p_radio
      elseif (projection.eq.'TAN') then
        xxproj(iproj) = convert(2,i)
        convert(2,i) = 0.0
        fits%gil%ptyp = p_gnomonic          ! Gnomonic
      elseif (projection.eq.'ATF') then
        xxproj(iproj) = convert(2,i)
        convert(2,i) = 0.0
        fits%gil%ptyp = p_aitoff            ! Aitof
      elseif (projection.eq.'SIN') then
        ! Note that Gildas does not support extended SIN (Slant
        ! orthographic). See Calabretta & Greisen 2002, sections
        ! 5.1.5 and 6.1.1. We should reject the cases when PVi_j
        ! are defined and non-zero.
        xxproj(iproj) = convert(2,i)
        convert(2,i) = 0.0
        fits%gil%ptyp = p_ortho             ! Orthographic or Dixon
      elseif (projection.eq.'ARC') then
        xxproj(iproj) = convert(2,i)
        convert(2,i) = 0.0
        fits%gil%ptyp = p_azimuthal         ! Schmidt or Azimuthal
      elseif (projection.eq.'STG') then
        xxproj(iproj) = convert(2,i)
        convert(2,i) = 0.0
        fits%gil%ptyp = p_stereo            ! Stereographic
      elseif (projection.eq.'GLS') then
        xxproj(iproj) = convert(2,i)
        convert(2,i) = 0.0
        fits%gil%ptyp = p_radio             ! Radio
      elseif (projection.eq.'CAR') then
        xxproj(iproj) = convert(2,i)
        convert(2,i) = 0.0
        fits%gil%ptyp = p_cartesian         ! Cartesian
      elseif (projection.eq.'NCP') then
        ! North Celestial Pole: Gildas offers native support.
        ! Note that according to Calabretta & Greisen 2002, NCP is
        ! obsolete and should be translated to (see section 6.1.2):
        !   SIN with PV2_1 = 0 and PV2_2 = 1/tan(d0)
        ! This defines an "extented" SIN projection (Slant orthographic,
        ! see section 5.1.5).
        ! However, Gildas supports only the (non-extended) orthographic
        ! projection with SIN with PV2_1 = PV2_2 = 0 (see section 6.1.1).
        xxproj(iproj) = convert(2,i)
        convert(2,i) = 0.0
        fits%gil%ptyp = p_ncp               ! North Celestial Pole
      else
        if (projection.ne.' ')  &
          call gio_message(seve%w,rname,'Unrecognized projection '//projection)
        fits%gil%ptyp = p_none
      endif
      if (iproj.eq.1) then
        fits%gil%xaxi = i
      elseif (iproj.eq.2) then
        fits%gil%yaxi = i
      endif
      xxproj(3) = pi*rota(i)/180.0d0
      fits%gil%pang = xxproj(3)
    else
      if (rota(i).ne.0d0) then
        write(mess,101) 'Axis',i,' rotation ignored'
        call gio_message(seve%w,rname,mess)
      endif
    endif
  enddo
  !
  ! Pointing center RA -- DEC can differ from Projection center A0 -- D0
  if (ihap) then
    fits%gil%ra = ihap_ra
    fits%gil%dec = ihap_de
  endif
  if (lhap_ra) fits%gil%ra = ihap_ra
  if (lhap_de) fits%gil%dec = ihap_de
  !
  xxx=0.0
  yyy=0.0
  !
  ! The projection center MUST NOT be reset here...
  if (fits%char%syst.eq.'EQUATORIAL') then
    if (fits%gil%epoc.ne.equinox_null) &
    call equ_to_gal(fits%gil%ra,fits%gil%dec,xxx,yyy,fits%gil%epoc,  &
                    fits%gil%lii,fits%gil%bii,xxx,yyy,error)
  elseif (fits%char%syst.eq.'GALACTIC') then
    if (fits%gil%epoc.eq.equinox_null) fits%gil%epoc = 2000.0
    call gal_to_equ(fits%gil%lii,fits%gil%bii,xxx,yyy,  &
                    fits%gil%ra,fits%gil%dec,xxx,yyy,fits%gil%epoc,error)
  endif
  !
  ! Special case for SPLINE : correct calibration
  !    at 60 micron
  !            Corrected Brightness = 0.75*(Tape Brightness) + 4.2E-8 W/M**2/SR
  !    at 100 micron
  !            Corrected Brightness = 0.68*(Tape Brightness) - 0.2E-8 W/M**2/SR
  !    (F. Boulanger, 25-July-1985)
  !
  if (istyle.eq.2) then
    if (my_convert(2,3).eq.60d-6) then
      bscal = 0.75*bscal
      bzero = 0.75*bzero + 4.2e-8
      fits%gil%rmin = 0.75*fits%gil%rmin + 4.2e-8
      fits%gil%rmax = 0.75*fits%gil%rmax + 4.2e-8
    elseif (my_convert(2,3).eq.100d-6) then
      bscal = 0.68*bscal
      bzero = 0.68*bzero - 0.2e-8
      fits%gil%rmin = 0.68*fits%gil%rmin - 0.2e-8
      fits%gil%rmax = 0.68*fits%gil%rmax - 0.2e-8
    else
      call gio_message(seve%w,'READ','Tape is not of style SPLINE')
    endif
  endif
  !
  ! Special patch for aips++ : SWAP AXIS 3 (STOKES) AND 4 (FREQ)
  ! if degenerate axis.
  if (rccode(3).eq.'STOKES'.and.rccode(4).eq.'FREQ') then
    if (fits%gil%dim(3).eq.1 .and. fits%gil%dim(4).gt.1) then
      fits%gil%dim(3)=fits%gil%dim(4)
      fits%gil%dim(4)=1
      rswap = convert(:,3)
      convert(:,3) = convert(:,4)
      convert(:,4) = rswap
    endif
  endif
  !
  ! Return : The image is effectively created and read elsewhere
  if (blanki) then
    if (nbit.ge.0) then        ! Integer input pixels
      fits%gil%bval = nint(rblank)*bscal + bzero
      fits%gil%eval = 0.5*bscal
    else
      fits%gil%bval = rblank*bscal + bzero
      fits%gil%eval = 0
    endif
    fits%gil%blan_words = 2
  endif
  !
  ! Protect against zero increments
  ! (which do occur, and tend to crash axis routines)
  do i=1,gdf_maxdims
    if (convert(3,i).eq.0.0) convert(3,i) = 1.0
  enddo
  !
  ! Setup the Frequency information if available
  if (fits%gil%spec_words.ne.0) then
    if (have_restfreq) then
      fits%gil%freq = f*1d-6  ! restfreq
    endif
    if (have_velo) then
      fits%gil%voff = v*1d-3
    endif
    if (have_velref) then
      if (fits%gil%vtyp.ne.vel_unk) then
        call gio_message(seve%w,rname,'Reference system set by SPECSYS, ignoring VELREF')
      else
        call gio_message(seve%w,rname,'Reference system set by VELREF')
        iv = nint(vv)
        if (vv-iv.eq.0.0) then
          if (iv.gt.256) then
            iv = iv-256
          endif
          if (iv.gt.0 .and. iv.lt.4) then
            fits%gil%vtyp = iv
          endif
        endif
        if (fits%gil%vtyp.eq.vel_unk) then
          call gio_message(seve%w,rname,'Invalid VELREF, Reference system set to Unknown')
          if (.not.(have_velo)) then
            call gio_message(seve%w,rname,'Obsolete use of VELREF')
            fits%gil%voff = vv*1d-5  ! Obsolete use of VELREF = source Vsys
          endif
        endif
      endif
    endif
    !
    ! Compute either velres or freqres, depending on which axis is defined
    if (fits%gil%faxi.ne.0) then
      !
      ! FREQUENCY axis
      if (fits%char%code(fits%gil%faxi).eq.'FREQUENCY') then
        fits%gil%fres = convert(3,fits%gil%faxi)
        if (.not.have_velo) then
           ! Deriving Voff from ALTRVAL/ALTRPIX is arbitrary. For instance, ALTRPIX is systematically
           ! the 1st channel in CASA FITS files, and there's no way to guess the source systemic velocity.
           ! Better to keep Voff = 0.
           !
           ! if (have_altrval) fits%gil%voff = altrval*1d-3
           ! if (have_altrpix) then
           !    ! Define Voff as the velocity of the ref. channel (which may be different from altrpix)
           !    fits%gil%voff = fits%gil%voff - (altrpix-convert(1,fits%gil%faxi)) * convert(3,fits%gil%faxi) *  &
           !       clight_kms / fits%gil%freq   ! Velocity at pixel
           ! endif
        endif
        !
        ! VELOCITY axis
      elseif (fits%char%code(fits%gil%faxi).eq.'VELOCITY') then
        fits%gil%vres = convert(3,fits%gil%faxi)
        if (.not.(have_restfreq)) then
           ! Note: if RESTFREQ is already known (usual case), we should check the
           ! consistency between RESTFREQ, convert[*,faxi], and ALTRVAL/ALTRPIX
           call gio_message(seve%w,rname,'Deriving REST FREQUENCY from ALTRVAL/ALTRPIX')
           if (have_altrval) fits%gil%freq = altrval*1d-6
           if (have_altrpix) then
              ! Frequency at ref. channel (in case it is different from altrpix)
              fits%gil%freq = fits%gil%freq - (altrpix-convert(1,fits%gil%faxi)) * convert(3,fits%gil%faxi)  *  &
                   fits%gil%freq / clight_kms
              ! Rest frequency corresponds to velocity = 0 (LSR frame)
              fits%gil%freq = fits%gil%freq - convert(2,fits%gil%faxi) * convert(3,fits%gil%faxi)  *  &
                   fits%gil%freq / clight_kms
           endif
        endif
      else
        call gio_message(seve%w,rname,'Unknown axis code '//fits%char%code(fits%gil%faxi))
      endif
    endif
    if (fits%gil%vres.eq.0) fits%gil%vres = - clight_kms * fits%gil%fres / fits%gil%freq
    if (fits%gil%fres.eq.0) fits%gil%fres = - fits%gil%freq * fits%gil%vres / clight_kms
  endif
  !
  ! Sent back
  fits%gil%convert = convert
  call r8tor8(xxproj,fits%gil%a0,3)
  !
  return
  !
99 error = .true.
  !
100 format(1x,a,a,a,i6)
101 format(1x,a,i6,a)
1003 format(f8.0)
end function togdf
!
subroutine no_blanc(chin,chout,n)
  use gio_dependencies_interfaces
  !---------------------------------------------------------------------
  ! @ public
  !---------------------------------------------------------------------
  character(len=*) :: chin          !
  character(len=*) :: chout         !
  integer :: n                      !
  !
  chout = chin
  n = len_trim(chout)
  call sic_black(chout,n)
end subroutine no_blanc
!
subroutine  unit_prefix_scale(prefix,scale,unit,error)
  character(len=*), intent(in) :: prefix
  character(len=*), intent(in) :: unit
  real(8), intent(out) :: scale
  logical, intent(out) :: error
  !
  integer :: kunit
  character(len=12) :: u_unit
  character(len=24) :: u_prefix
  !
  error = .false.
  scale = 1.d0
  !
  if (prefix.eq.' ') return
  !
  u_prefix = prefix
  call sic_upper(u_prefix)
  u_unit = unit
  call sic_upper(u_unit)
  kunit = index(u_prefix,u_unit)
  !
  if (kunit.eq.0) then
    ! No prefix multiplier
    return
  else if (u_prefix(kunit:).ne.u_unit) then
    ! Not a prefix multiplier
    return
  else if (kunit.gt.3)  then
    ! Prefix multiplier can only have 1 or 2 letters
    error = .true.
    return
  else if (kunit.eq.3) then
    ! The only Prefix multiplier with 2 letters is "da" for "deca"
    if (prefix(1:2).eq.'da') then
      scale = 10.d0
    else
      error = .true.
    endif
    return
  endif
  !
  select case (prefix(1:1))
  case ('d')
    scale = 1.d-1
  case ('c')
    scale = 1.d-2
  case ('m')
    scale = 1.d-3
  case ('u')
    scale = 1.d-6
  case ('n')
    scale = 1.d-9
  case ('p')
    scale = 1.d-12
  case ('f')
    scale = 1.d-15
  case ('a')
    scale = 1.d-18
  case ('z')
    scale = 1.d-21
  case ('y')
    scale = 1.d-24
  case ('h')
    scale = 1.d2
  case ('k')
    scale = 1.d3
  case ('M')
    scale = 1.d6
  case ('G')
    scale = 1.d9
  case ('T')
    scale = 1.d12
  case ('P')
    scale = 1.d15
  case ('E')
    scale = 1.d18
  case ('Z')
    scale = 1.d21
  case ('Y')
    scale = 1.d24
  case default
    scale = 1.d0
  end select
end subroutine unit_prefix_scale
