module cubedag_node
  use gkernel_interfaces
  use cubetools_list
  use cubedag_messaging
  use cubedag_parameters
  use cubedag_flag
  use cubedag_types
  use cubedag_type
  use cubedag_link

  integer(kind=code_k) :: code_type_node=0

  public :: code_type_node
  public :: cubedag_node_allocate,cubedag_node_deallocate
  public :: cubedag_node_destroy,cubedag_node_remove
  public :: cubedag_node_links,cubedag_node_link
  public :: cubedag_node_history
  public :: cubedag_node_set_spectro,cubedag_node_set_teles,cubedag_node_set_position,  &
            cubedag_node_set_unit,cubedag_node_set_origin,cubedag_node_set_family,  &
            cubedag_node_set_flags,cubedag_node_set_reso,  &
            cubedag_node_set_sicvar,cubedag_node_unset_sicvar
  private

contains
  !
  subroutine cubedag_node_allocate(object,error)
    !-------------------------------------------------------------------
    ! Allocate a 'cubedag_node_object_t' in memory
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: object
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='NODE>ALLOCATE'
    integer(kind=4) :: ier
    !
    allocate(cubedag_node_object_t::object,stat=ier)
    if (failed_allocate(rname,'object',ier,error)) return
    !
    ! Set up the list-type method
    object%ltype    => cubedag_node_ltype
    object%memsize  => cubedag_node_memsize
    object%disksize => cubedag_node_disksize
    object%datasize => cubedag_node_datasize
  end subroutine cubedag_node_allocate
  !
  subroutine cubedag_node_final(object,error)
    !-------------------------------------------------------------------
    ! Free the contents of a 'cubedag_node_object_t'
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: object
    logical,                      intent(inout) :: error
    !
    call cubetools_list_final(object%node%flag,error)
    if (error)  continue
    call cubedag_link_final(object%node%parents,error)
    if (error)  continue
    call cubedag_link_final(object%node%children,error)
    if (error)  continue
    call cubedag_link_final(object%node%twins,error)
    if (error)  continue
  end subroutine cubedag_node_final
  !
  subroutine cubedag_node_deallocate(object,error)
    !-------------------------------------------------------------------
    ! Deallocate a 'cubedag_node_object_t' in memory
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: object
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='NODE>DEALLOCATE'
    !
    if (.not.associated(object)) then
      call cubedag_message(seve%e,rname,'Internal error: object is not allocated')
      error = .true.
      return
    endif
    !
    select type (object)
    type is (cubedag_node_object_t)
      continue
    class default
      call cubedag_message(seve%e,rname,'Internal error: object has wrong type')
      error = .true.
      return
    end select
    !
    deallocate(object)  ! NB: deallocation is polymorphic
  end subroutine cubedag_node_deallocate

  function cubedag_node_ltype(obj)
    character(len=2) :: cubedag_node_ltype
    class(cubedag_node_object_t), intent(in) :: obj
    cubedag_node_ltype = '<>'
  end function cubedag_node_ltype

  function cubedag_node_memsize(obj)
    integer(kind=size_length) :: cubedag_node_memsize
    class(cubedag_node_object_t), intent(in) :: obj
    cubedag_node_memsize = 0
  end function cubedag_node_memsize

  function cubedag_node_disksize(obj)
    integer(kind=size_length) :: cubedag_node_disksize
    class(cubedag_node_object_t), intent(in) :: obj
    cubedag_node_disksize = 0
  end function cubedag_node_disksize

  function cubedag_node_datasize(obj)
    integer(kind=size_length) :: cubedag_node_datasize
    class(cubedag_node_object_t), intent(in) :: obj
    cubedag_node_datasize = 0
  end function cubedag_node_datasize

  subroutine cubedag_node_destroy(object,error)
    !-------------------------------------------------------------------
    ! Brute force destruction of a node object. Do not worry about
    ! family links between nodes.
    ! The object is a pointer which is expected to be deallocated and
    ! nullified in return.
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: object
    logical,                      intent(inout) :: error
    !
    ! Clean the object contents (common part). The polymorphic part
    ! must be cleaned by the specific deallocation subroutine invoked
    ! below
    call cubedag_node_final(object,error)
    if (error)  continue
    !
    ! Deallocate the object in memory
    call cubedag_type_deallocate(object,error)
    if (error)  return
  end subroutine cubedag_node_destroy

  subroutine cubedag_node_remove(object,error)
    use cubedag_tuple
    !-------------------------------------------------------------------
    ! Proper removal of a node object, taking care of the family links
    ! between the other nodes
    ! The object is a pointer which is expected to be deallocated and
    ! nullified in return.
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: object
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='REMOVE'
    integer(kind=entr_k) :: irel
    class(cubedag_node_object_t), pointer :: relative
    !
    if (object%node%children%n.gt.0) then
      call cubedag_message(seve%e,rname,'Node has one or more children')
      error = .true.
      return
    endif
    !
    ! Clean twins
    do irel=1,object%node%twins%n
      relative => object%node%twins%list(irel)%p
      call cubedag_link_unlink(relative%node%twins,object,error)
      if (error)  return
    enddo
    ! Clean parents
    do irel=1,object%node%parents%n
      relative => object%node%parents%list(irel)%p
      call cubedag_link_unlink(relative%node%children,object,error)
      if (error)  return
    enddo
    !
    ! Clean files on disk, only for non-raw cubes!
    if (object%node%origin.ne.code_origin_imported) then
      call cubedag_tuple_rmfiles(object%node%tuple,error)
      if (error)  return
    endif
    !
    ! Clean the object contents (common part). The polymorphic part
    ! must be cleaned by the specific deallocation subroutine invoked
    ! below
    call cubedag_node_final(object,error)
    if (error)  continue
    !
    ! Deallocate the object in memory
    call cubedag_type_deallocate(object,error)
    if (error)  return
  end subroutine cubedag_node_remove

  subroutine cubedag_node_links(parents,children,hid,error)
    !-------------------------------------------------------------------
    ! Insert all the links between parents, children, and twins
    ! ---
    ! Beware this subroutine can be called in an error recovery context
    !-------------------------------------------------------------------
    type(cubedag_link_t), intent(in)    :: parents
    type(cubedag_link_t), intent(in)    :: children
    integer(kind=entr_k), intent(in)    :: hid
    logical,              intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='NODE>LINKS'
    integer(kind=entr_k) :: ient,ic,it
    type(cubedag_link_t) :: twins
    !
    if (error)  return  ! Error recovery: unclear
    !
    ! Sanity check for cubes opened in UPDATE mode
    ! ZZZ Obviously this should be moved in cube/adm
    ! do ip=1,np
    !   ient = cubedag_dag_entrynum(pids(ip),error)
    !   if (error)  return
    !   if (ix%cube(ient)%desc%action.eq.code_update) then
    !     if (ix%children(ient)%n.gt.0) then
    !       call cubedag_message(seve%w,rname,  &
    !         'Updated tuple has children which should be updated too')
    !     endif
    !   endif
    ! enddo
    !
    if (children%n.eq.0) then
      ! This can happen if a cube was opened in update mode, or
      ! for commands with input-only cubes
      continue
    elseif (parents%n.eq.0) then
      ! Is this allowed, e.g. tuples created from scratch?
      call cubedag_message(seve%w,rname,'No parents for new tuple(s)')
    else
      call cubedag_link_reallocate(twins,children%n-1,error)
      if (error)  return
      twins%n = children%n-1
      do ient=1,children%n
        do ic=1,ient-1
          it = ic
          twins%list(it)%p => children%list(ic)%p
        enddo
        do ic=ient+1,children%n
          it = ic-1
          twins%list(it)%p => children%list(ic)%p
        enddo
        call cubedag_node_link(children%list(ient)%p,parents,twins,error)
        if (error)  exit
      enddo
      call cubedag_link_final(twins,error)
      if (error)  return
    endif
    !
    ! History
    call cubedag_node_history(children,hid,error)
    if (error)  return
    !
  end subroutine cubedag_node_links

  subroutine cubedag_node_link(child,parents,twins,error)
    !-------------------------------------------------------------------
    ! Add parent-children-twins links between the child and its parents
    ! and twins
    ! Also add the reference to the history index
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: child
    type(cubedag_link_t),         intent(in)    :: parents
    type(cubedag_link_t),         intent(in)    :: twins
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='NODE>LINK'
    !
    ! Insert parents
    if (parents%n.le.0) then
      call cubedag_message(seve%e,rname,'There should be at least 1 parent')
      error = .true.
      return
    endif
    !
    ! Fill the parents
    call cubedag_link_copy(parents,child%node%parents,error)
    if (error)  return
    !
    ! Add the children backpointers
    call cubedag_node_add_children(child,parents,error)
    if (error)  return
    !
    ! Fill the twins
    call cubedag_link_copy(twins,child%node%twins,error)
    if (error)  return
    !
    ! --- Insert on disk ---
    ! ZZZ TO BE DONE
    ! ZZZ Insert in real time, or later?
  end subroutine cubedag_node_link
  !
  subroutine cubedag_node_add_children(child,parents,error)
    !-------------------------------------------------------------------
    ! For each parent, add a reference to this new child
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: child
    type(cubedag_link_t),         intent(in)    :: parents
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='NODE>ADD>CHILDREN'
    integer(kind=entr_k) :: ip
    !
    do ip=1,parents%n
      call cubedag_node_add_onechild(parents%list(ip)%p,child,error)
      if (error)  exit
    enddo
  end subroutine cubedag_node_add_children
  !
  subroutine cubedag_node_add_onechild(parent,child,error)
    !-------------------------------------------------------------------
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: parent
    class(cubedag_node_object_t), pointer       :: child
    logical,                      intent(inout) :: error
    ! Local
    integer(kind=entr_k) :: nc
    !
    ! Insert one more child to the parent
    ! ZZZ Should we protect against duplicate insertions? Can this happen
    !     under normal conditions?
    nc = parent%node%children%n + 1
    call cubedag_link_reallocate(parent%node%children,nc,error)
    if (error)  return
    parent%node%children%list(nc)%p => child
    parent%node%children%n = nc
  end subroutine cubedag_node_add_onechild

  subroutine cubedag_node_history(link,hid,error)
    type(cubedag_link_t), intent(in)    :: link
    integer(kind=entr_k), intent(in)    :: hid
    logical,              intent(inout) :: error
    ! Local
    integer(kind=entr_k) :: iobj
    !
    do iobj=1,link%n
      link%list(iobj)%p%node%history = hid
    enddo
    !
  end subroutine cubedag_node_history
  !
  !---------------------------------------------------------------------
  !
  subroutine cubedag_node_set_origin(object,origin,error)
    !-------------------------------------------------------------------
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    integer(kind=code_k),         intent(in)    :: origin
    logical,                      intent(inout) :: error
    !
    object%node%origin = origin
  end subroutine cubedag_node_set_origin
  !
  subroutine cubedag_node_set_family(object,family,error)
    !-------------------------------------------------------------------
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    character(len=*),             intent(in)    :: family
    logical,                      intent(inout) :: error
    !
    object%node%family = family
  end subroutine cubedag_node_set_family
  !
  subroutine cubedag_node_set_flags(object,flags,error)
    !-------------------------------------------------------------------
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    type(flag_t),                 intent(in)    :: flags(:)
    logical,                      intent(inout) :: error
    !
    call cubedag_flaglist_create(flags,object%node%flag,error)
    if (error)  return
  end subroutine cubedag_node_set_flags
  !
  subroutine cubedag_node_set_unit(object,unit,error)
    !-------------------------------------------------------------------
    ! Set the unit for the given entry number
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    character(len=*),             intent(in)    :: unit
    logical,                      intent(inout) :: error
    !
    object%node%unit = unit
  end subroutine cubedag_node_set_unit
  !
  subroutine cubedag_node_set_position(object,source,ptype,a0,d0,pang,lres,mres,error)
    !-------------------------------------------------------------------
    ! Set the position information for the given entry number
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    character(len=*),             intent(in)    :: source
    integer(kind=code_k),         intent(in)    :: ptype
    real(kind=coor_k),            intent(in)    :: a0
    real(kind=coor_k),            intent(in)    :: d0
    real(kind=coor_k),            intent(in)    :: pang
    real(kind=coor_k),            intent(in)    :: lres
    real(kind=coor_k),            intent(in)    :: mres
    logical,                      intent(inout) :: error
    !
    object%node%source = source
    call sic_upper(object%node%source)
    object%node%ptype = ptype
    object%node%a0 = a0
    object%node%d0 = d0
    object%node%pang = pang
    object%node%lres = lres
    object%node%mres = mres
  end subroutine cubedag_node_set_position
  !
  subroutine cubedag_node_set_spectro(object,line,restf,fres,vsys,error)
    !-------------------------------------------------------------------
    ! Set the spectroscopic information for the given entry number
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    character(len=*),             intent(in)    :: line
    real(kind=coor_k),            intent(in)    :: restf
    real(kind=coor_k),            intent(in)    :: fres
    real(kind=coor_k),            intent(in)    :: vsys
    logical,                      intent(inout) :: error
    !
    object%node%line = line
    call sic_upper(object%node%line)
    object%node%restf = restf
    object%node%fres = fres
    object%node%vsys = vsys
  end subroutine cubedag_node_set_spectro
  !
  subroutine cubedag_node_set_reso(object,rmaj,rmin,rang,error)
    !-------------------------------------------------------------------
    ! Set the resolution information for the given entry number
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    real(kind=beam_k),            intent(in)    :: rmaj
    real(kind=beam_k),            intent(in)    :: rmin
    real(kind=beam_k),            intent(in)    :: rang
    logical,                      intent(inout) :: error
    !
    object%node%rmaj = rmaj
    object%node%rmin = rmin
    object%node%rang = rang
  end subroutine cubedag_node_set_reso
  !
  subroutine cubedag_node_set_teles(object,teles,error)
    !-------------------------------------------------------------------
    ! Set the telescope name for the given entry number
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    character(len=*),             intent(in)    :: teles(:)
    logical,                      intent(inout) :: error
    ! Local
    integer(kind=4) :: itel
    !
    object%node%nteles = size(teles)
    do itel=1,size(teles)
      object%node%teles(itel) = teles(itel)
      call sic_upper(object%node%teles(itel))
    enddo
    do itel=size(teles)+1,dag_mteles
      object%node%teles(itel) = ''
    enddo
  end subroutine cubedag_node_set_teles
  !
  subroutine cubedag_node_set_sicvar(object,sicvar,error)
    !-------------------------------------------------------------------
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    character(len=*),             intent(in)    :: sicvar
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='NODE>SET>SICVAR'
    integer(kind=4) :: nsicvar
    !
    nsicvar = object%node%nsicvar+1
    if (nsicvar.gt.dag_msicvar) then
      call cubedag_message(seve%e,rname,'Too many user variables pointing to the same object')
      error = .true.
      return
    endif
    object%node%nsicvar = nsicvar
    object%node%sicvar(nsicvar) = sicvar
  end subroutine cubedag_node_set_sicvar
  !
  subroutine cubedag_node_unset_sicvar(object,sicvar,error)
    !-------------------------------------------------------------------
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), intent(inout) :: object
    character(len=*),             intent(in)    :: sicvar
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='NODE>UNSET>SICVAR'
    integer(kind=4) :: isicvar,jsicvar
    !
    jsicvar = 0
    do isicvar=1,object%node%nsicvar
      if (object%node%sicvar(isicvar).eq.sicvar) then
        jsicvar = isicvar
        exit
      endif
    enddo
    !
    if (jsicvar.eq.0) then
      call cubedag_message(seve%w,rname,'Internal error: no such reference to variable '//sicvar)
      return
    endif
    !
    do isicvar=jsicvar+1,object%node%nsicvar
      object%node%sicvar(isicvar-1) = object%node%sicvar(isicvar)
    enddo
    object%node%nsicvar = object%node%nsicvar-1
  end subroutine cubedag_node_unset_sicvar
  !
end module cubedag_node
