!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_baseline
  use cubetools_structure
  use cube_types
  use cubeadm_cubeid_types
  use cubeadm_cubeprod_types
  use cubemain_messaging
  use cubemain_ancillary_mask_types
  use cubemain_windowing
  use cubemain_range
  !
  public :: baseline
  public :: cubemain_baseline_command
  private
  !
  integer(kind=code_k), parameter :: code_median     = 1
  integer(kind=code_k), parameter :: code_polynomial = 2
  integer(kind=code_k), parameter :: code_wavelet    = 3
  !
  type :: baseline_comm_t
     type(option_t), pointer     :: comm
     type(option_t), pointer     :: median
     type(option_t), pointer     :: wavelet
     type(option_t), pointer     :: polynomial   
     type(range_opt_t)           :: range
     type(cubeid_arg_t), pointer :: cube
     type(ancillary_mask_comm_t) :: mask
     type(cube_prod_t),  pointer :: base
     type(cube_prod_t),  pointer :: line
   contains
     procedure, public  :: register         => cubemain_baseline_comm_register
     procedure, private :: parse            => cubemain_baseline_comm_parse
     procedure, private :: parse_median     => cubemain_baseline_comm_parse_median
     procedure, private :: parse_wavelet    => cubemain_baseline_comm_parse_wavelet
     procedure, private :: parse_polynomial => cubemain_baseline_comm_parse_polynomial
     procedure, private :: main             => cubemain_baseline_comm_main
  end type baseline_comm_t
  type(baseline_comm_t) :: baseline
  !
  type baseline_user_t
     type(cubeid_user_t)         :: cubeids
     type(ancillary_mask_user_t) :: mask
     logical                     :: dorange = .false. ! Was the /range option present
     type(range_array_t)         :: range             ! Range(s) to be ignored when fitting a baseline
     logical                     :: domedian          ! Is /median present?
     real(kind=coor_k)           :: width             ! [MHz    ]  
     real(kind=coor_k)           :: sampling          ! [MHz    ] 
     logical                     :: dowavelet         ! [-------] Is /wavelet present?
     integer(kind=4)             :: degree = -1       ! [-------] degree for the wavelet
     logical                     :: dopolynomial      ! [-------] Is /polynomial present?
     integer(kind=4)             :: npol = 0          ! [-------] Number of polynomials to be used
     character(len=argu_l)       :: trkind            ! [-------] Unit kind for the transitions given
     integer(kind=4),   allocatable :: degrees(:)     ! [-------] degree(s) for the polynomials
     real(kind=coor_k), allocatable :: trans(:)       ! [MHz|kms] transition between different polynomials
   contains
     procedure, private :: toprog => cubemain_baseline_user_toprog
  end type baseline_user_t
  !
  type baseline_prog_t
     type(cube_t), pointer       :: cube             ! Input cube
     type(ancillary_mask_prog_t) :: mask             ! Input Mask
     type(cube_t), pointer       :: base             ! Output baseline
     type(cube_t), pointer       :: line             ! Output baselined cube
     integer(kind=code_k)  :: method                 ! Baselining method
     integer(kind=chan_k)  :: nwidth                 !
     integer(kind=chan_k)  :: nsampling              !
     integer(kind=chan_k)  :: nmedian                !
     integer(kind=4)       :: degree                 ! [---] degree for the wavelet
     type(window_array_t)  :: wind                   ! Window(s) to be ignored when fitting a baseline
     integer(kind=4)       :: npol = 0               ! [---] Number of polynomials to be used
     integer(kind=4),      allocatable :: degrees(:) ! [---] degree(s) for the polynomials 
     integer(kind=chan_k), allocatable :: trans(:)   ! [mhz|kms] transition between different polynomials
     procedure(cubemain_baseline_prog_median_loop), pointer :: loop => null()
   contains
     procedure, private :: header         => cubemain_baseline_prog_header
     procedure, private :: data           => cubemain_baseline_prog_data
     procedure, private :: median_act     => cubemain_baseline_prog_median_act
     procedure, private :: wavelet_act    => cubemain_baseline_prog_wavelet_act
     procedure, private :: polynomial_act => cubemain_baseline_prog_polynomial_act
  end type baseline_prog_t
  !
contains
  !
  subroutine cubemain_baseline_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(baseline_user_t) :: user
    character(len=*), parameter :: rname='BASELINE>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    call baseline%parse(line,user,error)
    if (error) return
    call baseline%main(user,error)
    if (error) continue
  end subroutine cubemain_baseline_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_baseline_comm_register(comm,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_comm_t), intent(inout) :: comm
    logical,                intent(inout) :: error
    !
    type(cubeid_arg_t) :: incube
    type(cube_prod_t) :: oucube
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract = 'Subtract a baseline from a cube'
    character(len=*), parameter :: comm_help =&
      'Three algorithms are available to compute a baseline:&
      & /MEDIAN, /WAVELET and /POLYNOMIAL. Only one of these three&
      & can be given at a time. If no algorithm option is given&
      & BASELINE defaults to /MEDIAN.'
    character(len=*), parameter :: rname='BASELINE>COMM>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Syntax
    call cubetools_register_command(&
         'BASELINE','[cube]',&
         comm_abstract,&
         comm_help,&
         cubemain_baseline_command,&
         comm%comm,&
         error)
    if (error) return
    call incube%register(&
         'CUBE',&
         'Input cube',&
         strg_id,&
         code_arg_optional,&
         [flag_cube],&
         code_access_speset,&
         code_read,&
         comm%cube,&
         error)
    if (error) return
    !
    call comm%range%register(&
         'RANGE',&
         'Define the signal velocity range(s)',&
         range_is_multiple,error)
    if (error) return
    !
    call comm%mask%register(&
         'Use a mask to define lines regions to be ignored during baselining',&
         code_access_speset,error)
    if (error) return
    !
    call cubetools_register_option(&
         'MEDIAN','[width [sampling]]',&
         'Use a median running filter to define the baseline',&
         'The median and associated median absolute deviation are computed&
         & in windows of the given width, sampled every sampling space.&
         & Intermediate values are then linearly interpolated so that the&
         & final cubes have the same number of channels as the input cube.&
         & Flat values are used (no extrapolation) For the first and last&
         & half windows (boundary conditions). When the input channels are&
         & blanked, the resulting channels are also blanked. Blank channels&
         & do not contribute to the surrounding windows.',&
         comm%median,&
         error)
    if (error) return
    call stdarg%register(&
         'width',&
         'Running filter width',&
         'In MHz. Default to 20 MHz.',&
         code_arg_optional,&
         error)
    if (error) return
    call stdarg%register(&
         'sampling',&
         'Running filter sampling',&
         'Default to width/2',&
         code_arg_optional,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'WAVELET','degree',&
         'Use a wavelet filter to define the baseline',&
         strg_id,&
         comm%wavelet,error)
    if (error) return
    call stdarg%register(&
         'degree',&
         'Wavelet degree',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'POLYNOMIAL','deg1 [tr12 deg2 [... [trij degj [kind]]]]',&
         'Fit Chebyshev polynomials to define the baseline',&
         strg_id,&
         comm%polynomial,error)
    if (error) return
    call stdarg%register(&
         'deg1',&
         '[First] Polynomial degree',&
         'Degree for the first polynomial',&
         code_arg_mandatory, error)
    if (error) return
    call stdarg%register(&
         'tr12',&
         'Transition between first and second polynomial',&
         strg_id,&
         code_arg_unlimited,&
         error)
    if (error) return
    call stdarg%register(&
         'deg2',&
         'Degree for the second polynomial',&
         strg_id,&
         code_arg_unlimited,&
         error)
    if (error) return
    call stdarg%register(&
         'trij',&
         'Transition between ith and jth polynomial',&
         strg_id,&
         code_arg_unlimited,&
         error)
    if (error) return
    call stdarg%register(&
         'degj',&
         'Degree for the jth polynomial',&
         strg_id,&
         code_arg_unlimited,&
         error)
    if (error) return
    call stdarg%register(&
         'kind',&
         'Kind of the transition(s) between polynomials',&
         'Available kinds: CHANNEL, VELOCITY (default), or FREQUENCY',&
         code_arg_optional,&
         error)
    if (error) return
    !
    ! Products
    call oucube%register(&
         'BASE',&
         'Baseline cube',&
         strg_id,&
         [flag_baseline,flag_base],&
         comm%base,&
         error)
    if (error)  return
    call oucube%register(&
         'LINE',&
         'Baseline subtracted cube',&
         strg_id,&
         [flag_baseline,flag_line,flag_cube],&
         comm%line,&
         error)
    if (error)  return
  end subroutine cubemain_baseline_comm_register
  !
  subroutine cubemain_baseline_comm_parse(comm,line,user,error)
    !----------------------------------------------------------------------
    ! BASELINE cubname
    ! /RANGE vfirst vlast
    ! /MEDIAN [width [sampling]]
    ! /WAVELET degree
    ! /POLYNOMIAL deg1 [tr12 deg2 [... degi trij degj] [unit]]
    !----------------------------------------------------------------------
    class(baseline_comm_t), intent(in)    :: comm
    character(len=*),       intent(in)    :: line
    type(baseline_user_t),  intent(out)   :: user
    logical,                intent(inout) :: error
    !
    logical :: combi
    character(len=*),parameter :: rname='BASELINE>COMM>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,comm%comm,user%cubeids,error)
    if (error) return
    !
    call comm%range%parse(line,user%dorange,user%range,error)
    if (error) return
    call comm%mask%parse(line,user%mask,error)
    if (error) return
    if (user%dorange.and.user%mask%present) then
       call cubemain_message(seve%e,rname,'Choose either the /RANGE or the /MASK option')
       error = .true.
       return
    endif
    !
    call comm%median%present(line,user%domedian,error)
    if (error) return
    call comm%wavelet%present(line,user%dowavelet,error)
    if (error) return
    call comm%polynomial%present(line,user%dopolynomial,error)
    if (error) return
    combi = ((user%domedian.and.user%dowavelet).or.(user%domedian.and.user%dopolynomial).or.(user%dowavelet.and.user%dopolynomial))
    !
    if ((.not.user%domedian).and.(.not.user%dowavelet).and.(.not.user%dopolynomial)) then
       user%domedian = .true.
    elseif (combi) then
       call cubemain_message(seve%e,rname,'Choose only one baselining method among /MEDIAN, /WAVELET, or /POLYNOMIAL')
       error = .true.
       return
    else
       if (user%domedian) then
          call comm%parse_median(line,user,error)
          if (error) return
       else if (user%dowavelet) then
          call comm%parse_wavelet(line,user,error)
          if (error) return
       else if (user%dopolynomial) then
          call comm%parse_polynomial(line,user,error)
          if (error) return
       else
          user%degree = -1
       endif
    endif
  end subroutine cubemain_baseline_comm_parse
  !
  subroutine cubemain_baseline_comm_parse_median(comm,line,user,error)
    !----------------------------------------------------------------------
    ! /MEDIAN [width [sampling]] 
    !----------------------------------------------------------------------
    class(baseline_comm_t), intent(in)    :: comm
    character(len=*),       intent(in)    :: line
    type(baseline_user_t),  intent(inout) :: user
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='BASELINE>COMM>PARSE>MEDIAN'
    !
    user%width = 20.d0 ! MHz
    call cubetools_getarg(line,comm%median,1,user%width,.not.mandatory,error)
    if (error) return
    user%sampling = 0.5d0*user%width
    call cubetools_getarg(line,comm%median,2,user%sampling,.not.mandatory,error)
    if (error) return
  end subroutine cubemain_baseline_comm_parse_median
  !
  subroutine cubemain_baseline_comm_parse_wavelet(comm,line,user,error)
    !----------------------------------------------------------------------
    ! /WAVELET degree 
    !----------------------------------------------------------------------
    class(baseline_comm_t), intent(in)    :: comm
    character(len=*),       intent(in)    :: line
    type(baseline_user_t),  intent(inout) :: user
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='BASELINE>COMM>PARSE>WAVELET'
    !
    call cubetools_getarg(line,comm%wavelet,1,user%degree,mandatory,error)
    if (error) return
    if (user%degree.lt.0) then
       ! *** JP Unclear to me that wavelet will work with degree.eq.0...
       call cubemain_message(seve%e,rname,'Degree must be positive')
       error = .true.
       return
    endif
  end subroutine cubemain_baseline_comm_parse_wavelet
  !
  subroutine cubemain_baseline_comm_parse_polynomial(comm,line,user,error)
    use gkernel_interfaces
    use cubetools_unit
    use cubetools_disambiguate
    !----------------------------------------------------------------------
    ! /POLYNOMIAL degree [tr12 deg2 [trij degj] [kind]]
    !----------------------------------------------------------------------
    class(baseline_comm_t), intent(in)    :: comm
    character(len=*),       intent(in)    :: line
    type(baseline_user_t),  intent(inout) :: user
    logical,                intent(inout) :: error
    !
    integer(kind=4), parameter :: nkinds=3
    integer(kind=4) :: narg,ipol,ier,pos,remain
    character(len=argu_l)  :: trkind,kinds(nkinds)
    type(unit_user_t) :: unit
    real(kind=coor_k) :: transfac
    character(len=*), parameter :: rname='BASELINE>COMM>PARSE>POLYNOMIAL'
    data kinds/'VELOCITY','FREQUENCY','CHANNEL'/
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    narg = comm%polynomial%getnarg()
    if (narg.le.1) then
       user%npol = 1
       call cubetools_getarg(line,comm%polynomial,1,user%degree,mandatory,error)
       if (error) return
       if (user%degree.lt.0) then
          call cubemain_message(seve%e,rname,'Degree must be positive')
          error = .true.
          return
       endif
    else
       if (narg.lt.3) then
          call cubemain_message(seve%e,rname,'Need at least 3 arguments')
          error = .true.
          return
       endif
       remain = modulo(narg,2)
       if (remain.eq.0) then
          call cubetools_getarg(line,comm%polynomial,narg,trkind,mandatory,error)
          if (error) return
          call cubetools_disambiguate_strict(trkind,kinds,pos,user%trkind,error)
          if (error) return
          user%npol = narg/2
       else
          user%trkind = kinds(1)
          user%npol = narg/2+1
       endif
       allocate(user%degrees(user%npol),user%trans(user%npol-1),stat=ier)
       if (failed_allocate(rname,'Degree arrays',ier,error)) then
          error = .true.
          return
       endif
       select case(trim(user%trkind))
       case('VELOCITY')
          call unit%get(strg_star,unit_velo%id,error)
          if (error) return
          transfac = unit%prog_per_user
       case('FREQUENCY')
          call unit%get(strg_star,unit_freq%id,error)
          if (error) return
          transfac = unit%prog_per_user
       case('CHANNEL')
          transfac = 1d0
       case default
          call cubemain_message(seve%e,rname,'Unknown transition kind '//trim(user%trkind))
          error = .true.
          return
       end select
       do ipol=1, user%npol
          call cubetools_getarg(line,comm%polynomial,(ipol-1)*2+1,user%degrees(ipol),mandatory,error)
          if (error) return
          if (user%degrees(ipol).lt.0) then
             call cubemain_message(seve%e,rname,'Degree must be positive')
             error = .true.
             return
          endif
          if (ipol.lt.user%npol) then
             call cubetools_getarg(line,comm%polynomial,2*ipol,user%trans(ipol),mandatory,error)
             if (error) return
             user%trans(ipol) = transfac*user%trans(ipol)
          endif
       enddo
    endif
  end subroutine cubemain_baseline_comm_parse_polynomial
  !
  subroutine cubemain_baseline_comm_main(comm,user,error)
    use cubedag_parameters
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_comm_t), intent(in)    :: comm
    type(baseline_user_t),  intent(in)    :: user
    logical,                intent(inout) :: error
    !
    type(baseline_prog_t) :: prog
    character(len=*), parameter :: rname='BASELINE>COMM>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(comm,prog,error)
    if (error) return
    call prog%header(comm,error)
    if (error) return
    !
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_baseline_comm_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_baseline_user_toprog(user,comm,prog,error)
    use gkernel_interfaces
    use cubetools_unit
    use cubetools_axis_types
    use cubetools_shape_types
    use cubetools_header_methods
    use cubeadm_get
    use cubetemplate_topology
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_user_t), intent(in)    :: user
    type(baseline_comm_t),  intent(in)    :: comm
    type(baseline_prog_t),  intent(out)   :: prog
    logical,                intent(inout) :: error
    !
    type(shape_t) :: n
    type(axis_t) :: axis
    type(unit_user_t) :: unit
    integer(kind=4) :: ier,itr
    real(kind=coor_k) :: width,sampling
    character(len=unit_l) :: ouunit
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='BASELINE>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(comm%cube,user%cubeids,prog%cube,error)
    if (error) return
    !
    call comm%range%user2prog(prog%cube,user%range,prog%wind,error)
    if (error) return
    call user%mask%toprog(comm%mask,prog%mask,error)
    if (error) return
    if (prog%mask%do) then
       call prog%mask%check_consistency(prog%cube,error)
       if (error) return
       call prog%mask%must_be_3d(error)
       if (error) return
    endif
    !
    if (user%domedian) then
       prog%loop => cubemain_baseline_prog_median_loop
       prog%method = code_median
       call cubetools_header_get_axis_head_f(prog%cube%head,axis,error)
       if (error) return
       axis%inc = abs(axis%inc)
       width = abs(user%width)
       sampling = abs(user%sampling)
       ! Sanity checks
       if (axis%inc.le.0) then
          call cubemain_message(seve%e,rname,'Cube frequency resolution must be > 0')
          error = .true.
          return
       endif
       ! Goto channel units
       prog%nwidth = nint(width/axis%inc)
       prog%nsampling = nint(sampling/axis%inc)
       ! Ensure that prog%nwidth will be odd because the median computation is simpler/faster
       if (mod(prog%nwidth,2).eq.0) then
          if (prog%nwidth.eq.axis%n) then
             prog%nwidth = prog%nwidth-1
          else
             prog%nwidth = prog%nwidth+1
          endif
       endif
       ! Ensure that we fall inside the spectral axis
       prog%nwidth = max(min(prog%nwidth,axis%n),1)
       prog%nsampling = min(max(prog%nsampling,1),axis%n)
       ! Compute the associated number of median values
       prog%nmedian = floor(dble(axis%n)/dble(prog%nsampling))
       if (prog%nsampling*prog%nmedian.lt.axis%n) then
          prog%nmedian = prog%nmedian+1
       endif
       ! User feedback
       write(mess,'(a,i0,a,i0,a)') 'Computing the median of ',prog%nwidth,' contiguous channels, every ',prog%nsampling,' channels'
       call cubemain_message(seve%i,rname,mess)
    else if (user%dowavelet) then
       prog%loop => cubemain_baseline_prog_wavelet_loop
       prog%method = code_wavelet
       prog%degree = user%degree
    else if (user%dopolynomial) then
       prog%loop => cubemain_baseline_prog_polynomial_loop
       prog%method = code_polynomial
       prog%npol = user%npol
       allocate(prog%degrees(prog%npol),prog%trans(prog%npol+1),stat=ier)
       if (failed_allocate(rname,'Degree arrays',ier,error)) then
          error = .true.
          return
       endif
       prog%trans(1) = 0
       call cubetools_header_get_array_shape(prog%cube%head,n,error)
       if (error) return
       prog%trans(prog%npol+1) = n%c
       if (prog%npol.eq.1) then
          prog%degrees(1) = user%degree          
       else
          prog%degrees(:) = user%degrees(:)
          select case(user%trkind)
          case('CHANNEL')
             do itr=2,prog%npol
                prog%trans(itr) = nint(user%trans(itr-1))
             enddo
             ouunit = 'th channel'
          case('VELOCITY')
             do itr=2,prog%npol
                call cubetemplate_topo_velocity2channel(prog%cube,user%trans(itr-1),prog%trans(itr),error)
                if (error) return
             enddo
             call unit%get(strg_star,unit_velo%id,error)
             if (error) return
             ouunit = unit%name
          case('FREQUENCY')
             do itr=2,prog%npol
                call cubetemplate_topo_frequency2channel(prog%cube,user%trans(itr-1),prog%trans(itr),error)
                if (error) return
             enddo
             call unit%get(strg_star,unit_freq%id,error)
             if (error) return
             ouunit = unit%name
          end select
          do itr=2,prog%npol
             if ((prog%trans(itr).le.1).or.(prog%trans(itr).ge.n%c)) then
                write(mess,'(a,1pg14.7,x,2a)') 'Transition at ',user%trans(itr-1),trim(ouunit),&
                     ' goes beyond bounds'
                call cubemain_message(seve%e,rname,mess)
                error = .true.
                return
             endif
             if (itr.gt.1) then
                if (prog%trans(itr).le.prog%trans(itr-1)) then
                   call cubemain_message(seve%e,rname,'Transitions must be in an ascending order of channels')
                   error = .true.
                   return
                endif
             endif
          enddo
       endif
    else
       call cubemain_message(seve%e,rname,'Unknown baselining method')
       error = .true.
       return
    endif
  end subroutine cubemain_baseline_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_baseline_prog_header(prog,comm,error)
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_prog_t), intent(inout) :: prog
    type(baseline_comm_t),  intent(in)    :: comm
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='BASELINE>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(comm%base,prog%cube,prog%base,error)
    if (error) return
    call cubeadm_clone_header(comm%line,prog%cube,prog%line,error)
    if (error) return
  end subroutine cubemain_baseline_prog_header
  !
  subroutine cubemain_baseline_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(baseline_prog_t), intent(inout) :: prog
    logical,                intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='BASELINE>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error) exit
       !$OMP TASK SHARED(prog,error) FIRSTPRIVATE(iter)
       if (.not.error) call prog%loop(iter,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_baseline_prog_data
  !
  subroutine cubemain_baseline_prog_median_loop(prog,iter,error)
    use cubeadm_taskloop
    use cubeadm_spectrum_types
    use cubemain_interpolate_spectrum_tool
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_prog_t),   intent(inout) :: prog
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    !
    type(spectrum_t) :: input,good,base,line
    type(spectrum_t) :: median
    type(interpolate_spectrum_prog_t) :: interp
    character(len=*), parameter :: rname='BASELINE>PROG>MEDIAN>LOOP'
    !
    call input%associate('input',prog%cube,iter,error)
    if (error) return
    call good%allocate('good',prog%cube,iter,error)
    if (error) return
    call line%allocate('line',prog%line,iter,error)
    if (error) return
    call base%allocate('base',prog%base,iter,error)
    if (error) return
    call median%y%reallocate('median',prog%nmedian,error)
    if (error) return
    ! *** JP TBF
!!$    call base%init_as(prog%cube,error)
!!$    if (error) return
    ! *** JP TBF
    ! Update median header information
    ! Use left edge of first channel to compute new ref channel
    median%ref = 0.5d0-(0.5d0-base%ref)/prog%nsampling
    median%val = base%val
    median%inc = prog%nsampling*base%inc
    call interp%init(median,base,error)
    if (error) return
    !
    do while (iter%iterate_entry(error))
      call prog%median_act(iter%ie,input,good,base,line,median,interp,error)
      if (error) return
    enddo ! ie
  end subroutine cubemain_baseline_prog_median_loop
  !
  subroutine cubemain_baseline_prog_median_act(prog,ie,input,good,base,line,median,interp,error)
    use cubeadm_spectrum_types
    use cubemain_interpolate_spectrum_tool
    use cubemain_statistics
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_prog_t),            intent(inout) :: prog
    integer(kind=entr_k),              intent(in)    :: ie
    type(spectrum_t),                  intent(inout) :: input
    type(spectrum_t),                  intent(inout) :: good
    type(spectrum_t),                  intent(inout) :: base
    type(spectrum_t),                  intent(inout) :: line
    type(spectrum_t),                  intent(inout) :: median
    type(interpolate_spectrum_prog_t), intent(in)    :: interp
    logical,                           intent(inout) :: error
    !
    integer(chan_k) :: im,ic,first,last
    type(spectrum_t)  :: extracted
    character(len=*), parameter :: rname='BASELINE>PROG>MEDIAN>ACT'
    !
    call input%get(ie,error)
    if (error) return
    ! Compute the median
    first = 1
    do im=1,median%n
       ! The following ensures that
       !    1) we never get past the number of channels
       !    2) we always compute the mad on nwidth contiguous samples
       last  = min(first+prog%nwidth-1,input%n)
       first = last-prog%nwidth+1
       call extracted%point_to(input,first,last,1.0,error)
       if (error) return
       call good%unblank(extracted,error)
       if (error) return
       median%y%val(im) = cubemain_median(good%y%val(1:good%n))
!!$       noise%y%val(im)  = cubemain_mad(good%y%val(1:good%n),median)
       first = first+prog%nsampling
    enddo ! im
    ! Interpolate the median as the baseline
    call interp%spectrum(median,base,error)
    if (error) return
    ! Deduce the line by subtraction
    do ic=1,input%n
       ! *** JP: We still need to handle the original NaN
       line%y%val(ic) = input%y%val(ic)-base%y%val(ic)
    enddo ! ic
    ! Write
    call base%put(ie,error)
    if (error) return
    call line%put(ie,error)
    if (error) return
  end subroutine cubemain_baseline_prog_median_act
  !
  subroutine cubemain_baseline_prog_wavelet_loop(prog,iter,error)
    use cubeadm_taskloop
    use cubeadm_spectrum_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_prog_t),   intent(inout) :: prog
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    !
    type(spectrum_t) :: input,line,base
    character(len=*), parameter :: rname='BASELINE>PROG>WAVELET>LOOP'
    !
    call input%associate('input',prog%cube,iter,error)
    if (error) return
    call line%allocate('line',prog%line,iter,error)
    if (error) return
    call base%allocate('base',prog%base,iter,error)
    if (error) return
    !
    do while (iter%iterate_entry(error))
      call prog%wavelet_act(iter%ie,input,base,line,error)
      if (error) return
    enddo
  end subroutine cubemain_baseline_prog_wavelet_loop
  !
  subroutine cubemain_baseline_prog_wavelet_act(prog,ie,input,base,line,error)
    use gkernel_interfaces
    use cubetools_nan
    use cubeadm_spectrum_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_prog_t), intent(inout) :: prog
    integer(kind=entr_k),   intent(in)    :: ie
    type(spectrum_t),       intent(inout) :: input
    type(spectrum_t),       intent(inout) :: base
    type(spectrum_t),       intent(inout) :: line
    logical,                intent(inout) :: error
    !
    real(kind=sign_k), allocatable :: wavelets(:,:)
    character(len=*), parameter :: rname='BASELINE>PROG>WAVELET>ACT'
    !
    call input%get(ie,error)
    if (error) return
    !
    if (input%y%isblanked()) then
       ! Nothing to do
       base%y%val(:) = gr4nan
       line%y%val(:) = gr4nan
    else if (input%y%hasblank()) then ! Expansive call when the input array is completely valid!
       ! Problematic case
       call cubemain_message(seve%e,rname,'Some NaN intensities in input spectrum')
       call cubemain_message(seve%e,rname,'Try replacing them with, eg, zeros before baselining')
       error = .true.
       return
    else
       line%y%val = input%y%val
       call gwavelet_gaps(line%y%val,wavelets,error)
       if (error) return
       call gwavelet_subtract(prog%degree,wavelets,line%y%val,error)
       if (error) return
       base%y%val = input%y%val-line%y%val
    endif
    !
    call base%put(ie,error)
    if (error) return
    call line%put(ie,error)
    if (error) return
  end subroutine cubemain_baseline_prog_wavelet_act
  !
  subroutine cubemain_baseline_prog_polynomial_loop(prog,iter,error)
    use cubeadm_taskloop
    use cubeadm_spectrum_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_prog_t),   intent(inout) :: prog
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    !
    type(spectrum_t) :: input,line,base,good,mask
    character(len=*), parameter :: rname='BASELINE>PROG>POLYNOMIAL>LOOP'
    !
    call input%associate('input',prog%cube,iter,error)
    if (error) return
    call input%associate_x(error)
    if (error) return
    call input%allocate_w(error)
    if (error) return
    call good%allocate('good',prog%cube,iter,error)
    if (error) return
    call good%allocate_xw(error)
    if (error) return
    ! The other ones do not need x or w arrays
    call line%allocate('line',prog%line,iter,error)
    if (error) return
    call base%allocate('base',prog%base,iter,error)
    if (error) return
    if (prog%mask%do) then
       call mask%associate('mask',prog%mask%cube,iter,error)
       if (error) return
    else
       call input%set_base_channels(prog%wind,error)
       if (error) return
    endif
    !
    do while (iter%iterate_entry(error))
      call prog%polynomial_act(iter%ie,input,mask,good,base,line,error)
      if (error) return
    enddo ! ie
  end subroutine cubemain_baseline_prog_polynomial_loop
  !
  subroutine cubemain_baseline_prog_polynomial_act(prog,ie,input,mask,good,base,line,error)
    use cubetools_nan
    use cubeadm_spectrum_types
    use cubemain_svd_tool
    use cubemain_chebyshev_tool
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(baseline_prog_t), intent(inout) :: prog
    integer(kind=entr_k),   intent(in)    :: ie
    type(spectrum_t),       intent(inout) :: input
    type(spectrum_t),       intent(inout) :: mask
    type(spectrum_t),       intent(inout) :: good
    type(spectrum_t),       intent(inout) :: base
    type(spectrum_t),       intent(inout) :: line
    logical,                intent(inout) :: error
    !
    integer(kind=4) :: ipol
    integer(kind=chan_k) :: ifirst,ilast
    type(chebyshev_t) :: poly
    type(svd_t)       :: svd
    type(spectrum_t)  :: extracted
    character(len=*), parameter :: rname='BASELINE>PROG>POLYNOMIAL>ACT'
    !
    call input%get(ie,error)
    if (error) return
    if (input%y%isblanked()) then
       line%y%val(:) = gr4nan
       base%y%val(:) = gr4nan
    else
       if (prog%mask%do) then
          call mask%get(ie,error)
          if (error) return
          input%w%val(:) = mask%y%val(:)
       endif
       do ipol=1,prog%npol
          ifirst = prog%trans(ipol)+1
          ilast = prog%trans(ipol+1)
          ! Select region to fit the baseline
          call extracted%point_to(input,ifirst,ilast,input%noi,error)
          if (error) return
          call good%mask(extracted,error)
          if (error) return
          if (good%n.gt.prog%degrees(ipol)) then
             call poly%fit(prog%degrees(ipol),good,svd,error)
             if (error) return
             call poly%subtract(input,ifirst,ilast,base,line,error)
             if (error) return
          else
             line%y%val(:) = input%y%val(:)
             base%y%val(:) = gr4nan
          endif
       enddo ! ipol
    endif
    call base%put(ie,error)
    if (error) return
    call line%put(ie,error)
    if (error) return
  end subroutine cubemain_baseline_prog_polynomial_act
end module cubemain_baseline
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
