!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_average
  use cubetools_structure
  use cubedag_flag
  use cube_types
  use cubeadm_cubeid_types
  use cubeadm_index
  use cubemain_messaging
  use cubemain_identifier
  !
  public :: average, average_user_t
  public :: code_current_index, code_cubeid_index, code_given_index
  private
  !
  integer(kind=code_k), parameter :: code_current_index = 1
  integer(kind=code_k), parameter :: code_cubeid_index  = 2
  integer(kind=code_k), parameter :: code_given_index   = 3
  !
  type :: average_comm_t
     type(option_t), pointer :: comm
     type(option_t), pointer :: noise
     type(option_t), pointer :: weight
     type(option_t), pointer :: index
     type(identifier_opt_t)  :: family
   contains
     procedure, public  :: register     => cubemain_average_register
     procedure, private :: parse        => cubemain_average_parse
     procedure, private :: parse_weight => cubemain_average_parse_weight
     procedure, private :: parse_noise  => cubemain_average_parse_noise
     procedure, public  :: main         => cubemain_average_main
  end type average_comm_t
  type(average_comm_t) :: average
  !
  integer(kind=4), parameter :: ione = 1 
  integer(kind=4), parameter :: itwo = 2
  type average_user_t
     type(cubeid_user_t)     :: cubeids
     type(cubeid_user_t)     :: noiseids
     real(kind=sign_k)       :: weights(2)
     type(identifier_user_t) :: family
     logical                 :: donoise
     logical                 :: doweight
     logical                 :: dosum
     !
     ! The adition of an index type in the user type is meant to
     ! circumvent the current state where there is no way of naming
     ! indexes and hence STITCH has to create an index of its on and
     ! then pass it to AVERAGE via the user type.  There is also a
     ! flag type here so that the calling command can set the flag of
     ! the products being created by average
     integer(kind=code_k)    :: index_code
     type(index_t)           :: inindex
     type(flag_t)            :: flag
   contains
     procedure, private :: toprog => cubemain_average_user_toprog
  end type average_user_t
  type average_prog_t
     real(kind=sign_k),allocatable :: weights(:)
     type(identifier_prog_t)       :: family
     logical                       :: donoise
     type(index_t)                 :: incubes
     type(index_t)                 :: noise
     type(cube_t), pointer         :: averaged
     type(cube_t), pointer         :: weight
     integer(kind=pixe_k)          :: nl,nm
     type(flag_t)                  :: flag
     logical                       :: dosum
     procedure(cubemain_average_prog_noise_loop), private, pointer :: loop => null()
   contains
     procedure, private :: header            => cubemain_average_prog_header
     procedure, private :: consistency       => cubemain_average_prog_consistency
     procedure, private :: data              => cubemain_average_prog_data
     procedure, private :: act_noise         => cubemain_average_prog_act_noise
     procedure, private :: act_weight        => cubemain_average_prog_act_weight
     procedure, private :: normalize_and_put => cubemain_average_prog_normalize_and_put
  end type average_prog_t
  !
contains
  !
  subroutine cubemain_average_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(average_user_t) :: user
    character(len=*), parameter :: rname='AVERAGE>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call average%parse(line,user,error)
    if (error) return
    call average%main(user,error)
    if (error) continue
  end subroutine cubemain_average_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_average_register(average,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_comm_t), intent(inout) :: average
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract = &
         'Average two cubes together'
    character(len=*), parameter :: comm_help = &
         'Average two cubes together with different weighting&
         & schemes. By default both cubes are assumed to have the&
         & same weight. This can be changed by using options /NOISE&
         & and /WEIGHT. These two options can be combined and the&
         & resulting weight will be w_i/(noise_i)**2. By default the&
         & resulting cube will have the same family name as cube1.&
         & This can be changed by using option /FAMILY.'
    character(len=*), parameter :: rname='AVERAGE>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Command
    call cubetools_register_command(&
         'AVERAGE','cube1 cube2',&
         comm_abstract,&
         comm_help,&
         cubemain_average_command,&
         average%comm,error)
    if (error) return
    call cubearg%register(&
         'CUBE1',&
         'Input cube #1',  &
         strg_id,&
         code_arg_optional,  &
         [flag_cube],&
         error)
    if (error) return
    call cubearg%register(&
         'CUBE2',&
         'Input cube #2',&
         strg_id,&
         code_arg_optional,&
         [flag_cube],&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'WEIGHT','w1 w2',&
         'Define explicit real weights',&
         strg_id,&
         average%weight,error)
    if (error) return
    call stdarg%register(&
         'w1',&
         'Weight for Input cube #1',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
    call stdarg%register(&
         'w2',&
         'Weight for Input cube #2',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
    !
    call average%family%register(&
         'Define the new family name for products',&
         .not.changeflags,error)
    if (error) return
    !
    call cubetools_register_option(&
         'NOISE','noise1 noise2',&
         'Cubes are weighted by noise',&
         'The weight is computed from the noise images as: 1/(noise&
         &**2). Currently only single noise per spectrum is supported',&
         average%noise,error)
    if (error) return
    call cubearg%register(&
         'NOISE1',&
         'Noise for Input cube #1',  &
         strg_id,&
         code_arg_mandatory,  &
         [flag_noise],&
         error)
    if (error) return
    call cubearg%register(&
         'NOISE2',&
         'Noise for Input cube #2',&
         strg_id,&
         code_arg_mandatory,&
         [flag_noise],&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'INDEX','',&
         'Average current index',&
         'Instead of giving two cubes explicitly use the current&
         & index as an input list, currently options /WEIGHT and &
         &/NOISE are not supported.',&
         average%index,error)
    if (error) return
  end subroutine cubemain_average_register
  !
  subroutine cubemain_average_parse(average,line,user,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(average_comm_t), intent(in)    :: average 
    character(len=*),      intent(in)    :: line
    type(average_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    logical :: docurrent
    character(len=*), parameter :: rname='AVERAGE>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,average%comm,user%cubeids,error)
    if (error) return
    call average%parse_weight(line,user,error)
    if (error) return
    call average%parse_noise(line,user,error)
    if (error) return
    call average%family%parse(line,user%family,error)
    if (error) return
    !
    ! The third type of index, code_given_index, is not currently
    ! accessible via the command line, only via edition of the user
    ! type in FORTRAN
    call average%index%present(line,docurrent,error)
    if (error) return
    if (docurrent) then
       user%index_code = code_current_index
    else
       user%index_code = code_cubeid_index
    endif
    user%flag = flag_average
    user%dosum = .false.
  end subroutine cubemain_average_parse
  !
  subroutine cubemain_average_parse_weight(average,line,user,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_comm_t), intent(in)    :: average 
    character(len=*),      intent(in)    :: line
    type(average_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    integer(kind=4) :: iw
    character(len=*), parameter :: rname='AVERAGE>PARSE>WEIGHT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call average%weight%present(line,user%doweight,error)
    if (error) return
    if (user%doweight) then
       do iw=1,2
          call cubetools_getarg(line,average%weight,iw,user%weights(iw),mandatory,error)
          if (error) return
          if (user%weights(iw).le.0) then
             call cubemain_message(seve%e,rname,'Weights must be positive')
             error = .true.
             return
          endif
       enddo
    else
       ! Do nothing
    endif
  end subroutine cubemain_average_parse_weight
  !
  subroutine cubemain_average_parse_noise(average,line,user,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_comm_t), intent(in)    :: average 
    character(len=*),      intent(in)    :: line
    type(average_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='AVERAGE>PARSE>NOISE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call average%noise%present(line,user%donoise,error)
    if (error) return
    if (user%donoise) then
       call cubeadm_cubeid_parse(line,average%noise,user%noiseids,error)
       if (error) return
    else
       ! Do nothing
    endif
  end subroutine cubemain_average_parse_noise
  !
  subroutine cubemain_average_main(average,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_comm_t), intent(in)    :: average
    type(average_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    type(average_prog_t) :: prog
    character(len=*), parameter :: rname='AVERAGE>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_average_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_average_user_toprog(user,prog,error)
    use gkernel_interfaces
    use cubeadm_get
    use cubetools_header_methods
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_user_t), intent(in)    :: user
    type(average_prog_t),  intent(out)   :: prog
    logical,               intent(inout) :: error
    !
    integer(kind=4) :: inoi,icub,ier
    integer(kind=chan_k) :: nchan
    type(cube_t), pointer :: pcub
    logical :: chanerror
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='AVERAGE>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    select case(user%index_code)
    case(code_current_index)
       if (user%doweight) then
          error = .true.
          call cubemain_message(seve%e,rname,'Option /WEIGHT not supported with option /INDEX')
       endif
       if (user%donoise) then
          error = .true.
          call cubemain_message(seve%e,rname,'Option /NOISE not supported with option /INDEX')
       endif
       if (error) return
       call prog%incubes%get_from_current(code_access_imaset,code_read,error)
       if (error) return
    case(code_cubeid_index)
       call prog%incubes%get_from_cubeid(average%comm,user%cubeids,code_access_imaset,code_read,error)
       if (error) return
    case(code_given_index)
       ! This is the case where an index is provided by a command
       ! editing the user type.
       call user%inindex%copy(prog%incubes,error)
       if (error) return
       ! Here we garantee that the cubes given by the command are
       ! indeed in the good access
       do icub=1,prog%incubes%n
          pcub => prog%incubes%get_cube(icub,error)
          if (error) return
          call cubeadm_access_header(pcub,code_access_imaset,code_read,error)
          if (error) return
       enddo
    case default
       call cubemain_message(seve%e,rname,'Unrecognized index code')
       error = .true.
       return
    end select
    !
    prog%flag  = user%flag
    prog%dosum = user%dosum
    !
    allocate(prog%weights(prog%incubes%n),stat=ier)
    if (failed_allocate(rname,'weight array',ier,error)) return
    if (user%doweight) then
       prog%weights(:) = user%weights(:)
    else
       prog%weights(:) = 1.0
    endif
    !
    prog%donoise = user%donoise
    if (prog%donoise) then
       call prog%noise%get_from_cubeid(average%noise,user%noiseids,code_access_imaset,code_read,error)
       if (error) return
       !
       chanerror = .false.
       do inoi=1,prog%noise%n
          pcub => prog%noise%get_cube(inoi,error)          
          if (error) return
          call cubetools_header_get_nchan(pcub%head,nchan,error)
          if (error) return
          if (nchan.gt.1) then
             write(mess,'(a,i0,a)') 'Noise cube #',inoi,' has more than one channel, not supported'
             call cubemain_message(seve%e,rname,mess)
             chanerror = .true.
          endif
       enddo
       error = chanerror
       if (error) return
       !
       prog%loop => cubemain_average_prog_noise_loop
       !
       if (prog%noise%n.ne.prog%incubes%n) then
          call cubemain_message(seve%e,rname,'Number of cubes and noise references mismatch')
          write(mess,'(2(a,i0),a)') 'There are ',prog%incubes%n,' cubes and ',prog%noise%n,' noise references'
          call cubemain_message(seve%e,rname,mess)
          error = .true.
          return
       endif
    else
       prog%loop => cubemain_average_prog_weight_loop
    endif
    !
    pcub => prog%incubes%get_cube(ione,error)
    if (error) return
    call user%family%toprog(pcub,prog%family,error)
    if (error) return
    call cubetools_header_get_nlnm(pcub%head,prog%nl,prog%nm,error)
    if (error) return
  end subroutine cubemain_average_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_average_prog_header(prog,error)
    use cubetools_header_methods
    use cubedag_allflags 
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cube_t), pointer :: pcub
    integer(kind=4) :: icub
    character(len=*), parameter :: rname='AVERAGE>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call prog%consistency(error)
    if (error) return
    pcub => prog%incubes%get_cube(ione,error)
    if (error) return
    call cubeadm_clone_header(pcub,[prog%flag,flag_cube],prog%averaged,error)
    if (error) return
    call cubeadm_clone_header(pcub,[prog%flag,flag_weight],prog%weight,error)
    if (error) return
    !
    call prog%family%apply(prog%averaged,error)
    if (error) return
    call prog%family%apply(prog%weight,error)
    if (error) return
    !
    do icub=2,prog%incubes%n
       pcub => prog%incubes%get_cube(icub,error)
       if (error) return
       call cubetools_header_add_observatories(pcub%head,prog%averaged%head,error)
       if (error) return
       call cubetools_header_add_observatories(pcub%head,prog%weight%head,error)
       if (error) return
    enddo
    call cubetools_header_put_array_unit('---', prog%weight%head,error)
    if (error) return
  end subroutine cubemain_average_prog_header
  !
  subroutine cubemain_average_prog_consistency(prog,error)
    use cubeadm_consistency
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(consistency_t) :: cons
    character(len=16) :: cubnam, noinam
    type(cube_t), pointer :: pcub, pnoi
    integer(kind=4) :: icub
    character(len=*), parameter :: rname='AVERAGE>PROG>CONSISTENCY'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cons%grid(prog%incubes,error)
    if (error) return
    !
    if (prog%donoise) then
       do icub=1,prog%incubes%n
          pcub => prog%incubes%get_cube(icub,error)
          if (error) return
          pnoi => prog%noise%get_cube(icub,error)
          if (error) return
          write(cubnam,'(a,i0)') 'Cube #',icub
          write(noinam,'(a,i0)') 'Noise #',icub
          call cons%signal_noise(cubnam,pcub,noinam,pnoi,error)
          if (error) return
       enddo
    endif
    !
    call cons%proceed(error)
    if (error) return
  end subroutine cubemain_average_prog_consistency
  !
  subroutine cubemain_average_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='AVERAGE>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error) exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error) &
         call prog%loop(iter%first,iter%last,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_average_prog_data
  !
  subroutine cubemain_average_prog_noise_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: inima,noise,average,weight
    type(cube_t), pointer :: pcub,pnoi
    character(len=*), parameter :: rname='AVERAGE>NOISE>LOOP'
    !
    pcub => prog%incubes%get_cube(ione,error)
    if (error) return
    pnoi => prog%noise%get_cube(ione,error)
    if (error) return
    call inima%init(pcub,error)
    if (error) return
    call noise%init(pnoi,error)
    if (error) return
    call average%reallocate('average',prog%nl,prog%nm,error)
    if (error) return
    call weight%reallocate('weight',prog%nl,prog%nm,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error) return
      call prog%act_noise(ie,inima,noise,average,weight,error)
      if (error) return
    enddo
  end subroutine cubemain_average_prog_noise_loop
  !
  subroutine cubemain_average_prog_act_noise(prog,ie,inima,noise,average,weight,error)
    use cubetools_nan
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t),         intent(inout) :: inima
    type(image_t),         intent(inout) :: noise
    type(image_t),         intent(inout) :: average
    type(image_t),         intent(inout) :: weight
    logical,               intent(inout) :: error
    !
    integer(kind=pixe_k) :: il,im
    integer(kind=4) :: icub
    real(kind=sign_k) :: wei
    type(cube_t), pointer :: pcub,pnoi
    character(len=*), parameter :: rname='AVERAGE>PROG>ACT>NOISE'
    !
    average%z(:,:) = 0
    weight%z(:,:) = 0
    do icub=1,prog%incubes%n
       pcub => prog%incubes%get_cube(icub,error)
       if (error) return
       pnoi => prog%noise%get_cube(icub,error)
       if (error) return
       call inima%get(pcub,ie,error)
       if (error) return
       call noise%get(pnoi,ie,error)
       if (error) return
       do im=1,prog%nm
          do il=1,prog%nl
             if (.not.ieee_is_nan(inima%z(il,im))) then
                wei = prog%weights(icub)/noise%z(il,im)**2
                average%z(il,im) = average%z(il,im)+inima%z(il,im)*wei
                weight%z(il,im) = weight%z(il,im)+wei
             endif
          enddo
       enddo
    enddo
    !
    call prog%normalize_and_put(ie,average,weight,error)
    if (error) return
  end subroutine cubemain_average_prog_act_noise
  !
  subroutine cubemain_average_prog_weight_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: inima,average,weight
    type(cube_t), pointer :: pcub
    character(len=*), parameter :: rname='AVERAGE>WEIGHT>LOOP'
    !
    pcub => prog%incubes%get_cube(ione,error)
    if (error) return
    call inima%init(pcub,error)
    if (error) return
    call average%reallocate('average',prog%nl,prog%nm,error)
    if (error) return
    call weight%reallocate('weight',prog%nl,prog%nm,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error) return
      call prog%act_weight(ie,inima,average,weight,error)
      if (error) return
    enddo
  end subroutine cubemain_average_prog_weight_loop
  !
  subroutine cubemain_average_prog_act_weight(prog,ie,inima,average,weight,error)
    use cubetools_nan
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t),         intent(inout) :: inima
    type(image_t),         intent(inout) :: average
    type(image_t),         intent(inout) :: weight
    logical,               intent(inout) :: error
    !
    integer(kind=pixe_k) :: il,im
    integer(kind=4) :: icub
    type(cube_t), pointer :: pcub
    character(len=*), parameter :: rname='AVERAGE>PROG>ACT>WEIGHT'
    !
    average%z(:,:) = 0
    weight%z(:,:) = 0
    do icub=1,prog%incubes%n
       pcub => prog%incubes%get_cube(icub,error)
       if (error) return
       call inima%get(pcub,ie,error)
       if (error) return
       do im=1,prog%nm
          do il=1,prog%nl
             if (.not.ieee_is_nan(inima%z(il,im)).and..not.ieee_is_nan(weight%z(il,im))) then
                average%z(il,im) = average%z(il,im)+inima%z(il,im)*prog%weights(icub)
                weight%z(il,im) = weight%z(il,im)+prog%weights(icub)
             endif
          enddo
       enddo
    enddo
    !
    call prog%normalize_and_put(ie,average,weight,error)
    if (error) return
  end subroutine cubemain_average_prog_act_weight
  !
  subroutine cubemain_average_prog_normalize_and_put(prog,ie,average,weight,error)
    use cubetools_nan
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t),         intent(inout) :: average
    type(image_t),         intent(inout) :: weight
    logical,               intent(inout) :: error
    !
    integer(kind=pixe_k) :: il,im
    character(len=*), parameter :: rname='AVERAGE>PROG>NORMALIZE>AND>PUT'
    !
    do im=1,prog%nm
       do il=1,prog%nl
          if (weight%z(il,im).eq.0) then
             average%z(il,im) = gr4nan
          else if (prog%dosum) then
             ! do nothing as average is already the sum
          else 
             average%z(il,im) = average%z(il,im)/weight%z(il,im)
          endif
       end do
    enddo
    call average%put(prog%averaged,ie,error)
    if (error) return
    call weight%put(prog%weight,ie,error)
    if (error) return
  end subroutine cubemain_average_prog_normalize_and_put
end module cubemain_average
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
