module cubedag_dag
  use cubedag_messaging
  use cubedag_parameters
  use cubedag_types
  use cubedag_index
  use cubedag_node
  use cubedag_tuple
  use cubedag_type

  integer(kind=iden_l) :: id_counter
  integer(kind=iden_l) :: id_null=-1  ! 0 reserved for root
  class(cubedag_node_object_t), pointer :: root=>null()
  type(cubedag_optimize_t), target :: ix,cx

  public :: ix,cx
  public :: cubedag_dag_root,cubedag_dag_newnode,cubedag_dag_newbranch,  &
            cubedag_dag_attach,cubedag_dag_removenode
  public :: cubedag_dag_updatecounter,cubedag_dag_resetcounter
  public :: cubedag_dag_destroy,cubedag_dag_entrynum
  public :: cubedag_dag_get_object,cubedag_dag_get_root
  public :: cubedag_dag_memsize,cubedag_dag_disksize
  public :: cubedag_dag_newid,cubedag_dag_nullid
  public :: cubedag_dag_contains,cubedag_dag_empty
  public :: cubedag_link_resolve
  private

contains

  subroutine cubedag_dag_root(error)
    !-------------------------------------------------------------------
    ! Create the root node, i.e. the one with no parent at all.
    !-------------------------------------------------------------------
    logical, intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='NODE>ROOT'
    !
    ! Sanity
    if (associated(root)) then
      call cubedag_message(seve%e,rname,'Root node already exists')
      error = .true.
      return
    endif
    if (ix%next.ne.1) then
      call cubedag_message(seve%e,rname,'Root node should be first in index!')
      error = .true.
      return
    endif
    !
    call cubedag_dag_newnode(root,code_type_node,error)
    if (error)  return
    root%node%family = '<root>'
    root%node%origin = code_origin_root
    root%node%history = 0
    !
    ! Insert in DAG
    call cubedag_dag_attach(root,error)
    if (error)  return
    !
    ! Also add a lightweight tag pointing to this root node
    ! ZZZ ?
  end subroutine cubedag_dag_root

  subroutine cubedag_dag_get_root(ptr)
    !-------------------------------------------------------------------
    ! Return a pointer to the root node
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer :: ptr
    ptr => root
  end subroutine cubedag_dag_get_root

  subroutine cubedag_dag_newbranch(object,ftype,error)
    !-------------------------------------------------------------------
    ! Insert a new node (already allocated out of this subroutine) in
    ! the DAG, attached to the root.
    ! This materializes the start of a new branch in the DAG.
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: object
    integer(kind=code_k),         intent(in)    :: ftype
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='DAG>NEWBRANCH'
    integer(kind=entr_k) :: np
    type(cubedag_link_t) :: parents,twins
    !
    ! Sanity
    if (.not.associated(root)) then
      call cubedag_message(seve%e,rname,'Root node does not exist yet')
      error = .true.
      return
    endif
    !
    ! Minimal setup of the node_t part
    object%node%type = ftype
    call cubedag_dag_newid(object%node%id)
    call cubedag_tuple_reset(object%node%tuple)
    !
    ! Insert in DAG
    call cubedag_dag_attach(object,error)
    if (error)  return
    !
    ! Link this node to the root
    np = 1
    call parents%reallocate(np,error)
    if (error)  return
    parents%n = 1
    parents%list(1)%p => root
    twins%n = 0  ! No twin brother
    call cubedag_node_link(object,parents,twins,error)
    if (error)  return
    call parents%final(error)
    call twins%final(error)
  end subroutine cubedag_dag_newbranch

  subroutine cubedag_dag_newnode(object,itype,error)
    !-------------------------------------------------------------------
    ! Create a new node standalone node (not attached to the DAG)
    ! This assumes connection and parents will be done later on
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: object
    integer(kind=code_k),         intent(in)    :: itype
    logical,                      intent(inout) :: error
    !
    ! Allocate the object in memory
    call cubedag_type_allocate(object,itype,error)
    if (error)  return
    !
    call cubedag_dag_newid(object%node%id)
    call cubedag_tuple_reset(object%node%tuple)
  end subroutine cubedag_dag_newnode

  subroutine cubedag_dag_attach(object,error)
    !-------------------------------------------------------------------
    ! Attach a node to the DAG
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: object
    logical,                      intent(inout) :: error
    !
    call cubedag_index_reallocate_expo(ix,ix%next,error)
    if (error)  return
    !
    ! --- Insert in memory ---
    object%node%ient = ix%next
    ix%object(ix%next)%p => object
    ix%next = ix%next+1
    !
    ! --- Insert on disk ---
    ! ZZZ TO BE DONE
    ! ZZZ Insert in real time, or later?
  end subroutine cubedag_dag_attach

  subroutine cubedag_dag_destroy(error)
    !-------------------------------------------------------------------
    ! Brute-force destroy the whole DAG
    !-------------------------------------------------------------------
    logical, intent(inout) :: error
    ! Local
    integer(kind=entr_k) :: ient
    !
    ! Bruce force clean, do not worry about family links between nodes
    do ient=1,ix%next-1
      call cubedag_node_destroy(ix%object(ient)%p,error)
      if (error)  continue
    enddo
    ix%next = 1
    root => null()
  end subroutine cubedag_dag_destroy

  subroutine cubedag_dag_removenode(id,error)
    !-------------------------------------------------------------------
    ! Properly remove a node from the DAG, taking care of its family
    ! links
    !-------------------------------------------------------------------
    integer(kind=iden_l), intent(in)    :: id
    logical,              intent(inout) :: error
    !
    ! Unreference from CX, before actual deletion from IX
    call cubedag_dag_removenode_from(cx,id,.false.,error)
    if (error)  return
    ! Unreference from IX
    call cubedag_dag_removenode_from(ix,id,.true.,error)
    if (error)  return
    !
  end subroutine cubedag_dag_removenode

  subroutine cubedag_dag_removenode_from(optx,id,rmnode,error)
    !-------------------------------------------------------------------
    ! Properly remove a node from one index, taking care of its family
    ! links
    !-------------------------------------------------------------------
    type(cubedag_optimize_t), intent(inout) :: optx
    integer(kind=iden_l),     intent(in)    :: id
    logical,                  intent(in)    :: rmnode
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='DAG>REMOVE>NODE'
    integer(kind=entr_k) :: ient,shift
    logical :: found
    !
    found = .false.
    shift = 0
    do ient=1,optx%next-1
      if (optx%object(ient)%p%node%id.eq.id) then
        if (rmnode) then
          call cubedag_node_remove(optx%object(ient)%p,error)
          if (error)  return
        endif
        found = .true.
        shift = shift+1
        cycle
      endif
      if (found) then
        optx%object(ient-shift)%p   => optx%object(ient)%p
        optx%topomarker(ient-shift) =  optx%topomarker(ient)  ! Probably useless
      endif
    enddo
    optx%next = optx%next-shift
    !
    if (rmnode .and. .not.found)  &
      call cubedag_message(seve%e,rname,'ID not found in index')
    !
  end subroutine cubedag_dag_removenode_from

  !---------------------------------------------------------------------

  subroutine cubedag_dag_newid(id)
    !-------------------------------------------------------------------
    ! Generate a unique (unused yet) identifier for a new node
    !-------------------------------------------------------------------
    integer(kind=iden_l), intent(out) :: id
    id = id_counter
    id_counter = id_counter+1
  end subroutine cubedag_dag_newid

  subroutine cubedag_dag_nullid(id)
    !-------------------------------------------------------------------
    ! Set the id to the 'null' (unset yet) value
    !-------------------------------------------------------------------
    integer(kind=iden_l), intent(out) :: id
    id = id_null
  end subroutine cubedag_dag_nullid

  subroutine cubedag_dag_updatecounter(error)
    !-------------------------------------------------------------------
    ! Ensure the ID counter is ready to give a unique identifier
    !-------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    ! Robust way:
    ! maxid = 0
    ! do ient=1,ix%next-1
    !   maxid = max(maxid,ix%object(ient)%p%node%id)
    ! enddo
    ! id_counter = maxid+1
    !
    ! Efficient way, assuming the identifiers are SORTED
    id_counter = ix%object(ix%next-1)%p%node%id+1  ! Last ID + 1
    !
  end subroutine cubedag_dag_updatecounter

  subroutine cubedag_dag_resetcounter(error)
    !-------------------------------------------------------------------
    ! Reset the ID counter
    !-------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    id_counter = 0
  end subroutine cubedag_dag_resetcounter

  function cubedag_dag_entrynum(id,error)
    !-------------------------------------------------------------------
    ! Resolve the entry number corresponding to the given id.
    ! This resolution is based on two strong assumptions:
    !  1) the identifier is UNIQUE,
    !  2) the identifier list is SORTED
    !-------------------------------------------------------------------
    integer(kind=entr_k) :: cubedag_dag_entrynum
    integer(kind=iden_l), intent(in)    :: id
    logical,              intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='NODE>ENTRYNUM'
    integer(kind=entr_k) :: inf,mid,sup
    !
    ! Dichotomic search
    if (ix%object(1)%p%node%id.eq.id) then
      cubedag_dag_entrynum = 1
      return
    endif
    !
    inf = 1
    sup = ix%next-1
    do while (sup.gt.inf+1)
      mid = (inf+sup)/2  ! Integer division
      if (ix%object(mid)%p%node%id.lt.id) then
        inf = mid
      else
        sup = mid
      endif
    enddo
    !
    if (ix%object(sup)%p%node%id.eq.id) then
      cubedag_dag_entrynum = sup
    else
      call cubedag_message(seve%e,rname,'No such identifier in DAG')
      cubedag_dag_entrynum = 0
      error = .true.
      return
    endif
    !
  end function cubedag_dag_entrynum

  subroutine cubedag_dag_get_object(id,object,error)
    !-------------------------------------------------------------------
    ! This subroutine should not be used, except when reconstructing
    ! graph links (parents, children, etc) from a list of IDs to a list
    ! of nodes in memory.
    !-------------------------------------------------------------------
    integer(kind=iden_l),         intent(in)    :: id
    class(cubedag_node_object_t), pointer       :: object ! Associated on return
    logical,                      intent(inout) :: error  !
    ! Local
    integer(kind=entr_k) :: ient
    !
    ient = cubedag_dag_entrynum(id,error)
    if (error)  return
    object => ix%object(ient)%p
  end subroutine cubedag_dag_get_object

  subroutine cubedag_link_resolve(link,error)
    !-------------------------------------------------------------------
    ! Resolve the cross-links (from IDs to pointer) for the given list.
    ! This assumes link%flag(:) contains the ids
    !-------------------------------------------------------------------
    type(cubedag_link_t), intent(inout) :: link
    logical,              intent(inout) :: error
    ! Local
    integer(kind=entr_k) :: il
    integer(kind=iden_l) :: id
    class(cubedag_node_object_t), pointer :: targ
    !
    do il=1,link%n
      id = link%flag(il)
      call cubedag_dag_get_object(id,targ,error)
      if (error)  return
      link%list(il)%p => targ
    enddo
    !
  end subroutine cubedag_link_resolve

  function cubedag_dag_memsize()
    !-------------------------------------------------------------------
    ! Return the DAG size in memory
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubedag_dag_memsize  ! [bytes]
    ! Local
    integer(kind=entr_k) :: ient
    !
    cubedag_dag_memsize = 0
    do ient=1,ix%next-1
      cubedag_dag_memsize = cubedag_dag_memsize + ix%object(ient)%p%memsize()
    enddo
  end function cubedag_dag_memsize

  function cubedag_dag_disksize()
    !-------------------------------------------------------------------
    ! Return the DAG size on disk
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubedag_dag_disksize  ! [bytes]
    ! Local
    integer(kind=entr_k) :: ient
    !
    cubedag_dag_disksize = 0
    do ient=1,ix%next-1
      cubedag_dag_disksize = cubedag_dag_disksize + ix%object(ient)%p%disksize()
    enddo
  end function cubedag_dag_disksize

  function cubedag_dag_contains(file)
    !-------------------------------------------------------------------
    ! Return .true. is the DAG contains a node referencing the named
    ! file.
    !-------------------------------------------------------------------
    logical :: cubedag_dag_contains
    character(len=*), intent(in) :: file
    ! Local
    integer(kind=entr_k) :: ient
    !
    cubedag_dag_contains = .false.
    do ient=1,ix%next-1
      if (ix%object(ient)%p%node%tuple%contains(file)) then
        cubedag_dag_contains = .true.
        return
      endif
    enddo
  end function cubedag_dag_contains

  function cubedag_dag_empty()
    !-------------------------------------------------------------------
    ! Return .true. if the DAG is empty
    !-------------------------------------------------------------------
    logical :: cubedag_dag_empty
    cubedag_dag_empty = ix%next.le.2  ! Ignore the root node
  end function cubedag_dag_empty

end module cubedag_dag
