!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_stitch
  use cubetools_structure
  use cubetools_axis_types
  use cubetools_spapro_types
  use cube_types
  use cubemain_messaging
  use cubemain_lists
  use cubemain_speline_types
  use cubemain_auxiliary
  !
  public :: cubemain_stitch_command,cubemain_stitch_register,cubemain_stitch_observatory
  private
  !
  type :: stitch_comm_t
     type(option_t), pointer   :: stitch     
     type(option_t), pointer   :: weight
     type(speline_opt_t)       :: freq
     type(axis_opt_t)          :: faxis
     type(spapro_type_opt_t)   :: ptype
     type(spapro_center_opt_t) :: pcenter
     type(spapro_angle_opt_t)  :: pangle
     type(axis_opt_t)          :: laxis         
     type(axis_opt_t)          :: maxis         
     type(option_t), pointer   :: like
     type(option_t), pointer   :: split
  end type stitch_comm_t
  type(stitch_comm_t) :: comm
  !
  type stitch_user_t
     type(cublist_t)            :: inlist     ! input cube names
     logical                    :: split      ! Use stitch v1 or v2?
     type(speline_user_t)       :: line       ! Optional new line name and freq
     type(axis_user_t)          :: faxis      ! [Mhz] User spectral axis description
     type(axis_user_t)          :: laxis      ! [RAD|DEG|MIN|SEC] User l axis description
     type(axis_user_t)          :: maxis      ! [RAD|DEG|MIN|SEC] User m axis description
     type(auxiliary_user_t)     :: like       ! Reference cube
     logical                    :: donoise    ! Weight according to noise?
     type(spapro_type_user_t)   :: ptype
     type(spapro_center_user_t) :: pcenter
     type(spapro_angle_user_t)  :: pangle
  end type stitch_user_t
  type stitch_prog_t
     type(cublist_t)       :: inlist            ! input cubes
     type(cube_t), pointer :: stitch            ! output cube
     type(cube_t), pointer :: weight            ! output weight
     logical               :: dorepro = .false. ! Reproject?
     logical               :: doresam = .false. ! Resample?
     logical               :: donoise           ! Weight according to noise?
     logical               :: split             ! Use stitch v1 or v2?
  end type stitch_prog_t
  !
contains
  !
  subroutine cubemain_stitch_register(error)
    use cubedag_allflags
    use cubetools_unit
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    character(len=*), parameter :: comm_abstract = 'Stitch cubes together'
    character(len=*), parameter :: comm_help = &
         'The spatial and spectral axes of the output cube are&
         & computed as the Union of the spectral and spatial grids of&
         & the input cubes, unless they are explicitly given'
    character(len=*), parameter :: rname='STITCH>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! call cubetools_register_command(&
    !      'STITCH',&
    !      comm_abstract,&
    !      comm_help,&
    !      cubemain_stitch_command,&
    !      error)
    ! if (error) return
    ! call cubemain_cublist_register(&
    !      'STITCH',&
    !      comm_abstract,&
    !      comm_help,&
    !      comm%stitch,error)
    ! if (error) return
    !
    call cubemain_cublist_register_weights(&
         'WEIGHT',&
         'Define the weighting scheme for the stitching',&
         comm%weight,error)
    if (error) return
    !
    call comm%freq%register(&
         'Define line name and frequency of the stitched cube',&
         error)
    if (error) return
    !
    call comm%faxis%register(&
         code_unit_freq, &
         'FAXIS',&
         'Define the frequency axis of the stitched cube',&
         error)
    if (error) return
    !
    call comm%ptype%register(&
         'PTYPE',&
         'Define the new projection type',&
         error)
    if (error) return
    call comm%pcenter%register(&
         'PCENTER',&
         'Define the new projection center',&
         error)
    if (error) return
    call comm%pangle%register(&
         'PANGLE',&
         'Define the new projection angle',&
         error)
    if (error) return
    !
    call comm%laxis%register(&
         code_unit_fov,&
         'LAXIS',&
         'Define the l axis of the stitched cube',&
         error)
    if (error) return
    !
    call comm%maxis%register(&
         code_unit_fov,&
         'MAXIS',&
         'Define the M axis of the stitched cube',&
         error)
    if (error) return
    !
    call cubemain_auxiliary_register(&
         'LIKE',&
         'Stitch onto the same grid as a refcube',&
         strg_id,&
         'Reference cube',&
         [flag_cube],&
         mandatory, &
         comm%like,error)
    if (error) return
    !
    call cubetools_register_option(&
         'SPLIT','',&
         'Stitch using split step algorithm',&
         strg_id,&
         comm%split,error)
    if (error) return
  end subroutine cubemain_stitch_register
  !
  subroutine cubemain_stitch_command(line,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(stitch_user_t) :: user
    type(stitch_prog_t) :: prog
    character(len=*), parameter :: rname='STITCH>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    call cubemain_stitch_parse(line,user,error)
    if (error) return
    call cubemain_stitch_main(user,prog,error)
    if (error) continue
  end subroutine cubemain_stitch_command
  !
  subroutine cubemain_stitch_parse(line,user,error)
    !----------------------------------------------------------------------
    ! STITCH cub1 cub2 ... cubn
    ! [/WEIGHT noise|equal|w1 w2 ... wn]
    ! [/FREQUENCY line freq]
    ! [/FAXIS n ref val inc]
    ! [/PROJECTION [type a0 d0 [pang [DEG|RAD]]]]
    ! [/LAXIS n ref val inc [unit]]
    ! [/MAXIS n ref val inc [unit]]
    ! [/LIKE refcube]
    !----------------------------------------------------------------------
    character(len=*),    intent(in)    :: line
    type(stitch_user_t), intent(out)   :: user
    logical,             intent(inout) :: error
    !
    logical :: conflict
    character(len=*), parameter :: rname='STITCH>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubemain_cublist_parse(line,comm%stitch,user%inlist,error)
    if(error) return
    call cubemain_cublist_parse_weights(line,comm%weight,user%inlist,user%donoise,error)
    if(error) return
    call comm%faxis%parse(line,user%faxis,error)
    if (error) return
    call comm%freq%parse(line,user%line,error)
    if (error) return
    call comm%ptype%parse(line,user%ptype,error)
    if (error) return
    call comm%pcenter%parse(line,user%pcenter,error)
    if (error) return
    call comm%pangle%parse(line,user%pangle,error)
    if (error) return
    call comm%laxis%parse(line,user%laxis,error)
    if (error) return
    call comm%maxis%parse(line,user%maxis,error)
    if (error) return
    call cubemain_auxiliary_parse(line,comm%like,user%like,error)
    if (error) return
    if (user%like%do) then
       conflict = user%faxis%do.or.user%line%do.or.user%ptype%do.or.user%pcenter%do.or.user%pangle%do
       conflict = conflict.or.user%laxis%do.or.user%maxis%do
       if (conflict) then
          call cubemain_message(seve%e,rname,&
               'Option /LIKE is exclusive to all options except /WEIGHT and /OUTPUT')
          error = .true.
          return
       endif
    endif
    !
    ! VVV Logical split determines wether we are doing the last
    ! reproject|resample operation split from the adding operation, its
    ! value is controlled by the presence of option /SPLIT.
    call comm%split%present(line,user%split,error)
    if (error) return
    !
    ! VVV stitch2 is a priori faster than stitch1, but it is also harder
    ! to maintain as parts of its code are duplicated to allow for
    ! efficiency
    !
    call cubemain_stitch_feedback(user)
  end subroutine cubemain_stitch_parse
  !
  subroutine cubemain_stitch_feedback(user)
    !----------------------------------------------------------------------
    ! Gives the user feedback on the inputs
    !----------------------------------------------------------------------
    type(stitch_user_t), intent(in) :: user
    !
    integer(kind=4) :: icub
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='STITCH>FEEDBACK'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')    
    !
    call cubemain_message(seve%r,rname,'')
    write(mess,*) 'Input cubes and their weights: '
    call cubemain_message(seve%r,rname,mess)
    call cubemain_message(seve%r,rname,'')
    do icub=1,user%inlist%n
       write(mess,'(a,f5.1,a)') 'w=',user%inlist%entries(icub)%weig,'; '//trim(user%inlist%entries(icub)%name)
       call cubemain_message(seve%r,rname,mess)
    enddo
    call cubemain_message(seve%r,rname,'')
  end subroutine cubemain_stitch_feedback
  !
  subroutine cubemain_stitch_main(user,prog,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(stitch_user_t), intent(in)    :: user
    type(stitch_prog_t), intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='STITCH>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubemain_stitch_header(user,prog,error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call cubemain_stitch_data(prog,error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_stitch_main
  !
  subroutine cubemain_stitch_header(user,prog,error)
    use cubetools_header_types
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(stitch_user_t), intent(in)    :: user
    type(stitch_prog_t), intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='STITCH>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubemain_cublist_copy(user%inlist,prog%inlist,error)
    if(error) return
    !
    prog%donoise=user%donoise
    prog%split = user%split
    !
    call cubemain_cublist_get_headers(code_access_imaset_or_speset,prog%inlist,error)
    if(error) return
    !
    if (user%like%do) then
       call cubemain_stitch_header_like(user,prog,error)
    else
       call cubemain_stitch_header_noref(user,prog,error)
    endif
    if(error) return
    !
    call cubemain_stitch_header_clone([flag_stitch,flag_weight],prog,prog%weight,error)
    if (error) return
    call cubetools_header_copy(prog%stitch%head,prog%weight%head,error)
    if (error) return
    !
    print *,'Stitched header:'
    call prog%stitch%head%list(error)
    !
  end subroutine cubemain_stitch_header
  !
  subroutine cubemain_stitch_header_like(user,prog,error)
    use cubetools_spatial_types
    use cubetools_header_methods
    use cubedag_allflags
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(stitch_user_t), intent(in)    :: user
    type(stitch_prog_t), intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    type(cube_t), pointer :: reference
    character(len=*), parameter :: rname='STITCH>HEADER>LIKE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubemain_auxiliary_user2prog(comm%like,code_access_imaset_or_speset,&
         user%like,reference,error)
    if (error) return
    call cubemain_stitch_consistency(reference,prog,error)
    if (error) return
    call cubemain_stitch_header_clone([flag_stitch,flag_cube],prog,prog%stitch,error)
    if (error) return
    ! Now update spectral section from the reference
    call cubetools_header_spectral_like(reference%head,prog%stitch%head,error)
    if (error) return
    ! Now copy the spatial section
    call cubetools_header_spatial_like(reference%head,prog%stitch%head,error)
    if (error) return
    ! Now stitch the telescope sections together
    call cubemain_stitch_observatory(prog%inlist,prog%stitch,error)
    if (error) return
  end subroutine cubemain_stitch_header_like
  !
  subroutine cubemain_stitch_header_noref(user,prog,error)
    use cubetools_header_methods
    use cubedag_allflags
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    type(stitch_user_t), intent(in)    :: user
    type(stitch_prog_t), intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    type(axis_t) :: laxis,maxis,faxis
    type(speline_prog_t) :: line
    character(len=*), parameter :: rname='STITCH>HEADER>NOREF'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !    
    call cubemain_stitch_consistency(prog%inlist%entries(1)%cube,prog,error)
    if (error) return
    prog%dorepro = prog%dorepro.or.user%ptype%do.or.user%pcenter%do.or.user%pangle%do
    prog%dorepro = prog%dorepro.or.user%laxis%do.or.user%maxis%do
    prog%doresam = prog%doresam.or.user%line%do.or.user%faxis%do
    !
    call cubemain_stitch_header_clone([flag_stitch,flag_cube],prog,prog%stitch,error)
    if (error) return
    !
    call comm%ptype%user2prog(user%ptype,prog%stitch%head%spa%pro,error)
    if (error) return
    call comm%pcenter%user2prog(prog%stitch%head%spa%fra,user%pcenter,prog%stitch%head%spa%pro,error)
    if (error) return
    call comm%pangle%user2prog(user%pangle,prog%stitch%head%spa%pro,error)
    if (error) return
    !
    if (prog%dorepro.or.user%laxis%do.or.user%maxis%do.or.&
         user%ptype%do.or.user%pcenter%do.or.user%pangle%do) then
       call cubetools_axis_init(laxis,error)
       if (error) return
       call cubetools_axis_init(maxis,error)
       if (error) return
       call cubemain_stitch_spatial(prog,laxis,maxis,error)
       if (error) return
       if (user%laxis%do) then
          call comm%laxis%user2prog(user%laxis,prog%stitch%head%spa%l,laxis,error)
          if (error) return
       endif
       if (user%maxis%do) then
          call comm%maxis%user2prog(user%maxis,prog%stitch%head%spa%m,maxis,error)
          if (error) return
       endif
       !
       call cubetools_header_update_axset_l(laxis,prog%stitch%head,error)
       if (error) return
       call cubetools_header_update_axset_m(maxis,prog%stitch%head,error)
       if (error) return
    endif
    if (prog%doresam.or.user%faxis%do.or.user%line%do) then
       call cubetools_axis_init(faxis,error)
       if (error) return
       if (user%line%do) then
          call user%line%toprog(prog%stitch,line,error)
          if (error) return
          call cubetools_header_modify_rest_frequency(line%freq,prog%stitch%head,error)
          if (error) return
          call cubetools_header_put_line(line%name,prog%stitch%head,error)
          if (error) return
       endif
       if (user%faxis%do) then
          if (user%line%do.and.(faxis%val.ne.line%freq)) then
             call cubemain_message(seve%w,rname,'Two different rest frequencies given')
             call cubemain_message(seve%w,rname,'Using the one from the frequency axis')
          endif
          call comm%faxis%user2prog(user%faxis,prog%stitch%head%spe%f,faxis,error)
          if (error) return
       else
          call cubemain_stitch_spectral(prog,faxis,error)
          if (error) return
       endif
       call cubetools_header_update_frequency_from_axis(faxis,prog%stitch%head,error)
       if (error) return
    endif
    call cubemain_stitch_observatory(prog%inlist,prog%stitch,error)
    if (error) return
  end subroutine cubemain_stitch_header_noref
  !
  subroutine cubemain_stitch_consistency(reference,prog,error)
    use cubetools_header_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(cube_t), pointer, intent(in)    :: reference
    type(stitch_prog_t),   intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    integer(kind=4) :: icub
    integer(kind=4) :: consisprob
    logical :: problem
    type(cube_header_cons_t) :: cons
    character(len=*), parameter :: rname='STITCH>CONSISTENCY'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !    
    consisprob = 0
    do icub=1,prog%inlist%n
       call cubetools_header_consistency_init(cons,error)
       if(error) return
       !
       ! Disabling less relevant checks
       cons%obs%check =.false.
       call cubetools_header_consistency_check(cons,reference%head,prog%inlist%entries(icub)%cube%head,error)
       if(error) return
       !
       if (cons%prob) then
          ! Check here what most be equal to continue
          problem = cons%spa%sou%prob.or.cons%spa%bea%prob
          problem = problem.or.cons%spe%frame%prob.or.cons%spe%conv%prob
          !
          prog%dorepro = cons%spa%prob
          prog%doresam = cons%spe%prob
          !
          if (problem) then
             call cubemain_message(seve%e,rname,'Cube '//trim(prog%inlist%entries(icub)%name)//&
                  ' is inconsistent with the reference cube')
             consisprob = consisprob+1
             call cubetools_header_consistency_print(cons,reference%head,&
                  prog%inlist%entries(icub)%cube%head,error)
             if(error) return
          endif
       endif
    enddo ! icub
    !
    if (consisprob.ne.0) then
       call cubemain_message(seve%e,rname,'There are severe inconsistencies between the input cubes')
       error = .true.
       goto 10
    endif
    !
10  continue
    call cubetools_header_consistency_final(cons,error)
    if(error) return
  end subroutine cubemain_stitch_consistency
  !
  subroutine cubemain_stitch_header_clone(flags,prog,oucube,error)
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(flag_t),          intent(in)    :: flags(:)
    type(stitch_prog_t),   intent(in)    :: prog
    type(cube_t), pointer, intent(inout) :: oucube
    logical,               intent(inout) :: error
    !
    integer(kind=code_k) :: ouaccess
    character(len=*), parameter :: rname='STITCH>HEADER>CLONE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    if (prog%inlist%access.eq.code_cube_imaset) then
       if (prog%doresam) then
          ouaccess = code_cube_speset
       else
          ouaccess = code_cube_imaset
       endif
    else if (prog%inlist%access.eq.code_cube_speset) then
       if (prog%dorepro) then
          ouaccess = code_cube_imaset
       else
          ouaccess = code_cube_speset
       endif
    else
       call cubemain_message(seve%e,rname,'Unknown access for entry list')
       error = .true.
       return
    endif
    !
    call cubeadm_clone_header(prog%inlist%entries(1)%cube,flags,&
         oucube,error,access=ouaccess)
    if (error) return
  end subroutine cubemain_stitch_header_clone
  !
  subroutine cubemain_stitch_observatory(cublist,oucube,error)
    use cubetools_header_methods
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(cublist_t),       intent(in)    :: cublist
    type(cube_t), pointer, intent(inout) :: oucube 
    logical,               intent(inout) :: error
    !
    integer(kind=4) :: icub
    character(len=*), parameter :: rname='STITCH>OBSERVATORY'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    do icub=1,cublist%n
       call cubetools_header_add_observatories(cublist%entries(icub)%cube%head,oucube%head,error)
       if (error) return
    enddo
  end subroutine cubemain_stitch_observatory
  !
  subroutine cubemain_stitch_spectral(prog,faxis,error)
    use cubetools_unit
    use cubemain_topology    
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(stitch_prog_t), intent(in)    :: prog
    type(axis_t),        intent(inout) :: faxis
    logical,             intent(inout) :: error
    !
    real(kind=coor_k) :: fmin, fmax, lfmin, lfmax
    real(kind=coor_k) :: lfres
    integer(kind=4) :: icub
    character(len=*), parameter :: rname='STITCH>SPECTRAL'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubemain_topo_fminfmax(prog%inlist%entries(1)%cube,fmin,fmax,error)
    if (error) return
    faxis%ref = prog%stitch%head%spe%ref%c
    faxis%val = prog%stitch%head%spe%ref%f
    faxis%inc = prog%stitch%head%spe%inc%f
    !
    do icub=1,prog%inlist%n
       call cubemain_topo_fminfmax(prog%inlist%entries(icub)%cube,lfmin,lfmax,error)
       if (error) return
       lfres = prog%inlist%entries(icub)%cube%head%spe%inc%f
       fmin = min(fmin,lfmin)
       fmax = max(fmax,lfmax)
       if (abs(lfres).gt.abs(faxis%inc)) faxis%inc = lfres
    enddo
    faxis%n = ceiling((fmax-fmin)/abs(faxis%inc))
    if (faxis%inc.lt.0) then
       faxis%ref  = -(fmax-faxis%val-0.5*faxis%inc)/faxis%inc
    else
       faxis%ref  = (faxis%val-fmin+0.5*faxis%inc)/faxis%inc
    endif
    !
    faxis%kind = code_unit_freq
    faxis%unit = prog%stitch%head%spe%f%unit
    faxis%name = 'FREQUENCY'
    faxis%genuine = .true.
  end subroutine cubemain_stitch_spectral
  !
  subroutine cubemain_stitch_spatial(prog,laxis,maxis,error)
    use cubetools_unit
    use cubetools_header_interface
    use cubemain_spatial_coordinates
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    type(stitch_prog_t), intent(in)    :: prog
    type(axis_t),        intent(inout) :: laxis
    type(axis_t),        intent(inout) :: maxis
    logical,             intent(inout) :: error
    !
    real(kind=coor_k) :: absmin(2),absmax(2),corners(2,4),projcorners(2,4)
    real(kind=coor_k) :: llinc,lminc
    integer(kind=4) :: iax,icub
    integer(kind=8), parameter :: four=4
    character(len=*), parameter :: rname='STITCH>SPATIAL'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! VVV There is a problem in using 8 points as the middle points
    ! may add a pixel at each side that should not be there.
    call cubemain_spatial_corners(prog%inlist%entries(1)%cube,corners,error)
    if(error) return
    do iax=1,2
       absmin(iax) = minval(corners(iax,:))
       absmax(iax) = maxval(corners(iax,:))
    enddo
    laxis%inc = tiny(1d0)
    maxis%inc = tiny(1d0)
    !
    do icub=1,prog%inlist%n
       
       call cubemain_spatial_corners(prog%inlist%entries(icub)%cube,corners,error)
       if (error) return
       call cubemain_spatial_reprojcoords(prog%inlist%entries(icub)%cube,corners,&
            prog%inlist%entries(1)%cube,four,projcorners,error)     
       if (error) return
       llinc = prog%inlist%entries(icub)%cube%head%spa%l%inc
       lminc = prog%inlist%entries(icub)%cube%head%spa%m%inc
       do iax=1,2
          absmin(iax) = min(absmin(iax),minval(projcorners(iax,:)))
          absmax(iax) = max(absmax(iax),maxval(projcorners(iax,:)))
       enddo
       if (abs(llinc).gt.abs(laxis%inc)) laxis%inc = llinc
       if (abs(lminc).gt.abs(maxis%inc)) maxis%inc = lminc
    enddo
    call cubemain_stitch_compute_spa_axis(absmin(1),absmax(1),laxis,error)
    if (error) return
    call cubemain_stitch_compute_spa_axis(absmin(2),absmax(2),maxis,error)
    if (error) return
    !
    laxis%kind = code_unit_fov
    laxis%unit = prog%stitch%head%spa%l%unit
    laxis%genuine = .true.
    maxis%kind = code_unit_fov
    maxis%unit = prog%stitch%head%spa%m%unit
    maxis%genuine = .true.
    select case(prog%stitch%head%spa%fra%code)
    case (code_spaframe_galactic)
       laxis%name = 'Lii'
       maxis%name = 'Bii'
    case (code_spaframe_equatorial,code_spaframe_icrs)
       laxis%name = 'RA'
       maxis%name = 'DEC'
    case default
       laxis%name = 'L'
       maxis%name = 'M'
    end select
  end subroutine cubemain_stitch_spatial
  !
  subroutine cubemain_stitch_compute_spa_axis(mini,maxi,axis,error)
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    real(kind=coor_k), intent(in)    :: mini
    real(kind=coor_k), intent(in)    :: maxi
    type(axis_t),      intent(inout) :: axis
    logical,           intent(inout) :: error
    !
    real(kind=coor_k) :: nrea, diff
    integer(kind=pixe_k) :: nnint
    real(kind=coor_k), parameter :: spatol = 0.1
    character(len=*), parameter :: rname='STITCH>COMPUTE>SPA>AXIS'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    nrea  = abs((maxi-mini)/axis%inc)
    nnint = nint(nrea)
    diff = abs((nnint-nrea)*abs(axis%inc))
    if (diff.le.abs(spatol*axis%inc)) then
       axis%n = nnint
    else
       axis%n = ceiling(nrea)
    endif
    if (axis%inc.lt.0) then
       axis%ref = -(maxi-0.5*axis%inc)/axis%inc
    else
       axis%ref = (-mini+0.5*axis%inc)/axis%inc
    endif
    axis%val = 0d0
  end subroutine cubemain_stitch_compute_spa_axis
  !
  subroutine cubemain_stitch_data(prog,error)
    use cubemain_stitch_v1
    use cubemain_stitch_v2
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(stitch_prog_t), intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='STITCH>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    if (prog%split) then
       call cubemain_stitch_v1_data(prog%inlist,prog%dorepro,prog%doresam,&
            prog%stitch,prog%weight,error)
    else
       call cubemain_stitch_v2_data(prog%inlist,prog%dorepro,prog%doresam,&
             prog%stitch,prog%weight,error)
    endif
    if (error) return
  end subroutine cubemain_stitch_data
end module cubemain_stitch
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
