!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubefit_minimize
  use cube_types
  use cubetools_structure
  use cubetools_keyword_arg
  use cubeadm_cubeid_types
  use cubefit_messaging
  use cubemain_windowing
  use cubemain_range
  use cubemain_auxiliary
  use cubefit_parameters
  use cubefit_hfs
  !
  public :: minimize
  public :: cubefit_minimize_command
  private
  !
  type :: minimize_comm_t
     type(option_t),      pointer :: comm
     type(range_opt_t)            :: range     
     type(option_t),      pointer :: initial
     type(option_t),      pointer :: line
     type(keyword_arg_t), pointer :: method_arg
     type(option_t),      pointer :: mask
     type(option_t),      pointer :: hfs
     type(option_t),      pointer :: window
   contains
     procedure, public  :: register     => cubefit_minimize_register
     procedure, private :: parse        => cubefit_minimize_parse
     procedure, private :: parse_line   => cubefit_minimize_parse_line
     procedure, private :: parse_window => cubefit_minimize_parse_window
     procedure, private :: main         => cubefit_minimize_main
  end type minimize_comm_t
  type(minimize_comm_t) :: minimize
  !
  integer(kind=4), parameter :: icube    = 1
  integer(kind=4), parameter :: iwindow  = 2
  type minimize_user_t
     type(cubeid_user_t)    :: cubeids
     type(range_array_t)    :: range
     type(auxiliary_user_t) :: initial
     type(auxiliary_user_t) :: mask
     logical                :: dowindow    = .false.
     logical                :: dorange     = .false.
     logical                :: doline      = .false.
     character(len=meth_l)  :: method
     integer(kind=line_k)   :: nline
     character(len=argu_l), allocatable :: lineinfo(:)
     type(hfs_user_t)       :: hfs
   contains
     procedure, private :: toprog => cubefit_minimize_user_toprog
  end type minimize_user_t
  type minimize_prog_t
     type(window_array_t)  :: wind
     type(hfs_prog_t)      :: hfs
     logical               :: doinitial
     logical               :: domask
     logical               :: dowindow
     integer(kind=code_k)  :: method
     integer(kind=line_k)  :: nline
     integer(kind=npar_k)  :: npar
     integer(kind=chan_k)  :: nout
     real(kind=para_k),    allocatable :: pars(:)
     integer(kind=flag_k), allocatable :: flag(:)
     type(cube_t), pointer :: incube
     type(cube_t), pointer :: oucube
     type(cube_t), pointer :: initial
     type(cube_t), pointer :: mask
     type(cube_t), pointer :: window
     procedure(cubefit_minimize_prog_loop_noguess), pointer  :: loop => null()
   contains
     procedure, private :: header      => cubefit_minimize_prog_header
     procedure, private :: data        => cubefit_minimize_prog_data
     procedure, private :: act_noguess => cubefit_minimize_prog_act_noguess     
     procedure, private :: act_window  => cubefit_minimize_prog_act_window     
     procedure, private :: act_initial => cubefit_minimize_prog_act_initial     
  end type minimize_prog_t
  !
contains
  !
  subroutine cubefit_minimize_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*),       intent(in)    :: line
    logical,                intent(inout) :: error
    !
    type(minimize_user_t) :: user
    character(len=*), parameter :: rname='MINIMIZE>COMMAND'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    call minimize%parse(line,user,error)
    if (error) return
    call minimize%main(user,error)
    if (error) return
  end subroutine cubefit_minimize_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubefit_minimize_register(minimize,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_comm_t), intent(inout) :: minimize
    logical,                intent(inout) :: error
    !
    type(cubeid_arg_t)   :: cubearg
    type(standard_arg_t) :: stdarg
    type(keyword_arg_t)  :: keyarg
    character(len=*), parameter :: comm_abstract = 'Fit functions to spectra'
    character(len=*), parameter :: comm_help = &
         strg_id
    character(len=*), parameter :: rname='MINIMIZE>REGISTER'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'MINIMIZE','[cube [window]]',&
         comm_abstract,&
         comm_help,&
         cubefit_minimize_command,&
         minimize%comm,error)
    if (error) return
    call cubearg%register( &
         'CUBE', &
         'Signal cube',  &
         strg_id,&
         code_arg_optional,  &
         [flag_cube], &
         error)
    if (error) return
    call cubearg%register( &
         'WINDOW', &
         'window cube',  &
         strg_id,&
         code_arg_optional,  &
         [flag_window], &
         error)
    if (error) return
    !
    call minimize%range%register(&
         'RANGE',&
         'Define velocity range(s) for first guess estimates',&
         range_is_multiple,error)
    if (error) return
    !
    call cubemain_auxiliary_register(&
         'INITIAL',&
         'Get initial guesses to a fit from a initial guess cube',&
         'The input CUBE can be the output of the command MINIMIZE or&
         &can be  created freely by the user',&
         'Initial guess cube',&
         [flag_minimize],&
         code_arg_optional,&
         minimize%initial,error)
    if (error) return
    !
    call cubemain_auxiliary_register(&
         'MASK',&
         'Use a 2D mask to define the regions to be fitted',&
         strg_id,&
         'Mask',&
         [flag_mask],&
         code_arg_optional,&
         minimize%mask,error)
    if (error) return
    !
    call cubetools_register_option(&
         'LINE','method [nlines f11 p11 ... f1m p1m ... fn1 pn1 ... fnm pnm]',&
         'Set fitting method and optionally its first guesses',&
         'gag_doc:hlp/cube-help-fit-minimize-line.hlp',&
         minimize%line,error)
    if (error) return
    call keyarg%register( &
         'method',  &
         'Method for fitting', &
         strg_id,&
         code_arg_mandatory, &
         spec_meth, &
         .not.flexible,&
         minimize%method_arg,&
         error)
    if (error) return
    call stdarg%register( &
         'nlines',  &
         'Number of lines to be fitted', &
         strg_id,&
         code_arg_mandatory, error)
    if (error) return
    call stdarg%register( &
         'f11',  &
         'Flag for the first parameter of the first line', &
         strg_id,&
         code_arg_unlimited, error)
    if (error) return
    call stdarg%register( &
         'p11',  &
         'First parameter of the first line', &
         strg_id,&
         code_arg_unlimited, error)
    if (error) return
    ! call stdarg%register( &
    !      'f1m',  &
    !      'Flag for the m-eth parameter of the first line', &
    !      strg_id,&
    !      code_arg_unlimited, error)
    ! if (error) return
    ! call stdarg%register( &
    !      'p1m',  &
    !      'm-eth parameter of the first line', &
    !      strg_id,&
    !      code_arg_unlimited, error)
    ! if (error) return
    ! call stdarg%register( &
    !      'fn1',  &
    !      'Flag for the first parameter of the n-eth line', &
    !      strg_id,&
    !      code_arg_unlimited, error)
    ! if (error) return
    ! call stdarg%register( &
    !      'pn1',  &
    !      'First parameter of the n-eth line', &
    !      strg_id,&
    !      code_arg_unlimited, error)
    ! if (error) return
    call stdarg%register( &
         'fnm',  &
         'Flag for the m-eth parameter of the n-eth line', &
         strg_id,&
         code_arg_unlimited, error)
    if (error) return
    call stdarg%register( &
         'pnm',  &
         'm-eth parameter of the n-eth line', &
         strg_id,&
         code_arg_unlimited, error)
    if (error) return
    !
    call cubefit_hfs_register(minimize%hfs,error)
    if (error) return
    !
    call cubetools_register_option(&
         'WINDOW','method',&
         'Estimate first guesses from a window CUBE',&
         'The window cube can be created with commad WINDOW. The&
         & number of lines to be fitted is assumed to be equal to the&
         & number of windows in the window CUBE',&
         minimize%window,error)
    if (error) return
    call keyarg%register( &
         'method',  &
         'Method for fitting', &
         strg_id,&
         code_arg_mandatory, &
         spec_meth, &
         .not.flexible,&
         minimize%method_arg,&
         error)
    if (error) return
  end subroutine cubefit_minimize_register
  !
  subroutine cubefit_minimize_parse(minimize,line,user,error)
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(minimize_comm_t), intent(in)    :: minimize
    character(len=*),       intent(in)    :: line
    type(minimize_user_t),  intent(out)   :: user
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='MINIMIZE>PARSE'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,minimize%comm,user%cubeids,error)
    if (error) return
    call minimize%range%parse(line,user%dorange,user%range,error)
    if (error) return
    call cubemain_auxiliary_parse(line,minimize%initial,user%initial,error)
    if (error) return
    call cubemain_auxiliary_parse(line,minimize%mask,user%mask,error)
    if (error) return
    call minimize%parse_line(line,user,error)
    if (error) return
    call cubefit_hfs_parse(line,minimize%hfs,user%hfs,error)
    if (error) return
    call minimize%parse_window(line,user,error)
    if (error) return
    !
    if (user%doline.and.user%initial%do) then
       call cubefit_message(seve%e,rname,"Options /LINES and /INITAL can not be given at the same time")
       error = .true.
       return
    else if (user%doline.and.user%dowindow) then
       call cubefit_message(seve%e,rname,"Options /LINES and /WINDOW can not be given at the same time")
       error = .true.
       return
    else if (user%dowindow.and.user%initial%do) then
       call cubefit_message(seve%e,rname,"Options /WINDOW and /INITAL can not be given at the same time")
       error = .true.
       return
    endif
  end subroutine cubefit_minimize_parse
  !
  subroutine cubefit_minimize_parse_line(minimize,line,user,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    ! /LINES method [nlines f11 p11 ... f1m p1m ... fn1 pn1 ... fnm pnm]
    !----------------------------------------------------------------------
    class(minimize_comm_t), intent(in)    :: minimize
    character(len=*),       intent(in)    :: line
    type(minimize_user_t),  intent(inout) :: user
    logical,                intent(inout) :: error
    !
    integer(kind=4) :: narg,iarg,ier
    character(len=*), parameter :: rname='MINIMIZE>PARSE>LINE'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    call minimize%line%present(line,user%doline,error)
    if (error) return
    if (user%doline) then
       call cubetools_getarg(line,minimize%line,1,user%method,mandatory,error)
       if (error) return
       narg = minimize%line%getnarg()
       if (narg.gt.1) then
          call cubetools_getarg(line,minimize%line,2,user%nline,mandatory,error)
          if (error) return
          allocate(user%lineinfo(narg-2),stat=ier)
          if (failed_allocate(rname,'user line guesses',ier,error)) return
          do iarg=3,narg
             call cubetools_getarg(line,minimize%line,iarg,user%lineinfo(iarg-2),mandatory,error)
             if (error) return
          enddo
       else
          user%nline = 0
       endif
    endif
  end subroutine cubefit_minimize_parse_line
  !
  subroutine cubefit_minimize_parse_window(minimize,line,user,error)
    !----------------------------------------------------------------------
    ! /FACTOR factor
    !----------------------------------------------------------------------
    class(minimize_comm_t), intent(in)    :: minimize
    character(len=*),       intent(in)    :: line
    type(minimize_user_t),  intent(inout) :: user
    logical,                intent(inout) :: error
    !
    character(len=*), parameter :: rname='MINIMIZE>PARSE>WINDOW'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    call minimize%window%present(line,user%dowindow,error)
    if (error) return
    if (user%dowindow) then
       call cubetools_getarg(line,minimize%window,1,user%method,mandatory,error)
       if (error) return
    endif
  end subroutine cubefit_minimize_parse_window
  !
  subroutine cubefit_minimize_main(minimize,user,error)    
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_comm_t), intent(in)    :: minimize
    type(minimize_user_t),  intent(in)    :: user
    logical,                intent(inout) :: error
    !
    type(minimize_prog_t) :: prog
    character(len=*), parameter :: rname='MINIMIZE>MAIN'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubefit_minimize_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubefit_minimize_user_toprog(user,prog,error)
    use gkernel_interfaces
    use cubetools_user2prog
    use cubetools_unit
    use cubetools_header_methods
    use cubetools_consistency_methods
    use cubeadm_get
    use cubefit_spectral_fit
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_user_t), intent(in)    :: user
    type(minimize_prog_t),  intent(out)   :: prog
    logical,                intent(inout) :: error
    !
    type(unit_user_t) :: nounit
    integer(kind=npar_k) :: ipar
    integer(kind=4) :: ier
    character(len=mess_l) :: mess
    character(len=meth_l) :: methodstr
    integer(kind=chan_k) :: masknc,nwindow
    logical :: prob
    character(len=*), parameter :: rname='MINIMIZE>USER>TOPROG'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    prob =.false.
    call cubeadm_cubeid_get_header(minimize%comm,icube,user%cubeids,&
         code_access_speset,code_read,prog%incube,error)
    if (error) return
    !
    call minimize%range%user2prog(prog%incube,user%range,prog%wind,error)
    if (error) return
    !
    prog%domask = user%mask%do
    if (prog%domask) then
       call cubemain_auxiliary_user2prog(minimize%mask,code_access_speset,  &
            user%mask,prog%mask,error)
       if (error) return
       call cubetools_header_get_nchan(prog%mask%head,masknc,error)
       if (error) return
       if (masknc.ne.1) then
          call cubefit_message(seve%e,rname,'Only 2D masks supported')
          error = .true.
          return
       endif
       call cubetools_consistency_spatial('Input cube',prog%incube%head,'Mask',prog%mask%head,prob,error)
       if (error) return
       if (cubetools_consistency_failed(rname,prob,error)) return
    endif
    !
    prog%dowindow = user%dowindow
    prog%doinitial = user%initial%do
    if (prog%doinitial) then
       call cubemain_auxiliary_user2prog(minimize%initial,code_access_speset,  &
            user%initial,prog%initial,error)
       if (error) return
       call cubetools_consistency_spatial('Input cube',prog%incube%head,'Initial guesses',prog%initial%head,&
            prob,error)
       if (error) return
       if (cubetools_consistency_failed(rname,prob,error)) return
       call cubetools_header_get_nchan(prog%initial%head,prog%nout,error)
       if (error) return
       prog%method = code_abs
       prog%loop => cubefit_minimize_prog_loop_initial
    else
       if (prog%dowindow) then
          call cubeadm_cubeid_get_header(minimize%comm,iwindow,user%cubeids,&
               code_access_speset,code_read,prog%window,error)
          if (error) return
          !
          call cubetools_consistency_spatial('Input cube',prog%incube%head,'Window',prog%window%head,prob,error)
          if (error) return
          if (cubetools_consistency_failed(rname,prob,error)) return
          call cubetools_keyword_user2prog(minimize%method_arg,user%method,prog%method,methodstr,error)
          if (error) return
          call cubetools_header_get_nchan(prog%window%head,nwindow,error)
          if (error) return
          prog%nline = nwindow/2
          prog%loop => cubefit_minimize_prog_loop_window
       else if (user%doline) then
          call cubetools_keyword_user2prog(minimize%method_arg,user%method,prog%method,methodstr,error)
          if (error) return
          prog%nline = user%nline
          prog%loop => cubefit_minimize_prog_loop_noguess
       else
          prog%method = igaussian
          prog%nline  = 0
          prog%loop => cubefit_minimize_prog_loop_noguess
       endif
       !
       call cubefit_parameters_npars_nout(prog%method,prog%nline,prog%npar,prog%nout,error)
       if (error) return
       if (user%doline.and.prog%nline.gt.0) then
          if (size(user%lineinfo).ne.2*prog%npar) then
             write(mess,'(2(a,i0))') "Expected ",2*prog%npar," flags + parameters, got ",size(user%lineinfo)
             call cubefit_message(seve%e,rname,mess)
             error = .true.
             return
          endif
          allocate(prog%pars(prog%npar),prog%flag(prog%npar),stat=ier)
          if (failed_allocate(rname,'initial guesses',ier,error)) return
          prog%flag(:) = 0
          prog%pars(:) = 0.
          call cubetools_unit_get(strg_star,code_unit_unk,nounit,error)
          if (error) return
          do ipar=1,prog%npar
             call cubetools_user2prog_resolve_star(user%lineinfo(2*ipar-1),0,prog%flag(ipar),error)
             if (error) return
             call cubetools_user2prog_resolve_star(user%lineinfo(2*ipar),nounit,0d0,prog%pars(ipar),error)
             if (error) return
          enddo
       endif
    endif
    !
    if (user%hfs%do) then
       call cubefit_hfs_user2prog(user%hfs,prog%hfs,error)
       if (error) return
    else 
       if (prog%method.eq.ihfs.or.prog%method.eq.iabsorption) then
          call cubefit_message(seve%e,rname,'Methods HFS and ABSORPTION require a HFS file')
          error = .true.
          return
       endif
    endif
  end subroutine cubefit_minimize_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubefit_minimize_prog_header(prog,error)
    use cubetools_header_methods
    use cubetools_axis_types
    use cubedag_allflags
    use cubeadm_clone
    use cubetools_unit
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_prog_t), intent(inout) :: prog
    logical,                intent(inout) :: error
    !
    type(axis_t) :: axis
    character(len=*), parameter :: rname='MINIMIZE>PROG>HEADER'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(prog%incube,[flag_fit,flag_minimize],prog%oucube,error)
    if (error) return
    !
    call cubetools_header_get_axis_head_c(prog%oucube%head,axis,error)
    if (error) return
    axis%name = 'Spectral fit'
    axis%unit = 'NONE'
    axis%kind = code_unit_unk
    axis%ref = 1d0
    axis%val = 1d0
    axis%inc = 1d0
    axis%n   = prog%nout
    call cubetools_header_update_axset_c(axis,prog%oucube%head,error)
    if (error) return
    !
  end subroutine cubefit_minimize_prog_header
  !
  subroutine cubefit_minimize_prog_data(prog,error)
    use gkernel_interfaces
    use cubeadm_opened
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(minimize_prog_t), intent(inout) :: prog
    logical,                intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='MINIMIZE>PROG>DATA'
    !
    call cubefit_message(fitseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !
    ! Kernel fitting routines raise a number of warnings when fit
    ! diverges, we don't want to see this messages polluting the
    ! terminal hence they are hidden here
    call exec_program('sic message cube s-w')
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error)  &
         call prog%loop(iter%first,iter%last,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
    !
    ! Reactivate cube warning messages after fits
    call exec_program('sic message cube s+w')
  end subroutine cubefit_minimize_prog_data
  !
  subroutine cubefit_minimize_prog_loop_noguess(prog,first,last,error)
    use cubemain_spectrum_real
    use cubeadm_entryloop
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_prog_t), intent(inout) :: prog
    integer(kind=entr_k),   intent(in)    :: first
    integer(kind=entr_k),   intent(in)    :: last
    logical,                intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(spectrum_t) :: inspec,ouspec,mask
    character(len=*), parameter :: rname='MINIMIZE>PROG>LOOP>NOGUESS'
    !
    call inspec%reassociate_and_init(prog%incube,error)
    if (error) return
    if (prog%domask) then
       call mask%reassociate_and_init(prog%mask,error)
       if (error) return
    end if
    call ouspec%reallocate('fit',prog%oucube%head%arr%n%c,error)
    if (error) return
    !
    do ie=first,last
       call cubeadm_entryloop_iterate(ie,error)
       if (error)  return
       call prog%act_noguess(ie,inspec,mask,ouspec,error)
       if (error)  return
    enddo
  end subroutine cubefit_minimize_prog_loop_noguess
  !
  subroutine cubefit_minimize_prog_act_noguess(prog,ie,inspec,mask,ouspec,error)
    use cubetools_nan
    use cubemain_spectrum_real
    use cubemain_spectrum_blanking
    use cubefit_spectral_fit
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_prog_t), intent(inout) :: prog
    integer(kind=entr_k),   intent(in)    :: ie
    type(spectrum_t),       intent(inout) :: inspec
    type(spectrum_t),       intent(inout) :: mask
    type(spectrum_t),       intent(inout) :: ouspec
    logical,                intent(inout) :: error
    !
    type(fit_spectral_t) :: fit
    logical :: dofit
    character(len=*), parameter :: rname='MINIMIZE>PROG>ACT>NOGUESS'
    !
    dofit = .true.
    if (prog%domask) then
       call mask%get(prog%mask,ie,error)
       if (error)  return
       dofit = mask%t(1).gt.0
    endif
    !
    if (dofit) then
       call inspec%get(prog%incube,ie,error)
       if (error)  return
       if (cubemain_spectrum_blank(inspec)) then
          ouspec%t(:) = gr4nan
       else
          call fit%user_import(prog%hfs,prog%npar,prog%method,prog%nline,prog%flag,prog%pars,error)
          if (error) return
          call fit%fit(inspec,prog%wind,error)
          if (error) return
          call fit%export(ouspec,error)
          if (error) return
       endif
    else
       ouspec%t(:) = gr4nan
    endif
    call ouspec%put(prog%oucube,ie,error)
    if (error)  return
  end subroutine cubefit_minimize_prog_act_noguess
  !
  subroutine cubefit_minimize_prog_loop_window(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_prog_t), intent(inout) :: prog
    integer(kind=entr_k),   intent(in)    :: first
    integer(kind=entr_k),   intent(in)    :: last
    logical,                intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    character(len=mess_l) :: mess
    type(spectrum_t) :: inspec,window,ouspec,mask
    character(len=*), parameter :: rname='MINIMIZE>PROG>LOOP>WINDOW'
    !
    call inspec%reassociate_and_init(prog%incube,error)
    if (error) return
    if (prog%domask) then
       call mask%reassociate_and_init(prog%mask,error)
       if (error) return
    end if
    call window%reassociate_and_init(prog%window,error)
    if (error) return
    call ouspec%reallocate('fit',prog%oucube%head%arr%n%c,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error)  return
      write(mess,'(a,i0)') 'Processing spectrum #',ie
      call cubefit_message(fitseve%others,rname,mess)
      call prog%act_window(ie,inspec,window,mask,ouspec,error)
      if (error)  return
    enddo
  end subroutine cubefit_minimize_prog_loop_window
  !   
  subroutine cubefit_minimize_prog_act_window(prog,ie,inspec,window,mask,ouspec,error)
    use gkernel_interfaces
    use cubetools_nan
    use cubemain_spectrum_real
    use cubemain_spectrum_blanking
    use cubemain_topology
    use cubefit_spectral_fit
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_prog_t), intent(inout) :: prog
    integer(kind=entr_k),   intent(in)    :: ie
    type(spectrum_t),       intent(inout) :: inspec
    type(spectrum_t),       intent(inout) :: window
    type(spectrum_t),       intent(inout) :: mask
    type(spectrum_t),       intent(inout) :: ouspec
    logical,                intent(inout) :: error
    !
    type(fit_spectral_t) :: fit
    logical :: dofit
    integer(kind=chan_k) :: iw
    integer(kind=4) :: ier
    integer(kind=chan_k),allocatable :: chanwin(:)
    real(kind=coor_k),allocatable :: r8win(:)
    character(len=*), parameter :: rname='MINIMIZE>PROG>ACT>WINDOW'
    !
    dofit = .true.
    if (prog%domask) then
       call mask%get(prog%mask,ie,error)
       if (error)  return
       dofit = mask%t(1).gt.0
    endif
    if (prog%dowindow.and.dofit) then
       call window%get(prog%window,ie,error)
       if (error)  return
       dofit = .not.ieee_is_nan(window%t(1))
    endif
    !
    if (dofit) then
       call inspec%get(prog%incube,ie,error)
       if (error)  return
       if (cubemain_spectrum_blank(inspec)) then
          ouspec%t(:) = gr4nan
       else
          allocate(r8win(window%n),chanwin(window%n),stat=ier)
          if (failed_allocate(rname,'Double window',ier,error)) return
          r8win(:) = window%t(:)
          do iw=1,window%n/2
             call cubemain_topo_vrange2crange(prog%incube,r8win(2*iw-1:2*iw),chanwin(2*iw-1:2*iw),error)
             if (error) return
          enddo
          call fit%wind_import(prog%hfs,prog%method,chanwin,error)
          if (error) return
          call fit%fit(inspec,prog%wind,error)
          if (error) return
          call fit%export(ouspec,error)
          if (error) return
       endif
    else
       ouspec%t(:) = gr4nan
    endif
    call ouspec%put(prog%oucube,ie,error)
    if (error)  return
  end subroutine cubefit_minimize_prog_act_window
  !
  subroutine cubefit_minimize_prog_loop_initial(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_prog_t), intent(inout) :: prog
    integer(kind=entr_k),   intent(in)    :: first
    integer(kind=entr_k),   intent(in)    :: last
    logical,                intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    character(len=mess_l) :: mess
    type(spectrum_t) :: inspec,initial,ouspec,mask
    character(len=*), parameter :: rname='MINIMIZE>PROG>LOOP>INITIAL'
    !
    call inspec%reassociate_and_init(prog%incube,error)
    if (error) return
    call initial%reassociate_and_init(prog%initial,error)
    if (error) return
    if (prog%domask) then
       call mask%reassociate_and_init(prog%mask,error)
       if (error) return
    end if
    call ouspec%reallocate('fit',prog%oucube%head%arr%n%c,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error)  return
      write(mess,'(a,i0)') 'Processing spectrum #',ie
      call cubefit_message(fitseve%others,rname,mess)
      call prog%act_initial(ie,inspec,initial,mask,ouspec,error)
      if (error)  return
    enddo
  end subroutine cubefit_minimize_prog_loop_initial
  !   
  subroutine cubefit_minimize_prog_act_initial(prog,ie,inspec,initial,mask,ouspec,error)
    use cubetools_nan
    use cubemain_spectrum_real
    use cubemain_spectrum_blanking
    use cubemain_topology
    use cubefit_spectral_fit
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(minimize_prog_t), intent(inout) :: prog
    integer(kind=entr_k),   intent(in)    :: ie
    type(spectrum_t),       intent(inout) :: inspec
    type(spectrum_t),       intent(inout) :: initial
    type(spectrum_t),       intent(inout) :: mask
    type(spectrum_t),       intent(inout) :: ouspec
    logical,                intent(inout) :: error
    !
    type(fit_spectral_t) :: fit
    logical :: dofit
    character(len=*), parameter :: rname='MINIMIZE>PROG>ACT>INITIAL'
    !
    dofit = .true.
    if (prog%domask) then
       call mask%get(prog%mask,ie,error)
       if (error)  return
       dofit = mask%t(1).gt.0
    endif
    if (prog%doinitial.and.dofit) then
       call initial%get(prog%initial,ie,error)
       if (error)  return
       dofit = .not.ieee_is_nan(initial%t(1))
    endif
    !
    if (dofit) then
       call inspec%get(prog%incube,ie,error)
       if (error)  return
       if (cubemain_spectrum_blank(inspec)) then
          ouspec%t(:) = gr4nan
       else
          call fit%spec_import(prog%hfs,initial,error)
          if (error)  return
          call fit%fit(inspec,prog%wind,error)
          if (error) return
          call fit%export(ouspec,error)
          if (error) return
       endif
    else
       ouspec%t(:) = gr4nan
    endif
    call ouspec%put(prog%oucube,ie,error)
    if (error)  return
  end subroutine cubefit_minimize_prog_act_initial
  !
endmodule cubefit_minimize
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
