!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_snr
  use cube_types
  use cubetools_structure
  use cubemain_messaging
  use cubeadm_cubeid_types
  !
  public :: snr
  public :: cubemain_snr_command
  private
  !
  type :: snr_comm_t
     type(option_t), pointer :: comm
   contains
     procedure, public  :: register => cubemain_snr_register
     procedure, private :: parse    => cubemain_snr_parse
     procedure, private :: main     => cubemain_snr_main
  end type snr_comm_t
  type(snr_comm_t) :: snr
  !
  integer(kind=4), parameter :: isig = 1
  integer(kind=4), parameter :: inoi = 2
  type snr_user_t
     type(cubeid_user_t) :: cubeids
   contains
     procedure, private :: toprog => cubemain_snr_user_toprog
  end type snr_user_t
  type snr_prog_t
     type(cube_t), pointer :: sig ! Input cube
     type(cube_t), pointer :: noi ! Noise reference
     type(cube_t), pointer :: snr ! Output cube
     procedure(cubemain_snr_prog_singlenoise_loop), pointer :: loop => null()
   contains
     procedure, private :: header      => cubemain_snr_prog_header
     procedure, private :: data        => cubemain_snr_prog_data
     procedure, private :: singlenoise => cubemain_snr_prog_singlenoise
     procedure, private :: multinoise  => cubemain_snr_prog_multinoise
  end type snr_prog_t
  !
contains
  !
  subroutine cubemain_snr_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(snr_user_t) :: user
    character(len=*), parameter :: rname='SNR>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call snr%parse(line,user,error)
    if (error) return
    call snr%main(user,error)
    if (error) continue
  end subroutine cubemain_snr_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_snr_register(snr,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(snr_comm_t), intent(inout) :: snr
    logical,           intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    character(len=*), parameter :: comm_abstract = &
         'Compute the signal to noise ratio'
    character(len=*), parameter :: comm_help = &
         strg_id
    character(len=*), parameter :: rname='SNR>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'SNR','[signal [noise]]',&
         comm_abstract,&
         comm_help,&
         cubemain_snr_command,&
         snr%comm,error)
    if (error) return
    call cubearg%register( &
         'SIGNAL', &
         'Signal cube',  &
         strg_id,&
         code_arg_optional,  &
         [flag_cube], &
         error)
    if (error) return
    call cubearg%register( &
         'NOISE', &
         'Noise cube', &
         strg_id,&
         code_arg_optional, &
         [flag_noise], &
         error)
    if (error) return
  end subroutine cubemain_snr_register
  !
  subroutine cubemain_snr_parse(snr,line,user,error)
    use cubetools_parse
    !----------------------------------------------------------------------
    ! SNR cubename
    !----------------------------------------------------------------------
    class(snr_comm_t), intent(in)    :: snr
    character(len=*),  intent(in)    :: line
    type(snr_user_t),  intent(out)   :: user
    logical,           intent(inout) :: error
    !
    character(len=*), parameter :: rname='SNR>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,snr%comm,user%cubeids,error)
    if (error) return
  end subroutine cubemain_snr_parse
  !
  subroutine cubemain_snr_main(snr,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(snr_comm_t), intent(in)    :: snr
    type(snr_user_t),  intent(in)    :: user
    logical,           intent(inout) :: error
    !
    type(snr_prog_t) :: prog
    character(len=*), parameter :: rname='SNR>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_snr_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_snr_user_toprog(user,prog,error)
    use cubetools_consistency_methods
    use cubetools_header_methods
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(snr_user_t), intent(in)    :: user
    type(snr_prog_t),  intent(out)   :: prog
    logical,           intent(inout) :: error
    !
    logical :: prob
    integer(kind=chan_k) :: nnoi
    character(len=*), parameter :: rname='SNR>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !    
    call cubeadm_cubeid_get_header(snr%comm,isig,user%cubeids,code_access_speset,code_read,prog%sig,error)
    if (error)  return
    call cubeadm_cubeid_get_header(snr%comm,inoi,user%cubeids,code_access_speset,code_read,prog%noi,error)
    if (error) return
    !
    prob = .false.
    call cubetools_consistency_signal_noise('Input cube',prog%sig%head,'Noise',prog%noi%head,prob,error)
    if(error) return
    if (cubetools_consistency_failed(rname,prob,error)) return
    !
    call cubetools_header_get_nchan(prog%noi%head,nnoi,error)
    if (error) return
    if (nnoi.eq.1) then
       prog%loop => cubemain_snr_prog_singlenoise_loop
    else
       prog%loop => cubemain_snr_prog_multinoise_loop
    endif
  end subroutine cubemain_snr_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_snr_prog_header(prog,error)
    use cubetools_header_methods
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(snr_prog_t), intent(inout) :: prog
    logical,           intent(inout) :: error
    !
    character(len=*), parameter :: rname='SNR>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(prog%sig,flag_snr,prog%snr,error)
    if (error) return
    call cubetools_header_put_array_unit('---',prog%snr%head,error)
    if (error) return
    !
  end subroutine cubemain_snr_prog_header
  !
  subroutine cubemain_snr_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(snr_prog_t), intent(inout) :: prog
    logical,           intent(inout) :: error
    !
    character(len=*), parameter :: rname='SNR>PROG>DATA'
    type(cubeadm_iterator_t) :: iter
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error) then
          call prog%loop(iter%first,iter%last,error)
       endif
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_snr_prog_data
  !
  subroutine cubemain_snr_prog_singlenoise_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(snr_prog_t),     intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='SNR>PROG>SINGLENOISE>LOOP'
    !
    type(spectrum_t) :: sig,rawnoi,snr
    integer(kind=entr_k) :: ie
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    ! 
    call sig%reassociate_and_init(prog%sig,error)
    if (error) return
    call rawnoi%reassociate_and_init(prog%noi,error)
    if (error) return
    call snr%reallocate('snr',prog%sig%head%arr%n%c,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error)  return
      call prog%singlenoise(ie,sig,rawnoi,snr,error)
      if (error)  return
    enddo
  end subroutine cubemain_snr_prog_singlenoise_loop
  !
  subroutine cubemain_snr_prog_singlenoise(prog,ie,sig,rawnoi,snr,error)
    use cubetools_nan
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(snr_prog_t),     intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(spectrum_t),      intent(inout) :: sig
    type(spectrum_t),      intent(inout) :: rawnoi
    type(spectrum_t),      intent(inout) :: snr
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='SNR>PROG>SINGLENOISE'
    !
    integer(kind=chan_k) :: ic,nc
    real(kind=sign_k), pointer :: noise
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    nc = prog%sig%head%arr%n%c
    !
    call sig%get(prog%sig,ie,error)
    if (error) return
    call rawnoi%get(prog%noi,ie,error)
    if (error) return
    noise => rawnoi%t(1)
    if ((noise.gt.0).and.(.not.ieee_is_nan(noise))) then
       do ic=1,nc
          if (.not.ieee_is_nan(sig%t(ic))) then
             snr%t(ic) = sig%t(ic)/noise
          endif
       enddo ! ic
    else
       do ic=1,nc
          snr%t(ic) = gr4nan
       enddo ! ic             
    endif
    call snr%put(prog%snr,ie,error)
    if (error) return
  end subroutine cubemain_snr_prog_singlenoise
  !
  subroutine cubemain_snr_prog_multinoise_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_interpolate
    use cubemain_spectrum_real
    use cubemain_spectrum_interpolate
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(snr_prog_t),     intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(spectrum_t) :: sig,rawnoi,snr,intnoi
    type(interpolate_t) :: interp
    character(len=*), parameter :: rname='SNR>PROG>MULTINOISE>LOOP'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    ! 
    call sig%reassociate_and_init(prog%sig,error)
    if (error) return
    call intnoi%reassociate_and_init(prog%sig,error)
    if (error) return
    call rawnoi%reassociate_and_init(prog%noi,error)
    if (error) return
    call snr%reallocate('snr',prog%sig%head%arr%n%c,error)
    if (error) return
    call cubemain_spectrum_interpolate_init(rawnoi,intnoi,interp,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error)  return
      call prog%multinoise(ie,sig,rawnoi,intnoi,interp,snr,error)
      if (error)  return
    enddo
  end subroutine cubemain_snr_prog_multinoise_loop
  !
  subroutine cubemain_snr_prog_multinoise(prog,ie,sig,rawnoi,intnoi,interp,snr,error)
    use cubetools_nan
    use cubemain_interpolate
    use cubemain_spectrum_real
    use cubemain_spectrum_blanking
    use cubemain_spectrum_interpolate
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(snr_prog_t),     intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(spectrum_t),      intent(inout) :: sig
    type(spectrum_t),      intent(inout) :: rawnoi
    type(spectrum_t),      intent(inout) :: intnoi
    type(interpolate_t),   intent(inout) :: interp
    type(spectrum_t),      intent(inout) :: snr
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='SNR>PROG>MULTINOISE'
    !
    real(kind=sign_k), pointer :: noise
    integer(kind=chan_k) :: ic,nc
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    nc = prog%sig%head%arr%n%c
    !
    call sig%get(prog%sig,ie,error)
    if (error) return
    call rawnoi%get(prog%noi,ie,error)
    if (error) return
    !          if (cube_spectrum_blank(rawnoi)) cycle
    if (interp%equal) then
       intnoi%t = rawnoi%t
    else
       call cubemain_spectrum_interpolate_compute(interp,rawnoi,intnoi,error)
       if (error) return
    endif
    do ic=1,nc
       noise => intnoi%t(ic) 
       if ((noise.gt.0).and.(.not.ieee_is_nan(noise))) then
          if (.not.ieee_is_nan(sig%t(ic))) then
             ! *** JP: The fact that only positive or both
             ! *** JP: positive and negative values should be considered
             ! *** JP: signal should be a user-domain decision
             snr%t(ic) = sig%t(ic)/noise
          endif
       endif
    enddo ! ic
    call snr%put(prog%snr,ie,error)
    if (error) return
  end subroutine cubemain_snr_prog_multinoise
end module cubemain_snr
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
