program uv_mosaic
  use gildas_def
  use gkernel_interfaces
  !----------------------------------------------------------------------
  !   Make a mosaic UV Tables from a set of single field UV tables.
  !
  ! The original UV Tables are assumed to have the same number of channels and
  ! 1) either to have the same phase tracking center
  ! 2) or a phase tracking center corresponding to the pointing center
  !
  ! It is assumed that the names of the original field UV tables
  ! the simple sequence 'NAME'-'I'.uvt
  !----------------------------------------------------------------------
  ! Local
  ! character(len=*), parameter :: version = 'Version 1.0 06-Feb-2015'
  character(len=*), parameter :: version = 'Version 1.1 19-Sep-2016'
  character(len=*), parameter :: pname = 'UV_MOSAIC'
  integer :: nf
  real :: bsize,bmin
  character(len=80) :: generic
  logical :: error
  !
  call gagout('I-'//pname//', '//version)
  !
  call gildas_open
  call gildas_inte('FIELDS$',nf,1)
  call gildas_char('NAME$',generic)
  call gildas_close
  !
  ! Do all the work in a subroutine...
  if (nf.ne.0) then
    call sub_gather_mosaic (trim(generic),nf,error)
  else
    call sub_explode_mosaic (trim(generic),nf,error)
  endif
  if (error) call sysexi(fatale)
  call gagout('I-'//pname//',  Successful completion')
  !
end program uv_mosaic
!
subroutine sub_gather_mosaic(generic,nf,error)
  use gkernel_interfaces
  use gkernel_types
  use image_def
  use gbl_message
  !
  ! Build a Mosaic from an ensemble of NF initial fields
  ! with a name following the convention name-'i' i=1,nf
  !
  character(len=*), intent(in) :: generic  ! Generic name
  integer, intent(in) :: nf                ! Number of field
  logical, intent(out) :: error
  !
  character(len=*), parameter :: rname = 'UV_MOSAIC'
  type(gildas) :: field
  type(gildas) :: mosaic
  !
  real, allocatable :: dmos  (:,:)      ! UV data
  real, allocatable :: dfield(:,:)      ! Temporary work space
  !
  character(len=filename_length) :: name
  integer :: nn,ier, nvisi, mvisi
  integer :: if,iv, i_xoff, i_yoff, luv
  real(kind=8) :: doffx,doffy
  type(projection_t) :: proj
  !
  integer :: code, code_x, code_y
  integer, parameter :: code_short_void=0    ! Old method: one file per field
  integer, parameter :: code_short_phase=1  ! Mosaic UV table with phase offsets
  integer, parameter :: code_short_point=2  ! Mosaic UV table with pointing offsets
  !
  real(8), parameter :: pi=3.14159265358979323846d0
  real(4), parameter :: point_accuracy=0.1*pi/180/3600 ! Pointing accuracy
  real(4), parameter :: phase_accuracy=0.001*pi/180/3600 ! Phase accuracy
  !
  call gildas_null(field,type='UVT')
  call gildas_null(mosaic,type='UVT')
  !
  if (nf.eq.0) then
    call gag_message(seve%e,rname,'SPLIT Not supported here')
    error = .true.
    return
  endif
  error = .false.
  nn = len_trim(generic)
  !
  name = generic
  if = 1
  call append_number(name,if)
  call sic_parsef (name,field%file,' ','.uvt')
  call gdf_read_header(field,error)
  if (error) then
    call gag_message(seve%e,rname,'  Cannot open '//trim(field%file))
    return
  endif
  call gdf_close_image(field,error)
  if (error) return
  !
  ! First pass: Check consistency
  call gdf_copy_header(field,mosaic,error)
  !
  code = code_short_void
  if (abs(field%gil%ra-field%gil%a0).gt.point_accuracy .or. &
      & abs(field%gil%dec-field%gil%d0).gt.point_accuracy) then
    ! Phase & Pointing center differ
    code = code_short_point  ! Must have common phase center
      Print *,'#0 RA ',field%gil%ra, field%gil%a0
      Print *,'#0 Dec ',field%gil%dec,field%gil%d0
  endif
  !
  ! Loop over individual fields
  nvisi = 0
  mvisi = 0
  do if = 1,nf
    name = generic
    call append_number(name,if)
    call sic_parsef (name,field%file,' ','.uvt')
    call gdf_read_header (field,error)
    if (error) then
      call gag_message(seve%e,rname,'  Cannot open '//trim(field%file))
      return
    endif
    call gdf_close_image(field,error)
    if (error) return
    if (field%gil%dim(1).ne.mosaic%gil%dim(1) .or. &
      & field%gil%nchan.ne.mosaic%gil%nchan) then
      call gag_message(seve%e,rname,'  Inconsistent channel number at '//trim(field%file))
      print *,'Dim ',field%gil%dim(1),mosaic%gil%dim(1)
      print *,'Dim ',field%gil%nchan,mosaic%gil%nchan
      error = .true.
    endif
    if (field%gil%ref(1).ne.mosaic%gil%ref(1) .or. &
      & field%gil%vres.ne.mosaic%gil%vres  .or. &
      & field%gil%fres.ne.mosaic%gil%fres  .or. &
      & field%gil%freq.ne.mosaic%gil%freq  .or. &
      & field%gil%voff.ne.mosaic%gil%voff) then
      call gag_message(seve%e,rname,'  Inconsistent frequency axis at '//trim(field%file))
      error = .true.
    endif
    !
    if (code.eq.code_short_point) then
      if (abs(mosaic%gil%a0-field%gil%a0).gt.phase_accuracy .or. &
        & abs(mosaic%gil%d0-field%gil%d0).gt.phase_accuracy) then
        call gag_message(seve%e,rname,' #1 Inconsistent phase center at '//trim(field%file))
        error = .true.
      endif
    else if (code.eq.code_short_phase) then
      if (field%gil%ra.ne.field%gil%a0 .or. field%gil%dec.ne.field%gil%d0) then
        call gag_message(seve%e,rname,'  Phase & Pointing center mismatch at '//trim(field%file))
        error = .true.
      endif
    else if (abs(field%gil%ra-field%gil%a0).gt.point_accuracy .or. &
      & abs(field%gil%dec-field%gil%d0).gt.point_accuracy) then
      ! Previously undefined, but phase center differ from pointing center
      ! - must have common phase center
      code = code_short_point
      if (abs(mosaic%gil%a0-field%gil%a0).gt.phase_accuracy .or. &
        & abs(mosaic%gil%d0-field%gil%d0).gt.phase_accuracy) then
        Print *,'A0 ',mosaic%gil%a0, field%gil%a0, 180*3600/pi*abs(mosaic%gil%a0-field%gil%a0)
        Print *,'D0 ',mosaic%gil%d0, field%gil%d0, 180*3600/pi*abs(mosaic%gil%d0-field%gil%d0)
        call gag_message(seve%e,rname,' #2 Inconsistent phase center at '//trim(field%file))
        error = .true.
      endif
    endif
    if (error) return
    !
    nvisi = nvisi + field%gil%nvisi
    mvisi = max(mvisi,field%gil%dim(2))
  enddo
  !
  ! Now allocate a suitable buffer and prepare the output UV table
  mosaic%gil%nvisi = nvisi
  mosaic%gil%dim(2) = nvisi
  !
  ! There may be problems if a column of the other type is already
  ! present in the initial UV table...
  if (code.eq.code_short_point) then
    code_x = code_uvt_xoff
    code_y = code_uvt_yoff
    if ( (mosaic%gil%column_pointer(code_uvt_loff).ne.0) .or. &
      & (mosaic%gil%column_pointer(code_uvt_moff).ne.0) ) then
      error = .true.
    endif
  else
    code_x = code_uvt_loff
    code_y = code_uvt_moff
    if ( (mosaic%gil%column_pointer(code_uvt_xoff).ne.0) .or. &
      & (mosaic%gil%column_pointer(code_uvt_yoff).ne.0) ) then
      error = .true.
    endif
  endif
  if (error) then
    call gag_message(seve%e,rname,'Combination of PHASE_OFF and POINT_OFF not allowed')
    return
  endif
  !
  ! Re-use of Add the extra columns after lcol
  if (mosaic%gil%column_pointer(code_x).eq.0) then
    mosaic%gil%dim(1) = mosaic%gil%dim(1)+1
    mosaic%gil%column_pointer(code_x) = mosaic%gil%dim(1)
    mosaic%gil%column_size(code_x) = 1
  endif
  if (mosaic%gil%column_pointer(code_y).eq.0) then
    mosaic%gil%dim(1) = mosaic%gil%dim(1)+1
    mosaic%gil%column_pointer(code_y) = mosaic%gil%dim(1)
    mosaic%gil%column_size(code_y) = 1
  endif
  i_xoff = mosaic%gil%column_pointer(code_x)
  i_yoff = mosaic%gil%column_pointer(code_y)
  !
  allocate(dmos(mosaic%gil%dim(1),mvisi), dfield(field%gil%dim(1),mvisi), stat=ier)
  if (ier.ne.0) then
    call gag_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  call gwcs_projec(mosaic%gil%a0,mosaic%gil%d0,mosaic%gil%pang,mosaic%gil%ptyp,proj,error)
  !
  if (code.eq.code_short_point) then
    mosaic%gil%ra = mosaic%gil%a0
    mosaic%gil%dec = mosaic%gil%d0
  else if (mosaic%gil%ra.ne.mosaic%gil%a0 .or. mosaic%gil%dec.ne.mosaic%gil%d0) then
    call gag_message(seve%e,rname,'Mismatch Pointing and Phase centers in PHASE offset mode')
    error = .true.
    return
  endif
  !
  ! Now create the output UV table
  !
  name = generic
  call sic_parsef (name,mosaic%file,' ','.uvt')
  call gdf_create_image(mosaic,error)
  if (error) return
  !
  ! And fill it
  field%blc = 0
  field%trc = 0
  mosaic%blc = 0
  mosaic%trc = 0
  do if=1,nf
    name = generic
    call append_number(name,if)
    call sic_parsef (name,field%file,' ','.uvt')
    call gdf_read_header (field,error)
    if (error) return
    call abs_to_rel(proj,field%gil%ra,field%gil%dec,doffx,doffy,1)
    call gdf_read_data (field,dfield,error)
    if (error) return
    call gdf_close_image(field,error)
    if (error) return
    luv = field%gil%dim(1)
    !
    do iv=1,field%gil%nvisi
      dmos(1:luv,iv) = dfield(1:luv,iv)
      dmos(i_xoff,iv) = doffx
      dmos(i_yoff,iv) = doffy
    enddo
    !
    mosaic%blc(2) = mosaic%trc(2)+1
    mosaic%trc(2) = mosaic%blc(2)+field%gil%nvisi-1
    !
    call gdf_write_data(mosaic,dmos,error)
    if (error) return
  enddo
  call gdf_close_image(mosaic,error)
  if (error) return
end subroutine sub_gather_mosaic
!
subroutine append_number(name,if)
  character(len=*), intent(inout) :: name
  integer, intent(in) :: if
  !
  character(len=12) :: rchain, lchain
  integer :: nn
  !
  nn = len_trim(name)
  !
  write(rchain,'(I12)') if
  lchain = adjustl(rchain)
  !
  name(nn+1:) = '-'//lchain
end subroutine append_number
!
subroutine sub_explode_mosaic(generic,nf,error)
  use gkernel_interfaces
  use gkernel_types
  use image_def
  use gbl_message
  !
  ! Explode a Mosaic to an ensemble of NF single fields
  ! with a name following the convention name-'i'
  !
  character(len=*), intent(in) :: generic  ! Generic name
  integer, intent(out) :: nf                ! Number of field
  logical, intent(out) :: error
  !
  character(len=*), parameter :: rname = 'UV_MOSAIC'
  real(8), parameter :: pi=3.14159265358979323846d0
  type(gildas) :: field
  type(gildas) :: mosaic
  !
  real, allocatable :: dmos  (:,:)      ! UV data
  real, allocatable :: dfield(:,:)      ! Temporary work space
  !
  character(len=filename_length) :: name
  integer :: nn,ier, nvisi, mvisi, nlast
  integer :: jv,iv, ifi, kfi, nfi, mfi, i_xoff, i_yoff
  real(kind=8) :: doffx,doffy,ra,dec
  real(kind=8), allocatable :: doff(:,:), dtmp(:,:)
  character(len=14) :: chra, chde
  type(projection_t) :: proj
  !
  call gildas_null(field,type='UVT')
  call gildas_null(mosaic,type='UVT')
  !
  if (nf.ne.0) then
    call gag_message(seve%e,rname,'GATHER Not supported here...')
    error = .true.
    return
  endif
  error = .false.
  nn = len_trim(generic)
  !
  name = generic
  call sic_parsef (name,mosaic%file,' ','.uvt')
  call gdf_read_header(mosaic,error)
  if (error) then
    call gag_message(seve%e,rname,'Cannot open '//trim(mosaic%file))
    return
  endif
  !
  ! First pass: Check number of fields
  ier = 0
  if (mosaic%gil%column_size(code_uvt_xoff).eq.0) then
    call gag_message(seve%w,rname,'No X offset column')
    ier = ier+1
  endif
  if (mosaic%gil%column_size(code_uvt_yoff).eq.0) then
    call gag_message(seve%w,rname,'No Y offset column')
    ier = ier+1
  endif
  if (ier.ne.0) then
    call gag_message(seve%e,rname,'Input UV table is not a mosaic')
    error = .true.
    return
  endif
  !
  allocate(dmos(mosaic%gil%dim(1),mosaic%gil%dim(2)),stat=ier)
  if (ier.ne.0) then
    call gag_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  call gwcs_projec(mosaic%gil%a0,mosaic%gil%d0,mosaic%gil%pang,mosaic%gil%ptyp,proj,error)
  if (error) return
  !
  call gdf_read_data(mosaic,dmos,error)
  if (error) return
  !
  ! Scan the list of offset
  i_xoff = mosaic%gil%column_pointer(code_uvt_xoff)
  i_yoff = mosaic%gil%column_pointer(code_uvt_yoff)
  doffx = dmos(i_xoff,1)
  doffy = dmos(i_yoff,1)
  nfi = 1
  mfi = 100
  allocate(doff(3,mfi),stat=ier)
  doff(1,1) = doffx
  doff(2,1) = doffy
  doff(3,1) = 1
  !
  do iv=2,mosaic%gil%nvisi
    kfi = 0
    do ifi=1,nfi
      if (dmos(i_xoff,iv).eq.doff(1,ifi) .and. &
      & dmos(i_yoff,iv).eq.doff(2,ifi) ) then
        kfi = ifi
        doff(3,kfi) = doff(3,kfi)+1
        exit
      endif
    enddo
    !
    ! New field
    if (kfi.eq.0) then
      if (nfi.eq.mfi) then
        allocate(dtmp(3,2*mfi),stat=ier)
        dtmp(:,1:mfi) = doff(:,:)
        deallocate(doff)
        allocate(doff(3,2*mfi),stat=ier)
        doff(:,:) = dtmp
        deallocate(dtmp)
        mfi = 2*mfi
      endif
      nfi = nfi+1
      doff(1,nfi) = dmos(i_xoff,iv)
      doff(2,nfi) = dmos(i_yoff,iv)
      doff(3,nfi) = 1
    endif
  enddo
  !
  mfi = 0
  do ifi=1,nfi
    call rel_to_abs(proj,doff(1,ifi),doff(2,ifi),ra,dec,1)
    call sexag(chra,ra,24)
    call sexag(chde,dec,360)
    mfi = max(nint(doff(3,ifi)),mfi)
  enddo
  !
  !
  call gdf_copy_header(mosaic,field,error)
  if (error) return
  !
  ! Get rid of the trailing columns if possible. Just verify
  ! the ntrail is 2, and that these are the two appropriate
  ! columns
  if (field%gil%ntrail.eq.2) then
    if ( (i_xoff.eq.field%gil%lcol+1 .and. &
      &  i_yoff.eq.field%gil%lcol+2) .or. &
      & (i_xoff.eq.field%gil%lcol+2 .and. &
      &  i_yoff.eq.field%gil%lcol+1) ) then
      field%gil%column_pointer(code_uvt_xoff) = 0
      field%gil%column_pointer(code_uvt_yoff) = 0
      field%gil%column_size(code_uvt_xoff) = 0
      field%gil%column_size(code_uvt_yoff) = 0
    endif
    field%gil%ntrail = 0
    field%gil%dim(1) = field%gil%lcol
    nlast = field%gil%lcol
  else
    nlast = field%gil%dim(1)
  endif
  !
  ! Now create the output UV tables
  !
  allocate (dfield(nlast,mfi),stat=ier)
  if (ier.ne.0) then
    call gag_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  !
  do ifi=1,nfi
    name = generic
    call append_number(name,ifi)
    call sic_parsef (name,field%file,' ','.uvt')
    field%gil%nvisi = doff(3,ifi)
    field%gil%dim(2) = field%gil%nvisi
    call rel_to_abs(proj,doff(1,ifi),doff(2,ifi),ra,dec,1)
    field%gil%ra = ra
    field%gil%dec = dec
    !
    ! And fill it
    !
    jv = 0
    do iv=1,mosaic%gil%nvisi
      if (dmos(i_xoff,iv).eq.doff(1,ifi) .and. &
      & dmos(i_yoff,iv).eq.doff(2,ifi) ) then
        jv = jv+1
        dfield(1:nlast,jv) = dmos(1:nlast,iv)
      endif
    enddo
    field%gil%nvisi = jv
    call gdf_create_image(field,error)
    if (error) return
    !
    field%blc = 0
    field%trc = 0
    call gdf_write_data(field,dfield,error)
    if (error) return
    call gdf_close_image(field,error)
    if (error) return
  enddo
end subroutine sub_explode_mosaic
!

