/*
 * Decompiled with CFR 0.152.
 */
package db.buffers;

import db.DBChangeSet;
import db.DBHandle;
import db.buffers.BufferFile;
import db.buffers.BufferFileAdapter;
import db.buffers.BufferFileBlock;
import db.buffers.BufferFileManager;
import db.buffers.BufferNode;
import db.buffers.DataBuffer;
import db.buffers.IndexProvider;
import db.buffers.InputBlockStream;
import db.buffers.LocalBufferFile;
import db.buffers.LocalManagedBufferFile;
import db.buffers.ManagedBufferFile;
import db.buffers.OutputBlockStream;
import db.buffers.RecoveryFile;
import db.buffers.RecoveryMgr;
import ghidra.framework.ShutdownHookRegistry;
import ghidra.framework.ShutdownPriority;
import ghidra.util.Msg;
import ghidra.util.SystemUtilities;
import ghidra.util.datastruct.ObjectArray;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.ClosedException;
import ghidra.util.task.TaskMonitor;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.NoSuchElementException;
import java.util.Stack;

public class BufferMgr {
    public static final String ALWAYS_PRECACHE_PROPERTY = "db.always.precache";
    private static boolean alwaysPreCache = SystemUtilities.getBooleanProperty((String)"db.always.precache", (boolean)false);
    public static final int DEFAULT_BUFFER_SIZE = 16384;
    public static final int DEFAULT_CHECKPOINT_COUNT = 10;
    public static final int DEFAULT_CACHE_SIZE = 0x400000;
    private static final int MINIMUM_CACHE_SIZE = 65536;
    private static final String CACHE_FILE_PREFIX = "ghidra";
    private static final String CACHE_FILE_EXT = ".cache";
    private static final int HEAD = -1;
    private static final int TAIL = -2;
    private static HashSet<BufferMgr> openInstances;
    private int maxCheckpoints;
    private int maxCacheSize;
    private int currentCheckpoint = -1;
    private boolean corruptedState = false;
    private BufferFile sourceFile;
    private LocalBufferFile cacheFile;
    private RecoveryMgr recoveryMgr;
    private Object snapshotLock = new Object();
    private boolean modifiedSinceSnapshot = false;
    private boolean hasNonUndoableChanges = false;
    private int bufferSize;
    private BufferNode cacheHead;
    private BufferNode cacheTail;
    private int cacheSize = 0;
    private int buffersOnHand = 0;
    private int lockCount = 0;
    private Stack<DataBuffer> freeBuffers = new Stack();
    private long cacheHits = 0L;
    private long cacheMisses = 0L;
    private int lowWaterMark = -1;
    private ArrayList<BufferNode> checkpointHeads = new ArrayList();
    private ArrayList<BufferNode> redoCheckpointHeads = new ArrayList();
    private BufferNode currentCheckpointHead;
    private BufferNode baselineCheckpointHead;
    private IndexProvider indexProvider;
    private IndexProvider cacheIndexProvider;
    private ObjectArray bufferTable;
    private static final int INITIAL_BUFFER_TABLE_SIZE = 1024;
    private PreCacheStatus preCacheStatus = PreCacheStatus.INIT;
    private Thread preCacheThread;
    private Object preCacheLock = new Object();

    public BufferMgr() throws IOException {
        this(null, 16384, 0x400000L, 10);
    }

    public BufferMgr(int requestedBufferSize, long approxCacheSize, int maxUndos) throws IOException {
        this(null, requestedBufferSize, approxCacheSize, maxUndos);
    }

    public BufferMgr(BufferFile sourceFile) throws IOException {
        this(sourceFile, 16384, 0x400000L, 10);
    }

    public BufferMgr(BufferFile sourceFile, long approxCacheSize, int maxUndos) throws IOException {
        this(sourceFile, 0, approxCacheSize, maxUndos);
    }

    private BufferMgr(BufferFile sourceFile, int requestedBufferSize, long approxCacheSize, int maxUndos) throws FileNotFoundException, IOException {
        this.bufferSize = requestedBufferSize;
        if (sourceFile != null) {
            this.sourceFile = sourceFile;
            int cnt = sourceFile.getIndexCount();
            this.indexProvider = new IndexProvider(cnt, sourceFile.getFreeIndexes());
            this.bufferTable = new ObjectArray(cnt + 1024);
            this.bufferSize = sourceFile.getBufferSize();
        } else {
            this.indexProvider = new IndexProvider();
            this.bufferTable = new ObjectArray(1024);
            this.bufferSize = LocalBufferFile.getRecommendedBufferSize(this.bufferSize);
        }
        this.maxCheckpoints = maxUndos < 1 ? 10 : maxUndos + 1;
        approxCacheSize = approxCacheSize < 65536L ? 65536L : approxCacheSize;
        this.maxCacheSize = (int)(approxCacheSize / (long)this.bufferSize);
        this.cacheHead = new BufferNode(-1, -1);
        this.cacheHead.nextCached = this.cacheTail = new BufferNode(-2, -1);
        this.cacheTail.prevCached = this.cacheHead;
        this.cacheFile = new LocalBufferFile(this.bufferSize, CACHE_FILE_PREFIX, CACHE_FILE_EXT);
        this.cacheIndexProvider = new IndexProvider();
        this.startCheckpoint();
        this.baselineCheckpointHead = this.currentCheckpointHead;
        this.currentCheckpointHead = null;
        if (sourceFile != null) {
            String[] parmNames;
            for (String name : parmNames = sourceFile.getParameterNames()) {
                this.cacheFile.setParameter(name, sourceFile.getParameter(name));
            }
        }
        BufferMgr.addInstance(this);
        if (alwaysPreCache) {
            this.startPreCacheIfNeeded();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void enablePreCache() {
        Object object = this.preCacheLock;
        synchronized (object) {
            if (this.preCacheStatus == PreCacheStatus.INIT) {
                this.startPreCacheIfNeeded();
            }
        }
    }

    private static synchronized void addInstance(BufferMgr bufMgr) {
        if (openInstances == null) {
            openInstances = new HashSet();
            Runnable cleanupTask = () -> {
                BufferMgr[] bufferMgrArray = BufferMgr.class;
                synchronized (BufferMgr.class) {
                    BufferMgr[] instanceList = openInstances.toArray(new BufferMgr[openInstances.size()]);
                    // ** MonitorExit[var1] (shouldn't be in output)
                    for (BufferMgr bufferMgr : instanceList) {
                        try {
                            bufferMgr.dispose(true);
                        }
                        catch (Throwable throwable) {
                            // empty catch block
                        }
                    }
                    return;
                }
            };
            ShutdownHookRegistry.addShutdownHook((Runnable)cleanupTask, (ShutdownPriority)ShutdownPriority.DISPOSE_FILE_HANDLES);
        }
        openInstances.add(bufMgr);
    }

    public void setCorruptedState() {
        this.corruptedState = true;
    }

    public boolean isCorrupted() {
        return this.corruptedState;
    }

    private static synchronized void removeInstance(BufferMgr bufMgr) {
        openInstances.remove(bufMgr);
    }

    public synchronized int getLockCount() {
        return this.lockCount;
    }

    public int getBufferSize() {
        return this.bufferSize;
    }

    public BufferFile getSourceFile() {
        return this.sourceFile;
    }

    protected void finalize() throws Throwable {
        this.dispose(true);
        super.finalize();
    }

    int getParameter(String name) throws NoSuchElementException {
        return this.cacheFile.getParameter(name);
    }

    void setParameter(String name, int value) {
        this.cacheFile.setParameter(name, value);
    }

    public void dispose() {
        this.dispose(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dispose(boolean keepRecoveryData) {
        Object object = this.snapshotLock;
        synchronized (object) {
            this.stopPreCache();
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                BufferNode next;
                if (this.recoveryMgr != null) {
                    if (!keepRecoveryData) {
                        this.recoveryMgr.dispose();
                    }
                    this.recoveryMgr = null;
                }
                if (this.sourceFile != null) {
                    this.sourceFile.dispose();
                    this.sourceFile = null;
                }
                if (this.cacheFile != null) {
                    this.cacheFile.delete();
                    this.cacheFile = null;
                }
                if (this.checkpointHeads != null) {
                    for (BufferNode node : this.checkpointHeads) {
                        while (node != null) {
                            next = node.nextInCheckpoint;
                            node.buffer = null;
                            node.nextCached = null;
                            node.prevCached = null;
                            node.nextInCheckpoint = null;
                            node.prevInCheckpoint = null;
                            node.nextVersion = null;
                            node.prevVersion = null;
                            node = next;
                        }
                    }
                    this.checkpointHeads = null;
                }
                if (this.redoCheckpointHeads != null) {
                    for (BufferNode node : this.redoCheckpointHeads) {
                        while (node != null) {
                            next = node.nextInCheckpoint;
                            node.buffer = null;
                            node.nextCached = null;
                            node.prevCached = null;
                            node.nextInCheckpoint = null;
                            node.prevInCheckpoint = null;
                            node.nextVersion = null;
                            node.prevVersion = null;
                            node = next;
                        }
                    }
                    this.redoCheckpointHeads = null;
                }
                this.bufferTable = null;
                this.currentCheckpointHead = null;
                this.baselineCheckpointHead = null;
                this.hasNonUndoableChanges = false;
                BufferMgr.removeInstance(this);
            }
        }
    }

    private void packCheckpoints() {
        if (this.checkpointHeads.size() <= this.maxCheckpoints) {
            return;
        }
        BufferNode cpHead = this.checkpointHeads.get(1);
        BufferNode cpNode = cpHead.nextInCheckpoint;
        while (cpNode.id != -2) {
            BufferNode baseline = cpNode.nextVersion;
            BufferNode cpNext = cpNode.nextInCheckpoint;
            if (baseline.id != -2) {
                this.disposeNode(baseline, true);
            }
            cpNode.checkpoint = 0;
            cpNode.addToCheckpoint(this.baselineCheckpointHead);
            cpNode = cpNext;
        }
        this.checkpointHeads.remove(1);
        this.hasNonUndoableChanges = true;
    }

    private void disposeNode(BufferNode node, boolean isVersioned) {
        node.removeFromCheckpoint();
        if (isVersioned) {
            node.removeFromVersion();
        }
        if (node.buffer != null) {
            this.freeBuffers.push(node.buffer);
            this.removeFromCache(node);
        }
        if (node.diskCacheIndex >= 0) {
            this.cacheIndexProvider.freeIndex(node.diskCacheIndex);
            node.diskCacheIndex = -1;
        }
    }

    private void disposeNodeList(BufferNode head) {
        BufferNode node = head.nextInCheckpoint;
        while (node.id != -2) {
            BufferNode nextNode = node.nextInCheckpoint;
            this.disposeNode(node, false);
            node = nextNode;
        }
    }

    private void disposeRedoCheckpoints() {
        int cnt = this.redoCheckpointHeads.size();
        if (cnt == 0) {
            return;
        }
        for (int i = 0; i < cnt; ++i) {
            this.disposeNodeList(this.redoCheckpointHeads.get(i));
        }
        this.redoCheckpointHeads.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setMaxUndos(int maxUndos) {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                int n = this.maxCheckpoints = maxUndos < 0 ? 10 : maxUndos + 1;
                while (this.checkpointHeads.size() > this.maxCheckpoints) {
                    this.packCheckpoints();
                }
                this.disposeRedoCheckpoints();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clearCheckpoints() {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                int oldMaxCheckpoints = this.maxCheckpoints;
                this.checkpoint();
                this.setMaxUndos(0);
                this.setMaxUndos(oldMaxCheckpoints);
            }
        }
    }

    public int getMaxUndos() {
        return this.maxCheckpoints;
    }

    private DataBuffer getCacheBuffer() throws IOException {
        if (this.cacheSize < this.maxCacheSize) {
            ++this.cacheSize;
            this.lowWaterMark = this.cacheSize;
            return new DataBuffer(this.cacheFile.getBufferSize());
        }
        if (!this.freeBuffers.isEmpty()) {
            --this.buffersOnHand;
            if (this.buffersOnHand < this.lowWaterMark) {
                this.lowWaterMark = this.buffersOnHand;
            }
            return this.freeBuffers.pop();
        }
        BufferNode oldNode = this.cacheTail.prevCached;
        if (oldNode.id == -1) {
            throw new IOException("Out of cache buffer space");
        }
        DataBuffer buf = oldNode.buffer;
        this.unloadCachedNode(oldNode);
        this.removeFromCache(oldNode);
        return buf;
    }

    private void removeFromCache(BufferNode node) {
        if (node.buffer != null) {
            node.removeFromCache();
            node.buffer = null;
            --this.buffersOnHand;
            if (this.buffersOnHand < this.lowWaterMark) {
                this.lowWaterMark = this.buffersOnHand;
            }
        }
    }

    private void returnToCache(BufferNode node, DataBuffer buf) {
        if (node.buffer != null || buf == null) {
            throw new AssertException();
        }
        node.buffer = buf;
        node.addToCache(this.cacheHead);
        ++this.buffersOnHand;
    }

    private void returnFreeBuffer(DataBuffer buf) {
        ++this.buffersOnHand;
        this.freeBuffers.push(buf);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void stopPreCache() {
        Object object = this.preCacheLock;
        synchronized (object) {
            if (this.preCacheThread == null) {
                return;
            }
            if (this.preCacheStatus == PreCacheStatus.RUNNING) {
                this.preCacheThread.interrupt();
                this.preCacheStatus = PreCacheStatus.INTERUPTED;
            }
            try {
                this.preCacheLock.wait();
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void startPreCacheIfNeeded() {
        if (this.preCacheThread != null) {
            throw new IllegalStateException("pre-cache thread already active");
        }
        if (!(this.sourceFile instanceof BufferFileAdapter)) {
            return;
        }
        BufferFileAdapter sourceAdapter = (BufferFileAdapter)this.sourceFile;
        if (!sourceAdapter.isRemote()) {
            return;
        }
        Object object = this.preCacheLock;
        synchronized (object) {
            this.preCacheThread = new Thread(() -> {
                try {
                    this.preCacheSourceFile();
                }
                catch (InterruptedIOException interruptedIOException) {
                    Object object = this.preCacheLock;
                    synchronized (object) {
                        this.preCacheStatus = PreCacheStatus.STOPPED;
                        this.preCacheThread = null;
                        this.preCacheLock.notifyAll();
                    }
                }
                catch (IOException e) {
                    Msg.error((Object)this, (Object)("pre-cache failure: " + e.getMessage()), (Throwable)e);
                }
                finally {
                    Object object = this.preCacheLock;
                    synchronized (object) {
                        this.preCacheStatus = PreCacheStatus.STOPPED;
                        this.preCacheThread = null;
                        this.preCacheLock.notifyAll();
                    }
                }
            });
            this.preCacheThread.setName("Pre-Cache");
            this.preCacheThread.setPriority(1);
            this.preCacheThread.start();
            this.preCacheStatus = PreCacheStatus.RUNNING;
        }
    }

    private void preCacheSourceFile() throws IOException {
        if (!(this.sourceFile instanceof BufferFileAdapter)) {
            throw new UnsupportedOperationException("unsupported use of preCacheSourceFile");
        }
        Msg.trace((Object)this, (Object)"Pre-cache started...");
        int cacheCount = 0;
        BufferFileAdapter sourceAdapter = (BufferFileAdapter)this.sourceFile;
        try (InputBlockStream inputBlockStream = sourceAdapter.getInputBlockStream();){
            BufferFileBlock block;
            while (!Thread.interrupted() && (block = inputBlockStream.readBlock()) != null) {
                DataBuffer buf = LocalBufferFile.getDataBuffer(block);
                if (buf == null || buf.isEmpty() || !this.preCacheBuffer(buf)) continue;
                ++cacheCount;
            }
            Msg.trace((Object)this, (Object)("Pre-cache added " + cacheCount + " of " + this.sourceFile.getIndexCount() + " buffers to cache"));
        }
    }

    private synchronized boolean preCacheBuffer(DataBuffer buf) throws IOException {
        int id = buf.getId();
        BufferNode node = this.getCachedBufferNode(id);
        if (node != null) {
            return false;
        }
        node = this.createNewBufferNode(id, this.baselineCheckpointHead, null);
        node.buffer = buf;
        this.unloadCachedNode(node);
        node.buffer = null;
        return true;
    }

    private BufferNode getBufferNode(int id, boolean load) throws IOException {
        BufferNode node = this.getCachedBufferNode(id);
        if (node == null) {
            if (this.sourceFile == null) {
                throw new IOException("Invalid buffer");
            }
            DataBuffer buf = this.getCacheBuffer();
            try {
                this.sourceFile.get(buf, id);
            }
            catch (IOException e) {
                this.returnFreeBuffer(buf);
                throw e;
            }
            node = this.createNewBufferNode(id, this.baselineCheckpointHead, null);
            this.returnToCache(node, buf);
            return node;
        }
        if (node.locked) {
            throw new IOException("Locked buffer: " + id);
        }
        if (load) {
            this.loadCachedNode(node);
        }
        return node;
    }

    private BufferNode createNewBufferNode(int id, BufferNode checkpointHead, BufferNode versionHead) {
        BufferNode node = new BufferNode(id, checkpointHead.checkpoint);
        node.addToCheckpoint(checkpointHead);
        if (versionHead == null) {
            this.createNewBufferList(id, node);
        } else {
            node.addToVersion(versionHead);
        }
        return node;
    }

    private BufferNode createNewBufferList(int id, BufferNode node) {
        BufferNode head = new BufferNode(-1, -1);
        BufferNode tail = new BufferNode(-2, -1);
        head.nextVersion = node;
        node.prevVersion = head;
        node.nextVersion = tail;
        tail.prevVersion = node;
        this.bufferTable.put(id, (Object)head);
        return head;
    }

    private BufferNode getCachedBufferNode(int id) throws IOException {
        if (this.bufferTable == null) {
            throw new ClosedException();
        }
        BufferNode bufListHead = (BufferNode)this.bufferTable.get(id);
        BufferNode node = null;
        if (bufListHead != null) {
            node = bufListHead.nextVersion;
        }
        return node;
    }

    private void loadCachedNode(BufferNode node) throws IOException {
        if (node.buffer == null) {
            if (node.locked || node.empty) {
                throw new IOException("Invalid or locked buffer");
            }
            this.returnToCache(node, this.cacheFile.get(this.getCacheBuffer(), node.diskCacheIndex));
            ++this.cacheMisses;
        } else {
            if (node.prevCached.id != -1) {
                node.removeFromCache();
                node.addToCache(this.cacheHead);
            }
            ++this.cacheHits;
        }
    }

    private void unloadCachedNode(BufferNode node) throws IOException {
        if (node.buffer == null) {
            throw new AssertException();
        }
        if (node.diskCacheIndex < 0) {
            node.diskCacheIndex = this.cacheIndexProvider.allocateIndex();
            this.cacheFile.put(node.buffer, node.diskCacheIndex);
        } else if (node.isDirty) {
            this.cacheFile.put(node.buffer, node.diskCacheIndex);
        }
        node.isDirty = false;
    }

    public synchronized DataBuffer getBuffer(int id) throws IOException {
        if (this.corruptedState) {
            throw new IOException("Corrupted BufferMgr state");
        }
        BufferNode node = this.getBufferNode(id, true);
        DataBuffer buf = node.buffer;
        if (node.empty || buf.isEmpty()) {
            throw new IOException("Invalid buffer: " + id);
        }
        if (node.checkpoint != this.currentCheckpoint || this.currentCheckpointHead == null) {
            this.unloadCachedNode(node);
        }
        this.removeFromCache(node);
        node.locked = true;
        ++this.lockCount;
        return buf;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DataBuffer createBuffer() throws IOException {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                if (this.corruptedState) {
                    throw new IOException("Corrupted BufferMgr state");
                }
                int id = this.indexProvider.allocateIndex();
                DataBuffer buf = null;
                BufferNode node = this.getCachedBufferNode(id);
                if (node != null) {
                    buf = node.buffer;
                    node.locked = true;
                    this.removeFromCache(node);
                }
                buf = buf != null ? buf : this.getCacheBuffer();
                buf.setId(id);
                buf.setDirty(true);
                buf.setEmpty(false);
                ++this.lockCount;
                return buf;
            }
        }
    }

    public void releaseBuffer(DataBuffer buf) throws IOException {
        try {
            if (buf.isDirty()) {
                this.releaseDirtyBuffer(buf);
            } else {
                this.releaseCleanBuffer(buf);
            }
        }
        catch (Exception e) {
            this.handleCorruptionException(e, "BufferMgr buffer release failed");
        }
    }

    private void handleCorruptionException(Exception exception, String errorText) throws IOException {
        if (exception instanceof ClosedException) {
            throw (IOException)exception;
        }
        Msg.error((Object)this, (Object)errorText, (Throwable)exception);
        this.corruptedState = true;
        if (exception instanceof IOException) {
            throw (IOException)exception;
        }
        if (!(exception instanceof RuntimeException)) {
            exception = new RuntimeException(errorText, exception);
        }
        throw (RuntimeException)exception;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void releaseCleanBuffer(DataBuffer buf) throws IOException {
        BufferMgr bufferMgr = this;
        synchronized (bufferMgr) {
            int id = buf.getId();
            BufferNode node = this.getCachedBufferNode(id);
            if (node == null || !node.locked) {
                throw new AssertException();
            }
            node.locked = false;
            --this.lockCount;
            this.returnToCache(node, buf);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void releaseDirtyBuffer(DataBuffer buf) throws IOException, AssertionError {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                int id = buf.getId();
                BufferNode node = this.getCachedBufferNode(id);
                if (node != null && !node.locked) {
                    throw new AssertException();
                }
                this.modifiedSinceSnapshot = true;
                if (this.currentCheckpointHead == null) {
                    this.startCheckpoint();
                }
                if (node == null) {
                    node = this.createNewBufferNode(id, this.currentCheckpointHead, null);
                } else {
                    if (node.buffer != null) {
                        throw new AssertionError((Object)"Invalid buffer state");
                    }
                    if (this.currentCheckpoint != node.checkpoint) {
                        BufferNode head = node.prevVersion;
                        if (head.id != -1) {
                            throw new AssertException("Head expected");
                        }
                        node.locked = false;
                        node = this.createNewBufferNode(id, this.currentCheckpointHead, head);
                    }
                }
                buf.setDirty(false);
                node.isDirty = true;
                node.modified = true;
                node.empty = buf.isEmpty();
                if (node.empty) {
                    this.indexProvider.freeIndex(id);
                }
                node.locked = false;
                --this.lockCount;
                this.returnToCache(node, buf);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteBuffer(int id) throws IOException {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                if (this.corruptedState) {
                    throw new IOException("Corrupted BufferMgr state");
                }
                try {
                    DataBuffer buf = this.getBuffer(id);
                    buf.setEmpty(true);
                    buf.setDirty(true);
                    this.releaseBuffer(buf);
                }
                catch (Exception e) {
                    this.handleCorruptionException(e, "BufferMgr buffer delete failed");
                }
            }
        }
    }

    public boolean atCheckpoint() {
        return this.currentCheckpointHead == null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean checkpoint() {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                if (this.currentCheckpointHead == null) {
                    return false;
                }
                if (this.lockCount != 0) {
                    throw new AssertException("Can't checkpoint with locked buffers (" + this.lockCount + " locks found)");
                }
                this.currentCheckpointHead = null;
                return true;
            }
        }
    }

    public synchronized boolean isChanged() {
        return this.currentCheckpoint != 0 || this.currentCheckpointHead != null || this.hasNonUndoableChanges;
    }

    private void startCheckpoint() {
        BufferNode tail;
        this.disposeRedoCheckpoints();
        ++this.currentCheckpoint;
        BufferNode head = new BufferNode(-1, this.currentCheckpoint);
        head.nextInCheckpoint = tail = new BufferNode(-2, this.currentCheckpoint);
        tail.prevInCheckpoint = head;
        this.checkpointHeads.add(head);
        this.currentCheckpointHead = head;
        this.packCheckpoints();
    }

    public boolean hasUndoCheckpoints() {
        return this.checkpointHeads.size() > 1;
    }

    public boolean hasRedoCheckpoints() {
        return this.redoCheckpointHeads.size() != 0;
    }

    public int getAvailableUndoCount() {
        return this.checkpointHeads.size() - 1;
    }

    public int getAvailableRedoCount() {
        return this.redoCheckpointHeads.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean undo(boolean redoable) throws IOException {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                if (this.lockCount != 0) {
                    throw new AssertException("Can't undo with locked buffers (" + this.lockCount + " locks found)");
                }
                int ix = this.checkpointHeads.size() - 1;
                if (ix < 1) {
                    return false;
                }
                this.modifiedSinceSnapshot = true;
                BufferNode cpHead = this.checkpointHeads.remove(ix);
                BufferNode node = cpHead.nextInCheckpoint;
                int srcIndexCnt = this.sourceFile != null ? this.sourceFile.getIndexCount() : 0;
                int revisedIndexCnt = this.indexProvider.getIndexCount();
                while (node.id != -2) {
                    BufferNode oldVer = node.nextVersion;
                    node.removeFromVersion();
                    if (oldVer.prevVersion.id != -1) {
                        throw new AssertException();
                    }
                    if (oldVer.id == -2) {
                        this.bufferTable.remove(node.id);
                        if (this.sourceFile == null || node.id >= srcIndexCnt) {
                            revisedIndexCnt = Math.min(node.id, revisedIndexCnt);
                        } else if (!node.empty) {
                            this.indexProvider.freeIndex(node.id);
                        }
                    } else {
                        if (node.empty) {
                            if (!oldVer.empty && !this.indexProvider.allocateIndex(node.id)) {
                                throw new AssertException();
                            }
                        } else if (oldVer.empty) {
                            this.indexProvider.freeIndex(node.id);
                        }
                        oldVer.clearSnapshotTaken();
                    }
                    node = node.nextInCheckpoint;
                }
                this.indexProvider.truncate(revisedIndexCnt);
                if (redoable) {
                    this.redoCheckpointHeads.add(cpHead);
                } else {
                    this.disposeNodeList(cpHead);
                    this.disposeRedoCheckpoints();
                }
                cpHead = this.checkpointHeads.get(ix - 1);
                this.currentCheckpoint = cpHead.checkpoint;
                this.currentCheckpointHead = null;
                return true;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean redo() {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                if (this.lockCount != 0) {
                    throw new AssertException("Can't redo with locked buffers (" + this.lockCount + " locks found)");
                }
                int ix = this.redoCheckpointHeads.size() - 1;
                if (ix < 0) {
                    return false;
                }
                this.modifiedSinceSnapshot = true;
                BufferNode cpHead = this.redoCheckpointHeads.remove(ix);
                BufferNode node = cpHead.nextInCheckpoint;
                while (node.id != -2) {
                    BufferNode head = (BufferNode)this.bufferTable.get(node.id);
                    if (head == null) {
                        if (!this.indexProvider.allocateIndex(node.id)) {
                            throw new AssertException();
                        }
                        if (node.empty) {
                            this.indexProvider.freeIndex(node.id);
                        }
                        node.clearSnapshotTaken();
                        head = this.createNewBufferList(node.id, node);
                    } else {
                        BufferNode curVer = head.nextVersion;
                        if (node.empty) {
                            if (!curVer.empty) {
                                this.indexProvider.freeIndex(node.id);
                            }
                        } else if (curVer.empty && !this.indexProvider.allocateIndex(node.id)) {
                            throw new AssertException();
                        }
                        node.clearSnapshotTaken();
                        node.addToVersion(head);
                    }
                    node = node.nextInCheckpoint;
                }
                this.checkpointHeads.add(cpHead);
                this.currentCheckpoint = cpHead.checkpoint;
                this.currentCheckpointHead = null;
                return true;
            }
        }
    }

    public boolean canSave() throws IOException {
        if (this.corruptedState) {
            return false;
        }
        if (this.sourceFile instanceof ManagedBufferFile) {
            return ((ManagedBufferFile)this.sourceFile).canSave();
        }
        return false;
    }

    public synchronized boolean modifiedSinceSnapshot() {
        return this.modifiedSinceSnapshot;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean takeRecoverySnapshot(DBChangeSet changeSet, TaskMonitor monitor) throws IOException, CancelledException {
        if (this.corruptedState) {
            throw new IOException("Corrupted BufferMgr state");
        }
        if (!(this.sourceFile instanceof LocalBufferFile)) {
            throw new RuntimeException("Invalid use of recovery manager");
        }
        Object object = this.snapshotLock;
        synchronized (object) {
            if (!this.canSave()) {
                throw new RuntimeException("Recovery snapshot only permitted for update of existing file");
            }
            if (this.currentCheckpointHead != null) {
                return false;
            }
            if (this.recoveryMgr == null) {
                this.recoveryMgr = new RecoveryMgr(this);
            }
            boolean success = false;
            try {
                this.recoveryMgr.startSnapshot(this.indexProvider.getIndexCount(), this.indexProvider.getFreeIndexes(), changeSet, monitor);
                int srcIndexCnt = this.sourceFile.getIndexCount();
                int indexCnt = this.indexProvider.getIndexCount();
                monitor.initialize((long)indexCnt);
                DataBuffer buf = new DataBuffer(this.cacheFile.getBufferSize());
                for (int id = 0; id < indexCnt; ++id) {
                    monitor.checkCanceled();
                    monitor.setProgress((long)id);
                    BufferNode node = null;
                    boolean writeBuffer = false;
                    BufferMgr bufferMgr = this;
                    synchronized (bufferMgr) {
                        node = this.getCachedBufferNode(id);
                        if (node != null) {
                            if (id < srcIndexCnt && node.checkpoint == 0 && !node.modified) {
                                continue;
                            }
                            if (!node.empty) {
                                if (node.buffer == null) {
                                    this.cacheFile.get(buf, node.diskCacheIndex);
                                } else {
                                    buf.copy(0, node.buffer, 0, node.buffer.length());
                                }
                                buf.setId(id);
                                writeBuffer = true;
                            }
                        }
                    }
                    if (!writeBuffer) continue;
                    this.recoveryMgr.putBuffer(buf, node);
                }
                this.modifiedSinceSnapshot = false;
                success = true;
            }
            finally {
                if (this.recoveryMgr.isSnapshotInProgress()) {
                    this.recoveryMgr.endSnapshot(success);
                }
            }
            return success;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public LocalBufferFile getRecoveryChangeSetFile() throws IOException {
        Object object = this.snapshotLock;
        synchronized (object) {
            if (this.recoveryMgr != null) {
                return this.recoveryMgr.getRecoveryChangeSetFile();
            }
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clearRecoveryFiles() {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                if (!(this.sourceFile instanceof LocalBufferFile) || this.bufferTable == null || this.isChanged() || this.recoveryMgr != null || this.lockCount != 0) {
                    return;
                }
                new RecoveryMgr(this);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean recover(TaskMonitor monitor) throws IOException, CancelledException {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                if (!(this.sourceFile instanceof LocalBufferFile) || this.bufferTable == null || this.isChanged() || this.recoveryMgr != null || this.lockCount != 0 || this.corruptedState) {
                    return false;
                }
                this.recoveryMgr = new RecoveryMgr(this, monitor);
                return this.recoveryMgr.recovered();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    synchronized void recover(RecoveryFile recoveryFile, int recoveryIndex, TaskMonitor monitor) throws IOException, CancelledException {
        boolean success;
        block8: {
            if (this.corruptedState) {
                throw new IOException("Corrupted BufferMgr state");
            }
            success = false;
            try {
                int[] freeIndexes;
                this.startCheckpoint();
                int[] bufferIndexes = recoveryFile.getBufferIndexes();
                monitor.initialize((long)bufferIndexes.length);
                int origIndexCount = this.indexProvider.getIndexCount();
                int recoveryIndexCount = recoveryFile.getIndexCount();
                if (recoveryIndexCount > origIndexCount) {
                    int maxIndex = recoveryIndexCount - 1;
                    this.indexProvider.allocateIndex(maxIndex);
                    this.indexProvider.freeIndex(maxIndex);
                }
                for (int index : freeIndexes = recoveryFile.getFreeIndexList()) {
                    monitor.checkCanceled();
                    if (index >= origIndexCount) {
                        BufferNode node = this.createNewBufferNode(index, this.currentCheckpointHead, null);
                        node.isDirty = true;
                        node.modified = true;
                        node.empty = true;
                        continue;
                    }
                    if (this.indexProvider.isFree(index)) continue;
                    this.deleteBuffer(index);
                }
                Arrays.sort(bufferIndexes);
                for (int i = 0; i < bufferIndexes.length; ++i) {
                    monitor.checkCanceled();
                    monitor.setProgress((long)(i + 1));
                    int index = bufferIndexes[i];
                    this.indexProvider.allocateIndex(index);
                    BufferNode node = this.createNewBufferNode(index, this.currentCheckpointHead, null);
                    DataBuffer buf = this.getCacheBuffer();
                    buf.setId(index);
                    buf.setDirty(true);
                    buf.setEmpty(false);
                    recoveryFile.getBuffer(buf, index);
                    node.isDirty = true;
                    node.modified = true;
                    node.empty = false;
                    node.snapshotTaken[recoveryIndex] = true;
                    this.returnToCache(node, buf);
                }
                this.checkpoint();
                success = true;
                if (success) break block8;
            }
            catch (Throwable throwable) {
                if (!success) {
                    Msg.error((Object)this, (Object)("Buffer file recover failed using: " + recoveryFile.getFile()));
                }
                this.corruptedState = !success;
                throw throwable;
            }
            Msg.error((Object)this, (Object)("Buffer file recover failed using: " + recoveryFile.getFile()));
        }
        this.corruptedState = !success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean canRecover(BufferFileManager bfMgr) {
        int ver = bfMgr.getCurrentVersion();
        if (ver < 1) {
            return false;
        }
        LocalBufferFile bf = null;
        try {
            bf = new LocalBufferFile(bfMgr.getBufferFile(ver), true);
            boolean bl = RecoveryMgr.canRecover(bf);
            return bl;
        }
        catch (IOException iOException) {
        }
        finally {
            if (bf != null) {
                try {
                    bf.close();
                }
                catch (IOException iOException) {}
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void save(String comment, DBChangeSet changeSet, TaskMonitor monitor) throws IOException, CancelledException {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                if (!(this.sourceFile instanceof ManagedBufferFile)) {
                    throw new IOException("Save not allowed");
                }
                if (this.corruptedState) {
                    throw new IOException("Corrupted BufferMgr state");
                }
                if (this.lockCount != 0) {
                    throw new IOException("Attempted save while buffers are locked");
                }
                if (monitor == null) {
                    monitor = TaskMonitor.DUMMY;
                }
                boolean oldCancelState = monitor.isCancelEnabled();
                ManagedBufferFile outFile = null;
                monitor.setMessage("Waiting for pre-save to complete...");
                if (this.sourceFile instanceof LocalManagedBufferFile) {
                    outFile = ((LocalManagedBufferFile)this.sourceFile).getSaveFile(monitor);
                } else {
                    monitor.setCancelEnabled(false);
                    outFile = ((ManagedBufferFile)this.sourceFile).getSaveFile();
                    monitor.setCancelEnabled(oldCancelState & !monitor.isCancelled());
                }
                if (outFile == null) {
                    throw new IOException("Save not allowed");
                }
                boolean success = false;
                try {
                    BufferFile changeFile;
                    if (comment != null) {
                        outFile.setVersionComment(comment);
                    }
                    this.doSave(outFile, monitor);
                    monitor.setCancelEnabled(false);
                    if (changeSet != null && (changeFile = ((ManagedBufferFile)this.sourceFile).getSaveChangeDataFile()) != null) {
                        monitor.setMessage("Saving change data...");
                        DBHandle cfh = new DBHandle(outFile.getBufferSize());
                        changeSet.write(cfh, false);
                        cfh.saveAs(changeFile, true, null);
                        cfh.close();
                    }
                    monitor.setMessage("Completing file save...");
                    success = true;
                }
                catch (Throwable throwable) {
                    ((ManagedBufferFile)this.sourceFile).saveCompleted(success);
                    monitor.setCancelEnabled(oldCancelState & !monitor.isCancelled());
                    throw throwable;
                }
                ((ManagedBufferFile)this.sourceFile).saveCompleted(success);
                monitor.setCancelEnabled(oldCancelState & !monitor.isCancelled());
                this.setSourceFile(outFile);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void saveAs(BufferFile outFile, boolean associateWithNewFile, TaskMonitor monitor) throws IOException, CancelledException {
        Object object = this.snapshotLock;
        synchronized (object) {
            BufferMgr bufferMgr = this;
            synchronized (bufferMgr) {
                if (this.corruptedState) {
                    throw new IOException("Corrupted BufferMgr state");
                }
                if (outFile.getIndexCount() != 0) {
                    throw new IllegalArgumentException("Empty buffer file must be provided");
                }
                if (this.lockCount != 0) {
                    throw new IOException("Attempted saveAs while buffers are locked");
                }
                if (monitor == null) {
                    monitor = TaskMonitor.DUMMY;
                }
                int indexCnt = this.indexProvider.getIndexCount();
                monitor.initialize((long)indexCnt);
                boolean success = false;
                try {
                    this.doSave(outFile, monitor);
                    monitor.setCancelEnabled(false);
                    monitor.setMessage("Completing file save...");
                    outFile.setReadOnly();
                    success = true;
                }
                finally {
                    if (!success) {
                        outFile.delete();
                    }
                    monitor.setCancelEnabled(true);
                }
                if (associateWithNewFile) {
                    this.setSourceFile(outFile);
                }
            }
        }
    }

    private void doSave(BufferFile outFile, TaskMonitor monitor) throws IOException, CancelledException {
        String[] parmNames;
        int indexCnt = this.indexProvider.getIndexCount();
        int preSaveCnt = outFile.getIndexCount();
        if (monitor == null) {
            monitor = TaskMonitor.DUMMY;
        }
        monitor.initialize((long)indexCnt);
        monitor.setMessage("Saving file...");
        int bufCount = 0;
        for (int id = 0; id < indexCnt; ++id) {
            DataBuffer buf;
            monitor.checkCanceled();
            BufferNode node = this.getCachedBufferNode(id);
            if (node != null) {
                if (node.empty || id < preSaveCnt && node.checkpoint == 0 && !node.modified) continue;
                ++bufCount;
                continue;
            }
            if (id < preSaveCnt || this.indexProvider.isFree(id) || (buf = this.getBuffer(id)) == null) continue;
            ++bufCount;
            this.releaseBuffer(buf);
        }
        try (OutputBlockStream out = LocalBufferFile.getOutputBlockStream(outFile, bufCount);){
            for (int id = 0; id < indexCnt; ++id) {
                monitor.checkCanceled();
                monitor.setProgress((long)id);
                BufferNode node = this.getCachedBufferNode(id);
                if (node == null || node.empty || id < preSaveCnt && node.checkpoint == 0 && !node.modified) continue;
                this.loadCachedNode(node);
                BufferFileBlock block = LocalBufferFile.getBufferFileBlock(node.buffer, this.bufferSize);
                out.writeBlock(block);
            }
        }
        outFile.setFreeIndexes(this.indexProvider.getFreeIndexes());
        for (String name : parmNames = this.cacheFile.getParameterNames()) {
            outFile.setParameter(name, this.cacheFile.getParameter(name));
        }
    }

    private void setSourceFile(BufferFile newFile) {
        if (this.sourceFile != null) {
            this.sourceFile.dispose();
            this.sourceFile = null;
        }
        int tempMaxCheckpoints = this.maxCheckpoints;
        this.setMaxUndos(0);
        this.currentCheckpointHead = null;
        this.setMaxUndos(tempMaxCheckpoints);
        BufferNode node = this.baselineCheckpointHead.nextInCheckpoint;
        while (node.id != -2) {
            node.modified = false;
            node.checkpoint = 0;
            node = node.nextInCheckpoint;
        }
        this.currentCheckpoint = 0;
        this.hasNonUndoableChanges = false;
        this.sourceFile = newFile;
        if (this.recoveryMgr != null) {
            this.recoveryMgr.clear();
        }
    }

    public long getCacheHits() {
        return this.cacheHits;
    }

    public long getCacheMisses() {
        return this.cacheMisses;
    }

    public int getLowBufferCount() {
        return this.lowWaterMark;
    }

    public void resetCacheStatistics() {
        this.cacheHits = 0L;
        this.cacheMisses = 0L;
        this.lowWaterMark = this.cacheSize;
    }

    public String getStatusInfo() {
        StringBuffer buf = new StringBuffer();
        if (this.corruptedState) {
            buf.append("BufferMgr is Corrupt!\n");
        }
        buf.append("Checkpoints: ");
        buf.append(this.currentCheckpoint);
        if (this.sourceFile != null) {
            buf.append("\n Source file: ");
            buf.append(this.sourceFile.toString());
        }
        buf.append("\n Cache file: ");
        buf.append(this.cacheFile.toString());
        buf.append("\n Buffer size: ");
        buf.append(this.bufferSize);
        buf.append("\n Cache size: ");
        buf.append(this.cacheSize);
        buf.append("\n Cache hits: ");
        buf.append(this.cacheHits);
        buf.append("\n Cache misses: ");
        buf.append(this.cacheMisses);
        buf.append("\n Locked buffers: ");
        buf.append(this.lockCount);
        buf.append("\n Low water buffer count: ");
        buf.append(this.lowWaterMark);
        buf.append("\n");
        return buf.toString();
    }

    public int getAllocatedBufferCount() {
        return this.indexProvider.getIndexCount() - this.indexProvider.getFreeIndexCount();
    }

    public int getFreeBufferCount() {
        return this.indexProvider.getFreeIndexCount();
    }

    public static void cleanupOldCacheFiles() {
        File tmpDir = new File(System.getProperty("java.io.tmpdir"));
        File[] cacheFiles = tmpDir.listFiles(new LocalBufferFile.BufferFileFilter(CACHE_FILE_PREFIX, CACHE_FILE_EXT));
        if (cacheFiles == null) {
            return;
        }
        for (File file : cacheFiles) {
            file.delete();
        }
    }

    private static enum PreCacheStatus {
        INIT,
        RUNNING,
        INTERUPTED,
        STOPPED;

    }
}

