#!/bin/sh
#**********************************************************************
# Copyright (C) 2023 - The Geeqie Team
#
# Author: Colin Clark
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#**********************************************************************

## @file
## @brief Run Lua tests
##
## $1 Geeqie executable
##
## Create a basic image and run all lua built-in functions on it.
## The image file and the Lua test file are created within this script.

if [ -z "$XDG_CONFIG_HOME" ]
then
	config_home="$HOME/.config"
else
	config_home="$XDG_CONFIG_HOME"
fi

lua_test_image=$(mktemp --suffix=".jpeg" "${TMPDIR:-/tmp}/geeqie.XXXXXXXXXX")
mkdir -p "$config_home/geeqie/lua/"
lua_test_file=$(mktemp --suffix=".lua" --tmpdir="$config_home/geeqie/lua/" lua-test-XXXXXX)

# Test image generated by:
# convert -size 32x32 xc:white empty.jpg
# exiftool -ISO=200 --printConv empty.jpg
# base64 --wrap=0 empty.jpg > <temp file>
lua_test_image_base64="/9j/4AAQSkZJRgABAQAAAQABAAD/4QCkRXhpZgAATU0AKgAAAAgABQEaAAUAAAABAAAASgEbAAUAAAABAAAAUgEoAAMAAAABAAEAAAITAAMAAAABAAEAAIdpAAQAAAABAAAAWgAAAAAAAAABAAAAAQAAAAEAAAABAAWIJwADAAAAAQDIAACQAAAHAAAABDAyMzKRAQAHAAAABAECAwCgAAAHAAAABDAxMDCgAQADAAAAAf//AAAAAAAA/9sAQwADAgICAgIDAgICAwMDAwQGBAQEBAQIBgYFBgkICgoJCAkJCgwPDAoLDgsJCQ0RDQ4PEBAREAoMEhMSEBMPEBAQ/8AACwgAIAAgAQERAP/EABUAAQEAAAAAAAAAAAAAAAAAAAAJ/8QAFBABAAAAAAAAAAAAAAAAAAAAAP/aAAgBAQAAPwCqYAAAAD//2Q=="
printf "%s" "$lua_test_image_base64" | base64 --decode > "$lua_test_image"

lua_test="path = Image:get_path()
name = Image:get_name()
extension = Image:get_extension()
date = math.floor(Image:get_date())
size = math.floor(Image:get_size())
marks = math.floor(Image:get_marks())
exif_structure = Image:get_exif()
iso = exif_structure:get_datum(\"Exif.Photo.ISOSpeedRatings\")
ret = name..\"\n\"..extension..\"\n\"..path..\"\n\"..date..\"\n\"..size..\"\n\"..marks..\"\n\"..iso
return ret
"
printf  "%s" "$lua_test" > "$lua_test_file"

xvfb-run --auto-servernum "$1" &

# Wait for remote to initialize
while [ ! -e "$config_home/geeqie/.command" ] ;
do
	sleep 1
done

sleep 2

base_lua=$(basename "$lua_test_file")
result=$(xvfb-run --auto-servernum "$1" --remote --lua:"$lua_test_image","$base_lua")
xvfb-run --auto-servernum "$1" --remote --quit

## @FIXME Running on GitHub gives additional dbind-WARNINGs. The data required is the last n lines.
result_tail=$(printf "%s" "$result" | tail --lines=7)

expected=$(printf "%s\n.%s\n%s\n%s\n%s\n%s\n%s" "$(basename "$lua_test_image")" jpeg "$lua_test_image" "$(stat -c %Y "$lua_test_image")" "$(stat -c %s "$lua_test_image")" 0 200)

printf "Result                              Expected\n"
printf '%s\n' "$result_tail" "$expected"| pr -2  --omit-header

rm "$lua_test_image"
rm "$lua_test_file"

if [ "$result_tail" = "$expected" ]
then
	exit 0
else
	exit 1
fi

