/* gEDA - GPL Electronic Design Automation
 * gschem - gEDA Schematic Capture
 * Copyright (C) 1998-2010 Ales Hvezda
 * Copyright (C) 1998-2019 gEDA Contributors (see ChangeLog for details)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
#include <config.h>
#include <version.h>

#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <glib.h>

#include "gschem.h"

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

/* These are generated by parse_commandline() */
extern SCM s_pre_load_expr;
extern SCM s_post_load_expr;

typedef struct {
  gschem_atexit_func func;
  gpointer arg;
} gschem_atexit_struct;

static GList *exit_functions = NULL;

/*! \brief Register a function to be called on program exit
 *
 *  \par Function Description
 *  This function registers a function to be called on
 *  program exit. Multiple functions will be executed in
 *  the order they are registered.
 *
 *  \param [in] func a pointer to the function to be registered
 *  \param [in] data an arbitrary argument provided to the function
 *                   when it is called
 */
void gschem_atexit(gschem_atexit_func func, gpointer data)
{
  gschem_atexit_struct *p;

  p = g_new(gschem_atexit_struct, 1);
  p->func = func;
  p->arg = data;
  exit_functions = g_list_append(exit_functions, p);
}

/*! \brief Cleanup gSchem on exit.
 *  \par Function Description
 *  This function cleans up all memory objects allocated during the
 *  gSchem runtime.
 */
void gschem_quit(void)
{
  GList *list;
  gschem_atexit_struct *p;

  /* Call all registered functions in order */
  list = exit_functions;
  while(list != NULL) {
    p = (gschem_atexit_struct *) list->data;
    p->func(p->arg);
    g_free(p);
    list = g_list_next(list);
  }
  g_list_free(exit_functions);

  x_controlfd_free ();

  s_clib_free();
  s_slib_free();
  /* o_text_freeallfonts();*/
  s_attrib_free();
  x_fam_free ();
  x_stroke_free ();
  o_undo_cleanup();
  /* s_stroke_free(); no longer needed */

  i_vars_freenames();
  i_vars_libgeda_freenames();

  /* x_window_free_head(); can't do this since it causes a
   * condition in which window_head->... is still being refered
   * after this */

  /* enable this to get more memory usage from glib */
  /* You also have to enable something in glib I think */
  /* g_mem_profile();*/


  gtk_main_quit();
}

/*! \brief Show warning dialog if certain configuration options are set.
 *
 * Some gnetlist configuration options, but not all, have been moved
 * from gnetlistrc to geda.conf in 1.9.1.  This is not compatible with
 * the way gnetlist configuration currently works, so the options have
 * been moved back to gnetlistrc.
 *
 * Chances are users which have used gEDA/gaf 1.9.1 or 1.9.2 changed
 * their gnetlist configuration to geda.conf, so their options are now
 * ignored.  Unfortunately, I don't see a way to resolve this silently
 * (short of adding geda.conf support to gnetlist just for these four
 * options).  Silently producing a broken netlist is a bad thing, so
 * warn about this issue on gschem startup.
 */
static void
warn_if_using_invalid_config ()
{
  EdaConfig *cfg = eda_config_get_context_for_file (NULL);
  EdaConfig *cfg_hierarchy_traversal =
    eda_config_get_source (cfg, "gnetlist", "traverse-hierarchy", NULL);
  EdaConfig *cfg_net_naming_priority =
    eda_config_get_source (cfg, "gnetlist", "net-naming-priority", NULL);
  EdaConfig *cfg_unnamed_netname =
    eda_config_get_source (cfg, "gnetlist", "default-net-name", NULL);
  EdaConfig *cfg_unnamed_busname =
    eda_config_get_source (cfg, "gnetlist", "default-bus-name", NULL);

  if (cfg_hierarchy_traversal == NULL && cfg_net_naming_priority == NULL &&
      cfg_unnamed_netname == NULL && cfg_unnamed_busname == NULL)
    return;

  GtkWidget *dialog;
  gchar *hierarchy_traversal_before = "", *hierarchy_traversal_after = "";
  gchar *net_naming_priority_before = "", *net_naming_priority_after = "";
  gchar *unnamed_netname_before = "", *unnamed_netname_after = "";
  gchar *unnamed_busname_before = "", *unnamed_busname_after = "";

  if (cfg_hierarchy_traversal != NULL) {
    hierarchy_traversal_before = g_strdup_printf (
      "  traverse-hierarchy (in %s)\n",
      eda_config_get_filename (cfg_hierarchy_traversal));
    hierarchy_traversal_after = g_strdup_printf (
      "  (hierarchy-traversal \"%s\")\n",
      eda_config_get_boolean (cfg, "gnetlist", "traverse-hierarchy", NULL)
        ? "enabled" : "disabled");
  }

  if (cfg_net_naming_priority != NULL) {
    gchar *str;
    net_naming_priority_before = g_strdup_printf (
      "  net-naming-priority (in %s)\n",
      eda_config_get_filename (cfg_net_naming_priority));
    str = eda_config_get_string (cfg, "gnetlist", "net-naming-priority", NULL);
    net_naming_priority_after = g_strdup_printf (
      "  (net-naming-priority \"%s\")\n",
      strcmp (str, "netname-attribute") == 0 ? "netname" : "netattrib");
    g_free (str);
  }

  if (cfg_unnamed_netname != NULL) {
    gchar *str;
    unnamed_netname_before = g_strdup_printf (
      "  default-net-name (in %s)\n",
      eda_config_get_filename (cfg_unnamed_netname));
    str = eda_config_get_string (cfg, "gnetlist", "default-net-name", NULL);
    unnamed_netname_after = g_strdup_printf (
      "  (unnamed-netname \"%s\")\n", str);
    g_free (str);
  }

  if (cfg_unnamed_busname != NULL) {
    gchar *str;
    unnamed_busname_before = g_strdup_printf (
      "  default-bus-name (in %s)\n",
      eda_config_get_filename (cfg_unnamed_busname));
    str = eda_config_get_string (cfg, "gnetlist", "default-bus-name", NULL);
    unnamed_busname_after = g_strdup_printf (
      "  (unnamed-busname \"%s\")\n", str);
    g_free (str);
  }

  dialog = gtk_message_dialog_new (
    NULL,
    GTK_DIALOG_MODAL,
    GTK_MESSAGE_WARNING,
    GTK_BUTTONS_OK,
    _("The following options are present in your configuration:\n\n%s%s%s%s\n"
      "These options were introduced in gEDA/gaf 1.9.1 as a replacement for "
      "the corresponding gnetlistrc options but were removed again in "
      "gEDA/gaf 1.10.0. The current version of gnetlist uses gnetlistrc "
      "options instead:\n\n%s%s%s%s\n"
      "Please make sure to update your configuration as the options currently "
      "set won't have the desired effect."),
    hierarchy_traversal_before, net_naming_priority_before,
    unnamed_netname_before, unnamed_busname_before,
    hierarchy_traversal_after, net_naming_priority_after,
    unnamed_netname_after, unnamed_busname_after);

  if (cfg_hierarchy_traversal != NULL) {
    g_free (hierarchy_traversal_before);
    g_free (hierarchy_traversal_after);
  }
  if (cfg_net_naming_priority != NULL) {
    g_free (net_naming_priority_before);
    g_free (net_naming_priority_after);
  }
  if (cfg_unnamed_netname != NULL) {
    g_free (unnamed_netname_before);
    g_free (unnamed_netname_after);
  }
  if (cfg_unnamed_busname != NULL) {
    g_free (unnamed_busname_before);
    g_free (unnamed_busname_after);
  }

  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}

/*! \brief Main Scheme(GUILE) program function.
 *  \par Function Description
 *  This function is the main program called from scm_boot_guile.
 *  It handles initializing all libraries and gSchem variables
 *  and passes control to the gtk main loop.
 */
void main_prog(void *closure, int argc, char *argv[])
{
  int i;
  char *cwd = NULL;
  GschemToplevel *w_current = NULL;
  TOPLEVEL *toplevel = NULL;
  char *input_str = NULL;
  int argv_index;
  GSList *filenames;
  char *filename;
  SCM scm_tmp;

#ifdef HAVE_GTHREAD
  /* Gschem isn't threaded, but some of GTK's file chooser
   * backends uses threading so we need to call g_thread_init().
   * GLib requires threading be initialised before any other GLib
   * functions are called. Do it now if its not already setup.  */
  if (!g_thread_supported ()) g_thread_init (NULL);
#endif

#if ENABLE_NLS
  /* This must be the same for all locales */
  setlocale(LC_NUMERIC, "C");
#endif

  gtk_init(&argc, &argv);

  argv_index = parse_commandline(argc, argv);
  cwd = g_get_current_dir();

  libgeda_init();

  /* create log file right away even if logging is enabled */
  s_log_init ("gschem");

  s_log_message(
                _("gEDA/gschem version %s%s.%s\n"), PREPEND_VERSION_STRING,
                PACKAGE_DOTTED_VERSION, PACKAGE_DATE_VERSION);
  s_log_message(
                _("gEDA/gschem comes with ABSOLUTELY NO WARRANTY; see COPYING for more details.\n"));
  s_log_message(
                _("This is free software, and you are welcome to redistribute it under certain\n"
                  "conditions; please see the COPYING file for more details.\n\n"));

#if defined(__MINGW32__) && defined(DEBUG)
  fprintf(stderr, _("This is the MINGW32 port.\n"));
#endif

#if DEBUG
  fprintf(stderr, _("Current locale settings: %s\n"), setlocale(LC_ALL, NULL));
#endif

  /* init global buffers */
  o_buffer_init();

  /* register guile (scheme) functions */
  g_register_funcs();
  g_init_window ();
  g_init_select ();
  g_init_hook ();
  g_init_attrib ();
  g_init_keys ();
  gschem_action_init ();
  g_init_util ();

  /* initialise color map (need to do this before reading rc files */
  x_color_init ();

  o_undo_init();

  if (s_path_sys_data () == NULL) {
    const gchar *message =
      _("You must set the GEDADATA environment variable!\n\n"
        "gschem cannot locate its data files. You must set the GEDADATA\n"
        "environment variable to point to the correct location.\n");
    GtkWidget* error_diag =
      gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR,
                              GTK_BUTTONS_OK,
                              "%s", message);
    gtk_dialog_run (GTK_DIALOG (error_diag));
    g_error ("%s", message);
  }

  scm_dynwind_begin (0);

  /* Run pre-load Scheme expressions */
  g_scm_eval_protected (s_pre_load_expr, scm_current_module ());

  /* By this point, libgeda should have setup the Guile load path, so
   * we can take advantage of that.  */
  scm_tmp = scm_sys_search_load_path (scm_from_utf8_string ("gschem.scm"));
  if (scm_is_false (scm_tmp)) {
    GtkWidget *dialog;
    GString *string;
    gchar *msg;
    dialog = gtk_message_dialog_new (
      NULL,
      GTK_DIALOG_MODAL,
      GTK_MESSAGE_ERROR,
      GTK_BUTTONS_CLOSE,
      _("Can't find Scheme initialization file \"%s\"."),
      "gschem.scm");
    string = g_string_new (NULL);
    g_string_printf (
      string,
      _("It appears the gschem data files are missing, or they are not where "
        "the gschem binary expects them to be.\n\n"
        "Are you sure you installed gschem?\n\n"
        "The following directories have been searched for \"%s\":"),
      "gschem.scm");
    for (SCM l = scm_variable_ref (scm_c_lookup ("%load-path"));
         scm_is_pair (l); l = scm_cdr (l)) {
      char *path = scm_to_utf8_string (scm_car (l));
      g_string_append_printf (string, "\n\t%s", path);
      free (path);
    }
    msg = g_string_free (string, FALSE);
    g_object_set (dialog, "secondary-text", msg, NULL);
    g_free (msg);
    gtk_window_set_title (GTK_WINDOW (dialog), _("gschem"));
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
    exit (1);
  }
  input_str = scm_to_utf8_string (scm_tmp);
  toplevel = s_toplevel_new ();
  if (g_read_file(toplevel, input_str, NULL)) {
    s_log_message(_("Read init scm file [%s]\n"), input_str);
  } else {
    /*! \todo These two messages are the same. Should be
     * integrated. */
    s_log_message(_("Failed to read init scm file [%s]\n"),
                  input_str);
  }
  free (input_str); /* M'allocated by scm_to_utf8_string() */
  scm_remember_upto_here_1 (scm_tmp);

  /* Set up default configuration */
  i_vars_init_gschem_defaults ();
  gschem_atexit (i_vars_atexit_save_user_config, NULL);

  /* Now read in RC files. */
  g_rc_parse_gtkrc();
  x_rc_parse_gschem (toplevel, NULL);

  /* Set default icon theme and make sure we can find our own icons */
  x_window_set_default_icon();
  x_window_init_icons ();

  /* At end, complete set up of window. */
  x_color_allocate();

  /* Allocate w_current */
  w_current = x_window_new (toplevel);

  g_dynwind_window (w_current);

  x_stroke_init ();
  x_fam_init ();

  filenames = NULL;

  for (i = argv_index; i < argc; i++) {

    if (g_path_is_absolute(argv[i]))
    {
      /* Path is already absolute so no need to do any concat of cwd */
      filename = g_strdup (argv[i]);
    } else {
      filename = g_build_filename (cwd, argv[i], NULL);
    }

    filenames = g_slist_prepend (filenames, filename);
    /*
     * SDB notes:  at this point the filename might be unnormalized, like
     *             /path/to/foo/../bar/baz.sch.  Bad filenames will be
     *             normalized in f_open (called by x_highlevel_open_pages).
     *             This works for Linux and MINGW32.
     */
  }

  filenames = g_slist_reverse (filenames);
  x_highlevel_open_pages (w_current, filenames, FALSE);
  g_slist_free_full (filenames, g_free);

  g_free(cwd);

  /* Create an empty page if necessary */
  if (w_current->toplevel->page_current == NULL)
    x_highlevel_new_page (w_current, NULL);


#if DEBUG
  scm_c_eval_string ("(display \"hello guile\n\")");
#endif

  /* Run post-load expressions */
  if (scm_is_false (g_scm_eval_protected (s_post_load_expr, scm_current_module ()))) {
    fprintf (stderr, _("ERROR: Failed to load or evaluate startup script.\n\n"
                       "The gschem log may contain more information.\n"));
    exit (1);
  }

  scm_dynwind_end ();

  x_controlfd_init ();

  warn_if_using_invalid_config ();

  /* enter main loop */
  gtk_main();
}

/*! \brief Main executable entrance point.
 *  \par Function Description
 *  This is the main function for gSchem. It sets up the Scheme(GUILE)
 *  environment and passes control to via scm_boot_guile to
 *  the #main_prog function.
 */
int main (int argc, char *argv[])
{

#if ENABLE_NLS
  setlocale(LC_ALL, "");
  setlocale(LC_NUMERIC, "C");
  bindtextdomain("geda-gschem", LOCALEDIR);
  textdomain("geda-gschem");
  bind_textdomain_codeset("geda-gschem", "UTF-8");
#endif

  scm_boot_guile (argc, argv, main_prog, 0);

  return 0;
}
