//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
//---------------------------------------------------------------------------
//
// GEANT4 Class file
//
// Description: Data on stopping power
//
// Author:      Anton Ivantchenko 21.04.2006
//
// Organisation:        QinetiQ Ltd, UK
// Customer:            ESA/ESTEC, NOORDWIJK
// Contract:            CSMAN-5288
//
// Modifications:
// 19.04.2010 A.Ivanchenko fixed data for G4_O, 
//                         checked all data 
// 25.04.2010 V.Ivanchenko introduce G4LPhysicsFreeVector
// 
//----------------------------------------------------------------------------
//

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "G4ASTARStopping.hh" 
#include "G4NISTStoppingData.hh" 

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4ASTARStopping::G4ASTARStopping() : nvectors(0), emin(CLHEP::keV)
{
  // 1st initialisation 
  Initialise();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4ASTARStopping::~G4ASTARStopping()
{
  if(0 < nvectors) {
    for(size_t i=0; i<nvectors; ++i) { delete sdata[i]; }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4ASTARStopping::PrintWarning(G4int i) const
{
  G4ExceptionDescription ed;
  ed << "index of data " << i << " is <0 or >= " << nvectors 
     << " request ignored!";
  G4Exception("G4ASTARStopping::PrintWarning()","em0033",JustWarning,ed);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
 
void G4ASTARStopping::Initialise()
{
  // this method may be called several times during initialisation
  G4int nmat = G4Material::GetNumberOfMaterials();
  if(nmat == (G4int)nvectors) { return; }

  // loop via material list to add extra data
  G4int j;
  for(G4int i=0; i<nmat; ++i) {
    const G4Material* mat = (*(G4Material::GetMaterialTable()))[i];

    G4bool isThere = false;  
    for(j=0; j<(G4int)nvectors; ++j) {
      if(mat == materials[j]) {
	isThere = true;
	break;
      }
    }
    if(!isThere) {
      // check list of NIST materials
      G4String mname = mat->GetName();
      for(j=0; j<74; ++j) {
        if(mname == nameNIST[j]) {
          FindData(j, mat);
          isThere = true;
          break;
	}
      }
      if(!isThere) {
	G4String form = mat->GetChemicalFormula();
	for(j=0; j<numberOfMolecula; ++j) {
	  if(form == molecularName[j]) {
	    FindData(molecularIndex[j], mat);
	    break;
	  }
	}
      }
    }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4ASTARStopping::FindData(G4int j, const G4Material* mat) 
{  
static const G4float e0[78] = { 168.8f, 204.2f, 233.8f, 259.7f, 283.f, 324.3f, 360.4f, 393.f, 422.8f, 450.5f, 476.5f, 501.f, 557.1f, 607.7f, 654.1f, 697.1f, 737.4f, 775.4f, 811.5f, 845.9f, 910.3f, 970.f, 1026.f, 1078.f, 1128.f, 1175.f, 1220.f, 1263.f, 1305.f, 1344.f, 1383.f, 1419.f, 1455.f, 1489.f, 1646.f, 1782.f, 1901.f, 2005.f, 2097.f, 2178.f, 2249.f, 2312.f, 2413.f, 2488.f, 2539.f, 2572.f, 2589.f, 2593.f, 2586.f, 2571.f, 2548.f, 2521.f, 2489.f, 2453.f, 2416.f, 2377.f, 2176.f, 1989.f, 1825.f, 1683.f, 1561.f, 1457.f, 1369.f, 1292.f, 1165.f, 1062.f, 978.5f, 908.4f, 848.8f, 797.4f, 752.4f, 712.9f, 677.7f, 646.1f, 617.7f, 591.9f, 568.4f, 547.f}; 
 
static const G4float e1[78] = { 192.2f, 229.4f, 260.2f, 287.f, 311.f, 353.1f, 389.7f, 422.5f, 452.4f, 480.1f, 506.f, 530.3f, 585.9f, 635.7f, 681.2f, 723.2f, 762.5f, 799.4f, 834.3f, 867.5f, 929.6f, 986.9f, 1040.f, 1090.f, 1137.f, 1182.f, 1224.f, 1265.f, 1304.f, 1341.f, 1376.f, 1411.f, 1444.f, 1475.f, 1619.f, 1741.f, 1847.f, 1938.f, 2017.f, 2085.f, 2144.f, 2195.f, 2275.f, 2331.f, 2367.f, 2388.f, 2397.f, 2395.f, 2386.f, 2370.f, 2350.f, 2326.f, 2299.f, 2271.f, 2241.f, 2210.f, 2056.f, 1913.f, 1783.f, 1666.f, 1562.f, 1468.f, 1384.f, 1308.f, 1177.f, 1072.f, 986.8f, 915.5f, 855.f, 802.8f, 757.2f, 717.1f, 681.4f, 649.5f, 620.7f, 594.7f, 570.9f, 549.2f}; 

static const G4float e2[78] = { 169.4f, 204.2f, 233.3f, 258.8f, 281.8f, 322.4f, 358.1f, 390.2f, 419.7f, 447.2f, 472.9f, 497.2f, 552.9f, 603.2f, 649.4f, 692.4f, 732.7f, 770.7f, 806.8f, 841.3f, 906.f, 966.1f, 1022.f, 1075.f, 1125.f, 1173.f, 1219.f, 1262.f, 1304.f, 1344.f, 1383.f, 1420.f, 1456.f, 1491.f, 1650.f, 1787.f, 1907.f, 2012.f, 2104.f, 2185.f, 2256.f, 2317.f, 2417.f, 2489.f, 2539.f, 2571.f, 2588.f, 2592.f, 2587.f, 2573.f, 2554.f, 2529.f, 2500.f, 2468.f, 2434.f, 2398.f, 2213.f, 2035.f, 1874.f, 1731.f, 1605.f, 1499.f, 1408.f, 1328.f, 1197.f, 1091.f, 1004.f, 932.2f, 870.7f, 817.8f, 771.5f, 730.8f, 694.5f, 662.1f, 632.9f, 606.4f, 582.3f, 560.2f}; 

static const G4float e3[78] = { 31.26f, 38.13f, 43.91f, 48.98f, 53.56f, 61.66f, 68.79f, 75.21f, 81.11f, 86.6f, 91.74f, 96.6f, 107.7f, 117.8f, 127.f, 135.6f, 143.6f, 151.2f, 158.4f, 165.3f, 178.2f, 190.2f, 201.4f, 212.f, 222.f, 231.6f, 240.7f, 249.5f, 257.9f, 266.f, 273.9f, 281.4f, 288.8f, 295.9f, 328.8f, 357.8f, 383.8f, 407.4f, 428.8f, 448.4f, 466.3f, 482.8f, 511.8f, 536.2f, 556.6f, 573.5f, 587.3f, 598.4f, 607.2f, 613.8f, 618.5f, 621.6f, 623.4f, 623.8f, 623.3f, 621.8f, 604.4f, 578.7f, 550.8f, 523.3f, 497.6f, 474.4f, 453.6f, 435.f, 402.7f, 375.9f, 353.2f, 333.6f, 316.9f, 302.f, 288.8f, 276.9f, 266.2f, 256.5f, 247.7f, 239.5f, 232.f, 225.1f}; 

static const G4float e4[78] = { 87.5f, 108.6f, 126.7f, 142.7f, 157.3f, 183.5f, 206.7f, 227.9f, 247.5f, 265.9f, 283.2f, 299.6f, 337.7f, 372.3f, 404.3f, 434.3f, 462.6f, 489.4f, 515.f, 539.5f, 585.8f, 629.f, 669.6f, 708.1f, 744.8f, 779.8f, 813.4f, 845.7f, 876.8f, 906.8f, 935.9f, 964.f, 991.3f, 1018.f, 1140.f, 1247.f, 1343.f, 1429.f, 1506.f, 1575.f, 1637.f, 1693.f, 1787.f, 1861.f, 1918.f, 1961.f, 1990.f, 2008.f, 2017.f, 2019.f, 2013.f, 2002.f, 1987.f, 1968.f, 1946.f, 1922.f, 1774.f, 1625.f, 1494.f, 1382.f, 1287.f, 1205.f, 1133.f, 1072.f, 968.6f, 885.9f, 818.f, 760.7f, 711.7f, 669.6f, 632.7f, 600.2f, 571.2f, 545.3f, 521.9f, 500.6f, 481.2f, 463.4f}; 

static const G4float e5[78] = { 55.8f, 71.89f, 86.05f, 98.92f, 110.9f, 132.7f, 152.5f, 170.9f, 188.2f, 204.5f, 220.1f, 235.1f, 270.1f, 302.5f, 332.8f, 361.5f, 388.6f, 414.6f, 439.4f, 463.3f, 508.4f, 550.4f, 589.6f, 626.5f, 661.1f, 693.7f, 724.3f, 753.2f, 780.3f, 805.9f, 830.1f, 852.8f, 874.3f, 894.5f, 980.f, 1045.f, 1096.f, 1136.f, 1168.f, 1195.f, 1216.f, 1234.f, 1262.f, 1280.f, 1291.f, 1297.f, 1298.f, 1297.f, 1293.f, 1286.f, 1279.f, 1269.f, 1259.f, 1248.f, 1237.f, 1225.f, 1161.f, 1098.f, 1039.f, 984.9f, 937.6f, 895.4f, 856.8f, 821.f, 756.1f, 698.6f, 647.8f, 604.8f, 567.8f, 535.7f, 507.4f, 482.4f, 460.f, 439.8f, 421.6f, 405.1f, 389.9f, 376.f}; 

static const G4float e6[78] = { 71.39f, 88.59f, 103.3f, 116.4f, 128.4f, 149.9f, 169.f, 186.5f, 202.8f, 218.f, 232.3f, 246.f, 277.7f, 306.6f, 333.4f, 358.5f, 382.1f, 404.6f, 426.f, 446.5f, 485.f, 520.9f, 554.3f, 585.8f, 615.5f, 643.5f, 670.f, 695.2f, 719.1f, 741.9f, 763.6f, 784.3f, 804.f, 822.9f, 905.9f, 974.f, 1031.f, 1080.f, 1122.f, 1159.f, 1191.f, 1220.f, 1267.f, 1304.f, 1332.f, 1353.f, 1368.f, 1378.f, 1383.f, 1385.f, 1384.f, 1381.f, 1376.f, 1370.f, 1362.f, 1352.f, 1308.f, 1253.f, 1187.f, 1115.f, 1044.f, 982.5f, 928.4f, 880.f, 801.2f, 737.f, 683.4f, 638.1f, 599.1f, 565.2f, 535.4f, 508.9f, 485.3f, 464.f, 444.8f, 427.3f, 411.4f, 396.7f}; 

static const G4float e7[78] = { 34.54f, 44.56f, 53.39f, 61.42f, 68.88f, 82.53f, 94.95f, 106.5f, 117.3f, 127.6f, 137.4f, 146.8f, 168.8f, 189.3f, 208.5f, 226.7f, 244.1f, 260.8f, 276.8f, 292.3f, 321.9f, 349.8f, 376.5f, 401.9f, 426.4f, 450.f, 472.8f, 494.9f, 516.3f, 537.1f, 557.3f, 577.f, 596.1f, 614.8f, 702.1f, 780.3f, 850.8f, 914.4f, 971.9f, 1024.f, 1070.f, 1111.f, 1181.f, 1233.f, 1272.f, 1298.f, 1314.f, 1321.f, 1321.f, 1315.f, 1305.f, 1291.f, 1276.f, 1258.f, 1240.f, 1222.f, 1130.f, 1047.f, 974.f, 911.2f, 856.7f, 809.f, 767.f, 729.8f, 666.1f, 613.2f, 569.7f, 532.7f, 501.f, 473.4f, 449.1f, 427.4f, 408.3f, 391.f, 375.4f, 361.1f, 348.1f, 336.f}; 

static const G4float e8[78] = { 11.5f, 14.55f, 17.2f, 19.57f, 21.76f, 25.72f, 29.27f, 32.54f, 35.59f, 38.46f, 41.18f, 43.78f, 49.83f, 55.39f, 60.56f, 65.44f, 70.06f, 74.46f, 78.68f, 82.74f, 90.45f, 97.69f, 104.5f, 111.1f, 117.3f, 123.3f, 129.1f, 134.6f, 140.f, 145.2f, 150.3f, 155.2f, 160.f, 164.7f, 186.4f, 205.8f, 223.5f, 239.5f, 254.2f, 267.7f, 280.1f, 291.5f, 311.6f, 328.5f, 342.6f, 354.3f, 363.8f, 371.5f, 377.5f, 382.1f, 385.5f, 387.8f, 389.1f, 389.6f, 389.4f, 388.7f, 384.4f, 375.7f, 362.6f, 346.8f, 330.5f, 316.4f, 304.f, 293.f, 274.1f, 258.4f, 245.f, 233.4f, 223.1f, 214.f, 205.8f, 198.4f, 191.6f, 185.5f, 179.7f, 174.5f, 169.6f, 165.f}; 

static const G4float e9[78] = { 138.6f, 167.7f, 192.f, 213.3f, 232.4f, 266.3f, 296.f, 322.7f, 347.2f, 369.9f, 391.2f, 411.3f, 457.3f, 498.8f, 536.8f, 572.f, 605.f, 636.1f, 665.7f, 693.8f, 746.5f, 795.4f, 841.f, 883.9f, 924.5f, 963.1f, 999.9f, 1035.f, 1069.f, 1101.f, 1133.f, 1163.f, 1192.f, 1220.f, 1348.f, 1460.f, 1557.f, 1643.f, 1719.f, 1786.f, 1845.f, 1897.f, 1983.f, 2047.f, 2093.f, 2124.f, 2142.f, 2149.f, 2148.f, 2139.f, 2125.f, 2106.f, 2084.f, 2059.f, 2031.f, 2003.f, 1861.f, 1726.f, 1600.f, 1484.f, 1379.f, 1290.f, 1214.f, 1147.f, 1035.f, 945.3f, 871.8f, 810.f, 757.7f, 712.5f, 672.9f, 638.1f, 607.f, 579.2f, 554.1f, 531.3f, 510.5f, 491.5f}; 

static const G4float e10[78] = { 146.3f, 177.1f, 202.9f, 225.4f, 245.6f, 281.3f, 312.5f, 340.6f, 366.2f, 390.f, 412.2f, 433.2f, 481.1f, 524.1f, 563.3f, 599.7f, 633.6f, 665.4f, 695.6f, 724.2f, 777.7f, 827.f, 872.7f, 915.5f, 955.7f, 993.7f, 1030.f, 1064.f, 1096.f, 1128.f, 1157.f, 1186.f, 1213.f, 1239.f, 1356.f, 1453.f, 1534.f, 1602.f, 1659.f, 1706.f, 1745.f, 1776.f, 1821.f, 1846.f, 1856.f, 1855.f, 1845.f, 1829.f, 1809.f, 1785.f, 1759.f, 1732.f, 1703.f, 1674.f, 1645.f, 1617.f, 1499.f, 1406.f, 1326.f, 1254.f, 1189.f, 1129.f, 1075.f, 1024.f, 935.3f, 858.7f, 792.9f, 737.6f, 690.f, 648.7f, 612.5f, 580.4f, 552.f, 526.4f, 503.3f, 482.4f, 463.3f, 445.8f}; 

static const G4float e11[78] = { 125.9f, 152.6f, 175.f, 194.8f, 212.5f, 244.1f, 271.8f, 296.8f, 319.8f, 341.1f, 361.2f, 380.2f, 423.8f, 463.1f, 499.3f, 533.f, 564.6f, 594.5f, 622.8f, 649.9f, 700.8f, 748.1f, 792.4f, 834.2f, 873.7f, 911.4f, 947.4f, 981.9f, 1015.f, 1047.f, 1078.f, 1107.f, 1136.f, 1164.f, 1291.f, 1401.f, 1498.f, 1584.f, 1660.f, 1727.f, 1787.f, 1839.f, 1926.f, 1991.f, 2039.f, 2072.f, 2092.f, 2103.f, 2105.f, 2100.f, 2089.f, 2075.f, 2056.f, 2035.f, 2011.f, 1986.f, 1846.f, 1705.f, 1576.f, 1460.f, 1358.f, 1271.f, 1196.f, 1130.f, 1020.f, 931.6f, 859.4f, 798.8f, 747.4f, 702.9f, 664.f, 629.7f, 599.1f, 571.7f, 547.1f, 524.7f, 504.3f, 485.6f}; 

static const G4float e12[78] = { 190.6f, 225.6f, 254.3f, 279.f, 300.9f, 339.2f, 372.1f, 401.4f, 427.9f, 452.3f, 475.f, 496.2f, 544.3f, 586.9f, 625.5f, 661.f, 693.8f, 724.6f, 753.5f, 780.8f, 831.5f, 877.9f, 920.8f, 960.6f, 998.f, 1033.f, 1066.f, 1098.f, 1128.f, 1156.f, 1184.f, 1210.f, 1235.f, 1259.f, 1366.f, 1455.f, 1531.f, 1595.f, 1650.f, 1698.f, 1738.f, 1772.f, 1825.f, 1862.f, 1886.f, 1900.f, 1906.f, 1906.f, 1901.f, 1892.f, 1880.f, 1866.f, 1850.f, 1833.f, 1814.f, 1795.f, 1684.f, 1571.f, 1466.f, 1372.f, 1288.f, 1213.f, 1145.f, 1085.f, 981.9f, 898.2f, 829.6f, 772.f, 722.9f, 680.4f, 643.1f, 610.2f, 580.8f, 554.4f, 530.6f, 509.f, 489.2f, 471.2f}; 

static const G4float e13[78] = { 192.3f, 228.9f, 259.f, 285.1f, 308.3f, 348.9f, 384.f, 415.3f, 443.7f, 469.9f, 494.2f, 517.f, 568.9f, 615.1f, 657.f, 695.5f, 731.3f, 764.8f, 796.3f, 826.2f, 881.8f, 932.7f, 979.8f, 1024.f, 1065.f, 1104.f, 1141.f, 1175.f, 1209.f, 1240.f, 1271.f, 1300.f, 1327.f, 1354.f, 1473.f, 1574.f, 1658.f, 1731.f, 1793.f, 1845.f, 1890.f, 1929.f, 1987.f, 2027.f, 2051.f, 2063.f, 2065.f, 2060.f, 2049.f, 2034.f, 2015.f, 1993.f, 1969.f, 1944.f, 1917.f, 1891.f, 1751.f, 1620.f, 1502.f, 1400.f, 1310.f, 1231.f, 1161.f, 1099.f, 994.3f, 909.f, 839.2f, 780.7f, 730.8f, 687.6f, 649.8f, 616.4f, 586.6f, 559.8f, 535.7f, 513.8f, 493.8f, 475.5f}; 

static const G4float e14[78] = { 159.8f, 195.8f, 226.f, 252.7f, 276.8f, 319.7f, 357.4f, 391.5f, 422.9f, 452.1f, 479.5f, 505.4f, 565.1f, 619.f, 668.6f, 714.8f, 758.1f, 799.1f, 838.1f, 875.3f, 945.4f, 1011.f, 1072.f, 1130.f, 1185.f, 1237.f, 1288.f, 1336.f, 1383.f, 1428.f, 1471.f, 1514.f, 1555.f, 1594.f, 1779.f, 1945.f, 2094.f, 2230.f, 2354.f, 2468.f, 2571.f, 2666.f, 2830.f, 2961.f, 3060.f, 3130.f, 3172.f, 3189.f, 3185.f, 3164.f, 3129.f, 3086.f, 3038.f, 2986.f, 2931.f, 2876.f, 2578.f, 2306.f, 2084.f, 1907.f, 1763.f, 1642.f, 1539.f, 1450.f, 1303.f, 1185.f, 1089.f, 1010.f, 941.9f, 883.6f, 832.8f, 788.1f, 748.4f, 712.9f, 680.9f, 652.f, 625.7f, 601.7f}; 

static const G4float e15[78] = { 124.1f, 148.8f, 169.3f, 187.2f, 203.3f, 231.5f, 256.1f, 278.2f, 298.4f, 317.1f, 334.5f, 351.f, 388.6f, 422.4f, 453.2f, 481.8f, 508.4f, 533.5f, 557.3f, 579.9f, 622.2f, 661.3f, 697.8f, 732.f, 764.3f, 794.9f, 824.f, 851.9f, 878.5f, 904.1f, 928.7f, 952.4f, 975.2f, 997.3f, 1097.f, 1184.f, 1259.f, 1325.f, 1384.f, 1435.f, 1481.f, 1521.f, 1588.f, 1639.f, 1678.f, 1706.f, 1726.f, 1738.f, 1744.f, 1745.f, 1743.f, 1736.f, 1727.f, 1716.f, 1703.f, 1688.f, 1603.f, 1511.f, 1421.f, 1338.f, 1261.f, 1190.f, 1126.f, 1067.f, 965.5f, 883.8f, 816.4f, 759.6f, 711.1f, 669.1f, 632.3f, 599.9f, 571.f, 545.f, 521.6f, 500.3f, 480.9f, 463.1f}; 

static const G4float e16[78] = { 93.28f, 114.2f, 131.9f, 147.5f, 161.6f, 186.6f, 208.6f, 228.5f, 246.8f, 263.8f, 279.8f, 295.f, 329.8f, 361.2f, 390.2f, 417.1f, 442.4f, 466.4f, 489.1f, 510.8f, 551.8f, 589.8f, 625.6f, 659.4f, 691.5f, 722.2f, 751.6f, 779.8f, 807.1f, 833.4f, 858.9f, 883.7f, 907.7f, 931.1f, 1040.f, 1137.f, 1225.f, 1306.f, 1380.f, 1449.f, 1513.f, 1571.f, 1676.f, 1763.f, 1834.f, 1891.f, 1933.f, 1961.f, 1977.f, 1982.f, 1978.f, 1967.f, 1952.f, 1932.f, 1911.f, 1888.f, 1752.f, 1614.f, 1490.f, 1383.f, 1290.f, 1209.f, 1139.f, 1077.f, 973.3f, 890.f, 821.5f, 764.f, 714.9f, 672.6f, 635.5f, 602.6f, 573.5f, 547.3f, 523.7f, 502.4f, 482.9f, 465.f}; 

static const G4float e17[78] = { 72.78f, 90.74f, 106.1f, 119.8f, 132.3f, 154.7f, 174.6f, 192.8f, 209.7f, 225.5f, 240.4f, 254.5f, 287.2f, 317.f, 344.6f, 370.4f, 394.6f, 417.6f, 439.5f, 460.5f, 499.9f, 536.5f, 570.8f, 603.1f, 633.6f, 662.6f, 690.1f, 716.4f, 741.5f, 765.6f, 788.6f, 810.8f, 832.1f, 852.6f, 944.4f, 1022.f, 1088.f, 1145.f, 1194.f, 1237.f, 1275.f, 1308.f, 1361.f, 1401.f, 1431.f, 1453.f, 1467.f, 1476.f, 1481.f, 1481.f, 1479.f, 1474.f, 1466.f, 1458.f, 1447.f, 1436.f, 1352.f, 1256.f, 1166.f, 1086.f, 1016.f, 954.4f, 900.2f, 852.3f, 772.1f, 708.1f, 655.5f, 610.7f, 573.3f, 540.6f, 511.9f, 486.6f, 464.f, 443.8f, 425.4f, 408.7f, 393.4f, 379.4f}; 

static const G4float e18[78] = { 99.73f, 123.7f, 144.1f, 162.3f, 178.8f, 208.3f, 234.6f, 258.5f, 280.5f, 301.2f, 320.6f, 339.1f, 381.8f, 420.6f, 456.5f, 490.f, 521.6f, 551.5f, 580.1f, 607.4f, 658.9f, 707.f, 752.2f, 795.f, 835.7f, 874.6f, 911.8f, 947.6f, 982.f, 1015.f, 1047.f, 1078.f, 1109.f, 1138.f, 1273.f, 1391.f, 1497.f, 1591.f, 1676.f, 1753.f, 1822.f, 1884.f, 1990.f, 2074.f, 2140.f, 2190.f, 2227.f, 2251.f, 2266.f, 2273.f, 2272.f, 2266.f, 2254.f, 2238.f, 2220.f, 2198.f, 2049.f, 1886.f, 1737.f, 1607.f, 1495.f, 1399.f, 1315.f, 1242.f, 1120.f, 1023.f, 942.8f, 875.6f, 818.4f, 769.2f, 726.1f, 688.f, 654.2f, 624.f, 596.7f, 572.f, 549.5f, 528.9f}; 

static const G4float e19[78] = { 157.6f, 186.4f, 209.9f, 230.3f, 248.3f, 279.7f, 306.8f, 330.8f, 352.6f, 372.6f, 391.2f, 408.7f, 448.1f, 483.2f, 514.9f, 544.1f, 571.1f, 596.4f, 620.3f, 642.8f, 684.8f, 723.2f, 758.8f, 792.f, 823.2f, 852.7f, 880.6f, 907.2f, 932.6f, 956.9f, 980.2f, 1003.f, 1024.f, 1045.f, 1139.f, 1220.f, 1290.f, 1353.f, 1408.f, 1457.f, 1502.f, 1541.f, 1608.f, 1662.f, 1704.f, 1736.f, 1761.f, 1778.f, 1789.f, 1796.f, 1798.f, 1796.f, 1790.f, 1782.f, 1772.f, 1760.f, 1674.f, 1574.f, 1476.f, 1385.f, 1302.f, 1227.f, 1160.f, 1099.f, 993.8f, 909.3f, 839.7f, 781.1f, 731.2f, 688.1f, 650.2f, 616.7f, 587.f, 560.3f, 536.2f, 514.4f, 494.5f, 476.2f}; 

static const G4float e20[78] = { 109.1f, 132.8f, 152.8f, 170.4f, 186.2f, 214.4f, 239.3f, 261.7f, 282.4f, 301.6f, 319.7f, 336.8f, 376.1f, 411.7f, 444.5f, 475.f, 503.6f, 530.7f, 556.5f, 581.1f, 627.4f, 670.4f, 710.8f, 748.9f, 785.f, 819.4f, 852.4f, 883.9f, 914.3f, 943.5f, 971.7f, 999.f, 1025.f, 1051.f, 1168.f, 1270.f, 1360.f, 1440.f, 1511.f, 1574.f, 1630.f, 1680.f, 1763.f, 1826.f, 1872.f, 1905.f, 1927.f, 1939.f, 1943.f, 1941.f, 1933.f, 1921.f, 1905.f, 1887.f, 1866.f, 1844.f, 1720.f, 1593.f, 1475.f, 1369.f, 1274.f, 1193.f, 1123.f, 1062.f, 959.2f, 877.1f, 809.7f, 753.f, 704.9f, 663.4f, 627.f, 594.9f, 566.3f, 540.6f, 517.5f, 496.5f, 477.4f, 459.8f}; 

static const G4float e21[78] = { 40.6f, 48.54f, 55.16f, 60.95f, 66.15f, 75.33f, 83.38f, 90.64f, 97.29f, 103.5f, 109.3f, 114.8f, 127.3f, 138.7f, 149.2f, 158.9f, 168.f, 176.7f, 184.9f, 192.8f, 207.6f, 221.3f, 234.2f, 246.4f, 258.f, 269.1f, 279.7f, 289.9f, 299.7f, 309.2f, 318.3f, 327.1f, 335.7f, 344.f, 382.3f, 415.8f, 445.6f, 472.1f, 495.7f, 516.7f, 535.4f, 551.8f, 578.9f, 599.2f, 613.8f, 623.8f, 629.9f, 633.f, 633.6f, 632.3f, 629.5f, 625.5f, 620.6f, 615.1f, 609.1f, 602.8f, 567.7f, 533.f, 501.8f, 474.4f, 450.3f, 429.f, 410.1f, 393.3f, 364.6f, 340.9f, 321.f, 304.f, 289.2f, 275.9f, 264.1f, 253.6f, 243.9f, 235.f, 226.7f, 218.9f, 211.9f, 205.4f}; 

static const G4float e22[78] = { 29.51f, 36.49f, 42.42f, 47.68f, 52.45f, 60.98f, 68.54f, 75.4f, 81.74f, 87.65f, 93.23f, 98.51f, 110.7f, 121.8f, 132.f, 141.5f, 150.5f, 159.f, 167.2f, 174.9f, 189.5f, 203.2f, 216.f, 228.1f, 239.7f, 250.7f, 261.3f, 271.4f, 281.2f, 290.7f, 299.8f, 308.7f, 317.3f, 325.6f, 364.2f, 398.4f, 429.1f, 456.8f, 482.1f, 505.1f, 526.2f, 545.4f, 579.f, 607.f, 630.1f, 649.1f, 664.4f, 676.6f, 686.2f, 693.4f, 698.8f, 702.4f, 704.6f, 705.7f, 705.7f, 704.9f, 692.5f, 672.6f, 650.3f, 627.6f, 606.f, 585.6f, 566.3f, 548.f, 514.f, 483.4f, 455.8f, 431.f, 408.4f, 388.f, 369.6f, 353.7f, 339.1f, 325.7f, 313.4f, 302.f, 291.7f, 282.3f}; 

static const G4float e23[78] = { 54.06f, 65.42f, 74.91f, 83.2f, 90.66f, 103.8f, 115.3f, 125.6f, 135.1f, 143.8f, 152.f, 159.8f, 177.4f, 193.3f, 207.8f, 221.3f, 233.9f, 245.7f, 256.9f, 267.6f, 287.7f, 306.2f, 323.5f, 339.8f, 355.2f, 369.8f, 383.8f, 397.1f, 410.f, 422.3f, 434.2f, 445.7f, 456.9f, 467.6f, 517.1f, 560.6f, 599.4f, 634.3f, 665.9f, 694.6f, 720.8f, 744.8f, 786.6f, 821.5f, 850.4f, 874.1f, 893.1f, 908.1f, 919.6f, 928.f, 933.6f, 936.8f, 938.f, 937.3f, 935.1f, 931.5f, 899.9f, 856.7f, 810.9f, 766.9f, 727.3f, 691.9f, 660.2f, 631.5f, 581.8f, 540.1f, 504.6f, 473.9f, 447.2f, 423.6f, 402.6f, 383.5f, 366.9f, 352.f, 338.3f, 325.7f, 314.1f, 303.6f}; 

static const G4float e24[78] = { 99.58f, 123.5f, 144.f, 162.1f, 178.6f, 208.1f, 234.3f, 258.2f, 280.2f, 300.8f, 320.3f, 338.7f, 381.4f, 420.2f, 456.f, 489.5f, 521.f, 551.f, 579.5f, 606.8f, 658.3f, 706.3f, 751.5f, 794.2f, 834.9f, 873.7f, 910.9f, 946.6f, 981.f, 1014.f, 1046.f, 1077.f, 1108.f, 1137.f, 1271.f, 1390.f, 1495.f, 1590.f, 1675.f, 1751.f, 1820.f, 1882.f, 1988.f, 2072.f, 2138.f, 2188.f, 2225.f, 2250.f, 2265.f, 2271.f, 2271.f, 2264.f, 2253.f, 2237.f, 2218.f, 2197.f, 2048.f, 1886.f, 1738.f, 1609.f, 1497.f, 1401.f, 1317.f, 1244.f, 1122.f, 1025.f, 944.6f, 877.4f, 820.f, 770.7f, 727.5f, 689.3f, 655.4f, 625.2f, 597.8f, 573.1f, 550.5f, 529.9f}; 

static const G4float e25[78] = { 38.11f, 45.4f, 51.41f, 56.61f, 61.25f, 69.34f, 76.36f, 82.61f, 88.29f, 93.53f, 98.41f, 103.f, 113.4f, 122.7f, 131.1f, 138.9f, 146.1f, 152.9f, 159.3f, 165.4f, 176.7f, 187.1f, 196.8f, 205.9f, 214.4f, 222.5f, 230.3f, 237.6f, 244.7f, 251.5f, 258.f, 264.3f, 270.3f, 276.2f, 303.1f, 326.6f, 347.4f, 366.2f, 383.1f, 398.5f, 412.5f, 425.4f, 447.8f, 466.5f, 482.1f, 494.8f, 505.1f, 513.2f, 519.4f, 523.9f, 526.9f, 528.6f, 529.1f, 528.6f, 527.3f, 525.2f, 505.6f, 481.f, 456.8f, 434.6f, 414.6f, 396.6f, 380.5f, 366.f, 340.9f, 319.7f, 301.6f, 285.8f, 272.f, 259.8f, 248.8f, 239.f, 230.1f, 222.f, 214.6f, 207.7f, 201.4f, 195.6f}; 

static const G4float e26[78] = { 47.65f, 57.57f, 65.83f, 73.05f, 79.53f, 90.94f, 100.9f, 109.9f, 118.f, 125.6f, 132.7f, 139.4f, 154.6f, 168.3f, 180.8f, 192.3f, 203.1f, 213.3f, 222.9f, 232.f, 249.1f, 264.8f, 279.5f, 293.2f, 306.1f, 318.4f, 330.f, 341.1f, 351.7f, 361.8f, 371.5f, 380.8f, 389.8f, 398.4f, 437.2f, 469.9f, 497.8f, 521.7f, 542.3f, 560.1f, 575.5f, 588.9f, 610.7f, 627.4f, 640.1f, 649.9f, 657.3f, 662.8f, 666.8f, 669.6f, 671.4f, 672.3f, 672.4f, 671.9f, 670.9f, 669.4f, 656.9f, 639.5f, 619.8f, 599.4f, 578.4f, 557.6f, 537.5f, 518.5f, 483.6f, 452.9f, 425.8f, 401.8f, 380.5f, 361.5f, 344.3f, 328.6f, 315.1f, 303.1f, 292.1f, 282.f, 272.6f, 263.8f}; 

static const G4float e27[78] = { 83.85f, 102.1f, 117.5f, 131.f, 143.3f, 165.f, 184.1f, 201.3f, 217.2f, 232.f, 245.9f, 259.f, 289.2f, 316.5f, 341.5f, 364.8f, 386.7f, 407.4f, 427.1f, 445.8f, 481.1f, 513.8f, 544.4f, 573.2f, 600.6f, 626.6f, 651.5f, 675.3f, 698.1f, 720.1f, 741.3f, 761.7f, 781.5f, 800.6f, 888.f, 963.9f, 1031.f, 1090.f, 1142.f, 1189.f, 1231.f, 1268.f, 1330.f, 1378.f, 1414.f, 1441.f, 1460.f, 1473.f, 1481.f, 1485.f, 1484.f, 1481.f, 1475.f, 1466.f, 1456.f, 1445.f, 1389.f, 1323.f, 1249.f, 1171.f, 1095.f, 1029.f, 971.4f, 920.7f, 835.9f, 768.f, 711.5f, 663.8f, 622.8f, 587.4f, 556.1f, 528.4f, 503.7f, 481.5f, 461.4f, 443.2f, 426.5f, 411.2f}; 

static const G4float e28[78] = { 212.f, 275.4f, 331.7f, 383.f, 430.9f, 518.8f, 599.2f, 674.f, 744.5f, 811.5f, 875.6f, 937.1f, 1082.f, 1217.f, 1344.f, 1465.f, 1580.f, 1691.f, 1798.f, 1901.f, 2098.f, 2285.f, 2463.f, 2634.f, 2798.f, 2956.f, 3108.f, 3256.f, 3399.f, 3538.f, 3674.f, 3805.f, 3933.f, 4058.f, 4637.f, 5152.f, 5610.f, 6019.f, 6383.f, 6705.f, 6988.f, 7235.f, 7632.f, 7915.f, 8100.f, 8203.f, 8240.f, 8222.f, 8161.f, 8067.f, 7947.f, 7808.f, 7655.f, 7494.f, 7327.f, 7157.f, 6345.f, 5647.f, 5068.f, 4588.f, 4188.f, 3860.f, 3585.f, 3351.f, 2971.f, 2675.f, 2438.f, 2242.f, 2078.f, 1939.f, 1818.f, 1713.f, 1620.f, 1538.f, 1464.f, 1398.f, 1337.f, 1283.f}; 

static const G4float e29[78] = { 87.26f, 110.8f, 131.3f, 149.8f, 166.8f, 197.7f, 225.5f, 251.1f, 275.f, 297.5f, 318.9f, 339.4f, 387.f, 430.9f, 471.8f, 510.4f, 547.f, 581.9f, 615.3f, 647.5f, 708.7f, 766.1f, 820.5f, 872.3f, 921.8f, 969.3f, 1015.f, 1059.f, 1102.f, 1143.f, 1183.f, 1221.f, 1259.f, 1296.f, 1465.f, 1614.f, 1748.f, 1867.f, 1975.f, 2071.f, 2156.f, 2233.f, 2361.f, 2460.f, 2535.f, 2588.f, 2624.f, 2644.f, 2652.f, 2650.f, 2639.f, 2621.f, 2598.f, 2571.f, 2540.f, 2507.f, 2323.f, 2139.f, 1976.f, 1832.f, 1706.f, 1593.f, 1493.f, 1402.f, 1249.f, 1130.f, 1033.f, 953.8f, 886.8f, 829.5f, 779.8f, 736.4f, 697.9f, 663.7f, 632.9f, 605.2f, 580.f, 557.f}; 

static const G4float e30[78] = { 165.5f, 197.6f, 224.2f, 247.2f, 267.8f, 304.f, 335.4f, 363.5f, 389.1f, 412.8f, 434.8f, 455.6f, 502.9f, 545.2f, 583.8f, 619.4f, 652.6f, 683.8f, 713.3f, 741.2f, 793.5f, 841.6f, 886.3f, 928.2f, 967.7f, 1005.f, 1041.f, 1074.f, 1107.f, 1138.f, 1167.f, 1196.f, 1223.f, 1250.f, 1369.f, 1471.f, 1560.f, 1636.f, 1703.f, 1762.f, 1812.f, 1857.f, 1928.f, 1981.f, 2018.f, 2042.f, 2056.f, 2062.f, 2061.f, 2054.f, 2043.f, 2028.f, 2010.f, 1990.f, 1969.f, 1946.f, 1819.f, 1688.f, 1565.f, 1451.f, 1350.f, 1264.f, 1190.f, 1125.f, 1017.f, 929.8f, 858.3f, 798.2f, 747.f, 702.8f, 664.f, 629.8f, 599.3f, 571.9f, 547.2f, 524.8f, 504.4f, 485.7f}; 

static const G4float e31[78] = { 10.15f, 13.69f, 16.93f, 19.96f, 22.84f, 28.23f, 33.29f, 38.08f, 42.66f, 47.08f, 51.35f, 55.5f, 65.43f, 74.84f, 83.85f, 92.52f, 100.9f, 109.1f, 117.f, 124.7f, 139.7f, 154.1f, 168.f, 181.5f, 194.6f, 207.4f, 219.9f, 232.1f, 244.f, 255.7f, 267.2f, 278.4f, 289.5f, 300.3f, 351.9f, 399.6f, 443.7f, 484.4f, 522.1f, 556.7f, 588.4f, 617.2f, 666.6f, 705.7f, 735.5f, 757.f, 771.4f, 779.9f, 783.4f, 783.f, 779.4f, 773.6f, 765.9f, 757.f, 747.2f, 736.9f, 683.7f, 635.6f, 595.2f, 561.5f, 533.f, 508.6f, 487.3f, 468.5f, 439.9f, 417.9f, 398.8f, 381.4f, 364.9f, 349.2f, 334.2f, 319.8f, 306.2f, 293.7f, 282.8f, 272.8f, 263.6f, 255.2f}; 

static const G4float e32[78] = { 132.7f, 159.4f, 181.6f, 200.9f, 218.2f, 248.6f, 275.f, 298.7f, 320.4f, 340.4f, 359.1f, 376.6f, 416.8f, 452.7f, 485.5f, 515.8f, 544.f, 570.6f, 595.8f, 619.7f, 664.3f, 705.6f, 743.9f, 779.9f, 813.9f, 846.1f, 876.8f, 906.1f, 934.2f, 961.1f, 987.f, 1012.f, 1036.f, 1059.f, 1166.f, 1258.f, 1338.f, 1410.f, 1474.f, 1531.f, 1583.f, 1628.f, 1706.f, 1767.f, 1814.f, 1849.f, 1874.f, 1891.f, 1899.f, 1902.f, 1899.f, 1891.f, 1875.f, 1852.f, 1825.f, 1795.f, 1638.f, 1494.f, 1375.f, 1276.f, 1193.f, 1121.f, 1058.f, 1002.f, 908.8f, 833.2f, 770.5f, 717.6f, 672.4f, 633.2f, 598.8f, 568.3f, 541.2f, 516.9f, 494.8f, 474.9f, 456.6f, 439.9f}; 

static const G4float e33[78] = { 118.7f, 142.6f, 162.5f, 179.7f, 195.2f, 222.4f, 246.f, 267.2f, 286.5f, 304.3f, 321.f, 336.6f, 372.4f, 404.4f, 433.5f, 460.4f, 485.5f, 509.1f, 531.4f, 552.6f, 592.1f, 628.5f, 662.4f, 694.1f, 724.1f, 752.4f, 779.4f, 805.1f, 829.6f, 853.2f, 875.9f, 897.6f, 918.7f, 938.9f, 1031.f, 1110.f, 1179.f, 1240.f, 1294.f, 1341.f, 1383.f, 1421.f, 1483.f, 1530.f, 1566.f, 1592.f, 1610.f, 1621.f, 1627.f, 1627.f, 1624.f, 1618.f, 1609.f, 1598.f, 1585.f, 1571.f, 1483.f, 1387.f, 1296.f, 1214.f, 1140.f, 1074.f, 1015.f, 962.2f, 872.f, 799.7f, 739.5f, 688.5f, 644.8f, 606.9f, 573.7f, 544.4f, 518.3f, 494.9f, 473.6f, 454.3f, 436.7f, 420.6f}; 

static const G4float e34[78] = { 173.6f, 209.f, 238.5f, 264.3f, 287.6f, 328.7f, 364.7f, 397.1f, 426.9f, 454.5f, 480.4f, 504.9f, 561.f, 611.6f, 658.f, 701.1f, 741.6f, 779.8f, 816.f, 850.6f, 915.4f, 975.6f, 1032.f, 1085.f, 1135.f, 1182.f, 1228.f, 1271.f, 1313.f, 1353.f, 1392.f, 1429.f, 1465.f, 1499.f, 1657.f, 1792.f, 1910.f, 2012.f, 2101.f, 2179.f, 2247.f, 2305.f, 2398.f, 2463.f, 2507.f, 2533.f, 2544.f, 2544.f, 2534.f, 2517.f, 2494.f, 2468.f, 2437.f, 2405.f, 2371.f, 2335.f, 2154.f, 1981.f, 1825.f, 1686.f, 1564.f, 1460.f, 1372.f, 1295.f, 1167.f, 1064.f, 980.4f, 910.2f, 850.4f, 798.9f, 753.8f, 714.2f, 678.9f, 647.3f, 618.8f, 593.f, 569.4f, 547.9f}; 

static const G4float e35[78] = { 163.5f, 196.6f, 224.1f, 248.2f, 269.8f, 308.f, 341.4f, 371.4f, 398.9f, 424.4f, 448.3f, 470.9f, 522.5f, 569.f, 611.6f, 651.1f, 688.f, 722.9f, 756.f, 787.5f, 846.6f, 901.2f, 952.3f, 1000.f, 1046.f, 1089.f, 1130.f, 1169.f, 1207.f, 1243.f, 1278.f, 1311.f, 1344.f, 1375.f, 1516.f, 1638.f, 1744.f, 1836.f, 1916.f, 1986.f, 2048.f, 2101.f, 2185.f, 2247.f, 2288.f, 2314.f, 2327.f, 2330.f, 2325.f, 2312.f, 2295.f, 2273.f, 2249.f, 2222.f, 2193.f, 2163.f, 2002.f, 1845.f, 1703.f, 1575.f, 1463.f, 1368.f, 1286.f, 1215.f, 1097.f, 1001.f, 923.2f, 857.7f, 802.f, 753.9f, 711.7f, 674.6f, 641.6f, 611.9f, 585.2f, 561.f, 539.f, 518.8f}; 

static const G4float e36[78] = { 197.5f, 241.8f, 279.3f, 312.2f, 342.f, 394.9f, 441.5f, 483.7f, 522.4f, 558.5f, 592.4f, 624.4f, 698.1f, 764.8f, 826.f, 883.1f, 936.6f, 987.3f, 1035.f, 1081.f, 1168.f, 1249.f, 1324.f, 1396.f, 1464.f, 1529.f, 1591.f, 1650.f, 1708.f, 1764.f, 1818.f, 1870.f, 1920.f, 1970.f, 2198.f, 2402.f, 2586.f, 2754.f, 2907.f, 3047.f, 3175.f, 3291.f, 3492.f, 3652.f, 3773.f, 3856.f, 3905.f, 3921.f, 3899.f, 3849.f, 3783.f, 3708.f, 3627.f, 3544.f, 3460.f, 3378.f, 2998.f, 2685.f, 2432.f, 2226.f, 2055.f, 1911.f, 1789.f, 1683.f, 1508.f, 1370.f, 1257.f, 1163.f, 1084.f, 1015.f, 955.9f, 903.7f, 857.5f, 816.2f, 779.1f, 745.5f, 714.9f, 687.1f}; 

static const G4float e37[78] = { 58.22f, 68.98f, 77.79f, 85.4f, 92.16f, 103.9f, 114.1f, 123.1f, 131.3f, 138.8f, 145.8f, 152.4f, 167.3f, 180.5f, 192.5f, 203.5f, 213.8f, 223.4f, 232.4f, 241.f, 256.9f, 271.6f, 285.1f, 297.8f, 309.8f, 321.1f, 331.8f, 342.1f, 351.9f, 361.3f, 370.3f, 379.f, 387.4f, 395.5f, 432.4f, 464.5f, 492.9f, 518.2f, 541.f, 561.6f, 580.4f, 597.4f, 626.9f, 651.2f, 671.2f, 687.3f, 700.1f, 710.f, 717.3f, 722.3f, 725.4f, 726.7f, 726.5f, 725.f, 722.3f, 718.8f, 690.9f, 655.f, 617.8f, 582.6f, 552.1f, 525.7f, 502.3f, 481.5f, 445.5f, 415.5f, 389.9f, 367.7f, 348.1f, 330.6f, 314.8f, 300.9f, 288.6f, 277.3f, 266.9f, 257.6f, 249.f, 241.2f}; 

static const G4float e38[78] = { 111.7f, 137.5f, 159.4f, 178.8f, 196.5f, 227.9f, 255.7f, 281.f, 304.4f, 326.2f, 346.7f, 366.2f, 411.2f, 452.f, 489.7f, 525.f, 558.1f, 589.5f, 619.5f, 648.1f, 702.2f, 752.5f, 799.9f, 844.6f, 887.2f, 927.9f, 966.8f, 1004.f, 1040.f, 1075.f, 1108.f, 1141.f, 1172.f, 1203.f, 1343.f, 1467.f, 1576.f, 1674.f, 1762.f, 1841.f, 1912.f, 1975.f, 2081.f, 2165.f, 2230.f, 2277.f, 2310.f, 2331.f, 2342.f, 2344.f, 2339.f, 2329.f, 2313.f, 2293.f, 2270.f, 2244.f, 2074.f, 1896.f, 1739.f, 1607.f, 1494.f, 1398.f, 1314.f, 1241.f, 1120.f, 1022.f, 942.4f, 875.3f, 818.1f, 768.9f, 725.8f, 687.7f, 653.9f, 623.7f, 596.4f, 571.7f, 549.2f, 528.6f}; 

static const G4float e39[78] = { 108.2f, 133.5f, 155.f, 174.1f, 191.4f, 222.4f, 249.8f, 274.8f, 297.8f, 319.3f, 339.6f, 358.9f, 403.3f, 443.7f, 481.f, 515.9f, 548.7f, 579.8f, 609.5f, 637.9f, 691.4f, 741.4f, 788.3f, 832.7f, 875.f, 915.3f, 953.9f, 991.1f, 1027.f, 1061.f, 1095.f, 1127.f, 1158.f, 1189.f, 1328.f, 1451.f, 1560.f, 1658.f, 1745.f, 1824.f, 1895.f, 1958.f, 2065.f, 2150.f, 2215.f, 2263.f, 2298.f, 2320.f, 2332.f, 2335.f, 2331.f, 2321.f, 2306.f, 2287.f, 2265.f, 2240.f, 2075.f, 1901.f, 1746.f, 1613.f, 1501.f, 1404.f, 1320.f, 1247.f, 1125.f, 1027.f, 946.2f, 878.8f, 821.4f, 772.f, 728.6f, 690.4f, 656.4f, 626.1f, 598.7f, 573.9f, 551.3f, 530.6f}; 

static const G4float e40[78] = { 104.5f, 129.2f, 150.2f, 168.8f, 185.8f, 216.1f, 242.9f, 267.4f, 290.f, 311.1f, 331.f, 349.8f, 393.5f, 433.1f, 469.7f, 503.9f, 536.2f, 566.7f, 595.9f, 623.7f, 676.3f, 725.4f, 771.5f, 815.1f, 856.6f, 896.2f, 934.1f, 970.6f, 1006.f, 1040.f, 1072.f, 1104.f, 1135.f, 1165.f, 1302.f, 1422.f, 1530.f, 1626.f, 1712.f, 1790.f, 1860.f, 1922.f, 2029.f, 2113.f, 2178.f, 2228.f, 2263.f, 2286.f, 2299.f, 2304.f, 2302.f, 2293.f, 2280.f, 2262.f, 2242.f, 2218.f, 2059.f, 1888.f, 1735.f, 1604.f, 1492.f, 1396.f, 1312.f, 1240.f, 1118.f, 1021.f, 941.1f, 874.1f, 817.f, 767.9f, 724.9f, 686.8f, 653.1f, 622.9f, 595.7f, 571.f, 548.6f, 528.f}; 

static const G4float e41[78] = { 104.7f, 129.4f, 150.5f, 169.2f, 186.2f, 216.5f, 243.4f, 267.9f, 290.5f, 311.7f, 331.6f, 350.5f, 394.2f, 433.9f, 470.6f, 504.9f, 537.2f, 567.8f, 596.9f, 624.9f, 677.5f, 726.7f, 772.8f, 816.5f, 858.1f, 897.8f, 935.8f, 972.3f, 1007.f, 1041.f, 1074.f, 1106.f, 1137.f, 1167.f, 1304.f, 1425.f, 1532.f, 1629.f, 1715.f, 1793.f, 1863.f, 1925.f, 2032.f, 2116.f, 2181.f, 2230.f, 2265.f, 2289.f, 2302.f, 2306.f, 2304.f, 2295.f, 2282.f, 2264.f, 2244.f, 2220.f, 2063.f, 1894.f, 1742.f, 1611.f, 1498.f, 1401.f, 1318.f, 1244.f, 1123.f, 1025.f, 944.5f, 877.2f, 819.9f, 770.6f, 727.3f, 689.2f, 655.3f, 625.f, 597.7f, 572.9f, 550.4f, 529.7f}; 

static const G4float e42[78] = { 82.8f, 103.7f, 121.6f, 137.7f, 152.3f, 178.7f, 202.3f, 223.8f, 243.8f, 262.5f, 280.2f, 297.1f, 336.2f, 372.f, 405.1f, 436.2f, 465.5f, 493.4f, 520.1f, 545.7f, 594.1f, 639.3f, 682.f, 722.5f, 761.f, 797.9f, 833.4f, 867.4f, 900.3f, 932.f, 962.8f, 992.5f, 1021.f, 1049.f, 1179.f, 1293.f, 1395.f, 1486.f, 1567.f, 1640.f, 1706.f, 1765.f, 1864.f, 1940.f, 1999.f, 2041.f, 2069.f, 2086.f, 2093.f, 2091.f, 2083.f, 2069.f, 2051.f, 2030.f, 2005.f, 1978.f, 1819.f, 1660.f, 1522.f, 1406.f, 1308.f, 1224.f, 1151.f, 1088.f, 982.9f, 898.5f, 829.5f, 771.1f, 721.3f, 678.5f, 641.f, 608.f, 578.6f, 552.3f, 528.5f, 506.9f, 487.2f, 469.1f}; 

static const G4float e43[78] = { 35.86f, 44.19f, 51.24f, 57.48f, 63.13f, 73.21f, 82.12f, 90.2f, 97.64f, 104.6f, 111.1f, 117.3f, 131.6f, 144.5f, 156.4f, 167.6f, 178.f, 187.9f, 197.3f, 206.3f, 223.3f, 239.1f, 253.9f, 268.f, 281.3f, 294.f, 306.2f, 317.9f, 329.2f, 340.1f, 350.6f, 360.8f, 370.6f, 380.2f, 424.5f, 463.5f, 498.4f, 529.9f, 558.2f, 583.8f, 607.f, 627.8f, 663.4f, 691.6f, 713.4f, 729.6f, 740.9f, 748.1f, 751.7f, 752.4f, 750.7f, 747.f, 741.7f, 735.1f, 727.6f, 719.5f, 664.4f, 611.2f, 569.4f, 538.3f, 514.f, 491.6f, 472.1f, 454.f, 422.9f, 393.7f, 370.2f, 349.6f, 331.4f, 315.3f, 301.2f, 288.5f, 276.8f, 266.1f, 256.3f, 247.1f, 238.9f, 231.5f}; 

static const G4float e44[78] = { 68.73f, 83.7f, 96.26f, 107.3f, 117.2f, 134.8f, 150.3f, 164.2f, 177.f, 188.8f, 199.9f, 210.4f, 234.5f, 256.2f, 276.1f, 294.6f, 311.9f, 328.3f, 343.8f, 358.6f, 386.3f, 412.f, 436.1f, 458.8f, 480.3f, 500.8f, 520.3f, 539.1f, 557.1f, 574.5f, 591.2f, 607.5f, 623.2f, 638.4f, 708.4f, 770.3f, 825.6f, 875.7f, 921.2f, 962.8f, 1001.f, 1036.f, 1098.f, 1150.f, 1194.f, 1230.f, 1260.f, 1285.f, 1304.f, 1319.f, 1330.f, 1338.f, 1343.f, 1344.f, 1344.f, 1341.f, 1296.f, 1229.f, 1160.f, 1095.f, 1035.f, 981.4f, 932.6f, 888.5f, 812.1f, 748.3f, 695.6f, 650.4f, 611.3f, 577.f, 546.8f, 520.f, 496.f, 474.4f, 454.8f, 437.f, 420.7f, 405.7f}; 

static const G4float e45[78] = { 167.4f, 201.6f, 230.3f, 255.4f, 278.f, 317.9f, 352.9f, 384.4f, 413.4f, 440.3f, 465.5f, 489.3f, 544.f, 593.3f, 638.6f, 680.6f, 720.1f, 757.3f, 792.7f, 826.4f, 889.7f, 948.5f, 1003.f, 1055.f, 1104.f, 1151.f, 1195.f, 1238.f, 1279.f, 1318.f, 1356.f, 1392.f, 1427.f, 1461.f, 1617.f, 1750.f, 1867.f, 1969.f, 2059.f, 2138.f, 2207.f, 2267.f, 2363.f, 2433.f, 2482.f, 2512.f, 2527.f, 2531.f, 2525.f, 2511.f, 2491.f, 2467.f, 2439.f, 2408.f, 2375.f, 2340.f, 2160.f, 1987.f, 1830.f, 1690.f, 1567.f, 1463.f, 1374.f, 1297.f, 1169.f, 1066.f, 981.3f, 910.8f, 850.9f, 799.3f, 754.1f, 714.4f, 679.f, 647.4f, 618.9f, 593.f, 569.5f, 547.9f}; 

static const G4float e46[78] = { 1.058E+02f, 1.283E+02f, 1.471E+02f, 1.636E+02f, 1.784E+02f, 2.046E+02f, 2.275E+02f, 2.481E+02f, 2.670E+02f, 2.845E+02f, 3.009E+02f, 3.163E+02f, 3.518E+02f, 3.836E+02f, 4.127E+02f, 4.398E+02f, 4.650E+02f, 4.889E+02f, 5.115E+02f, 5.330E+02f, 5.733E+02f, 6.106E+02f, 6.455E+02f, 6.783E+02f, 7.094E+02f, 7.389E+02f, 7.671E+02f, 7.941E+02f, 8.200E+02f, 8.450E+02f, 8.690E+02f, 8.923E+02f, 9.147E+02f, 9.365E+02f, 1.036E+03f, 1.124E+03f, 1.202E+03f, 1.272E+03f, 1.335E+03f, 1.392E+03f, 1.444E+03f, 1.491E+03f, 1.573E+03f, 1.639E+03f, 1.693E+03f, 1.735E+03f, 1.768E+03f, 1.792E+03f, 1.808E+03f, 1.818E+03f, 1.821E+03f, 1.820E+03f, 1.814E+03f, 1.805E+03f, 1.792E+03f, 1.777E+03f, 1.667E+03f, 1.544E+03f, 1.430E+03f, 1.329E+03f, 1.241E+03f, 1.164E+03f, 1.097E+03f, 1.038E+03f, 9.395E+02f, 8.602E+02f, 7.947E+02f, 7.395E+02f, 6.924E+02f, 6.519E+02f, 6.161E+02f, 5.845E+02f, 5.564E+02f, 5.313E+02f, 5.086E+02f, 4.880E+02f, 4.692E+02f, 4.520E+02};

static const G4float e47[78] = { 193.8f, 233.f, 265.8f, 294.4f, 320.3f, 365.9f, 405.9f, 441.9f, 475.f, 505.7f, 534.5f, 561.7f, 624.2f, 680.6f, 732.3f, 780.4f, 825.5f, 868.2f, 908.6f, 947.2f, 1020.f, 1087.f, 1150.f, 1209.f, 1265.f, 1319.f, 1370.f, 1419.f, 1465.f, 1510.f, 1554.f, 1595.f, 1636.f, 1675.f, 1852.f, 2004.f, 2137.f, 2253.f, 2354.f, 2442.f, 2518.f, 2584.f, 2688.f, 2762.f, 2810.f, 2837.f, 2848.f, 2845.f, 2831.f, 2810.f, 2782.f, 2749.f, 2713.f, 2674.f, 2633.f, 2592.f, 2381.f, 2184.f, 2007.f, 1851.f, 1714.f, 1598.f, 1500.f, 1414.f, 1272.f, 1158.f, 1066.f, 988.4f, 922.7f, 866.1f, 816.7f, 773.2f, 734.5f, 699.9f, 668.8f, 640.6f, 614.9f, 591.4f}; 

static const G4float e48[78] = { 17.96f, 22.18f, 25.76f, 28.92f, 31.8f, 36.93f, 41.47f, 45.6f, 49.4f, 52.95f, 56.3f, 59.46f, 66.77f, 73.41f, 79.52f, 85.23f, 90.6f, 95.69f, 100.5f, 105.2f, 113.9f, 122.f, 129.7f, 136.9f, 143.8f, 150.3f, 156.6f, 162.7f, 168.5f, 174.1f, 179.5f, 184.8f, 189.9f, 194.9f, 217.7f, 237.9f, 256.f, 272.3f, 287.f, 300.4f, 312.6f, 323.6f, 342.6f, 358.2f, 370.7f, 380.6f, 388.3f, 394.2f, 398.4f, 401.3f, 403.1f, 404.f, 404.1f, 403.5f, 402.4f, 401.f, 392.5f, 380.4f, 365.3f, 348.6f, 332.f, 317.6f, 304.9f, 293.7f, 274.4f, 258.3f, 244.7f, 232.9f, 222.5f, 213.2f, 205.f, 197.5f, 190.7f, 184.4f, 178.7f, 173.4f, 168.4f, 163.9f}; 

static const G4float e49[78] = { 46.69f, 57.18f, 66.06f, 73.91f, 81.04f, 93.75f, 105.f, 115.2f, 124.7f, 133.5f, 141.8f, 149.6f, 167.8f, 184.3f, 199.6f, 213.8f, 227.3f, 240.f, 252.1f, 263.8f, 285.7f, 306.2f, 325.4f, 343.6f, 361.f, 377.5f, 393.4f, 408.7f, 423.3f, 437.5f, 451.2f, 464.5f, 477.3f, 489.8f, 547.f, 597.3f, 641.7f, 681.2f, 716.3f, 747.6f, 775.5f, 800.2f, 841.5f, 873.5f, 897.8f, 915.8f, 928.6f, 937.1f, 940.f, 938.4f, 933.8f, 927.f, 918.6f, 909.2f, 899.1f, 888.5f, 833.7f, 781.8f, 735.1f, 693.7f, 657.1f, 624.6f, 595.5f, 569.4f, 524.5f, 487.2f, 455.6f, 428.5f, 404.9f, 383.9f, 365.2f, 348.6f, 333.8f, 320.5f, 308.3f, 297.2f, 287.f, 277.8f}; 

static const G4float e50[78] = { 192.4f, 229.8f, 260.8f, 287.8f, 312.f, 354.4f, 391.4f, 424.6f, 454.8f, 482.8f, 509.f, 533.7f, 590.f, 640.5f, 686.6f, 729.3f, 769.2f, 806.7f, 842.2f, 876.f, 939.2f, 997.5f, 1052.f, 1103.f, 1151.f, 1197.f, 1240.f, 1281.f, 1321.f, 1359.f, 1395.f, 1430.f, 1464.f, 1497.f, 1644.f, 1769.f, 1878.f, 1971.f, 2053.f, 2123.f, 2184.f, 2236.f, 2319.f, 2377.f, 2414.f, 2436.f, 2445.f, 2443.f, 2433.f, 2417.f, 2396.f, 2371.f, 2343.f, 2314.f, 2283.f, 2251.f, 2091.f, 1936.f, 1791.f, 1657.f, 1537.f, 1435.f, 1349.f, 1273.f, 1148.f, 1047.f, 964.8f, 895.9f, 837.3f, 786.7f, 742.4f, 703.5f, 668.8f, 637.7f, 609.7f, 584.3f, 561.2f, 540.f}; 

static const G4float e51[78] = { 146.3f, 178.1f, 204.7f, 228.1f, 249.2f, 286.4f, 319.1f, 348.5f, 375.5f, 400.6f, 424.1f, 446.2f, 497.f, 542.8f, 584.7f, 623.5f, 659.9f, 694.2f, 726.6f, 757.6f, 815.5f, 869.1f, 919.f, 965.9f, 1010.f, 1052.f, 1092.f, 1130.f, 1167.f, 1202.f, 1235.f, 1268.f, 1299.f, 1329.f, 1465.f, 1581.f, 1681.f, 1768.f, 1843.f, 1908.f, 1965.f, 2013.f, 2090.f, 2144.f, 2180.f, 2201.f, 2210.f, 2209.f, 2202.f, 2188.f, 2170.f, 2149.f, 2125.f, 2099.f, 2072.f, 2044.f, 1912.f, 1785.f, 1661.f, 1543.f, 1433.f, 1340.f, 1261.f, 1191.f, 1076.f, 982.4f, 906.f, 842.f, 787.5f, 740.5f, 699.2f, 662.9f, 630.5f, 601.5f, 575.3f, 551.6f, 529.9f, 510.1f}; 

static const G4float e52[78] = { 168.2f, 205.8f, 237.5f, 265.4f, 290.7f, 335.4f, 374.8f, 410.5f, 443.2f, 473.7f, 502.3f, 529.3f, 591.4f, 647.6f, 699.1f, 747.1f, 792.1f, 834.6f, 874.9f, 913.5f, 985.9f, 1053.f, 1116.f, 1175.f, 1232.f, 1285.f, 1336.f, 1385.f, 1432.f, 1478.f, 1521.f, 1564.f, 1604.f, 1644.f, 1825.f, 1983.f, 2122.f, 2246.f, 2355.f, 2453.f, 2539.f, 2615.f, 2740.f, 2833.f, 2899.f, 2940.f, 2962.f, 2967.f, 2958.f, 2938.f, 2908.f, 2871.f, 2829.f, 2782.f, 2732.f, 2681.f, 2418.f, 2183.f, 1987.f, 1826.f, 1691.f, 1577.f, 1480.f, 1396.f, 1257.f, 1145.f, 1053.f, 977.1f, 912.4f, 856.5f, 807.8f, 764.9f, 726.7f, 692.6f, 661.8f, 633.9f, 608.6f, 585.4f}; 

static const G4float e53[78] = { 163.2f, 195.3f, 221.8f, 244.9f, 265.6f, 301.9f, 333.5f, 361.8f, 387.6f, 411.5f, 433.8f, 454.8f, 502.7f, 545.7f, 584.8f, 621.f, 654.8f, 686.6f, 716.7f, 745.3f, 798.7f, 847.9f, 893.8f, 936.8f, 977.4f, 1016.f, 1052.f, 1087.f, 1121.f, 1153.f, 1183.f, 1213.f, 1241.f, 1269.f, 1393.f, 1500.f, 1593.f, 1673.f, 1744.f, 1806.f, 1860.f, 1908.f, 1985.f, 2042.f, 2083.f, 2110.f, 2126.f, 2133.f, 2133.f, 2126.f, 2115.f, 2100.f, 2082.f, 2061.f, 2038.f, 2014.f, 1874.f, 1731.f, 1599.f, 1482.f, 1379.f, 1290.f, 1215.f, 1148.f, 1038.f, 948.5f, 875.3f, 813.8f, 761.5f, 716.3f, 676.6f, 641.6f, 610.5f, 582.5f, 557.3f, 534.4f, 513.6f, 494.5f}; 

static const G4float e54[78] = { 118.7f, 146.6f, 170.3f, 191.4f, 210.4f, 244.5f, 274.7f, 302.1f, 327.3f, 350.9f, 373.1f, 394.2f, 442.7f, 486.8f, 527.4f, 565.2f, 600.9f, 634.6f, 666.7f, 697.4f, 755.1f, 808.9f, 859.3f, 906.9f, 952.f, 995.f, 1036.f, 1075.f, 1113.f, 1150.f, 1185.f, 1219.f, 1252.f, 1283.f, 1428.f, 1554.f, 1665.f, 1762.f, 1848.f, 1923.f, 1990.f, 2049.f, 2144.f, 2216.f, 2266.f, 2300.f, 2320.f, 2328.f, 2327.f, 2318.f, 2302.f, 2282.f, 2259.f, 2232.f, 2203.f, 2173.f, 2008.f, 1847.f, 1702.f, 1574.f, 1464.f, 1369.f, 1288.f, 1217.f, 1099.f, 1004.f, 925.6f, 860.1f, 804.4f, 756.3f, 714.1f, 676.9f, 643.8f, 614.2f, 587.4f, 563.2f, 541.1f, 520.8f}; 

static const G4float e55[78] = { 193.7f, 232.8f, 265.4f, 293.9f, 319.6f, 365.f, 404.8f, 440.6f, 473.4f, 503.9f, 532.5f, 559.6f, 621.6f, 677.5f, 728.8f, 776.5f, 821.2f, 863.4f, 903.5f, 941.8f, 1014.f, 1080.f, 1142.f, 1201.f, 1257.f, 1309.f, 1360.f, 1408.f, 1454.f, 1499.f, 1541.f, 1583.f, 1622.f, 1661.f, 1836.f, 1986.f, 2117.f, 2231.f, 2331.f, 2417.f, 2492.f, 2557.f, 2660.f, 2732.f, 2779.f, 2806.f, 2817.f, 2814.f, 2801.f, 2780.f, 2752.f, 2720.f, 2684.f, 2646.f, 2606.f, 2565.f, 2359.f, 2165.f, 1991.f, 1836.f, 1700.f, 1585.f, 1488.f, 1403.f, 1263.f, 1150.f, 1058.f, 981.3f, 916.1f, 860.f, 810.9f, 767.8f, 729.5f, 695.1f, 664.3f, 636.2f, 610.8f, 587.5f}; 

static const G4float e56[78] = { 153.f, 187.5f, 216.6f, 242.2f, 265.3f, 306.5f, 342.7f, 375.5f, 405.6f, 433.6f, 460.f, 484.9f, 542.1f, 593.8f, 641.3f, 685.4f, 726.9f, 766.f, 803.1f, 838.5f, 905.f, 966.6f, 1024.f, 1078.f, 1130.f, 1178.f, 1225.f, 1269.f, 1312.f, 1352.f, 1392.f, 1429.f, 1466.f, 1501.f, 1661.f, 1797.f, 1915.f, 2017.f, 2106.f, 2183.f, 2249.f, 2305.f, 2394.f, 2454.f, 2492.f, 2511.f, 2516.f, 2509.f, 2494.f, 2471.f, 2443.f, 2410.f, 2376.f, 2339.f, 2301.f, 2262.f, 2070.f, 1895.f, 1742.f, 1609.f, 1493.f, 1395.f, 1312.f, 1239.f, 1119.f, 1021.f, 941.4f, 874.6f, 817.8f, 768.7f, 725.7f, 687.9f, 654.1f, 623.9f, 596.6f, 571.9f, 549.4f, 528.7f}; 

static const G4float e57[78] = { 82.7f, 99.97f, 114.4f, 127.1f, 138.4f, 158.5f, 176.2f, 192.f, 206.5f, 220.f, 232.6f, 244.5f, 271.9f, 296.4f, 319.f, 339.8f, 359.4f, 377.8f, 395.3f, 412.f, 443.3f, 472.2f, 499.3f, 524.7f, 548.8f, 571.7f, 593.6f, 614.5f, 634.6f, 653.9f, 672.5f, 690.4f, 707.8f, 724.6f, 801.6f, 868.9f, 928.6f, 981.9f, 1030.f, 1073.f, 1112.f, 1148.f, 1209.f, 1260.f, 1301.f, 1334.f, 1361.f, 1382.f, 1397.f, 1408.f, 1415.f, 1419.f, 1420.f, 1418.f, 1414.f, 1409.f, 1399.f, 1373.f, 1319.f, 1246.f, 1165.f, 1095.f, 1034.f, 979.4f, 887.4f, 813.7f, 752.5f, 700.8f, 656.5f, 618.1f, 584.5f, 554.8f, 528.3f, 504.6f, 483.f, 463.5f, 445.6f, 429.3f}; 

static const G4float e58[78] = { 94.27f, 114.4f, 131.3f, 146.3f, 159.9f, 184.f, 205.5f, 224.9f, 242.9f, 259.7f, 275.5f, 290.5f, 325.3f, 357.f, 386.3f, 413.7f, 439.5f, 464.1f, 487.6f, 510.1f, 552.6f, 592.4f, 629.9f, 665.5f, 699.4f, 731.8f, 762.9f, 792.9f, 821.8f, 849.7f, 876.7f, 902.9f, 928.2f, 952.9f, 1067.f, 1167.f, 1255.f, 1335.f, 1405.f, 1468.f, 1525.f, 1574.f, 1657.f, 1719.f, 1764.f, 1795.f, 1814.f, 1823.f, 1824.f, 1818.f, 1807.f, 1792.f, 1774.f, 1753.f, 1731.f, 1708.f, 1591.f, 1482.f, 1385.f, 1299.f, 1222.f, 1153.f, 1092.f, 1037.f, 941.7f, 862.4f, 795.8f, 739.8f, 692.3f, 651.2f, 615.4f, 583.7f, 555.6f, 530.3f, 507.5f, 486.9f, 468.f, 450.8f}; 

static const G4float e59[78] = { 194.5f, 234.7f, 268.4f, 298.f, 324.7f, 372.f, 413.6f, 451.2f, 485.8f, 518.f, 548.2f, 576.8f, 642.6f, 702.2f, 756.9f, 808.f, 855.9f, 901.2f, 944.4f, 985.6f, 1063.f, 1135.f, 1203.f, 1267.f, 1327.f, 1385.f, 1440.f, 1492.f, 1543.f, 1592.f, 1639.f, 1684.f, 1728.f, 1770.f, 1964.f, 2131.f, 2277.f, 2404.f, 2516.f, 2613.f, 2698.f, 2771.f, 2887.f, 2969.f, 3022.f, 3053.f, 3065.f, 3061.f, 3046.f, 3021.f, 2990.f, 2953.f, 2912.f, 2868.f, 2822.f, 2775.f, 2564.f, 2372.f, 2191.f, 2022.f, 1868.f, 1739.f, 1629.f, 1534.f, 1377.f, 1252.f, 1150.f, 1065.f, 993.f, 931.1f, 877.2f, 829.9f, 787.8f, 750.2f, 716.4f, 685.8f, 658.f, 632.5f}; 

static const G4float e60[78] = { 14.45f, 17.87f, 20.78f, 23.37f, 25.71f, 29.9f, 33.61f, 36.99f, 40.1f, 43.01f, 45.75f, 48.35f, 54.36f, 59.81f, 64.84f, 69.54f, 73.97f, 78.17f, 82.17f, 86.f, 93.22f, 99.96f, 106.3f, 112.3f, 118.f, 123.5f, 128.7f, 133.8f, 138.7f, 143.4f, 147.9f, 152.4f, 156.7f, 160.8f, 180.3f, 197.6f, 213.3f, 227.7f, 240.9f, 253.f, 264.3f, 274.7f, 293.2f, 309.f, 322.4f, 333.7f, 343.1f, 350.7f, 356.8f, 361.6f, 365.1f, 367.6f, 369.1f, 369.8f, 369.9f, 369.3f, 364.7f, 357.f, 347.3f, 336.6f, 325.5f, 314.4f, 303.6f, 293.2f, 274.3f, 258.5f, 245.1f, 233.4f, 223.1f, 214.f, 205.8f, 198.4f, 191.6f, 185.4f, 179.7f, 174.4f, 169.5f, 164.9f}; 

static const G4float e61[78] = { 47.32f, 61.63f, 74.34f, 85.98f, 96.82f, 116.8f, 135.1f, 152.1f, 168.2f, 183.4f, 198.1f, 212.1f, 245.3f, 276.1f, 305.2f, 332.7f, 359.1f, 384.4f, 408.7f, 432.2f, 477.f, 519.3f, 559.4f, 597.5f, 633.9f, 668.6f, 701.9f, 733.7f, 764.3f, 793.5f, 821.6f, 848.5f, 874.4f, 899.1f, 1008.f, 1097.f, 1168.f, 1225.f, 1270.f, 1307.f, 1336.f, 1359.f, 1390.f, 1408.f, 1416.f, 1417.f, 1413.f, 1405.f, 1395.f, 1383.f, 1370.f, 1356.f, 1341.f, 1326.f, 1310.f, 1295.f, 1218.f, 1146.f, 1081.f, 1023.f, 970.6f, 923.9f, 881.4f, 842.4f, 773.1f, 713.1f, 661.2f, 617.4f, 579.7f, 547.f, 518.2f, 492.7f, 470.f, 449.5f, 430.9f, 414.f, 398.6f, 384.4f}; 

static const G4float e62[78] = { 68.9f, 85.1f, 98.85f, 111.f, 122.1f, 141.8f, 159.3f, 175.1f, 189.8f, 203.4f, 216.3f, 228.5f, 256.6f, 282.1f, 305.7f, 327.6f, 348.3f, 367.9f, 386.5f, 404.4f, 437.9f, 469.2f, 498.5f, 526.3f, 552.6f, 577.7f, 601.7f, 624.7f, 646.9f, 668.2f, 688.8f, 708.7f, 728.f, 746.7f, 832.2f, 906.8f, 972.6f, 1031.f, 1083.f, 1129.f, 1170.f, 1207.f, 1268.f, 1315.f, 1351.f, 1377.f, 1396.f, 1409.f, 1417.f, 1421.f, 1421.f, 1419.f, 1414.f, 1406.f, 1397.f, 1387.f, 1349.f, 1301.f, 1237.f, 1163.f, 1087.f, 1021.f, 964.f, 913.7f, 829.4f, 762.2f, 706.3f, 659.f, 618.4f, 583.3f, 552.3f, 524.9f, 500.4f, 478.4f, 458.5f, 440.4f, 423.9f, 408.7f}; 

static const G4float e63[78] = { 192.f, 228.9f, 259.4f, 286.f, 309.7f, 351.2f, 387.4f, 419.7f, 449.2f, 476.4f, 501.9f, 525.8f, 580.4f, 629.2f, 673.8f, 714.9f, 753.3f, 789.4f, 823.5f, 856.f, 916.5f, 972.3f, 1024.f, 1073.f, 1119.f, 1162.f, 1203.f, 1243.f, 1280.f, 1316.f, 1351.f, 1384.f, 1416.f, 1446.f, 1585.f, 1703.f, 1805.f, 1892.f, 1968.f, 2034.f, 2090.f, 2139.f, 2215.f, 2268.f, 2303.f, 2323.f, 2331.f, 2330.f, 2321.f, 2307.f, 2287.f, 2265.f, 2240.f, 2212.f, 2184.f, 2155.f, 1995.f, 1841.f, 1703.f, 1582.f, 1477.f, 1385.f, 1305.f, 1233.f, 1113.f, 1015.f, 936.f, 869.6f, 813.f, 764.1f, 721.3f, 683.7f, 650.1f, 620.f, 592.9f, 568.3f, 545.9f, 525.4f}; 

static const G4float e64[78] = { 59.24f, 72.81f, 84.28f, 94.41f, 103.6f, 119.9f, 134.3f, 147.4f, 159.4f, 170.6f, 181.1f, 191.1f, 214.f, 234.7f, 253.8f, 271.6f, 288.3f, 304.1f, 319.1f, 333.5f, 360.5f, 385.5f, 409.f, 431.2f, 452.2f, 472.2f, 491.4f, 509.7f, 527.3f, 544.3f, 560.6f, 576.4f, 591.7f, 606.5f, 674.2f, 733.1f, 785.f, 831.f, 871.7f, 907.9f, 940.f, 968.3f, 1015.f, 1051.f, 1077.f, 1095.f, 1106.f, 1112.f, 1114.f, 1112.f, 1107.f, 1100.f, 1092.f, 1082.f, 1071.f, 1059.f, 995.6f, 933.7f, 877.6f, 827.9f, 785.2f, 747.8f, 714.5f, 684.4f, 631.9f, 587.1f, 548.3f, 514.3f, 484.2f, 457.3f, 433.4f, 412.5f, 393.7f, 376.7f, 361.5f, 347.6f, 334.9f, 323.2f}; 

static const G4float e65[78] = { 27.39f, 33.96f, 39.55f, 44.52f, 49.04f, 57.11f, 64.28f, 70.8f, 76.82f, 82.45f, 87.76f, 92.79f, 104.4f, 115.f, 124.8f, 133.9f, 142.5f, 150.6f, 158.4f, 165.8f, 179.8f, 192.9f, 205.2f, 216.8f, 227.8f, 238.4f, 248.4f, 258.1f, 267.4f, 276.4f, 285.1f, 293.5f, 301.6f, 309.5f, 345.7f, 377.3f, 405.3f, 430.1f, 452.1f, 471.8f, 489.3f, 504.8f, 530.7f, 550.9f, 566.3f, 577.9f, 586.3f, 592.2f, 596.1f, 598.4f, 599.3f, 599.1f, 598.1f, 596.4f, 594.2f, 591.5f, 573.8f, 552.8f, 531.2f, 510.1f, 488.3f, 467.f, 447.f, 428.6f, 396.3f, 369.4f, 346.8f, 327.7f, 311.4f, 297.1f, 283.9f, 271.9f, 260.8f, 250.6f, 241.7f, 233.6f, 226.f, 218.9f}; 

static const G4float e66[78] = { 135.1f, 165.6f, 191.3f, 213.9f, 234.4f, 270.7f, 302.8f, 331.8f, 358.4f, 383.2f, 406.5f, 428.6f, 479.3f, 525.2f, 567.4f, 606.7f, 643.6f, 678.5f, 711.7f, 743.4f, 803.1f, 858.6f, 910.7f, 960.1f, 1007.f, 1052.f, 1095.f, 1136.f, 1176.f, 1214.f, 1251.f, 1287.f, 1322.f, 1356.f, 1514.f, 1655.f, 1783.f, 1899.f, 2006.f, 2104.f, 2193.f, 2275.f, 2418.f, 2535.f, 2626.f, 2693.f, 2737.f, 2760.f, 2764.f, 2752.f, 2728.f, 2695.f, 2655.f, 2612.f, 2566.f, 2519.f, 2278.f, 2060.f, 1878.f, 1727.f, 1602.f, 1494.f, 1403.f, 1323.f, 1191.f, 1085.f, 998.2f, 926.f, 864.6f, 811.7f, 765.6f, 724.9f, 688.8f, 656.5f, 627.4f, 601.1f, 577.1f, 555.2f}; 

static const G4float e67[78] = { 149.7f, 181.5f, 208.2f, 231.7f, 252.8f, 290.4f, 323.5f, 353.4f, 380.9f, 406.5f, 430.5f, 453.3f, 505.7f, 553.f, 596.6f, 637.2f, 675.4f, 711.5f, 745.8f, 778.6f, 840.4f, 897.8f, 951.6f, 1003.f, 1051.f, 1097.f, 1141.f, 1183.f, 1224.f, 1263.f, 1301.f, 1337.f, 1372.f, 1407.f, 1563.f, 1701.f, 1822.f, 1929.f, 2024.f, 2109.f, 2185.f, 2252.f, 2362.f, 2446.f, 2507.f, 2549.f, 2573.f, 2583.f, 2581.f, 2569.f, 2550.f, 2524.f, 2494.f, 2461.f, 2426.f, 2389.f, 2211.f, 2043.f, 1887.f, 1744.f, 1616.f, 1507.f, 1415.f, 1334.f, 1201.f, 1094.f, 1006.f, 933.3f, 871.4f, 818.1f, 771.5f, 730.5f, 694.1f, 661.5f, 632.1f, 605.6f, 581.4f, 559.3f}; 

static const G4float e68[78] = { 192.5f, 230.f, 261.f, 288.1f, 312.3f, 354.9f, 392.f, 425.3f, 455.6f, 483.7f, 510.f, 534.8f, 591.3f, 642.1f, 688.4f, 731.3f, 771.4f, 809.1f, 844.8f, 878.8f, 942.4f, 1001.f, 1056.f, 1107.f, 1155.f, 1201.f, 1245.f, 1287.f, 1327.f, 1365.f, 1401.f, 1437.f, 1471.f, 1504.f, 1652.f, 1779.f, 1888.f, 1983.f, 2065.f, 2136.f, 2197.f, 2250.f, 2333.f, 2392.f, 2430.f, 2452.f, 2461.f, 2459.f, 2449.f, 2433.f, 2411.f, 2386.f, 2358.f, 2328.f, 2296.f, 2264.f, 2109.f, 1960.f, 1817.f, 1683.f, 1560.f, 1456.f, 1368.f, 1291.f, 1163.f, 1060.f, 976.6f, 906.6f, 847.1f, 795.7f, 750.8f, 711.3f, 676.1f, 644.6f, 616.2f, 590.5f, 567.f, 545.6f}; 

static const G4float e69[78] = { 13.2f, 16.72f, 19.77f, 22.52f, 25.04f, 29.61f, 33.73f, 37.51f, 41.03f, 44.35f, 47.5f, 50.51f, 57.52f, 63.96f, 69.97f, 75.62f, 80.99f, 86.11f, 91.01f, 95.73f, 104.7f, 113.1f, 121.1f, 128.7f, 136.f, 143.f, 149.8f, 156.3f, 162.6f, 168.8f, 174.7f, 180.5f, 186.2f, 191.7f, 217.4f, 240.5f, 261.5f, 280.8f, 298.5f, 314.8f, 329.8f, 343.7f, 368.f, 388.4f, 405.3f, 419.f, 429.8f, 438.1f, 444.2f, 448.4f, 450.8f, 451.8f, 451.6f, 450.3f, 448.2f, 445.4f, 420.f, 390.1f, 362.9f, 339.6f, 319.8f, 303.1f, 288.9f, 276.7f, 256.9f, 241.7f, 229.4f, 218.6f, 209.1f, 200.6f, 193.f, 186.1f, 179.8f, 174.f, 168.6f, 163.7f, 159.1f, 154.8f}; 

static const G4float e70[78] = { 20.75f, 25.23f, 28.99f, 32.29f, 35.26f, 40.51f, 45.11f, 49.26f, 53.06f, 56.59f, 59.9f, 63.02f, 70.18f, 76.63f, 82.54f, 88.02f, 93.16f, 98.01f, 102.6f, 107.f, 115.2f, 122.9f, 130.f, 136.7f, 143.1f, 149.2f, 155.f, 160.6f, 165.9f, 171.1f, 176.1f, 180.9f, 185.6f, 190.1f, 211.1f, 229.6f, 246.2f, 261.3f, 275.1f, 287.7f, 299.3f, 310.f, 328.8f, 344.7f, 358.1f, 369.2f, 378.3f, 385.6f, 391.3f, 395.6f, 398.7f, 400.6f, 401.7f, 401.9f, 401.4f, 400.4f, 389.5f, 374.8f, 359.5f, 344.8f, 331.f, 318.3f, 306.5f, 295.7f, 276.7f, 260.8f, 247.3f, 235.5f, 225.2f, 216.1f, 207.8f, 200.3f, 193.5f, 187.3f, 181.5f, 176.2f, 171.3f, 166.7f}; 

static const G4float e71[78] = { 98.91f, 122.7f, 143.1f, 161.1f, 177.5f, 206.9f, 233.f, 256.8f, 278.8f, 299.3f, 318.6f, 337.f, 379.5f, 418.1f, 453.8f, 487.1f, 518.6f, 548.3f, 576.7f, 603.9f, 655.2f, 703.f, 748.f, 790.5f, 831.f, 869.6f, 906.6f, 942.2f, 976.4f, 1009.f, 1041.f, 1072.f, 1102.f, 1131.f, 1265.f, 1383.f, 1488.f, 1582.f, 1666.f, 1743.f, 1811.f, 1873.f, 1978.f, 2062.f, 2128.f, 2178.f, 2215.f, 2240.f, 2255.f, 2262.f, 2262.f, 2256.f, 2245.f, 2230.f, 2211.f, 2190.f, 2050.f, 1896.f, 1752.f, 1624.f, 1511.f, 1414.f, 1329.f, 1255.f, 1132.f, 1034.f, 952.7f, 884.8f, 826.9f, 777.1f, 733.4f, 694.9f, 660.7f, 630.1f, 602.5f, 577.6f, 554.8f, 534.f}; 

static const G4float e72[78] = { 126.3f, 154.7f, 178.7f, 199.8f, 218.8f, 252.7f, 282.5f, 309.5f, 334.3f, 357.3f, 379.f, 399.5f, 446.7f, 489.3f, 528.5f, 565.f, 599.2f, 631.6f, 662.4f, 691.9f, 747.3f, 798.8f, 847.2f, 892.9f, 936.3f, 977.8f, 1018.f, 1056.f, 1093.f, 1128.f, 1163.f, 1196.f, 1228.f, 1260.f, 1405.f, 1535.f, 1652.f, 1758.f, 1855.f, 1942.f, 2022.f, 2094.f, 2216.f, 2309.f, 2377.f, 2422.f, 2446.f, 2455.f, 2453.f, 2446.f, 2436.f, 2423.f, 2409.f, 2392.f, 2371.f, 2346.f, 2169.f, 1971.f, 1797.f, 1653.f, 1537.f, 1437.f, 1351.f, 1275.f, 1149.f, 1049.f, 965.8f, 896.6f, 837.6f, 786.9f, 742.5f, 703.3f, 668.6f, 637.5f, 609.5f, 584.2f, 561.1f, 539.9f}; 

static const G4float e73[78] = { 18.11f, 23.3f, 27.86f, 31.99f, 35.83f, 42.84f, 49.2f, 55.1f, 60.63f, 65.87f, 70.87f, 75.65f, 86.89f, 97.3f, 107.1f, 116.3f, 125.1f, 133.5f, 141.7f, 149.5f, 164.4f, 178.6f, 192.f, 204.9f, 217.3f, 229.2f, 240.7f, 251.8f, 262.6f, 273.1f, 283.4f, 293.3f, 303.f, 312.5f, 356.7f, 396.5f, 432.5f, 465.2f, 494.9f, 521.8f, 546.1f, 567.9f, 604.7f, 633.2f, 654.5f, 669.3f, 678.8f, 683.6f, 684.6f, 682.5f, 678.f, 671.7f, 663.9f, 655.1f, 645.6f, 635.6f, 584.6f, 538.7f, 500.4f, 468.8f, 446.7f, 430.5f, 417.1f, 405.3f, 383.9f, 364.f, 344.9f, 326.8f, 309.7f, 294.6f, 281.3f, 269.5f, 258.7f, 248.8f, 239.6f, 231.1f, 223.2f, 216.2f}; 

  if(0 == j)      { AddData(e0, mat); }
  else if(1 == j) { AddData(e1, mat); }
  else if(2 == j) { AddData(e2, mat); }
  else if(3 == j) { AddData(e3, mat); }
  else if(4 == j) { AddData(e4, mat); }
  else if(5 == j) { AddData(e5, mat); }
  else if(6 == j) { AddData(e6, mat); }
  else if(7 == j) { AddData(e7, mat); }
  else if(8 == j) { AddData(e8, mat); }
  else if(9 == j) { AddData(e9, mat); }
  else if(10 == j) { AddData(e10, mat); }
  else if(11 == j) { AddData(e11, mat); }
  else if(12 == j) { AddData(e12, mat); }
  else if(13 == j) { AddData(e13, mat); }
  else if(14 == j) { AddData(e14, mat); }
  else if(15 == j) { AddData(e15, mat); }
  else if(16 == j) { AddData(e16, mat); }
  else if(17 == j) { AddData(e17, mat); }
  else if(18 == j) { AddData(e18, mat); }
  else if(19 == j) { AddData(e19, mat); }
  else if(20 == j) { AddData(e20, mat); }
  else if(21 == j) { AddData(e21, mat); }
  else if(22 == j) { AddData(e22, mat); }
  else if(23 == j) { AddData(e23, mat); }
  else if(24 == j) { AddData(e24, mat); }
  else if(25 == j) { AddData(e25, mat); }
  else if(26 == j) { AddData(e26, mat); }
  else if(27 == j) { AddData(e27, mat); }
  else if(28 == j) { AddData(e28, mat); }
  else if(29 == j) { AddData(e29, mat); }
  else if(30 == j) { AddData(e30, mat); }
  else if(31 == j) { AddData(e31, mat); }
  else if(32 == j) { AddData(e32, mat); }
  else if(33 == j) { AddData(e33, mat); }
  else if(34 == j) { AddData(e34, mat); }
  else if(35 == j) { AddData(e35, mat); }
  else if(36 == j) { AddData(e36, mat); }
  else if(37 == j) { AddData(e37, mat); }
  else if(38 == j) { AddData(e38, mat); }
  else if(39 == j) { AddData(e39, mat); }
  else if(40 == j) { AddData(e40, mat); }
  else if(41 == j) { AddData(e41, mat); }
  else if(42 == j) { AddData(e42, mat); }
  else if(43 == j) { AddData(e43, mat); }
  else if(44 == j) { AddData(e44, mat); }
  else if(45 == j) { AddData(e45, mat); }
  else if(46 == j) { AddData(e46, mat); }
  else if(47 == j) { AddData(e47, mat); }
  else if(48 == j) { AddData(e48, mat); }
  else if(49 == j) { AddData(e49, mat); }
  else if(50 == j) { AddData(e50, mat); }
  else if(51 == j) { AddData(e51, mat); }
  else if(52 == j) { AddData(e52, mat); }
  else if(53 == j) { AddData(e53, mat); }
  else if(54 == j) { AddData(e54, mat); }
  else if(55 == j) { AddData(e55, mat); }
  else if(56 == j) { AddData(e56, mat); }
  else if(57 == j) { AddData(e57, mat); }
  else if(58 == j) { AddData(e58, mat); }
  else if(59 == j) { AddData(e59, mat); }
  else if(60 == j) { AddData(e60, mat); }
  else if(61 == j) { AddData(e61, mat); }
  else if(62 == j) { AddData(e62, mat); }
  else if(63 == j) { AddData(e63, mat); }
  else if(64 == j) { AddData(e64, mat); }
  else if(65 == j) { AddData(e65, mat); }
  else if(66 == j) { AddData(e66, mat); }
  else if(67 == j) { AddData(e67, mat); }
  else if(68 == j) { AddData(e68, mat); }
  else if(69 == j) { AddData(e69, mat); }
  else if(70 == j) { AddData(e70, mat); }
  else if(71 == j) { AddData(e71, mat); }
  else if(72 == j) { AddData(e72, mat); }
  else if(73 == j) { AddData(e73, mat); }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4ASTARStopping::AddData(const G4float* stop, const G4Material* mat)
{
  G4LPhysicsFreeVector* v = new G4LPhysicsFreeVector(78, T0[0], T0[77]);
  for(size_t i=0; i<78; ++i) { v->PutValues(i, T0[i], ((G4double)stop[i])*fac); }
  v->SetSpline(true);
  v->FillSecondDerivatives();
  materials.push_back(mat);
  sdata.push_back(v);
  ++nvectors;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
