//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file eventgenerator/particleGun/src/TrackingAction.cc
/// \brief Implementation of the TrackingAction class
//
//
// $Id: TrackingAction.cc 68734 2013-04-05 09:47:02Z gcosmo $
// 
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "TrackingAction.hh"
#include "HistoManager.hh"

#include "G4Track.hh"
#include "G4PhysicalConstants.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

TrackingAction::TrackingAction()
:G4UserTrackingAction()
 { }

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void TrackingAction::PreUserTrackingAction(const G4Track* track)
{
 G4AnalysisManager* analysis = G4AnalysisManager::Instance();

 //vertex position
 //
 G4ThreeVector ur(0.,0.,1.); 
 G4ThreeVector vertex   = track->GetVertexPosition();
 G4double r = vertex.mag();
 if (r > 0.0) {
   //vertex position: dN = f(radius)
   //
   analysis->FillH1(1, r); 

   //vertex position: angular distribution : dN = f(costheta)
   //
   ur = vertex/r;   
   G4double costheta  = ur.z();
   analysis->FillH1(2, costheta);

   //vertex position: angular distribution : dN = f(phi)
   //
   G4double phi  = std::atan2(ur.y(), ur.x());
   if (phi < 0.) phi += twopi;
   analysis->FillH1(3, phi);
 }
 
 //particle direction: angular distr in vetex frame dN = f(cosalpha)
 //
 G4ThreeVector um = track->GetMomentumDirection();
 G4double cosalpha  = ur*um;
 analysis->FillH1(4, cosalpha);
 
 //particle direction: angular distr in vertex frame dN = f(psi)
 //
 // complete local frame
 G4ThreeVector u1(1.,0.,0.);  u1.rotateUz(ur);
 G4ThreeVector u2(0.,1.,0.);  u2.rotateUz(ur);
 //
 G4double psi = std::atan2(u2*um, u1*um);
 if (psi < 0.) psi += twopi;
 analysis->FillH1(5, psi);
 
 //particle direction: angl distr in master frame
 //
 G4double cosalphaM = um.z();
 G4double psiM = um.phi(); if (psiM < 0.) psiM += twopi;
 analysis->FillH1(6, cosalphaM);
 analysis->FillH1(7, psiM);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void TrackingAction::PostUserTrackingAction(const G4Track*)
{ }

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

