//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// INCL++ intra-nuclear cascade model
// Alain Boudard, CEA-Saclay, France
// Joseph Cugnon, University of Liege, Belgium
// Jean-Christophe David, CEA-Saclay, France
// Pekka Kaitaniemi, CEA-Saclay, France, and Helsinki Institute of Physics, Finland
// Sylvie Leray, CEA-Saclay, France
// Davide Mancusi, CEA-Saclay, France
//
#define INCLXX_IN_GEANT4_MODE 1

#include "globals.hh"

#include "G4INCLNKbToSpiChannel.hh"
#include "G4INCLKinematicsUtils.hh"
#include "G4INCLBinaryCollisionAvatar.hh"
#include "G4INCLRandom.hh"
#include "G4INCLGlobals.hh"
#include "G4INCLLogger.hh"
#include <algorithm>
#include "G4INCLPhaseSpaceGenerator.hh"

namespace G4INCL {
	
	NKbToSpiChannel::NKbToSpiChannel(Particle *p1, Particle *p2)
		: particle1(p1), particle2(p2)
		{}
	
	NKbToSpiChannel::~NKbToSpiChannel(){}
	
	void NKbToSpiChannel::fillFinalState(FinalState *fs) {
		
        // p K0b -> S+ pi0 (2/3)
        // p K0b -> S0 pi+ (2/3)
        //
        // p K-  -> S+ pi- (1)
        // p K-  -> S0 pi0 (1/2)
        // p K-  -> S- pi+ (2/3)
        
		Particle *nucleon;
		Particle *kaon;
		
		if(particle1->isNucleon()){
			nucleon = particle1;
			kaon = particle2;
		}
		else{
			nucleon = particle2;
			kaon = particle1;
		}
		
		ParticleType SigmaType, PionType;
		
		const G4int iso = ParticleTable::getIsospin(nucleon->getType()) + ParticleTable::getIsospin(kaon->getType());
		const G4int iso_n = ParticleTable::getIsospin(nucleon->getType());
		const G4double rdm = Random::shoot();
		
		G4int WhichChannel = 99;
		
		if(iso == 2 || iso == -2){
			if(rdm < 0.5){
				PionType = ParticleTable::getPionType(iso);
				SigmaType = SigmaZero;
				WhichChannel = 0;
			}
			else{
				SigmaType = ParticleTable::getSigmaType(iso);
				PionType = PiZero;
				WhichChannel = iso/2;
			}
		}
		else{
			if(rdm*13. < 6.){
				PionType = ParticleTable::getPionType(-2.*iso_n);
				SigmaType = ParticleTable::getSigmaType(2.*iso_n);
				WhichChannel = iso_n;
			}
			else if(rdm*13. < 9.){
				PionType = PiZero;
				SigmaType = SigmaZero;
				WhichChannel = 0;
			}
			else{
				PionType = ParticleTable::getPionType(2.*iso_n);
				SigmaType = ParticleTable::getSigmaType(-2.*iso_n);
				WhichChannel = -iso_n;
			}
		}
		
		ThreeVector mom_pion = KaonMomentum(kaon,nucleon,WhichChannel); // based on K- p collision
		
		kaon->setType(PionType);
		nucleon->setType(SigmaType);
		
		G4double norm = KinematicsUtils::momentumInCM(kaon,nucleon);

		kaon->setMomentum(mom_pion*norm);
		nucleon->setMomentum(-mom_pion*norm);
		
		kaon->adjustEnergyFromMomentum();
		nucleon->adjustEnergyFromMomentum();
		
		fs->addModifiedParticle(nucleon);
		fs->addModifiedParticle(kaon);
				
	}
	
	ThreeVector NKbToSpiChannel::KaonMomentum(Particle const * const kaon, Particle const * const nucleon, G4int WhichChannel){
		
		const G4double pLab = KinematicsUtils::momentumInLab(kaon,nucleon);
		G4double max = 0;
		
		if(pLab < 435.) return Random::normVector(); // isotropic
		
		if(WhichChannel == 0) max = 865;
		else if(std::abs(WhichChannel) == 1) max = 1845;
		else{
			INCL_WARN("Bad channel choosed in Kb N To Sigma pi angulard distribution determination. Return an isotropic distribution.\n");
			return Random::normVector();
		}
		
		G4double cos_theta = 1.;
		G4double sin_theta = 0.;
		const G4double cos_phi = std::cos(Random::shoot()*Math::twoPi);
		const G4double sin_phi = std::sqrt(1-cos_phi*cos_phi);
		
		const G4double x = kaon->getMomentum().getX();
		const G4double y = kaon->getMomentum().getY();
		const G4double z = kaon->getMomentum().getZ();
		
		const G4double r = std::sqrt(x*x+y*y+z*z);
		const G4double rho = std::sqrt(x*x+y*y);
		
		if(pLab >= max){
			const G4double b = 12. * pLab/2375.; // correspond to the forward slope description at 2375 MeV/c
			cos_theta = std::log(Random::shoot()*(std::exp(b)-std::exp(-b))+std::exp(-b))/b;
			sin_theta = std::sqrt(1-cos_theta*cos_theta);
			
		}
		else{
			const G4double Legendre_coef[3][283][9] = {
				{// K- p -> Sigma- pi+
				{435,0.20268,0.89605,0.00315,-0.02043,0.00013,1e-05,0,0},
				{440,0.2116,0.91169,0.00737,-0.02297,2e-04,0,0,0},
				{445,0.21976,0.92725,0.01151,-0.02554,0.00027,-1e-05,0,0},
				{450,0.22691,0.94264,0.01546,-0.02817,0.00033,-2e-05,0,0},
				{455,0.23293,0.95777,0.0191,-0.03089,4e-04,-3e-05,0,0},
				{460,0.23762,0.97259,0.02233,-0.03372,0.00046,-4e-05,0,0},
				{465,0.24084,0.98722,0.025,-0.03662,0.00053,-5e-05,0,0},
				{470,0.24244,1.00183,0.02695,-0.0395,0.00059,-6e-05,0,0},
				{475,0.24243,1.0166,0.02804,-0.0423,0.00065,-7e-05,0,0},
				{480,0.2408,1.03171,0.0281,-0.04494,0.00071,-8e-05,0,0},
				{485,0.23759,1.04732,0.02704,-0.04728,0.00076,-8e-05,0,0},
				{490,0.23289,1.06364,0.02475,-0.04917,0.00081,-9e-05,0,0},
				{495,0.22693,1.08085,0.02112,-0.05045,0.00086,-1e-04,0,0},
				{500,0.21985,1.0991,0.01607,-0.05099,0.00089,-0.00011,0,0},
				{505,0.21186,1.11838,0.00962,-0.05067,0.00092,-0.00012,0,0},
				{510,0.20311,1.13865,0.0018,-0.04938,0.00094,-0.00012,0,0},
				{515,0.19378,1.15987,-0.00734,-0.04703,0.00095,-0.00013,0,0},
				{520,0.18394,1.18199,-0.01776,-0.04353,0.00094,-0.00013,0,0},
				{525,0.17362,1.20497,-0.02941,-0.03891,0.00092,-0.00014,0,0},
				{530,0.16277,1.22874,-0.04223,-0.03321,0.00088,-0.00014,1e-05,0},
				{535,0.15135,1.25326,-0.05615,-0.02646,0.00082,-0.00014,1e-05,0},
				{540,0.13922,1.2785,-0.07107,-0.01873,0.00073,-0.00014,1e-05,0},
				{545,0.12624,1.30447,-0.08684,-0.01017,0.00061,-0.00013,1e-05,0},
				{550,0.1123,1.3312,-0.10326,-0.00094,0.00047,-0.00012,1e-05,0},
				{555,0.09727,1.35871,-0.12015,0.00879,0.00029,-0.00011,1e-05,0},
				{560,0.08117,1.38699,-0.13734,0.01891,7e-05,-9e-05,1e-05,0},
				{565,0.064,1.41599,-0.15466,0.02935,-0.00019,-7e-05,2e-05,0},
				{570,0.0459,1.44562,-0.17197,0.04009,-5e-04,-5e-05,2e-05,0},
				{575,0.02704,1.47583,-0.18911,0.05108,-0.00085,-2e-05,2e-05,0},
				{580,0.00769,1.50654,-0.20595,0.06228,-0.00126,2e-05,2e-05,0},
				{585,-0.01183,1.53766,-0.22246,0.07364,-0.00172,6e-05,2e-05,0},
				{590,-0.03118,1.56911,-0.23858,0.08511,-0.00224,0.00011,2e-05,0},
				{595,-0.05002,1.60083,-0.25425,0.09665,-0.00282,0.00016,3e-05,0},
				{600,-0.06802,1.63271,-0.26943,0.10821,-0.00346,0.00023,3e-05,0},
				{605,-0.08492,1.66463,-0.28414,0.1197,-0.00417,3e-04,3e-05,0},
				{610,-0.10045,1.69641,-0.29846,0.131,-0.00494,0.00038,3e-05,0},
				{615,-0.11445,1.72786,-0.31248,0.14199,-0.00578,0.00047,3e-05,0},
				{620,-0.12685,1.75882,-0.32626,0.15255,-0.00669,0.00057,3e-05,0},
				{625,-0.13761,1.78907,-0.3398,0.16263,-0.00766,0.00068,3e-05,0},
				{630,-0.14665,1.81843,-0.353,0.17219,-0.00869,8e-04,3e-05,0},
				{635,-0.15404,1.84667,-0.36578,0.18122,-0.00977,0.00093,3e-05,0},
				{640,-0.15984,1.8736,-0.37806,0.18969,-0.0109,0.00107,3e-05,0},
				{645,-0.1641,1.89903,-0.38974,0.19753,-0.01207,0.00121,3e-05,0},
				{650,-0.1668,1.92283,-0.40077,0.20457,-0.01326,0.00137,3e-05,0},
				{655,-0.16802,1.94484,-0.41108,0.2107,-0.01447,0.00153,3e-05,0},
				{660,-0.16778,1.96491,-0.4206,0.21575,-0.01567,0.0017,2e-05,0},
				{665,-0.16609,1.98295,-0.42932,0.21971,-0.01684,0.00187,2e-05,1e-05},
				{670,-0.16293,1.99895,-0.4373,0.22266,-0.01797,0.00204,1e-05,1e-05},
				{675,-0.15835,2.01289,-0.4446,0.2247,-0.01902,0.00221,0,1e-05},
				{680,-0.15235,2.02475,-0.45128,0.22593,-0.01997,0.00238,-1e-05,1e-05},
				{685,-0.14502,2.03452,-0.45737,0.22638,-0.02077,0.00255,-2e-05,1e-05},
				{690,-0.1364,2.04219,-0.46282,0.22601,-0.02139,0.0027,-3e-05,1e-05},
				{695,-0.12665,2.04773,-0.46761,0.22477,-0.02178,0.00284,-4e-05,1e-05},
				{700,-0.11594,2.05112,-0.47172,0.22263,-0.0219,0.00296,-6e-05,1e-05},
				{705,-0.10451,2.05233,-0.47513,0.21958,-0.0217,0.00305,-8e-05,1e-05},
				{710,-0.09257,2.05134,-0.47789,0.21563,-0.02111,0.00311,-1e-04,1e-05},
				{715,-0.08045,2.0481,-0.48004,0.21082,-0.02008,0.00314,-0.00012,1e-05},
				{720,-0.06841,2.04258,-0.48164,0.20519,-0.01853,0.00312,-0.00015,1e-05},
				{725,-0.05677,2.03476,-0.4827,0.19876,-0.01641,0.00305,-0.00018,1e-05},
				{730,-0.04577,2.02468,-0.48321,0.19159,-0.01364,0.00292,-0.00021,1e-05},
				{735,-0.03567,2.01241,-0.48312,0.18371,-0.01013,0.00271,-0.00024,1e-05},
				{740,-0.02666,1.99799,-0.4824,0.17518,-0.0058,0.00243,-0.00028,1e-05},
				{745,-0.01887,1.98148,-0.48101,0.16603,-0.00058,0.00205,-0.00031,1e-05},
				{750,-0.01239,1.96293,-0.47891,0.15626,0.00561,0.00157,-0.00035,1e-05},
				{755,-0.00733,1.9424,-0.47608,0.14587,0.01287,0.00098,-0.00039,1e-05},
				{760,-0.00369,1.91993,-0.47246,0.13483,0.02128,0.00026,-0.00043,0},
				{765,-0.0015,1.8956,-0.46801,0.12316,0.03091,-0.00059,-0.00048,0},
				{770,-0.00083,1.86947,-0.46257,0.1109,0.04184,-0.0016,-0.00052,0},
				{775,-0.00169,1.84167,-0.45593,0.09812,0.05414,-0.00277,-0.00056,0},
				{780,-0.0042,1.81233,-0.44791,0.0849,0.06785,-0.0041,-0.00061,-1e-05},
				{785,-0.0085,1.78162,-0.43841,0.07136,0.08299,-0.00562,-0.00065,-1e-05},
				{790,-0.0149,1.74974,-0.4273,0.05758,0.09958,-0.00733,-0.00068,-2e-05},
				{795,-0.02359,1.71687,-0.41448,0.04365,0.11765,-0.00925,-0.00072,-2e-05},
				{800,-0.03492,1.68318,-0.3999,0.02967,0.13718,-0.01137,-0.00074,-3e-05},
				{805,-0.04918,1.64882,-0.38363,0.01571,0.15809,-0.0137,-0.00076,-4e-05},
				{810,-0.06654,1.61392,-0.36573,0.00182,0.18032,-0.01625,-0.00077,-4e-05},
				{815,-0.08712,1.57864,-0.34628,-0.01193,0.20379,-0.01901,-0.00077,-5e-05},
				{820,-0.11012,1.54311,-0.32533,-0.02547,0.22841,-0.02199,-0.00075,-6e-05},
				{825,-0.13493,1.50749,-0.30295,-0.03874,0.25413,-0.02519,-0.00072,-7e-05},
				{830,-0.16027,1.47191,-0.2792,-0.05167,0.28087,-0.02861,-0.00068,-8e-05},
				{835,-0.18509,1.43652,-0.25416,-0.0642,0.30854,-0.03226,-0.00061,-9e-05},
				{840,-0.20852,1.40147,-0.22789,-0.07626,0.33709,-0.03613,-0.00053,-1e-04},
				{845,-0.23013,1.3669,-0.20046,-0.08779,0.36642,-0.04024,-0.00042,-0.00012},
				{850,-0.2499,1.33295,-0.17192,-0.09871,0.39647,-0.04458,-3e-04,-0.00013},
				{855,-0.26818,1.29977,-0.14236,-0.10898,0.42717,-0.04916,-0.00014,-0.00015},
				{860,-0.2854,1.2675,-0.11182,-0.11851,0.45844,-0.05397,4e-05,-0.00016},
				{865,-0.30205,1.23628,-0.08039,-0.12725,0.4902,-0.05903,0.00025,-0.00018},
				{870,-0.31869,1.20621,-0.0481,-0.13519,0.52231,-0.06438,0.00049,-2e-04},
				{875,-0.33574,1.17732,-0.01498,-0.14236,0.5546,-0.07012,0.00076,-0.00021},
				{880,-0.35356,1.14966,0.01893,-0.14877,0.58687,-0.07634,0.00107,-0.00023},
				{885,-0.37245,1.12329,0.05362,-0.15447,0.61893,-0.08312,0.00141,-0.00025},
				{890,-0.3926,1.09818,0.08898,-0.15955,0.65061,-0.09049,0.00179,-0.00027},
				{895,-0.41404,1.07429,0.12481,-0.16417,0.68173,-0.09828,0.00221,-0.00029},
				{900,-0.43679,1.05155,0.16084,-0.16846,0.7122,-0.10616,0.00266,-0.00031},
				{905,-0.46071,1.02994,0.1968,-0.17253,0.74193,-0.11383,0.00314,-0.00033},
				{910,-0.4856,1.00944,0.23246,-0.17643,0.77089,-0.12104,0.00366,-0.00034},
				{915,-0.51107,0.99007,0.26757,-0.18017,0.79911,-0.12764,0.00421,-0.00035},
				{920,-0.53674,0.97186,0.3019,-0.18373,0.8266,-0.13347,0.00478,-0.00036},
				{925,-0.56217,0.95486,0.33525,-0.18707,0.85342,-0.13846,0.00537,-0.00036},
				{930,-0.58681,0.9392,0.36744,-0.19012,0.87966,-0.14268,0.00598,-0.00036},
				{935,-0.61023,0.92501,0.39833,-0.1928,0.9054,-0.14621,0.00659,-0.00035},
				{940,-0.63197,0.91241,0.42775,-0.19502,0.93074,-0.14913,0.0072,-0.00034},
				{945,-0.65175,0.90149,0.45562,-0.19679,0.95586,-0.15152,0.0078,-0.00031},
				{950,-0.66926,0.89231,0.4819,-0.19816,0.98094,-0.15347,0.00838,-0.00027},
				{955,-0.68439,0.88494,0.50655,-0.19918,1.00619,-0.15508,0.00891,-0.00022},
				{960,-0.69707,0.87945,0.52952,-0.19989,1.03181,-0.15642,0.00939,-0.00016},
				{965,-0.70731,0.87589,0.55076,-0.20033,1.05793,-0.15758,0.0098,-8e-05},
				{970,-0.71516,0.87419,0.5703,-0.20045,1.08454,-0.15852,0.01012,2e-05},
				{975,-0.7207,0.87429,0.58813,-0.20021,1.11152,-0.1592,0.01032,0.00013},
				{980,-0.72394,0.87607,0.60419,-0.1995,1.13868,-0.1595,0.01035,0.00027},
				{985,-0.72503,0.87939,0.61843,-0.19819,1.16573,-0.15928,0.01018,0.00043},
				{990,-0.72394,0.88412,0.63075,-0.19616,1.1924,-0.15842,0.00978,0.00061},
				{995,-0.72067,0.89012,0.64109,-0.19327,1.21844,-0.15676,0.0091,0.00081},
				{1000,-0.71516,0.89727,0.64936,-0.1894,1.24357,-0.15418,0.0081,0.00105},
				{1005,-0.70731,0.90541,0.6555,-0.18445,1.26753,-0.15055,0.00676,0.00131},
				{1010,-0.697,0.91431,0.65947,-0.17846,1.29013,-0.1459,0.00501,0.00161},
				{1015,-0.68411,0.92376,0.66132,-0.17145,1.31127,-0.14023,0.00282,0.00193},
				{1020,-0.66835,0.93356,0.66107,-0.16347,1.3309,-0.13354,0.00014,0.00228},
				{1025,-0.64976,0.94348,0.65877,-0.15451,1.34895,-0.12582,-0.0031,0.00266},
				{1030,-0.62823,0.95334,0.65446,-0.14459,1.36535,-0.11708,-0.00694,0.00307},
				{1035,-0.6038,0.96292,0.64817,-0.13375,1.38003,-0.10731,-0.01143,0.00351},
				{1040,-0.57654,0.97201,0.63996,-0.12199,1.39292,-0.0965,-0.01662,0.00397},
				{1045,-0.54672,0.9804,0.62985,-0.10933,1.40396,-0.08465,-0.02256,0.00446},
				{1050,-0.51475,0.98795,0.61792,-0.09577,1.41311,-0.07177,-0.02927,0.00498},
				{1055,-0.48114,0.99462,0.60434,-0.08128,1.42048,-0.05788,-0.03657,0.00552},
				{1060,-0.44627,1.00047,0.58928,-0.06579,1.42621,-0.04301,-0.04426,0.00608},
				{1065,-0.41097,1.00553,0.57295,-0.04925,1.43042,-0.02719,-0.05212,0.00665},
				{1070,-0.37567,1.00983,0.55554,-0.03158,1.43327,-0.01044,-0.05994,0.00724},
				{1075,-0.34091,1.01341,0.53725,-0.01274,1.43488,0.00719,-0.06751,0.00782},
				{1080,-0.30701,1.0163,0.51825,0.00733,1.4354,0.02567,-0.07463,0.00841},
				{1085,-0.27413,1.01854,0.49874,0.0287,1.43496,0.04499,-0.08107,0.009},
				{1090,-0.2425,1.02016,0.47892,0.0514,1.4337,0.06511,-0.08666,0.00957},
				{1095,-0.21201,1.02123,0.45893,0.0754,1.43171,0.08607,-0.09129,0.01013},
				{1100,-0.18239,1.02178,0.43894,0.10065,1.4291,0.10791,-0.09489,0.01066},
				{1105,-0.15367,1.02189,0.41909,0.12708,1.42598,0.13065,-0.09738,0.01114},
				{1110,-0.1254,1.02159,0.39954,0.15464,1.42243,0.15435,-0.09869,0.01156},
				{1115,-0.09757,1.02096,0.38043,0.18328,1.41856,0.17903,-0.09875,0.01192},
				{1120,-0.06982,1.02003,0.36193,0.21294,1.41446,0.20473,-0.09748,0.0122},
				{1125,-0.04213,1.01887,0.34419,0.24357,1.41025,0.23151,-0.0948,0.01238},
				{1130,-0.0142,1.01753,0.32732,0.27509,1.40599,0.25935,-0.09067,0.01246},
				{1135,0.01406,1.01605,0.31137,0.30741,1.40167,0.28815,-0.08514,0.01242},
				{1140,0.04282,1.01449,0.29633,0.3404,1.39724,0.31776,-0.07828,0.01222},
				{1145,0.0719,1.0129,0.28221,0.37395,1.39267,0.34804,-0.07016,0.01185},
				{1150,0.10146,1.01132,0.26901,0.40794,1.3879,0.37885,-0.06087,0.01128},
				{1155,0.13127,1.00979,0.25673,0.44225,1.38291,0.41003,-0.05049,0.0105},
				{1160,0.1612,1.00837,0.24537,0.47676,1.37764,0.44145,-0.03908,0.00947},
				{1165,0.19131,1.00711,0.23494,0.51136,1.37206,0.47296,-0.02672,0.00818},
				{1170,0.22156,1.00605,0.22546,0.54595,1.36612,0.5044,-0.01351,0.0066},
				{1175,0.25183,1.00528,0.21695,0.58048,1.35981,0.53559,0.00041,0.00471},
				{1180,0.28191,1.00489,0.20947,0.61496,1.35311,0.56633,0.01488,0.00247},
				{1185,0.31173,1.00498,0.20307,0.64937,1.34598,0.59642,0.02974,-0.00015},
				{1190,0.34118,1.00562,0.1978,0.68368,1.33842,0.62566,0.04483,-0.00318},
				{1195,0.37021,1.0069,0.1937,0.71789,1.33041,0.65385,0.05998,-0.00664},
				{1200,0.39843,1.00892,0.19083,0.75199,1.32192,0.6808,0.07505,-0.01057},
				{1205,0.42584,1.01177,0.18923,0.78595,1.31293,0.7063,0.08986,-0.015},
				{1210,0.45231,1.01553,0.18896,0.81978,1.30344,0.7302,0.10428,-0.01995},
				{1215,0.47765,1.02033,0.19014,0.85355,1.29349,0.7525,0.11823,-0.02545},
				{1220,0.50177,1.02629,0.19287,0.88735,1.28312,0.77325,0.13164,-0.03151},
				{1225,0.52457,1.03352,0.19729,0.92127,1.27241,0.79248,0.14447,-0.03814},
				{1230,0.54599,1.04214,0.2035,0.95541,1.26139,0.81024,0.15665,-0.04537},
				{1235,0.56609,1.05228,0.21163,0.98984,1.25012,0.82657,0.16812,-0.05321},
				{1240,0.5847,1.06406,0.2218,1.02465,1.23865,0.84152,0.17883,-0.06168},
				{1245,0.60195,1.07759,0.23413,1.05995,1.22704,0.85513,0.18872,-0.07079},
				{1250,0.61786,1.09296,0.24867,1.09575,1.21531,0.86741,0.19769,-0.08061},
				{1255,0.63245,1.11008,0.26523,1.13191,1.20336,0.87829,0.20553,-0.09138},
				{1260,0.64572,1.1288,0.28356,1.16821,1.19107,0.88763,0.21202,-0.10339},
				{1265,0.65767,1.14902,0.30341,1.20443,1.1783,0.89534,0.2169,-0.11694},
				{1270,0.6683,1.17065,0.32457,1.24045,1.16511,0.90145,0.22018,-0.13214},
				{1275,0.67758,1.19375,0.34687,1.27626,1.15175,0.90623,0.22212,-0.14893},
				{1280,0.68536,1.21833,0.37014,1.31188,1.1385,0.90994,0.22304,-0.16725},
				{1285,0.69167,1.24444,0.39422,1.34729,1.12563,0.91284,0.22321,-0.187},
				{1290,0.69645,1.27212,0.41894,1.38249,1.11339,0.91518,0.22293,-0.20808},
				{1295,0.69969,1.30139,0.44417,1.41745,1.10188,0.91705,0.22235,-0.23029},
				{1300,0.70139,1.33231,0.46975,1.45211,1.09119,0.91853,0.22163,-0.25337},
				{1305,0.70159,1.36489,0.49554,1.48643,1.0814,0.91967,0.22091,-0.27708},
				{1310,0.70037,1.39918,0.52138,1.52037,1.07258,0.92056,0.22033,-0.30119},
				{1315,0.69788,1.43522,0.54714,1.55387,1.06482,0.92125,0.22005,-0.32545},
				{1320,0.69426,1.47301,0.57267,1.5869,1.05819,0.92182,0.22018,-0.34964},
				{1325,0.68978,1.51246,0.59789,1.61947,1.05272,0.92232,0.22064,-0.37368},
				{1330,0.68469,1.55333,0.62279,1.65156,1.04837,0.92281,0.22135,-0.39738},
				{1335,0.6793,1.59543,0.64739,1.68314,1.04513,0.92337,0.22227,-0.4205},
				{1340,0.67393,1.63853,0.67168,1.71418,1.04298,0.92406,0.22335,-0.44283},
				{1345,0.66891,1.68242,0.69567,1.74465,1.04188,0.92496,0.22454,-0.46412},
				{1350,0.66455,1.72688,0.71937,1.77453,1.04181,0.92613,0.22578,-0.48416},
				{1355,0.66117,1.7717,0.74277,1.80377,1.04274,0.92764,0.22704,-0.50271},
				{1360,0.65897,1.81664,0.76591,1.83233,1.04468,0.92957,0.2283,-0.51949},
				{1365,0.65815,1.86145,0.78889,1.86002,1.0477,0.93203,0.2298,-0.53402},
				{1370,0.65879,1.90585,0.81186,1.88663,1.05191,0.93512,0.23178,-0.54579},
				{1375,0.66083,1.94962,0.8348,1.91201,1.05725,0.93889,0.23444,-0.55462},
				{1380,0.66406,1.9926,0.85756,1.93612,1.06349,0.94336,0.23782,-0.56076},
				{1385,0.66812,2.03463,0.87996,1.95888,1.07037,0.94852,0.24197,-0.56449},
				{1390,0.67247,2.07556,0.90183,1.98025,1.07762,0.95437,0.24693,-0.5661},
				{1395,0.67646,2.11524,0.92297,2.00017,1.08501,0.9609,0.25275,-0.56588},
				{1400,0.67917,2.15351,0.94322,2.01858,1.09227,0.96812,0.25948,-0.56411},
				{1405,0.67999,2.19022,0.96239,2.03543,1.09914,0.97601,0.26715,-0.56106},
				{1410,0.67813,2.2252,0.98032,2.05066,1.10537,0.98459,0.27582,-0.55704},
				{1415,0.67307,2.25832,0.99681,2.06423,1.11071,0.99384,0.28552,-0.55231},
				{1420,0.66456,2.28945,1.01176,2.07609,1.11496,1.00374,0.29628,-0.54712},
				{1425,0.65267,2.31863,1.02525,2.08636,1.11811,1.01425,0.30798,-0.54148},
				{1430,0.6378,2.34595,1.03744,2.09517,1.12022,1.02527,0.32046,-0.53535},
				{1435,0.62062,2.37149,1.04848,2.10265,1.12135,1.03672,0.33357,-0.52871},
				{1440,0.60209,2.39534,1.05852,2.10894,1.12156,1.04854,0.34718,-0.52152},
				{1445,0.5832,2.41757,1.06772,2.11416,1.1209,1.06063,0.36112,-0.51375},
				{1450,0.56496,2.43828,1.07622,2.11846,1.11943,1.07293,0.37526,-0.50535},
				{1455,0.54826,2.45754,1.08418,2.12197,1.1172,1.08535,0.38943,-0.49631},
				{1460,0.53374,2.47544,1.09175,2.12481,1.11428,1.09782,0.40349,-0.48658},
				{1465,0.52181,2.49206,1.09908,2.12713,1.11072,1.11025,0.41731,-0.47614},
				{1470,0.51254,2.50744,1.10619,2.12895,1.10663,1.12251,0.4309,-0.46499},
				{1475,0.50575,2.52157,1.11304,2.13027,1.10217,1.13444,0.44441,-0.4532},
				{1480,0.50085,2.53446,1.11958,2.13104,1.09748,1.14586,0.45799,-0.44082},
				{1485,0.49715,2.5461,1.12576,2.13126,1.09271,1.15662,0.47178,-0.4279},
				{1490,0.49385,2.55652,1.13152,2.13089,1.08803,1.16654,0.48593,-0.4145},
				{1495,0.49019,2.56569,1.13682,2.12991,1.08357,1.17545,0.50059,-0.40067},
				{1500,0.48551,2.57364,1.14159,2.1283,1.07949,1.1832,0.51591,-0.38647},
				{1505,0.47939,2.58035,1.14579,2.12603,1.07594,1.18959,0.53203,-0.37195},
				{1510,0.47148,2.58582,1.14937,2.12307,1.07308,1.19448,0.5491,-0.35716},
				{1515,0.46175,2.59007,1.15228,2.11941,1.07105,1.1977,0.56728,-0.34216},
				{1520,0.45033,2.59313,1.15453,2.11509,1.06998,1.19914,0.58667,-0.32702},
				{1525,0.43753,2.5951,1.15631,2.11032,1.06994,1.19888,0.60734,-0.31178},
				{1530,0.42376,2.59609,1.15782,2.10533,1.071,1.19702,0.62932,-0.29653},
				{1535,0.40952,2.59624,1.15926,2.10037,1.07322,1.19365,0.65266,-0.28133},
				{1540,0.39543,2.59565,1.16083,2.09565,1.07667,1.18887,0.6774,-0.26624},
				{1545,0.38215,2.59444,1.16275,2.09142,1.08141,1.18276,0.7036,-0.25134},
				{1550,0.37039,2.59273,1.1652,2.08789,1.08751,1.17544,0.73127,-0.2367},
				{1555,0.36091,2.59062,1.16833,2.08526,1.09503,1.16716,0.76032,-0.22237},
				{1560,0.35456,2.58822,1.17224,2.08366,1.10406,1.15822,0.79059,-0.20842},
				{1565,0.35177,2.58562,1.17705,2.08327,1.11465,1.14894,0.82195,-0.19492},
				{1570,0.35337,2.58293,1.18288,2.08422,1.1269,1.13961,0.85423,-0.18192},
				{1575,0.35961,2.58025,1.18983,2.08669,1.14085,1.13056,0.88728,-0.1695},
				{1580,0.37063,2.57768,1.19803,2.09082,1.1566,1.12209,0.92096,-0.1577},
				{1585,0.3863,2.57533,1.20758,2.09678,1.17421,1.1145,0.95511,-0.1466},
				{1590,0.40627,2.57329,1.2186,2.10471,1.19376,1.10811,0.98957,-0.13626},
				{1595,0.43008,2.57166,1.2312,2.11477,1.21531,1.10322,1.0242,-0.12673},
				{1600,0.45695,2.57056,1.2455,2.12712,1.23895,1.10014,1.05885,-0.11809},
				{1605,0.48626,2.57007,1.2616,2.14191,1.26474,1.09918,1.09336,-0.1104},
				{1610,0.51735,2.57029,1.27959,2.15926,1.29271,1.1006,1.12758,-0.1037},
				{1615,0.54962,2.57116,1.2992,2.17896,1.3226,1.10434,1.16131,-0.09794},
				{1620,0.58245,2.57256,1.32004,2.20065,1.35402,1.11019,1.19434,-0.09302},
				{1625,0.61536,2.5744,1.34172,2.224,1.38657,1.11794,1.22646,-0.08883},
				{1630,0.64786,2.57654,1.36383,2.24863,1.41985,1.12739,1.25746,-0.08529},
				{1635,0.67945,2.57889,1.38599,2.27422,1.45346,1.13832,1.28713,-0.08227},
				{1640,0.70969,2.58132,1.40779,2.30039,1.48701,1.15051,1.31525,-0.07969},
				{1645,0.73811,2.58373,1.42884,2.32681,1.52009,1.16377,1.34161,-0.07744},
				{1650,0.7643,2.58599,1.44873,2.35312,1.55232,1.17789,1.36601,-0.07541},
				{1655,0.78784,2.58801,1.46709,2.37897,1.58329,1.19264,1.38822,-0.07352},
				{1660,0.80845,2.58964,1.4836,2.40404,1.61266,1.20785,1.40808,-0.07166},
				{1665,0.8259,2.59072,1.49807,2.42802,1.64013,1.22332,1.42543,-0.06977},
				{1670,0.84003,2.5911,1.51034,2.45063,1.6654,1.2389,1.44012,-0.06776},
				{1675,0.85087,2.59061,1.52021,2.47156,1.68818,1.25439,1.45202,-0.06558},
				{1680,0.85853,2.58908,1.52752,2.49054,1.7082,1.26964,1.46098,-0.06314},
				{1685,0.86321,2.58636,1.53208,2.50725,1.72514,1.28447,1.46686,-0.06037},
				{1690,0.86521,2.58228,1.53373,2.52141,1.73872,1.29869,1.4695,-0.05719},
				{1695,0.86497,2.57669,1.53227,2.53272,1.74866,1.31215,1.46878,-0.05354},
				{1700,0.86261,2.56941,1.52753,2.54089,1.75465,1.32465,1.46453,-0.04935},
				{1705,0.85865,2.5603,1.51934,2.54562,1.7564,1.33604,1.45663,-0.04453},
				{1710,0.85338,2.54921,1.50758,2.54666,1.75368,1.34611,1.44499,-0.03906},
				{1715,0.84709,2.53615,1.49237,2.54388,1.74639,1.35461,1.42984,-0.03313},
				{1720,0.83998,2.52115,1.4739,2.5372,1.73451,1.36127,1.41148,-0.02693},
				{1725,0.83223,2.50424,1.45236,2.52653,1.718,1.36581,1.3902,-0.02071},
				{1730,0.82398,2.48546,1.42794,2.51179,1.69681,1.36795,1.36629,-0.01468},
				{1735,0.81532,2.46484,1.40083,2.49289,1.67092,1.36742,1.34005,-0.00906},
				{1740,0.80635,2.4424,1.37121,2.46973,1.6403,1.36394,1.31177,-0.00407},
				{1745,0.79717,2.4182,1.33927,2.44227,1.60491,1.35727,1.28174,8e-05},
				{1750,0.78787,2.39228,1.30516,2.41059,1.56494,1.34739,1.25012,0.00337},
				{1755,0.77864,2.36474,1.26903,2.37487,1.52061,1.33437,1.21704,0.00581},
				{1760,0.76947,2.33565,1.231,2.33528,1.47217,1.31829,1.18262,0.00745},
				{1765,0.76063,2.3051,1.19124,2.29197,1.41987,1.29923,1.14697,0.00832},
				{1770,0.75229,2.27315,1.14986,2.24514,1.36393,1.27726,1.11021,0.00845},
				{1775,0.74462,2.23989,1.10702,2.19493,1.3046,1.25247,1.07248,0.00788},
				{1780,0.73776,2.20541,1.06285,2.14153,1.24212,1.22492,1.03388,0.00663},
				{1785,0.73178,2.16977,1.0175,2.0851,1.17673,1.1947,0.99455,0.00475},
				{1790,0.72674,2.13307,0.9711,2.02582,1.10867,1.16189,0.95459,0.00226},
				{1795,0.7226,2.09537,0.9238,1.96385,1.03819,1.12655,0.91414,-0.00079},
				{1800,0.7193,2.05676,0.87573,1.89936,0.96551,1.08878,0.8733,-0.00439},
				{1805,0.71674,2.01732,0.82704,1.83253,0.89088,1.04864,0.83221,-0.00849},
				{1810,0.71485,1.97712,0.77787,1.76351,0.81455,1.00621,0.79099,-0.01305},
				{1815,0.7135,1.93625,0.72835,1.69249,0.73675,0.96158,0.74975,-0.01806},
				{1820,0.71257,1.89479,0.67863,1.61963,0.65772,0.91481,0.70861,-0.02347},
				{1825,0.71198,1.85282,0.62885,1.5451,0.5777,0.86598,0.66769,-0.02924},
				{1830,0.71165,1.81041,0.57915,1.46907,0.49694,0.81518,0.62713,-0.03536},
				{1835,0.7115,1.76765,0.52966,1.39172,0.41567,0.76248,0.58703,-0.04177},
				{1840,0.71149,1.72461,0.48053,1.3132,0.33413,0.70795,0.54752,-0.04846},
				{1845,0.71158,1.68139,0.43186,1.23376,0.25256,0.65182,0.50865,-0.05537}},
				{// K- p -> Sigma0 pi0
				{435,0.01082,1.10989,0.01663,-0.00335,0,0,0,0},
				{440,-0.01888,1.07693,-0.00805,-0.01206,0,0,0,0},
				{445,-0.04853,1.04408,-0.03276,-0.02075,0,0,0,0},
				{450,-0.07803,1.01152,-0.0575,-0.02944,0,0,0,0},
				{455,-0.10733,0.97938,-0.0823,-0.03809,0,0,0,0},
				{460,-0.13633,0.9478,-0.10716,-0.0467,0,0,0,0},
				{465,-0.16491,0.91681,-0.13197,-0.0552,0,0,0,0},
				{470,-0.19294,0.88642,-0.15661,-0.06351,0,0,0,0},
				{475,-0.2203,0.85665,-0.18094,-0.07154,0,0,0,0},
				{480,-0.24683,0.82754,-0.20479,-0.07921,0,0,0,0},
				{485,-0.27226,0.79925,-0.22775,-0.08632,0,0,0,0},
				{490,-0.29631,0.77198,-0.24937,-0.09267,0,0,0,0},
				{495,-0.31868,0.74592,-0.26919,-0.09804,0,0,0,0},
				{500,-0.33912,0.72125,-0.28682,-0.10221,0,0,0,0},
				{505,-0.35756,0.69815,-0.30212,-0.10482,0,0,0,0},
				{510,-0.37395,0.67677,-0.31501,-0.10547,0,0,0,0},
				{515,-0.38825,0.65729,-0.32543,-0.10379,0,0,0,0},
				{520,-0.40045,0.63982,-0.33324,-0.09964,0,0,0,0},
				{525,-0.41054,0.62436,-0.33823,-0.09347,0,0,0,0},
				{530,-0.41855,0.61093,-0.34015,-0.0858,0,0,0,0},
				{535,-0.42447,0.59951,-0.33876,-0.07716,0,0,0,0},
				{540,-0.42842,0.59011,-0.33407,-0.06792,0,0,0,0},
				{545,-0.43073,0.5827,-0.32663,-0.05813,0,0,0,0},
				{550,-0.43174,0.57727,-0.31706,-0.04781,0,0,0,0},
				{555,-0.4318,0.57379,-0.30599,-0.03697,0,0,0,0},
				{560,-0.43126,0.57219,-0.29405,-0.0257,0,0,0,0},
				{565,-0.43044,0.57227,-0.28185,-0.01436,0,0,0,0},
				{570,-0.4298,0.57397,-0.26991,-0.00328,0,0,0,0},
				{575,-0.42982,0.57729,-0.25874,0.0072,0,0,0,0},
				{580,-0.4308,0.58214,-0.24872,0.01687,0,0,0,0},
				{585,-0.4328,0.58835,-0.2401,0.02571,0,0,0,0},
				{590,-0.43612,0.59583,-0.23321,0.03376,0,0,0,0},
				{595,-0.44115,0.6045,-0.22839,0.04108,0,0,0,0},
				{600,-0.4482,0.61432,-0.22597,0.04772,0,0,0,0},
				{605,-0.45692,0.62534,-0.22614,0.0535,0,0,0,0},
				{610,-0.46673,0.63752,-0.22906,0.05801,0,0,0,0},
				{615,-0.47702,0.65085,-0.23489,0.06082,0,0,0,0},
				{620,-0.48723,0.66531,-0.24375,0.06153,0,0,0,0},
				{625,-0.4968,0.68072,-0.25539,0.06011,0,0,0,0},
				{630,-0.50526,0.69682,-0.26923,0.05678,0,0,0,0},
				{635,-0.51223,0.71341,-0.28474,0.05178,0,0,0,0},
				{640,-0.51759,0.73044,-0.30159,0.04536,0,0,0,0},
				{645,-0.52116,0.74789,-0.31929,0.0379,0,0,0,0},
				{650,-0.52267,0.76582,-0.33727,0.02987,0,0,0,0},
				{655,-0.52186,0.78423,-0.35493,0.02172,0,0,0,0},
				{660,-0.51846,0.80318,-0.3717,0.01394,0,0,0,0},
				{665,-0.51218,0.82275,-0.387,0.007,0,0,0,0},
				{670,-0.50273,0.84317,-0.40028,0.00144,0,0,0,0},
				{675,-0.48982,0.86469,-0.41096,-0.0022,0,0,0,0},
				{680,-0.47326,0.88743,-0.41875,-0.00347,0,0,0,0},
				{685,-0.45292,0.91119,-0.42381,-0.0022,0,0,0,0},
				{690,-0.42867,0.93557,-0.42646,0.00161,0,0,0,0},
				{695,-0.40062,0.96016,-0.42703,0.00789,0,0,0,0},
				{700,-0.36903,0.98453,-0.42585,0.01652,0,0,0,0},
				{705,-0.33405,1.00813,-0.42325,0.02726,0,0,0,0},
				{710,-0.29565,1.03016,-0.41947,0.03953,0,0,0,0},
				{715,-0.25378,1.0498,-0.41477,0.05274,0,0,0,0},
				{720,-0.20851,1.06626,-0.40938,0.06618,0,0,0,0},
				{725,-0.16053,1.07916,-0.40351,0.07896,0,0,0,0},
				{730,-0.11059,1.08856,-0.39716,0.09047,0,0,0,0},
				{735,-0.05947,1.09453,-0.3903,0.10015,0,0,0,0},
				{740,-0.00792,1.09717,-0.38288,0.10747,0,0,0,0},
				{745,0.0433,1.09656,-0.37485,0.11203,0,0,0,0},
				{750,0.09344,1.09277,-0.36603,0.11403,0,0,0,0},
				{755,0.14178,1.0859,-0.35625,0.11383,0,0,0,0},
				{760,0.18761,1.07615,-0.34549,0.1117,0,0,0,0},
				{765,0.23037,1.06383,-0.33392,0.10794,0,0,0,0},
				{770,0.26959,1.04936,-0.32217,0.1028,0,0,0,0},
				{775,0.30481,1.03313,-0.311,0.09648,0,0,0,0},
				{780,0.33569,1.0155,-0.3011,0.08904,0,0,0,0},
				{785,0.36209,0.9969,-0.29309,0.08029,0,0,0,0},
				{790,0.38386,0.97771,-0.28758,0.07006,0,0,0,0},
				{795,0.40088,0.95833,-0.28517,0.05815,0,0,0,0},
				{800,0.41307,0.93883,-0.28627,0.04427,0,0,0,0},
				{805,0.42048,0.91889,-0.29099,0.02801,0,0,0,0},
				{810,0.42339,0.89844,-0.29914,0.0094,0,0,0,0},
				{815,0.42217,0.87751,-0.31045,-0.01136,0,0,0,0},
				{820,0.4172,0.85614,-0.32463,-0.03409,0,0,0,0},
				{825,0.40886,0.83439,-0.34137,-0.05857,0,0,0,0},
				{830,0.39754,0.8123,-0.36041,-0.08462,0,0,0,0},
				{835,0.38361,0.7899,-0.38143,-0.11203,0,0,0,0},
				{840,0.36745,0.76725,-0.40416,-0.14061,0,0,0,0},
				{845,0.34944,0.74439,-0.4283,-0.17016,0,0,0,0},
				{850,0.32996,0.72136,-0.45357,-0.20047,0,0,0,0},
				{855,0.30939,0.6982,-0.47967,-0.23135,0,0,0,0},
				{860,0.2881,0.67496,-0.50631,-0.26261,0,0,0,0},
				{865,0.26649,0.65168,-0.53321,-0.29404,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0},
				{0,0,0,0,0,0,0,0,0}},
				{// K- p -> Sigma+ pi-
				{435,-0.41506,0.51794,-0.39061,-0.17259,-0.00101,-0.00057,0,1e-05},
				{440,-0.41233,0.48944,-0.38429,-0.1651,-0.00059,-0.00043,0,1e-05},
				{445,-0.4096,0.46112,-0.37795,-0.15764,-0.00017,-0.00029,0,0},
				{450,-0.40685,0.4332,-0.37154,-0.15026,0.00025,-0.00016,0,0},
				{455,-0.40408,0.40591,-0.36504,-0.14301,0.00067,-2e-05,0,0},
				{460,-0.40131,0.37946,-0.35841,-0.13594,0.00109,0.00012,0,0},
				{465,-0.39874,0.35403,-0.35162,-0.12907,0.00152,0.00025,0,0},
				{470,-0.39658,0.32979,-0.34467,-0.12244,0.00194,0.00039,0,0},
				{475,-0.39505,0.30691,-0.33753,-0.11608,0.00237,0.00053,0,0},
				{480,-0.3944,0.28554,-0.3302,-0.11002,0.00279,0.00067,0,-1e-05},
				{485,-0.39493,0.26577,-0.32275,-0.10426,0.00322,0.00081,0,-1e-05},
				{490,-0.39696,0.24765,-0.31529,-0.09883,0.00364,0.00095,0,-1e-05},
				{495,-0.4008,0.23126,-0.30789,-0.09372,0.00407,0.00109,0,-1e-05},
				{500,-0.40672,0.21665,-0.30065,-0.08895,0.00449,0.00124,0,-1e-05},
				{505,-0.41477,0.20379,-0.2936,-0.08456,0.00491,0.00138,0,-1e-05},
				{510,-0.42493,0.19264,-0.2868,-0.0806,0.00533,0.00152,0,-1e-05},
				{515,-0.4372,0.18316,-0.28027,-0.07711,0.00575,0.00167,0,-2e-05},
				{520,-0.45145,0.17527,-0.27403,-0.07413,0.00616,0.00181,0,-2e-05},
				{525,-0.46723,0.16882,-0.268,-0.07163,0.00656,0.00195,0,-2e-05},
				{530,-0.48408,0.16365,-0.26212,-0.06961,0.00696,0.0021,0,-2e-05},
				{535,-0.50152,0.15962,-0.25632,-0.06805,0.00734,0.00224,0,-2e-05},
				{540,-0.5191,0.1566,-0.25056,-0.06694,0.00772,0.00238,0,-2e-05},
				{545,-0.53644,0.15456,-0.24489,-0.06628,0.00807,0.00251,0,-3e-05},
				{550,-0.55315,0.15349,-0.23935,-0.06605,0.00841,0.00264,0,-3e-05},
				{555,-0.56883,0.15338,-0.23399,-0.06626,0.00872,0.00277,0,-3e-05},
				{560,-0.58327,0.15419,-0.22886,-0.0669,0.00901,0.0029,0,-3e-05},
				{565,-0.59663,0.15588,-0.2239,-0.06794,0.00927,0.00301,0,-3e-05},
				{570,-0.60911,0.15838,-0.21909,-0.06938,0.00949,0.00312,0,-3e-05},
				{575,-0.62091,0.16164,-0.21439,-0.07118,0.00967,0.00322,0,-4e-05},
				{580,-0.63222,0.1656,-0.20978,-0.07333,0.00981,0.00331,0,-4e-05},
				{585,-0.64322,0.17023,-0.20527,-0.07583,0.0099,0.00339,0,-4e-05},
				{590,-0.65408,0.17551,-0.20089,-0.07867,0.00993,0.00345,0,-4e-05},
				{595,-0.66499,0.18138,-0.19664,-0.08184,0.0099,0.0035,0,-4e-05},
				{600,-0.6761,0.18784,-0.19254,-0.08532,0.0098,0.00353,0,-4e-05},
				{605,-0.68744,0.19486,-0.18869,-0.08913,0.00962,0.00354,0,-4e-05},
				{610,-0.69895,0.2025,-0.18519,-0.09324,0.00936,0.00353,0,-4e-05},
				{615,-0.71054,0.21077,-0.18219,-0.09768,0.009,0.00349,0,-4e-05},
				{620,-0.72215,0.21971,-0.17979,-0.10242,0.00855,0.00343,0,-4e-05},
				{625,-0.73381,0.22936,-0.17813,-0.1075,0.00798,0.00333,0,-4e-05},
				{630,-0.74564,0.23978,-0.17729,-0.11293,0.0073,0.00321,0,-4e-05},
				{635,-0.75775,0.25101,-0.17738,-0.11874,0.0065,0.00305,0,-4e-05},
				{640,-0.77025,0.26312,-0.17851,-0.12496,0.00555,0.00285,1e-05,-4e-05},
				{645,-0.78311,0.27609,-0.18077,-0.1316,0.00446,0.00261,1e-05,-4e-05},
				{650,-0.79619,0.28987,-0.18424,-0.13869,0.00322,0.00232,1e-05,-3e-05},
				{655,-0.80933,0.30439,-0.18901,-0.14623,0.00181,0.00199,1e-05,-3e-05},
				{660,-0.8224,0.3196,-0.19518,-0.15426,0.00022,0.0016,2e-05,-3e-05},
				{665,-0.83513,0.3354,-0.2028,-0.16277,-0.00156,0.00116,2e-05,-3e-05},
				{670,-0.84713,0.35164,-0.21197,-0.1718,-0.00353,0.00066,2e-05,-2e-05},
				{675,-0.85802,0.36815,-0.22273,-0.18137,-0.00571,1e-04,3e-05,-2e-05},
				{680,-0.86734,0.38479,-0.23516,-0.19149,-0.00811,-0.00053,3e-05,-1e-05},
				{685,-0.87414,0.4013,-0.24915,-0.2022,-0.01074,-0.00123,4e-05,0},
				{690,-0.87701,0.41738,-0.26452,-0.2135,-0.0136,-0.00201,4e-05,0},
				{695,-0.87457,0.43271,-0.28104,-0.22543,-0.01672,-0.00286,4e-05,1e-05},
				{700,-0.86542,0.447,-0.29851,-0.238,-0.02008,-0.00379,4e-05,2e-05},
				{705,-0.8484,0.46,-0.31674,-0.25122,-0.02371,-0.00481,5e-05,3e-05},
				{710,-0.82293,0.47163,-0.33558,-0.26507,-0.02761,-0.00591,5e-05,4e-05},
				{715,-0.7885,0.48183,-0.35485,-0.2795,-0.03179,-0.00711,4e-05,5e-05},
				{720,-0.74459,0.49053,-0.37441,-0.29449,-0.03625,-0.0084,4e-05,7e-05},
				{725,-0.69112,0.49769,-0.39405,-0.31,-0.041,-0.00979,3e-05,8e-05},
				{730,-0.62899,0.50329,-0.41345,-0.32601,-0.04604,-0.01127,2e-05,9e-05},
				{735,-0.55919,0.50731,-0.43227,-0.34248,-0.05136,-0.01286,1e-05,0.00011},
				{740,-0.48275,0.50974,-0.45017,-0.3594,-0.05697,-0.01455,-1e-05,0.00013},
				{745,-0.40083,0.51061,-0.46688,-0.37671,-0.06287,-0.01634,-3e-05,0.00015},
				{750,-0.31522,0.51019,-0.48236,-0.39436,-0.06905,-0.01823,-6e-05,0.00017},
				{755,-0.22787,0.50878,-0.49662,-0.41225,-0.0755,-0.02023,-1e-04,0.00019},
				{760,-0.14072,0.5067,-0.5097,-0.43031,-0.08221,-0.02233,-0.00014,0.00021},
				{765,-0.05569,0.50425,-0.52159,-0.44847,-0.08917,-0.02453,-0.00019,0.00023},
				{770,0.0256,0.50172,-0.53224,-0.46666,-0.09637,-0.02683,-0.00024,0.00026},
				{775,0.10169,0.49939,-0.54158,-0.48485,-0.10377,-0.02923,-0.00031,0.00028},
				{780,0.17164,0.49748,-0.54958,-0.503,-0.11136,-0.03171,-0.00038,0.00031},
				{785,0.23528,0.49609,-0.55635,-0.52109,-0.11908,-0.03426,-0.00045,0.00034},
				{790,0.29246,0.49532,-0.56201,-0.53909,-0.1269,-0.03689,-0.00053,0.00037},
				{795,0.34305,0.49529,-0.56666,-0.55697,-0.13479,-0.03958,-0.00061,4e-04},
				{800,0.38717,0.49602,-0.57044,-0.5747,-0.14269,-0.04231,-0.00069,0.00043},
				{805,0.42531,0.49745,-0.57352,-0.59223,-0.15055,-0.04508,-0.00077,0.00046},
				{810,0.45797,0.49951,-0.57609,-0.60951,-0.15829,-0.04787,-0.00084,0.00049},
				{815,0.48565,0.50213,-0.57833,-0.6265,-0.16587,-0.05066,-0.00089,0.00052},
				{820,0.50886,0.50523,-0.58041,-0.64314,-0.17322,-0.05344,-0.00093,0.00056},
				{825,0.5281,0.50875,-0.58253,-0.65939,-0.18027,-0.05618,-0.00095,0.00059},
				{830,0.54388,0.51262,-0.58485,-0.6752,-0.18697,-0.05889,-0.00095,0.00062},
				{835,0.55669,0.51675,-0.58757,-0.69052,-0.19326,-0.06154,-0.00091,0.00065},
				{840,0.56705,0.5211,-0.59085,-0.70531,-0.19907,-0.06411,-0.00084,0.00069},
				{845,0.57545,0.52557,-0.59488,-0.71951,-0.20435,-0.06659,-0.00074,0.00072},
				{850,0.5824,0.53011,-0.59985,-0.73309,-0.20902,-0.06896,-0.00059,0.00075},
				{855,0.5884,0.53464,-0.60592,-0.74598,-0.21304,-0.07122,-4e-04,0.00078},
				{860,0.59395,0.53909,-0.61329,-0.75815,-0.21633,-0.07334,-0.00016,0.00081},
				{865,0.59954,0.54338,-0.62212,-0.76955,-0.21885,-0.0753,0.00013,0.00084},
				{870,0.60541,0.54745,-0.63256,-0.78013,-0.22056,-0.0771,0.00048,0.00087},
				{875,0.61159,0.55119,-0.64468,-0.78984,-0.22145,-0.07869,0.00089,9e-04},
				{880,0.61811,0.55452,-0.65859,-0.79866,-0.22151,-0.08006,0.00137,0.00093},
				{885,0.62501,0.55734,-0.67437,-0.80652,-0.22076,-0.08119,0.00192,0.00095},
				{890,0.63226,0.55962,-0.69201,-0.81339,-0.21917,-0.08204,0.00254,0.00097},
				{895,0.63991,0.56144,-0.71124,-0.81921,-0.21677,-0.08265,0.00322,0.00099},
				{900,0.6482,0.56292,-0.73165,-0.82391,-0.21351,-0.08304,0.00396,0.00101},
				{905,0.65734,0.56418,-0.7528,-0.82746,-0.20938,-0.08324,0.00472,0.00102},
				{910,0.66707,0.56537,-0.77433,-0.82982,-0.20437,-0.08329,0.0055,0.00103},
				{915,0.67677,0.56667,-0.79588,-0.83094,-0.19849,-0.08322,0.00626,0.00103},
				{920,0.68581,0.56824,-0.81712,-0.83079,-0.19176,-0.08306,0.00696,0.00102},
				{925,0.6938,0.57027,-0.83765,-0.82936,-0.18417,-0.08286,0.00755,0.00101},
				{930,0.70068,0.57295,-0.85705,-0.8266,-0.17573,-0.08266,0.00799,0.00099},
				{935,0.70637,0.57649,-0.8749,-0.8225,-0.16642,-0.08249,0.00819,0.00097},
				{940,0.71082,0.58109,-0.89077,-0.81703,-0.15624,-0.08242,0.00809,0.00093},
				{945,0.71395,0.58698,-0.90423,-0.81016,-0.14519,-0.08246,0.0076,0.00088},
				{950,0.71565,0.59441,-0.91489,-0.80182,-0.13326,-0.08261,0.00663,0.00082},
				{955,0.71583,0.60361,-0.92234,-0.79199,-0.12046,-0.08291,0.00506,0.00075},
				{960,0.71439,0.61483,-0.92615,-0.78061,-0.10679,-0.08336,0.0028,0.00067},
				{965,0.71134,0.62829,-0.92599,-0.76765,-0.09225,-0.08398,-0.00027,0.00056},
				{970,0.70709,0.64411,-0.92166,-0.7531,-0.07689,-0.08479,-0.00425,0.00045},
				{975,0.7021,0.66238,-0.91307,-0.73694,-0.06076,-0.0858,-0.00928,0.00031},
				{980,0.69672,0.68301,-0.90027,-0.7192,-0.04391,-0.08702,-0.01547,0.00016},
				{985,0.69117,0.70589,-0.8834,-0.69991,-0.02642,-0.08844,-0.02292,-1e-05},
				{990,0.68572,0.73088,-0.8626,-0.6791,-0.00834,-0.09004,-0.03174,-0.00021},
				{995,0.6806,0.75785,-0.838,-0.6568,0.01026,-0.09181,-0.04202,-0.00043},
				{1000,0.67606,0.78666,-0.80974,-0.63304,0.02931,-0.09375,-0.05388,-0.00067},
				{1005,0.67236,0.81718,-0.77798,-0.60786,0.04877,-0.09583,-0.06739,-0.00094},
				{1010,0.66984,0.84916,-0.74299,-0.5813,0.06857,-0.09799,-0.08243,-0.00124},
				{1015,0.66874,0.88239,-0.70509,-0.5534,0.08867,-0.10021,-0.09858,-0.00156},
				{1020,0.66925,0.91668,-0.66453,-0.52422,0.10903,-0.10243,-0.11538,-0.00191},
				{1025,0.67157,0.95184,-0.62161,-0.49382,0.12959,-0.10462,-0.13236,-0.00229},
				{1030,0.67589,0.98766,-0.57661,-0.46226,0.15032,-0.10673,-0.14906,-0.00271},
				{1035,0.68241,1.02396,-0.52979,-0.42958,0.17118,-0.10873,-0.165,-0.00316},
				{1040,0.69132,1.06053,-0.48143,-0.39584,0.1921,-0.11057,-0.17973,-0.00364},
				{1045,0.70281,1.09718,-0.43183,-0.36109,0.21306,-0.11223,-0.19278,-0.00415},
				{1050,0.71699,1.13373,-0.38124,-0.3254,0.234,-0.11366,-0.2038,-0.0047},
				{1055,0.73356,1.17007,-0.32997,-0.28883,0.25489,-0.11484,-0.21289,-0.00529},
				{1060,0.75214,1.20612,-0.2783,-0.25144,0.27572,-0.11578,-0.22029,-0.00591},
				{1065,0.77234,1.24178,-0.22653,-0.2133,0.29644,-0.11646,-0.22622,-0.00657},
				{1070,0.79378,1.27697,-0.17493,-0.17448,0.31703,-0.11688,-0.23091,-0.00727},
				{1075,0.81607,1.31159,-0.1238,-0.13504,0.33746,-0.11701,-0.23458,-0.008},
				{1080,0.83883,1.34556,-0.07343,-0.09504,0.3577,-0.11687,-0.23746,-0.00877},
				{1085,0.86167,1.37879,-0.02411,-0.05456,0.37772,-0.11643,-0.23978,-0.00959},
				{1090,0.88422,1.41119,0.02389,-0.01366,0.39751,-0.11569,-0.24169,-0.01044},
				{1095,0.90616,1.44276,0.07039,0.02762,0.41704,-0.11467,-0.24307,-0.01132},
				{1100,0.92718,1.47349,0.11521,0.06924,0.43632,-0.11339,-0.24371,-0.01225},
				{1105,0.94696,1.50338,0.15818,0.11115,0.45534,-0.11187,-0.24342,-0.01321},
				{1110,0.9652,1.53241,0.19913,0.15331,0.47409,-0.11012,-0.242,-0.0142},
				{1115,0.9816,1.5606,0.23789,0.19569,0.49258,-0.10817,-0.23926,-0.01523},
				{1120,0.99584,1.58794,0.27429,0.23824,0.51078,-0.10603,-0.23498,-0.01629},
				{1125,1.00761,1.61442,0.30816,0.28092,0.52871,-0.10373,-0.22898,-0.01738},
				{1130,1.01668,1.64004,0.33938,0.32369,0.54636,-0.10127,-0.22118,-0.0185},
				{1135,1.02313,1.66484,0.36806,0.36651,0.56375,-0.09867,-0.21198,-0.01964},
				{1140,1.02709,1.68883,0.39433,0.40934,0.58091,-0.09593,-0.20188,-0.02081},
				{1145,1.02872,1.71205,0.41837,0.45217,0.59787,-0.09304,-0.1914,-0.02199},
				{1150,1.02817,1.73453,0.4403,0.49494,0.61465,-0.09001,-0.18106,-0.02319},
				{1155,1.02559,1.75629,0.4603,0.53762,0.63128,-0.08684,-0.17138,-0.0244},
				{1160,1.02111,1.77737,0.47851,0.58018,0.64779,-0.08352,-0.16287,-0.02561},
				{1165,1.01489,1.7978,0.49508,0.62259,0.6642,-0.08006,-0.15604,-0.02682},
				{1170,1.00709,1.81759,0.51017,0.66481,0.68055,-0.07647,-0.15123,-0.02803},
				{1175,0.99786,1.83673,0.52401,0.70683,0.69689,-0.07271,-0.14805,-0.02923},
				{1180,0.98737,1.8552,0.53683,0.74862,0.71327,-0.06878,-0.14595,-0.03041},
				{1185,0.9758,1.87296,0.54887,0.79018,0.72977,-0.06466,-0.14436,-0.03156},
				{1190,0.96331,1.89,0.56034,0.8315,0.74642,-0.06033,-0.14272,-0.03268},
				{1195,0.95008,1.90628,0.57149,0.87255,0.7633,-0.05577,-0.14046,-0.03375},
				{1200,0.93626,1.92179,0.58256,0.91332,0.78046,-0.05097,-0.13703,-0.03476},
				{1205,0.92203,1.9365,0.59376,0.95379,0.79796,-0.04591,-0.13185,-0.03572},
				{1210,0.90757,1.95038,0.60531,0.99396,0.81585,-0.04057,-0.12448,-0.0366},
				{1215,0.89304,1.96341,0.61726,1.03379,0.83412,-0.03495,-0.1149,-0.0374},
				{1220,0.87864,1.97556,0.62967,1.07325,0.85278,-0.02905,-0.10319,-0.0381},
				{1225,0.86453,1.9868,0.64256,1.11232,0.87181,-0.02286,-0.08944,-0.03869},
				{1230,0.85091,1.9971,0.65596,1.15095,0.8912,-0.01639,-0.07374,-0.03915},
				{1235,0.83795,2.00644,0.66993,1.18913,0.91096,-0.00962,-0.05619,-0.03947},
				{1240,0.82582,2.01479,0.68448,1.22682,0.93106,-0.00255,-0.03687,-0.03963},
				{1245,0.81473,2.02212,0.69966,1.26398,0.9515,0.00481,-0.01587,-0.03962},
				{1250,0.80486,2.0284,0.71551,1.3006,0.97227,0.01247,0.00658,-0.03943},
				{1255,0.7966,2.03353,0.73205,1.33662,0.99334,0.0204,0.02967,-0.03908},
				{1260,0.79032,2.03741,0.74932,1.372,1.0147,0.02857,0.05245,-0.0386},
				{1265,0.78642,2.03993,0.76735,1.40669,1.03632,0.03694,0.07397,-0.038},
				{1270,0.78492,2.04109,0.78615,1.44068,1.05815,0.04549,0.09368,-0.03732},
				{1275,0.78537,2.04098,0.80574,1.47397,1.08018,0.05421,0.11146,-0.03658},
				{1280,0.78729,2.03971,0.8261,1.50657,1.10237,0.0631,0.12727,-0.0358},
				{1285,0.79021,2.03737,0.84724,1.53849,1.12468,0.07215,0.14103,-0.035},
				{1290,0.79375,2.03407,0.86916,1.56972,1.14706,0.08134,0.15264,-0.03421},
				{1295,0.79786,2.02989,0.89186,1.60028,1.16947,0.09069,0.16178,-0.03347},
				{1300,0.80263,2.02488,0.91536,1.63014,1.19182,0.10018,0.16808,-0.03285},
				{1305,0.80811,2.01914,0.93965,1.65929,1.21405,0.10981,0.17119,-0.03239},
				{1310,0.81435,2.01272,0.96476,1.68774,1.23609,0.11959,0.17074,-0.03214},
				{1315,0.82144,2.0057,0.99069,1.71547,1.25787,0.12949,0.16635,-0.03216},
				{1320,0.82936,1.99815,1.01744,1.74248,1.27932,0.13954,0.15783,-0.03249},
				{1325,0.83779,1.99015,1.04496,1.76877,1.30041,0.14973,0.14596,-0.03314},
				{1330,0.84665,1.9818,1.07325,1.79437,1.32111,0.1601,0.13175,-0.03405},
				{1335,0.85598,1.97322,1.10228,1.81929,1.3414,0.17067,0.11614,-0.03517},
				{1340,0.86584,1.96452,1.13207,1.84353,1.36125,0.18147,0.10006,-0.03645},
				{1345,0.87626,1.9558,1.16259,1.86713,1.38065,0.19253,0.08445,-0.03781},
				{1350,0.88729,1.94717,1.19385,1.89008,1.39956,0.20387,0.07024,-0.0392},
				{1355,0.89897,1.93876,1.22584,1.9124,1.41797,0.21552,0.05839,-0.04056},
				{1360,0.91133,1.93063,1.25852,1.9341,1.43584,0.22751,0.04964,-0.04184},
				{1365,0.92428,1.92278,1.29175,1.95515,1.45313,0.23983,0.04407,-0.043},
				{1370,0.93773,1.91515,1.32538,1.97551,1.46982,0.25248,0.04159,-0.04401},
				{1375,0.95159,1.90768,1.35924,1.99516,1.48588,0.26547,0.0418,-0.04485},
				{1380,0.96583,1.90033,1.39318,2.01405,1.50132,0.27876,0.04395,-0.04548},
				{1385,0.98042,1.89304,1.42703,2.03215,1.51615,0.29237,0.04726,-0.0459},
				{1390,0.9953,1.88573,1.46066,2.04944,1.53039,0.30627,0.05095,-0.04606},
				{1395,1.01046,1.87835,1.49391,2.06587,1.54404,0.32046,0.05424,-0.04595},
				{1400,1.02585,1.87084,1.52662,2.08142,1.55712,0.33492,0.05634,-0.04555},
				{1405,1.04143,1.86314,1.55865,2.09605,1.56963,0.34966,0.05648,-0.04482},
				{1410,1.05718,1.8552,1.58983,2.10973,1.58159,0.36465,0.05388,-0.04374},
				{1415,1.07304,1.84694,1.62001,2.12243,1.59301,0.37988,0.04775,-0.04229},
				{1420,1.08898,1.83832,1.64908,2.13412,1.6039,0.39535,0.03758,-0.04044},
				{1425,1.10484,1.82934,1.67699,2.14484,1.6143,0.41106,0.02383,-0.03819},
				{1430,1.12048,1.81999,1.70374,2.15463,1.62425,0.427,0.00724,-0.03552},
				{1435,1.13574,1.81027,1.72932,2.16352,1.63379,0.44316,-0.01145,-0.03242},
				{1440,1.15046,1.80018,1.75373,2.17157,1.64296,0.45955,-0.03152,-0.02889},
				{1445,1.16449,1.78973,1.77696,2.17881,1.65179,0.47617,-0.05224,-0.0249},
				{1450,1.17767,1.77892,1.799,2.18528,1.66033,0.493,-0.07288,-0.02046},
				{1455,1.18985,1.76774,1.81985,2.19103,1.66861,0.51005,-0.09269,-0.01555},
				{1460,1.20086,1.75621,1.8395,2.1961,1.67668,0.52732,-0.11096,-0.01015},
				{1465,1.21057,1.74432,1.85795,2.20053,1.68457,0.54479,-0.12698,-0.00427},
				{1470,1.219,1.7322,1.87524,2.20436,1.69232,0.56245,-0.14044,0.00207},
				{1475,1.22629,1.72006,1.89148,2.20761,1.69992,0.58024,-0.15137,0.00883},
				{1480,1.2326,1.7081,1.90674,2.21031,1.70741,0.59813,-0.15979,0.01594},
				{1485,1.23807,1.69653,1.92113,2.2125,1.71479,0.61607,-0.16572,0.02336},
				{1490,1.24286,1.68556,1.93474,2.21421,1.72208,0.63402,-0.16918,0.03103},
				{1495,1.24712,1.67541,1.94765,2.21547,1.7293,0.65192,-0.17019,0.03891},
				{1500,1.251,1.66628,1.95997,2.21629,1.73646,0.66975,-0.16879,0.04693},
				{1505,1.25464,1.65837,1.97178,2.21673,1.74357,0.68744,-0.16499,0.05505},
				{1510,1.25822,1.65191,1.98318,2.2168,1.75066,0.70496,-0.15881,0.06321},
				{1515,1.26186,1.64709,1.99426,2.21654,1.75773,0.72227,-0.15028,0.07137},
				{1520,1.26574,1.64404,2.00509,2.21596,1.7648,0.73931,-0.13954,0.07946},
				{1525,1.27001,1.6427,2.01568,2.21509,1.77186,0.75605,-0.12701,0.08744},
				{1530,1.27484,1.64297,2.02605,2.21392,1.77888,0.77244,-0.11314,0.09525},
				{1535,1.28041,1.64477,2.03621,2.21246,1.78586,0.78844,-0.09836,0.10284},
				{1540,1.28686,1.64799,2.04616,2.21072,1.79279,0.80401,-0.08314,0.11014},
				{1545,1.29439,1.65256,2.05591,2.2087,1.79966,0.8191,-0.06791,0.11712},
				{1550,1.3031,1.65836,2.06547,2.20642,1.80644,0.83367,-0.05309,0.12371},
				{1555,1.3129,1.66525,2.07475,2.20382,1.81312,0.84769,-0.03882,0.1299},
				{1560,1.32356,1.67306,2.08365,2.20085,1.81969,0.86115,-0.02508,0.1357},
				{1565,1.33485,1.68162,2.09205,2.19746,1.82612,0.87402,-0.0119,0.14112},
				{1570,1.34654,1.69074,2.09985,2.19357,1.83239,0.88627,0.00073,0.14615},
				{1575,1.35842,1.70026,2.10691,2.18914,1.83849,0.89789,0.01281,0.1508},
				{1580,1.37025,1.71001,2.11314,2.1841,1.8444,0.90885,0.02433,0.15507},
				{1585,1.38181,1.71981,2.11843,2.17841,1.8501,0.91914,0.03529,0.15897},
				{1590,1.39288,1.72949,2.12265,2.17199,1.85557,0.92872,0.04568,0.1625},
				{1595,1.40323,1.73887,2.1257,2.16479,1.86078,0.93759,0.0555,0.16566},
				{1600,1.41263,1.74778,2.12746,2.15675,1.86573,0.9457,0.06474,0.16846},
				{1605,1.42086,1.75605,2.12782,2.14782,1.8704,0.95306,0.07341,0.1709},
				{1610,1.42772,1.76352,2.12669,2.13793,1.87475,0.95962,0.08151,0.17299},
				{1615,1.43328,1.77017,2.12406,2.12708,1.87879,0.96539,0.08916,0.17472},
				{1620,1.43768,1.77599,2.11996,2.11527,1.88249,0.97035,0.09653,0.17611},
				{1625,1.4411,1.78103,2.11445,2.10249,1.88584,0.97449,0.10377,0.17715},
				{1630,1.44369,1.7853,2.10756,2.08876,1.88883,0.97781,0.11105,0.17786},
				{1635,1.44561,1.78882,2.09933,2.07406,1.89145,0.98031,0.11854,0.17823},
				{1640,1.44703,1.79161,2.0898,2.05839,1.89369,0.98196,0.1264,0.17827},
				{1645,1.4481,1.7937,2.07902,2.04178,1.89552,0.98277,0.1348,0.17798},
				{1650,1.449,1.79509,2.06703,2.0242,1.89693,0.98272,0.14389,0.17737},
				{1655,1.44987,1.79582,2.05386,2.00566,1.89793,0.98181,0.15384,0.17644},
				{1660,1.4508,1.79591,2.03956,1.98617,1.89846,0.98003,0.16463,0.17518},
				{1665,1.45177,1.79536,2.02422,1.96574,1.89851,0.97737,0.17604,0.17354},
				{1670,1.45273,1.79421,2.0079,1.94436,1.89801,0.9738,0.18781,0.1715},
				{1675,1.45366,1.79245,1.99068,1.92205,1.89694,0.96931,0.1997,0.16901},
				{1680,1.45453,1.79012,1.97262,1.89882,1.89525,0.9639,0.21145,0.16602},
				{1685,1.4553,1.78723,1.95379,1.87466,1.8929,0.95754,0.22282,0.16251},
				{1690,1.45594,1.78379,1.93428,1.84959,1.88984,0.95022,0.23355,0.15843},
				{1695,1.45642,1.77982,1.91414,1.82361,1.88604,0.94193,0.24341,0.15375},
				{1700,1.4567,1.77534,1.89345,1.79673,1.88144,0.93265,0.25213,0.14842},
				{1705,1.45676,1.77036,1.87229,1.76895,1.87602,0.92237,0.25948,0.1424},
				{1710,1.45653,1.76489,1.85069,1.74029,1.86972,0.91108,0.26518,0.13566},
				{1715,1.45585,1.75888,1.8286,1.71072,1.86253,0.89877,0.26888,0.12821},
				{1720,1.45454,1.75226,1.80593,1.68023,1.85444,0.88545,0.27022,0.12005},
				{1725,1.45241,1.74498,1.78259,1.6488,1.84541,0.87112,0.26884,0.11119},
				{1730,1.44926,1.73696,1.75849,1.61643,1.83543,0.8558,0.26436,0.10163},
				{1735,1.44492,1.72815,1.73354,1.58309,1.82449,0.83947,0.25641,0.0914},
				{1740,1.43919,1.71847,1.70766,1.54877,1.81256,0.82215,0.24464,0.08048},
				{1745,1.43192,1.70788,1.68076,1.51346,1.79963,0.80384,0.22872,0.06891},
				{1750,1.42321,1.69643,1.6529,1.4772,1.78573,0.78458,0.20876,0.05669},
				{1755,1.41325,1.68419,1.62417,1.44003,1.77092,0.76442,0.18503,0.04387},
				{1760,1.40226,1.67124,1.59466,1.40201,1.75523,0.74341,0.15781,0.03049},
				{1765,1.39045,1.65768,1.56447,1.3632,1.73872,0.72159,0.12737,0.01657},
				{1770,1.37802,1.64358,1.53368,1.32364,1.72144,0.69903,0.094,0.00214},
				{1775,1.36518,1.62903,1.5024,1.28339,1.70344,0.67576,0.05798,-0.01274},
				{1780,1.35214,1.61411,1.47071,1.24251,1.68477,0.65185,0.01957,-0.02806},
				{1785,1.3391,1.59891,1.43872,1.20104,1.66547,0.62732,-0.02093,-0.04378},
				{1790,1.32627,1.58351,1.4065,1.15905,1.6456,0.60225,-0.06326,-0.05985},
				{1795,1.31385,1.56799,1.37416,1.11658,1.6252,0.57668,-0.10713,-0.07626},
				{1800,1.30206,1.55243,1.3418,1.07368,1.60433,0.55065,-0.15227,-0.09296},
				{1805,1.29111,1.53692,1.3095,1.03042,1.58303,0.52422,-0.1984,-0.10991},
				{1810,1.28119,1.52155,1.27735,0.98684,1.56136,0.49744,-0.24524,-0.12709},
				{1815,1.27252,1.50639,1.24546,0.943,1.53936,0.47035,-0.29251,-0.14447},
				{1820,1.2653,1.49154,1.21391,0.89895,1.51708,0.44301,-0.33993,-0.16199},
				{1825,1.25974,1.47706,1.18281,0.85475,1.49457,0.41547,-0.38723,-0.17964},
				{1830,1.25605,1.46305,1.15224,0.81044,1.47189,0.38777,-0.43413,-0.19737},
				{1835,1.25443,1.4496,1.12229,0.76609,1.44907,0.35997,-0.48035,-0.21516},
				{1840,1.25509,1.43678,1.09306,0.72174,1.42618,0.33212,-0.5256,-0.23297},
				{1845,1.25803,1.42461,1.06458,0.67744,1.40325,0.30425,-0.56974,-0.25076}}};
			
			
			
			const G4int coef_ener = G4int((pLab-Legendre_coef[WhichChannel+1][0][0])/5);
			const G4double sup_ener = pLab/5. - coef_ener - Legendre_coef[WhichChannel+1][0][0]/5;
				
// assert(pLab >= Legendre_coef[WhichChannel+1][coef_ener][0] && pLab < Legendre_coef[WhichChannel+1][coef_ener+1][0]);
			
			// Legendre coefficient normalized
			const G4double A0 = 1.;
			const G4double A1 = (1-sup_ener)*Legendre_coef[WhichChannel+1][coef_ener][1] + sup_ener*Legendre_coef[WhichChannel+1][coef_ener+1][1];
			const G4double A2 = (1-sup_ener)*Legendre_coef[WhichChannel+1][coef_ener][2] + sup_ener*Legendre_coef[WhichChannel+1][coef_ener+1][2];
			const G4double A3 = (1-sup_ener)*Legendre_coef[WhichChannel+1][coef_ener][3] + sup_ener*Legendre_coef[WhichChannel+1][coef_ener+1][3];
			const G4double A4 = (1-sup_ener)*Legendre_coef[WhichChannel+1][coef_ener][4] + sup_ener*Legendre_coef[WhichChannel+1][coef_ener+1][4];
			const G4double A5 = (1-sup_ener)*Legendre_coef[WhichChannel+1][coef_ener][5] + sup_ener*Legendre_coef[WhichChannel+1][coef_ener+1][5];
			const G4double A6 = (1-sup_ener)*Legendre_coef[WhichChannel+1][coef_ener][6] + sup_ener*Legendre_coef[WhichChannel+1][coef_ener+1][6];
			const G4double A7 = (1-sup_ener)*Legendre_coef[WhichChannel+1][coef_ener][7] + sup_ener*Legendre_coef[WhichChannel+1][coef_ener+1][7];
			const G4double A8 = (1-sup_ener)*Legendre_coef[WhichChannel+1][coef_ener][8] + sup_ener*Legendre_coef[WhichChannel+1][coef_ener+1][8];
			
			// Theoritical max if all Ai > 0 (often the case)
			const G4double A = std::fabs(A0) + std::fabs(A1) + std::fabs(A2) + std::fabs(A3) + std::fabs(A4) + std::fabs(A5) + std::fabs(A6) + std::fabs(A7) + std::fabs(A8);
			
			G4bool success = false;
			G4int maxloop = 0;
			
			while(!success && maxloop < 1000){
				
				cos_theta = Random::shoot()*2-1.; // not optimized
				
				// Legendre Polynomial
				G4double P0 = A0;
				G4double P1 = A1*cos_theta;
				G4double P2 = A2/2.*(3*std::pow(cos_theta,2)-1);
				G4double P3 = A3/2.*(5*std::pow(cos_theta,3)-3*cos_theta);
				G4double P4 = A4/8.*(35*std::pow(cos_theta,4)-30*std::pow(cos_theta,2)+3);
				G4double P5 = A5/8.*(63*std::pow(cos_theta,5)-70*std::pow(cos_theta,3)+15*cos_theta);
				G4double P6 = A6/16.*(231*std::pow(cos_theta,6)-315*std::pow(cos_theta,4)+105*std::pow(cos_theta,2)-5);
				G4double P7 = A7/16.*(429*std::pow(cos_theta,7)-693*std::pow(cos_theta,5)+315*std::pow(cos_theta,3)-35*cos_theta);
				G4double P8 = A8/128.*(6435*std::pow(cos_theta,8)-12012*std::pow(cos_theta,6)+6930*std::pow(cos_theta,4)-1260*std::pow(cos_theta,2)+35);
				
				G4double P = (P0 + P1 + P2 + P3 + P4 + P5 + P6 + P7 + P8)/2; // /2 for the normalisation
				
				if(Random::shoot()*A < P) success = true;
				maxloop +=1 ;
				if(maxloop==1000) cos_theta = std::log(Random::shoot()*(std::exp(10.)-std::exp(-10.))+std::exp(-10.))/10.; // if no success in 1E4 shoot, probably angulard distribution piked very forward 
			}
			sin_theta = std::sqrt(1-cos_theta*cos_theta);
		}
		
		if(rho == 0) return ThreeVector(sin_theta*cos_phi,sin_theta*sin_phi,cos_theta);
		// Rotation in the direction of the incident kaon
		const G4double px = x/r*cos_theta - y/rho*sin_theta*cos_phi + z/r*x/rho*sin_theta*sin_phi;
		const G4double py = y/r*cos_theta + x/rho*sin_theta*cos_phi + z/r*y/rho*sin_theta*sin_phi;
		const G4double pz = z/r*cos_theta - rho/r*sin_theta*sin_phi;
		
		return ThreeVector(px,py,pz);
	}

}
