//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//      G4MottCoefficients.cc
//-------------------------------------------------------------------
//
// GEANT4 Class header file
//
// File name:    G4MottCoefficients
//
// Author:      Cristina Consolandi
//
// Creation date: 27.05.2012
//
// Class Description:
//
//	Mott Coulomb Cross section coefficients:
//
//      Reference:
//      M.J. Boschini et al.
//      "Non Ionizing Energy Loss induced by Electrons in the Space Environment"
//      Proc. of the 13th International Conference on Particle Physics and Advanced Technology
//      (13th ICPPAT, Como 3-7/10/2011), World Scientific (Singapore).
//
//      Available at: http://arxiv.org/abs/1111.4042v4
//      coeffb of par. 2.1 , eq. (17) were recalculated by M. Tacconi
//      following the same procedur as:
//
//      T. Lijian et al. "Analytic Fitting to the Mott Cross Section of Electrons"
//      Radiat. Phys. Chem. 45 (1995), 235–245.
//
//
//
//
//
// -------------------------------------------------------------------------------------
//
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

#include "G4MottCoefficients.hh"
#include "G4Pow.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......


using namespace std;

G4MottCoefficients::G4MottCoefficients()
{
  fG4pow = G4Pow::GetInstance();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

G4MottCoefficients::~G4MottCoefficients()
{
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

G4double  G4MottCoefficients::GetTransitionRandom(G4double Z, G4double energy){

static const G4double prm[92][11]={
{-3.45427,-1.17273,-0.674098,0.00696853,0.141878,0.0121233,-0.018557,0.00236715,-0.000372957,0.000177557,-2.10679e-05},
{-3.29628,-1.18151,-0.703359,0.0103044,0.163054,0.0115247,-0.0215392,0.00261679,-0.000348795,0.000209188,-2.68608e-05},
{-3.21835,-1.18266,-0.735985,0.0115607,0.187515,0.0101161,-0.0252226,0.00302583,-0.00031054,0.00024924,-3.53293e-05},
{-3.1717,-1.14925,-0.790004,-0.0198764,0.234438,0.0105132,-0.033369,0.00468367,-0.000489407,0.000336593,-4.92297e-05},
{-3.17209,-1.22705,-0.688282,0.0457405,0.159246,0.0111483,-0.0209024,0.00159142,-5.23442e-05,0.000192063,-2.82055e-05},
{-3.16083,-1.22387,-0.70212,0.0399292,0.171785,0.0119655,-0.0233146,0.00193347,-6.22793e-05,0.000213781,-3.21789e-05},
{-3.15413,-1.19529,-0.746885,0.00939304,0.211887,0.0152694,-0.0314518,0.00326366,-0.000138169,0.00028801,-4.55196e-05},
{-3.17206,-1.22774,-0.698953,0.0372708,0.18142,0.0152531,-0.0263482,0.00200525,2.7775e-05,0.000234207,-3.74789e-05},
{-3.17449,-1.20587,-0.734414,0.00491526,0.217546,0.0215135,-0.0349861,0.0032042,-4.9299e-06,0.000307788,-5.12635e-05},
{-3.1924,-1.23048,-0.698328,0.0268472,0.184787,0.0260339,-0.0279849,-0.000229421,0.000577754,0.000276219,-5.16374e-05},
{-3.2049,-1.24254,-0.706055,0.0356984,0.19275,0.0245999,-0.029428,-4.55243e-05,0.000620556,0.000286538,-5.49918e-05},
{-3.22766,-1.24018,-0.699531,0.0292779,0.194941,0.0272524,-0.0304795,-0.000102565,0.000639556,0.000298322,-5.69012e-05},
{-3.24588,-1.2344,-0.705018,0.0181285,0.206355,0.0312091,-0.0340683,0.000193923,0.000686569,0.000325252,-6.30398e-05},
{-3.26373,-1.23951,-0.70502,0.0212337,0.20878,0.0312006,-0.0347649,0.000306563,0.000692132,0.000328136,-6.38534e-05},
{-3.28305,-1.23997,-0.710294,0.0142502,0.219202,0.0349216,-0.038322,0.00061188,0.000739166,0.000355053,-7.00194e-05},
{-3.30593,-1.23538,-0.703515,0.00260105,0.22212,0.040248,-0.040836,0.000805208,0.000736592,0.000373285,-7.28831e-05},
{-3.33916,-1.22704,-0.625808,-0.0575292,0.17402,0.0944467,-0.0434333,-0.0092778,0.00412198,2.87672e-05,-6.93206e-05},
{-3.36453,-1.21502,-0.609634,-0.0965305,0.17717,0.117298,-0.0512211,-0.0108645,0.00501469,-3.45852e-05,-7.5006e-05},
{-3.38155,-1.43456,-0.609212,0.270927,0.107979,-0.0306118,-0.00218581,-1.34583e-05,-0.000136473,0.000111238,-1.07532e-05},
{-3.40227,-1.40841,-0.634279,0.25675,0.126197,-0.0331101,-0.00316589,0.00023027,-0.000190379,0.000129144,-1.26628e-05},
{-3.39149,-1.34102,-0.799316,0.211216,0.241378,-0.0375171,-0.023294,0.00279561,0.000871935,-5.13124e-05,-1.41801e-05},
{-2.4747,-0.673854,-0.949469,-0.150542,0.272209,0.0188266,-0.0308029,0.000659245,0.00130619,-0.000117714,-3.56591e-06},
{-2.23681,-0.925239,-0.747925,-0.0739567,0.170753,0.0137901,-0.0155923,0.000134395,0.000465198,-6.45333e-05,7.19535e-06},
{-2.11934,-1.01535,-0.609797,-0.055407,0.0974267,0.014963,-0.00347492,-0.000553922,-0.000278579,-1.42579e-05,1.83269e-05},
{-2.00521,-1.03932,-0.6315,-0.0662658,0.09963,0.018045,-0.00354355,-0.000700334,-0.000299444,-1.79189e-05,2.05635e-05},
{-1.92535,-1.05016,-0.611446,-0.0595249,0.0924364,0.0166651,-0.00304578,-0.00064247,-0.000285856,-1.58822e-05,1.8711e-05},
{-1.85487,-1.0455,-0.595443,-0.0568245,0.0869614,0.0156046,-0.00261736,-0.000574645,-0.000286237,-1.46623e-05,1.75923e-05},
{-1.78958,-1.03358,-0.593189,-0.066245,0.0851314,0.0169641,-0.00254946,-0.00058829,-0.00028881,-1.94165e-05,1.84522e-05},
{-1.7312,-1.03187,-0.595721,-0.0687198,0.0844325,0.0173088,-0.00241511,-0.000603525,-0.000298154,-1.85105e-05,1.85247e-05},
{-1.67552,-1.02443,-0.595395,-0.072281,0.0828622,0.017723,-0.00225957,-0.000610151,-0.000302192,-1.94811e-05,1.86356e-05},
{-1.62282,-1.01332,-0.602885,-0.0787764,0.0838998,0.0185986,-0.00230456,-0.000621724,-0.000310777,-2.11935e-05,1.92922e-05},
{-1.57541,-1.00961,-0.606542,-0.0805026,0.0839081,0.018709,-0.00229554,-0.000614519,-0.000311766,-2.125e-05,1.92361e-05},
{-1.53386,-1.00016,-0.607855,-0.0861356,0.0831995,0.0195493,-0.00219244,-0.000638326,-0.000322125,-2.25062e-05,1.98282e-05},
{-1.49079,-0.993843,-0.608259,-0.0892669,0.0825349,0.020055,-0.00218695,-0.000646426,-0.000319155,-2.4519e-05,2.00785e-05},
{-1.45611,-0.989,-0.609355,-0.0925922,0.0819072,0.0204973,-0.00209186,-0.000657213,-0.000327024,-2.4845e-05,2.03551e-05},
{-1.41871,-0.982753,-0.613789,-0.0939993,0.0823134,0.0206126,-0.00207367,-0.000661344,-0.00033119,-2.43004e-05,2.03755e-05},
{-1.3876,-0.974781,-0.612345,-0.0976259,0.0813288,0.0210619,-0.00201616,-0.000665815,-0.00033189,-2.53876e-05,2.04942e-05},
{-1.35484,-0.968802,-0.612766,-0.0987116,0.080421,0.0210492,-0.00192582,-0.000655693,-0.000331531,-2.55126e-05,2.025e-05},
{-1.34772,-1.00326,-0.556153,-0.0424339,0.070231,0.0104262,-0.00175413,-0.000279392,-0.000219016,-7.48141e-06,1.08526e-05},
{-1.32033,-0.998614,-0.556946,-0.0433613,0.0698885,0.0104265,-0.00173528,-0.000269835,-0.00021763,-7.70613e-06,1.07236e-05},
{-1.29387,-0.991914,-0.556351,-0.0433399,0.0690369,0.0101818,-0.00166188,-0.00025643,-0.000214616,-6.96604e-06,1.02468e-05},
{-1.26937,-0.984208,-0.559893,-0.0473158,0.0696842,0.0107802,-0.00169281,-0.000264224,-0.000223084,-8.77498e-06,1.10006e-05},
{-1.24606,-0.982953,-0.557063,-0.044914,0.0685251,0.010376,-0.00159731,-0.000266324,-0.000219274,-6.68118e-06,1.03695e-05},
{-1.22265,-0.977172,-0.560847,-0.0470713,0.069072,0.0107057,-0.00160698,-0.000277367,-0.00022425,-6.96396e-06,1.0673e-05},
{-1.20578,-0.971237,-0.557586,-0.0490051,0.067968,0.0109327,-0.00152961,-0.000278309,-0.000225988,-7.684e-06,1.07652e-05},
{-1.18467,-0.969531,-0.558205,-0.0474022,0.0680256,0.0106047,-0.00155372,-0.000264934,-0.000222263,-7.10051e-06,1.04828e-05},
{-1.16668,-0.962652,-0.557634,-0.0505,0.067703,0.0110136,-0.00156156,-0.000267733,-0.000221753,-8.39898e-06,1.06854e-05},
{-1.15074,-0.957517,-0.560071,-0.0541672,0.0682883,0.0117987,-0.0015924,-0.000298938,-0.000227628,-9.6769e-06,1.13059e-05},
{-1.13288,-0.957159,-0.557679,-0.049911,0.0674192,0.0109158,-0.00153791,-0.000266411,-0.00022149,-8.01081e-06,1.05714e-05},
{-1.1196,-0.950271,-0.559097,-0.0547589,0.0680769,0.0117711,-0.00166561,-0.000276567,-0.000219439,-1.12569e-05,1.12041e-05},
{-1.10563,-0.943237,-0.557926,-0.0565505,0.0681072,0.0120531,-0.00171725,-0.000282879,-0.000218682,-1.21928e-05,1.14843e-05},
{-1.09235,-0.941569,-0.560505,-0.0561123,0.0685157,0.0119646,-0.00169261,-0.000280428,-0.000223617,-1.1682e-05,1.15322e-05},
{-1.08165,-0.939346,-0.555938,-0.0560682,0.0673082,0.0119112,-0.0016401,-0.000274998,-0.000219648,-1.139e-05,1.11769e-05},
{-1.0696,-0.937992,-0.557616,-0.0570681,0.0680815,0.0121915,-0.00172775,-0.000284477,-0.000220627,-1.21141e-05,1.15469e-05},
{-1.05804,-0.937207,-0.558408,-0.0566075,0.0682875,0.012231,-0.00174882,-0.000293511,-0.000220088,-1.18869e-05,1.1573e-05},
{-1.0487,-0.93416,-0.559485,-0.0573018,0.0687981,0.0123716,-0.0017948,-0.000298165,-0.000220727,-1.20823e-05,1.17117e-05},
{-1.03853,-0.930536,-0.561213,-0.0596194,0.0693916,0.0127633,-0.0018757,-0.000302286,-0.00022093,-1.35729e-05,1.21118e-05},
{-1.03259,-0.931439,-0.556726,-0.0570483,0.0685184,0.0123419,-0.00183882,-0.000290425,-0.000216266,-1.27107e-05,1.16801e-05},
{-1.02709,-0.933279,-0.555008,-0.0568133,0.0682819,0.0124695,-0.00185261,-0.00029575,-0.000215381,-1.34199e-05,1.1844e-05},
{-1.01793,-0.929331,-0.562398,-0.0612057,0.070269,0.0133683,-0.00199229,-0.000324079,-0.000221687,-1.53905e-05,1.27335e-05},
{-1.01629,-0.929382,-0.55768,-0.061975,0.0694549,0.013656,-0.00198758,-0.000327635,-0.000220765,-1.71386e-05,1.30514e-05},
{-1.01017,-0.933211,-0.553837,-0.0563843,0.069119,0.0127631,-0.00202446,-0.000306269,-0.000209907,-1.50295e-05,1.22372e-05},
{-1.00402,-0.931527,-0.561663,-0.0614812,0.0714191,0.0138037,-0.00221284,-0.000334504,-0.000214929,-1.75699e-05,1.32174e-05},
{-1.00296,-0.935577,-0.556482,-0.0566492,0.0707889,0.0131248,-0.00222224,-0.000315959,-0.000207854,-1.66993e-05,1.2789e-05},
{-0.999308,-0.940971,-0.558419,-0.0565687,0.0713035,0.0134232,-0.00225806,-0.000337324,-0.000209122,-1.69497e-05,1.30385e-05},
{-0.99743,-0.937256,-0.558277,-0.0576755,0.0720174,0.0136793,-0.00232713,-0.000344131,-0.000212815,-1.77581e-05,1.34929e-05},
{-0.996147,-0.942928,-0.559895,-0.0571104,0.0728195,0.0138271,-0.00242734,-0.000350052,-0.000210827,-1.80175e-05,1.36015e-05},
{-0.997667,-0.950669,-0.555255,-0.0523833,0.0725868,0.0133248,-0.00251216,-0.000343447,-0.000199857,-1.73841e-05,1.32229e-05},
{-0.993545,-0.951196,-0.562655,-0.0541101,0.0747495,0.0136734,-0.00266748,-0.000346536,-0.000202929,-1.84821e-05,1.37021e-05},
{-0.996861,-0.959249,-0.558053,-0.0499803,0.0740782,0.0133638,-0.0026,-0.000356007,-0.000206035,-1.74446e-05,1.37177e-05},
{-0.998852,-0.959127,-0.559367,-0.0530054,0.0751304,0.0140818,-0.00276284,-0.000372799,-0.000203002,-1.99556e-05,1.43134e-05},
{-0.999091,-0.968269,-0.559251,-0.0479489,0.0758764,0.0134949,-0.00287584,-0.00036583,-0.000194827,-1.85467e-05,1.39202e-05},
{-1.00359,-0.972198,-0.556711,-0.0477494,0.0757621,0.0137758,-0.00288699,-0.000378367,-0.000197018,-1.97346e-05,1.43232e-05},
{-1.00288,-0.975142,-0.561481,-0.0490123,0.0775024,0.0142525,-0.00304164,-0.000397013,-0.000198895,-2.09334e-05,1.49274e-05},
{-1.00826,-0.98024,-0.560716,-0.0475238,0.0777307,0.0142664,-0.00306904,-0.000404107,-0.000198918,-2.11886e-05,1.50728e-05},
{-1.01165,-0.987835,-0.560303,-0.0449431,0.0786238,0.0141298,-0.00319235,-0.000406331,-0.000195207,-2.10731e-05,1.51339e-05},
{-1.018,-0.997307,-0.556998,-0.041257,0.0782376,0.0138629,-0.00318327,-0.000410616,-0.000192401,-2.05472e-05,1.49881e-05},
{-1.02086,-1.00489,-0.561339,-0.0386772,0.0800913,0.0137721,-0.00332262,-0.00041823,-0.000194708,-2.03251e-05,1.53389e-05},
{-1.02511,-1.00748,-0.563121,-0.0416363,0.0811907,0.0146575,-0.00349921,-0.000446843,-0.000191104,-2.32522e-05,1.60824e-05},
{-1.0274,-1.0166,-0.565418,-0.0382969,0.0822905,0.0143542,-0.00358594,-0.000445608,-0.000189679,-2.23899e-05,1.60135e-05},
{-1.03437,-1.02268,-0.565814,-0.035928,0.0832699,0.0142663,-0.00367908,-0.000454381,-0.000189162,-2.20657e-05,1.61487e-05},
{-1.04332,-1.02938,-0.561205,-0.0342207,0.0829085,0.0142911,-0.00371948,-0.000457825,-0.000185521,-2.29898e-05,1.6307e-05},
{-1.04429,-1.03749,-0.568732,-0.0314777,0.0849283,0.0140019,-0.00383956,-0.000454762,-0.00018503,-2.19763e-05,1.6222e-05},
{-1.04911,-1.04439,-0.571889,-0.0319427,0.0862762,0.0145259,-0.00395989,-0.000485614,-0.000187376,-2.29388e-05,1.68404e-05},
{-1.05909,-1.04857,-0.566255,-0.0309344,0.0861182,0.0147465,-0.00407832,-0.000497202,-0.000179714,-2.4611e-05,1.71362e-05},
{-1.06439,-1.05789,-0.570318,-0.0266983,0.0878402,0.0143828,-0.00418574,-0.000502199,-0.000180593,-2.32303e-05,1.71511e-05},
{-1.06775,-1.06261,-0.572843,-0.0257022,0.0889957,0.0143758,-0.00430065,-0.000498425,-0.000178364,-2.38618e-05,1.72926e-05},
{-1.07551,-1.06847,-0.571036,-0.0246921,0.088754,0.0145911,-0.00425427,-0.000527694,-0.000182927,-2.34593e-05,1.75615e-05},
{-1.07971,-1.0743,-0.571836,-0.0217792,0.0892357,0.0142461,-0.00430431,-0.000520164,-0.000178707,-2.28916e-05,1.73228e-05},
{-1.04956,-0.951005,-0.714805,-0.212099,0.178657,0.085347,-0.0264126,-0.010159,0.00231653,0.000412161,-8.84648e-05},
{-1.08715,-1.07732,-0.576722,-0.0221064,0.0917223,0.0146632,-0.0045922,-0.000535837,-0.000172049,-2.46466e-05,1.78337e-05},
{-1.09157,-1.07821,-0.57956,-0.0222743,0.0927873,0.0148675,-0.00463775,-0.000551921,-0.000176996,-2.45488e-05,1.81678e-05}
};

G4int indexZ=int(Z)-1;


G4double result=prm[indexZ][0]
                +prm[indexZ][1]*log10(energy)
                +prm[indexZ][2]*fG4pow->powN(log10(energy),2)
                +prm[indexZ][3]*fG4pow->powN(log10(energy),3)
                +prm[indexZ][4]*fG4pow->powN(log10(energy),4)
                +prm[indexZ][5]*fG4pow->powN(log10(energy),5)
                +prm[indexZ][6]*fG4pow->powN(log10(energy),6)
                +prm[indexZ][7]*fG4pow->powN(log10(energy),7)
                +prm[indexZ][8]*fG4pow->powN(log10(energy),8)
                +prm[indexZ][9]*fG4pow->powN(log10(energy),9)
                +prm[indexZ][10]*fG4pow->powN(log10(energy),10);

return  G4Exp(G4Log(10)*result);

}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

void G4MottCoefficients::SetMottCoeff(G4double Z, G4double coeffb[5][6]){

//H.......................................
static const G4double bH[5][6]={
{1.e0,2.67363e-8,7.1153e-8,-9.7703e-8,-6.69132e-7,-3.09263e-7},
{1.17182e-2,1.62222e-2,-5.90397e-5,-1.05585e-4,4.17873e-4,9.13843e-4},
{-2.65955e-1,-7.29531e-1,-4.99796e-1,2.83507e-4,-9.09042e-4,-2.20244e-3},
{-1.82348e-4,-8.86355e-5,-1.90554e-4,-2.49708e-4,6.35004e-4,1.73523e-3},
{4.70966e-5,-4.09705e-6,3.75218e-5,8.05645e-5,-1.90534e-4,-5.42847e-4}
};
//He.......................................
static const G4double bHe[5][6]={
{1.e0,3.76476e-8,-3.05313e-7,-3.27422e-7,2.44235e-6,4.08754e-6},
{2.35767e-2,3.24642e-2,-6.37269e-4,-7.6916e-4,5.28004e-3,9.45642e-3},
{-2.73743e-1,-7.40767e-1,-4.98195e-1,1.74337e-3,-1.25798e-2,-2.24046e-2},
{-7.79128e-4,-4.14495e-4,-1.62657e-3,-1.37286e-3,1.04319e-2,1.83488e-2},
{2.02855e-4,1.94598e-6,4.30102e-4,4.3218e-4,-3.31526e-3,-5.81788e-3}
};
//Li..................................................
static const G4double bLi[5][6]={
{1.e0,7.00357e-8,-3.15076e-7,-4.24915e-7,2.45516e-6,4.90187e-6},
{3.55657e-2,4.87956e-2,-1.95525e-3,-2.7866e-3,1.6549e-2,3.11496e-2},
{-2.81171e-1,-7.52015e-1,-4.95329e-1,5.83548e-3,-3.3983e-2,-6.55379e-2},
{-1.83452e-3,-8.12746e-4,-3.84675e-3,-4.44467e-3,2.55871e-2,4.99483e-2},
{4.79031e-4,-3.89615e-5,1.01022e-3,1.39133e-3,-7.99398e-3,-1.56366e-2}
};
//Be..................................................
static const G4double bBe[5][6]={
{1.e0,7.58881e-8,4.705e-8,2.48041e-7,-2.06053e-6,-1.97319e-6},
{4.76788e-2,6.522e-2,-4.54331e-3,-6.50318e-3,3.76564e-2,7.17176e-2},
{-2.88203e-1,-7.63217e-1,-4.90337e-1,1.22839e-2,-6.86398e-2,-1.35769e-1},
{-3.37733e-3,-1.36514e-3,-7.51614e-3,-8.78592e-3,4.78572e-2,9.69021e-2},
{8.81822e-4,-1.02577e-4,1.99797e-3,2.72661e-3,-1.48296e-2,-3.0106e-2}
};
//B....................................................
static const G4double bB[5][6]={
{9.99999e-1,7.91498e-8,1.84164e-6,2.68534e-6,-1.8163e-5,-2.69021e-5},
{5.98818e-2,8.17654e-2,-7.70811e-3,-1.12378e-2,6.38329e-2,1.25339e-1},
{-2.94716e-1,-7.74405e-1,-4.8622e-1,1.77367e-2,-9.46825e-2,-2.01789e-1},
{-5.52375e-3,-2.05348e-3,-9.44915e-3,-1.08135e-2,5.41024e-2,1.25257e-1},
{1.44555e-3,-1.99404e-4,2.36742e-3,3.29655e-3,-1.64122e-2,-3.8375e-2}
};
//C...................................................
static const G4double bC[5][6]={
{9.99999e-1,7.68158e-8,5.18185e-6,7.34245e-6,-4.9478e-5,-7.71923e-5},
{7.21461e-2,9.84618e-2,-1.06535e-2,-1.62358e-2,8.59238e-2,1.78727e-1},
{-3.00622e-1,-7.85616e-1,-4.85735e-1,1.91563e-2,-8.10204e-2,-2.15386e-1},
{-8.34809e-3,-2.85241e-3,-7.03252e-3,-7.56786e-3,1.44975e-2,8.79093e-2},
{2.18964e-3,-3.42022e-4,1.3293e-3,2.20108e-3,-3.57927e-3,-2.5928e-2}
};
//N....................................................
static const G4double bN[5][6]={
{9.99999e-1,8.36312e-8,1.09116e-5,1.47812e-5,-1.02733e-4,-1.62724e-4},
{8.44142e-2,1.1531e-1,-1.1723e-2,-1.94732e-2,8.92604e-2,2.09303e-1},
{-3.05743e-1,-7.96809e-1,-4.93957e-1,1.01607e-2,1.67761e-2,-1.05909e-1},
{-1.2009e-2,-3.80678e-3,4.51195e-3,6.93472e-3,-1.12405e-1,-8.15484e-2},
{3.16048e-3,-5.22237e-4,-2.58261e-3,-2.38303e-3,3.63393e-2,2.75127e-2}
};
//O...................................................
static const G4double bO[5][6]={
{9.99998e-1,1.57323e-8,1.77595e-5,2.56082e-5,-1.67537e-4,-2.73755e-4},
{9.66438e-2,1.32264e-1,-9.53841e-3,-1.83707e-2,6.01664e-2,1.93357e-1},
{-3.09969e-1,-8.0779e-1,-5.14392e-1,-1.67153e-2,2.3387e-1,1.916e-1},
{-1.65906e-2,-5.11585e-3,2.80424e-2,3.94663e-2,-3.5572e-1,-4.39251e-1},
{4.37866e-3,-6.81795e-4,-1.0152e-2,-1.24875e-2,1.11484e-1,1.38105e-1}
};
//F.................................................
static const G4double bF[5][6]={
{9.99997e-1,-8.06132e-8,2.49797e-5,3.8512e-5,-2.37451e-4,-3.99607e-4},
{1.08782e-1,1.49306e-1,-2.50975e-3,-1.05471e-2,-1.64831e-2,1.05733e-1},
{-3.13165e-1,-8.18489e-1,-5.50832e-1,-6.74447e-2,6.06357e-1,7.39717e-1},
{-2.21976e-2,-6.84023e-3,6.65411e-2,9.48702e-2,-7.43989e-1,-1.03582e0},
{5.87088e-3,-8.1048e-4,-2.21731e-2,-2.94422e-2,2.2954e-1,3.19669e-1}
};
//Ne.................................................
static const G4double bNe[5][6]={
{9.99997e-1,-1.87404e-7,3.10276e-5,5.20007e-5,-2.98132e-4,-5.19259e-4},
{1.20783e-1,1.66407e-1,1.06608e-2,6.48772e-3,-1.53031e-1,-7.59354e-2},
{-3.15222e-1,-8.28793e-1,-6.0574e-1,-1.47812e-1,1.1576e0,1.58565e0},
{-2.89055e-2,-9.08096e-3,1.21467e-1,1.77575e-1,-1.2911e0,-1.90333e0},
{7.65342e-3,-8.85417e-4,-3.89092e-2,-5.4404e-2,3.93087e-1,5.79439e-1}
};
//Na.................................................
static const G4double bNa[5][6]={
{9.99996e-1,-2.44548e-7,3.31019e-5,6.29483e-5,-3.24667e-4,-5.95527e-4},
{1.32615e-1,1.83566e-1,3.04158e-2,3.40925e-2,-3.54681e-1,-3.63044e-1},
{-3.16092e-1,-8.38704e-1,-6.78558e-1,-2.59346e-1,1.88547e0,2.73632e0},
{-3.67233e-2,-1.18139e-2,1.91089e-1,2.87408e-1,-1.98397e0,-3.03075e0},
{9.72033e-3,-9.2638e-4,-5.95654e-2,-8.69829e-2,5.95744e-1,9.10242e-1}
};
//Mg.................................................
static const G4double bMg[5][6]={
{9.99995e-1,-2.12227e-7,2.95645e-5,6.92848e-5,-3.02153e-4,-6.05145e-4},
{1.44258e-1,2.00775e-1,5.67845e-2,7.35166e-2,-6.22861e-1,-7.62213e-1},
{-3.15754e-1,-8.48196e-1,-7.67318e-1,-4.02984e-1,2.77477e0,4.18114e0},
{-4.56307e-2,-1.50425e-2,2.72232e-1,4.23528e-1,-2.79606e0,-4.38863e0},
{1.2056e-2,-9.44637e-4,-8.28738e-2,-1.26564e-1,8.26726e-1,1.29882e0}
};
//Al.....................................................
static const G4double bAl[5][6]={
{9.99995e-1,-4.03407e-8,1.86047e-5,6.85201e-5,-2.14503e-4,-5.22528e-4},
{1.55704e-1,2.18048e-1,8.88994e-2,1.24878e-1,-9.51331e-1,-1.26824e0},
{-3.14244e-1,-8.57322e-1,-8.6719e-1,-5.75787e-1,3.78571e0,5.87052e0},
{-5.55526e-2,-1.86861e-2,3.5886e-1,5.81094e-1,-3.67623e0,-5.908e0},
{1.46269e-2,-9.79742e-4,-1.06652e-1,-1.71226e-1,1.06737e0,1.71918e0}
};
//Si.....................................................
static const G4double bSi[5][6]={
{9.99994e-1,3.00267e-7,-1.1184e-6,5.88256e-5,-4.78456e-5,-3.25731e-4},
{1.6696e-1,2.35405e-1,1.25215e-1,1.87646e-1,-1.32685e0,-1.86549e0},
{-3.1163e-1,-8.66152e-1,-9.71254e-1,-7.72715e-1,4.85654e0,7.7215e0},
{-6.63778e-2,-2.26481e-2,4.42898e-1,7.53182e-1,-4.55172e0,-7.4867e0},
{1.73883e-2,-1.07669e-3,-1.28075e-1,-2.18389e-1,1.29217e0,2.13475e0}
};
//P.....................................................
static const G4double bP[5][6]={
{9.99994e-1,8.94829e-7,-2.98434e-5,3.82193e-5,2.00584e-4,-6.40482e-6},
{1.78039e-1,2.52912e-1,1.63761e-1,2.60132e-1,-1.73287e0,-2.53185e0},
{-3.08007e-1,-8.74905e-1,-1.07146e0,-9.85062e-1,5.91697e0,9.63265e0},
{-7.79747e-2,-2.66797e-2,5.15288e-1,9.29261e-1,-5.34252e0,-9.00574e0},
{2.02892e-2,-1.33011e-3,-1.44039e-1,-2.6433e-1,1.4736e0,2.50398e0}
};
//S....................................................
static const G4double bS[5][6]={
{9.99994e-1,1.75397e-6,-6.7331e-5,6.29524e-6,5.29623e-4,4.35288e-4},
{1.88968e-1,2.70612e-1,2.01975e-1,3.40574e-1,-2.14737e0,-3.23836e0},
{-3.03499e-1,-8.83717e-1,-1.15816e0,-1.20414e0,6.88176e0,1.14841e1},
{-9.01806e-2,-3.06202e-2,5.65581e-1,1.09902e0,-5.95552e0,-1.03302e1},
{2.32694e-2,-1.80614e-3,-1.51041e-1,-3.05449e-1,1.58037e0,2.78083e0}
};
//Cl....................................................
static const G4double bCl[5][6]={
{9.99994e-1,3.07931e-6,-1.11876e-4,-4.10164e-5,9.17095e-4,9.80145e-4},
{1.99765e-1,2.88611e-1,2.37501e-1,4.25803e-1,-2.55105e0,-3.95585e0},
{-2.98206e-1,-8.9293e-1,-1.22279e0,-1.4169e0,7.67836e0,1.31601e1},
{-1.02865e-1,-3.40967e-2,5.84677e-1,1.24786e0,-6.31301e0,-1.13328e1},
{2.628e-2,-2.63995e-3,-1.46076e-1,-3.36795e-1,1.58677e0,2.92251e0}
};
//Ar....................................................
static const G4double bAr[5][6]={
{9.99993e-1,4.49776e-6,-1.65136e-4,-9.76754e-5,1.39664e-3,1.66293e-3},
{2.10469e-1,3.06924e-1,2.66793e-1,5.13797e-1,-2.90958e0,-4.63816e0},
{-2.92294e-1,-9.0256e-1,-1.25307e0,-1.6147e0,8.18574e0,1.44912e1},
{-1.15831e-1,-3.70891e-2,5.59807e-1,1.36619e0,-6.28824e0,-1.18327e1},
{2.92513e-2,-3.84903e-3,-1.24976e-1,-3.55149e-1,1.45127e0,2.86925e0}
};
//K.................................................
static const G4double bK[5][6]={
{9.99993e-1,6.01488e-6,-2.22125e-4,-1.61774e-4,1.92058e-3,2.41975e-3},
{2.21091e-1,3.25648e-1,2.87732e-1,6.01632e-1,-3.20436e0,-5.25724e0},
{-2.85814e-1,-9.12889e-1,-1.24213e0,-1.78635e0,8.34196e0,1.53776e1},
{-1.29005e-1,-3.92986e-2,4.84255e-1,1.44206e0,-5.81999e0,-1.17275e1},
{3.21555e-2,-5.54272e-3,-8.56572e-2,-3.56589e-1,1.1548e0,2.58829e0}
};
//Ca.................................................
static const G4double bCa[5][6]={
{9.99993e-1,8.01467e-6,-2.79242e-4,-2.3682e-4,2.45459e-3,3.21683e-3},
{2.31651e-1,3.44948e-1,2.9782e-1,6.85187e-1,-3.41294e0,-5.77715e0},
{-2.78858e-1,-9.24428e-1,-1.18215e0,-1.91691e0,8.07489e0,1.56969e1},
{-1.42276e-1,-4.01888e-2,3.50466e-1,1.45983e0,-4.83806e0,-1.08936e1},
{3.49529e-2,-7.90933e-3,-2.58002e-2,-3.36028e-1,6.7574e-1,2.04052e0}
};
//Sc.................................................
static const G4double bSc[5][6]={
{9.99992e-1,1.04277e-5,-3.35126e-4,-3.21042e-4,2.98507e-3,4.03325e-3},
{2.42172e-1,3.64954e-1,2.94606e-1,7.60693e-1,-3.51409e0,-6.1646e0},
{-2.71512e-1,-9.37543e-1,-1.0657e0,-1.99328e0,7.31863e0,1.53396e1},
{-1.5554e-1,-3.93862e-2,1.51477e-1,1.40614e0,-3.28024e0,-9.22338e0},
{3.76066e-2,-1.10812e-2,5.66831e-2,-2.8921e-1,-4.60274e-3,1.19273e0}
};
//Ti.................................................
static const G4double bTi[5][6]={
{9.99992e-1,1.30838e-5,-3.8407e-4,-4.09294e-4,3.47025e-3,4.81071e-3},
{2.52646e-1,3.85718e-1,2.7633e-1,8.25665e-1,-3.48888e0,-6.39017e0},
{-2.63758e-1,-9.52326e-1,-8.88179e-1,-2.0072e0,6.0196e0,1.42157e1},
{-1.68806e-1,-3.68095e-2,-1.16429e-1,1.27309e0,-1.09991e0,-6.63409e0},
{4.01184e-2,-1.50938e-2,1.6274e-1,-2.1376e-1,-8.99142e-1,2.08129e-2}
};
//V.................................................
static const G4double bV[5][6]={
{9.99991e-1,1.59363e-5,-4.27315e-4,-5.01461e-4,3.91365e-3,5.55096e-3},
{2.63096e-1,4.07357e-1,2.40649e-1,8.7645e-1,-3.31641e0,-6.42069e0},
{-2.55682e-1,-9.6909e-1,-6.43149e-1,-1.94678e0,4.11915e0,1.22255e1},
{-1.81974e-1,-3.21462e-2,-4.58817e-1,1.04913e0,1.75388e0,-3.03434e0},
{4.24541e-2,-2.00595e-2,2.93916e-1,-1.06138e-1,-2.02203e0,-1.50181e0}
};
//Cr.................................................
static const G4double bCr[5][6]={
{9.9999e-1,1.8895e-5,-4.59994e-4,-5.93663e-4,4.27684e-3,6.2009e-3},
{2.73504e-1,4.2999e-1,1.86473e-1,9.09921e-1,-2.98441e0,-6.23344e0},
{-2.47225e-1,-9.88118e-1,-3.28759e-1,-1.80252e0,1.5909e0,9.30968e0},
{-1.951e-1,-2.51242e-2,-8.76244e-1,7.25592e-1,5.2965e0,1.62177e0},
{4.46307e-2,-2.60754e-2,4.50054e-1,3.61638e-2,-3.37524e0,-3.38601e0}
};
//Mn.................................................
static const G4double bMn[5][6]={
{9.99989e-1,2.18906e-5,-4.79199e-4,-6.83164e-4,4.53467e-3,6.72485e-3},
{2.83854e-1,4.53722e-1,1.12877e-1,9.23203e-1,-2.48247e0,-5.80855e0},
{-2.38338e-1,-1.00965e0,5.61453e-2,-1.56605e0,-1.58353e0,5.42138e0},
{-2.08226e-1,-1.55138e-2,-1.36846e0,2.95168e-1,9.53392e0,7.3655e0},
{4.66619e-2,-3.32255e-2,6.30711e-1,2.15167e-1,-4.95748e0,-5.63744e0}
};
//Fe...................................................
static const G4double bFe[5][6]={
{9.99987e-1,2.482e-5,-4.82895e-4,-7.67488e-4,4.669e-3,7.09581e-3},
{2.94123e-1,4.78653e-1,1.93256e-2,9.13569e-1,-1.80354e0,-5.1304e0},
{-2.28945e-1,-1.0339e0,5.1121e-1,-1.22996e0,-5.40939e0,5.3149e-1},
{-2.21426e-1,-3.12679e-3,-1.93353e0,-2.48179e-1,1.44576e1,1.42077e1},
{4.85718e-2,-4.15796e-2,8.34879e-1,4.32418e-1,-6.76244e0,-8.25464e0}
};
//Co..................................................
static const G4double bCo[5][6]={
{9.99985e-1,2.76168e-5,-4.67522e-4,-8.43978e-4,4.65008e-3,7.27476e-3},
{3.0428e-1,5.04856e-1,-9.42256e-2,8.78905e-1,-9.44256e-1,-4.18847e0},
{-2.18939e-1,-1.06098e0,1.03431e0,-7.89123e-1,-9.87815e0,-5.36997e0},
{-2.34805e-1,1.21281e-2,-2.56765e0,-9.08015e-1,2.00436e1,2.21379e1},
{5.03942e-2,-5.11768e-2,1.0609e0,6.88631e-1,-8.77867e0,-1.12289e1}
};
//Ni..................................................
static const G4double bNi[5][6]={
{9.99982e-1,3.00792e-5,-4.33447e-4,-9.09366e-4,4.47786e-3,7.25072e-3},
{3.14283e-1,5.32444e-1,-2.27528e-1,8.16951e-1,9.53704e-2,-2.9773e0},
{-2.08186e-1,-1.09112e0,1.62237e0,-2.38394e-1,-1.49699e1,-1.22743e1},
{-2.48493e-1,3.04543e-2,-3.26601e0,-1.6876e0,2.62568e1,3.11259e1},
{5.21709e-2,-6.20908e-2,1.30683e0,9.84353e-1,-1.09909e1,-1.45449e1}
};
//Cu...................................................
static const G4double bCu[5][6]={
{9.99979e-1,3.24569e-5,-3.76717e-4,-9.64e-4,4.11997e-3,6.98761e-3},
{3.24082e-1,5.61534e-1,-3.79993e-1,7.25313e-1,1.31342e0,-1.49163e0},
{-1.96521e-1,-1.12457e0,2.27091e0,4.27678e-1,-2.06558e1,-2.0169e1},
{-2.62655e-1,5.20812e-2,-4.02224e0,-2.59048e0,3.30508e1,4.1135e1},
{5.39556e-2,-7.44077e-2,1.57015e0,1.32022e0,-1.33802e1,-1.81848e1}
};
//Zn...................................................
static const G4double bZn[5][6]={
{9.99976e-1,3.39628e-5,-2.98845e-4,-9.99651e-4,3.58523e-3,6.47782e-3},
{3.33647e-1,5.92068e-1,-5.5102e-1,6.0363e-1,2.70712e0,2.67853e-1},
{-1.83849e-1,-1.16095e0,2.97559e0,1.20728e0,-2.69053e1,-2.90214e1},
{-2.77375e-1,7.65838e-2,-4.83023e0,-3.61238e0,4.03786e1,5.21084e1},
{5.57745e-2,-8.79952e-2,1.84848e0,1.69425e0,-1.59274e1,-2.21242e1}
};
//Ga...................................................
static const G4double bGa[5][6]={
{9.99972e-1,3.48473e-5,-1.97828e-4,-1.01659e-3,2.85509e-3,5.69725e-3},
{3.42904e-1,6.24164e-1,-7.39023e-1,4.50306e-1,4.26553e0,2.29299e0},
{-1.6994e-1,-1.20051e0,3.72868e0,2.10276e0,-3.36594e1,-3.87726e1},
{-2.92882e-1,1.04194e-1,-5.68033e0,-4.75339e0,4.81633e1,6.39614e1},
{5.77008e-2,-1.02941e-1,2.13826e0,2.10589e0,-1.86034e1,-2.63294e1}
};
//Ge.................................................
static const G4double bGe[5][6]={
{9.99968e-1,3.47804e-5,-7.11898e-5,-1.01028e-3,1.90919e-3,4.61426e-3},
{3.51793e-1,6.57815e-1,-9.42175e-1,2.6494e-1,5.97606e0,4.57222e0},
{-1.54595e-1,-1.24305e0,4.5217e0,3.11205e0,-4.08539e1,-4.93518e1},
{-3.09367e-1,1.34661e-1,-6.56214e0,-6.00882e0,5.63228e1,7.65965e1},
{5.97948e-2,-1.19174e-1,2.4357e0,2.553e0,-2.13779e1,-3.07628e1}
};
//As.................................................
static const G4double bAs[5][6]={
{9.99963e-1,3.37519e-5,8.13037e-5,-9.78638e-4,7.41412e-4,3.21498e-3},
{3.60246e-1,6.93065e-1,-1.1588e0,4.68519e-2,7.82662e0,7.09456e0},
{-1.37615e-1,-1.28855e0,5.34673e0,4.23402e0,-4.84263e1,-6.06897e1},
{-3.27017e-1,1.67941e-1,-7.46593e0,-7.37492e0,6.47773e1,8.99181e1},
{6.21158e-2,-1.36692e-1,2.73726e0,3.03374e0,-2.42209e1,-3.53873e1}
};
//Se.................................................
static const G4double bSe[5][6]={
{9.99958e-1,3.14983e-5,2.59741e-4,-9.19008e-4,-6.49202e-4,1.49008e-3},
{3.68196e-1,7.29888e-1,-1.38664e0,-2.03807e-1,9.80062e0,9.84155e0},
{-1.18788e-1,-1.33674e0,6.194e0,5.46446e0,-5.62994e1,-7.26915e1},
{-3.46033e-1,2.03729e-1,-8.38012e0,-8.84466e0,7.34319e1,1.03805e2},
{6.47255e-2,-1.55409e-1,3.03879e0,3.54516e0,-2.7098e1,-4.01572e1}
};
//Br.................................................
static const G4double bBr[5][6]={
{9.99952e-1,2.79961e-5,4.60479e-4,-8.33486e-4,-2.23214e-3,-5.16285e-4},
{3.7558e-1,7.68292e-1,-1.62392e0,-4.87674e-1,1.18854e1,1.28025e1},
{-9.79355e-2,-1.38749e0,7.0556e0,6.80198e0,-6.44113e1,-8.52915e1},
{-3.66572e-1,2.41863e-1,-9.29524e0,-1.04141e1,8.22093e1,1.18166e2},
{6.76714e-2,-1.75289e-1,3.33691e0,4.08538e0,-2.99809e1,-4.50381e1}
};
//Kr.................................................
static const G4double bKr[5][6]={
{9.99947e-1,2.2952e-5,6.82639e-4,-7.17139e-4,-4.00522e-3,-2.81601e-3},
{3.82332e-1,8.08194e-1,-1.86848e0,-8.03415e-1,1.4064e1,1.5956e1},
{-7.48735e-2,-1.44031e0,7.92233e0,8.2382e0,-7.26858e1,-9.83865e1},
{-3.88797e-1,2.81837e-1,-1.02005e1,-1.20716e1,9.10175e1,1.32873e2},
{7.10017e-2,-1.96182e-1,3.6278e0,4.65e0,-3.28365e1,-4.99823e1}
};
//Rb.................................................
static const G4double bRb[5][6]={
{9.99941e-1,1.63607e-5,9.28242e-4,-5.69257e-4,-5.98245e-3,-5.42476e-3},
{3.88363e-1,8.49548e-1,-2.11701e0,-1.15003e0,1.63119e1,1.92737e1},
{-4.93364e-2,-1.49491e0,8.78129e0,9.76596e0,-8.10212e1,-1.11851e2},
{-4.12953e-1,3.23322e-1,-1.10814e1,-1.38073e1,9.97386e1,1.47775e2},
{7.47911e-2,-2.18001e-1,3.90642e0,5.23509e0,-3.56231e1,-5.49352e1}
};
//Sr.................................................
static const G4double bSr[5][6]={
{9.99935e-1,8.3152e-6,1.19244e-3,-3.95258e-4,-8.12525e-3,-8.28836e-3},
{3.93603e-1,8.92358e-1,-2.36688e0,-1.52786e0,1.86071e1,2.27314e1},
{-2.11345e-2,-1.55112e0,9.62205e0,1.13827e1,-8.93266e1,-1.25574e2},
{-4.39199e-1,3.66161e-1,-1.19262e1,-1.56159e1,1.08267e2,1.62736e2},
{7.90853e-2,-2.40719e-1,4.16872e0,5.83839e0,-3.8304e1,-5.98484e1}
};
//Y.................................................
static const G4double bY[5][6]={
{9.99929e-1,-1.65608e-6,1.47476e-3,-1.8625e-4,-1.04363e-2,-1.14318e-2},
{3.97989e-1,9.3643e-1,-2.61569e0,-1.93387e0,2.09305e1,2.63021e1},
{9.88744e-3,-1.60813e0,1.04351e1,1.3074e1,-9.75207e1,-1.39437e2},
{-4.67657e-1,4.09494e-1,-1.2724e1,-1.74797e1,1.16508e2,1.77614e2},
{8.39169e-2,-2.64077e-1,4.41092e0,6.45344e0,-4.08455e1,-6.46698e1}
};
//Zr.................................................
static const G4double bZr[5][6]={
{9.99922e-1,-1.37624e-5,1.77335e-3,5.994e-5,-1.29013e-2,-1.48443e-2},
{4.0145e-1,9.816e-1,-2.86053e0,-2.36562e0,2.32591e1,2.99551e1},
{4.3915e-2,-1.66522e0,1.12093e1,1.48279e1,-1.05512e2,-1.5331e2},
{-4.98475e-1,4.52607e-1,-1.34628e1,-1.93836e1,1.24357e2,1.92256e2},
{8.93258e-2,-2.87869e-1,4.62891e0,7.07473e0,-4.32116e1,-6.93461e1}
};
//Nb.................................................
static const G4double bNb[5][6]={
{9.99916e-1,-2.78975e-5,2.08753e-3,3.41607e-4,-1.55141e-2,-1.85156e-2},
{4.03889e-1,1.02783e0,-3.09762e0,-2.82183e0,2.55629e1,3.36548e1},
{8.12122e-2,-1.7221e0,1.1931e1,1.66362e1,-1.13185e2,-1.67048e2},
{-5.3188e-1,4.95236e-1,-1.41275e1,-2.13166e1,1.31687e2,2.06495e2},
{9.53781e-2,-3.12041e-1,4.81765e0,7.69813e0,-4.53587e1,-7.38182e1}
};
//Mo.................................................
static const G4double bMo[5][6]={
{9.9991e-1,-4.3988e-5,2.41439e-3,6.54636e-4,-1.82489e-2,-2.24062e-2},
{4.05249e-1,1.07495e0,-3.32445e0,-3.30073e0,2.78221e1,3.7376e1},
{1.21896e-1,-1.77813e0,1.25907e1,1.84893e1,-1.20461e2,-1.80542e2},
{-5.67942e-1,5.36741e-1,-1.4708e1,-2.32664e1,1.38408e2,2.20203e2},
{1.02086e-1,-3.36419e-1,4.97373e0,8.31905e0,-4.72562e1,-7.80409e1}
};
//Tc.................................................
static const G4double bTc[5][6]={
{9.99904e-1,-6.22073e-5,2.75066e-3,1.00063e-3,-2.10817e-2,-2.64933e-2},
{4.05472e-1,1.12279e0,-3.53825e0,-3.79964e0,3.00132e1,4.10871e1},
{1.66089e-1,-1.83255e0,1.31785e1,2.03743e1,-1.27252e2,-1.93664e2},
{-6.06729e-1,5.76392e-1,-1.51936e1,-2.52171e1,1.44423e2,2.33232e2},
{1.09463e-1,-3.60803e-1,5.09358e0,8.93178e0,-4.88706e1,-8.19632e1}
};
//Ru.................................................
static const G4double bRu[5][6]={
{9.99898e-1,-8.26232e-5,3.09353e-3,1.37924e-3,-2.39902e-2,-3.07502e-2},
{4.04498e-1,1.17115e0,-3.73628e0,-4.3158e0,3.21142e1,4.4758e1},
{2.13904e-1,-1.88458e0,1.36848e1,2.22782e1,-1.33471e2,-2.06293e2},
{-6.48301e-1,6.13439e-1,-1.55742e1,-2.71531e1,1.49638e2,2.45444e2},
{1.17516e-1,-3.8499e-1,5.17383e0,9.5307e0,-5.01709e1,-8.55368e1}
};
//Rh.................................................
static const G4double bRh[5][6]={
{9.99893e-1,-1.05293e-4,3.43932e-3,1.78981e-3,-2.69462e-2,-3.51436e-2},
{4.02272e-1,1.21982e0,-3.91575e0,-4.84621e0,3.41013e1,4.83563e1},
{2.65438e-1,-1.93341e0,1.40998e1,2.41876e1,-1.39034e2,-2.183e2},
{-6.92691e-1,6.47135e-1,-1.58398e1,-2.90581e1,1.5396e2,2.56697e2},
{1.26243e-1,-4.08783e-1,5.21125e0,1.011e1,-5.11262e1,-8.8713e1}
};
//Pd.................................................
static const G4double bPd[5][6]={
{9.99888e-1,-1.30188e-4,3.78436e-3,2.22971e-3,-2.99185e-2,-3.96315e-2},
{3.9875e-1,1.26855e0,-4.07427e0,-5.38796e0,3.5955e1,5.18546e1},
{3.20748e-1,-1.97817e0,1.44153e1,2.6089e1,-1.43866e2,-2.29578e2},
{-7.39891e-1,6.76679e-1,-1.59819e1,-3.09162e1,1.57312e2,2.66867e2},
{1.3563e-1,-4.31972e-1,5.2031e0,1.06642e1,-5.17104e1,-9.14493e1}
};
//Ag.................................................
static const G4double bAg[5][6]={
{9.99884e-1,-1.57459e-4,4.12052e-3,2.69984e-3,-3.28468e-2,-4.41512e-2},
{3.93891e-1,1.3171e0,-4.20944e0,-5.93691e0,3.76535e1,5.52188e1},
{3.79857e-1,-2.01791e0,1.46238e1,2.79653e1,-1.47893e2,-2.39997e2},
{-7.89852e-1,7.01196e-1,-1.59929e1,-3.27077e1,1.59613e2,2.75813e2},
{1.45646e-1,-4.54328e-1,5.14701e0,1.11863e1,-5.18977e1,-9.36987e1}
};
//Cd.................................................
static const G4double bCd[5][6]={
{9.99881e-1,-1.86816e-4,4.45491e-3,3.19574e-3,-3.57812e-2,-4.87457e-2},
{3.87612e-1,1.36524e0,-4.31686e0,-6.49005e0,3.91612e1,5.84032e1},
{4.4294e-1,-2.05189e0,1.47103e1,2.98033e1,-1.50988e2,-2.49389e2},
{-8.42683e-1,7.20045e-1,-1.58579e1,-3.44169e1,1.60733e2,2.83354e2},
{1.56312e-1,-4.75705e-1,5.0381e0,1.16708e1,-5.16454e1,-9.53999e1}
};
//In.................................................
static const G4double bIn[5][6]={
{9.99878e-1,-2.1848e-4,4.78376e-3,3.71897e-3,-3.86926e-2,-5.33866e-2},
{3.79879e-1,1.41265e0,-4.39391e0,-7.04282e0,4.0455e1,6.13721e1},
{5.10001e-1,-2.07896e0,1.46666e1,3.15843e1,-1.53072e2,-2.57624e2},
{-8.98306e-1,7.32213e-1,-1.55686e1,-3.60229e1,1.60591e2,2.89348e2},
{1.67591e-1,-4.95837e-1,4.87389e0,1.21106e1,-5.09274e1,-9.6506e1}
};
//Sn................................................
static const G4double bSn[5][6]={
{9.99876e-1,-2.52173e-4,5.09558e-3,4.26518e-3,-4.14939e-2,-5.79712e-2},
{3.70689e-1,1.45908e0,-4.43983e0,-7.59192e0,4.15261e1,6.4108e1},
{5.80924e-1,-2.09831e0,1.44911e1,3.32945e1,-1.54118e2,-2.64634e2},
{-9.56518e-1,7.37009e-1,-1.51243e1,-3.75098e1,1.59159e2,2.93723e2},
{1.79397e-1,-5.14573e-1,4.65426e0,1.25003e1,-4.97362e1,-9.69932e1}
};
//Sb.................................................
static const G4double bSb[5][6]={
{9.99874e-1,-2.87917e-4,5.39066e-3,4.83313e-3,-4.41852e-2,-6.24953e-2},
{3.60001e-1,1.5042e0,-4.4515e0,-8.1327e0,4.23476e1,6.65709e1},
{6.55716e-1,-2.10882e0,1.41741e1,3.49157e1,-1.54034e2,-2.70276e2},
{-1.01724e0,7.33501e-1,-1.45156e1,-3.88572e1,1.56348e2,2.96328e2},
{1.9169e-1,-5.3169e-1,4.37639e0,1.28328e1,-4.80435e1,-9.68125e1}
};
//Te.................................................
static const G4double bTe[5][6]={
{9.99874e-1,-3.25847e-4,5.67083e-3,5.42945e-3,-4.67877e-2,-6.701e-2},
{3.47783e-1,1.54767e0,-4.42606e0,-8.65995e0,4.28934e1,6.87183e1},
{7.34353e-1,-2.10941e0,1.3707e1,3.64273e1,-1.52736e2,-2.74401e2},
{-1.08036e0,7.20771e-1,-1.37344e1,-4.00427e1,1.52073e2,2.97007e2},
{2.04417e-1,-5.46972e-1,4.03784e0,1.31007e1,-4.58224e1,-9.59123e1}
};
//I.................................................
static const G4double bI[5][6]={
{9.99875e-1,-3.65406e-4,5.92115e-3,6.03332e-3,-4.91674e-2,-7.12951e-2},
{3.34062e-1,1.58924e0,-4.36361e0,-9.17241e0,4.31638e1,7.05522e1},
{8.16604e-1,-2.09933e0,1.30915e1,3.78235e1,-1.50231e2,-2.77015e2},
{-1.14554e0,6.98298e-1,-1.2784e1,-4.10591e1,1.46348e2,2.9577e2},
{2.17451e-1,-5.60344e-1,3.63994e0,1.33015e1,-4.30802e1,-9.42981e1}
};
//Xe..................................................
static const G4double bXe[5][6]={
{9.99877e-1,-4.06637e-4,6.14278e-3,6.64937e-3,-5.13391e-2,-7.53881e-2},
{3.18822e-1,1.62854e0,-4.26169e0,-9.66459e0,4.31357e1,7.20331e1},
{9.02373e-1,-2.07743e0,1.23209e1,3.90832e1,-1.46445e2,-2.77984e2},
{-1.21259e0,6.65174e-1,-1.16583e1,-4.18839e1,1.39104e2,2.92479e2},
{2.3071e-1,-5.71608e-1,3.18104e0,1.34277e1,-3.97963e1,-9.19255e1}
};
//Cs..................................................
static const G4double bCs[5][6]={
{9.99881e-1,-4.49175e-4,6.32631e-3,7.27184e-3,-5.32297e-2,-7.91972e-2},
{3.02086e-1,1.66529e0,-4.11996e0,-1.0133e1,4.28031e1,7.31464e1},
{9.91429e-1,-2.04292e0,1.13959e1,4.01931e1,-1.41369e2,-2.7726e2},
{-1.28117e0,6.2087e-1,-1.03596e1,-4.25024e1,1.30338e2,2.8709e2},
{2.44065e-1,-5.80702e-1,2.66217e0,1.34745e1,-3.59721e1,-8.87824e1}
};
//Ba..................................................
static const G4double bBa[5][6]={
{9.99886e-1,-4.93136e-4,6.47914e-3,7.90029e-3,-5.48909e-2,-8.27736e-2},
{2.8384e-1,1.69905e0,-3.93534e0,-1.05724e1,4.21392e1,7.38521e1},
{1.08366e0,-1.99451e0,1.03076e1,4.1134e1,-1.34917e2,-2.7471e2},
{-1.35107e0,5.64397e-1,-8.87993e0,-4.28945e1,1.19972e2,2.7947e2},
{2.57431e-1,-5.87417e-1,2.08111e0,1.34353e1,-3.15844e1,-8.48268e1}
};
//La..................................................
static const G4double bLa[5][6]={
{9.99892e-1,-5.38122e-4,6.59206e-3,8.52848e-3,-5.62508e-2,-8.60251e-2},
{2.6413e-1,1.72952e0,-3.70812e0,-1.09794e1,4.1143e1,7.4141e1},
{1.17876e0,-1.93141e0,9.05928e0,4.18928e1,-1.27099e2,-2.70309e2},
{-1.42185e0,4.95292e-1,-7.22427e0,-4.30461e1,1.08024e2,2.69601e2},
{2.70647e-1,-5.91727e-1,1.43982e0,1.33056e1,-2.66423e1,-8.00557e1}
};
//Ce..................................................
static const G4double bCe[5][6]={
{9.999e-1,-5.83896e-4,6.66373e-3,9.15389e-3,-5.72988e-2,-8.89368e-2},
{2.42961e-1,1.75635e0,-3.43619e0,-1.13482e1,3.97928e1,7.39747e1},
{1.27654e0,-1.85263e0,7.64589e0,4.24492e1,-1.17853e2,-2.63933e2},
{-1.49322e0,4.12926e-1,-5.38909e0,-4.29363e1,9.44419e1,2.57363e2},
{2.83597e-1,-5.93557e-1,7.3754e-1,1.30785e1,-2.11316e1,-7.44329e1}
};
//Pr..................................................
static const G4double bPr[5][6]={
{9.99909e-1,-6.301e-4,6.69295e-3,9.77077e-3,-5.8022e-2,-9.14799e-2},
{2.20373e-1,1.77919e0,-3.11907e0,-1.16754e1,3.80818e1,7.33384e1},
{1.37668e0,-1.75734e0,6.06819e0,4.27905e1,-1.07169e2,-2.55543e2},
{-1.56475e0,3.16885e-1,-3.37687e0,-4.25518e1,7.92269e1,2.42726e2},
{2.96124e-1,-5.92915e-1,-2.45072e-2,1.27501e1,-1.50562e1,-6.79514e1}
};
//Nd..................................................
static const G4double bNd[5][6]={
{9.9992e-1,-6.77211e-4,6.67917e-3,1.03866e-2,-5.84208e-2,-9.36775e-2},
{1.96406e-1,1.79753e0,-2.7553e0,-1.19533e1,3.59929e1,7.21943e1},
{1.47887e0,-1.64416e0,4.32381e0,4.28906e1,-9.50034e1,-2.45021e2},
{-1.636e0,2.06169e-1,-1.18704e0,-4.18661e1,6.23476e1,2.25579e2},
{3.08067e-1,-5.89622e-1,-8.4608e-1,1.23116e1,-8.40939e0,-6.05788e1}
};
//Pm..................................................
static const G4double bPm[5][6]={
{9.99932e-1,-7.23543e-4,6.61026e-3,1.09756e-2,-5.83892e-2,-9.53462e-2},
{1.71135e-1,1.81127e0,-2.34665e0,-1.21818e1,3.35368e1,7.05547e1},
{1.58269e0,-1.5131e0,2.4214e0,4.27487e1,-8.14091e1,-2.32422e2},
{-1.70642e0,8.13302e-2,1.16945e0,-4.0879e1,4.38728e1,2.05995e2},
{3.19226e-1,-5.84023e-1,-1.72313e0,1.17636e1,-1.21776e0,-5.23434e1}
};
//Sm..................................................
static const G4double bSm[5][6]={
{9.99946e-1,-7.69368e-4,6.48923e-3,1.15425e-2,-5.79523e-2,-9.65284e-2},
{1.44613e-1,1.81998e0,-1.89196e0,-1.23548e1,3.06998e1,6.83894e1},
{1.68776e0,-1.36316e0,3.59801e-1,4.23443e1,-6.63559e1,-2.17658e2},
{-1.77552e0,-5.81387e-2,3.69184e0,-3.95701e1,2.37855e1,1.83897e2},
{3.2942e-1,-5.76119e-1,-2.65496e0,1.10995e1,6.52027e0,-4.32241e1}
};
//Eu..................................................
static const G4double bEu[5][6]={
{9.99962e-1,-8.13654e-4,6.31217e-3,1.20763e-2,-5.70782e-2,-9.71683e-2},
{1.16915e-1,1.82343e0,-1.39208e0,-1.24689e1,2.74833e1,6.56914e1},
{1.79364e0,-1.194e0,-1.85513e0,4.16665e1,-4.9868e1,-2.00724e2},
{-1.84272e0,-2.12012e-1,6.37189e0,-3.79293e1,2.12742e0,1.59295e2},
{3.38449e-1,-5.66167e-1,-3.63838e0,1.03165e1,1.47866e1,-3.32296e1}
};
//Gd..................................................
static const G4double bGd[5][6]={
{9.9998e-1,-8.57331e-4,6.07736e-3,1.25819e-2,-5.57471e-2,-9.72475e-2},
{8.8154e-2,1.82105e0,-8.47718e-1,-1.25186e1,2.38891e1,6.2451e1},
{1.89976e0,-1.00428e0,-4.21799e0,4.0697e1,-3.19703e1,-1.81604e2},
{-1.90731e0,-3.8108e-1,9.2019e0,-3.59386e1,-2.10594e1,1.32192e2},
{3.46059e-1,-5.54093e-1,-4.67039e0,9.40924e0,2.35631e1,-2.23653e1}
};
//Tb..................................................
static const G4double bTb[5][6]={
{9.99999e-1,-8.99343e-4,5.7868e-3,1.30512e-2,-5.39764e-2,-9.67723e-2},
{5.84024e-2,1.81266e0,-2.59193e-1,-1.25002e1,1.99139e1,5.86545e1},
{2.00567e0,-7.93885e-1,-6.72461e0,3.94244e1,-1.26714e1,-1.60273e2},
{-1.96873e0,-5.64782e-1,1.21753e1,-3.3588e1,-4.57487e1,1.02578e2},
{3.5205e-1,-5.40282e-1,-5.74837e0,8.37487e0,3.28364e1,-1.06341e1}
};
//Dy..................................................
static const G4double bDy[5][6]={
{1.00002e0,-9.39311e-4,5.43607e-3,1.34763e-2,-5.17249e-2,-9.56765e-2},
{2.78005e-2,1.79795e0,3.70737e-1,-1.24114e1,1.55759e1,5.43184e1},
{2.1107e0,-5.62473e-1,-9.36239e0,3.78424e1,7.94516e0,-1.36806e2},
{-2.02617e0,-7.62769e-1,1.52768e1,-3.08728e1,-7.18354e1,7.05554e1},
{3.5614e-1,-5.25061e-1,-6.86679e0,7.21247e0,4.25672e1,1.92463e0}
};
//Ho...................................................
static const G4double bHo[5][6]={
{1.00004e0,-9.76747e-4,5.02266e-3,1.3853e-2,-4.89737e-2,-9.3934e-2},
{-3.54744e-3,1.77664e0,1.04132e0,-1.22473e1,1.08741e1,4.94269e1},
{2.21429e0,-3.09743e-1,-1.21254e1,3.59359e1,2.986e1,-1.11177e2},
{-2.07893e0,-9.74603e-1,1.84979e1,-2.77795e1,-9.928e1,3.61173e1},
{3.5809e-1,-5.08796e-1,-8.02233e0,5.91833e0,5.27374e1,1.53069e1}
};
//Er...................................................
static const G4double bEr[5][6]={
{1.00007e0,-1.01117e-3,4.54349e-3,1.4172e-2,-4.56907e-2,-9.14834e-2},
{-3.54805e-2,1.74848e0,1.74921e0,-1.20063e1,5.83106e0,4.40029e1},
{2.3157e0,-3.56472e-2,-1.49988e1,3.37013e1,5.29727e1,-8.34879e1},
{-2.12613e0,-1.19953e0,2.18208e1,-2.43067e1,-1.27958e2,-6.05797e-1},
{3.57593e-1,-4.91966e-1,-9.2087e0,4.49274e0,6.33009e1,2.94631e1}
};
//Tm...................................................
static const G4double bTm[5][6]={
{1.00009e0,-1.04219e-3,3.99979e-3,1.44301e-2,-4.18849e-2,-8.83308e-2},
{-6.78832e-2,1.71323e0,2.49363e0,-1.16834e1,4.45978e-1,3.80313e1},
{2.41433e0,2.59886e-1,-1.79765e1,3.11249e1,7.72617e1,-5.37152e1},
{-2.16704e0,-1.4368e0,2.52369e1,-2.04429e1,-1.57828e2,-3.9614e1},
{3.544e-1,-4.75067e-1,-1.04225e1,2.93269e0,7.42385e1,4.43866e1}
};
//Yb...................................................
static const G4double bYb[5][6]={
{1.00012e0,-1.06927e-3,3.39091e-3,1.46208e-2,-3.75471e-2,-8.44506e-2},
{-1.00611e-1,1.67068e0,3.27246e0,-1.12756e1,-5.27058e0,3.15152e1},
{2.50948e0,5.76648e-1,-2.10477e1,2.81988e1,1.02665e2,-2.18989e1},
{-2.20084e0,-1.68528e0,2.87325e1,-1.61828e1,-1.88802e2,-8.08394e1},
{3.48226e-1,-4.58722e-1,-1.16589e1,1.23735e0,8.55166e1,6.00477e1}
};
//Lu...................................................
static const G4double bLu[5][6]={
{1.00015e0,-1.09216e-3,2.71745e-3,1.47404e-2,-3.26779e-2,-7.98343e-2},
{-1.33495e-1,1.62059e0,4.08307e0,-1.07793e1,-1.13042e1,2.44607e1},
{2.60038e0,9.14519e-1,-2.42e1,2.49142e1,1.29108e2,1.19101e1},
{-2.22661e0,-1.94391e0,3.22924e1,-1.15206e1,-2.20784e2,-1.24202e2},
{3.38764e-1,-4.43546e-1,-1.29122e1,-5.9428e-1,9.70972e1,7.64128e1}
};
//Hf..................................................
static const G4double bHf[5][6]={
{1.00018e0,-1.11005e-3,1.98019e-3,1.47814e-2,-2.72814e-2,-7.44728e-2},
{-1.66392e-1,1.56288e0,4.92359e0,-1.01915e1,-1.76472e1,1.68656e1},
{2.68631e0,1.27279e0,-2.74231e1,2.12642e1,1.56537e2,4.76851e1},
{-2.24352e0,-2.21096e0,3.59035e1,-6.45262e0,-2.53694e2,-1.69646e2},
{3.25734e-1,-4.30391e-1,-1.41779e1,-2.56236e0,1.08949e2,9.34555e1}
};
//Ta..................................................
static const G4double bTa[5][6]={
{1.00021e0,-1.1227e-3,1.1875e-3,1.47467e-2,-2.14258e-2,-6.84575e-2},
{-1.99145e-1,1.49739e0,5.79226e0,-9.50812e0,-2.42938e1,8.72312e0},
{2.76656e0,1.65088e0,-3.07077e1,1.72391e1,1.84905e2,8.54178e1},
{-2.25073e0,-2.48478e0,3.95541e1,-9.72452e-1,-2.87462e2,-2.17133e2},
{3.0885e-1,-4.20088e-1,-1.54515e1,-4.66797e0,1.21043e2,1.11156e2}
};
//W...................................................
static const G4double bW[5][6]={
{1.00025e0,-1.12974e-3,3.25083e-4,1.46196e-2,-1.49818e-2,-6.15916e-2},
{-2.31499e-1,1.42403e0,6.68301e0,-8.72894e0,-3.12005e1,8.13039e-2},
{2.84004e0,2.04799e0,-3.40292e1,1.28426e1,2.14035e2,1.24907e2},
{-2.24701e0,-2.76343e0,4.3216e1,4.91177e0,-3.2188e2,-2.66423e2},
{2.87708e-1,-4.13579e-1,-1.67235e1,-6.90724e0,1.33305e2,1.29426e2}
};
//Re..................................................
static const G4double bRe[5][6]={
{1.00028e0,-1.13018e-3,-5.85208e-4,1.44008e-2,-8.12721e-3,-5.40793e-2},
{-2.63328e-1,1.34285e0,7.59541e0,-7.85118e0,-3.83723e1,-9.07534e0},
{2.90612e0,2.46267e0,-3.73829e1,8.06978e0,2.43918e2,1.66172e2},
{-2.23164e0,-3.04426e0,4.68824e1,1.12007e1,-3.56918e2,-3.17506e2},
{2.62068e-1,-4.12056e-1,-1.79911e1,-9.27922e0,1.45719e2,1.48255e2}
};
//Os..................................................
static const G4double bOs[5][6]={
{1.00032e0,-1.12442e-3,-1.55891e-3,1.40881e-2,-7.3226e-4,-4.57754e-2},
{-2.94323e-1,1.25374e0,8.52167e0,-6.87354e0,-4.57537e1,-1.8691e1},
{2.96357e0,2.89403e0,-4.07385e1,2.92128e0,2.74335e2,2.08986e2},
{-2.20324e0,-3.32515e0,5.05197e1,1.78893e1,-3.92323e2,-3.70113e2},
{2.31474e-1,-4.16536e-1,-1.92429e1,-1.1781e1,1.58197e2,1.67545e2}
};
//Ir...................................................
static const G4double bIr[5][6]={
{1.00036e0,-1.11107e-3,-2.58076e-3,1.36692e-2,7.08911e-3,-3.6773e-2},
{-3.24316e-1,1.15695e0,9.45947e0,-5.79675e0,-5.33328e1,-2.87528e1},
{3.01159e0,3.33981e0,-4.4085e1,-2.59468e0,3.05217e2,2.53266e2},
{-2.16093e0,-3.60244e0,5.41141e1,2.49638e1,-4.28002e2,-4.24126e2},
{1.95645e-1,-4.28567e-1,-2.04742e1,-1.44066e1,1.70702e2,1.87249e2}
};
//Pt...................................................
static const G4double bPt[5][6]={
{1.00039e0,-1.09011e-3,-3.65347e-3,1.31413e-2,1.53649e-2,-2.70335e-2},
{-3.53028e-1,1.05257e0,1.04028e1,-4.62034e0,-6.10681e1,-3.92186e1},
{3.04907e0,3.79831e0,-4.73979e1,-8.47391e0,3.36394e2,2.98829e2},
{-2.10347e0,-3.87306e0,5.76385e1,3.24147e1,-4.63751e2,-4.79318e2},
{1.54174e-1,-4.49506e-1,-2.16756e1,-1.71516e1,1.83161e2,2.07283e2}
};
//Au...................................................
static const G4double bAu[5][6]={
{1.00043e0,-1.06114e-3,-4.7734e-3,1.25001e-2,2.40713e-2,-1.65728e-2},
{-3.80207e-1,9.40778e-1,1.13464e1,-3.34423e0,-6.89259e1,-5.00549e1},
{3.07493e0,4.26732e0,-5.06567e1,-1.47102e1,3.67721e2,3.45521e2},
{-2.0297e0,-4.13334e0,6.10696e1,4.02298e1,-4.99399e2,-5.35495e2},
{1.06694e-1,-4.80932e-1,-2.28393e1,-2.00106e1,1.95512e2,2.27574e2}
};
//Hg.................................................
static const G4double bHg[5][6]={
{1.00047e0,-1.02377e-3,-5.93877e-3,1.17393e-2,3.32021e-2,-5.38025e-3},
{-4.05577e-1,8.21834e-1,1.22846e1,-1.96909e0,-7.68675e1,-6.12207e1},
{3.08808e0,4.74434e0,-5.38383e1,-2.12945e1,3.99035e2,3.93158e2},
{-1.93842e0,-4.37923e0,6.43816e1,4.83938e1,-5.3475e2,-5.92428e2},
{5.28167e-2,-5.24544e-1,-2.39565e1,-2.29768e1,2.07685e2,2.48037e2}
};
//Tl.................................................
static const G4double bTl[5][6]={
{1.00051e0,-9.77691e-4,-7.14416e-3,1.08563e-2,4.27196e-2,6.50699e-3},
{-4.28882e-1,6.96049e-1,1.32126e1,-4.94649e-1,-8.48638e1,-7.26907e1},
{3.08745e0,5.22659e0,-5.69235e1,-2.82201e1,4.30208e2,4.41611e2},
{-1.82849e0,-4.60634e0,6.7553e1,5.68935e1,-5.69648e2,-6.49946e2},
{-7.81506e-3,-5.82177e-1,-2.50201e1,-2.60444e1,2.19621e2,2.68607e2}
};
//Pb.................................................
static const G4double bPb[5][6]={
{1.00055e0,-9.22411e-4,-8.38967e-3,9.83942e-3,5.26394e-2,1.91419e-2},
{-4.49833e-1,5.63865e-1,1.41246e1,1.0768e0,-9.28737e1,-8.44177e1},
{3.07189e0,5.71077e0,-5.98888e1,-3.54715e1,4.61066e2,4.9067e2},
{-1.69869e0,-4.80968e0,7.05579e1,6.57062e1,-6.03887e2,-7.07793e2},
{-7.55851e-2,-6.55871e-1,-2.60213e1,-2.92042e1,2.31249e2,2.89188e2}
};
//Bi.................................................
static const G4double bBi[5][6]={
{1.00059e0,-8.58152e-4,-9.6602e-3,8.70229e-3,6.28328e-2,3.23376e-2},
{-4.68192e-1,4.25668e-1,1.50174e1,2.74763e0,-1.00883e2,-9.64012e1},
{3.04043e0,6.19366e0,-6.27207e1,-4.30487e1,4.91532e2,5.40291e2},
{-1.54799e0,-4.98431e0,7.3381e1,7.48252e1,-6.37365e2,-7.65885e2},
{-1.5081e-1,-7.47659e-1,-2.69551e1,-3.24524e1,2.42528e2,3.09744e2}
};
//Po.................................................
static const G4double bPo[5][6]={
{1.00062e0,-7.84582e-4,-1.09553e-2,7.43533e-3,7.33106e-2,4.61358e-2},
{-4.83646e-1,2.8205e-1,1.58842e1,4.51493e0,-1.08846e2,-1.08589e2},
{2.99185e0,6.67128e0,-6.53938e1,-5.09329e1,5.21419e2,5.90246e2},
{-1.37512e0,-5.12439e0,7.59949e1,8.42238e1,-6.69862e2,-8.23945e2},
{-2.33878e-1,-8.59879e-1,-2.78125e1,-3.57783e1,2.53381e2,3.30173e2}
};
//At.................................................
static const G4double bAt[5][6]={
{1.00066e0,-7.01484e-4,-1.22684e-2,6.03454e-3,8.40298e-2,6.05018e-2},
{-4.95899e-1,1.3369e-1,1.67195e1,6.37574e0,-1.16723e2,-1.20936e2},
{2.92499e0,7.13921e0,-6.78861e1,-5.91051e1,5.50562e2,6.40337e2},
{-1.17889e0,-5.22361e0,7.8376e1,9.38748e1,-7.01185e2,-8.81728e2},
{-3.25146e-1,-9.95032e-1,-2.85858e1,-3.91713e1,2.63738e2,3.50385e2}
};
//Rn.................................................
static const G4double bRn[5][6]={
{1.0007e0,-6.08729e-4,-1.35873e-2,4.50061e-3,9.48991e-2,7.53277e-2},
{-5.04647e-1,-1.86445e-2,1.75175e1,8.32729e0,-1.24477e2,-1.33405e2},
{2.83871e0,7.59269e0,-7.01766e1,-6.75463e1,5.78809e2,6.90388e2},
{-9.58143e-1,-5.27523e0,8.0502e1,1.03751e2,-7.31155e2,-9.39013e2},
{-4.24964e-1,-1.15578e0,-2.92679e1,-4.26205e1,2.73537e2,3.70298e2}
};
//Fr.................................................
static const G4double bFr[5][6]={
{1.00073e0,-5.06594e-4,-1.49195e-2,2.82752e-3,1.05984e-1,9.07179e-2},
{-5.09498e-1,-1.74114e-1,1.82691e1,1.0364e1,-1.32042e2,-1.45912e2},
{2.73154e0,8.02658e0,-7.22315e1,-7.62278e1,6.05902e2,7.40063e2},
{-7.11399e-1,-5.27209e0,8.23375e1,1.13815e2,-7.59479e2,-9.95409e2},
{-5.33773e-1,-1.34491e0,-2.98474e1,-4.61115e1,2.82675e2,3.89772e2}
};
//Ra.................................................
static const G4double bRa[5][6]={
{1.00076e0,-3.94938e-4,-1.62461e-2,1.01616e-3,1.17156e-1,1.06529e-1},
{-5.10179e-1,-3.3169e-1,1.89693e1,1.24811e1,-1.39382e2,-1.58412e2},
{2.60247e0,8.43503e0,-7.4033e1,-8.51228e1,6.31692e2,7.89166e2},
{-4.37673e-1,-5.2062e0,8.38642e1,1.2403e2,-7.85984e2,-1.05067e3},
{-6.5185e-1,-1.56551e0,-3.03186e1,-4.96307e1,2.91091e2,4.08719e2}
};
//Ac.................................................
static const G4double bAc[5][6]={
{1.00079e0,-2.74014e-4,-1.75614e-2,-9.33046e-4,1.28369e-1,1.22714e-1},
{-5.06317e-1,-4.90282e-1,1.96102e1,1.46725e1,-1.4644e2,-1.70838e2},
{2.45014e0,8.81187e0,-7.5553e1,-9.42005e1,6.55963e2,8.37419e2},
{-1.35635e-1,-5.06921e0,8.50532e1,1.34356e2,-8.10425e2,-1.10448e3},
{-7.79579e-1,-1.82077e0,-3.06725e1,-5.31628e1,2.987e2,4.2702e2}
};
//Th................................................
static const G4double bTh[5][6]={
{1.00082e0,-1.43895e-4,-1.8859e-2,-3.02404e-3,1.39581e-1,1.39236e-1},
{-4.97537e-1,-6.48585e-1,2.01838e1,1.69296e1,-1.53157e2,-1.83108e2},
{2.27321e0,9.15011e0,-7.6763e1,-1.0342e2,6.78484e2,8.84496e2},
{1.9599e-1,-4.85189e0,8.5876e1,1.44742e2,-8.32543e2,-1.15645e3},
{-9.17319e-1,-2.1142e0,-3.09003e1,-5.66889e1,3.05413e2,4.44542e2}
};
//Pa.................................................
static const G4double bPa[5][6]={
{1.00085e0,-5.30411e-6,-2.01239e-2,-5.2478e-3,1.50683e-1,1.55963e-1},
{-4.83526e-1,-8.05262e-1,2.06846e1,1.92461e1,-1.59496e2,-1.95177e2},
{2.07059e0,9.44256e0,-7.76442e1,-1.12749e2,6.99103e2,9.30193e2},
{5.58224e-1,-4.54471e0,8.63141e1,1.55146e2,-8.52163e2,-1.20633e3},
{-1.06534e0,-2.44938e0,-3.09967e1,-6.01932e1,3.11168e2,4.61192e2}
};
//U....................................................
static const G4double bU[5][6]={
{1.00087e0,1.42049e-4,-2.13455e-2,-7.61031e-3,1.61591e-1,1.72814e-1},
{-4.6392e-1,-9.58725e-1,2.11056e1,2.16134e1,-1.65402e2,-2.06974e2},
{1.841e0,9.68106e0,-7.81727e1,-1.22145e2,7.17614e2,9.74218e2},
{9.52222e-1,-4.13715e0,8.63446e1,1.65515e2,-8.69054e2,-1.25378e3},
{-1.22395e0,-2.83024e0,-3.09548e1,-6.36564e1,3.15887e2,4.76849e2}
};

        for(int j=0 ;j<5;j++){
	  for(int k=0;k<6;k++ ){
                     if (Z==1)coeffb[j][k]= bH[j][k];
                else if (Z==2)coeffb[j][k]= bHe[j][k];
                else if (Z==3)coeffb[j][k]= bLi[j][k];
                else if (Z==4)coeffb[j][k]= bBe[j][k];
                else if (Z==5)coeffb[j][k]= bB[j][k];
                else if (Z==6)coeffb[j][k]= bC[j][k];
                else if (Z==7)coeffb[j][k]= bN[j][k];
                else if (Z==8)coeffb[j][k]= bO[j][k];
                else if (Z==9)coeffb[j][k]= bF[j][k];
                else if (Z==10)coeffb[j][k]= bNe[j][k];
                else if (Z==11)coeffb[j][k]= bNa[j][k];
                else if (Z==12)coeffb[j][k]= bMg[j][k];
                else if (Z==13)coeffb[j][k]= bAl[j][k];
                else if (Z==14)coeffb[j][k]= bSi[j][k];
                else if (Z==15)coeffb[j][k]= bP[j][k];
                else if (Z==16)coeffb[j][k]= bS[j][k];
                else if (Z==17)coeffb[j][k]= bCl[j][k];
                else if (Z==18)coeffb[j][k]= bAr[j][k];
                else if (Z==19)coeffb[j][k]= bK[j][k];
                else if (Z==20)coeffb[j][k]= bCa[j][k];
                else if (Z==21)coeffb[j][k]= bSc[j][k];
                else if (Z==22)coeffb[j][k]= bTi[j][k];
                else if (Z==23)coeffb[j][k]= bV[j][k];
                else if (Z==24)coeffb[j][k]= bCr[j][k];
                else if (Z==25)coeffb[j][k]= bMn[j][k];
                else if (Z==26)coeffb[j][k]= bFe[j][k];
                else if (Z==27)coeffb[j][k]= bCo[j][k];
                else if (Z==28)coeffb[j][k]= bNi[j][k];
                else if (Z==29)coeffb[j][k]= bCu[j][k];
                else if (Z==30)coeffb[j][k]= bZn[j][k];
                else if (Z==31)coeffb[j][k]= bGa[j][k];
                else if (Z==32)coeffb[j][k]= bGe[j][k];
                else if (Z==33)coeffb[j][k]= bAs[j][k];
                else if (Z==34)coeffb[j][k]= bSe[j][k];
                else if (Z==35)coeffb[j][k]= bBr[j][k];
                else if (Z==36)coeffb[j][k]= bKr[j][k];
                else if (Z==37)coeffb[j][k]= bRb[j][k];
                else if (Z==38)coeffb[j][k]= bSr[j][k];
                else if (Z==39)coeffb[j][k]= bY[j][k];
                else if (Z==40)coeffb[j][k]= bZr[j][k];
                else if (Z==41)coeffb[j][k]= bNb[j][k];
                else if (Z==42)coeffb[j][k]= bMo[j][k];
                else if (Z==43)coeffb[j][k]= bTc[j][k];
                else if (Z==44)coeffb[j][k]= bRu[j][k];
                else if (Z==45)coeffb[j][k]= bRh[j][k];
                else if (Z==46)coeffb[j][k]= bPd[j][k];
                else if (Z==47)coeffb[j][k]= bAg[j][k];
                else if (Z==48)coeffb[j][k]= bCd[j][k];
                else if (Z==49)coeffb[j][k]= bIn[j][k];
                else if (Z==50)coeffb[j][k]= bSn[j][k];
                else if (Z==51)coeffb[j][k]= bSb[j][k];
                else if (Z==52)coeffb[j][k]= bTe[j][k];
                else if (Z==53)coeffb[j][k]= bI[j][k];
                else if (Z==54)coeffb[j][k]= bXe[j][k];
                else if (Z==55)coeffb[j][k]= bCs[j][k];
                else if (Z==56)coeffb[j][k]= bBa[j][k];
                else if (Z==57)coeffb[j][k]= bLa[j][k];
                else if (Z==58)coeffb[j][k]= bCe[j][k];
                else if (Z==59)coeffb[j][k]= bPr[j][k];
                else if (Z==60)coeffb[j][k]= bNd[j][k];
                else if (Z==61)coeffb[j][k]= bPm[j][k];
                else if (Z==62)coeffb[j][k]= bSm[j][k];
                else if (Z==63)coeffb[j][k]= bEu[j][k];
                else if (Z==64)coeffb[j][k]= bGd[j][k];
                else if (Z==65)coeffb[j][k]= bTb[j][k];
                else if (Z==66)coeffb[j][k]= bDy[j][k];
                else if (Z==67)coeffb[j][k]= bHo[j][k];
                else if (Z==68)coeffb[j][k]= bEr[j][k];
                else if (Z==69)coeffb[j][k]= bTm[j][k];
                else if (Z==70)coeffb[j][k]= bYb[j][k];
                else if (Z==71)coeffb[j][k]= bLu[j][k];
                else if (Z==72)coeffb[j][k]= bHf[j][k];
                else if (Z==73)coeffb[j][k]= bTa[j][k];
                else if (Z==74)coeffb[j][k]= bW[j][k];
                else if (Z==75)coeffb[j][k]= bRe[j][k];
                else if (Z==76)coeffb[j][k]= bOs[j][k];
                else if (Z==77)coeffb[j][k]= bIr[j][k];
                else if (Z==78)coeffb[j][k]= bPt[j][k];
                else if (Z==79)coeffb[j][k]= bAu[j][k];
                else if (Z==80)coeffb[j][k]= bHg[j][k];
                else if (Z==81)coeffb[j][k]= bTl[j][k];
                else if (Z==82)coeffb[j][k]= bPb[j][k];
                else if (Z==83)coeffb[j][k]= bBi[j][k];
                else if (Z==84)coeffb[j][k]= bPo[j][k];
                else if (Z==85)coeffb[j][k]= bAt[j][k];
                else if (Z==86)coeffb[j][k]= bRn[j][k];
                else if (Z==87)coeffb[j][k]= bFr[j][k];
                else if (Z==88)coeffb[j][k]= bRa[j][k];
                else if (Z==89)coeffb[j][k]= bAc[j][k];
                else if (Z==90)coeffb[j][k]= bTh[j][k];
                else if (Z==91)coeffb[j][k]= bPa[j][k];
                else if (Z==92)coeffb[j][k]= bU[j][k];

                else coeffb[j][k]=0;
	  }
        }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....
