//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// $Id: G4E1Probability.cc 92144 2015-08-19 14:25:18Z gcosmo $
//
//---------------------------------------------------------------------
//
// Geant4 class G4E1Probability
//
// by V. Lara (May 2003)
//
// Modifications:
// 18.05.2010 V.Ivanchenko trying to speedup the most slow method
//            by usage of G4Pow, integer A and introduction of const members
// 17.11.2010 V.Ivanchenko perform general cleanup and simplification
//            of integration method; low-limit of integration is defined
//            by gamma energy or is zero (was always zero before)
//

#include "G4E1Probability.hh"
#include "Randomize.hh"
#include "G4Pow.hh"
#include "G4Exp.hh"
#include "G4SystemOfUnits.hh"

static const G4double tolerance = 0.1*CLHEP::keV;

G4E1Probability::G4E1Probability():G4VEmissionProbability()
{
  G4double x = CLHEP::pi*CLHEP::hbarc;
  normC = 1.0 / (x*x);
  theLevelDensityPerNucleon = 0.125/MeV;
  aLevelDensityParam = sigma0 = Egdp = GammaR = 0.0;
  Afrag = 0;
  fG4pow = G4Pow::GetInstance(); 
}

G4E1Probability::~G4E1Probability()
{}


G4double G4E1Probability::EmissionProbDensity(const G4Fragment& frag, 
					      G4double gammaE)
{
  // Calculate the probability density here

  // From nuclear fragment properties and the excitation energy, calculate
  // the probability density for photon evaporation from U to U - gammaE
  // (U = nucleus excitation energy, gammaE = total evaporated photon
  // energy). Fragment = nuclear fragment BEFORE de-excitation

  G4double theProb = 0.0;

  if(gammaE > tolerance) {

    G4double Uexcite = frag.GetExcitationEnergy();
    G4double U = std::max(0.0, Uexcite - gammaE);

    // Need a level density parameter.
    // For now, just use the constant approximation (not reliable near magic
    // nuclei) - is equivalent to G4ConstantLevelDensityParameter class

    if(Afrag != frag.GetA_asInt()) {
      Afrag = frag.GetA_asInt();
      aLevelDensityParam = Afrag*theLevelDensityPerNucleon;

      // Define constants for the photoabsorption cross-section (the reverse
      // process of our de-excitation)
      sigma0 = 2.5 * Afrag * millibarn;
      Egdp   = (40.3 / fG4pow->powZ(Afrag,0.2) )*MeV;
      GammaR = 0.30 * Egdp;
    }
    // VI reduce number of calls to exp 
    G4double levelDens = 
      G4Exp(2*(std::sqrt(aLevelDensityParam*U)-
	       std::sqrt(aLevelDensityParam*Uexcite)));
 
    //G4cout<<" Uexcite, gammaE = "<<Uexcite<<"  "<<gammaE<<G4endl;
    //cout<<" lev density param = "<<aLevelDensityParam<<G4endl;
    //cout<<" level densities = "<<levelDensBef<<"  "<<levelDensAft<<G4endl;
    //cout<<" sigma0 = "<<sigma0<<G4endl;
    //cout<<" Egdp, GammaR = "<<Egdp<<"  "<<GammaR<<G4endl;
    //cout<<" normC = "<<normC<<G4endl;

    // VI implementation 18.05.2010
    G4double gammaE2 = gammaE*gammaE;
    G4double gammaR2 = gammaE2*GammaR*GammaR;
    G4double egdp2   = gammaE2 - Egdp*Egdp;
    G4double sigmaAbs = sigma0*gammaR2/(egdp2*egdp2 + gammaR2); 
    theProb = normC * sigmaAbs * gammaE2 * levelDens;

    //G4cout<<" sigmaAbs = "<<sigmaAbs <<" Probability = "<<theProb<<G4endl;
  }
  return theProb;
}

G4double G4E1Probability::EmissionProbability(const G4Fragment& frag, 
                                              G4double gammaE)
{
  // From nuclear fragment properties and the excitation energy, calculate
  // the probability for photon evaporation down to last ground level.
  // fragment = nuclear fragment BEFORE de-excitation

  G4double upperLim = gammaE;
  G4double lowerLim = 0.0; 

  //G4cout << "G4E1Probability::EmissionProbability:  Emin= " << lowerLim
  //	 << " Emax= " << upperLim << G4endl;
  if( upperLim - lowerLim <=  tolerance) { return 0.0; } 

  // Need to integrate EmissionProbDensity from lowerLim to upperLim 
  // and multiply by factor 3 (?!)

  G4double integ = EmissionIntegration(frag,lowerLim,upperLim);
  return integ;
}

G4double G4E1Probability::EmissionIntegration(const G4Fragment& frag, 
					      G4double lowLim, G4double upLim)

{
  // Simple integration
  // VI replace by direct integration over 100 point

  static const G4int numIters = 100;
  G4double Step = (upLim-lowLim)/G4double(numIters);

  G4double res = 0.0;
  G4double x = lowLim - 0.5*Step;

  for(G4int i = 0; i < numIters; ++i) {
    x += Step;
    res += EmissionProbDensity(frag, x);
  }

  if(res > 0.0) { res *= Step; }
  else { res = 0.0; }

  return res;

}


