//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// $Id: G4PSTARStopping.cc 83008 2014-07-24 14:49:52Z gcosmo $

//---------------------------------------------------------------------------
//
// GEANT4 Class file
//
// Description: Data on stopping power
//
// Author:      Anton Ivantchenko 18.04.2006
//
// Organisation:        QinetiQ Ltd, UK
// Customer:            ESA/ESTEC, NOORDWIJK
// Contract:            CSMAN-5288
//
// Modifications:  
// 25.04.2010 V.Ivanchenko introduce G4LPhysicsFreeVector
// 26.04.2010 A.Ivanchenko fixed data for G4_PROPANE and G4_Pt, 
//                         checked all data
// 
//----------------------------------------------------------------------------
//

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "G4PSTARStopping.hh" 
#include "G4NISTStoppingData.hh" 

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......


G4PSTARStopping::G4PSTARStopping() : nvectors(0), emin(CLHEP::keV)
{
  // 1st initialisation 
  Initialise();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4PSTARStopping::~G4PSTARStopping()
{
  if(0 < nvectors) {
    for(size_t i=0; i<nvectors; ++i) { delete sdata[i]; }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4PSTARStopping::PrintWarning(G4int i) const
{
  G4ExceptionDescription ed;
  ed << "index of data " << i << " is <0 or >= " << nvectors 
     << " request ignored!";
  G4Exception("G4PSTARStopping::PrintWarning()","em0033",JustWarning,ed);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
 
void G4PSTARStopping::Initialise()
{
  // this method may be called several times during initialisation
  G4int nmat = G4Material::GetNumberOfMaterials();
  if(nmat == (G4int)nvectors) { return; }

  // loop via material list to add extra data
  G4int j;
  for(G4int i=0; i<nmat; ++i) {
    const G4Material* mat = (*(G4Material::GetMaterialTable()))[i];

    G4bool isThere = false;  
    for(j=0; j<(G4int)nvectors; ++j) {
      if(mat == materials[j]) {
	isThere = true;
	break;
      }
    }
    if(!isThere) {
      // check list of NIST materials
      G4String mname = mat->GetName();
      for(j=0; j<74; ++j) {
        if(mname == nameNIST[j]) {
          FindData(j, mat);
          isThere = true;
          break;
	}
      }
      if(!isThere) {
	G4String form = mat->GetChemicalFormula();
	for(j=0; j<numberOfMolecula; ++j) {
	  if(form == molecularName[j]) {
	    FindData(molecularIndex[j], mat);
	    break;
	  }
	}
      }
    }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4PSTARStopping::FindData(G4int j, const G4Material* mat) 
{
  static const G4double e0[60] = { 176.8, 216.6, 250.1, 279.6, 306.3, 351.1, 390.9, 426.8, 459.8, 489.9, 517.8, 544, 596.4, 641.3, 680.6, 715.7, 747, 775, 800, 822.6, 861.7, 894.2, 920.7, 941.5, 957.3, 968.6, 976.2, 980.5, 982, 981, 978, 973.2, 967, 959.5, 910.4, 853.6, 797.9, 746.9, 701.5, 661.5, 626.4, 595.6, 544.8, 504.4, 471.1, 442.8, 416.4, 392, 369.9, 350, 332.2, 316.4, 302.2, 289.5, 278.1, 267.9, 228.9, 200.8, 179.5, 162.6 }; 
  
  static const G4double e1[60] = { 185.5, 227.2, 262.3, 293.2, 321.2, 364.8, 404, 439.3, 471.6, 500.3, 526.4, 550.6, 600.2, 641.6, 677.4, 709.2, 737.3, 762, 783.7, 803.1, 837.2, 866.8, 891.7, 911.6, 926.6, 937.4, 944.7, 949, 950.8, 950.3, 947.9, 943.9, 938.4, 931.8, 887.1, 833.7, 779.9, 729.5, 686.8, 650.4, 618.5, 590.2, 541.4, 500.4, 465.3, 434.8, 408, 384.6, 364.4, 346.5, 330.5, 316.1, 303.2, 291.3, 280.4, 270.4, 230.7, 202.2, 180.5, 163.5 }; 
  
  static const G4double e2[60] = { 195.9, 239.9, 277, 309.7, 339.3, 387.5, 430.6, 469.4, 505, 537.2, 566.9, 594.6, 649.9, 696.7, 737.4, 773.3, 805.2, 833.4, 858.3, 880.5, 918.9, 951, 977.2, 997.6, 1013, 1023, 1030, 1034, 1034, 1033, 1029, 1023, 1016, 1008, 955.6, 895, 835.1, 779.6, 729.7, 685.6, 646.6, 612.3, 555, 509.1, 471.6, 440.3, 413.7, 390.4, 370.1, 352.1, 336, 321.4, 308.3, 296.3, 285.3, 275.2, 234.9, 206, 184, 166.7 }; 
  
  static const G4double e3[60] = { 29.73, 36.41, 42.05, 47.01, 51.5, 59.46, 66.48, 72.83, 78.66, 84.09, 89.2, 94.02, 103.7, 112.1, 119.7, 126.6, 132.8, 138.5, 143.7, 148.5, 157.1, 164.3, 170.5, 175.7, 180.1, 183.7, 186.7, 189.1, 191.1, 192.5, 193.6, 194.4, 194.8, 195, 193.4, 189.3, 184.4, 179.2, 174.1, 169.3, 164.8, 160.5, 152.7, 145.7, 139.5, 133.8, 128.6, 123.9, 119.5, 115.4, 111.7, 108.1, 104.8, 101.7, 98.74, 95.98, 84.49, 75.92, 69.3, 64.05 }; 
  
  static const G4double e4[60] = { 119.7, 146.7, 169.3, 189.3, 207.4, 239.5, 267.8, 293.3, 316.8, 338.7, 359.2, 378.7, 417, 450.4, 480.1, 506.7, 530.7, 552.6, 572.4, 590.5, 622.1, 648.3, 670, 687.7, 702, 713.2, 721.7, 727.8, 731.9, 734.1, 734.8, 734, 732, 729, 702.9, 667.2, 629.1, 592.2, 558.3, 527.8, 500.6, 476.3, 434.9, 401.2, 373.3, 349.8, 329.7, 312.1, 296.4, 282.4, 269.9, 258.7, 248.5, 239.1, 230.6, 222.7, 191.1, 168.2, 150.8, 137 }; 
  
  static const G4double e5[60] = { 92.38, 113.1, 130.6, 146.1, 160, 184.8, 206.6, 226.3, 244.4, 261.3, 277.1, 292.1, 320.6, 344.8, 365.7, 383.8, 399.6, 413.2, 425, 435.1, 451, 462, 469.2, 473.4, 475.2, 475.1, 473.7, 471.2, 468, 464.2, 460.1, 455.8, 451.3, 446.8, 424.5, 404.5, 386.7, 371, 356.8, 344, 332.3, 321.5, 301.7, 284.2, 268.6, 254.8, 242.5, 231.4, 221.5, 212.4, 204.2, 196.6, 189.7, 183.3, 177.4, 171.9, 149.4, 132.7, 119.8, 109.4 }; 
  
  static const G4double e6[60] = { 73.51, 90.03, 104, 116.2, 127.3, 147, 164.4, 180.1, 194.5, 207.9, 220.5, 232.5, 256.1, 276.8, 295.3, 311.9, 327, 340.8, 353.4, 364.9, 385.3, 402.5, 417, 429.2, 439.3, 447.6, 454.3, 459.6, 463.7, 466.6, 468.6, 469.8, 470.2, 470, 461.9, 447.2, 430.1, 412.7, 396.5, 381.4, 367.5, 354.5, 331.1, 310.6, 292.6, 276.5, 262.2, 249.4, 238.1, 227.9, 218.7, 210.2, 202.6, 195.7, 189.2, 183.2, 158.9, 140.9, 127, 116 }; 
  
  static const G4double e7[60] = { 86.08, 105.4, 121.7, 136.1, 149.1, 172.2, 192.5, 210.9, 227.7, 243.5, 258.2, 272.2, 299.7, 323.5, 344.5, 363.3, 380.2, 395.3, 409, 421.4, 442.5, 459.4, 472.8, 483.1, 490.7, 496, 499.2, 500.7, 500.8, 499.5, 497.2, 494, 490, 485.5, 457.4, 426.7, 397.7, 371.9, 349.5, 330.1, 313.2, 298.5, 274.2, 254.9, 239, 225.6, 214.4, 204.7, 196.1, 188.4, 181.3, 174.9, 168.9, 163.4, 158.2, 153.3, 133, 118.2, 106.8, 97.72 }; 
  
  static const G4double e8[60] = { 14.76, 18.07, 20.87, 23.33, 25.56, 29.51, 33, 36.15, 39.04, 41.74, 44.27, 46.66, 51.51, 55.8, 59.66, 63.18, 66.43, 69.43, 72.23, 74.84, 79.58, 83.77, 87.5, 90.81, 93.77, 96.4, 98.74, 100.8, 102.6, 104.3, 105.7, 106.9, 108, 108.9, 111.5, 112, 111.2, 109.6, 107.6, 105.4, 103.2, 101, 96.33, 91.8, 87.7, 84.05, 80.83, 78, 75.47, 73.18, 71.11, 69.21, 67.46, 65.83, 64.32, 62.91, 56.98, 52.39, 48.68, 45.58 }; 
  
  static const G4double e9[60] = { 142.5, 174.5, 201.5, 225.2, 246.7, 283.1, 315.3, 344.4, 371.1, 395.5, 418.2, 439.6, 482.3, 519, 551.3, 580.1, 606, 629.2, 650.1, 669, 701.9, 729.5, 752.2, 770.4, 784.5, 795.1, 802.6, 807.4, 810, 810.5, 809.3, 806.6, 802.7, 797.7, 762.5, 719.5, 676.2, 635.8, 599.3, 566.7, 537.9, 512.3, 468.1, 431.4, 400.8, 375, 353, 333.7, 316.9, 301.9, 288.5, 276.3, 265.3, 255.1, 245.9, 237.4, 203.4, 178.9, 160.2, 145.4 }; 
 
  static const G4double e10[60] = { 149.7, 183.3, 211.7, 236.7, 259.2, 299.3, 334.7, 366.6, 396, 423.3, 449, 473.3, 517.2, 553.4, 583.6, 608.7, 629.6, 646.8, 660.7, 671.9, 687.3, 695.5, 698.4, 697.6, 694.2, 689.1, 682.8, 675.7, 668.2, 660.5, 652.7, 644.9, 637.2, 629.6, 594.2, 563, 535.4, 510.7, 488.4, 468.2, 449.8, 432.8, 404.4, 380.1, 358.1, 337.6, 318.8, 302.2, 287.4, 274.2, 262.3, 251.5, 241.7, 232.7, 224.4, 216.8, 186, 163.6, 146.3, 132.6 }; 
 
  static const G4double e11[60] = { 141.6, 173.4, 200.2, 223.8, 245.2, 281.3, 313.3, 342.2, 368.8, 393.1, 415.6, 436.8, 479, 515.2, 547, 575.3, 600.7, 623.4, 643.8, 662.1, 694.2, 721, 743.2, 760.9, 774.8, 785.2, 792.6, 797.6, 800.3, 801.1, 800.3, 797.9, 794.4, 789.9, 756.4, 714.5, 671.4, 630.5, 593.2, 559.7, 530, 503.5, 459, 423, 393.3, 368.3, 346.9, 328.1, 311.6, 296.9, 283.7, 271.8, 261, 251.1, 242, 233.7, 200.3, 176.3, 158, 143.5 }; 
 
  static const G4double e12[60] = { 138.4, 169.5, 195.7, 218.8, 239.7, 270.2, 297.8, 322.7, 345.4, 365.1, 382.6, 398.7, 433.5, 462.4, 487.4, 509.9, 530, 547.7, 563.4, 577.6, 603.9, 628.3, 650.1, 668.6, 683.5, 695.3, 704.5, 711.4, 716.3, 719.4, 720.8, 720.9, 719.7, 717.4, 694.5, 661.1, 624.7, 589.1, 556, 526, 499.1, 475, 435.8, 404.4, 377.9, 354.6, 334, 315.9, 300.2, 286.2, 273.7, 262.4, 252.2, 242.7, 234.1, 226.1, 194.2, 171.1, 153.4, 139.4 }; 
 
  static const G4double e13[60] = { 142.6, 175.3, 203.1, 228, 250.5, 283.4, 313.3, 340.4, 365.1, 386.7, 405.9, 423.5, 462, 494, 521.9, 546.9, 569.3, 589.2, 606.9, 622.9, 652.4, 679.9, 704.5, 725.3, 742.2, 755.6, 766.1, 774, 779.6, 783.1, 784.7, 784.8, 783.4, 780.7, 753.8, 714.5, 671.4, 629.1, 590.2, 555.6, 524.9, 497.7, 453.9, 419.3, 390.3, 365.1, 343, 323.7, 307.1, 292.3, 279.1, 267.3, 256.6, 246.8, 237.8, 229.5, 196.7, 173, 155, 140.8 }; 
 
  static const G4double e14[60] = { 225.8, 276.5, 319.3, 357, 391.1, 445.9, 494.9, 539.1, 579.6, 616, 649.5, 680.6, 742.9, 795.1, 840.1, 879.9, 914.8, 945.5, 972.4, 996.1, 1037, 1071, 1099, 1120, 1135, 1145, 1150, 1152, 1152, 1148, 1142, 1135, 1126, 1115, 1052, 981.5, 912.8, 849.7, 795.6, 749, 708.3, 672.4, 611.7, 562.1, 520.6, 485.3, 454.9, 428.4, 405.5, 385.2, 367.1, 350.8, 336.2, 322.8, 310.5, 299.3, 254.6, 222.7, 198.6, 179.6 }; 
 
  static const G4double e15[60] = { 119.2, 146, 168.6, 188.5, 206.5, 235.1, 260.7, 283.8, 305, 323.9, 341.3, 357.4, 390.7, 418.9, 443.7, 465.8, 485.7, 503.6, 519.5, 534.1, 560.1, 583.2, 603.2, 620, 633.7, 644.7, 653.4, 660.1, 665, 668.4, 670.3, 671.1, 670.7, 669.5, 652.9, 626.4, 596.2, 565.7, 537, 510.7, 486.8, 465.1, 427.5, 396.1, 369.6, 347, 327.4, 310.1, 294.9, 281.3, 269, 258, 247.9, 238.6, 230.2, 222.4, 191, 168.2, 150.8, 137 }; 
 
  static const G4double e16[60] = { 106.4, 130.3, 150.4, 168.2, 184.2, 212.7, 237.9, 260.6, 281.4, 300.9, 319.1, 336.4, 370.9, 401.2, 428.3, 452.8, 475.1, 495.6, 514.4, 531.8, 562.6, 588.8, 611.2, 630.2, 646.1, 659.3, 670.1, 678.6, 685.2, 690, 693.2, 694.9, 695.4, 694.8, 679, 650.5, 617.3, 583.6, 552.1, 523.6, 497.8, 474.6, 434.7, 401.8, 374.3, 351, 330.9, 313.2, 297.6, 283.7, 271.2, 259.9, 249.6, 240.3, 231.7, 223.7, 191.9, 168.9, 151.4, 137.5 }; 
 
  static const G4double e17[60] = { 74.48, 91.21, 105.3, 117.8, 129, 149, 166.5, 182.4, 197, 210.6, 223.4, 235.5, 259.6, 280.7, 299.6, 316.6, 332, 346.2, 359.1, 371.1, 392.2, 410.1, 425.2, 438, 448.7, 457.5, 464.6, 470.3, 474.7, 477.9, 480.1, 481.4, 481.9, 481.7, 473, 456.7, 437.6, 417.8, 398.6, 380.6, 364, 348.7, 322.9, 301.7, 283.5, 267.5, 253.4, 240.9, 229.8, 219.9, 210.8, 202.5, 194.8, 187.7, 181.3, 175.5, 151.6, 134.4, 121.1, 110.6 }; 
 
  static const G4double e18[60] = { 135.4, 165.8, 191.5, 214.1, 234.5, 270.8, 302.8, 331.7, 358.2, 383, 406.2, 428.2, 471.7, 509.8, 543.6, 574, 601.5, 626.5, 649.3, 670.1, 706.4, 736.5, 761.2, 781.4, 797.4, 809.8, 819, 825.3, 829.2, 830.8, 830.6, 828.7, 825.4, 820.9, 785.9, 741.4, 696, 653.6, 618.4, 588.8, 562.8, 539.6, 499.3, 464.8, 434.6, 407.9, 384.2, 363.4, 344.9, 328.4, 313.6, 300.3, 288.2, 277.2, 267, 257.7, 220.4, 193.5, 173, 156.9 }; 
 
  static const G4double e19[60] = { 132.2, 161.9, 187, 209, 229, 262.6, 292.5, 319.4, 344.2, 366.8, 387.8, 407.5, 447, 481, 510.8, 537.6, 561.6, 583.2, 602.7, 620.4, 651.5, 677.9, 700, 718.2, 732.7, 744.1, 752.7, 759, 763.2, 765.5, 766.2, 765.5, 763.5, 760.6, 734.4, 698.2, 659.3, 621.4, 586, 553.8, 524.9, 498.9, 452.4, 413.9, 382.9, 358, 337.7, 319.7, 303.9, 289.7, 277.1, 265.6, 255.2, 245.6, 236.9, 228.8, 196.4, 173, 155, 140.8 }; 
 
  static const G4double e20[60] = { 123.5, 151.3, 174.7, 195.3, 214, 246.1, 274.5, 300.2, 323.8, 345.6, 365.9, 385, 423, 455.7, 484.7, 510.5, 533.8, 554.8, 573.7, 590.9, 620.8, 645.9, 666.5, 683.3, 696.6, 706.8, 714.4, 719.7, 723, 724.5, 724.5, 723.1, 720.6, 717.1, 689.6, 653.5, 615.8, 579.8, 546.6, 516.7, 490, 466.2, 426.1, 393.6, 366.6, 343.7, 324, 306.7, 291.5, 278, 265.8, 254.7, 244.7, 235.5, 227.1, 219.4, 188.4, 166.1, 149, 135.4 }; 
 
  static const G4double e21[60] = { 34.67, 42.46, 49.03, 54.82, 60.05, 69.34, 77.52, 84.92, 91.73, 98.06, 104, 109.6, 121, 130.9, 139.9, 148, 155.5, 162.3, 168.6, 174.5, 184.9, 194, 201.8, 208.5, 214.2, 219, 223, 226.3, 228.9, 231, 232.5, 233.4, 234, 234.1, 230.4, 221.9, 211.4, 200.6, 190.1, 180.6, 171.9, 164.1, 151.1, 140.6, 132.1, 125.1, 119.1, 114, 109.6, 105.6, 102.1, 98.92, 95.83, 92.83, 89.95, 87.22, 75.76, 67.41, 61.26, 56.63 }; 
 
  static const G4double e22[60] = { 34.9, 42.74, 49.35, 55.18, 60.45, 69.8, 78.04, 85.48, 92.33, 98.71, 104.7, 110.4, 121.4, 130.9, 139.2, 146.6, 153.2, 159, 164.3, 168.9, 176.9, 183.2, 188.4, 192.5, 195.9, 198.7, 201, 202.8, 204.4, 205.6, 206.7, 207.5, 208.2, 208.7, 209.9, 209.2, 207.5, 205.1, 202.3, 199.1, 195.8, 192.3, 185.4, 178.6, 172, 165.8, 159.6, 153.6, 148, 142.7, 137.8, 133.3, 129.1, 125.2, 121.7, 118.3, 104.7, 94.43, 86.45, 79.87 }; 
 
  static const G4double e23[60] = { 37.81, 46.31, 53.47, 59.78, 65.49, 75.62, 84.54, 92.61, 100, 106.9, 113.4, 119.6, 132, 143, 153, 162, 170.4, 178.2, 185.4, 192.2, 204.6, 215.6, 225.4, 234.2, 242.1, 249.2, 255.5, 261.2, 266.2, 270.7, 274.6, 278.1, 281, 283.6, 290.9, 291.5, 287.6, 281, 272.8, 263.9, 254.9, 246, 229.5, 214.8, 202.1, 191.1, 181.6, 173.4, 166.2, 159.7, 153.8, 148.5, 143.6, 139.1, 135, 131.1, 115.3, 103.5, 94.02, 86.52 }; 
 
  static const G4double e24[60] = { 135.1, 165.5, 191.1, 213.6, 234, 270.2, 302.1, 330.9, 357.4, 382.1, 405.3, 427.2, 470.7, 508.7, 542.4, 572.8, 600.3, 625.3, 648, 668.8, 705, 735.1, 759.9, 780.1, 796.1, 808.5, 817.7, 824.1, 827.9, 829.6, 829.4, 827.5, 824.2, 819.7, 784.9, 740.5, 695.1, 652.8, 617.8, 588.4, 562.8, 539.9, 499.9, 465.6, 435.5, 408.8, 385, 364.1, 345.6, 329.1, 314.3, 300.9, 288.8, 277.7, 267.6, 258.3, 220.8, 193.9, 173.4, 157.2 }; 
 
  static const G4double e25[60] = { 25.71, 31.49, 36.37, 40.66, 44.54, 51.43, 57.5, 62.99, 68.03, 72.73, 77.14, 81.32, 89.78, 97.24, 104, 110.1, 115.8, 121, 125.8, 130.4, 138.5, 145.7, 152.1, 157.7, 162.6, 166.9, 170.7, 173.9, 176.7, 179, 181, 182.5, 183.8, 184.7, 185.6, 182.1, 176.3, 169.4, 162.2, 155.2, 148.7, 142.6, 132, 123.3, 116.2, 110.2, 105, 100.6, 96.67, 93.2, 90.08, 87.27, 84.72, 82.38, 80.24, 78.26, 70.14, 63.87, 58.87, 54.76 }; 
 
  static const G4double e26[60] = { 45.91, 56.23, 64.93, 72.59, 79.52, 91.82, 102.7, 112.5, 121.5, 129.9, 137.7, 145.2, 159.7, 172.3, 183.3, 193.1, 201.7, 209.4, 216.3, 222.3, 232.3, 240, 245.6, 249.6, 252.3, 253.9, 254.5, 254.5, 253.9, 252.9, 251.5, 250, 248.2, 246.4, 236.5, 227.2, 218.9, 211.6, 205.1, 199.3, 193.9, 189, 180.1, 172.3, 165.2, 158.8, 152.6, 146.6, 140.9, 135.7, 130.8, 126.3, 122.1, 118.3, 114.7, 111.3, 97.72, 87.77, 80.18, 74.18 }; 
 
  static const G4double e27[60] = { 88.74, 108.7, 125.5, 140.3, 153.7, 177.5, 198.4, 217.4, 234.8, 251, 266.2, 280.6, 308.9, 333.6, 355.6, 375.3, 393.1, 409.2, 423.9, 437.3, 460.8, 480.4, 496.7, 510.2, 521.2, 530.1, 537.1, 542.5, 546.4, 549.1, 550.6, 551.2, 551, 550.1, 537.6, 518.1, 496.1, 473.9, 451.2, 429.2, 408.8, 390.1, 357.9, 331.5, 309.8, 291.7, 276.2, 262.4, 250.1, 239.1, 229.2, 220.2, 212, 204.6, 197.7, 191.4, 165.6, 146.7, 132.1, 120.4 }; 
 
  static const G4double e28[60] = { 746.5, 914.3, 1056, 1180, 1293, 1493, 1669, 1829, 1975, 2112, 2240, 2361, 2586, 2777, 2942, 3084, 3208, 3316, 3409, 3490, 3618, 3710, 3771, 3808, 3823, 3822, 3807, 3781, 3745, 3702, 3654, 3601, 3545, 3487, 3183, 2890, 2629, 2402, 2204, 2031, 1882, 1753, 1544, 1384, 1258, 1159, 1077, 1007, 947, 894.2, 847.5, 805.9, 768.6, 734.9, 704.3, 676.4, 566.7, 489.7, 432.5, 388.1 }; 
 
  static const G4double e29[60] = { 184.2, 225.6, 260.6, 291.3, 319.1, 368.5, 412, 451.3, 487.4, 521.1, 552.7, 582.6, 641.3, 691.9, 736.7, 776.6, 812.5, 844.9, 874.2, 900.8, 946.6, 984.1, 1014, 1039, 1058, 1072, 1083, 1089, 1093, 1094, 1093, 1090, 1085, 1079, 1032, 973, 911.2, 852.1, 797.5, 748.3, 704.5, 665.8, 600.4, 547.7, 504.5, 468.3, 437.7, 411.3, 388.2, 368, 349.9, 333.8, 319.3, 306.2, 294.2, 283.2, 239.6, 208.6, 185.2, 167 }; 
 
  static const G4double e30[60] = { 145.4, 178.1, 205.6, 229.9, 251.8, 286.2, 317.1, 344.9, 370.4, 393.1, 413.7, 432.9, 472.6, 506.1, 535.2, 561.3, 584.6, 605.3, 623.8, 640.4, 670.2, 696.5, 719.1, 737.7, 752.5, 763.9, 772.5, 778.6, 782.6, 784.6, 784.9, 783.8, 781.4, 778, 749.2, 710.6, 669.6, 630, 593.5, 560.4, 530.8, 504.4, 459.6, 423.3, 393.3, 368.2, 346.8, 328, 311.6, 296.9, 283.8, 272, 261.3, 251.5, 242.4, 234.1, 200.8, 176.8, 158.4, 143.8 }; 
 
  static const G4double e31[60] = { 45.92, 56.24, 64.94, 72.6, 79.53, 91.84, 102.7, 112.5, 121.5, 129.9, 137.8, 145.2, 160.1, 173.2, 184.9, 195.5, 205.1, 213.8, 221.8, 229.1, 242, 252.7, 261.6, 268.8, 274.6, 279, 282.3, 284.5, 285.8, 286.2, 286, 285.1, 283.8, 281.9, 268.6, 252.2, 236.2, 221.7, 209, 198.2, 188.9, 180.9, 168.1, 158.1, 150.2, 143.7, 138.1, 133.2, 128.8, 124.8, 121.2, 117.8, 114.6, 111.6, 108.8, 106.1, 94.63, 85.43, 77.91, 71.84 }; 
 
  static const G4double e32[60] = { 111, 135.9, 157, 175.5, 192.2, 222, 248.2, 271.9, 293.6, 313.9, 333, 351, 386.5, 417.4, 444.8, 469.4, 491.6, 511.7, 530.1, 546.7, 575.8, 599.9, 619.8, 636, 649.1, 659.3, 667.2, 672.9, 676.7, 678.9, 679.6, 679.1, 677.6, 675.1, 653, 622.4, 589.5, 557.3, 525.5, 495.6, 468.4, 444.1, 403.1, 370.6, 344.6, 323.5, 305.9, 290.3, 276.4, 263.9, 252.7, 242.4, 233.2, 224.7, 216.9, 209.6, 180.4, 159.2, 142.8, 129.9 }; 
 
  static const G4double e33[60] = { 80.87, 99.04, 114.4, 127.9, 140.1, 161.7, 180.8, 198.1, 214, 228.7, 242.6, 255.7, 281.9, 304.9, 325.4, 344, 360.9, 376.5, 390.8, 404, 427.6, 447.8, 465.3, 480.3, 493.1, 504.1, 513.3, 521.1, 527.4, 532.5, 536.5, 539.6, 541.7, 543, 540.3, 527.4, 509.2, 488.8, 467.4, 446.5, 426.9, 408.7, 376.9, 350.4, 328, 309.1, 292.7, 278.1, 265.1, 253.3, 242.5, 232.8, 223.9, 215.6, 208.2, 201.2, 173.1, 152.5, 136.8, 124.4 }; 
 
  static const G4double e34[60] = { 198.1, 242.6, 280.2, 313.3, 343.1, 391.9, 435.4, 474.6, 510.6, 543.1, 573, 601, 656.7, 703.7, 744.4, 780.4, 812.2, 840.3, 865, 887, 925, 956.5, 982.1, 1002, 1016, 1026, 1032, 1035, 1036, 1033, 1029, 1023, 1016, 1007, 953.3, 891.9, 831.3, 775.4, 725.1, 680.6, 641.3, 606.6, 546.6, 497.8, 458.8, 427.4, 401.7, 379.2, 359.6, 342.2, 326.7, 312.7, 300.1, 288.5, 277.8, 268.1, 229.1, 201, 179.7, 162.8 }; 
 
  static const G4double e35[60] = { 174.4, 213.6, 246.7, 275.8, 302.1, 345, 383.2, 417.7, 449.4, 477.9, 504.3, 528.8, 578.1, 619.8, 656, 688.2, 716.7, 742, 764.4, 784.4, 819.3, 848.7, 872.8, 892, 906.4, 916.8, 923.7, 927.8, 929.4, 928.8, 926.4, 922.3, 916.9, 910.4, 866.6, 814.5, 762, 712.8, 668.3, 628.6, 593.3, 562.1, 507.7, 463.3, 427.6, 398.9, 375.3, 354.7, 336.7, 320.6, 306.2, 293.3, 281.6, 270.9, 261, 252, 215.7, 189.5, 169.6, 153.9 }; 
 
  static const G4double e36[60] = { 272.8, 334.1, 385.8, 431.4, 472.5, 545.6, 610, 668.2, 721.8, 771.6, 818.4, 862.7, 949.2, 1024, 1090, 1148, 1200, 1246, 1288, 1325, 1387, 1436, 1473, 1500, 1518, 1528, 1532, 1530, 1524, 1514, 1501, 1485, 1467, 1447, 1338, 1227, 1127, 1041, 967.3, 904.6, 850.4, 803.2, 724.6, 661.8, 610.5, 567.7, 531.4, 499.8, 472.5, 448.3, 426.7, 407.4, 390, 374.1, 359.6, 346.4, 293.7, 256.2, 227.9, 205.7 }; 
 
  static const G4double e37[60] = { 40.18, 49.21, 56.82, 63.53, 69.59, 80.35, 89.84, 98.41, 106.3, 113.6, 120.5, 127, 140.2, 151.7, 162.1, 171.5, 180.1, 188, 195.4, 202.1, 214.3, 224.8, 233.8, 241.6, 248.3, 253.9, 258.7, 262.6, 265.7, 268.2, 270, 271.3, 272.1, 272.4, 268.7, 259.5, 248, 235.8, 224, 213.1, 203.1, 194.1, 178.9, 166.6, 156.5, 148.1, 141, 134.9, 129.5, 124.7, 120.4, 116.4, 112.6, 109.1, 105.8, 102.6, 89.8, 80.34, 73.15, 67.51 }; 
 
  static const G4double e38[60] = { 149.4, 183, 211.3, 236.2, 258.7, 298.3, 333.3, 364.8, 393.8, 420.8, 446, 469.8, 516.6, 557.2, 593.1, 625.2, 654, 680, 703.5, 724.8, 761.7, 792, 816.7, 836.5, 851.9, 863.6, 871.9, 877.4, 880.4, 881.2, 880.1, 877.3, 873.2, 867.9, 829.5, 781.9, 733.4, 688, 647.9, 612.8, 581.8, 554.2, 507.1, 468.4, 435.9, 408.3, 384.4, 363.5, 344.9, 328.4, 313.7, 300.3, 288.2, 277.2, 267, 257.7, 220.4, 193.5, 173, 156.9 }; 
 
  static const G4double e39[60] = { 146.6, 179.5, 207.3, 231.8, 253.9, 292.9, 327.3, 358.4, 386.9, 413.5, 438.4, 461.9, 508.2, 548.4, 584, 615.8, 644.5, 670.4, 693.8, 715.1, 752, 782.3, 807.1, 827, 842.5, 854.3, 862.9, 868.5, 871.6, 872.6, 871.6, 869, 865, 859.8, 822, 774.9, 726.9, 682, 643, 609.2, 579.4, 552.9, 507.4, 469.6, 437.7, 410.1, 386.1, 365.1, 346.5, 329.9, 315, 301.6, 289.4, 278.3, 268.2, 258.8, 221.3, 194.3, 173.7, 157.5 }; 
 
  static const G4double e40[60] = { 141.3, 173.1, 199.9, 223.5, 244.8, 282.5, 315.8, 345.8, 373.5, 399.2, 423.3, 446.1, 491.1, 530.2, 565, 596.1, 624.2, 649.7, 672.8, 693.8, 730.4, 760.6, 785.3, 805.3, 821.1, 833.1, 841.9, 847.9, 851.4, 852.6, 852, 849.7, 846.1, 841.2, 804.8, 758.9, 712.2, 668.5, 631.2, 599, 570.8, 545.6, 502.2, 465.8, 434.6, 407.5, 383.7, 362.8, 344.4, 327.9, 313.1, 299.8, 287.7, 276.7, 266.6, 257.3, 220, 193.2, 172.8, 156.7 }; 
 
  static const G4double e41[60] = { 141.7, 173.5, 200.4, 224, 245.4, 283.2, 316.5, 346.6, 374.3, 400, 424.2, 447, 492.1, 531.3, 566.1, 597.3, 625.4, 650.9, 674, 695.1, 731.6, 761.9, 786.7, 806.6, 822.4, 834.4, 843.3, 849.2, 852.6, 853.9, 853.2, 851, 847.3, 842.4, 805.8, 759.8, 713, 669.2, 632, 600.1, 572.1, 547.1, 504, 467.7, 436.5, 409.3, 385.3, 364.3, 345.7, 329.2, 314.4, 300.9, 288.8, 277.8, 267.6, 258.3, 220.8, 193.9, 173.4, 157.2 }; 
 
  static const G4double e42[60] = { 126.5, 155, 179, 200.1, 219.2, 253.1, 283, 310, 334.8, 357.9, 379.6, 400.2, 440.6, 475.8, 507, 535, 560.2, 583.1, 603.8, 622.7, 655.5, 682.7, 704.9, 722.9, 737.3, 748.4, 756.6, 762.3, 765.9, 767.5, 767.5, 766, 763.3, 759.4, 729.3, 689.8, 648.6, 609.1, 572.9, 540.4, 511.6, 486.2, 443.4, 408.8, 380.1, 356, 335.4, 317.2, 301.2, 286.8, 274.1, 262.6, 252.2, 242.7, 234, 225.9, 193.8, 170.5, 152.8, 138.8 }; 
 
  static const G4double e43[60] = { 41.1, 50.34, 58.12, 64.98, 71.19, 82.2, 91.9, 100.7, 108.7, 116.2, 123.3, 130, 143.4, 155.2, 165.8, 175.4, 184.2, 192.3, 199.7, 206.6, 219, 229.6, 238.9, 246.8, 253.5, 259.2, 264, 268, 271.1, 273.6, 275.4, 276.7, 277.4, 277.7, 273.7, 264.2, 252.3, 239.7, 227.6, 216.2, 205.9, 196.6, 180.7, 167.8, 157.3, 148.5, 141.1, 134.7, 129.1, 124.1, 119.7, 115.7, 111.9, 108.4, 105.1, 101.9, 88.79, 78.99, 71.48, 65.54 }; 
 
  static const G4double e44[60] = { 58.01, 71.05, 82.04, 91.73, 100.5, 116, 129.7, 142.1, 153.5, 164.1, 174, 183.5, 202.5, 219.3, 234.5, 248.3, 261, 272.8, 283.8, 294, 312.7, 329.2, 343.8, 356.9, 368.6, 379, 388.3, 396.5, 403.8, 410.2, 415.8, 420.7, 424.8, 428.4, 438.1, 437.9, 431.3, 420.8, 408.2, 394.6, 380.7, 367.1, 339, 313.8, 293.3, 277.2, 264.3, 252.6, 241.9, 232.2, 223.2, 215, 207.4, 200.4, 193.8, 187.7, 163, 144.6, 130.3, 118.9 }; 
 
  static const G4double e45[60] = { 191, 233.9, 270.1, 302, 330.8, 377.8, 419.7, 457.5, 492.1, 523.4, 552.3, 579.2, 633, 678.5, 717.9, 752.9, 783.8, 811.1, 835.3, 856.8, 894.2, 925.4, 951, 971, 985.8, 996.3, 1003, 1007, 1008, 1006, 1003, 997.7, 991.2, 983.5, 933.5, 875.4, 817.5, 763.6, 715, 671.7, 633.5, 599.6, 542.9, 497.4, 460.4, 429.7, 403.7, 381, 361.2, 343.6, 327.9, 313.8, 301, 289.3, 278.6, 268.8, 229.5, 201.3, 179.8, 162.9 }; 
 
  static const G4double e46[60] = { 99.46, 121.8, 140.7, 157.3, 172.3, 198.9, 222.4, 243.6, 263.1, 281.3, 298.4, 314.5, 346.6, 374.7, 399.8, 422.4, 443, 461.9, 479.1, 495, 523.1, 547, 567.2, 584.4, 598.7, 610.6, 620.2, 628, 633.9, 638.3, 641.3, 643, 643.6, 643.3, 630.5, 606.7, 578.3, 549.1, 520.9, 494.6, 470.9, 449.7, 413.2, 383.1, 357.8, 336.3, 317.6, 301.1, 286.3, 273.2, 261.3, 250.5, 240.8, 231.9, 223.7, 216.1, 185.7, 163.7, 146.7, 133.4 }; 
 
  static const G4double e47[60] = { 228.8, 280.2, 323.5, 361.7, 396.2, 451.9, 501.6, 546.5, 587.6, 624.6, 658.6, 690.2, 753.4, 806.4, 852.1, 892.4, 927.9, 959.1, 986.3, 1010, 1052, 1086, 1114, 1135, 1150, 1160, 1166, 1168, 1166, 1163, 1157, 1149, 1140, 1129, 1064, 992.4, 922.6, 858.6, 801.5, 751.1, 706.8, 667.8, 601.8, 548.9, 506.2, 471.3, 442.2, 416.9, 394.9, 375.4, 358, 342.3, 328.2, 315.3, 303.4, 292.6, 249.3, 218.3, 194.8, 176.3 }; 
 
  static const G4double e48[60] = { 15.4, 18.86, 21.78, 24.35, 26.68, 30.81, 34.44, 37.73, 40.75, 43.57, 46.21, 48.71, 53.82, 58.34, 62.45, 66.21, 69.7, 72.94, 75.98, 78.85, 84.11, 88.84, 93.13, 97.02, 100.6, 103.8, 106.8, 109.5, 111.9, 114.1, 116.2, 118, 119.6, 121.1, 126.1, 128.1, 127.9, 126.3, 123.6, 120.4, 116.9, 113.3, 106.4, 99.98, 94.37, 89.5, 85.18, 81.37, 78.01, 75.05, 72.42, 70.08, 67.99, 66.12, 64.44, 62.92, 56.88, 52.22, 48.46, 45.34 }; 
 
  static const G4double e49[60] = { 61.33, 75.12, 86.74, 96.97, 106.2, 122.2, 136.3, 149.1, 160.8, 171.6, 181.6, 191.1, 210, 226.3, 240.6, 253.5, 265, 275.4, 284.8, 293.3, 308, 320.3, 330.3, 338.4, 344.7, 349.4, 352.8, 355.1, 356.4, 356.9, 356.6, 355.7, 354.3, 352.4, 338.9, 322.3, 305.5, 289.9, 275.7, 263, 251.7, 241.5, 224.4, 210.3, 198.3, 188, 179, 171, 163.9, 157.5, 151.6, 146.3, 141.3, 136.6, 132.1, 128, 110.6, 97.75, 87.92, 80.2 }; 
 
  static const G4double e50[60] = { 190.1, 232.8, 268.8, 300.5, 329.2, 374.1, 414.4, 450.7, 483.9, 513.5, 540.4, 565.4, 616.5, 659.2, 696, 728.7, 757.6, 783, 805.3, 825.2, 860.1, 890.2, 915.4, 935.4, 950.4, 961.1, 968.2, 972.3, 973.8, 973, 970.2, 965.7, 959.9, 952.8, 906, 850.6, 795.1, 743.3, 696.4, 654.7, 617.9, 585.3, 530.7, 487.1, 451.3, 421.4, 395.9, 373.7, 354.4, 337.2, 321.9, 308.1, 295.7, 284.2, 273.8, 264.1, 225.7, 198.1, 177.1, 160.5 }; 
 
  static const G4double e51[60] = { 164.8, 201.8, 233, 260.5, 285.4, 324.6, 359.7, 391.4, 420.3, 446.2, 469.7, 491.7, 536.5, 574.3, 607.1, 636.2, 662.2, 685.1, 705.4, 723.6, 755.9, 783.9, 807.7, 826.9, 841.7, 852.8, 860.8, 866, 868.8, 869.6, 868.6, 866, 862.1, 857.1, 820.5, 774.5, 727.2, 682.2, 641.1, 604.1, 571.2, 542, 492.3, 452.1, 419.1, 391.8, 368.6, 348.4, 330.7, 314.9, 300.8, 288.1, 276.7, 266.1, 256.5, 247.6, 212, 186.3, 166.8, 151.3 }; 
 
  static const G4double e52[60] = { 184.9, 226.5, 261.5, 292.4, 320.3, 369.9, 413.5, 453, 489.3, 523, 554.8, 584.8, 643.9, 695.3, 740.8, 781.4, 817.9, 850.9, 880.6, 907.5, 953.7, 990.9, 1021, 1044, 1061, 1073, 1081, 1084, 1085, 1083, 1078, 1072, 1064, 1054, 994.1, 927.8, 865, 808.9, 760.1, 717.9, 681.4, 649.7, 591.2, 541, 499.2, 464.9, 436.3, 411.4, 389.8, 370.6, 353.4, 338.1, 324.2, 311.4, 299.8, 289.1, 246.4, 215.9, 192.7, 174.4 }; 
 
  static const G4double e53[60] = { 150.9, 184.9, 213.5, 238.7, 261.4, 297.8, 330.2, 359.6, 386.4, 410.5, 432.5, 453, 494.9, 530.3, 561.2, 588.8, 613.4, 635.3, 654.7, 672.3, 703.4, 730.5, 753.5, 772.3, 787, 798.3, 806.6, 812.4, 816, 817.6, 817.4, 815.8, 812.9, 808.9, 777.6, 736.5, 693.4, 651.9, 613.6, 579.1, 548.2, 520.6, 472.9, 434, 402.2, 376.1, 354.3, 335, 318.2, 303.2, 289.8, 277.7, 266.7, 256.6, 247.4, 238.9, 204.8, 180.2, 161.4, 146.5 }; 
 
  static const G4double e54[60] = { 174.9, 214.2, 247.4, 276.6, 303, 345.9, 384.1, 418.7, 450.4, 479, 505.2, 529.8, 579.1, 620.8, 657.1, 689.3, 717.9, 743.2, 765.7, 785.8, 820.8, 850.5, 875, 894.5, 909.3, 920.1, 927.4, 931.9, 933.9, 933.7, 931.6, 928, 922.9, 916.8, 874.4, 823.2, 771.3, 722.4, 676.2, 634, 596.2, 562.8, 507.2, 463.4, 428.5, 400.2, 376.7, 356.1, 338, 321.9, 307.5, 294.5, 282.8, 272, 262.1, 253, 216.6, 190.4, 170.3, 154.5 }; 
 
  static const G4double e55[60] = { 197.6, 242, 279.4, 312.4, 342.2, 395.1, 441.8, 483.9, 522.7, 558.8, 592.7, 624.8, 687.3, 741.3, 789, 831.4, 869.4, 903.6, 934.3, 962, 1009, 1048, 1078, 1101, 1119, 1132, 1140, 1145, 1146, 1145, 1141, 1135, 1128, 1119, 1060, 991.7, 924.3, 862, 806.1, 756.5, 712.8, 674.2, 605.4, 548.6, 503.3, 467.5, 438.7, 413.6, 391.8, 372.4, 355.2, 339.7, 325.7, 312.9, 301.1, 290.4, 247.5, 216.8, 193.5, 175.1 }; 
 
  static const G4double e56[60] = { 186, 227.8, 263.1, 294.1, 322.2, 372.1, 416, 455.7, 492.2, 526.2, 558.1, 588.3, 645.3, 693.8, 735.8, 772.4, 804.5, 832.6, 857.2, 878.7, 913.5, 939.2, 957.2, 969, 975.6, 978, 976.8, 972.8, 966.5, 958.4, 948.9, 938.3, 926.9, 914.9, 851.3, 789.6, 734.2, 685.4, 642.9, 605.7, 573, 543.9, 499.5, 465.2, 435.8, 409.1, 384.6, 363.2, 344.6, 328.1, 313.3, 300, 288, 277, 266.8, 257.5, 220.4, 193.6, 173.2, 157 }; 
 
  static const G4double e57[60] = { 83.27, 102, 117.8, 131.7, 144.2, 165, 183.4, 200.1, 215.3, 229.2, 242, 254, 278.7, 300, 319, 336.2, 351.9, 366.1, 379.2, 391.2, 413.1, 432.7, 450, 465.1, 478, 489.1, 498.5, 506.4, 513, 518.4, 522.7, 526, 528.4, 530.1, 528.6, 516.7, 499.4, 479.6, 460.4, 442.3, 425.4, 409.6, 381.2, 356.4, 334.6, 315.4, 298.3, 283.1, 269.6, 257.4, 246.5, 236.5, 227.4, 219.1, 211.5, 204.4, 175.9, 155.2, 139.3, 126.7 }; 
 
  static const G4double e58[60] = { 140.2, 171.7, 198.3, 221.7, 242.8, 277.9, 309, 337.2, 363, 386.4, 408.1, 428.4, 468.9, 503.4, 533.6, 560.3, 584.1, 605.3, 624.1, 640.9, 670.1, 694.5, 714.3, 729.8, 741.4, 749.6, 755, 757.9, 758.6, 757.6, 755, 751, 746, 740, 701.3, 656.7, 612.9, 572.6, 538.7, 509.8, 484.7, 462.4, 424.2, 392.4, 365.3, 341.8, 321.1, 302.9, 287.3, 273.6, 261.3, 250.1, 240.1, 231.1, 222.7, 215, 184.5, 162.6, 145.9, 132.6 }; 
 
  static const G4double e59[60] = { 2.183E+02, 2.673E+02, 3.087E+02, 3.451E+02, 3.780E+02, 4.365E+02, 4.880E+02, 5.346E+02, 5.775E+02, 6.173E+02, 6.548E+02, 6.902E+02, 7.595E+02, 8.190E+02, 8.714E+02, 9.180E+02, 9.597E+02, 9.970E+02, 1.031E+03, 1.061E+03, 1.112E+03, 1.153E+03, 1.186E+03, 1.211E+03, 1.229E+03, 1.242E+03, 1.250E+03, 1.254E+03, 1.254E+03, 1.252E+03, 1.247E+03, 1.239E+03, 1.230E+03, 1.220E+03, 1.153E+03, 1.078E+03, 1.005E+03, 9.379E+02, 8.769E+02, 8.222E+02, 7.736E+02, 7.305E+02, 6.583E+02, 6.006E+02, 5.537E+02, 5.150E+02, 4.825E+02, 4.542E+02, 4.297E+02, 4.079E+02, 3.886E+02, 3.712E+02, 3.556E+02, 3.413E+02, 3.282E+02, 3.163E+02, 2.687E+02, 2.348E+02, 2.092E+02, 1.890E+02 };

  static const G4double e60[60] =  { 1.377E+01, 1.687E+01, 1.947E+01, 2.177E+01, 2.385E+01, 2.754E+01, 3.079E+01, 3.373E+01, 3.643E+01, 3.895E+01, 4.131E+01, 4.355E+01, 4.811E+01, 5.216E+01, 5.583E+01, 5.920E+01, 6.232E+01, 6.523E+01, 6.795E+01, 7.051E+01, 7.522E+01, 7.946E+01, 8.330E+01, 8.679E+01, 8.997E+01, 9.287E+01, 9.551E+01, 9.792E+01, 1.001E+02, 1.021E+02, 1.039E+02, 1.055E+02, 1.070E+02, 1.083E+02, 1.127E+02, 1.145E+02, 1.144E+02, 1.130E+02, 1.108E+02, 1.081E+02, 1.053E+02, 1.024E+02, 9.692E+01, 9.212E+01, 8.791E+01, 8.422E+01, 8.097E+01, 7.811E+01, 7.556E+01, 7.326E+01, 7.117E+01, 6.926E+01, 6.750E+01, 6.587E+01, 6.436E+01, 6.294E+01, 5.700E+01, 5.239E+01, 4.867E+01, 4.557E+01 };

  static const G4double e61[60] = { 105, 128.6, 148.5, 166, 181.8, 210, 234.8, 257.2, 277.8, 296.9, 315, 332, 364.6, 392.5, 416.7, 437.8, 456.3, 472.5, 486.6, 498.9, 518.4, 532.3, 541.6, 547, 549.2, 549, 546.8, 543, 538, 532.2, 525.7, 518.8, 511.7, 504.4, 468.7, 437.3, 410.7, 388.4, 369.4, 353, 338.7, 325.9, 304.1, 285.9, 270.3, 256.7, 245.2, 235.1, 225.9, 217.2, 209, 201.1, 193.6, 187, 180.9, 175.3, 152.3, 135.4, 122.2, 111.7 }; 
 
  static const G4double e62[60] = { 85.64, 104.9, 121.1, 135.4, 148.3, 171.3, 191.5, 209.8, 226.6, 242.2, 256.9, 270.8, 298.1, 322, 343.1, 362.1, 379.3, 394.9, 409.1, 422.1, 444.8, 463.8, 479.7, 492.8, 503.6, 512.3, 519.2, 524.5, 528.5, 531.2, 532.9, 533.7, 533.6, 532.9, 521.9, 503.9, 483.5, 462.8, 442.9, 424.2, 406.7, 390.6, 358.9, 330.8, 307.6, 288.9, 273.6, 259.9, 247.8, 236.9, 227.1, 218.2, 210.1, 202.7, 195.9, 189.7, 164.2, 145.5, 131.1, 119.6 }; 
 
  static const G4double e63[60] = { 179.2, 219.5, 253.4, 283.3, 310.4, 352.2, 389.8, 423.8, 454.7, 482.3, 507.2, 530.3, 577.9, 617.7, 652.1, 682.6, 709.7, 733.5, 754.4, 773, 806.1, 835.1, 859.5, 879.2, 894.2, 905.1, 912.7, 917.4, 919.6, 919.6, 917.7, 914.2, 909.3, 903.3, 861.4, 810.7, 759.2, 710.8, 667.1, 628.3, 593.9, 563.4, 511.8, 470.2, 435.9, 407.2, 382.8, 361.4, 342.9, 326.4, 311.6, 298.4, 286.4, 275.4, 265.3, 256.1, 219, 192.4, 172.1, 156.1 }; 
 
  static const G4double e64[60] = { 60.88, 74.56, 86.1, 96.26, 105.4, 121.8, 136.1, 149.1, 161.1, 172.2, 182.6, 192.5, 212.3, 229.6, 245.1, 259.1, 271.8, 283.5, 294.2, 304.1, 321.6, 336.5, 349.1, 359.8, 368.7, 376, 382, 386.7, 390.2, 392.7, 394.4, 395.2, 395.3, 394.8, 385.4, 369.3, 351, 332.6, 315.4, 299.6, 285.5, 272.8, 251.3, 233.9, 219.6, 207.6, 197.4, 188.7, 181, 174.1, 167.9, 162.2, 157, 152.2, 147.7, 143.5, 126, 112.5, 101.7, 93.07 }; 
 
  static const G4double e65[60] = { 32.4, 39.68, 45.82, 51.23, 56.12, 64.8, 72.45, 79.36, 85.72, 91.64, 97.2, 102.5, 113.1, 122.4, 130.8, 138.5, 145.5, 151.9, 157.9, 163.4, 173.4, 182, 189.5, 196, 201.6, 206.4, 210.4, 213.8, 216.5, 218.7, 220.4, 221.6, 222.4, 222.9, 220.7, 213.8, 204.9, 195.4, 186, 177.3, 169.4, 162.3, 150.1, 140.3, 132.2, 125.5, 119.7, 114.7, 110.1, 106.1, 102.5, 99.14, 96.12, 93.34, 90.77, 88.4, 78.69, 71.47, 65.7, 60.79 }; 
 
  static const G4double e66[60] = { 174.5, 213.8, 246.8, 276, 302.3, 349.1, 390.3, 427.6, 461.8, 493.7, 523.6, 552, 607.8, 656.2, 699.1, 737.4, 771.9, 803, 831.1, 856.5, 900.2, 935.6, 963.8, 985.9, 1003, 1015, 1022, 1027, 1028, 1027, 1023, 1018, 1011, 1002, 947.8, 885.5, 824.9, 769.7, 720.9, 678.2, 641, 608.6, 553.2, 507.9, 470.5, 439.1, 412.4, 389, 368.6, 350.5, 334.4, 319.8, 306.6, 294.6, 283.6, 273.5, 233.2, 204.4, 182.5, 165.2 }; 
 
  static const G4double e67[60] = { 169.9, 208.1, 240.3, 268.6, 294.3, 339.8, 379.9, 416.2, 449.5, 480.6, 509.7, 537.3, 591.5, 638.4, 679.9, 716.9, 750.2, 780.2, 807.4, 832, 874.4, 909.1, 937.1, 959.3, 976.6, 989.5, 998.7, 1005, 1008, 1008, 1006, 1003, 998, 991.7, 946.9, 891.9, 836, 783.4, 735.7, 693.2, 655.4, 622, 563.5, 515.3, 475.8, 443.4, 416.2, 392.6, 371.9, 353.6, 337.3, 322.5, 309.3, 297.1, 286, 275.8, 235.1, 206, 183.9, 166.5 }; 
 
  static const G4double e68[60] = { 191.6, 234.7, 271, 302.9, 331.9, 377.2, 417.8, 454.5, 488, 517.9, 545.1, 570.4, 621.9, 665, 702.2, 735.2, 764.3, 790, 812.4, 832.5, 867.7, 897.9, 923.2, 943.3, 958.3, 969, 976, 980, 981.4, 980.4, 977.5, 972.9, 966.9, 959.8, 912.2, 856.2, 800.1, 747.8, 700.6, 658.5, 621.4, 588.5, 535.1, 492.8, 457.8, 427.9, 401.8, 379.1, 359.4, 341.9, 326.3, 312.3, 299.6, 287.9, 277.3, 267.5, 228.5, 200.4, 179.1, 162.3 }; 
 
  static const G4double e69[60] = { 18.38, 22.51, 25.99, 29.06, 31.83, 36.75, 41.09, 45.01, 48.62, 51.98, 55.13, 58.11, 64.19, 69.57, 74.44, 78.9, 83.02, 86.84, 90.42, 93.78, 99.92, 105.4, 110.3, 114.8, 118.8, 122.4, 125.7, 128.6, 131.2, 133.5, 135.6, 137.4, 139, 140.4, 144.4, 144.8, 142.7, 139.1, 134.5, 129.6, 124.2, 118.8, 108.7, 100, 92.78, 86.72, 81.64, 77.35, 73.71, 70.6, 67.92, 65.59, 63.56, 61.77, 60.19, 58.78, 53.33, 49.08, 45.62, 42.73 }; 
 
  static const G4double e70[60] = { 14.97, 18.34, 21.17, 23.67, 25.93, 29.94, 33.47, 36.67, 39.61, 42.34, 44.91, 47.34, 52.3, 56.7, 60.68, 64.33, 67.7, 70.84, 73.78, 76.55, 81.61, 86.15, 90.24, 93.95, 97.3, 100.3, 103.1, 105.6, 107.8, 109.9, 111.7, 113.3, 114.7, 116, 120, 121.2, 120.5, 118.5, 115.9, 112.9, 109.8, 106.7, 100.2, 94.14, 89.08, 85, 81.68, 78.78, 76.2, 73.88, 71.78, 69.86, 68.09, 66.45, 64.93, 63.5, 57.52, 52.9, 49.15, 46.04 }; 
 
  static const G4double e71[60] = { 133.7, 163.8, 189.1, 211.4, 231.6, 267.5, 299, 327.6, 353.8, 378.2, 401.2, 422.9, 466, 503.6, 537.2, 567.3, 594.6, 619.5, 642.1, 662.8, 698.9, 729, 753.8, 774, 790.1, 802.6, 811.9, 818.3, 822.3, 824.1, 823.9, 822.2, 819, 814.5, 780.1, 736, 695.9, 660.4, 628.6, 599.9, 573.7, 549.7, 507.5, 471.4, 440.1, 412.8, 388.8, 367.6, 348.9, 332.2, 317.2, 303.7, 291.4, 280.3, 270, 260.6, 222.8, 195.5, 174.8, 158.5 }; 
 
  static const G4double e72[60] = { 152.2, 186.5, 215.3, 240.7, 263.7, 304.5, 340.4, 372.9, 402.8, 430.6, 456.7, 481.4, 530.5, 573.2, 611.3, 645.4, 676.4, 704.4, 730, 753.3, 793.8, 827.4, 855, 877.2, 894.8, 908.2, 918.1, 924.7, 928.5, 929.9, 929.1, 926.5, 922.3, 916.9, 876.3, 825.9, 775.3, 728.5, 686.8, 650.1, 618, 589.9, 536.1, 489.2, 450.8, 420.2, 395.5, 373.8, 354.6, 337.5, 322.2, 308.3, 295.8, 284.4, 273.9, 264.3, 225.7, 198, 176.9, 160.4 }; 
 
  static const G4double e73[60] = { 38.1, 46.67, 53.88, 60.24, 65.99, 76.2, 85.2, 93.33, 100.8, 107.8, 114.3, 120.5, 132.8, 143.6, 153.2, 161.8, 169.6, 176.7, 183.2, 189.1, 199.4, 208, 215, 220.7, 225.2, 228.7, 231.2, 232.9, 233.9, 234.3, 234.1, 233.5, 232.5, 231.1, 221.3, 209.3, 197.4, 186.5, 177, 168.6, 161.3, 155, 144.4, 135.9, 128.8, 122.8, 117.5, 112.7, 108.4, 104.5, 101, 97.75, 94.8, 92.09, 89.6, 87.29, 77.8, 70.52, 64.84, 60.13 }; 
 
  if(0 == j)      { AddData(e0, mat); }
  else if(1 == j) { AddData(e1, mat); }
  else if(2 == j) { AddData(e2, mat); }
  else if(3 == j) { AddData(e3, mat); }
  else if(4 == j) { AddData(e4, mat); }
  else if(5 == j) { AddData(e5, mat); }
  else if(6 == j) { AddData(e6, mat); }
  else if(7 == j) { AddData(e7, mat); }
  else if(8 == j) { AddData(e8, mat); }
  else if(9 == j) { AddData(e9, mat); }
  else if(10 == j) { AddData(e10, mat); }
  else if(11 == j) { AddData(e11, mat); }
  else if(12 == j) { AddData(e12, mat); }
  else if(13 == j) { AddData(e13, mat); }
  else if(14 == j) { AddData(e14, mat); }
  else if(15 == j) { AddData(e15, mat); }
  else if(16 == j) { AddData(e16, mat); }
  else if(17 == j) { AddData(e17, mat); }
  else if(18 == j) { AddData(e18, mat); }
  else if(19 == j) { AddData(e19, mat); }
  else if(20 == j) { AddData(e20, mat); }
  else if(21 == j) { AddData(e21, mat); }
  else if(22 == j) { AddData(e22, mat); }
  else if(23 == j) { AddData(e23, mat); }
  else if(24 == j) { AddData(e24, mat); }
  else if(25 == j) { AddData(e25, mat); }
  else if(26 == j) { AddData(e26, mat); }
  else if(27 == j) { AddData(e27, mat); }
  else if(28 == j) { AddData(e28, mat); }
  else if(29 == j) { AddData(e29, mat); }
  else if(30 == j) { AddData(e30, mat); }
  else if(31 == j) { AddData(e31, mat); }
  else if(32 == j) { AddData(e32, mat); }
  else if(33 == j) { AddData(e33, mat); }
  else if(34 == j) { AddData(e34, mat); }
  else if(35 == j) { AddData(e35, mat); }
  else if(36 == j) { AddData(e36, mat); }
  else if(37 == j) { AddData(e37, mat); }
  else if(38 == j) { AddData(e38, mat); }
  else if(39 == j) { AddData(e39, mat); }
  else if(40 == j) { AddData(e40, mat); }
  else if(41 == j) { AddData(e41, mat); }
  else if(42 == j) { AddData(e42, mat); }
  else if(43 == j) { AddData(e43, mat); }
  else if(44 == j) { AddData(e44, mat); }
  else if(45 == j) { AddData(e45, mat); }
  else if(46 == j) { AddData(e46, mat); }
  else if(47 == j) { AddData(e47, mat); }
  else if(48 == j) { AddData(e48, mat); }
  else if(49 == j) { AddData(e49, mat); }
  else if(50 == j) { AddData(e50, mat); }
  else if(51 == j) { AddData(e51, mat); }
  else if(52 == j) { AddData(e52, mat); }
  else if(53 == j) { AddData(e53, mat); }
  else if(54 == j) { AddData(e54, mat); }
  else if(55 == j) { AddData(e55, mat); }
  else if(56 == j) { AddData(e56, mat); }
  else if(57 == j) { AddData(e57, mat); }
  else if(58 == j) { AddData(e58, mat); }
  else if(59 == j) { AddData(e59, mat); }
  else if(60 == j) { AddData(e60, mat); }
  else if(61 == j) { AddData(e61, mat); }
  else if(62 == j) { AddData(e62, mat); }
  else if(63 == j) { AddData(e63, mat); }
  else if(64 == j) { AddData(e64, mat); }
  else if(65 == j) { AddData(e65, mat); }
  else if(66 == j) { AddData(e66, mat); }
  else if(67 == j) { AddData(e67, mat); }
  else if(68 == j) { AddData(e68, mat); }
  else if(69 == j) { AddData(e69, mat); }
  else if(70 == j) { AddData(e70, mat); }
  else if(71 == j) { AddData(e71, mat); }
  else if(72 == j) { AddData(e72, mat); }
  else if(73 == j) { AddData(e73, mat); }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4PSTARStopping::AddData(const G4double* stop, const G4Material* mat)
{
  static const G4double T0[60] = { 
    0.001, 0.0015, 0.002, 0.0025, 0.003, 0.004, 0.005, 0.006, 0.007, 0.008,
    0.009, 0.01,  0.0125, 0.015, 0.0175, 0.02, 0.0225, 0.025, 0.0275, 0.03, 
    0.035, 0.04,   0.045, 0.05,   0.055, 0.06,  0.065,  0.07, 0.075, 0.08, 
    0.085, 0.09,   0.095, 0.1,    0.125, 0.15,  0.175,   0.2, 0.225, 0.25, 
    0.275, 0.3,    0.35,  0.4,    0.45,  0.5,    0.55,   0.6, 0.65,  0.7, 
    0.75,  0.8,    0.85,  0.9,    0.95,  1.,     1.25,   1.5, 1.75,  2. };

  G4LPhysicsFreeVector* v = 
    new G4LPhysicsFreeVector(60, T0[0]*CLHEP::MeV, T0[59]*CLHEP::MeV);
  for(size_t i=0; i<60; ++i) { v->PutValues(i, T0[i]*CLHEP::MeV, stop[i]*fac); }
  v->SetSpline(true);
  materials.push_back(mat);
  sdata.push_back(v);
  ++nvectors;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
