//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
// $Id: G4NucleiPropertiesTableAME03.cc 72955 2013-08-14 14:23:14Z gcosmo $
//
// -------------------------------------------------------------------
//
//      File name:     G4NucleiPropertiesTableAME03.cc 
//
//      Authors:       Tatsumi Koi (tkoi@slac.stanford.edu)
// 
//      Data are update to 
//      The Ame2003 atomic mass evaluation (II)
//      by G.Audi, A.H.Wapstra and C.Thibault 
//      Nuclear Physics A729 p. 337-676, December 22, 2003.
//
//      Creation date: Apr. 2009 
//                     based on G4NucleiPropertiesTable
//
//      Modifications: 
//

#include "G4ios.hh"
#include "G4PhysicalConstants.hh"
#include "G4SystemOfUnits.hh"
#include "G4NucleiPropertiesTableAME03.hh"

// Class G4NucleiPropertiesTableAME03 
// Determine the table index for a Nuclide with Z protons and A nucleons

G4ThreadLocal G4bool  G4NucleiPropertiesTableAME03::isIntialized = false;
G4ThreadLocal G4double G4NucleiPropertiesTableAME03::electronMass[ZMax];

G4NucleiPropertiesTableAME03::G4NucleiPropertiesTableAME03()
{
}

G4int G4NucleiPropertiesTableAME03::GetIndex(G4int Z, G4int A) 
{
  if(A>G4NucleiPropertiesTableAME03::MaxA) {
    G4Exception("G4NucleiPropertiesTableAME03::GetIndex",
		"PART201",
		EventMustBeAborted,"Nucleon number larger than 293");
    return -1;
  } else if(A<1) {
    G4Exception("G4NucleiPropertiesTableAME03::GetIndex",
		"Illegal arguemntPART201",
		EventMustBeAborted," Nucleon number is negative"); 
    return -1;
  } else if(Z>A) {
    G4Exception("G4NucleiPropertiesTableAME03::GetIndex",
		"PART201",
		EventMustBeAborted, "Nucleon number smaller than Z"); 
    return -1;
  }
   
  G4int i = shortTable[A-1];
  while ( i < shortTable[A] ) {
    if (indexArray[0][i] != Z ) {
      i++;
    } else {
      return i;
    }
  }
  return -1;
}


G4int G4NucleiPropertiesTableAME03::MinZ(G4int A)
{
  G4int i = shortTable[A-1];
  return indexArray[0][i];
}


G4int G4NucleiPropertiesTableAME03::MaxZ(G4int A)
{
  G4int i = shortTable[A]-1;
  return indexArray[0][i];
}


G4double G4NucleiPropertiesTableAME03::GetNuclearMass(G4int Z, G4int A)
{
  if (!isIntialized) {
    // calculate electron mass in orbit with binding energy
    isIntialized = true;
    for (int iz=1; iz<ZMax; iz+=1){
      electronMass[iz] =  iz*electron_mass_c2 
	  - ( 14.4381 * std::pow( G4double(iz) , 2.39 )) *eV
	  - ( 1.55468*1e-6 * std::pow( G4double(iz) , 5.35 ) ) *eV;
    }
  }

  G4double nuclearMass = GetAtomicMass(Z,A) - electronMass[Z];

  if (nuclearMass <0.0) nuclearMass = 0.0;

  return nuclearMass;

}


G4double G4NucleiPropertiesTableAME03::GetMassExcess(G4int Z, G4int A) 
{
    G4int i=GetIndex(Z, A);
    if (i >= 0) {
      return MassExcess[i]*keV;
    } else {
      return 0.0;
    }
}

G4double G4NucleiPropertiesTableAME03::GetBindingEnergy(G4int Z, G4int A)
{
  G4int i=GetIndex(Z, A);
  if (i >= 0){
      return (G4double(A-Z)*MassExcess[0] + G4double(Z)*MassExcess[1] - MassExcess[i])*keV;
  } else { 
    return 0.0;
  }
}

G4double  G4NucleiPropertiesTableAME03::GetBetaDecayEnergy(G4int Z, G4int A)
{
  G4int i=GetIndex(Z, A);
    if (i >= 0){
      return BetaEnergy[i]*keV;
    } else { 
      return 0.0;
    }
}

G4double  G4NucleiPropertiesTableAME03::GetAtomicMass(G4int Z, G4int A)
{
  G4int i=GetIndex(Z, A);	
  if (i >= 0){
    return MassExcess[i]*keV + G4double(A)*amu_c2;
  } else { 
    return 0.0;
  }	
}


G4bool G4NucleiPropertiesTableAME03::IsInTable(G4int Z, G4int A)
{
  return (Z <= A && A >= 1 && A <= 273 && Z >= 0 && Z <= 110 && GetIndex(Z, A) >= 0);
}


//+------------------------------------------------------------+
//| Table of Z (number of protons) and A (number of nucleons)  |
//|        indexArray[0][ ] --> Z                              |
//|        indexArray[1][ ] --> A                              |
//+------------------------------------------------------------

const G4int G4NucleiPropertiesTableAME03::indexArray[2][G4NucleiPropertiesTableAME03::nEntries] = {
  // ---------- Z ----------
  {

   0,   1,   1,   1,   2,   3,   1,   2,   3,   1,
   2,   3,   4,   1,   2,   3,   4,   5,   1,   2,
   3,   4,   5,   2,   3,   4,   5,   6,   2,   3,
   4,   5,   6,   2,   3,   4,   5,   6,   7,   3,
   4,   5,   6,   7,   3,   4,   5,   6,   7,   8,
   4,   5,   6,   7,   8,   4,   5,   6,   7,   8,
   9,   4,   5,   6,   7,   8,   9,   4,   5,   6,
   7,   8,   9,  10,   5,   6,   7,   8,   9,  10,
   5,   6,   7,   8,   9,  10,  11,   5,   6,   7,
   8,   9,  10,  11,  12,   6,   7,   8,   9,  10,
  11,  12,   6,   7,   8,   9,  10,  11,  12,  13,
   6,   7,   8,   9,  10,  11,  12,  13,  14,   7,
   8,   9,  10,  11,  12,  13,  14,   7,   8,   9,
  10,  11,  12,  13,  14,  15,   7,   8,   9,  10,
  11,  12,  13,  14,  15,   8,   9,  10,  11,  12,
  13,  14,  15,  16,   8,   9,  10,  11,  12,  13,
  14,  15,  16,   8,   9,  10,  11,  12,  13,  14,
  15,  16,  17,   9,  10,  11,  12,  13,  14,  15,
  16,  17,   9,  10,  11,  12,  13,  14,  15,  16,
  17,  18,   9,  10,  11,  12,  13,  14,  15,  16,
  17,  18,  10,  11,  12,  13,  14,  15,  16,  17,
  18,  19,  10,  11,  12,  13,  14,  15,  16,  17,
  18,  19,  10,  11,  12,  13,  14,  15,  16,  17,
  18,  19,  20,  11,  12,  13,  14,  15,  16,  17,
  18,  19,  20,  11,  12,  13,  14,  15,  16,  17,
  18,  19,  20,  21,  11,  12,  13,  14,  15,  16,
  17,  18,  19,  20,  21,  12,  13,  14,  15,  16,
  17,  18,  19,  20,  21,  22,  12,  13,  14,  15,
  16,  17,  18,  19,  20,  21,  22,  12,  13,  14,
  15,  16,  17,  18,  19,  20,  21,  22,  23,  13,
  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,
  13,  14,  15,  16,  17,  18,  19,  20,  21,  22,
  23,  24,  14,  15,  16,  17,  18,  19,  20,  21,
  22,  23,  24,  14,  15,  16,  17,  18,  19,  20,
  21,  22,  23,  24,  25,  15,  16,  17,  18,  19,
  20,  21,  22,  23,  24,  25,  26,  15,  16,  17,
  18,  19,  20,  21,  22,  23,  24,  25,  26,  16,
  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,
  27,  16,  17,  18,  19,  20,  21,  22,  23,  24,
  25,  26,  27,  28,  16,  17,  18,  19,  20,  21,
  22,  23,  24,  25,  26,  27,  28,  17,  18,  19,
  20,  21,  22,  23,  24,  25,  26,  27,  28,  17,
  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,
  28,  18,  19,  20,  21,  22,  23,  24,  25,  26,
  27,  28,  29,  18,  19,  20,  21,  22,  23,  24,
  25,  26,  27,  28,  29,  19,  20,  21,  22,  23,
  24,  25,  26,  27,  28,  29,  30,  19,  20,  21,
  22,  23,  24,  25,  26,  27,  28,  29,  30,  20,
  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,
  31,  20,  21,  22,  23,  24,  25,  26,  27,  28,
  29,  30,  31,  21,  22,  23,  24,  25,  26,  27,
  28,  29,  30,  31,  32,  21,  22,  23,  24,  25,
  26,  27,  28,  29,  30,  31,  32,  21,  22,  23,
  24,  25,  26,  27,  28,  29,  30,  31,  32,  33,
  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,
  32,  33,  22,  23,  24,  25,  26,  27,  28,  29,
  30,  31,  32,  33,  22,  23,  24,  25,  26,  27,
  28,  29,  30,  31,  32,  33,  23,  24,  25,  26,
  27,  28,  29,  30,  31,  32,  33,  23,  24,  25,
  26,  27,  28,  29,  30,  31,  32,  33,  34,  24,
  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,
  24,  25,  26,  27,  28,  29,  30,  31,  32,  33,
  34,  35,  25,  26,  27,  28,  29,  30,  31,  32,
  33,  34,  35,  25,  26,  27,  28,  29,  30,  31,
  32,  33,  34,  35,  36,  26,  27,  28,  29,  30,
  31,  32,  33,  34,  35,  36,  26,  27,  28,  29,
  30,  31,  32,  33,  34,  35,  36,  37,  26,  27,
  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,
  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,
  37,  38,  27,  28,  29,  30,  31,  32,  33,  34,
  35,  36,  37,  38,  27,  28,  29,  30,  31,  32,
  33,  34,  35,  36,  37,  38,  28,  29,  30,  31,
  32,  33,  34,  35,  36,  37,  38,  39,  28,  29,
  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,
  38,  39,  40,  29,  30,  31,  32,  33,  34,  35,
  36,  37,  38,  39,  40,  29,  30,  31,  32,  33,
  34,  35,  36,  37,  38,  39,  40,  30,  31,  32,
  33,  34,  35,  36,  37,  38,  39,  40,  41,  30,
  31,  32,  33,  34,  35,  36,  37,  38,  39,  40,
  41,  30,  31,  32,  33,  34,  35,  36,  37,  38,
  39,  40,  41,  42,  31,  32,  33,  34,  35,  36,
  37,  38,  39,  40,  41,  42,  31,  32,  33,  34,
  35,  36,  37,  38,  39,  40,  41,  42,  43,  31,
  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,
  42,  43,  32,  33,  34,  35,  36,  37,  38,  39,
  40,  41,  42,  43,  44,  32,  33,  34,  35,  36,
  37,  38,  39,  40,  41,  42,  43,  44,  32,  33,
  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,
  44,  45,  33,  34,  35,  36,  37,  38,  39,  40,
  41,  42,  43,  44,  45,  33,  34,  35,  36,  37,
  38,  39,  40,  41,  42,  43,  44,  45,  46,  33,
  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,
  44,  45,  46,  34,  35,  36,  37,  38,  39,  40,
  41,  42,  43,  44,  45,  46,  47,  34,  35,  36,
  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,
  47,  35,  36,  37,  38,  39,  40,  41,  42,  43,
  44,  45,  46,  47,  48,  35,  36,  37,  38,  39,
  40,  41,  42,  43,  44,  45,  46,  47,  48,  35,
  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,
  46,  47,  48,  49,  36,  37,  38,  39,  40,  41,
  42,  43,  44,  45,  46,  47,  48,  49,  36,  37,
  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,
  48,  49,  50,  36,  37,  38,  39,  40,  41,  42,
  43,  44,  45,  46,  47,  48,  49,  50,  37,  38,
  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,
  49,  50,  37,  38,  39,  40,  41,  42,  43,  44,
  45,  46,  47,  48,  49,  50,  38,  39,  40,  41,
  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,
  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,
  48,  49,  50,  51,  38,  39,  40,  41,  42,  43,
  44,  45,  46,  47,  48,  49,  50,  51,  52,  39,
  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
  50,  51,  52,  39,  40,  41,  42,  43,  44,  45,
  46,  47,  48,  49,  50,  51,  52,  39,  40,  41,
  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,
  52,  53,  40,  41,  42,  43,  44,  45,  46,  47,
  48,  49,  50,  51,  52,  53,  40,  41,  42,  43,
  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,
  54,  41,  42,  43,  44,  45,  46,  47,  48,  49,
  50,  51,  52,  53,  54,  41,  42,  43,  44,  45,
  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,
  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,
  51,  52,  53,  54,  55,  42,  43,  44,  45,  46,
  47,  48,  49,  50,  51,  52,  53,  54,  55,  56,
  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,
  52,  53,  54,  55,  56,  43,  44,  45,  46,  47,
  48,  49,  50,  51,  52,  53,  54,  55,  56,  43,
  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,
  54,  55,  56,  57,  43,  44,  45,  46,  47,  48,
  49,  50,  51,  52,  53,  54,  55,  56,  57,  44,
  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,
  55,  56,  57,  58,  44,  45,  46,  47,  48,  49,
  50,  51,  52,  53,  54,  55,  56,  57,  58,  45,
  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,
  56,  57,  58,  59,  45,  46,  47,  48,  49,  50,
  51,  52,  53,  54,  55,  56,  57,  58,  59,  46,
  47,  48,  49,  50,  51,  52,  53,  54,  55,  56,
  57,  58,  59,  46,  47,  48,  49,  50,  51,  52,
  53,  54,  55,  56,  57,  58,  59,  60,  47,  48,
  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,
  59,  60,  47,  48,  49,  50,  51,  52,  53,  54,
  55,  56,  57,  58,  59,  60,  61,  47,  48,  49,
  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
  60,  61,  47,  48,  49,  50,  51,  52,  53,  54,
  55,  56,  57,  58,  59,  60,  61,  62,  47,  48,
  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,
  59,  60,  61,  62,  47,  48,  49,  50,  51,  52,
  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,
  63,  48,  49,  50,  51,  52,  53,  54,  55,  56,
  57,  58,  59,  60,  61,  62,  63,  48,  49,  50,
  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,
  61,  62,  63,  49,  50,  51,  52,  53,  54,  55,
  56,  57,  58,  59,  60,  61,  62,  63,  49,  50,
  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,
  61,  62,  63,  64,  49,  50,  51,  52,  53,  54,
  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,
  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
  60,  61,  62,  63,  64,  65,  50,  51,  52,  53,
  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
  64,  65,  51,  52,  53,  54,  55,  56,  57,  58,
  59,  60,  61,  62,  63,  64,  65,  66,  51,  52,
  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,
  63,  64,  65,  66,  52,  53,  54,  55,  56,  57,
  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,
  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,
  62,  63,  64,  65,  66,  67,  52,  53,  54,  55,
  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,
  66,  67,  53,  54,  55,  56,  57,  58,  59,  60,
  61,  62,  63,  64,  65,  66,  67,  68,  53,  54,
  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,
  65,  66,  67,  68,  54,  55,  56,  57,  58,  59,
  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
  64,  65,  66,  67,  68,  69,  54,  55,  56,  57,
  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,
  68,  69,  55,  56,  57,  58,  59,  60,  61,  62,
  63,  64,  65,  66,  67,  68,  69,  70,  55,  56,
  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,
  67,  68,  69,  70,  55,  56,  57,  58,  59,  60,
  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,
  71,  55,  56,  57,  58,  59,  60,  61,  62,  63,
  64,  65,  66,  67,  68,  69,  70,  71,  56,  57,
  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,
  68,  69,  70,  71,  56,  57,  58,  59,  60,  61,
  62,  63,  64,  65,  66,  67,  68,  69,  70,  71,
  72,  57,  58,  59,  60,  61,  62,  63,  64,  65,
  66,  67,  68,  69,  70,  71,  72,  57,  58,  59,
  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
  70,  71,  72,  73,  58,  59,  60,  61,  62,  63,
  64,  65,  66,  67,  68,  69,  70,  71,  72,  73,
  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,
  68,  69,  70,  71,  72,  73,  59,  60,  61,  62,
  63,  64,  65,  66,  67,  68,  69,  70,  71,  72,
  73,  74,  59,  60,  61,  62,  63,  64,  65,  66,
  67,  68,  69,  70,  71,  72,  73,  74,  60,  61,
  62,  63,  64,  65,  66,  67,  68,  69,  70,  71,
  72,  73,  74,  75,  60,  61,  62,  63,  64,  65,
  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,
  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,
  71,  72,  73,  74,  75,  76,  61,  62,  63,  64,
  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,
  75,  76,  62,  63,  64,  65,  66,  67,  68,  69,
  70,  71,  72,  73,  74,  75,  76,  77,  62,  63,
  64,  65,  66,  67,  68,  69,  70,  71,  72,  73,
  74,  75,  76,  77,  63,  64,  65,  66,  67,  68,
  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,
  63,  64,  65,  66,  67,  68,  69,  70,  71,  72,
  73,  74,  75,  76,  77,  78,  64,  65,  66,  67,
  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,
  78,  64,  65,  66,  67,  68,  69,  70,  71,  72,
  73,  74,  75,  76,  77,  78,  79,  65,  66,  67,
  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,
  78,  79,  65,  66,  67,  68,  69,  70,  71,  72,
  73,  74,  75,  76,  77,  78,  79,  80,  66,  67,
  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,
  78,  79,  80,  66,  67,  68,  69,  70,  71,  72,
  73,  74,  75,  76,  77,  78,  79,  80,  67,  68,
  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,
  79,  80,  67,  68,  69,  70,  71,  72,  73,  74,
  75,  76,  77,  78,  79,  80,  68,  69,  70,  71,
  72,  73,  74,  75,  76,  77,  78,  79,  80,  81,
  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,
  78,  79,  80,  81,  69,  70,  71,  72,  73,  74,
  75,  76,  77,  78,  79,  80,  81,  82,  69,  70,
  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,
  81,  82,  70,  71,  72,  73,  74,  75,  76,  77,
  78,  79,  80,  81,  82,  70,  71,  72,  73,  74,
  75,  76,  77,  78,  79,  80,  81,  82,  71,  72,
  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,
  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,
  81,  82,  71,  72,  73,  74,  75,  76,  77,  78,
  79,  80,  81,  82,  83,  72,  73,  74,  75,  76,
  77,  78,  79,  80,  81,  82,  83,  72,  73,  74,
  75,  76,  77,  78,  79,  80,  81,  82,  83,  72,
  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,
  83,  72,  73,  74,  75,  76,  77,  78,  79,  80,
  81,  82,  83,  84,  73,  74,  75,  76,  77,  78,
  79,  80,  81,  82,  83,  84,  73,  74,  75,  76,
  77,  78,  79,  80,  81,  82,  83,  84,  74,  75,
  76,  77,  78,  79,  80,  81,  82,  83,  84,  74,
  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,
  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,
  85,  75,  76,  77,  78,  79,  80,  81,  82,  83,
  84,  85,  76,  77,  78,  79,  80,  81,  82,  83,
  84,  85,  86,  76,  77,  78,  79,  80,  81,  82,
  83,  84,  85,  86,  77,  78,  79,  80,  81,  82,
  83,  84,  85,  86,  77,  78,  79,  80,  81,  82,
  83,  84,  85,  86,  77,  78,  79,  80,  81,  82,
  83,  84,  85,  86,  87,  78,  79,  80,  81,  82,
  83,  84,  85,  86,  87,  78,  79,  80,  81,  82,
  83,  84,  85,  86,  87,  78,  79,  80,  81,  82,
  83,  84,  85,  86,  87,  88,  79,  80,  81,  82,
  83,  84,  85,  86,  87,  88,  79,  80,  81,  82,
  83,  84,  85,  86,  87,  88,  79,  80,  81,  82,
  83,  84,  85,  86,  87,  88,  80,  81,  82,  83,
  84,  85,  86,  87,  88,  89,  80,  81,  82,  83,
  84,  85,  86,  87,  88,  89,  80,  81,  82,  83,
  84,  85,  86,  87,  88,  89,  80,  81,  82,  83,
  84,  85,  86,  87,  88,  89,  90,  80,  81,  82,
  83,  84,  85,  86,  87,  88,  89,  90,  81,  82,
  83,  84,  85,  86,  87,  88,  89,  90,  81,  82,
  83,  84,  85,  86,  87,  88,  89,  90,  91,  82,
  83,  84,  85,  86,  87,  88,  89,  90,  91,  82,
  83,  84,  85,  86,  87,  88,  89,  90,  91,  82,
  83,  84,  85,  86,  87,  88,  89,  90,  91,  83,
  84,  85,  86,  87,  88,  89,  90,  91,  83,  84,
  85,  86,  87,  88,  89,  90,  91,  92,  83,  84,
  85,  86,  87,  88,  89,  90,  91,  92,  84,  85,
  86,  87,  88,  89,  90,  91,  92,  84,  85,  86,
  87,  88,  89,  90,  91,  92,  85,  86,  87,  88,
  89,  90,  91,  92,  85,  86,  87,  88,  89,  90,
  91,  92,  85,  86,  87,  88,  89,  90,  91,  92,
  86,  87,  88,  89,  90,  91,  92,  86,  87,  88,
  89,  90,  91,  92,  93,  86,  87,  88,  89,  90,
  91,  92,  93,  86,  87,  88,  89,  90,  91,  92,
  93,  86,  87,  88,  89,  90,  91,  92,  93,  94,
  87,  88,  89,  90,  91,  92,  93,  94,  87,  88,
  89,  90,  91,  92,  93,  94,  87,  88,  89,  90,
  91,  92,  93,  94,  95,  87,  88,  89,  90,  91,
  92,  93,  94,  95,  88,  89,  90,  91,  92,  93,
  94,  95,  96,  88,  89,  90,  91,  92,  93,  94,
  95,  96,  89,  90,  91,  92,  93,  94,  95,  96,
  97,  89,  90,  91,  92,  93,  94,  95,  96,  97,
  90,  91,  92,  93,  94,  95,  96,  97,  98,  90,
  91,  92,  93,  94,  95,  96,  97,  98,  91,  92,
  93,  94,  95,  96,  97,  98,  91,  92,  93,  94,
  95,  96,  97,  98,  99,  92,  93,  94,  95,  96,
  97,  98,  99,  92,  93,  94,  95,  96,  97,  98,
  99, 100,  93,  94,  95,  96,  97,  98,  99, 100,
  93,  94,  95,  96,  97,  98,  99, 100,  94,  95,
  96,  97,  98,  99, 100, 101,  94,  95,  96,  97,
  98,  99, 100, 101,  94,  95,  96,  97,  98,  99,
 100, 101,  95,  96,  97,  98,  99, 100, 101, 102,
  95,  96,  97,  98,  99, 100, 101, 102,  96,  97,
  98,  99, 100, 101, 102,  96,  97,  98,  99, 100,
 101, 102, 103,  96,  97,  98,  99, 100, 101, 102,
 103,  97,  98,  99, 100, 101, 102, 103, 104,  97,
  98,  99, 100, 101, 102, 103, 104,  98,  99, 100,
 101, 102, 103, 104, 105,  98,  99, 100, 101, 102,
 103, 104, 105,  99, 100, 101, 102, 103, 104, 105,
  99, 100, 101, 102, 103, 104, 105, 106, 100, 101,
 102, 103, 104, 105, 106, 100, 101, 102, 103, 104,
 105, 106, 107, 101, 102, 103, 104, 105, 106, 107,
 101, 102, 103, 104, 105, 106, 107, 102, 103, 104,
 105, 106, 107, 108, 102, 103, 104, 105, 106, 107,
 108, 103, 104, 105, 106, 107, 108, 109, 103, 104,
 105, 106, 107, 108, 109, 104, 105, 106, 107, 108,
 109, 110, 104, 105, 106, 107, 108, 109, 110, 105,
 106, 107, 108, 109, 110, 105, 106, 107, 108, 109,
 110, 106, 107, 108, 109, 110, 106, 107, 108, 109,
 110, 111, 106, 107, 108, 109, 110, 111, 107, 108,
 109, 110, 111, 107, 108, 109, 110, 111, 108, 109,
 110, 111, 108, 109, 110, 111, 112, 109, 110, 111,
 112, 109, 110, 111, 112, 110, 111, 112, 110, 111,
 112, 111, 112, 111, 112, 113, 112, 113, 112, 113,
 114, 113, 114, 113, 114, 115, 114, 115, 114, 115,
 116, 115, 116, 115, 116, 117, 116, 117, 118
  },
  // --------- A -----------
  {
   1,   1,   2,   3,   3,   3,   4,   4,   4,   5,
   5,   5,   5,   6,   6,   6,   6,   6,   7,   7,
   7,   7,   7,   8,   8,   8,   8,   8,   9,   9,
   9,   9,   9,  10,  10,  10,  10,  10,  10,  11,
  11,  11,  11,  11,  12,  12,  12,  12,  12,  12,
  13,  13,  13,  13,  13,  14,  14,  14,  14,  14,
  14,  15,  15,  15,  15,  15,  15,  16,  16,  16,
  16,  16,  16,  16,  17,  17,  17,  17,  17,  17,
  18,  18,  18,  18,  18,  18,  18,  19,  19,  19,
  19,  19,  19,  19,  19,  20,  20,  20,  20,  20,
  20,  20,  21,  21,  21,  21,  21,  21,  21,  21,
  22,  22,  22,  22,  22,  22,  22,  22,  22,  23,
  23,  23,  23,  23,  23,  23,  23,  24,  24,  24,
  24,  24,  24,  24,  24,  24,  25,  25,  25,  25,
  25,  25,  25,  25,  25,  26,  26,  26,  26,  26,
  26,  26,  26,  26,  27,  27,  27,  27,  27,  27,
  27,  27,  27,  28,  28,  28,  28,  28,  28,  28,
  28,  28,  28,  29,  29,  29,  29,  29,  29,  29,
  29,  29,  30,  30,  30,  30,  30,  30,  30,  30,
  30,  30,  31,  31,  31,  31,  31,  31,  31,  31,
  31,  31,  32,  32,  32,  32,  32,  32,  32,  32,
  32,  32,  33,  33,  33,  33,  33,  33,  33,  33,
  33,  33,  34,  34,  34,  34,  34,  34,  34,  34,
  34,  34,  34,  35,  35,  35,  35,  35,  35,  35,
  35,  35,  35,  36,  36,  36,  36,  36,  36,  36,
  36,  36,  36,  36,  37,  37,  37,  37,  37,  37,
  37,  37,  37,  37,  37,  38,  38,  38,  38,  38,
  38,  38,  38,  38,  38,  38,  39,  39,  39,  39,
  39,  39,  39,  39,  39,  39,  39,  40,  40,  40,
  40,  40,  40,  40,  40,  40,  40,  40,  40,  41,
  41,  41,  41,  41,  41,  41,  41,  41,  41,  41,
  42,  42,  42,  42,  42,  42,  42,  42,  42,  42,
  42,  42,  43,  43,  43,  43,  43,  43,  43,  43,
  43,  43,  43,  44,  44,  44,  44,  44,  44,  44,
  44,  44,  44,  44,  44,  45,  45,  45,  45,  45,
  45,  45,  45,  45,  45,  45,  45,  46,  46,  46,
  46,  46,  46,  46,  46,  46,  46,  46,  46,  47,
  47,  47,  47,  47,  47,  47,  47,  47,  47,  47,
  47,  48,  48,  48,  48,  48,  48,  48,  48,  48,
  48,  48,  48,  48,  49,  49,  49,  49,  49,  49,
  49,  49,  49,  49,  49,  49,  49,  50,  50,  50,
  50,  50,  50,  50,  50,  50,  50,  50,  50,  51,
  51,  51,  51,  51,  51,  51,  51,  51,  51,  51,
  51,  52,  52,  52,  52,  52,  52,  52,  52,  52,
  52,  52,  52,  53,  53,  53,  53,  53,  53,  53,
  53,  53,  53,  53,  53,  54,  54,  54,  54,  54,
  54,  54,  54,  54,  54,  54,  54,  55,  55,  55,
  55,  55,  55,  55,  55,  55,  55,  55,  55,  56,
  56,  56,  56,  56,  56,  56,  56,  56,  56,  56,
  56,  57,  57,  57,  57,  57,  57,  57,  57,  57,
  57,  57,  57,  58,  58,  58,  58,  58,  58,  58,
  58,  58,  58,  58,  58,  59,  59,  59,  59,  59,
  59,  59,  59,  59,  59,  59,  59,  60,  60,  60,
  60,  60,  60,  60,  60,  60,  60,  60,  60,  60,
  61,  61,  61,  61,  61,  61,  61,  61,  61,  61,
  61,  61,  62,  62,  62,  62,  62,  62,  62,  62,
  62,  62,  62,  62,  63,  63,  63,  63,  63,  63,
  63,  63,  63,  63,  63,  63,  64,  64,  64,  64,
  64,  64,  64,  64,  64,  64,  64,  65,  65,  65,
  65,  65,  65,  65,  65,  65,  65,  65,  65,  66,
  66,  66,  66,  66,  66,  66,  66,  66,  66,  66,
  67,  67,  67,  67,  67,  67,  67,  67,  67,  67,
  67,  67,  68,  68,  68,  68,  68,  68,  68,  68,
  68,  68,  68,  69,  69,  69,  69,  69,  69,  69,
  69,  69,  69,  69,  69,  70,  70,  70,  70,  70,
  70,  70,  70,  70,  70,  70,  71,  71,  71,  71,
  71,  71,  71,  71,  71,  71,  71,  71,  72,  72,
  72,  72,  72,  72,  72,  72,  72,  72,  72,  72,
  73,  73,  73,  73,  73,  73,  73,  73,  73,  73,
  73,  73,  74,  74,  74,  74,  74,  74,  74,  74,
  74,  74,  74,  74,  75,  75,  75,  75,  75,  75,
  75,  75,  75,  75,  75,  75,  76,  76,  76,  76,
  76,  76,  76,  76,  76,  76,  76,  76,  77,  77,
  77,  77,  77,  77,  77,  77,  77,  77,  77,  77,
  78,  78,  78,  78,  78,  78,  78,  78,  78,  78,
  78,  78,  78,  79,  79,  79,  79,  79,  79,  79,
  79,  79,  79,  79,  79,  80,  80,  80,  80,  80,
  80,  80,  80,  80,  80,  80,  80,  81,  81,  81,
  81,  81,  81,  81,  81,  81,  81,  81,  81,  82,
  82,  82,  82,  82,  82,  82,  82,  82,  82,  82,
  82,  83,  83,  83,  83,  83,  83,  83,  83,  83,
  83,  83,  83,  83,  84,  84,  84,  84,  84,  84,
  84,  84,  84,  84,  84,  84,  85,  85,  85,  85,
  85,  85,  85,  85,  85,  85,  85,  85,  85,  86,
  86,  86,  86,  86,  86,  86,  86,  86,  86,  86,
  86,  86,  87,  87,  87,  87,  87,  87,  87,  87,
  87,  87,  87,  87,  87,  88,  88,  88,  88,  88,
  88,  88,  88,  88,  88,  88,  88,  88,  89,  89,
  89,  89,  89,  89,  89,  89,  89,  89,  89,  89,
  89,  89,  90,  90,  90,  90,  90,  90,  90,  90,
  90,  90,  90,  90,  90,  91,  91,  91,  91,  91,
  91,  91,  91,  91,  91,  91,  91,  91,  91,  92,
  92,  92,  92,  92,  92,  92,  92,  92,  92,  92,
  92,  92,  92,  93,  93,  93,  93,  93,  93,  93,
  93,  93,  93,  93,  93,  93,  93,  94,  94,  94,
  94,  94,  94,  94,  94,  94,  94,  94,  94,  94,
  94,  95,  95,  95,  95,  95,  95,  95,  95,  95,
  95,  95,  95,  95,  95,  96,  96,  96,  96,  96,
  96,  96,  96,  96,  96,  96,  96,  96,  96,  97,
  97,  97,  97,  97,  97,  97,  97,  97,  97,  97,
  97,  97,  97,  97,  98,  98,  98,  98,  98,  98,
  98,  98,  98,  98,  98,  98,  98,  98,  99,  99,
  99,  99,  99,  99,  99,  99,  99,  99,  99,  99,
  99,  99,  99, 100, 100, 100, 100, 100, 100, 100,
 100, 100, 100, 100, 100, 100, 100, 100, 101, 101,
 101, 101, 101, 101, 101, 101, 101, 101, 101, 101,
 101, 101, 102, 102, 102, 102, 102, 102, 102, 102,
 102, 102, 102, 102, 102, 102, 103, 103, 103, 103,
 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
 104, 104, 104, 104, 104, 104, 104, 104, 104, 104,
 104, 104, 104, 104, 105, 105, 105, 105, 105, 105,
 105, 105, 105, 105, 105, 105, 105, 105, 105, 106,
 106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
 106, 106, 106, 107, 107, 107, 107, 107, 107, 107,
 107, 107, 107, 107, 107, 107, 107, 108, 108, 108,
 108, 108, 108, 108, 108, 108, 108, 108, 108, 108,
 108, 108, 109, 109, 109, 109, 109, 109, 109, 109,
 109, 109, 109, 109, 109, 109, 110, 110, 110, 110,
 110, 110, 110, 110, 110, 110, 110, 110, 110, 110,
 110, 111, 111, 111, 111, 111, 111, 111, 111, 111,
 111, 111, 111, 111, 111, 112, 112, 112, 112, 112,
 112, 112, 112, 112, 112, 112, 112, 112, 112, 112,
 113, 113, 113, 113, 113, 113, 113, 113, 113, 113,
 113, 113, 113, 113, 113, 114, 114, 114, 114, 114,
 114, 114, 114, 114, 114, 114, 114, 114, 114, 114,
 115, 115, 115, 115, 115, 115, 115, 115, 115, 115,
 115, 115, 115, 115, 115, 116, 116, 116, 116, 116,
 116, 116, 116, 116, 116, 116, 116, 116, 116, 117,
 117, 117, 117, 117, 117, 117, 117, 117, 117, 117,
 117, 117, 117, 117, 118, 118, 118, 118, 118, 118,
 118, 118, 118, 118, 118, 118, 118, 118, 118, 119,
 119, 119, 119, 119, 119, 119, 119, 119, 119, 119,
 119, 119, 119, 119, 120, 120, 120, 120, 120, 120,
 120, 120, 120, 120, 120, 120, 120, 120, 120, 121,
 121, 121, 121, 121, 121, 121, 121, 121, 121, 121,
 121, 121, 121, 121, 122, 122, 122, 122, 122, 122,
 122, 122, 122, 122, 122, 122, 122, 122, 122, 123,
 123, 123, 123, 123, 123, 123, 123, 123, 123, 123,
 123, 123, 123, 124, 124, 124, 124, 124, 124, 124,
 124, 124, 124, 124, 124, 124, 124, 124, 125, 125,
 125, 125, 125, 125, 125, 125, 125, 125, 125, 125,
 125, 125, 126, 126, 126, 126, 126, 126, 126, 126,
 126, 126, 126, 126, 126, 126, 126, 127, 127, 127,
 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
 127, 127, 128, 128, 128, 128, 128, 128, 128, 128,
 128, 128, 128, 128, 128, 128, 128, 128, 129, 129,
 129, 129, 129, 129, 129, 129, 129, 129, 129, 129,
 129, 129, 129, 129, 130, 130, 130, 130, 130, 130,
 130, 130, 130, 130, 130, 130, 130, 130, 130, 130,
 130, 131, 131, 131, 131, 131, 131, 131, 131, 131,
 131, 131, 131, 131, 131, 131, 131, 132, 132, 132,
 132, 132, 132, 132, 132, 132, 132, 132, 132, 132,
 132, 132, 132, 133, 133, 133, 133, 133, 133, 133,
 133, 133, 133, 133, 133, 133, 133, 133, 134, 134,
 134, 134, 134, 134, 134, 134, 134, 134, 134, 134,
 134, 134, 134, 134, 135, 135, 135, 135, 135, 135,
 135, 135, 135, 135, 135, 135, 135, 135, 135, 135,
 136, 136, 136, 136, 136, 136, 136, 136, 136, 136,
 136, 136, 136, 136, 136, 136, 137, 137, 137, 137,
 137, 137, 137, 137, 137, 137, 137, 137, 137, 137,
 137, 137, 138, 138, 138, 138, 138, 138, 138, 138,
 138, 138, 138, 138, 138, 138, 138, 138, 139, 139,
 139, 139, 139, 139, 139, 139, 139, 139, 139, 139,
 139, 139, 139, 139, 140, 140, 140, 140, 140, 140,
 140, 140, 140, 140, 140, 140, 140, 140, 140, 140,
 141, 141, 141, 141, 141, 141, 141, 141, 141, 141,
 141, 141, 141, 141, 141, 141, 142, 142, 142, 142,
 142, 142, 142, 142, 142, 142, 142, 142, 142, 142,
 142, 142, 143, 143, 143, 143, 143, 143, 143, 143,
 143, 143, 143, 143, 143, 143, 143, 143, 144, 144,
 144, 144, 144, 144, 144, 144, 144, 144, 144, 144,
 144, 144, 144, 144, 145, 145, 145, 145, 145, 145,
 145, 145, 145, 145, 145, 145, 145, 145, 145, 145,
 146, 146, 146, 146, 146, 146, 146, 146, 146, 146,
 146, 146, 146, 146, 146, 146, 147, 147, 147, 147,
 147, 147, 147, 147, 147, 147, 147, 147, 147, 147,
 147, 147, 148, 148, 148, 148, 148, 148, 148, 148,
 148, 148, 148, 148, 148, 148, 148, 148, 149, 149,
 149, 149, 149, 149, 149, 149, 149, 149, 149, 149,
 149, 149, 149, 149, 150, 150, 150, 150, 150, 150,
 150, 150, 150, 150, 150, 150, 150, 150, 150, 150,
 150, 151, 151, 151, 151, 151, 151, 151, 151, 151,
 151, 151, 151, 151, 151, 151, 151, 151, 152, 152,
 152, 152, 152, 152, 152, 152, 152, 152, 152, 152,
 152, 152, 152, 152, 153, 153, 153, 153, 153, 153,
 153, 153, 153, 153, 153, 153, 153, 153, 153, 153,
 153, 154, 154, 154, 154, 154, 154, 154, 154, 154,
 154, 154, 154, 154, 154, 154, 154, 155, 155, 155,
 155, 155, 155, 155, 155, 155, 155, 155, 155, 155,
 155, 155, 155, 155, 156, 156, 156, 156, 156, 156,
 156, 156, 156, 156, 156, 156, 156, 156, 156, 156,
 157, 157, 157, 157, 157, 157, 157, 157, 157, 157,
 157, 157, 157, 157, 157, 157, 158, 158, 158, 158,
 158, 158, 158, 158, 158, 158, 158, 158, 158, 158,
 158, 158, 159, 159, 159, 159, 159, 159, 159, 159,
 159, 159, 159, 159, 159, 159, 159, 159, 160, 160,
 160, 160, 160, 160, 160, 160, 160, 160, 160, 160,
 160, 160, 160, 160, 161, 161, 161, 161, 161, 161,
 161, 161, 161, 161, 161, 161, 161, 161, 161, 161,
 162, 162, 162, 162, 162, 162, 162, 162, 162, 162,
 162, 162, 162, 162, 162, 162, 163, 163, 163, 163,
 163, 163, 163, 163, 163, 163, 163, 163, 163, 163,
 163, 163, 164, 164, 164, 164, 164, 164, 164, 164,
 164, 164, 164, 164, 164, 164, 164, 164, 165, 165,
 165, 165, 165, 165, 165, 165, 165, 165, 165, 165,
 165, 165, 165, 165, 166, 166, 166, 166, 166, 166,
 166, 166, 166, 166, 166, 166, 166, 166, 166, 166,
 167, 167, 167, 167, 167, 167, 167, 167, 167, 167,
 167, 167, 167, 167, 167, 167, 168, 168, 168, 168,
 168, 168, 168, 168, 168, 168, 168, 168, 168, 168,
 168, 169, 169, 169, 169, 169, 169, 169, 169, 169,
 169, 169, 169, 169, 169, 169, 169, 170, 170, 170,
 170, 170, 170, 170, 170, 170, 170, 170, 170, 170,
 170, 170, 171, 171, 171, 171, 171, 171, 171, 171,
 171, 171, 171, 171, 171, 171, 171, 171, 172, 172,
 172, 172, 172, 172, 172, 172, 172, 172, 172, 172,
 172, 172, 172, 173, 173, 173, 173, 173, 173, 173,
 173, 173, 173, 173, 173, 173, 173, 173, 174, 174,
 174, 174, 174, 174, 174, 174, 174, 174, 174, 174,
 174, 174, 175, 175, 175, 175, 175, 175, 175, 175,
 175, 175, 175, 175, 175, 175, 176, 176, 176, 176,
 176, 176, 176, 176, 176, 176, 176, 176, 176, 176,
 177, 177, 177, 177, 177, 177, 177, 177, 177, 177,
 177, 177, 177, 177, 178, 178, 178, 178, 178, 178,
 178, 178, 178, 178, 178, 178, 178, 178, 179, 179,
 179, 179, 179, 179, 179, 179, 179, 179, 179, 179,
 179, 179, 180, 180, 180, 180, 180, 180, 180, 180,
 180, 180, 180, 180, 180, 181, 181, 181, 181, 181,
 181, 181, 181, 181, 181, 181, 181, 181, 182, 182,
 182, 182, 182, 182, 182, 182, 182, 182, 182, 182,
 183, 183, 183, 183, 183, 183, 183, 183, 183, 183,
 183, 183, 184, 184, 184, 184, 184, 184, 184, 184,
 184, 184, 184, 184, 184, 185, 185, 185, 185, 185,
 185, 185, 185, 185, 185, 185, 185, 186, 186, 186,
 186, 186, 186, 186, 186, 186, 186, 186, 186, 187,
 187, 187, 187, 187, 187, 187, 187, 187, 187, 187,
 187, 188, 188, 188, 188, 188, 188, 188, 188, 188,
 188, 188, 188, 188, 189, 189, 189, 189, 189, 189,
 189, 189, 189, 189, 189, 189, 190, 190, 190, 190,
 190, 190, 190, 190, 190, 190, 190, 190, 191, 191,
 191, 191, 191, 191, 191, 191, 191, 191, 191, 192,
 192, 192, 192, 192, 192, 192, 192, 192, 192, 192,
 193, 193, 193, 193, 193, 193, 193, 193, 193, 193,
 193, 194, 194, 194, 194, 194, 194, 194, 194, 194,
 194, 194, 195, 195, 195, 195, 195, 195, 195, 195,
 195, 195, 195, 196, 196, 196, 196, 196, 196, 196,
 196, 196, 196, 196, 197, 197, 197, 197, 197, 197,
 197, 197, 197, 197, 198, 198, 198, 198, 198, 198,
 198, 198, 198, 198, 199, 199, 199, 199, 199, 199,
 199, 199, 199, 199, 199, 200, 200, 200, 200, 200,
 200, 200, 200, 200, 200, 201, 201, 201, 201, 201,
 201, 201, 201, 201, 201, 202, 202, 202, 202, 202,
 202, 202, 202, 202, 202, 202, 203, 203, 203, 203,
 203, 203, 203, 203, 203, 203, 204, 204, 204, 204,
 204, 204, 204, 204, 204, 204, 205, 205, 205, 205,
 205, 205, 205, 205, 205, 205, 206, 206, 206, 206,
 206, 206, 206, 206, 206, 206, 207, 207, 207, 207,
 207, 207, 207, 207, 207, 207, 208, 208, 208, 208,
 208, 208, 208, 208, 208, 208, 209, 209, 209, 209,
 209, 209, 209, 209, 209, 209, 209, 210, 210, 210,
 210, 210, 210, 210, 210, 210, 210, 210, 211, 211,
 211, 211, 211, 211, 211, 211, 211, 211, 212, 212,
 212, 212, 212, 212, 212, 212, 212, 212, 212, 213,
 213, 213, 213, 213, 213, 213, 213, 213, 213, 214,
 214, 214, 214, 214, 214, 214, 214, 214, 214, 215,
 215, 215, 215, 215, 215, 215, 215, 215, 215, 216,
 216, 216, 216, 216, 216, 216, 216, 216, 217, 217,
 217, 217, 217, 217, 217, 217, 217, 217, 218, 218,
 218, 218, 218, 218, 218, 218, 218, 218, 219, 219,
 219, 219, 219, 219, 219, 219, 219, 220, 220, 220,
 220, 220, 220, 220, 220, 220, 221, 221, 221, 221,
 221, 221, 221, 221, 222, 222, 222, 222, 222, 222,
 222, 222, 223, 223, 223, 223, 223, 223, 223, 223,
 224, 224, 224, 224, 224, 224, 224, 225, 225, 225,
 225, 225, 225, 225, 225, 226, 226, 226, 226, 226,
 226, 226, 226, 227, 227, 227, 227, 227, 227, 227,
 227, 228, 228, 228, 228, 228, 228, 228, 228, 228,
 229, 229, 229, 229, 229, 229, 229, 229, 230, 230,
 230, 230, 230, 230, 230, 230, 231, 231, 231, 231,
 231, 231, 231, 231, 231, 232, 232, 232, 232, 232,
 232, 232, 232, 232, 233, 233, 233, 233, 233, 233,
 233, 233, 233, 234, 234, 234, 234, 234, 234, 234,
 234, 234, 235, 235, 235, 235, 235, 235, 235, 235,
 235, 236, 236, 236, 236, 236, 236, 236, 236, 236,
 237, 237, 237, 237, 237, 237, 237, 237, 237, 238,
 238, 238, 238, 238, 238, 238, 238, 238, 239, 239,
 239, 239, 239, 239, 239, 239, 240, 240, 240, 240,
 240, 240, 240, 240, 240, 241, 241, 241, 241, 241,
 241, 241, 241, 242, 242, 242, 242, 242, 242, 242,
 242, 242, 243, 243, 243, 243, 243, 243, 243, 243,
 244, 244, 244, 244, 244, 244, 244, 244, 245, 245,
 245, 245, 245, 245, 245, 245, 246, 246, 246, 246,
 246, 246, 246, 246, 247, 247, 247, 247, 247, 247,
 247, 247, 248, 248, 248, 248, 248, 248, 248, 248,
 249, 249, 249, 249, 249, 249, 249, 249, 250, 250,
 250, 250, 250, 250, 250, 251, 251, 251, 251, 251,
 251, 251, 251, 252, 252, 252, 252, 252, 252, 252,
 252, 253, 253, 253, 253, 253, 253, 253, 253, 254,
 254, 254, 254, 254, 254, 254, 254, 255, 255, 255,
 255, 255, 255, 255, 255, 256, 256, 256, 256, 256,
 256, 256, 256, 257, 257, 257, 257, 257, 257, 257,
 258, 258, 258, 258, 258, 258, 258, 258, 259, 259,
 259, 259, 259, 259, 259, 260, 260, 260, 260, 260,
 260, 260, 260, 261, 261, 261, 261, 261, 261, 261,
 262, 262, 262, 262, 262, 262, 262, 263, 263, 263,
 263, 263, 263, 263, 264, 264, 264, 264, 264, 264,
 264, 265, 265, 265, 265, 265, 265, 265, 266, 266,
 266, 266, 266, 266, 266, 267, 267, 267, 267, 267,
 267, 267, 268, 268, 268, 268, 268, 268, 268, 269,
 269, 269, 269, 269, 269, 270, 270, 270, 270, 270,
 270, 271, 271, 271, 271, 271, 272, 272, 272, 272,
 272, 272, 273, 273, 273, 273, 273, 273, 274, 274,
 274, 274, 274, 275, 275, 275, 275, 275, 276, 276,
 276, 276, 277, 277, 277, 277, 277, 278, 278, 278,
 278, 279, 279, 279, 279, 280, 280, 280, 281, 281,
 281, 282, 282, 283, 283, 283, 284, 284, 285, 285,
 285, 286, 286, 287, 287, 287, 288, 288, 289, 289,
 289, 290, 290, 291, 291, 291, 292, 292, 293
  }
};




//+----------------------+
//| Table of Mass Excess |
//+----------------------+
// unit: keV
const G4double G4NucleiPropertiesTableAME03::MassExcess[G4NucleiPropertiesTableAME03::nEntries] = {
    8071.3171,    7288.9705,  13135.72158,    14949.806,  14931.21475,        28667,    25901.518,   2424.91565,    25323.185,     32892.44,
    11386.233,    11678.886,        37996,    41863.757,    17595.106,    14086.793,    18374.947,        43603,        49135,    26101.038,
    14908.141,    15770.034,    27868.346,    31598.044,    20946.844,     4941.672,     22921.49,     35094.06,    40939.429,    24954.264,
    11347.648,    12415.681,    28910.491,    48809.203,    33050.581,     12606.67,    12050.731,    15698.682,    38800.148,     40797.31,
    20174.064,     8667.931,    10650.342,    24303.569,        50096,    25076.506,    13368.899,            0,    17338.082,    32047.954,
    33247.823,    16562.166,   3125.01129,     5345.481,    23112.428,    39954.498,    23663.683,   3019.89305,   2863.41704,     8007.356,
        32658,        49798,    28972.278,     9873.144,    101.43805,     2855.605,    16775.372,        57678,    37081.686,    13694.129,
     5683.658,  -4737.00141,    10680.254,    23996.462,    43770.816,    21038.832,     7871.368,     -808.813,     1951.701,    16460.901,
        52322,    24926.178,    13114.466,     -781.522,      873.701,     5317.166,    24189.968,        59364,    32420.666,    15862.129,
      3334.87,    -1487.386,      1751.44,    12926.808,    33040.092,     37557.61,     21765.11,     3797.462,      -17.404,  -7041.93131,
     6847.719,    17570.348,        45960,    25251.164,     8062.906,      -47.551,    -5731.776,    -2184.161,    10910.506,        26119,
        53281,    32038.675,     9284.152,     2793.378,    -8024.715,    -5182.436,     -396.963,        18183,        32164,        38396,
     14612.96,     3329.747,    -5154.045,  -9529.85358,    -5473.766,      6769.57,        23772,        47543,      19070.4,     7559.527,
    -5951.521,    -8418.114,   -13933.567,      -56.946,    10754.673,        31997,        56504,        27442,    11272.706,    -2108.075,
    -9357.818,   -13192.826,    -8916.172,     3824.318,        18872,        35713,    18271.772,      429.611,    -6862.316,   -16214.582,
   -12210.309,    -7144.632,        10973,        25970,        44954,     24926.86,     7069.949,    -5517.436,   -14586.651,   -17196.658,
   -12384.301,      -717.03,        17543,        53850,        33226,    11244.601,     -989.247,   -15018.641,   -16850.441, -21492.79678,
    -7158.753,     4073.203,        26557,        40296,    18057.881,     2665.004,   -10619.032,   -18215.322,   -21895.046,   -16952.626,
    -3159.582,        13143,        48903,    23102.025,      8361.09,    -8910.672,   -15872.419,   -24432.928,   -20200.575,   -14062.532,
         4443,        20083,        56289,        30842,    12654.768,     -3217.38,   -14953.628,   -22949.006,   -24440.885,   -19044.648,
    -7067.165,        11293,        37278,    19064.478,     -954.781,   -11061.966,   -24080.907,   -24305.218,   -26015.697,   -13329.771,
    -2200.204,        20418,        45997,    24889.293,      4894.07,    -8529.377,   -20492.662,   -26337.486,   -26585.994,   -21003.432,
    -9384.141,         6763,        53121,        32761,     8808.603,    -2932.495,    -19956.77,   -24557.669,   -29931.788,   -24439.776,
   -18377.217,        -1481,        13153,        39582,        16152,      -130.19,   -14360.307,    -24857.74,   -28846.356,    -29013.54,
   -23047.411,     -11168.9,         4602,        47953,        21424,     5781.974,   -12482.507,   -20250.977,   -30664.075,   -29521.857,
    -30231.54,   -17426.171,    -6439.359,        13898,        55275,        29249,     9946.326,    -6579.998,   -18994.145,    -26896.36,
   -31761.532,   -30947.659,   -24800.199,    -13161.76,         2841,        34996,    16050.594,    -4067.274,    -14757.82,   -26861.197,
   -29798.097,   -34714.551,   -28800.691,    -22059.22,        -4937,         9101,        43568,    21396.417,     1928.205,   -12873.735,
   -23162.245,   -29800.203,   -33242.011,   -33807.011,     -27274.4,   -14168.021,         1500,        50235,        29295,     5467.052,
    -8106.838,   -22866.568,    -27557.81, -35039.89602,   -33535.205,   -34846.275,   -20523.228,    -8850.275,        10330,        35704,
    13562.553,    -5276.508,   -19019.105,   -27307.189,   -33067.467,   -35559.074,   -35137.759,   -28642.392,       -15700,         -205,
        43678,        18434,      938.865,   -17677.503,    -24912.99,   -34422.675,   -35021.556,   -38547.072,   -32121.239,   -25121.552,
        -8169,         5990,        26697,     5765.948,   -11965.235,   -24168.168,   -32009.808,   -36593.239,   -38408.639,   -36187.929,
   -29321.103,       -18024,        -2133,        32844,        12100,    -9116.168,   -20231.052,   -32673.053,   -35809.606,   -41468.479,
   -37816.093,   -37548.459,    -24116.38,       -13461,         6399,        17903,    -3252.672,   -18362.647,   -29770.589,   -36608.186,
    -40811.95,   -41067.792,   -39005.737,   -31879.629,   -18965.218,        -5114,        13579,        25504,          699,   -14708.253,
   -29720.127,   -35418.323,   -43135.077,   -41757.115,   -44123.422,   -37073.013,   -29473.742,       -12370,          755,         8002,
       -10517,   -25907.836,   -35696.272,   -42340.123,   -44332.121,   -44932.394,   -42002.051,   -34558.385,       -22263,        -6623,
        10703,        13199,        -4704,       -23716,   -32123.935,   -44214.129,   -44496.101,   -48487.727,   -44475.385,    -42819.18,
   -29323.431,       -18160,         1639,        18397,        22001,          298,       -18146,   -30319.265,   -41289.265,   -46552.368,
   -48558.799,   -47956.943,   -45330.484,   -37615.586,       -24582,        -9576,         8998,         7303,       -14503,   -25352.141,
   -39570.832,   -44536.885,   -51426.672,   -49221.554,   -50259.499,   -42626.814,   -34475.541,       -17195,        -3791,        13497,
        -7797,       -22002,   -35863.251,   -43218.184,   -49727.849,   -52201.383,   -51448.807,   -48241.341,   -40222.341,       -27274,
       -11439,        -2999,       -16199,   -32509.141,   -40356.541,   -49464.837,    -51441.31,   -55416.933,   -50705.444,   -48331.615,
       -33916,       -22654,        -2627,         4602,       -11998,       -27898,       -37623,   -46828.839,   -51848.839,   -55284.741,
   -54687.904,   -50945.323,   -42644.824,       -29370,       -13460,        -5403,       -23893,   -34218.841,   -45594.395,   -49890.954,
   -56932.545,    -55555.37,   -56252.456,   -48009.541,   -39210.779,       -21694,        -6567,         -270,       -18118,   -29580.571,
   -41670.332,   -49151.491,   -55107.491,    -57710.58,   -57479.368,   -54027.557,   -45335.579,       -31624,       -14923,       -13441,
       -25271,   -38936.785,   -46080.109,   -55281.245,    -56909.71,   -60605.352,   -56039.352,   -53903.674,       -38601,       -25728,
        -4741,        -7120,       -20688,   -33543.903,   -44188.742,    -52524.14,     -57486.8,    -60180.13,   -59344.204,   -56081.969,
   -47309.576,       -32800,       -15901,       -15174,       -30767,   -40208.743,   -51834.726,   -55906.827,   -62153.418,   -59845.868,
   -60227.694,   -51662.055,   -42297.694,       -23986,        -8374,       -10042,       -25216,   -37066.562,    -47891.49,   -55479.562,
   -60663.114,   -62228.412,    -61155.65,   -56357.224,   -47260.499,       -34121,       -17000,        -3996,       -21648,   -32577.268,
   -46503.876,   -53177.832,   -61411.832,   -61649.012,   -64472.079,   -58344.099,   -54187.768,       -39998,       -27768,        -6399,
       -15649,       -29361,   -42180.653,   -51555.736,   -58921.391,   -62898.422,   -64220.892,    -61983.64,    -56345.48,    -47090.48,
       -33729,       -18052,       -11653,       -24424,   -40414.553,   -48038.804,   -58900.749,   -61431.505,   -66746.096,   -62797.837,
   -61171.431,   -52000.431,       -42243,       -24964,        -5198,       -20912,       -35527,   -46351.151,   -55545.834,   -61840.387,
   -65512.556,   -65579.531,   -62213.025,   -56547.093,       -46910,       -33823,       -15398,       -33152,   -42616.698,   -54770.668,
   -59792.686,   -67099.277,   -65424.243,   -66003.595,   -58834.328,   -54349.881,       -39521,       -11252,       -27796,   -40672.693,
   -50877.951,   -59169.934,   -65126.052,   -67263.661,   -65911.599,   -62657.173,   -56414.625,       -46981,       -32919,       -24796,
       -36254,   -49573.517,   -56111.332,   -66006.285,   -66258.274,   -68899.427,   -63724.427,   -61624.427,   -51502.304,       -41722,
       -19049,       -33403,   -45692.348,   -55061.049,    -63742.68,   -67318.779,   -67880.441,   -66879.683,    -62657.81,    -56647.81,
       -46491,       -32798,       -28597,   -43128.173,   -51350.415,   -63463.815,   -65567.035,    -70007.22,    -67086.12,   -66979.785,
   -58899.233,   -54214.814,       -38642,       -25299,       -38396,   -50002.598,   -59978.648,   -65736.213,   -68417.973,   -69327.758,
   -67100.605,   -63086.666,   -56301.531,       -46476,       -32435,       -35900,   -45643.206,    -59149.87,   -62976.127,   -69564.648,
   -68910.089,   -70563.111,   -64343.111,   -62046.216,       -51426,       -41676,       -31000,   -43873.368,   -55203.797,   -62711.127,
   -67326.897,   -70140.242,   -69907.736,   -67894.336,   -63116.336,   -57063.323,   -46923.323,       -32304,       -28299,       -39300,
   -53940.319,   -59782.999,    -68131.38,    -68589.38,   -72585.911,   -68229.809,   -67894.407,   -59015.201,   -53940.919,       -38117,
       -37036,       -49863,   -58986.595,   -65410.343,   -69699.335,   -71297.534,   -70956.701,   -68217.642,   -63628.936,   -56551.751,
       -46052,       -31699,       -32248,       -48372,   -56006.205,   -65708.883,   -68049.585,   -73422.437,   -70859.967,   -72212.735,
   -65306.081,   -62331.509,    -51917.05,       -40697,       -29500,       -43901,   -54119.802,   -62469.023,    -68464.58,   -71856.427,
    -73032.41,   -72169.018,   -69139.018,   -64323.624,   -57221.677,   -46621.677,       -41610,   -50975.985,    -62136.64,    -66296.64,
   -73213.046,   -72289.504,    -75252.05,   -70289.169,   -69014.318,   -60479.832,   -54243.893,       -38704,       -36747,       -48577,
   -58722.344,   -65992.344,   -71214.029,   -73916.577,   -74599.594,   -73234.914,   -70169.443,   -64824.531,   -57804.063,       -46905,
       -34298,       -44749,    -57342.57,    -63706.57,   -71862.211,   -72817.419,   -77026.086,   -73452.302,   -74179.727,   -66936.228,
   -63173.924,       -52527,       -41703,       -42327,       -53420,   -62509.526,   -69488.526,   -73636.526,   -75917.602,   -76068.514,
   -74442.736,   -70803.362,   -65476.577,   -58356.577,       -47357,       -36449,   -51844.769,   -59135.169,   -69515.169,   -72159.286,
   -77759.936,   -75889.472,   -77892.492,   -72172.854,   -70308.223,   -61217.786,   -55517.043,       -46128,   -57983.308,   -66303.308,
   -72533.308,   -76389.519,   -77974.839,   -77694.038,   -75454.821,   -71527.705,   -66017.338,   -58488.484,       -47478,       -42457,
       -53104,   -65624.008,   -70324.008,   -77594.008,   -77496.465,   -80589.508,   -76188.201,   -76008.384,   -68192.393,       -64192,
       -52974,       -36300,       -49388,       -60901,   -69880.657,   -75340.657,    -79008.93,   -79981.709,   -79074.805,   -76795.439,
   -72326.557,   -66458.557,   -58958.557,       -47748,       -44106,       -58246,       -66082,   -75951.829,   -77799.335,   -82430.991,
   -79750.025,   -80643.837,   -74157.855,       -71492,       -61879,       -55806,       -40054,       -53067,       -63323,   -72428.267,
   -78610.267,   -81480.267,   -82167.331,   -81102.572,   -77842.123,   -73149.123,   -67149.123,       -59103,       -47665,       -34353,
       -49844,       -59150,    -70540.57,    -75639.57,    -83265.57,   -82747.017,   -84523.576,   -79283.576,    -77804.35,    -69826.35,
    -64556.35,       -53207,       -44237,       -55983,   -66581.926,   -73856.926,   -80709.426,   -84597.795,   -84880.413,   -83018.723,
    -79348.15,    -74183.15,   -67694.927,       -59122,       -47339,       -40138,       -51288,   -63878.135,   -70732.135,   -79692.135,
   -82608.998,    -87921.74,    -84299.14,   -83623.101,   -76073.101,   -72700.088,       -62710,       -55647,       -33692,       -47143,
       -59196,    -68571.62,    -76726.62,   -81712.502,   -86209.141,   -87701.749,   -84868.884,   -80650.386,   -75003.889,       -67844,
       -59513,       -47658,       -41451,       -55927,   -64619.846,   -74969.846,   -79361.712,   -85941.604,   -86487.462,   -88767.265,
   -82656.265,   -80167.265,   -71206.603,       -65307,       -53216,       -36859,       -50338,   -61508.323,   -71310.323,   -77745.323,
   -83645.279,   -86345.031,   -87890.402,   -86632.442,   -82204.165,   -75984.165,       -68658,       -59103,       -47403,       -30926,
       -46649,   -56580.144,   -68785.048,   -74772.048,   -82867.702,   -84813.327,   -88453.882,   -86448.337,   -86805.003,   -78934.648,
       -74408,       -63360,       -55498,       -40716,       -53049,   -64017.525,   -72617.525,   -80084.606,    -84223.16,    -87117.04,
   -87208.278,   -86803.495,   -83602.533,   -77265.533,       -69173,       -59699,       -46780,       -36803,       -47804,       -61143,
   -68553.352,    -78840.43,   -82348.499,   -87266.837,   -86364.502,   -88409.708,   -84153.961,   -82567.898,       -72938,       -66350,
       -53300,       -43901,       -56039,   -65853.935,   -75116.826,   -81207.068,   -85657.766,   -86781.901,   -87707.492,   -86016.873,
   -83449.819,   -78339.819,       -70151,       -60100,       -46696,       -38629,       -53030,   -61224.644,   -72938.959,   -78346.709,
   -85442.791,   -85603.696,   -88790.496,   -85817.254,   -86072.062,   -79679.409,   -76229.409,       -64571,       -56104,       -34652,
       -47916,   -58356.314,   -68788.109,   -76257.693,   -82946.645,   -85605.644,   -87540.442,   -87220.108,   -86112.242,   -82589.242,
   -77799.242,   -70819.242,       -60603,       -47003,       -44796,   -54221.644,   -66645.662,    -72467.42,   -81286.925,   -83528.547,
   -88111.723,   -86427.771,   -88224.469,   -83174.815,   -81299.957,   -73060.614,   -67630.614,       -53896,       -39495,    -50878.87,
   -62185.677,   -70200.924,   -77768.473,   -82326.954,    -85965.84,   -87323.141,   -87616.977,   -85574.394,   -82187.735,   -76757.735,
       -69853,       -61274,       -47199,       -36198,       -46696,   -60219.307,   -67294.307,   -76604.307,   -79939.307,   -86184.307,
   -86016.224,   -89218.984,   -85584.225,   -85226.218,   -78148.384,   -74249.829,   -64169.829,   -56779.829,   -43597.231,   -55407.231,
   -64912.231,   -73457.231,   -78942.231,   -83511.231,    -86335.84,    -87949.72,   -87408.021,   -85428.021,   -81224.197,    -75747.66,
       -68614,       -59560,       -38312,   -53077.471,   -61892.471,   -71742.471,   -76347.471,   -83557.471,   -84565.665,   -89098.043,
   -86775.004,   -87925.075,   -82264.893,   -79677.893,   -70709.488,   -64929.488,       -47553,       -58936,   -68372.027,   -75317.027,
   -80847.027,   -84597.027,   -87258.775,   -88022.185,    -87479.11,   -84791.366,   -80649.453,   -74599.453,       -66974,       -56178,
       -44404,       -54912,       -66341,   -72223.666,   -80328.666,   -82486.166,   -88088.872,   -86949.825,   -89390.045,   -85111.392,
   -83974.674,   -76106.627,   -71591.627,       -59176,       -38582,       -51353,       -62364,   -70852.653,   -77337.653,   -82287.653,
   -85927.653,   -87845.641,   -88412.828,   -87067.992,   -84330.103,   -79481.085,   -73262.528,   -63820.056,       -52499,       -46770,
       -59699,       -67096,   -76255.078,   -79775.078,   -86322.078,   -86361.478,   -89902.478,    -86937.34,   -87132.499,   -80606.451,
   -77425.204,       -66330,   -58214.429,       -42718,       -55191,       -64916,   -72942.869,   -79102.869,   -83922.869,   -86863.285,
   -88367.594,   -88401.743,   -86984.841,   -83559.577,   -78576.801,       -70654,       -60541,       -37744,       -52201,       -60696,
       -71303,   -75952.879,   -83672.879,   -85019.304,   -89524.304,   -87601.836,   -89252.325,   -84115.603,   -82040.983,       -72507,
   -65721.935,       -52652,       -47283,       -58097,       -67245,   -74535.668,   -80850.668,   -85010.668,   -87606.591,   -88722.669,
   -88508.424,   -86488.746,   -82639.154,   -76259.154,   -67612.012,   -57613.447,       -43901,       -53617,       -65456,   -70960.763,
   -79981.763,   -82775.901,   -88349.175,   -87460.552,    -90352.99,    -86474.99,   -85843.888,       -77544,    -72277.12,       -60321,
   -51904.646,       -50627,       -61097,   -69216.683,   -76665.683,   -82357.192,   -86004.158,   -88220.719,   -89257.519,   -88395.728,
   -85944.797,   -80888.145,   -73484.917,       -64947,       -54397,       -45802,       -58833,   -65999.617,   -75483.617,   -79741.327,
   -86336.399,   -86624.458,   -90580.518,   -87996.067,   -88661.269,   -81600.729,   -77301.267,       -67096,   -59966.685,       -46294,
       -42197,       -54138,       -63724,   -72202.714,   -78682.714,   -83692.028,   -87032.672,   -89049.279,    -89369.62,   -88333.039,
   -84419.744,    -78347.03,   -71128.339,   -62092.297,   -51704.182,       -51307,       -59727,       -70532,   -75631.725,   -83496.665,
   -84948.803,   -90020.941,   -88572.155,   -90560.901,   -84515.383,    -81888.57,       -72796,   -67085.913,       -54539,   -45945.564,
       -46305,       -57110,   -66428.402,   -74208.402,       -80403,       -84987,   -88090.485,   -89536.616,   -90035.978,   -87003.403,
   -82062.759,   -76337.797,   -68656.771,       -59699,       -49025,       -52751,       -64450,   -70735.792,   -79960.691,   -82567.691,
   -88719.392,   -88250.019,   -91528.107,   -86821.176,   -85268.962,    -77492.26,   -73046.747,       -62068,       -54604,       -49854,
       -60007,       -68949,   -76530.301,   -82265.301,   -86425.301,   -88945.042,    -90399.95,    -88644.75,   -85096.897,   -80434.508,
   -74185.361,   -66442.814,       -57288,       -46512,       -45196,       -57920,       -65139,   -75465.639,   -79565.639,   -86708.557,
   -87230.346,    -91656.06,    -87999.42,   -87721.044,   -80971.048,   -78079.081,   -68409.391,       -62373,       -49621,       -53244,
       -63239,       -71623,   -78557.492,   -83907.492,   -87704.492,   -90068.363,   -89477.443,   -87184.443,    -83765.53,   -78794.437,
   -72305.075,    -64590.11,       -54967,       -44004,       -50943,       -59234,   -70149.064,   -75649.064,   -83974.064,   -85735.073,
   -91105.073,   -88424.465,   -89404.587,   -83789.587,   -82172.448,   -73888.663,   -68888.663,       -57687,       -49705,       -57082,
       -66257,   -74661.064,   -81061.064,   -85841.064,   -89204.076,   -89595.111,   -88551.151,    -86287.26,   -82472.775,   -77100.495,
   -70742.779,       -62401,       -52704,       -41579,       -52900,       -64692,       -71231,    -80730.32,   -83577.359,    -89945.95,
   -88330.175,   -90314.028,   -86080.028,   -85355.002,   -78139.833,   -74608.944,       -64543,       -57836,       -44889,       -60612,
       -69955,   -77311.209,   -83426.209,   -87820.474,   -89224.113,   -89171.894,   -87943.313,   -85248.552,   -81043.671,   -75654.978,
       -68707,       -60175,       -50338,       -58796,       -66471,   -76710.752,   -80876.752,   -88236.752,   -87620.291,   -90524.548,
   -87364.961,   -87660.103,   -81731.335,     -79089.8,    -70258.61,       -64823,       -53132,       -44497,       -64804,   -73358.534,
   -80480.534,   -85898.534,   -88255.501,   -89022.201,   -88836.431,   -87192.064,   -84087.575,    -79667.97,   -73759.389,       -66658,
       -57911,       -47618,       -61013,   -72327.416,   -77813.416,   -86020.416,   -86398.416,   -90064.575,   -87910.536,   -89168.536,
    -84344.94,   -82669.928,   -74973.468,   -70820.558,       -60258,       -52890,       -39570,       -58898,     -68517.1,     -76985.1,
     -83499.1,     -86700.1,     -88281.1,   -88983.125,   -88320.793,    -86240.02,   -82815.595,   -77895.769,   -71975.611,       -64431,
       -55424,       -45056,       -54800,   -67288.998,   -74358.998,   -83334.598,   -84608.531,   -88992.091,   -87737.939,   -89860.039,
   -85931.378,   -85401.514,   -78631.901,   -75533.918,   -66330.757,       -60184,       -48046,       -39048,       -52452,       -63202,
   -72938.793,   -80593.793,   -84627.681,   -87003.181,   -88503.367,   -88697.386,   -87500.424,   -85064.555,    -81326.12,   -76287.496,
   -69773.559,       -62235,       -52946,       -42253,       -46157,   -61569.949,   -69889.949,   -80138.949,   -82291.605,    -87351.41,
   -86932.379,   -89881.713,   -86900.424,   -87261.603,   -81628.008,   -79422.905,   -71175.457,   -66596.233,       -55470,       -47581,
       -33936,       -55266,   -68137.141,   -77314.218,   -81987.983,   -85209.473,   -87444.363,   -88415.211,   -88059.787,    -86683.79,
   -83769.256,   -79715.394,   -74278.264,   -67768.984,       -59737,       -50198,       -39353,       -50720,   -62419.171,   -76554.171,
   -79673.573,   -85182.183,   -85699.888,    -89280.48,   -87155.926,   -88434.841,   -83740.245,   -82474.025,   -75213.484,   -71425.808,
       -61711,       -55247,       -42504,       -57930,   -70952.598,   -78942.598,   -82944.598,   -85886.598,   -87643.598,   -88070.958,
   -87553.459,   -85494.383,   -82423.228,   -77937.607,   -72332.373,   -65407.646,       -57129,       -47283,       -52024,   -66795.791,
   -74165.791,    -82559.49,    -84072.49,    -88124.49,   -86891.181,   -88949.868,    -85218.65,   -84835.983,   -78514.011,   -75646.459,
   -66738.751,       -61507,       -49826,       -41573,       -47199,       -60799,   -69707.636,   -77827.636,   -83789.636,   -86417.033,
   -87581.853,   -87850.512,   -86650.512,    -84624.93,   -80935.888,   -76213.758,   -69977.556,   -62857.216,       -54194,       -44181,
       -56504,       -64879,    -74425.21,   -79499.294,   -86425.137,   -86338.711,   -88886.935,   -86036.945,   -86468.332,   -81327.241,
   -79199.314,   -71197.972,   -66810.917,       -56262,       -49052,       -35974,       -50310,       -60258,   -69561.221,    -76502.82,
    -82379.35,   -86545.599,   -87721.227,   -87100.653,   -85878.553,   -83177.333,   -79580.199,   -74072.875,   -68025.381,       -60016,
       -51214,       -41004,       -55154,       -65931,    -72330.89,    -80150.89,   -82887.407,   -88261.631,   -86524.681,   -87568.521,
   -83131.521,   -82018.083,   -74940.294,    -71497.79,   -61749.669,       -55778,       -43631,       -34940,       -50319,       -60799,
   -68837.893,   -75643.893,   -80700.915,   -84913.745,   -87231.371,   -86952.496,   -84823.335,   -81991.697,   -77496.499,   -72380.247,
    -65398.07,       -57529,       -48168,       -37688,       -56961,       -64273,   -72990.992,   -77050.992,   -83271.368,   -84321.031,
   -88083.278,   -84695.278,    -84251.77,    -78206.57,   -75455.963,   -66985.963,   -61782.272,   -50482.272,       -42839,       -29305,
       -51558,       -60519,   -68326.902,   -74476.902,   -79725.632,   -82938.221,   -85440.105,   -86020.892,   -84197.879,   -80522.949,
   -75938.662,   -69926.584,   -63224.224,   -54540.837,       -45317,       -34374,       -47432,       -55722,   -65475.096,   -70515.096,
   -77823.147,   -80034.775,   -84538.479,   -83792.724,   -85955.195,   -81156.907,   -78992.889,   -71319.889,   -66959.515,       -57060,
       -49960,       -37474,       -51642,       -60445,    -67671.41,   -73935.736,   -78187.073,   -81611.999,   -83073.499,   -84007.448,
   -82965.734,   -79523.191,   -74242.392,   -68232.392,     -60434.4,       -52322,       -42281,       -31354,       -46584,       -57278,
   -63269.947,   -71768.956,   -74892.447,   -80436.989,   -80755.645,   -83753.165,   -81421.106,   -81971.958,   -75621.643,   -71759.504,
   -62368.181,   -56584.535,       -45196,       -36906,       -52098,   -60056.986,   -67414.986,   -72986.839,   -77096.839,   -79631.839,
   -81437.134,   -81273.762,   -80657.737,   -77998.429,   -72927.362,   -65880.845,   -58288.238,       -49183,       -39691,       -27877,
       -48671,   -55620.044,    -65000.05,   -69122.947,   -75675.496,   -76713.808,    -80931.05,    -79459.88,    -81001.88,   -77122.285,
   -76093.179,    -67769.37,   -62554.136,       -51568,       -44712,       -31275,       -43259,   -52019.275,       -60598,   -66848.403,
   -72028.748,   -75454.748,   -78151.936,   -79047.936,   -79272.075,   -77550.499,   -75363.063,   -70752.013,   -64187.855,   -55837.477,
       -47050,       -36365,   -47302.986,   -58013.403,   -63128.403,   -70390.756,   -72530.756,   -77413.404,   -76871.898,   -79342.169,
   -76302.498,    -76275.83,   -70540.457,   -67859.496,    -58015.31,       -51651,       -39272,       -30348,       -43845,       -53486,
       -60795,    -66695.08,    -71056.55,   -74380.875,   -76071.245,   -77141.922,    -76446.56,   -75133.454,   -71495.975,   -67715.154,
   -61688.403,   -53741.615,       -44041,       -33497,       -38964,       -50599,       -57035,   -64823.663,   -68303.663,   -73689.663,
   -73603.339,   -77057.339,   -74797.274,   -75768.769,   -71110.545,   -69316.955,   -61947.908,   -57832.887,       -46612,       -38732,
       -24938,       -35220,       -45820,       -54287,   -61500.777,   -66770.777,   -70952.896,   -73395.232,   -74582.481,   -74659.094,
   -74194.911,    -71629.52,   -68758.601,   -63632.086,   -58265.971,   -50781.802,   -41543.916,       -30202,       -42597,       -50068,
       -59113,   -63808.061,   -70158.061,   -71262.276,   -74768.765,   -72894.497,   -74714.206,   -70724.206,   -70124.452,   -63608.266,
   -60500.173,   -51770.574,   -46305.574,       -33422,       -37623,       -46929,       -55349,   -61628.663,   -67348.663,   -70684.663,
   -72565.846,   -73373.467,   -72889.831,   -71320.222,   -69149.764,    -65019.41,   -60487.968,    -54015.37,       -47059,   -38407.984,
       -27302,       -42383,       -52704,   -58201.466,   -65691.466,   -68498.396,   -72461.596,   -71744.378,   -73713.221,   -70161.974,
   -70398.165,   -64644.213,   -62612.157,   -54429.236,   -49933.735,       -39568,       -32733,       -38797,       -48400,       -55778,
       -62473,   -66973.239,   -70197.239,   -71824.466,   -72077.123,   -71254.414,   -69159.914,   -66039.673,   -62215.463,   -56635.339,
   -50503.433,   -42554.171,       -34102,       -23668,       -45401,       -51912,   -60530.237,   -64220.237,   -69370.326,   -70092.829,
   -72542.197,    -70097.52,   -70529.829,   -65354.551,   -64212.821,   -56839.827,    -53264.49,   -43749.923,   -37852.167,       -25799,
       -40669,       -48969,       -56793,   -62373.426,   -66733.426,   -69467.426,   -70830.678,   -70770.626,   -69427.886,    -66828.93,
   -63419.838,   -58709.273,   -53441.815,   -46483.134,       -38754,   -29628.547,       -44730,       -54399,   -59092.669,   -65212.669,
   -67211.669,   -70696.751,   -69477.216,   -70412.109,   -66191.026,   -65303.809,   -58703.194,   -56014.817,   -47214.373,   -42104.119,
       -31019,       -23695,       -41451,       -50217,       -56849,   -62213.301,   -66053.301,   -68568.524,   -69539.048,   -69173.476,
   -67335.876,   -64567.376,   -60570.398,   -55842.973,   -49714.975,   -42853.503,    -34448.35,       -25227,       -47422,       -53104,
       -60417,       -63369,   -67948.626,   -67842.938,   -69678.064,   -66388.064,   -66058.488,   -60302.313,   -58169.617,   -50269.937,
   -45937.205,   -35875.507,   -29361.804,       -16660,       -42961,       -50431,       -56979,       -61777,   -65512.709,   -67468.186,
   -68061.133,   -67202.843,    -65208.95,   -61898.708,   -57844.214,   -52562.344,   -46318.685,       -38734,       -30407,   -20875.601,
       -46305,       -54753,       -58647,    -64287.29,   -65681.287,   -68186.808,   -66047.112,    -66342.62,   -61483.558,   -59831.527,
   -52836.866,   -49173.105,   -39782.377,   -34001.937,       -22354,       -14503,       -43147,       -50897,       -56626,       -61488,
   -64601.404,   -66386.498,   -66383.942,   -65174.074,   -62735.074,   -59304.213,   -54791.409,    -49286.28,   -42541.078,   -34909.095,
   -26006.814,       -16124,       -48177,       -53104,       -59746,   -62083.294,   -65973.294,   -64987.069,   -65949.562,   -61888.462,
   -61022.716,    -54642.37,   -51821.541,   -43282.801,   -38233.747,       -27644,   -20459.661,        -7265,       -43799,       -50561,
       -56467,       -60659,   -63617.935,   -64904.574,   -64528.312,   -62935.934,   -60287.224,   -56442.273,   -51635.507,   -45855.107,
   -38861.977,   -30656.812,       -21646,       -11625,       -46603,       -54399,   -57760.118,   -62590.118,   -63076.897,   -64931.595,
   -61893.929,   -61588.481,   -56020.981,   -53858.984,   -46097.776,   -41891.844,       -31850,     -25438.4,       -13205,        -4792,
       -43585,       -50701,       -55843,   -59936.551,   -62286.551,   -63296.733,   -62548.311,   -60594.053,   -57501.125,   -53467.756,
    -48351.06,   -42088.612,       -34837,   -26502.896,   -17078.797,        -6540,       -48102,       -52499,   -58564.175,   -60066.731,
   -62996.731,   -61317.664,   -61574.646,   -57064.151,   -55360.552,   -48393.908,   -44890.192,   -35794.885,    -29990.68,       -18741,
   -11037.512,       -43901,       -50096,   -55603.099,   -58803.099,   -60928.684,   -61279.963,   -60370.311,   -58077.311,    -54716.89,
    -50290.43,   -44917.768,   -38385.847,   -30721.352,   -22081.119,       -12375,        -1788,       -46342,       -53663,   -56244.606,
   -60114.606,   -59800.614,   -60768.957,   -57310.199,   -56253.855,   -50137.665,   -47293.364,   -38917.754,    -33927.78,       -23320,
   -16305.533,        -3612,       -43501,       -50114,   -54524.862,   -57724.862,   -59215.595,   -59312.137,   -57833.542,   -55431.345,
   -51720.273,   -47086.091,   -41250.281,    -34293.12,   -26430.172,   -17470.596,    -7564.773,         3502,       -47730,       -51400,
   -56489.417,    -57379.99,    -59260.28,   -56741.334,   -56403.544,   -51329.977,   -49097.186,   -41523.244,   -37238.053,       -27520,
   -21101.014,        -9282,    -1087.345,       -43780,       -49099,       -53654,   -56258.878,   -57556.282,   -56885.778,   -55411.784,
   -52396.538,   -48727.383,   -43553.865,   -37438.225,   -30271.935,    -21941.57,   -12819.798,        -2569,       -45503,       -51949,
   -53869.598,   -56949.598,   -55575.279,   -55846.626,   -51740.766,   -50227.088,   -43673.097,   -39996.301,   -30868.738,   -25319.155,
       -14195,    -6647.425,       -42802,       -48652,   -52315.634,   -54700.634,   -55170.695,   -54483.847,   -52408.647,   -49632.795,
   -45288.307,   -40104.697,   -33428.622,    -25690.09,   -17443.057,    -7989.172,       -46500,   -49374.133,   -53494.133,   -53387.359,
   -54577.509,   -51365.374,   -50641.603,   -45062.886,    -42097.94,   -33861.029,   -28927.898,       -18537,   -11779.132,          550,
       -42802,       -47469,   -50989.216,   -52389.034,   -52889.623,   -51723.623,   -49701.726,    -46269.17,    -41949.53,   -36047.421,
   -29370.489,   -21550.199,   -12780.882,    -3327.962,       -44116,   -49698.297,   -50343.004,   -52444.262,   -50507.262,   -50415.962,
   -45653.453,   -43546.189,   -36251.884,   -31998.007,   -22326.122,   -16316.846,        -4754,       3567.8,       -41601,       -46416,
    -49064.17,   -50471.936,   -50366.314,   -49303.561,   -46586.212,   -43020.103,   -38077.364,   -32263.781,   -24952.104,   -16921.649,
    -8300.467,         2003,       -44404,   -46685.398,   -49788.398,   -48936.195,   -49644.477,   -45839.673,   -44358.859,   -37977.526,
   -34435.958,   -25596.408,   -20244.723,        -9403,    -1939.209,       -40846,       -44740,   -47411.884,   -48441.634,   -48253.952,
   -46511.436,   -43552.934,   -39471.782,   -34374.658,   -27871.187,   -20661.178,   -12801.105,    -3144.762,       -41880,   -46058.563,
   -46433.254,   -48247.518,   -45447.518,   -44609.074,   -39051.679,   -36169.301,   -28300.768,   -23576.146,   -13351.007,    -6826.135,
       -39523,   -43286.117,   -45296.117,   -46367.023,   -45811.023,   -43662.754,   -40197.266,   -35772.441,   -30186.894,   -23799.819,
   -16587.297,    -7568.734,       -36412,   -41501.304,   -42841.304,   -45707.304,   -44226.631,   -44256.145,   -39610.851,   -37332.183,
    -30318.71,   -26349.123,   -16885.078,   -11045.339,         1047,       -38359,   -41396.176,   -43389.676,   -43822.152,   -42809.355,
   -40335.554,   -36683.166,   -31866.958,   -26175.833,   -19755.772,   -11541.263,        -2213,       -36431,   -38608.542,   -42509.542,
   -41930.192,   -42999.479,   -39172.952,   -37864.474,   -31714.853,   -28539.309,   -20190.133,   -14681.328,    -3169.291,       -32984,
       -36766,   -39904.768,   -41215.714,   -41218.183,   -39715.774,   -36712.973,   -33005.122,   -28117.858,   -22443.512,   -14979.941,
    -6373.435,       -30879,       -33813,    -38667.15,    -39016.15,   -41136.426,   -38328.071,    -37822.95,   -32300.802,   -30201.784,
   -22346.744,   -17815.439,    -7204.962,     -538.359,       -31829,   -35477.935,   -37977.935,    -38985.38,   -38453.064,   -36483.186,
   -33581.955,   -29630.792,   -24602.221,   -17878.165,   -10061.055,    -1415.347,       -28662,   -34296.326,   -35566.326,   -38706.326,
   -36751.194,   -37323.422,   -32881.422,   -31370.436,   -24333.279,   -20416.935,   -10903.017,    -4563.217,       -31112,   -34348.616,
   -36393.733,   -36706.409,    -35697.96,   -33809.297,    -30592.96,   -26281.028,   -20246.022,   -13240.145,    -5053.834,       -29649,
       -31708,   -35880.506,   -34833.207,   -36292.864,   -32776.523,   -32011.418,   -25872.246,   -22555.968,   -13545.828,    -8071.257,
       -30302,   -33392.604,   -34533.808,   -34477.014,   -33394.325,   -31050.961,   -27318.856,    -22194.49,   -15872.871,    -8359.902,
      -146.14,       -27554,   -32432.681,   -32529.281,   -34763.121,   -32262.062,   -32192.983,   -26827.027,     -24207.6,   -15990.063,
   -11005.037,    -1187.575,   -29689.824,   -31689.824,   -32796.847,   -32570.023,   -31000.015,   -28155.026,   -23713.927,   -18023.722,
   -11074.785,    -3476.244,     5065.181,   -28280.779,   -29438.431,   -32647.448,   -31140.018,   -31826.683,   -27496.631,   -25360.754,
   -18009.031,   -13474.451,     -3923.38,     1970.318,   -28267.783,   -30422.425,   -31141.091,   -30540.979,    -28341.16,   -24748.749,
   -19687.634,   -13357.968,    -6344.205,     1475.814,       -25821,   -29907.673,   -29582.104,   -30954.447,   -27494.447,   -26050.199,
   -19369.486,   -15473.405,    -6672.103,    -1230.817,   -24400.873,   -27392.356,   -29095.035,   -29547.053,   -28059.394,   -25227.978,
   -20798.434,   -15214.964,    -8819.148,    -1518.058,     6760.921,   -26602.838,   -27268.884,   -29504.137,   -27048.096,   -26243.283,
    -20370.07,   -16954.491,    -8987.739,    -4006.076,     6122.235,   -23741.111,   -26401.111,   -27663.259,   -27182.028,   -25257.915,
   -21415.944,   -16524.923,   -10789.496,    -4072.179,     3596.375,       -22598,   -24399.705,   -27345.859,   -25983.272,     -25933.6,
   -20732.892,   -17924.235,   -10590.867,    -6275.017,     3141.787,     9213.115,   -23143.394,   -25269.119,   -25761.192,    -24786.57,
   -21539.866,   -17307.062,   -12163.464,    -6160.261,      861.303,     8636.458,       -20750,   -24690.242,   -24345.972,   -25109.735,
   -20667.303,   -18333.551,   -11875.313,    -7984.077,      608.456,     6054.402,       -18751,   -22287.497,   -23820.592,   -23770.091,
    -21061.67,   -17508.993,   -12971.536,    -7713.889,    -1309.717,     5839.136,   -20945.512,   -22253.094,    -23785.44,   -20027.931,
   -18181.739,   -12419.576,    -9115.503,    -1242.551,     3565.079,    13511.303,   -16218.666,   -21033.666,   -22451.905,   -20054.433,
   -17145.849,   -13242.582,    -8631.014,    -2841.602,     3537.912,    11131.119,       -13097,   -16749.473,   -21748.455,   -18870.022,
   -17469.516,   -12491.356,    -9647.976,    -2665.206,     1713.894,    10760.201,        -8346,   -13638.048,    -17614.44,   -18258.461,
   -16365.944,   -12879.634,     -8928.61,    -3769.239,     1854.952,     8844.409,    16502.052,        -5114,    -9246.298,   -14728.292,
   -14791.778,   -15953.071,    -11971.83,    -9597.913,     -3346.17,      461.069,     8789.565,    14042.591,        -6076,    -10491.45,
   -11858.421,   -12432.507,   -11647.148,    -8755.556,    -4157.682,       836.47,     7204.953,    13905.728,        -1651,    -7547.389,
    -8117.295,    -10369.42,     -8621.19,    -8659.606,    -3537.587,     -191.422,     7278.529,    12091.061,    21614.516,    -3184.313,
    -5230.649,      -6653.4,    -6579.472,    -5698.258,    -3549.848,      357.656,      6154.98,    12118.868,    19663.224,     -181.261,
    -1200.193,    -4469.913,    -3379.708,    -4319.753,     -958.372,      100.503,     6428.984,    10711.967,     19485.38,         4477,
     1648.537,     -540.277,    -1255.123,    -1168.574,      318.103,     2533.509,     6011.514,    10926.733,    17871.519,     5873.948,
     1783.844,      2257.25,      255.574,     2978.909,     3291.002,     8122.698,    10304.294,    17800.445,         8821,     5900.825,
     4395.555,     3658.607,     4314.635,     5887.347,     8706.595,    12215.918,    17068.683,    22699.383,        13335,     8358.331,
     8098.722,     5217.537,     7059.162,     6651.082,    10843.944,    12374.432,      18668.9,    21923.337,        12802,    10397.048,
     8830.754,     8618.322,      9394.19,    11569.518,    14472.525,    18521.029,    23212.048,        15465,    14352.164,    10613.426,
    11482.904,    10272.874,    13751.627,    14668.946,    20376.714,        23029,        16813,     14472.42,    13278.226,    12963.917,
    14523.155,    16937.984,    20379.211,        24591,        20800,    16373.558,    16349.332,    14321.283,    16621.441,    17202.945,
        22116,        24299,        23464,        20297,    18383.833,    17234.662,    17826.437,    19385.739,    22320.714,     25838.34,
        22440,     21657.19,     18827.19,     20234.72,    19996.285,    23870.222,    25713.685,        26492,    23814.031,    21994.031,
     21638.22,    22310.233,     24340.57,    27377.277,    31590.626,        28774,    27373.099,    23669.099,    24310.214,     23197.06,
    26033.165,    27328.826,        32738,        32981,    29654.986,    27178.986,    25850.941,    25806.176,    26831.753,     29021.97,
     32562.04,        35384,        33282,    28941.792,    28895.981,    26772.188,    28924.171,    29224.699,        33701,    36088.247,
    35816.157,    32562.774,    30753.502,    29586.514,    29897.971,    31210.582,    33779.521,    37399.683,        39598,    34517.809,
    33807.809,    30863.976,    32174.506,    31614.706,    35236.181,    36933.632,        42328,        38396,    35917.278,    33817.278,
    33425.722,    33807.368,    35625.068,    38285.435,        42439,        46363,        40649,    39148.307,    35448.307,    35947.837,
    34610.734,        37361,    38365.534,        43398,        44767,        41498,    38733.238,    37490.098,    36919.958,    37949.575,
    40051.797,        43173,    47293.098,        47230,        45103,    40614.285,    40341.197,    38146.625,    39956.471,    40349.597,
        44534,    46723.592,        47722,     44255.35,    42330.456,    40920.456,    41044.669,    42183.684,        44662,        47910,
        52704,        51508,        46454,    45346.325,    42446.325,    43379.304,    42902.718,        46183,        47890,        53403,
        50202,    47641.875,    45391.875,    44873.275,    45093.307,        46571,        49277,        53098,        57818,        52625,
    50768.948,    47308.948,    47456.272,    46164.745,    48423.087,    49395.914,        54288,        57203,        53337,    50573.883,
    49312.385,    48589.877,    49391.985,        51192,        54287,        58145,        56803,    52715.098,    52314.736,    50126.995,
    51511.785,    51725.434,        55665,        58034,        64199,        56197,    54261.791,    52956.791,    52936.008,    53703.425,
        56103,        59361,        63843,        58620,     57418.39,     54718.39,    55469.685,    54805.218,        57735,    59337.614,
        64967,        68400,        59875,    57755.509,    57176.109,    57183.593,    58691.176,        60945,        64783,        69259,
        63202,    59805.555,    59880.951,    58453.651,    60715.501,    61479.247,        66027,        69009,    63106.068,    61899.746,
    61004.706,    61815.448,    63386.875,        66438,        70221,        75292,     65395.19,     64994.64,    62618.439,    63968.439,
    64091.733,        67902,    70140.589,        76276,        68996,        67154,    65533.901,    65490.624,    66136.624,        68610,
        71583,        76043,        70562,    67392.202,        68080,    67239.766,        70301,    71906.414,        77149,        80664,
        73104,     70750.15,    69849.622,    69725.622,        71176,        73619,        77326,        81816,    72989.038,     72951.37,
    71171.793,        73227,    74073.581,        78636,        81516,    76647.617,    75227.617,    74134.617,    74512.202,     75986.64,
        79027,        82914,        87896,        79056,        78534,    76033.987,    77293.987,    76817.382,        80630,    82881.097,
        88837,        80929,    79301.015,    79013.697,    79349.538,        81301,        84466,        88687,        93791,        84393,
    81340.767,    81991.986,    80904.186,        83514,    84724.274,        89847,        93320,        84809,    84088.873,    83799.253,
    84842.994,    86853.555,        90057,        94397,       100041,        87039,        87186,    85486.109,    87615.418,    87823.739,
        91872,    94235.704,       100720,        89403,    88589.481,    88996.213,    90240.522,        92735,        95934,       100342,
        92702,        90426,    91688.184,        91479,        94839,        96399,       101748,       105415,        93704,        93624,
        94109,        95852,        98400,       102101,       106656,        95644,        96551,        95611,        98276,        99149,
       103676,   106583.474,       113614,        98478,        98504,        99561,   101315.395,       104379,       108162,       113329,
       101407,        99951,       102123,       102394,       106269,       108424,       114473,       102979,       103669,       104837,
       107111,       110216,       114606,       119751,       104649,       106226,       106176,       109361,       110784,       116068,
   119599.066,       107903,       108709,       110476,   112817.611,       116574,       121173,       126824,       111132,       109876,
       112738,       113703,       118246,       121185,       127893,       113204,       113994,       115901,       118906,       122761,
       127901,       134453,       115174,       116851,       117001,       120866,       123108,       129224,       133944,       118728,
       119934,       121741,       124872,       129529,       135183,       121757,       121400,       124463,       125426,       131021,
       134806,       124329,       125919,       128226,       131470,       136056,       125898,       128576,       129526,       133891,
       136293,       143091,       128751,       130053,       132259,       134986,       138665,       143154,       132682,       133325,
       137388,       139251,       145046,       134368,       135953,       138463,       141751,       145445,       137123,       140801,
       142551,       147636,       139576,       141978,       144984,       148591,       152712,       144207,       145750,       150533,
       153056,       145493,       147978,       151338,       155136,       148848,       153206,       155596,       150959,       154043,
       157689,       156012,       158135,       156878,       160023,       164363,       160574,       165881,       162177,       166488,
       171114,       168117,       171260,       168643,       172884,       178088,       172968,       179305,       174449,       179513,
       185239,       180842,       184985,       181068,       186309,       192413,       186100,       193330,       199964
};


//+----------------------------+
//| Table of Beta Decay Energy |
//+----------------------------+
//unit: keV
const G4double G4NucleiPropertiesTableAME03::BetaEnergy[G4NucleiPropertiesTableAME03::nEntries] = {

      782.347,            0,            0,       18.591,       -13736,            0,    23476.602,    -22898.27,            0,    21506.207,
     -292.653,       -26317,            0,    24268.651,     3508.313,    -4288.154,       -25228,            0,        23034,    11192.898,
     -861.893,   -12098.312,            0,      10651.2,    16005.172,   -17979.819,    -12172.57,            0,    15985.165,    13606.616,
    -1068.034,   -16494.809,            0,    15758.623,     20443.91,      555.939,    -3647.951,   -23101.466,            0,    20623.246,
    11506.133,    -1982.411,   -13653.227,            0,        25019,    11707.607,    13368.899,   -17338.082,   -14709.872,            0,
    16685.657,    13437.155,    -2220.469,   -17766.948,            0,    16290.816,     20643.79,      156.476,    -5143.939,       -24651,
            0,        20825,    19099.134,     9771.706,    -2754.166,   -13919.767,            0,        20596,    23387.557,      8010.47,
     10420.66,   -15417.255,   -13316.208,            0,    22731.984,    13167.463,     8680.181,    -2760.514,     -14509.2,            0,
        27396,    11811.712,    13895.988,    -1655.223,    -4443.465,   -18872.802,            0,        26943,    16558.536,    12527.259,
     4822.257,    -3238.827,   -11175.367,   -20113.285,            0,      15792.5,    17967.648,     3814.866,     7024.527,    -13889.65,
    -10722.63,            0,        20709,    17188.258,     8110.457,     5684.226,    -3547.616,   -13094.667,       -15209,            0,
        21243,    22754.523,     6490.774,    10818.093,    -2842.278,    -4785.474,       -18580,       -13982,            0,        23783,
    11283.213,     8483.792,     4375.809,    -4056.087,   -12243.337,       -17002,            0,        28473,    11510.874,    13511.047,
     2466.593,     5515.453,   -13876.621,   -10811.619,       -21242,            0,        29063,        16169,    13380.781,     7249.743,
     3835.008,    -4276.654,    -12740.49,       -15048,            0,        17441,    17842.161,     7291.928,     9352.266,    -4004.273,
    -5065.677,       -18118,       -14997,            0,        20027,    17856.911,    12587.386,     9069.214,     2610.007,    -4812.357,
   -11667.271,       -18260,            0,        20623,        21982,    12233.847,    14029.394,       1831.8,     4642.356,   -14334.044,
   -11231.956,       -22484,            0,        22239,    15392.877,    13284.036,      7596.29,     3679.724,     -4942.42,   -13793.044,
       -16303,            0,        25801,    14740.935,    17271.762,     6961.747,     8560.509,    -4232.352,    -6138.043,       -18506,
       -15640,            0,        25447,        18187,    15872.148,    11736.247,     7995.378,     1491.879,    -5396.237,   -11977.483,
       -18360,            0,        18214,     20019.26,    10107.185,     13018.94,      224.311,     1710.479,   -12685.926,   -11129.568,
       -22619,            0,        21108,    19995.224,    13423.447,    11963.285,     5844.824,      248.508,    -5582.562,   -11619.291,
       -16147,            0,        20360,        23952,    11741.097,    17024.276,     4600.898,      5374.12,    -5492.012,    -6062.559,
       -16896,       -14634,            0,        23430,        16282,    14230.117,    10497.433,     3988.616,      167.184,    -5966.129,
   -11878.511,       -15770,            0,        26529,        15642,    18264.481,     7768.471,    10413.097,    -1142.217,      709.682,
   -12805.369,   -10986.812,       -20337,            0,        26026,        19303,    16526.324,    12414.146,     7902.216,     4865.172,
     -813.873,     -6147.46,   -11638.439,       -16003,            0,        18946,    20117.867,    10690.547,    12103.377,       2936.9,
     4916.453,     -5913.86,    -6741.472,       -17122,       -14038,            0,        22171,    19468.213,     14801.94,    10288.509,
     6637.958,     3441.808,          565,    -6532.611,   -13106.379,       -15668,            0,        20940,        23828,    13573.889,
     14759.73,     4691.242,     7482.086,    -1504.691,      1311.07,   -14323.047,   -11672.953,       -19181,            0,        22142,
    18839.061,    13742.597,     8288.083,     5760.279,     2491.606,     -421.315,    -6495.367,       -12942,       -15495,            0,
        25243,        17495,    18616.368,     7235.487,     9509.685,      598.881,     3525.516,    -6425.833,    -6999.687,       -16952,
       -14159,            0,        20931,    17731.183,    12202.933,      7841.64,     4583.431,       1815.4,     -2220.71,    -6866.826,
       -11297,       -15891,            0,        20744,        21216,    11114.884,    12442.001,     3136.554,     5658.873,    -3652.386,
     -267.634,   -13432.079,       -10655,       -19860,            0,        21156,    15109.975,    11407.942,     6837.597,     4203.764,
      255.842,    -2062.055,    -7126.108,   -12914.411,       -13851,       -18693,            0,        24806,        15407,    15011.874,
     5698.196,     7716.754,    -1377.962,     2366.307,    -7050.409,    -7599.272,       -17104,       -13125,            0,        18518,
        15391,     9788.436,      6643.85,     1991.998,      600.273,    -2930.343,    -7443.667,       -12296,       -15640,       -17326,
            0,        17903,        19012,         8408,    12090.194,      281.972,     3991.625,    -4012.342,    -1656.205,   -13495.749,
       -11163,       -19800,       -16758,            0,        21703,        18444,        12174,        10970,     5263.103,     2006.431,
     -601.856,    -2626.459,    -7714.898,       -13033,       -15006,       -18574,            0,        21806,        10849,    14218.691,
     4966.053,     6889.787,    -2205.118,     1037.945,    -7632.685,    -8151.273,       -17280,       -13404,            0,        21294,
        14205,        13861,     7354.933,     6509.665,     2473.534,     -752.576,    -3207.466,    -8018.999,       -12948,       -15835,
            0,        13199,        16310,       7847.4,     9108.295,     1976.474,     3975.623,    -4711.489,    -2373.829,       -14416,
       -11262,       -20027,            0,        16599,        15901,         9725,         9206,         5020,     3435.902,     -596.837,
    -3742.581,      -8300.5,       -13275,       -15910,            0,        18490,        10326,    11375.554,     4296.559,     7041.591,
    -1377.175,      697.086,    -8242.916,    -8798.762,       -17516,       -15127,            0,        17847,        11463,    12089.761,
     7481.159,         5956,     2603.089,     -231.212,    -3451.811,    -8691.977,       -13711,       -16702,            0,        11830,
        13665,     7143.324,     9201.137,     1628.464,     3695.642,        -4566,    -2135.678,       -15303,       -12873,       -20987,
            0,        13568,        12855,    10644.838,     8335.398,      4962.66,      2693.33,     -835.927,    -3262.234,    -8772.393,
       -14510,       -16899,            0,        15593,         9441,    11625.983,       4072.1,     6246.591,     -2307.55,      381.826,
    -8565.639,    -9364.361,       -18312,       -15612,            0,        15174,        11851,    10824.929,     7588.072,     5183.552,
     1565.297,    -1072.761,    -4798.427,    -9096.724,       -13140,       -17121,            0,        17652,        10929,    13926.608,
     6673.956,         8234,      237.181,     2823.067,    -6127.981,    -4156.331,       -14189,       -12231,       -21368,            0,
        13712,        12820,     9375.083,     7365.656,     3977.031,      1322.47,    -2237.252,     -5638.16,        -9255,       -13361,
       -15677,            0,        12771,        15991,     7624.251,    10861.945,     2530.756,     5314.591,    -3948.259,    -1626.406,
        -9171,        -9757,       -17279,            0,        15714,        14615,        10824,     9194.683,     6294.553,     3672.169,
       66.975,    -3366.505,    -5665.933,        -9637,       -13087,            0,        17754,         9465,    12153.971,     5022.017,
     7306.591,    -1675.034,      579.353,    -7169.267,    -4484.447,       -14829,            0,        16543,        12877,    10205.257,
     8291.983,     5956.118,      2137.61,    -1352.063,    -3254.426,    -6242.549,        -9434,       -14062,            0,        11457,
        13320,     6537.816,     9894.953,      251.989,     2641.152,        -5175,        -2100,   -10122.123,        -9781,            0,
        14354,        12289,     9368.702,     8681.631,     3576.099,      561.662,    -1000.758,    -4221.873,        -6010,       -10157,
       -13693,            0,        14531,     8222.243,    12113.399,      2103.22,     4440.185,      -2921.1,     -106.336,    -8080.552,
    -4684.419,       -15573,            0,        13097,        11606,      9976.05,     5757.564,      2681.76,      909.785,    -2227.152,
    -4013.939,    -6785.135,        -9826,       -14041,            0,         9743,    13506.663,     3826.257,     6588.522,     -654.559,
     1653.022,        -6220,    -2296.895,       -10620,        -9750,            0,        12873,    11330.429,      7507.33,     4615.771,
     2813.345,     -232.506,      -2013.4,        -4778,    -6053.013,       -10140,       -14619,            0,        11001,        14641,
      5842.68,     8348.381,          458,     3996.532,    -4356.102,     -335.402,    -8879.206,    -5074.282,       -15824,            0,
        12827,         9124,     6423.748,     4288.992,     1598.199,     -340.833,    -2739.059,    -4588.706,    -7077.185,       -10500,
       -14353,            0,        16124,         7634,     9702.678,     2340.702,     5372.853,     -2562.47,     1352.768,    -6906.653,
    -2974.573,   -10414.458,       -11220,            0,        14401,        10218,     8349.221,     5995.557,     3391.847,     1175.983,
     -863.392,        -3030,    -4815.395,    -7101.947,       -10600,            0,         9366,    11160.655,         4160,     6916.407,
     -923.543,     2962.547,    -4962.881,    -1274.851,    -8534.486,    -6235.939,       -15540,            0,        11830,        10145,
         7270,     5221.685,     2702.548,      683.016,     -1364.68,    -3065.471,    -5344.912,    -7020.468,       -10899,            0,
        10451,        12594,         6364,     8155.641,      955.208,     4208.667,    -3573.784,      727.424,    -7243.499,    -3762.304,
       -10647,       -10824,            0,        11092,         9090,         6979,         4148,     2281.076,      150.912,    -1625.778,
    -3639.374,    -5326.785,        -7120,       -10999,            0,        15395,       7290.4,        10380,     2644.117,     5600.649,
    -1870.463,      2003.02,    -5719.638,    -1864.631,    -9090.437,    -5700.743,            0,        11856,         8320,         6230,
     3856.211,      1585.32,     -280.801,    -2239.217,    -3927.116,    -5510.367,    -7528.855,       -11010,            0,        10647,
        12520,         4700,         7270,      -97.544,     3093.044,    -4401.308,     -179.817,    -7815.991,        -4000,       -11218,
            0,        13087,        11513,         8980,         5460,     3668.272,       972.78,     -906.905,    -2279.366,    -4468.882,
        -5868,        -7500,       -11210,            0,        14140,         7836,         9870,     1847.507,     4631.656,    -2680.965,
      893.812,    -6485.983,        -2666,        -9613,        -6073,            0,        13013,        10256,         9105,         6182,
         2870,      687.064,    -1064.758,    -3260.449,        -4693,        -6000,        -8046,       -11439,            0,        15491,
         9306,        11391,         5099,         7626,     -518.554,     1776.559,        -5240,    -1479.226,        -7978,        -5270,
       -11349,            0,        11746,        10599,         7275,       6852.5,     3888.369,      282.618,     -1861.69,    -3670.573,
        -5165,    -6488.223,        -8573,       -11783,            0,        11150,        12590,         6854,         8960,     2916.863,
     5312.742,      -3622.6,     -676.039,        -7550,    -3373.013,        -9990,        -7063,            0,        13451,        12054,
         9375,         8155,     4985.882,     4496.639,     1492.607,    -2832.864,    -4218.499,    -5646.497,        -7160,        -8331,
       -11855,            0,        14475,         8693,        10350,     4391.865,     6579.892,      545.858,     2279.804,        -6111,
        -2489,    -8960.663,        -5900,       -12091,            0,        13479,        11170,         9802,         6435,     5899.956,
     2699.752,     1545.371,     -1257.96,    -4428.277,        -6220,        -7326,        -9555,       -11700,            0,        15724,
         9931,    12204.904,         5987,     8095.654,     1945.625,     3640.555,    -2005.544,      356.666,    -7870.356,        -4527,
       -11048,        -7862,            0,        12333,        10969,         8600,     7467.082,     4138.554,     2893.879,       91.238,
     -404.783,    -3200.962,        -6337,        -8093,        -9473,       -12920,            0,        11001,        13339,         7410,
    10287.079,     3508.069,     4918.338,     -902.335,     2045.206,    -4255.748,    -1586.063,        -9630,        -6588,       -13050,
            0,        12137,         9815,     9262.891,     6090.243,     4450.698,     1124.134,      925.591,    -1690.619,    -2567.053,
        -5110,        -8189,       -10051,       -13404,            0,        14401,         8195,    11714.315,      5407.75,     7096.082,
      160.905,       3186.8,    -2973.242,      254.808,    -6392.653,        -3450,       -11658,        -8467,            0,        13264,
        10440,    10431.794,     7469.584,     6688.952,     2658.998,     1934.798,     -320.334,    -1107.865,        -3523,        -4790,
        -6980,       -10216,       -13600,            0,         9426,    12424.019,     5821.758,     8819.505,     2241.622,     4583.177,
    -1683.952,     1796.697,    -5049.653,    -1874.858,    -8239.343,        -5430,       -13734,            0,        11384,    11306.807,
     8015.248,     7567.548,     4558.481,     3638.886,     1357.302,      293.835,    -2042.582,    -3386.659,        -5430,        -6905,
        -8579,       -14075,            0,        10498,        13524,         7075,         9310,         3335,         6245,     -168.083,
     3202.759,    -3634.759,     -358.007,    -7077.835,    -3898.554,       -10080,        -7390,            0,        11810,         9505,
         8545,         5485,         4569,     2824.609,      1613.88,     -541.699,        -1980,    -4203.824,    -5476.538,        -7134,
        -9054,            0,        14765,         8815,         9850,         4605,         7210,     1008.194,     4532.378,    -2323.038,
     1150.071,    -5660.182,        -2587,    -8968.406,        -5780,            0,        11383,         9436,         6945,         5530,
         3750,     2661.748,      763.411,     -543.075,    -2687.744,    -4141.912,        -6050,        -7625,       -10796,            0,
        10507,        11429,         5883,         8105,       2157.5,     5602.706,    -1139.047,     2440.221,    -4278.653,    -1136.719,
    -7868.047,        -4515,       -12415,            0,        12771,        11010,         8489,         6485,         4950,         3640,
     1917.988,      567.188,    -1344.837,    -2737.888,    -4849.018,    -6218.558,    -9442.472,       -11321,            0,        12929,
         7396,         9160,         3520,         6547,         39.4,         3541,    -2965.138,      195.159,    -6526.048,    -3181.246,
       -11095,        -8116,            0,        12473,         9725,         8027,         6160,         4820,     2940.416,     1504.308,
        34.15,    -1416.903,    -3425.264,    -4982.775,        -7923,       -10113,            0,        14457,         8495,        10607,
         4650,         7720,     1346.426,         4505,    -1922.468,     1650.489,    -5136.721,    -2074.621,        -9533,        -6786,
       -13070,            0,        10815,         9147,         7291,         6315,         4160,     2595.923,     1116.078,     -214.245,
    -2019.678,    -3849.592,        -6380,    -8647.142,    -9998.565,            0,         9715,        11839,         5505,         9021,
     2794.138,     5573.274,     -888.623,     2892.439,        -3878,     -631.103,        -8300,        -5267,       -11956,        -8417,
            0,        10470,         8120,         7449,     5691.509,     3646.966,     2216.561,       1036.8,     -861.791,     -2450.93,
    -5056.652,    -7403.228,        -8538,       -10550,            0,        13032,         7166,         9484,      4257.71,     6595.072,
       288.06,      3956.06,    -2584.451,      665.202,     -7060.54,    -4299.462,       -10206,        -7129,       -13673,            0,
        11942,         9585,         8479,         6480,     5009.314,     3340.644,     2016.607,      320.341,    -1036.581,    -3913.295,
    -6072.714,    -7218.691,    -9036.042,   -10388.114,            0,         8421,        10804,         5100,     7864.941,     1452.137,
     5072.138,    -1448.786,     1988.746,    -6045.518,    -2626.813,        -9092,        -5710,       -12547,        -8594,            0,
        10805,         9319,         7780,     6194.598,         4584,     3103.485,     1446.131,      499.361,    -3032.574,    -4940.644,
    -5724.962,    -7681.026,        -8957,       -10675,            0,        11700,         6286,     9224.899,         2607,     6151.701,
     -469.373,     3278.088,    -4706.932,    -1552.214,    -7776.702,    -4445.512,       -10979,        -7464,            0,        10153,
         8942,         7581,         5735,         4160,     2519.741,     1454.908,      -1755.2,    -3547.854,    -4662.389,    -6249.147,
    -7742.546,        -9155,       -10776,            0,        12724,         7219,        10326,         4100,     7142.918,      521.789,
     4425.714,    -3656.639,     -278.377,    -6749.995,    -2891.968,     -9669.69,        -6037,       -12752,            0,         9995,
         8383,         6935,         5350,         3797,     2363.871,      -590.92,        -2293,    -3418.913,    -4971.093,    -6489.361,
    -7714.965,        -9623,       -10964,            0,         8290,        10915,         5500,         8325,     1761.008,         5370,
    -2680.608,      980.122,        -5615,     -1617.14,    -8283.785,        -5000,       -11201,        -7983,            0,         9175,
         8404,         6400,         4780,     3363.012,      391.036,    -1043.961,    -2263.891,    -3814.485,     -5372.28,    -6357.716,
        -8342,        -9697,       -11125,            0,        11793,         6539,         9499,     2847.038,     6368.591,    -1615.774,
     1983.852,        -4234,     -725.026,    -7215.169,    -3530.889,       -10066,        -6707,       -12948,            0,         9343,
         7356,         6115,     4394.265,     1403.639,      -52.219,    -1228.581,    -2694.761,    -4204.882,    -5388.693,        -6948,
        -8532,        -9837,            0,         7676,        10239,         4166,         7360,      -616.46,     2904.256,    -3159.587,
      295.143,    -5928.769,    -2641.535,     -8831.19,        -5436,       -11690,        -8635,            0,         8554,         7122,
         5418,     2356.966,        766.7,      -185.77,    -1644.366,    -3104.489,    -4419.606,     -5908.58,        -7102,        -8747,
       -10293,            0,        11315,         5486,         8207,          378,     3666.159,    -2154.039,         1258,    -4823.596,
    -1675.013,     -7696.46,     -4152.91,       -10562,        -7368,       -13320,            0,         9619,         8468,         6514,
         3201,         1581,      702.025,     -662.331,    -2080.773,    -3424.425,    -4919.826,    -5920.158,        -7544,        -9008,
       -10368,            0,        12489,         7070,       8975.6,     1273.933,      4383.56,    -1254.152,       2122.1,    -3928.661,
     -529.864,    -6769.613,    -3097.984,    -9203.161,        -6147,       -12137,        -8998,            0,        10749,         9737,
         7655,     4033.888,       2375.5,     1500.186,      194.019,    -1196.962,    -2435.868,    -3738.436,    -5038.623,    -6513.938,
        -7539,        -9289,       -10694,            0,        15413,         8320,        10249,     2152.656,     5059.805,     -419.031,
     2949.334,    -2981.289,      361.178,    -5633.594,    -2205.103,    -8247.448,    -4579.225,       -11126,        -7890,       -13645,
            0,        12872,     9177.077,     4673.766,     3221.489,     2234.891,      970.848,     -355.425,    -1375.996,    -2914.534,
    -4053.862,    -5437.131,     -6509.28,        -8032,        -9538,       -10846,            0,        11699,        14135,     3119.402,
      5508.61,      517.704,     3580.592,    -2124.554,     1278.915,    -4694.596,    -1266.219,    -7260.542,    -3787.676,        -9714,
        -6465,       -12743,            0,        13023,         7990,         4002,         2942,         1757,       427.36,     -517.499,
    -2059.076,    -3071.155,    -4485.621,    -5605.235,    -6924.726,        -8279,        -9846,            0,        14772,         7370,
     8393.699,         1513,         4052,    -1233.309,     2058.687,    -3731.218,     -382.667,    -6321.972,    -2867.552,    -8907.708,
        -5232,       -11681,        -8253,            0,        13600,         8909,         8120,         5962,     2627.397,     1164.821,
      268.659,        -1200,    -2025.582,    -3689.042,     -4722.13,    -6236.202,     -7120.34,        -8663,       -10014,            0,
         8374,         9547,     5074.084,     6925.844,      -86.426,     2548.224,    -2849.991,      431.388,    -5141.091,    -2127.927,
    -8001.342,    -4387.055,       -10549,        -7210,       -13078,            0,         9948,         9303,       6941.6,      5876.53,
     4166.249,     1175.628,     -620.575,      -1222.1,    -2701.219,    -3597.134,    -5507.324,    -6047.495,        -8009,        -8803,
       -10209,            0,        10777,         6400,         7820,     2736.517,     5374.224,     -1736.95,     1043.841,        -4437,
    -1113.438,    -7077.789,    -3442.504,    -9748.121,        -5972,       -12147,        -8691,            0,        10479,         8039,
         6806,     5057.023,     4212.829,     2317.626,     -278.875,    -2129.161,    -2831.638,    -4495.198,    -5116.252,    -6982.177,
        -7869,        -9362,       -10479,            0,         7312,         8718,         4060,     6220.376,     1049.663,     3762.248,
        -3388,     -443.508,      -6045.2,    -2750.607,        -8470,    -5203.691,       -11300,        -7643,       -13534,            0,
         8961,         7808,         6150,      5248.73,     3212.589,     2501.884,      580.787,    -1823.014,    -3674.929,    -4584.287,
    -6012.078,     -6702.36,    -8683.387,        -9224,       -10944,            0,         8290,         9753,         5040,     7308.052,
     2211.628,     4503.704,     -745.754,     2162.471,    -4798.288,    -2164.018,        -7673,    -4360.374,        -9900,        -7100,
       -12486,            0,         8803,         7227,     6264.326,     4251.337,     3424.926,     1461.499,       933.95,    -1041.714,
    -3442.543,    -5280.799,        -6010,    -7797.992,        -8112,       -10042,       -10926,            0,        10694,         5992,
     8499.008,     3123.492,     5544.542,      318.656,     2997.521,     -2332.06,      550.853,    -6350.316,    -3862.139,    -9391.323,
    -5783.646,       -11388,        -8290,            0,         7959,         7358,     5571.853,         4110,         2535,     1805.295,
     -163.372,     -616.026,    -2659.307,    -5071.067,    -7046.517,    -7592.608,        -9105,        -9492,       -11814,            0,
         6949,     9380.006,     4122.897,     6552.549,     1038.312,     4217.242,     -1471.17,         1542,    -3879.595,    -1029.106,
    -8323.809,    -5215.234,       -10987,        -6856,       -13437,            0,         8761,         8579,         6250,     5180.345,
         3426,     2697.188,          896,      224.139,    -1721.576,    -2187.436,    -4611.049,    -6564.159,    -8350.378,        -8788,
       -10685,            0,    10710.417,         5115,     7262.354,         2140,     4882.647,     -541.505,      2470.27,    -3039.671,
      -26.668,    -5735.373,    -2680.961,    -9844.186,        -6364,       -12380,        -8924,            0,         9641,         7309,
         5900,     4361.471,     3324.324,      1690.37,     1070.677,     -695.362,    -1313.107,    -3637.478,    -3780.821,    -6026.751,
    -7946.788,        -9701,       -10545,            0,        11634,         6437,         7788,         3480,         5386,      -86.324,
         3454,    -2260.066,      971.496,    -4658.225,     -1793.59,    -7369.047,    -4115.021,       -11221,        -7880,       -13794,
            0,        10600,         8467,         7213,         5270,     4182.119,     2442.336,     1187.249,       76.613,     -464.183,
    -2565.391,    -2870.919,    -5126.515,    -5366.115,    -7484.169,    -9237.886,       -11342,            0,         7471,         9045,
         4695,         6350,     1104.215,     3506.488,    -1874.268,     1819.709,        -3990,     -599.754,    -6516.186,    -3108.093,
    -8729.599,        -5465,       -12884,            0,         9306,         8421,         6279,         5720,         3336,     1881.183,
      807.621,     -483.636,    -1569.609,    -2170.458,    -4130.355,    -4531.442,    -6472.598,        -6956,        -8651,       -11106,
            0,        10321,         5498,         7490,      2806.93,       3963.2,     -717.217,     1968.842,    -3551.247,      236.191,
    -5753.952,    -2032.055,    -8182.921,    -4495.502,       -10366,        -6835,            0,         9604,         7377,         6695,
         4500,         3224,     1627.228,      252.657,     -822.709,      -2094.5,    -3120.241,    -3824.209,    -5580.124,    -6131.907,
    -7949.262,        -8452,       -10434,            0,         6511,         8618,         3690,     5150.089,      722.503,     2449.369,
    -2444.677,      432.309,    -5175.278,     -1141.73,    -7372.994,    -3575.337,    -9514.567,    -5897.756,       -12053,            0,
         8300,         7825,         5580,         4360,         2734,     1363.252,      -60.052,     -1342.74,    -2598.956,    -3409.092,
    -4710.565,    -5267.458,    -6958.682,        -7729,        -9126,            0,         9669,         4693,         6120,         1999,
     3485.082,    -1219.535,      934.893,    -4221.083,     -887.217,    -6600.615,    -2688.377,    -8800.444,    -5110.253,       -11085,
        -7324,            0,         8765,         6632,         5364,         3840,     2515.223,      970.524,     -365.571,      -1837.6,
      -2768.5,    -3996.979,    -4727.425,    -6127.998,    -6861.472,    -8405.153,        -9222,            0,         5682,         7312,
         2952,         4580,     -105.688,     1835.126,        -3290,     -329.576,    -5756.175,    -2132.696,     -7899.68,    -4332.732,
   -10061.697,    -6513.704,       -12702,            0,         7471,         6548,         4797,         3736,     1955.478,      592.947,
      -858.29,    -1993.894,    -3310.241,    -4054.494,     -5281.87,    -6243.659,        -7585,        -8327,        -9531,            0,
         8449,         3894,         5640,     1393.997,     2505.521,    -2139.695,      295.507,    -4859.062,    -1652.031,    -6994.661,
    -3663.761,    -9390.728,     -5780.44,       -11648,        -7851,            0,         7750,         5729,         4862,         3113,
     1785.094,       -2.555,    -1209.868,        -2439,    -3430.861,    -4512.804,     -5505.13,    -6745.201,    -7631.983,    -8902.281,
        -9883,            0,         4928,         6642,         2337,         3890,     -986.225,      962.493,      -4061.1,     -865.746,
    -6380.346,    -2820.829,     -8538.74,    -5049.053,       -10590,        -7184,       -13195,            0,         6763,         5906,
         4192,         2959,     1286.638,     -376.261,    -1592.378,    -2648.711,    -3844.951,    -4806.766,      -5780.4,     -6993.13,
    -8205.166,        -9011,       -10021,            0,         7797,         3361,         4830,      486.778,     1854.699,    -3037.667,
     -305.447,      -5567.5,    -2161.998,    -7761.208,    -4205.931,       -10042,        -6411,       -12233,        -8413,            0,
         7117,         5142,         4093,         2350,     1010.182,     -748.422,    -1954.258,    -3092.928,    -4033.369,    -5116.697,
    -6262.448,        -7251,        -8334,    -9424.099,       -10539,            0,         4397,         6065,     1502.556,         2930,
    -1679.066,      256.981,    -4510.495,    -1703.599,    -6966.644,    -3503.716,    -9095.307,    -5804.205,       -11250,        -7704,
            0,         6194,         5507,         3200,     2125.585,      351.279,     -909.652,        -2293,    -3360.421,     -4426.46,
    -5372.662,    -6531.921,    -7664.495,    -8640.233,        -9706,       -10586,            0,         7322,         2581,         3870,
     -313.992,      968.344,    -3458.759,    -1056.344,     -6116.19,    -2844.301,    -8375.611,    -4989.974,       -10608,        -7014,
       -12693,            0,         6614,         4410,         3200,     1490.734,       96.541,    -1478.595,    -2402.196,    -3711.072,
    -4634.183,     -5835.81,    -6957.161,    -7862.948,    -8959.575,    -9905.823,       -11067,            0,         3670,         5090,
      890.573,     1880.291,    -2518.946,      -337.79,    -5073.567,    -2232.791,    -7573.942,    -4285.191,        -9718,        -6419,
       -11819,        -8195,            0,         5319,         4555,         2605,     1297.403,     -670.504,    -1473.994,    -3015.246,
    -3669.155,    -5173.518,     -6115.64,     -7166.29,    -8330.365,    -9121.772,       -10251,            0,         6446,         1920,
         3080,    -1374.318,      271.346,    -4105.859,    -1513.678,    -6553.992,    -3676.796,    -9127.563,    -5549.583,       -11124,
        -7548,            0,         5850,         3664,         2385,      470.061,     -686.849,    -2075.199,    -2775.852,    -4344.488,
    -5183.611,    -6676.074,    -7738.532,    -8247.033,    -9453.885,            0,         2874,         4120,     -106.774,      1190.15,
    -3212.134,     -723.771,    -5578.718,    -2964.945,    -8236.912,     -4933.13,       -10391,        -6758,       -12329,            0,
         4667,         3520,     1399.818,      500.589,        -1166,    -2021.897,    -3432.555,     -4319.64,    -5902.109,    -6676.932,
    -7820.289,    -8769.318,     -9452.92,            0,         5583,      644.707,     2101.257,        -1937,        -91.3,    -4762.509,
    -2107.264,    -7294.305,    -4253.877,    -9671.885,    -6009.276,       -11563,        -8321,            0,         4816,         2648,
     1407.766,     -105.622,    -1062.753,    -2717.349,     -3566.11,    -4942.739,    -5813.583,    -7311.677,    -8030.455,    -8621.183,
       -10303,            0,         2281,         3103,     -852.203,      708.282,    -3804.804,    -1480.814,    -6381.333,    -3541.568,
    -8839.549,    -5351.685,       -10842,        -7464,            0,         3894,         2672,     1029.751,     -187.682,    -1742.516,
    -2958.502,    -4081.152,    -5097.124,     -6503.47,    -7210.009,    -7860.073,    -9656.343,            0,         4179,      374.691,
     1814.264,        -2800,     -838.444,    -5557.395,    -2882.378,    -7868.533,    -4724.622,    -10225.14,    -6524.872,            0,
         3763,         2010,     1070.906,         -556,    -2148.269,    -3465.489,    -4424.825,    -5585.547,    -6387.075,    -7212.522,
    -9018.564,            0,         5089,         1340,         2866,    -1480.673,       29.515,    -4645.294,    -2278.668,    -7013.473,
    -3969.587,    -9464.046,    -5839.738,       -12092,            0,         3037,       1993.5,      432.476,    -1012.797,    -2473.801,
    -3652.388,    -4816.207,    -5691.125,    -6420.061,    -8214.509,        -9329,            0,         2178,         3901,      -579.35,
     1069.286,    -3826.527,    -1308.478,    -6149.622,    -3175.544,    -8349.176,    -5508.804,   -11512.037,            0,         3782,
         3139,     1310.945,        2.469,    -1502.409,      -3002.8,    -3707.851,    -4887.264,    -5674.346,     -7463.57,    -8606.506,
            0,         2934,         4854,          349,     2120.276,    -2808.355,     -505.122,    -5522.148,    -2099.018,     -7855.04,
    -4531.305,   -10610.477,    -6666.603,            0,         3649,         2500,     1007.445,     -532.316,    -1969.877,    -2901.232,
    -3951.163,    -5028.571,    -6724.056,    -7817.109,    -8645.708,            0,         5634,         1270,         3140,    -1955.132,
      572.229,        -4442,    -1510.986,    -7037.157,    -3916.344,    -9513.918,      -6339.8,            0,         3237,     2045.117,
      312.677,    -1008.449,    -1888.664,    -3216.337,    -4311.931,    -6035.006,    -7005.878,    -8186.311,            0,         2059,
         4172,      -1047.3,     1459.658,    -3516.341,     -765.105,    -6139.172,    -3316.278,     -9010.14,    -5474.572,            0,
         3091,     1141.204,      -56.794,    -1082.689,    -2343.365,    -3732.104,    -5124.367,    -6321.619,    -7512.969,    -8213.762,
            0,         4879,         96.6,      2233.84,    -2501.059,      -69.079,    -5365.955,    -2619.427,    -8217.537,    -4985.027,
    -9817.461,            0,         2000,     1107.023,     -226.824,    -1570.008,    -2844.989,    -4441.098,    -5690.205,    -6948.938,
    -7598.541,    -8541.426,            0,     1157.652,     3209.016,    -1507.429,      686.664,    -4330.052,    -2135.877,    -7351.723,
    -4534.579,    -9551.071,    -5893.698,            0,     2154.642,      718.666,     -600.112,    -2199.819,    -3592.411,    -5061.115,
    -6329.665,    -7013.763,    -7820.019,            0,         4087,     -325.569,     1372.343,        -3460,    -1444.248,    -6680.712,
    -3896.082,    -8801.302,    -5441.286,            0,     2991.483,     1702.679,      452.018,    -1487.659,    -2831.416,    -4429.544,
     -5583.47,    -6395.816,     -7301.09,    -8278.979,            0,      666.046,     2235.253,     -2456.04,     -804.813,    -5873.213,
    -3415.579,    -7966.751,    -4981.664,   -10128.311,            0,         2660,     1262.148,     -481.231,    -1924.113,    -3841.971,
    -4891.021,    -5735.427,    -6717.316,    -7668.554,            0,         1802,     2946.154,    -1362.587,      -49.672,    -5200.709,
    -2808.656,    -7333.369,     -4315.85,    -9416.804,    -6071.328,            0,     2125.724,      492.074,     -974.622,    -3246.704,
    -4232.803,    -5143.599,    -6003.202,    -7021.565,    -7775.154,            0,         3940,      -344.27,      763.762,    -4442.431,
    -2333.752,    -6458.239,    -3891.235,    -8592.534,    -5445.946,            0,         3537,     1533.095,        -50.5,    -2708.421,
    -3552.677,    -4537.457,    -5257.647,    -6404.172,    -7148.854,            0,     1307.582,     1532.346,    -3757.508,    -1846.193,
    -5762.163,    -3304.073,    -7872.952,     -4807.63,    -9946.224,            0,         4815,     1418.239,    -2397.471,    -2908.585,
    -3903.266,    -4611.569,    -5789.412,    -6379.514,    -7593.207,            0,         3653,     4998.983,    -2878.433,    -1400.506,
     -4978.16,     -2843.38,     -6982.77,      -4379.1,    -9046.307,            0,         5292,     3976.391,      644.022,    -1892.517,
    -3486.311,    -3951.024,    -5159.371,    -5624.191,    -6989.457,    -7657.643,            0,         4132,     5481.994,       63.486,
     1161.292,     -3981.24,    -2373.918,    -6251.743,    -3807.239,    -8328.495,    -5253.027,            0,         4415,     1366.971,
      574.086,     -785.358,    -2891.592,    -4597.874,    -4994.152,    -6368.483,    -6700.775,            0,         5896,      569.906,
     2252.125,     -1748.23,       38.416,     -5122.02,    -3346.165,    -7469.951,    -4812.532,    -9523.456,            0,     2046.336,
     1422.751,      -73.929,     -881.214,     -2148.41,    -3907.504,    -5797.324,    -5963.888,    -7544.357,            0,     1018.932,
      3269.72,    -1090.205,      940.044,    -3361.381,    -1058.875,    -6328.481,    -4282.983,    -8773.413,            0,         2829,
     2188.814,      714.845,      -86.548,    -1486.677,    -2215.407,    -3478.005,    -4915.219,    -6944.786,            0,     4090.104,
     -473.406,     2001.676,    -2723.335,     -312.093,    -4831.697,    -2181.596,     -7496.15,            0,         2920,      1505.27,
      736.949,     -656.028,    -1572.712,    -2819.248,    -3509.323,    -4852.765,      -5630.7,            0,         4977,      259.609,
     2881.185,    -1841.625,       408.08,    -4192.862,    -1530.488,    -6294.468,    -3254.437,            0,         2405,     1566.295,
      212.432,     -775.869,    -2175.328,    -2903.007,    -4048.505,    -4691.019,            0,         1113,     3738.738,     -869.479,
     1210.031,    -3478.753,      -917.32,    -5707.768,        -2652,            0,         2341,     1194.194,      314.309,    -1559.237,
    -2414.829,    -3441.227,        -4212,            0,         4427,       24.226,     2028.049,    -2300.158,     -581.504,        -4913,
        -2184,            0,         3167,         1913,     1149.171,     -591.776,    -1559.302,    -2934.975,    -3517.627,            0,
          783,         2830,     -1407.53,      238.435,    -3873.937,    -1843.463,            0,         2678,         1820,      355.811,
     -672.013,    -2030.337,    -3036.707,    -4213.349,            0,         1401,         3704,     -641.115,     1113.153,    -2836.105,
    -1295.661,        -5409,            0,         3326,         2476,     1328.045,       44.765,    -1025.577,    -2190.217,     -3540.07,
            0,         2102,         4340,       45.811,     2123.793,    -2151.983,     -300.528,        -4477,        -2387,            0,
     3253.383,     1809.272,     1166.988,     -311.457,    -1312.611,    -2568.939,    -3620.162,            0,         5080,          710,
     2943.833,    -1310.529,        559.8,    -3621.475,    -1697.451,            0,         3931,         2479,         2100,      391.556,
     -381.646,      -1817.7,    -2660.367,        -4153,            0,         5714,         1500,         3700,      -499.53,     1337.103,
        -2750,        -1005,        -5033,            0,         3269,         2765,      1243.14,       570.14,    -1029.617,    -2102.222,
        -3121,        -4120,            0,         2128,         4489,      273.088,     2194.572,    -1809.846,     -393.126,        -4184,
        -2190,            0,         3467,     1924.894,         1410,     -124.214,    -1139.015,        -2478,        -3249,        -4794,
            0,         5054,         1107,         2900,     -932.978,      476.585,        -3280,        -1708,        -5512,            0,
         2560,         2250,        518.6,     -220.032,        -1477,        -2706,        -3821,        -4720,            0,         1856,
         3460,     -147.324,     1291.527,    -2258.342,     -972.827,        -4893,        -2915,            0,         2763,     1261.498,
      722.508,     -802.108,        -1800,        -3095,        -3859,            0,         4087,      400.361,     2187.741,     -1384.79,
     -213.648,        -3940,        -2369,        -6164,            0,         1935,         1305,       20.783,     -767.417,        -2400,
        -3257,        -4482,            0,         1202,         2700,     -751.295,      664.467,        -2930,        -1602,        -5629,
        -3433,            0,         2120,      579.401,       -7.484,    -1507.583,        -2254,        -3839,        -4476,            0,
         3396,      -75.396,       1427.3,     -2261.85,     -763.746,        -4547,        -2982,            0,     1206.322,       895.04,
     -810.742,    -1571.427,        -3051,        -3783,        -5071,            0,       400.55,       2376.2,        -1350,     -123.294,
        -3810,        -2239,        -6136,            0,         1842,         1620,       43.277,         -646,        -2473,        -2973,
        -4460,            0,         3170,         -687,          840,        -3061,        -1606,        -5242,        -3515,            0,
         2354,      900.528,          124,        -1451,        -2443,        -3706,        -4490,            0,       37.668,     1779.577,
        -2055,         -847,        -4563,        -2879,            0,         1420,         1093,     -377.585,    -1474.438,        -3040,
        -3887,        -4982,            0,          522,         2500,        -1260,      476.605,        -3813,        -2251,        -5956,
            0,         1628,      287.318,      -335.84,        -1951,        -3165,        -4222,        -5103,            0,         3053,
     -651.218,       1087.8,        -2610,        -1210,        -5122,        -3474,            0,          720,       289.62,    -1043.742,
    -2010.561,        -3204,        -4340,        -5644,            0,         -147,         1700,    -2129.309,     -208.321,        -4049,
        -2363,        -6484,            0,          814,     -406.732,     -1244.31,        -2494,        -3199,        -4408,            0,
         2277,        -1263,          209,        -3360,        -1560,        -5348,        -3668,            0,           80,         -486,
        -1742,        -2548,        -3701,        -4555,            0,         -907,          940,        -2665,         -873,        -4527,
        -2908,        -7031,            0,          -26,        -1057,        -1754,        -3064,        -3783,        -5167,            0,
         1457,        -2173,         -271,        -3875,        -2155,        -6049,            0,         -690,        -1168,        -2274,
        -3105,        -4390,        -5145,            0,        -1577,           50,        -3185,        -1423,        -5284,        -3531,
            0,         -806,        -1767,        -2342,        -3756,        -4599,        -5651,            0,         1257,        -2863,
         -965,        -4543,        -2939,        -6709,            0,         -790,        -1908,        -3004,        -3855,        -5140,
        -6552,            0,        -1677,         -150,        -3865,        -2243,        -6116,        -4719,            0,        -1206,
        -1807,        -3131,        -4657,        -5654,            0,          357,        -3063,         -963,        -5595,        -3785,
            0,        -1590,        -2308,        -3244,        -4585,            0,        -2677,         -950,        -4365,        -2403,
        -6798,            0,        -1302,        -2206,        -2727,        -3679,        -4489,            0,         -643,        -4063,
        -1863,        -5795,            0,        -1585,        -2510,        -3288,        -3694,            0,        -3677,        -1750,
        -5085,            0,        -2402,        -3006,        -3607,        -4121,            0,        -1543,        -4783,        -2523,
            0,        -2485,        -3360,        -3798,            0,        -4357,        -2390,            0,        -3084,        -3646,
            0,        -2123,            0,        -3145,        -4340,            0,        -5306,            0,        -4311,        -4626,
            0,        -3143,            0,        -4241,        -5204,            0,        -6338,            0,        -5064,        -5726,
            0,        -4143,            0,        -5241,        -6104,            0,        -7231,            0,            0
};

// Reduced Table of A for shorter index search.
//         The index in this table coincide with A-1
//         For each A value shortTable[A-1] has the index of 
//         the 1st occurrence in the indexArray[][]
const G4int G4NucleiPropertiesTableAME03::shortTable[G4NucleiPropertiesTableAME03::MaxA+1]=
{
    0,    2,    3,    6,    9,   13,   18,   23,   28,   33,
   39,   44,   50,   55,   61,   67,   74,   80,   87,   95,
  102,  110,  119,  127,  136,  145,  154,  163,  173,  182,
  192,  202,  212,  222,  233,  243,  254,  265,  276,  287,
  299,  310,  322,  333,  345,  357,  369,  381,  394,  407,
  419,  431,  443,  455,  467,  479,  491,  503,  515,  527,
  540,  552,  564,  576,  587,  599,  610,  622,  633,  645,
  656,  668,  680,  692,  704,  716,  728,  740,  753,  765,
  777,  789,  801,  814,  826,  839,  852,  865,  878,  892,
  905,  919,  933,  947,  961,  975,  989, 1004, 1018, 1033,
 1048, 1062, 1076, 1090, 1104, 1119, 1133, 1147, 1162, 1176,
 1191, 1205, 1220, 1235, 1250, 1265, 1279, 1294, 1309, 1324,
 1339, 1354, 1369, 1383, 1398, 1412, 1427, 1442, 1458, 1474,
 1491, 1507, 1523, 1538, 1554, 1570, 1586, 1602, 1618, 1634,
 1650, 1666, 1682, 1698, 1714, 1730, 1746, 1762, 1778, 1794,
 1811, 1828, 1844, 1861, 1877, 1894, 1910, 1926, 1942, 1958,
 1974, 1990, 2006, 2022, 2038, 2054, 2070, 2086, 2101, 2117,
 2132, 2148, 2163, 2178, 2192, 2206, 2220, 2234, 2248, 2262,
 2275, 2288, 2300, 2312, 2325, 2337, 2349, 2361, 2374, 2386,
 2398, 2409, 2420, 2431, 2442, 2453, 2464, 2474, 2484, 2495,
 2505, 2515, 2526, 2536, 2546, 2556, 2566, 2576, 2586, 2597,
 2608, 2618, 2629, 2639, 2649, 2659, 2668, 2678, 2688, 2697,
 2706, 2714, 2722, 2730, 2737, 2745, 2753, 2761, 2770, 2778,
 2786, 2795, 2804, 2813, 2822, 2831, 2840, 2849, 2858, 2866,
 2875, 2883, 2892, 2900, 2908, 2916, 2924, 2932, 2940, 2948,
 2955, 2963, 2971, 2979, 2987, 2995, 3003, 3010, 3018, 3025,
 3033, 3040, 3047, 3054, 3061, 3068, 3075, 3082, 3089, 3095,
 3101, 3106, 3112, 3118, 3123, 3128, 3132, 3137, 3141, 3145,
 3148, 3151, 3153, 3156, 3158, 3161, 3163, 3166, 3168, 3171,
 3173, 3176, 3178
};

