//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// $Id: G4VAnalysisManager.cc 70604 2013-06-03 11:27:06Z ihrivnac $

// Author: Ivana Hrivnacova, 09/07/2013  (ivana@ipno.in2p3.fr)

#include "G4VAnalysisManager.hh"
#include "G4AnalysisMessenger.hh"
#include "G4AnalysisUtilities.hh"
#include "G4HnManager.hh"
#include "G4VH1Manager.hh"
#include "G4VH2Manager.hh"
#include "G4VNtupleManager.hh"
#include "G4VFileManager.hh"

#include <iostream>

using namespace G4Analysis;

//_____________________________________________________________________________
G4VAnalysisManager::G4VAnalysisManager(const G4String& type, G4bool isMaster)
 : fState(type, isMaster),
   fMessenger(0),
   fH1HnManager(0),
   fH2HnManager(0),
   fVH1Manager(0),
   fVH2Manager(0),
   fVNtupleManager(0),
   fVFileManager(0)
{
  fMessenger = new G4AnalysisMessenger(this);
}

//_____________________________________________________________________________
G4VAnalysisManager::~G4VAnalysisManager()
{
  delete fMessenger;
  delete fVH1Manager;
  delete fVH2Manager;
  delete fVNtupleManager;
  delete fVFileManager;
}

// 
// protected methods
//
    
//_____________________________________________________________________________
void G4VAnalysisManager::SetH1Manager(G4VH1Manager* h1Manager)
{
  fVH1Manager = h1Manager;
  fH1HnManager = h1Manager->fHnManager;
  fMessenger->SetH1HnManager(fH1HnManager);
} 

//_____________________________________________________________________________
void G4VAnalysisManager::SetH2Manager(G4VH2Manager* h2Manager)
{
  fVH2Manager = h2Manager;
  fH2HnManager = h2Manager->fHnManager;
  fMessenger->SetH2HnManager(fH2HnManager);
}  

//_____________________________________________________________________________
void G4VAnalysisManager::SetNtupleManager(G4VNtupleManager* ntupleManager)
{
  fVNtupleManager = ntupleManager;
}  

//_____________________________________________________________________________
void G4VAnalysisManager::SetFileManager(G4VFileManager* fileManager)
{
  fVFileManager = fileManager;
}  

//_____________________________________________________________________________
G4bool G4VAnalysisManager::WriteAscii(const G4String& fileName)
{
  G4bool finalResult = true;

  // Replace or add file extension .ascii
  G4String name(fileName);
  if ( name.find(".") != std::string::npos ) { 
    name.erase(name.find("."), name.length()); 
  }
  name.append(".ascii");

#ifdef G4VERBOSE
  if ( fState.GetVerboseL3() ) 
    fState.GetVerboseL3()->Message("write ASCII", "file", name);
#endif
     
  std::ofstream output(name, std::ios::out);
  if ( ! output ) {
    G4ExceptionDescription description;
    description 
      << "Cannot open file. File name is not defined.";
    G4Exception("G4VAnalysisManager::WriteAscii()",
                "Analysis_W009", JustWarning, description);
    return false;
  }
  output.setf( std::ios::scientific, std::ios::floatfield );

  G4bool result = fVH1Manager->WriteOnAscii(output);
  finalResult = finalResult && result;
  
  result = fVH2Manager->WriteOnAscii(output);
  finalResult = finalResult && result;  

#ifdef G4VERBOSE
    if ( fState.GetVerboseL1() ) 
      fState.GetVerboseL1()->Message("write ASCII", "file",  name, result);
#endif
  
  return finalResult;
}     

// 
// public methods
//

//_____________________________________________________________________________
G4bool G4VAnalysisManager::OpenFile()
{
  if ( fVFileManager->GetFileName() == "" ) {
    G4ExceptionDescription description;
    description 
      << "Cannot open file. File name is not defined.";
    G4Exception("G4VFileManager::OpenFile()",
                "Analysis_W009", JustWarning, description);
    return false;
  }           
  
  return OpenFileImpl(fVFileManager->GetFileName());
} 

//_____________________________________________________________________________
G4bool G4VAnalysisManager::OpenFile(const G4String& fileName)
{
  return OpenFileImpl(fileName);
}  

//_____________________________________________________________________________
G4bool G4VAnalysisManager::Write()
{
  return WriteImpl();
}  

//_____________________________________________________________________________
G4bool G4VAnalysisManager::CloseFile()
{
  return CloseFileImpl();
}  

//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetFileName(const G4String& fileName)
{ 
  return fVFileManager->SetFileName(fileName); 
}

//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetHistoDirectoryName(const G4String& dirName)
{ 
  return fVFileManager->SetHistoDirectoryName(dirName); 
}

//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetNtupleDirectoryName(const G4String& dirName)
{ 
  return fVFileManager->SetNtupleDirectoryName(dirName); 
}

//_____________________________________________________________________________
G4String G4VAnalysisManager::GetFileName() const 
{  
  return fVFileManager->GetFileName(); 
}

//_____________________________________________________________________________
G4String G4VAnalysisManager::GetHistoDirectoryName() const 
{  
  return fVFileManager->GetHistoDirectoryName(); 
}
 
//_____________________________________________________________________________
G4String G4VAnalysisManager::GetNtupleDirectoryName() const
{
  return fVFileManager->GetNtupleDirectoryName(); 
}

//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateH1(const G4String& name,  const G4String& title,
                               G4int nbins, G4double xmin, G4double xmax,
                               const G4String& unitName, const G4String& fcnName,
                               const G4String& binSchemeName)
{
  if ( ! CheckName(name, "H1") ) return kInvalidId;
  if ( ! CheckNbins(nbins) ) return kInvalidId;
  if ( ! CheckMinMax(xmin, xmax, binSchemeName) ) return kInvalidId;

  return fVH1Manager->CreateH1(name, title, nbins, xmin, xmax, 
                               unitName, fcnName, binSchemeName);
}                                         

//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateH1(const G4String& name,  const G4String& title,
                               const std::vector<G4double>& edges,
                               const G4String& unitName, const G4String& fcnName)
{
  if ( ! CheckName(name, "H1") ) return kInvalidId;
  if ( ! CheckEdges(edges) ) return kInvalidId;

  return fVH1Manager->CreateH1(name, title, edges, unitName, fcnName);
}                                         

//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateH2(const G4String& name,  const G4String& title,
                               G4int nxbins, G4double xmin, G4double xmax,
                               G4int nybins, G4double ymin, G4double ymax,
                               const G4String& xunitName, const G4String& yunitName,
                               const G4String& xfcnName, const G4String& yfcnName,
                               const G4String& xbinSchemeName, 
                               const G4String& ybinSchemeName)
                               
{
  if ( ! CheckName(name, "H2") ) return kInvalidId;
  
  if ( ! CheckNbins(nxbins) ) return kInvalidId;
  if ( ! CheckMinMax(xmin, xmax, xbinSchemeName) ) return kInvalidId;

  if ( ! CheckNbins(nybins) ) return kInvalidId;
  if ( ! CheckMinMax(ymin, ymax, ybinSchemeName) ) return kInvalidId;

  return fVH2Manager->CreateH2(name, title, 
                               nxbins, xmin, xmax, nybins, ymin, ymax, 
                               xunitName, yunitName, xfcnName, yfcnName, 
                               xbinSchemeName, ybinSchemeName);
}                                         

//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateH2(const G4String& name,  const G4String& title,
                               const std::vector<G4double>& xedges,
                               const std::vector<G4double>& yedges,
                               const G4String& xunitName, const G4String& yunitName,
                               const G4String& xfcnName, const G4String& yfcnName)
                               
{
  if ( ! CheckName(name, "H2") ) return kInvalidId;
  
  if ( ! CheckEdges(xedges) ) return kInvalidId;
  if ( ! CheckEdges(yedges) ) return kInvalidId;

  return fVH2Manager->CreateH2(name, title, 
                               xedges, yedges,
                               xunitName, yunitName, xfcnName, yfcnName);
}                                         

//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetH1(G4int id,
                                G4int nbins, G4double xmin, G4double xmax,
                                const G4String& unitName, const G4String& fcnName,
                                const G4String& binSchemeName)
{                                
  if ( ! CheckNbins(nbins) ) return kInvalidId;
  if ( ! CheckMinMax(xmin, xmax, binSchemeName) ) return kInvalidId;

  return fVH1Manager->SetH1(id, nbins, xmin, xmax, unitName, fcnName, binSchemeName); 
}
  
//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetH1(G4int id,
                                const std::vector<G4double>& edges,
                                const G4String& unitName, const G4String& fcnName)
{                                
  if ( ! CheckEdges(edges) ) return kInvalidId;

  return fVH1Manager->SetH1(id, edges, unitName, fcnName); 
}
  
//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetH2(G4int id,
                                G4int nxbins, G4double xmin, G4double xmax, 
                                G4int nybins, G4double ymin, G4double ymax,
                                const G4String& xunitName, const G4String& yunitName,
                                const G4String& xfcnName, const G4String& yfcnName,
                                const G4String& xbinSchemeName, 
                                const G4String& ybinSchemeName)
{                                
  if ( ! CheckNbins(nxbins) ) return kInvalidId;
  if ( ! CheckMinMax(xmin, xmax, xbinSchemeName) ) return kInvalidId;

  if ( ! CheckNbins(nybins) ) return kInvalidId;
  if ( ! CheckMinMax(ymin, ymax, ybinSchemeName) ) return kInvalidId;

  return fVH2Manager->SetH2(id, nxbins, xmin, xmax, nybins, ymin, ymax, 
                            xunitName, yunitName, xfcnName, yfcnName,
                            xbinSchemeName, ybinSchemeName);
}
                                  
//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetH2(G4int id,
                                const std::vector<G4double>& xedges,
                                const std::vector<G4double>& yedges,
                                const G4String& xunitName, const G4String& yunitName,
                                const G4String& xfcnName, const G4String& yfcnName)
{                                
  if ( ! CheckEdges(xedges) ) return kInvalidId;
  if ( ! CheckEdges(yedges) ) return kInvalidId;

  return fVH2Manager->SetH2(id, xedges, yedges, 
                            xunitName, yunitName, xfcnName, yfcnName);
}
                                  
//_____________________________________________________________________________
G4bool G4VAnalysisManager::ScaleH1(G4int id, G4double factor)
{
  return fVH1Manager->ScaleH1(id, factor);
}  

//_____________________________________________________________________________
G4bool G4VAnalysisManager::ScaleH2(G4int id, G4double factor)
{
  return fVH2Manager->ScaleH2(id, factor);
}  
                           
//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateNtuple(const G4String& name, 
                                          const G4String& title)
{
  if ( ! CheckName(name, "Ntuple") ) return kInvalidId;

  return fVNtupleManager->CreateNtuple(name, title);
}                                         

//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateNtupleIColumn(const G4String& name)
{
  if ( ! CheckName(name, "NtupleIColumn") ) return kInvalidId;

  return fVNtupleManager->CreateNtupleIColumn(name);
}  

//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateNtupleFColumn(const G4String& name)
{
  if ( ! CheckName(name, "NtupleFColumn") ) return kInvalidId;

  return fVNtupleManager->CreateNtupleFColumn(name);
}  

//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateNtupleDColumn(const G4String& name)
{
  if ( ! CheckName(name, "NtupleDColumn") ) return kInvalidId;

  return fVNtupleManager->CreateNtupleDColumn(name);
}  

//_____________________________________________________________________________
void G4VAnalysisManager::FinishNtuple()
{ 
  return fVNtupleManager->FinishNtuple();
}
   
//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateNtupleIColumn(G4int ntupleId, 
                                                 const G4String& name)
{
  if ( ! CheckName(name, "NtupleIColumn") ) return kInvalidId;

  return fVNtupleManager->CreateNtupleIColumn(ntupleId, name);
}                                         

//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateNtupleFColumn(G4int ntupleId, const G4String& name)
{
  if ( ! CheckName(name, "NtupleFColumn") ) return kInvalidId;

  return fVNtupleManager->CreateNtupleFColumn(ntupleId, name);
}                                         


//_____________________________________________________________________________
G4int G4VAnalysisManager::CreateNtupleDColumn(G4int ntupleId, const G4String& name)   
{
  if ( ! CheckName(name, "NtupleDColumn") ) return kInvalidId;

  return fVNtupleManager->CreateNtupleDColumn(ntupleId, name);
}                                         

//_____________________________________________________________________________
void G4VAnalysisManager::FinishNtuple(G4int ntupleId)
{ 
  return fVNtupleManager->FinishNtuple(ntupleId);
}
   
//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetFirstHistoId(G4int firstId) 
{
  G4bool finalResult = true;
  G4bool result = fVH1Manager->SetFirstId(firstId);
  finalResult = finalResult && result;
  
  result = fH1HnManager->SetFirstId(firstId);
  finalResult = finalResult && result;
  
  result = fVH2Manager->SetFirstId(firstId);
  finalResult = finalResult && result;

  result = fH2HnManager->SetFirstId(firstId);
  finalResult = finalResult && result;
   
  return finalResult; 
}  

//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetFirstH1Id(G4int firstId) 
{
  G4bool finalResult = true;
  G4bool result = fVH1Manager->SetFirstId(firstId);
  finalResult = finalResult && result;
  
  result = fH1HnManager->SetFirstId(firstId);
  finalResult = finalResult && result;
  
  return finalResult; 
}  

//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetFirstH2Id(G4int firstId) 
{
  G4bool finalResult = true;
  G4bool result = fVH2Manager->SetFirstId(firstId);
  finalResult = finalResult && result;

  result = fH2HnManager->SetFirstId(firstId);
  finalResult = finalResult && result;
   
  return finalResult; 
}  

//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetFirstNtupleId(G4int firstId) 
{
  return fVNtupleManager->SetFirstId(firstId);
}  

//_____________________________________________________________________________
G4bool G4VAnalysisManager::SetFirstNtupleColumnId(G4int firstId) 
{
  return fVNtupleManager->SetFirstNtupleColumnId(firstId);
}

// Fill methods in .icc

//_____________________________________________________________________________
void  G4VAnalysisManager::SetActivation(G4bool activation) 
{
  fState.SetIsActivation(activation);
}

// GetActivation() in .icc

//_____________________________________________________________________________
G4bool G4VAnalysisManager::IsActive() const
{
// Return true if activation option is selected and any of managers has 
// an activated object.

  return fState.GetIsActivation() && 
         ( fH1HnManager->IsActive() || fH2HnManager->IsActive() );
}  


//_____________________________________________________________________________
G4bool G4VAnalysisManager::IsAscii() const
{
// Return true any of managers has an object with activated ASCII option.

  return ( fH1HnManager->IsAscii() || fH2HnManager->IsAscii() );
}  

//_____________________________________________________________________________
G4int G4VAnalysisManager::GetNofH1s() const
{
  return fH1HnManager->GetNofHns();
}  

//_____________________________________________________________________________
G4int G4VAnalysisManager::GetNofH2s() const
{
  return fH2HnManager->GetNofHns();
}  

//_____________________________________________________________________________
G4int G4VAnalysisManager::GetNofNtuples() const
{
  return fVNtupleManager->GetNofNtuples();
}  

// GetH1Id(), GetH2Id in .icc 

//_____________________________________________________________________________
void  G4VAnalysisManager::SetH1Activation(G4int id, G4bool activation)
{
// Set activation to a given H1 object

  fH1HnManager->SetActivation(id, activation);
}    

//_____________________________________________________________________________
void  G4VAnalysisManager::SetH1Activation(G4bool activation)
{
// Set activation to all H1 objects

  fH1HnManager->SetActivation(activation);
}    

//_____________________________________________________________________________
void  G4VAnalysisManager::SetH1Ascii(G4int id, G4bool ascii)
{
  fH1HnManager->SetAscii(id, ascii);
}    

//_____________________________________________________________________________
void  G4VAnalysisManager::SetH2Activation(G4int id, G4bool activation)
{
// Set activation to a given H2 object

  fH2HnManager->SetActivation(id, activation);
}    

//_____________________________________________________________________________
void  G4VAnalysisManager::SetH2Activation(G4bool activation)
{
// Set activation to all H2 objects

  fH2HnManager->SetActivation(activation);
}    

//_____________________________________________________________________________
void  G4VAnalysisManager::SetH2Ascii(G4int id, G4bool ascii)
{
  fH2HnManager->SetAscii(id, ascii);
}    

// Access methods in .icc

//_____________________________________________________________________________
void G4VAnalysisManager::SetVerboseLevel(G4int verboseLevel) 
{
  fState.SetVerboseLevel(verboseLevel);
} 

// GetVerboseLevel() in .icc 

