/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//  Table manipulation utility
//
#include <iostream>
#include <sstream>
#include <fstream>
#include <cstdlib>
#include <unistd.h>
#include "table.hh"
#include "eval.hh"
#include "ParseLine.hh"
#include "VArg.hh"
//
#include "TCanvas.h"
#include "TH1.h"
#include "TH2.h"
#include "TPolyMarker.h"
#include "TStyle.h"
#include <sys/times.h>
#include <limits.h>
#include <cmath>

using namespace table;
using namespace std;

void help(ostream& out);
void exec(istream& in, Table& t, eval::dict_type& d);

int debug(0);
int verbose(0);
TCanvas* Canvas=0;

int 
main(int argc, const char* argv[]) {
    if (argc < 2) {
        help(cerr);
	return 1;
    }
    Table t;
    eval::dict_type d;
    bool title(true), body(true), sum(false), syntax (false);
    string tablename, cmdfile, command, ofile("-"), ftype, layoutcmd;
    for (int i=1; i<argc; i++) {
        string argi(argv[i]);
	if (argi == "-b") {
	    body = false;
	} 
	else if (argi == "-s") {
	    sum = true;
	} 
	else if (argi == "-h") {
	    title = false;
	} 
	else if (argi == "--help") {
	    help(cout);
	    return 0;
	} 
	else if (argi == "-html") {
	    ftype = "html";
	} 
	else if (argi == "-i") {
	    if (++i < argc) cmdfile = argv[i];
	    else            syntax = true;
	} 
	else if (argi == "-c") {
	    if (++i < argc) {
	        if (!command.empty() && command[command.size()-1] != ';') {
		    command += ";";
		}
	        command += argv[i];
	    } else {
	        syntax = true;
	    }
	}

	//------------------------------  Layout (-l) command file
	else if (argi == "-l") {
	    if (++i < argc) layoutcmd = argv[i];
	    else            syntax    = true;
	}

	//------------------------------  Get the output file name
	else if (argi == "-o") {
	    if (++i < argc) ofile = argv[i];
	    else            syntax = true;
	} 

	//------------------------------  Get the tablename
	else if (argi == "-t") {
	    if (++i < argc) tablename = argv[i];
	    else            syntax = true;
	}
	else if (argi == "-v") {
	    verbose = 1;
	} 
	else if (argi == "-xml") {
	    ftype = "xml";
	} 
	else if (argi == "--debug") {
	    if (++i < argc) debug = strtol(argv[i], 0, 0);
	    else            debug = 1;
	    t.setDebug(debug);
	    verbose = 1;
	} 
	else if (argi == "--version") {
	    cout << "DMT table manipulation utility - $Id: tablepgm.cc 7596 2016-04-15 06:51:07Z john.zweizig@LIGO.ORG $" << endl;
	    return 0;
	} 
	else if (argi[0] == '-' && argi != "-") {
            cerr << "Undefined keyword: " << argi << endl;
	    syntax = true;
	    break;
	} 
	else {
	    string thisfile = argi;
	    string thisname = tablename;
	    cerr << "Reading file: " << thisfile << endl;
	    try {
	        Table thistable(thisname, "");
		thistable.setDebug(debug);
		if (!layoutcmd.empty()) {
		    if (debug) cout << "execfrom(\"" << layoutcmd << "\")" 
				    << endl;
		    ifstream lcmd(layoutcmd.c_str());
		    exec(lcmd, thistable, d);
		}
		thistable.read(thisfile);
		if (thistable.defined()) {
		    if (debug) cerr << "New table defined with " 
				    << thistable.getNRows() 
				    << " rows." << endl;
		    if (thisname.empty()) thisname = thisfile;
		    if (!t.size()) t=thistable;
		    else           t.insertRows(t.getNRows(), thistable);
		    if (debug) cerr << "Default table size is now: " 
				    << t.getNRows() << "x" << t.size() 
				    << endl;
		} else {
		    cerr << "Empty table file: " << thisfile << endl;
		}
	    } catch (std::exception& e) {
	        cerr << "Caught exception reading table " << thisfile
		     << ": " << e.what() << endl;
		return 1;
	    }
	}
    }
    if (syntax) {
        help(cerr);
	return 1;
    }

    //----------------------------------  Make the table
    while (!command.empty()) {
        string::size_type l = command.find(";");
	if (debug) cerr << "exec(\"" << command.substr(0, l) << "\")" << endl;
        istringstream in(command.substr(0, l));
	exec(in, t, d);
	if (l != string::npos) command.erase(0, l+1);
	else                   command.clear();
    }

    
    if (cmdfile == "-") {
        if (debug) cerr << "execfrom(cin)" << endl;
	exec(cin, t, d);
    } else if (!cmdfile.empty()) {
	if (debug) cerr << "execfrom(" << cmdfile << ")" << endl;
        ifstream in(cmdfile.c_str());
	exec(in, t, d);
    }

    //----------------------------------  Write the canvas if necessary
    if (Canvas) {
	Canvas->Print("tablepgm.ps");
	delete Canvas;
	Canvas = 0;
    }

    //----------------------------------  All done, write the table
    int flags = 0;
    if (title) flags += 1;
    if (body)  flags += 2;
    if (sum)   flags += 4;
    t.write(ofile ,flags, ftype); 
    return 0;
}

//======================================  remove blanks
void
deblank(string& s) {
    unsigned int i=0;
    while (i<s.size()) {
        if (s[i] == ' ') s.erase(i, 1);
        else if (s[i] == '\t') s.erase(i, 1);
	else i++;
    }
}

//======================================  Print user time
double tDelta(0.0);
void
usertime(void) {
    struct tms t;
    times(&t);
    double tNow(t.tms_utime/double(sysconf(_SC_CLK_TCK)));
    cerr<< "user time: " << tNow-tDelta <<  " (total: " << tNow << ")" << endl;
    tDelta = tNow;
}

//======================================  perform an operation on a column list
typedef std::vector<Column*> col_list;
void
ListColOper(const col_list& list1, const col_list& list2, 
	    Column::row_index i, Column::row_index j, const string& oper) 
{
    Column::row_index N=list1.size();
    for (Column::row_index k=0; k<N; ++k) {
	if (oper == "|") {
	    (*list1[k])[i] |= (*list2[k])[j];
	} else {
	    (*list1[k])[i] += (*list2[k])[j];
	}	
    }
}

//======================================  Execute a list of commands.
void 
exec(istream& in, Table& t, eval::dict_type& d) {
    ParseLine pl(in);
    if (verbose) pl.setLog(cerr);
    VArg va;
    while (pl.getLine() >= 0) {
        va.erase();
        try {
	    unsigned int nArg=pl.getCount();
	    string cmd(pl[0]);

	    //--------------------------  Add a column
	    if (cmd == "addcolumn") {
	        va << "-before" << "-eval" << "-extern" << "-title";
		va.scan(pl, 1);
		int inx(0);
		if (!va["-before"].empty()) inx = t.find(va["-before"]);
		else                        inx = t.size();
		t.insertColumn(inx, va["-title"]);
		Column& col(t[inx]);
		if (!va["-extern"].empty()) {
		    col.setXType(va["-extern"]);
		}
		if (!va["-eval"].empty()) {
		    eval evcmd(t, va["-eval"], debug);
		    evcmd.setDictionary(d);
		    int N = t.getNRows();
		    for (int i=0; i<N; i++) {
		        col.refCell(i) = Cell(evcmd.evalRow(i));
		    }
		}

	    //--------------------------  Add rows from another table
	    } else if (cmd == "addrows" || cmd == "insertrows") {
	        va << "-table";
		va.scan(pl, 1);
		if (!va["-table"].empty()) {
		    Table t2(va["-table"], va["-table"]);
		    t.insertRows(t.getNRows(), t2);
		}

	    //--------------------------  Execute a command file.
	    } else if (cmd == "exec") {
		if (nArg < 2) 
		    throw runtime_error("exec must have a filename argument.");
		va << "-cwd";
		va.scan(pl, 1, nArg-1);
		string wd = va["-cwd"];
		string macro = pl[nArg-1];
		if (macro[0] != '/' && !wd.empty()) macro = wd + "/" + macro;

		ifstream in1(macro.c_str());
		if (!in1.is_open()) {
		    throw runtime_error(string("Can't open: ") + macro);
		} else if (wd.empty()) {
		    exec(in1, t, d);
		} else {
		    char savewd[1024];
		    if (!getcwd(savewd, sizeof(savewd))) {
			throw runtime_error("Unable to save current directory");
		    }
		    try {
			if (!chdir(wd.c_str())) {
			    throw runtime_error("Unable to set directory");
			}
			exec(in1, t, d);
			if (!chdir(savewd)) {
			    cerr << "Default directory was lost" << endl;
			    exit(1);
			}
		    } catch (exception& e) {
			if (!chdir(savewd)) {
			    cerr << "Default directory was lost" <<endl;
			    exit(1);
			}
			throw;
		    }
		}

	    //--------------------------  Plot a histogram
	    } else if (cmd == "histogram") {
		int N = t.getNRows();
		if (!N) throw runtime_error("Empty table");

	        va << "-color" = "0";
	        va << "-eval";
		va << "-file" = "tablepgm.ps";
		va << "-fill" = "0";
		va << "-hold";
		va << "-logbin" = "false";
		va << "-nbin" = "100";
		va << "-title";
		va << "-rows";
		va << "-weight";
		va << "-xaxis";
		va << "-xlog" = "false";
		va << "-xmin" = "0";
		va << "-xmax" = "0";
		va << "-yaxis";
		va << "-ylog" = "false";
		va << "-ymin" = "0";
		va << "-ymax" = "0";
		va.scan(pl, 1);

		//----------------------  Get row selection.
		vector<bool> mask(N);
		int Nsel = 0;
		string select=va["-rows"];
		if (!select.empty()) {
		    eval selcmd(t, select, debug);
		    selcmd.setDictionary(d);
		    for (int i=0; i<N; ++i) {
			mask[i] = selcmd.evalRow_bool(i);
			if (mask[i]) Nsel++;
		    }
		} else {
		    for (int i=0; i<N; ++i) mask[i] = true;
		    Nsel = N;
		}
		if (!Nsel) throw runtime_error("No rows selected");

		//----------------------  Calculate the values & weights
		std::vector<double> val(Nsel);
		std::vector<double> wt(Nsel);

		eval evcmd(t, va["-eval"], debug);
		evcmd.setDictionary(d);
		string weight=va["-weight"];
		if (weight.empty()) {
		    int j=0;
		    for (int i=0; i<N; ++i) {
			if (mask[i]) {
			    val[j] = evcmd.evalRow_numeric(i);
			    wt[j]  = 1.0;
			    j++;
			}
		    }
		} else {
		    eval evalwt(t, weight, debug);
		    evalwt.setDictionary(d);
		    int j=0;
		    for (int i=0; i<N; ++i) {
			if (mask[i]) {
			    val[j] = evcmd.evalRow_numeric(i);
			    wt[j]  = evalwt.evalRow_numeric(i);
			    j++;
			}
		    }
		}

		//----------------------  Get the histogram limits
		double xmin = va.getDouble("-xmin");
		double xmax = va.getDouble("-xmax");
		if (xmax <= xmin) {
		    xmin = val[0];
		    xmax = val[0];
		    for (int i=1; i<Nsel; ++i) {
		        double x = val[i];
			if      (x < xmin) xmin = x;
			else if (x > xmax) xmax = x;
		    }
		}

		//----------------------  Book and fill the histogram
		string title = va["-title"];
		if (title.empty()) title = va["-eval"] + " histogram";
		long nbin = long(va.getDouble("-nbin"));
		TH1D histo("TblPlot", title.c_str(), nbin, xmin, xmax);
		if (va.getBool("-logbin")) {
		    double lbin = xmin;
		    double dbin = exp(log(xmax/xmin)/nbin);
		    double* bedge = new double[nbin+1];
		    for (int i=0; i<=nbin; i++) {
			bedge[i] = lbin;
			lbin *= dbin;
		    }
		    histo.SetBins(nbin, bedge);
		    delete[] bedge;
		}
		if (!va["-xaxis"].empty()) {
		    TAxis* xax = histo.GetXaxis();
		    xax->CenterTitle(kTRUE);
		    histo.SetXTitle(va["-xaxis"].c_str());
		}
		if (!va["-yaxis"].empty()) {
		    TAxis* yax = histo.GetYaxis();
		    yax->CenterTitle(kTRUE);
		    histo.SetYTitle(va["-yaxis"].c_str());
		}
		if (va.getDouble("-ymax") != 0) {
		    histo.SetMaximum(va.getDouble("-ymax"));
		}
		if (va.getDouble("-ymin") != 0) {
		    histo.SetMinimum(va.getDouble("-ymin"));
		}
		if (va.getDouble("-color")) {
		    histo.SetLineColor(int(va.getDouble("-color")));
		    histo.SetLineWidth(2);
		}
		if (va.getDouble("-fill")) {
		    histo.SetFillColor(int(va.getDouble("-fill")));
		}
		
		for (int i=1; i<Nsel; ++i) histo.Fill(val[i], wt[i]);

		//----------------------  Plot it
		if (!Canvas) {
		    Canvas =  new TCanvas("tablepgm", "Table Histo");
		    if (va.getBool("-ylog")) Canvas->SetLogy(1);
		    if (va.getBool("-xlog")) Canvas->SetLogx(1);
		    Canvas->SetTicks(1, 1);
		    gStyle->SetOptStat(0);
		    histo.DrawCopy("");
		} else {
		    histo.DrawCopy("Same");
		}

		if (!va.getBool("-hold")) {
		    Canvas->Update();
		    Canvas->Print(va["-file"].c_str());
		    delete Canvas;
		    Canvas = 0;
		}

	    //--------------------------  Merge two or more columns
	    } else if (cmd == "merge") {
	        va << "-join" = " ";
	        Table::col_index inx = va.scanx(pl, 1, nArg);
		string join = va["-join"];

		string name = pl[inx++];
		if (!t.exists(name)) t.insertColumn(t.size(), name);

	        for (Table::col_index i=inx; i<nArg; ++i) {
		    Table::col_index icol = t.find(name);
		    Table::col_index jcol = t.find(pl[i]);
		    if (jcol == t.size()) {
		        throw runtime_error("merge::unrecognized column");
		    }
		    t.merge(icol, jcol, join);
		    if (icol != jcol) t.removeColumn(jcol);
		}
	    }
	    
	    //--------------------------  Merge two tables
	    else if (cmd == "mergetable") {
	        va << "-columns" << "-file" << "-key" << "-range" << "-table";
		va << "-oper"    = "+";
		// Specify sorted search explicitly - note that sorted
		// searches behave differently (1xN or Nx1 instead of NxN)
		va << "-ordered"  = "false";
		va.scan(pl, 1);

		string file = va["-file"];
		if (file.empty()) throw runtime_error("File not specified");
		string table = va["-table"];
		string key   = va["-key"];
		string key2  = key;
		string range = va["-range"];
		if (!range.empty()) {
		    string::size_type ixsep = range.find(":");
		    key = range.substr(0,ixsep);
		    range.erase(0,ixsep+1);
		} else {
		    string::size_type ixsep = key.find(",");
		    if (ixsep != string::npos) {
			key.erase(ixsep, string::npos);
			key2.erase(0, ixsep+1);
		    }
		}
		const Column& keyCol1 = t[key];
		const Column* rngCol1 = 0;
		if (!range.empty()) rngCol1 = &t[range];

		//----------------------  Get the table
		if (debug > 1) cerr << "Read table: " << table << " from: "
				    << file << " ... ";
		Table t2(table, file);
		if (debug > 1) cerr << t2.getNRows() << " rows read." << endl;
		const Column& keyCol2 = t2[key2];

		//----------------------  Build two lists of column pointers
		string cols  = va["-columns"];
		col_list pCol1, pCol2;
		while (!cols.empty()) {
		    string::size_type inx = cols.find(",");
		    string name;
		    if (inx == string::npos) {
		        name = cols;
			cols.clear();
		    } else {
		        name = cols.substr(0,inx);
			cols.erase(0,inx+1);
		    }
		    pCol1.push_back(&t[name]);
		    pCol2.push_back(&t2[name]);
		}

		//----------------------  List merge criterion in verbose mode
		Column::col_order order1 = Column::kUnordered;
		if (va.getBool("-ordered")) order1 = keyCol1.getOrdering();

		if (verbose) {
		    if (rngCol1) cerr << "Merge key: " << key2 << " in range: "
				      << key << ":" << range << endl;
		    else         cerr << "Merge key: " << key2 << " == "
				      << " key: " << key << endl;
		    
		    switch (order1) {
		    case Column::kUnordered:
		    case Column::kUndefined:
			cerr << "Warning: Unsorted merge is VERY slow" << endl;
		    default:
			break;
		    }
		}

		//----------------------  Test ordering class
		string oper  = va["-oper"];
		Column::row_index n1 = keyCol1.size();
		Column::row_index n2 = keyCol2.size();
		if (!rngCol1) {
		    switch (order1) {
		    case Column::kIncreasing: {
			t2.sort(t2.find(key2), false);

			//------------------  For each t1 loop over leq t2 rows.
			Column::row_index j=0;
			for (Column::row_index i=0; i<n1; ++i) {
			    while (j<n2 && !(keyCol1[i] < keyCol2[j])) {
				if (keyCol1[i] == keyCol2[j])
				    ListColOper(pCol1, pCol2, i, j, oper);
				++j;
			    }
			}
			break;
		    }
		    case Column::kDecreasing: {
			t2.sort(t2.find(key2), true);

			//------------------  For each t1 loop over geq t2 rows.
			Column::row_index j=0;
			for (Column::row_index i=0; i<n1; ++i) {
			    while (j<n2 && !(keyCol2[j] < keyCol1[i])) {
				if (keyCol1[i] == keyCol2[j]) 
				    ListColOper(pCol1, pCol2, i, j, oper);
				++j;
			    }
			}
			break;
		    }
		    default:
			//------------------  Loop over row pairs
			for (Column::row_index i=0; i<n1; ++i) {
			    for (Column::row_index j=0; j<n2; ++j) {
				if (keyCol1[i] == keyCol2[j])
				    ListColOper(pCol1, pCol2, i, j, oper);
			    }
			}
		    }
		} else {
		    switch (order1) {
		    case Column::kIncreasing: {
			t2.sort(t2.find(key2), false);

			//------------------  Loop over row pairs
			Column::row_index j=0;
			for (Column::row_index i=0; i<n1; ++i) {
			    while (j<n2 && keyCol2[j] < (*rngCol1)[i]) {
				if (!(keyCol2[j] < keyCol1[i]))
				    ListColOper(pCol1, pCol2, i, j, oper);
				++j;
			    }
			}
			break;
		    }
		    case Column::kDecreasing: {
			t2.sort(t2.find(key2), true);

			//------------------  Loop over row pairs
			Column::row_index j=0;
			for (Column::row_index i=0; i<n1; ++i) {
			    while (j<n2 && !(keyCol2[j] < keyCol1[i])) {
				if (!(keyCol2[j] < keyCol1[i]))
				    ListColOper(pCol1, pCol2, i, j, oper);
				++j;
			    }
			}
			break;
		    }
		    default:
			//------------------  Loop over row pairs
			for (Column::row_index i=0; i<n1; ++i) {
			    for (Column::row_index j=0; j<n2; ++j) {
				if (!(keyCol2[j] < keyCol2[j]) &&
				    keyCol2[j] < (*rngCol1)[i])
				    ListColOper(pCol1, pCol2, i, j, oper);
			    }
			}
		    }
		}
	    }
	    
	    //--------------------------  Remove one or more rows or columns
	    else if (cmd == "remove") {
	        for (unsigned int i=1; i<nArg; ++i) {
		    string argi = pl[i];
		    if (argi == "-column") {
		        Table::col_index inx = t.find(pl[++i]);
			if (inx < t.size()) t.removeColumn(inx);
			else                cerr << "Remove: Column " << pl[i]
						 << " not found." << endl;
		    } else if (argi == "-rows") {
		        eval evcmd(t, pl[++i], debug);
			evcmd.setDictionary(d);
			int N = t.getNRows();
			vector<bool> mask(N);
			for (int i=0; i<N; ++i) {
			    mask[i] = !evcmd.evalRow_bool(i);
			}
			t.selectRows(mask);
			if (verbose) {
			    cerr << "Removed " << N-t.getNRows() << " rows, "
				 << t.getNRows() << " remain." << endl;
			}
		    } else {
		        cerr << "Unrecognized key: " << argi << endl;
			break;
		    }
		}
	    }
	    
	    //--------------------------  Replace values in a column
	    else if (cmd == "replace") {
	        va << "-column" << "-rows" << "-with";
		va << "-order" = "up" ;
		va.scan(pl,1);

		//----------------------  Get the order option.
		bool order_up = true;
		if (va["-order"] == "down") {
		    order_up = false;
		} else if (va["-order"] != "up") {
		    cerr << "replace: Unrecognized order option '"
			 << va["-order"] << "'" << endl;
		    break;
		}

		eval evwith(t, va["-with"], debug);
		evwith.setDictionary(d);

		Column& col(t[ va["-column"] ]);
		Column::row_index N = t.getNRows();

		if (va["-rows"].empty()) {
		    if (order_up) {
			for (Column::row_index i=0; i<N; ++i) {
			    col.refCell(i) = Cell(evwith.evalRow(i));
			}
		    } else {
			for (Column::row_index i=N; i != 0; --i) {
			    col.refCell(i-1) = Cell(evwith.evalRow(i-1));
			}
		    }
		} else {
		    eval evsel(t, va["-rows"], debug);
		    evsel.setDictionary(d);
		    if (order_up) {
			for (Column::row_index i=0; i<N; ++i) {
			    if (evsel.evalRow_bool(i)) {
				col.refCell(i) = Cell(evwith.evalRow(i));
			    }
			}
		    } else {
			for (Column::row_index i=N; i != 0; --i) {
			    if (evsel.evalRow_bool(i-1)) {
				col.refCell(i-1) = Cell(evwith.evalRow(i-1));
			    }
			}
		    }
		}
	    }

	    //--------------------------  Set column attributes
	    else if (cmd == "setcolumn") {
		Column& col(t[ pl[1] ]);

		va << "-title" << "-format" << "-extern" << "-justify";
		va.scan(pl, 2, nArg);

		if (!va["-title"].empty())   col.setTitle(va["-title"]);
		if (!va["-extern"].empty())  col.setXType(va["-extern"]);
		if (!va["-format"].empty())  col.setFormat(va["-format"]);
		if (!va["-justify"].empty()) col.setJustify(va["-justify"]);
	    }

	    //--------------------------  Sort rows
	    else if (cmd == "sort") {
	        bool   decrease = false;
		Table::sort_enum sType = Table::kAuto;
	        for (unsigned int i=1; i<nArg; ++i) {
		    string argi = pl[i];
		    if (argi == "-down") {
		        decrease = true;
		    } else if (argi == "-key") {
		        Table::col_index inx = t.find(pl[++i]);
			if (inx < t.size()) t.sort(inx, decrease, sType);
			else                cerr << "Sort: key " << pl[i]
						 << " not found." << endl;
		    } else if (argi == "-up") {
		        decrease = false;
		    } else if (argi == "-sort-type") {
		        argi = pl[++i];
			if (argi == "auto") {
			    sType = Table::kAuto;
			} else if (argi == "bubble") {
			    sType = Table::kBubble;
			} else if (argi == "merge") {
			    sType = Table::kMerge;
			} else {
			    cerr << "Unrecognized sort type: " << argi << endl;
			}			
		    } else {
		        cerr << "Unrecognized sort option: " << argi << endl;
			break;
		    }
		}
	    }

	    //--------------------------  Print column status
	    else if (cmd == "status") {
		t.status();
	    }

	    //--------------------------  Write out the table
	    else if (cmd == "write") {
	        va << "-html" << "-xml" << "-noheader" << "-rows";
		va << "-summary" << "-nobody" << "-template";
		va << "-file" = "-";
		va.scan(pl, 1);

		string ftype;
		if (va.getBool("-html")) ftype = "html";
		if (va.getBool("-xml"))  ftype = "xml";

		int  flags=3;
		if (va.getBool("-nobody"))   flags &= ~2;
		if (va.getBool("-noheader")) flags &= ~1;
		if (va.getBool("-summary"))  flags |= 4;

		//----------------------  Test for template request.
		if (va.getBool("-template")) {

		    //------------------  Open the output stream.
		    ostream* out = 0;
		    if (va["-file"] == "-") {
			out = &cout;
		    } else {
			out = new ofstream(va["-file"].c_str());
		    }
		    *out << "#" << endl;
		    *out << "# Template file for " << t.getTitle() << " table."
			 << endl;
		    *out << "#" << endl;

		    //------------------  Loop over columns, add definitions.
		    for (size_t i=0; i<t.size(); ++i) {
			const Column& ci(t[i]); 
			*out << "addcolumn -title " << ci.getTitle();
			if (!ci.refXType().empty()) {
			    *out << " -extern " << ci.refXType();
			}
			*out << endl;
		    }
		    *out << "#" << endl;
		    if (out != &cout) delete out;
		    continue;
		}

		//----------------------  Write table data
		string selcmd=va["-rows"];
		if (selcmd.empty()) {
		    t.write(va["-file"], flags, ftype);
		} else {
		    eval evcmd(t, selcmd, debug);
		    evcmd.setDictionary(d);
		    Column::row_index N = t.getNRows();
		    vector<bool> mask(N);
		    for (Column::row_index i=0; i<N; ++i) {
			mask[i] = evcmd.evalRow_bool(i);
		    }
		    Table tmp(t);
		    tmp.selectRows(mask);
		    tmp.write(va["-file"], flags, ftype);
		}
	    }

	    //--------------------------  Make an X-Y plot
	    else if (cmd == "xyplot") {
		int N = t.getNRows();
		if (!N) throw runtime_error("Empty table");

		va << "-color" = "2";
		va << "-file" = "tablepgm.ps";
		va << "-hold";
		va << "-rows";
		va << "-title";
		va << "-xaxis";
		va << "-xlog" = "false";
		va << "-xmin" = "0";
		va << "-xmax" = "0";
	        va << "-xval";
		va << "-yaxis";
		va << "-ylog" = "false";
		va << "-ymin" = "0";
		va << "-ymax" = "0";
	        va << "-yval";
		va.scan(pl, 1);

		//----------------------  Get row selection.
		vector<bool> mask(N);
		int Nsel = 0;
		string select=va["-rows"];
		if (!select.empty()) {
		    eval selcmd(t, select, debug);
		    selcmd.setDictionary(d);
		    for (int i=0; i<N; ++i) {
			mask[i] = selcmd.evalRow_bool(i);
			if (mask[i]) Nsel++;
		    }
		} else {
		    for (int i=0; i<N; ++i) mask[i] = true;
		    Nsel = N;
		}
		if (!Nsel) throw runtime_error("No rows selected");


		//----------------------  Calculate the values & weights
		std::vector<double> xval(Nsel);
		std::vector<double> yval(Nsel);

		eval xcmd(t, va["-xval"], debug);
		xcmd.setDictionary(d);
		eval ycmd(t, va["-yval"], debug);
		ycmd.setDictionary(d);

		int j=0;
		for (int i=0; i<N; ++i) {
		    if (mask[i]) {
			xval[j] = xcmd.evalRow_numeric(i);
			yval[j] = ycmd.evalRow_numeric(i);
			j++;
		    }
		}

		//----------------------  Get the histogram limits
		double xmin = va.getDouble("-xmin");
		double xmax = va.getDouble("-xmax");
		double ymin = va.getDouble("-ymin");
		double ymax = va.getDouble("-ymax");
		if (xmax <= xmin) {
		    xmin = xval[0];
		    xmax = xval[0];
		    for (int i=1; i<Nsel; ++i) {
		        double x = xval[i];
			if      (x < xmin) xmin = x;
			else if (x > xmax) xmax = x;
		    }
		}

		if (ymax <= ymin) {
		    ymin = yval[0];
		    ymax = yval[0];
		    for (int i=1; i<Nsel; ++i) {
		        double x = xval[i];
			if (x >= xmin && x < xmax) {
			    double y = yval[i];
			    if      (y < ymin) ymin = y;
			    else if (y > ymax) ymax = y;
			}
		    }
		}

		//----------------------  Set up the canvas
		if (!Canvas) {
		    Canvas = new TCanvas("tablepgm", "Table Histo");
		    if (va.getBool("-ylog")) Canvas->SetLogy(1);
		    if (va.getBool("-xlog")) Canvas->SetLogx(1);
		    gStyle->SetOptStat(0);
		    Canvas->SetTicks(1, 1);

		    //------------------  Plot the axes
		    string title = va["-title"];
		    if (title.empty()) {
			title = va["-yval"] + " vs. " + va["-xval"];
		    }
		    TH2D axis("TblPlot", title.c_str(), 
			      100, xmin, xmax, 100, ymin, ymax);
		    if (!va["-xaxis"].empty()) {
			TAxis* xax = axis.GetXaxis();
			xax->CenterTitle(kTRUE);
			axis.SetXTitle(va["-xaxis"].c_str());
		    }
		    if (!va["-yaxis"].empty()) {
			TAxis* yax = axis.GetYaxis();
			yax->CenterTitle(kTRUE);
			axis.SetYTitle(va["-yaxis"].c_str());
		    }
		    axis.DrawCopy();
		}

		//----------------------  Make a PolyMarker
		TPolyMarker xyplot;
		xyplot.SetMarkerStyle(20);
		xyplot.SetMarkerColor(int(va.getDouble("-color")));
		xyplot.DrawPolyMarker(Nsel, &xval.front(), &yval.front());
		Canvas->Update();
		if (!va.getBool("-hold")){
		    Canvas->Print(va["-file"].c_str());
		    delete Canvas;
		    Canvas = 0;
		}

	    //--------------------------  Define a symbol or error
	    } else {
	        unsigned int inx=cmd.find("=");
		if (!inx || inx >= cmd.size()-1) {
		    cerr << "Unrecognized command: " << cmd << endl;
		} else {
		    string sym = cmd.substr(0,inx);
		    inx++;
		    string val = cmd.substr(inx, cmd.size()-inx);
		    deblank(sym);
		    deblank(val);
		    d[sym]=val;
		    if (debug) cerr << "Symbol: " << sym << " set to " 
				    << val << endl;
		}
	    }

	//------------------------------  Error in a command.
	} catch (exception& e) {
	    cerr << "Exception: " << e.what() << ", caught at line: " 
		 << pl.getLineNumber() << endl;
	}
	if (verbose) usertime();
    }
}

void
help(ostream& out) {
    out << "Syntax: " << endl;
    out << "  tablepgm [options] <file1> [... <fileN>]" << endl;
    out << "Processing options: " << endl;
    out << "  -c <cmd>    Execute a list of commands" << endl;
    out << "  -i <cfile>  Execute commands in the specified file" << endl;
    out << "  -l <layout> Execute layout file before input tables" << endl;
    out << "  -t <name>   Table name" << endl;
    out << "  --debug [n] Set the debug level." << endl;
    out << "  --help      Print this message and exit." << endl;
    out << "  --version   Print cvs version ID and exit." << endl; 
    out << "Output options: " << endl;
    out << "  -b          Print table body only" << endl;
    out << "  -html       Write table as html" << endl;
    out << "  -o <ofile>  output file name" << endl;
    out << "  -s          Print sum of all numeric columns" << endl;
    out << "  -xml        Write table as xml" << endl;
    out << "Commands: " << endl;
    out << "addcolumn addrows histogram merge remove replace setcolumn "
	<< "sort write" << endl;
}
