/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TABLE_CELL_HH
#define TABLE_CELL_HH
//#include "format.hh"
#include <string>
#include <stdexcept>

//======================================  Cell class definition
namespace table {

    class format_spec;

    /**  The Cell class contains the contents of a table cell. The cell data
      *  may be a numeric or non-numeric string. If the string is detemined 
      *  to be numeric, it is converted and stored as a double precision 
      *  floating point number in addition to the string.
      *  @memo Table cell class.
      *  @author John Zweizig (john.zweizig@ligo.org)
      *  @version $Id: cell.hh 7980 2018-01-05 01:26:54Z john.zweizig@LIGO.ORG $
      */
    class Cell {
    public:
	/**  Cell contents enumerator
	  */
	enum cell_type {
	    kUnknown,    ///< Contents type is unknown
	    kNull,       ///< Cell contains a null string.
	    kNumeric,    ///< Cell contains a number
	    kString      ///< Cell contains a string
	};

	/**  Construct an empty cell.
	  *  \brief Default constructor
	  */
	explicit Cell(void);

	/**  Construct a cell and set it to a string value.
	  *  \brief String constructor
	  *  \param str Initial string value.
	  */
	explicit Cell(const std::string& str);

	/**  Construct a cell and set it to a string value.
	  *  \brief String constructor
	  *  \param str Initial string value.
	  */
	explicit Cell(const char* str);

	/**  Construct a cell and set it to a numeric value.
	  *  \brief String constructor
	  *  \param num Initial numeric value.
	  */
	explicit Cell(double num);

	/**  Compare the cell contents to that of another cell. If the 
	  *  cell content is numeric, the numeric values are compared. 
	  *  \brief Compare cell contents (equal)
	  *  \param c Cell to be compared to this.
	  *  \return True if the contents of this cell are equal to the 
	  *          contents of the argument.
	  */
	bool operator==(const Cell& c) const;

	/**  Compare the cell contents to that of another cell.  If the 
	  *  cell content is numeric, the numeric values are compared. 
	  *  \brief Compare cell contents (less than)
	  *  \param c Cell to be compared to this.
	  *  \return True if the contents of this cell are equal to the 
	  *          contents of the argument.
	  */
	bool operator<(const Cell& c) const;

	/**  Add the contents of the argument cell to those of the current cell.
	  *  If the contents of both cells are numeric, the numeric values 
	  *  are added and the cell is set to the resulting value. Otherwise,
	  *  the argument cell string is appended to the this cell string.
	  *  \brief Add cell contents.
	  *  \param c Cell to be added to this.
	  *  \return Reference to this cell.
	  */
	Cell& operator+=(const Cell& c);

	/**  Append the contents of the argument cell to those of the current 
	  *  cell. The result is assumed to be string type.
	  *  \brief Append cell contents.
	  *  \param c Cell to be appended to this.
	  *  \return Reference to this cell.
	  */
	Cell& operator|=(const Cell& c);

	/**  Append a string to the this cell contents.
	  *  \brief Append a string.
	  *  \param x String to be appended to the cell contents.
	  */
	void append(const std::string& x);

	/**  Get the numeric value of this cell.
	  *  \brief Get the cell value.
	  *  \exception A \c std::runtime_error exception is thrown if the
	  *             cell contents are not numeric,
	  *  \return Double precision numeric value.
	  */
	double getNumeric(void) const;

	/**  Get the string value of this cell.
	  *  \brief Get the cell contents.
	  *  \return Constant reference to the Cell contents string.
	  */
	const std::string& getString(void) const;

	/**  Get the cell contents data type.
	  *  \brief get cell data type
	  *  \return Enumerated Cell data type.
	  */
	cell_type getType(void) const;

	/**  Get the width of the cell data when represnted as a atring.
	  *  \brief Get the string width.
	  *  \return Width of the string representation of the cell contents.
	  */
	unsigned int getWidth(void) const;

	/**  Reformat numeric string accordnig to the specified format
	  *  \brief reformat numeric data
	  *  \param fmt Format specifier
	  */
	void reformat(const format_spec& fmt);

	/**  Detemine the cell contents data type. The internal type code is
	  *  set according to whether the contents string is empty (kNull),
	  *  convertable to a number (kNumeric) or other (kString).
	  *  \brief Set data type code.
	  */
	void setType(void);

	/**  Set the cell contents to the specified numeric value.
	  *  \brief Set cell to a numeric value.
	  *  \param x Numeric value.
	  */
	void setNumeric(double x);

    private:
	cell_type mType;
	std::string mContent;
	double mNumeric;
    };

    //=====================================  Inline methods
    inline bool
    Cell::operator==(const Cell& c) const {
	if (mType == kNumeric && c.mType == kNumeric) 
	    return (mNumeric == c.mNumeric);
	return (mContent == c.mContent);
    }
    
    inline bool
    Cell::operator<(const Cell& c) const {
	if (mType == kNumeric && c.mType == kNumeric) 
	    return (mNumeric < c.mNumeric);
	return (mContent < c.mContent);
    }
    
    inline unsigned int
    Cell::getWidth(void) const {
	return mContent.size();
    }
    
    inline const std::string& 
    Cell::getString(void) const {
	return mContent;
    }
    
    inline Cell::cell_type
    Cell::getType(void) const {
	return mType;
    }
}
#endif // !defined(TABLE_CELL_HH)
