/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "seg_ioxsil.hh"
#include "xsil/MetaIO.hh"
#include "xsil/ligolw.hh"
#include "S6SegWriter.hh"
#include "Time.hh"
#include <iostream>
#include <sstream>
#include <stdexcept>

#define PIDCVSHDR "$Header$"
#define PIDTITLE  "seg_calc S6 format segment writer"
#include "ProcIdent.hh"

//======================================  Segment xml constants
static const char* tab_segdef    = "segment_definer";
static const char* tab_segment   = "segment";
static const char* tab_summary   = "segment_summary";

static const char* col_def_id      = "segment_def_id";
static const char* col_def_ifo     = "ifos";
static const char* col_def_name    = "name";
static const char* col_def_version = "version";

static const char* col_seg_defid   = "segment_def_id";
static const char* col_seg_start   = "start_time";
static const char* col_seg_end     = "end_time";

static const char* col_sum_defid   = "segment_def_id";
static const char* col_sum_start   = "start_time";
static const char* col_sum_end     = "end_time";

//======================================  Segment xml I/O segment definer
struct seg_definer {
public:
    seg_definer(const std::string& defID, const segID& seg);
    const std::string& def_id(void) const;
    const std::string& ifo(void) const;
    const std::string& name(void) const;
    int                version(void) const;
    const segID&       seg_id(void)const ;
private:
    std::string _def_id;
    segID       _seg_id;
};

inline
seg_definer::seg_definer(const std::string& defID, const segID& segid) 
    : _def_id(defID), _seg_id(segid)
{}

inline const std::string& 
seg_definer::def_id(void) const {
    return _def_id;
}

inline const std::string& 
seg_definer::ifo(void) const {
    return _seg_id.ifo();
}

inline const std::string& 
seg_definer::name(void) const {
    return _seg_id.name();
}

inline const segID& 
seg_definer::seg_id(void) const {
    return _seg_id;
}

inline int
seg_definer::version(void) const {
    return _seg_id.version();
}

typedef std::vector<seg_definer> segdef_vect;

using namespace std;


//======================================  Constructor
seg_ioxsil::seg_ioxsil(void)
{}

//======================================  Destructor
seg_ioxsil::~seg_ioxsil(void)
{}

//======================================  Read a segment file
void 
seg_ioxsil::read_seg(const std::string& file, const segID& select, 
		     const std::string& format) {
    xsil::MetaIO meta(file.c_str(), tab_segdef);
    if (!meta.is_open()) {
	string msg = "Unable to open file: ";
	msg += file;
	throw runtime_error(msg);
    }

    //----------------------------------  Initialize column indices
    int inx_def_id =-1, inx_def_name=-1, inx_def_version=-1;
    try {
	int nDim = meta.getNColumn();
	for (int i=0; i<nDim; ++i) {
	    string colname = meta.getColumnName(i);
	    if (colname == col_def_id) {
		inx_def_id = i;
	    }
	    else if (colname == col_def_name) {
		inx_def_name = i;
	    }
	    else if (colname == col_def_version) {
		inx_def_version = i;
	    }
	}
    } catch (...) {
	cerr << "Caught exception while reading column headers" << endl;
	throw;
    }
    if ( inx_def_id<0 || inx_def_name<0 || inx_def_version < 0) {
	throw runtime_error("Missing definer column");
    }

    //----------------------------------  Loop over rows
    segdef_vect sdvect;
    while (meta.getRow() == 1) {
	string name  = meta.getString(col_def_name);
	string ifo   = meta.getString(col_def_ifo);
	int    versn = meta.getInt(col_def_version);
	segID seg_id(name, ifo, versn);
	if (!seg_id.test(select)) continue;
	string id   = meta.getString(col_def_id);
	sdvect.push_back(seg_definer(id, seg_id)); 
    }
    meta.close();

    //----------------------------------  Print selected segment definitions
    int nDef = sdvect.size();
    if (mDebug > 1) {
	cout << "Selected definers: " << endl;
	for (int i=0; i<nDef; ++i) {
	    cout << sdvect[i].def_id() << " "  
		 << sdvect[i].seg_id().full_name() 
		 << endl;
	}
    }

    //----------------------------------  Select segment table
    meta.open(file.c_str(), tab_segment);
    //if (!meta.is_open()) {
    //throw runtime_error(string("Unable to find table: ")+tab_segment);
    //}

    //----------------------------------  Loop over rows
    string last_def;
    segID  seg_id("", "");
    while (meta.getRow() == 1) {
	string id = meta.getString(col_seg_defid);
	if (id != last_def) {
	    for (int i=0; i<nDef; ++i) {
		if (sdvect[i].def_id() == id) {
		    last_def = id;
		    seg_id = sdvect[i].seg_id();
		    if (mSegMap.find(seg_id) == mSegMap.end()) {
			seg_map::value_type node(seg_id, LockSegList());
			mSegMap.insert(node);
		    }
		    break;
		}
	    }
	    if (id != last_def) continue;	    
	}
	int start = meta.getInt(col_seg_start);
	int end   = meta.getInt(col_seg_end);
	mSegMap[seg_id].stuff(LockSegment(0, Time(start), Time(end), 0));
    }
    meta.close();

    //----------------------------------  Select segment summary table
    meta.open(file.c_str(), tab_summary);
    if (!meta.is_open()) return; //-----  Missing summary not fatal.

    //----------------------------------  Loop over rows
    last_def.clear();
    while (meta.getRow() == 1) {
	string id = meta.getString(col_sum_defid);
	if (id != last_def) {
	    for (int i=0; i<nDef; ++i) {
		if (sdvect[i].def_id() == id) {
		    last_def = id;
		    seg_id = segID(sdvect[i].name() + "_summary",
				   sdvect[i].ifo(), 
				   sdvect[i].version());
		    if (mSegMap.find(seg_id) == mSegMap.end()) {
			seg_map::value_type node(seg_id, LockSegList());
			mSegMap.insert(node);
		    }
		    break;
		}
	    }
	    if (id != last_def) continue;	    
	}
	int start = meta.getInt(col_sum_start);
	int end   = meta.getInt(col_sum_end);
	mSegMap[seg_id].stuff(LockSegment(0, Time(start), Time(end), 0));
    }
    meta.close();
}

//======================================  Write an xsil segment file
void 
seg_ioxsil::write_seg(const seg_map& smap, const segID& select, 
		      const std::string& format, const std::string& file) 
{
    if (debug()) cerr << "Writing segments matching: " << select 
		      << " to XML file: " << file << endl;

    trig::S6SegWriter writer;
    Time early(0), late(0);
    size_t nTypes(0), nSelect(0), nTotal(0); 
    for (const_seg_iter i=smap.begin(); i!=smap.end(); ++i) {
	nTypes++;
	segID myId = i->first;
	if (myId.test(select)) {
	    //--------------------------  Get the segment list
	    const LockSegList& ll = i->second;
	    size_t nSeg = ll.size();
	    
	    //--------------------------  Get summary list (for unique segID)
	    const_seg_iter sitr = smap.end();
	    if (!select.name().empty()) {
		segID sumId = myId;
		sumId.set_name(myId.name() + "_summary");
		sitr = smap.find(sumId);
	    }
	    size_t nSum = 0, jSum = 0;
	    if (sitr != smap.end()) nSum = sitr->second.size();
	    const LockSegList& sl = sitr->second;
	    
	    nSelect++;
	    nTotal += nSeg;
	    Time endSum(0);
	    for (size_t j=0; j<nSeg; j++) {
		//----------------------  Fill in summary segments before active
		while (jSum < nSum &&
		       sl[jSum].getEndTime() <= ll[j].getStartTime()) {
		    if (endSum < sl[jSum].getStartTime()) {
			endSum = sl[jSum].getStartTime();
		    }
		    Time eTime = ll[j].getStartTime();
		    if (sl[jSum].getEndTime() < eTime) {
			eTime =  sl[jSum].getEndTime();
			jSum++;
		    }
		    if (endSum < eTime) {
			trig::Segment sumSeg(myId.name(), myId.version(), 
					     endSum, eTime);
			sumSeg.setIfos(myId.ifo().c_str());
			sumSeg.setActivity(0);
			if (!early || endSum < early) early = endSum;
			if (!late  || eTime  > late)  late  = eTime;
			writer.addSegment(sumSeg, *ProcIdent);
			endSum = eTime;
		    }
		}
		//----------------------  Fill active segment
		trig::Segment seg(myId.name(), myId.version(), 
				  ll[j].getStartTime(), ll[j].getEndTime());
		seg.setIfos(myId.ifo().c_str());
		if (!early || seg.getStartTime() < early) 
		    early = seg.getStartTime();
		if (!late  || seg.getEndTime()   > late)
		    late  = seg.getEndTime();
		writer.addSegment(seg, *ProcIdent); 
		endSum = seg.getEndTime();
	    }
	    while (jSum < nSum) {
		if (endSum < sl[jSum].getStartTime()) {
		    endSum = sl[jSum].getStartTime();
		}
		Time eTime = sl[jSum].getEndTime();
		jSum++;
		if (endSum < eTime) {
		    trig::Segment sumSeg(myId.name(), myId.version(), 
					 endSum, eTime);
		    sumSeg.setIfos(myId.ifo().c_str());
		    sumSeg.setActivity(0);
		    if (!early || endSum < early) early = endSum;
		    if (!late  || eTime  > late)  late  = eTime;
		    writer.addSegment(sumSeg, *ProcIdent);
		    endSum = eTime;
		}
	    }
	    
	}
    }
    if (debug()) {
	cerr << "writing: " << nTotal << " segments from: " << nSelect 
	     << "/" << nTypes << " selected segment types." << endl;
    }
    writer.write(file, early, late);
}
