/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "seg_iotext.hh"
#include "Time.hh"
#include "ParseLine.hh"
#include "fzstream.hh"
#include <iostream>
#include <fstream>
#include <sstream>
#include <stdexcept>

using namespace std;

//======================================  Constructor
seg_iotext::seg_iotext(void)
{}

//======================================  Destructor
seg_iotext::~seg_iotext(void)
{}

//======================================  Read a segment file
void 
seg_iotext::read_seg(const std::string& file, const segID& select, 
		     const std::string& fmt) {
    int iStart(-1), iName(-1), iEnd(-1), iDt(-1), iID(-1), iVsn(-1), iAct(-1);
    int nCol(1);

    //----------------------------------  Parse format specifier
    if (!fmt.empty()) {
	string pfmt(fmt);
	for (int i=0; !pfmt.empty(); ++i) {
	    switch (pfmt[0]) {
	    case 's':
		iStart = i;
		break;
	    case 'e':
		iEnd = i;
		break;
	    case 'd':
		iDt = i;
		break;
	    case 'n':
		iName = i;
		break;
	    case 'v':
		iVsn = i;
		break;
	    case 'a':
		iAct = i;
		break;
	    case 'i':
		iID = i;
		break;
	    default:
		break;
	    }
	    string::size_type n=pfmt.find(":");
	    if (n != string::npos) {
		pfmt.erase(0,pfmt.find(":")+1);
		nCol++;
	    } else {
		pfmt.clear();
	    }
	}

	if (mDebug) cerr << "Start/Name/End/Dt/ID=" << iStart << "/" << iName
			 << "/"  << iEnd << "/" << iDt << "/" << iID << endl;

	if (iStart < 0 || (iEnd < 0 && iDt < 0)) {
	    throw runtime_error(string("Format is insufficient: ")+fmt);
	}
    }

    //----------------------------------  Open the input file
    istream* istrm = 0;
    if ( file == "-") {
	istrm = &cin;
    } else {
	istrm = new fz_istream(file.c_str());
    }
    ParseLine pl(*istrm);
    if (!pl.isOpen()) 
	throw runtime_error(string("Unable to open segment file: ")+file);

    //----------------------------------  Loop over segment lines.
    seg_iter lclitr = mSegMap.end();
    segID lclID("", "", 0);
    while (pl.getLine()>=0) {
	int nArg = pl.getCount();
	if (!nArg) continue;
	if (iStart < 0) {
	    if (mDebug) cerr << "Identifying fields in: " << endl;
	    for (int i=0; i<nArg; ++i) {
		if (mDebug) cerr << pl[i] << " ";
		if (string(pl[i]).find_first_not_of("0123456789") == string::npos) {
		    double t=pl.getDouble(i);
		    if (t>6e8 && t<2e9) {
			if (iStart < 0) iStart = i;
			else iEnd = i;
		    } else if (t>0 && iStart >= 0) {
			iDt = i;
		    } else {
			iID = i;
		    }
		} else {
		    iName = i;
		}
	    }
	    if (iStart<0 || (iEnd<0 && iDt<<0)) {
		if (istrm != &cin) delete istrm;
		throw runtime_error("Unable to identify segment list fields");
	    } else if (mDebug) {
		cerr << endl;
		cerr << "Start/Name/End/Dt/ID=" << iStart << "/" << iName
		     << "/"  << iEnd << "/" << iDt << "/" << iID << endl;
	    }
	    nCol = nArg;
	}

	//-----------------------------  Check that the line is wide enough
	if (nArg < nCol) {
	    cerr << "Line: " << pl.getLineNumber() << " is too short ("
		 << nArg << " words) for format." << endl;
	    continue;
	}

	//-----------------------------  check for active
	if (iAct >= 0 && pl.getInt(iAct) == 0) {
	    if (mDebug) cout << "Not active" << endl;
	    continue;
	}

	//-----------------------------  Get the segment name
	segID myID(select);
	if (iName >= 0) {
	    segID newID(pl[iName]);
	    myID.set_name(newID.name());
	    if (!newID.ifo().empty()) myID.set_ifo(newID.ifo());
	    if (newID.version() != segID::kUnknown) {
	        myID.set_version(newID.version());
	    }
	}
	if (iVsn  >= 0) myID.set_version(pl.getInt(iVsn));

	if (mDebug > 2) {
	    cout << "text:read_seg: My ID: " << myID.full_name() 
		 << " select ID: " << select.full_name() << endl;
	}
	if (!myID.test(select)) {
	    if (mDebug > 1) cerr << myID.full_name() << " not selected" << endl;
	    continue;
	}

	//-----------------------------  Get the segment time
	Time tStart(pl.getInt(iStart));
	Interval dT;
	if (iDt>=0) dT = pl.getDouble(iDt);
	else        dT = Time(pl.getInt(iEnd)) - tStart;
	int ID = iID<0 ? 0: pl.getInt(iID);
	LockSegment seg(ID, tStart, dT);

	//-----------------------------  Fill in the segment
	if (lclitr == mSegMap.end() || myID != lclID) {
	    lclitr = mSegMap.find(myID);
	    if (lclitr == mSegMap.end()) {
		seg_value node(myID, LockSegList(myID.full_name().c_str()));
		mSegMap.insert(node);
		lclitr = mSegMap.find(myID);
	    }
	    lclID = myID;
	}
	lclitr->second.stuff(seg);
    }
    if (istrm != &cin) delete istrm;
}


//======================================  Write a text file
void 
seg_iotext::write_seg(const seg_map& smap, const segID& select, 
		      const string& format, const string& sfile) {

    //----------------------------------  Open the output stream
    ostream* of=0;
    bool     alloc_stream=false;
    if (sfile.empty() || sfile == "-") {
	of = &cout;
    } else {
	of = new ofstream(sfile.c_str());
	if (!dynamic_cast<ofstream*>(of)->is_open()) 
	    throw runtime_error(string("Unable to open file:") + sfile);
	alloc_stream = true;
    }

    //----------------------------------  Loop over segment lists
    for (const_seg_iter i=smap.begin(); i!=smap.end(); ++i) {
	if (i->first.test(select)) {

	    //--------------------------  Fix name if appropriate
	    string fmt(format);
	    string::size_type inx = fmt.find("%n");
	    if (inx != string::npos) fmt.replace(inx, 2, i->first.name());
	    inx = fmt.find("%f");
	    if (inx != string::npos) fmt.replace(inx, 2, i->first.full_name());

	    //--------------------------  Loop over segments.
	    int N=i->second.size();
	    for (int inx=0; inx<N; ++inx) {
		i->second[inx].write(*of, fmt) << endl;
	    }
	}
    }
    if (alloc_stream) delete of;
}
