#include "LscCalib.hh"
#include "Dacc.hh"
#include "DVecType.hh"
#include "ParseLine.hh"
#include "xsil/Xwriter.hh"

using namespace std;

double dF(0.25);
double fMax(4096);

//======================================  Read a flat text file into an FSeries
typedef pair<double,fComplex>  dc_pair;
FSeries
readTxtFSeries(const char* file) {
    ParseLine pl(file);
    if (!pl.isOpen()) throw runtime_error("Unable to open file");
    pl.setComment("#%");
    vector<dc_pair> vec;
    vec.reserve(1572864);
    while (pl.getLine() >= 0) {
        int nArg = pl.getCount();
	if (!nArg) continue;
	if (nArg != 3) throw runtime_error("nColumns != 3 in FSeries file.");
	vec.push_back(dc_pair(pl.getDouble(0), 0));
	vec.back().second.setMArg(pl.getDouble(1), pl.getDouble(2));
    }
    long nRow = vec.size();
    double f0 = vec[0].first;
    double df = (vec[nRow-1].first - f0) / double(nRow - 1);
    DVectC cv;
    cv.Extend(nRow);
    for (long i=0; i<nRow; i++) cv[i] = vec[i].second;
    FSeries r(f0, df, Time(0), Interval(1./df), cv);
    return r.interpolate(0, fMax, dF);
}

//======================================  Read a flat text file into a TSeries
typedef std::pair<long,double> ld_pair;

TSeries
readTxtTSeries(const char* file) {
    ParseLine pl(file);
    if (!pl.isOpen()) throw runtime_error("Unable to open file");
    pl.setComment("#%");
    vector<ld_pair> vec;
    vec.reserve(1572864);
    while (pl.getLine() >= 0) {
        int nArg = pl.getCount();
	if (!nArg) continue;
	if (nArg != 2) throw runtime_error("nColumns != 2 in TSeries file.");
	vec.push_back(ld_pair(pl.getInt(0), pl.getDouble(1)));
    }
    long nRow = vec.size();
    Time     t0 = Time(vec[0].first);
    Interval dt = Time(vec[1].first) - t0;
    DVectF dv;
    dv.Extend(nRow);
    for (long i=0; i<nRow; i++) {
        long j = long((Time(vec[i].first) - t0) / dt + 0.5);
	if (j >= nRow) dv.Extend(j + nRow - i);
        dv[j]  = vec[i].second;
    }
    TSeries r(t0, dt, dv);
    return r;
}

//====================================== dumpFS
inline void
dumpFS(const std::string& name, const FSeries& fs) {
    cerr << "       " << name << " fMax = " << fs.getHighFreq()
	 << " dF = " << fs.getFStep() << " len = " << fs.getLength() << endl;
}

//====================================== Main program
int
main(int argc, const char* argv[]) {
    LscCalib cal("", "");
    string chan, out;
    bool syntax=false;
    CalibChanList::cal_format format(CalibChanList::cfmt_default);
    for (int i=1; i<argc; i++) {
        string argi = argv[i];
	if (argi == "-alpha") {
	    cal.setAlpha(readTxtTSeries(argv[++i]));
	} else if (argi == "-c") {
	    chan = argv[++i];
	    if (!chan.empty()) cal.setChannel(chan);
	} else if (argi == "-cav") {
	    cal.setSensingFunction(readTxtFSeries(argv[++i]));
	} else if (argi == "--debug") {
	    cal.setDebug(strtol(argv[++i], 0, 0));
	} else if (argi == "-df") {
	    dF = strtod(argv[++i], 0);
	} else if (argi == "-f") {
	    cal.readFrame(argv[++i], "", chan);
	} else if (argi == "-fmax") {
	    fMax = strtod(argv[++i], 0);
	} else if (argi == "-gamma") {
	    cal.setAlphaBeta(readTxtTSeries(argv[++i]));
	} else if (argi == "-i") {
	    chan = string(argv[++i]) + ":LSC-AS_Q";
	} else if (argi == "-o") {
	    out = argv[++i];
	} else if (argi == "-olg") {
	    cal.setOpenLoopGain(readTxtFSeries(argv[++i]));
	} else if (argi == "-resp") {
	    cal.setResponseFunction(readTxtFSeries(argv[++i]));
	} else if (argi == "-v1") {
	    format=CalibChanList::cfmt_framev1;
	} else if (argi == "-v2") {
	    format=CalibChanList::cfmt_framev2;
	} else if (argi == "-x") {
	    try {
	        cal.readXml(argv[++i], "", chan);
	    } catch (std::exception& e) {
	        cerr << "Exception: " << e.what() 
		     << " caught while reading file: " << argv[i] << endl;
		syntax=true;
		break;	
	    }
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    syntax=true;
	    break;
	}
    }
    if (syntax) {
        cerr << "Command line syntax:" << endl;
        cerr << "mkcalibfile [-c <channel>] [-cav <sensing-text>] "
	     << "[-f <frame-in>] [-fmax <max-freq>] [-i <ifo>]" << endl;
	cerr << "[-o <out>] [-olg <open-loop-gain>] [-x <xml-in>]"
	     << "[-gamma <gamma-file>] [-alpha <alpha-file>] [-v1 | -v2]" 
	     << endl;
	cerr << "[-resp <response-function>]" << endl;
	return 1;
    }

    //----------------------------------  Interpolate transfer functions
    if (cal.getDebug()) {
        cerr << "Starting interpolation: fMax = " << fMax 
	     << " dF = " << dF << endl;
	dumpFS("    OpenLoopGain", cal.refOpenLoopGain());
	dumpFS("ResponseFunction", cal.refResponseFunction());
	dumpFS(" SensingFunction", cal.refSensingFunction());
    }
    cal.prepare();
    cal.setOpenLoopGain(cal.refOpenLoopGain().interpolate(0,fMax,dF));
    cal.setResponseFunction(cal.refResponseFunction().interpolate(0,fMax,dF));
    cal.setSensingFunction(cal.refSensingFunction().interpolate(0,fMax,dF));
    if (cal.getDebug()) {
        cerr << "Finished interpolation: " << endl;
	dumpFS("    OpenLoopGain", cal.refOpenLoopGain());
	dumpFS("ResponseFunction", cal.refResponseFunction());
	dumpFS(" SensingFunction", cal.refSensingFunction());
    }

    string::size_type olen = out.size();

    try {
        switch (format) {
	case CalibChanList::cfmt_framev1:
	case CalibChanList::cfmt_framev2:
	    cal.writeFrame(out.c_str(), format);
	    break;
	default:
	    if (!olen) {
	        xsil::Xwriter xw(cout);
		cal.writeXml(xw);
	    } else if (olen > 4 && out.substr(olen-4, 4) == ".gwf") {
	        cal.writeFrame(out.c_str(), CalibChanList::cfmt_framev1);
	    } else {
	        cal.writeXml(out.c_str());
	    }
	}
    } catch (std::exception& e) {
        cerr << "Exception during write: " << e.what() << endl;
    }
    return 0;
}
