/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef TRIGRSLT_HH
#define TRIGRSLT_HH

#include "Interval.hh"
#include "TrigBase.hh"
#include <vector>

namespace trig {
   class TrigBase;
   class TrigRaw;

   /**  The trigger result object contains a trigger dependent list of data
    *  that describes the trigger. In the present implementation, the data 
    *  list is a vector of double precision floats. Other trigger-dependent 
    *  result classes may be defined based on the TrigBase class.
    *  \brief Trigger result class.
    *  \author J. Zweizig
    *  \version $Id: TrigRslt.hh 8013 2018-03-01 22:33:34Z john.zweizig@LIGO.ORG $
    */ 
   class TrigRslt : public TrigBase {
   public:
      /** Result data type.
       */
      typedef double param_type;

      TrigRslt(void) {}
      
      /**  A result object is created and the parameter data are copied into 
       *  the trigger data list.
       *  \brief Create a result object.
       *  \param ID    Trigger type identifier string.
       *  \param SubID Trigger sub-type string.
       *  \param nData Number of data words.
       *  \param DList Array of data words.
       */
      TrigRslt(const std::string& ID, const std::string& SubID=0, 
	       index_type nData=0, const param_type* DList=0);

      /**  Release the data after recording the trigger.
       *  @memo Result destructor.
       */
      virtual ~TrigRslt(void);

      /**  The number of defined data words is returned.
       *  \brief Number of data words.
       *  \return Number of data words.
       */
      index_type getNData(void) const;

      /**  Get a specified result data word. 
       *  @memo Get a data word.
       *  \param i Number of parameter to be read.
       *  \return Constant reference to selected parameter.
       */
      const param_type& operator[](unsigned int i) const;

      /**  Get or set a specified data word.
       *  \brief Read/Modify a data word.
       *  \param i Number of parameter to be read/modified.
       *  \return Reference to selected parameter.
       */
      param_type& operator[](unsigned int i);

   private:
      typedef std::vector<param_type> result_vect;
      result_vect mResult;
   };   // trig::TrigRslt

}    //  namespace trig

//--------------------------------------  Get the number of data words.
inline trig::TrigBase::index_type 
trig::TrigRslt::getNData(void) const {
   return mResult.size();
}

//--------------------------------------  Get a data word.
inline const trig::TrigRslt::param_type& 
trig::TrigRslt::operator[](unsigned int i) const {
   return mResult[i];
}

//--------------------------------------  Read/Modify a data word
inline trig::TrigRslt::param_type& 
trig::TrigRslt::operator[](unsigned int i) {
   return mResult[i];
}

#endif   //  TRIGRSLT_HH
