/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef TRIGBASE_HH
#define TRIGBASE_HH

#include "Interval.hh"
#include "Time.hh"
#include <string>
#include <iostream>

namespace trig {
   class TrigRaw;

   /**  The trigger disposition is a bit-mask indicating where a trigger 
    *  is to be routed. The current options include logging in the meta-
    *  data database (#d_metaDB#) and issuing an operator alarm through 
    *  Epics (#d_alarm#). By default, triggers are routed only to the
    *  database.
    *  @memo Trigger disposition codes
    */
   enum TrigDisp {
      /** Trigger is not routed past the trigger manager
       */
      d_none = 0,

      /**  Record trigger in the meta-database.
       */
      d_metaDB = 1,

      /**  Generate an epics alarm
       */
      d_alarm  = 2,

      /**  Record trigger in the test database.
       */
      d_testDB = 4,

      /**  All defined destinations.
       */
      d_all = 7
   };

   /**  The TrigPrio enumerator indicates the importance of a trigger.
     *  Four levels are currently defined:
     *  @memo Trigger priority enumeration
     */
   enum TrigPrio {
      /**  Triggered condition is not expected to affect data integrity.
        *  @memo Information only.
	*/
      p_info = 0,

      /**  Triggered condition will affect the recorded data and should 
        *  be considered in the data analysis.
	*  @memo Warning level.
	*/
      p_warn,

      /**  The trigger describes a major short term disruption. It is
        *  likely that the data will not be usable until the effects
	*  of the triggered disruption subside.
	*  @memo Major disturbance to the data.
	*/
      p_error,

      /**  The trigger results from a fault in the apparatus which must
        *  be repaired or reset before data taking can proceed.
	*  @memo Hardware fault.
	*/
      p_severe
   };

   /**  The trigger base class contains the information needed from all 
     *  triggers. It may be used as a base for a specific trigger object. 
     *  Data contained in the trigger base provide all the information 
     *  needed to select triggers for cross correlation studies, etc. 
     *  \brief Trigger result base class.
     *  \author J. Zweizig
     *  \version $Id: TrigBase.hh 8013 2018-03-01 22:33:34Z john.zweizig@LIGO.ORG $
     */
   class TrigBase {
   public:
      /**  Index data type.
        */
      typedef unsigned int index_type;
      // static const int maxDataLength = 1024;

   public:

      //----------------------------------  Constructors
      /**  Construct an empty trigger object.
        *  @memo Default constructor.
	*/
      TrigBase(void);

      /**  A trigger object is created and initialized as specified by the
        *  constructor arguments.
	*  @memo Create a trigger object.
	*  @param ID     Primary trigger identifier
	*  @param SubID  Secondary trigger i dentifier.
	*  @param When   Trigger time
	*  @param dTime  Trigger duration
	*  @param Size   Trigger Peak amplitude
	*  @param Signif Trigger significance (S/N)
	*  @param Frequency Trigger center frequency.
	*  @param ifo    Trigger ifo name.
	*/
      TrigBase(const std::string& ID, const std::string& SubID="",
	       const Time& When=Time(0), const Interval& dTime=0.0,
	       double Size=1.0, double Signif=1.0, double Frequency=0.0,
	       const std::string& ifo="");

      /**  Release the data after recording the trigger.
        *  @memo Destroy a TrigBase object.
	*/
      virtual ~TrigBase(void) {}

      /**  Clone a trigger object.
        *  \brief  Clone a TrigBase instance.
	*  \return Pointer to the TrigBase clone.
        */
      virtual TrigBase* clone(void) const;
      
      //----------------------------------  Accessors
      /**  Get the triggered frequency bandwidth.
        *  @memo Trigger frequency.
	*  @return Trigger effect low frequency.
	*/
      double getBandwidth(void) const;

      /**  Get the trigger disposition mask.
        *  @memo Get the trigger disposition mask.
	*  @return Trigger disposition mask.
	*/
      int getDisposition(void) const;

      /**  The trigger time offset from the start of the data epoch is
        *  returned.
	*  @memo Get trigger duration.
	*  @return The interval corresponding to the duration of the 
	*          effect being trigger on.
	*/
      Interval getDt(void) const;

      /**  Get the midpoint of triggered frequency band.
        *  @memo Trigger middle frequency.
	*  @return Trigger effect middle frequency.
	*/
      double getFrequency(void) const;

      /**  Get the trigger ID.
        *  @memo Get the trigger ID.
	*  @return Trigger ID.
	*/
      const char* getID(void) const;

      /**  Get the IFO(s) in which data the trigger was found.
        *  \brief Get the trigger IFO list.
	*  \return Trigger IFO list.
	*/
      const char* getIFOs(void) const;

      /**  Get the lower edge of the triggered frequency band.
        *  @memo Low trigger frequency.
	*  @return Trigger effect low frequency.
	*/
      double getLowF(void) const;

      /**  Get the trigger effect intensity in units of the trigger threshold.
        *  @memo Trigger Intensity.
	*  @return Trigger effect intensity.
	*/
      double getIntensity(void) const;

      /**  Get the upper edge of the triggered frequency band.
        *  @memo High trigger frequency.
	*  @return Trigger effect high frequency.
	*/
      double getHighF(void) const;

      /**  Get the priority level for this trigger.
        *  @memo   Get trigger priority.
	*  @return Trigger priority level.
	*/
      TrigPrio getPriority(void) const;

      /**  Get the process ID for this trigger.
        *  @memo Trigger process ID.
	*  @return Unique process ID.
	*/
      const char* getProcess(void) const;

      /**  Get the trigger effect significance in units of the error on
        *  the trigger amplitude.
	*  @memo Trigger Significance.
	*  @return Trigger effect significance.
	*/
      double getSignificance(void) const;

      /**  The trigger sub-ID is returned. The sub-ID is defined by the 
        *  trigger generator to indicate more precisely the reason for the
	*  trigger. This is typically the channel name.
	*  \brief Get the trigger sub-ID.
	*  @return trigger sub-ID.
	*/
      const char* getSubID(void) const;

      /**  Get the time at which the effect being triggered on starts.
        *  @memo Get trigger time.
	*  @return GPS start time of effect.
	*/
      Time getTime(void) const;

      /**  Get the time offset from the start time to the time of the peak 
        *	 sample in the trigger
	*  @return Peak time offset.
	*/
      Interval getPeakOffset(void) const;

      /**  Get the time offset from the start time to the power-weighted time
        *	 average.
	*  @return Average time offset.
	*/
      Interval getAvgOffset(void) const;

      /**  Get the time of the peak sample or pixel in the trigger
        *  \brief Peak time.
	*  @return Trigger peak time.
	*/
      Time     getPeakTime(void) const;

      /**  Get the power-weighted signal time.
        *  \brief Average time.
	*  \return Trigger average time.
	*/
      Time     getAvgTime(void) const;

      /**  Get the width of the power distribution in time.
        *  \return Width of the time distribution.
	*/
      Interval getTimeSigma(void) const;

      /**  Get the peak frequency of the trigger.
        *  \return Peak frequency.
	*/
      double   getFreqPeak(void) const;

      /**  Get the power weighted average frequency.
        *  \return Average frequency.
	*/
      double   getFreqAvg(void) const;

      /**  Get the sigma of the power distribution in frequency.
        *  \return Sigma of frequency distribution.
	*/
      double   getFreqSigma(void) const;

      /**  Get the noise power in the trigger time-frequency interval.
        *  \return Noise power.
	*/
      double   getNoisePower(void) const;

      /**  Get the signal power in the trigger time-frequency interval.
        *  \return Signal power.
	*/
      double   getSignalPower(void) const;

      /**  Get the number of time frequency pixels needed to span the 
        *  trigger.
	*  \return Number of pixels.
	*/
      int      getPixelCount(void) const;

      /**  Get the confidence limit for the trigger.
        *  \return Confidence level.
	*/
      double   getConfidence(void) const;

      //-----------------------------------  Overloaded operators
      /**  Compare two triggers.
        *  \brief Compare operator.
	*  \param trigger Trigger to be compared to this instance.
	*  \return True if trgger are the same.
	*/
      bool operator==(const TrigBase& trigger) const;

      //-----------------------------------  Mutators
      /**  The specified data are appended to the trigger result data. 
        *  The result data are assumed to be binary numbers, meaning 
	*  that no content checking is performed and any number of 
	*  embedded null characters are allowed. The total result data 
	*  length must not exceed \c maxDataLength bytes.
	*  @memo Append data to result data text.
	*  @param data Pointer to result data increment.
	*  @param len  Length of result data increment in bytes.
	*/
      void appData(const char* data, index_type len);

      /**  The specified data replace a substring of the trigger result data. 
        *  The result data are treated as a binary string in that no content 
	*  checking is performed and any number of embedded null characters 
	*  are allowed. The result data length must not exceed 1024 bytes.
	*  If the initial offset is greater than the maximum data length, 
	*  the request is ignored. If the data would be copied past the end of
	*  the result array, the input data are truncated. The result length is
	*  set to the end of the new data if this value exceeds the current 
	*  result length.
	*  \brief Replace a substring of the result data.
	*  @param data Pointer to result data increment.
	*  @param inx  Offset into the result data of text to be replaced.
	*  @param len  Length of result data increment in bytes.
	*/
      void modData(const char* data, index_type inx, index_type len);

      /**  The bandwidth field of the trigger is replaced with the argument 
        *  value.
	*  @memo Set the bandwidth.
	*  @param bw Bandwidth in Hz.
	*/
      void setBandwidth(double bw);

      /**  The specified data replace the trigger result data. The result 
        *  data are assumed to be binary numbers, meaning that no content 
	*  checking is performed and any number of embedded null characters 
	*  are allowed. The result data length must not exceed 1024 bytes.
	*  @memo Append data to result data text.
	*  @param data Pointer to replacement data.
	*  @param len  Length of result data increment in bytes.
	*/
      void setData(const char* data, index_type len);

      /**  Set the disposition for this trigger.
        *  @memo Set the trigger disposition.
	*  @param disp Trigger disposition.
	*/
      void setDisposition(int disp);

      /**  Set the duration for this trigger.
        *  \brief Set the trigger duration.
	*  \param dT Trigger duration.
	*/
      void setDuration(Interval dT);

      /**  Set the typical frequency (i.e. the midpoint of the frequency band) 
        *  of the effect being triggered on.
	*  @memo Set the frequency.
	*  @param F0 Trigger effect frequency in Hz.
	*/
      void setFrequency(double F0);

      /**  Set the ID and sub-ID for this trigger.
        *  \brief Set the trigger ID.
	*  \param id    Trigger ID string.
	*  \param subid Trigger sub-ID string.
	*/
      void setID(const std::string& id, const std::string& subid);

      /**  Set the interferometer ID list for this trigger.
        *  @memo Set the interferometer list.
	*  @param IFO interferometer list.
	*/
      void setIfos(const char* IFO);

      /**  Set the intensity for this trigger.
        *  \brief Set the intensit.
	*  \param Mag trigger intensity.
	*/
      void setIntensity(double Mag);

      /**  Set the significance for this trigger.
        *  \brief Set the significance.
	*  \param sig Trigger significance.
	*/
      void setSignificance(double sig);

      /**  Set the trigger time stamp. The time stamp can be any time that 
        *  best defines the particular type of trigger, e.g. start time
	*  peak time, critical time, etc.
	*  @memo Set the trigger time.
	*  @param T0 %Time to be recorded in the trigger object.
	*/
      void setTime(const Time& T0);

      /**  Set the priority level for this trigger.
        *  @memo Set the trigger priority level.
	*  @param prio Trigger priority level.
	*/
      void setPriority(TrigPrio prio);

      /**  Set the process ID for this trigger.
        *  \brief Set the trigger process ID.
	*  \param Process Unique process ID.
	*/
      void setProcess(const char* Process);

      /**  Set the peak sample time for the trigger.
        *  \param dT Offset to peak time.
	*/
      void setPeakOffset(Interval dT);

      /**  Set the power weighted time average.
        *  \param dT %Time offset to Average time.
	*/
      void setAvgOffset(Interval dT);

      /**  Set the peak time sample for the trigger.
        *  \param t Peak time.
	*/
      void setPeakTime(const Time& t);

      /**  Set the power weighted time average.
        *  \param t Average time.
	*/
      void setAvgTime(const Time& t);

      /**  Set the sigma of the power distribution in time.
        *  \brief Time sigma
	*  \param s Sigma value.
	*/
      void setTimeSigma(Interval s);

      /**  Set the frequency of the peak in the power distribution in frequency.
        *  \param f Peak frequency.
	*/
      void setFreqPeak(double f);

      /**  Set the power weighted average frequency.
        *  \param f Average frequency.
	*/
      void setFreqAvg(double f);

      /**  Set the sigma of the power distribution in frequency.
        *  \param s Frequency sigma.
	*/
      void setFreqSigma(double s);

      /**  Set the total noise power in the time-frequency region containing 
        *  the trigger effect.
	*  \param np Noise power.
	*/
      void setNoisePower(double np);

      /**  Set the total signal power in the tim-frequency region containing 
        *  the trigger effect.
	*  \param sp Signal Power
	*/
      void setSignalPower(double sp);

      /**  Set the number of time-frequency pixels covering the trigger.
        *  \param N Number of pixels in trigger.
	*/
      void setPixelCount(int N);

      /**  Set log of the confidence of the trigger effect.
       *  \param cl Confidence level.
       */
      void setConfidence(double cl);

   private:
      /**  Creator database.
       */
      int mCreatorDB;

      /**  Identifier of the trigger source process.
       */
      std::string mProcess;

      /**  Identifier of the trigger filter.
       */
      std::string mFilter;

      /**  Descriptive name for the generated trigger.
       *  @memo Trigger ID.
       */
      std::string mName;

      /**  The sub-ID of the specified trigger indicates a program defined
       *  subdivision of the trigger type.
       *  @memo Trigger sub-ID.
       */
      std::string mSubID;

      /**  Interferometer ID.
       */
      std::string mIfo;

      /**  mTime contains the time at which the trigger occurred, or the
       *  start time of a triggered interval.
       *  @memo Trigger time.
       */
      Time mTime;

      /**  mDuration contains the duration of the effect causing the trigger.
       *  A duration <= 0 indicates that the effect preceded mTIme. Use 
       *  1/<sample-rate> to indicate a single sample transient.
       *  @memo Trigger duration.
       */
      Interval mDuration;

      /**  Trigger priority
       *  @memo Trigger priority.
       */
      TrigPrio mPriority;

      /**  Trigger disposition
       *  @memo Trigger disposition.
       */
      int mDisposition;

      /**  The trigger intensity in local units (e.g. threshold value).
       *  @memo Trigger intensity.
       */
      double mSize;

      /**  The trigger intensity in units of expected noise sigma.
       *  @memo Trigger significance.
       */
      double mSignificance;

      /**  The frequency band of the trigger disturbance.
       *  @memo Trigger signal frequency.
       */
      double mFrequency;

      /**  Width of the frequency band of the triggered disturbance.
       *  @memo Trigger signal banwidth.
       */
      double mBandwidth;

      /**  Unique event ID assigned by the database.
       *  @memo Event ID.
       */
      std::string mEventID;

      Interval mPeakTime;
      Interval mAvgTime;
      Interval mTimeSigma;
      double   mFreqPeak;
      double   mFreqAvg;
      double   mFreqSigma;
      double   mNoisePower;
      double   mSignalPower;
      int      mPixelCount;
      double   mConfidence;

   };   // TrigBase

}    //  namespace trig

#ifndef __CINT__
//======================================  Get time offset.
inline Interval 
trig::TrigBase::getAvgOffset(void) const {
   return mAvgTime;
}

inline Time     
trig::TrigBase::getAvgTime(void) const {
   return mTime + mAvgTime;
}

inline double
trig::TrigBase::getBandwidth(void) const {
   return mBandwidth;
}

inline double   
trig::TrigBase::getConfidence(void) const {
   return mConfidence;
}

inline Interval
trig::TrigBase::getDt(void) const {
   return mDuration;
}
inline int
trig::TrigBase::getDisposition(void) const {
   return mDisposition;
}

//======================================  Get frequency band floor
inline double 
trig::TrigBase::getFrequency(void) const {
   return mFrequency;
}

inline double   
trig::TrigBase::getFreqAvg(void) const {
   return mFreqAvg;
}

inline double   
trig::TrigBase::getFreqPeak(void) const {
   return mFreqPeak;
}

inline double   
trig::TrigBase::getFreqSigma(void) const {
   return mFreqSigma;
}

inline double 
trig::TrigBase::getHighF(void) const {
   return mFrequency + 0.5*mBandwidth;
}

//======================================  Get trigger ID.
inline const char*
trig::TrigBase::getID(void) const {
   return mName.c_str();
}

//======================================  Get IFO ID.
inline const char*
trig::TrigBase::getIFOs(void) const {
   return mIfo.c_str();
}

//======================================  Get trigger Intensity.
inline double
trig::TrigBase::getIntensity(void) const {
   return mSize;
}

//======================================  Get frequency band floor
inline double 
trig::TrigBase::getLowF(void) const {
   return mFrequency - 0.5*mBandwidth;
}

inline double   
trig::TrigBase::getNoisePower(void) const {
   return mNoisePower;
}

inline Interval 
trig::TrigBase::getPeakOffset(void) const {
   return mPeakTime;
}

inline Time     
trig::TrigBase::getPeakTime(void) const {
   return mTime + mPeakTime;
}

inline int      
trig::TrigBase::getPixelCount(void) const {
   return mPixelCount;
}

//======================================  Get pointer to the proces ID
inline const char*
trig::TrigBase::getProcess(void) const {
   return mProcess.c_str();
}

inline trig::TrigPrio
trig::TrigBase::getPriority(void) const {
   return mPriority;
}

inline double   
trig::TrigBase::getSignalPower(void) const {
   return mSignalPower;
}

//======================================  Get trigger Intensity.
inline double
trig::TrigBase::getSignificance(void) const {
   return mSignificance;
}

//======================================  Get sub-ID.
inline const char*
trig::TrigBase::getSubID(void) const {
   return mSubID.c_str();
}

//======================================  Get time offset.
inline Time
trig::TrigBase::getTime(void) const {
   return mTime;
}

inline Interval 
trig::TrigBase::getTimeSigma(void) const {
   return mTimeSigma;
}
#endif

#endif   //  TRIGBASE_HH
