/* -*- mode: c++; c-basic-offset: 4; -*- */
#include <unistd.h>
#include "LSCSegFind.hh"
#include <sstream>
#include <iostream>
#include <stdlib.h>

using namespace std;

//======================================  Default constructor
LSCSegFind::LSCSegFind(void) 
  : LockSegList("Xn"), mStartGPS(0), mEndGPS(0), mMode(kUnknown), mDebug(0)
{}

//======================================  Data constructor
LSCSegFind::LSCSegFind(const string& type, const string& ifo, 
		       const Time& start, const Time& end) 
  : LockSegList(ifo.c_str()), mStartGPS(start), mEndGPS(end), 
    mMode(kUnknown), mIfo(ifo), mType(type), mDebug(0)
{}

//======================================  LSCSegFind Destructor
LSCSegFind::~LSCSegFind(void) {
}

//======================================  Build a query command
void 
LSCSegFind::buildQuery(void) {
    ostringstream out;
    switch (mMode) {
    case kFile:
	break;
    case kWeb:
        out << "curl -s -S " << mUrl << "/" << mSource << " > " << mTmpFile;
	if (!mDebug) out << " 2> /dev/null";
	break;
    case kDataBase:
        if (!mUrl.empty()) {
	    string env="SEGMENTDBSERVER=";
	    env+=mSource;
	    putenv(const_cast<char*>(env.c_str()));
	}
        out << "LSCsegFind";
	out << " -t " << mType << " -s " << mStartGPS.getS() 
	    << " -e " << mEndGPS.getS();
	if (!mIfo.empty()) out << " -i " << mIfo;
	out << " > " << mTmpFile;
	if (!mDebug) out << " 2> /dev/null";
	break;
    default:
        throw runtime_error("No segment data source");
    }
    mQuery = out.str();
}

//======================================  Query the database (or file).
void 
LSCSegFind::query(void) {
    if (mMode == kFile) {
        if (mDebug) cerr << "Ingesting segments from file: " << mSource 
			 << endl;
	read(mSource);
    } else if (mMode == kUnknown) {
        throw runtime_error("No segment data source");
    } else {
        if (mTmpFile.empty()) mTmpFile = "/tmp/abcdSegFind";
	if (mQuery.empty()) buildQuery();
	if (mDebug) cerr << "Issuing query: " << mQuery << endl;
	if (system(mQuery.c_str()) != 0) {
	    throw runtime_error(string("Unable to execute command: ")+mQuery);
	}
	if (mDebug) cerr << "Ingesting segments from file: " << mTmpFile 
			 << endl;
	read(mTmpFile);
	unlink(mTmpFile.c_str());
    }
    if (mDebug) cerr << "Read segments: " << mIfo << ":" << mType
		     << " # seg read: " << size() << endl;
}

//======================================  Reset the query
void 
LSCSegFind::reset(void) {
    mQuery.clear();
    clear();
}

//======================================  Set the debug level
void 
LSCSegFind::setDebug(int lvl) {
    mDebug = lvl;
}

//======================================  Specify the IFO name
void 
LSCSegFind::setIfo(const string& ifo) {
    mIfo = ifo;
}

//======================================  Set the time Epoch
void 
LSCSegFind::setRange(const Time& start, const Time& end) {
    mStartGPS = start;
    mEndGPS   = end;
}

//======================================  Specify the source type and location
void 
LSCSegFind::setSource(in_mode mode, const string& file) {
    mMode   = mode;
    mSource = file;
}

//======================================  Set the segment type
void 
LSCSegFind::setType(const string& type) {
    mType = type;
}

//======================================  Set the source url
void 
LSCSegFind::setUrl(const string& url) {
    mUrl  = url;
    mMode = kWeb;
}
