#ifndef _LIGO_EVENTTYPE_H
#define _LIGO_EVENTTYPE_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Type							*/
/*                                                         		*/
/* Module Description: Event type					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Type.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <string>
#include <iosfwd>

namespace events {

   class TypeInfo;
   class LayoutInfo;


/** Describes the event type. An event type consists of a name and
    a subtype. Type and subtype names can not contain the characters 
    "(", ")", "[", "]" or ".".
   
    @memo Defines an event type
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Type {
   public:      
      /** Creates an empty event type.
          @memo Default constructor
       ******************************************************************/
      Type() : mInfo (0) {
      }
      /** Creates an event type.
          @memo Default constructor
   	  @param name Name representing the event type
   	  @param subtype Subtype of the event
       ******************************************************************/
      explicit Type (const char* name) 
      : mInfo (0) {
         SetType (name); }
      /** Creates an event type.
          @memo Default constructor
   	  @param name Name representing the event type
   	  @param subtype Subtype of the event
       ******************************************************************/
      explicit Type (const std::string& name)
      : mInfo (0) {
         SetType (name.c_str()); }
      /** Copy constructor.
          @memo Copy constructor
   	  @param type Event type
       ******************************************************************/
      Type (const Type& type) : mInfo (0) {
         *this = type; }
      /** Destroys the event type.
          @memo Destructor
       ******************************************************************/
      ~Type();
      /** Assignment operator.
          @memo Assignment operator
   	  @param type Event type
   	  @return *this
       ******************************************************************/
      Type& operator= (const Type& type);
   
      /** Sets the event type.
          @memo Set type
   	  @param name Name representing the event type
   	  @param subtype Subtype of the event
   	  @return true if successful
       ******************************************************************/
      bool SetType (const char* name);
      /** Get the event name.
          @memo Get event type
   	  @return name
       ******************************************************************/
      const char* GetName() const;
      /** Get the event ID.
          @memo Get event ID
   	  @return name
       ******************************************************************/
      int GetId () const;
      /** Get the type information record.
          @memo Get type information
   	  @return type information record
       ******************************************************************/
      const TypeInfo* GetTypeInfo() const {
         return mInfo; }
      /** Get the the layout information record.
          @memo Get layout information
   	  @return layout information record
       ******************************************************************/
      const LayoutInfo* GetLayout() const;
   
      /** Test if two event types are equal.
          @memo Equality operator
   	  @return true if identical event types
       ******************************************************************/
      bool operator== (const Type& t) const {
         return (mInfo == t.mInfo); }
      /** Test if two event types are unequal.
          @memo Inequality operator
   	  @return true if different event types
       ******************************************************************/
      bool operator!= (const Type& t) const {
         return  (mInfo != t.mInfo); }
      /** Smaller than operator. Compares the type Id and subtype Id.
          @memo Smaller than operator
   	  @return true if smaller 
       ******************************************************************/
      bool operator< (const Type& t) const {
         return (mInfo < t.mInfo); }
   
      /** Tests if the event type exists.
          @memo Test type
   	  @param name Name representing the event type
   	  @param subtype Subtype of the event
   	  @return true if type exists
       ******************************************************************/
      static bool Test (const char* name);
      /** Tests if the pattern matches the event type. The pattern
          can contain a wildcard.
          @memo Match operator
   	  @param pat Pattern for event name
   	  @return true if event type matches pattern
       ******************************************************************/
      bool Match (const char* pat) const;
   
      /** Check if a valid type name is specified.
          @memo Check name
          @param name Type or subtype name
          @return True if valid
       ******************************************************************/
      static bool CheckName (const char* name);
   
      /** Dump all registered types to the specified output stream.
          @memo Dump all registered types.
          @param os output stream
       ******************************************************************/
      static void DumpAll (std::ostream& os);
      /** Dump all registered types to the standard output stream.
          @memo Dump all registered types.
          @param os output stream
       ******************************************************************/
      static void DumpAll();
   
   private:
      /// Event type information
      TypeInfo*		mInfo;
   };


}

#endif // _LIGO_EVENTTYPE_H
