//////////////////////////////////////////////////////////////////////////
//  									//
//  List								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <time.h>
#include <stdio.h>
#include <iostream>
#include <fstream>
#include "events/Set.hh"
#include "events/Iterator.hh"
#include "events/Algorithm.hh"


namespace events {

   using namespace std;

//______________________________________________________________________________
   class IteratorImpSet : public IteratorImp {
   
   public:
      // List iterator
      typedef Chain::iterator chain_iterator;
      // List of chain iterators
      typedef std::vector<chain_iterator> iterator_array;
   
      // Create event list iterator implementation
      IteratorImpSet (Set& set, iterator_array iters, int cur) 
      : mSet (set), mIters (iters), mCur (cur) {
      }
      // Copy
      virtual IteratorImp* Copy() const {
         return new IteratorImpSet (*this); }
      // Get
      virtual Event* Get() const {
         return (mCur >= 0) ? &*mIters[mCur] : 0; }
      // Find next
      void Next() {
         // look for the oldest
         mCur = mIters.size() - 1;
         bool found = false;
         for (int i = 0; i < (int)mIters.size(); ++i) {
            if (mIters[i] == mSet.GetChain(i).End()) {
               continue; }
            if (!found) {
               mCur = i; found = true;
            }
            else if (mIters[i]->GetTime() < mIters[mCur]->GetTime()) {
               mCur = i;
            }
         }
      }
      // Increment
      virtual void Inc() {
         if (mCur < 0) {
            return; }
         // increment the current if not end
         if (mIters[mCur] != mSet.GetChain(mCur).End()) {
            ++mIters[mCur];
         }
         // now look for the oldest
         Next();
      }
      // Decrement
      virtual void Dec() {
         if (mCur < 0) {
            return; }
         // look for the newest iterator among the ones prior 
         // to the ones in the list
         // mCur = 0;
         bool found = false;
         chain_iterator ci, ii;
         for (int i = 0; i < (int)mIters.size(); ++i) {
            if (mIters[i] == mSet.GetChain(i).Begin()) {
               continue; }
            if (!found) {
               mCur = i; found = true;
               --(ci = mIters[mCur]);
            }
            else {
               --(ii = mIters[i]);
               if (ii->GetTime() >= ci->GetTime()) {
                  mCur = i;
               }
            }
         }
         // decrement the current if not begin
         if (mIters[mCur] != mSet.GetChain(mCur).Begin()) {
            --mIters[mCur];
         }
      }
      // Get iterator array
      iterator_array& GetIterators() {
         return mIters; }
      // Get iterator array
      const iterator_array& GetIterators() const {
         return mIters; }
      // Get chain number of current iterator
      int GetChainNumber() const {
         return mCur; }
   private:
      // Refernce to original set
      Set&		mSet;
      // List of chain iterators
      iterator_array	mIters;
      // Current iterator position
      int		mCur;
   };


//______________________________________________________________________________
   Set::Set (int N)
   : mDefault (0)
   {
      for (int i = 0; i < N; ++i) {
         AddChain (Chain());
      }
   }

//______________________________________________________________________________
   bool Set::AddChain (const char* filename)
   {
      AddChain (Chain());
      mSet.back()->AddList (filename);
      mDefault = N() - 1;
      return true;
   }

//______________________________________________________________________________
   bool Set::AddChain (const Chain& chain) 
   {
      mSet.push_back (chain); 
      mDefault = N() - 1;
      return true;
   }

//______________________________________________________________________________
   bool Set::RemoveChain (int lnum)
   {
      if ((lnum < 0) || (lnum >= N())) {
         return false;
      }
      mSet.erase (mSet.begin() + lnum);
      if ((mDefault >= lnum) && (mDefault > 0)) {
         --mDefault; 
      }
      return true;
   }

//______________________________________________________________________________
   void Set::Merge()
   {
      Chain* c = new Chain();
      std::copy (Begin(), End(), std::back_inserter (*c));
      Clear();
      mSet.push_back (Chain());
      mSet.back().Reset (c);
      mDefault = 0;
   }

//______________________________________________________________________________
   void Set::Join (Set& set)
   {
      for (eventset::iterator i = set.mSet.begin(); 
          i != set.mSet.end(); ++i) {
         mSet.push_back (Chain());
         mSet.back().Reset (i->Release());
      }
      set.Clear();
   }

//______________________________________________________________________________
   bool Set::Configure (const char* filename)
   {
   // :TODO:
      return false;
   }

//______________________________________________________________________________
   bool Set::Save (const char* filename, int perfile, 
                  int maxevents) const
   {
      // write a single file
      if (perfile <= 0) {
         std::ofstream out (filename);
         if (!out) {
            return false;
         }
         Write (out, Begin(), End(), maxevents);
         return !!out;
      }
      // write multiple files
      else {
         ConstIterator next = Begin();
         for (int i = 0; (next != End()) && maxevents; ++i) {
            char name[4*1024];
            sprintf (name, "%s.%i", filename, i);
            std::ofstream out (name);
            if (!out) {
               return false;
            }
            next = Write (out, next, End(), perfile);
            if (maxevents >= 0) {
               maxevents -= perfile;
               if (maxevents < 0) maxevents = 0;
            }
            if (!out) {
               return false;
            }
         }
         return true;
      }
   }

//______________________________________________________________________________
   bool Set::Restore(const char* filename) {
      Clear();
      return AddChain (filename);
   }

//______________________________________________________________________________
   int Set::Size() const 
   {
      int n = 0;
      for (eventset::const_iterator i = mSet.begin();
          i != mSet.end(); ++i) {
         n += (*i)->Size();
      }
      return n;
   }

//______________________________________________________________________________
   bool Set::Empty() const 
   {
      for (eventset::const_iterator i = mSet.begin();
          i != mSet.end(); ++i) {
         if (!(*i)->Empty()) {
            return false;
         }
      }
      return true;
   }

//______________________________________________________________________________
   bool Set::operator== (const Set& s) const
   {
      if (Size() != s.Size()) {
         return false;
      }
      const_iterator j = s.Begin();
      for (const_iterator i = Begin(); i != End(); ++i, ++j) {
         if (*i != *j) {
            return false;
         }
      }
      return true;
   }

//______________________________________________________________________________
   bool Set::CheckOrder() const 
   {
      return events::CheckOrder (Begin(), End());
   }

//______________________________________________________________________________
   void Set::Swap (Set& l) 
   {
      std::swap (mSet, l.mSet);
   }

//______________________________________________________________________________

   Set::iterator Set::LowerBound (const Event& e)
   {
      // :TODO: more efficient algorithm
      return std::lower_bound (Begin(), End(), e);
   }

//______________________________________________________________________________
   Set::const_iterator Set::LowerBound (const Event& e) const
   {
      // :TODO: more efficient algorithm
      return std::lower_bound (Begin(), End(), e);
   }

//______________________________________________________________________________
   Set::iterator Set::UpperBound (const Event& e)
   {
      // :TODO: more efficient algorithm
      return std::upper_bound (Begin(), End(), e);
   }

//______________________________________________________________________________
   Set::const_iterator Set::UpperBound (const Event& e) const
   {
      // :TODO: more efficient algorithm
      return std::upper_bound (Begin(), End(), e);
   }

//______________________________________________________________________________
   void Set::Sort()
   {
      Merge();
      if (N() == 1) {
         GetChain(0).Sort();
      }
   }

//______________________________________________________________________________
   Set::reference Set::At (size_type idx)
   {
      Set::iterator i = Begin();
      advance (i, idx);
      return *i;
   }

//______________________________________________________________________________
   Set::const_reference Set::At (size_type idx) const
   {
      Set::const_iterator i = Begin();
      advance (i, idx);
      return *i;
   }

//______________________________________________________________________________
   Set::reference Set::Front()
   {
      return *Begin();
   }

//______________________________________________________________________________
   Set::const_reference Set::Front() const
   {
      return *Begin();
   }

//______________________________________________________________________________
   Set::reference Set::Back()
   {
      return *--End();
   }

//______________________________________________________________________________
   Set::const_reference Set::Back() const
   {
      return *--End();
   }

//______________________________________________________________________________
   Set::iterator Set::Begin()
   {
      IteratorImpSet::iterator_array iarr;
      for (int i = 0; i < (int)mSet.size(); ++i) {
         iarr.push_back (mSet[i]->Begin());
      }
      Set::iterator i (IteratorImpSet (*this, iarr, 0));
      IteratorImpSet* imp = 
         dynamic_cast<IteratorImpSet*> (i.GetImplementation());
      imp->Next();
      return i;
   }

//______________________________________________________________________________
   Set::const_iterator Set::Begin() const
   {
      return ((Set*)this)->Begin();
   }

//______________________________________________________________________________
   Set::iterator Set::End()
   {
      IteratorImpSet::iterator_array iarr;
      for (int i = 0; i < (int)mSet.size(); ++i) {
         iarr.push_back (mSet[i]->End());
      }
      Set::iterator i (IteratorImpSet (*this, iarr, mSet.size() - 1));
      return i;
   }

//______________________________________________________________________________
   Set::const_iterator Set::End() const
   {
      return ((Set*)this)->End();
   }

//______________________________________________________________________________
   void Set::Insert (const Event& event)
   {
      CheckChain();
      mSet[mDefault]->Insert (event);
   }

//______________________________________________________________________________
   Set::iterator Set::Insert (const iterator& pos, const Event& event)
   {
      iterator p = pos;
      IteratorImpSet* imp = 
         dynamic_cast<IteratorImpSet*> (p.GetImplementation());
      int i = 0;
      if ((imp == 0) || ((i = imp->GetChainNumber()) < 0)) {
         return End();
      }
      Chain::iterator j = mSet[i]->Insert (imp->GetIterators()[i], event);
      imp->GetIterators()[i] = j;
      return p;
   }

//______________________________________________________________________________
   void Set::Insert (const iterator& beg, const iterator& end)
   {
      for (iterator i = beg; i != end; ++i) {
         Insert (*i);
      }
   }

//______________________________________________________________________________
   void Set::PushBack (const Event& event)
   {
      CheckChain();
      mSet[mDefault]->PushBack (event);
   }

//______________________________________________________________________________
   Set::iterator Set::Erase (const iterator& pos)
   {
      iterator p = pos;
      IteratorImpSet* imp = 
         dynamic_cast<IteratorImpSet*> (p.GetImplementation());
      int i = 0;
      if ((imp == 0) || ((i = imp->GetChainNumber()) < 0)) {
         return End();
      }
      Chain::iterator r = mSet[i]->Erase (imp->GetIterators()[i]);
      imp->GetIterators()[i] = r;
      imp->Next();
      return p;
   }

//______________________________________________________________________________
   Set::iterator Set::Erase (const iterator& beg, const iterator& end)
   {
      int n = 0;
      for (iterator i = beg; i != end; ++i) ++n;
      iterator j = beg;
      for (int i = 0; i < n; ++i) {
         j = Erase (j);
      }
      return j;
   }

//______________________________________________________________________________
   void Set::PopBack()
   {
      CheckChain();
      mSet[mDefault]->PopBack();
   }

//______________________________________________________________________________
   void Set::SetColumn (const Column& column, const Function& expression)
   {
      events::SetColumn (Begin(), End(), column, expression);
   }

//______________________________________________________________________________
   void Set::SetColumn (const Column& column, const Function& expression,
                     const Condition& cond, const TimeWindow& window) 
   {
      events::SetColumn (Begin(), End(), column, expression, cond, window); 
   }


//______________________________________________________________________________
   int Set::Select (const Condition& cond)
   {
      Set s;
      events::Select (Begin(), End(), std::back_inserter (s), cond, mWindow);
      *this = s;
      return Size();
   }

//______________________________________________________________________________
   int Set::Select (const Set& events, const Condition& cond)
   {
      Clear();
      events::Select (events.Begin(), events.End(), 
                     std::back_inserter (*this), cond, mWindow);
      return Size();
   }

//______________________________________________________________________________
   int Set::SelectAdd (const Set& events, const Condition& cond)
   {
      CheckChain();
      int prev = Size();
      int prevchain = mSet[mDefault]->Size();
      // iterator last_of_prev = mSet[mDefault]->End();
      // if (prevchain) --last_of_prev;
      events::Select (events.Begin(), events.End(), 
                     std::back_inserter (*this), cond, mWindow);
      int num = Size() - prev;
      if (prevchain && num) {
         iterator last_of_prev = mSet[mDefault]->Begin() + (prevchain - 1);
         iterator first_of_new = last_of_prev;
         ++first_of_new;
         if (*last_of_prev > *first_of_new) {
            Sort();
         }
      }
      return num;
   }

//______________________________________________________________________________
   int Set::Sort (const Function& func, bool ascending, int n)
   {
      if (n == 0) {
         Merge();
         if (N() == 1) {
            GetChain(0).Sort (func, ascending);
         }
      }
      else {
         Set s;
         events::Sort (Begin(), End(), std::back_inserter (s), 
                      func, ascending, n);
         *this = s;
      }
      return Size();
   }

//______________________________________________________________________________
   int Set::Sort (const Set& events, const Function& func, bool ascending, int n)
   {
      Clear();
      events::Sort (events.Begin(), events. End(), 
                   std::back_inserter (*this), func, ascending, n);
      return Size();
   }

//______________________________________________________________________________
   int Set::MultiCoincidence (int n, const TimeWindow& window, const Condition& cond)
   {
      Set s;
      events::Coincidence (Begin(), End(), n, 
                          std::back_inserter (s), window, cond);
      *this = s;
      return Size();
   }

//______________________________________________________________________________
   int Set::MultiCoincidence (int n, const Set& events, const TimeWindow& window, 
                     const Condition& cond)
   {
      Clear();
      events::Coincidence (events.Begin(), events.End(), n, 
                          std::back_inserter (*this), window, cond);
      return Size();
   }

//______________________________________________________________________________
   int Set::MultiCoincidenceAdd (int n, const Set& events, 
                     const TimeWindow& window, const Condition& cond)
   {
      CheckChain();
      int prev = Size();
      int prevchain = mSet[mDefault]->Size();
      // iterator last_of_prev = mSet[mDefault]->End();
      // if (prevchain) --last_of_prev;
      events::Coincidence (events.Begin(), events.End(), n,
                          std::back_inserter (*this), window, cond);
      int num = Size() - prev;
      if (prevchain && num) {
         iterator last_of_prev = mSet[mDefault]->Begin() + (prevchain - 1);
         iterator first_of_new = last_of_prev;
         ++first_of_new;
         if (*last_of_prev > *first_of_new) {
            Sort();
         }
      }
      return num;
   }   

//______________________________________________________________________________
   int Set::ChainCoincidence (const TimeWindow& window, const Condition& cond)
   {
      Set s;
      WindowIterator::InputStateList state;
      for (eventset::iterator i = mSet.begin(); i != mSet.end(); ++i) {
         state.push_back (WindowIterator::InputState (
                                           (*i)->Begin(), (*i)->End()));
      }
      events::Coincidence (state, std::back_inserter (s), window, cond);
      *this = s;
      return Size();
   }

//______________________________________________________________________________
   int Set::ChainCoincidence (const Set& events, const TimeWindow& window, 
                     const Condition& cond)
   {
      Clear();
      WindowIterator::InputStateList state;
      for (eventset::const_iterator i = events.mSet.begin(); 
          i != events.mSet.end(); ++i) {
         state.push_back (WindowIterator::InputState (
                                           (*i)->Begin(), (*i)->End()));
      }
      events::Coincidence (state, std::back_inserter (*this), window, cond);
      return Size();
   }

//______________________________________________________________________________
   int Set::ChainCoincidenceAdd (const Set& events, 
                     const TimeWindow& window, const Condition& cond)
   {
      CheckChain();
      int prev = Size();
      int prevchain = mSet[mDefault]->Size();
      // iterator last_of_prev = mSet[mDefault]->End();
      // if (prevchain) --last_of_prev;
      // make coincidence input list
      WindowIterator::InputStateList state;
      for (eventset::const_iterator i = events.mSet.begin(); 
          i != events.mSet.end(); ++i) {
         state.push_back (WindowIterator::InputState (
                                           (*i)->Begin(), (*i)->End()));
      }
      int num = Size() - prev;
      // check order
      if (prevchain && num) {
         iterator last_of_prev = mSet[mDefault]->Begin() + (prevchain - 1);
         iterator first_of_new = last_of_prev;
         ++first_of_new;
         if (*last_of_prev > *first_of_new) {
            Sort();
         }
      }
      return num;
   }   

//______________________________________________________________________________
   int Set::Clusters (int threshold, const TimeWindow& window, 
                     const Condition& cond)
   {
      Set s;
      events::SelectClusters (Begin(), End(), 
                           std::back_inserter (s), threshold, window, cond);
      *this = s;
      return Size();
   }

//______________________________________________________________________________
   int Set::Clusters (const Set& events, int threshold, const TimeWindow& window, 
                     const Condition& cond)
   {
      Clear();
      events::SelectClusters (events.Begin(), events.End(), 
                           std::back_inserter (*this), threshold, 
                           window, cond);
      return Size();
   }

//______________________________________________________________________________
   int Set::ClusterAdd (const Set& events, int threshold, const TimeWindow& window, 
                     const Condition& cond)
   {
      CheckChain();
      int prev = Size();
      int prevchain = mSet[mDefault]->Size();
      // iterator last_of_prev = mSet[mDefault]->End();
      // if (prevchain) --last_of_prev;
      // select clusters
      events::SelectClusters (events.Begin(), events.End(), 
                           std::back_inserter (*this), threshold,
                           window, cond);
      int num = Size() - prev;
      // check order
      if (prevchain && num) {
         iterator last_of_prev = mSet[mDefault]->Begin() + (prevchain - 1);
         iterator first_of_new = last_of_prev;
         ++first_of_new;
         if (*last_of_prev > *first_of_new) {
            Sort();
         }
      }
      return Size() - prev;
   }

//______________________________________________________________________________
   int Set::Histogram (Histogram1& hist, const Function& func, 
                     const TimeWindow& window,
                     const Condition& cond) const
   {
      return MakeHistogram (hist, Begin(), End(), func, cond, window);
   }

//______________________________________________________________________________
   int Set::Histogram (Histogram2& hist, const Function& f1, 
                     const Function& f2, const TimeWindow& window,
                     const Condition& cond) const
   
   {
      return MakeHistogram (hist, Begin(), End(), f1, f2, cond, window);
   }

//______________________________________________________________________________
   int Set::TimeSeries (TSeries& ts, const TimeWindow& window,
                     bool integrated, const Condition& cond) const
   {
      return MakeTimeSeries (ts, Begin(), End(), integrated, cond, window);
   }

//______________________________________________________________________________
   int Set::TimeSeries (TSeries& ts, const Function& func, 
                     const Condition& cond) const
   {
      return MakeTimeSeries (ts, Begin(), End(), func, cond, mWindow);
   }

//______________________________________________________________________________
   void Set::Dump(std::ostream& os, int num) const
   {
      int count = 0;
      for (ConstIterator i = Begin(); i != End(); ++i) {
         i->Dump(os);
         cout << endl;
         if (num) {
            ++count;
            if (count == num) 
               break;
         }
      }
   
   }

//______________________________________________________________________________
   void Set::Dump (int num) const
   {
      Dump (cout, num);
   }

//______________________________________________________________________________
   void Set::DumpColumn(const char* name, std::ostream& os, int num) const
   {
      int count = 0;
      for (ConstIterator i = Begin(); i != End(); ++i) {
         i->DumpColumn(name,os);
         if (num) {
            ++count;
            if (count == num) 
               break;
         }
      }
   
   }

//______________________________________________________________________________
   void Set::DumpColumn (const char* name, int num) const
   {
      DumpColumn (name, cout, num);
   }

//______________________________________________________________________________
   void Set::CheckChain()
   {
      if (N() == 0) {
         AddChain (Chain());
      }
      if ((mDefault < 0) || (mDefault >= N())) {
         mDefault = 0;
      }
   }


}
