#ifndef _LIGO_EVENTNAME_H
#define _LIGO_EVENTNAME_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Name							*/
/*                                                         		*/
/* Module Description: Event name					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Name.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <string>
#include <iosfwd>

namespace events {


/** Describes the event name. An event name can not contain the 
    characters "(", ")", "[", "]" or ".".
   
    @memo Defines an event name
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Name {
   public:      
      /** Creates an empty event name.
          @memo Default constructor
       ******************************************************************/
      Name () : mId (0) {
      }
      /** Creates an event name.
          @memo Default constructor
   	  @param name Name representing the event name
       ******************************************************************/
      explicit Name (int id);
      /** Creates an event name.
          @memo Constructor
   	  @param name Name representing the event name
       ******************************************************************/
      explicit Name (const char* name) 
      : mId (0) {
         SetName (name); }
      /** Creates an event name.
          @memo Constructor
   	  @param name Name representing the event name
       ******************************************************************/
      explicit Name (const std::string& name)
      : mId (0) {
         SetName (name.c_str()); }
   
      /** Sets the event name.
          @memo Set name
   	  @param name Name representing the event name
   	  @return true if successful
       ******************************************************************/
      bool SetName (const char* name);
      /** Get the event name.
          @memo Get event name
   	  @return name
       ******************************************************************/
      const char* GetName() const;
      /** Get the event ID.
          @memo Get event ID
   	  @return name
       ******************************************************************/
      int GetId () const {
         return mId; }
      /** Test if two event names are equal.
          @memo Equality operator
   	  @return true if identical event names
       ******************************************************************/
      bool operator== (const Name& n) const {
         return (mId == n.mId); }
      /** Test if two event names are unequal.
          @memo Inequality operator
   	  @return true if different event names
       ******************************************************************/
      bool operator!= (const Name& n) const {
         return  (mId != n.mId); }
   
      /** Get the event ID of the given string. Will not register
          the name but rather return 0 if the name does not exists.
          @memo Get event ID
   	  @return ID
       ******************************************************************/
      static int GetNameId (const char* name);
      /** Tests if the event name exists.
          @memo Test name
   	  @param name Name representing the event name
   	  @return true if type exists
       ******************************************************************/
      static bool Test (const char* name);
      /** Tests if the pattern matches the event name. The pattern
          can contain a wildcard.
          @memo Match operator
   	  @param pat Pattern for event name
   	  @return true if event name matches pattern
       ******************************************************************/
      bool Match (const char* pat) const;
   
      /** Check if a valid name name is specified.
          @memo Check name
          @param name Name
          @return True if valid
       ******************************************************************/
      static bool CheckName (const char* name);
   
      /** Dump all registered names to the specified output stream.
          @memo Dump all registered names.
          @param os output stream
       ******************************************************************/
      static void DumpAll (std::ostream& os);
      /** Dump all registered names to the standard output stream.
          @memo Dump all registered names.
          @param os output stream
       ******************************************************************/
      static void DumpAll();
   
   private:
      /// Event name id
      int		mId;
   };


}

#endif // _LIGO_EVENTNAME_H
